-- SPDX-FileCopyrightText: 2017-2021 
--
-- SPDX-License-Identifier: AGPL-3.0-or-later

{-# LANGUAGE FlexibleInstances #-}
{-# LANGUAGE ScopedTypeVariables #-}
{-# OPTIONS_GHC -fno-warn-orphans #-}
module QuickCheck where

import Test.QuickCheck
import Test.Tasty
import Test.Tasty.QuickCheck

import qualified Data.List as List
import qualified Data.Set as Set

import Reloto

quickchecks :: TestTree
quickchecks =
  testGroup "QuickCheck"
   [ testProperty "rankOfCombin n (combinOfRank n k r) == r" $ \(NKR (n,k,r)) ->
    rankOfCombin n (combinOfRank n k r) == r
   , testProperty "combinOfRank n (length ns) (rankOfCombin n ns) == ns" $ \(SortedNNS (n,ns)) ->
    combinOfRank n (toInteger $ length ns) (rankOfCombin n ns) == ns
   , testProperty "rankOfSequence n (sequenceOfRank n k r) == r" $ \(NKR (n,k,r)) ->
    rankOfSequence n (sequenceOfRank n k r) == r
   , testProperty "sequenceOfRank n (length ns) (rankOfSequence n ns) == ns" $ \(NNS (n,ns)) ->
    sequenceOfRank n (toInteger $ length ns) (rankOfSequence n ns) == ns
   ]

-- * Type 'NNS'
newtype NNS = NNS (Integer, [Integer])
 deriving (Eq, Show)
instance Arbitrary NNS where
  arbitrary = do
    n <- arbitrarySizedNatural
    k <- choose (0,n)
    ns <- List.take (fromInteger k) . nubList <$> infiniteListOf (choose (1,n))
    return $ NNS (n,ns)

-- * Type 'SortedNNS'
newtype SortedNNS = SortedNNS (Integer, [Integer])
 deriving (Eq, Show)
instance Arbitrary SortedNNS where
  arbitrary = do
    n <- arbitrarySizedNatural
    k <- choose (0,n)
    ns <- List.sort . List.take (fromInteger k) . nubList <$> infiniteListOf (choose (1,n))
    return $ SortedNNS (n,ns)

-- * Type 'NKR'
newtype NKR = NKR (Integer, Integer, Integer)
 deriving (Eq, Show)
instance Arbitrary NKR where
  arbitrary = do
    n <- arbitrarySizedNatural
    k <- choose (0, n)
    r <- choose (0, (n`nCk`k) - 1)
    return $ NKR (n,k,r)

-- | Like 'nub', but O(n * log n).
nubList :: Ord a => [a] -> [a]
nubList = go Set.empty where
  go _ [] = []
  go s (x:xs) | x`Set.member`s = go s xs
              | otherwise      = x:go (Set.insert x s) xs
