-- --------------------------------------------------------------------------
--  $Revision: 808 $ $Date: 2009-02-10 00:19:07 +0100 (Tue, 10 Feb 2009) $
-- --------------------------------------------------------------------------

-- |
--
-- Module      :  Exec.Encode.Main
-- Copyright   :  Otakar Smrz 2005-2009
-- License     :  GPL
--
-- Maintainer  :  otakar.smrz mff.cuni.cz
-- Stability   :  provisional
-- Portability :  portable
--
-- "Encode.Arabic" "Encode.Mapper" "Encode.Unicode"


module Main where

import Encode
import Encode.Arabic
import Encode.Unicode

import System.Environment
import System.Console.GetOpt
import System.IO

import Data.Char
import Data.List
import Data.Version (showVersion)

import Version

version = Version [1, 1, build] []

    where Version [build] [] = revised "$Revision: 808 $"


data Opts = DisplayUsage | PrintVersion | LineOriented |

            Prefix String | Suffix String

    deriving (Eq, Ord)


options :: [OptDescr Opts]

options = [ Option ['h'] ["help"]       (NoArg  DisplayUsage)
                                                "show usage information",

            Option ['l'] ["lines"]      (NoArg  LineOriented)
                                                "use line-oriented mode",

            Option ['p'] ["prefix"]     (ReqArg Prefix "text")
                                                "prefix input with text",

            Option ['s'] ["suffix"]     (ReqArg Suffix "text")
                                                "suffix input with text",

            Option ['v'] ["version"]    (NoArg  PrintVersion)
                                                "show program's version" ]


copyleft = unlines ["Encode Arabic (C) 2009-2005 Otakar Smrz",
                    "GNU General Public License http://www.gnu.org/licenses/"]

synopsis = unlines [copyleft,
                    "         http://sourceforge.net/projects/encode-arabic/",
                    "                    http://quest.ms.mff.cuni.cz/encode/",
                    "                              <otakar.smrz mff.cuni.cz>",
                    "",
                    "encode [--OPTIONS] [ENCODING]"]


main = do   argv <- getArgs

            let (opts, nons, errs) = getOpt Permute options argv

            if null errs then case (sort . nub) opts of

                DisplayUsage : _    ->  warn (usageInfo synopsis options)
                PrintVersion : _    ->  warn (unlines [copyleft,
                                              unwords ["Encode Arabic",
                                                       showVersion Main.version,
                                                       "February 2009"]])
                LineOriented : _    ->  execute True (tail opts) nons
                _                   ->  execute False      opts  nons

                         else           warn (usageInfo synopsis options)


warn = hPutStr stderr


execute l opts nons = case opts of

                        [Prefix p]  -> interactPlus l e (p ++)
                        [Prefix p, Suffix s]
                                    -> interactPlus l e ((p ++) . (++ s))
                        [Suffix s]  -> interactPlus l e (++ s)
                        _           -> interactPlus l e id

    where e = case nons of  [] -> ""
                            _  -> head nons


interactPlus :: Bool -> String -> (String -> String) -> IO ()

interactPlus l e f = interact (if l then unlines . map decodes . lines
                                    else decodes)

    where decodes = encodes . decode UTF . f

          encodes = case map toLower e of

                        "arabtex"       ->  encode ArabTeX
                        "lagally"       ->  encode Lagally
                        "tex"           ->  encode TeX

                        "zdmg"          ->  encode ZDMG
                        "arabtex_zdmg"  ->  encode ArabTeX_ZDMG
                        "arabtex-zdmg"  ->  encode ArabTeX_ZDMG

                        "buckwalter"    ->  encode Buckwalter
                        "tim"           ->  encode Tim

                        "winarabic"     ->  encode WinArabic
                        "cp1256"        ->  encode CP1256
                        "windows_1256"  ->  encode Windows_1256
                        "windows-1256"  ->  encode Windows_1256

                        "isoarabic"     ->  encode ISOArabic
                        "iso_8859_6"    ->  encode ISO_8859_6
                        "iso-8859-6"    ->  encode ISO_8859_6

                        "macarabic"     ->  encode MacArabic

                        "macfarsi"      ->  encode MacFarsi

                        "dosarabic"     ->  encode DOSArabic
                        "cp864"         ->  encode CP864

                        "dosfarsi"      ->  encode DOSFarsi
                        "cp1006"        ->  encode CP1006

                        "asmo449"       ->  encode ASMO449

                        "isiri3342"     ->  encode ISIRI3342

                        "utf8"          ->  encode UTF8
                        "utf"           ->  encode UTF

                        _               ->  encode Unicode
