{-# OPTIONS_GHC -Wall #-}
{-# LANGUAGE NoImplicitPrelude #-}
{-# LANGUAGE FlexibleInstances #-}
{-# LANGUAGE FunctionalDependencies #-}

module Data.Aviation.Navigation.WindParameters (
  WindParameters(..)
, WindParameters'
, HasWindParameters(..)
, optWindParameters
, optWindParametersVersion
) where

import Control.Category ( Category(id, (.)) )
import Control.Lens ( Lens' )
import Data.Aviation.Navigation.Vector
    ( Vector, vectorDegrees )
import Data.Eq ( Eq )
import Data.Functor ( Functor(fmap), (<$>) )
import Data.Maybe ( Maybe(Just, Nothing) )
import Data.Ord ( Ord )
import Data.Semigroup((<>))
import GHC.Show(Show)
import Options.Applicative
    ( Applicative((<*>)),
      Alternative((<|>)),
      auto,
      flag',
      help,
      long,
      metavar,
      option,
      short,
      Parser )
import Prelude(Double)

data WindParameters a =
  WindParameters
    (Vector a) -- trk, TAS
    (Vector a) -- wind direction, wind speed
  deriving (Eq, Ord, Show)

type WindParameters' =
  WindParameters Double

class HasWindParameters a c | a -> c where
  windParameters ::
    Lens' a (WindParameters c)
  {-# INLINE trackTAS #-}
  trackTAS ::
    Lens' a (Vector c)
  trackTAS =
    windParameters . trackTAS
  {-# INLINE windDirSpeed #-}
  windDirSpeed ::
    Lens' a (Vector c)
  windDirSpeed =
    windParameters . windDirSpeed

instance HasWindParameters (WindParameters a) a where
  windParameters =
    id
  {-# INLINE trackTAS #-}
  trackTAS f (WindParameters tt wsd) =
    fmap (`WindParameters` wsd) (f tt)
  {-# INLINE windDirSpeed #-}
  windDirSpeed f (WindParameters tt wsd) =
    fmap (WindParameters tt) (f wsd)

optWindParameters ::
  Parser WindParameters'
optWindParameters =
  (\trk tas wd ws -> WindParameters (vectorDegrees trk tas) (vectorDegrees wd ws)) <$>
    option auto (
      long "trk" <>
      short 't' <>
      metavar "TRACK" <>
      help "Track Direction °"
    )
    <*>
    option auto (
      long "tas" <>
      short 'a' <>
      metavar "TRUE_AIRSPEED" <>
      help "True Air Speed kt"
    )
    <*>
    option auto (
      long "wd" <>
      short 'd' <>
      metavar "WIND_DIRECTION" <>
      help "Wind Direction °"
    )
    <*>
    option auto (
      long "ws" <>
      short 's' <>
      metavar "WIND_SPEED" <>
      help "Wind Speed kt"
    )

optWindParametersVersion ::
  Parser (Maybe WindParameters')
optWindParametersVersion =
  flag'
    Nothing
    (
      short 'v' <>
      long "version" <>
      help "the program version"
    ) <|>
    Just <$> optWindParameters
