module Text.BBCode.Internal.Helper
  ( module Text.BBCode.Internal.Helper
  , module Control.Lens
  , module Data.Text
  , module Data.String
  , module Text.BBCode.Internal.Types
  )
where

import Control.Lens hiding (List)
import Data.String (IsString, fromString)
import Data.Text (Text)
import Text.BBCode.Internal.Types

{- | Get representation of element in form of @IsString a => a@
@elName HR == "hr"@
@elName Align == "align"@
@elName ListElement == "*"@
-}
elName :: IsString a => El -> a
-- Void elements
elName HR = "hr"
elName BR = "br"
elName Clear = "clear"
elName ListElement = "*"
-- No argumets
elName Bold = "b"
elName Italic = "i"
elName Underline = "u"
elName Strikethrough = "s"
elName Indent = "indent"
elName NFO = "nfo"
elName Oneline = "oneline"
elName Code = "code"
elName Preformatted = "pre"
-- Optional argument
elName Box = "box"
elName Image = "img"
elName Quote = "quote"
elName Spoiler = "spoiler"
elName List = "list"
-- One argument
elName Color = "color"
elName Size = "size"
elName URL = "url"
elName Align = "align"
elName Font = "font"

{- | Elements that can be opening
e.g. @HR@: @"[hr]"@ or @Bold@: @"[bold]"@
-}
elementOpenings :: [El]
elementOpenings = [HR .. Font]

{- | Elements that can be opening and can take argument
e.g. @Box@: @"[box=right]"@
-}
elementArgOpenings :: [El]
elementArgOpenings = [Box .. Font]

{- | Elements that can be closing
e.g. @Bold@: @"[/bold]"@
-}
elementClosings :: [El]
elementClosings = [Bold .. Font]

show' :: (Show a, IsString b) => a -> b
show' = fromString . show
{-# INLINE show' #-}

surround :: Text -> Text -> Text
surround s x = mconcat [s, x, s]
{-# INLINE surround #-}

opening :: El -> Text
opening = (\x -> mconcat ["[", x, "]"]) . elName
{-# INLINE opening #-}

openingArg :: El -> Text -> Text
openingArg el arg = mconcat ["[", elName el, "=", arg, "]"]
{-# INLINE openingArg #-}

closing :: El -> Text
closing x = mconcat ["[/", elName x, "]"]
{-# INLINE closing #-}

wrap :: El -> Text -> Text
wrap el x = mconcat ["[", elName el, "]", x, "[/", elName el, "]"]
{-# INLINE wrap #-}

wrapArg :: El -> Text -> Text -> Text
wrapArg el arg x = mconcat ["[", elName el, "=", arg, "]", x, "[/", elName el, "]"]

-- | Combine two predicates with OR
(|||) :: (a -> Bool) -> (a -> Bool) -> (a -> Bool)
(|||) pL pR c = pL c || pR c
{-# INLINE (|||) #-}
