{-# LANGUAGE NumericUnderscores #-}
{-# LANGUAGE OverloadedStrings #-}
{-# LANGUAGE QuasiQuotes #-}
{-# LANGUAGE TemplateHaskell #-}

module Cardano.Blocks
  where

import Data.Aeson (decode, eitherDecode, encode)
import Data.Text (Text)
import qualified Money
import Test.Hspec
import Test.Tasty.Hspec
import Text.RawString.QQ

import Blockfrost.Types

spec_blocks :: Spec
spec_blocks = do
  it "parses block sample" $ do
    eitherDecode blockSample
    `shouldBe`
    Right blockExpected

  it "parses affected addresses sample" $ do
    eitherDecode affectedAddrsSample
    `shouldBe`
    Right affectedAddrsExpected

  it "parses transactions with cbor sample" $ do
    eitherDecode txHashesCBORSample
    `shouldBe`
    Right txHashesCBORExpected


blockSample = [r|
{
    "time": 1641338934,
    "height": 15243593,
    "hash": "4ea1ba291e8eef538635a53e59fddba7810d1679631cc3aed7c8e6c4091a516a",
    "slot": 412162133,
    "epoch": 425,
    "epoch_slot": 12,
    "slot_leader": "pool1pu5jlj4q9w9jlxeu370a3c9myx47md5j5m2str0naunn2qnikdy",
    "size": 3,
    "tx_count": 1,
    "output": "128314491794",
    "fees": "592661",
    "block_vrf": "vrf_vk1wf2k6lhujezqcfe00l6zetxpnmh9n6mwhpmhm0dvfh3fxgmdnrfqkms8ty",
    "op_cert": "da905277534faf75dae41732650568af545134ee08a3c0392dbefc8096ae177c",
    "op_cert_counter": "18",
    "previous_block": "43ebccb3ac72c7cebd0d9b755a4b08412c9f5dcb81b8a0ad1e3c197d29d47b05",
    "next_block": "8367f026cf4b03e116ff8ee5daf149b55ba5a6ec6dec04803b8dc317721d15fa",
    "confirmations": 4698
}
|]

blockExpected = Block
  { _blockTime = 1641338934
  , _blockHeight = pure 15243593
  , _blockHash = "4ea1ba291e8eef538635a53e59fddba7810d1679631cc3aed7c8e6c4091a516a"
  , _blockSlot = pure 412162133
  , _blockEpoch = pure 425
  , _blockEpochSlot = pure 12
  , _blockSlotLeader = "pool1pu5jlj4q9w9jlxeu370a3c9myx47md5j5m2str0naunn2qnikdy"
  , _blockSize = 3
  , _blockTxCount = 1
  , _blockOutput = pure $ 128314491794
  , _blockFees = pure $ 592661
  , _blockBlockVrf = pure "vrf_vk1wf2k6lhujezqcfe00l6zetxpnmh9n6mwhpmhm0dvfh3fxgmdnrfqkms8ty"
  , _blockOpCert = pure "da905277534faf75dae41732650568af545134ee08a3c0392dbefc8096ae177c"
  , _blockOpCertCounter = pure 18
  , _blockPreviousBlock = pure "43ebccb3ac72c7cebd0d9b755a4b08412c9f5dcb81b8a0ad1e3c197d29d47b05"
  , _blockNextBlock = pure "8367f026cf4b03e116ff8ee5daf149b55ba5a6ec6dec04803b8dc317721d15fa"
  , _blockConfirmations = 4698
  }

affectedAddrsSample = [r|
[
  {
    "address": "addr1q9ld26v2lv8wvrxxmvg90pn8n8n5k6tdst06q2s856rwmvnueldzuuqmnsye359fqrk8hwvenjnqultn7djtrlft7jnq7dy7wv",
    "transactions": [
      {
        "tx_hash": "1a0570af966fb355a7160e4f82d5a80b8681b7955f5d44bec0dce628516157f0"
      }
    ]
  },
  {
    "address": "addr1qxqs59lphg8g6qndelq8xwqn60ag3aeyfcp33c2kdp46a09re5df3pzwwmyq946axfcejy5n4x0y99wqpgtp2gd0k09qsgy6pz",
    "transactions": [
      {
        "tx_hash": "1a0570af966fb355a7160e4f82d5a80b8681b7955f5d44bec0dce628516157d0"
      }
    ]
  }
]
|]

affectedAddrsExpected :: [(Address, [TxHash])]
affectedAddrsExpected =
  [ (Address "addr1q9ld26v2lv8wvrxxmvg90pn8n8n5k6tdst06q2s856rwmvnueldzuuqmnsye359fqrk8hwvenjnqultn7djtrlft7jnq7dy7wv",
     [TxHash "1a0570af966fb355a7160e4f82d5a80b8681b7955f5d44bec0dce628516157f0"])
  , (Address "addr1qxqs59lphg8g6qndelq8xwqn60ag3aeyfcp33c2kdp46a09re5df3pzwwmyq946axfcejy5n4x0y99wqpgtp2gd0k09qsgy6pz",
     [TxHash "1a0570af966fb355a7160e4f82d5a80b8681b7955f5d44bec0dce628516157d0"])
  ]

txHashesCBORSample = [r|
[
  {
    "tx_hash": "6f044565223657acdde3c569d6555b2edbd71bc2bfb4df0b1cce0ef805f606d4",
    "cbor": "84a300d9010281825820787b3a89b1d32806968b867a4a31f1e33054b573821293a1c915559e34810a3602018282583900bd71b1547ab3ec95725100d0c0fb06da5ffae9cf54fb97e0f52fb9cab51adb784c4997143ba56990c0584111137d02898950245f5db5e2631a05f5e10082583900bd71b1547ab3ec95725100d0c0fb06da5ffae9cf54fb97e0f52fb9cab51adb784c4997143ba56990c0584111137d02898950245f5db5e2631b0000001588345487021a0002922da100818258206a7884092084c018eed71f9df5ff61935ab021f578b2d57d8ffb0b0e8ac0ea285840a2045496acf31996336ef187f9f3c2bd24c24995b540541586cbbc73a1550a50c2fdc3d4b99766c2823eb520866b03a38a60a4eb94f24994f441fb2852447c0ef5f6"
  },
  {
    "tx_hash": "9b9ddb7ee8101e5a2638b5b79fe4dbafafcda56df12eebe2fd456e9862b848ea",
    "cbor": "84a300d90102818258207df720d99d8323830709439492876005ff752b94dafeb7bec529ab1679ec236901018282583900f2b425e12b7d90128fff6368c9c3cee21e29ae0273dda6f4c21c66f8364d4566878a66fab0bbc596dac020e3188cb58e40f7069e3c6652fa1a00e4e1c082583900c8cbbfc1db08891d2c2ec9f8df43c2904a61c0626c7e1a57627eb0753e9649b2a3596889c539a17e4d4cbefd9ed73a73d4d8f6089e02d1df1b00000004a6558a56021a0002922da10081825820844e231d37f9b76a2b02978d634ec7fa9fa7bc74a4ce3bc403e3091321a672405840c3cebb97014af06991be1a15fbc86adf71c2441dcbf3078edb11d9407306a13694f7ca5d8fc366bf28ea30fda713c2368e648ce7c87bd2981b43e4fecdd05403f5f6"
  },
  {
    "tx_hash": "52aec9ace9b710bc2e662d4fe20e71c1bc3a0c0f3b8076f069af9713963d4559",
    "cbor": "84a300d9010281825820c8e9b710248220c28f5fbf6fb040b3a392707d19d0bfdaa5474c882d02b0c3ab0101828258390036e0cf1e52e05ef92e52c7bc2a04493d6bae481b8acbab12ec4300d7f9c9e87246d2f0373885896ad2804b7229673204cac9208345c1ea5b1a001e848082583900263bdbb58275ab6396c39a3188b6594bd5d46f54e08b6868c9fb3e498141d8651e0139b3bfc9c31f76d885431f83957329d75e0e53a65c2a1b000000730e797dda021a0002a2ada100d90102818258200909d10ce4850cd340b155d1dceae0bdcd77eba09524296ffe40f061dfc537c7584073a9efedf31dafc428605836d1ce2e2f62c1c6a9e733a5e538de034449b8ade504521e5fff96df5712487d214c83ac69babaa604bec9bc7a278f9f38b58ef70cf5f6"
  },
  {
    "tx_hash": "b49ccb4ee194da2d97bfa5374509b2ed11a3b8265d346287928e6d25b82f48cb",
    "cbor": "84a300d9010281825820c5952d922ea00bbb4b6be104c647e6e7b8366c290db5173655f871243494d1001818018182581d60da0eb5ed7611482ec5089b69d870e0c56c1c45180256112398e0835b1b00000002540be400021a00030d40a100d90102818258206855ffb8cbde9163c55c2405f93a8cfa063a631d811281d954e16782123956a75840a4208e0e2236b7f04f77f47ba07f3ec40cbf567ddfd7c7562a451cd51b38f58936e233be2a1a4f202c704fc9963215cdfdde19970c7d05d928403ee688e4ac08f5f6"
  }
]
|]

txHashesCBORExpected :: [TxHashCBOR]
txHashesCBORExpected =
  [ TxHashCBOR
      ( TxHash "6f044565223657acdde3c569d6555b2edbd71bc2bfb4df0b1cce0ef805f606d4"
      , TransactionCBOR "84a300d9010281825820787b3a89b1d32806968b867a4a31f1e33054b573821293a1c915559e34810a3602018282583900bd71b1547ab3ec95725100d0c0fb06da5ffae9cf54fb97e0f52fb9cab51adb784c4997143ba56990c0584111137d02898950245f5db5e2631a05f5e10082583900bd71b1547ab3ec95725100d0c0fb06da5ffae9cf54fb97e0f52fb9cab51adb784c4997143ba56990c0584111137d02898950245f5db5e2631b0000001588345487021a0002922da100818258206a7884092084c018eed71f9df5ff61935ab021f578b2d57d8ffb0b0e8ac0ea285840a2045496acf31996336ef187f9f3c2bd24c24995b540541586cbbc73a1550a50c2fdc3d4b99766c2823eb520866b03a38a60a4eb94f24994f441fb2852447c0ef5f6"
      )
  , TxHashCBOR
      ( TxHash "9b9ddb7ee8101e5a2638b5b79fe4dbafafcda56df12eebe2fd456e9862b848ea"
      , TransactionCBOR "84a300d90102818258207df720d99d8323830709439492876005ff752b94dafeb7bec529ab1679ec236901018282583900f2b425e12b7d90128fff6368c9c3cee21e29ae0273dda6f4c21c66f8364d4566878a66fab0bbc596dac020e3188cb58e40f7069e3c6652fa1a00e4e1c082583900c8cbbfc1db08891d2c2ec9f8df43c2904a61c0626c7e1a57627eb0753e9649b2a3596889c539a17e4d4cbefd9ed73a73d4d8f6089e02d1df1b00000004a6558a56021a0002922da10081825820844e231d37f9b76a2b02978d634ec7fa9fa7bc74a4ce3bc403e3091321a672405840c3cebb97014af06991be1a15fbc86adf71c2441dcbf3078edb11d9407306a13694f7ca5d8fc366bf28ea30fda713c2368e648ce7c87bd2981b43e4fecdd05403f5f6"
      )
  , TxHashCBOR
      ( TxHash "52aec9ace9b710bc2e662d4fe20e71c1bc3a0c0f3b8076f069af9713963d4559"
      , TransactionCBOR "84a300d9010281825820c8e9b710248220c28f5fbf6fb040b3a392707d19d0bfdaa5474c882d02b0c3ab0101828258390036e0cf1e52e05ef92e52c7bc2a04493d6bae481b8acbab12ec4300d7f9c9e87246d2f0373885896ad2804b7229673204cac9208345c1ea5b1a001e848082583900263bdbb58275ab6396c39a3188b6594bd5d46f54e08b6868c9fb3e498141d8651e0139b3bfc9c31f76d885431f83957329d75e0e53a65c2a1b000000730e797dda021a0002a2ada100d90102818258200909d10ce4850cd340b155d1dceae0bdcd77eba09524296ffe40f061dfc537c7584073a9efedf31dafc428605836d1ce2e2f62c1c6a9e733a5e538de034449b8ade504521e5fff96df5712487d214c83ac69babaa604bec9bc7a278f9f38b58ef70cf5f6"
      )
  , TxHashCBOR
      ( TxHash "b49ccb4ee194da2d97bfa5374509b2ed11a3b8265d346287928e6d25b82f48cb"
      , TransactionCBOR "84a300d9010281825820c5952d922ea00bbb4b6be104c647e6e7b8366c290db5173655f871243494d1001818018182581d60da0eb5ed7611482ec5089b69d870e0c56c1c45180256112398e0835b1b00000002540be400021a00030d40a100d90102818258206855ffb8cbde9163c55c2405f93a8cfa063a631d811281d954e16782123956a75840a4208e0e2236b7f04f77f47ba07f3ec40cbf567ddfd7c7562a451cd51b38f58936e233be2a1a4f202c704fc9963215cdfdde19970c7d05d928403ee688e4ac08f5f6"
      )
  ]
