import java.io.IOException;
import java.util.Collections;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.EEnumLiteral;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.impl.EObjectImpl;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.xmi.impl.EcoreResourceFactoryImpl;

public class Ecore2Clafer extends EObjectImpl {
	public static final String[] claferKeywords = {"this","ref","parent","children","enum","min","max","xor","or","mux","opt","all","disj","if","then","else","not","in","sum","no","lone","one","some","abstract"};
	public static final String[] intPrimitives = {"EInt","Integer","UnlimitedNatural","EBigInt","Real","EBigInteger"};
	public static final String[] stringPrimitives = {"EString","String"};
	public static final String[] classifiersToBeEliminated = {"EBoolean","Boolean","int","string"};
	public static void main(String[] args) {
		try {
		    Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("ecore", new EcoreResourceFactoryImpl());
			new Ecore2Clafer().translate(args[0]);
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}	  

	public void translate(String filepath) throws IOException {
        URI fileURI = URI.createFileURI(filepath);
        
        Resource resource = new ResourceSetImpl().createResource(fileURI);
        
        resource.load(Collections.emptyMap());
        
        for (EObject eobject : resource.getContents())
        {
            EPackage epackage = (EPackage)eobject;
            
            for (EClassifier eclassifier : epackage.getEClassifiers())
            {
                System.out.print(translateEClassifier(eclassifier));
            }
        }
    }

	public String translateEClassifier(EClassifier eclassifier)
	{
	    StringBuilder sb = new StringBuilder();
	    
        if (eclassifier instanceof EClass)
        {
           	sb.append("abstract " + formatName(eclassifier.getName()) + 
        			translateEClass((EClass) eclassifier));
        	
        }
        else if (eclassifier instanceof EEnum)
        {
            sb.append("enum " + formatName(eclassifier.getName()) + " = " + translateEEnum((EEnum) eclassifier) + "\n");
        }
        else
        {
            sb.append("abstract " + formatName(eclassifier.getName()) + "\n");
        }
        sb.append("\n");
        String sbString = sb.toString();
        for(int i = 0; i < classifiersToBeEliminated.length;i++){
        	if(sbString.startsWith("abstract "+classifiersToBeEliminated[i])||sbString.startsWith("enum "+classifiersToBeEliminated)){
        		return "";
        	}
        }
        return sbString;
	}
	
	public String translateEClass(EClass eClass)
	{
	    StringBuilder sb = new StringBuilder();
	    
	    sb.append(translateESuperTypes(eClass) + "\n");
	    
 
        for (EAttribute eAttribute : eClass.getEAttributes())
        {
            sb.append(translateEAttribute(eAttribute));
        }
        
        for (EReference eReference : eClass.getEReferences())
        {
            sb.append(translateEReference(eReference));
        }
        return sb.toString();
	}
	
	public String translateEEnum(EEnum eEnum)
	{
	    StringBuilder sb = new StringBuilder();
	    
	    for (EEnumLiteral eEnumLiteral : eEnum.getELiterals())
	    {
	        sb.append(" | " +  replaceClaferKeywords(eEnumLiteral.getName()));
	    }
	    
	    return sb.substring(3);
	}
	
	public String translateESuperTypes(EClass eClass)
	{
	    StringBuilder sb = new StringBuilder();
	    
	    EList<EClass> eSuperTypes = eClass.getESuperTypes();
        
        if (!eSuperTypes.isEmpty())
        {
            sb.append(" : ");
            for (EClass ec : eSuperTypes)
            {
                sb.append(formatName(ec.getName()) + " ");    
            }
        }
        return sb.toString();
	}
	
	public String translateEAttribute(EAttribute eAttribute)
	{
        StringBuilder sb = new StringBuilder();
        
        // if attribute of type EBoolean, then translate to optional clafer
        String attrTypeName = formatName(eAttribute.getEAttributeType().getName());
        String card = formatCard(eAttribute.getLowerBound(),
                eAttribute.getUpperBound());
        if(attrTypeName.equals("EBoolean") || attrTypeName.equals("Boolean")){
        	sb.append(indent(1) + formatName(eAttribute.getName()) + 
                " " + ((!card.isEmpty())?card:"?") //Needed to be refined.
                 + "\n");
        }
        else{
        	sb.append(indent(1) + formatName(eAttribute.getName()) + 
        			(eAttribute.getEAttributeType() instanceof EEnum ? " -> " : " : ") +
        			attrTypeName + " " +
        			card + "\n");
        }
        return sb.toString();
	}

	public String translateEReference(EReference eReference)
	{

        StringBuilder sb = new StringBuilder();
        
        sb.append(indent(1) + formatName(eReference.getName()) + " " +
            (eReference.isContainment() ? ":" : "->") + " " +
            formatName((null == eReference.getEReferenceType()) ? "#ERROR#" : eReference.getEReferenceType().getName()) + " " +
            formatCard(eReference.getLowerBound(), eReference.getUpperBound()) + "\n");
        
        if (null != eReference.getEOpposite())
        {
            sb.append(indent(2) + "[parent in this." + formatName(eReference.getEOpposite().getName()) +"]\n");
        }
        return sb.toString();
	}
	
	public String indent(int n)
	{
	    StringBuilder sb = new StringBuilder();
	    
	    for (int i = 0; i < n; ++i)
	    {
	        sb.append("    ");
	    }
	    return sb.toString();
	}
	
	public String formatCard(int lowerBound, int upperBound)
	{
	    if (0 == lowerBound && 1 == upperBound)
	        return "?";
	    else if (0 == lowerBound && -1 == upperBound)
	        return "*";
	    else if (1 == lowerBound && -1 == upperBound)
	        return "+";
	    else if (1 == lowerBound && 1 == upperBound)
	        return "";
	    else if (upperBound == -2)
	    	return "*";
	    else
	        return lowerBound + ".." + ((-1 == upperBound) ? "*" : upperBound);
	}
	
	public String formatName(String name)
	{
	    // resKey - list of reserved keywords
	    // if name in resKey then name = name ++ "_"
	    // enum, ref, parent, this, min, max, 
	    
	    // if name in EString, EInt, then replace string, int
		
		String temp = (null == name) ? "" : name.replace("_", "");
		for(int i = 0; i < intPrimitives.length;i++){
			if(temp.equals(intPrimitives[i]))
				return "int";
		}
		for(int i = 0; i < stringPrimitives.length;i++){
			if(temp.equals(stringPrimitives[i]))
				return "string";
		}
		return replaceClaferKeywords(temp);
	}
	
	private String replaceClaferKeywords(String name){
		String temp = (null == name) ? "" : name;
		for(int i = 0; i < claferKeywords.length;i++){
			if(temp.equals(claferKeywords[i]))
				return temp + "_";			
		}
	    return temp;
	}
	
}
