{-|
This module is for creating 'LogAction' instances that use the logging and
formatting functions in this library.

@
import Colog.Monad (usingLoggerT)
import Colog.Simple (Severity (..), colorSevFmt, logger, logNotice, logTextStdout)

usingLoggerT (logger colorSevFmt logTextStdout Debug) $ logNotice "Take note of this!"
@

The 'LogAction' produced by 'logger' above is what you would put in your
'ReaderT' environment as well. See co-log documentation for more info on
integrating with transformers.

<https://github.com/co-log/co-log/blob/main/tutorials/custom/Custom.md>
-}
module Colog.Simple
  (
    -- * Types
    Target

    -- * Constructing LogAction loggers
  , logger

    -- * Utility functions
  , logTest

    -- * Re-exported from co-log-simple
  , module Colog.Simple.Message
  , module Colog.Simple.Severity

    -- * Re-exported from co-log
  , HasLog (..)
  , LogAction
  , logTextStderr, logTextStdout
  )
  where

import Colog (LogAction)
import Colog.Actions (logTextStderr, logTextStdout)
import Colog.Message (formatWith)
import Colog.Monad (HasLog (..), WithLog)
import Control.Monad.Trans (MonadIO)
import Data.Text (Text)

import Colog.Simple.Message
import Colog.Simple.Severity


-- | A synonym for ordinary co-log 'LogAction' types parameterized with 'Text'
type Target m = LogAction m Text


-- | Construct a logging action
logger :: MonadIO m
  => Formatter  -- ^ Format function
  -> Target m   -- ^ Where log message output will go
  -> Severity   -- ^ The severity level to filter at or above
  -> LogAction m Message
logger formatter target sev = setSeverity sev . formatWith formatter $ target


-- | Test function to generate messages with every severity in this library
logTest :: WithLog env Message m => m ()
logTest = do
  logDebug "log test message Debug 1 of 5"
  logInfo "log test message Info 2 of 5"
  logNotice "log test message Notice 3 of 5"
  logWarning "log test message Warning 4 of 5"
  logError "log test message Error 5 of 5"
