-- |
-- Module      :  ELynx.MarkovProcess.CXXModelsData
-- Description :  Stationary distributions and weights
-- Copyright   :  2021 Dominik Schrempf
-- License     :  GPL-3.0-or-later
--
-- Maintainer  :  dominik.schrempf@gmail.com
-- Stability   :  unstable
-- Portability :  portable
--
-- Creation date: Tue Feb 26 17:17:35 2019.
--
-- Quang, BL. S., Gascuel, O. & Lartillot, N. Empirical profile mixture models for
-- phylogenetic reconstruction. Bioinformatics 24, 2317–2323 (2008).
module ELynx.MarkovProcess.CXXModelsData
  ( c10StatDists,
    c10Weights,
    c20StatDists,
    c20Weights,
    c30StatDists,
    c30Weights,
    c40StatDists,
    c40Weights,
    c50StatDists,
    c50Weights,
    c60StatDists,
    c60Weights,
  )
where

import qualified Data.Vector.Storable as V
import ELynx.MarkovProcess.RateMatrix

-- | Stationary distribution of C10 model.
c10StatDists :: [StationaryDistribution]
c10StatDists =
  map
    V.fromList
    [ [ 0.408257,
        0.0349388,
        0.00698709,
        0.00978467,
        0.00616043,
        0.122161,
        0.00391518,
        0.0125784,
        0.00596702,
        0.0158339,
        0.00813132,
        0.00962854,
        0.0394156,
        0.00752797,
        0.0081783,
        0.168245,
        0.0658133,
        0.0604427,
        0.00187516,
        0.00415797
      ],
      [ 0.102776,
        0.0149663,
        0.0155944,
        0.0419667,
        0.0180729,
        0.0138806,
        0.0158865,
        0.106608,
        0.0436344,
        0.113194,
        0.04378,
        0.0213272,
        0.0223251,
        0.0440685,
        0.0418664,
        0.0529608,
        0.108174,
        0.160665,
        0.00451472,
        0.0137374
      ],
      [ 0.0351766,
        0.00787065,
        0.000676874,
        0.00196868,
        0.0126221,
        0.00224206,
        0.00128783,
        0.351582,
        0.00188565,
        0.127818,
        0.0242632,
        0.00165915,
        0.00297716,
        0.00165596,
        0.00196786,
        0.00499981,
        0.0255378,
        0.388864,
        0.00119078,
        0.00375393
      ],
      [ 0.0408514,
        0.00376029,
        0.233381,
        0.0901239,
        0.00251082,
        0.115833,
        0.0373197,
        0.00255236,
        0.0485017,
        0.00521646,
        0.00225718,
        0.218565,
        0.0108334,
        0.0380451,
        0.0269887,
        0.0804527,
        0.030288,
        0.00444811,
        0.00108153,
        0.00698909
      ],
      [ 0.0185493,
        0.00704165,
        0.000977506,
        0.00248916,
        0.073333,
        0.00289529,
        0.0040104,
        0.163242,
        0.00435709,
        0.444308,
        0.120282,
        0.00248957,
        0.00488276,
        0.00835394,
        0.00623624,
        0.00516424,
        0.0131807,
        0.0968581,
        0.00687598,
        0.0144734
      ],
      [ 0.110675,
        0.00148349,
        0.163644,
        0.263846,
        0.00232568,
        0.0325228,
        0.0163804,
        0.00683349,
        0.0677158,
        0.014068,
        0.00489881,
        0.0405186,
        0.0298982,
        0.0877962,
        0.035219,
        0.0562888,
        0.0426922,
        0.0181079,
        0.0010339,
        0.00405223
      ],
      [ 0.0522658,
        0.0143325,
        0.0297745,
        0.0388387,
        0.0624033,
        0.0228101,
        0.155164,
        0.0187406,
        0.0439469,
        0.065378,
        0.0207189,
        0.0714837,
        0.0145475,
        0.073654,
        0.0668295,
        0.0549018,
        0.037014,
        0.0267512,
        0.0193757,
        0.111069
      ],
      [ 0.0116587,
        0.0105341,
        0.00217425,
        0.00242511,
        0.365099,
        0.00347091,
        0.0366787,
        0.0187185,
        0.00266947,
        0.067649,
        0.0143535,
        0.00640111,
        0.00311599,
        0.00402037,
        0.00509901,
        0.00948485,
        0.00737139,
        0.0206341,
        0.0509565,
        0.357486
      ],
      [ 0.0627196,
        0.00526629,
        0.0236193,
        0.0686285,
        0.00391818,
        0.0256175,
        0.0332612,
        0.0128968,
        0.227084,
        0.0305628,
        0.0124037,
        0.0428629,
        0.0140441,
        0.109811,
        0.203878,
        0.0483152,
        0.0463378,
        0.0197063,
        0.00251435,
        0.00655211
      ],
      [ 0.114552,
        0.00985495,
        0.0416192,
        0.0364908,
        0.0046606,
        0.0503818,
        0.0165233,
        0.00929495,
        0.0423027,
        0.0139154,
        0.00822408,
        0.0750615,
        0.0379222,
        0.0339625,
        0.0324009,
        0.261065,
        0.184583,
        0.0195769,
        0.0017549,
        0.00585383
      ]
    ]

-- | Weights of C10 model.
c10Weights :: [Double]
c10Weights =
  [ 0.119134,
    0.0874372,
    0.103711,
    0.0922585,
    0.107049,
    0.132995,
    0.0538028,
    0.0691986,
    0.131994,
    0.10242
  ]

-- | Stationary distribution of C20 model.
c20StatDists :: [StationaryDistribution]
c20StatDists =
  map
    V.fromList
    [ [ 0.0862413,
        0.0130505,
        0.0329909,
        0.0184527,
        0.00441553,
        0.0366905,
        0.0108013,
        0.00979071,
        0.0220195,
        0.0112826,
        0.00878215,
        0.0791293,
        0.0189273,
        0.0169047,
        0.0171944,
        0.317815,
        0.27117,
        0.0179753,
        0.00153173,
        0.00483429
      ],
      [ 0.203558,
        0.0348667,
        0.00316561,
        0.00708594,
        0.0112429,
        0.0195236,
        0.0024392,
        0.115257,
        0.00423808,
        0.0789777,
        0.0309187,
        0.00770524,
        0.0164189,
        0.00640441,
        0.00509808,
        0.0496777,
        0.111895,
        0.284906,
        0.00177626,
        0.00484482
      ],
      [ 0.0211547,
        0.00481886,
        0.000549287,
        0.00145396,
        0.0128252,
        0.00114309,
        0.00113464,
        0.392846,
        0.00135799,
        0.125064,
        0.0209789,
        0.0012755,
        0.00202472,
        0.00123288,
        0.00149462,
        0.00262407,
        0.0171914,
        0.386068,
        0.00115911,
        0.0036028
      ],
      [ 0.0376904,
        0.00640738,
        0.0109469,
        0.0358365,
        0.00363498,
        0.0191107,
        0.0329514,
        0.0101712,
        0.289763,
        0.0237496,
        0.00965289,
        0.0365411,
        0.0105337,
        0.0893564,
        0.28852,
        0.0356314,
        0.0355927,
        0.0144622,
        0.00279252,
        0.00665572
      ],
      [ 0.00845978,
        0.0084909,
        0.00244879,
        0.00250555,
        0.342046,
        0.00242771,
        0.0433214,
        0.0097713,
        0.0026741,
        0.0380507,
        0.00807248,
        0.00725259,
        0.00214187,
        0.00427815,
        0.00535899,
        0.00804189,
        0.00553221,
        0.012141,
        0.049484,
        0.4375
      ],
      [ 0.17599,
        0.00175587,
        0.130126,
        0.218217,
        0.0025277,
        0.0409535,
        0.0130708,
        0.00856221,
        0.0542946,
        0.0159531,
        0.00540458,
        0.0332846,
        0.037102,
        0.0707184,
        0.0290429,
        0.0793481,
        0.0540083,
        0.0249553,
        0.00105921,
        0.00362591
      ],
      [ 0.16344,
        0.00886599,
        0.0374273,
        0.0220612,
        0.00306413,
        0.529672,
        0.00900061,
        0.00175694,
        0.0167118,
        0.00611563,
        0.00293908,
        0.0438702,
        0.0126458,
        0.0137555,
        0.0195541,
        0.0829343,
        0.0142836,
        0.00579857,
        0.00286407,
        0.00323983
      ],
      [ 0.0917469,
        0.0284015,
        0.0133819,
        0.0196876,
        0.0998479,
        0.0249899,
        0.0449766,
        0.0583556,
        0.0164916,
        0.115501,
        0.0395995,
        0.0290699,
        0.0209916,
        0.0255085,
        0.0265853,
        0.0736483,
        0.0661518,
        0.0831856,
        0.0246464,
        0.0972327
      ],
      [ 0.0646701,
        0.00771176,
        0.0168734,
        0.0544978,
        0.0219148,
        0.0148894,
        0.0313852,
        0.0505983,
        0.0907931,
        0.184428,
        0.077484,
        0.0228907,
        0.0105004,
        0.0996415,
        0.0988016,
        0.0321196,
        0.0411766,
        0.0505824,
        0.0084303,
        0.0206106
      ],
      [ 0.0135994,
        0.010009,
        0.00079517,
        0.00180118,
        0.264097,
        0.00267946,
        0.00724019,
        0.0814027,
        0.00251581,
        0.366142,
        0.0734965,
        0.00184694,
        0.00389941,
        0.00464208,
        0.00434084,
        0.00436688,
        0.00752485,
        0.0573473,
        0.0261565,
        0.0660971
      ],
      [ 0.147804,
        0.00488258,
        0.0534743,
        0.0727246,
        0.00299039,
        0.0907726,
        0.0262289,
        0.00357811,
        0.105166,
        0.0126777,
        0.00596218,
        0.072663,
        0.0156558,
        0.0757166,
        0.0842845,
        0.14599,
        0.0634877,
        0.00927198,
        0.00159285,
        0.00507607
      ],
      [ 0.0186377,
        0.00549689,
        0.00083297,
        0.00202485,
        0.0385383,
        0.00217135,
        0.0023666,
        0.202081,
        0.00291207,
        0.437038,
        0.124186,
        0.00198652,
        0.00406723,
        0.00658901,
        0.00420552,
        0.00461774,
        0.0149904,
        0.118938,
        0.00268717,
        0.00563241
      ],
      [ 0.0477624,
        0.00757917,
        0.0141349,
        0.0462688,
        0.0130691,
        0.00523279,
        0.0165352,
        0.17415,
        0.0577575,
        0.112125,
        0.0330288,
        0.0209574,
        0.0124375,
        0.0429297,
        0.0505743,
        0.0264989,
        0.0951755,
        0.20937,
        0.00316605,
        0.0112466
      ],
      [ 0.416419,
        0.0406938,
        0.00451317,
        0.00632298,
        0.00484384,
        0.0946185,
        0.00310574,
        0.00764432,
        0.00389418,
        0.00998854,
        0.00693232,
        0.00917014,
        0.0187841,
        0.00613205,
        0.00561008,
        0.236077,
        0.0746275,
        0.0459225,
        0.00121726,
        0.00348258
      ],
      [ 0.0402296,
        0.0124783,
        0.0365524,
        0.0372197,
        0.0459095,
        0.0233618,
        0.210831,
        0.00934787,
        0.0482411,
        0.0360561,
        0.010029,
        0.103665,
        0.0098504,
        0.0826558,
        0.0735203,
        0.0533383,
        0.0310209,
        0.015248,
        0.0140077,
        0.106438
      ],
      [ 0.0323453,
        0.00359763,
        0.24315,
        0.0710274,
        0.00244293,
        0.101607,
        0.0366225,
        0.00314108,
        0.0470129,
        0.00519805,
        0.00240287,
        0.252045,
        0.00948378,
        0.0330831,
        0.0236283,
        0.0848355,
        0.0359083,
        0.00487046,
        0.000873093,
        0.00672477
      ],
      [ 0.147626,
        0.00323272,
        0.0403052,
        0.0576893,
        0.00471772,
        0.0330851,
        0.0146393,
        0.0108267,
        0.0451351,
        0.0256201,
        0.00586514,
        0.0211973,
        0.347371,
        0.0371554,
        0.0334507,
        0.0892065,
        0.0485899,
        0.0282336,
        0.00163587,
        0.00441772
      ],
      [ 0.103145,
        0.00617625,
        0.0386402,
        0.0923369,
        0.00676664,
        0.0202338,
        0.0246762,
        0.0376904,
        0.0921699,
        0.0376284,
        0.0161883,
        0.0435172,
        0.0128302,
        0.0786603,
        0.0717748,
        0.095145,
        0.137857,
        0.0740454,
        0.00221447,
        0.00830416
      ],
      [ 0.0837543,
        0.00207351,
        0.0804871,
        0.194776,
        0.00230634,
        0.022903,
        0.0268459,
        0.00740798,
        0.145929,
        0.019025,
        0.00673952,
        0.0518811,
        0.0085616,
        0.14565,
        0.0899383,
        0.045574,
        0.0451081,
        0.0150303,
        0.00107713,
        0.00493253
      ],
      [ 0.0578736,
        0.00111308,
        0.294674,
        0.34021,
        0.00170349,
        0.0293911,
        0.0139817,
        0.00305257,
        0.0363365,
        0.00626119,
        0.0027296,
        0.0491422,
        0.0156106,
        0.059825,
        0.0138314,
        0.0358045,
        0.0249942,
        0.00876742,
        0.000866434,
        0.0038313
      ]
    ]

-- | Weights of C20 model.
c20Weights :: [Double]
c20Weights =
  [ 0.0559911,
    0.0514825,
    0.0812922,
    0.0721977,
    0.0556719,
    0.0331003,
    0.0589502,
    0.0263757,
    0.0307584,
    0.0376701,
    0.0303058,
    0.0808776,
    0.0263349,
    0.0579101,
    0.0371248,
    0.0586868,
    0.0561479,
    0.0349811,
    0.0544937,
    0.0596472
  ]

-- | Stationary distribution of C30 model.
c30StatDists :: [StationaryDistribution]
c30StatDists =
  map
    V.fromList
    [ [ 0.110045,
        0.00190472,
        0.159541,
        0.109896,
        0.00166295,
        0.0684302,
        0.0137951,
        0.00262831,
        0.0358554,
        0.00733965,
        0.00247064,
        0.0640338,
        0.166936,
        0.0310187,
        0.0171295,
        0.138179,
        0.0568343,
        0.00823656,
        0.000466112,
        0.00359702
      ],
      [ 0.0874125,
        0.00498264,
        0.032612,
        0.0951701,
        0.00489966,
        0.0144043,
        0.0210627,
        0.0399884,
        0.11472,
        0.0301585,
        0.0126489,
        0.0382152,
        0.0137397,
        0.0798169,
        0.080632,
        0.087377,
        0.155862,
        0.0793881,
        0.00151228,
        0.00539745
      ],
      [ 0.0225477,
        0.00500182,
        0.000595928,
        0.00150305,
        0.0089216,
        0.0011571,
        0.000937432,
        0.394469,
        0.00136009,
        0.0889573,
        0.0189103,
        0.00130346,
        0.0018312,
        0.00114366,
        0.00149005,
        0.00283364,
        0.0189813,
        0.425056,
        0.000669375,
        0.00233037
      ],
      [ 0.0602158,
        0.000952546,
        0.290008,
        0.361087,
        0.00146256,
        0.0281926,
        0.0130501,
        0.00305162,
        0.0352705,
        0.00604019,
        0.00274606,
        0.0414988,
        0.0127175,
        0.0621611,
        0.0136833,
        0.0318109,
        0.022528,
        0.00932584,
        0.000794803,
        0.00340246
      ],
      [ 0.0101224,
        0.00859894,
        0.000637919,
        0.00112496,
        0.278538,
        0.00240852,
        0.00477534,
        0.0701153,
        0.00167485,
        0.413591,
        0.0744863,
        0.00129289,
        0.00404666,
        0.00350286,
        0.00283449,
        0.00370872,
        0.00523793,
        0.040886,
        0.0200223,
        0.0523939
      ],
      [ 0.133583,
        0.00105145,
        0.112578,
        0.209957,
        0.00167936,
        0.0207552,
        0.012133,
        0.00735265,
        0.0771772,
        0.0133278,
        0.00305717,
        0.0213892,
        0.18902,
        0.0565844,
        0.028479,
        0.0484054,
        0.0373318,
        0.0225174,
        0.000926699,
        0.00269464
      ],
      [ 0.0408277,
        0.0153918,
        0.00306349,
        0.00660109,
        0.157505,
        0.00581131,
        0.0245212,
        0.148751,
        0.00759232,
        0.16378,
        0.0385527,
        0.00804649,
        0.00583522,
        0.0102922,
        0.0124492,
        0.0151579,
        0.033222,
        0.154771,
        0.0264937,
        0.121334
      ],
      [ 0.246906,
        0.103941,
        0.00274183,
        0.00549448,
        0.0251776,
        0.0373263,
        0.00857523,
        0.0292404,
        0.00561231,
        0.0535091,
        0.0302246,
        0.016893,
        0.00780989,
        0.0103988,
        0.0106279,
        0.164235,
        0.123989,
        0.0955868,
        0.00531559,
        0.0163954
      ],
      [ 0.0549429,
        0.0099281,
        0.00929153,
        0.0417085,
        0.024386,
        0.0105564,
        0.0363512,
        0.0569585,
        0.115252,
        0.168183,
        0.0592328,
        0.0202958,
        0.00830554,
        0.0906036,
        0.130543,
        0.0283779,
        0.0412594,
        0.0592101,
        0.00963554,
        0.024978
      ],
      [ 0.0462773,
        0.0172727,
        0.0182504,
        0.0224266,
        0.133632,
        0.0160971,
        0.135785,
        0.0164967,
        0.0239396,
        0.0598936,
        0.0164507,
        0.0412365,
        0.0117413,
        0.0348991,
        0.0362984,
        0.0454156,
        0.0304388,
        0.0253979,
        0.0330338,
        0.235016
      ],
      [ 0.047438,
        0.00823324,
        0.0112117,
        0.0388101,
        0.0116644,
        0.00559986,
        0.0149157,
        0.183217,
        0.0467851,
        0.110069,
        0.0356444,
        0.0222454,
        0.0100245,
        0.0374051,
        0.041018,
        0.0317171,
        0.111435,
        0.219931,
        0.00266856,
        0.00996601
      ],
      [ 0.0213608,
        0.00610249,
        0.00129412,
        0.00362973,
        0.0371808,
        0.0030017,
        0.00384259,
        0.130947,
        0.00545678,
        0.456699,
        0.194784,
        0.0039879,
        0.00407473,
        0.0139566,
        0.00699762,
        0.00769915,
        0.0198019,
        0.0693208,
        0.00340864,
        0.00645457
      ],
      [ 0.0919632,
        0.0117031,
        0.0306717,
        0.0171908,
        0.00415894,
        0.0370685,
        0.0100793,
        0.00931237,
        0.0205386,
        0.0097241,
        0.00757673,
        0.0764682,
        0.0179686,
        0.016006,
        0.0160005,
        0.325447,
        0.274438,
        0.0178218,
        0.00138874,
        0.00447397
      ],
      [ 0.464925,
        0.0233329,
        0.00507317,
        0.00579942,
        0.00255882,
        0.149524,
        0.00232984,
        0.00433612,
        0.00285254,
        0.00559955,
        0.00393132,
        0.00753048,
        0.0186467,
        0.00435713,
        0.00430132,
        0.215019,
        0.047703,
        0.0292668,
        0.00090381,
        0.00200872
      ],
      [ 0.205133,
        0.00209159,
        0.107098,
        0.198973,
        0.00182351,
        0.0487574,
        0.0127143,
        0.00581247,
        0.0667787,
        0.0133472,
        0.00437834,
        0.0339418,
        0.0110998,
        0.0822742,
        0.0439661,
        0.0873962,
        0.0519782,
        0.0193174,
        0.00073616,
        0.00238214
      ],
      [ 0.026369,
        0.00394624,
        0.311772,
        0.0694354,
        0.00233519,
        0.0799842,
        0.0309615,
        0.00275212,
        0.0288631,
        0.00385797,
        0.00183637,
        0.272716,
        0.00624576,
        0.0218979,
        0.0133614,
        0.0798729,
        0.0324143,
        0.00427053,
        0.000722966,
        0.00638577
      ],
      [ 0.15265,
        0.00402153,
        0.0237392,
        0.0377949,
        0.00585844,
        0.0306214,
        0.014193,
        0.0123318,
        0.0375543,
        0.0290062,
        0.00644732,
        0.016823,
        0.38645,
        0.0341734,
        0.0332784,
        0.0880336,
        0.0489543,
        0.0313553,
        0.00182526,
        0.00488778
      ],
      [ 0.00802476,
        0.00234121,
        0.000338596,
        0.000781895,
        0.0311968,
        0.000540383,
        0.00100264,
        0.317706,
        0.00125983,
        0.373789,
        0.0488212,
        0.00063274,
        0.00206875,
        0.00155079,
        0.00174086,
        0.00120951,
        0.00656968,
        0.19446,
        0.00163092,
        0.00433436
      ],
      [ 0.059995,
        0.0016885,
        0.0889731,
        0.0962701,
        0.00118007,
        0.0588796,
        0.0327277,
        0.00214673,
        0.182586,
        0.00708764,
        0.00339794,
        0.133492,
        0.0141408,
        0.0864856,
        0.100054,
        0.0779002,
        0.0448817,
        0.00422414,
        0.000624903,
        0.00326411
      ],
      [ 0.0393521,
        0.00990719,
        0.0431198,
        0.039464,
        0.0188681,
        0.0282866,
        0.209572,
        0.00761014,
        0.0596434,
        0.025834,
        0.00845867,
        0.142548,
        0.00968405,
        0.101979,
        0.0838171,
        0.0624999,
        0.0347088,
        0.0127686,
        0.00546458,
        0.0564145
      ],
      [ 0.00727155,
        0.00678525,
        0.00276038,
        0.00250694,
        0.127148,
        0.00808347,
        0.0113218,
        0.00566096,
        0.00177351,
        0.03942,
        0.00798661,
        0.00197568,
        0.00410981,
        0.00433393,
        0.0140999,
        0.00522444,
        0.00430223,
        0.00942264,
        0.627357,
        0.108456
      ],
      [ 0.090707,
        0.00636999,
        0.0745873,
        0.0386802,
        0.00288592,
        0.475005,
        0.016809,
        0.00147211,
        0.0343856,
        0.00551498,
        0.00246921,
        0.0860678,
        0.0112151,
        0.0259377,
        0.0290062,
        0.073111,
        0.0153706,
        0.00392819,
        0.00229148,
        0.00418607
      ],
      [ 0.00552919,
        0.00721059,
        0.00114134,
        0.000968304,
        0.374409,
        0.0016071,
        0.0325831,
        0.00829184,
        0.00138494,
        0.0353678,
        0.00744868,
        0.00460866,
        0.00133746,
        0.00226922,
        0.00246263,
        0.00574027,
        0.00372796,
        0.00988026,
        0.0330334,
        0.460998
      ],
      [ 0.244326,
        0.027388,
        0.00315909,
        0.00767156,
        0.00606194,
        0.019609,
        0.00201894,
        0.101744,
        0.00454923,
        0.0468424,
        0.0201286,
        0.00624227,
        0.0185219,
        0.0053594,
        0.00453866,
        0.0497754,
        0.11708,
        0.310493,
        0.000957725,
        0.00353337
      ],
      [ 0.0863111,
        0.00244198,
        0.0600903,
        0.20361,
        0.00293931,
        0.0175221,
        0.0245245,
        0.0105994,
        0.148579,
        0.027121,
        0.00958244,
        0.0313963,
        0.00682768,
        0.167235,
        0.0984812,
        0.03478,
        0.0408211,
        0.0202271,
        0.00140013,
        0.00551054
      ],
      [ 0.0643926,
        0.00231659,
        0.162821,
        0.17627,
        0.00430667,
        0.0297139,
        0.0303504,
        0.0088163,
        0.072714,
        0.0148017,
        0.00567484,
        0.103121,
        0.00992703,
        0.0752535,
        0.0369049,
        0.0926434,
        0.083313,
        0.0161551,
        0.00112371,
        0.00938015
      ],
      [ 0.173668,
        0.00835966,
        0.0285985,
        0.0483894,
        0.0058274,
        0.0781901,
        0.0266135,
        0.00686419,
        0.0964012,
        0.0219499,
        0.0112303,
        0.0520405,
        0.0169661,
        0.0722447,
        0.0943629,
        0.15478,
        0.0751702,
        0.0172524,
        0.00287745,
        0.00821304
      ],
      [ 0.0347857,
        0.00627549,
        0.00923552,
        0.0323568,
        0.00343035,
        0.0170288,
        0.0306439,
        0.00919323,
        0.302085,
        0.0224429,
        0.00937208,
        0.0314157,
        0.0104447,
        0.0861073,
        0.307598,
        0.0326883,
        0.0328713,
        0.0130832,
        0.00252449,
        0.00641713
      ],
      [ 0.108774,
        0.0247877,
        0.00158232,
        0.00417699,
        0.0316523,
        0.0134703,
        0.00247658,
        0.164346,
        0.00270004,
        0.233715,
        0.0539213,
        0.00326798,
        0.0154887,
        0.0057932,
        0.0051781,
        0.0188188,
        0.0474912,
        0.251283,
        0.00376565,
        0.00731064
      ],
      [ 0.110101,
        0.00726998,
        0.0579269,
        0.0828181,
        0.0269154,
        0.0314463,
        0.0308557,
        0.0530866,
        0.029386,
        0.109679,
        0.0458729,
        0.0435099,
        0.0296431,
        0.0615197,
        0.0324325,
        0.0715888,
        0.0685882,
        0.0754042,
        0.00623241,
        0.0257238
      ]
    ]

-- | Weights of C30 model.
c30Weights :: [Double]
c30Weights =
  [ 0.00957833,
    0.0248476,
    0.0636309,
    0.0537939,
    0.0295886,
    0.0117588,
    0.0132013,
    0.0236869,
    0.0261688,
    0.0239822,
    0.0257101,
    0.0465072,
    0.0546795,
    0.0536085,
    0.0270623,
    0.0403914,
    0.0474213,
    0.0458816,
    0.0214037,
    0.0290386,
    0.0123392,
    0.056935,
    0.0419688,
    0.0339027,
    0.0388777,
    0.0196344,
    0.0233086,
    0.0622723,
    0.0184803,
    0.0203395
  ]

-- | Stationary distribution of C40 model.
c40StatDists :: [StationaryDistribution]
c40StatDists =
  map
    V.fromList
    [ [ 0.066026,
        0.00565867,
        0.105447,
        0.0440361,
        0.00131048,
        0.0711239,
        0.0168195,
        0.00390887,
        0.036669,
        0.0055316,
        0.00374124,
        0.159982,
        0.0176359,
        0.0273928,
        0.0231862,
        0.249769,
        0.150708,
        0.0065529,
        0.000672321,
        0.00382902
      ],
      [ 0.0232377,
        0.00379875,
        0.353209,
        0.0739378,
        0.00240321,
        0.0576668,
        0.0315867,
        0.00310928,
        0.0259363,
        0.00387116,
        0.00173556,
        0.275965,
        0.00631169,
        0.0197339,
        0.0122683,
        0.0657068,
        0.0270484,
        0.00475317,
        0.000760289,
        0.00696025
      ],
      [ 0.0166487,
        0.00366657,
        0.000565145,
        0.00133563,
        0.00827757,
        0.000889475,
        0.000823185,
        0.412937,
        0.00119041,
        0.0884689,
        0.0186055,
        0.00126222,
        0.001403,
        0.00106698,
        0.00125948,
        0.00213394,
        0.0162167,
        0.420686,
        0.000608205,
        0.00195532
      ],
      [ 0.239474,
        0.0283812,
        0.00447417,
        0.010553,
        0.00559911,
        0.013511,
        0.00389298,
        0.0765957,
        0.0071093,
        0.0358495,
        0.0199496,
        0.0120537,
        0.0114266,
        0.00865589,
        0.00729013,
        0.0847799,
        0.179728,
        0.245468,
        0.0009838,
        0.00422407
      ],
      [ 0.119461,
        0.0150527,
        0.0134273,
        0.0192173,
        0.0550467,
        0.0337676,
        0.0214746,
        0.0579002,
        0.0147261,
        0.144631,
        0.0561243,
        0.0294552,
        0.0631355,
        0.0301538,
        0.0233256,
        0.0925267,
        0.083123,
        0.0811758,
        0.0131636,
        0.0331118
      ],
      [ 0.0567044,
        0.00089248,
        0.29555,
        0.379515,
        0.00129723,
        0.023047,
        0.0118361,
        0.0031182,
        0.0314206,
        0.00601375,
        0.00285841,
        0.0364734,
        0.0124746,
        0.0609517,
        0.0117359,
        0.0300335,
        0.0227051,
        0.00946396,
        0.000773876,
        0.00313438
      ],
      [ 0.0179027,
        0.016076,
        0.000887041,
        0.00231821,
        0.334486,
        0.00398298,
        0.0127293,
        0.0404651,
        0.00279947,
        0.167614,
        0.0424172,
        0.00356977,
        0.00201151,
        0.00453955,
        0.00409671,
        0.00758416,
        0.00682273,
        0.0326045,
        0.0518381,
        0.245254
      ],
      [ 0.271217,
        0.200383,
        0.0021017,
        0.002323,
        0.020299,
        0.0502501,
        0.0053728,
        0.0150685,
        0.00206463,
        0.0330003,
        0.0154811,
        0.0141045,
        0.0045351,
        0.00482641,
        0.00564808,
        0.17642,
        0.0839578,
        0.0741934,
        0.00462652,
        0.0141271
      ],
      [ 0.0894737,
        0.00455383,
        0.0272183,
        0.127508,
        0.00565902,
        0.0115686,
        0.0215746,
        0.0469424,
        0.138205,
        0.0512035,
        0.0147657,
        0.0190192,
        0.00955465,
        0.116809,
        0.104003,
        0.0383954,
        0.0836653,
        0.0819556,
        0.00170794,
        0.00621813
      ],
      [ 0.0495441,
        0.0182506,
        0.0143641,
        0.0215379,
        0.141805,
        0.01402,
        0.110854,
        0.0247066,
        0.0258142,
        0.0700288,
        0.0188272,
        0.0315864,
        0.0112101,
        0.0316504,
        0.0375346,
        0.0456094,
        0.0361428,
        0.0369178,
        0.0371985,
        0.222397
      ],
      [ 0.170431,
        0.000974733,
        0.109856,
        0.253646,
        0.00133213,
        0.0249846,
        0.010139,
        0.00587494,
        0.0903324,
        0.0116526,
        0.00365127,
        0.0271109,
        0.0293614,
        0.09173,
        0.0415784,
        0.0561766,
        0.0479046,
        0.02033,
        0.000669682,
        0.00226373
      ],
      [ 0.0162725,
        0.00506141,
        0.00101821,
        0.00251413,
        0.0376246,
        0.00219354,
        0.00299143,
        0.132817,
        0.00401204,
        0.490444,
        0.192993,
        0.00218762,
        0.00343332,
        0.0104414,
        0.00548261,
        0.00401221,
        0.0127074,
        0.064772,
        0.00321076,
        0.00581006
      ],
      [ 0.0823766,
        0.00656943,
        0.0311745,
        0.0675531,
        0.00647179,
        0.0178962,
        0.0251144,
        0.0291162,
        0.0982302,
        0.0287904,
        0.0168023,
        0.059839,
        0.0114045,
        0.0686451,
        0.0734226,
        0.1303,
        0.182037,
        0.0540271,
        0.00227246,
        0.00795733
      ],
      [ 0.359497,
        0.0251417,
        0.00314844,
        0.00649627,
        0.00920205,
        0.119468,
        0.00229704,
        0.0458767,
        0.00501688,
        0.0468054,
        0.0215569,
        0.00334215,
        0.0443916,
        0.00490143,
        0.00724072,
        0.0465271,
        0.0477755,
        0.194216,
        0.00245402,
        0.00464504
      ],
      [ 0.201558,
        0.00323653,
        0.095415,
        0.153491,
        0.00256433,
        0.0667292,
        0.0155219,
        0.00677408,
        0.0547323,
        0.0165114,
        0.0060163,
        0.0425386,
        0.00919706,
        0.0772011,
        0.0430162,
        0.118598,
        0.0625473,
        0.0202798,
        0.000956551,
        0.003115
      ],
      [ 0.104273,
        0.00224501,
        0.242407,
        0.177482,
        0.00125703,
        0.169782,
        0.0132649,
        0.00189295,
        0.0220652,
        0.00425426,
        0.00164412,
        0.0621646,
        0.0317042,
        0.0356499,
        0.0147062,
        0.0778636,
        0.0288516,
        0.00602502,
        0.00069309,
        0.00177419
      ],
      [ 0.0781183,
        0.0194449,
        0.00415417,
        0.0116634,
        0.0262794,
        0.0111524,
        0.00635894,
        0.135453,
        0.00937298,
        0.245757,
        0.108778,
        0.015927,
        0.0055294,
        0.0240152,
        0.0111498,
        0.0408519,
        0.0860514,
        0.148276,
        0.00315476,
        0.00851085
      ],
      [ 0.0856592,
        0.0136073,
        0.0135062,
        0.00786026,
        0.0047153,
        0.0245401,
        0.00553791,
        0.0100592,
        0.0127319,
        0.0103344,
        0.00806758,
        0.0441923,
        0.0175274,
        0.00925906,
        0.0101233,
        0.340648,
        0.357329,
        0.019367,
        0.00142431,
        0.00350998
      ],
      [ 0.0674595,
        0.00216342,
        0.0662588,
        0.0865501,
        0.00182127,
        0.0368557,
        0.0381149,
        0.00332388,
        0.189974,
        0.009384,
        0.00394874,
        0.116311,
        0.0151208,
        0.093936,
        0.116173,
        0.0842204,
        0.0565954,
        0.00645142,
        0.00071873,
        0.00461894
      ],
      [ 0.0572262,
        0.00153015,
        0.179393,
        0.199226,
        0.00137018,
        0.0316472,
        0.0291392,
        0.00458046,
        0.101562,
        0.010074,
        0.00402046,
        0.108388,
        0.00636741,
        0.0903669,
        0.0494724,
        0.0621143,
        0.0496102,
        0.00859413,
        0.000666929,
        0.00464976
      ],
      [ 0.00360202,
        0.00454848,
        0.00208716,
        0.00178577,
        0.0855715,
        0.00563916,
        0.00649688,
        0.00292929,
        0.00104198,
        0.0232635,
        0.00445923,
        0.00134555,
        0.0024992,
        0.00327181,
        0.0102713,
        0.00306718,
        0.00259003,
        0.00586684,
        0.761782,
        0.067881
      ],
      [ 0.203202,
        0.00981316,
        0.0135012,
        0.00838182,
        0.00196196,
        0.618489,
        0.00277479,
        0.00118285,
        0.00445989,
        0.00398268,
        0.00206318,
        0.0143744,
        0.00858704,
        0.00445146,
        0.00838957,
        0.073992,
        0.0108922,
        0.00607691,
        0.00186061,
        0.00156387
      ],
      [ 0.00508988,
        0.00617765,
        0.00161262,
        0.00120404,
        0.356313,
        0.00163342,
        0.0393461,
        0.00590888,
        0.00137137,
        0.0249344,
        0.00497952,
        0.0057093,
        0.00141364,
        0.00246931,
        0.00287408,
        0.00595277,
        0.00365368,
        0.00819341,
        0.0357987,
        0.485365
      ],
      [ 0.0403336,
        0.00815495,
        0.00982186,
        0.0375407,
        0.0119141,
        0.00479344,
        0.0176736,
        0.189342,
        0.0607377,
        0.105186,
        0.03056,
        0.0216052,
        0.00775506,
        0.0383639,
        0.0540186,
        0.025711,
        0.100991,
        0.221091,
        0.002878,
        0.0115277
      ],
      [ 0.0790086,
        0.00249479,
        0.0546012,
        0.199788,
        0.00237734,
        0.0192656,
        0.02707,
        0.00756675,
        0.155311,
        0.0254542,
        0.00980244,
        0.0309384,
        0.00566407,
        0.184338,
        0.106544,
        0.0332371,
        0.0359575,
        0.0145306,
        0.00116828,
        0.00488208
      ],
      [ 0.0722241,
        0.00647553,
        0.119488,
        0.134589,
        0.0348233,
        0.0287815,
        0.0699011,
        0.0173589,
        0.0490342,
        0.051987,
        0.0154411,
        0.067893,
        0.0145597,
        0.070897,
        0.0489728,
        0.058958,
        0.0425973,
        0.0317884,
        0.00879138,
        0.0554387
      ],
      [ 0.108584,
        0.00125026,
        0.152967,
        0.166485,
        0.00145535,
        0.0336098,
        0.0134902,
        0.00388218,
        0.0576227,
        0.00898614,
        0.0024339,
        0.0441956,
        0.19901,
        0.0405398,
        0.020645,
        0.084675,
        0.0454715,
        0.0113416,
        0.000590283,
        0.00276502
      ],
      [ 0.0309526,
        0.0065594,
        0.00823521,
        0.0291974,
        0.00368916,
        0.0154206,
        0.0310385,
        0.00982516,
        0.306263,
        0.02379,
        0.00970717,
        0.0301337,
        0.00950291,
        0.0832608,
        0.319589,
        0.0295285,
        0.0303052,
        0.0133037,
        0.00281253,
        0.00688506
      ],
      [ 0.00989537,
        0.00282767,
        0.000374823,
        0.00091821,
        0.0298607,
        0.000699707,
        0.00104195,
        0.311504,
        0.00139605,
        0.375039,
        0.0474451,
        0.000730793,
        0.00252963,
        0.0017337,
        0.00196045,
        0.0014628,
        0.0075739,
        0.1973,
        0.00167998,
        0.00402599
      ],
      [ 0.116321,
        0.00347923,
        0.0731918,
        0.138088,
        0.00941177,
        0.0193193,
        0.0160241,
        0.0712243,
        0.035512,
        0.0771474,
        0.0242841,
        0.0250164,
        0.0508927,
        0.0586677,
        0.0273321,
        0.047556,
        0.0726552,
        0.123571,
        0.00268927,
        0.0076166
      ],
      [ 0.128522,
        0.0172929,
        0.040275,
        0.0250692,
        0.0199118,
        0.112703,
        0.0606981,
        0.010935,
        0.028875,
        0.0258416,
        0.0167593,
        0.117984,
        0.0180675,
        0.0439706,
        0.0373073,
        0.174149,
        0.0648968,
        0.0182067,
        0.0063575,
        0.0321772
      ],
      [ 0.0372287,
        0.014494,
        0.00237032,
        0.00485851,
        0.146377,
        0.00464339,
        0.0186795,
        0.182046,
        0.00581985,
        0.17801,
        0.0371334,
        0.00533773,
        0.00485386,
        0.00790971,
        0.0094528,
        0.0103571,
        0.0284162,
        0.184992,
        0.0211294,
        0.0958905
      ],
      [ 0.0535644,
        0.00962562,
        0.0113537,
        0.0391699,
        0.0264214,
        0.0120279,
        0.0384888,
        0.0522748,
        0.0996038,
        0.189239,
        0.0712219,
        0.0239173,
        0.00837206,
        0.0928585,
        0.11598,
        0.0299114,
        0.0389485,
        0.0500948,
        0.0104232,
        0.026503
      ],
      [ 0.133242,
        0.0246514,
        0.00127392,
        0.00404615,
        0.0156995,
        0.00891392,
        0.00158647,
        0.197128,
        0.00237132,
        0.125129,
        0.0286947,
        0.0022705,
        0.0118846,
        0.00308435,
        0.00331477,
        0.0171462,
        0.0563298,
        0.356621,
        0.00173418,
        0.00487784
      ],
      [ 0.149866,
        0.00354374,
        0.0280355,
        0.0435381,
        0.00475339,
        0.0311113,
        0.0140626,
        0.0101953,
        0.0393125,
        0.0251434,
        0.00515483,
        0.0176453,
        0.39238,
        0.0348151,
        0.0326607,
        0.0874497,
        0.0473307,
        0.0271597,
        0.00152152,
        0.00432083
      ],
      [ 0.421437,
        0.018761,
        0.00733051,
        0.00868378,
        0.00271561,
        0.0902333,
        0.0030262,
        0.00393628,
        0.00515087,
        0.00471933,
        0.00383066,
        0.012159,
        0.020894,
        0.00727486,
        0.0061426,
        0.290119,
        0.0651922,
        0.0252211,
        0.000810824,
        0.00236228
      ],
      [ 0.177071,
        0.00783489,
        0.02265,
        0.0509767,
        0.00405142,
        0.089739,
        0.0220667,
        0.00595198,
        0.125769,
        0.020537,
        0.00929825,
        0.0311657,
        0.0264088,
        0.0752471,
        0.133278,
        0.116959,
        0.0565567,
        0.0165087,
        0.00299471,
        0.00493467
      ],
      [ 0.0293984,
        0.00317291,
        0.109971,
        0.046427,
        0.00150396,
        0.422242,
        0.0272495,
        0.000799733,
        0.0622314,
        0.00376343,
        0.00166571,
        0.148362,
        0.00564818,
        0.0388688,
        0.0370902,
        0.0472252,
        0.0086569,
        0.00203639,
        0.000917602,
        0.00276931
      ],
      [ 0.0265779,
        0.0101369,
        0.0280314,
        0.0269057,
        0.0276961,
        0.0173377,
        0.281513,
        0.0064647,
        0.0474749,
        0.026821,
        0.00723753,
        0.13186,
        0.0083015,
        0.0989711,
        0.0791105,
        0.0426277,
        0.0259043,
        0.0100147,
        0.00785289,
        0.0891598
      ],
      [ 0.00960965,
        0.00773017,
        0.000633186,
        0.00104719,
        0.263017,
        0.00202274,
        0.00390014,
        0.0733098,
        0.00149315,
        0.445169,
        0.0732575,
        0.00131044,
        0.00427681,
        0.00338994,
        0.00271362,
        0.00361174,
        0.00579284,
        0.0425173,
        0.0181276,
        0.0370698
      ]
    ]

-- | Weights of C40 model.
c40Weights :: [Double]
c40Weights =
  [ 0.0223854,
    0.0338892,
    0.0577169,
    0.0252416,
    0.0108608,
    0.0462374,
    0.0102293,
    0.0147524,
    0.0143161,
    0.0182303,
    0.0204025,
    0.0425505,
    0.0248627,
    0.0105893,
    0.0188239,
    0.00866634,
    0.0148496,
    0.0343037,
    0.0225335,
    0.0174069,
    0.0112208,
    0.0443532,
    0.0392573,
    0.0196757,
    0.028769,
    0.0114441,
    0.0112339,
    0.0582694,
    0.0444272,
    0.0112011,
    0.0145176,
    0.0114629,
    0.0239628,
    0.0266266,
    0.0481201,
    0.0371147,
    0.0160477,
    0.0237249,
    0.0235226,
    0.0261998
  ]

-- | Stationary distribution of C50 model.
c50StatDists :: [StationaryDistribution]
c50StatDists =
  map
    V.fromList
    [ [ 0.115682,
        0.00212412,
        0.125699,
        0.0919063,
        0.0016175,
        0.0391977,
        0.0108596,
        0.00344834,
        0.0344439,
        0.0125587,
        0.00467132,
        0.0530851,
        0.206661,
        0.0264814,
        0.0147638,
        0.154493,
        0.0841319,
        0.0134754,
        0.000392551,
        0.00430762
      ],
      [ 0.0983768,
        0.00160659,
        0.0456633,
        0.0935028,
        0.00112009,
        0.0251388,
        0.0191777,
        0.00367483,
        0.151407,
        0.0115555,
        0.00258422,
        0.0266318,
        0.229718,
        0.0694385,
        0.0855866,
        0.0696565,
        0.0498854,
        0.0123983,
        0.00069296,
        0.00218381
      ],
      [ 0.0214598,
        0.00446298,
        0.000555089,
        0.00135258,
        0.00855351,
        0.00112246,
        0.000813505,
        0.399631,
        0.00128242,
        0.0868331,
        0.0170323,
        0.00117488,
        0.00183519,
        0.00101872,
        0.00142423,
        0.00233511,
        0.0161891,
        0.430205,
        0.00061736,
        0.00210206
      ],
      [ 0.117155,
        0.032654,
        0.00620066,
        0.00998032,
        0.0657132,
        0.0333659,
        0.0134621,
        0.0557335,
        0.00794829,
        0.199356,
        0.0938018,
        0.0221831,
        0.0115415,
        0.0204903,
        0.0146873,
        0.0842555,
        0.0920314,
        0.0798031,
        0.0127314,
        0.0269048
      ],
      [ 0.0371284,
        0.0131607,
        0.00255462,
        0.00465054,
        0.156922,
        0.00416133,
        0.019977,
        0.178808,
        0.00498614,
        0.165159,
        0.0352649,
        0.0057612,
        0.00512406,
        0.00679862,
        0.00826841,
        0.0102949,
        0.0275498,
        0.181807,
        0.0221255,
        0.109498
      ],
      [ 0.0540087,
        0.000836323,
        0.306245,
        0.38187,
        0.00111469,
        0.0245756,
        0.0116353,
        0.00254509,
        0.030937,
        0.00536416,
        0.00256903,
        0.0359123,
        0.0122515,
        0.0586791,
        0.0115134,
        0.0277199,
        0.020314,
        0.00828799,
        0.000720693,
        0.00290032
      ],
      [ 0.0404572,
        0.028652,
        0.00495693,
        0.0068813,
        0.242849,
        0.0103569,
        0.0318377,
        0.0329142,
        0.0105561,
        0.143431,
        0.0485779,
        0.0105988,
        0.00489299,
        0.0118557,
        0.0126873,
        0.0229136,
        0.0212457,
        0.0349069,
        0.0593874,
        0.220042
      ],
      [ 0.319667,
        0.138968,
        0.00144847,
        0.00200138,
        0.00997903,
        0.0503464,
        0.00296588,
        0.0182219,
        0.00202126,
        0.0260477,
        0.0136422,
        0.00936868,
        0.00591365,
        0.00358722,
        0.00435407,
        0.1656,
        0.118413,
        0.0996664,
        0.00164228,
        0.00614589
      ],
      [ 0.0977328,
        0.00251083,
        0.0724816,
        0.197027,
        0.00556062,
        0.0117997,
        0.0154288,
        0.0563751,
        0.0613631,
        0.047368,
        0.0159721,
        0.0241106,
        0.0149,
        0.0909736,
        0.034432,
        0.0408092,
        0.0929289,
        0.111482,
        0.00122744,
        0.00551621
      ],
      [ 0.0195453,
        0.0107627,
        0.0127492,
        0.011794,
        0.14032,
        0.0061666,
        0.215761,
        0.00826718,
        0.0118897,
        0.0328258,
        0.00664433,
        0.0423533,
        0.00528472,
        0.0258824,
        0.024497,
        0.0203711,
        0.0124449,
        0.0133889,
        0.0330587,
        0.345993
      ],
      [ 0.185338,
        0.00185224,
        0.161345,
        0.234878,
        0.00192286,
        0.054278,
        0.0109591,
        0.00550122,
        0.0382027,
        0.00925008,
        0.00343293,
        0.033406,
        0.0233824,
        0.0551124,
        0.0201268,
        0.0884267,
        0.051789,
        0.0177494,
        0.000835994,
        0.00221142
      ],
      [ 0.0143603,
        0.0036047,
        0.000411253,
        0.00108887,
        0.0196479,
        0.00187385,
        0.000991664,
        0.285685,
        0.00213672,
        0.317288,
        0.150556,
        0.00155703,
        0.00122034,
        0.0028111,
        0.00194519,
        0.00344669,
        0.0196686,
        0.167294,
        0.000843478,
        0.00356936
      ],
      [ 0.150577,
        0.00957632,
        0.0227565,
        0.0358641,
        0.0104881,
        0.0285062,
        0.0171219,
        0.0577109,
        0.0276149,
        0.0531781,
        0.0298859,
        0.0514957,
        0.0243637,
        0.0354202,
        0.0272841,
        0.132633,
        0.169554,
        0.104387,
        0.00219421,
        0.00938724
      ],
      [ 0.277615,
        0.0280749,
        0.00194439,
        0.00537302,
        0.0070772,
        0.0234861,
        0.0012257,
        0.0949678,
        0.00308357,
        0.0474515,
        0.0189616,
        0.0034483,
        0.0134815,
        0.00374076,
        0.00354151,
        0.0423818,
        0.0932708,
        0.326207,
        0.000963513,
        0.00370386
      ],
      [ 0.205879,
        0.00249091,
        0.075426,
        0.161651,
        0.00210732,
        0.0463391,
        0.0159993,
        0.00515537,
        0.0913036,
        0.0166911,
        0.00536596,
        0.0366042,
        0.00849072,
        0.0973331,
        0.0696431,
        0.088244,
        0.0501877,
        0.0172747,
        0.000776174,
        0.0030381
      ],
      [ 0.0704473,
        0.00149732,
        0.187469,
        0.208434,
        0.0013217,
        0.0409359,
        0.0247326,
        0.00339884,
        0.0739754,
        0.00761491,
        0.00270559,
        0.0965448,
        0.00673196,
        0.0820493,
        0.0355811,
        0.0832004,
        0.0610002,
        0.00788473,
        0.000646918,
        0.00382803
      ],
      [ 0.108767,
        0.00735776,
        0.0100122,
        0.0249269,
        0.015562,
        0.0114116,
        0.00730171,
        0.0923916,
        0.0138538,
        0.112308,
        0.0251594,
        0.00831075,
        0.259979,
        0.0187962,
        0.0142911,
        0.0329858,
        0.0608007,
        0.162244,
        0.0049709,
        0.00857003
      ],
      [ 0.109878,
        0.0295589,
        0.000867374,
        0.00267084,
        0.028604,
        0.0101436,
        0.00159952,
        0.171472,
        0.0015037,
        0.241889,
        0.0488425,
        0.00201796,
        0.00631863,
        0.00393786,
        0.00336164,
        0.0148413,
        0.0393773,
        0.273667,
        0.00359303,
        0.00585493
      ],
      [ 0.058736,
        0.00692528,
        0.0144651,
        0.0412403,
        0.0154694,
        0.015436,
        0.0243546,
        0.0388984,
        0.0996366,
        0.207192,
        0.103832,
        0.0289715,
        0.00833815,
        0.108218,
        0.0976056,
        0.0350037,
        0.0432645,
        0.0338957,
        0.00652556,
        0.0119916
      ],
      [ 0.0642736,
        0.00525071,
        0.144305,
        0.153274,
        0.0286093,
        0.0265312,
        0.0545105,
        0.0231701,
        0.0452879,
        0.058794,
        0.0192273,
        0.071063,
        0.0146855,
        0.066931,
        0.0405954,
        0.054046,
        0.043602,
        0.0350962,
        0.00655203,
        0.0441953
      ],
      [ 0.00487716,
        0.00503139,
        0.00209095,
        0.00192293,
        0.0983777,
        0.00615619,
        0.00725617,
        0.00400789,
        0.000993075,
        0.0277171,
        0.00537714,
        0.00145717,
        0.0029039,
        0.00350461,
        0.0112195,
        0.00329284,
        0.00295187,
        0.00708726,
        0.724652,
        0.0791231
      ],
      [ 0.0552756,
        0.0115894,
        0.00846155,
        0.0432312,
        0.00961795,
        0.0074182,
        0.0218432,
        0.0843826,
        0.163327,
        0.0631707,
        0.0228359,
        0.0221225,
        0.00652522,
        0.0639538,
        0.146077,
        0.0387464,
        0.100713,
        0.118118,
        0.00298298,
        0.00960875
      ],
      [ 0.00449551,
        0.00619902,
        0.000890609,
        0.000674178,
        0.400884,
        0.0013356,
        0.0223834,
        0.00805107,
        0.00105813,
        0.0341606,
        0.00701178,
        0.00318917,
        0.00130624,
        0.00157564,
        0.00181707,
        0.00502515,
        0.00337803,
        0.00946203,
        0.0345278,
        0.452575
      ],
      [ 0.0505469,
        0.00768183,
        0.0101983,
        0.0470721,
        0.0160596,
        0.00517598,
        0.0176387,
        0.169463,
        0.0497501,
        0.209444,
        0.0474166,
        0.0132404,
        0.00844474,
        0.053753,
        0.051076,
        0.0149657,
        0.03846,
        0.172926,
        0.00437246,
        0.0123139
      ],
      [ 0.0749936,
        0.0124353,
        0.0203869,
        0.0100161,
        0.0038092,
        0.0270453,
        0.00581932,
        0.00743392,
        0.014732,
        0.00838354,
        0.00640172,
        0.0530316,
        0.0176075,
        0.0096947,
        0.0117746,
        0.3482,
        0.348235,
        0.0157071,
        0.00137596,
        0.00291601
      ],
      [ 0.0569785,
        0.00176591,
        0.102737,
        0.106157,
        0.00132179,
        0.0360979,
        0.034465,
        0.00311879,
        0.185459,
        0.00858711,
        0.00431374,
        0.139374,
        0.00855798,
        0.0898676,
        0.101042,
        0.0669467,
        0.0422839,
        0.00552428,
        0.000686497,
        0.00471478
      ],
      [ 0.0746973,
        0.00394557,
        0.0423454,
        0.0906599,
        0.00363883,
        0.0183504,
        0.0229406,
        0.0156166,
        0.12297,
        0.0167059,
        0.00901115,
        0.0672502,
        0.0127747,
        0.0811334,
        0.0755385,
        0.132886,
        0.170553,
        0.0324157,
        0.00152093,
        0.00504616
      ],
      [ 0.17313,
        0.00938612,
        0.0203504,
        0.0387504,
        0.00448721,
        0.0867075,
        0.0227177,
        0.00492238,
        0.125544,
        0.0156727,
        0.00787132,
        0.0400476,
        0.0180405,
        0.0680443,
        0.133335,
        0.142136,
        0.0670614,
        0.0136107,
        0.00287027,
        0.00531475
      ],
      [ 0.00751784,
        0.00212117,
        0.000341917,
        0.000667876,
        0.0310677,
        0.000466913,
        0.000898937,
        0.320302,
        0.00103903,
        0.384419,
        0.0335748,
        0.000602706,
        0.00243131,
        0.00137426,
        0.00152778,
        0.00115498,
        0.00552329,
        0.199295,
        0.00172716,
        0.00394622
      ],
      [ 0.113459,
        0.0281671,
        0.0268294,
        0.0283293,
        0.0928796,
        0.0494616,
        0.0733971,
        0.0152335,
        0.0176739,
        0.037425,
        0.0140126,
        0.0539481,
        0.0259048,
        0.0303861,
        0.0259068,
        0.121621,
        0.0621544,
        0.0350518,
        0.0240853,
        0.124074
      ],
      [ 0.0810974,
        0.00899714,
        0.0930492,
        0.0372909,
        0.002713,
        0.0907986,
        0.0275326,
        0.00533957,
        0.0362449,
        0.00831441,
        0.00615441,
        0.177093,
        0.0120848,
        0.0316411,
        0.025247,
        0.233557,
        0.106302,
        0.00814021,
        0.000950692,
        0.00745168
      ],
      [ 0.00757098,
        0.00716117,
        0.00049456,
        0.000891326,
        0.297029,
        0.00155605,
        0.00342729,
        0.0711141,
        0.00107067,
        0.416918,
        0.0635143,
        0.000864035,
        0.00349893,
        0.00251506,
        0.00186372,
        0.00271598,
        0.00424124,
        0.041488,
        0.0202374,
        0.0518279
      ],
      [ 0.086899,
        0.00542266,
        0.0274343,
        0.117345,
        0.00568342,
        0.0162895,
        0.0386176,
        0.0182492,
        0.149526,
        0.048245,
        0.0150062,
        0.0265186,
        0.00577334,
        0.159013,
        0.149227,
        0.0416278,
        0.0492401,
        0.0283951,
        0.00272523,
        0.00876225
      ],
      [ 0.0501736,
        0.0113806,
        0.00768113,
        0.0191841,
        0.0119036,
        0.00521331,
        0.00988204,
        0.203482,
        0.0206446,
        0.0870652,
        0.036001,
        0.0197961,
        0.0061624,
        0.0180415,
        0.0178215,
        0.0399964,
        0.162255,
        0.262373,
        0.00172457,
        0.00921833
      ],
      [ 0.177098,
        0.00357587,
        0.0740854,
        0.0713923,
        0.00385864,
        0.211235,
        0.0161904,
        0.0040472,
        0.0370753,
        0.00826202,
        0.00369757,
        0.0494737,
        0.0543871,
        0.0396583,
        0.0340392,
        0.140615,
        0.0532238,
        0.0108131,
        0.00212448,
        0.00514764
      ],
      [ 0.242384,
        0.0115188,
        0.00720908,
        0.00544261,
        0.00229719,
        0.593273,
        0.00205723,
        0.00208173,
        0.00287156,
        0.00491,
        0.00271204,
        0.00899551,
        0.00763059,
        0.00285962,
        0.00600369,
        0.0738385,
        0.0108329,
        0.00968775,
        0.00178961,
        0.00160391
      ],
      [ 0.032912,
        0.00319267,
        0.143899,
        0.0532755,
        0.00166462,
        0.463509,
        0.0175283,
        0.000876108,
        0.0317056,
        0.00297349,
        0.0012156,
        0.131603,
        0.00688601,
        0.0251373,
        0.0184844,
        0.0491059,
        0.00984294,
        0.00239178,
        0.0008917,
        0.00290456
      ],
      [ 0.0388306,
        0.00489943,
        0.0491655,
        0.0413874,
        0.00170354,
        0.345044,
        0.0421499,
        0.00103798,
        0.123725,
        0.00726049,
        0.00335164,
        0.119727,
        0.00562115,
        0.0664068,
        0.0881106,
        0.0460192,
        0.00791813,
        0.0020819,
        0.00195492,
        0.00360516
      ],
      [ 0.0373301,
        0.00916334,
        0.0396965,
        0.0362168,
        0.0144959,
        0.0269081,
        0.220512,
        0.00638711,
        0.0634517,
        0.0225147,
        0.00718608,
        0.152708,
        0.00995802,
        0.108449,
        0.0882553,
        0.0609295,
        0.0332848,
        0.0105312,
        0.00429315,
        0.047729
      ],
      [ 0.0171508,
        0.00565843,
        0.00111564,
        0.00268175,
        0.0490847,
        0.00219974,
        0.00370226,
        0.116938,
        0.00407925,
        0.500942,
        0.1832,
        0.00280387,
        0.00400697,
        0.0115255,
        0.00613355,
        0.00539456,
        0.0140434,
        0.0579479,
        0.00419195,
        0.00719895
      ],
      [ 0.0238117,
        0.00329378,
        0.370777,
        0.0831676,
        0.0022275,
        0.053433,
        0.0290274,
        0.00288609,
        0.0258098,
        0.00370314,
        0.00159506,
        0.259554,
        0.00668902,
        0.0201877,
        0.0117854,
        0.0626213,
        0.0278627,
        0.00464381,
        0.000688992,
        0.0062349
      ],
      [ 0.209736,
        0.0105937,
        0.0311812,
        0.0613839,
        0.00991105,
        0.0855144,
        0.0226071,
        0.0296058,
        0.0455281,
        0.0792591,
        0.0306389,
        0.019447,
        0.0213982,
        0.0610672,
        0.0588191,
        0.0820738,
        0.0527883,
        0.0722389,
        0.00513,
        0.0110785
      ],
      [ 0.0295833,
        0.00588894,
        0.00852018,
        0.0296344,
        0.00323424,
        0.0154495,
        0.02992,
        0.00752729,
        0.317117,
        0.0212988,
        0.00874154,
        0.0312424,
        0.00843308,
        0.0831654,
        0.323587,
        0.0294177,
        0.0283085,
        0.0100345,
        0.00257807,
        0.00631797
      ],
      [ 0.0937098,
        0.00289961,
        0.0255803,
        0.0577481,
        0.0140839,
        0.0182984,
        0.0300827,
        0.0312422,
        0.0867966,
        0.067914,
        0.01767,
        0.0206426,
        0.213942,
        0.0611154,
        0.0902988,
        0.0526724,
        0.0464314,
        0.0509513,
        0.00365078,
        0.01427
      ],
      [ 0.419507,
        0.0176715,
        0.00719436,
        0.00880545,
        0.00285112,
        0.0865515,
        0.00330323,
        0.00419601,
        0.00503365,
        0.0048203,
        0.00393323,
        0.012443,
        0.0192167,
        0.00750218,
        0.00609116,
        0.295973,
        0.0660748,
        0.0253174,
        0.000942289,
        0.00257126
      ],
      [ 0.0368106,
        0.0124259,
        0.0102859,
        0.0343205,
        0.0759532,
        0.00772969,
        0.0938663,
        0.0458547,
        0.0701169,
        0.133403,
        0.031916,
        0.0265402,
        0.00749485,
        0.0673302,
        0.10388,
        0.0271625,
        0.0353789,
        0.0485031,
        0.0254607,
        0.105568
      ],
      [ 0.0665479,
        0.00185956,
        0.183575,
        0.149215,
        0.00127224,
        0.0634659,
        0.019684,
        0.00262142,
        0.0655694,
        0.00750609,
        0.00151282,
        0.0699101,
        0.182105,
        0.0446839,
        0.0292035,
        0.075452,
        0.0278833,
        0.00484561,
        0.000723776,
        0.00236318
      ],
      [ 0.16002,
        0.00397537,
        0.019682,
        0.0286161,
        0.00458768,
        0.0326662,
        0.0114325,
        0.00722291,
        0.0232558,
        0.0202542,
        0.00379358,
        0.0142832,
        0.460797,
        0.0260458,
        0.0217875,
        0.0913633,
        0.0427326,
        0.0227118,
        0.0012566,
        0.0035159
      ],
      [ 0.0843584,
        0.00129715,
        0.0847585,
        0.260552,
        0.00143564,
        0.0191268,
        0.0174708,
        0.00556223,
        0.144172,
        0.014456,
        0.00550323,
        0.0336163,
        0.00724868,
        0.160704,
        0.0742581,
        0.0323411,
        0.0355032,
        0.0136662,
        0.000760206,
        0.00320994
      ],
      [ 0.152149,
        0.000925399,
        0.116471,
        0.220438,
        0.00150071,
        0.0172039,
        0.0095692,
        0.00747151,
        0.0529323,
        0.0148756,
        0.00356173,
        0.0157411,
        0.217335,
        0.0489135,
        0.0171707,
        0.044102,
        0.0323715,
        0.023915,
        0.000908902,
        0.00244332
      ]
    ]

-- | Weights of C50 model.
c50Weights :: [Double]
c50Weights =
  [ 0.00509464,
    0.00753314,
    0.0595801,
    0.0100458,
    0.0109109,
    0.0427412,
    0.00632106,
    0.0192362,
    0.0119168,
    0.0154209,
    0.0148037,
    0.014293,
    0.0164064,
    0.0272538,
    0.0178737,
    0.0125183,
    0.00698661,
    0.0140027,
    0.0140399,
    0.0109043,
    0.011579,
    0.0138661,
    0.0377329,
    0.0118586,
    0.0356688,
    0.0161327,
    0.0200601,
    0.0133087,
    0.0357641,
    0.0119353,
    0.0241183,
    0.0243098,
    0.0178039,
    0.0200988,
    0.00579569,
    0.0397441,
    0.0240799,
    0.00868054,
    0.0239822,
    0.042803,
    0.0335193,
    0.00636191,
    0.0547546,
    0.00544598,
    0.0354219,
    0.0120158,
    0.0047506,
    0.036392,
    0.0270346,
    0.00709747
  ]

-- | Stationary distribution of C60 model.
c60StatDists :: [StationaryDistribution]
c60StatDists =
  map
    V.fromList
    [ [ 0.153436,
        0.00473066,
        0.0546757,
        0.0529324,
        0.00355543,
        0.110378,
        0.017448,
        0.00503439,
        0.0603929,
        0.0130294,
        0.00755506,
        0.0796727,
        0.0249524,
        0.0514333,
        0.0444389,
        0.202963,
        0.0957668,
        0.0101809,
        0.00144445,
        0.00598003
      ],
      [ 0.0281985,
        0.00195035,
        0.00915493,
        0.038853,
        0.00159193,
        0.00840283,
        0.0155385,
        0.0107873,
        0.347604,
        0.0217787,
        0.0109905,
        0.0312955,
        0.00715399,
        0.0939884,
        0.303106,
        0.0197479,
        0.0328352,
        0.0135303,
        0.000920999,
        0.0025714
      ],
      [ 0.00836807,
        0.00204339,
        0.000222837,
        0.000473154,
        0.0123984,
        0.000482575,
        0.000518999,
        0.376845,
        0.000629617,
        0.260833,
        0.0315701,
        0.000612345,
        0.000959592,
        0.000949869,
        0.000731977,
        0.000974638,
        0.00499908,
        0.293808,
        0.000865776,
        0.00171323
      ],
      [ 0.222723,
        0.164029,
        0.000799759,
        0.00272808,
        0.0228667,
        0.0202703,
        0.00371839,
        0.0455406,
        0.00228329,
        0.088335,
        0.0348095,
        0.00612065,
        0.00354716,
        0.00510519,
        0.00648461,
        0.085004,
        0.101285,
        0.169858,
        0.00484248,
        0.00965
      ],
      [ 0.041214,
        0.0204716,
        0.00174344,
        0.00372634,
        0.2239,
        0.00691075,
        0.0180294,
        0.115428,
        0.00429003,
        0.169356,
        0.0414067,
        0.00510677,
        0.00584164,
        0.00575385,
        0.00676271,
        0.0149106,
        0.0239548,
        0.120034,
        0.0332237,
        0.137935
      ],
      [ 0.048055,
        0.00378018,
        0.208461,
        0.185518,
        0.0196001,
        0.0191402,
        0.0872162,
        0.00946854,
        0.0375741,
        0.0277341,
        0.00883084,
        0.0940629,
        0.00812678,
        0.0747677,
        0.0308438,
        0.0439681,
        0.0324589,
        0.0181769,
        0.00346657,
        0.03875
      ],
      [ 0.00628487,
        0.00730276,
        0.000532415,
        0.000974149,
        0.397455,
        0.00195782,
        0.0102225,
        0.0180592,
        0.00162059,
        0.117906,
        0.0234722,
        0.00303425,
        0.00201656,
        0.00344091,
        0.00262469,
        0.00569033,
        0.00370918,
        0.0152754,
        0.0598639,
        0.318557
      ],
      [ 0.181501,
        0.420563,
        0.00251455,
        0.000708814,
        0.00689764,
        0.0461854,
        0.00143746,
        0.00417455,
        0.000647425,
        0.00983105,
        0.00416114,
        0.0148485,
        0.00387672,
        0.0014097,
        0.00268454,
        0.186454,
        0.068719,
        0.0345742,
        0.00270835,
        0.0061033
      ],
      [ 0.0600741,
        0.00707198,
        0.0170572,
        0.0127487,
        0.114091,
        0.0343564,
        0.0305859,
        0.0204571,
        0.0212367,
        0.0994551,
        0.0318166,
        0.0134869,
        0.0297628,
        0.014247,
        0.0367643,
        0.0505793,
        0.0339368,
        0.0217422,
        0.231281,
        0.119249
      ],
      [ 0.0708395,
        0.0131641,
        0.0324483,
        0.0508264,
        0.059436,
        0.0183309,
        0.0567273,
        0.0650369,
        0.0343618,
        0.128226,
        0.0390363,
        0.0416822,
        0.0135608,
        0.0494875,
        0.0474098,
        0.0551343,
        0.064226,
        0.078961,
        0.0137118,
        0.0673934
      ],
      [ 0.0617689,
        0.000719984,
        0.343023,
        0.379251,
        0.000981419,
        0.0284659,
        0.00795921,
        0.00169996,
        0.0216077,
        0.00399453,
        0.00197343,
        0.0303082,
        0.0174791,
        0.0307856,
        0.00763329,
        0.0337832,
        0.0203427,
        0.00589923,
        0.000613027,
        0.00171028
      ],
      [ 0.042156,
        0.00444503,
        0.0164385,
        0.0531949,
        0.00867283,
        0.0134711,
        0.0177765,
        0.0267728,
        0.132374,
        0.196724,
        0.118283,
        0.0286981,
        0.00518379,
        0.139369,
        0.104207,
        0.0255853,
        0.0333292,
        0.0217067,
        0.00458523,
        0.00702815
      ],
      [ 0.281481,
        0.0258338,
        0.00643857,
        0.0115046,
        0.00504817,
        0.0270055,
        0.00546297,
        0.0188216,
        0.00987128,
        0.0190993,
        0.015872,
        0.0172868,
        0.012951,
        0.0133102,
        0.0100367,
        0.188681,
        0.24271,
        0.0837524,
        0.00122746,
        0.00360529
      ],
      [ 0.276919,
        0.0318293,
        0.00116725,
        0.00247525,
        0.00708439,
        0.0199647,
        0.000517086,
        0.0983109,
        0.00162322,
        0.0489264,
        0.0173415,
        0.00213153,
        0.00701797,
        0.00182163,
        0.0017227,
        0.0336349,
        0.0814141,
        0.362092,
        0.000711814,
        0.00329423
      ],
      [ 0.15778,
        0.00533181,
        0.064829,
        0.0913587,
        0.00250478,
        0.090621,
        0.053381,
        0.00291716,
        0.0782149,
        0.0156226,
        0.0045759,
        0.0570403,
        0.00670778,
        0.106537,
        0.111214,
        0.092931,
        0.0393123,
        0.00850409,
        0.00285758,
        0.00775903
      ],
      [ 0.0593735,
        0.00153322,
        0.218948,
        0.181942,
        0.00108383,
        0.0813707,
        0.0220478,
        0.00209936,
        0.0750172,
        0.00561913,
        0.00218717,
        0.115118,
        0.0109737,
        0.0688752,
        0.0354741,
        0.0726449,
        0.0380238,
        0.00426697,
        0.000734646,
        0.00266649
      ],
      [ 0.0978066,
        0.00641384,
        0.0120781,
        0.0291283,
        0.0198964,
        0.0128913,
        0.0128037,
        0.0723904,
        0.0245631,
        0.127944,
        0.0303267,
        0.0101844,
        0.272352,
        0.0307876,
        0.0265576,
        0.0350549,
        0.0484557,
        0.115203,
        0.00468425,
        0.0104774
      ],
      [ 0.0124023,
        0.00418857,
        0.000603732,
        0.0012524,
        0.0610615,
        0.00113068,
        0.00135421,
        0.234417,
        0.00167182,
        0.455056,
        0.0667308,
        0.00092391,
        0.00370762,
        0.00329574,
        0.00306804,
        0.00194209,
        0.00676129,
        0.129048,
        0.00389372,
        0.00749118
      ],
      [ 0.0794231,
        0.00199195,
        0.0587236,
        0.12469,
        0.001287,
        0.0325432,
        0.0238605,
        0.00362772,
        0.21476,
        0.0097988,
        0.00418462,
        0.0662793,
        0.014241,
        0.114388,
        0.129474,
        0.0615807,
        0.0477334,
        0.00781872,
        0.000652537,
        0.00294202
      ],
      [ 0.0248149,
        0.00278397,
        0.427883,
        0.143239,
        0.00146804,
        0.0643968,
        0.0185737,
        0.00225069,
        0.0179274,
        0.00345586,
        0.00157145,
        0.188892,
        0.0073768,
        0.0210778,
        0.00835529,
        0.0377003,
        0.0187768,
        0.00460727,
        0.000589186,
        0.00426027
      ],
      [ 0.00170034,
        0.00290594,
        0.00107115,
        0.00117317,
        0.0513377,
        0.00355796,
        0.00276305,
        0.00122912,
        0.00042738,
        0.0127421,
        0.00256713,
        0.00042229,
        0.00135367,
        0.00164242,
        0.00606743,
        0.00118717,
        0.00140331,
        0.00331373,
        0.864044,
        0.0390913
      ],
      [ 0.0468361,
        0.00599541,
        0.0185616,
        0.0705436,
        0.00801772,
        0.00454353,
        0.0152063,
        0.155061,
        0.0866248,
        0.0824253,
        0.0245854,
        0.0205604,
        0.00814856,
        0.0557031,
        0.0639797,
        0.0237026,
        0.0962054,
        0.204749,
        0.00183687,
        0.00671317
      ],
      [ 0.0258765,
        0.0142504,
        0.0107037,
        0.010565,
        0.187643,
        0.0073065,
        0.141108,
        0.0114778,
        0.0119179,
        0.0407992,
        0.0098799,
        0.0298384,
        0.00512288,
        0.0158529,
        0.0201097,
        0.02757,
        0.0170765,
        0.0187502,
        0.0405125,
        0.353639
      ],
      [ 0.0296285,
        0.00900465,
        0.000885102,
        0.00330465,
        0.0408798,
        0.00279695,
        0.00507015,
        0.25834,
        0.00466909,
        0.266809,
        0.0770825,
        0.00349444,
        0.00269182,
        0.00554811,
        0.00464003,
        0.00685381,
        0.0322266,
        0.22719,
        0.00355061,
        0.0153353
      ],
      [ 0.0555726,
        0.00974186,
        0.014039,
        0.00694432,
        0.00344967,
        0.0157957,
        0.00416313,
        0.00698265,
        0.0139225,
        0.00752712,
        0.00587627,
        0.0409064,
        0.0119733,
        0.00687277,
        0.00984479,
        0.348247,
        0.421366,
        0.0131119,
        0.00100615,
        0.00265768
      ],
      [ 0.0682671,
        0.0141433,
        0.0360279,
        0.0497415,
        0.0803376,
        0.0268697,
        0.112767,
        0.0132647,
        0.0482609,
        0.0544494,
        0.0170034,
        0.0530661,
        0.019195,
        0.0612274,
        0.0615207,
        0.067184,
        0.0443996,
        0.0267714,
        0.0199958,
        0.125507
      ],
      [ 0.0792619,
        0.00495033,
        0.0436646,
        0.0829639,
        0.00491311,
        0.0183429,
        0.0233169,
        0.0249427,
        0.0932578,
        0.0221483,
        0.0120893,
        0.0635289,
        0.012636,
        0.0666365,
        0.0638377,
        0.133485,
        0.191675,
        0.0503102,
        0.00180401,
        0.00623531
      ],
      [ 0.0731759,
        0.0123459,
        0.0110007,
        0.0349943,
        0.00931488,
        0.0224509,
        0.0386903,
        0.0246226,
        0.178334,
        0.0508307,
        0.0185741,
        0.0324201,
        0.0148723,
        0.0858952,
        0.210534,
        0.0603181,
        0.0649575,
        0.038504,
        0.00510464,
        0.0130597
      ],
      [ 0.0878403,
        0.0171147,
        0.00328039,
        0.0101466,
        0.0191954,
        0.00870909,
        0.00549022,
        0.198776,
        0.00820969,
        0.175646,
        0.0417233,
        0.00608012,
        0.0111284,
        0.0109832,
        0.0110332,
        0.0209863,
        0.0697719,
        0.28252,
        0.0031744,
        0.00819055
      ],
      [ 0.0990216,
        0.0108996,
        0.0118798,
        0.0279,
        0.0214442,
        0.024025,
        0.0123445,
        0.0776565,
        0.0322818,
        0.235451,
        0.120767,
        0.021115,
        0.00756555,
        0.0557711,
        0.0349352,
        0.052417,
        0.0649785,
        0.0771067,
        0.00470758,
        0.00773287
      ],
      [ 0.0601641,
        0.0315067,
        0.0337189,
        0.00598395,
        0.00695259,
        0.0543081,
        0.0531524,
        0.00706508,
        0.0139598,
        0.00706981,
        0.00882987,
        0.278352,
        0.00758343,
        0.0210646,
        0.0161995,
        0.28298,
        0.0860317,
        0.0100954,
        0.00149666,
        0.0134849
      ],
      [ 0.00497817,
        0.00529787,
        0.000531537,
        0.000763055,
        0.335112,
        0.00150513,
        0.00415752,
        0.0443829,
        0.00116151,
        0.441742,
        0.0602807,
        0.000701221,
        0.00278477,
        0.00240899,
        0.00184123,
        0.00257958,
        0.00302885,
        0.0237677,
        0.0171303,
        0.0458456
      ],
      [ 0.0251997,
        0.00970995,
        0.00410123,
        0.0262572,
        0.0429092,
        0.00380673,
        0.0431939,
        0.0974043,
        0.0824313,
        0.24472,
        0.0539323,
        0.0142032,
        0.00526585,
        0.0620071,
        0.111447,
        0.00960931,
        0.0251183,
        0.0783303,
        0.0146572,
        0.0456965
      ],
      [ 0.0230362,
        0.00485801,
        0.000626444,
        0.00157626,
        0.0072127,
        0.00115653,
        0.000889924,
        0.39529,
        0.00146635,
        0.0576537,
        0.0140987,
        0.00135344,
        0.00201779,
        0.000987005,
        0.00147487,
        0.00287702,
        0.0205581,
        0.460366,
        0.000547769,
        0.00195391
      ],
      [ 0.140878,
        0.00322137,
        0.0285077,
        0.0423839,
        0.00412311,
        0.0299558,
        0.0131321,
        0.00669145,
        0.0383782,
        0.019512,
        0.00362769,
        0.0171298,
        0.438347,
        0.0320632,
        0.0297808,
        0.08514,
        0.0422766,
        0.0198195,
        0.00132369,
        0.00370876
      ],
      [ 0.444249,
        0.0235405,
        0.0057193,
        0.00648899,
        0.00244132,
        0.0822688,
        0.00255057,
        0.00336151,
        0.00380971,
        0.00409901,
        0.00286831,
        0.0102305,
        0.0162891,
        0.00559976,
        0.00502166,
        0.299997,
        0.0559665,
        0.0226608,
        0.000773543,
        0.00206398
      ],
      [ 0.0898718,
        0.00441315,
        0.0129167,
        0.0058957,
        0.000807357,
        0.808756,
        0.00165179,
        0.000433928,
        0.0033939,
        0.00155645,
        0.000425342,
        0.0130068,
        0.00341281,
        0.00238065,
        0.00709583,
        0.0362877,
        0.00328875,
        0.00202896,
        0.00152239,
        0.000853745
      ],
      [ 0.055084,
        0.00358249,
        0.12738,
        0.0655885,
        0.0040375,
        0.0637608,
        0.0404883,
        0.00755742,
        0.0867683,
        0.0136305,
        0.00816842,
        0.187783,
        0.0110682,
        0.0527969,
        0.0472254,
        0.126338,
        0.0752544,
        0.0102728,
        0.00135637,
        0.011859
      ],
      [ 0.0117681,
        0.00703883,
        0.0144712,
        0.0110901,
        0.0338266,
        0.00496266,
        0.433719,
        0.0061337,
        0.0137929,
        0.0298795,
        0.00762376,
        0.0844145,
        0.00813461,
        0.103834,
        0.0442559,
        0.0140571,
        0.0108277,
        0.00837578,
        0.00806831,
        0.143725
      ],
      [ 0.0159286,
        0.00805199,
        0.00109669,
        0.00213242,
        0.0743651,
        0.00250864,
        0.00491921,
        0.0501544,
        0.00355994,
        0.530763,
        0.216009,
        0.00326926,
        0.00452474,
        0.0134552,
        0.00480987,
        0.00669222,
        0.0119092,
        0.0281012,
        0.00709281,
        0.0106565
      ],
      [ 0.0195973,
        0.0034539,
        0.309985,
        0.0250778,
        0.00197065,
        0.0627529,
        0.0295961,
        0.00326504,
        0.0240964,
        0.00282469,
        0.000842506,
        0.32891,
        0.00490628,
        0.0116197,
        0.0105143,
        0.106498,
        0.0438054,
        0.0040302,
        0.000633396,
        0.0056198
      ],
      [ 0.0833804,
        0.00819811,
        0.0686821,
        0.0227415,
        0.00241326,
        0.0982292,
        0.00739549,
        0.00174712,
        0.0129342,
        0.00396531,
        0.0019558,
        0.0969824,
        0.0355924,
        0.0121521,
        0.0125871,
        0.311561,
        0.211337,
        0.00471386,
        0.0016329,
        0.00179911
      ],
      [ 0.0181409,
        0.00851436,
        0.00333335,
        0.00965319,
        0.00354269,
        0.0224553,
        0.0375239,
        0.00350905,
        0.320807,
        0.0149147,
        0.00460989,
        0.0233205,
        0.00871975,
        0.0526694,
        0.412966,
        0.0262309,
        0.0131791,
        0.00447465,
        0.0034767,
        0.00795882
      ],
      [ 0.249423,
        0.0122863,
        0.0169234,
        0.0370491,
        0.00773609,
        0.0347468,
        0.00870696,
        0.0595718,
        0.0177065,
        0.0451065,
        0.0204556,
        0.016412,
        0.0686404,
        0.0228502,
        0.0185482,
        0.0889296,
        0.0986017,
        0.167252,
        0.00286039,
        0.00619385
      ],
      [ 0.141974,
        0.00688567,
        0.0537452,
        0.040754,
        0.00402861,
        0.398811,
        0.0152896,
        0.00166276,
        0.0314274,
        0.00923483,
        0.00554255,
        0.0576297,
        0.0180329,
        0.028624,
        0.0373946,
        0.112373,
        0.0242478,
        0.00484629,
        0.00259091,
        0.00490542
      ],
      [ 0.0178903,
        0.0290304,
        0.00310543,
        0.00405034,
        0.0238913,
        0.010048,
        0.12527,
        0.0016708,
        0.023334,
        0.0722357,
        0.0116142,
        0.0155854,
        0.000993842,
        0.105182,
        0.195884,
        0.0181676,
        0.018626,
        0.0113865,
        0.226055,
        0.0859787
      ],
      [ 0.145476,
        0.00141863,
        0.129425,
        0.247135,
        0.00175521,
        0.0319651,
        0.0130426,
        0.00585254,
        0.0818154,
        0.0123593,
        0.00441789,
        0.040042,
        0.0151136,
        0.0906469,
        0.0420979,
        0.0656688,
        0.0511289,
        0.0169401,
        0.000773144,
        0.00292584
      ],
      [ 0.0169799,
        0.00220872,
        0.104366,
        0.037638,
        0.000678059,
        0.540447,
        0.0181137,
        0.000745968,
        0.0508618,
        0.0021147,
        0.000947377,
        0.131805,
        0.00387544,
        0.026935,
        0.0242347,
        0.029703,
        0.00458362,
        0.00158917,
        0.000603189,
        0.00157041
      ],
      [ 0.0402646,
        0.00393887,
        0.0293968,
        0.160352,
        0.00411534,
        0.012926,
        0.0617968,
        0.00984913,
        0.14488,
        0.0354919,
        0.0124819,
        0.0323829,
        0.00433742,
        0.249701,
        0.115202,
        0.0243247,
        0.0305645,
        0.0156027,
        0.00266766,
        0.00972278
      ],
      [ 0.225691,
        0.00433901,
        0.0637125,
        0.115483,
        0.00274401,
        0.0867335,
        0.0131067,
        0.00850862,
        0.0660469,
        0.0193314,
        0.00648772,
        0.0244309,
        0.0611149,
        0.0578159,
        0.0523417,
        0.107088,
        0.0507677,
        0.0299883,
        0.00136959,
        0.00289829
      ],
      [ 0.00331642,
        0.00518907,
        0.000826647,
        0.000513413,
        0.380176,
        0.00104328,
        0.0278451,
        0.00418953,
        0.000714992,
        0.0111212,
        0.00236218,
        0.00308302,
        0.000836508,
        0.00110249,
        0.00153108,
        0.00358767,
        0.00236089,
        0.00607663,
        0.0333347,
        0.510789
      ],
      [ 0.199501,
        0.058459,
        0.00949551,
        0.0125535,
        0.0634434,
        0.0939787,
        0.0341857,
        0.014021,
        0.0118723,
        0.0449388,
        0.0246991,
        0.0392255,
        0.0145385,
        0.0254265,
        0.0236079,
        0.166392,
        0.0533159,
        0.0367303,
        0.0129803,
        0.0606346
      ],
      [ 0.0319449,
        0.00660403,
        0.0378075,
        0.0279488,
        0.0108916,
        0.0365541,
        0.208864,
        0.00475423,
        0.0868665,
        0.0156546,
        0.00432533,
        0.208471,
        0.00608996,
        0.0766373,
        0.101167,
        0.0577657,
        0.0302051,
        0.00605852,
        0.00260019,
        0.0387897
      ],
      [ 0.07768,
        0.0140093,
        0.00806517,
        0.0112622,
        0.0173585,
        0.0136868,
        0.013373,
        0.123964,
        0.0129145,
        0.0724671,
        0.0420745,
        0.0403217,
        0.0117084,
        0.0179996,
        0.0142519,
        0.0922724,
        0.23169,
        0.167914,
        0.00281536,
        0.0141727
      ],
      [ 0.118366,
        0.00356248,
        0.0495595,
        0.153759,
        0.00686657,
        0.0194993,
        0.0229373,
        0.0302661,
        0.0982304,
        0.0571251,
        0.0171727,
        0.0259525,
        0.0175153,
        0.120492,
        0.0805193,
        0.0486588,
        0.0635796,
        0.0553336,
        0.00230083,
        0.00830274
      ],
      [ 0.052856,
        0.000862588,
        0.209276,
        0.402419,
        0.00126507,
        0.0155838,
        0.0124149,
        0.00548648,
        0.0497017,
        0.00902565,
        0.00423571,
        0.0264744,
        0.00631856,
        0.121241,
        0.0193569,
        0.0197263,
        0.0235464,
        0.0159764,
        0.000838161,
        0.00339487
      ],
      [ 0.0344366,
        0.0020986,
        0.113901,
        0.054178,
        0.00315576,
        0.336164,
        0.0461777,
        0.000346342,
        0.0667553,
        0.00483557,
        0.00197045,
        0.163672,
        0.00403698,
        0.0605414,
        0.0426222,
        0.0481173,
        0.00891481,
        0.00201106,
        0.00065101,
        0.00541456
      ],
      [ 0.115309,
        0.00149624,
        0.175552,
        0.174941,
        0.00157002,
        0.0394181,
        0.0132402,
        0.0056913,
        0.0433118,
        0.010141,
        0.00303321,
        0.0458477,
        0.16658,
        0.0366731,
        0.0151279,
        0.0871536,
        0.0468261,
        0.0141932,
        0.00075157,
        0.00314327
      ],
      [ 0.386515,
        0.0218928,
        0.00223668,
        0.00313878,
        0.00392814,
        0.369435,
        0.00146729,
        0.00853761,
        0.00188405,
        0.0127257,
        0.00805817,
        0.00304205,
        0.0158688,
        0.00214647,
        0.00375793,
        0.0808877,
        0.0305196,
        0.0410635,
        0.000992288,
        0.00190203
      ],
      [ 0.0146571,
        0.00243179,
        0.000521057,
        0.00148744,
        0.00916375,
        0.00209533,
        0.00101819,
        0.19139,
        0.00228984,
        0.44328,
        0.221743,
        0.00129983,
        0.000768515,
        0.00493628,
        0.00288413,
        0.00272515,
        0.0170997,
        0.0778557,
        0.000877938,
        0.0014756
      ]
    ]

-- | Weights of C60 model.
c60Weights :: [Double]
c60Weights =
  [ 0.0169699,
    0.0211683,
    0.0276589,
    0.0065676,
    0.0141221,
    0.00687748,
    0.014691,
    0.00672258,
    0.00183967,
    0.0102547,
    0.0230896,
    0.0057941,
    0.0125395,
    0.0204526,
    0.00706296,
    0.0117983,
    0.00683347,
    0.0433776,
    0.0318279,
    0.0222546,
    0.0102265,
    0.0150546,
    0.013416,
    0.0148552,
    0.0239112,
    0.0128776,
    0.0222319,
    0.0247445,
    0.0214275,
    0.0115002,
    0.00760174,
    0.0130259,
    0.0093702,
    0.0467194,
    0.044194,
    0.0322263,
    0.0403,
    0.0150234,
    0.010459,
    0.0214742,
    0.0154958,
    0.010179,
    0.022798,
    0.0123205,
    0.00667776,
    0.000415008,
    0.0344385,
    0.0113663,
    0.0127143,
    0.0124324,
    0.0262124,
    0.0064995,
    0.0103203,
    0.0142464,
    0.02156,
    0.0199151,
    0.00389642,
    0.0113449,
    0.0128596,
    0.0117657
  ]
