-- |
--
-- Module      :  From
-- Copyright   :  (C) 2024-2025 XT et al.
-- License     :  Apache-2.0 (see the LICENSE file)
-- Portability :  FlexibleInstances, MultiParamTypeClasses
--
-- If you are an experienced Haskeller, you'd immediately guess what this is
-- about. Your guess is correct. There are 8 types covered by this module:
--
-- * 'String' (legacy)
-- * Sequence of bytes
--
--      * Strict
--      * Lazy
--      * Builder
--      * Unpinned
--
-- * Sequence of Unicode characters
--
--      * Strict
--      * Lazy
--      * Builder
--
-- Check the [implementation](#i:From), and go enjoy the convenience of 'from'.
--
-- But if you are not familiar with the history of how Haskell dealt with the
-- concept of /string/ (whatever it means), here's a bit of introduction:
--
-- == Introduction
--
-- Haskell has 'String', but Haskellers have been not entirely happy with it.
-- You probably shouldn't rely on it either. It's a legacy type. Please kindly
-- understand the situation; Haskell is older than Unicode. Haskell is older
-- than the WWW itself.
--
-- There are alternatives in the modern Haskell ecosystem. The two major
-- relevant packages are:
--
-- 1. "bytestring" for "sequences of bytes" where a "byte" is any value from
-- @0x00@ to @0xFF@.
--
-- 2. "text" for sequences of Unicode characters.
--
-- Each package offers "strict" and "lazy" variants. If you have no idea what
-- this means, you can just start using the strict ones
-- ('Data.ByteString.ByteString' and 'Data.Text.Text'). They are just like the
-- immutable contiguous byte/character sequence types commonly found in other
-- programming languages.
--
-- There are also "builder" types for efficient construction (eventual
-- concatenation or output) of sequences: 'Data.ByteString.Builder.Builder'
-- (bytestring) and 'Data.Text.Lazy.Builder.Builder' (text).
--
-- 'Data.ByteString.ByteString' is /pinned/ in the memory. This means the GC
-- cannot move it around. For large data, this is the desired behavior, but for
-- small sequences of bytes, this can lead to memory fragmentation. For that
-- reason, there is an /unpinned/ variant called
-- 'Data.ByteString.Short.ShortByteString'. If you still don't understand, just
-- stick to 'Data.ByteString.ByteString' for now.
--
-- So that's 8 distinct types to deal with. It is difficult to remember the
-- conversion functions. This module aims to provide the "practical defaults"
-- for that, so that you can simply use 'from' in most cases.
--
-- Each instance's documentation in the [instances](#i:From) section reveals
-- the underlying implementation. If you need a different assumption (e.g.
-- conversion not in UTF-8), you should express the accurate conversion in your
-- code, instead of 'from'.
module From.String (
  -- ** Instances #instances#
  From (..),
) where

import From (From (..))
import From.String.AutoGen ()
