# `GhcCompat` plugin

[![Hackage Version](https://img.shields.io/hackage/v/ghc-compat-plugin)](https://hackage.haskell.org/package/ghc-compat-plugin)
[![Packaging status](https://repology.org/badge/tiny-repos/haskell:ghc-compat-plugin.svg)](https://repology.org/project/haskell:ghc-compat-plugin/versions)
[![latest packaged versions](https://repology.org/badge/latest-versions/haskell:ghc-compat-plugin.svg)](https://repology.org/project/haskell:ghc-compat-plugin/versions)

Eases support for multiple GHC versions

Controls various GHC options and extensions to make compilation across multiple versions easier, and to alert you to incompatibilities.

Often GHC will add warnings to encourage users to make use of newer language features. However, if you’re like me, you use `-Weverything` and also support a large number of GHC versions. This can lead to a bunch of fragile conditionalization as you need to disable certain warnings _but not before those warnings were added to GHC_.

This plugin does three things:

1. reports if you use any [language extensions]() that aren’t supported by your minimum GHC version,
2. automatically disables any [warnings]() where addressing them requires features that aren’t part of your minimum GHC version, and
3. reports any warnings from no. 2 that were introduced before GHC 8.10.1[^1].

[^1]: This is because before GHC 8.10.1, plugins couldn’t modify warning flags, so we report them to allow them to be manually disabled.

So, currently, if you enable this plugin,

| oldest GHC < | then this plugin will allow      | from GHC | auto unset |
| ------------ | -------------------------------- | -------- | ---------- |
| 6.8.1        | [`missing-kind-signatures`]      | 9.2.1    | ✔         |
| 7.4.1        | [`missing-poly-kind-signatures`] | 9.8.1    | ✔         |
| 7.8.1        | [`deriving-typeable`]            | 7.10.1   |            |
| 〃           | [`missing-role-annotations`]     | 9.8.1    | ✔         |
| 8.2.1        | [`missing-deriving-strategies`]  | 8.8.1    |            |
| 8.6.1        | [`star-is-type`]                 | 8.6.1    |            |
| 8.10.1       | [`prepositive-qualified-module`] | 8.10.1   | ✔         |
| 9.14.1       | [`pattern-namespace-specifier`]  | 9.14.1   | ✔         |

Note that there are often versions between the oldest one you support and the one where the plugin disables the warning. This is because the flag warning about a feature can be added long after the feature (for example, [role annotations](https://ghc.gitlab.haskell.org/ghc/doc/users_guide/exts/roles.html#role-annotations) were added in GHC 7.8.1, but the warning to use them wasn’t added until GHC 9.8.1).

## usage

Add the following to any stanzas[^2] in your Cabal package files.

[^2]: I like to put it in a `common` section (provided you’re using at least Cabal 2.2) that’s imported by all my stanzas.

```cabal
  build-depends: ghc-compat-plugin >=0.0.2 && <0.1
  ghc-options:
    -fplugin=GhcCompat
    -fplugin-opt=GhcCompat:minVersion=8.8.1
```

Note that in the last line, you must provide the oldest GHC version you support to the plugin, so it knows which flags to disable.

You can also add

```cabal
    -fplugin-opt=GhcCompat:reportIncompatibleExtensions=error
```

(where `error` can also be `warn` (the default) or `no`) in order to control how the plugin informs you of enabled extensions that aren’t compatible with all of your supported GHC versions.

**NB**: This plugin should load from GHC 7.2 (the first version of GHC to support plugins[^3]). However, it currently won’t _do_ anything before GHC 7.10.1. If you do need to support a GHC prior to 7.2, you can use the plugin conditionally, like

```cabal
  if impl(ghc >= 7.2.1)
    build-depends: ghc-compat-plugin >=0.0.2 && <0.1
    ghc-options:
      -fplugin=GhcCompat
      -fplugin-opt=GhcCompat:minVersion=6.8.1
```

[^3]: What good would a compatibility plugin be if it wasn’t extremely compatible?

<!-- links to warning flags controlled by the plugin -->

[`deriving-typeable`]: https://ghc.gitlab.haskell.org/ghc/doc/users_guide/using-warnings.html#ghc-flag-Wderiving-typeable
[`missing-deriving-strategies`]: https://ghc.gitlab.haskell.org/ghc/doc/users_guide/using-warnings.html#ghc-flag-Wmissing-deriving-strategies
[`missing-kind-signatures`]: https://ghc.gitlab.haskell.org/ghc/doc/users_guide/using-warnings.html#ghc-flag-Wmissing-kind-signatures
[`missing-poly-kind-signatures`]: https://ghc.gitlab.haskell.org/ghc/doc/users_guide/using-warnings.html#ghc-flag-Wmissing-poly-kind-signatures
[`missing-role-annotations`]: https://ghc.gitlab.haskell.org/ghc/doc/users_guide/using-warnings.html#ghc-flag-Wmissing-role-annotations
[`pattern-namespace-specifier`]: https://ghc.gitlab.haskell.org/ghc/doc/users_guide/using-warnings.html#ghc-flag-Wpattern-namespace-specifier
[`prepositive-qualified-module`]: https://ghc.gitlab.haskell.org/ghc/doc/users_guide/using-warnings.html#ghc-flag-Wprepositive-qualified-module
[`star-is-type`]: https://ghc.gitlab.haskell.org/ghc/doc/users_guide/using-warnings.html#ghc-flag-Wstar-is-type

## versioning

This project largely follows the [Haskell Package Versioning Policy](https://pvp.haskell.org/) (PVP), but is more strict in some ways.

The version always has four components, `A.B.C.D`. The first three correspond to those required by PVP, while the fourth matches the “patch” component from [Semantic Versioning](https://semver.org/).

Here is a breakdown of some of the constraints:

### sensitivity to additions to the API

PVP recommends that clients follow [these import guidelines](https://wiki.haskell.org/Import_modules_properly) in order that they may be considered insensitive to additions to the API. However, this isn’t sufficient. We expect clients to follow these additional recommendations for API insensitivity

If you don’t follow these recommendations (in addition to the ones made by PVP), you should ensure your dependencies don’t allow a range of `C` values. That is, your dependencies should look like

```cabal
yaya >=1.2.3 && <1.2.4
```

rather than

```cabal
yaya >=1.2.3 && <1.3
```

#### use package-qualified imports everywhere

If your imports are [package-qualified](https://downloads.haskell.org/ghc/latest/docs/users_guide/exts/package_qualified_imports.html?highlight=packageimports#extension-PackageImports), then a dependency adding new modules can’t cause a conflict with modules you already import.

#### avoid orphans

Because of the transitivity of instances, orphans make you sensitive to your dependencies’ instances. If you have an orphan instance, you are sensitive to the APIs of the packages that define the class and the types of the instance.

One way to minimize this sensitivity is to have a separate package (or packages) dedicated to any orphans you have. Those packages can be sensitive to their dependencies’ APIs, while the primary package remains insensitive, relying on the tighter ranges of the orphan packages to constrain the solver.

### transitively breaking changes (increments `A`)

#### removing a type class instance

Type class instances are imported transitively, and thus changing them can impact packages that only have your package as a transitive dependency.

#### widening a dependency range with new major versions

This is a consequence of instances being transitively imported. A new major version of a dependency can remove instances, and that can break downstream clients that unwittingly depended on those instances.

A library _may_ declare that it always bumps the `A` component when it removes an instance (as this policy dictates). In that case, only `A` widenings need to induce `A` bumps. `B` widenings can be `D` bumps like other widenings, Alternatively, one may compare the APIs when widening a dependency range, and if no instances have been removed, make it a `D` bump.

### breaking changes (increments `B`)

#### restricting an existing dependency’s version range in any way

Consumers have to contend not only with our version bounds, but also with those of other libraries. It’s possible that some dependency overlapped in a very narrow way, and even just restricting a particular patch version of a dependency could make it impossible to find a dependency solution.

#### restricting the license in any way

Making a license more restrictive may prevent clients from being able to continue using the package.

#### adding a dependency

A new dependency may make it impossible to find a solution in the face of other packages dependency ranges.

### non-breaking changes (increments `C`)

#### adding a module

This is also what PVP recommends. However, unlike in PVP, this is because we recommend that package-qualified imports be used on all imports.

### other changes (increments `D`)

#### widening a dependency range for non-major versions

This is fairly uncommon, in the face of `^>=`-style ranges, but it can happen in a few situations.

#### deprecation

**NB**: This case is _weaker_ than PVP, which indicates that packages should bump their major version when adding `deprecation` pragmas.

We disagree with this because packages shouldn’t be _publishing_ with `-Werror`. The intent of deprecation is to indicate that some API _will_ change. To make that signal a major change itself defeats the purpose. You want people to start seeing that warning as soon as possible. The major change occurs when you actually remove the old API.

Yes, in development, `-Werror` is often (and should be) used. However, that just helps developers be aware of deprecations more immediately. They can always add `-Wwarn=deprecation` in some scope if they need to avoid updating it for the time being.

## licensing

This package is licensed under [The GNU AGPL 3.0 only](./LICENSE). If you need a license for usage that isn’t covered under the AGPL, please contact [Greg Pfeil](mailto:greg@technomadic.org?subject=licensing%20ghc-compat-plugin).

You should review the [license report](docs/license-report.md) for details about dependency licenses.

## comparisons

This was inspired by compatibility flags in other language implementations, like `-Wc++11-compat` in [GCC](https://gcc.gnu.org/onlinedocs/gcc-15.2.0/gcc/Warning-Options.html#index-Wc_002b_002b11-compat) and [Clang](https://clang.llvm.org/docs/DiagnosticsReference.html#wc-11-compat).
