{-# LANGUAGE DeriveGeneric         #-}
{-# LANGUAGE DuplicateRecordFields #-}
-- |
-- Module: Network.Greskell.WebSocket.Request
-- Description: Request to Gremlin Server
-- Maintainer: Toshio Ito <debug.ito@gmail.com>
--
--
module Network.Greskell.WebSocket.Request
    ( -- * RequestMessage
      RequestMessage (..)
    , Operation (..)
    , toRequestMessage
    , makeRequestMessage
    ) where

import           Control.Applicative                       ((<$>), (<*>))
import           Data.Aeson                                (FromJSON (..), Object, ToJSON (..))
import           Data.Text                                 (Text)
import           Data.UUID                                 (UUID)
import           Data.UUID.V4                              (nextRandom)
import           GHC.Generics                              (Generic)

import qualified Network.Greskell.WebSocket.Request.Aeson  as GAeson
import           Network.Greskell.WebSocket.Request.Common (Operation (..))


-- | RequestMessage to a Gremlin Server. See
-- <http://tinkerpop.apache.org/docs/current/dev/provider/>.
data RequestMessage
  = RequestMessage
      { requestId :: !UUID
      , op        :: !Text
      , processor :: !Text
      , args      :: !Object
      }
  deriving (Eq, Generic, Show)

instance ToJSON RequestMessage where
  toJSON = GAeson.genericToJSON GAeson.opt
  toEncoding = GAeson.genericToEncoding GAeson.opt

instance FromJSON RequestMessage where
  parseJSON = GAeson.genericParseJSON GAeson.opt

-- | Convert an 'Operation' object to 'RequestMessage'.
toRequestMessage :: Operation o => UUID -> o -> RequestMessage
toRequestMessage rid o =
  RequestMessage { requestId = rid,
                   op = opName o,
                   processor = opProcessor o,
                   args = opArgs o
                 }

-- | Create a 'RequestMessage' from an 'Operation' object. The
-- 'requestId' is generated by the random number generator of the
-- system.
makeRequestMessage :: Operation o => o -> IO RequestMessage
makeRequestMessage o = toRequestMessage <$> nextRandom <*> pure o
