/**********************************************************************************************
*
*   rl_gputex v1.1 - GPU compressed textures loading and saving
*
*   DESCRIPTION:
*
*     Load GPU compressed image data from image files provided as memory data arrays,
*     data is loaded compressed, ready to be loaded into GPU
*
*     Note that some file formats (DDS, PVR, KTX) also support uncompressed data storage.
*     In those cases data is loaded uncompressed and format is returned
*
*   FIXME: This library still depends on Raylib due to the following reasons:
*     - rl_save_ktx_to_memory() requires rlGetGlTextureFormats() from rlgl.h
*         though this is not a problem, if you don't need KTX support
*
*   TODO:
*     - Review rl_load_ktx_from_memory() to support KTX v2.2 specs
*
*   CONFIGURATION:
*
*   #define RL_GPUTEX_SUPPORT_DDS
*   #define RL_GPUTEX_SUPPORT_PKM
*   #define RL_GPUTEX_SUPPORT_KTX
*   #define RL_GPUTEX_SUPPORT_PVR
*   #define RL_GPUTEX_SUPPORT_ASTC
*       Define desired file formats to be supported
*
*   #define RL_GPUTEX_SHOW_LOG_INFO
*       Define, if you wish to see warnings generated by the library
*       This will include <stdio.h> unless you provide your own RL_GPUTEX_LOG
*   #define RL_GPUTEX_LOG
*       Define, if you wish to provide your own warning function
*       Make sure that this macro puts newline character '\n' at the end
*
*   #define RL_GPUTEX_MALLOC
*   #define RL_GPUTEX_FREE
*       Define those macros in order to provide your own libc-compliant allocator;
*       not doing so will include <stdlib.h>
*       If you're providing any of those, you must provide ALL of them,
*       otherwise the code will (most likely) crash...
*
*   #define RL_GPUTEX_NULL
*       Define in order to provide your own libc-compliant NULL pointer;
*       not doing so will include <stdlib.h>
*
*   #define RL_GPUTEX_MEMCPY
*       Define in order to provide your own libc-compliant 'memcpy' function;
*       not doing so will include <string.h>
*
*   #define RLGPUTEXAPI
*       Define to compiler-specific intrinsic, if you wish to export public functions
*       There is no need to do so when statically linking
*
*   VERSIONS HISTORY:
*       1.1 (15-Jul-2025) Several minor fixes related to specific image formats; some work has been done
*           in order to decouple the library from Raylib by introducing few new macros; library still
*           requires Raylib in order to function properly
*
*       1.0 (17-Sep-2022) First version has been created by migrating part of compressed-texture-loading
*           functionality from Raylib src/rtextures.c into self-contained library
*
*   LICENSE: zlib/libpng
*
*   Copyright (c) 2013-2026 Ramon Santamaria (@raysan5)
*
*   This software is provided "as-is", without any express or implied warranty. In no event
*   will the authors be held liable for any damages arising from the use of this software.
*
*   Permission is granted to anyone to use this software for any purpose, including commercial
*   applications, and to alter it and redistribute it freely, subject to the following restrictions:
*
*     1. The origin of this software must not be misrepresented; you must not claim that you
*     wrote the original software. If you use this software in a product, an acknowledgment
*     in the product documentation would be appreciated but is not required.
*
*     2. Altered source versions must be plainly marked as such, and must not be misrepresented
*     as being the original software.
*
*     3. This notice may not be removed or altered from any source distribution.
*
**********************************************************************************************/

#ifndef RL_GPUTEX_H
#define RL_GPUTEX_H

#ifndef RLGPUTEXAPI
    #define RLGPUTEXAPI  // Functions defined as 'extern' by default (implicit specifiers)
#endif

// Texture pixel formats
// NOTE: Support depends on OpenGL version
// WARNING: Enum values aligned with raylib/rlgl equivalent PixelFormat/rlPixelFormat enum,
// to avoid value mapping between the 3 libraries format values
typedef enum {
    RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_GRAYSCALE = 1,     // 8 bit per pixel (no alpha)
    RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_GRAY_ALPHA,        // 8*2 bpp (2 channels)
    RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R5G6B5,            // 16 bpp
    RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R8G8B8,            // 24 bpp
    RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R5G5B5A1,          // 16 bpp (1 bit alpha)
    RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R4G4B4A4,          // 16 bpp (4 bit alpha)
    RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R8G8B8A8,          // 32 bpp
    RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R32,               // 32 bpp (1 channel - float)
    RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R32G32B32,         // 32*3 bpp (3 channels - float)
    RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R32G32B32A32,      // 32*4 bpp (4 channels - float)
    RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R16,               // 16 bpp (1 channel - half float)
    RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R16G16B16,         // 16*3 bpp (3 channels - half float)
    RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R16G16B16A16,      // 16*4 bpp (4 channels - half float)
    RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT1_RGB,            // 4 bpp (no alpha)
    RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT1_RGBA,           // 4 bpp (1 bit alpha)
    RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT3_RGBA,           // 8 bpp
    RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT5_RGBA,           // 8 bpp
    RL_GPUTEX_PIXELFORMAT_COMPRESSED_ETC1_RGB,            // 4 bpp
    RL_GPUTEX_PIXELFORMAT_COMPRESSED_ETC2_RGB,            // 4 bpp
    RL_GPUTEX_PIXELFORMAT_COMPRESSED_ETC2_EAC_RGBA,       // 8 bpp
    RL_GPUTEX_PIXELFORMAT_COMPRESSED_PVRT_RGB,            // 4 bpp
    RL_GPUTEX_PIXELFORMAT_COMPRESSED_PVRT_RGBA,           // 4 bpp
    RL_GPUTEX_PIXELFORMAT_COMPRESSED_ASTC_4x4_RGBA,       // 8 bpp
    RL_GPUTEX_PIXELFORMAT_COMPRESSED_ASTC_8x8_RGBA        // 2 bpp
} rlGpuTexPixelFormat;

//----------------------------------------------------------------------------------
// Module Functions Declaration
//----------------------------------------------------------------------------------
#if defined(__cplusplus)
extern "C" {            // Prevents name mangling of functions
#endif

// Load image data from memory data files
RLGPUTEXAPI void *rl_load_dds_from_memory(const unsigned char *file_data, unsigned int file_size, int *width, int *height, int *format, int *mips);
RLGPUTEXAPI void *rl_load_pkm_from_memory(const unsigned char *file_data, unsigned int file_size, int *width, int *height, int *format, int *mips);
RLGPUTEXAPI void *rl_load_ktx_from_memory(const unsigned char *file_data, unsigned int file_size, int *width, int *height, int *format, int *mips);
RLGPUTEXAPI void *rl_load_pvr_from_memory(const unsigned char *file_data, unsigned int file_size, int *width, int *height, int *format, int *mips);
RLGPUTEXAPI void *rl_load_astc_from_memory(const unsigned char *file_data, unsigned int file_size, int *width, int *height, int *format, int *mips);

RLGPUTEXAPI int rl_save_ktx_to_memory(const char *fileName, void *data, int width, int height, int format, int mipmaps);  // Save image data as KTX file

#if defined(__cplusplus)
}
#endif

#endif // RL_GPUTEX_H

/***********************************************************************************
*
*   RL_GPUTEX IMPLEMENTATION
*
************************************************************************************/

#if defined(RL_GPUTEX_IMPLEMENTATION)

#if defined(RL_GPUTEX_SHOW_LOG_INFO) && !defined(RL_GPUTEX_LOG)
    #include <stdio.h>          // Required for: printf()
#endif
#if !defined(RL_GPUTEX_MALLOC) || !defined(RL_GPUTEX_NULL)
    #include <stdlib.h>         // Required for: NULL, malloc(), calloc(), free()
#endif
#if !defined(RL_GPUTEX_MEMCPY)
    #include <string.h>         // Required for: memcpy()
#endif

#if defined(RL_GPUTEX_MALLOC) || defined(RL_GPUTEX_FREE)
    #if !defined(RL_GPUTEX_MALLOC) || !defined(RL_GPUTEX_FREE)
        #warning "RL_GPUTEX_MALLOC and RL_GPUTEX_FREE allocation functions are required to be provided"
    #endif
#endif
#if !defined(RL_GPUTEX_MALLOC)
    #define RL_GPUTEX_MALLOC(sz)        malloc(sz)
    #define RL_GPUTEX_FREE(ptr)         free(ptr)
#endif

#if !defined(RL_GPUTEX_NULL)
    #define RL_GPUTEX_NULL NULL
#endif
#if !defined(RL_GPUTEX_MEMCPY)
    #define RL_GPUTEX_MEMCPY(dest, src, num)  memcpy(dest, src, num)
#endif

// Simple warning logging system to avoid LOG() calls if required
// NOTE: Avoiding those calls, also avoids const strings memory usage
// WARN: This macro expects that newline character is added automatically
//       in order to match the functionality of Raylib's TRACELOG()
#if defined(RL_GPUTEX_SHOW_LOG_INFO)
    #if !defined(RL_GPUTEX_LOG)
        #define RL_GPUTEX_LOG(...) (void)(printf("RL_GPUTEX: WARNING: " __VA_ARGS__), printf("\n"))
    #endif
#else
    #if defined(RL_GPUTEX_LOG)
        // Undefine it first in order to supress warnings about macro redefinition
        #undef RL_GPUTEX_LOG
    #endif
    #define RL_GPUTEX_LOG(...)
#endif

//----------------------------------------------------------------------------------
// Module Internal Functions Declaration
//----------------------------------------------------------------------------------
// Get pixel data size in bytes for certain pixel format
static int get_pixel_data_size(int width, int height, int format);

// Get OpenGL internal formats and data type from rlGpuTexPixelFormat
void get_gl_texture_formats(int format, unsigned int *gl_internal_format, unsigned int *gl_format, unsigned int *gl_type);

//----------------------------------------------------------------------------------
// Module Functions Definition
//----------------------------------------------------------------------------------
#if defined(RL_GPUTEX_SUPPORT_DDS)
// Loading DDS from memory image data (compressed or uncompressed)
void *rl_load_dds_from_memory(const unsigned char *file_data, unsigned int file_size, int *width, int *height, int *format, int *mips)
{
    void *image_data = RL_GPUTEX_NULL;  // Image data pointer
    int image_pixel_size = 0;           // Image pixel size

    unsigned char *file_data_ptr = (unsigned char *)file_data;

    // Required extension:
    // GL_EXT_texture_compression_s3tc

    // Supported tokens (defined by extensions)
    // GL_COMPRESSED_RGB_S3TC_DXT1_EXT      0x83F0
    // GL_COMPRESSED_RGBA_S3TC_DXT1_EXT     0x83F1
    // GL_COMPRESSED_RGBA_S3TC_DXT3_EXT     0x83F2
    // GL_COMPRESSED_RGBA_S3TC_DXT5_EXT     0x83F3

    #define FOURCC_DXT1 0x31545844  // Equivalent to "DXT1" in ASCII
    #define FOURCC_DXT3 0x33545844  // Equivalent to "DXT3" in ASCII
    #define FOURCC_DXT5 0x35545844  // Equivalent to "DXT5" in ASCII

    // DDS Pixel Format
    typedef struct {
        unsigned int size;
        unsigned int flags;
        unsigned int fourcc;
        unsigned int rgb_bit_count;
        unsigned int r_bit_mask;
        unsigned int g_bit_mask;
        unsigned int b_bit_mask;
        unsigned int a_bit_mask;
    } dds_pixel_format;

    // DDS Header (124 bytes)
    typedef struct {
        unsigned int size;
        unsigned int flags;
        unsigned int height;
        unsigned int width;
        unsigned int pitch_or_linear_size;
        unsigned int depth;
        unsigned int mipmap_count;
        unsigned int reserved1[11];
        dds_pixel_format ddspf;
        unsigned int caps;
        unsigned int caps2;
        unsigned int caps3;
        unsigned int caps4;
        unsigned int reserved2;
    } dds_header;

    if (file_data_ptr != RL_GPUTEX_NULL)
    {
        // Verify the type of file
        unsigned char *dds_header_id = file_data_ptr;
        file_data_ptr += 4;

        if ((dds_header_id[0] != 'D') || (dds_header_id[1] != 'D') || (dds_header_id[2] != 'S') || (dds_header_id[3] != ' '))
        {
            RL_GPUTEX_LOG("DDS file data not valid");
        }
        else
        {
            dds_header *header = (dds_header *)file_data_ptr;

            file_data_ptr += sizeof(dds_header);        // Skip header

            *width = header->width;
            *height = header->height;

            if (*width % 4 != 0) RL_GPUTEX_LOG("DDS file width must be multiple of 4. Image will not display correctly");
            if (*height % 4 != 0) RL_GPUTEX_LOG("DDS file height must be multiple of 4. Image will not display correctly");

            image_pixel_size = header->width*header->height;

            if (header->mipmap_count == 0) *mips = 1;   // Parameter not used
            else *mips = header->mipmap_count;

            if (header->ddspf.rgb_bit_count == 16)      // 16bit mode, no compressed
            {
                if (header->ddspf.flags == 0x40)        // No alpha channel
                {
                    int data_size = image_pixel_size*sizeof(unsigned short);
                    if (header->mipmap_count > 1) data_size = data_size + data_size/3;
                    image_data = RL_GPUTEX_MALLOC(data_size);

                    RL_GPUTEX_MEMCPY(image_data, file_data_ptr, data_size);

                    *format = RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R5G6B5;
                }
                else if (header->ddspf.flags == 0x41)           // With alpha channel
                {
                    if (header->ddspf.a_bit_mask == 0x8000)     // 1bit alpha
                    {
                        int data_size = image_pixel_size*sizeof(unsigned short);
                        if (header->mipmap_count > 1) data_size = data_size + data_size/3;
                        image_data = RL_GPUTEX_MALLOC(data_size);

                        RL_GPUTEX_MEMCPY(image_data, file_data_ptr, data_size);

                        unsigned char alpha = 0;

                        // NOTE: Data comes as A1R5G5B5, it must be reordered to R5G5B5A1
                        for (int i = 0; i < data_size/sizeof(unsigned short); i++)
                        {
                            alpha = ((unsigned short *)image_data)[i] >> 15;
                            ((unsigned short *)image_data)[i] = ((unsigned short *)image_data)[i] << 1;
                            ((unsigned short *)image_data)[i] += alpha;
                        }

                        *format = RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R5G5B5A1;
                    }
                    else if (header->ddspf.a_bit_mask == 0xf000)   // 4bit alpha
                    {
                        int data_size = image_pixel_size*sizeof(unsigned short);
                        if (header->mipmap_count > 1) data_size = data_size + data_size/3;
                        image_data = RL_GPUTEX_MALLOC(data_size);

                        RL_GPUTEX_MEMCPY(image_data, file_data_ptr, data_size);

                        unsigned char alpha = 0;

                        // NOTE: Data comes as A4R4G4B4, it must be reordered R4G4B4A4
                        for (int i = 0; i < data_size/sizeof(unsigned short); i++)
                        {
                            alpha = ((unsigned short *)image_data)[i] >> 12;
                            ((unsigned short *)image_data)[i] = ((unsigned short *)image_data)[i] << 4;
                            ((unsigned short *)image_data)[i] += alpha;
                        }

                        *format = RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R4G4B4A4;
                    }
                }
            }
            else if ((header->ddspf.flags == 0x40) && (header->ddspf.rgb_bit_count == 24)) // DDS_RGB, no compressed
            {
                int data_size = image_pixel_size*3*sizeof(unsigned char);
                if (header->mipmap_count > 1) data_size = data_size + data_size/3;
                image_data = RL_GPUTEX_MALLOC(data_size);

                RL_GPUTEX_MEMCPY(image_data, file_data_ptr, data_size);

                *format = RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R8G8B8;
            }
            else if ((header->ddspf.flags == 0x41) && (header->ddspf.rgb_bit_count == 32)) // DDS_RGBA, no compressed
            {
                int data_size = image_pixel_size*4*sizeof(unsigned char);
                if (header->mipmap_count > 1) data_size = data_size + data_size/3;
                image_data = RL_GPUTEX_MALLOC(data_size);

                RL_GPUTEX_MEMCPY(image_data, file_data_ptr, data_size);

                unsigned char blue = 0;

                // NOTE: Data comes as A8R8G8B8, it must be reordered R8G8B8A8 (view next comment)
                // DirecX understand ARGB as a 32bit DWORD but the actual memory byte alignment is BGRA
                // So, we must realign B8G8R8A8 to R8G8B8A8
                for (int i = 0; i < data_size; i += 4)
                {
                    blue = ((unsigned char *)image_data)[i];
                    ((unsigned char *)image_data)[i] = ((unsigned char *)image_data)[i + 2];
                    ((unsigned char *)image_data)[i + 2] = blue;
                }

                *format = RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R8G8B8A8;
            }
            else if (((header->ddspf.flags == 0x04) || (header->ddspf.flags == 0x05)) && (header->ddspf.fourcc > 0)) // Compressed
            {
                int data_size = 0;

                // Calculate data size, including all mipmaps
                if (header->mipmap_count > 1) data_size = header->pitch_or_linear_size + header->pitch_or_linear_size/3;
                else data_size = header->pitch_or_linear_size;

                image_data = RL_GPUTEX_MALLOC(data_size*sizeof(unsigned char));

                RL_GPUTEX_MEMCPY(image_data, file_data_ptr, data_size);

                switch (header->ddspf.fourcc)
                {
                    case FOURCC_DXT1:
                    {
                        if (header->ddspf.flags == 0x04) *format = RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT1_RGB;
                        else *format = RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT1_RGBA;
                    } break;
                    case FOURCC_DXT3: *format = RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT3_RGBA; break;
                    case FOURCC_DXT5: *format = RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT5_RGBA; break;
                    default: break;
                }
            }
        }
    }

    return image_data;
}
#endif

#if defined(RL_GPUTEX_SUPPORT_PKM)
// Loading PKM image data (ETC1/ETC2 compression)
// NOTE: KTX is the standard Khronos Group compression format (ETC1/ETC2, mipmaps)
// PKM is a much simpler file format used mainly to contain a single ETC1/ETC2 compressed image (no mipmaps)
void *rl_load_pkm_from_memory(const unsigned char *file_data, unsigned int file_size, int *width, int *height, int *format, int *mips)
{
    void *image_data = RL_GPUTEX_NULL;        // Image data pointer

    unsigned char *file_data_ptr = (unsigned char *)file_data;

    // Required extensions:
    // GL_OES_compressed_ETC1_RGB8_texture  (ETC1) (OpenGL ES 2.0)
    // GL_ARB_ES3_compatibility  (ETC2/EAC) (OpenGL ES 3.0)

    // Supported tokens (defined by extensions)
    // GL_ETC1_RGB8_OES                 0x8D64
    // GL_COMPRESSED_RGB8_ETC2          0x9274
    // GL_COMPRESSED_RGBA8_ETC2_EAC     0x9278

    // PKM file (ETC1) Header (16 bytes)
    typedef struct {
        char id[4];                 // "PKM "
        char version[2];            // "10" or "20"
        unsigned short format;      // Data format (big-endian) (Check list below)
        unsigned short width;       // Texture width (big-endian) (orig_width rounded to multiple of 4)
        unsigned short height;      // Texture height (big-endian) (orig_height rounded to multiple of 4)
        unsigned short orig_width;   // Original width (big-endian)
        unsigned short orig_height;  // Original height (big-endian)
    } pkm_header;

    // Formats list
    // version 10: format: 0=ETC1_RGB, [1=ETC1_RGBA, 2=ETC1_RGB_MIP, 3=ETC1_RGBA_MIP] (not used)
    // version 20: format: 0=ETC1_RGB, 1=ETC2_RGB, 2=ETC2_RGBA_OLD, 3=ETC2_RGBA, 4=ETC2_RGBA1, 5=ETC2_R, 6=ETC2_RG, 7=ETC2_SIGNED_R, 8=ETC2_SIGNED_R

    // NOTE: The extended width and height are the widths rounded up to a multiple of 4
    // NOTE: ETC is always 4bit per pixel (64 bit for each 4x4 block of pixels)

    if (file_data_ptr != RL_GPUTEX_NULL)
    {
        pkm_header *header = (pkm_header *)file_data_ptr;

        if ((header->id[0] != 'P') || (header->id[1] != 'K') || (header->id[2] != 'M') || (header->id[3] != ' '))
        {
            RL_GPUTEX_LOG("PKM file data not valid");
        }
        else
        {
            file_data_ptr += sizeof(pkm_header);   // Skip header

            // NOTE: format, width and height come as big-endian, data must be swapped to little-endian
            header->format = ((header->format & 0x00FF) << 8) | ((header->format & 0xFF00) >> 8);
            header->width = ((header->width & 0x00FF) << 8) | ((header->width & 0xFF00) >> 8);
            header->height = ((header->height & 0x00FF) << 8) | ((header->height & 0xFF00) >> 8);

            *width = header->width;
            *height = header->height;
            *mips = 1;

            int bpp = 4;
            if (header->format == 3) bpp = 8;

            int data_size = (*width)*(*height)*bpp/8;  // Total data size in bytes

            image_data = RL_GPUTEX_MALLOC(data_size*sizeof(unsigned char));

            RL_GPUTEX_MEMCPY(image_data, file_data_ptr, data_size);

            if (header->format == 0) *format = RL_GPUTEX_PIXELFORMAT_COMPRESSED_ETC1_RGB;
            else if (header->format == 1) *format = RL_GPUTEX_PIXELFORMAT_COMPRESSED_ETC2_RGB;
            else if (header->format == 3) *format = RL_GPUTEX_PIXELFORMAT_COMPRESSED_ETC2_EAC_RGBA;
        }
    }

    return image_data;
}
#endif

#if defined(RL_GPUTEX_SUPPORT_KTX)
// Load KTX compressed image data (ETC1/ETC2 compression)
// TODO: Review KTX loading, many things changed!
void *rl_load_ktx_from_memory(const unsigned char *file_data, unsigned int file_size, int *width, int *height, int *format, int *mips)
{
    void *image_data = RL_GPUTEX_NULL;        // Image data pointer

    unsigned char *file_data_ptr = (unsigned char *)file_data;

    // Required extensions:
    // GL_OES_compressed_ETC1_RGB8_texture  (ETC1)
    // GL_ARB_ES3_compatibility  (ETC2/EAC)

    // Supported tokens (defined by extensions)
    // GL_ETC1_RGB8_OES                 0x8D64
    // GL_COMPRESSED_RGB8_ETC2          0x9274
    // GL_COMPRESSED_RGBA8_ETC2_EAC     0x9278

    // KTX file Header (64 bytes)
    // v1.1 - https://www.khronos.org/opengles/sdk/tools/KTX/file_format_spec/
    // v2.0 - http://github.khronos.org/KTX-Specification/

    // KTX 1.1 Header
    // TODO: Support KTX 2.2 specs!
    typedef struct {
        char id[12];                            // Identifier: "«KTX 11»\r\n\x1A\n"
        unsigned int endianness;                // Little endian: 0x01 0x02 0x03 0x04
        unsigned int gl_type;                   // For compressed textures, glType must equal 0
        unsigned int gl_type_size;              // For compressed texture data, usually 1
        unsigned int gl_format;                 // For compressed textures is 0
        unsigned int gl_internal_format;        // Compressed internal format
        unsigned int gl_base_internal_format;   // Same as glFormat (RGB, RGBA, ALPHA...)
        unsigned int width;                     // Texture image width in pixels
        unsigned int height;                    // Texture image height in pixels
        unsigned int depth;                     // For 2D textures is 0
        unsigned int elements;                  // Number of array elements, usually 0
        unsigned int faces;                     // Cubemap faces, for no-cubemap = 1
        unsigned int mipmap_levels;             // Non-mipmapped textures = 1
        unsigned int key_value_data_size;       // Used to encode any arbitrary data...
    } ktx_header;

    // NOTE: Before start of every mipmap data block, we have: unsigned int data_size

    if (file_data_ptr != RL_GPUTEX_NULL)
    {
        ktx_header *header = (ktx_header *)file_data_ptr;

        if ((header->id[1] != 'K') || (header->id[2] != 'T') || (header->id[3] != 'X') ||
            (header->id[4] != ' ') || (header->id[5] != '1') || (header->id[6] != '1'))
        {
            RL_GPUTEX_LOG("KTX file data not valid");
        }
        else
        {
            file_data_ptr += sizeof(ktx_header);           // Move file data pointer

            *width = header->width;
            *height = header->height;
            *mips = header->mipmap_levels;

            file_data_ptr += header->key_value_data_size; // Skip value data size

            int data_size = ((int *)file_data_ptr)[0];
            file_data_ptr += sizeof(int);

            image_data = RL_GPUTEX_MALLOC(data_size*sizeof(unsigned char));

            RL_GPUTEX_MEMCPY(image_data, file_data_ptr, data_size);

            if (header->gl_internal_format == 0x8D64) *format = RL_GPUTEX_PIXELFORMAT_COMPRESSED_ETC1_RGB;
            else if (header->gl_internal_format == 0x9274) *format = RL_GPUTEX_PIXELFORMAT_COMPRESSED_ETC2_RGB;
            else if (header->gl_internal_format == 0x9278) *format = RL_GPUTEX_PIXELFORMAT_COMPRESSED_ETC2_EAC_RGBA;

            // TODO: Support uncompressed data formats? Right now it returns format = 0!
        }
    }

    return image_data;
}

// Save image data as KTX file
// NOTE: By default KTX 1.1 spec is used, 2.0 is still on draft (01Oct2018)
// TODO: Review KTX saving, many things changed!
int rl_save_ktx(const char *file_name, void *data, int width, int height, int format, int mipmaps)
{
    // KTX file Header (64 bytes)
    // v1.1 - https://www.khronos.org/opengles/sdk/tools/KTX/file_format_spec/
    // v2.0 - https://github.khronos.org/KTX-Specification/ktxspec.v2.html
    typedef struct {
        char id[12];                            // Identifier: "«KTX 11»\r\n\x1A\n"         // KTX 2.0: "«KTX 20»\r\n\x1A\n"
        unsigned int endianness;                // Little endian: 0x01 0x02 0x03 0x04
        unsigned int gl_type;                   // For compressed textures, glType must equal 0
        unsigned int gl_type_size;              // For compressed texture data, usually 1
        unsigned int gl_format;                 // For compressed textures is 0
        unsigned int gl_internal_format;        // Compressed internal format
        unsigned int gl_base_internal_format;   // Same as glFormat (RGB, RGBA, ALPHA...)   // KTX 2.0: UInt32 vkFormat
        unsigned int width;                     // Texture image width in pixels
        unsigned int height;                    // Texture image height in pixels
        unsigned int depth;                     // For 2D textures is 0
        unsigned int elements;                  // Number of array elements, usually 0
        unsigned int faces;                     // Cubemap faces, for no-cubemap = 1
        unsigned int mipmap_levels;             // Non-mipmapped textures = 1
        unsigned int key_value_data_size;       // Used to encode any arbitrary data...     // KTX 2.0: UInt32 levelOrder - ordering of the mipmap levels, usually 0
                                                                                            // KTX 2.0: UInt32 supercompressionScheme - 0 (None), 1 (Crunch CRN), 2 (Zlib DEFLATE)...
        // KTX 2.0 defines additional header elements...
    } ktx_header;

    /*
    Byte[12] identifier
    UInt32 vkFormat
    UInt32 typeSize
    UInt32 pixelWidth
    UInt32 pixelHeight
    UInt32 pixelDepth
    UInt32 layerCount
    UInt32 faceCount
    UInt32 levelCount
    UInt32 supercompressionScheme
    */

    // Calculate file data_size required
    int data_size = sizeof(ktx_header);

    for (int i = 0, w = width, h = height; i < mipmaps; i++)
    {
        data_size += get_pixel_data_size(w, h, format);
        w /= 2; h /= 2;
    }

    unsigned char *file_data = RL_GPUTEX_MALLOC(data_size);
    unsigned char *file_data_ptr = file_data;

    ktx_header header = { 0 };

    // KTX identifier (v1.1)
    //unsigned char id[12] = { '«', 'K', 'T', 'X', ' ', '1', '1', '»', '\r', '\n', '\x1A', '\n' };
    //unsigned char id[12] = { 0xAB, 0x4B, 0x54, 0x58, 0x20, 0x31, 0x31, 0xBB, 0x0D, 0x0A, 0x1A, 0x0A };

    const char ktx_identifier[12] = { 0xAB, 'K', 'T', 'X', ' ', '1', '1', 0xBB, '\r', '\n', 0x1A, '\n' };

    // Get the image header
    RL_GPUTEX_MEMCPY(header.id, ktx_identifier, 12);  // KTX 1.1 signature
    header.endianness = 0;
    header.gl_type = 0;                     // Obtained from format
    header.gl_type_size = 1;
    header.gl_format = 0;                   // Obtained from format
    header.gl_internal_format = 0;          // Obtained from format
    header.gl_base_internal_format = 0;
    header.width = width;
    header.height = height;
    header.depth = 0;
    header.elements = 0;
    header.faces = 1;
    header.mipmap_levels = mipmaps;         // If it was 0, it means mipmaps should be generated on loading (not for compressed formats)
    header.key_value_data_size = 0;         // No extra data after the header

    // TODO: WARNING: Function dependant on rlgl library!
    rlGetGlTextureFormats(format, &header.gl_internal_format, &header.gl_format, &header.gl_type); // rlgl module function

    header.gl_base_internal_format = header.gl_format; // TODO: WARNING: KTX 1.1 only

    // NOTE: We can save into a .ktx all PixelFormats supported by raylib, including compressed formats like DXT, ETC or ASTC

    if (header.gl_format == -1) RL_GPUTEX_LOG("RL_GPUTEX: GL format not supported for KTX export (%i)", header.gl_format);
    else
    {
        RL_GPUTEX_MEMCPY(file_data_ptr, &header, sizeof(ktx_header));
        file_data_ptr += sizeof(ktx_header);

        int temp_width = width;
        int temp_height = height;
        int data_offset = 0;

        // Save all mipmaps data
        for (int i = 0; i < mipmaps; i++)
        {
            unsigned int data_size = get_pixel_data_size(temp_width, temp_height, format);

            RL_GPUTEX_MEMCPY(file_data_ptr, &data_size, sizeof(unsigned int));
            RL_GPUTEX_MEMCPY(file_data_ptr + 4, (unsigned char *)data + data_offset, data_size);

            temp_width /= 2;
            temp_height /= 2;
            data_offset += data_size;
            file_data_ptr += (4 + data_size);
        }
    }

    // Save file data to file
    int success = false;
    FILE *file = fopen(file_name, "wb");

    if (file != RL_GPUTEX_NULL)
    {
        unsigned int count = (unsigned int)fwrite(file_data, sizeof(unsigned char), data_size, file);

        if (count == 0) RL_GPUTEX_LOG("FILEIO: [%s] Failed to write file", file_name);
        else if (count != data_size) RL_GPUTEX_LOG("FILEIO: [%s] File partially written", file_name);
        else (void)0; // WARN: this branch is handled by Raylib, since rl_gputex only prints warnings

        int result = fclose(file);
        if (result != 0) RL_GPUTEX_LOG("FILEIO: [%s] Failed to close file", file_name);

        if (result == 0 && count == data_size) success = true;
    }
    else RL_GPUTEX_LOG("FILEIO: [%s] Failed to open file", file_name);

    RL_GPUTEX_FREE(file_data);    // Free file data buffer

    // If all data has been written correctly to file, success = 1
    return success;
}
#endif

#if defined(RL_GPUTEX_SUPPORT_PVR)
// Loading PVR image data (uncompressed or PVRT compression)
// NOTE: PVR v2 not supported, use PVR v3 instead
void *rl_load_pvr_from_memory(const unsigned char *file_data, unsigned int file_size, int *width, int *height, int *format, int *mips)
{
    void *image_data = RL_GPUTEX_NULL;        // Image data pointer

    unsigned char *file_data_ptr = (unsigned char *)file_data;

    // Required extension:
    // GL_IMG_texture_compression_pvrtc

    // Supported tokens (defined by extensions)
    // GL_COMPRESSED_RGB_PVRTC_4BPPV1_IMG       0x8C00
    // GL_COMPRESSED_RGBA_PVRTC_4BPPV1_IMG      0x8C02

#if 0   // Not used...
    // PVR file v2 Header (52 bytes)
    typedef struct {
        unsigned int headerLength;
        unsigned int height;
        unsigned int width;
        unsigned int numMipmaps;
        unsigned int flags;
        unsigned int dataLength;
        unsigned int bpp;
        unsigned int bitmaskRed;
        unsigned int bitmaskGreen;
        unsigned int bitmaskBlue;
        unsigned int bitmaskAlpha;
        unsigned int pvrTag;
        unsigned int numSurfs;
    } PVRHeaderV2;
#endif

    // PVR file v3 Header (52 bytes)
    // NOTE: After it could be metadata (15 bytes?)
    typedef struct {
        char id[4];
        unsigned int flags;
        unsigned char channels[4];      // pixelFormat high part
        unsigned char channel_depth[4];  // pixelFormat low part
        unsigned int color_space;
        unsigned int channel_type;
        unsigned int height;
        unsigned int width;
        unsigned int depth;
        unsigned int num_surfaces;
        unsigned int num_faces;
        unsigned int num_mipmaps;
        unsigned int metadata_size;
    } pvr_header;

#if 0   // Not used...
    // Metadata (usually 15 bytes)
    typedef struct {
        unsigned int devFOURCC;
        unsigned int key;
        unsigned int data_size;      // Not used?
        unsigned char *data;        // Not used?
    } PVRMetadata;
#endif

    if (file_data_ptr != RL_GPUTEX_NULL)
    {
        // Check PVR image version
        unsigned char pvr_version = file_data_ptr[0];

        // Load different PVR data formats
        if (pvr_version == 0x50)
        {
            pvr_header *header = (pvr_header *)file_data_ptr;

            if ((header->id[0] != 'P') || (header->id[1] != 'V') || (header->id[2] != 'R') || (header->id[3] != 3))
            {
                RL_GPUTEX_LOG("PVR file data not valid");
            }
            else
            {
                file_data_ptr += sizeof(pvr_header);   // Skip header

                *width = header->width;
                *height = header->height;
                *mips = header->num_mipmaps;

                // Check data format
                if (((header->channels[0] == 'l') && (header->channels[1] == 0)) && (header->channel_depth[0] == 8)) *format = RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_GRAYSCALE;
                else if (((header->channels[0] == 'l') && (header->channels[1] == 'a')) && ((header->channel_depth[0] == 8) && (header->channel_depth[1] == 8))) *format = RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_GRAY_ALPHA;
                else if ((header->channels[0] == 'r') && (header->channels[1] == 'g') && (header->channels[2] == 'b'))
                {
                    if (header->channels[3] == 'a')
                    {
                        if ((header->channel_depth[0] == 5) && (header->channel_depth[1] == 5) && (header->channel_depth[2] == 5) && (header->channel_depth[3] == 1)) *format = RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R5G5B5A1;
                        else if ((header->channel_depth[0] == 4) && (header->channel_depth[1] == 4) && (header->channel_depth[2] == 4) && (header->channel_depth[3] == 4)) *format = RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R4G4B4A4;
                        else if ((header->channel_depth[0] == 8) && (header->channel_depth[1] == 8) && (header->channel_depth[2] == 8) && (header->channel_depth[3] == 8)) *format = RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R8G8B8A8;
                    }
                    else if (header->channels[3] == 0)
                    {
                        if ((header->channel_depth[0] == 5) && (header->channel_depth[1] == 6) && (header->channel_depth[2] == 5)) *format = RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R5G6B5;
                        else if ((header->channel_depth[0] == 8) && (header->channel_depth[1] == 8) && (header->channel_depth[2] == 8)) *format = RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R8G8B8;
                    }
                }
                else if (header->channels[0] == 2) *format = RL_GPUTEX_PIXELFORMAT_COMPRESSED_PVRT_RGB;
                else if (header->channels[0] == 3) *format = RL_GPUTEX_PIXELFORMAT_COMPRESSED_PVRT_RGBA;

                file_data_ptr += header->metadata_size;    // Skip meta data header

                // Calculate data size (depends on format)
                int bpp = 0;
                switch (*format)
                {
                    case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_GRAYSCALE: bpp = 8; break;
                    case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_GRAY_ALPHA:
                    case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R5G5B5A1:
                    case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R5G6B5:
                    case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R4G4B4A4: bpp = 16; break;
                    case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R8G8B8A8: bpp = 32; break;
                    case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R8G8B8: bpp = 24; break;
                    case RL_GPUTEX_PIXELFORMAT_COMPRESSED_PVRT_RGB:
                    case RL_GPUTEX_PIXELFORMAT_COMPRESSED_PVRT_RGBA: bpp = 4; break;
                    default: break;
                }

                int data_size = (*width)*(*height)*bpp/8;  // Total data size in bytes
                image_data = RL_GPUTEX_MALLOC(data_size*sizeof(unsigned char));

                RL_GPUTEX_MEMCPY(image_data, file_data_ptr, data_size);
            }
        }
        else if (pvr_version == 52) RL_GPUTEX_LOG("PVRv2 format not supported, update your files to PVRv3");
    }

    return image_data;
}
#endif

#if defined(RL_GPUTEX_SUPPORT_ASTC)
// Load ASTC compressed image data (ASTC compression)
void *rl_load_astc_from_memory(const unsigned char *file_data, unsigned int file_size, int *width, int *height, int *format, int *mips)
{
    void *image_data = RL_GPUTEX_NULL;        // Image data pointer

    unsigned char *file_data_ptr = (unsigned char *)file_data;

    // Required extensions:
    // GL_KHR_texture_compression_astc_hdr
    // GL_KHR_texture_compression_astc_ldr

    // Supported tokens (defined by extensions)
    // GL_COMPRESSED_RGBA_ASTC_4x4_KHR      0x93b0
    // GL_COMPRESSED_RGBA_ASTC_8x8_KHR      0x93b7

    // ASTC file Header (16 bytes)
    typedef struct {
        unsigned char id[4];        // Signature: 0x13 0xAB 0xA1 0x5C
        unsigned char blockX;       // Block X dimensions
        unsigned char blockY;       // Block Y dimensions
        unsigned char blockZ;       // Block Z dimensions (1 for 2D images)
        unsigned char width[3];     // void *width in pixels (24bit value)
        unsigned char height[3];    // void *height in pixels (24bit value)
        unsigned char length[3];    // void *Z-size (1 for 2D images)
    } astc_header;

    if (file_data_ptr != RL_GPUTEX_NULL)
    {
        astc_header *header = (astc_header *)file_data_ptr;

        if ((header->id[3] != 0x5c) || (header->id[2] != 0xa1) || (header->id[1] != 0xab) || (header->id[0] != 0x13))
        {
            RL_GPUTEX_LOG("ASTC file data not valid");
        }
        else
        {
            file_data_ptr += sizeof(astc_header);   // Skip header

            // NOTE: Assuming Little Endian (could it be wrong?)
            *width = 0x00000000 | ((int)header->width[2] << 16) | ((int)header->width[1] << 8) | ((int)header->width[0]);
            *height = 0x00000000 | ((int)header->height[2] << 16) | ((int)header->height[1] << 8) | ((int)header->height[0]);
            *mips = 1;      // NOTE: ASTC format only contains one mipmap level

            // NOTE: Each block is always stored in 128bit so we can calculate the bpp
            int bpp = 128/(header->blockX*header->blockY);

            // NOTE: Currently we only support 2 blocks configurations: 4x4 and 8x8
            if ((bpp == 8) || (bpp == 2))
            {
                int data_size = (*width)*(*height)*bpp/8;  // Data size in bytes

                image_data = RL_GPUTEX_MALLOC(data_size*sizeof(unsigned char));

                RL_GPUTEX_MEMCPY(image_data, file_data_ptr, data_size);

                if (bpp == 8) *format = RL_GPUTEX_PIXELFORMAT_COMPRESSED_ASTC_4x4_RGBA;
                else if (bpp == 2) *format = RL_GPUTEX_PIXELFORMAT_COMPRESSED_ASTC_8x8_RGBA;
            }
            else RL_GPUTEX_LOG("ASTC block size configuration not supported");
        }
    }

    return image_data;
}
#endif

//----------------------------------------------------------------------------------
// Module Internal Functions Definition
//----------------------------------------------------------------------------------
// Get pixel data size in bytes for certain pixel format
static int get_pixel_data_size(int width, int height, int format)
{
    int data_size = 0;       // Size in bytes
    int bpp = 0;            // Bits per pixel

    switch (format)
    {
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_GRAYSCALE: bpp = 8; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_GRAY_ALPHA:
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R5G6B5:
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R5G5B5A1:
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R4G4B4A4: bpp = 16; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R8G8B8A8: bpp = 32; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R8G8B8: bpp = 24; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R32: bpp = 32; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R32G32B32: bpp = 32*3; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R32G32B32A32: bpp = 32*4; break;
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT1_RGB:
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT1_RGBA:
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_ETC1_RGB:
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_ETC2_RGB:
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_PVRT_RGB:
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_PVRT_RGBA: bpp = 4; break;
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT3_RGBA:
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT5_RGBA:
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_ETC2_EAC_RGBA:
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_ASTC_4x4_RGBA: bpp = 8; break;
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_ASTC_8x8_RGBA: bpp = 2; break;
        default: break;
    }

    data_size = width*height*bpp/8;  // Total data size in bytes

    // Most compressed formats works on 4x4 blocks,
    // if texture is smaller, minimum dataSize is 8 or 16
    if ((width < 4) && (height < 4))
    {
        if ((format >= RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT1_RGB) && (format < RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT3_RGBA)) data_size = 8;
        else if ((format >= RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT3_RGBA) && (format < RL_GPUTEX_PIXELFORMAT_COMPRESSED_ASTC_8x8_RGBA)) data_size = 16;
    }

    return data_size;
}

// Get OpenGL internal formats and data type from rlGpuTexPixelFormat
void get_gl_texture_formats(int format, unsigned int *gl_internal_format, unsigned int *gl_format, unsigned int *gl_type)
{
    // KTX 1.1 uses OpenGL formats on header info but KTX 2.0 uses Vulkan texture formats,
    // if this library is being improved to support KTX 2.0, it requires those formats to be provided -> View list at the end!

    /*
    *gl_internal_format = 0;
    *gl_format = 0;
    *gl_type = 0;

    switch (format)
    {
    #if defined(GRAPHICS_API_OPENGL_11) || defined(GRAPHICS_API_OPENGL_21) || defined(GRAPHICS_API_OPENGL_ES2)
        // NOTE: on OpenGL ES 2.0 (WebGL), internalFormat must match format and options allowed are: GL_LUMINANCE, GL_RGB, GL_RGBA
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_GRAYSCALE: *gl_internal_format = GL_LUMINANCE; *gl_format = GL_LUMINANCE; *gl_type = GL_UNSIGNED_BYTE; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_GRAY_ALPHA: *gl_internal_format = GL_LUMINANCE_ALPHA; *gl_format = GL_LUMINANCE_ALPHA; *gl_type = GL_UNSIGNED_BYTE; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R5G6B5: *gl_internal_format = GL_RGB; *gl_format = GL_RGB; *gl_type = GL_UNSIGNED_SHORT_5_6_5; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R8G8B8: *gl_internal_format = GL_RGB; *gl_format = GL_RGB; *gl_type = GL_UNSIGNED_BYTE; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R5G5B5A1: *gl_internal_format = GL_RGBA; *gl_format = GL_RGBA; *gl_type = GL_UNSIGNED_SHORT_5_5_5_1; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R4G4B4A4: *gl_internal_format = GL_RGBA; *gl_format = GL_RGBA; *gl_type = GL_UNSIGNED_SHORT_4_4_4_4; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R8G8B8A8: *gl_internal_format = GL_RGBA; *gl_format = GL_RGBA; *gl_type = GL_UNSIGNED_BYTE; break;
        #if !defined(GRAPHICS_API_OPENGL_11)
        #if defined(GRAPHICS_API_OPENGL_ES3)
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R32: *gl_internal_format = GL_R32F_EXT; *gl_format = GL_RED_EXT; *gl_type = GL_FLOAT; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R32G32B32: *gl_internal_format = GL_RGB32F_EXT; *gl_format = GL_RGB; *gl_type = GL_FLOAT; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R32G32B32A32: *gl_internal_format = GL_RGBA32F_EXT; *gl_format = GL_RGBA; *gl_type = GL_FLOAT; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R16: *gl_internal_format = GL_R16F_EXT; *gl_format = GL_RED_EXT; *gl_type = GL_HALF_FLOAT; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R16G16B16: *gl_internal_format = GL_RGB16F_EXT; *gl_format = GL_RGB; *gl_type = GL_HALF_FLOAT; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R16G16B16A16: *gl_internal_format = GL_RGBA16F_EXT; *gl_format = GL_RGBA; *gl_type = GL_HALF_FLOAT; break;
        #else
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R32: *gl_internal_format = GL_LUMINANCE; *gl_format = GL_LUMINANCE; *gl_type = GL_FLOAT; break;            // NOTE: Requires extension OES_texture_float
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R32G32B32: *gl_internal_format = GL_RGB; *gl_format = GL_RGB; *gl_type = GL_FLOAT; break;                  // NOTE: Requires extension OES_texture_float
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R32G32B32A32: *gl_internal_format = GL_RGBA; *gl_format = GL_RGBA; *gl_type = GL_FLOAT; break;             // NOTE: Requires extension OES_texture_float
        #if defined(GRAPHICS_API_OPENGL_21)
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R16: *gl_internal_format = GL_LUMINANCE; *gl_format = GL_LUMINANCE; *gl_type = GL_HALF_FLOAT_ARB; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R16G16B16: *gl_internal_format = GL_RGB; *gl_format = GL_RGB; *gl_type = GL_HALF_FLOAT_ARB; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R16G16B16A16: *gl_internal_format = GL_RGBA; *gl_format = GL_RGBA; *gl_type = GL_HALF_FLOAT_ARB; break;
        #else // defined(GRAPHICS_API_OPENGL_ES2)
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R16: *gl_internal_format = GL_LUMINANCE; *gl_format = GL_LUMINANCE; *gl_type = GL_HALF_FLOAT_OES; break;   // NOTE: Requires extension OES_texture_half_float
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R16G16B16: *gl_internal_format = GL_RGB; *gl_format = GL_RGB; *gl_type = GL_HALF_FLOAT_OES; break;         // NOTE: Requires extension OES_texture_half_float
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R16G16B16A16: *gl_internal_format = GL_RGBA; *gl_format = GL_RGBA; *gl_type = GL_HALF_FLOAT_OES; break;    // NOTE: Requires extension OES_texture_half_float
        #endif
        #endif
        #endif
    #elif defined(GRAPHICS_API_OPENGL_33)
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_GRAYSCALE: *gl_internal_format = GL_R8; *gl_format = GL_RED; *gl_type = GL_UNSIGNED_BYTE; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_GRAY_ALPHA: *gl_internal_format = GL_RG8; *gl_format = GL_RG; *gl_type = GL_UNSIGNED_BYTE; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R5G6B5: *gl_internal_format = GL_RGB565; *gl_format = GL_RGB; *gl_type = GL_UNSIGNED_SHORT_5_6_5; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R8G8B8: *gl_internal_format = GL_RGB8; *gl_format = GL_RGB; *gl_type = GL_UNSIGNED_BYTE; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R5G5B5A1: *gl_internal_format = GL_RGB5_A1; *gl_format = GL_RGBA; *gl_type = GL_UNSIGNED_SHORT_5_5_5_1; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R4G4B4A4: *gl_internal_format = GL_RGBA4; *gl_format = GL_RGBA; *gl_type = GL_UNSIGNED_SHORT_4_4_4_4; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R8G8B8A8: *gl_internal_format = GL_RGBA8; *gl_format = GL_RGBA; *gl_type = GL_UNSIGNED_BYTE; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R32: *gl_internal_format = GL_R32F; *gl_format = GL_RED; *gl_type = GL_FLOAT; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R32G32B32: *gl_internal_format = GL_RGB32F; *gl_format = GL_RGB; *gl_type = GL_FLOAT; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R32G32B32A32: *gl_internal_format = GL_RGBA32F; *gl_format = GL_RGBA; *gl_type = GL_FLOAT; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R16: *gl_internal_format = GL_R16F; *gl_format = GL_RED; *gl_type = GL_HALF_FLOAT; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R16G16B16: *gl_internal_format = GL_RGB16F; *gl_format = GL_RGB; *gl_type = GL_HALF_FLOAT; break;
        case RL_GPUTEX_PIXELFORMAT_UNCOMPRESSED_R16G16B16A16: *gl_internal_format = GL_RGBA16F; *gl_format = GL_RGBA; *gl_type = GL_HALF_FLOAT; break;
    #endif
    #if !defined(GRAPHICS_API_OPENGL_11)
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT1_RGB: *gl_internal_format = GL_COMPRESSED_RGB_S3TC_DXT1_EXT; break;
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT1_RGBA: *gl_internal_format = GL_COMPRESSED_RGBA_S3TC_DXT1_EXT; break;
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT3_RGBA: *gl_internal_format = GL_COMPRESSED_RGBA_S3TC_DXT3_EXT; break;
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_DXT5_RGBA: *gl_internal_format = GL_COMPRESSED_RGBA_S3TC_DXT5_EXT; break;
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_ETC1_RGB: *gl_internal_format = GL_ETC1_RGB8_OES; break;                      // NOTE: Requires OpenGL ES 2.0 or OpenGL 4.3
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_ETC2_RGB: *gl_internal_format = GL_COMPRESSED_RGB8_ETC2; break;               // NOTE: Requires OpenGL ES 3.0 or OpenGL 4.3
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_ETC2_EAC_RGBA: *gl_internal_format = GL_COMPRESSED_RGBA8_ETC2_EAC; break;     // NOTE: Requires OpenGL ES 3.0 or OpenGL 4.3
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_PVRT_RGB: *gl_internal_format = GL_COMPRESSED_RGB_PVRTC_4BPPV1_IMG; break;    // NOTE: Requires PowerVR GPU
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_PVRT_RGBA: *gl_internal_format = GL_COMPRESSED_RGBA_PVRTC_4BPPV1_IMG; break;  // NOTE: Requires PowerVR GPU
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_ASTC_4x4_RGBA: *gl_internal_format = GL_COMPRESSED_RGBA_ASTC_4x4_KHR; break;  // NOTE: Requires OpenGL ES 3.1 or OpenGL 4.3
        case RL_GPUTEX_PIXELFORMAT_COMPRESSED_ASTC_8x8_RGBA: *gl_internal_format = GL_COMPRESSED_RGBA_ASTC_8x8_KHR; break;  // NOTE: Requires OpenGL ES 3.1 or OpenGL 4.3
    #endif
        default: RL_GPUTEX_LOG("GPUTEX: Current format not supported (%i)", format); break;
    }
    */
}
#endif // RL_GPUTEX_IMPLEMENTATION

/*
// OpenGL texture data formats
// NOTE: Those values can be useful for KTX 1.1 saving,
// probably only the latest OpenGL ones, not the extensions ones
// So, there is no need to include full OpenGL headers
#define GL_UNSIGNED_BYTE                      0x1401
#define GL_UNSIGNED_SHORT_5_6_5               0x8363
#define GL_UNSIGNED_SHORT_4_4_4_4             0x8033
#define GL_UNSIGNED_SHORT_5_5_5_1             0x8034
#define GL_FLOAT                              0x1406
#define GL_HALF_FLOAT                         0x140b
#define GL_HALF_FLOAT_ARB                     0x140b
#define GL_HALF_FLOAT_OES                     0x8d61

#define GL_RED_EXT                            0x1903
#define GL_RGB                                0x1907
#define GL_RGBA                               0x1908
#define GL_LUMINANCE                          0x1909
#define GL_R8                                 0x8229
#define GL_RG8                                0x822b
#define GL_RGB565                             0x8d62
#define GL_RGB8                               0x8051
#define GL_RGB5_A1                            0x8057
#define GL_RGBA4                              0x8056
#define GL_RGBA8                              0x8058
#define GL_R16F                               0x822d
#define GL_RGB16F                             0x881b
#define GL_RGBA16F                            0x881a
#define GL_R32F                               0x822e
#define GL_RGB32F                             0x8815
#define GL_RGBA32F                            0x8814

#define GL_R32F_EXT                           0x822e
#define GL_RGB32F_EXT                         0x8815
#define GL_RGBA32F_EXT                        0x8814
#define GL_R16F_EXT                           0x822d
#define GL_RGB16F_EXT                         0x881b
#define GL_RGBA16F_EXT                        0x881a

// S3TC (DXT) compression
#define GL_COMPRESSED_RGB_S3TC_DXT1_EXT       0x83f0
#define GL_COMPRESSED_RGBA_S3TC_DXT1_EXT      0x83f1
#define GL_COMPRESSED_RGBA_S3TC_DXT3_EXT      0x83f2
#define GL_COMPRESSED_RGBA_S3TC_DXT5_EXT      0x83f3

// ETC formats
#define GL_ETC1_RGB8_OES                      0x8d64
#define GL_COMPRESSED_RGB8_ETC2               0x9274
#define GL_COMPRESSED_RGBA8_ETC2_EAC          0x9278

// PVRTC
#define GL_COMPRESSED_RGB_PVRTC_4BPPV1_IMG    0x8c00
#define GL_COMPRESSED_RGBA_PVRTC_4BPPV1_IMG   0x8c02

// ASTC
#define GL_COMPRESSED_RGBA_ASTC_4x4_KHR       0x93b0
#define GL_COMPRESSED_RGBA_ASTC_8x8_KHR       0x93b7
*/

/*
// Vulkan texture formats
typedef enum VkFormat {
    // Provided by VK_VERSION_1_0
    VK_FORMAT_UNDEFINED = 0,
    VK_FORMAT_R4G4_UNORM_PACK8 = 1,
    VK_FORMAT_R4G4B4A4_UNORM_PACK16 = 2,
    VK_FORMAT_B4G4R4A4_UNORM_PACK16 = 3,
    VK_FORMAT_R5G6B5_UNORM_PACK16 = 4,
    VK_FORMAT_B5G6R5_UNORM_PACK16 = 5,
    VK_FORMAT_R5G5B5A1_UNORM_PACK16 = 6,
    VK_FORMAT_B5G5R5A1_UNORM_PACK16 = 7,
    VK_FORMAT_A1R5G5B5_UNORM_PACK16 = 8,
    VK_FORMAT_R8_UNORM = 9,
    VK_FORMAT_R8_SNORM = 10,
    VK_FORMAT_R8_USCALED = 11,
    VK_FORMAT_R8_SSCALED = 12,
    VK_FORMAT_R8_UINT = 13,
    VK_FORMAT_R8_SINT = 14,
    VK_FORMAT_R8_SRGB = 15,
    VK_FORMAT_R8G8_UNORM = 16,
    VK_FORMAT_R8G8_SNORM = 17,
    VK_FORMAT_R8G8_USCALED = 18,
    VK_FORMAT_R8G8_SSCALED = 19,
    VK_FORMAT_R8G8_UINT = 20,
    VK_FORMAT_R8G8_SINT = 21,
    VK_FORMAT_R8G8_SRGB = 22,
    VK_FORMAT_R8G8B8_UNORM = 23,
    VK_FORMAT_R8G8B8_SNORM = 24,
    VK_FORMAT_R8G8B8_USCALED = 25,
    VK_FORMAT_R8G8B8_SSCALED = 26,
    VK_FORMAT_R8G8B8_UINT = 27,
    VK_FORMAT_R8G8B8_SINT = 28,
    VK_FORMAT_R8G8B8_SRGB = 29,
    VK_FORMAT_B8G8R8_UNORM = 30,
    VK_FORMAT_B8G8R8_SNORM = 31,
    VK_FORMAT_B8G8R8_USCALED = 32,
    VK_FORMAT_B8G8R8_SSCALED = 33,
    VK_FORMAT_B8G8R8_UINT = 34,
    VK_FORMAT_B8G8R8_SINT = 35,
    VK_FORMAT_B8G8R8_SRGB = 36,
    VK_FORMAT_R8G8B8A8_UNORM = 37,
    VK_FORMAT_R8G8B8A8_SNORM = 38,
    VK_FORMAT_R8G8B8A8_USCALED = 39,
    VK_FORMAT_R8G8B8A8_SSCALED = 40,
    VK_FORMAT_R8G8B8A8_UINT = 41,
    VK_FORMAT_R8G8B8A8_SINT = 42,
    VK_FORMAT_R8G8B8A8_SRGB = 43,
    VK_FORMAT_B8G8R8A8_UNORM = 44,
    VK_FORMAT_B8G8R8A8_SNORM = 45,
    VK_FORMAT_B8G8R8A8_USCALED = 46,
    VK_FORMAT_B8G8R8A8_SSCALED = 47,
    VK_FORMAT_B8G8R8A8_UINT = 48,
    VK_FORMAT_B8G8R8A8_SINT = 49,
    VK_FORMAT_B8G8R8A8_SRGB = 50,
    VK_FORMAT_A8B8G8R8_UNORM_PACK32 = 51,
    VK_FORMAT_A8B8G8R8_SNORM_PACK32 = 52,
    VK_FORMAT_A8B8G8R8_USCALED_PACK32 = 53,
    VK_FORMAT_A8B8G8R8_SSCALED_PACK32 = 54,
    VK_FORMAT_A8B8G8R8_UINT_PACK32 = 55,
    VK_FORMAT_A8B8G8R8_SINT_PACK32 = 56,
    VK_FORMAT_A8B8G8R8_SRGB_PACK32 = 57,
    VK_FORMAT_A2R10G10B10_UNORM_PACK32 = 58,
    VK_FORMAT_A2R10G10B10_SNORM_PACK32 = 59,
    VK_FORMAT_A2R10G10B10_USCALED_PACK32 = 60,
    VK_FORMAT_A2R10G10B10_SSCALED_PACK32 = 61,
    VK_FORMAT_A2R10G10B10_UINT_PACK32 = 62,
    VK_FORMAT_A2R10G10B10_SINT_PACK32 = 63,
    VK_FORMAT_A2B10G10R10_UNORM_PACK32 = 64,
    VK_FORMAT_A2B10G10R10_SNORM_PACK32 = 65,
    VK_FORMAT_A2B10G10R10_USCALED_PACK32 = 66,
    VK_FORMAT_A2B10G10R10_SSCALED_PACK32 = 67,
    VK_FORMAT_A2B10G10R10_UINT_PACK32 = 68,
    VK_FORMAT_A2B10G10R10_SINT_PACK32 = 69,
    VK_FORMAT_R16_UNORM = 70,
    VK_FORMAT_R16_SNORM = 71,
    VK_FORMAT_R16_USCALED = 72,
    VK_FORMAT_R16_SSCALED = 73,
    VK_FORMAT_R16_UINT = 74,
    VK_FORMAT_R16_SINT = 75,
    VK_FORMAT_R16_SFLOAT = 76,
    VK_FORMAT_R16G16_UNORM = 77,
    VK_FORMAT_R16G16_SNORM = 78,
    VK_FORMAT_R16G16_USCALED = 79,
    VK_FORMAT_R16G16_SSCALED = 80,
    VK_FORMAT_R16G16_UINT = 81,
    VK_FORMAT_R16G16_SINT = 82,
    VK_FORMAT_R16G16_SFLOAT = 83,
    VK_FORMAT_R16G16B16_UNORM = 84,
    VK_FORMAT_R16G16B16_SNORM = 85,
    VK_FORMAT_R16G16B16_USCALED = 86,
    VK_FORMAT_R16G16B16_SSCALED = 87,
    VK_FORMAT_R16G16B16_UINT = 88,
    VK_FORMAT_R16G16B16_SINT = 89,
    VK_FORMAT_R16G16B16_SFLOAT = 90,
    VK_FORMAT_R16G16B16A16_UNORM = 91,
    VK_FORMAT_R16G16B16A16_SNORM = 92,
    VK_FORMAT_R16G16B16A16_USCALED = 93,
    VK_FORMAT_R16G16B16A16_SSCALED = 94,
    VK_FORMAT_R16G16B16A16_UINT = 95,
    VK_FORMAT_R16G16B16A16_SINT = 96,
    VK_FORMAT_R16G16B16A16_SFLOAT = 97,
    VK_FORMAT_R32_UINT = 98,
    VK_FORMAT_R32_SINT = 99,
    VK_FORMAT_R32_SFLOAT = 100,
    VK_FORMAT_R32G32_UINT = 101,
    VK_FORMAT_R32G32_SINT = 102,
    VK_FORMAT_R32G32_SFLOAT = 103,
    VK_FORMAT_R32G32B32_UINT = 104,
    VK_FORMAT_R32G32B32_SINT = 105,
    VK_FORMAT_R32G32B32_SFLOAT = 106,
    VK_FORMAT_R32G32B32A32_UINT = 107,
    VK_FORMAT_R32G32B32A32_SINT = 108,
    VK_FORMAT_R32G32B32A32_SFLOAT = 109,
    VK_FORMAT_R64_UINT = 110,
    VK_FORMAT_R64_SINT = 111,
    VK_FORMAT_R64_SFLOAT = 112,
    VK_FORMAT_R64G64_UINT = 113,
    VK_FORMAT_R64G64_SINT = 114,
    VK_FORMAT_R64G64_SFLOAT = 115,
    VK_FORMAT_R64G64B64_UINT = 116,
    VK_FORMAT_R64G64B64_SINT = 117,
    VK_FORMAT_R64G64B64_SFLOAT = 118,
    VK_FORMAT_R64G64B64A64_UINT = 119,
    VK_FORMAT_R64G64B64A64_SINT = 120,
    VK_FORMAT_R64G64B64A64_SFLOAT = 121,
    VK_FORMAT_B10G11R11_UFLOAT_PACK32 = 122,
    VK_FORMAT_E5B9G9R9_UFLOAT_PACK32 = 123,
    VK_FORMAT_D16_UNORM = 124,
    VK_FORMAT_X8_D24_UNORM_PACK32 = 125,
    VK_FORMAT_D32_SFLOAT = 126,
    VK_FORMAT_S8_UINT = 127,
    VK_FORMAT_D16_UNORM_S8_UINT = 128,
    VK_FORMAT_D24_UNORM_S8_UINT = 129,
    VK_FORMAT_D32_SFLOAT_S8_UINT = 130,
    VK_FORMAT_BC1_RGB_UNORM_BLOCK = 131,
    VK_FORMAT_BC1_RGB_SRGB_BLOCK = 132,
    VK_FORMAT_BC1_RGBA_UNORM_BLOCK = 133,
    VK_FORMAT_BC1_RGBA_SRGB_BLOCK = 134,
    VK_FORMAT_BC2_UNORM_BLOCK = 135,
    VK_FORMAT_BC2_SRGB_BLOCK = 136,
    VK_FORMAT_BC3_UNORM_BLOCK = 137,
    VK_FORMAT_BC3_SRGB_BLOCK = 138,
    VK_FORMAT_BC4_UNORM_BLOCK = 139,
    VK_FORMAT_BC4_SNORM_BLOCK = 140,
    VK_FORMAT_BC5_UNORM_BLOCK = 141,
    VK_FORMAT_BC5_SNORM_BLOCK = 142,
    VK_FORMAT_BC6H_UFLOAT_BLOCK = 143,
    VK_FORMAT_BC6H_SFLOAT_BLOCK = 144,
    VK_FORMAT_BC7_UNORM_BLOCK = 145,
    VK_FORMAT_BC7_SRGB_BLOCK = 146,
    VK_FORMAT_ETC2_R8G8B8_UNORM_BLOCK = 147,
    VK_FORMAT_ETC2_R8G8B8_SRGB_BLOCK = 148,
    VK_FORMAT_ETC2_R8G8B8A1_UNORM_BLOCK = 149,
    VK_FORMAT_ETC2_R8G8B8A1_SRGB_BLOCK = 150,
    VK_FORMAT_ETC2_R8G8B8A8_UNORM_BLOCK = 151,
    VK_FORMAT_ETC2_R8G8B8A8_SRGB_BLOCK = 152,
    VK_FORMAT_EAC_R11_UNORM_BLOCK = 153,
    VK_FORMAT_EAC_R11_SNORM_BLOCK = 154,
    VK_FORMAT_EAC_R11G11_UNORM_BLOCK = 155,
    VK_FORMAT_EAC_R11G11_SNORM_BLOCK = 156,
    VK_FORMAT_ASTC_4x4_UNORM_BLOCK = 157,
    VK_FORMAT_ASTC_4x4_SRGB_BLOCK = 158,
    VK_FORMAT_ASTC_5x4_UNORM_BLOCK = 159,
    VK_FORMAT_ASTC_5x4_SRGB_BLOCK = 160,
    VK_FORMAT_ASTC_5x5_UNORM_BLOCK = 161,
    VK_FORMAT_ASTC_5x5_SRGB_BLOCK = 162,
    VK_FORMAT_ASTC_6x5_UNORM_BLOCK = 163,
    VK_FORMAT_ASTC_6x5_SRGB_BLOCK = 164,
    VK_FORMAT_ASTC_6x6_UNORM_BLOCK = 165,
    VK_FORMAT_ASTC_6x6_SRGB_BLOCK = 166,
    VK_FORMAT_ASTC_8x5_UNORM_BLOCK = 167,
    VK_FORMAT_ASTC_8x5_SRGB_BLOCK = 168,
    VK_FORMAT_ASTC_8x6_UNORM_BLOCK = 169,
    VK_FORMAT_ASTC_8x6_SRGB_BLOCK = 170,
    VK_FORMAT_ASTC_8x8_UNORM_BLOCK = 171,
    VK_FORMAT_ASTC_8x8_SRGB_BLOCK = 172,
    VK_FORMAT_ASTC_10x5_UNORM_BLOCK = 173,
    VK_FORMAT_ASTC_10x5_SRGB_BLOCK = 174,
    VK_FORMAT_ASTC_10x6_UNORM_BLOCK = 175,
    VK_FORMAT_ASTC_10x6_SRGB_BLOCK = 176,
    VK_FORMAT_ASTC_10x8_UNORM_BLOCK = 177,
    VK_FORMAT_ASTC_10x8_SRGB_BLOCK = 178,
    VK_FORMAT_ASTC_10x10_UNORM_BLOCK = 179,
    VK_FORMAT_ASTC_10x10_SRGB_BLOCK = 180,
    VK_FORMAT_ASTC_12x10_UNORM_BLOCK = 181,
    VK_FORMAT_ASTC_12x10_SRGB_BLOCK = 182,
    VK_FORMAT_ASTC_12x12_UNORM_BLOCK = 183,
    VK_FORMAT_ASTC_12x12_SRGB_BLOCK = 184,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G8B8G8R8_422_UNORM = 1000156000,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_B8G8R8G8_422_UNORM = 1000156001,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G8_B8_R8_3PLANE_420_UNORM = 1000156002,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G8_B8R8_2PLANE_420_UNORM = 1000156003,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G8_B8_R8_3PLANE_422_UNORM = 1000156004,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G8_B8R8_2PLANE_422_UNORM = 1000156005,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G8_B8_R8_3PLANE_444_UNORM = 1000156006,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_R10X6_UNORM_PACK16 = 1000156007,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_R10X6G10X6_UNORM_2PACK16 = 1000156008,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_R10X6G10X6B10X6A10X6_UNORM_4PACK16 = 1000156009,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G10X6B10X6G10X6R10X6_422_UNORM_4PACK16 = 1000156010,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_B10X6G10X6R10X6G10X6_422_UNORM_4PACK16 = 1000156011,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_420_UNORM_3PACK16 = 1000156012,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G10X6_B10X6R10X6_2PLANE_420_UNORM_3PACK16 = 1000156013,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_422_UNORM_3PACK16 = 1000156014,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G10X6_B10X6R10X6_2PLANE_422_UNORM_3PACK16 = 1000156015,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_444_UNORM_3PACK16 = 1000156016,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_R12X4_UNORM_PACK16 = 1000156017,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_R12X4G12X4_UNORM_2PACK16 = 1000156018,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_R12X4G12X4B12X4A12X4_UNORM_4PACK16 = 1000156019,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G12X4B12X4G12X4R12X4_422_UNORM_4PACK16 = 1000156020,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_B12X4G12X4R12X4G12X4_422_UNORM_4PACK16 = 1000156021,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_420_UNORM_3PACK16 = 1000156022,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G12X4_B12X4R12X4_2PLANE_420_UNORM_3PACK16 = 1000156023,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_422_UNORM_3PACK16 = 1000156024,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G12X4_B12X4R12X4_2PLANE_422_UNORM_3PACK16 = 1000156025,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_444_UNORM_3PACK16 = 1000156026,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G16B16G16R16_422_UNORM = 1000156027,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_B16G16R16G16_422_UNORM = 1000156028,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G16_B16_R16_3PLANE_420_UNORM = 1000156029,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G16_B16R16_2PLANE_420_UNORM = 1000156030,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G16_B16_R16_3PLANE_422_UNORM = 1000156031,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G16_B16R16_2PLANE_422_UNORM = 1000156032,
    // Provided by VK_VERSION_1_1
    VK_FORMAT_G16_B16_R16_3PLANE_444_UNORM = 1000156033,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_G8_B8R8_2PLANE_444_UNORM = 1000330000,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_G10X6_B10X6R10X6_2PLANE_444_UNORM_3PACK16 = 1000330001,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_G12X4_B12X4R12X4_2PLANE_444_UNORM_3PACK16 = 1000330002,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_G16_B16R16_2PLANE_444_UNORM = 1000330003,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_A4R4G4B4_UNORM_PACK16 = 1000340000,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_A4B4G4R4_UNORM_PACK16 = 1000340001,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_ASTC_4x4_SFLOAT_BLOCK = 1000066000,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_ASTC_5x4_SFLOAT_BLOCK = 1000066001,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_ASTC_5x5_SFLOAT_BLOCK = 1000066002,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_ASTC_6x5_SFLOAT_BLOCK = 1000066003,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_ASTC_6x6_SFLOAT_BLOCK = 1000066004,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_ASTC_8x5_SFLOAT_BLOCK = 1000066005,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_ASTC_8x6_SFLOAT_BLOCK = 1000066006,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_ASTC_8x8_SFLOAT_BLOCK = 1000066007,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_ASTC_10x5_SFLOAT_BLOCK = 1000066008,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_ASTC_10x6_SFLOAT_BLOCK = 1000066009,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_ASTC_10x8_SFLOAT_BLOCK = 1000066010,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_ASTC_10x10_SFLOAT_BLOCK = 1000066011,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_ASTC_12x10_SFLOAT_BLOCK = 1000066012,
    // Provided by VK_VERSION_1_3
    VK_FORMAT_ASTC_12x12_SFLOAT_BLOCK = 1000066013,
    // Provided by VK_VERSION_1_4
    VK_FORMAT_A1B5G5R5_UNORM_PACK16 = 1000470000,
    // Provided by VK_VERSION_1_4
    VK_FORMAT_A8_UNORM = 1000470001,
    // Provided by VK_IMG_format_pvrtc
    VK_FORMAT_PVRTC1_2BPP_UNORM_BLOCK_IMG = 1000054000,
    // Provided by VK_IMG_format_pvrtc
    VK_FORMAT_PVRTC1_4BPP_UNORM_BLOCK_IMG = 1000054001,
    // Provided by VK_IMG_format_pvrtc
    VK_FORMAT_PVRTC2_2BPP_UNORM_BLOCK_IMG = 1000054002,
    // Provided by VK_IMG_format_pvrtc
    VK_FORMAT_PVRTC2_4BPP_UNORM_BLOCK_IMG = 1000054003,
    // Provided by VK_IMG_format_pvrtc
    VK_FORMAT_PVRTC1_2BPP_SRGB_BLOCK_IMG = 1000054004,
    // Provided by VK_IMG_format_pvrtc
    VK_FORMAT_PVRTC1_4BPP_SRGB_BLOCK_IMG = 1000054005,
    // Provided by VK_IMG_format_pvrtc
    VK_FORMAT_PVRTC2_2BPP_SRGB_BLOCK_IMG = 1000054006,
    // Provided by VK_IMG_format_pvrtc
    VK_FORMAT_PVRTC2_4BPP_SRGB_BLOCK_IMG = 1000054007,
    // Provided by VK_ARM_tensors
    VK_FORMAT_R8_BOOL_ARM = 1000460000,
    // Provided by VK_NV_optical_flow
    VK_FORMAT_R16G16_SFIXED5_NV = 1000464000,
    // Provided by VK_ARM_format_pack
    VK_FORMAT_R10X6_UINT_PACK16_ARM = 1000609000,
    // Provided by VK_ARM_format_pack
    VK_FORMAT_R10X6G10X6_UINT_2PACK16_ARM = 1000609001,
    // Provided by VK_ARM_format_pack
    VK_FORMAT_R10X6G10X6B10X6A10X6_UINT_4PACK16_ARM = 1000609002,
    // Provided by VK_ARM_format_pack
    VK_FORMAT_R12X4_UINT_PACK16_ARM = 1000609003,
    // Provided by VK_ARM_format_pack
    VK_FORMAT_R12X4G12X4_UINT_2PACK16_ARM = 1000609004,
    // Provided by VK_ARM_format_pack
    VK_FORMAT_R12X4G12X4B12X4A12X4_UINT_4PACK16_ARM = 1000609005,
    // Provided by VK_ARM_format_pack
    VK_FORMAT_R14X2_UINT_PACK16_ARM = 1000609006,
    // Provided by VK_ARM_format_pack
    VK_FORMAT_R14X2G14X2_UINT_2PACK16_ARM = 1000609007,
    // Provided by VK_ARM_format_pack
    VK_FORMAT_R14X2G14X2B14X2A14X2_UINT_4PACK16_ARM = 1000609008,
    // Provided by VK_ARM_format_pack
    VK_FORMAT_R14X2_UNORM_PACK16_ARM = 1000609009,
    // Provided by VK_ARM_format_pack
    VK_FORMAT_R14X2G14X2_UNORM_2PACK16_ARM = 1000609010,
    // Provided by VK_ARM_format_pack
    VK_FORMAT_R14X2G14X2B14X2A14X2_UNORM_4PACK16_ARM = 1000609011,
    // Provided by VK_ARM_format_pack
    VK_FORMAT_G14X2_B14X2R14X2_2PLANE_420_UNORM_3PACK16_ARM = 1000609012,
    // Provided by VK_ARM_format_pack
    VK_FORMAT_G14X2_B14X2R14X2_2PLANE_422_UNORM_3PACK16_ARM = 1000609013,
    // Provided by VK_EXT_texture_compression_astc_hdr
    VK_FORMAT_ASTC_4x4_SFLOAT_BLOCK_EXT = VK_FORMAT_ASTC_4x4_SFLOAT_BLOCK,
    // Provided by VK_EXT_texture_compression_astc_hdr
    VK_FORMAT_ASTC_5x4_SFLOAT_BLOCK_EXT = VK_FORMAT_ASTC_5x4_SFLOAT_BLOCK,
    // Provided by VK_EXT_texture_compression_astc_hdr
    VK_FORMAT_ASTC_5x5_SFLOAT_BLOCK_EXT = VK_FORMAT_ASTC_5x5_SFLOAT_BLOCK,
    // Provided by VK_EXT_texture_compression_astc_hdr
    VK_FORMAT_ASTC_6x5_SFLOAT_BLOCK_EXT = VK_FORMAT_ASTC_6x5_SFLOAT_BLOCK,
    // Provided by VK_EXT_texture_compression_astc_hdr
    VK_FORMAT_ASTC_6x6_SFLOAT_BLOCK_EXT = VK_FORMAT_ASTC_6x6_SFLOAT_BLOCK,
    // Provided by VK_EXT_texture_compression_astc_hdr
    VK_FORMAT_ASTC_8x5_SFLOAT_BLOCK_EXT = VK_FORMAT_ASTC_8x5_SFLOAT_BLOCK,
    // Provided by VK_EXT_texture_compression_astc_hdr
    VK_FORMAT_ASTC_8x6_SFLOAT_BLOCK_EXT = VK_FORMAT_ASTC_8x6_SFLOAT_BLOCK,
    // Provided by VK_EXT_texture_compression_astc_hdr
    VK_FORMAT_ASTC_8x8_SFLOAT_BLOCK_EXT = VK_FORMAT_ASTC_8x8_SFLOAT_BLOCK,
    // Provided by VK_EXT_texture_compression_astc_hdr
    VK_FORMAT_ASTC_10x5_SFLOAT_BLOCK_EXT = VK_FORMAT_ASTC_10x5_SFLOAT_BLOCK,
    // Provided by VK_EXT_texture_compression_astc_hdr
    VK_FORMAT_ASTC_10x6_SFLOAT_BLOCK_EXT = VK_FORMAT_ASTC_10x6_SFLOAT_BLOCK,
    // Provided by VK_EXT_texture_compression_astc_hdr
    VK_FORMAT_ASTC_10x8_SFLOAT_BLOCK_EXT = VK_FORMAT_ASTC_10x8_SFLOAT_BLOCK,
    // Provided by VK_EXT_texture_compression_astc_hdr
    VK_FORMAT_ASTC_10x10_SFLOAT_BLOCK_EXT = VK_FORMAT_ASTC_10x10_SFLOAT_BLOCK,
    // Provided by VK_EXT_texture_compression_astc_hdr
    VK_FORMAT_ASTC_12x10_SFLOAT_BLOCK_EXT = VK_FORMAT_ASTC_12x10_SFLOAT_BLOCK,
    // Provided by VK_EXT_texture_compression_astc_hdr
    VK_FORMAT_ASTC_12x12_SFLOAT_BLOCK_EXT = VK_FORMAT_ASTC_12x12_SFLOAT_BLOCK,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G8B8G8R8_422_UNORM_KHR = VK_FORMAT_G8B8G8R8_422_UNORM,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_B8G8R8G8_422_UNORM_KHR = VK_FORMAT_B8G8R8G8_422_UNORM,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G8_B8_R8_3PLANE_420_UNORM_KHR = VK_FORMAT_G8_B8_R8_3PLANE_420_UNORM,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G8_B8R8_2PLANE_420_UNORM_KHR = VK_FORMAT_G8_B8R8_2PLANE_420_UNORM,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G8_B8_R8_3PLANE_422_UNORM_KHR = VK_FORMAT_G8_B8_R8_3PLANE_422_UNORM,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G8_B8R8_2PLANE_422_UNORM_KHR = VK_FORMAT_G8_B8R8_2PLANE_422_UNORM,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G8_B8_R8_3PLANE_444_UNORM_KHR = VK_FORMAT_G8_B8_R8_3PLANE_444_UNORM,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_R10X6_UNORM_PACK16_KHR = VK_FORMAT_R10X6_UNORM_PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_R10X6G10X6_UNORM_2PACK16_KHR = VK_FORMAT_R10X6G10X6_UNORM_2PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_R10X6G10X6B10X6A10X6_UNORM_4PACK16_KHR = VK_FORMAT_R10X6G10X6B10X6A10X6_UNORM_4PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G10X6B10X6G10X6R10X6_422_UNORM_4PACK16_KHR = VK_FORMAT_G10X6B10X6G10X6R10X6_422_UNORM_4PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_B10X6G10X6R10X6G10X6_422_UNORM_4PACK16_KHR = VK_FORMAT_B10X6G10X6R10X6G10X6_422_UNORM_4PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_420_UNORM_3PACK16_KHR = VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_420_UNORM_3PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G10X6_B10X6R10X6_2PLANE_420_UNORM_3PACK16_KHR = VK_FORMAT_G10X6_B10X6R10X6_2PLANE_420_UNORM_3PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_422_UNORM_3PACK16_KHR = VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_422_UNORM_3PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G10X6_B10X6R10X6_2PLANE_422_UNORM_3PACK16_KHR = VK_FORMAT_G10X6_B10X6R10X6_2PLANE_422_UNORM_3PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_444_UNORM_3PACK16_KHR = VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_444_UNORM_3PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_R12X4_UNORM_PACK16_KHR = VK_FORMAT_R12X4_UNORM_PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_R12X4G12X4_UNORM_2PACK16_KHR = VK_FORMAT_R12X4G12X4_UNORM_2PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_R12X4G12X4B12X4A12X4_UNORM_4PACK16_KHR = VK_FORMAT_R12X4G12X4B12X4A12X4_UNORM_4PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G12X4B12X4G12X4R12X4_422_UNORM_4PACK16_KHR = VK_FORMAT_G12X4B12X4G12X4R12X4_422_UNORM_4PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_B12X4G12X4R12X4G12X4_422_UNORM_4PACK16_KHR = VK_FORMAT_B12X4G12X4R12X4G12X4_422_UNORM_4PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_420_UNORM_3PACK16_KHR = VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_420_UNORM_3PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G12X4_B12X4R12X4_2PLANE_420_UNORM_3PACK16_KHR = VK_FORMAT_G12X4_B12X4R12X4_2PLANE_420_UNORM_3PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_422_UNORM_3PACK16_KHR = VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_422_UNORM_3PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G12X4_B12X4R12X4_2PLANE_422_UNORM_3PACK16_KHR = VK_FORMAT_G12X4_B12X4R12X4_2PLANE_422_UNORM_3PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_444_UNORM_3PACK16_KHR = VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_444_UNORM_3PACK16,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G16B16G16R16_422_UNORM_KHR = VK_FORMAT_G16B16G16R16_422_UNORM,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_B16G16R16G16_422_UNORM_KHR = VK_FORMAT_B16G16R16G16_422_UNORM,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G16_B16_R16_3PLANE_420_UNORM_KHR = VK_FORMAT_G16_B16_R16_3PLANE_420_UNORM,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G16_B16R16_2PLANE_420_UNORM_KHR = VK_FORMAT_G16_B16R16_2PLANE_420_UNORM,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G16_B16_R16_3PLANE_422_UNORM_KHR = VK_FORMAT_G16_B16_R16_3PLANE_422_UNORM,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G16_B16R16_2PLANE_422_UNORM_KHR = VK_FORMAT_G16_B16R16_2PLANE_422_UNORM,
    // Provided by VK_KHR_sampler_ycbcr_conversion
    VK_FORMAT_G16_B16_R16_3PLANE_444_UNORM_KHR = VK_FORMAT_G16_B16_R16_3PLANE_444_UNORM,
    // Provided by VK_EXT_ycbcr_2plane_444_formats
    VK_FORMAT_G8_B8R8_2PLANE_444_UNORM_EXT = VK_FORMAT_G8_B8R8_2PLANE_444_UNORM,
    // Provided by VK_EXT_ycbcr_2plane_444_formats
    VK_FORMAT_G10X6_B10X6R10X6_2PLANE_444_UNORM_3PACK16_EXT = VK_FORMAT_G10X6_B10X6R10X6_2PLANE_444_UNORM_3PACK16,
    // Provided by VK_EXT_ycbcr_2plane_444_formats
    VK_FORMAT_G12X4_B12X4R12X4_2PLANE_444_UNORM_3PACK16_EXT = VK_FORMAT_G12X4_B12X4R12X4_2PLANE_444_UNORM_3PACK16,
    // Provided by VK_EXT_ycbcr_2plane_444_formats
    VK_FORMAT_G16_B16R16_2PLANE_444_UNORM_EXT = VK_FORMAT_G16_B16R16_2PLANE_444_UNORM,
    // Provided by VK_EXT_4444_formats
    VK_FORMAT_A4R4G4B4_UNORM_PACK16_EXT = VK_FORMAT_A4R4G4B4_UNORM_PACK16,
    // Provided by VK_EXT_4444_formats
    VK_FORMAT_A4B4G4R4_UNORM_PACK16_EXT = VK_FORMAT_A4B4G4R4_UNORM_PACK16,
    // Provided by VK_NV_optical_flow
    // VK_FORMAT_R16G16_S10_5_NV is a deprecated alias
    VK_FORMAT_R16G16_S10_5_NV = VK_FORMAT_R16G16_SFIXED5_NV,
    // Provided by VK_KHR_maintenance5
    VK_FORMAT_A1B5G5R5_UNORM_PACK16_KHR = VK_FORMAT_A1B5G5R5_UNORM_PACK16,
    // Provided by VK_KHR_maintenance5
    VK_FORMAT_A8_UNORM_KHR = VK_FORMAT_A8_UNORM,
} VkFormat;
*/
