{-# LANGUAGE OverloadedStrings #-}

{- | Configuration constants and settings for OpenTimestamps.

This module contains all the configuration values used throughout
the OpenTimestamps library, including HTTP settings, timeouts,
file format constants, and operational limits.
-}
module OpenTimestamps.Config
  ( accept
  , apiUrlDigest
  , apiUrlTimeStamp
  , contentType
  , userAgent
  , bitcoinRpcTimeoutMicroseconds
  , upgradeMaxIterations
  , maxTimestampMessageLength
  , opMaxVarBytesLength
  , attestationMaxVarBytesLength
  , detachedTimestampFileMagic
  , detachedTimestampFileVersion
  ) where

import Data.ByteString (ByteString)
import Data.Word (Word8)

-- | Accept string for HTTP requests.
accept :: ByteString
accept = "application/vnd.opentimestamps.v1"

-- | Content-Type string for HTTP requests.
contentType :: ByteString
contentType = "application/octet-stream"

-- | User-Agent string for HTTP requests.
userAgent :: ByteString
userAgent = "Haskell-OpenTimestamps-Client/0.5.5.1"

-- | URL string for digest HTTP requests.
apiUrlDigest :: String
apiUrlDigest = "/digest"

-- | URL string for timestamp HTTP requests.
apiUrlTimeStamp :: String
apiUrlTimeStamp = "/timestamp/"

-- | Timeout for Bitcoin RPC requests in microseconds (180 seconds).
bitcoinRpcTimeoutMicroseconds :: Int
bitcoinRpcTimeoutMicroseconds = 180 * 1000 * 1000

-- | Maximum number of iterations for the upgrade process.
upgradeMaxIterations :: Int
upgradeMaxIterations = 10

-- | Maximum length for Timestamp message byte strings in Timestamp.hs.
maxTimestampMessageLength :: Int
maxTimestampMessageLength = 4096

-- | Maximum length for variable-length byte strings in Op.hs (Append/Prepend).
opMaxVarBytesLength :: Int
opMaxVarBytesLength = 4096

-- | Maximum length for variable-length byte strings in Attestation.hs (Pending).
attestationMaxVarBytesLength :: Int
attestationMaxVarBytesLength = 8192

-- | Magic string for detached timestamp files.
detachedTimestampFileMagic :: ByteString
detachedTimestampFileMagic = "\x00OpenTimestamps\x00\x00Proof\x00\xbf\x89\xe2\xe8\x84\xe8\x92\x94"

-- | Version for detached timestamp files.
detachedTimestampFileVersion :: Word8
detachedTimestampFileVersion = 1
