{-# LANGUAGE OverloadedStrings #-}
module Language.Cimple.Hic.Inference.TaggedUnionSpec where

import           Test.Hspec                        (Spec, describe, it,
                                                    shouldBe)

import           Language.Cimple.Hic.Inference     (inferProgram)
import           Language.Cimple.Hic.InferenceSpec (checkInference,
                                                    checkRefactoring, mustParse)

spec :: Spec
spec = do
    describe "TaggedUnion inference with mistakes" $ do
        it "issues a diagnostic when a union member is missing" $ do
            prog <- mustParse
                [ "typedef enum Tox_Event_Type {"
                , "    TOX_EVENT_FRIEND_MESSAGE,"
                , "    TOX_EVENT_SOMETHING_ELSE"
                , "} Tox_Event_Type;"
                , "typedef union Tox_Event_Data {"
                , "    struct Tox_Event_Friend_Message *friend_message;"
                , "} Tox_Event_Data;"
                , "struct Tox_Event {"
                , "    Tox_Event_Type type;"
                , "    Tox_Event_Data data;"
                , "};"
                ]
            let (_, diags) = inferProgram prog
            diags `shouldBe` ["TaggedUnion Tox_Event: could not find union member for enum value TOX_EVENT_SOMETHING_ELSE"]

    describe "TaggedUnion inference" $ do
        checkInference
            [ "typedef enum Tox_Event_Type {"
            , "    TOX_EVENT_FRIEND_MESSAGE"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "    struct Tox_Event_Friend_Message *friend_message;"
            , "} Tox_Event_Data;"
            , "struct Tox_Event {"
            , "    Tox_Event_Type type;"
            , "    Tox_Event_Data data;"
            , "};"
            ]
            [ "typedef enum Tox_Event_Type {"
            , "  TOX_EVENT_FRIEND_MESSAGE,"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "  struct Tox_Event_Friend_Message* friend_message;"
            , "} Tox_Event_Data;"
            , "tagged union Tox_Event {"
            , "  tag field: type"
            , "  union field: data"
            , "  TOX_EVENT_FRIEND_MESSAGE => friend_message"
            , "};"
            ]

        checkInference
            [ "typedef enum Tox_Event_Type {"
            , "    TOX_EVENT_FRIEND_MESSAGE"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "    struct Tox_Event_Friend_Message *friend_message;"
            , "} Tox_Event_Data;"
            , "struct Tox_Event {"
            , "    int extra_field;"
            , "    Tox_Event_Type type;"
            , "    Tox_Event_Data data;"
            , "};"
            ]
            [ "typedef enum Tox_Event_Type {"
            , "  TOX_EVENT_FRIEND_MESSAGE,"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "  struct Tox_Event_Friend_Message* friend_message;"
            , "} Tox_Event_Data;"
            , "struct Tox_Event {"
            , "  int extra_field;"
            , "  Tox_Event_Type type;"
            , "  Tox_Event_Data data;"
            , "};"
            ]

    describe "TaggedUnion match inference" $ do
        checkInference
            [ "typedef enum Tox_Event_Type {"
            , "    TOX_EVENT_FRIEND_MESSAGE"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "    struct Tox_Event_Friend_Message *friend_message;"
            , "} Tox_Event_Data;"
            , "struct Tox_Event {"
            , "    Tox_Event_Type type;"
            , "    Tox_Event_Data data;"
            , "};"
            , "void handle_event_direct(struct Tox_Event event) {"
            , "    switch (event.type) {"
            , "        case TOX_EVENT_FRIEND_MESSAGE: {"
            , "            break;"
            , "        }"
            , "    }"
            , "}"
            ]
            [ "typedef enum Tox_Event_Type {"
            , "  TOX_EVENT_FRIEND_MESSAGE,"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "  struct Tox_Event_Friend_Message* friend_message;"
            , "} Tox_Event_Data;"
            , "tagged union Tox_Event {"
            , "  tag field: type"
            , "  union field: data"
            , "  TOX_EVENT_FRIEND_MESSAGE => friend_message"
            , "};"
            , "void handle_event_direct(struct Tox_Event event) {"
            , "  match event {"
            , "    TOX_EVENT_FRIEND_MESSAGE => {"
            , ""
            , "    }"
            , "  }"
            , "}"
            ]

        checkInference
            [ "typedef enum Tox_Event_Type {"
            , "    TOX_EVENT_FRIEND_MESSAGE"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "    struct Tox_Event_Friend_Message *friend_message;"
            , "} Tox_Event_Data;"
            , "struct Tox_Event {"
            , "    Tox_Event_Type type;"
            , "    Tox_Event_Data data;"
            , "};"
            , "void handle_event(const struct Tox_Event *event) {"
            , "    switch (event->type) {"
            , "        case TOX_EVENT_FRIEND_MESSAGE: {"
            , "            break;"
            , "        }"
            , "    }"
            , "}"
            ]
            [ "typedef enum Tox_Event_Type {"
            , "  TOX_EVENT_FRIEND_MESSAGE,"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "  struct Tox_Event_Friend_Message* friend_message;"
            , "} Tox_Event_Data;"
            , "tagged union Tox_Event {"
            , "  tag field: type"
            , "  union field: data"
            , "  TOX_EVENT_FRIEND_MESSAGE => friend_message"
            , "};"
            , "void handle_event(struct Tox_Event const* event) {"
            , "  match event {"
            , "    TOX_EVENT_FRIEND_MESSAGE => {"
            , ""
            , "    }"
            , "  }"
            , "}"
            ]

        checkInference
            [ "typedef enum TCP_Proxy_Type {"
            , "    TCP_PROXY_HTTP"
            , "} TCP_Proxy_Type;"
            , "struct IP_Port { int x; };"
            , "struct TCP_Proxy_Info {"
            , "    struct IP_Port ip_port;"
            , "    TCP_Proxy_Type proxy_type;"
            , "};"
            , "void handle_proxy(struct TCP_Proxy_Info *proxy_info) {"
            , "    switch (proxy_info->proxy_type) {"
            , "        case TCP_PROXY_HTTP: break;"
            , "    }"
            , "}"
            ]
            [ "typedef enum TCP_Proxy_Type {"
            , "  TCP_PROXY_HTTP,"
            , "} TCP_Proxy_Type;"
            , "struct IP_Port {"
            , "  int x;"
            , "};"
            , "struct TCP_Proxy_Info {"
            , "  struct IP_Port ip_port;"
            , "  TCP_Proxy_Type proxy_type;"
            , "};"
            , "void handle_proxy(struct TCP_Proxy_Info* proxy_info) {"
            , "  switch (proxy_info->proxy_type) {"
            , "    case TCP_PROXY_HTTP: break;"
            , "  }"
            , "}"
            ]

        checkInference
            [ "typedef enum Tox_Event_Type {"
            , "    TOX_EVENT_FRIEND_MESSAGE"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "    struct Tox_Event_Friend_Message *friend_message;"
            , "} Tox_Event_Data;"
            , "struct Tox_Event {"
            , "    Tox_Event_Type type;"
            , "    Tox_Event_Data data;"
            , "};"
            , "void handle_event(const struct Tox_Event *event) {"
            , "    switch (event->type) {"
            , "        case TOX_EVENT_FRIEND_MESSAGE: {"
            , "            handle_message(event->data.friend_message);"
            , "            break;"
            , "        }"
            , "        default: {"
            , "            break;"
            , "        }"
            , "    }"
            , "}"
            ]
            [ "typedef enum Tox_Event_Type {"
            , "  TOX_EVENT_FRIEND_MESSAGE,"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "  struct Tox_Event_Friend_Message* friend_message;"
            , "} Tox_Event_Data;"
            , "tagged union Tox_Event {"
            , "  tag field: type"
            , "  union field: data"
            , "  TOX_EVENT_FRIEND_MESSAGE => friend_message"
            , "};"
            , "void handle_event(struct Tox_Event const* event) {"
            , "  match event {"
            , "    TOX_EVENT_FRIEND_MESSAGE => {"
            , "      handle_message(event.friend_message);"
            , "    }"
            , "    default => {"
            , ""
            , "    }"
            , "  }"
            , "}"
            ]

    describe "TaggedUnion construction" $ do
        checkRefactoring
            [ "typedef enum Tox_Event_Type {"
            , "    TOX_EVENT_FRIEND_MESSAGE"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "    struct Tox_Event_Friend_Message *friend_message;"
            , "} Tox_Event_Data;"
            , "struct Tox_Event {"
            , "    Tox_Event_Type type;"
            , "    Tox_Event_Data data;"
            , "};"
            , "void f(struct Tox_Event_Friend_Message *msg) {"
            , "    Tox_Event event;"
            , "    event.type = TOX_EVENT_FRIEND_MESSAGE;"
            , "    event.data.friend_message = msg;"
            , "}"
            ]
            [ "typedef enum Tox_Event_Type {"
            , "  TOX_EVENT_FRIEND_MESSAGE,"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "  struct Tox_Event_Friend_Message* friend_message;"
            , "} Tox_Event_Data;"
            , "tagged union Tox_Event {"
            , "  tag field: type"
            , "  union field: data"
            , "  TOX_EVENT_FRIEND_MESSAGE => friend_message"
            , "};"
            , "void f(struct Tox_Event_Friend_Message* msg) {"
            , "  Tox_Event event;"
            , ""
            , "  event.type = TOX_EVENT_FRIEND_MESSAGE <= msg;"
            , "}"
            ]
            [ "typedef enum Tox_Event_Type {"
            , "    TOX_EVENT_FRIEND_MESSAGE"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "    struct Tox_Event_Friend_Message *friend_message;"
            , "} Tox_Event_Data;"
            , "struct Tox_Event {"
            , "    Tox_Event_Type type;"
            , "    Tox_Event_Data data;"
            , "};"
            , "void f(struct Tox_Event_Friend_Message *msg) {"
            , "    Tox_Event event;"
            , "    event.type = TOX_EVENT_FRIEND_MESSAGE;"
            , "    event.data.friend_message = msg;"
            , "}"
            ]

        checkInference
            [ "typedef enum Tox_Event_Type {"
            , "    TOX_EVENT_FRIEND_MESSAGE"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "    struct Tox_Event_Friend_Message *friend_message;"
            , "} Tox_Event_Data;"
            , "struct Tox_Event {"
            , "    Tox_Event_Type type;"
            , "    Tox_Event_Data data;"
            , "};"
            , "void f(struct Tox_Event_Friend_Message *msg) {"
            , "    Tox_Event event;"
            , "    event.type = TOX_EVENT_FRIEND_MESSAGE;"
            , "    event.data.friend_message = msg;"
            , "}"
            ]
            [ "typedef enum Tox_Event_Type {"
            , "  TOX_EVENT_FRIEND_MESSAGE,"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "  struct Tox_Event_Friend_Message* friend_message;"
            , "} Tox_Event_Data;"
            , "tagged union Tox_Event {"
            , "  tag field: type"
            , "  union field: data"
            , "  TOX_EVENT_FRIEND_MESSAGE => friend_message"
            , "};"
            , "void f(struct Tox_Event_Friend_Message* msg) {"
            , "  Tox_Event event;"
            , ""
            , "  event.type = TOX_EVENT_FRIEND_MESSAGE <= msg;"
            , "}"
            ]

    describe "TaggedUnionGet inference" $ do
        checkInference
            [ "typedef enum Tox_Event_Type {"
            , "    TOX_EVENT_FRIEND_MESSAGE"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "    struct Tox_Event_Friend_Message *friend_message;"
            , "} Tox_Event_Data;"
            , "struct Tox_Event {"
            , "    Tox_Event_Type type;"
            , "    Tox_Event_Data data;"
            , "};"
            , "const Tox_Event_Friend_Message *tox_event_get_friend_message_direct(struct Tox_Event event) {"
            , "    return event.type == TOX_EVENT_FRIEND_MESSAGE ? event.data.friend_message : nullptr;"
            , "}"
            ]
            [ "typedef enum Tox_Event_Type {"
            , "  TOX_EVENT_FRIEND_MESSAGE,"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "  struct Tox_Event_Friend_Message* friend_message;"
            , "} Tox_Event_Data;"
            , "tagged union Tox_Event {"
            , "  tag field: type"
            , "  union field: data"
            , "  TOX_EVENT_FRIEND_MESSAGE => friend_message"
            , "};"
            , "get event.type == ? event.friend_message"
            ]

        checkInference
            [ "typedef enum Tox_Event_Type {"
            , "    TOX_EVENT_FRIEND_MESSAGE"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "    struct Tox_Event_Friend_Message *friend_message;"
            , "} Tox_Event_Data;"
            , "struct Tox_Event {"
            , "    Tox_Event_Type type;"
            , "    Tox_Event_Data data;"
            , "};"
            , "const Tox_Event_Friend_Message *tox_event_get_friend_message(const struct Tox_Event *event) {"
            , "    return event->type == TOX_EVENT_FRIEND_MESSAGE ? event->data.friend_message : nullptr;"
            , "}"
            ]
            [ "typedef enum Tox_Event_Type {"
            , "  TOX_EVENT_FRIEND_MESSAGE,"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "  struct Tox_Event_Friend_Message* friend_message;"
            , "} Tox_Event_Data;"
            , "tagged union Tox_Event {"
            , "  tag field: type"
            , "  union field: data"
            , "  TOX_EVENT_FRIEND_MESSAGE => friend_message"
            , "};"
            , "get event.type == ? event.friend_message"
            ]

        checkInference
            [ "typedef enum Tox_Event_Type {"
            , "    TOX_EVENT_FRIEND_MESSAGE"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "    struct Tox_Event_Friend_Message *friend_message;"
            , "} Tox_Event_Data;"
            , "struct Tox_Event {"
            , "    Tox_Event_Type type;"
            , "    Tox_Event_Data data;"
            , "};"
            , "bool tox_event_pack(const struct Tox_Event *event) {"
            , "    switch (event->type) {"
            , "        case TOX_EVENT_FRIEND_MESSAGE:"
            , "            return pack_msg(event->data.friend_message);"
            , "        default:"
            , "            return false;"
            , "    }"
            , "}"
            ]
            [ "typedef enum Tox_Event_Type {"
            , "  TOX_EVENT_FRIEND_MESSAGE,"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "  struct Tox_Event_Friend_Message* friend_message;"
            , "} Tox_Event_Data;"
            , "tagged union Tox_Event {"
            , "  tag field: type"
            , "  union field: data"
            , "  TOX_EVENT_FRIEND_MESSAGE => friend_message"
            , "};"
            , "bool tox_event_pack(struct Tox_Event const* event) {"
            , "  match event {"
            , "    TOX_EVENT_FRIEND_MESSAGE => {"
            , "      return pack_msg(event.friend_message);"
            , "    }"
            , "    default => {"
            , "      return false;"
            , "    }"
            , "  }"
            , "}"
            ]

    describe "TaggedUnionGetTag inference" $ do
        checkInference
            [ "typedef enum Tox_Event_Type {"
            , "    TOX_EVENT_FRIEND_MESSAGE"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "    struct Tox_Event_Friend_Message *friend_message;"
            , "} Tox_Event_Data;"
            , "struct Tox_Event {"
            , "    Tox_Event_Type type;"
            , "    Tox_Event_Data data;"
            , "};"
            , "Tox_Event_Type tox_event_get_type(const struct Tox_Event *event) {"
            , "    return event->type;"
            , "}"
            ]
            [ "typedef enum Tox_Event_Type {"
            , "  TOX_EVENT_FRIEND_MESSAGE,"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "  struct Tox_Event_Friend_Message* friend_message;"
            , "} Tox_Event_Data;"
            , "tagged union Tox_Event {"
            , "  tag field: type"
            , "  union field: data"
            , "  TOX_EVENT_FRIEND_MESSAGE => friend_message"
            , "};"
            , "get tag event->type"
            ]

        checkInference
            [ "typedef enum Tox_Event_Type {"
            , "    TOX_EVENT_FRIEND_MESSAGE"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "    struct Tox_Event_Friend_Message *friend_message;"
            , "} Tox_Event_Data;"
            , "struct Tox_Event {"
            , "    Tox_Event_Type type;"
            , "    Tox_Event_Data data;"
            , "};"
            , "void handle_event(const struct Tox_Event *event) {"
            , "    switch (event->type) {"
            , "        case TOX_EVENT_FRIEND_MESSAGE: {"
            , "            handle_message(event->data.friend_message);"
            , "            log_message(event->data.friend_message);"
            , "            break;"
            , "        }"
            , "    }"
            , "}"
            ]
            [ "typedef enum Tox_Event_Type {"
            , "  TOX_EVENT_FRIEND_MESSAGE,"
            , "} Tox_Event_Type;"
            , "typedef union Tox_Event_Data {"
            , "  struct Tox_Event_Friend_Message* friend_message;"
            , "} Tox_Event_Data;"
            , "tagged union Tox_Event {"
            , "  tag field: type"
            , "  union field: data"
            , "  TOX_EVENT_FRIEND_MESSAGE => friend_message"
            , "};"
            , "void handle_event(struct Tox_Event const* event) {"
            , "  match event {"
            , "    TOX_EVENT_FRIEND_MESSAGE => {"
            , "      handle_message(event.friend_message);"
            , ""
            , "      log_message(event.friend_message);"
            , "    }"
            , "  }"
            , "}"
            ]

    describe "TaggedUnion low-level access diagnostics" $ do
        it "issues a diagnostic for wrong member access in match" $ do
            prog <- mustParse
                [ "typedef enum Tox_Event_Type {"
                , "    TOX_EVENT_FRIEND_MESSAGE,"
                , "    TOX_EVENT_FRIEND_REQUEST"
                , "} Tox_Event_Type;"
                , "typedef union Tox_Event_Data {"
                , "    struct Tox_Event_Friend_Message *friend_message;"
                , "    struct Tox_Event_Friend_Request *friend_request;"
                , "} Tox_Event_Data;"
                , "struct Tox_Event {"
                , "    Tox_Event_Type type;"
                , "    Tox_Event_Data data;"
                , "};"
                , "void handle_event(const struct Tox_Event *event) {"
                , "    switch (event->type) {"
                , "        case TOX_EVENT_FRIEND_MESSAGE: {"
                , "            handle_request(event->data.friend_request);"
                , "            break;"
                , "        }"
                , "    }"
                , "}"
                ]
            let (_, diags) = inferProgram prog
            diags `shouldBe` ["test.c:16: in function 'handle_event': Unrecognized high-level access to tagged union 'Tox_Event' field 'friend_request'"]
        it "issues a diagnostic for raw member access" $ do
            prog <- mustParse
                [ "typedef enum Tox_Event_Type {"
                , "    TOX_EVENT_FRIEND_MESSAGE"
                , "} Tox_Event_Type;"
                , "typedef union Tox_Event_Data {"
                , "    struct Tox_Event_Friend_Message *friend_message;"
                , "} Tox_Event_Data;"
                , "struct Tox_Event {"
                , "    Tox_Event_Type type;"
                , "    Tox_Event_Data data;"
                , "};"
                , "void f(struct Tox_Event *event) {"
                , "    int x = event->type;"
                , "}"
                ]
            let (_, diags) = inferProgram prog
            diags `shouldBe` ["test.c:12: in function 'f': Unrecognized low-level access to tagged union 'Tox_Event' field 'type'"]

        it "does not issue a diagnostic for non-tagged union IP" $ do
            prog <- mustParse
                [ "typedef struct Family { uint8_t value; } Family;"
                , "typedef union IP_Union { int v4; int v6; } IP_Union;"
                , "struct IP {"
                , "    Family family;"
                , "    IP_Union ip;"
                , "};"
                , "void f(struct IP *ip) {"
                , "    int x = ip->family;"
                , "}"
                ]
            let (_, diags) = inferProgram prog
            diags `shouldBe` []

-- end of tests
