{- |
Module      : IMP.Util
Description : Utility functions for the IMP language parser
Copyright   : (c) Basil Feitknecht, 2025
License     : MIT
Maintainer  : bfeitknecht@ethz.ch
Stability   : stable
Portability : portable

This module provides utility functions and definitions used in the
IMP parser. It includes token parsers, reserved keywords, operators,
and helper functions for parsing identifiers, symbols, and whitespace.
-}
module IMP.Util where

import Text.Parsec
import Text.Parsec.Language (emptyDef)
import Text.Parsec.String (Parser)

import qualified Text.Parsec.Token as Tok

-- | Token parser for IMP, including operators and reserved keywords.
lexer :: Tok.TokenParser ()
lexer = Tok.makeTokenParser style
    where
        ops =
            [ "+"
            , "-"
            , "*"
            , "/"
            , "%"
            , "="
            , "#"
            , "<"
            , "<="
            , ">"
            , ">="
            , "not"
            , "and"
            , "or"
            , ":="
            , "+="
            , "-="
            , "*="
            , "/="
            , "%="
            , "par"
            , "[]"
            ]
        keywords =
            [ "skip"
            , "if"
            , "then"
            , "else"
            , "end"
            , "while"
            , "do"
            , "true"
            , "false"
            , "print"
            , "read"
            , "var"
            , "in"
            , "procedure"
            , "begin"
            , "break"
            , "for"
            , "to"
            , "repeat"
            , "until"
            , "time"
            , "match"
            , "on"
            , "default"
            , "havoc"
            , "assert"
            , "flip"
            , "flop"
            , "raise"
            , "try"
            , "catch"
            , "with"
            , "swap"
            ]
        style =
            emptyDef
                { Tok.commentLine = "//"
                , Tok.commentStart = "/*"
                , Tok.commentEnd = "*/"
                , Tok.reservedOpNames = ops
                , Tok.reservedNames = keywords
                , Tok.identStart = letter
                , Tok.identLetter = alphaNum
                }

-- | Parses an identifier (variable or procedure name).
identifier :: Parser String
identifier = Tok.identifier lexer

-- | Parses a variable name or the placeholder '_'.
variable :: Parser String
variable = identifier <|> symbol "_"

-- | Parses a reserved keyword.
reserved :: String -> Parser ()
reserved = Tok.reserved lexer

-- | Parses a reserved operator.
operator :: String -> Parser ()
operator = Tok.reservedOp lexer

-- | Parses parentheses around another parser.
parens :: Parser a -> Parser a
parens = Tok.parens lexer

-- | Parses an integer literal.
integer :: Parser Integer
integer = Tok.integer lexer

-- | Parses semicolon.
semi :: Parser String
semi = Tok.semi lexer

-- | Parses whitespace.
whitespace :: Parser ()
whitespace = Tok.whiteSpace lexer

-- | Parses a symbol.
symbol :: String -> Parser String
symbol = Tok.symbol lexer
