-- SPDX-FileCopyrightText: 2020 Tocqueville Group
--
-- SPDX-License-Identifier: LicenseRef-MIT-TQ

-- | File Generation machinery that resembles @summoner@ library.
module FileGen
  ( run
  ) where

import System.Directory (setCurrentDirectory)
import Summoner.Tree (TreeFs(..), showBoldTree, traverseTree)

import qualified FileGen.Files as F

run :: Text -> IO ()
run projectName =
  createProjectDirectoryCustom (CustomSettings projectName)

-- | From the given 'CustomSettings' creates the project.
createProjectDirectoryCustom :: CustomSettings -> IO ()
createProjectDirectoryCustom settings@CustomSettings{..} = do
  let tree = createProjectTemplateCustom settings
  traverseTree tree
  putTextLn "The project with the following structure has been created:"
  putTextLn $ showBoldTree tree
  setCurrentDirectory (toString sProjectName)

-- | Creating tree structure of the project.
createProjectTemplateCustom :: CustomSettings -> TreeFs
createProjectTemplateCustom settings@CustomSettings{..} =
  Dir (toString sProjectName) $ generateFiles settings

-- | Files Template
generateFiles :: CustomSettings -> [TreeFs]
generateFiles CustomSettings{..} =
  [ Dir "app"
    [ File "Main.hs" F.main ]
  , Dir "src"
    [ File "Basic.hs" F.basic ]
  , Dir "test"
    [ Dir "Test"
        [ File "Basic.hs" F.basicTest ]
    , File "Main.hs" F.mainTest
    ]
  , File "ChangeLog.md" (F.changelog sProjectName)
  , File "README.md" (F.readme sProjectName)
  , File ".gitignore" F.gitignore
  , File "package.yaml" (F.packageYaml sProjectName)
  , File "stack.yaml" F.stackYaml
  ]

data CustomSettings = CustomSettings
  { sProjectName :: Text
  }
