{-# LANGUAGE Haskell2010
 #-}

module Main where

import Options
import Haskell2Java

import Foreign.Java.Bindings

import System.Console.GetOpt
import System.Environment (getArgs, getProgName)

import Data.Maybe
import Data.NamedRecord
import Data.List
import Data.List.Split
import Data.Word

import Text.Printf

versionInfo = "Java <-> Haskell Bindings Generator"

-- | The flags (needed for the options list / GetOpt).
data Flag = Help
          | Version
          | Classpath [String]
          | SearchDepth Word32
          | TargetDirectory String
          | CompleteSE6
          | Verbose

-- | The Command line options as fed to the GetOpt parser.
options = [
    Option "h" ["help"] (NoArg Help)
        "Print this help text.",

    Option "V" ["version"] (NoArg Version)
        "Print version information.",

    Option "c" ["classpath"] (ReqArg readClasspath "")
        "The Java classpath to be used.",

    Option "d" ["search-depth"] (ReqArg readSearchDepth "")
        "The maximum search depth for dependent Java classes.",

    Option "v" ["--verbose"] (NoArg Verbose)
        "Verbose output, prints extra diagnostic information.",

    Option "X" [] (NoArg CompleteSE6)
        "Completely translate Java SE 6"

 ] where
    readClasspath = Classpath . splitOneOf ":;,"
    readSearchDepth = SearchDepth . read

parseArgs :: IO (Either (Options, [String]) [String])
-- ^ Parse command line arguments with the help of the GetOpt library.
parseArgs = do
    (opts, args, errs) <- getArgs >>= return . getOpt Permute options
    if null errs
        then return $ Left  (parseOpts newOptions opts, args)
        else return $ Right errs
  where
    parseOpts opts [] = opts
    parseOpts opts (x:xs) = flip parseOpts xs $
        case x of
            Help ->          opts `set` optShowHelp    := True
            Version ->       opts `set` optShowVersion := True
            Verbose ->       opts `set` optVerbose     := True
            CompleteSE6 ->   opts `set` optCompleteSE6 := True
            Classpath cp ->  opts `set` optClasspath   := cp
            SearchDepth d -> opts `set` optSearchDepth := d

main :: IO ()
-- ^ main: parseArgs and either 'run' or show errors.
main = parseArgs >>= either (uncurry run) (mapM_ putStr)

run :: Options -> [String] -> IO ()
-- ^ The actual main function.
run opts args
    | opts `get` optShowHelp = do
        progName <- getProgName
        putStrLn (usageInfo ((\_ -> let { __ = 
{-# LINE 84 "Main.hss" #-}
concat ["", (versionInfo), "\n\nUsage: ", (progName), " [options] command [files...]\n\nUse `", (progName), " help' for information about commands.\n"]
{-# LINE 90 "Main.hss" #-}
} in __) undefined) options)

    | opts `get` optShowVersion = putStrLn versionInfo

    | otherwise = getProgName >>= \progName -> putStr ((\_ -> let { __ = 
{-# LINE 94 "Main.hss" #-}
concat ["No command given. Use `", (progName), " help' for help.\n"]
{-# LINE 96 "Main.hss" #-}
} in __) undefined)

