# jsonpatch

[![Hackage](https://img.shields.io/hackage/v/jsonpatch.svg?style=flat)](https://hackage.haskell.org/package/jsonpatch)
[![Stackage Nightly](http://stackage.org/package/jsonpatch/badge/nightly)](http://stackage.org/nightly/package/jsonpatch)
[![Stackage LTS](http://stackage.org/package/jsonpatch/badge/lts)](http://stackage.org/lts/package/jsonpatch)
[![CI](https://github.com/pbrisbin/jsonpatch/actions/workflows/ci.yml/badge.svg)](https://github.com/pbrisbin/jsonpatch/actions/workflows/ci.yml)

Haskell package for parsing and applying [JSON Patches][jsonpatch].

[jsonpatch]: https://jsonpatch.com/

## Example

<!--

```haskell
{-# OPTIONS_GHC -Wno-incomplete-patterns #-}
{-# OPTIONS_GHC -Wno-incomplete-uni-patterns #-}

module Main (main) where
import Prelude
import Text.Markdown.Unlit ()
```

-->

Typical use cases need only one import:

```haskell
import Data.JSON.Patch
```

Our example will make use of a few more libraries:


```haskell
import Control.Exception (displayException)
import Data.Aeson (Value, Result(..), fromJSON)
import Data.Aeson.Encode.Pretty
import Data.Aeson.QQ (aesonQQ)
import Data.ByteString.Lazy qualified as BSL
```

The `FromJSON` instance can be used to build a `[Patch]`:

```haskell
patch :: [Patch]
patch = fromResult $ fromJSON [aesonQQ|
  [
    { "op": "replace", "path": "/baz", "value": "boo" },
    { "op": "add", "path": "/hello", "value": ["world"] },
    { "op": "remove", "path": "/foo" }
  ]
|]


-- | Unsafe unwrapping for the sake of example
fromResult :: Result a -> a
fromResult (Success a) = a
```

The patches can then be applied to a document:

```haskell
result :: Either PatchError Value
result = applyPatches patch [aesonQQ|
  {
    "baz": "qux",
    "foo": "bar"
  }
|]
```

The result is in `Either PatchError`, with `displayException` available to get
a user-friendly message.

```haskell
main :: IO ()
main = either (fail . displayException) (BSL.putStr . encodePretty) result
```

The above program outputs:

```json
{
  "baz": "boo",
  "hello": ["world"]
}
```


## Quality

The full test suite from [`json-patch/json-patch-tests`][json-patch-tests]
passes. However, some error cases have poor (or misleading) error messages at
this time.

[json-patch-tests]: https://github.com/json-patch/json-patch-tests

## License

This package is licensed AGPLv3. See [COPYING](./COPYING).
