module KB.Text.Shape.Font
 ( extractBlob

 , withFont
 , Handles.Font

 , getFontInfo
 , Structs.FontInfo

 , withLoader
 , Loader(..)
 , loadFont
 , LoadFontResult(..)
 , placeBlobThen
 ) where

import Prelude hiding (id)
import Foreign
import Foreign.C

import Control.Monad (when)
import Data.ByteString (ByteString)
import Data.ByteString qualified as ByteString
import Data.ByteString.Unsafe qualified as ByteString
import Data.Text (Text)
import Data.Text.Foreign qualified as Text

import KB.Text.Shape.FFI.API.Direct qualified as ShapeDirect
import KB.Text.Shape.FFI.Enums qualified as Enums
import KB.Text.Shape.FFI.Handles qualified as Handles
import KB.Text.Shape.FFI.Structs qualified as Structs

-- | Extract and pre-process font data needed for shaping.
extractBlob :: ByteString -> Int -> IO ByteString
extractBlob fontData fontIndex =
  withLoader \loader -> do
    loadFont fontData fontIndex loader >>= \case
      Left err ->
        error $ show err
      Right LoadFontNeedsBlob{scratchSize, outputSize} ->
        placeBlobThen loader scratchSize outputSize
          -- copy the blob data out before leaving the loader
          ByteString.packCStringLen
      Right LoadFontReady ->
        pure fontData

withFont :: ByteString -> Int -> (Handles.Font -> IO r) -> IO r
withFont fontData fontIndex action =
  withLoader \loader -> do
    loadFont fontData fontIndex loader >>= \case
      Left err ->
        error $ show err
      Right LoadFontNeedsBlob{scratchSize, outputSize} ->
        placeBlobThen loader scratchSize outputSize \_blob -> pure ()
      Right LoadFontReady ->
        pure ()
    action loader.font

getFontInfo :: Handles.Font -> IO Info
getFontInfo font =
  alloca \fontInfoPtr -> do
    ShapeDirect.kbts_GetFontInfo font fontInfoPtr
    Structs.FontInfo{strings=stringPtrs, ..} <- peek fontInfoPtr
    strings <- loadStrings 0 stringPtrs stringLengths
    pure Info{..}
  where
    loadStrings ix strPtr lenPtr = do
      if strPtr == nullPtr || ix + 1 >= Enums.FONT_INFO_STRING_ID_COUNT then
        pure []
      else do
        strElemPtr <- peek strPtr
        lenElem <- peek lenPtr
        str <- Text.peekCStringLen (strElemPtr, fromIntegral lenElem)
        (str :) <$> loadStrings (ix + 1) (advancePtr strPtr 1) (advancePtr lenPtr 1)

data Info = Info
  { strings :: [Text]
  , styleFlags :: Enums.FontStyleFlags
  , weight :: Enums.FontWeight
  , width :: Enums.FontWidth
  }
  deriving (Eq, Show)

data Loader = Loader
  { font :: Handles.Font
  , state :: Ptr ShapeDirect.LoadFontState
  } deriving (Show)

withLoader :: (Loader -> IO a) -> IO a
withLoader action =
  alloca \fontPtr ->
    alloca \statePtr -> do
      fillBytes fontPtr 0x00 $ sizeOf (undefined :: Handles.Font)
      fillBytes statePtr 0x00 $ sizeOf (undefined :: ShapeDirect.LoadFontState)
      action Loader
        { font = Handles.Font fontPtr
        , state = statePtr
        }

data LoadFontResult
  = LoadFontReady
  | LoadFontNeedsBlob { scratchSize :: Int, outputSize :: Int}

loadFont
  :: ByteString
  -> Int
  -> Loader
  -> IO (Either Enums.LoadFontError LoadFontResult)
loadFont ttfData fontIndex loader =
  alloca \scratchSizePtr ->
    alloca \outputSizePtr ->
      ByteString.unsafeUseAsCStringLen ttfData \(ttfDataPtr, ttfDataSize) -> do
        err <- ShapeDirect.kbts_LoadFont
          loader.font
          loader.state
          (castPtr ttfDataPtr)
          (fromIntegral ttfDataSize)
          (fromIntegral fontIndex)
          scratchSizePtr
          outputSizePtr
        case err of
          Enums.LOAD_FONT_ERROR_NONE ->
            pure $ Right LoadFontReady
          Enums.LOAD_FONT_ERROR_NEED_TO_CREATE_BLOB -> do
            scratchSize <- peek scratchSizePtr
            outputSize <- peek outputSizePtr
            pure $ Right LoadFontNeedsBlob{..}
          _ ->
            pure $ Left err

placeBlobThen :: Loader -> Int -> Int -> (CStringLen -> IO r) -> IO r
placeBlobThen loader scratchSize outputSize next =
  allocaBytes scratchSize \scratchPtr ->
    allocaBytes outputSize \outputPtr -> do
      err <- ShapeDirect.kbts_PlaceBlob loader.font loader.state scratchPtr outputPtr
      when (err /= Enums.LOAD_FONT_ERROR_NONE) $
        error $ show err
      next (castPtr outputPtr, outputSize)
