module Lazy.Scope.Scoped where

import Data.ByteString.Lazy qualified as L
import Data.ByteString.Lazy.Char8 qualified as L8
import Lazy.Scope.Type
import Relude hiding (Handle)

unScope :: (NFData a, Monad m) => Scoped s a -> LazyT s m a
unScope (Scoped !a) =
  case rnf a of
    () -> pure a

bs2Scoped :: (LByteString -> a) -> Bs s -> Scoped s a
bs2Scoped f = fmap f

scoped2Bs :: (a -> LByteString) -> Scoped s a -> Bs s
scoped2Bs f = fmap f

toBs :: LByteString -> Bs s
toBs = Scoped
{-# INLINE toBs #-}

condM :: [(Scoped s Bool, m a)] -> m a -> m a
condM [] o = o
condM ((Scoped True, a):_) _ = a
condM ((Scoped False, _):t) o = condM t o

empty :: Bs s
empty = pure L.empty

singleton :: W8 s -> Bs s
singleton = pure . L.singleton . unScoped

pack :: [W8 s] -> Bs s
pack = pure . L.pack . fmap unScoped

cons :: W8 s -> Bs s -> Bs s
cons = liftA2 L.cons

snoc :: Bs s -> W8 s -> Bs s
snoc = liftA2 L.snoc

uncons :: Bs s -> Scoped s (Maybe (Word8, LByteString))
uncons = fmap L.uncons

unsnoc :: Bs s -> Scoped s (Maybe (LByteString, Word8))
unsnoc = fmap L.unsnoc

null :: Bs s -> B s
null = fmap L.null

reverse :: Bs s -> Bs s
reverse =  fmap L.reverse

intersperse :: W8 s -> Bs s -> Bs s
intersperse = liftA2 L.intersperse

intercalate :: Bs s -> [Bs s] -> Bs s
intercalate (Scoped bs) = pure . L.intercalate bs . fmap unScoped

transpose :: [Bs s] -> [Bs s]
transpose = fmap pure . L.transpose . fmap unScoped

foldl' :: (Scoped s a -> Word8 -> Scoped s a) -> Scoped s a -> Bs s -> Scoped s a
foldl' f (Scoped b) = fmap (L.foldl' (\a i -> unScoped $ f (pure a) i) b)

foldr' :: (Word8 -> Scoped s a -> Scoped s a) -> Scoped s a -> Bs s -> Scoped s a
foldr' f (Scoped b) = fmap (L.foldr' (\i a -> unScoped $ f i (pure a)) b)

concat :: [Bs s] -> Bs s
concat = pure . L.concat . fmap unScoped

concatMap :: (Word8 -> LByteString) -> Bs s -> Bs s
concatMap f = fmap (L.concatMap f)

any :: (Word8 -> Bool) -> Bs s -> B s
any p = fmap (L.any p)

all :: (Word8 -> Bool) -> Bs s -> B s
all p = fmap (L.all p)

maximum :: HasCallStack => Bs s -> W8 s
maximum = fmap L.maximum

minimum :: HasCallStack => Bs s -> W8 s
minimum = fmap L.minimum

compareLength :: Bs s -> I64 s -> Scoped s Ordering
compareLength = liftA2 L.compareLength

scanl :: (W8 s -> W8 s -> W8 s) -> W8 s -> Bs s -> Bs s
scanl f i = fmap (L.scanl (\a e -> unScoped $ f (pure a) (pure e)) $ unScoped i)

scanl1 :: (W8 s -> W8 s -> W8 s) -> Bs s -> Bs s
scanl1 f = fmap (L.scanl1 (\a e -> unScoped $ f (pure a) (pure e)))
scanr :: (W8 s -> W8 s -> W8 s) -> W8 s -> Bs s -> Bs s
scanr f i = fmap (L.scanr (\a e -> unScoped $ f (pure a) (pure e)) $ unScoped i)
scanr1 :: (W8 s -> W8 s -> W8 s) -> Bs s -> Bs s
scanr1 f = fmap (L.scanr1 (\a e -> unScoped $ f (pure a) (pure e)))

mapAccumL :: (acc -> W8 s -> (acc, W8 s)) -> acc -> Bs s -> (acc, Bs s)
mapAccumL f a (Scoped lbs) =
  case L.mapAccumL (\acc (b :: Word8) -> mapSnd unScoped $ f acc (pure b)) a lbs of
    (acc, lbs') -> (acc, toBs lbs')

mapAccumR :: (acc -> W8 s -> (acc, W8 s)) -> acc -> Bs s -> (acc, Bs s)
mapAccumR f a (Scoped lbs) =
  case L.mapAccumR (\acc b -> mapSnd unScoped $ f acc (pure b)) a lbs of
    (acc, lbs') -> (acc, toBs lbs')

repeat :: W8 s -> Bs s
repeat = fmap L.repeat
replicate :: I64 s -> W8 s -> Bs s
replicate = liftA2 L.replicate
cycle :: HasCallStack => Bs s -> Bs s
cycle = fmap L.cycle
iterate :: (W8 s -> W8 s) -> W8 s -> Bs s
iterate f (Scoped i) = toBs $ L.iterate (unScoped . f . pure) i

unfoldr :: (a -> Maybe (W8 s, a)) -> a -> Bs s
unfoldr f = toBs . L.unfoldr (\a -> fmap (mapFst unScoped) $ f a)

isPrefixOf :: Bs s -> Bs s -> B s
isPrefixOf = liftA2 L.isPrefixOf

isSuffixOf :: Bs s -> Bs s -> B s
isSuffixOf = liftA2 L.isSuffixOf

elem :: W8 s -> Bs s -> B s
elem = liftA2 L.elem

notElem :: W8 s -> Bs s -> B s
notElem = liftA2 L.notElem

find :: (Word8 -> Bool) -> Bs s -> Scoped s (Maybe Word8)
find f = fmap (L.find f)

filter :: (Word8 -> Bool) -> Bs s -> Bs s
filter f = fmap (L.filter f)
partition :: (Word8 -> Bool) -> Bs s -> (Bs s, Bs s)
partition f =  both toBs . L.partition f . unScoped
index :: HasCallStack => Bs s -> I64 s -> W8 s
index = liftA2 L.index
indexMaybe :: Bs s -> I64 s -> Scoped s (Maybe Word8)
indexMaybe = liftA2 L.indexMaybe

(!?) :: Bs s -> I64 s -> Scoped s (Maybe Word8)
(!?)= liftA2 (L.!?)

elemIndex :: W8 s -> Bs s -> Scoped s (Maybe Int64)
elemIndex = liftA2 L.elemIndex
elemIndexEnd :: W8 s -> Bs s -> Scoped s (Maybe Int64)
elemIndexEnd = liftA2 L.elemIndexEnd
elemIndices :: W8 s -> Bs s -> Scoped s [Int64]
elemIndices = liftA2 L.elemIndices
findIndex :: (Word8 -> Bool) -> Bs s -> Scoped s (Maybe Int64)
findIndex p = fmap (L.findIndex p)
findIndexEnd :: (Word8 -> Bool) -> Bs s -> Scoped s (Maybe Int64)
findIndexEnd p = fmap (L.findIndexEnd p)
findIndices :: (Word8 -> Bool) -> Bs s -> Scoped s [Int64]
findIndices p = fmap (L.findIndices p)
count :: W8 s -> Bs s -> I64 s
count = liftA2 L.count
zip :: Bs s -> Bs s -> Scoped s [(Word8, Word8)]
zip = liftA2 L.zip
zipWith :: (Word8 -> Word8 -> a) -> Bs s -> Bs s -> Scoped s [a]
zipWith f = liftA2 (L.zipWith f)
packZipWith :: (Word8 -> Word8 -> Word8) -> Bs s -> Bs s -> Bs s
packZipWith f = liftA2 (L.packZipWith f)
unzip :: Scoped s [(Word8, Word8)] -> (Bs s, Bs s)
unzip = both toBs . L.unzip . unScoped
copy :: Bs s -> Bs s
copy = fmap L.copy

take :: Int64 -> Bs s -> Bs s
take n = pure . L.take n . unScoped
{-# INLINE take #-}

length :: Bs s -> I64 s
length = fmap L.length
{-# INLINE length #-}

drop :: I64 s -> Bs s -> Bs s
drop (Scoped n) = pure . L.drop n . unScoped
{-# INLINE drop #-}

dropEnd :: I64 s -> Bs s -> Bs s
dropEnd (Scoped n) = pure . L.dropEnd n . unScoped
{-# INLINE dropEnd #-}

dropWhile :: (Word8 -> Bool) -> Bs s -> Bs s
dropWhile p = pure . L.dropWhile p . unScoped
{-# INLINE dropWhile #-}

dropWhileEnd :: (Word8 -> Bool) -> Bs s -> Bs s
dropWhileEnd p = pure . L.dropWhileEnd p . unScoped
{-# INLINE dropWhileEnd #-}

takeWhile :: (Word8 -> Bool) -> Bs s -> Bs s
takeWhile p = pure . L.takeWhile p . unScoped
{-# INLINE takeWhile #-}

takeWhileEnd :: (Word8 -> Bool) -> Bs s -> Bs s
takeWhileEnd p = pure . L.takeWhileEnd p . unScoped
{-# INLINE takeWhileEnd #-}

span :: (Word8 -> Bool) -> Bs s -> (Bs s, Bs s)
span p = both pure . L.span p . unScoped

spanEnd :: (Word8 -> Bool) -> Bs s -> (Bs s, Bs s)
spanEnd p = both pure . L.spanEnd p . unScoped

break :: (Word8 -> Bool) -> Bs s -> (Bs s, Bs s)
break p = both pure . L.break p . unScoped

breakEnd :: (Word8 -> Bool) -> Bs s -> (Bs s, Bs s)
breakEnd p = both pure . L.breakEnd p . unScoped

splitAt :: I64 s -> Bs s -> (Bs s, Bs s)
splitAt (Scoped n) = both pure . L.splitAt n . unScoped

mapLbs :: (LByteString -> LByteString) -> Bs s -> Bs s
mapLbs f = fmap f

unpack8 :: Monad m => Bs s -> LazyT s m String
unpack8 = pure . L8.unpack . unScoped

toLbs :: Monad m => Bs s -> LazyT s m LByteString
toLbs (Scoped a) =
  case rnf a of
    () -> pure a

group :: Bs s -> Scoped s [LByteString]
group = fmap L.group

groupBy :: (Word8 -> Word8 -> Bool) -> Bs s -> Scoped s [LByteString]
groupBy p = fmap (L.groupBy p)

inits :: Bs s -> Scoped s [LByteString]
inits = fmap L.inits
tails :: Bs s -> Scoped s [LByteString]
tails = fmap L.tails
initsNE :: Bs s -> Scoped s (NonEmpty LByteString)
initsNE = fmap L.initsNE
tailsNE :: Bs s -> Scoped s (NonEmpty LByteString)
tailsNE = fmap L.tailsNE
stripPrefix :: Bs s -> Bs s -> Scoped s (Maybe LByteString)
stripPrefix = liftA2 L.stripPrefix
stripSuffix :: Bs s -> Bs s -> Scoped s (Maybe LByteString)
stripSuffix = liftA2 L.stripSuffix
splitWith :: (Word8 -> Bool) -> Bs s -> Scoped s [LByteString]
splitWith p = fmap (L.splitWith p)
