module LessPass.System.Clipboard (copyToClipboard) where

import Control.Exception (IOException, catch)
import System.IO (hClose, hPutStr, hPutStrLn, stderr)
import System.Process

-- | Attempts to copy the given string to the clipoard.
--   This depends on certain system programs being available
--   and will print an error if they are not there.
copyToClipboard :: String -> IO ()
copyToClipboard v = try [tryCommand (xclip v), giveUp]

-- Try all functions in the list until one returns True
try :: [IO Bool] -> IO ()
try [] = error "This should not happen, please contact daniel@dangarbri.tech"
try (fn : rest) =
  fn >>= \success ->
    if success
      then return ()
      else try rest

tryCommand :: IO Bool -> IO Bool
tryCommand fn =
  catch
    fn
    ( \e ->
        let _ = e :: IOException
         in return False
    )

-- | Attempt to copy the given string to the clipboard via xclip.
--   If xclip is not available, then tries the next function provided.
xclip :: String -> IO Bool
xclip v = do
  (Just xclipStdin, _, _, _) <- createProcess (proc "xclip" ["-sel", "clipboard"]) {std_in = CreatePipe}
  hPutStr xclipStdin v
  hClose xclipStdin
  putStrLn "Password copied to clipboard"
  return True

giveUp :: IO Bool
giveUp = do
  hPutStrLn stderr "Could not find a program for copying to clipboard. Tried xclip. (TODO: Implement pbcopy for mac, wl-copy for linux/wayland, clip for windows)"
  return True
