{-# LANGUAGE TypeFamilies #-}
{-# LANGUAGE TypeOperators #-}
module LLVM.Extra.ArithmeticPrivate where

import qualified LLVM.Util.Intrinsic as Intrinsic

import qualified LLVM.Core as LLVM
import LLVM.Core
   (CodeGenFunction, valueOf, Value,
    CmpPredicate(CmpLE, CmpGE), FPPredicate, CmpRet, CmpResult,
    IsConst, IsPrimitive, IsArithmetic, IsInteger, IsFloating,
    getElementPtr, )

import Data.Word (Word32, )
import Data.Int (Int32, )

import Prelude hiding (and, or, sqrt, sin, cos, exp, log, abs, min, max, )


add ::
   (IsArithmetic a) =>
   Value a -> Value a -> CodeGenFunction r (Value a)
add = LLVM.add

sub ::
   (IsArithmetic a) =>
   Value a -> Value a -> CodeGenFunction r (Value a)
sub = LLVM.sub


inc ::
   (IsArithmetic a, IsConst a, Num a) =>
   Value a -> CodeGenFunction r (Value a)
inc x = add x (valueOf 1)

dec ::
   (IsArithmetic a, IsConst a, Num a) =>
   Value a -> CodeGenFunction r (Value a)
dec x = sub x (valueOf 1)

advanceArrayElementPtr ::
   (LLVM.IsType a) =>
   Value (LLVM.Ptr a) ->
   CodeGenFunction r (Value (LLVM.Ptr a))
advanceArrayElementPtr p =
   getElementPtr p (valueOf 1 :: Value Word32, ())

decreaseArrayElementPtr ::
   (LLVM.IsType a) =>
   Value (LLVM.Ptr a) ->
   CodeGenFunction r (Value (LLVM.Ptr a))
decreaseArrayElementPtr p =
   getElementPtr p (valueOf (-1) :: Value Int32, ())



mul ::
   (IsArithmetic a) =>
   Value a -> Value a -> CodeGenFunction r (Value a)
mul = LLVM.mul


{- |
This would also work for vectors,
but LLVM-3.1 crashes when actually doing this.
-}
min :: (CmpRet a) => Value a -> Value a -> CodeGenFunction r (Value a)
min = cmpSelect (cmp CmpLE)

max :: (CmpRet a) => Value a -> Value a -> CodeGenFunction r (Value a)
max = cmpSelect (cmp CmpGE)

abs :: (IsArithmetic a, CmpRet a) =>
   Value a -> CodeGenFunction r (Value a)
abs x = do
   b <- cmp LLVM.CmpGE x (LLVM.value LLVM.zero)
   LLVM.select b x =<< LLVM.neg x


signumGen ::
   (CmpRet a, IsPrimitive a) =>
   Value a -> Value a ->
   Value a -> CodeGenFunction r (Value a)
signumGen minusOne one x = do
   let zero = LLVM.value LLVM.zero
   negative <- cmp LLVM.CmpLT x zero
   positive <- cmp LLVM.CmpGT x zero
   LLVM.select negative minusOne
      =<< LLVM.select positive one zero

signum ::
   (Num a, CmpRet a, IsConst a, IsPrimitive a) =>
   Value a -> CodeGenFunction r (Value a)
signum = signumGen (LLVM.valueOf (-1)) (LLVM.valueOf 1)


cmpSelect ::
   (CmpRet a) =>
   (Value a -> Value a -> CodeGenFunction r (Value (CmpResult a))) ->
   (Value a -> Value a -> CodeGenFunction r (Value a))
cmpSelect f x y =
   f x y >>= \b -> LLVM.select b x y


fcmp ::
   (IsFloating a, CmpRet a, CmpResult a ~ b) =>
   FPPredicate -> Value a -> Value a -> CodeGenFunction r (Value b)
fcmp = LLVM.fcmp

cmp ::
   (CmpRet a, CmpResult a ~ b) =>
   CmpPredicate -> Value a -> Value a -> CodeGenFunction r (Value b)
cmp = LLVM.cmp



and ::
   (IsInteger a) =>
   Value a -> Value a -> CodeGenFunction r (Value a)
and = LLVM.and

or ::
   (IsInteger a) =>
   Value a -> Value a -> CodeGenFunction r (Value a)
or = LLVM.or


fraction :: (IsFloating a) => Value a -> CodeGenFunction r (Value a)
fraction x = sub x =<< Intrinsic.floor x
