{-# LANGUAGE TypeFamilies #-}
{-# LANGUAGE TypeOperators #-}
{-# LANGUAGE MultiParamTypeClasses #-}
module LLVM.Extra.Nice.Value.Vector (
   cons,
   fst, snd,
   fst3, snd3, thd3,
   zip, zip3,
   unzip, unzip3,

   swap,
   mapFst, mapSnd,
   mapFst3, mapSnd3, mapThd3,

   extract, insert,
   replicate,
   iterate,
   dissect,
   dissect1,
   select,
   cmp,
   take, takeRev,

   NativeInteger,
   NativeFloating,
   fromIntegral,
   truncateToInt,
   splitFractionToInt,
   ) where

import qualified LLVM.Extra.Nice.Vector.Instance as Inst
import qualified LLVM.Extra.Nice.Vector as NiceVector
import qualified LLVM.Extra.Nice.Value.Private as NiceValue
import qualified LLVM.Extra.ScalarOrVector as SoV
import LLVM.Extra.Nice.Vector.Instance (NVVector)

import qualified LLVM.Core as LLVM

import qualified Type.Data.Num.Decimal as TypeNum


import qualified Data.NonEmpty as NonEmpty
import qualified Data.Tuple.HT as TupleHT
import qualified Data.Tuple as Tuple
import Data.Word (Word8, Word16, Word32, Word64, Word)
import Data.Int (Int8, Int16, Int32, Int64, Int)

import Prelude (Float, Double, Bool, fmap, (.))


cons ::
   (TypeNum.Positive n, NiceVector.C a) =>
   LLVM.Vector n a -> NVVector n a
cons = Inst.toNiceValue . NiceVector.cons

fst :: NVVector n (a,b) -> NVVector n a
fst = NiceValue.lift1 Tuple.fst

snd :: NVVector n (a,b) -> NVVector n b
snd = NiceValue.lift1 Tuple.snd

swap :: NVVector n (a,b) -> NVVector n (b,a)
swap = NiceValue.lift1 TupleHT.swap

mapFst ::
   (NVVector n a0 -> NVVector n a1) ->
   NVVector n (a0,b) -> NVVector n (a1,b)
mapFst f = Tuple.uncurry zip . TupleHT.mapFst f . unzip

mapSnd ::
   (NVVector n b0 -> NVVector n b1) ->
   NVVector n (a,b0) -> NVVector n (a,b1)
mapSnd f = Tuple.uncurry zip . TupleHT.mapSnd f . unzip


fst3 :: NVVector n (a,b,c) -> NVVector n a
fst3 = NiceValue.lift1 TupleHT.fst3

snd3 :: NVVector n (a,b,c) -> NVVector n b
snd3 = NiceValue.lift1 TupleHT.snd3

thd3 :: NVVector n (a,b,c) -> NVVector n c
thd3 = NiceValue.lift1 TupleHT.thd3

mapFst3 ::
   (NVVector n a0 -> NVVector n a1) ->
   NVVector n (a0,b,c) -> NVVector n (a1,b,c)
mapFst3 f = TupleHT.uncurry3 zip3 . TupleHT.mapFst3 f . unzip3

mapSnd3 ::
   (NVVector n b0 -> NVVector n b1) ->
   NVVector n (a,b0,c) -> NVVector n (a,b1,c)
mapSnd3 f = TupleHT.uncurry3 zip3 . TupleHT.mapSnd3 f . unzip3

mapThd3 ::
   (NVVector n c0 -> NVVector n c1) ->
   NVVector n (a,b,c0) -> NVVector n (a,b,c1)
mapThd3 f = TupleHT.uncurry3 zip3 . TupleHT.mapThd3 f . unzip3


zip :: NVVector n a -> NVVector n b -> NVVector n (a,b)
zip (NiceValue.Cons a) (NiceValue.Cons b) = NiceValue.Cons (a,b)

zip3 :: NVVector n a -> NVVector n b -> NVVector n c -> NVVector n (a,b,c)
zip3 (NiceValue.Cons a) (NiceValue.Cons b) (NiceValue.Cons c) =
   NiceValue.Cons (a,b,c)

unzip :: NVVector n (a,b) -> (NVVector n a, NVVector n b)
unzip (NiceValue.Cons (a,b)) = (NiceValue.Cons a, NiceValue.Cons b)

unzip3 :: NVVector n (a,b,c) -> (NVVector n a, NVVector n b, NVVector n c)
unzip3 (NiceValue.Cons (a,b,c)) =
   (NiceValue.Cons a, NiceValue.Cons b, NiceValue.Cons c)


extract ::
   (TypeNum.Positive n, NiceVector.C a) =>
   LLVM.Value Word32 -> NVVector n a ->
   LLVM.CodeGenFunction r (NiceValue.T a)
extract k v = NiceVector.extract k (Inst.fromNiceValue v)

insert ::
   (TypeNum.Positive n, NiceVector.C a) =>
   LLVM.Value Word32 -> NiceValue.T a ->
   NVVector n a -> LLVM.CodeGenFunction r (NVVector n a)
insert k a = Inst.liftNiceValueM (NiceVector.insert k a)


replicate ::
   (TypeNum.Positive n, NiceVector.C a) =>
   NiceValue.T a -> LLVM.CodeGenFunction r (NVVector n a)
replicate = fmap Inst.toNiceValue . NiceVector.replicate

iterate ::
   (TypeNum.Positive n, NiceVector.C a) =>
   (NiceValue.T a -> LLVM.CodeGenFunction r (NiceValue.T a)) ->
   NiceValue.T a -> LLVM.CodeGenFunction r (NVVector n a)
iterate f = fmap Inst.toNiceValue . NiceVector.iterate f

take ::
   (TypeNum.Positive n, TypeNum.Positive m, NiceVector.C a) =>
   NVVector n a -> LLVM.CodeGenFunction r (NVVector m a)
take = Inst.liftNiceValueM NiceVector.take

takeRev ::
   (TypeNum.Positive n, TypeNum.Positive m, NiceVector.C a) =>
   NVVector n a -> LLVM.CodeGenFunction r (NVVector m a)
takeRev = Inst.liftNiceValueM NiceVector.takeRev


dissect ::
   (TypeNum.Positive n, NiceVector.C a) =>
   NVVector n a -> LLVM.CodeGenFunction r [NiceValue.T a]
dissect = NiceVector.dissect . Inst.fromNiceValue

dissect1 ::
   (TypeNum.Positive n, NiceVector.C a) =>
   NVVector n a -> LLVM.CodeGenFunction r (NonEmpty.T [] (NiceValue.T a))
dissect1 = NiceVector.dissect1 . Inst.fromNiceValue

select ::
   (TypeNum.Positive n, NiceVector.Select a) =>
   NVVector n Bool ->
   NVVector n a -> NVVector n a ->
   LLVM.CodeGenFunction r (NVVector n a)
select = Inst.liftNiceValueM3 NiceVector.select

cmp ::
   (TypeNum.Positive n, NiceVector.Comparison a) =>
   LLVM.CmpPredicate ->
   NVVector n a -> NVVector n a ->
   LLVM.CodeGenFunction r (NVVector n Bool)
cmp = Inst.liftNiceValueM2 . NiceVector.cmp


{-
ToDo: make this a super-class of NiceValue.NativeInteger
problem: we need NiceValue.Repr, which provokes an import cycle
maybe we should break the cycle using a ConstraintKind,
i.e. define class NativeIntegerVec in NiceValue,
and define NativeInteger = NiceValue.NativeIntegerVec here
and export only NiceValueVec.NativeInteger constraint synonym.
-}
class
   (NiceValue.Repr i ~ LLVM.Value ir,
    LLVM.CmpRet ir, LLVM.IsInteger ir, SoV.IntegerConstant ir) =>
      NativeInteger i ir where

instance NativeInteger Word   Word   where
instance NativeInteger Word8  Word8  where
instance NativeInteger Word16 Word16 where
instance NativeInteger Word32 Word32 where
instance NativeInteger Word64 Word64 where

instance NativeInteger Int   Int   where
instance NativeInteger Int8  Int8  where
instance NativeInteger Int16 Int16 where
instance NativeInteger Int32 Int32 where
instance NativeInteger Int64 Int64 where

instance
   (TypeNum.Positive n, n ~ m,
    NiceVector.NativeInteger n i ir,
    NiceValue.NativeInteger i ir) =>
      NativeInteger (LLVM.Vector n i) (LLVM.Vector m ir) where


class
   (NiceValue.Repr a ~ LLVM.Value ar,
    LLVM.CmpRet ar,  SoV.RationalConstant ar, LLVM.IsFloating ar) =>
      NativeFloating a ar where

instance NativeFloating Float  Float  where
instance NativeFloating Double Double where

instance
   (TypeNum.Positive n, n ~ m,
    NiceVector.NativeFloating n a ar,
    NiceValue.NativeFloating a ar) =>
      NativeFloating (LLVM.Vector n a) (LLVM.Vector m ar) where

fromIntegral ::
   (NativeInteger i ir, NativeFloating a ar,
    LLVM.ShapeOf ir ~ LLVM.ShapeOf ar) =>
   NiceValue.T i -> LLVM.CodeGenFunction r (NiceValue.T a)
fromIntegral = NiceValue.liftM LLVM.inttofp


truncateToInt ::
   (NativeInteger i ir, NativeFloating a ar,
    LLVM.ShapeOf ir ~ LLVM.ShapeOf ar) =>
   NiceValue.T a -> LLVM.CodeGenFunction r (NiceValue.T i)
truncateToInt = NiceValue.liftM LLVM.fptoint

splitFractionToInt ::
   (NativeInteger i ir, NativeFloating a ar,
    LLVM.ShapeOf ir ~ LLVM.ShapeOf ar) =>
   NiceValue.T a -> LLVM.CodeGenFunction r (NiceValue.T (i,a))
splitFractionToInt = NiceValue.liftM SoV.splitFractionToInt
