# lr-acts

[![Haskell](https://img.shields.io/badge/language-Haskell-orange.svg)](https://haskell.org) [![BSD3 License](https://img.shields.io/badge/license-BSD3-blue.svg)](https://github.com/AliceRixte/lr-acts/LICENSE) [![Hackage](https://img.shields.io/hackage/v/lr-acts.svg)](https://hackage.haskell.org/package/lr-acts) [![Nightly](https://www.stackage.org/package/lr-acts/badge/nightly)](https://www.stackage.org/nightly/package/lr-acts) [![LTS](https://www.stackage.org/package/lr-acts/badge/lts)](https://www.stackage.org/lts/package/lr-acts)

## Features

* Left and right actions of
  * sets
  * semigroup
  * monoids
  * groups
* Semidirect product
* Group torsors
* Cyclic actions
* Generated actions


### Fine-grained class hierarchy

Left and right actions with a fine-grained class hierarchy for action properties. For left actions, here are the provided classes :

``` haskell
class LAct               -- Set action
 => LActSg               -- Semigroup action
     => LActMn           -- Monoid action
          => LTorsor     -- Torsor
 => LActDistrib          -- Distributive action
 => LActNeutral          -- Neutral preserving action
 => LActGen              -- Action generated by a set
     => LActCyclic       -- Cyclic action (generated by a single element)

```

### Derive most of you action instances

The acting type is always the second parameter. Use this with `DerivingVia` language extension to derive action instances :

``` haskell
import Data.Act
import Data.Semigroup

newtype Seconds = Seconds Float
newtype Duration = Duration Seconds
  deriving (Semigroup, Monoid) via (Sum Float)

  deriving (LAct Seconds, RAct Seconds) via (ActSelf' (Sum Float))
  -- derives LAct Second  Duration

  deriving (LAct [Seconds], RAct [Seconds]) via (ActMap (ActSelf' (Sum Float)))
   -- derives LAct [Second] Duration

newtype Durations = Durations [Duration]
  deriving (LAct Seconds, RAct Seconds) via (ActFold [Duration])
  -- derives LAct Second Durations

```

``` haskell
ghci> Duration 2 `lact` Seconds 3
Seconds 5.0

ghci> Duration 2 `lact` [Seconds 3, Seconds 4]
[Seconds 5.0,Seconds 6.0]

ghci> [Duration 2, Duration 3] `lact` Seconds 4
[Seconds 5.0,Seconds 6.0]

ghci> Durations [Duration 2, Duration 3] `lact` Seconds 4
Seconds 9.0
```

### Semidirect products

This fine-grained hierarchy allows to check for associativity and existence of neutral elements using _semidirect products_.

``` haskell
>>> import Data.Semigroup
>>> LPair (Sum 1) (Product 2) <> LPair (Sum (3 :: Int)) (Product (4 :: Int))
LPair {lactee = Sum {getSum = 7}, lactor = Product {getProduct = 8}}
```

GHC will complain when using a semigroup action that is not distributive :

```haskell
>>> LPair (Sum 1) (Sum 2) <> LPair (Sum (3 :: Int)) (Sum (4 :: Int))
No instance for `LActDistrib (Sum Int) (Sum Int)'
  arising from a use of `<>'
```

## Comparison with other action libraries

Here is a list of action libraries on hackage :

- [monoid-extra](https://github.com/diagrams/monoid-extras)
- [acts](https://hackage.haskell.org/package/acts)
- [semigroup-actions](https://hackage.haskell.org/package/semigroups-actions)
- [raaz](https://hackage.haskell.org/package/raaz-0.0.1/docs/Raaz-Core-MonoidalAction.html)


In comparison with these libraries, `lr-acts`is the only library that :
- Implements right actions
- Implements cyclic actions and generated actions
- Ensures the associativity and the neutrality of `mempty` in semidirect products
- Proposes several newtypes for deriving instances (note that [acts](https://hackage.haskell.org/package/acts) proposes a deriving mechanism, but centered around the actee type, not the actor type as in this library)

The main drawback of providing right actions and checking properties for semidirect products is that the number of instances can quickly be overwhelming. It can be a lot of boiler plate to declare them all, especially when the acting semigroup is commutative.
