{-# LANGUAGE FlexibleContexts, RankNTypes, DeriveFunctor, TupleSections, FlexibleInstances #-}

module MVD.Finders where

import MVD.STR
import qualified Data.Set as S
import Data.Function
import Data.List (find)
import Data.Graph


type Step c a l = TR c a -> Graph c a -> (c -> l) -> Graph c a

type Label c a l = Graph c a -> (c -> l)

class Labelling l where
    isAccepting :: l -> Bool


extractAccepting :: (Labelling l) => (c -> l) -> Graph c a -> S.Set c
extractAccepting l (Graph v _) = S.filter (isAccepting . l) v

mfinder :: (Ord c, Ord a, Labelling l) => Step c a l -> Label c a l -> STR c a m -> Graph c a -> (Graph c a, S.Set c)
mfinder step lab str = fix fprocess
    where
        tr l = TR { tinitial = initial str
                , accepting = isAccepting . l
                , next = \c -> concat [map (a,) (execute str c a) | a <- actions str c]
                }
        fprocess frec g = if g == g' then (g', extractAccepting (lab g') g') else frec g'
            where
                l = lab g
                g' = step (tr l) g l



data SimpleLabel = Accepted | Disabled | Enabled
    deriving (Eq, Show, Ord)

instance Labelling SimpleLabel where
    isAccepting = (==Accepted)


joinSimpleLabel :: (c -> SimpleLabel) -> (c -> SimpleLabel) -> (c -> SimpleLabel)
joinSimpleLabel f g c = f c `ljoin` g c


ljoin :: SimpleLabel -> SimpleLabel -> SimpleLabel
ljoin Accepted _ = Accepted
ljoin _ Accepted = Accepted
ljoin Disabled _ = Disabled
ljoin _ Disabled = Disabled
ljoin _ _ = Enabled


enabledLeavesStepper :: (Ord c, Ord a) => Step c a SimpleLabel
enabledLeavesStepper tr g@(Graph v e) l =
    let enleaves = filter ((==) Enabled . l) $ gleaves g
        new = foldr ((\s acc -> S.fromList s `S.union` acc) . (\c -> map (\(a, c') -> (c, a, c')) $ next tr c)) S.empty enleaves
    in Graph (v `S.union` S.map (\(_, _, c) -> c) new) (new `S.union` e)


-- leavesReducer == redReducer identity
leavesReducer :: Ord c => Label c a SimpleLabel
leavesReducer g c = if c `elem` gleaves g then Enabled else Disabled

redReducer :: (Ord alpha, Ord a, Eq c, Reduce r c alpha) => r -> Label c a SimpleLabel
redReducer r g@(Graph v _) c = if c `elem` enabled then Enabled else Disabled
    where
        enabled = concatMap invert $ gleaves $ gcmap (rstate r) g
        invert cl = [cc | cc <- S.toList v, rstate r cc == cl]


estateBreaker :: (Eq c, Evaluate e c Bool) => e -> Label c a SimpleLabel
estateBreaker e (Graph v _) c =
    case find (estate e) $ S.toList v of
        (Just c') -> if c == c' then Accepted else Disabled
        Nothing -> Enabled

originalBreakerReducer :: (Reduce r c alpha, Evaluate e c Bool, Eq c, Ord a, Ord alpha)
    => r -> e -> Label c a SimpleLabel
originalBreakerReducer r e g = joinSimpleLabel (redReducer r g) (estateBreaker e g)
