{-|
Module      : Nauty.Encoding.Internal
Description : Internal functions for encoding.
Copyright   : (c) Marcelo Garlet Milani, 2026
License     : MIT
Maintainer  : mgmilani@pm.me
Stability   : unstable

This module contains internal functions used by other modules.
Except for test cases, you should not import this module.
-}

module Nauty.Internal.Encoding where

import qualified Data.Text.Lazy as T
import Data.Bits
import Data.Word

-- | Encode an 8-bit vector as 6-bit words.
encodeVector :: Word64 -> [Word8] -> Word8 -> Int -> [Word8]
encodeVector lastValidBits bs acc 0 = acc : encodeVector lastValidBits bs 0 6
encodeVector lastValidBits [b] acc s
  | (fromIntegral lastValidBits) > s = 
      (acc .|. shiftR b (8 - s))
      : [shiftR (shiftL b s) 2]
  | otherwise = [acc .|. shiftR b (8 - s)]
encodeVector lastValidBits (b:bs) acc r = 
  (acc .|. (shiftR b (8 - r)))
  : encodeVector lastValidBits bs (shiftR (shiftL b r) 2) ((r - 2))
encodeVector _ [] _ _ = []

-- | Minimum amount of bits required to represent the number.
numBits :: (Num a) => Word64 -> a
numBits n = numBits' (shiftR n 1)
  where
    numBits' 0 = 1
    numBits' n' = 1 + numBits' (shiftR n' 1)

-- | Encode a number.
encodeNumber :: Word64 -> T.Text
encodeNumber n 
  | n < 63 = T.pack [toEnum $ fromIntegral $ n + 63]
  | otherwise = 
    let bits = fromIntegral $ (18 :: Int) * (((numBits n) + 7) `div` 18) :: Int
        encode' :: Int -> [Word8]
        encode' 0 = [fromIntegral n]
        encode' s = fromIntegral (shiftR n s .&. 63) : encode' (s - 6)
    in  T.pack $ map (toEnum . fromIntegral . (+63)) 
        $ (take (bits `div` 18) $ repeat 63)
        ++ encode' (bits - 6)
