{-|
Module      : Nauty.Parsing.Internal
Description : Internal functions for parsing.
Copyright   : (c) Marcelo Garlet Milani, 2026
License     : MIT
Maintainer  : mgmilani@pm.me
Stability   : unstable

This module contains internal functions used by other modules.
Except for test cases, you should not import this module.
-}

module Nauty.Internal.Parsing where

import Control.Monad.Trans.State
import Control.Monad.Trans.Class
import qualified Data.ByteString.Lazy as B
import qualified Data.Text.Lazy as T
import Data.Bits
import Data.Word

-- | Read the given number of bytes.
consume :: Word64 -> StateT B.ByteString (Either T.Text) B.ByteString
consume l = do
  b <- get
  let l' = fromIntegral l
  if (fromIntegral $ B.length b) < l then
     lift $ Left $ T.concat
       [ "Error: ByteString has length "
       , (T.show $ B.length b)
       , ", but "
       , T.show l
       , " bytes were requested"
       ]
  else do
    put $ B.drop l' b
    return $ B.take l' b

-- | Read bytes until the given condition evaluates to false.
consumeWhile :: (Word8 -> Bool) -> StateT B.ByteString (Either T.Text) B.ByteString
consumeWhile p = do
  b <- get
  let (b0, b1) = B.span p b
  put b1
  return b0

-- | Parse a single number.
parseNumber :: StateT B.ByteString (Either T.Text) Word64
parseNumber = do
  size <- consumeWhile (==126)
  if B.length size == 0 then do
    be <- consume 1
    return $ (fromIntegral $ B.head be) - 63
  else do 
    be <- parseVector (18 * (fromIntegral $ 1 + B.length size))
    return $ bigendian be

-- | Parse a single number with big-endian encoding.
bigendian :: B.ByteString -> Word64
bigendian bs = foldr (\b n -> (shiftL n 8) .|. (fromIntegral b)) 0 $ B.unpack bs

-- | Parse a vector with the given amount bits.
parseVector :: Word64 -> StateT B.ByteString (Either T.Text) B.ByteString
parseVector len = do
  v <- consume ((len + 5) `div` 6)
  let (bs, _) = B.foldl'
              append6Bits
              ([], 0)
              v
  if (8 * ((len + 7)`div` 8)) < (6 * ((len + 5) `div` 6)) then
    return $ B.pack $ reverse $ drop 1 bs
  else
    return $ B.pack $ reverse $ bs

-- |Put 6 bits to a list of bytes.
append6Bits :: ([Word8], Word8) -> Word8 ->  ([Word8], Word8)
append6Bits ([], _) b     = (shiftL (b - 63) 2 : [], 2)
append6Bits (bs, 0) b = (shiftL (b - 63) 2 : bs, 2)
append6Bits ((b0:bs), s) b = 
  let b' = b - 63
      s' = fromIntegral $ 6 - s
      b1 = shiftR b' s'
      b2 = shiftL b' $ fromIntegral $ s + 2
  in 
  if s == 6 then
    ((b0 .|. b1) : bs, 0)
  else
    (b2 : (b0 .|. b1) : bs, (s + 2) `mod` 8)

-- | Parse an optional header.
-- Returns the rest of the file.
header :: T.Text -> T.Text -> T.Text
header h t
  | h `T.isPrefixOf` t = T.drop (T.length h) t
  | otherwise = t

-- | Skips the header. Returns the rest of the text.
ignoreHeader :: T.Text -> T.Text
ignoreHeader txt
  | ">>" `T.isPrefixOf` txt = 
      let t1 = T.dropWhile (/= '<') txt in
      if "<<" `T.isPrefixOf` t1 then
        T.drop 2 t1
      else
        txt
  | otherwise = txt
      
