{-# LANGUAGE OverloadedStrings, OverloadedRecordDot #-}

module Main where

import System.Exit (exitFailure, exitSuccess)
import Test.HUnit hiding (Node)

import Data.Either
import qualified Data.Array.Unboxed as A
import qualified Data.Text.Lazy as T
import Nauty.Internal
import qualified Nauty.Digraph6 as D
import qualified Nauty.Graph6   as G
import qualified Nauty.Sparse6  as S

testParsing :: Graph -> T.Text -> IO ()
testParsing g str = do
  assertEqual ("parse " ++ T.unpack str) ([Right g]) (parse str)
  assertEqual ("encode " ++ T.unpack str) str (encode g)

tests :: Test
tests = TestList                         
  [ TestLabel "g = parse $ encode g" $ TestCase
    ( do 
      let g0 = G.fromEdgeList 4 [(1,0), (2,1), (3,2), (3,0)]
          s0 = S.fromEdgeList 4 [(1,0), (2,1), (3,2), (3,0)]
          d0 = D.fromArcList  4 [(1,0), (2,1), (3,2), (3,0)]
          g0' = Graph
                  { numberOfVertices = 4
                  , adjacencyList = A.array (0,7) $ zip [0..]
                      [0,1,1,2,0,3,2,3]
                  , graphType = Undirected
                  , format = Graph6
                  , adjacencyMatrix = g0.adjacency
                  }
      assertEqual (T.unpack $ G.encode g0)
        ([Right g0'])
        (parse $ G.encode g0)
      assertEqual (T.unpack $ S.encode s0)
        ([Right g0'{format= Sparse6, adjacencyList = s0.adjacency}])
        (parse $ S.encode s0)
      assertEqual (T.unpack $ D.encode d0)
        ([Right g0'{format = Digraph6
                  , adjacencyList = A.array (0,7) $ zip [0..]
                      [1,0,2,1,3,0,3,2]
                  , adjacencyMatrix = d0.adjacency
                  , graphType = Directed
                  }])
        (parse $ D.encode d0)
      testParsing (g0')
                  (G.encode g0)
      testParsing (g0'{format = Sparse6, adjacencyList = s0.adjacency})
                  (S.encode s0)
    )
  , TestLabel "Invalid characters 1" $ TestCase
    ( do
      let gtext = T.intercalate "\n"
                    [ "D??"
                    , "D?_"
                    , "D?o"
                    , "D?w"
                    ]
          (errs, gs) = partitionEithers $ parse gtext
      assertEqual "" [] errs
      let incs = encodeMany $ map (\g -> g{format = Incremental}) gs
      assertEqual ""
        [ ":D"
        , ";ON"
        , ";O^"
        , ";On"
        ]
        (T.lines incs)

    )
  ]

main :: IO ()
main = do 
  count <- runTestTT tests
  if errors count + failures count > 0 then exitFailure else exitSuccess
