{-#LANGUAGE MultiParamTypeClasses #-}
module Text.Parcom.Core
( ParcomError (..)
, SourcePosition (..)
, Parcom
, parse
, peek, next, atEnd
, try, handle
, notFollowedBy
, (<?>), (<|>), empty
, Stream, Token
)
where

import qualified Text.Parcom.Stream as Stream
import Text.Parcom.Stream (Stream, Token)
import Control.Monad (liftM, ap)
import Control.Applicative

data SourcePosition =
    SourcePosition
        { posFileName :: String
        , posLine :: Int
        , posColumn :: Int
        }
        deriving (Show)

data ParcomError =
    ParcomError
        { peErrorDescription :: String
        , peSourcePosition :: SourcePosition
        }
        deriving (Show)

data ParcomState s =
    ParcomState
        { psSourcePosition :: SourcePosition
        , psStream :: s
        }

newtype Parcom s t a = Parcom { runParcom :: ParcomState s -> (Either ParcomError a, ParcomState s) }

instance Monad (Parcom s t) where
    return x = Parcom (\s -> (Right x, s))
    fail err = Parcom (\s -> (Left $ ParcomError err (psSourcePosition s), s))
    m >>= f = Parcom $ \s ->
        let (a, s') = runParcom m s
        in case a of
            Left e -> (Left e, s')
            Right x -> runParcom (f x) s'

instance Functor (Parcom s t) where
    fmap f xs = xs >>= return . f

instance Applicative (Parcom s t) where
    pure = return
    (<*>) = ap

instance Alternative (Parcom s t) where
    (<|>) = alt
    empty = fail "empty"

runParser :: (Stream s t, Token t) => Parcom s t a -> String -> s -> (Either ParcomError a, ParcomState s)
runParser p fn str =
    runParcom p state
    where
        state =
            ParcomState
                { psSourcePosition = SourcePosition fn 1 1
                , psStream = str }

parse :: (Stream s t, Token t) => Parcom s t a -> String -> s -> Either ParcomError a
parse p fn str = fst $ runParser p fn str

getState :: Parcom s t (ParcomState s)
getState = Parcom $ \s -> (Right s, s)

useState :: (ParcomState s -> a) -> Parcom s t a
useState f = Parcom $ \s -> (Right $ f s, s)

setState :: ParcomState s -> Parcom s t ()
setState s = Parcom $ \_ -> (Right (), s)

modifyState :: (ParcomState s -> ParcomState s) -> Parcom s t ()
modifyState f = Parcom $ \s -> (Right (), f s)

handle :: Parcom s t a -> (ParcomError -> Parcom s t b) -> (a -> Parcom s t b) -> Parcom s t b
handle p f t = Parcom $ \s ->
    let (r', s') = runParcom p s
    in case r' of
        -- parse failed: run the error handler
        Left e -> runParcom (f e) s'
        -- parse succeeded: run the success handler
        Right x -> runParcom (t x) s'


-- | Backtracking modifier; restores the parser state to the previous situation
-- if the wrapped parser fails.
try :: Parcom s t a -> Parcom s t a
try p = Parcom $ \s ->
    let (r', s') = runParcom p s
    in case r' of
        -- parse failed: return the error and restore the old state
        Left e -> (Left e, s)
        -- parse succeeded: return the result and the new state
        Right x -> (Right x, s')

-- | Return the result of the first parser that succeeds.
alt :: Parcom s t a -> Parcom s t a -> Parcom s t a
alt a b = Parcom $ \s ->
            let (r', s') = runParcom a s
            in case r' of
                Left _ -> runParcom b s
                Right x -> (Right x, s')

-- | Succeeds iff the given parser fails
notFollowedBy :: (Stream s t) => Parcom s t a -> Parcom s t ()
notFollowedBy p = Parcom $ \s ->
    let (r', s') = runParcom p s
    in case r' of
        Left _ -> (Right (), s)
        Right x -> runParcom (fail "something followed that shouldn't") s
    
-- | Gets the next token from the stream without consuming it.
-- Fails at end-of-input.
peek :: (Stream s t) => Parcom s t t
peek = do
    str <- psStream `liftM` getState
    if Stream.atEnd str
        then fail "Unexpected end of input"
        else return (Stream.peek str)

-- | Checks whether end-of-input has been reached.
atEnd :: (Stream s t) => Parcom s t Bool
atEnd = useState (Stream.atEnd . psStream)

nextLine :: SourcePosition -> SourcePosition
nextLine s =
    s { posLine = posLine s + 1, posColumn = 1 }

nextColumn :: SourcePosition -> SourcePosition
nextColumn s =
    s { posColumn = posColumn s + 1 }

-- | Gets the next token from the stream and consumes it.
-- Fails at end-of-input.
next :: (Stream s t, Token t) => Parcom s t t
next = do
    str <- psStream `liftM` getState
    if Stream.atEnd str
        then fail "Unexpected end of input"
        else do
            let (t, str') = Stream.pop str
            modifyState $ \state -> state { psStream = str' }
            if Stream.isLineDelimiter t
                then modifyState $ \s -> s { psSourcePosition = nextLine (psSourcePosition s) }
                else modifyState $ \s -> s { psSourcePosition = nextColumn (psSourcePosition s) }
            return t

(<?>) :: (Stream s t) => Parcom s t a -> String -> Parcom s t a
p <?> expected = p <|> fail ("Expected " ++ expected)
