# pure-noise

Performant, modern noise generation for Haskell with a minimal dependency footprint.

The algorithms used in this library are ported from [FastNoiseLite](https://github.com/Auburn/FastNoiseLite). The library structure has been retuned to fit better with Haskell semantics.

The public interface for this library is unlikely to change much, although the implementations (`noiseBaseN` functions and anything in `Numeric.Noise.Internal`) are subject to change and may change between minor versions.

## Usage

The library exports newtypes for N-dimensional noise. Currently, these are just functions that accept a seed and a point in N-dimensional space. They can be arbitrarily unwrapped by with the `noiseNAt` family of functions. Since they abstract over the given seed and parameters, they can be composed with `Num` or `Fractional` methods at will with little-to-no performance cost.

Noise values are generally clamped to `[-1, 1]`, although some noise functions may occasionally produce values slightly outside this range.

```haskell
import Numeric.Noise qualified as Noise

myNoise2 :: (RealFrac a) => Seed -> a -> a -> a
myNoise2 =
  let fractalConfig = Noise.defaultFractalConfig
  in Noise.noise2At $
      Noise.fractal2 fractalConfig ((perlin2 + superSimplex2) / 2)
```

More examples can be found in `bench` and `demo`.

## Performance notes

- This library benefits considerably from compilation with the LLVM backend (`-fllvm`). Benchmarks suggest a ~50-80% difference depending on the kind of noise.

## Benchmarks

### Results

Measured by values / second generated by the noise functions. These results come from a benchmark with `-fllvm` enabled.

All results are for `Float`s.

There's inevitably some noise in the measurements because all of the results are forced into an unboxed vector.

#### 2D

| name          | values / second |
| ------------- | --------------- |
| value2        | 156_797_694     |
| perlin2       | 138_048_921     |
| superSimplex2 | 65_204_214      |
| openSimplex2  | 64_483_692      |
| valueCubic2   | 50_666_467      |
| cellular2     | 20_819_883      |

#### 3D

| name        | values / second |
| ----------- | --------------- |
| value3      | 83_034_432      |
| perlin3     | 60_233_650      |
| valueCubic3 | 15_220_433      |

## Examples

There's an interactive [demo app](https://github.com/jtnuttall/pure-noise/tree/main/demo) in the `demo` directory.

_OpenSimplex2_

![OpenSimplex2](https://raw.githubusercontent.com/jtnuttall/pure-noise/main/demo/images/opensimplex.png)
![OpenSimplex2 ridged](https://raw.githubusercontent.com/jtnuttall/pure-noise/main/demo/images/opensimplex-ridged.png)

_Perlin_

![Perlin fBm](https://raw.githubusercontent.com/jtnuttall/pure-noise/main/demo/images/perlin-fbm.png)

_Cellular_

![value](https://raw.githubusercontent.com/jtnuttall/pure-noise/main/demo/images/cell-value.png)
![distance2add](https://raw.githubusercontent.com/jtnuttall/pure-noise/main/demo/images/cell-d2.png)
