{-# LANGUAGE BinaryLiterals #-}
{-# LANGUAGE NoImplicitPrelude #-}
{-# LANGUAGE OverloadedStrings #-}

module Codec.QRCode.Mode.Kanji
  ( kanji
  , kanjiB
  , kanjiMap
  ) where

import           Codec.QRCode.Base

import           Data.Binary                          (decode)
import qualified Data.Map.Strict                      as M

import qualified Codec.QRCode.Data.ByteStreamBuilder  as BSB
import           Codec.QRCode.Data.QRSegment.Internal
import           Codec.QRCode.Data.Result
import           Codec.QRCode.Data.ToInput

-- | Generate a segment representing the specified text data encoded as QR code Kanji.
--
--   Since this encoding does neither contain ASCII nor the half width katakana characters
--   it may be impossible to encode all text in this encoding.
--
--   But it is possible to encode some of it and combine it with others like ISO-8859-1.
kanji :: ToText a => a -> Result QRSegment
kanji s =
  case toString s of
    [] -> pure (constStream mempty)
    s' -> ((encodeBits 4 0b1000 <> lengthSegment (8, 10, 12) (length s')) <>) . constStream <$> kanjiB s'

kanjiB :: [Char] -> Result BSB.ByteStreamBuilder
kanjiB s = Result $ mconcat <$> traverse (fmap (BSB.encodeBits 13 . fromIntegral) . (`M.lookup` kanjiMap)) s

-- This map is generated, see below on how it was done.
kanjiMap :: M.Map Char Word16
kanjiMap = decode
  "\0\0\0\0\0\0\26ßÂ¢\0QÂ£\0RÂ§\0XÂ¨\0\14Â¬\0\138Â°\0KÂ±\0=Â´\0\12Â¶\0·Ã\
  \\151\0>Ã·\0@Î\145\1ßÎ\146\1àÎ\147\1áÎ\148\1âÎ\149\1ãÎ\150\1äÎ\151\1åÎ\
  \\152\1æÎ\153\1çÎ\154\1èÎ\155\1éÎ\156\1êÎ\157\1ëÎ\158\1ìÎ\159\1íÎ \1îÎ¡\1\
  \ïÎ£\1ðÎ¤\1ñÎ¥\1òÎ¦\1óÎ§\1ôÎ¨\1õÎ©\1öÎ±\1ÿÎ²\2\0Î³\2\1Î´\2\2Îµ\2\3Î¶\2\4Î\
  \·\2\5Î¸\2\6Î¹\2\7Îº\2\8Î»\2\9Î¼\2\10Î½\2\11Î¾\2\12Î¿\2\13Ï\128\2\14Ï\129\
  \\2\15Ï\131\2\16Ï\132\2\17Ï\133\2\18Ï\134\2\19Ï\135\2\20Ï\136\2\21Ï\137\2\
  \\22Ð\129\2FÐ\144\2@Ð\145\2AÐ\146\2BÐ\147\2CÐ\148\2DÐ\149\2EÐ\150\2GÐ\151\
  \\2HÐ\152\2IÐ\153\2JÐ\154\2KÐ\155\2LÐ\156\2MÐ\157\2NÐ\158\2OÐ\159\2PÐ \2Q\
  \Ð¡\2RÐ¢\2SÐ£\2TÐ¤\2UÐ¥\2VÐ¦\2WÐ§\2XÐ¨\2YÐ©\2ZÐª\2[Ð«\2\92Ð¬\2]Ð\173\2^Ð®\
  \\2_Ð¯\2`Ð°\2pÐ±\2qÐ²\2rÐ³\2sÐ´\2tÐµ\2uÐ¶\2wÐ·\2xÐ¸\2yÐ¹\2zÐº\2{Ð»\2|Ð¼\2\
  \}Ð½\2~Ð¾\2\128Ð¿\2\129Ñ\128\2\130Ñ\129\2\131Ñ\130\2\132Ñ\131\2\133Ñ\132\
  \\2\134Ñ\133\2\135Ñ\134\2\136Ñ\135\2\137Ñ\136\2\138Ñ\137\2\139Ñ\138\2\140\
  \Ñ\139\2\141Ñ\140\2\142Ñ\141\2\143Ñ\142\2\144Ñ\143\2\145Ñ\145\2vâ\128\144\
  \\0\29â\128\149\0\28â\128\150\0!â\128\152\0%â\128\153\0&â\128\156\0'â\128\
  \\157\0(â\128 \0µâ\128¡\0¶â\128¥\0$â\128¦\0#â\128°\0±â\128²\0Lâ\128³\0Mâ\
  \\128»\0fâ\132\131\0Nâ\132«\0°â\134\144\0iâ\134\145\0jâ\134\146\0hâ\134\
  \\147\0kâ\135\146\0\139â\135\148\0\140â\136\128\0\141â\136\130\0\157â\136\
  \\131\0\142â\136\135\0\158â\136\136\0xâ\136\139\0yâ\136\146\0<â\136\154\0\
  \£â\136\157\0¥â\136\158\0Gâ\136 \0\154â\136§\0\136â\136¨\0\137â\136©\0\
  \\127â\136ª\0~â\136«\0§â\136¬\0¨â\136´\0Hâ\136µ\0¦â\136½\0¤â\137\146\0 â\
  \\137 \0Bâ\137¡\0\159â\137¦\0Eâ\137§\0Fâ\137ª\0¡â\137«\0¢â\138\130\0|â\
  \\138\131\0}â\138\134\0zâ\138\135\0{â\138¥\0\155â\140\146\0\156â\148\128\
  \\2\159â\148\129\2ªâ\148\130\2 â\148\131\2«â\148\140\2¡â\148\143\2¬â\148\
  \\144\2¢â\148\147\2\173â\148\148\2¤â\148\151\2¯â\148\152\2£â\148\155\2®â\
  \\148\156\2¥â\148\157\2ºâ\148 \2µâ\148£\2°â\148¤\2§â\148¥\2¼â\148¨\2·â\
  \\148«\2²â\148¬\2¦â\148¯\2¶â\148°\2»â\148³\2±â\148´\2¨â\148·\2¸â\148¸\2½â\
  \\148»\2³â\148¼\2©â\148¿\2¹â\149\130\2¾â\149\139\2´â\150 \0aâ\150¡\0`â\
  \\150²\0câ\150³\0bâ\150¼\0eâ\150½\0dâ\151\134\0_â\151\135\0^â\151\139\0[â\
  \\151\142\0]â\151\143\0\92â\151¯\0¼â\152\133\0Zâ\152\134\0Yâ\153\128\0Jâ\
  \\153\130\0Iâ\153ª\0´â\153\173\0³â\153¯\0²ã\128\128\0\0ã\128\129\0\1ã\128\
  \\130\0\2ã\128\131\0\22ã\128\133\0\24ã\128\134\0\25ã\128\135\0\26ã\128\
  \\136\01ã\128\137\02ã\128\138\03ã\128\139\04ã\128\140\05ã\128\141\06ã\128\
  \\142\07ã\128\143\08ã\128\144\09ã\128\145\0:ã\128\146\0gã\128\147\0lã\128\
  \\148\0+ã\128\149\0,ã\128\156\0 ã\129\129\1\31ã\129\130\1 ã\129\131\1!ã\
  \\129\132\1\34ã\129\133\1#ã\129\134\1$ã\129\135\1%ã\129\136\1&ã\129\137\1\
  \'ã\129\138\1(ã\129\139\1)ã\129\140\1*ã\129\141\1+ã\129\142\1,ã\129\143\1\
  \-ã\129\144\1.ã\129\145\1/ã\129\146\10ã\129\147\11ã\129\148\12ã\129\149\1\
  \3ã\129\150\14ã\129\151\15ã\129\152\16ã\129\153\17ã\129\154\18ã\129\155\1\
  \9ã\129\156\1:ã\129\157\1;ã\129\158\1<ã\129\159\1=ã\129 \1>ã\129¡\1?ã\129\
  \¢\1@ã\129£\1Aã\129¤\1Bã\129¥\1Cã\129¦\1Dã\129§\1Eã\129¨\1Fã\129©\1Gã\129\
  \ª\1Hã\129«\1Iã\129¬\1Jã\129\173\1Kã\129®\1Lã\129¯\1Mã\129°\1Nã\129±\1Oã\
  \\129²\1Pã\129³\1Qã\129´\1Rã\129µ\1Sã\129¶\1Tã\129·\1Uã\129¸\1Vã\129¹\1Wã\
  \\129º\1Xã\129»\1Yã\129¼\1Zã\129½\1[ã\129¾\1\92ã\129¿\1]ã\130\128\1^ã\130\
  \\129\1_ã\130\130\1`ã\130\131\1aã\130\132\1bã\130\133\1cã\130\134\1dã\130\
  \\135\1eã\130\136\1fã\130\137\1gã\130\138\1hã\130\139\1iã\130\140\1jã\130\
  \\141\1kã\130\142\1lã\130\143\1mã\130\144\1nã\130\145\1oã\130\146\1pã\130\
  \\147\1qã\130\155\0\10ã\130\156\0\11ã\130\157\0\20ã\130\158\0\21ã\130¡\1\
  \\128ã\130¢\1\129ã\130£\1\130ã\130¤\1\131ã\130¥\1\132ã\130¦\1\133ã\130§\1\
  \\134ã\130¨\1\135ã\130©\1\136ã\130ª\1\137ã\130«\1\138ã\130¬\1\139ã\130\
  \\173\1\140ã\130®\1\141ã\130¯\1\142ã\130°\1\143ã\130±\1\144ã\130²\1\145ã\
  \\130³\1\146ã\130´\1\147ã\130µ\1\148ã\130¶\1\149ã\130·\1\150ã\130¸\1\151ã\
  \\130¹\1\152ã\130º\1\153ã\130»\1\154ã\130¼\1\155ã\130½\1\156ã\130¾\1\157ã\
  \\130¿\1\158ã\131\128\1\159ã\131\129\1 ã\131\130\1¡ã\131\131\1¢ã\131\132\
  \\1£ã\131\133\1¤ã\131\134\1¥ã\131\135\1¦ã\131\136\1§ã\131\137\1¨ã\131\138\
  \\1©ã\131\139\1ªã\131\140\1«ã\131\141\1¬ã\131\142\1\173ã\131\143\1®ã\131\
  \\144\1¯ã\131\145\1°ã\131\146\1±ã\131\147\1²ã\131\148\1³ã\131\149\1´ã\131\
  \\150\1µã\131\151\1¶ã\131\152\1·ã\131\153\1¸ã\131\154\1¹ã\131\155\1ºã\131\
  \\156\1»ã\131\157\1¼ã\131\158\1½ã\131\159\1¾ã\131 \1Àã\131¡\1Áã\131¢\1Âã\
  \\131£\1Ãã\131¤\1Äã\131¥\1Åã\131¦\1Æã\131§\1Çã\131¨\1Èã\131©\1Éã\131ª\1Êã\
  \\131«\1Ëã\131¬\1Ìã\131\173\1Íã\131®\1Îã\131¯\1Ïã\131°\1Ðã\131±\1Ñã\131²\
  \\1Òã\131³\1Óã\131´\1Ôã\131µ\1Õã\131¶\1Öã\131»\0\5ã\131¼\0\27ã\131½\0\18ã\
  \\131¾\0\19ä¸\128\5êä¸\129\13\26ä¸\131\105ä¸\135\16\28ä¸\136\11$ä¸\137\9Ï\
  \ä¸\138\11#ä¸\139\6zä¸\141\153ä¸\142\16\158ä¸\144\17 ä¸\145\6\14ä¸\148\7\
  \\14ä¸\149\17¡ä¸\150\11¢ä¸\151\18\128ä¸\152\7µä¸\153\15xä¸\158\11%ä¸¡\16ü\
  \ä¸¦\15\128ä¸ª\17¢ä¸\173\13\6ä¸±\17£ä¸²\88ä¸¶\17¤ä¸¸\7[ä¸¹\12Ïä¸»\10eä¸¼\
  \\17¥ä¸¿\17¦ä¹\130\17§ä¹\131\14Tä¹\133\7¶ä¹\139\14Vä¹\141\14!ä¹\142\8Áä¹\
  \\143\15Òä¹\149\27(ä¹\150\17¨ä¹\151\11&ä¹\152\17©ä¹\153\6sä¹\157\8#ä¹\158\
  \\8îä¹\159\16gä¹¢\19ää¹±\16Ðä¹³\14;ä¹¾\7#äº\128\7\148äº\130\17ªäº\133\17«\
  \äº\134\16ùäº\136\16\156äº\137\12Häº\138\17\173äº\139\10\22äº\140\141äº\
  \\142\17°äº\145\6\29äº\146\8Ýäº\148\8Üäº\149\5ääº\152\17jäº\153\17iäº\155\
  \\9qäº\156\5\159äº\158\17±äº\159\17²äº \17³äº¡\15Óäº¢\17´äº¤\8ðäº¥\5åäº¦\
  \\16\18äº¨\7Üäº«\7Ýäº¬\7Þäº\173\13`äº®\16úäº°\17µäº³\17¶äº¶\17·äºº\11lä»\
  \\128\10\153ä»\129\11mä»\130\17¼ä»\132\17ºä»\134\17»ä»\135\7·ä»\138\9aä»\
  \\139\6®ä»\141\17¹ä»\142\17¸ä»\143\15gä»\148\9åä»\149\9ää»\150\12|ä»\151\
  \\17½ä»\152\154ä»\153\11åä»\157\0\23ä»\158\17¾ä»\159\17Àä»£\12£ä»¤\17\31ä\
  \»¥\5Èä»\173\17¿ä»®\6|ä»°\8\2ä»²\13\7ä»¶\8\143ä»·\17Áä»»\14Cä¼\129\7iä¼\
  \\137\17Âä¼\138\5Éä¼\141\8Þä¼\142\7jä¼\143\15Zä¼\144\14°ä¼\145\7¸ä¼\154\6\
  \¯ä¼\156\17åä¼\157\13 ä¼¯\14\140ä¼°\17Ää¼´\14ºä¼¶\17 ä¼¸\11Lä¼º\9æä¼¼\10\
  \\23ä¼½\6~ä½\131\13Oä½\134\12Áä½\135\17Èä½\141\5Êä½\142\13aä½\143\10\154ä\
  \½\144\9rä½\145\16\131ä½\147\12\140ä½\149\6}ä½\151\17Çä½\153\16\157ä½\154\
  \\17Ãä½\155\17Åä½\156\9¬ä½\157\17Æä½\158\19\131ä½©\17Îä½¯\17Ñä½°\17Ïä½³\6\
  \\128ä½µ\15yä½¶\17Éä½»\17Íä½¼\8ñä½¿\9çä¾\131\7$ä¾\134\17Òä¾\136\17Êä¾\139\
  \\17!ä¾\141\10\24ä¾\143\17Ëä¾\145\17Ðä¾\150\17Óä¾\152\17Ìä¾\155\7ßä¾\157\
  \\5Ëä¾ \7àä¾¡\6\127ä¾«\19\132ä¾\173\16\25ä¾®\15Nä¾¯\8òä¾µ\11Nä¾¶\16õä¾¿\
  \\15\150ä¿\130\8Wä¿\131\12cä¿\132\6¢ä¿\138\10²ä¿\142\17×ä¿\144\17Üä¿\145\
  \\17Úä¿\148\17Õä¿\151\12mä¿\152\17Øä¿\154\17Ûä¿\155\17Ùä¿\157\15\155ä¿\
  \\159\17Öä¿¡\11Mä¿£\16\19ä¿¤\17Ýä¿¥\17Þä¿®\10\131ä¿¯\17ëä¿³\14oä¿µ\15\21ä\
  \¿¶\17æä¿¸\15®ä¿º\6tä¿¾\17êå\128\133\17äå\128\134\17íå\128\137\121å\128\
  \\139\8Âå\128\141\14{å\128\143\23Åå\128\145\17ìå\128\146\13¼å\128\148\17á\
  \å\128\150\8ôå\128\153\8óå\128\154\17ßå\128\159\10Xå\128¡\17çå\128£\15\
  \\173å\128¤\12ìå\128¥\17ãå\128¦\8\145å\128¨\17àå\128©\17èå\128ª\17âå\128«\
  \\17\15å\128¬\17éå\128\173\17`å\128¶\8$å\128¹\8\144å\129\131\17îå\129\135\
  \\17ïå\129\136\17óå\129\137\5Ìå\129\143\15\142å\129\144\17òå\129\149\17ñå\
  \\129\150\17õå\129\154\17ôå\129\156\13bå\129¥\8\146å\129¬\17öå\129²\10Cå\
  \\129´\12då\129µ\13cå\129¶\84å\129¸\17÷å\129½\7\149å\130\128\17øå\130\133\
  \\17úå\130\141\15Ôå\130\145\8\134å\130\152\9Ðå\130\153\14õå\130\154\17ùå\
  \\130¬\9\131å\130\173\16¢å\130²\17üå\130³\18\2å\130´\17ûå\130µ\9\130å\130\
  \·\10Ýå\130¾\8Xå\131\130\18\3å\131\133\8\13å\131\137\18\0å\131\138\18\1å\
  \\131\141\13íå\131\143\12\92å\131\145\7áå\131\149\15ìå\131\150\18\4å\131\
  \\154\16ûå\131\158\18\5å\131£\18\8å\131¥\18\6å\131§\12-å\131\173\18\7å\
  \\131®\18\9å\131µ\18\11å\131¹\18\10å\131»\15\134å\132\128\7\150å\132\129\
  \\18\13å\132\130\18\14å\132\132\6må\132\137\18\12å\132\146\10rå\132\148\
  \\18\17å\132\149\18\16å\132\150\18\15å\132\152\17Ôå\132\154\18\18å\132\
  \\159\10Þå\132¡\18\19å\132ª\16\132å\132²\16Wå\132·\18\21å\132º\18\20å\132\
  \»\18\23å\132¼\18\22å\132¿\18\24å\133\128\18\25å\133\129\5òå\133\131\8³å\
  \\133\132\8Zå\133\133\10\155å\133\134\13\27å\133\135\7âå\133\136\11æå\133\
  \\137\8õå\133\139\9Nå\133\140\18\27å\133\141\16Få\133\142\13¥å\133\144\10\
  \\25å\133\146\18\26å\133\148\18\28å\133\154\13½å\133\156\7\21å\133¢\18\29\
  \å\133¥\14<å\133¨\12\19å\133©\18\31å\133ª\18 å\133«\14ªå\133¬\8öå\133\173\
  \\17Zå\133®\18!å\133±\7äå\133µ\15zå\133¶\12tå\133·\8/å\133¸\13\148å\133¼\
  \\8\147å\134\128\18\34å\134\130\18#å\134\133\14 å\134\134\6>å\134\137\18&\
  \å\134\138\9»å\134\140\18%å\134\141\9\132å\134\143\18'å\134\144\26,å\134\
  \\145\18(å\134\146\15àå\134\147\18)å\134\149\18*å\134\150\18+å\134\151\11\
  \'å\134\153\10Jå\134 \7%å\134¢\18.å\134¤\18,å\134¥\16;å\134¦\18-å\134¨\15\
  \9å\134©\18/å\134ª\180å\134«\181å\134¬\13¾å\134°\185å\134±\183å\134²\184å\
  \\134³\182å\134´\9¡å\134µ\186å\134¶\16hå\134·\17\34å\134½\187å\135\132\11\
  \¦å\135\133\188å\135\134\10¹å\135\137\189å\135\139\13\28å\135\140\16ýå\
  \\135\141\13Àå\135\150\18\131å\135\155\18:å\135\156\31#å\135\157\8\3å\135\
  \ \18;å\135¡\15ýå\135¦\10Èå\135§\12ºå\135©\18=å\135ª\14\34å\135\173\18>å\
  \\135°\18@å\135±\6Íå\135µ\18Aå\135¶\7åå\135¸\14\10å\135¹\6Zå\135º\10¯å\
  \\135½\14\159å\135¾\18Bå\136\128\13Áå\136\131\11nå\136\132\18Cå\136\134\
  \\15jå\136\135\11Øå\136\136\7 å\136\138\7'å\136\139\18Då\136\142\18Få\136\
  \\145\8Yå\136\148\18Eå\136\151\171å\136\157\10Éå\136¤\14»å\136¥\15\138å\
  \\136§\18Gå\136©\16Øå\136ª\18Hå\136®\18Iå\136°\13Þå\136³\18Jå\136¶\11§å\
  \\136·\9¼å\136¸\8\148å\136¹\18Kå\136º\9èå\136»\9Oå\137\131\13då\137\132\
  \\18Må\137\135\12eå\137\138\9\173å\137\139\18Nå\137\140\18Oå\137\141\12\
  \\15å\137\143\18Lå\137\148\18Qå\137\150\15Õå\137\155\9Då\137\158\18På\137\
  \£\8\149å\137¤\9\156å\137¥\14\141å\137©\18Tå\137ª\18Rå\137¯\15[å\137°\11(\
  \å\137±\18[å\137²\7\4å\137³\18Uå\137´\18Så\137µ\12.å\137½\18Wå\137¿\18Vå\
  \\138\131\6ãå\138\135\8\128å\138\136\18\92å\138\137\16ëå\138\141\18Xå\138\
  \\145\18]å\138\146\18Zå\138\148\18Yå\138\155\17\13å\138\159\8÷å\138 \6\
  \\129å\138£\172å\138©\10Õå\138ª\13·å\138«\9Eå\138¬\18`å\138\173\18aå\138±\
  \\17#å\138´\17Jå\138µ\18cå\138¹\8øå\138¼\18bå\138¾\6Îå\139\129\18då\139\
  \\131\15õå\139\133\13:å\139\135\16\133å\139\137\15\151å\139\141\18eå\139\
  \\146\29Óå\139\149\13îå\139\151\18få\139\152\7(å\139\153\161å\139\157\10ß\
  \å\139\158\18gå\139\159\15¥å\139 \18kå\139¢\11¨å\139£\18hå\139¤\8\14å\139\
  \¦\18iå\139§\7)å\139²\8Må\139³\18lå\139µ\18må\139¸\18nå\139¹\18oå\139º\10\
  \Yå\139¾\8ùå\139¿\16\92å\140\129\16få\140\130\145å\140\133\15¯å\140\134\
  \\18på\140\136\18qå\140\141\18så\140\143\18uå\140\144\18tå\140\149\18vå\
  \\140\150\6{å\140\151\15ëå\140\153\9ºå\140\154\18wå\140\157\128å\140 \10à\
  \å\140¡\7çå\140£\18xå\140ª\14Ùå\140¯\18yå\140±\18zå\140³\18{å\140¸\18|å\
  \\140¹\15\3å\140º\8&å\140»\5ãå\140¿\13ýå\141\128\18}å\141\129\10\156å\141\
  \\131\11çå\141\133\18\127å\141\134\18~å\141\135\10áå\141\136\8ßå\141\137\
  \\18\129å\141\138\14¼å\141\141\18\130å\141\145\14Úå\141\146\12rå\141\147\
  \\12¬å\141\148\7æå\141\151\14,å\141\152\12Ðå\141\154\14\142å\141\156\15íå\
  \\141\158\18\132å\141 \11èå\141¦\8Tå\141©\18\133å\141®\18\134å\141¯\6\11å\
  \\141°\5óå\141±\7kå\141³\12få\141´\7°å\141µ\16Ñå\141·\18\137å\141¸\6uå\
  \\141»\18\136å\141¿\7èå\142\130\18\138å\142\132\16oå\142\150\18\139å\142\
  \\152\17\16å\142\154\8úå\142\159\8´å\142 \18\140å\142¥\18\142å\142¦\18\
  \\141å\142¨\11~å\142©\6\24å\142\173\6=å\142®\18\143å\142°\18\144å\142³\8µ\
  \å\142¶\18\145å\142»\7Îå\143\130\9Ñå\143\131\18\146å\143\136\16\20å\143\
  \\137\9så\143\138\7¹å\143\139\16\134å\143\140\12/å\143\141\14½å\143\142\
  \\10{å\143\148\10¦å\143\150\10få\143\151\10så\143\153\10Öå\143\155\14¾å\
  \\143\159\18\149å\143¡\6\34å\143¢\120å\143£\8ûå\143¤\8Ãå\143¥\8%å\143¨\18\
  \\153å\143©\12Àå\143ª\12¼å\143«\7éå\143¬\10âå\143\173\18\154å\143®\18\152\
  \å\143¯\6\130å\143°\12¤å\143±\106å\143²\9êå\143³\6\5å\143¶\7\16å\143·\9Få\
  \\143¸\9éå\143º\18\155å\144\129\18\156å\144\131\7¨å\144\132\6åå\144\136\9\
  \Gå\144\137\7§å\144\138\13]å\144\139\6\4å\144\140\13ïå\144\141\16<å\144\
  \\142\9\0å\144\143\16Ùå\144\144\13¦å\144\145\8üå\144\155\8Nå\144\157\18¥å\
  \\144\159\8!å\144 \15éå\144¦\14Ûå\144©\18¤å\144«\7\92å\144¬\18\159å\144\
  \\173\18 å\144®\18¢å\144¶\18£å\144¸\7ºå\144¹\11\129å\144»\15kå\144¼\18¡å\
  \\144½\18\157å\144¾\8áå\145\128\18\158å\145\130\17Cå\145\134\15°å\145\136\
  \\13få\145\137\8àå\145\138\9På\145\142\18¦å\145\145\14\27å\145\159\18ªå\
  \\145¨\10|å\145ª\10tå\145°\18\173å\145±\18«å\145³\16!å\145µ\18¨å\145¶\18±\
  \å\145·\18¬å\145»\18¯å\145¼\8Äå\145½\16=å\146\128\18°å\146\132\18²å\146\
  \\134\18´å\146\139\9®å\146\140\17aå\146\142\18©å\146\143\18§å\146\144\18³\
  \å\146\146\18®å\146¢\18¶å\146¤\18Âå\146¥\18¸å\146¨\18¼å\146«\18Àå\146¬\18\
  \¹å\146¯\18Ýå\146²\9§å\146³\6Ðå\146¸\18·å\146¼\18Äå\146½\5ôå\146¾\18Ãå\
  \\147\128\5£å\147\129\15)å\147\130\18Áå\147\132\18ºå\147\135\18µå\147\136\
  \\18»å\147\137\9\134å\147\152\18Åå\147¡\5õå\147¢\18Îå\147¥\18Æå\147¦\18Çå\
  \\147¨\10ãå\147©\16\9å\147\173\18Ìå\147®\18Ëå\147²\13\142å\147º\18Íå\147½\
  \\18Êå\148\132\6\19å\148\134\9tå\148\135\11Oå\148\143\18Èå\148\144\13Âå\
  \\148\148\18Éå\148\150\5 å\148®\18Óå\148¯\16\130å\148±\10åå\148³\18Ùå\148\
  \¸\18Øå\148¹\18Ïå\148¾\12\129å\149\128\18Ðå\149\132\12\173å\149\133\18Õå\
  \\149\134\10äå\149\140\18Òå\149\143\16bå\149\147\8[å\149\150\18Öå\149\151\
  \\18×å\149\156\18Ôå\149\157\18Úå\149£\18Ñå\149»\18àå\149¼\18åå\149¾\18áå\
  \\150\128\18Üå\150\131\18æå\150\132\12\16å\150\135\18èå\150\137\9\1å\150\
  \\138\18Þå\150\139\13\29å\150\152\18âå\150\153\18Ûå\150\154\7+å\150\156\7\
  \lå\150\157\7\5å\150\158\18ãå\150\159\18ßå\150§\8\150å\150¨\18éå\150©\18ç\
  \å\150ª\122å\150«\7©å\150¬\7êå\150®\18äå\150°\82å\150¶\6#å\151\132\18íå\
  \\151\133\18ëå\151\135\19%å\151\148\18ðå\151\154\18êå\151\156\18îå\151\
  \\159\18ìå\151£\9ëå\151¤\18ïå\151·\18òå\151¹\18÷å\151½\18õå\151¾\18ôå\152\
  \\134\12Ñå\152\137\6\131å\152\148\18ñå\152\150\18óå\152\151\10æå\152\152\
  \\6\18å\152\155\18öå\152©\6\156å\152¯\19\2å\152±\11:å\152²\18ýå\152´\18ûå\
  \\152¶\18üå\152¸\18þå\153\130\6\28å\153\140\12\24å\153\142\18øå\153\144\
  \\18ùå\153\155\7\26å\153¤\19\1å\153¨\7må\153ª\19\4å\153«\19\0å\153¬\19\3å\
  \\153´\15lå\153¸\14\19å\153º\14¶å\154\128\19\6å\154\134\19\5å\154\135\6äå\
  \\154\138\19\7å\154\143\19\10å\154\148\19\9å\154 \19\8å\154¢\14Xå\154¥\19\
  \\11å\154®\19\12å\154´\19\14å\154¶\19\13å\154¼\19\16å\155\128\19\19å\155\
  \\129\19\17å\155\130\19\15å\155\131\19\18å\155\136\19\20å\155\142\19\21å\
  \\155\145\19\22å\155\147\19\23å\155\151\19\24å\155\152\18$å\155\154\10zå\
  \\155\155\9ìå\155\158\6±å\155 \5öå\155£\12ãå\155®\19\25å\155°\9bå\155²\5Í\
  \å\155³\11}å\155¹\19\26å\155º\8Åå\155½\9Qå\155¿\19\28å\156\128\19\27å\156\
  \\131\15\158å\156\132\19\29å\156\136\19\31å\156\137\19\30å\156\139\19 å\
  \\156\141\19!å\156\143\8\151å\156\146\6@å\156\147\19\34å\156\150\19$å\156\
  \\152\19#å\156\156\19&å\156\159\13¹å\156¦\19'å\156§\5³å\156¨\9\157å\156\
  \\173\8\92å\156°\12îå\156·\19(å\156¸\19)å\156»\19+å\157\128\19,å\157\130\
  \\9¢å\157\135\8\15å\157\138\15Öå\157\142\19*å\157\143\19-å\157\144\9\127å\
  \\157\145\9\2å\157¡\191å\157¤\9cå\157¦\12Òå\157©\19.å\157ª\13Xå\157¿\192å\
  \\158\130\11\130å\158\136\190å\158\137\193å\158\139\8^å\158\147\194å\158 \
  \\195å\158¢\9\3å\158£\6ßå\158¤\197å\158ª\198å\158°\199å\158³\196å\159\128\
  \\19/å\159\131\19:å\159\134\19;å\159\139\16\4å\159\142\11)å\159\146\19=å\
  \\159\147\19>å\159\148\19<å\159\150\19@å\159\156\14Wå\159\159\5æå\159 \15\
  \5å\159£\19Aå\159´\11;å\159·\107å\159¹\14|å\159º\7nå\159¼\9©å \128\15øå \
  \\130\13ðå \133\8\152å \134\12\141å \138\19?å \139\19Bå \149\12\130å \153\
  \\19Cå \157\19Då ¡\19Få ¤\13gå ª\7,å ¯\31\31å °\6Aå ±\15±å ´\11*å µ\13§å \
  \º\9¤å ½\19Lå¡\128\15{å¡\129\17\27å¡\138\6²å¡\139\19Hå¡\145\12\25å¡\146\
  \\19Kå¡\148\13Ãå¡\151\13¨å¡\152\13Äå¡\153\14·å¡\154\13Kå¡\158\9\135å¡¢\19\
  \Gå¡©\6Vå¡«\13\149å¡°\19Iå¡²\19Eå¡µ\11oå¡¹\19Må¡¾\10\173å¢\131\7ëå¢\133\
  \\19Nå¢\147\15¦å¢\151\12]å¢\156\13Då¢\159\19På¢¨\15îå¢«\19Qå¢®\19Vå¢³\15m\
  \å¢¸\19Uå¢¹\19Oå¢º\19Rå¢»\19Tå¢¾\9då£\129\15\135å£\133\19Wå£\135\12äå£\
  \\138\6³å£\140\11+å£\145\19Yå£\147\19Xå£\149\9Hå£\151\19Zå£\152\19\92å£\
  \\153\19[å£\156\19^å£\158\19Så£\159\19`å£¤\19_å£¥\19]å£«\9íå£¬\11på£®\123\
  \å£¯\19aå£°\11ºå£±\5ëå£²\14\132å£·\13Yå£¹\19cå£º\19bå£»\19då£¼\19eå£½\19f\
  \å¤\130\19gå¤\137\15\143å¤\138\19hå¤\143\6\132å¤\144\19iå¤\149\16\155å¤\
  \\150\6Ïå¤\152\18\135å¤\153\10§å¤\154\12}å¤\155\19jå¤\156\16iå¤¢\162å¤¥\
  \\19lå¤§\12¥å¤©\13\150å¤ª\12~å¤«\156å¤¬\19må¤\173\19nå¤®\6[å¤±\108å¤²\19o\
  \å¤·\5Îå¤¸\19på¤¾\19qå¥\132\6Bå¥\135\7oå¥\136\14\30å¥\137\15²å¥\142\19uå¥\
  \\143\124å¥\144\19tå¥\145\8_å¥\148\15úå¥\149\19så¥\151\13Åå¥\152\19wå¥\
  \\154\19vå¥ \19yå¥¢\19xå¥¥\6\92å¥§\19zå¥¨\10çå¥©\19|å¥ª\12Äå¥¬\19{å¥®\15q\
  \å¥³\10×å¥´\13ºå¥¸\19\128å¥½\9\4å¦\129\19\129å¦\130\14@å¦\131\14Üå¦\132\
  \\16Oå¦\138\14Då¦\141\19\138å¦\147\7\151å¦\150\16¤å¦\153\16-å¦\155\19êå¦\
  \\157\19\130å¦£\19\133å¦¥\12\131å¦¨\15×å¦¬\13©å¦²\19\134å¦¹\16\5å¦»\9\136\
  \å¦¾\10èå§\134\19\135å§\137\9ïå§\139\9îå§\144\5·å§\145\8Æå§\147\11©å§\148\
  \\5Ïå§\153\19\139å§\154\19\140å§\156\19\137å§¥\6\23å§¦\7-å§¨\19\136å§ª\16\
  \Cå§«\15\16å§¶\5¦å§»\5÷å§¿\9ðå¨\129\5Ðå¨\131\5¡å¨\137\19\145å¨\145\19\143\
  \å¨\152\16:å¨\154\19\146å¨\156\19\144å¨\159\19\142å¨ \11På¨¥\19\141å¨©\15\
  \\152å¨¯\8âå¨µ\19\150å¨¶\19\151å¨¼\10éå©\128\19\147å©\129\17Kå©\134\14kå©\
  \\137\19\149å©\154\9eå©¢\19\152å©¦\157å©ª\19\153å©¬\19\148å©¿\169åª\146\
  \\14}åª\154\19\154åª\155\15\17åª¼\19\155åª½\19\159åª¾\19\156å«\129\6\133å\
  \«\130\19\158å«\137\109å«\139\19\157å«\140\8\153å«\144\19«å«\150\19¤å«\
  \\151\19¡å«¡\13\4å«£\19 å«¦\19¢å«©\19£å«º\19¥å«»\19¦å¬\137\7på¬\139\19¨å¬\
  \\140\19§å¬\150\19©å¬¢\11,å¬ª\19¬å¬¬\13Zå¬°\6$å¬²\19ªå¬¶\19\173å¬¾\19®å\
  \\173\128\19±å\173\131\19¯å\173\133\19°å\173\144\9ñå\173\145\19²å\173\148\
  \\9\5å\173\149\19³å\173\151\10\26å\173\152\12vå\173\154\19´å\173\155\19µå\
  \\173\156\9ùå\173\157\9\6å\173\159\16På\173£\7\135å\173¤\8Çå\173¥\19¶å\
  \\173¦\6÷å\173©\19·å\173«\12wå\173°\19¸å\173±\19áå\173³\19¹å\173µ\19ºå\
  \\173¸\19»å\173º\19½å®\128\19¾å®\131\19Àå®\133\12®å®\135\6\6å®\136\10gå®\
  \\137\5Àå®\139\126å®\140\7.å®\141\103å®\143\9\7å®\149\13Æå®\151\10\128å®\
  \\152\7/å®\153\13\8å®\154\13hå®\155\5¶å®\156\7\152å®\157\15³å®\159\10@å®¢\
  \\7±å®£\11éå®¤\10:å®¥\16\135å®¦\19Áå®®\7»å®°\9\137å®³\6Ñå®´\6Cå®µ\10êå®¶\
  \\6\134å®¸\19Âå®¹\16¥å®¿\10¨å¯\130\10bå¯\131\19Ãå¯\132\7qå¯\133\14\16å¯\
  \\134\16'å¯\135\19Äå¯\137\19Åå¯\140\158å¯\144\19Çå¯\146\7&å¯\147\85å¯\148\
  \\19Æå¯\155\70å¯\157\11Qå¯\158\19Ëå¯\159\9Àå¯¡\6\135å¯¢\19Êå¯¤\19Èå¯¥\19Ì\
  \å¯¦\19Éå¯§\14Jå¯¨\22Kå¯©\11Rå¯«\19Íå¯®\16þå¯°\19Îå¯³\19Ðå¯µ\13\30å¯¶\19Ï\
  \å¯¸\11¡å¯º\10\27å¯¾\12\142å¯¿\10uå°\129\15Uå°\130\11êå°\132\10Kå°\133\19\
  \Ñå°\134\10ëå°\135\19Òå°\136\19Óå°\137\5Ñå°\138\12xå°\139\11qå°\141\19Ôå°\
  \\142\13ñå°\143\10ìå°\145\10íå°\147\19Õå°\150\11ëå°\154\10îå° \19Öå°¢\19×\
  \å°¤\16^å°¨\19Øå°\173\8\4å°±\10\129å°¸\19Ùå°¹\19Úå°º\10Zå°»\11Kå°¼\142å°½\
  \\11så°¾\14öå°¿\14Aå±\128\8\7å±\129\19Ûå±\133\7Ïå±\134\19Üå±\136\8<å±\138\
  \\14\13å±\139\6nå±\141\9òå±\142\19Ýå±\143\19àå±\144\19ßå±\145\8;å±\147\19\
  \Þå±\149\13\151å±\158\12nå± \13ªå±¡\10Få±¤\127å±¥\16Úå±¬\19âå±®\19ãå±¯\14\
  \\20å±±\9Òå±¶\19åå±¹\19æå²\140\19çå²\144\7rå²\145\19èå²\148\19éå²¡\6jå²¨\
  \\12\26å²©\7bå²«\19ëå²¬\16&å²±\12\144å²³\6øå²¶\19íå²·\19ïå²¸\7]å²»\19ìå²¼\
  \\19îå²¾\19ñå³\133\19ðå³\135\19òå³\153\19óå³ \13ûå³¡\7ìå³¨\6£å³©\19ôå³ª\
  \\19ùå³\173\19÷å³¯\15µå³°\15´å³¶\13Çå³º\19öå³»\10³å³½\19õå´\135\11\146å´\
  \\139\19úå´\142\9¨å´\145\20\0å´\148\20\1å´\149\19ûå´\150\6Òå´\151\19üå´\
  \\152\20\5å´\153\20\4å´\154\20\3å´\155\19ÿå´\159\19þå´¢\20\2å´©\15¶åµ\139\
  \\20\9åµ\140\20\6åµ\142\20\8åµ\144\16Òåµ\146\20\7åµ\156\19ýåµ©\11\147åµ¬\
  \\20\10åµ¯\9uåµ³\20\11åµ¶\20\12å¶\130\20\15å¶\132\20\14å¶\135\20\13å¶\139\
  \\13Èå¶\140\19øå¶\144\20\21å¶\157\20\17å¶¢\20\16å¶¬\20\18å¶®\20\19å¶·\20\
  \\22å¶º\17$å¶¼\20\23å¶½\20\20å·\137\20\24å·\140\7^å·\141\20\25å·\146\20\
  \\27å·\147\20\26å·\150\20\28å·\155\20\29å·\157\11ìå·\158\10\130å·¡\10Äå·£\
  \\12Cå·¥\9\8å·¦\9vå·§\9\9å·¨\7Ðå·«\20\30å·®\9wå·±\8Èå·²\20\31å·³\16$å·´\
  \\14bå·µ\20 å··\9\10å·»\7*å·½\12Æå·¾\8\16å¸\130\9óå¸\131\15:å¸\134\14¿å¸\
  \\139\20!å¸\140\7så¸\145\20$å¸\150\13\31å¸\153\20#å¸\154\20\34å¸\155\20%å\
  \¸\157\13iå¸¥\11\131å¸«\9ôå¸\173\11Èå¸¯\12\145å¸°\7\129å¸³\13 å¸¶\20&å¸·\
  \\20'å¸¸\11-å¸½\15Øå¹\128\20*å¹\131\20)å¹\132\20(å¹\133\15]å¹\135\201å¹\
  \\140\15ùå¹\142\20+å¹\148\20-å¹\149\16\11å¹\151\20,å¹\159\20.å¹¡\14¦å¹¢\
  \\20/å¹£\15|å¹¤\200å¹²\71å¹³\15}å¹´\14Nå¹µ\202å¹¶\203å¹¸\9\11å¹¹\72å¹º\20\
  \4å¹»\8¶å¹¼\16£å¹½\16\136å¹¾\7tå¹¿\206åº\129\13!åº\131\9\12åº\132\10ïåº\
  \\135\14Ýåº\138\10ðåº\143\10Øåº\149\13jåº\150\15·åº\151\13\152åº\154\9\13\
  \åº\156\15;åº \207åº¦\13¸åº§\9\128åº«\8Éåº\173\13kåºµ\5Áåº¶\10Îåº·\9\14åº\
  \¸\16¦å»\129\208å»\130\209å»\131\14på»\136\20:å»\137\175å»\138\17Lå»\143\
  \\20<å»\144\20;å»\147\6æå»\150\20@å»\154\20Cå»\155\20Då»\157\20Bå»\159\15\
  \\31å» \10ñå»¡\20Få»¢\20Eå»£\20Aå»¨\20Gå»©\20Hå»¬\20Iå»°\20Lå»±\20Jå»³\20\
  \Kå»´\20Må»¶\6Då»·\13lå»¸\20Nå»º\8\154å»»\6´å»¼\14Uå»¾\20Oå»¿\149å¼\129\
  \\15\153å¼\131\20På¼\132\17Må¼\137\20Qå¼\138\15~å¼\139\20Tå¼\140\17\159å¼\
  \\141\17¯å¼\143\10.å¼\144\143å¼\145\20Uå¼\147\7¼å¼\148\13\34å¼\149\5øå¼\
  \\150\20Vå¼\151\15då¼\152\9\15å¼\155\12ïå¼\159\13må¼¥\16må¼¦\8·å¼§\8Êå¼©\
  \\20Wå¼\173\20Xå¼¯\20^å¼±\10cå¼µ\13#å¼·\7íå¼¸\20Yå¼¼\15\10å¼¾\12åå½\129\
  \\20Zå½\136\20[å½\138\7îå½\140\20\92å½\142\20]å½\145\20_å½\147\13Öå½\150\
  \\20`å½\151\20aå½\153\20bå½\156\20Så½\157\20Rå½¡\20cå½¢\8`å½¦\15\6å½©\9\
  \\138å½ª\15\22å½«\13$å½¬\15*å½\173\20då½°\10òå½±\6%å½³\20eå½·\20få½¹\16på\
  \½¼\14Þå½¿\20iå¾\128\6]å¾\129\11ªå¾\130\20hå¾\131\20gå¾\132\8aå¾\133\12\
  \\146å¾\135\20må¾\136\20kå¾\138\20jå¾\139\16åå¾\140\8ãå¾\144\10Ùå¾\145\20\
  \lå¾\146\13«å¾\147\10\157å¾\151\13þå¾\152\20på¾\153\20oå¾\158\20nå¾ \20qå\
  \¾¡\8äå¾¨\20rå¾©\15\92å¾ª\10ºå¾\173\20så¾®\14÷å¾³\13ÿå¾´\13%å¾¹\13\143å¾¼\
  \\20tå¾½\7\138å¿\131\11Så¿\133\15\11å¿\140\7uå¿\141\14Eå¿\150\20uå¿\151\9\
  \õå¿\152\15Ùå¿\153\15Úå¿\156\6^å¿\157\20zå¿ \13\9å¿¤\20wå¿«\6µå¿°\20«å¿±\
  \\20yå¿µ\14Oå¿¸\20xå¿»\20vå¿½\9Zå¿¿\20|æ\128\142\20\131æ\128\143\20\137æ\
  \\128\144\20\129æ\128\146\13»æ\128\149\20\134æ\128\150\15<æ\128\153\20\
  \\128æ\128\155\20\133æ\128\156\17%æ\128\157\9öæ\128 \12\147æ\128¡\20}æ\
  \\128¥\7½æ\128¦\20\136æ\128§\11«æ\128¨\6Eæ\128©\20\130æ\128ª\6¶æ\128«\20\
  \\135æ\128¯\7ïæ\128±\20\132æ\128º\20\138æ\129\129\20\140æ\129\130\20\150æ\
  \\129\131\20\148æ\129\134\20\145æ\129\138\20\144æ\129\139\176æ\129\141\20\
  \\146æ\129\144\7ðæ\129\146\9\16æ\129\149\10Úæ\129\153\20\153æ\129\154\20\
  \\139æ\129\159\20\143æ\129 \20~æ\129¢\6¸æ\129£\20\147æ\129¤\20\149æ\129¥\
  \\12ðæ\129¨\9fæ\129©\6væ\129ª\20\141æ\129«\20\152æ\129¬\20\151æ\129\173\7\
  \ñæ\129¯\12gæ\129°\7\6æ\129µ\8bæ\129·\20\142æ\130\129\20\154æ\130\131\20\
  \\157æ\130\132\20\159æ\130\137\10;æ\130\139\20¥æ\130\140\13næ\130\141\20\
  \\155æ\130\146\20£æ\130\148\6·æ\130\150\20¡æ\130\151\20¢æ\130\154\20\158æ\
  \\130\155\20 æ\130\159\8åæ\130 \16\137æ\130£\73æ\130¦\68æ\130§\20¤æ\130©\
  \\14Yæ\130ª\5«æ\130²\14ßæ\130³\20{æ\130´\20ªæ\130µ\20®æ\130¶\16cæ\130¸\20\
  \§æ\130¼\13Éæ\130½\20¬æ\131\133\11.æ\131\134\20\173æ\131\135\14\21æ\131\
  \\145\17fæ\131\147\20©æ\131\152\20¯æ\131\154\9[æ\131\156\11Éæ\131\159\5Òæ\
  \\131 \20¨æ\131¡\20¦æ\131£\129æ\131§\20\156æ\131¨\9Óæ\131°\12\132æ\131±\
  \\20»æ\131³\12:æ\131´\20¶æ\131¶\20³æ\131·\20´æ\131¹\10dæ\131º\20·æ\131»\
  \\20ºæ\132\128\20µæ\132\129\10\132æ\132\131\20¸æ\132\134\20²æ\132\136\16z\
  \æ\132\137\16yæ\132\141\20¼æ\132\142\20½æ\132\143\5Óæ\132\149\20±æ\132\
  \\154\80æ\132\155\5¤æ\132\159\74æ\132¡\20¹æ\132§\20Áæ\132¨\20Àæ\132¬\20Åæ\
  \\132´\20Ææ\132¼\20Äæ\132½\20Çæ\132¾\20¿æ\132¿\20Ãæ\133\130\20Èæ\133\132\
  \\20Éæ\133\135\20¾æ\133\136\10\28æ\133\138\20Âæ\133\139\12\148æ\133\140\9\
  \\17æ\133\141\20°æ\133\142\11Tæ\133\147\20Öæ\133\149\15§æ\133\152\20Ìæ\
  \\133\153\20Íæ\133\154\20Îæ\133\157\20Õæ\133\159\20Ôæ\133¢\16\29æ\133£\75\
  \æ\133¥\20Òæ\133§\8dæ\133¨\6Óæ\133«\20Ïæ\133®\16öæ\133¯\20Ñæ\133°\5Ôæ\133\
  \±\20Óæ\133³\20Êæ\133´\20Ðæ\133µ\20×æ\133¶\8cæ\133·\20Ëæ\133¾\16¼æ\134\
  \\130\16\138æ\134\135\20Úæ\134\138\20Þæ\134\142\12^æ\134\144\177æ\134\145\
  \\20ßæ\134\148\20Üæ\134\150\20Ùæ\134\153\20Øæ\134\154\20Ýæ\134¤\15næ\134§\
  \\13òæ\134©\8eæ\134«\20àæ\134¬\20Ûæ\134®\20áæ\134²\8\155æ\134¶\6oæ\134º\
  \\20éæ\134¾\76æ\135\131\20çæ\135\134\20èæ\135\135\9gæ\135\136\20ææ\135\
  \\137\20äæ\135\138\20ãæ\135\139\20êæ\135\140\20âæ\135\141\20ìæ\135\144\6¹\
  \æ\135£\20îæ\135¦\20íæ\135²\13&æ\135´\20ñæ\135¶\20ïæ\135·\20åæ\135¸\8\156\
  \æ\135º\20ðæ\135¼\20ôæ\135½\20óæ\135¾\20õæ\135¿\20òæ\136\128\20öæ\136\136\
  \\20÷æ\136\137\20øæ\136\138\15¨æ\136\140\20úæ\136\141\20ùæ\136\142\10\158\
  \æ\136\144\11¬æ\136\145\6¤æ\136\146\6ºæ\136\148\20ûæ\136\150\5½æ\136\154\
  \\11Êæ\136\155\20üæ\136\157\28Aæ\136\158\21\0æ\136\159\8\129æ\136¡\21\1æ\
  \\136¦\11íæ\136ª\21\2æ\136®\21\3æ\136¯\7\153æ\136°\21\4æ\136²\21\5æ\136³\
  \\21\6æ\136´\12\149æ\136¸\8Ëæ\136»\16_æ\136¿\15Ûæ\137\128\10Êæ\137\129\21\
  \\7æ\137\135\11îæ\137\136\28ûæ\137\137\14àæ\137\139\10hæ\137\141\9\139æ\
  \\137\142\21\8æ\137\147\12\133æ\137\149\15eæ\137\152\12¯æ\137\155\21\11æ\
  \\137\158\21\9æ\137 \21\12æ\137£\21\10æ\137¨\21\13æ\137®\15oæ\137±\5µæ\
  \\137¶\15=æ\137¹\14áæ\137¼\21\14æ\137¾\21\17æ\137¿\10óæ\138\128\7\154æ\
  \\138\130\21\15æ\138\131\21\22æ\138\132\10ôæ\138\137\21\16æ\138\138\14cæ\
  \\138\145\16½æ\138\146\21\18æ\138\147\21\19æ\138\148\21\23æ\138\149\13Êæ\
  \\138\150\21\20æ\138\151\9\18æ\138\152\11Üæ\138\155\21%æ\138\156\14²æ\138\
  \\158\12°æ\138«\14âæ\138¬\21kæ\138±\15¸æ\138µ\13oæ\138¹\16\21æ\138»\21\26\
  \æ\138¼\6_æ\138½\13\10æ\139\130\21#æ\139\133\12Óæ\139\134\21\29æ\139\135\
  \\21$æ\139\136\21\31æ\139\137\21&æ\139\138\21\34æ\139\140\21!æ\139\141\14\
  \\143æ\139\143\21\27æ\139\144\6»æ\139\145\21\25æ\139\146\7Ñæ\139\147\12±æ\
  \\139\148\21\21æ\139\151\21\24æ\139\152\9\19æ\139\153\11Ùæ\139\155\10õæ\
  \\139\156\21 æ\139\157\14qæ\139 \7Òæ\139¡\6çæ\139¬\7\7æ\139\173\11@æ\139®\
  \\21(æ\139¯\21-æ\139±\21)æ\139³\8\157æ\139µ\21.æ\139¶\9Áæ\139·\9Iæ\139¾\
  \\10\133æ\139¿\21\28æ\140\129\10\29æ\140\130\21+æ\140\135\9÷æ\140\136\21,\
  \æ\140\137\5Âæ\140\140\21'æ\140\145\13'æ\140\153\7Óæ\140\159\7òæ\140§\21*\
  \æ\140¨\5¥æ\140«\9\129æ\140¯\11Uæ\140º\13pæ\140½\14Òæ\140¾\210æ\140¿\12=æ\
  \\141\137\12hæ\141\140\9Êæ\141\141\211æ\141\143\213æ\141\144\21/æ\141\149\
  \\15\159æ\141\151\13;æ\141\156\12;æ\141§\15¹æ\141¨\10Læ\141©\21@æ\141«\21\
  \>æ\141®\11\152æ\141²\8\158æ\141¶\218æ\141·\10÷æ\141º\14&æ\141»\14Pæ\142\
  \\128\216æ\142\131\12<æ\142\136\10væ\142\137\21;æ\142\140\10öæ\142\142\21\
  \5æ\142\143\21:æ\142\146\14ræ\142\150\214æ\142\152\8@æ\142\155\6üæ\142\
  \\159\21<æ\142 \16éæ\142¡\9\140æ\142¢\12Ôæ\142£\219æ\142¥\11Úæ\142§\9\20æ\
  \\142¨\11\132æ\142©\6Fæ\142ª\12\27æ\142«\217æ\142¬\7¤æ\142²\8fæ\142´\13Mæ\
  \\142µ\21=æ\142»\12>æ\142¾\21Aæ\143\128\21Cæ\143\131\12uæ\143\132\21Iæ\
  \\143\134\21Dæ\143\137\21Fæ\143\143\15 æ\143\144\13qæ\143\146\21Gæ\143\
  \\150\16\139æ\143\154\16§æ\143\155\77æ\143¡\5¬æ\143£\21Eæ\143©\21Bæ\143®\
  \\7væ\143´\6Gæ\143¶\21Hæ\143º\16¨æ\144\134\21Læ\144\141\12yæ\144\143\21Sæ\
  \\144\147\21Mæ\144\150\21Jæ\144\151\21Qæ\144\156\212æ\144¦\21Næ\144¨\21Ræ\
  \\144¬\14Àæ\144\173\13Ëæ\144´\21Kæ\144¶\21Oæ\144º\8gæ\144¾\9¯æ\145\130\11\
  \Ûæ\145\142\21Wæ\145\152\13\133æ\145§\21Tæ\145©\16\0æ\145¯\21Uæ\145¶\21Væ\
  \\145¸\16Læ\145º\11 æ\146\131\8\130æ\146\136\21]æ\146\146\9Ôæ\146\147\21Z\
  \æ\146\149\21Yæ\146\154\14Qæ\146\158\13óæ\146¤\13\144æ\146¥\21[æ\146©\21\
  \\92æ\146«\15Oæ\146\173\14dæ\146®\9Âæ\146°\11ïæ\146²\15ïæ\146¹\6èæ\146»\
  \\21cæ\146¼\21^æ\147\129\16©æ\147\130\21eæ\147\133\21aæ\147\135\21bæ\147\
  \\141\12@æ\147\146\21`æ\147\148\21\30æ\147\152\21dæ\147\154\21_æ\147 \21i\
  \æ\147¡\21jæ\147¢\13\134æ\147£\21læ\147¦\9Ãæ\147§\21gæ\147¬\7\155æ\147¯\
  \\21mæ\147±\21fæ\147²\21qæ\147´\21pæ\147¶\21oæ\147º\21ræ\147½\21tæ\147¾\
  \\11/æ\148\128\21sæ\148\133\21wæ\148\152\21uæ\148\156\21væ\148\157\21Pæ\
  \\148£\21yæ\148¤\21xæ\148ª\21Xæ\148«\21zæ\148¬\21næ\148¯\9øæ\148´\21{æ\
  \\148µ\21|æ\148¶\21~æ\148·\21}æ\148¸\21\127æ\148¹\6¼æ\148»\9\21æ\148¾\15º\
  \æ\148¿\11\173æ\149\133\8Ìæ\149\136\21\129æ\149\141\21\132æ\149\143\151æ\
  \\149\145\7¾æ\149\149\21\131æ\149\150\21\130æ\149\151\14sæ\149\152\21\133\
  \æ\149\153\7óæ\149\157\21\135æ\149\158\21\134æ\149¢\78æ\149£\9Õæ\149¦\14\
  \\22æ\149¬\8hæ\149°\11\148æ\149²\21\136æ\149´\11®æ\149µ\13\135æ\149·\15>æ\
  \\149¸\21\137æ\150\130\21\138æ\150\131\21\139æ\150\135\15væ\150\136\19¼æ\
  \\150\137\11Äæ\150\140\15+æ\150\142\9\150æ\150\144\14ãæ\150\145\14Áæ\150\
  \\151\13¬æ\150\153\16ÿæ\150\155\21\141æ\150\156\10Næ\150\159\21\142æ\150¡\
  \\5´æ\150¤\8\18æ\150¥\11Ëæ\150§\15@æ\150«\21\143æ\150¬\9áæ\150\173\12ææ\
  \\150¯\9úæ\150°\11Væ\150·\21\144æ\150¹\15»æ\150¼\6Wæ\150½\9ûæ\151\129\21\
  \\147æ\151\131\21\145æ\151\132\21\148æ\151\133\16÷æ\151\134\21\146æ\151\
  \\139\11ùæ\151\140\21\149æ\151\143\12pæ\151\146\21\150æ\151\151\7xæ\151\
  \\153\21\152æ\151\155\21\151æ\151 \21\153æ\151¡\21\154æ\151¢\7yæ\151¥\14:\
  \æ\151¦\12Õæ\151§\7Ìæ\151¨\9üæ\151©\12Aæ\151¬\10»æ\151\173\5®æ\151±\21\
  \\155æ\151º\6`æ\151»\21\159æ\152\130\9\22æ\152\131\21\158æ\152\134\9iæ\
  \\152\135\10øæ\152\138\21\157æ\152\140\10ùæ\152\142\16>æ\152\143\9hæ\152\
  \\147\5Õæ\152\148\11Ìæ\152\156\21¤æ\152\159\11¯æ\152 \6&æ\152¥\10´æ\152§\
  \\16\6æ\152¨\9°æ\152\173\10úæ\152¯\11¥æ\152´\21£æ\152µ\21¡æ\152¶\21¢æ\152\
  \¼\13\11æ\152¿\21Åæ\153\129\21¨æ\153\130\10\30æ\153\131\9\23æ\153\132\21¦\
  \æ\153\137\21§æ\153\139\11Wæ\153\143\21¥æ\153\146\9Îæ\153\157\21ªæ\153\
  \\158\21©æ\153\159\21®æ\153¢\21¯æ\153¤\21«æ\153¦\6Áæ\153§\21¬æ\153¨\21\
  \\173æ\153©\14Óæ\153®\15Aæ\153¯\8iæ\153°\21°æ\153´\11°æ\153¶\10ûæ\153º\12\
  \ñæ\154\129\8\5æ\154\131\21±æ\154\132\21µæ\154\135\6\137æ\154\136\21²æ\
  \\154\137\21´æ\154\142\21³æ\154\145\10Ëæ\154\150\12çæ\154\151\5Ãæ\154\152\
  \\21¶æ\154\157\21·æ\154¢\13(æ\154¦\17/æ\154«\9âæ\154®\15©æ\154´\15Üæ\154¸\
  \\21Áæ\154¹\21¹æ\154¼\21¼æ\154¾\21»æ\155\129\21¸æ\155\132\21Àæ\155\135\14\
  \\28æ\155\137\21ºæ\155\150\21Âæ\155\153\10Ìæ\155\154\21Ãæ\155\156\16ªæ\
  \\155\157\14\152æ\155 \21Äæ\155¦\21Ææ\155©\21Çæ\155°\21Èæ\155²\8\8æ\155³\
  \\6'æ\155´\9\24æ\155µ\21Éæ\155·\21Êæ\155¸\10Ñæ\155¹\12Bæ\155¼\18\150æ\155\
  \½\12\29æ\155¾\12\28æ\155¿\12\150æ\156\128\9\133æ\156\131\17ðæ\156\136\8\
  \\142æ\156\137\16\140æ\156\139\15¼æ\156\141\15^æ\156\143\21Ëæ\156\148\9±æ\
  \\156\149\13=æ\156\150\21Ìæ\156\151\17Næ\156\155\15Ýæ\156\157\13)æ\156\
  \\158\21Íæ\156\159\7zæ\156¦\21Îæ\156§\21Ïæ\156¨\16Xæ\156ª\16\34æ\156«\16\
  \\22æ\156¬\15ûæ\156\173\9Äæ\156®\21Ñæ\156±\10iæ\156´\15ðæ\156¶\21Óæ\156·\
  \\21Öæ\156¸\21Õæ\156º\7wæ\156½\7Àæ\156¿\21Òæ\157\129\21Ôæ\157\134\21×æ\
  \\157\137\11\153æ\157\142\16Ûæ\157\143\5Çæ\157\144\9\158æ\157\145\12zæ\
  \\157\147\10[æ\157\150\111æ\157\153\21Úæ\157\156\13\173æ\157\158\21Øæ\157\
  \\159\12iæ\157 \21Ùæ\157¡\110æ\157¢\16[æ\157£\21Ûæ\157¤\21Üæ\157¥\16Èæ\
  \\157ª\21áæ\157\173\9\25æ\157¯\14tæ\157°\21Þæ\157±\13Ìæ\157²\21\156æ\157³\
  \\21 æ\157µ\7®æ\157·\14fæ\157¼\21àæ\157¾\10üæ\157¿\14Âæ\158\133\21ææ\158\
  \\135\14øæ\158\137\21Ýæ\158\139\21ãæ\158\140\21âæ\158\144\11Íæ\158\149\16\
  \\13æ\158\151\17\17æ\158\154\16\7æ\158\156\6\138æ\158\157\9ýæ\158 \17gæ\
  \\158¡\21åæ\158¢\11\149æ\158¦\21äæ\158©\21ßæ\158¯\8Íæ\158³\21ëæ\158´\21éæ\
  \\158¶\6\139æ\158·\21çæ\158¸\21íæ\158¹\21óæ\159\129\12\134æ\159\132\15\
  \\127æ\159\134\21õæ\159\138\15\1æ\159\142\21ôæ\159\143\14\144æ\159\144\15\
  \Þæ\159\145\79æ\159\147\11õæ\159\148\10\159æ\159\152\13Qæ\159\154\16\141æ\
  \\159\157\21ðæ\159\158\21ïæ\159¢\21ñæ\159¤\21îæ\159§\21öæ\159©\21ìæ\159¬\
  \\21êæ\159®\21òæ\159¯\21èæ\159±\13\12æ\159³\16væ\159´\10Dæ\159µ\9²æ\159»\
  \\9xæ\159¾\16\15æ\159¿\6àæ \130\13Læ \131\14\8æ \132\6(æ \147\11ðæ \150\
  \\11²æ \151\8Iæ \158\21øæ ¡\9\26æ ¢\7\28æ ©\21úæ ª\7\20æ «\22\1æ ²\21ýæ ´\
  \\11ñæ ¸\6êæ ¹\9jæ ¼\6éæ ½\9\141æ¡\128\21ûæ¡\129\8\133æ¡\130\8jæ¡\131\13Í\
  \æ¡\134\21ùæ¡\136\5Äæ¡\141\21üæ¡\142\21þæ¡\144\8\11æ¡\145\8Kæ¡\147\7:æ¡\
  \\148\7ªæ¡\153\22\2æ¡\156\9·æ¡\157\16\17æ¡\159\9Öæ¡£\22\3æ¡§\15\15æ¡´\22\
  \\15æ¡¶\6qæ¡·\22\4æ¡¾\22\21æ¡¿\22\5æ¢\129\17\0æ¢\131\22\12æ¢\133\14~æ¢\
  \\141\22\20æ¢\143\22\7æ¢\147\5²æ¢\148\22\9æ¢\151\9\27æ¢\155\22\11æ¢\157\
  \\22\10æ¢\159\22\6æ¢ \22\17æ¢¢\10ýæ¢¦\19kæ¢§\8ææ¢¨\16Üæ¢\173\22\8æ¢¯\13ræ\
  \¢°\6Âæ¢±\9kæ¢³\22\0æ¢µ\22\16æ¢¶\7\1æ¢¹\22\14æ¢º\22\18æ¢¼\13Îæ£\132\7|æ£\
  \\134\220æ£\137\16Gæ£\138\22\23æ£\139\7{æ£\141\22\30æ£\146\15ßæ£\148\22\
  \\31æ£\149\22!æ£\151\22%æ£\152\22\25æ£\154\12Éæ£\159\13Ïæ£ \22)æ£¡\22\28æ\
  \££\22&æ£§\22 æ£®\11Xæ£¯\22*æ£²\11±æ£¹\22(æ£º\7;æ¤\128\17oæ¤\129\22\22æ¤\
  \\132\22$æ¤\133\5Öæ¤\136\22\24æ¤\139\168æ¤\140\22\29æ¤\141\11Aæ¤\142\13Eæ\
  \¤\143\22\19æ¤\146\22#æ¤\153\11\154æ¤\154\22-æ¤\155\7\17æ¤\156\8\159æ¤¡\
  \\22/æ¤¢\22\26æ¤£\22.æ¤¥\22'æ¤¦\22\27æ¤¨\22+æ¤ª\22,æ¤°\22=æ¤´\14\12æ¤¶\22\
  \\34æ¤¹\229æ¤½\22;æ¤¿\13Væ¥\138\16«æ¥\147\15Væ¥\148\226æ¥\149\12\136æ¥\
  \\153\22<æ¥\154\12\30æ¥\156\223æ¥\157\22@æ¥\158\22?æ¥ \14-æ¥¡\22>æ¥¢\14(æ\
  \¥ª\22Bæ¥«\225æ¥\173\8\6æ¥®\228æ¥¯\10¼æ¥³\14\128æ¥´\22:æ¥µ\8\9æ¥·\222æ¥¸\
  \\224æ¥¹\221æ¥¼\17Oæ¥½\6ùæ¥¾\227æ¦\129\22Aæ¦\130\6Ôæ¦\138\9¥æ¦\142\6<æ¦\
  \\145\22Ræ¦\148\17Pæ¦\149\22Uæ¦\155\11Yæ¦\156\22Tæ¦ \22Sæ¦§\22Pæ¦®\22Dæ¦±\
  \\22aæ¦²\22Cæ¦´\22Væ¦»\22Næ¦¾\22Iæ¦¿\22Fæ§\129\22Gæ§\131\22Oæ§\135\31 æ§\
  \\138\22Læ§\139\9\28æ§\140\13Fæ§\141\12Dæ§\142\22Jæ§\144\22Eæ§\147\22Hæ§\
  \\152\16¬æ§\153\16\10æ§\157\22Mæ§\158\22Wæ§§\22_æ§¨\22Xæ§«\22eæ§\173\22cæ\
  \§²\22^æ§¹\22]æ§»\13Næ§½\12Eæ§¿\22[æ¨\130\22Yæ¨\133\22`æ¨\138\22fæ¨\139\
  \\14óæ¨\140\22læ¨\146\22gæ¨\147\22jæ¨\148\22dæ¨\151\13\20æ¨\153\15\23æ¨\
  \\155\22Zæ¨\158\22bæ¨\159\10þæ¨¡\16Mæ¨¢\22væ¨£\22iæ¨©\8 æ¨ª\6aæ¨«\6þæ¨®\
  \\22Qæ¨µ\10ÿæ¨¶\22næ¨¸\22uæ¨¹\10wæ¨º\7\18æ¨½\12Íæ©\132\22kæ©\135\22pæ©\
  \\136\22tæ©\139\7ôæ©\152\7«æ©\153\22ræ©\159\7\128æ©¡\14\9æ©¢\22qæ©¦\22sæ©\
  \²\22mæ©¸\22oæ©¿\7\0æª\128\12èæª\132\22zæª\141\22xæª\142\8çæª\144\22wæª\
  \\151\22\128æª\156\21÷æª \22yæª¢\22{æª£\22|æªª\22\139æª¬\22\135æª®\22\13æ\
  \ª³\22\134æª¸\22\133æª»\22\130æ«\129\22hæ«\130\22\132æ«\131\22\131æ«\145\
  \\22\137æ«\147\17Eæ«\154\22\140æ«\155\89æ«\158\22\136æ«\159\22\138æ«¨\14¥\
  \æ«ª\22\141æ«º\22\145æ«»\22\142æ¬\132\16Óæ¬\133\22\143æ¬\138\22\92æ¬\146\
  \\22\146æ¬\150\22\147æ¬\157\6\20æ¬\159\22\149æ¬ \8\135æ¬¡\10\31æ¬£\8\19æ¬\
  \§\6bæ¬²\16¾æ¬·\22\151æ¬¸\22\150æ¬¹\22\153æ¬º\7\156æ¬½\8\20æ¬¾\7<æ\173\
  \\131\22\156æ\173\135\22\155æ\173\137\22\157æ\173\140\6\140æ\173\142\12Öæ\
  \\173\144\22\158æ\173\147\7=æ\173\148\22 æ\173\153\22\159æ\173\155\22¡æ\
  \\173\159\22¢æ\173¡\22£æ\173¢\9þæ\173£\11³æ\173¤\9_æ\173¦\15Pæ\173©\15 æ\
  \\173ª\17cæ\173¯\10\21æ\173³\9\142æ\173´\170æ\173¸\22¤æ\173¹\22¥æ\173»\10\
  \\0æ\173¿\22¦æ®\128\22§æ®\131\22©æ®\132\22¨æ®\134\15÷æ®\137\10½æ®\138\10j\
  \æ®\139\9ãæ®\141\22ªæ®\149\22¬æ®\150\11Bæ®\152\22«æ®\158\22\173æ®¤\22®æ®ª\
  \\22¯æ®«\22°æ®¯\22±æ®±\22³æ®²\22²æ®³\22´æ®´\6cæ®µ\12éæ®·\22µæ®º\9Åæ®»\6ëæ\
  \®¼\22¶æ®¿\13¡æ¯\128\19Jæ¯\133\7\130æ¯\134\22·æ¯\139\22¸æ¯\141\15ªæ¯\142\
  \\16\8æ¯\146\14\5æ¯\147\22¹æ¯\148\14äæ¯\152\14ùæ¯\155\16Qæ¯\159\22ºæ¯«\22\
  \¼æ¯¬\22»æ¯¯\22¾æ¯³\22½æ°\136\22Áæ°\143\10\1æ°\145\16/æ°\147\22Âæ°\148\22\
  \Ãæ°\151\7\131æ°\155\22Äæ°£\22Ææ°¤\22Åæ°´\11\133æ°·\15\24æ°¸\6)æ°¾\14Ãæ±\
  \\128\13sæ±\129\10 æ±\130\7Áæ±\142\14Äæ±\144\10,æ±\149\22Èæ±\151\7>æ±\154\
  \\6Xæ±\157\140æ±\158\22Çæ±\159\9\29æ± \12òæ±¢\22Éæ±¨\22Ñæ±ª\22Êæ±°\12\127\
  \æ±²\7Âæ±³\22Òæ±º\8\136æ±½\7\132æ±¾\22Ðæ²\129\22Îæ²\130\22Ëæ²\131\16Àæ²\
  \\136\13>æ²\140\14\23æ²\141\22Ìæ²\144\22Ôæ²\146\22Óæ²\147\8Bæ²\150\6kæ²\
  \\153\9yæ²\154\22Íæ²\155\22Ïæ²¡\15öæ²¢\12²æ²«\16\23æ²®\22Üæ²±\22Ýæ²³\6\
  \\141æ²¸\15fæ²¹\16{æ²º\22ßæ²»\10!æ²¼\11\0æ²½\22Øæ²¾\22Þæ²¿\6Hæ³\129\7õæ³\
  \\132\22Õæ³\133\22Úæ³\137\11òæ³\138\14\145æ³\140\14åæ³\147\22×æ³\149\15Àæ\
  \³\151\22Ùæ³\153\22âæ³\155\22àæ³\157\22Ûæ³¡\15Áæ³¢\14gæ³£\7Ãæ³¥\13\132æ³¨\
  \\13\13æ³ª\22ãæ³¯\22áæ³°\12\151æ³±\22Öæ³³\6*æ´\139\16\173æ´\140\22îæ´\146\
  \\22íæ´\151\11ôæ´\153\22êæ´\155\16Ìæ´\158\13ôæ´\159\22äæ´¥\13Cæ´©\6+æ´ª\9\
  \\30æ´«\22çæ´²\10\134æ´³\22ìæ´µ\22ëæ´¶\22ææ´¸\22éæ´»\7\8æ´½\22èæ´¾\14hæµ\
  \\129\16ìæµ\132\112æµ\133\11óæµ\153\22ôæµ\154\22òæµ\156\15,æµ£\22ïæµ¤\22ñ\
  \æµ¦\6\25æµ©\9\31æµª\17Qæµ¬\6Üæµ®\15Bæµ´\16Áæµ·\6Ãæµ¸\11Zæµ¹\22óæ¶\133\22\
  \øæ¶\136\11\1æ¶\140\16\143æ¶\142\22õæ¶\147\22ðæ¶\149\22öæ¶\153\17\28æ¶\
  \\155\13Óæ¶\156\14\0æ¶¯\6Õæ¶²\64æ¶µ\22üæ¶¸\22ÿæ¶¼\17\1æ·\128\16Äæ·\133\23\
  \\6æ·\134\23\0æ·\135\22ýæ·\139\17\18æ·\140\23\3æ·\145\10©æ·\146\23\5æ·\
  \\149\23\10æ·\152\13Ñæ·\153\23\8æ·\158\23\2æ·¡\12×æ·¤\23\9æ·¦\22þæ·¨\23\4\
  \æ·ª\23\11æ·«\5úæ·¬\23\1æ·®\23\12æ·±\11[æ·³\10¾æ·µ\15cæ··\9læ·¹\22ùæ·º\23\
  \\7æ·»\13\153æ¸\133\11´æ¸\135\7\9æ¸\136\9\143æ¸\137\11\2æ¸\138\22ûæ¸\139\
  \\10¡æ¸\147\8kæ¸\149\22úæ¸\153\23\16æ¸\154\10Íæ¸\155\8¸æ¸\157\23\31æ¸\159\
  \\23\25æ¸ \7Ôæ¸¡\13®æ¸£\23\20æ¸¤\23\29æ¸¥\5\173æ¸¦\6\17æ¸©\6wæ¸«\23\22æ¸¬\
  \\12jæ¸\173\23\13æ¸®\23\15æ¸¯\9 æ¸¸\23 æ¸º\23\27æ¸¾\23\19æ¹\131\23\26æ¹\
  \\138\16)æ¹\141\23\24æ¹\142\23\28æ¹\150\8Îæ¹\152\11\3æ¹\155\12Øæ¹\159\23\
  \\18æ¹§\16\142æ¹«\23\21æ¹®\23\14æ¹¯\13Òæ¹²\23\17æ¹¶\23\23æ¹¾\17pæ¹¿\10<æº\
  \\128\16\30æº\130\23!æº\140\14¬æº\143\23-æº\144\8¹æº\150\10Àæº\152\23#æº\
  \\156\16íæº\157\9!æº\159\230æº¢\5ìæº¥\23.æºª\23\34æº¯\23(æº²\23*æº¶\16®æº\
  \·\23%æºº\13\141æº½\23'æ»\130\23/æ»\132\23)æ»\133\16Eæ»\137\23$æ»\139\10 \
  \æ»\140\23<æ»\145\7\10æ»\147\23&æ»\148\23+æ»\149\23,æ»\157\12ªæ»\158\12\
  \\152æ»¬\234æ»¯\23:æ»²\238æ»´\13\136æ»·\23Bæ»¸\235æ»¾\236æ»¿\23\30æ¼\129\
  \\7Ùæ¼\130\15\25æ¼\134\10=æ¼\137\9Wæ¼\143\17Ræ¼\145\232æ¼\147\23Aæ¼\148\6\
  \Iæ¼\149\12Fæ¼ \14\153æ¼¢\7?æ¼£\178æ¼«\16\31æ¼¬\13Pæ¼±\239æ¼²\23;æ¼¸\12\
  \\17æ¼¾\23@æ¼¿\237æ½\129\231æ½\133\7Aæ½\148\8\137æ½\152\23Næ½\155\23Iæ½\
  \\156\11öæ½\159\7\3æ½¤\10Áæ½¦\23Ræ½\173\23Kæ½®\13*æ½¯\23Hæ½°\13Wæ½´\23kæ½\
  \¸\23Eæ½º\23Dæ½¼\23Mæ¾\128\23Gæ¾\129\23Fæ¾\130\23Læ¾\132\11\159æ¾\134\23C\
  \æ¾\142\23Oæ¾\145\23Pæ¾\151\7@æ¾¡\23Uæ¾£\23Tæ¾¤\23Væ¾ª\23Yæ¾±\13¢æ¾³\23Sæ\
  \¾¹\23Wæ¿\128\8\131æ¿\129\12·æ¿\130\23Qæ¿\131\14Zæ¿\134\23Xæ¿\148\23]æ¿\
  \\149\23[æ¿\152\23^æ¿\155\23aæ¿\159\23Zæ¿ \9Jæ¿¡\14Gæ¿¤\22÷æ¿«\16Ôæ¿¬\23\
  \\92æ¿®\23`æ¿¯\12³æ¿±\23_æ¿³\23Jæ¿¶\29\137æ¿º\23dæ¿¾\23hç\128\129\23fç\
  \\128\137\23bç\128\139\23cç\128\143\23gç\128\145\23eç\128\149\15-ç\128\
  \\152\23mç\128\154\23jç\128\155\23iç\128\157\23lç\128\158\14\18ç\128\159\
  \\23nç\128¦\13\21ç\128§\12«ç\128¬\11£ç\128°\23oç\128²\23qç\128¾\23pç\129\
  \\140\233ç\129\145\23rç\129\152\14%ç\129£\23sç\129«\6\142ç\129¯\13Ôç\129°\
  \\6Äç\129¸\7Äç\129¼\10\92ç\129½\9\144ç\130\137\17Fç\130\138\11\134ç\130\
  \\142\6Jç\130\146\23uç\130\153\23tç\130¬\23xç\130\173\12Ùç\130®\23{ç\130¯\
  \\23vç\130³\23zç\130¸\23yç\130¹\13\159ç\130º\5×ç\131\136\173ç\131\139\23}\
  \ç\131\143\6\7ç\131\153\23\128ç\131\157\23~ç\131\159\23|ç\131±\23wç\131¹\
  \\15Âç\131½\23\130ç\132\137\23\129ç\132\148\6Kç\132\153\23\132ç\132\154\
  \\15pç\132\156\23\131ç\132¡\163ç\132¦\11\5ç\132¶\12\18ç\132¼\11\4ç\133\
  \\137\179ç\133\140\23\138ç\133\142\11÷ç\133\149\23\134ç\133\150\23\139ç\
  \\133\153\6Lç\133¢\23\137ç\133¤\14\129ç\133¥\23\133ç\133¦\23\136ç\133§\11\
  \\6ç\133©\14Ïç\133¬\23\140ç\133®\10Oç\133½\11øç\134\132\23\143ç\134\136\
  \\23\135ç\134\138\8Fç\134\143\23\141ç\134\148\16¯ç\134\149\23\144ç\134\
  \\153\31$ç\134\159\10®ç\134¨\23\145ç\134¬\23\146ç\134±\14Mç\134¹\23\148ç\
  \\134¾\23\149ç\135\131\14Rç\135\136\13Õç\135\137\23\151ç\135\142\23\153ç\
  \\135\144\17\19ç\135\146\23\150ç\135\148\23\152ç\135\149\6Mç\135\151\23\
  \\147ç\135\159\18úç\135 \23\154ç\135¥\12Gç\135¦\9×ç\135§\23\156ç\135¬\23\
  \\155ç\135\173\11Cç\135®\18\151ç\135µ\23\157ç\135¹\23\159ç\135»\23\142ç\
  \\135¼\23\158ç\135¿\23 ç\136\134\14\154ç\136\141\23¡ç\136\144\23¢ç\136\
  \\155\23£ç\136¨\23¤ç\136ª\13\92ç\136¬\23¦ç\136\173\23¥ç\136°\23§ç\136²\23\
  \¨ç\136µ\10]ç\136¶\15Cç\136º\16jç\136»\23©ç\136¼\23ªç\136½\125ç\136¾\10\
  \\34ç\136¿\23«ç\137\128\23¬ç\137\134\23\173ç\137\135\15\144ç\137\136\14Åç\
  \\137\139\23®ç\137\140\14vç\137\146\13+ç\137\152\23¯ç\137\153\6¥ç\137\155\
  \\7Íç\137\157\16Dç\137\159\164ç\137¡\6rç\137¢\17Sç\137§\15ñç\137©\15hç\
  \\137²\11µç\137´\23°ç\137¹\14\1ç\137½\8¡ç\137¾\23±ç\138\128\9\146ç\138\
  \\129\23³ç\138\130\23²ç\138\135\23´ç\138\146\23µç\138\150\23¶ç\138 \7\157\
  \ç\138¢\23·ç\138§\23¸ç\138¬\8¢ç\138¯\14Æç\138²\23ºç\138¶\113ç\138¹\23¹ç\
  \\139\130\7öç\139\131\23»ç\139\132\23½ç\139\134\23¼ç\139\142\23¾ç\139\144\
  \\8Ïç\139\146\23¿ç\139\151\8'ç\139\153\12\31ç\139\155\9]ç\139 \23Áç\139¡\
  \\23Âç\139¢\23Àç\139©\10kç\139¬\14\6ç\139\173\7÷ç\139·\23Äç\139¸\12Ëç\139\
  \¹\23Ãç\139¼\17Tç\139½\14\130ç\140\138\23Çç\140\150\23Éç\140\151\23Æç\140\
  \\155\16Rç\140\156\23Èç\140\157\23Êç\140\159\17\2ç\140¥\23Îç\140©\23Íç\
  \\140ª\13\22ç\140«\14Lç\140®\8£ç\140¯\23Ìç\140´\23Ëç\140¶\16\144ç\140·\16\
  \\145ç\140¾\23Ïç\140¿\6Nç\141\132\9Vç\141\133\10\2ç\141\142\23Ðç\141\143\
  \\23Ñç\141\151\23Óç\141£\10¢ç\141¨\23Õç\141ª\23Ôç\141°\23Öç\141²\6ìç\141µ\
  \\23Øç\141¸\23×ç\141º\23Úç\141»\23Ùç\142\132\8ºç\142\135\16æç\142\137\8\
  \\10ç\142\139\6dç\142\150\8(ç\142©\7_ç\142²\17&ç\142³\23Üç\142»\23Þç\143\
  \\128\23ßç\143\130\6\143ç\143\136\23Ûç\143\138\9Øç\143\141\13?ç\143\142\
  \\23Ýç\143\158\23âç\143 \10lç\143¥\23àç\143ª\8]ç\143\173\14Çç\143®\23áç\
  \\143±\23üç\143¸\23çç\143¾\8»ç\144\131\7Åç\144\133\23äç\144\134\16Ýç\144\
  \\137\16îç\144¢\12´ç\144¥\23æç\144²\23èç\144³\17\20ç\144´\8\21ç\144µ\14úç\
  \\144¶\14iç\144º\23éç\144¿\23ëç\145\129\23îç\145\149\23êç\145\153\23íç\
  \\145\154\8èç\145\155\6,ç\145\156\23ïç\145\158\11\144ç\145\159\23ìç\145 \
  \\17\26ç\145£\23òç\145¤\31\34ç\145©\23ðç\145ª\23óç\145¯\23åç\145°\23ñç\
  \\145³\9zç\145¶\23ôç\145¾\23õç\146\131\16Þç\146\139\23öç\146\158\23÷ç\146\
  \¢\23ãç\146§\23øç\146°\7Bç\146½\10#ç\147\138\23ùç\147\143\23úç\147\148\23\
  \ûç\147\156\6\26ç\147 \24\0ç\147¢\15\26ç\147£\24\1ç\147¦\7\34ç\147§\24\2ç\
  \\147©\24\3ç\147®\24\4ç\147°\24\6ç\147±\24\7ç\147²\24\5ç\147¶\152ç\147·\
  \\24\9ç\147¸\24\8ç\148\131\24\11ç\148\132\24\10ç\148\133\24\12ç\148\140\
  \\24\13ç\148\141\24\15ç\148\142\24\14ç\148\145\9Yç\148\147\24\17ç\148\149\
  \\24\16ç\148\152\7Cç\148\154\11rç\148\156\13\155ç\148\158\24\18ç\148\159\
  \\11¶ç\148£\9Ùç\148¥\6Yç\148¦\24\19ç\148¨\16°ç\148«\15¡ç\148¬\24\20ç\148°\
  \\13£ç\148±\16\146ç\148²\9\34ç\148³\11\92ç\148·\12êç\148¸\18rç\148º\13,ç\
  \\148»\6¦ç\148¼\24\21ç\149\132\24\22ç\149\134\24\27ç\149\137\24\25ç\149\
  \\138\24\24ç\149\139\21\128ç\149\140\6Åç\149\141\24\23ç\149\143\5Øç\149\
  \\145\14¨ç\149\148\14Èç\149\153\16ïç\149\154\24\28ç\149\155\24\26ç\149\
  \\156\12ûç\149\157\11¤ç\149 \14©ç\149¢\15\12ç\149¤\24\30ç\149¥\16êç\149¦\
  \\8lç\149§\24\31ç\149©\24\29ç\149ª\14Ôç\149«\24 ç\149\173\24!ç\149°\5Ùç\
  \\149³\114ç\149´\24&ç\149¶\24#ç\149·\14+ç\149¸\24\34ç\149¿\7\133ç\150\130\
  \\24)ç\150\134\24$ç\150\135\24%ç\150\137\24(ç\150\138\24'ç\150\139\15\4ç\
  \\150\142\12!ç\150\143\12 ç\150\145\7\158ç\150\148\24*ç\150\154\24+ç\150\
  \\157\24,ç\150£\24.ç\150¥\24-ç\150«\65ç\150±\246ç\150²\14æç\150³\240ç\150\
  \µ\242ç\150¸\244ç\150¹\11]ç\150¼\245ç\150½\243ç\150¾\10>ç\151\130\24/ç\
  \\151\131\241ç\151\133\15!ç\151\135\11\7ç\151\138\248ç\151\141\247ç\151\
  \\146\249ç\151\148\10$ç\151\149\9mç\151\152\13×ç\151\153\24:ç\151\155\13I\
  \ç\151\158\24<ç\151¢\16ßç\151£\24;ç\151©\12Iç\151°\24Bç\151²\24Dç\151³\24\
  \Eç\151´\12óç\151º\24Cç\151¼\24@ç\151¾\24=ç\151¿\24>ç\152\129\24Aç\152\
  \\137\24Hç\152\139\24Fç\152\141\24Gç\152\159\24Iç\152 \24Kç\152¡\24Lç\152\
  \¢\24Mç\152¤\24Nç\152§\24Jç\152°\24Pç\152´\24Oç\152»\24Qç\153\130\17\3ç\
  \\153\134\24Tç\153\135\24Rç\153\136\24Sç\153\140\7`ç\153\146\16|ç\153\150\
  \\15\136ç\153\152\24Vç\153\156\24Uç\153¡\24Wç\153¢\24Xç\153§\24\92ç\153¨\
  \\24Yç\153©\24Zç\153ª\24[ç\153¬\24]ç\153°\24^ç\153²\24_ç\153¶\24`ç\153¸\
  \\24aç\153º\14\173ç\153»\13¯ç\153¼\24bç\153½\14\146ç\153¾\15\19ç\154\128\
  \\24cç\154\131\24dç\154\132\13\137ç\154\134\6Æç\154\135\9#ç\154\136\24eç\
  \\154\139\24fç\154\142\24gç\154\144\9Èç\154\147\24iç\154\150\24hç\154\153\
  \\24jç\154\154\24kç\154®\14çç\154°\24lç\154´\24mç\154·\31\9ç\154¸\24nç\
  \\154¹\24oç\154º\24pç\154¿\9Íç\155\130\24qç\155\131\14uç\155\134\15þç\155\
  \\136\6-ç\155\138\66ç\155\141\24rç\155\146\24tç\155\150\24sç\155\151\13Ðç\
  \\155\155\11·ç\155\156\22\152ç\155\158\24uç\155\159\16?ç\155¡\24vç\155£\7\
  \Dç\155¤\14Õç\155¥\24wç\155§\24xç\155ª\24yç\155®\16Zç\155²\16Sç\155´\13<ç\
  \\155¸\12Jç\155»\24{ç\155¾\10Âç\156\129\11\8ç\156\132\24~ç\156\135\24}ç\
  \\156\136\24|ç\156\137\14ûç\156\139\7Eç\156\140\8§ç\156\155\24\132ç\156\
  \\158\24\129ç\156\159\11^ç\156 \160ç\156¤\24\128ç\156¥\24\130ç\156¦\24\
  \\131ç\156©\24\127ç\156·\24\133ç\156¸\24\134ç\156º\13-ç\156¼\7aç\157\128\
  \\13\5ç\157\135\24\135ç\157\154\24\136ç\157\155\24\139ç\157¡\11\135ç\157£\
  \\14\2ç\157¥\24\140ç\157¦\15òç\157¨\24\137ç\157«\24\138ç\157¹\24\143ç\157\
  \¾\24\142ç\157¿\24\141ç\158\139\24\145ç\158\142\24\144ç\158\145\24\146ç\
  \\158\158\24\148ç\158 \24\147ç\158¥\15\139ç\158¬\10µç\158\173\17\4ç\158°\
  \\24\149ç\158³\13õç\158¶\24\150ç\158¹\24\151ç\158»\24\155ç\158¼\24\153ç\
  \\158½\24\154ç\158¿\24\152ç\159\135\24\156ç\159\141\24\157ç\159\151\24\
  \\158ç\159\154\24\159ç\159\155\165ç\159\156\24 ç\159¢\16nç\159£\24¡ç\159¥\
  \\12íç\159§\14\138ç\159©\8)ç\159\173\12Úç\159®\24¢ç\159¯\7øç\159³\11Îç\
  \\159¼\24£ç \130\9{ç \140\24¤ç \146\24¥ç \148\8¤ç \149\9\147ç  \24§ç ¥\13\
  \µç ¦\9\148ç §\7\173ç ²\15Ãç ´\14jç º\13¶ç ¿\9;ç¡\133\24©ç¡\157\11\9ç¡«\
  \\16ðç¡¬\9$ç¡¯\8¥ç¡²\14¡ç¡´\24«ç¡¼\24\173ç¢\129\8éç¢\134\24¬ç¢\135\13tç¢\
  \\140\24¯ç¢\141\6Öç¢\142\24ªç¢\145\14èç¢\147\6\15ç¢\149\9ªç¢\151\17qç¢\
  \\154\24®ç¢£\24°ç¢§\15\137ç¢©\11×ç¢ª\24²ç¢¯\24³ç¢µ\24±ç¢º\6íç¢¼\24¹ç¢¾\24\
  \¸ç£\129\10%ç£\133\24ºç£\134\24µç£\138\24»ç£\139\24¶ç£\144\14Öç£\145\24´ç\
  \£\148\24·ç£\154\24Áç£§\24Àç£¨\16\1ç£¬\24¼ç£¯\5éç£´\24Ãç£½\24Âç¤\129\11\
  \\10ç¤\135\24Äç¤\142\12\34ç¤\145\24Æç¤\146\24Åç¤\153\24Çç¤¦\24¦ç¤ª\24¨ç¤«\
  \\24Éç¤¬\24Èç¤º\10&ç¤¼\17'ç¤¾\10Pç¥\128\24Êç¥\129\8Vç¥\135\7\159ç¥\136\7\
  \\134ç¥\137\10\3ç¥\144\16\147ç¥\147\24Ðç¥\149\24Ïç¥\150\12#ç¥\151\24Ìç¥\
  \\154\24Îç¥\157\10ªç¥\158\11_ç¥\159\24Íç¥ \24Ëç¥¢\14Iç¥¥\11\11ç¥¨\15\27ç¥\
  \\173\9\149ç¥·\13Øç¥º\24Ñç¥¿\24Òç¦\128\24èç¦\129\8\22ç¦\132\17\92ç¦\133\
  \\12\20ç¦\138\24Óç¦\141\6\144ç¦\142\13uç¦\143\15_ç¦\157\24Ôç¦¦\7Úç¦§\24Õç\
  \¦ª\24×ç¦®\24Øç¦°\14Hç¦³\24Ùç¦¹\24Úç¦º\24Ûç¦½\8\23ç¦¾\6\145ç¦¿\14\3ç§\128\
  \\10\135ç§\129\10\4ç§\137\24Üç§\139\10\136ç§\145\6\136ç§\146\15\34ç§\149\
  \\24Ýç§\152\14éç§\159\12$ç§¡\24àç§£\24áç§¤\14\137ç§¦\11`ç§§\24Þç§©\13\1ç§\
  \¬\24ßç§°\11\12ç§»\5Úç¨\128\7\136ç¨\136\24âç¨\139\13vç¨\141\24ãç¨\142\11Å\
  \ç¨\148\16+ç¨\151\15\2ç¨\152\24äç¨\153\24åç¨\154\12ôç¨\156\17\5ç¨\159\24ç\
  \ç¨ \24æç¨®\10mç¨±\24éç¨²\5îç¨·\24ìç¨»\24êç¨¼\6\146ç¨½\8mç¨¾\24ëç¨¿\9%ç©\
  \\128\9Rç©\130\15¤ç©\131\24íç©\134\15óç©\137\24ïç©\141\11Ïç©\142\6.ç©\143\
  \\6xç©\144\5ªç©\151\24îç©¡\24ðç©¢\24ñç©£\115ç©©\24òç©«\6îç©°\24ôç©´\8\138\
  \ç©¶\7Æç©¹\24õç©º\83ç©½\24öç©¿\11úçª\129\14\11çª\131\11Þçª\132\9³çª\136\
  \\24÷çª\146\13\2çª\147\12Kçª\149\24ùçª\150\24ûçª\151\24øçª\152\24úçª\159\
  \\8Açª©\24üçªª\8Eçª®\7Ççª¯\16±çª°\24þçª¶\25\0çªº\6\13çª¿\25\3ç«\131\7\22ç\
  \«\132\25\2ç«\133\25\1ç«\135\25\5ç«\136\24ýç«\138\25\6ç«\139\16çç«\141\25\
  \\7ç«\143\25\8ç«\146\19rç«\147\25\10ç«\149\25\9ç«\153\25\11ç«\154\25\12ç«\
  \\156\16óç«\157\25\13ç«\159\29íç« \11\13ç«¡\25\14ç«¢\25\15ç«£\10¶ç«¥\13öç\
  \«¦\25\16ç«ª\12Çç«\173\25\17ç«¯\12Ûç«°\25\18ç«¶\7ãç«¸\18\30ç«¹\12üç«º\101\
  \ç«¿\7Fç¬\130\25\19ç¬\132\25 ç¬\134\25\22ç¬\136\7Èç¬\138\25\21ç¬\139\25\
  \\34ç¬\143\25\20ç¬\145\11\14ç¬\152\25\24ç¬\153\25\25ç¬\155\13\138ç¬\158\
  \\25\26ç¬ \6ýç¬¥\11yç¬¦\15Dç¬¨\25\28ç¬¬\12¦ç¬³\25\23ç¬µ\25\27ç¬¶\25\29ç¬¹\
  \\9¹ç\173\133\25$ç\173\134\15\13ç\173\136\14¤ç\173\137\13Ùç\173\139\8\24ç\
  \\173\140\25#ç\173\141\25!ç\173\143\14³ç\173\144\25\30ç\173\145\12ýç\173\
  \\146\13Ûç\173\148\13Úç\173\150\9´ç\173\157\256ç\173¥\25&ç\173§\25(ç\173¬\
  \\25+ç\173®\25,ç\173°\25)ç\173±\25*ç\173´\25'ç\173µ\25%ç\173º\25\31ç®\134\
  \\15\141ç®\135\6\147ç®\139\253ç®\141\250ç®\143\255ç®\146\254ç®\148\14\147\
  \ç®\149\16%ç®\151\9Úç®\152\25.ç®\153\257ç®\154\252ç®\156\251ç®\157\25-ç®\
  \\159\25/ç®¡\7Gç®ª\12Üç®\173\11ûç®±\14 ç®´\25<ç®¸\14¢ç¯\128\11ßç¯\129\259\
  \ç¯\132\14Íç¯\134\25=ç¯\135\15\145ç¯\137\12úç¯\139\258ç¯\140\25:ç¯\143\25\
  \;ç¯\157\25>ç¯ \10Bç¯¤\14\4ç¯¥\25Cç¯¦\25Bç¯©\25?ç¯\173\17Uç¯³\25Hç¯¶\25Lç\
  \¯·\25Iç°\128\25Eç°\135\25Fç°\141\25Kç°\145\25@ç°\146\18\147ç°\147\25Gç°\
  \\148\25Aç°\151\25Jç°\159\25Pç°¡\7Hç°£\25Mç°§\25Nç°ª\25Oç°«\25Rç°·\25Qç°¸\
  \\14ôç°½\25Sç°¾\17:ç°¿\15«ç±\128\25Xç±\131\25Uç±\140\25Tç±\141\11Ðç±\143\
  \\25Wç±\144\25Yç±\148\25Vç±\150\25]ç±\152\25Zç±\159\25[ç± \25Dç±¤\25\92ç±\
  \¥\25^ç±¬\25_ç±³\15\132ç±µ\25`ç±¾\16`ç²\129\8\12ç²\130\8Hç²\131\25aç²\137\
  \\15rç²\139\11\136ç²\141\16.ç²\144\25bç²\146\16ñç²\149\14\148ç²\151\12%ç²\
  \\152\14Sç²\155\10¬ç²\159\5¾ç²¡\25gç²¢\25eç²¤\25cç²¥\7\31ç²§\11\15ç²¨\25h\
  \ç²«\25fç²\173\25dç²®\25lç²±\25kç²²\25jç²³\25iç²¹\25mç²½\25nç²¾\11¸ç³\128\
  \\25oç³\130\25qç³\133\25pç³\138\8Ðç³\142\12\23ç³\146\25sç³\150\13Üç³\152\
  \\25rç³\156\25tç³\158\15sç³\159\12Lç³ \9&ç³¢\25uç³§\17\6ç³¯\25wç³²\25xç³´\
  \\25yç³¶\25zç³¸\10\5ç³º\25{ç³»\8nç³¾\7Êç´\128\7\137ç´\130\25\128ç´\132\16\
  \qç´\133\9'ç´\134\25|ç´\138\25\131ç´\139\16dç´\141\14[ç´\144\15\18ç´\148\
  \\10Ãç´\149\25\130ç´\151\10Qç´\152\9(ç´\153\10\6ç´\154\7Éç´\155\15tç´\156\
  \\25\129ç´ \12&ç´¡\15áç´¢\9µç´«\10\7ç´¬\13[ç´®\25\134ç´¯\17\29ç´°\9\151ç´\
  \²\25\135ç´³\11aç´µ\25\137ç´¹\11\16ç´º\9nç´¿\25\136çµ\130\10\137çµ\131\8¼\
  \çµ\132\12'çµ\133\25\132çµ\134\25\138çµ\139\25\133çµ\140\8oçµ\142\25\141ç\
  \µ\143\25\145çµ\144\8\139çµ\150\25\140çµ\155\25\149çµ\158\9)çµ¡\16Íçµ¢\5º\
  \çµ£\25\146çµ¦\7Ëçµ¨\25\143çµ®\25\144çµ±\13Ýçµ²\25\142çµ³\25\139çµµ\6Ççµ¶\
  \\11âçµ¹\8¦çµ½\25\151ç¶\137\25\148ç¶\143\25\150ç¶\147\25\147ç¶\153\8pç¶\
  \\154\12qç¶\155\25\152ç¶\156\12Nç¶\159\25¥ç¶¢\25¡ç¶£\25\155ç¶«\25\159ç¶¬\
  \\10xç¶\173\5Ûç¶®\25\154ç¶¯\25¢ç¶°\25¦ç¶±\9*ç¶²\16Tç¶´\13Tç¶µ\25\156ç¶¸\
  \\25¤ç¶º\25\153ç¶»\12Ýç¶½\25\158ç¶¾\5»ç¶¿\16Hç·\135\25\157ç·\138\8\25ç·\
  \\139\14êç·\143\12Mç·\145\17\14ç·\146\10Ïç·\149\25Îç·\152\25§ç·\154\11üç·\
  \\156\25£ç·\157\25¨ç·\158\25ªç· \13wç·¡\25\173ç·¤\25©ç·¨\15\146ç·©\7Iç·¬\
  \\16Iç·¯\5Üç·²\25¬ç·´\17;ç·»\25«ç¸\129\6Oç¸\132\14*ç¸\133\25®ç¸\137\25µç¸\
  \\138\25¯ç¸\139\25¶ç¸\146\25²ç¸\155\14\155ç¸\158\10Hç¸\159\25´ç¸¡\25±ç¸¢\
  \\25·ç¸£\25°ç¸¦\10£ç¸«\15Äç¸®\10«ç¸±\25³ç¸²\25Àç¸µ\25»ç¸·\25¾ç¸¹\25¼ç¸º\
  \\25Áç¸»\25ºç¸½\25 ç¸¾\11Ñç¹\129\14Éç¹\131\25½ç¹\134\25¸ç¹\138\12\0ç¹\139\
  \\8qç¹\141\10\138ç¹\148\11Dç¹\149\12\21ç¹\150\25Äç¹\153\25Æç¹\154\25Çç¹\
  \\157\25Ãç¹\158\25Åç¹¦\25¹ç¹§\25Âç¹©\25Êç¹ª\25Éç¹\173\16\26ç¹°\8Jç¹¹\25Èç\
  \¹»\25Ìç¹¼\25Ëç¹½\25Ïç¹¿\25Ñçº\130\9Ûçº\131\25Íçº\136\25Òçº\137\25Óçº\140\
  \\25Ôçº\142\25Úçº\143\13\154çº\144\25Öçº\146\25Õçº\147\25×çº\148\25Øçº\
  \\150\25Ùçº\155\25Ûçº\156\25Üç¼¶\7Jç¼¸\25Ýç¼º\25Þç½\133\25ßç½\140\25àç½\
  \\141\25áç½\142\25âç½\144\25ãç½\145\25äç½\148\25æç½\149\25åç½\152\25çç½\
  \\159\25èç½ \25éç½§\25ìç½¨\25êç½©\25ëç½ª\9\159ç½«\8rç½®\12õç½°\14±ç½²\10Ð\
  \ç½µ\14lç½·\14ëç½¸\25íç½¹\20ëç¾\130\25îç¾\131\25ðç¾\133\16Åç¾\134\25ïç¾\
  \\135\25òç¾\136\25ñç¾\138\16²ç¾\140\25óç¾\142\14üç¾\148\25ôç¾\154\25÷ç¾\
  \\157\25öç¾\158\25õç¾£\25øç¾¤\8Qç¾¨\12\1ç¾©\7 ç¾®\25üç¾¯\25ùç¾²\25úç¾¶\25\
  \ýç¾¸\25þç¾¹\25ûç¾½\6\8ç¿\129\6eç¿\133\26\0ç¿\134\26\1ç¿\138\26\2ç¿\140\
  \\16Âç¿\146\10\139ç¿\148\26\4ç¿\149\26\3ç¿ \11\137ç¿¡\26\5ç¿¦\26\6ç¿©\26\
  \\7ç¿«\7cç¿°\7Kç¿³\26\8ç¿¹\26\9ç¿»\15üç¿¼\16Ãè\128\128\16³è\128\129\17Vè\
  \\128\131\9,è\128\132\26\12è\128\133\10Rè\128\134\26\11è\128\139\26\13è\
  \\128\140\10'è\128\144\12\143è\128\146\26\14è\128\149\9+è\128\151\16Uè\
  \\128\152\26\15è\128\153\26\16è\128\156\26\17è\128¡\26\18è\128¨\26\19è\
  \\128³\10(è\128¶\16kè\128»\26\21è\128½\12Þè\128¿\26\20è\129\134\26\23è\
  \\129\138\26\22è\129\146\26\24è\129\150\11¹è\129\152\26\25è\129\154\26\26\
  \è\129\158\15wè\129\159\26\27è\129¡\12Oè\129¢\26\28è\129¨\26\29è\129¯\17<\
  \è\129°\26 è\129²\26\31è\129³\26\30è\129´\13.è\129¶\26!è\129·\11Eè\129¹\
  \\26\34è\129½\26#è\129¾\17Wè\129¿\26$è\130\132\26%è\130\133\26'è\130\134\
  \\26&è\130\135\14£è\130\137\147è\130\139\17]è\130\140\14§è\130\147\26)è\
  \\130\150\11\17è\130\152\15\9è\130\154\26*è\130\155\26(è\130\157\7Lè\130¡\
  \\8Òè\130¢\10\8è\130¥\14ìè\130©\8¨è\130ª\15âè\130¬\26-è\130\173\26+è\130¯\
  \\9-è\130±\9.è\130²\5çè\130´\9¦è\130º\14xè\131\131\5Ýè\131\132\262è\131\
  \\134\12ßè\131\140\14wè\131\142\12\153è\131\150\264è\131\153\260è\131\154\
  \\263è\131\155\26.è\131\157\261è\131\158\15Åè\131¡\8Óè\131¤\5ûè\131¥\26/è\
  \\131¯\266è\131±\267è\131´\13÷è\131¸\7ùè\131¼\26Eè\131½\14\92è\132\130\10\
  \\9è\132\133\7úè\132\134\11Æè\132\135\17eè\132\136\16,è\132\137\265è\132\
  \\138\11Òè\132\154\7²è\132\155\268è\132£\26:è\132©\269è\132¯\26;è\132±\12\
  \Åè\132³\14]è\132¹\13/è\132¾\26Bè\133\134\26Aè\133\139\26<è\133\142\11tè\
  \\133\144\15Eè\133\145\26Dè\133\147\26Cè\133\148\9/è\133\149\17rè\133\159\
  \\26Tè\133¥\26Hè\133¦\26Iè\133«\10nè\133®\26Gè\133°\9Xè\133±\26Fè\133´\26\
  \Jè\133¸\130è\133¹\15`è\133º\12\2è\133¿\12\154è\134\128\26Nè\134\130\26Oè\
  \\134\131\26Kè\134\136\26Lè\134\138\26Mè\134\143\90è\134\147\26Uè\134\149\
  \\26Qè\134\154\15Fè\134\156\16\12è\134\157\15\7è\134 \26Pè\134£\26Sè\134¤\
  \\26Rè\134¨\15ãè\134©\26Vè\134°\26Wè\134³\12\22è\134µ\26Xè\134¸\26Zè\134º\
  \\26^è\134½\26[è\134¾\26Yè\134¿\14^è\135\128\26\92è\135\130\26]è\135\134\
  \\6pè\135\136\26dè\135\137\26_è\135\141\26`è\135\145\26aè\135\147\12_è\
  \\135\152\26cè\135\153\26bè\135\154\26eè\135\159\26fè\135 \26gè\135£\11bè\
  \\135¥\6§è\135§\26hè\135¨\17\21è\135ª\10)è\135\173\10\140è\135³\10\10è\
  \\135´\12öè\135º\26iè\135»\26jè\135¼\6\16è\135¾\26kè\136\129\26lè\136\130\
  \\26mè\136\133\26nè\136\135\26oè\136\136\7ûè\136\137\21hè\136\138\26pè\
  \\136\140\11ãè\136\141\26qè\136\142\10Iè\136\144\26rè\136\146\17®è\136\
  \\150\26sè\136\151\15\156è\136\152\7Zè\136\155\12\3è\136\156\10·è\136\158\
  \\15Qè\136\159\10\141è\136©\26tè\136ª\91è\136«\26uè\136¬\14Êè\136®\26\132\
  \è\136³\26wè\136µ\12\135è\136¶\14\149è\136·\8½è\136¸\26vè\136¹\12\4è\137\
  \\128\26xè\137\135\13xè\137\152\26zè\137\153\26yè\137\154\26|è\137\157\26\
  \{è\137\159\26}è\137¢\26\128è\137¤\26~è\137¦\7Mè\137¨\26\129è\137ª\26\130\
  \è\137«\26\131è\137®\9oè\137¯\17\7è\137±\26\133è\137²\11Fè\137¶\6Pè\137·\
  \\26\134è\137¸\26\135è\137¾\26\136è\138\139\5ðè\138\141\26\137è\138\146\
  \\26\138è\138\153\15Gè\138\157\10Eè\138\159\26\140è\138¥\6Èè\138¦\5°è\138\
  \«\26\139è\138¬\26\142è\138\173\14mè\138¯\11cè\138±\6\148è\138³\15Æè\138¸\
  \\8|è\138¹\8\26è\138»\26\141è\138½\6¨è\139\133\7!è\139\145\6Qè\139\146\26\
  \\146è\139\147\17(è\139\148\12\155è\139\151\15#è\139\153\26\158è\139\155\
  \\6\149è\139\156\26\156è\139\158\26\154è\139\159\26\145è\139¡\26\143è\139\
  \£\26\144è\139¥\10aè\139¦\8*è\139§\13\23è\139«\14\15è\139±\60è\139³\26\
  \\148è\139´\26\147è\139¹\26\153è\139º\26\149è\139»\26\152è\140\130\16Nè\
  \\140\131\26\151è\140\132\6\150è\140\133\7\29è\140\134\26\155è\140\137\26\
  \\157è\140\142\8sè\140\150\26¡è\140\151\26ªè\140\152\26«è\140\156\5©è\140\
  \£\26²è\140¨\5ïè\140«\26©è\140¯\26¨è\140±\26£è\140²\26¢è\140´\26 è\140µ\
  \\26\159è\140¶\13\3è\140¸\12¹è\140¹\26¥è\141\128\26¤è\141\133\26§è\141\
  \\137\12Pè\141\138\8tè\141\143\6 è\141\144\26¦è\141\146\92è\141\152\12Qè\
  \\141³\26¸è\141µ\26¹è\141·\6\151è\141»\6lè\141¼\26¶è\142\133\26¬è\142\135\
  \\26´è\142\137\26»è\142\138\26µè\142\142\26³è\142\147\26\150è\142\150\26±\
  \è\142\154\26\173è\142\158\7Nè\142\159\26¯è\142 \26ºè\142¢\26°è\142¨\26¼è\
  \\142ª\26®è\142«\14\156è\142±\16Éè\142µ\26·è\142½\26Íè\143\129\26Åè\143\
  \\133\11\155è\143\138\7¥è\143\140\8\27è\143\142\26Àè\143\147\6\153è\143\
  \\150\11\18è\143\152\26Ãè\143\156\9\152è\143\159\13°è\143 \26Èè\143©\15¬è\
  \\143«\26¿è\143¯\6\152è\143°\8Ôè\143±\15\8è\143²\26Éè\143´\26½è\143·\26Æè\
  \\143»\26Ðè\143½\26Áè\144\131\26Âè\144\132\13øè\144\135\26Çè\144\139\26Äè\
  \\144\140\15Çè\144\141\26Êè\144\142\5Þè\144\147\26¾è\144 \26Ìè\144¢\26Ëè\
  \\144©\14\139è\144ª\26Òè\144¬\26Ýè\144±\7\30è\144µ\26àè\144¸\26Îè\144¼\26\
  \Óè\144½\16Îè\145\134\26Üè\145\137\16´è\145\142\16èè\145\151\13\24è\145\
  \\155\7\11è\145¡\15Rè\145¢\26âè\145£\13ßè\145¦\5¯è\145©\26Ûè\145«\26×è\
  \\145¬\12Rè\145\173\26Ñè\145®\26Ùè\145¯\26Þè\145±\14Kè\145µ\5¨è\145·\26Öè\
  \\145¹\26ßè\145º\15Xè\146\130\26Úè\146\132\26Õè\146\139\11\19è\146\144\10\
  \\142è\146\148\10*è\146\153\16Vè\146\156\15&è\146\159\26åè\146¡\26îè\146\
  \\173\26Øè\146²\7\23è\146¸\116è\146¹\26ãè\146»\26èè\146¼\12Sè\146¿\26äè\
  \\147\129\26ëè\147\132\12þè\147\134\26ìè\147\137\16µè\147\138\26áè\147\
  \\139\6×è\147\141\26çè\147\144\26êè\147\145\16*è\147\150\26íè\147\153\26æ\
  \è\147\154\26éè\147¬\15Èè\147®\17@è\147´\26ñè\147¼\26øè\147¿\26ðè\148\128\
  \\10Aè\148\134\26Ïè\148\145\15\140è\148\147\16 è\148\148\26÷è\148\149\26ö\
  \è\148\151\26òè\148\152\26óè\148\154\6\21è\148\159\26õè\148¡\26ïè\148¦\13\
  \Sè\148¬\26ôè\148\173\5üè\148µ\12`è\148½\15\129è\149\128\26ùè\149\129\27\
  \\0è\149\131\14×è\149\136\26üè\149\137\11\20è\149\138\10Gè\149\139\27\2è\
  \\149\142\7üè\149\149\27\3è\149\151\15Yè\149\152\26ûè\149\154\26Ôè\149£\
  \\26úè\149¨\17nè\149©\13àè\149ª\15Sè\149\173\27\10è\149·\27\16è\149¾\27\
  \\17è\150\128\27\4è\150\132\14\150è\150\135\27\14è\150\136\27\6è\150\138\
  \\27\8è\150\144\27\18è\150\145\27\7è\150\148\27\11è\150\151\6Rè\150\153\
  \\14#è\150\155\27\12è\150\156\27\15è\150¤\27\5è\150¦\12\5è\150¨\27\9è\150\
  \©\9Æè\150ª\11dè\150«\8Oè\150¬\16rè\150®\16wè\150¯\10Òè\150¹\27\22è\150º\
  \\27\20è\151\129\17mè\151\137\27\19è\151\141\16Õè\151\143\27\21è\151\144\
  \\27\23è\151\149\27\24è\151\156\27\27è\151\157\27\25è\151¤\13áè\151¥\27\
  \\26è\151©\14Ëè\151ª\27\13è\151·\10Óè\151¹\27\28è\151º\27!è\151»\12Tè\151\
  \¾\27 è\152\130\27\1è\152\134\27\34è\152\135\12(è\152\138\27\29è\152\139\
  \\27\31è\152\147\27\30è\152\150\22\144è\152\151\22\129è\152\154\27$è\152¢\
  \\27#è\152\173\16Öè\152¯\24zè\152°\27%è\152¿\27&è\153\141\27'è\153\142\8Õ\
  \è\153\144\7³è\153\148\27)è\153\149\18<è\153\154\7Õè\153\156\16øè\153\158\
  \\81è\153\159\27*è\153§\27+è\153«\13\14è\153±\27,è\153¹\148è\153»\5¸è\154\
  \\138\6¡è\154\139\271è\154\140\272è\154\147\27-è\154\149\9Üè\154£\27.è\
  \\154¤\14aè\154©\27/è\154ª\270è\154«\27:è\154¯\274è\154°\277è\154¶\273è\
  \\155\132\275è\155\134\276è\155\135\10Vè\155\137\278è\155\139\12àè\155\
  \\141\8uè\155\142\6áè\155\148\27;è\155\153\6Þè\155\155\27Aè\155\158\27<è\
  \\155\159\27@è\155¤\14¸è\155©\27=è\155¬\27>è\155\173\15'è\155®\14Øè\155¯\
  \\27Bè\155¸\12»è\155¹\27Lè\155»\27Hè\155¾\6©è\156\128\27Fè\156\130\15Éè\
  \\156\131\27Gè\156\134\27Dè\156\136\27Eè\156\137\27Jè\156\138\27Mè\156\
  \\141\27Kè\156\145\27Iè\156\146\27Cè\156\152\12÷è\156\154\27Tè\156\156\16\
  \(è\156¥\27Rè\156©\27Sè\156´\27Nè\156·\27Pè\156»\27Qè\156¿\27Oè\157\137\
  \\11äè\157\139\17Xè\157\140\27Xè\157\142\27Yè\157\147\27_è\157\149\11Iè\
  \\157\151\27[è\157\153\27^è\157\159\27Vè\157 \27Uè\157£\27`è\157¦\6\154è\
  \\157¨\27\92è\157ª\27aè\157®\27]è\157´\27Zè\157¶\131è\157¸\27Wè\157¿\14\
  \\136è\158\130\27eè\158\141\16\154è\158\159\27dè\158¢\27cè\158«\27lè\158¯\
  \\27fè\158³\27nè\158º\16Æè\158»\27qè\158½\27hè\159\128\27iè\159\132\27mè\
  \\159\134\27pè\159\135\27oè\159\139\27gè\159\144\27jè\159\146\27{è\159 \
  \\27tè\159¯\27rè\159²\27sè\159¶\27xè\159·\27yè\159¹\6Éè\159»\7¡è\159¾\27w\
  \è \133\27bè \141\27vè \142\27zè \143\27uè \145\27|è \149\27~è \150\27}è \
  \¡\27\128è ¢\27\127è £\279è §\27\132è ±\27\129è ¶\27\130è ¹\27\131è »\27\
  \\133è¡\128\8\140è¡\130\27\135è¡\132\27\134è¡\134\10\143è¡\140\93è¡\141\
  \\22åè¡\146\27\136è¡\147\10°è¡\151\6Øè¡\153\27\137è¡\155\61è¡\157\11\21è¡\
  \\158\27\138è¡¡\94è¡¢\27\139è¡£\5ßè¡¨\15\28è¡«\27\140è¡°\11\138è¡²\27\147\
  \è¡µ\27\144è¡·\13\15è¡½\27\145è¡¾\27\142è¡¿\8\28è¢\129\27\141è¢\130\27\
  \\148è¢\136\8Uè¢\139\12\156è¢\141\27\154è¢\146\27\150è¢\150\12sè¢\151\27\
  \\149è¢\153\27\152è¢\158\27\143è¢¢\27\153è¢¤\27\155è¢«\14íè¢®\27\151è¢°\
  \\27\156è¢±\27\158è¢´\8Ñè¢µ\27\146è¢·\5¿è¢¿\27\157è£\129\9\153è£\130\174è\
  \£\131\27\159è£\132\27 è£\133\12Uè£\143\16àè£\148\27¡è£\149\16\148è£\152\
  \\27¢è£\153\27£è£\156\15¢è£\157\27¤è£\159\9~è£¡\16áè£¨\27©è£²\27ªè£³\11\
  \\22è£´\27¨è£¸\16Çè£¹\27¥è£¼\27§è£½\11»è£¾\11\158è¤\130\27¦è¤\132\27«è¤\
  \\135\15aè¤\138\27\173è¤\140\27¬è¤\144\7\12è¤\146\15Êè¤\147\27®è¤\157\27º\
  \è¤\158\27°è¤¥\27±è¤ª\27²è¤«\27³è¤¶\27·è¤¸\27¸è¤»\27¶è¥\129\27´è¥\131\27¯\
  \è¥\132\27µè¥\140\27¹è¥\141\29µè¥\150\6fè¥\158\27¼è¥\159\8\29è¥ \27»è¥¤\
  \\27Áè¥¦\27Àè¥ª\27Ãè¥\173\27Âè¥¯\27Äè¥²\10\144è¥´\27Åè¥·\27Æè¥¾\27Çè¥¿\11\
  \¼è¦\129\16¶è¦\131\27Èè¦\134\15bè¦\135\14eè¦\136\27Éè¦\138\27Êè¦\139\8©è¦\
  \\143\7\139è¦\147\27Ëè¦\150\10\11è¦\151\14`è¦\152\27Ìè¦\154\6ïè¦¡\27Íè¦¦\
  \\27Ïè¦§\16×è¦©\27Îè¦ª\11eè¦¬\27Ðè¦¯\27Ñè¦²\27Òè¦³\7Oè¦º\27Óè¦½\27Ôè¦¿\27\
  \Õè§\128\27Öè§\146\6ðè§\154\27×è§\156\27Øè§\157\27Ùè§£\6°è§¦\11Gè§§\27Úè§\
  \´\27Ûè§¸\27Üè¨\128\8¾è¨\130\13yè¨\131\27Ýè¨\136\8vè¨\138\11uè¨\140\27àè¨\
  \\142\13âè¨\144\27ßè¨\147\8Pè¨\150\27Þè¨\151\12µè¨\152\7\140è¨\155\27áè¨\
  \\157\27âè¨\159\11\23è¨£\8\141è¨¥\27ãè¨ª\15Ëè¨\173\11Ýè¨±\7Öè¨³\16sè¨´\12\
  \)è¨¶\27äè¨º\11fè¨»\13\16è¨¼\11\24è©\129\27åè©\134\27èè©\136\27éè©\144\9|\
  \è©\145\12\128è©\146\27çè©\148\11\25è©\149\15\29è©\155\27æè©\158\10\12è© \
  \\62è©¢\27íè©£\8wè©¦\10\14è©©\10\13è©«\17lè©¬\27ìè©\173\27ëè©®\12\6è©°\7¬\
  \è©±\17bè©²\6Ùè©³\11\26è©¼\27êèª\130\27ïèª\132\27ðèª\133\27îèª\135\8Öèª\
  \\137\16\159èª\140\10\15èª\141\14Fèª\145\27óèª\147\11¾èª\149\12áèª\152\16\
  \\149èª\154\27öèª\158\8êèª \11½èª¡\27òèª£\27÷èª¤\8ëèª¥\27ôèª¦\27õèª¨\27ñè\
  \ª¬\11àèª\173\14\7èª°\12Îèª²\6\155èª¹\14îèª¼\7¢èª¿\132è«\130\27úè«\132\27\
  \øè«\135\12ëè«\139\11¿è«\140\7Pè«\141\27ùè«\143\11zè«\146\17\8è«\150\17_è\
  \«\154\27ûè«\155\28\7è«\156\133è«\158\28\6è« \28\3è«¡\28\11è«¢\28\4è«¤\28\
  \\0è«¦\13zè«§\27þè««\27üè«\173\16\128è«®\10\16è«±\28\1è«³\27ýè«·\28\5è«¸\
  \\10Ôè«º\8¿è«¾\12¸è¬\128\15äè¬\129\69è¬\130\5àè¬\132\13ãè¬\135\28\9è¬\140\
  \\28\8è¬\142\14$è¬\144\28\13è¬\148\28\2è¬\150\28\12è¬\151\28\14è¬\153\8ªè\
  \¬\154\28\10è¬\155\95è¬\157\10Sè¬ \28\15è¬¡\16·è¬¦\28\18è¬¨\28\21è¬«\28\
  \\19è¬¬\15\20è¬³\28\16è¬¹\8\30è¬¾\28\20è\173\129\28\22è\173\137\28\26è\
  \\173\140\28\23è\173\142\28\25è\173\143\28\24è\173\150\28\27è\173\152\10/\
  \è\173\154\28\29è\173\155\28\28è\173\156\15Hè\173\159\28\31è\173¦\8xè\173\
  \«\28\30è\173¬\28 è\173¯\28!è\173°\7£è\173±\25ÿè\173²\117è\173´\28\34è\
  \\173·\8ìè\173½\28#è®\128\28$è®\131\9Ýè®\138\21\140è®\140\28%è®\142\28&è®\
  \\144\10\145è®\146\28'è®\147\28(è®\150\28)è®\153\28*è®\154\28+è°·\12Êè°º\
  \\28,è°¿\28.è±\129\28-è±\134\13äè±\136\28/è±\138\15Ìè±\140\280è±\142\281è\
  \±\144\282è±\149\283è±\154\14\24è±¡\11\27è±¢\284è±ª\9Kè±«\17¬è±¬\285è±¸\
  \\286è±¹\15\30è±º\287è±¼\28?è²\130\288è²\133\28:è²\137\289è²\138\28;è²\
  \\140\15åè²\141\28<è²\142\28=è²\148\28>è²\152\28@è²\157\6Ìè²\158\13eè² \
  \\15Iè²¡\9 è²¢\96è²§\15.è²¨\6\157è²©\14Ìè²ª\28Cè²«\7Qè²¬\11Óè²\173\28Bè²®\
  \\28Gè²¯\13\25è²°\16aè²²\28Eè²³\28Fè²´\7\141è²¶\28Hè²·\14\131è²¸\12\157è²\
  \»\14ïè²¼\13\156è²½\28Dè²¿\15æè³\128\6ªè³\129\28Jè³\130\17Gè³\131\13@è³\
  \\132\17dè³\135\10\17è³\136\28Iè³\138\12oè³\141\28Zè³\142\12\7è³\145\146è\
  \³\147\15/è³\154\28Mè³\155\9Þè³\156\10\18è³\158\11\28è³ \14\133è³¢\8«è³£\
  \\28Lè³¤\28Kè³¦\15Jè³ª\10?è³\173\13±è³º\28Oè³»\28Pè³¼\97è³½\28Nè´\132\28Q\
  \è´\133\28Rè´\135\28Tè´\136\12aè´\138\28Sè´\139\7dè´\141\28Vè´\143\28Uè´\
  \\144\28Wè´\147\28Yè´\148\28[è´\150\28\92èµ¤\11Ôèµ¦\10Mèµ§\28]èµ«\6ñèµ\
  \\173\28^èµ°\12Vèµ±\28_èµ³\28`èµ´\15Kèµ·\7\142è¶\129\28aè¶\133\134è¶\138\
  \\6:è¶\153\28bè¶£\10oè¶¨\11\150è¶³\12kè¶º\28eè¶¾\28dè·\130\28cè·\139\28kè\
  \·\140\28iè·\143\28fè·\150\28hè·\154\28gè·\155\28jè·\157\7×è·\159\28nè·¡\
  \\11Õè·£\28oè·¨\8×è·ª\28lè·«\28mè·¯\17Hè·³\135è·µ\12\8è·¼\28pè·¿\28sè¸\
  \\136\28qè¸\137\28rè¸\138\16¸è¸\143\13åè¸\144\28vè¸\157\28tè¸\158\28uè¸\
  \\159\28wè¸ª\28\136è¸°\28zè¸´\28{è¸µ\28yè¹\130\28xè¹\132\13{è¹\135\28\128\
  \è¹\136\28\132è¹\137\28\129è¹\138\28|è¹\140\28\130è¹\144\28\131è¹\149\28\
  \\138è¹\153\28\133è¹\159\11Öè¹ \28\135è¹£\28\137è¹¤\28\134è¹²\28\140è¹´\
  \\10\146è¹¶\28\139è¹¼\28\141èº\129\28\142èº\132\28\145èº\133\28\144èº\135\
  \\28\143èº\138\28\147èº\139\28\146èº\141\16tèº\145\28\149èº\147\28\148èº\
  \\148\28\150èº\153\28\151èº¡\28\153èºª\28\152èº«\11gèº¬\28\154èº¯\8+èº°\
  \\28\155èº±\28\157èº¾\28\158è»\133\28\159è»\134\28\156è»\136\28 è»\138\10\
  \Tè»\139\28¡è»\140\7\143è»\141\8Rè»\146\8¬è»\155\28¢è»\159\14.è»¢\13\157è\
  \»£\28£è»«\28¦è»¸\102è»»\28¥è»¼\28¤è»½\8yè»¾\28§è¼\131\6òè¼\133\28©è¼\137\
  \\9\154è¼\138\28¨è¼\140\28±è¼\146\28«è¼\147\28\173è¼\148\15£è¼\149\28ªè¼\
  \\153\28¬è¼\155\28°è¼\156\28®è¼\157\7\144è¼\159\28¯è¼¦\28²è¼©\14yè¼ª\17\
  \\22è¼¯\10\147è¼³\28³è¼¸\16\129è¼¹\28µè¼»\28´è¼¾\28¸è¼¿\16 è½\130\28·è½\
  \\132\7\13è½\133\28¶è½\134\28»è½\137\28ºè½\140\28¹è½\141\13\145è½\142\28¼\
  \è½\151\28½è½\156\28¾è½\159\9Lè½¡\8Dè½¢\28Àè½£\28Áè½¤\28Âè¾\155\11hè¾\156\
  \\28Ãè¾\158\10+è¾\159\28Äè¾£\28Åè¾§\18_è¾¨\18^è¾\173\28Æè¾®\25Ðè¾¯\28Çè¾°\
  \\12Ãè¾±\11Jè¾²\14_è¾·\28Èè¾º\15\147è¾»\13Rè¾¼\9^è¾¿\12Èè¿\130\6\9è¿\132\
  \\16\24è¿\133\11vè¿\142\8}è¿\145\8\31è¿\148\15\148è¿\154\28Éè¿¢\28Ëè¿¥\28\
  \Êè¿¦\6\158è¿©\144è¿ª\28Ìè¿«\14\151è¿\173\13\146è¿¯\28Íè¿°\10±è¿´\28Ïè¿·\
  \\16@è¿¸\28Þè¿¹\28Ñè¿º\28Òè¿½\13Gé\128\128\12\158é\128\129\12Wé\128\131\
  \\13æé\128\133\28Ðé\128\134\7´é\128\139\28Ùé\128\141\28Öé\128\142\28ãé\
  \\128\143\13çé\128\144\13\0é\128\145\28Óé\128\147\13|é\128\148\13²é\128\
  \\149\28Ôé\128\150\28Øé\128\151\11\128é\128\153\14\135é\128\154\13Jé\128\
  \\157\11Àé\128\158\28×é\128\159\12lé\128 \12bé\128¡\28Õé\128¢\5§é\128£\17\
  \Aé\128§\28Úé\128®\12\159é\128±\10\148é\128²\11ié\128µ\28Üé\128¶\28Ûé\128\
  \¸\5íé\128¹\28Ýé\128¼\15\14é\128¾\28åé\129\129\14\25é\129\130\11\139é\129\
  \\133\12øé\129\135\86é\129\137\28äé\129\138\16\150é\129\139\6\30é\129\141\
  \\15\149é\129\142\6\159é\129\143\28ßé\129\144\28àé\129\145\28áé\129\146\
  \\28âé\129\147\13ùé\129\148\12Âé\129\149\5áé\129\150\28æé\129\152\28çé\
  \\129\153\31!é\129\156\12{é\129\158\28èé\129 \6Sé\129¡\12+é\129£\8\173é\
  \\129¥\16¹é\129¨\28éé\129©\13\139é\129\173\12Xé\129®\10Ué\129¯\28êé\129²\
  \\28íé\129µ\10Åé\129¶\28ëé\129·\12\10é\129¸\12\9é\129º\5âé\129¼\17\9é\129\
  \½\28ïé\129¿\14ðé\130\128\28ñé\130\129\28ðé\130\130\28îé\130\131\25\4é\
  \\130\132\7Ré\130\135\28Îé\130\137\28óé\130\138\28òé\130\143\28ôé\130\145\
  \\16\151é\130£\14\31é\130¦\15Íé\130¨\28õé\130ª\10Wé\130¯\28öé\130±\28÷é\
  \\130µ\28øé\130¸\13\128é\131\129\5èé\131\138\98é\131\142\17Yé\131\155\28ü\
  \é\131¡\8Sé\131¢\28ùé\131¤\28úé\131¨\15Té\131\173\6óé\131µ\16\152é\131·\7\
  \ýé\131½\13³é\132\130\28ýé\132\146\28þé\132\153\28ÿé\132\173\13\129é\132°\
  \\29\1é\132²\29\0é\133\137\14\17é\133\138\29\2é\133\139\10\149é\133\140\
  \\10^é\133\141\14zé\133\142\13\17é\133\146\10pé\133\148\11\140é\133\150\
  \\29\3é\133\152\29\4é\133¢\11|é\133£\29\5é\133¥\29\6é\133©\29\7é\133ª\16Ï\
  \é\133¬\10\150é\133²\29\9é\133³\29\8é\133µ\99é\133·\9Sé\133¸\9ßé\134\130\
  \\29\12é\134\135\10Æé\134\137\29\11é\134\139\29\10é\134\141\12§é\134\144\
  \\8íé\134\146\11Áé\134\151\14®é\134\156\10\152é\134¢\29\13é\134¤\11\29é\
  \\134ª\29\16é\134«\29\14é\134¯\29\15é\134´\29\18é\134µ\29\17é\134¸\118é\
  \\134º\29\19é\135\128\29\20é\135\129\29\21é\135\134\14Îé\135\135\9\145é\
  \\135\136\10_é\135\137\29\22é\135\139\29\23é\135\140\16âé\135\141\10¤é\
  \\135\142\16lé\135\143\17\10é\135\144\29\24é\135\145\8 é\135\150\29\25é\
  \\135\152\13\130é\135\155\29\28é\135\156\7\24é\135\157\11jé\135\159\29\26\
  \é\135¡\29\27é\135£\13^é\135¦\15ôé\135§\8:é\135µ\29\30é\135¶\29\31é\135¼\
  \\29\29é\135¿\29!é\136\141\14\29é\136\142\6âé\136\145\29%é\136\148\29\34é\
  \\136\149\29$é\136\158\29 é\136©\29né\136¬\29#é\136´\17)é\136·\8Øé\136¿\
  \\29-é\137\132\13\147é\137\133\29(é\137\136\29+é\137\137\29)é\137\139\29.\
  \é\137\144\29/é\137\151\29'é\137\154\294é\137\155\6Té\137\158\29&é\137¢\
  \\14«é\137¤\29*é\137¦\11\30é\137±\9:é\137¾\15çé\138\128\8\34é\138\131\10¥\
  \é\138\133\13úé\138\145\12\12é\138\147\292é\138\149\29,é\138\150\291é\138\
  \\152\16Aé\138\154\136é\138\155\293é\138\156\290é\138\173\12\11é\138·\297\
  \é\138¹\296é\139\143\295é\139\146\15Îé\139¤\10Ûé\139©\298é\139ª\15\157é\
  \\139\173\63é\139²\15%é\139³\13\18é\139¸\7Øé\139º\29:é\139¼\9<é\140\134\9\
  \Ëé\140\143\299é\140\144\11\141é\140\152\11\142é\140\153\29@é\140\154\29B\
  \é\140 \119é\140¢\29Aé\140£\29Cé\140¦\8\17é\140¨\15$é\140«\10`é\140¬\17Bé\
  \\140®\29<é\140¯\9¶é\140²\17^é\140µ\29Eé\140º\29Dé\140»\29Fé\141\132\29;é\
  \\141\139\14'é\141\141\13´é\141\148\13Ué\141\150\29Ké\141\155\12âé\141\
  \\156\29Gé\141 \29Hé\141¬\8Lé\141®\29Jé\141µ\8®é\141¼\29Ié\141¾\11\31é\
  \\142\140\7\25é\142\148\29Oé\142\150\9}é\142\151\12Yé\142\154\13Hé\142§\6\
  \Úé\142¬\29Mé\142\173\29Né\142®\13Aé\142°\29Lé\142¹\29Pé\143\131\29Vé\143\
  \\136\29Yé\143\144\29Xé\143\145\13\140é\143\150\29Qé\143\151\29Ré\143\152\
  \\29Ué\143\157\29Wé\143¡\7þé\143¤\29Zé\143¥\29Té\143¨\29Sé\144\131\29^é\
  \\144\135\29_é\144\144\29`é\144\147\29]é\144\148\29\92é\144\152\11 é\144\
  \\153\13èé\144\154\29[é\144¡\29dé\144«\29bé\144µ\29cé\144¶\29aé\144¸\12¶é\
  \\144º\29eé\145\129\29fé\145\132\29hé\145\145\7Sé\145\146\29gé\145\147\16\
  \xé\145\154\29sé\145\155\29ié\145\158\29lé\145 \29jé\145¢\29ké\145ª\29mé\
  \\145°\29oé\145µ\29pé\145·\29qé\145¼\29té\145½\29ré\145¾\29ué\145¿\29wé\
  \\146\129\29vé\149·\137é\150\128\16eé\150\130\29xé\150\131\12\13é\150\135\
  \\29yé\150\137\15\130é\150\138\29zé\150\139\6Êé\150\143\6\27é\150\145\7Ué\
  \\150\147\7Té\150\148\29{é\150\150\29|é\150\152\29}é\150\153\29~é\150 \29\
  \\128é\150¢\7Vé\150£\6ôé\150¤\9=é\150¥\14´é\150§\29\130é\150¨\29\129é\150\
  \\173\29\131é\150²\6;é\150¹\29\134é\150»\29\133é\150¼\29\132é\150¾\29\135\
  \é\151\131\29\138é\151\135\5Åé\151\138\29\136é\151\140\29\140é\151\141\29\
  \\139é\151\148\29\142é\151\149\29\141é\151\150\29\143é\151\152\13ìé\151\
  \\156\29\144é\151¡\29\145é\151¢\29\147é\151¥\29\146é\152\156\15Lé\152¡\29\
  \\148é\152¨\29\149é\152ª\9£é\152®\29\150é\152¯\29\151é\152²\15èé\152»\12*\
  \é\152¿\5¢é\153\128\12\137é\153\130\29\152é\153\132\15Mé\153\139\29\155é\
  \\153\140\29\153é\153\141\9>é\153\143\29\154é\153\144\8Àé\153\155\15\131é\
  \\153\156\29\157é\153\157\29\159é\153\158\29\158é\153\159\29 é\153¢\6\0é\
  \\153£\11wé\153¤\10Üé\153¥\7Wé\153¦\29¡é\153ª\14\134é\153¬\29£é\153°\6\1é\
  \\153²\29¢é\153³\13Bé\153µ\17\11é\153¶\13éé\153·\29\156é\153¸\16äé\153º\8\
  \¯é\153½\16ºé\154\133\87é\154\134\16òé\154\136\8Gé\154\138\12 é\154\139\
  \\26@é\154\141\29¤é\154\142\6Ëé\154\143\11\143é\154\148\6õé\154\149\29¦é\
  \\154\151\29§é\154\152\29¥é\154\153\8\132é\154\155\9\155é\154\156\11!é\
  \\154 \6\2é\154£\17\23é\154§\29©é\154¨\28ìé\154ª\29¨é\154°\29¬é\154±\29ªé\
  \\154²\29«é\154´\29\173é\154¶\29®é\154·\17*é\154¸\29¯é\154¹\29°é\154»\11Ç\
  \é\154¼\14¹é\155\128\11\157é\155\129\7eé\155\132\16\153é\155\133\6«é\155\
  \\134\10\151é\155\135\8Ùé\155\137\29³é\155\139\29²é\155\140\10\19é\155\
  \\141\29´é\155\142\29±é\155\145\9Çé\155\149\29¸é\155\150\27ké\155\153\18\
  \\148é\155\155\11\151é\155\156\29¶é\155¢\16ãé\155£\14/é\155¨\6\10é\155ª\
  \\11áé\155«\104é\155°\15ué\155²\6\31é\155¶\17+é\155·\16Ëé\155¹\29¹é\155»\
  \\13¤é\156\128\10yé\156\132\29ºé\156\134\29»é\156\135\11ké\156\136\29¼é\
  \\156\138\17,é\156\141\29·é\156\142\29¾é\156\143\29Àé\156\145\29¿é\156\
  \\147\29½é\156\150\29Áé\156\153\29Âé\156\156\12Zé\156\158\6 é\156¤\29Ãé\
  \\156§\166é\156ª\29Äé\156°\29Åé\156²\17Ié\156¸\21Ðé\156¹\29Æé\156½\29Çé\
  \\156¾\29Èé\157\130\29Ìé\157\132\29Éé\157\134\29Êé\157\136\29Ëé\157\137\
  \\29Íé\157\146\11Âé\157\150\16ué\157\153\11Ãé\157\156\29Îé\157\158\14ñé\
  \\157 \29Ïé\157¡\30òé\157¢\16Jé\157¤\29Ðé\157¦\29Ñé\157¨\29Òé\157©\6öé\
  \\157«\29Ôé\157\173\11xé\157±\29Õé\157´\8Cé\157¹\29Öé\157º\29Úé\157¼\29Øé\
  \\158\129\29Ùé\158\132\7\19é\158\133\29×é\158\134\29Ûé\158\139\29Üé\158\
  \\141\5Æé\158\143\29Ýé\158\144\29Þé\158\152\11\34é\158\156\29ßé\158 \7¦é\
  \\158£\29âé\158¦\29áé\158¨\29àé\158«\28\17é\158\173\15\154é\158³\29ãé\158\
  \´\29äé\159\131\29åé\159\134\29æé\159\136\29çé\159\139\29èé\159\147\7Xé\
  \\159\156\29éé\159\173\29êé\159®\14Bé\159²\29ìé\159³\6yé\159µ\29ïé\159¶\
  \\29îé\159»\6\3é\159¿\7ÿé \129\15\133é \130\138é \131\9`é \133\9@é \134\
  \\10Çé \136\11{é \140\29ñé \143\29ðé \144\16¡é \145\7fé \146\14Ðé \147\14\
  \\26é \151\11\156é \152\17\12é \154\8zé ¡\29ôé ¤\29óé ¬\15êé \173\13êé ´\
  \\6/é ·\29õé ¸\29òé »\150é ¼\16Êé ½\29öé¡\134\29÷é¡\139\29ùé¡\140\12¨é¡\
  \\141\6úé¡\142\6ûé¡\143\29øé¡\148\7gé¡\149\8°é¡\152\7hé¡\155\13\158é¡\158\
  \\17\30é¡§\8Úé¡«\29úé¡¯\29ûé¡°\29üé¡±\30\0é¡³\30\2é¡´\30\1é¢¨\15Wé¢ª\30\3\
  \é¢¯\30\4é¢±\30\5é¢¶\30\6é£\131\30\8é£\132\30\7é£\134\30\9é£\155\14òé£\
  \\156\26\10é£\159\11Hé£¢\7\145é£©\30\10é£«\30\11é£\173\18jé£®\22\154é£¯\
  \\14Ñé£²\5ùé£´\5¹é£¼\10\20é£½\15Ïé£¾\11<é¤\131\30\12é¤\133\16]é¤\137\30\
  \\13é¤\138\16»é¤\140\6!é¤\144\9àé¤\146\30\14é¤\147\6¬é¤\148\30\15é¤\152\
  \\30\16é¤\157\30\18é¤\158\30\19é¤ \30\21é¤¡\30\17é¤¤\30\20é¤¨\7Yé¤¬\30\22\
  \é¤®\30\23é¤½\30\24é¤¾\30\25é¥\130\30\26é¥\133\30\28é¥\137\30\27é¥\139\30\
  \\30é¥\140\30!é¥\144\30\29é¥\145\30\31é¥\146\30 é¥\149\30\34é¥\151\8\0é¦\
  \\150\10qé¦\151\30#é¦\152\30$é¦\153\9Aé¦¥\30%é¦¨\6Ýé¦¬\14né¦\173\30&é¦®\
  \\30'é¦³\12ùé¦´\14)é¦¼\30(é§\129\14\157é§\132\12\138é§\133\67é§\134\8,é§\
  \\136\8-é§\144\13\19é§\145\30-é§\146\8.é§\149\6\173é§\152\30,é§\155\30*é§\
  \\157\30+é§\159\30)é§¢\307é§\173\30.é§®\30/é§±\300é§²\301é§¸\303é§»\302é§\
  \¿\10¸é¨\129\304é¨\133\306é¨\142\7\146é¨\143\305é¨\146\12[é¨\147\8±é¨\153\
  \\308é¨¨\12\139é¨«\309é¨°\13ëé¨·\30:é¨¾\30@é©\128\30=é©\130\30<é©\131\30>\
  \é©\133\30;é©\141\30Bé©\149\30Aé©\151\30Dé©\154\8\1é©\155\30Cé©\159\30Eé©\
  \¢\30Fé©¤\30Hé©¥\30Gé©©\30Ié©ª\30Ké©«\30Jéª¨\9\92éª\173\30Léª°\30Méª¸\6Ûé\
  \ª¼\30Né«\128\30Oé«\132\11\145é«\143\30Pé«\145\30Qé«\147\30Ré«\148\30Sé«\
  \\152\9Bé«\158\30Té«\159\30Ué«¢\30Vé«£\30Wé«¦\30Xé«ª\14¯é««\30Zé«\173\15\
  \\5é«®\30[é«¯\30Yé«±\30]é«´\30\92é«·\30^é«»\30_é¬\134\30`é¬\152\30aé¬\154\
  \\30bé¬\159\30cé¬¢\30dé¬£\30eé¬¥\30fé¬§\30gé¬¨\30hé¬©\30ié¬ª\30jé¬®\30ké¬\
  \¯\30lé¬±\22\148é¬²\30mé¬»\25vé¬¼\7\147é\173\129\6Àé\173\130\9pé\173\131\
  \\30oé\173\132\30né\173\133\16#é\173\141\30qé\173\142\30ré\173\143\30pé\
  \\173\145\30sé\173\148\16\2é\173\152\30té\173\154\7Ûé\173¯\17Dé\173´\30ué\
  \®\131\30wé®\142\5¼é®\145\30xé®\146\15ié®\147\30vé®\150\30yé®\151\30zé®\
  \\159\30{é® \30|é®¨\30}é®ª\16\14é®«\9Ìé®\173\9¸é®®\12\14é®´\30~é®¹\30\129\
  \é¯\128\30\127é¯\134\30\130é¯\137\8ïé¯\138\30\128é¯\143\30\131é¯\145\30\
  \\132é¯\146\30\133é¯\148\30\137é¯\150\9Éé¯\155\12¢é¯¡\30\138é¯¢\30\135é¯£\
  \\30\134é¯¤\30\136é¯¨\8~é¯°\30\142é¯±\30\141é¯²\30\140é¯µ\5±é°\132\30\152\
  \é°\134\30\148é°\136\30\149é°\137\30\145é°\138\30\151é°\140\30\147é°\141\
  \\7\2é°\144\17ké°\146\30\150é°\147\30\146é°\148\30\144é°\149\30\143é°\155\
  \\30\154é°¡\30\157é°¤\30\156é°¥\30\155é°\173\15(é°®\30\153é°¯\5ñé°°\30\
  \\158é°²\30 é°¹\7\15é°º\30\139é°»\6\22é°¾\30¢é±\134\30¡é±\135\30\159é±\
  \\136\12Ìé±\146\16\16é±\151\17\24é±\154\30£é± \30¤é±§\30¥é±¶\30¦é±¸\30§é³\
  \¥\139é³§\30¨é³©\14µé³«\30\173é³¬\30©é³°\30ªé³³\15Ðé³´\16Bé³¶\14\14é´\131\
  \\30®é´\134\30¯é´\135\13üé´\136\30¬é´\137\30«é´\142\6hé´\146\30·é´\149\30\
  \¶é´\155\6Ué´\159\30´é´£\30³é´¦\30±é´¨\7\27é´ª\30°é´«\100é´¬\6gé´»\9Cé´¾\
  \\30ºé´¿\30¹éµ\129\30¸éµ\132\30µéµ\134\30»éµ\136\30¼éµ\144\30Äéµ\145\30Ãé\
  \µ\153\30Åéµ\156\6\12éµ\157\30Àéµ\158\30Áéµ \9Téµ¡\167éµ¤\30Âéµ¬\15Ñéµ¯\
  \\30Êéµ²\30Æéµº\30Ëé¶\135\30Èé¶\137\30Çé¶\143\8{é¶\154\30Ìé¶¤\30Íé¶©\30Îé\
  \¶«\30Éé¶¯\30²é¶²\30Ïé¶´\13_é¶¸\30Óé¶º\30Ôé¶»\30Òé·\129\30Ñé·\130\30×é·\
  \\132\30Ðé·\134\30Õé·\143\30Öé·\147\30Ùé·\153\30Øé·¦\30Ûé·\173\30Üé·¯\30Ý\
  \é·²\17hé·¸\30Úé·¹\12©é·º\9«é·½\30Þé¸\154\30ßé¸\155\30àé¸\158\30áé¹µ\30âé\
  \¹¸\8²é¹¹\30ãé¹½\30äé¹¿\10-éº\129\30åéº\136\30æéº\139\30çéº\140\30èéº\145\
  \\30ëéº\146\30ééº\147\17[éº\149\30êéº\151\17-éº\157\30ìéº\159\17\25éº¥\30\
  \íéº¦\14\158éº©\30îéºª\30ðéº\173\30ñéº¸\30ïéº¹\9Méºº\16Kéº»\16\3éº¼\205éº\
  \¾\22Àéº¿\16\27é»\132\6ié»\140\30óé»\141\7¯é»\142\30ôé»\143\30õé»\144\30ö\
  \é»\146\9Ué»\148\30÷é»\152\23Òé»\153\16Yé»\155\12¡é»\156\30øé»\157\30úé»\
  \\158\30ùé» \30ûé»¥\30üé»¨\30ýé»¯\30þé»´\31\0é»¶\31\1é»·\31\2é»¹\31\3é»»\
  \\31\4é»¼\31\5é»½\31\6é¼\135\31\7é¼\136\31\8é¼\142\13\131é¼\147\8Ûé¼\149\
  \\31\10é¼ \12,é¼¡\31\11é¼¬\31\12é¼»\15\0é¼¾\31\13é½\138\31\14é½\139\24Öé½\
  \\142\28Xé½\143\29ëé½\146\31\15é½\148\31\16é½\159\31\18é½ \31\19é½¡\31\20\
  \é½¢\17.é½£\31\17é½¦\31\21é½§\31\22é½ª\31\24é½¬\31\23é½²\31\26é½¶\31\27é½\
  \·\31\25é¾\141\16ôé¾\149\31\28é¾\156\31\29é¾\157\24óé¾ \31\30ï¼\129\0\9ï¼\
  \\131\0Tï¼\132\0Pï¼\133\0Sï¼\134\0Uï¼\136\0)ï¼\137\0*ï¼\138\0Vï¼\139\0;ï¼\
  \\140\0\3ï¼\142\0\4ï¼\143\0\30ï¼\144\0Ïï¼\145\0Ðï¼\146\0Ñï¼\147\0Òï¼\148\
  \\0Óï¼\149\0Ôï¼\150\0Õï¼\151\0Öï¼\152\0×ï¼\153\0Øï¼\154\0\6ï¼\155\0\7ï¼\
  \\156\0Cï¼\157\0Aï¼\158\0Dï¼\159\0\8ï¼ \0Wï¼¡\0àï¼¢\0áï¼£\0âï¼¤\0ãï¼¥\0äï\
  \¼¦\0åï¼§\0æï¼¨\0çï¼©\0èï¼ª\0éï¼«\0êï¼¬\0ëï¼\173\0ìï¼®\0íï¼¯\0îï¼°\0ïï¼±\
  \\0ðï¼²\0ñï¼³\0òï¼´\0óï¼µ\0ôï¼¶\0õï¼·\0öï¼¸\0÷ï¼¹\0øï¼º\0ùï¼»\0-ï¼¼\0\31ï\
  \¼½\0.ï¼¾\0\15ï¼¿\0\17ï½\128\0\13ï½\129\1\1ï½\130\1\2ï½\131\1\3ï½\132\1\4\
  \ï½\133\1\5ï½\134\1\6ï½\135\1\7ï½\136\1\8ï½\137\1\9ï½\138\1\10ï½\139\1\11\
  \ï½\140\1\12ï½\141\1\13ï½\142\1\14ï½\143\1\15ï½\144\1\16ï½\145\1\17ï½\146\
  \\1\18ï½\147\1\19ï½\148\1\20ï½\149\1\21ï½\150\1\22ï½\151\1\23ï½\152\1\24ï\
  \½\153\1\25ï½\154\1\26ï½\155\0/ï½\156\0\34ï½\157\00ï¿£\0\16ï¿¥\0O"

{-
-- Compile with:
--   stack ghc --package iconv --package bytestring --package containers --package binary -- <name>.hs

module Main where

import           Codec.Text.IConv     (convertStrictly)
import           Data.Binary          (encode)
import           Data.Bits            (shiftL, shiftR, (.&.))
import qualified Data.ByteString.Lazy as BL
import           Data.Char            (chr, isDigit)
import           Data.List            (intercalate)
import qualified Data.Map             as M
import           Data.Maybe           (mapMaybe)
import           Data.Word            (Word16, Word32, Word8)

unicodeToKanji :: Word32 -> Maybe (Char, Word16)
unicodeToKanji cw =
  case convertStrictly "UTF-32BE" "Shift_JIS" cb of
    Left kc
      | BL.length kc == 2 ->
          case fromIntegral (kc `BL.index` 0) `shiftL` 8 + fromIntegral (kc `BL.index` 1) of
            k
              | k >= 0x8140 && k <= 0x9ffc -> Just (cc, go (k - 0x8140))
              | k >= 0xe040 && k <= 0xebbf -> Just (cc, go (k - 0xc140))
              | otherwise -> Nothing
    _ -> Nothing
  where
    go :: Word16 -> Word16
    go k = (k `shiftR` 8) * 0xc0 + (k .&. 0xff)
    cc :: Char
    cc = chr (fromIntegral cw)
    cb :: BL.ByteString
    cb =
      BL.pack
        [ fromIntegral (cw `shiftR` 24)
        , fromIntegral (cw `shiftR` 16)
        , fromIntegral (cw `shiftR` 8)
        , fromIntegral cw
        ]

showB :: Word8 -> String
showB c
  -- c0-ctrl    quote        backslash    del + c1-ctrl-chars        shy
  | c < 0x20 || c == 0x22 || c == 0x5c || (c >= 0x7f && c < 0xa0) || c == 0xad =
      '\\' : show c
  | otherwise =
      -- This character can be encoded directly
      [chr (fromIntegral c)]

addEmptyString :: [String] -> [String]
addEmptyString (x@('\\':_) : y@(y1:_) : zs)
  | isDigit y1 = x : "\&" : y : addEmptyString zs
addEmptyString (x:xs) = x : addEmptyString xs
addEmptyString [] = []

chunksOf :: Int -> [[a]] -> [[a]]
chunksOf n0 x0 = go [] n0 x0
  where
    go :: [a] -> Int -> [[a]] -> [[a]]
    go p _ [] = [p]
    go p n (x:xs) =
      let
        n' = n - length x
      in
        case n' `compare` 0 of
          LT -> p : go [] n0 (x:xs)
          EQ -> (p++x) : go [] n0 xs
          GT -> go (p++x) n' xs

main :: IO ()
main = do
  let
    kanjiMap :: M.Map Char Word16
    kanjiMap = M.fromList (mapMaybe unicodeToKanji [0 .. 0x10ffff])
    stream :: [String]
    stream = addEmptyString (map showB (BL.unpack (encode kanjiMap)))
  putStrLn "kanjiMap :: M.Map Char Word16"
  putStrLn "kanjiMap = decode"
  putStrLn ("  \"" ++ intercalate "\\\n  \\" (chunksOf 72 stream) ++ "\"")
-}
