module Data.CompactSpace (Compact(..)) where
import Data.Searchable (K(..),list2K)
import Data.Void (Void)
import Control.Applicative ((<|>))

-- | Stone spaces are compact zero-dimensional topological spaces. 
-- In Haskell, *zero-dimensional* is the same as a well-behaved 'Eq' instance: 
-- any two distinct values can be distingished by a Boolean predicate. 
-- *Compactness* is the same as computable universal quantification.  
-- The 'K' monad admits exhaustive search by total predicates 
-- and thereby witnesses universal quantification. 
-- 
-- Stone spaces are dual to Boolean algebras. 
-- The canonical Boolean algebra associated with a Stone space X
-- is the algebra of its clopen sets, that is, 
-- the algebra of extents of the predicates on X. 
-- Therefore, a function 
-- 
-- @
-- f :: Y -> X
-- @
-- 
-- between Compact spaces can be encoded as a predicate transformer 
-- 
-- @
-- (X -> Bool) -> (Y -> Bool)
-- @
-- 
-- that preserves the Boolean algebra structure. 
class Eq space => Compact space where
    theType :: K space

instance Compact Void where
    theType = Emptyset

instance Compact () where
    theType = pure ()

instance Compact Bool where
    theType = list2K [False,True]

instance Compact Ordering where
    theType = list2K [LT,EQ,GT]

instance (Compact x, Compact y) => Compact (x,y) where
    theType = (,) <$> theType <*> theType

instance (Compact x, Compact y) => Compact (Either x y) where
    theType = fmap Left theType <|> fmap Right theType

-- | countable products of Compact spaces are Compact spaces
instance Compact x => Compact [x] where
    theType = sequence (repeat theType)
