--------------------------------------------------------------------
-- |
-- Module    : Selenium.Syntax
-- Copyright : (c) Galois, Inc. 2007
-- License   : BSD3
--
-- Maintainer: Aaron Tomb <atomb@galois.com>
-- Stability : provisional
-- Portability: non-portable (multi-parameter type classes)
--
-- Haskell data structures describing the commands understood by the Selenium
-- web application testing system. 
--
--------------------------------------------------------------------
--
-- The syntax of the selenium command and assert language
--

module Selenium.Syntax (
    SCommand(..),
    Locator(..),
    SelectOptionLocator(..)
  ) where

import Selenium.Pretty

-- | An SCommand: an action or accessor
data SCommand
    = SClick
    | SDoubleClick
    | SClickAt
    | SDoubleClickAt
    | SFireEvent
    | SKeyPress
    | SShiftKeyDown
    | SShiftKeyUp
    | SMetaKeyDown
    | SMetaKeyUp
    | SAltKeyDown
    | SAltKeyUp
    | SControlKeyDown
    | SControlKeyUp
    | SKeyDown
    | SKeyUp
    | SMouseOver
    | SMouseOut
    | SMouseDown
    | SMouseDownAt
    | SMouseUp
    | SMouseUpAt
    | SMouseMove
    | SMouseMoveAt
    | SType
    | STypeKeys
    | SSetSpeed
    | SGetSpeed
    | SCheck
    | SUncheck
    | SSelect
    | SAddSelection
    | SRemoveSelection
    | SRemoveAllSelections
    | SSubmit
    | SOpen
    | SOpenWindow
    | SSelectWindow
    | SSelectFrame
    | SFrameMatchesExpression
    | SWindowMatchesExpression
    | SWaitForPopup
    | SCancelNextPrompt
    | SAnswerNextPrompt
    | SGoBack
    | SRefresh
    | SClose
    | SIsAlertPresent
    | SIsPromptPresent
    | SIsConfirmationPresent
    | SGetAlert
    | SGetConfirmation
    | SGetPrompt
    | SGetLocation
    | SGetTitle
    | SGetBodyText
    | SGetValue
    | SGetText
    | SHighlight
    | SEval
    | SIsChecked
    | SGetTable
    | SGetSelectedLabels
    | SGetSelectedLabel
    | SGetSelectedValues
    | SGetSelectedValue
    | SGetSelectedIndexes
    | SGetSelectedIndex
    | SGetSelectedIds
    | SGetSelectedId
    | SIsSomethingSelected
    | SGetSelectOptions
    | SGetAttribute
    | SIsTextPresent
    | SIsElementPresent
    | SIsVisible
    | SIsEditable
    | SGetAllButtons
    | SGetAllLinks
    | SGetAllFields
    | SGetAttrFromAllWindows
    | SSetMouseSpeed
    | SGetMouseSpeed
    | SDragAndDrop
    | SDragAndDropToObject
    | SWindowFocus
    | SWindowMaximize
    | SGetAllWindowIds
    | SGetAllWindowNames
    | SGetAllWindowTitles
    | SGetHTMLSource
    | SSetCursorPosition
    | SGetElementIndex
    | SIsOrdered
    | SGetElementPositionLeft
    | SGetElementPositionTop
    | SGetElementWidth
    | SGetElementHeight
    | SGetCursorPosition
    | SSetContext
    | SGetExpression
    | SWaitForCondition
    | SGetXPathCount
    | SAddLocationStrategy
    | SSetTimeout
    | SWaitForPage
    | SGetCookie
    | SCreateCookie
    | SDeleteCookie
    | STestComplete
    | SNewSession

instance Show SCommand where
    show SClick                   = "click"
    show SDoubleClick             = "doubleClick"
    show SClickAt                 = "clickAt"
    show SDoubleClickAt           = "doubleClickAt"
    show SFireEvent               = "fireEvent"
    show SKeyPress                = "keyPress"
    show SShiftKeyDown            = "shiftKeyDown"
    show SShiftKeyUp              = "shiftKeyUp"
    show SMetaKeyDown             = "metaKeyDown"
    show SMetaKeyUp               = "metaKeyUp"
    show SAltKeyDown              = "altKeyDown"
    show SAltKeyUp                = "altKeyUp"
    show SControlKeyDown          = "controlKeyDown"
    show SControlKeyUp            = "controlKeyUp"
    show SKeyDown                 = "keyDown"
    show SKeyUp                   = "keyUp"
    show SMouseOver               = "mouseOver"
    show SMouseOut                = "mouseOut"
    show SMouseDown               = "mouseDown"
    show SMouseDownAt             = "mouseDownAt"
    show SMouseUp                 = "mouseUp"
    show SMouseUpAt               = "mouseUpAt"
    show SMouseMove               = "mouseMove"
    show SMouseMoveAt             = "mouseMoveAt"
    show SType                    = "type"
    show STypeKeys                = "typeKeys"
    show SSetSpeed                = "setSpeed"
    show SGetSpeed                = "getSpeed"
    show SCheck                   = "check"
    show SUncheck                 = "uncheck"
    show SSelect                  = "select"
    show SAddSelection            = "addSelection"
    show SRemoveSelection         = "removeSelection"
    show SRemoveAllSelections     = "removeAllSelections"
    show SSubmit                  = "submit"
    show SOpen                    = "open"
    show SOpenWindow              = "openWindow"
    show SSelectWindow            = "selectWindow"
    show SSelectFrame             = "selectFrame"
    show SFrameMatchesExpression  = "getWhetherThisFrameMatchFrameExpression"
    show SWindowMatchesExpression = "getWhetherThisWindowMatchWindowExpression"
    show SWaitForPopup            = "waitForPopup"
    show SCancelNextPrompt        = "chooseCancelOnNextPrompt"
    show SAnswerNextPrompt        = "answerOnNextPrompt"
    show SGoBack                  = "goBack"
    show SRefresh                 = "refresh"
    show SClose                   = "close"
    show SIsAlertPresent          = "isAlertPresent"
    show SIsPromptPresent         = "isPromptPresent"
    show SIsConfirmationPresent   = "isConfirmationPresent"
    show SGetAlert                = "getAlert"
    show SGetConfirmation         = "getConfirmation"
    show SGetPrompt               = "getPrompt"
    show SGetLocation             = "getLocation"
    show SGetTitle                = "getTitle"
    show SGetBodyText             = "getBodyText"
    show SGetValue                = "getValue"
    show SGetText                 = "getText"
    show SHighlight               = "highlight"
    show SEval                    = "getEval"
    show SIsChecked               = "isChecked"
    show SGetTable                = "getTable"
    show SGetSelectedLabels       = "getSelectedLabels"
    show SGetSelectedLabel        = "getSelectedLabel"
    show SGetSelectedValues       = "getSelectedValues"
    show SGetSelectedValue        = "getSelectedValue"
    show SGetSelectedIndexes      = "getSelectedIndexes"
    show SGetSelectedIndex        = "getSelectedIndex"
    show SGetSelectedIds          = "getSelectedIds"
    show SGetSelectedId           = "getSelectedId"
    show SIsSomethingSelected     = "isSomethingSelected"
    show SGetSelectOptions        = "getSelectOptions"
    show SGetAttribute            = "getAttribute"
    show SIsTextPresent           = "isTextPresent"
    show SIsElementPresent        = "isElementPresent"
    show SIsVisible               = "isVisible"
    show SIsEditable              = "isEditable"
    show SGetAllButtons           = "getAllButtons"
    show SGetAllLinks             = "getAllLinks"
    show SGetAllFields            = "getAllFields"
    show SGetAttrFromAllWindows   = "getAttributeFromAllWindows"
    show SSetMouseSpeed           = "setMouseSpeed"
    show SGetMouseSpeed           = "getMouseSpeed"
    show SDragAndDrop             = "dragAndDrop"
    show SDragAndDropToObject     = "dragAndDropToObject"
    show SWindowFocus             = "windowFocus"
    show SWindowMaximize          = "windowMaximize"
    show SGetAllWindowIds         = "getAllWindowIds"
    show SGetAllWindowNames       = "getAllWindowNames"
    show SGetAllWindowTitles      = "getAllWindowTitles"
    show SGetHTMLSource           = "getHtmlSource"
    show SSetCursorPosition       = "setCursorPosition"
    show SGetElementIndex         = "getElementIndex"
    show SIsOrdered               = "isOrdered"
    show SGetElementPositionLeft  = "getElementPositionLeft"
    show SGetElementPositionTop   = "getElementPositionTop"
    show SGetElementWidth         = "getElementWidth"
    show SGetElementHeight        = "getElementHeight"
    show SGetCursorPosition       = "getCursorPosition"
    show SSetContext              = "setContext"
    show SGetExpression           = "getExpression"
    show SWaitForCondition        = "waitForCondition"
    show SGetXPathCount           = "getXpathCount"
    show SAddLocationStrategy     = "addLocationStrategy"
    show SSetTimeout              = "setTimeout"
    show SWaitForPage             = "waitForPageToLoad"
    show SGetCookie               = "getCookie"
    show SCreateCookie            = "createCookie"
    show SDeleteCookie            = "deleteCookie"
    show STestComplete            = "testComplete"
    show SNewSession              = "getNewBrowserSession"

------------------------------------------------------------------------

newtype Coordinate = C (Int, Int)

data Locator
    = IdOrName String
    | Id       String
    | Name     String
    | DOM      String
    | XPath    String
    | Link     String
    | CSS      String

data SelectOptionLocator
    = OptionLabel String
    | OptionValue String
    | OptionId    String
    | OptionIndex Int

instance Show Coordinate where
    show (C (x, y)) = show x ++ "," ++ show y

instance Show Locator where
    show (IdOrName s)     = "identifier" <=> s
    show (Id s)           = "id"         <=> s
    show (Name s)         = "name"       <=> s
    show (DOM s)          = "dom"        <=> s
    show (XPath s)        = "xpath"      <=> s
    show (Link s)         = "link"       <=> s
    show (CSS s)          = "css"        <=> s

instance Show SelectOptionLocator where
    show (OptionLabel s) = "label"       <=> s
    show (OptionValue s) = "value"       <=> s
    show (OptionId s)    = "id"          <=> s
    show (OptionIndex s) = "index"       <=> show s
