{-# LANGUAGE ForeignFunctionInterface, CPP #-}

module SQL.CLI where

import Foreign.C.Types
import Foreign.Ptr (Ptr, nullPtr)
import Data.Bits ((.&.), complement)
-- API declaration data types
type SQLCHAR = CUChar

type SQLINTEGER = CLong

type SQLSMALLINT = CShort
type SQLDOUBLE = CDouble
type SQLFLOAT = CDouble
type SQLREAL = CFloat
type SQLPOINTER = Ptr ()
type SQLDATE = CUChar
type SQLTIME = CUChar
type SQLTIMESTAMP = CUChar
type SQLVARCHAR = CUChar
type SQLDECIMAL = CUChar
type SQLNUMERIC = CUChar

-- function return type
type SQLRETURN = SQLSMALLINT

-- generic data structures
-- we use SQLHANDLE type to be compatible with ODBC definitions
#if ((defined WIN32) || (defined _WIN64))
type SQLHANDLE = Ptr ()
#else
type SQLHANDLE = SQLINTEGER
#endif

type SQLHENV = SQLHANDLE
type SQLHDBC = SQLHANDLE
type SQLHSTMT = SQLHANDLE
type SQLHDESC = SQLHANDLE

-- using SQLLEN and SQLULEN types to be compatible with ODBC definitions
type SQLLEN = CLong
type SQLULEN = CULong

-- special length/indicator values
-- The value of 'sql_null_data' should
-- not be changed; it was chosen for compatibility with the
-- ISO SQL definition of the value returned to indicate a null value. */
sql_null_data :: Num a => a
sql_null_data        = -1

sql_data_at_exec :: Num a => a
sql_data_at_exec     = -2

-- return values from functions
-- These codes should not be changed;
-- they were chosen for compatibility with the ISO SQL definition * of SQLCODE.
sql_success             :: SQLRETURN
sql_success             = 0

sql_success_with_info   :: SQLRETURN
sql_success_with_info   = 1

sql_no_data             :: SQLRETURN
sql_no_data             = 100

sql_error               :: SQLRETURN
sql_error               = -1

sql_invalid_handle      :: SQLRETURN
sql_invalid_handle      = -2

sql_need_data           :: SQLRETURN
sql_need_data           = 99

-- | Test for 'sql_success' or 'sql_success_with_info'
sql_succeeded :: SQLRETURN -> Bool
sql_succeeded rc = (rc .&. (complement 1)) == 0

-- flags for null-terminated strings
sql_nts :: Num a => a
sql_nts = -3
sql_ntsl :: Num a => a
sql_ntsl = -3

-- | maximum message lenth
-- Vendors may increase this constant but its value must be at lieast 512
sql_max_message_length :: Num a => a
sql_max_message_length = 512

-- | maximum identifier length
sql_max_id_length :: Num a => a
sql_max_id_length = 18


-- date/time length constants
sql_date_len :: Num a => a
sql_date_len = 10

sql_time_len :: Num a => a
sql_time_len = 8 -- add P+1 if precision is nonzero

sql_timestamp_len :: Num a => a
sql_timestamp_len = 19 -- add P+1 if precision is nonzero

-- handle type identifiers
sql_handle_env :: Num a => a
sql_handle_env = 1

sql_handle_dbc :: Num a => a
sql_handle_dbc = 2

sql_handle_stmt :: Num a => a
sql_handle_stmt = 3

sql_handle_desc :: Num a => a
sql_handle_desc = 4

-- | environment attribute
sql_attr_output_nts :: Num a => a
sql_attr_output_nts = 10001

-- | connection attribute
sql_attr_auto_ipd :: Num a => a
sql_attr_auto_ipd = 10001

-- | connection and schema attributes
sql_attr_metadata_id :: Num a => a
sql_attr_metadata_id = 10014

-- statement attributes
sql_attr_app_row_desc           :: Num a => a
sql_attr_app_row_desc           = 10010

sql_attr_app_param_desc         :: Num a => a
sql_attr_app_param_desc         = 10011

sql_attr_imp_row_desc           :: Num a => a
sql_attr_imp_row_desc           = 10012

sql_attr_imp_param_desc         :: Num a => a
sql_attr_imp_param_desc         = 10013

sql_attr_cursor_scrollable      :: Num a => a
sql_attr_cursor_scrollable      = -1

sql_attr_cursor_sensitivity     :: Num a => a
sql_attr_cursor_sensitivity     = -2

-- identifiers of fields in the sql descriptor
sql_desc_count                  :: Num a => a
sql_desc_count                  = 1001

sql_desc_type                   :: Num a => a
sql_desc_type                   = 1002

sql_desc_length                 :: Num a => a
sql_desc_length                 = 1003

sql_desc_octet_length_ptr        :: Num a => a
sql_desc_octet_length_ptr        = 1004

sql_desc_precision              :: Num a => a
sql_desc_precision              = 1005

sql_desc_scale                  :: Num a => a
sql_desc_scale                  = 1006

sql_desc_datetime_interval_code :: Num a => a
sql_desc_datetime_interval_code = 1007

sql_desc_nullable               :: Num a => a
sql_desc_nullable               = 1008

sql_desc_indicator_ptr          :: Num a => a
sql_desc_indicator_ptr          = 1009

sql_desc_data_ptr               :: Num a => a
sql_desc_data_ptr               = 1010

sql_desc_name                   :: Num a => a
sql_desc_name                   = 1011

sql_desc_unnamed                :: Num a => a
sql_desc_unnamed                = 1012

sql_desc_octet_length           :: Num a => a
sql_desc_octet_length           = 1013

sql_desc_alloc_type             :: Num a => a
sql_desc_alloc_type             = 1099

-- identifiers of fields in the diagnostics area
sql_diag_returncode :: Num a => a
sql_diag_returncode = 1

sql_diag_number :: Num a => a
sql_diag_number = 2

sql_diag_row_count :: Num a => a
sql_diag_row_count = 3

sql_diag_sqlstate :: Num a => a
sql_diag_sqlstate = 4


sql_diag_native :: Num a => a
sql_diag_native = 5

sql_diag_message_text :: Num a => a
sql_diag_message_text = 6

sql_diag_dynamic_function :: Num a => a
sql_diag_dynamic_function = 7

sql_diag_class_origin :: Num a => a
sql_diag_class_origin = 8

sql_diag_subclass_origin :: Num a => a
sql_diag_subclass_origin = 9

sql_diag_connection_name :: Num a => a
sql_diag_connection_name = 10

sql_diag_server_name :: Num a => a
sql_diag_server_name = 11

sql_diag_dynamic_function_code :: Num a => a
sql_diag_dynamic_function_code = 12


-- dynamic function codes
-- These codes should not be changed;
-- they were chosen for compatibility with the ISO SQL definition * of dynamic function codes.
sql_diag_alter_table :: Num a => a
sql_diag_alter_table = 4

sql_diag_create_index :: Num a => a
sql_diag_create_index = -1

sql_diag_create_table :: Num a => a
sql_diag_create_table = 77

sql_diag_create_view :: Num a => a
sql_diag_create_view = 84

sql_diag_delete_where :: Num a => a
sql_diag_delete_where = 19

sql_diag_drop_index :: Num a => a
sql_diag_drop_index = -2

sql_diag_drop_table :: Num a => a
sql_diag_drop_table = 32

sql_diag_drop_view :: Num a => a
sql_diag_drop_view = 36

sql_diag_dynamic_delete_cursor :: Num a => a
sql_diag_dynamic_delete_cursor = 38

sql_diag_dynamic_update_cursor :: Num a => a
sql_diag_dynamic_update_cursor = 81

sql_diag_grant :: Num a => a
sql_diag_grant = 48

sql_diag_insert :: Num a => a
sql_diag_insert = 50

sql_diag_revoke :: Num a => a
sql_diag_revoke = 59

sql_diag_select_cursor :: Num a => a
sql_diag_select_cursor = 85

sql_diag_unknown_statement :: Num a => a
sql_diag_unknown_statement = 0

sql_diag_update_where :: Num a => a
sql_diag_update_where = 82

-- SQL data type codes
-- These codes should not be changed;
-- they were chosen for compatibility with the ISO SQL definition * of data type codes.
sql_char :: Num a => a
sql_char = 1

sql_numeric :: Num a => a
sql_numeric = 2

sql_decimal :: Num a => a
sql_decimal = 3

sql_integer :: Num a => a
sql_integer = 4

sql_smallint :: Num a => a
sql_smallint = 5

sql_float :: Num a => a
sql_float = 6

sql_real :: Num a => a
sql_real = 7

sql_double :: Num a => a
sql_double = 8

sql_datetime :: Num a => a
sql_datetime = 9

sql_varchar :: Num a => a
sql_varchar = 12

-- One-parameter shortcuts for date/time data types
sql_type_date :: Num a => a
sql_type_date = 91

sql_type_time :: Num a => a
sql_type_time = 92

sql_type_timestamp :: Num a => a
sql_type_timestamp = 93


-- Statement attribute values for cursor sensitivity
sql_unspecified :: Num a => a
sql_unspecified = 0

sql_insensitive :: Num a => a
sql_insensitive = 1

sql_sensitive :: Num a => a
sql_sensitive = 2


-- GetTypeInfo() request for all data types
sql_all_types :: Num a => a
sql_all_types = 0


-- Default conversion code for BindCol(), BindParam() and GetData()
sql_default :: Num a => a
sql_default = 99


-- GetData() code indicating that the application row descriptor * specifies the data type
sql_ard_type :: Num a => a
sql_ard_type = -99


-- SQL date/time type subcodes
-- These codes should not be changed;
-- they were chosen for compatibility with the ISO SQL definition * of date/time type subcodes.
sql_code_date :: Num a => a
sql_code_date = 1

sql_code_time :: Num a => a
sql_code_time = 2

sql_code_timestamp :: Num a => a
sql_code_timestamp = 3


-- CLI option values
sql_false :: Num a => a
sql_false = 0

sql_true :: Num a => a
sql_true = 1


-- values of NULLABLE field in descriptor
-- These codes should not be changed;
-- they were chosen for compatibility with the ISO SQL definition * of the NULLABLE descriptor field.
sql_no_nulls :: Num a => a
sql_no_nulls = 0

sql_nullable :: Num a => a
sql_nullable = 1


-- Value returned by GetTypeInfo() to denote that it is
-- not known whether or not a data type supports null values
sql_nullable_unknown :: Num a => a
sql_nullable_unknown = 2


-- Values returned by GetTypeInfo() to show WHERE clause supported
sql_pred_none :: Num a => a
sql_pred_none = 0

sql_pred_char :: Num a => a
sql_pred_char = 1

sql_pred_basic :: Num a => a
sql_pred_basic = 2


-- values of UNNAMED field in descriptor
-- These codes should not be changed;
-- they were chosen for compatibility with the ISO SQL definition
-- of the UNNAMED descriptor field.
sql_named :: Num a => a
sql_named = 0

sql_unnamed :: Num a => a
sql_unnamed = 1


--values of ALLOC_TYPE field in descriptor
sql_desc_alloc_auto :: Num a => a
sql_desc_alloc_auto = 1

sql_desc_alloc_user :: Num a => a
sql_desc_alloc_user = 2


-- FreeStmt() options
sql_close :: Num a => a
sql_close = 0

sql_drop :: Num a => a
sql_drop = 1

sql_unbind :: Num a => a
sql_unbind = 2

sql_reset_params :: Num a => a
sql_reset_params = 3


-- Codes used for FetchOrientation in FetchScroll(), and in DataSources()
sql_fetch_next :: Num a => a
sql_fetch_next = 1

sql_fetch_first :: Num a => a
sql_fetch_first = 2


-- Other codes used for FetchOrientation in FetchScroll() */
sql_fetch_last :: Num a => a
sql_fetch_last = 3

sql_fetch_prior :: Num a => a
sql_fetch_prior = 4

sql_fetch_absolute :: Num a => a
sql_fetch_absolute = 5
sql_fetch_relative :: Num a => a
sql_fetch_relative = 6


-- EndTran() options
sql_commit :: Num a => a
sql_commit = 0

sql_rollback :: Num a => a
sql_rollback = 1


-- null handles returned by AllocHandle()
sql_null_henv :: SQLHANDLE
sql_null_henv = sql_null_handle

sql_null_hdbc :: SQLHANDLE
sql_null_hdbc = sql_null_handle

sql_null_hstmt :: SQLHANDLE
sql_null_hstmt = sql_null_handle

sql_null_hdesc :: SQLHANDLE
sql_null_hdesc = sql_null_handle


-- null handle used in place of parent handle when allocating HENV
sql_null_handle :: SQLHANDLE
#if ((defined WIN32) || (defined _WIN64))
sql_null_handle = nullPtr
#else
sql_null_handle = 0
#endif


-- Values that may appear in the result set of SpecialColumns()
sql_scope_currow :: Num a => a
sql_scope_currow = 0

sql_scope_transaction :: Num a => a
sql_scope_transaction = 1

sql_scope_session :: Num a => a
sql_scope_session = 2


sql_pc_unknown :: Num a => a
sql_pc_unknown = 0

sql_pc_non_pseudo :: Num a => a
sql_pc_non_pseudo = 1

sql_pc_pseudo :: Num a => a
sql_pc_pseudo = 2


-- Reserved value for the IdentifierType argument of SpecialColumns()
sql_row_identifier :: Num a => a
sql_row_identifier = 1


-- Reserved values for UNIQUE argument of Statistics()
sql_index_unique :: Num a => a
sql_index_unique = 0

sql_index_all :: Num a => a
sql_index_all = 1


-- Values that may appear in the result set of Statistics()
sql_index_clustered :: Num a => a
sql_index_clustered = 1

sql_index_hashed :: Num a => a
sql_index_hashed = 2

sql_index_other :: Num a => a
sql_index_other = 3


-- GetFunctions() values to identify CLI functions
sql_api_sqlallocconnect :: Num a => a
sql_api_sqlallocconnect = 1

sql_api_sqlallocenv :: Num a => a
sql_api_sqlallocenv = 2

sql_api_sqlallochandle :: Num a => a
sql_api_sqlallochandle = 1001

sql_api_sqlallocstmt :: Num a => a
sql_api_sqlallocstmt = 3

sql_api_sqlbindcol :: Num a => a
sql_api_sqlbindcol = 4

sql_api_sqlbindparam :: Num a => a
sql_api_sqlbindparam = 22

sql_api_sqlcancel :: Num a => a
sql_api_sqlcancel = 5

sql_api_sqlclosecursor :: Num a => a
sql_api_sqlclosecursor = 1003

sql_api_sqlcolattribute :: Num a => a
sql_api_sqlcolattribute = 6

sql_api_sqlcolumns :: Num a => a
sql_api_sqlcolumns = 40

sql_api_sqlconnect :: Num a => a
sql_api_sqlconnect = 7

sql_api_sqlcopydesc :: Num a => a
sql_api_sqlcopydesc = 1004

sql_api_sqldatasources :: Num a => a
sql_api_sqldatasources = 57

sql_api_sqldescribecol :: Num a => a
sql_api_sqldescribecol = 8

sql_api_sqldisconnect :: Num a => a
sql_api_sqldisconnect = 9

sql_api_sqlendtran :: Num a => a
sql_api_sqlendtran = 1005

sql_api_sqlerror :: Num a => a
sql_api_sqlerror = 10

sql_api_sqlexecdirect :: Num a => a
sql_api_sqlexecdirect = 11

sql_api_sqlexecute :: Num a => a
sql_api_sqlexecute = 12

sql_api_sqlfetch :: Num a => a
sql_api_sqlfetch = 13

sql_api_sqlfetchscroll :: Num a => a
sql_api_sqlfetchscroll = 1021

sql_api_sqlfreeconnect :: Num a => a
sql_api_sqlfreeconnect = 14

sql_api_sqlfreeenv :: Num a => a
sql_api_sqlfreeenv = 15

sql_api_sqlfreehandle :: Num a => a
sql_api_sqlfreehandle = 1006

sql_api_sqlfreestmt :: Num a => a
sql_api_sqlfreestmt = 16

sql_api_sqlgetconnectattr :: Num a => a
sql_api_sqlgetconnectattr = 1007

sql_api_sqlgetconnectoption :: Num a => a
sql_api_sqlgetconnectoption = 42

sql_api_sqlgetcursorname :: Num a => a
sql_api_sqlgetcursorname = 17

sql_api_sqlgetdata :: Num a => a
sql_api_sqlgetdata = 43

sql_api_sqlgetdescfield :: Num a => a
sql_api_sqlgetdescfield = 1008

sql_api_sqlgetdescrec :: Num a => a
sql_api_sqlgetdescrec = 1009

sql_api_sqlgetdiagfield :: Num a => a
sql_api_sqlgetdiagfield = 1010

sql_api_sqlgetdiagrec :: Num a => a
sql_api_sqlgetdiagrec = 1011

sql_api_sqlgetenvattr :: Num a => a
sql_api_sqlgetenvattr = 1012

sql_api_sqlgetfunctions :: Num a => a
sql_api_sqlgetfunctions = 44

sql_api_sqlgetinfo :: Num a => a
sql_api_sqlgetinfo = 45

sql_api_sqlgetstmtattr :: Num a => a
sql_api_sqlgetstmtattr = 1014

sql_api_sqlgetstmtoption :: Num a => a
sql_api_sqlgetstmtoption = 46

sql_api_sqlgettypeinfo :: Num a => a
sql_api_sqlgettypeinfo = 47

sql_api_sqllanguages :: Num a => a
sql_api_sqllanguages = 2001

sql_api_sqlnumresultcols :: Num a => a
sql_api_sqlnumresultcols = 18

sql_api_sqlparamdata :: Num a => a
sql_api_sqlparamdata = 48

sql_api_sqlprepare :: Num a => a
sql_api_sqlprepare = 19

sql_api_sqlputdata :: Num a => a
sql_api_sqlputdata = 49

sql_api_sqlreleaseenv :: Num a => a
sql_api_sqlreleaseenv = 1015

sql_api_sqlrowcount :: Num a => a
sql_api_sqlrowcount = 20

sql_api_sqlserverinfo :: Num a => a
sql_api_sqlserverinfo = 2002

sql_api_sqlsetconnectattr :: Num a => a
sql_api_sqlsetconnectattr = 1016

sql_api_sqlsetconnectoption :: Num a => a
sql_api_sqlsetconnectoption = 50

sql_api_sqlsetcursorname :: Num a => a
sql_api_sqlsetcursorname = 21

sql_api_sqlsetdescfield :: Num a => a
sql_api_sqlsetdescfield = 1017

sql_api_sqlsetdescrec :: Num a => a
sql_api_sqlsetdescrec = 1018

sql_api_sqlsetenvattr :: Num a => a
sql_api_sqlsetenvattr = 1019

sql_api_sqlsetparam :: Num a => a
sql_api_sqlsetparam = 22

sql_api_sqlsetstmtattr :: Num a => a
sql_api_sqlsetstmtattr = 1020

sql_api_sqlsetstmtoption :: Num a => a
sql_api_sqlsetstmtoption = 51

sql_api_sqlspecialcolumns :: Num a => a
sql_api_sqlspecialcolumns = 52

sql_api_sqlstatistics :: Num a => a
sql_api_sqlstatistics = 53

sql_api_sqltables :: Num a => a
sql_api_sqltables = 54

sql_api_sqltransact :: Num a => a
sql_api_sqltransact = 23


-- Information requested by GetInfo()
sql_max_driver_connections :: Num a => a
sql_max_driver_connections = 0

sql_max_concurrent_activities :: Num a => a
sql_max_concurrent_activities = 1

sql_data_source_name :: Num a => a
sql_data_source_name = 2

sql_fetch_direction :: Num a => a
sql_fetch_direction = 8

sql_server_name :: Num a => a
sql_server_name = 13

sql_search_pattern_escape :: Num a => a
sql_search_pattern_escape = 14

sql_dbms_name :: Num a => a
sql_dbms_name = 17

sql_dbms_ver :: Num a => a
sql_dbms_ver = 18

sql_accessible_tables :: Num a => a
sql_accessible_tables = 19

sql_cursor_commit_behavior :: Num a => a
sql_cursor_commit_behavior = 23

sql_data_source_read_only :: Num a => a
sql_data_source_read_only = 25

sql_default_txn_isolation :: Num a => a
sql_default_txn_isolation = 26

sql_identifier_case :: Num a => a
sql_identifier_case = 28

sql_identifier_quote_char :: Num a => a
sql_identifier_quote_char = 29

sql_max_column_name_len :: Num a => a
sql_max_column_name_len = 30

sql_max_cursor_name_len :: Num a => a
sql_max_cursor_name_len = 31

sql_max_schema_name_len :: Num a => a
sql_max_schema_name_len = 32

sql_max_catalog_name_len :: Num a => a
sql_max_catalog_name_len = 34

sql_max_table_name_len :: Num a => a
sql_max_table_name_len = 35

sql_scroll_concurrency :: Num a => a
sql_scroll_concurrency = 43

sql_txn_capable :: Num a => a
sql_txn_capable = 46

sql_user_name :: Num a => a
sql_user_name = 47

sql_txn_isolation_option :: Num a => a
sql_txn_isolation_option = 72

sql_integrity :: Num a => a
sql_integrity = 73

sql_getdata_extensions :: Num a => a
sql_getdata_extensions = 81

sql_null_collation :: Num a => a
sql_null_collation = 85

sql_alter_table :: Num a => a
sql_alter_table = 86

sql_order_by_columns_in_select :: Num a => a
sql_order_by_columns_in_select = 90

sql_special_characters :: Num a => a
sql_special_characters = 94

sql_max_columns_in_group_by :: Num a => a
sql_max_columns_in_group_by = 97

sql_max_columns_in_index :: Num a => a
sql_max_columns_in_index = 98

sql_max_columns_in_order_by :: Num a => a
sql_max_columns_in_order_by = 99

sql_max_columns_in_select :: Num a => a
sql_max_columns_in_select = 100

sql_max_columns_in_table :: Num a => a
sql_max_columns_in_table = 101

sql_max_index_size :: Num a => a
sql_max_index_size = 102

sql_max_row_size :: Num a => a
sql_max_row_size = 104

sql_max_statement_len :: Num a => a
sql_max_statement_len = 105

sql_max_tables_in_select :: Num a => a
sql_max_tables_in_select = 106

sql_max_user_name_len :: Num a => a
sql_max_user_name_len = 107

sql_oj_capabilities :: Num a => a
sql_oj_capabilities = 115

sql_xopen_cli_year :: Num a => a
sql_xopen_cli_year = 10000

sql_cursor_sensitivity :: Num a => a
sql_cursor_sensitivity = 10001

sql_describe_parameter :: Num a => a
sql_describe_parameter = 10002

sql_catalog_name :: Num a => a
sql_catalog_name = 10003

sql_collation_seq :: Num a => a
sql_collation_seq = 10004

sql_max_identifier_len :: Num a => a
sql_max_identifier_len = 10005


-- SQL_ALTER_TABLE bitmasks
sql_at_add_column :: CLong
sql_at_add_column = 0x01

sql_at_drop_column :: CLong
sql_at_drop_column = 0x02

sql_at_alter_column :: CLong
sql_at_alter_column = 0x04

sql_at_add_constraint :: CLong
sql_at_add_constraint = 0x08

sql_at_drop_constraint :: CLong
sql_at_drop_constraint = 0x10

-- SQL_CURSOR_COMMIT_BEHAVIOR values
sql_cb_delete :: Num a => a
sql_cb_delete = 0

sql_cb_close :: Num a => a
sql_cb_close = 1

sql_cb_preserve :: Num a => a
sql_cb_preserve = 2

-- SQL_FETCH_DIRECTION bitmasks
sql_fd_fetch_next :: CLong
sql_fd_fetch_next = 0x01

sql_fd_fetch_first :: CLong
sql_fd_fetch_first = 0x02

sql_fd_fetch_last :: CLong
sql_fd_fetch_last = 0x04

sql_fd_fetch_prior :: CLong
sql_fd_fetch_prior = 0x08

sql_fd_fetch_absolute :: CLong
sql_fd_fetch_absolute = 0x10

sql_fd_fetch_relative :: CLong
sql_fd_fetch_relative = 0x20

-- SQL_GETDATA_EXTENSIONS bitmasks
sql_gd_any_column :: CLong
sql_gd_any_column = 0x01

sql_gd_any_order :: CLong
sql_gd_any_order = 0x02

-- SQL_IDENTIFIER_CASE values
sql_ic_upper :: Num a => a
sql_ic_upper = 1

sql_ic_lower :: Num a => a
sql_ic_lower = 2

sql_ic_sensitive :: Num a => a
sql_ic_sensitive = 3

sql_ic_mixed :: Num a => a
sql_ic_mixed = 4


-- SQL_OJ_CAPABILITIES bitmasks
sql_oj_left :: CLong
sql_oj_left = 0x01

sql_oj_right :: CLong
sql_oj_right = 0x02

sql_oj_full :: CLong
sql_oj_full = 0x04

sql_oj_nested :: CLong
sql_oj_nested = 0x08

sql_oj_not_ordered :: CLong
sql_oj_not_ordered = 0x10

sql_oj_inner :: CLong
sql_oj_inner = 0x20

sql_oj_all_comparison_ops :: CLong
sql_oj_all_comparison_ops = 0x40


-- SQL_SCROLL_CONCURRENCY bitmasks
sql_scco_read_only :: CLong
sql_scco_read_only = 0x01

sql_scco_lock :: CLong
sql_scco_lock = 0x02

sql_scco_opt_rowver :: CLong
sql_scco_opt_rowver = 0x04

sql_scco_opt_values :: CLong
sql_scco_opt_values = 0x08

-- SQL_TXN_CAPABLE values
sql_tc_none :: Num a => a
sql_tc_none = 0

sql_tc_dml :: Num a => a
sql_tc_dml = 1

sql_tc_all :: Num a => a
sql_tc_all = 2

sql_tc_ddl_commit :: Num a => a
sql_tc_ddl_commit = 3

sql_tc_ddl_ignore :: Num a => a
sql_tc_ddl_ignore = 4

-- SQL_TXN_ISOLATION_OPTION bitmasks
sql_txn_read_uncommitted :: CLong
sql_txn_read_uncommitted = 0x01

sql_txn_read_committed :: CLong
sql_txn_read_committed = 0x02

sql_txn_repeatable_read :: CLong
sql_txn_repeatable_read = 0x04

sql_txn_serializable :: CLong
sql_txn_serializable = 0x08

foreign import ccall "SQLAllocConnect" sqlallocconnect :: SQLHENV -> Ptr SQLHDBC -> IO SQLRETURN
foreign import ccall "SQLAllocEnv" sqlallocenv :: Ptr SQLHENV -> IO SQLRETURN
foreign import ccall "SQLAllocHandle" sqlallochandle :: SQLSMALLINT -> SQLHANDLE -> Ptr SQLHANDLE -> IO SQLRETURN
foreign import ccall "SQLAllocStmt" sqlallocstmt :: SQLHDBC -> Ptr SQLHSTMT -> IO SQLRETURN
foreign import ccall "SQLBindCol" sqlbindcol :: SQLHSTMT -> SQLSMALLINT -> SQLSMALLINT -> SQLPOINTER -> SQLLEN -> Ptr SQLLEN -> IO SQLRETURN
foreign import ccall "SQLBindParam" sqlbindparam :: SQLHSTMT -> SQLSMALLINT -> SQLSMALLINT -> SQLSMALLINT -> SQLULEN -> SQLSMALLINT -> SQLPOINTER -> Ptr SQLLEN -> IO SQLRETURN
foreign import ccall "SQLCancel" sqlcancel :: SQLHSTMT -> IO SQLRETURN
foreign import ccall "SQLCloseCursor" sqlclosecursor :: SQLHSTMT -> IO SQLRETURN
#ifdef ODBC
foreign import ccall "SQLColAttribute" sqlcolattribute :: SQLHSTMT -> SQLSMALLINT -> SQLSMALLINT -> SQLPOINTER -> SQLSMALLINT -> Ptr SQLSMALLINT -> Ptr SQLLEN -> IO SQLRETURN
#else
foreign import ccall "SQLColAttribute" sqlcolattribute :: SQLHSTMT -> SQLSMALLINT -> SQLSMALLINT -> SQLPOINTER -> SQLSMALLINT -> Ptr SQLSMALLINT -> SQLPOINTER -> IO SQLRETURN
#endif
foreign import ccall "SQLColumns" sqlcolumns :: SQLHSTMT -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLCHAR -> SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLConnect" sqlconnect :: SQLHDBC -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLCHAR -> SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLCopyDesc" sqlopydesc :: SQLHDESC -> SQLHDESC -> IO SQLRETURN
foreign import ccall "SQLDataSources" sqldatasources :: SQLHENV -> SQLSMALLINT -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLSMALLINT -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLDescribeCol" sqldescribecol :: SQLHSTMT -> SQLSMALLINT -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLSMALLINT -> Ptr SQLSMALLINT -> Ptr SQLULEN -> Ptr SQLSMALLINT -> Ptr SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLDisconnect" sqldisconnect :: SQLHDBC -> IO SQLRETURN
foreign import ccall "SQLEndTran" sqlendtran :: SQLSMALLINT -> SQLHANDLE -> SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLError" sqlerror :: SQLHENV -> SQLHDBC -> SQLHSTMT -> Ptr SQLCHAR -> Ptr SQLINTEGER -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLExecDirect" sqlexecdirect :: SQLHSTMT -> Ptr SQLCHAR -> SQLINTEGER -> IO SQLRETURN
foreign import ccall "SQLExecute" sqlexecute :: SQLHSTMT -> IO SQLRETURN
foreign import ccall "SQLFetch" sqlfetch :: SQLHSTMT -> IO SQLRETURN
foreign import ccall "SQLFetchScroll" sqlfetchscroll :: SQLHSTMT -> SQLSMALLINT -> SQLLEN -> IO SQLRETURN
foreign import ccall "SQLFreeConnect" sqlfreeconnect :: SQLHDBC -> IO SQLRETURN
foreign import ccall "SQLFreeEnv" sqlfreeenv :: SQLHENV -> IO SQLRETURN
foreign import ccall "SQLFreeHandle" sqlfreehandle :: SQLSMALLINT -> SQLHANDLE -> IO SQLRETURN
foreign import ccall "SQLFreeStmt" sqlfreestmt :: SQLHSTMT -> SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLGetConnectAttr" sqlgetconnectattr :: SQLHDBC -> SQLINTEGER -> SQLPOINTER -> SQLINTEGER -> Ptr SQLINTEGER -> IO SQLRETURN
foreign import ccall "SQLGetConnectOption" sqlgetconnectoption :: SQLHDBC -> SQLSMALLINT -> SQLPOINTER -> IO SQLRETURN
foreign import ccall "SQLGetCursorName" sqlgetcursorname :: SQLHSTMT -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLGetData" sqlgetdata :: SQLHSTMT -> SQLSMALLINT -> SQLSMALLINT -> SQLPOINTER -> SQLLEN -> Ptr SQLLEN -> IO SQLRETURN
foreign import ccall "SQLGetDescField" sqlgetdescfield :: SQLHDESC -> SQLSMALLINT -> SQLSMALLINT -> SQLPOINTER -> SQLINTEGER -> Ptr SQLINTEGER -> IO SQLRETURN
foreign import ccall "SQLGetDescRec" sqlgetdescrec :: SQLHDESC -> SQLSMALLINT -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLSMALLINT -> Ptr SQLSMALLINT -> Ptr SQLSMALLINT -> Ptr SQLLEN -> Ptr SQLSMALLINT -> Ptr SQLSMALLINT -> Ptr SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLGetDiagField" sqlgetdiagfield :: SQLSMALLINT -> SQLHANDLE -> SQLSMALLINT -> SQLSMALLINT -> SQLPOINTER -> SQLSMALLINT -> Ptr SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLGetDiagRec" sqlgetdiagrec :: SQLSMALLINT -> SQLHANDLE -> SQLSMALLINT -> Ptr SQLCHAR -> Ptr SQLINTEGER -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLGetEnvAttr" sqlgetenvattr :: SQLHENV -> SQLINTEGER -> SQLPOINTER -> SQLINTEGER -> Ptr SQLINTEGER -> IO SQLRETURN
foreign import ccall "SQLGetFunctions" sqlgetfunctions :: SQLHDBC -> SQLSMALLINT -> Ptr SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLGetInfo" sqlgetinfo :: SQLHDBC -> SQLSMALLINT -> SQLPOINTER -> SQLSMALLINT -> Ptr SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLGetStmtAttr" sqlgetstmtattr :: SQLHSTMT -> SQLINTEGER -> SQLPOINTER -> SQLINTEGER -> Ptr SQLINTEGER -> IO SQLRETURN
foreign import ccall "SQLGetStmtOption" sqlgetstmtoption :: SQLHSTMT -> SQLSMALLINT -> SQLPOINTER -> IO SQLRETURN
foreign import ccall "SQLGetTypeInfo" sqlgettypeinfo :: SQLHSTMT -> SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLNumResultCols" sqlnumresultcols :: SQLHSTMT -> Ptr SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLParamData" sqlparamdata :: SQLHSTMT -> Ptr SQLPOINTER -> IO SQLRETURN
foreign import ccall "SQLPrepare" sqlprepare :: SQLHSTMT -> Ptr SQLCHAR -> SQLINTEGER -> IO SQLRETURN
foreign import ccall "SQLPutData" sqlputdata :: SQLHSTMT -> SQLPOINTER -> SQLLEN -> IO SQLRETURN
foreign import ccall "SQLRowCount" sqlrowcount :: SQLHSTMT -> Ptr SQLLEN -> IO SQLRETURN
foreign import ccall "SQLSetConnectAttr" sqlsetconnectattr :: SQLHDBC -> SQLINTEGER -> SQLPOINTER -> SQLINTEGER -> IO SQLRETURN
foreign import ccall "SQLSetConnectOption" sqlsetconnectoption :: SQLHDBC -> SQLSMALLINT -> SQLPOINTER -> IO SQLRETURN
foreign import ccall "SQLSetCursorName" sqlsetcursorname :: SQLHSTMT -> Ptr SQLCHAR -> SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLSetDescField" sqlsetdescfield :: SQLHDESC -> SQLSMALLINT -> SQLSMALLINT -> SQLPOINTER -> SQLINTEGER -> IO SQLRETURN
foreign import ccall "SQLSetDescRec" sqlsetdescrec :: SQLHDESC -> SQLSMALLINT -> SQLSMALLINT -> SQLSMALLINT -> SQLINTEGER -> SQLSMALLINT -> SQLSMALLINT -> SQLPOINTER -> Ptr SQLLEN -> Ptr SQLLEN -> IO SQLRETURN
foreign import ccall "SQLSetEnvAttr" sqlsetenvattr :: SQLHENV -> SQLINTEGER -> SQLPOINTER -> SQLINTEGER -> IO SQLRETURN

-- sqlsetparam is another name for sqlbindparam
foreign import ccall "SQLBindParam" sqlsetparam :: SQLHSTMT -> SQLSMALLINT -> SQLSMALLINT -> SQLSMALLINT -> SQLULEN -> SQLSMALLINT -> SQLPOINTER -> Ptr SQLLEN -> IO SQLRETURN

foreign import ccall "SQLSetStmtAttr" sqlsetstmtattr :: SQLHSTMT -> SQLINTEGER -> SQLPOINTER -> SQLULEN -> IO SQLRETURN
foreign import ccall "SQLSetStmtOption" sqlsetstmtoption :: SQLHSTMT -> SQLSMALLINT -> SQLPOINTER -> IO SQLRETURN
foreign import ccall "SQLSpecialColumns" sqlspecialcolumns :: SQLHSTMT -> SQLSMALLINT -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLCHAR -> SQLSMALLINT -> SQLSMALLINT -> SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLStatistics" sqlstatistics :: SQLHSTMT -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLCHAR -> SQLSMALLINT -> SQLSMALLINT -> SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLTables" sqltables :: SQLHSTMT -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLCHAR -> SQLSMALLINT -> Ptr SQLCHAR -> SQLSMALLINT -> IO SQLRETURN
foreign import ccall "SQLTransact" sqltransact :: SQLHENV -> SQLHDBC -> SQLSMALLINT -> IO SQLRETURN
