{- |
Module                  : Summoner.Default
Copyright               : (c) 2017-2026 Kowainik
SPDX-License-Identifier : MPL-2.0
Maintainer              : Kowainik <xrom.xkov@gmail.com>
Stability               : Stable
Portability             : Portable

This module contains some default values to use.
-}

module Summoner.Default
       ( defaultGHC
       , defaultCabal
       , defaultCabalVersion
       , defaultStack
       , defaultLicenseName
       , defaultOwner
       , defaultFullName
       , defaultEmail
       , defaultTomlFile
       , defaultConfigFile
       , defaultConfigFileContent
       , defaultDescription
       , currentYear

         -- * GitHub Actions defaults
       , ghcActionsCacheVersion
       , ghcActionsCheckoutVersion
       , ghcActionsSetupHaskellVersion
       ) where

import Relude.Extra.Enum (prev)

import Data.Time (getCurrentTime, getCurrentTimeZone, localDay, toGregorian, utcToLocalTime)
import System.Directory (getHomeDirectory)
import System.FilePath ((</>))

import Summoner.GhcVer (GhcVer, showGhcVer)
import Summoner.License (LicenseName (MIT))
import Summoner.Text (quote)


-- | Default GHC version is the latest available.
defaultGHC :: GhcVer
defaultGHC = maxBound

-- | Default version of the Cabal.
defaultCabal :: Text
defaultCabal = "3.14.2.0"

-- | Default version of the Cabal specified in the *.cabal* file.
defaultCabalVersion :: Text
defaultCabalVersion = "3.0"

-- | Default version of the Stack.
defaultStack :: Text
defaultStack = "3.3.1"

defaultLicenseName :: LicenseName
defaultLicenseName = MIT

defaultOwner :: Text
defaultOwner = "kowainik"

defaultFullName :: Text
defaultFullName = "Kowainik"

defaultEmail :: Text
defaultEmail = "xrom.xkov@gmail.com"

defaultTomlFile :: FilePath
defaultTomlFile = ".summoner.toml"

defaultConfigFile :: IO FilePath
defaultConfigFile = (</> defaultTomlFile) <$> getHomeDirectory

defaultDescription :: Text
defaultDescription = "See README for more info"

ghcActionsCheckoutVersion :: Text
ghcActionsCheckoutVersion = "@v4"

ghcActionsSetupHaskellVersion :: Text
ghcActionsSetupHaskellVersion = "@v2"

ghcActionsCacheVersion :: Text
ghcActionsCacheVersion = "@v4"

currentYear :: IO Text
currentYear = do
    now <- getCurrentTime
    zone <- getCurrentTimeZone
    let localNow = utcToLocalTime zone now
    let (year, _, _) = toGregorian $ localDay localNow
    pure $ show year

{- | Default content of the @~/summoner.toml@ file to be generated by
the @summon config@ command.
-}
defaultConfigFileContent :: Text
defaultConfigFileContent = unlines
    [ "# This file is automatically generated by Summoner."
    , "# Edit required fields, uncomment additional settings or delete irrelevant"
    , "# lines for the personalized configuration."
    , ""
    , "# GitHub user name"
    , "owner = " <> quote defaultOwner
    , ""
    , "# First and last name"
    , "fullName = " <> quote defaultFullName
    , ""
    , "# Email used at GitHub"
    , "email = " <> quote defaultEmail
    , ""
    , "# Default OSS license for your projects."
    , "# Run the 'summon show license' command to see the list of supported licenses."
    , "license = " <> quote licenseName
    , ""
    , "# Create projects with the Cabal support"
    , "# cabal = true"
    , ""
    , "# Create projects with the Stack support"
    , "# stack = true"
    , ""
    , "# Should your projects have the library stanza by default?"
    , "# lib = true"
    , ""
    , "# Should your projects have an executable stanza by default?"
    , "# exe = true"
    , ""
    , "# Should your projects have the test-suite stanza by default?"
    , "# test = true"
    , ""
    , "# Should your projects have the benchmark stanza by default?"
    , "# bench = true"
    , ""
    , "# Summoner suports 'git' version control system and GitHub as a platform for"
    , "# hosting 'git' repos. The following set of flags controls integration with GitHub."
    , "# github = true    # enabled GitHub integration"
    , "# noUpload = true  # Init git repo, but don't create it on GitHub"
    , "# private = true  # create private repos by default"
    , "# gitignore =      # list of additional entries to be added in .gitignore"
    , "#     [ " <> quote "build"
    , "#     , " <> quote "result"
    , "#     ]"
    , ""
    , "# Summoner supports various CI services. Uncomment those you want to use by default."
    , "# githubActions = true  # GitHub Actions CI"
    , "# travis = true         # Travis CI"
    , "# appveyor = true       # AppVeyor CI"
    , ""
    , "# List of additional GHC versions to support besides " <> defaultGhcVer <> "."
    , "# Run the 'summon show ghc' command to see the list of all supported GHC versions."
    , "# ghcVersions = [" <> quote defaultGhcPrevPrev <> ", " <> quote defaultGhcPrev <> "]"
    , ""
    , "# List of default-extensions in the .cabal file"
    , "# extensions = [ " <> quote "ConstraintKinds"
    , "#              , " <> quote "DeriveGeneric"
    , "#              , " <> quote "GeneralizedNewtypeDeriving"
    , "#              , " <> quote "InstanceSigs"
    , "#              , " <> quote "KindSignatures"
    , "#              , " <> quote "LambdaCase"
    , "#              , " <> quote "OverloadedStrings"
    , "#              , " <> quote "RecordWildCards"
    , "#              , " <> quote "ScopedTypeVariables"
    , "#              , " <> quote "StandaloneDeriving"
    , "#              , " <> quote "TupleSections"
    , "#              , " <> quote "TypeApplications"
    , "#              , " <> quote "ViewPatterns"
    , "#              ]"
    , ""
    , "# List of additional files to add after creating the project"
    , "# files ="
    , "#     [ { path = " <> quote ".stylish-haskell.yaml"
    , "#       , url  = " <> quote "https://raw.githubusercontent.com/kowainik/org/main/.stylish-haskell.yaml"
    , "#       }"
    , "#     , { path = " <> quote ".github/CODEOWNERS"
    , "#       , raw  = " <> quote ("*  @" <> defaultOwner)
    , "#       }"
    , "#     , { path  = " <> quote "src/Foo.hs"
    , "#       , local = " <> quote "/home/user/.default/Foo.hs"
    , "#       }"
    , "#     ]"
    , ""
    , "# Alternative Prelude to be used instead of the default one if you prefer"
    , "# [prelude]"
    , "#     package = " <> quote "relude"
    , "#     module  = " <> quote "Relude"
    ]
  where
    licenseName :: Text
    licenseName = show defaultLicenseName

    defaultGhcVer, defaultGhcPrev, defaultGhcPrevPrev :: Text
    defaultGhcVer = showGhcVer defaultGHC
    defaultGhcPrev = showGhcVer $ prev defaultGHC
    defaultGhcPrevPrev = showGhcVer $ prev $ prev defaultGHC
