-- File created: 2008-11-12 14:16:48

{-# LANGUAGE CPP, MultiParamTypeClasses, FlexibleInstances
           , FlexibleContexts, UndecidableInstances, KindSignatures #-}

#include "exports.h"

-- | The base implementation of a Patricia trie representing a map with list
-- keys, generalized over any type of map from element values to tries.
--
-- Worst-case complexities are given in terms of @n@, @m@, and @s@. @n@ refers
-- to the number of keys in the map and @m@ to their maximum length. @s@ refers
-- to the length of a key given to the function, not any property of the map.
--
-- In addition, the trie's branching factor plays a part in almost every
-- operation, but the complexity depends on the underlying 'Map'. Thus, for
-- instance, 'member' is actually @O(m f(b))@ where @f(b)@ is the complexity of
-- a lookup operation on the 'Map' used. This complexity depends on the
-- underlying operation, which is not part of the specification of the visible
-- function. Thus it could change whilst affecting the complexity only for
-- certain Map types: hence this \"b factor\" is not shown explicitly.
--
-- Disclaimer: the complexities have not been proven.
--
-- Strict versions of functions are provided for those who want to be certain
-- that their 'TrieMap' doesn't contain values consisting of unevaluated
-- thunks. Note, however, that they do not evaluate the whole trie strictly,
-- only the values. And only to one level of depth: for instance, 'alter'' does
-- not 'seq' the value within the 'Maybe', only the 'Maybe' itself. The user
-- should add the strictness in such cases himself, if he so wishes.
--
-- Many functions come in both ordinary and @WithKey@ forms, where the former
-- takes a function of type @a -> b@ and the latter of type @v k -> a -> b@,
-- where @v k@ is the key associated with the value @a@. For most of these
-- functions, there is additional overhead involved in keeping track of the
-- key: don't use the latter form of the function unless you need it.
module Data.ListTrie.Patricia.Vector.Map (MAP_EXPORTS) where

import Control.Applicative ((<*>),(<$>))
import Control.Arrow       ((***), second)
import Control.Monad       (liftM3)
import Data.Binary         (Binary,get,put)
import Data.Either         (partitionEithers)
import Data.Function       (on)
import qualified Data.Foldable as F
import qualified Data.List.NonEmpty as NE
import qualified Data.Maybe as Maybe
import Data.Monoid         (Monoid(..))
import Data.Semigroup      (Semigroup(..), stimesIdempotent)
import Data.Traversable    (Traversable(traverse))
import qualified Data.Vector.Generic as VG
import Prelude hiding      (filter, foldl, foldr, lookup, map, null)
import qualified Prelude

#if __GLASGOW_HASKELL__
import Text.Read (readPrec, lexP, parens, prec, Lexeme(Ident))
#endif

import qualified Data.ListTrie.Base.Map      as Map
import qualified Data.ListTrie.Patricia.Vector.Base as Base
import Data.ListTrie.Base.Classes (fmap')
import Data.ListTrie.Base.Map     (Map, OrdMap)

#include "docs.h"

-- Invariant: any (Tr Nothing _ _) has at least two children, all of which are
-- Just or have a Just descendant.
--
-- In order to avoid a lot of special casing it has to be the case that there's
-- only one way to represent a given trie. The above property makes sure of
-- that, so that, for instance, 'fromList [("foo",1)]' can only be 'Tr (Just 1)
-- "foo" Map.empty', and not 'Tr Nothing "fo" (Map.fromList [('o',Tr (Just 1)
-- "" Map.empty)])'. Base.tryCompress is a function which takes care of this.
--
-- | The data structure itself: a map from keys of type @v k@ to values of type
-- @v@ implemented as a trie, using @map@ to map keys of type @k@ to sub-tries.
--
-- Regarding the instances:
--
-- - The @Trie@ class is internal, ignore it.
--
-- - The 'Eq' constraint for the 'Ord' instance is misleading: it is needed
--   only because 'Eq' is a superclass of 'Ord'.
--
-- - The 'Foldable' and 'Traversable' instances allow folding over and
--   traversing only the values, not the keys.
--
-- - The 'Monoid' instance defines 'mappend' as 'union' and 'mempty' as
--   'empty'.
data TrieMap map v k a = Tr (Maybe a) !(v k) !(CMap map v k a)

type CMap map (v :: * -> *) k a = map k (TrieMap map v k a)

instance (VG.Vector v k, Monoid (v k), Map map k) => Base.Trie TrieMap Maybe map v k where
   mkTrie = Tr
   tParts (Tr v p m) = (v,p,m)

-- Don't use CMap in these instances since Haddock won't expand it
instance (VG.Vector v k, Monoid (v k), Map map k, Eq (map k (TrieMap map v k a)), Eq a)
      => Eq (TrieMap map v k a)
 where
   Tr v1 p1 m1 == Tr v2 p2 m2 =
      v1 == v2 && Base.eqComparePrefixes (Map.eqCmp m1) p1 p2
               && m1 == m2

-- Eq constraint only needed because of superclassness... sigh
instance (VG.Vector v k, Monoid (v k), Ord (v k), Eq (map k (TrieMap map v k a)), OrdMap map k, Ord k, Ord a)
      => Ord (TrieMap map v k a)
 where
   compare = compare `on` toAscList

instance (VG.Vector v k, Monoid (v k), Map map k) => Semigroup (TrieMap map v k a) where
   (<>) = union
   sconcat = unions . NE.toList
   stimes = stimesIdempotent

instance (VG.Vector v k, Monoid (v k), Map map k) => Monoid (TrieMap map v k a) where
   mempty  = empty
   mappend = (<>)
   mconcat = unions

instance (VG.Vector v k, Monoid (v k), Map map k) => Functor (TrieMap map v k) where
   fmap = map

instance (VG.Vector v k, Monoid (v k), Map map k) => F.Foldable (TrieMap map v k) where
   foldl = foldl . flip
   foldr = foldr

instance (VG.Vector v k, Monoid (v k), Map map k, Traversable (map k)) => Traversable (TrieMap map v k) where
   traverse f (Tr v p m) =
      flip Tr p <$> traverse f v <*> traverse (traverse f) m

instance (VG.Vector v k, Monoid (v k), Map map k, Show (v k), Show a) => Show (TrieMap map v k a) where
   showsPrec p s = showParen (p > 10) $
      showString "fromList " . shows (toList s)

instance (VG.Vector v k, Monoid (v k), Map map k, Read (v k), Read a) => Read (TrieMap map v k a) where
#if __GLASGOW_HASKELL__
   readPrec = parens $ prec 10 $ do
      Ident "fromList" <- lexP
      fmap fromList readPrec
#else
   readsPrec p = readParen (p > 10) $ \r -> do
      ("fromList", list) <- lex r
      (xs, rest) <- readsPrec (p+1) list
      [(fromList xs, rest)]
#endif

instance (Map map k, Binary k, Binary (v k), Binary a) => Binary (TrieMap map v k a) where
   put (Tr v k m) = put v >> put k >> (put . Map.serializeToList $ m)
   get = liftM3 Tr get get (get >>= return . Map.deserializeFromList)

-- * Construction

-- | @O(1)@. The empty map.
empty :: (VG.Vector v k, Monoid (v k), Map map k) => TrieMap map v k a
empty = Base.empty

-- | @O(1)@. The singleton map containing only the given key-value pair.
singleton :: (VG.Vector v k, Monoid (v k), Map map k) => v k -> a -> TrieMap map v k a
singleton = Base.singleton

-- * Modification

-- | @O(min(m,s))@. Inserts the key-value pair into the map. If the key is
-- already a member of the map, the given value replaces the old one.
--
-- > insert = insertWith const
insert :: (VG.Vector v k, Monoid (v k), Map map k) => v k -> a -> TrieMap map v k a -> TrieMap map v k a
insert = Base.insert

-- | @O(min(m,s))@. Like 'insert', but the new value is reduced to weak head
-- normal form before being placed into the map.
--
-- > insert' = insertWith' const
insert' :: (VG.Vector v k, Monoid (v k), Map map k) => v k -> a -> TrieMap map v k a -> TrieMap map v k a
insert' = Base.insert'

-- | @O(min(m,s))@. Inserts the key-value pair into the map. If the key is
-- already a member of the map, the old value is replaced by @f givenValue
-- oldValue@ where @f@ is the given function.
insertWith :: (VG.Vector v k, Monoid (v k), Map map k)
           => (a -> a -> a) -> v k -> a -> TrieMap map v k a -> TrieMap map v k a
insertWith = Base.insertWith

-- | @O(min(m,s))@. Like 'insertWith', but the new value is reduced to weak
-- head normal form before being placed into the map, whether it is the given
-- value or a result of the combining function.
insertWith' :: (VG.Vector v k, Monoid (v k), Map map k)
            => (a -> a -> a) -> v k -> a -> TrieMap map v k a -> TrieMap map v k a
insertWith' = Base.insertWith'

-- | @O(min(m,s))@. Removes the key from the map along with its associated
-- value. If the key is not a member of the map, the map is unchanged.
delete :: (VG.Vector v k, Monoid (v k), Map map k) => v k -> TrieMap map v k a -> TrieMap map v k a
delete = Base.delete

-- | @O(min(m,s))@. Adjusts the value at the given key by calling the given
-- function on it. If the key is not a member of the map, the map is unchanged.
adjust :: (VG.Vector v k, Monoid (v k), Map map k) => (a -> a) -> v k -> TrieMap map v k a -> TrieMap map v k a
adjust = Base.adjust

-- | @O(min(m,s))@. Like 'adjust', but the function is applied strictly.
adjust' :: (VG.Vector v k, Monoid (v k), Map map k) => (a -> a) -> v k -> TrieMap map v k a -> TrieMap map v k a
adjust' = Base.adjust'

-- | @O(min(m,s))@. Updates the value at the given key: if the given
-- function returns 'Nothing', the value and its associated key are removed; if
-- 'Just'@ a@ is returned, the old value is replaced with @a@. If the key is
-- not a member of the map, the map is unchanged.
update :: (VG.Vector v k, Monoid (v k), Map map k)
       => (a -> Maybe a) -> v k -> TrieMap map v k a -> TrieMap map v k a
update f k = snd . updateLookup f k

-- | @O(min(m,s))@. Like 'update', but also returns 'Just' the original value,
-- or 'Nothing' if the key is not a member of the map.
updateLookup :: (VG.Vector v k, Monoid (v k), Map map k)
             => (a -> Maybe a)
             -> v k
             -> TrieMap map v k a
             -> (Maybe a, TrieMap map v k a)
updateLookup = Base.updateLookup

-- | @O(min(m,s))@. The most general modification function, allowing you to
-- modify the value at the given key, whether or not it is a member of the map.
-- In short: the given function is passed 'Just' the value at the key if it is
-- present, or 'Nothing' otherwise; if the function returns 'Just' a value, the
-- new value is inserted into the map, otherwise the old value is removed. More
-- precisely, for @alter f k m@:
--
-- If @k@ is a member of @m@, @f (@'Just'@ oldValue)@ is called. Now:
--
-- - If @f@ returned 'Just'@ newValue@, @oldValue@ is replaced with @newValue@.
--
-- - If @f@ returned 'Nothing', @k@ and @oldValue@ are removed from the map.
--
-- If, instead, @k@ is not a member of @m@, @f @'Nothing' is called, and:
--
-- - If @f@ returned 'Just'@ value@, @value@ is inserted into the map, at @k@.
--
-- - If @f@ returned 'Nothing', the map is unchanged.
--
-- The function is applied lazily only if the given key is a prefix of more
-- than one key in the map.
alter :: (VG.Vector v k, Monoid (v k), Map map k)
      => (Maybe a -> Maybe a) -> v k -> TrieMap map v k a -> TrieMap map v k a
alter = Base.alter

-- | @O(min(m,s))@. Like 'alter', but the function is always applied strictly.
alter' :: (VG.Vector v k, Monoid (v k), Map map k)
       => (Maybe a -> Maybe a) -> v k -> TrieMap map v k a -> TrieMap map v k a
alter' = Base.alter'

-- * Querying

-- | @O(1)@. 'True' iff the map is empty.
null :: (VG.Vector v k, Monoid (v k), Map map k) => TrieMap map v k a -> Bool
null = Base.null

-- | @O(n m)@. The number of elements in the map. The value is built up lazily,
-- allowing for delivery of partial results without traversing the whole map.
size :: (VG.Vector v k, Monoid (v k), Map map k, Num n) => TrieMap map v k a -> n
size = Base.size

-- | @O(n m)@. The number of elements in the map. The value is built strictly:
-- no value is returned until the map has been fully traversed.
size' :: (VG.Vector v k, Monoid (v k), Map map k, Num n) => TrieMap map v k a -> n
size' = Base.size'

-- | @O(min(m,s))@. 'True' iff the given key is associated with a value in the
-- map.
member :: (VG.Vector v k, Monoid (v k), Map map k) => v k -> TrieMap map v k a -> Bool
member = Base.member

-- | @O(min(m,s))@. 'False' iff the given key is associated with a value in the
-- map.
notMember :: (VG.Vector v k, Monoid (v k), Map map k) => v k -> TrieMap map v k a -> Bool
notMember = Base.notMember

-- | @O(min(m,s))@. 'Just' the value in the map associated with the given key,
-- or 'Nothing' if the key is not a member of the map.
lookup :: (VG.Vector v k, Monoid (v k), Map map k) => v k -> TrieMap map v k a -> Maybe a
lookup = Base.lookup

-- | @O(min(m,s))@. Like 'lookup', but returns the given value when the key is
-- not a member of the map.
lookupWithDefault :: (VG.Vector v k, Monoid (v k), Map map k) => a -> v k -> TrieMap map v k a -> a
lookupWithDefault = Base.lookupWithDefault

-- | @O(min(n1 m1,n2 m2))@. 'True' iff the first map is a submap of the second,
-- i.e. all keys that are members of the first map are also members of the
-- second map, and their associated values are the same.
--
-- > isSubmapOf = isSubmapOfBy (==)
isSubmapOf :: (VG.Vector v k, Monoid (v k), Map map k, Eq a) => TrieMap map v k a -> TrieMap map v k a -> Bool
isSubmapOf = isSubmapOfBy (==)

-- | @O(min(n1 m1,n2 m2))@. Like 'isSubmapOf', but one can specify the equality
-- relation applied to the values.
--
-- 'True' iff all keys that are members of the first map are also members of
-- the second map, and the given function @f@ returns 'True' for all @f
-- firstMapValue secondMapValue@ where @firstMapValue@ and @secondMapValue@ are
-- associated with the same key.
isSubmapOfBy :: (VG.Vector v k, Monoid (v k), Map map k)
             => (a -> b -> Bool) -> TrieMap map v k a -> TrieMap map v k b -> Bool
isSubmapOfBy = Base.isSubmapOfBy

-- | @O(min(n1 m1,n2 m2))@. 'True' iff the first map is a proper submap of the
-- second, i.e. all keys that are members of the first map are also members of
-- the second map, and their associated values are the same, but the maps are
-- not equal. That is, at least one key was a member of the second map but not
-- the first.
--
-- > isProperSubmapOf = isProperSubmapOfBy (==)
isProperSubmapOf :: (VG.Vector v k, Monoid (v k), Map map k, Eq a)
                 => TrieMap map v k a -> TrieMap map v k a -> Bool
isProperSubmapOf = isProperSubmapOfBy (==)

-- | @O(min(n1 m1,n2 m2))@. Like 'isProperSubmapOf', but one can specify the
-- equality relation applied to the values.
--
-- 'True' iff all keys that are members of the first map are also members of
-- the second map, and the given function @f@ returns 'True' for all @f
-- firstMapValue secondMapValue@ where @firstMapValue@ and @secondMapValue@ are
-- associated with the same key, and at least one key in the second map is not
-- a member of the first.
isProperSubmapOfBy :: (VG.Vector v k, Monoid (v k), Map map k) => (a -> b -> Bool)
                                -> TrieMap map v k a
                                -> TrieMap map v k b
                                -> Bool
isProperSubmapOfBy = Base.isProperSubmapOfBy

-- * Combination

defaultUnion :: a -> a -> a
defaultUnion = const

-- | @O(min(n1 m1,n2 m2))@. The union of the two maps: the map which contains
-- all keys that are members of either map. This union is left-biased: if a key
-- is a member of both maps, the value from the first map is chosen.
--
-- The worst-case performance occurs when the two maps are identical.
--
-- > union = unionWith const
union :: (VG.Vector v k, Monoid (v k), Map map k) => TrieMap map v k a -> TrieMap map v k a -> TrieMap map v k a
union = unionWith defaultUnion

-- | @O(min(n1 m1,n2 m2))@. Like 'union', but the combining function ('const')
-- is applied strictly.
--
-- > union' = unionWith' const
union' :: (VG.Vector v k, Monoid (v k), Map map k) => TrieMap map v k a -> TrieMap map v k a -> TrieMap map v k a
union' = unionWith' defaultUnion

-- | @O(min(n1 m1,n2 m2))@. Like 'union', but the given function is used to
-- determine the new value if a key is a member of both given maps. For a
-- function @f@, the new value is @f firstMapValue secondMapValue@.
unionWith :: (VG.Vector v k, Monoid (v k), Map map k) => (a -> a -> a)
                       -> TrieMap map v k a
                       -> TrieMap map v k a
                       -> TrieMap map v k a
unionWith = Base.unionWith

-- | @O(min(n1 m1,n2 m2))@. Like 'unionWith', but the combining function is
-- applied strictly.
unionWith' :: (VG.Vector v k, Monoid (v k), Map map k) => (a -> a -> a)
                        -> TrieMap map v k a
                        -> TrieMap map v k a
                        -> TrieMap map v k a
unionWith' = Base.unionWith'

-- | @O(min(n1 m1,n2 m2))@. Like 'unionWith', but in addition to the two
-- values, the key is passed to the combining function.
unionWithKey :: (VG.Vector v k, Monoid (v k), Map map k) => (v k -> a -> a -> a)
                          -> TrieMap map v k a
                          -> TrieMap map v k a
                          -> TrieMap map v k a
unionWithKey = Base.unionWithKey

-- | @O(min(n1 m1,n2 m2))@. Like 'unionWithKey', but the combining function is
-- applied strictly.
unionWithKey' :: (VG.Vector v k, Monoid (v k), Map map k) => (v k -> a -> a -> a)
                           -> TrieMap map v k a
                           -> TrieMap map v k a
                           -> TrieMap map v k a
unionWithKey' = Base.unionWithKey'

-- | @O(sum(n))@. The union of all the maps: the map which contains all keys
-- that are members of any of the maps. If a key is a member of multiple maps,
-- the value that occurs in the earliest of the maps (according to the order of
-- the given list) is chosen.
--
-- The worst-case performance occurs when all the maps are identical.
--
-- > unions = unionsWith const
unions :: (VG.Vector v k, Monoid (v k), Map map k) => [TrieMap map v k a] -> TrieMap map v k a
unions = unionsWith defaultUnion

-- | @O(sum(n))@. Like 'unions', but the combining function ('const') is
-- applied strictly.
--
-- > unions' = unionsWith' const
unions' :: (VG.Vector v k, Monoid (v k), Map map k) => [TrieMap map v k a] -> TrieMap map v k a
unions' = unionsWith' defaultUnion

-- | @O(sum(n))@. Like 'unions', but the given function determines the final
-- value if a key is a member of more than one map. The function is applied as
-- a left fold over the values in the given list's order. For example:
--
-- > unionsWith (-) [fromList [("a",1)],fromList [("a",2)],fromList [("a",3)]]
-- >    == fromList [("a",(1-2)-3)]
-- >    == fromList [("a",-4)]
unionsWith :: (VG.Vector v k, Monoid (v k), Map map k)
           => (a -> a -> a) -> [TrieMap map v k a] ->  TrieMap map v k a
unionsWith = Base.unionsWith

-- | @O(sum(n))@. Like 'unionsWith', but the combining function is applied
-- strictly.
unionsWith' :: (VG.Vector v k, Monoid (v k), Map map k)
            => (a -> a -> a) -> [TrieMap map v k a] ->  TrieMap map v k a
unionsWith' = Base.unionsWith'

-- | @O(sum(n))@. Like 'unionsWith', but in addition to the two values under
-- consideration, the key is passed to the combining function.
unionsWithKey :: (VG.Vector v k, Monoid (v k), Map map k)
              => (v k -> a -> a -> a) -> [TrieMap map v k a] ->  TrieMap map v k a
unionsWithKey = Base.unionsWithKey

-- | @O(sum(n))@. Like 'unionsWithKey', but the combining function is applied
-- strictly.
unionsWithKey' :: (VG.Vector v k, Monoid (v k), Map map k)
               => (v k -> a -> a -> a) -> [TrieMap map v k a] ->  TrieMap map v k a
unionsWithKey' = Base.unionsWithKey'

-- | @O(min(n1 m1,n2 m2))@. The difference of the two maps: the map which
-- contains all keys that are members of the first map and not of the second.
--
-- The worst-case performance occurs when the two maps are identical.
--
-- > difference = differenceWith (\_ _ -> Nothing)
difference :: (VG.Vector v k, Monoid (v k), Map map k)
           => TrieMap map v k a -> TrieMap map v k b -> TrieMap map v k a
difference = differenceWith (\_ _ -> Nothing)

-- | @O(min(n1 m1,n2 m2))@. Like 'difference', but the given function
-- determines what to do when a key is a member of both maps. If the function
-- returns 'Nothing', the key is removed; if it returns 'Just' a new value,
-- that value replaces the old one in the first map.
differenceWith :: (VG.Vector v k, Monoid (v k), Map map k) => (a -> b -> Maybe a)
                            -> TrieMap map v k a
                            -> TrieMap map v k b
                            -> TrieMap map v k a
differenceWith = Base.differenceWith

-- | @O(min(n1 m1,n2 m2))@. Like 'differenceWith', but in addition to the two
-- values, the key they are associated with is passed to the combining
-- function.
differenceWithKey :: (VG.Vector v k, Monoid (v k), Map map k) => (v k -> a -> b -> Maybe a)
                               -> TrieMap map v k a
                               -> TrieMap map v k b
                               -> TrieMap map v k a
differenceWithKey = Base.differenceWithKey

-- | @O(min(n1 m1,n2 m2))@. The intersection of the two maps: the map which
-- contains all keys that are members of both maps.
--
-- The worst-case performance occurs when the two maps are identical.
--
-- > intersection = intersectionWith const
intersection :: (VG.Vector v k, Monoid (v k), Map map k)
             => TrieMap map v k a -> TrieMap map v k b -> TrieMap map v k a
intersection = intersectionWith const

-- | @O(min(n1 m1,n2 m2))@. Like 'intersection', but the combining function is
-- applied strictly.
--
-- > intersection' = intersectionWith' const
intersection' :: (VG.Vector v k, Monoid (v k), Map map k)
              => TrieMap map v k a -> TrieMap map v k b -> TrieMap map v k a
intersection' = intersectionWith' const

-- | @O(min(n1 m1,n2 m2))@. Like 'intersection', but the given function
-- determines the new values.
intersectionWith :: (VG.Vector v k, Monoid (v k), Map map k) => (a -> b -> c)
                              -> TrieMap map v k a
                              -> TrieMap map v k b
                              -> TrieMap map v k c
intersectionWith = Base.intersectionWith

-- | @O(min(n1 m1,n2 m2))@. Like 'intersectionWith', but the combining function
-- is applied strictly.
intersectionWith' :: (VG.Vector v k, Monoid (v k), Map map k) => (a -> b -> c)
                               -> TrieMap map v k a
                               -> TrieMap map v k b
                               -> TrieMap map v k c
intersectionWith' = Base.intersectionWith'

-- | @O(min(n1 m1,n2 m2))@. Like 'intersectionWith', but in addition to the two
-- values, the key they are associated with is passed to the combining
-- function.
intersectionWithKey :: (VG.Vector v k, Monoid (v k), Map map k) => (v k -> a -> b -> c)
                                 -> TrieMap map v k a
                                 -> TrieMap map v k b
                                 -> TrieMap map v k c
intersectionWithKey = Base.intersectionWithKey

-- | @O(min(n1 m1,n2 m2))@. Like 'intersectionWithKey', but the combining
-- function is applied strictly.
intersectionWithKey' :: (VG.Vector v k, Monoid (v k), Map map k) => (v k -> a -> b -> c)
                                  -> TrieMap map v k a
                                  -> TrieMap map v k b
                                  -> TrieMap map v k c
intersectionWithKey' = Base.intersectionWithKey'

-- * Filtering

-- | @O(n m)@. Apply the given function to the elements in the map, discarding
-- those for which the function returns 'False'.
filter :: (VG.Vector v k, Monoid (v k), Map map k) => (a -> Bool) -> TrieMap map v k a -> TrieMap map v k a
filter = filterWithKey . const

-- | @O(n m)@. Like 'filter', but the key associated with the element is also
-- passed to the given predicate.
filterWithKey :: (VG.Vector v k, Monoid (v k), Map map k)
              => (v k -> a -> Bool) -> TrieMap map v k a -> TrieMap map v k a
filterWithKey = Base.filterWithKey

-- | @O(n m)@. A pair of maps: the first element contains those values for
-- which the given predicate returns 'True', and the second contains those for
-- which it was 'False'.
partition :: (VG.Vector v k, Monoid (v k), Map map k) => (a -> Bool)
                       -> TrieMap map v k a
                       -> (TrieMap map v k a, TrieMap map v k a)
partition = partitionWithKey . const

-- | @O(n m)@. Like 'partition', but the key associated with the element is
-- also passed to the given predicate.
partitionWithKey :: (VG.Vector v k, Monoid (v k), Map map k) => (v k -> a -> Bool)
                              -> TrieMap map v k a
                              -> (TrieMap map v k a, TrieMap map v k a)
partitionWithKey = Base.partitionWithKey

-- | @O(n m)@. Apply the given function to the elements in the map, preserving
-- only the 'Just' results.
mapMaybe :: (VG.Vector v k, Monoid (v k), Map map k)
         => (a -> Maybe b) -> TrieMap map v k a -> TrieMap map v k b
mapMaybe = mapMaybeWithKey . const

-- | @O(n m)@. Like 'mapMaybe', but the key associated with the element is also
-- passed to the given function.
mapMaybeWithKey :: (VG.Vector v k, Monoid (v k), Map map k)
                => (v k -> a -> Maybe b) -> TrieMap map v k a -> TrieMap map v k b
mapMaybeWithKey f =
   fromList . Maybe.mapMaybe (\(k,v) -> fmap ((,) k) (f k v)) . toList

-- | @O(n m)@. Apply the given function to the elements in the map, separating
-- the 'Left' results from the 'Right'. The first element of the pair contains
-- the former results, and the second the latter.
mapEither :: (VG.Vector v k, Monoid (v k), Map map k) => (a -> Either b c)
                       -> TrieMap map v k a
                       -> (TrieMap map v k b, TrieMap map v k c)
mapEither = mapEitherWithKey . const

-- | @O(n m)@. Like 'mapEither', but the key associated with the element is
-- also passed to the given function.
mapEitherWithKey :: (VG.Vector v k, Monoid (v k), Map map k) => (v k -> a -> Either b c)
                              -> TrieMap map v k a
                              -> (TrieMap map v k b, TrieMap map v k c)
mapEitherWithKey f =
   (fromList *** fromList) . partitionEithers .
   Prelude.map (\(k,v) -> either (Left . (,) k) (Right . (,) k) (f k v)) .
   toList

-- * Mapping

-- | @O(n m)@. Apply the given function to all the elements in the map.
map :: (VG.Vector v k, Monoid (v k), Map map k) => (a -> b) -> TrieMap map v k a -> TrieMap map v k b
map = genericMap fmap

-- | @O(n m)@. Like 'map', but apply the function strictly.
map' :: (VG.Vector v k, Monoid (v k), Map map k) => (a -> b) -> TrieMap map v k a -> TrieMap map v k b
map' = genericMap fmap'

genericMap :: (VG.Vector v k, Monoid (v k), Map map k) => ((a -> b) -> Maybe a -> Maybe b)
                        -> (a -> b) -> TrieMap map v k a -> TrieMap map v k b
genericMap myFmap f (Tr v p m) = Tr (myFmap f v) p
                                    (Map.map (genericMap myFmap f) m)

-- | @O(n m)@. Like 'map', but also pass the key associated with the element to
-- the given function.
mapWithKey :: (VG.Vector v k, Monoid (v k), Map map k)
           => (v k -> a -> b) -> TrieMap map v k a -> TrieMap map v k b
mapWithKey = genericMapWithKey fmap

-- | @O(n m)@. Like 'mapWithKey', but apply the function strictly.
mapWithKey' :: (VG.Vector v k, Monoid (v k), Map map k)
            => (v k -> a -> b) -> TrieMap map v k a -> TrieMap map v k b
mapWithKey' = genericMapWithKey fmap'

genericMapWithKey :: (VG.Vector v k, Monoid (v k), Map map k)
                  => ((a -> b) -> Maybe a -> Maybe b)
                  -> (v k -> a -> b) -> TrieMap map v k a -> TrieMap map v k b
genericMapWithKey = go mempty
 where
   go k myFmap f (Tr v p m) =
      let k' = k `mappend` p
       in Tr (myFmap (f $ k') v)
             p
             (Map.mapWithKey (\x -> go (k' `VG.snoc` x) myFmap f) m)

-- | @O(n m)@. Apply the given function to all the keys in a map.
--
-- > mapKeys = mapKeysWith const
mapKeys :: (VG.Vector v k1, VG.Vector v k2, Monoid (v k1), Monoid (v k2), Map map k1, Map map k2)
        => (v k1 -> v k2) -> TrieMap map v k1 a -> TrieMap map v k2 a
mapKeys = mapKeysWith const

-- | @O(n m)@. Like 'mapKeys', but use the first given function to combine
-- elements if the second function gives two keys the same value.
mapKeysWith :: (VG.Vector v k1, VG.Vector v k2, Monoid (v k1), Monoid (v k2), Map map k1, Map map k2)
            => (a -> a -> a)
            -> (v k1 -> v k2)
            -> TrieMap map v k1 a
            -> TrieMap map v k2 a
mapKeysWith = Base.mapKeysWith . fromListWith

-- | @O(n m)@. Apply the given function to the contents of all the keys in the
-- map.
--
-- > mapInKeys = mapInKeysWith const
mapInKeys :: (VG.Vector v k1, VG.Vector v k2, Monoid (v k1), Monoid (v k2), Map map k1, Map map k2)
          => (k1 -> k2) -> TrieMap map v k1 a -> TrieMap map v k2 a
mapInKeys = mapInKeysWith defaultUnion

-- | @O(n m)@. Like 'mapInKeys', but combine identical keys strictly.
--
-- > mapInKeys' = mapInKeysWith' const
mapInKeys' :: (VG.Vector v k1, VG.Vector v k2, Monoid (v k1), Monoid (v k2), Map map k1, Map map k2)
           => (k1 -> k2) -> TrieMap map v k1 a -> TrieMap map v k2 a
mapInKeys' = mapInKeysWith' defaultUnion

-- | @O(n m)@. Like 'mapInKeys', but use the first given function to combine
-- elements if the second function gives two keys the same value.
mapInKeysWith :: (VG.Vector v k1, VG.Vector v k2, Monoid (v k1), Monoid (v k2), Map map k1, Map map k2)
              => (a -> a -> a)
              -> (k1 -> k2)
              -> TrieMap map v k1 a
              -> TrieMap map v k2 a
mapInKeysWith = Base.mapInKeysWith

-- | @O(n m)@. Like 'mapInKeysWith', but apply the combining function strictly.
mapInKeysWith' :: (VG.Vector v k1, VG.Vector v k2, Monoid (v k1), Monoid (v k2), Map map k1, Map map k2)
               => (a -> a -> a)
               -> (k1 -> k2)
               -> TrieMap map v k1 a
               -> TrieMap map v k2 a
mapInKeysWith' = Base.mapInKeysWith'

-- | @O(n m)@. Like "Data.List".@mapAccumL@ on the 'toList' representation.
--
-- Essentially a combination of 'map' and 'foldl': the given
-- function is applied to each element of the map, resulting in a new value for
-- the accumulator and a replacement element for the map.
mapAccum :: (VG.Vector v k, Monoid (v k), Map map k) => (acc -> a -> (acc, b))
                      -> acc
                      -> TrieMap map v k a
                      -> (acc, TrieMap map v k b)
mapAccum = genericMapAccum Map.mapAccum (flip const)

-- | @O(n m)@. Like 'mapAccum', but the function is applied strictly.
mapAccum' :: (VG.Vector v k, Monoid (v k), Map map k) => (acc -> a -> (acc, b))
                       -> acc
                       -> TrieMap map v k a
                       -> (acc, TrieMap map v k b)
mapAccum' = genericMapAccum Map.mapAccum seq

-- | @O(n m)@. Like 'mapAccum', but the function receives the key in addition
-- to the value associated with it.
mapAccumWithKey :: (VG.Vector v k, Monoid (v k), Map map k) => (acc -> v k -> a -> (acc, b))
                             -> acc
                             -> TrieMap map v k a
                             -> (acc, TrieMap map v k b)
mapAccumWithKey = genericMapAccumWithKey Map.mapAccumWithKey (flip const)

-- | @O(n m)@. Like 'mapAccumWithKey', but the function is applied strictly.
mapAccumWithKey' :: (VG.Vector v k, Monoid (v k), Map map k) => (acc -> v k -> a -> (acc, b))
                              -> acc
                              -> TrieMap map v k a
                              -> (acc, TrieMap map v k b)
mapAccumWithKey' = genericMapAccumWithKey Map.mapAccumWithKey seq

-- | @O(n m)@. Like 'mapAccum', but in ascending order, as though operating on
-- the 'toAscList' representation.
mapAccumAsc :: (VG.Vector v k, Monoid (v k), OrdMap map k)
            => (acc -> a -> (acc, b))
            -> acc
            -> TrieMap map v k a
            -> (acc, TrieMap map v k b)
mapAccumAsc = genericMapAccum Map.mapAccumAsc (flip const)

-- | @O(n m)@. Like 'mapAccumAsc', but the function is applied strictly.
mapAccumAsc' :: (VG.Vector v k, Monoid (v k), OrdMap map k)
             => (acc -> a -> (acc, b))
             -> acc
             -> TrieMap map v k a
             -> (acc, TrieMap map v k b)
mapAccumAsc' = genericMapAccum Map.mapAccumAsc seq

-- | @O(n m)@. Like 'mapAccumAsc', but the function receives the key in
-- addition to the value associated with it.
mapAccumAscWithKey :: (VG.Vector v k, Monoid (v k), OrdMap map k)
                   => (acc -> v k -> a -> (acc, b))
                   -> acc
                   -> TrieMap map v k a
                   -> (acc, TrieMap map v k b)
mapAccumAscWithKey = genericMapAccumWithKey Map.mapAccumAscWithKey (flip const)

-- | @O(n m)@. Like 'mapAccumAscWithKey', but the function is applied strictly.
mapAccumAscWithKey' :: (VG.Vector v k, Monoid (v k), OrdMap map k)
                    => (acc -> v k -> a -> (acc, b))
                    -> acc
                    -> TrieMap map v k a
                    -> (acc, TrieMap map v k b)
mapAccumAscWithKey' = genericMapAccumWithKey Map.mapAccumAscWithKey seq

-- | @O(n m)@. Like 'mapAccum', but in descending order, as though operating on
-- the 'toDescList' representation.
mapAccumDesc :: (VG.Vector v k, Monoid (v k), OrdMap map k)
             => (acc -> a -> (acc, b))
             -> acc
             -> TrieMap map v k a
             -> (acc, TrieMap map v k b)
mapAccumDesc = genericMapAccum Map.mapAccumDesc (flip const)

-- | @O(n m)@. Like 'mapAccumDesc', but the function is applied strictly.
mapAccumDesc' :: (VG.Vector v k, Monoid (v k), OrdMap map k)
              => (acc -> a -> (acc, b))
              -> acc
              -> TrieMap map v k a
              -> (acc, TrieMap map v k b)
mapAccumDesc' = genericMapAccum Map.mapAccumDesc seq

-- | @O(n m)@. Like 'mapAccumDesc', but the function receives the key in
-- addition to the value associated with it.
mapAccumDescWithKey :: (VG.Vector v k, Monoid (v k), OrdMap map k)
                    => (acc -> v k -> a -> (acc, b))
                    -> acc
                    -> TrieMap map v k a
                    -> (acc, TrieMap map v k b)
mapAccumDescWithKey =
   genericMapAccumWithKey Map.mapAccumDescWithKey (flip const)

-- | @O(n m)@. Like 'mapAccumDescWithKey', but the function is applied
-- strictly.
mapAccumDescWithKey' :: (VG.Vector v k, Monoid (v k), OrdMap map k)
                     => (acc -> v k -> a -> (acc, b))
                     -> acc
                     -> TrieMap map v k a
                     -> (acc, TrieMap map v k b)
mapAccumDescWithKey' = genericMapAccumWithKey Map.mapAccumDescWithKey seq

genericMapAccum :: (VG.Vector v k, Monoid (v k), Map map k)
                => (  (acc -> TrieMap map v k a -> (acc, TrieMap map v k b))
                   -> acc
                   -> CMap map v k a
                   -> (acc, CMap map v k b)
                   )
                -> (b -> (acc, Maybe b) -> (acc, Maybe b))
                -> (acc -> a -> (acc, b))
                -> acc
                -> TrieMap map v k a
                -> (acc, TrieMap map v k b)
genericMapAccum subMapAccum seeq f acc (Tr mv p m) =
   let (acc', mv') =
          case mv of
               Nothing -> (acc, Nothing)
               Just v  ->
                  let (acc'', v') = f acc v
                   in v' `seeq` (acc'', Just v')
    in second (Tr mv' p) $
         subMapAccum (genericMapAccum subMapAccum seeq f) acc' m

genericMapAccumWithKey :: (VG.Vector v k, Monoid (v k), Map map k) => (  (  acc
                                          -> k
                                          -> TrieMap map v k a
                                          -> (acc, TrieMap map v k b)
                                          )
                                       -> acc
                                       -> CMap map v k a
                                       -> (acc, CMap map v k b)
                                       )
                                    -> (b -> (acc, Maybe b) -> (acc, Maybe b))
                                    -> (acc -> v k -> a -> (acc, b))
                                    -> acc
                                    -> TrieMap map v k a
                                    -> (acc, TrieMap map v k b)
genericMapAccumWithKey = go mempty
 where
   go k subMapAccum seeq f acc (Tr mv p m) =
      let k'         = k `mappend` p
          (acc', mv') =
             case mv of
                  Nothing -> (acc, Nothing)
                  Just v  ->
                     let (acc'', v') = f acc k' v
                      in v' `seeq` (acc'', Just v')
       in second (Tr mv' p) $
             subMapAccum (\a x -> go (k' `VG.snoc` x) subMapAccum seeq f a)
                         acc' m

-- * Folding

-- | @O(n m)@. Equivalent to a list @foldr@ on the 'toList' representation,
-- folding only over the elements.
foldr :: (VG.Vector v k, Monoid (v k), Map map k) => (a -> b -> b) -> b -> TrieMap map v k a -> b
foldr = foldrWithKey . const

-- | @O(n m)@. Equivalent to a list @foldr@ on the 'toList' representation,
-- folding over both the keys and the elements.
foldrWithKey :: (VG.Vector v k, Monoid (v k), Map map k) => (v k -> a -> b -> b) -> b -> TrieMap map v k a -> b
foldrWithKey = Base.foldrWithKey

-- | @O(n m)@. Equivalent to a list @foldr@ on the 'toAscList' representation.
foldrAsc :: (VG.Vector v k, Monoid (v k), OrdMap map k)
         => (a -> b -> b) -> b -> TrieMap map v k a -> b
foldrAsc = foldrAscWithKey . const

-- | @O(n m)@. Equivalent to a list @foldr@ on the 'toAscList' representation,
-- folding over both the keys and the elements.
foldrAscWithKey :: (VG.Vector v k, Monoid (v k), OrdMap map k)
                => (v k -> a -> b -> b) -> b -> TrieMap map v k a -> b
foldrAscWithKey = Base.foldrAscWithKey

-- | @O(n m)@. Equivalent to a list @foldr@ on the 'toDescList' representation.
foldrDesc :: (VG.Vector v k, Monoid (v k), OrdMap map k)
          => (a -> b -> b) -> b -> TrieMap map v k a -> b
foldrDesc = foldrDescWithKey . const

-- | @O(n m)@. Equivalent to a list @foldr@ on the 'toDescList' representation,
-- folding over both the keys and the elements.
foldrDescWithKey :: (VG.Vector v k, Monoid (v k), OrdMap map k)
                 => (v k -> a -> b -> b) -> b -> TrieMap map v k a -> b
foldrDescWithKey = Base.foldrDescWithKey

-- | @O(n m)@. Equivalent to a list @foldl@ on the toList representation.
foldl :: (VG.Vector v k, Monoid (v k), Map map k) => (a -> b -> b) -> b -> TrieMap map v k a -> b
foldl = foldlWithKey . const

-- | @O(n m)@. Equivalent to a list @foldl@ on the toList representation,
-- folding over both the keys and the elements.
foldlWithKey :: (VG.Vector v k, Monoid (v k), Map map k) => (v k -> a -> b -> b) -> b -> TrieMap map v k a -> b
foldlWithKey = Base.foldlWithKey

-- | @O(n m)@. Equivalent to a list @foldl@ on the toAscList representation.
foldlAsc :: (VG.Vector v k, Monoid (v k), OrdMap map k)
         => (a -> b -> b) -> b -> TrieMap map v k a -> b
foldlAsc = foldlAscWithKey . const

-- | @O(n m)@. Equivalent to a list @foldl@ on the toAscList representation,
-- folding over both the keys and the elements.
foldlAscWithKey :: (VG.Vector v k, Monoid (v k), OrdMap map k)
                => (v k -> a -> b -> b) -> b -> TrieMap map v k a -> b
foldlAscWithKey = Base.foldlAscWithKey

-- | @O(n m)@. Equivalent to a list @foldl@ on the toDescList representation.
foldlDesc :: (VG.Vector v k, Monoid (v k), OrdMap map k)
          => (a -> b -> b) -> b -> TrieMap map v k a -> b
foldlDesc = foldlDescWithKey . const

-- | @O(n m)@. Equivalent to a list @foldl@ on the toDescList representation,
-- folding over both the keys and the elements.
foldlDescWithKey :: (VG.Vector v k, Monoid (v k), OrdMap map k)
                 => (v k -> a -> b -> b) -> b -> TrieMap map v k a -> b
foldlDescWithKey = Base.foldlDescWithKey

-- | @O(n m)@. Equivalent to a list @foldl'@ on the 'toList' representation.
foldl' :: (VG.Vector v k, Monoid (v k), Map map k) => (a -> b -> b) -> b -> TrieMap map v k a -> b
foldl' = foldlWithKey' . const

-- | @O(n m)@. Equivalent to a list @foldl'@ on the 'toList' representation,
-- folding over both the keys and the elements.
foldlWithKey' :: (VG.Vector v k, Monoid (v k), Map map k) => (v k -> a -> b -> b) -> b -> TrieMap map v k a -> b
foldlWithKey' = Base.foldlWithKey'

-- | @O(n m)@. Equivalent to a list @foldl'@ on the 'toAscList' representation.
foldlAsc' :: (VG.Vector v k, Monoid (v k), OrdMap map k)
          => (a -> b -> b) -> b -> TrieMap map v k a -> b
foldlAsc' = foldlAscWithKey' . const

-- | @O(n m)@. Equivalent to a list @foldl'@ on the 'toAscList' representation,
-- folding over both the keys and the elements.
foldlAscWithKey' :: (VG.Vector v k, Monoid (v k), OrdMap map k)
                 => (v k -> a -> b -> b) -> b -> TrieMap map v k a -> b
foldlAscWithKey' = Base.foldlAscWithKey'

-- | @O(n m)@. Equivalent to a list @foldl'@ on the 'toDescList'
-- representation.
foldlDesc' :: (VG.Vector v k, Monoid (v k), OrdMap map k)
           => (a -> b -> b) -> b -> TrieMap map v k a -> b
foldlDesc' = foldlDescWithKey' . const

-- | @O(n m)@. Equivalent to a list @foldl'@ on the 'toDescList'
-- representation, folding over both the keys and the elements.
foldlDescWithKey' :: (VG.Vector v k, Monoid (v k), OrdMap map k)
                  => (v k -> a -> b -> b) -> b -> TrieMap map v k a -> b
foldlDescWithKey' = Base.foldlDescWithKey'

foldTrie :: (VG.Vector v k, Monoid (v k), Map map k)
         => (Maybe a -> v k -> map k (TrieMap map v k a) -> (Maybe a, v k, map k (TrieMap map v k a)))
         -> TrieMap map v k a -> TrieMap map v k a
foldTrie = Base.foldTrie

-- * Conversion between lists

-- | @O(n m)@. Converts the map to a list of the key-value pairs contained
-- within, in undefined order.
toList :: (VG.Vector v k, Monoid (v k), Map map k) => TrieMap map v k a -> [(v k,a)]
toList = Base.toList

-- | @O(n m)@. Converts the map to a list of the key-value pairs contained
-- within, in ascending order.
toAscList :: (VG.Vector v k, Monoid (v k), OrdMap map k) => TrieMap map v k a -> [(v k,a)]
toAscList = Base.toAscList

-- | @O(n m)@. Converts the map to a list of the key-value pairs contained
-- within, in descending order.
toDescList :: (VG.Vector v k, Monoid (v k), OrdMap map k) => TrieMap map v k a -> [(v k,a)]
toDescList = Base.toDescList

-- | @O(n m)@. Creates a map from a list of key-value pairs. If a key occurs
-- more than once, the value from the last pair (according to the list's order)
-- is the one which ends up in the map.
--
-- > fromList = fromListWith const
fromList :: (VG.Vector v k, Monoid (v k), Map map k) => [(v k,a)] -> TrieMap map v k a
fromList = Base.fromList

-- | @O(n m)@. Like 'fromList', but the given function is used to determine the
-- final value if a key occurs more than once. The function is applied as
-- though it were flipped and then applied as a left fold over the values in
-- the given list's order. Or, equivalently (except as far as performance is
-- concerned), as though the function were applied as a right fold over the
-- values in the reverse of the given list's order. For example:
--
-- > fromListWith (-) [("a",1),("a",2),("a",3),("a",4)]
-- >    == fromList [("a",4-(3-(2-1)))]
-- >    == fromList [("a",2)]
fromListWith :: (VG.Vector v k, Monoid (v k), Map map k) => (a -> a -> a) -> [(v k,a)] -> TrieMap map v k a
fromListWith = Base.fromListWith

-- | @O(n m)@. Like 'fromListWith', but the combining function is applied
-- strictly.
fromListWith' :: (VG.Vector v k, Monoid (v k), Map map k) => (a -> a -> a) -> [(v k,a)] -> TrieMap map v k a
fromListWith' = Base.fromListWith'

-- | @O(n m)@. Like 'fromListWith', but the key, in addition to the values to
-- be combined, is passed to the combining function.
fromListWithKey :: (VG.Vector v k, Monoid (v k), Map map k)
                => (v k -> a -> a -> a) -> [(v k,a)] -> TrieMap map v k a
fromListWithKey = Base.fromListWithKey

-- | @O(n m)@. Like 'fromListWithKey', but the combining function is applied
-- strictly.
fromListWithKey' :: (VG.Vector v k, Monoid (v k), Map map k)
                 => (v k -> a -> a -> a) -> [(v k,a)] -> TrieMap map v k a
fromListWithKey' = Base.fromListWithKey'

-- * Ordering ops

-- | @O(m)@. Removes and returns the minimal key in the map, along with the
-- value associated with it. If the map is empty, 'Nothing' and the original
-- map are returned.
minView :: (VG.Vector v k, Monoid (v k), OrdMap map k) => TrieMap map v k a -> (Maybe (v k, a), TrieMap map v k a)
minView = Base.minView

-- | @O(m)@. Removes and returns the maximal key in the map, along with the
-- value associated with it. If the map is empty, 'Nothing' and the original
-- map are returned.
maxView :: (VG.Vector v k, Monoid (v k), OrdMap map k) => TrieMap map v k a -> (Maybe (v k, a), TrieMap map v k a)
maxView = Base.maxView

-- | @O(m)@. Like 'fst' composed with 'minView'. 'Just' the minimal key in the
-- map and its associated value, or 'Nothing' if the map is empty.
findMin :: (VG.Vector v k, Monoid (v k), OrdMap map k) => TrieMap map v k a -> Maybe (v k, a)
findMin = Base.findMin

-- | @O(m)@. Like 'fst' composed with 'maxView'. 'Just' the minimal key in the
-- map and its associated value, or 'Nothing' if the map is empty.
findMax :: (VG.Vector v k, Monoid (v k), OrdMap map k) => TrieMap map v k a -> Maybe (v k, a)
findMax = Base.findMax

-- | @O(m)@. Like 'snd' composed with 'minView'. The map without its minimal
-- key, or the unchanged original map if it was empty.
deleteMin :: (VG.Vector v k, Monoid (v k), OrdMap map k) => TrieMap map v k a -> TrieMap map v k a
deleteMin = Base.deleteMin

-- | @O(m)@. Like 'snd' composed with 'maxView'. The map without its maximal
-- key, or the unchanged original map if it was empty.
deleteMax :: (VG.Vector v k, Monoid (v k), OrdMap map k) => TrieMap map v k a -> TrieMap map v k a
deleteMax = Base.deleteMax

-- | @O(min(m,s))@. Splits the map in two about the given key. The first
-- element of the resulting pair is a map containing the keys lesser than the
-- given key; the second contains those keys that are greater.
split :: (VG.Vector v k, Monoid (v k), OrdMap map k)
      => v k -> TrieMap map v k a -> (TrieMap map v k a, TrieMap map v k a)
split = Base.split

-- | @O(min(m,s))@. Like 'split', but also returns the value associated with
-- the given key, if any.
splitLookup :: (VG.Vector v k, Monoid (v k), OrdMap map k) => v k
                            -> TrieMap map v k a
                            -> (TrieMap map v k a, Maybe a, TrieMap map v k a)
splitLookup = Base.splitLookup

-- | @O(m)@. 'Just' the key of the map which precedes the given key in order,
-- along with its associated value, or 'Nothing' if the map is empty.
findPredecessor :: (VG.Vector v k, Monoid (v k), OrdMap map k) => v k -> TrieMap map v k a -> Maybe (v k, a)
findPredecessor = Base.findPredecessor

-- | @O(m)@. 'Just' the key of the map which succeeds the given key in order,
-- along with its associated value, or 'Nothing' if the map is empty.
findSuccessor :: (VG.Vector v k, Monoid (v k), OrdMap map k) => v k -> TrieMap map v k a -> Maybe (v k, a)
findSuccessor = Base.findSuccessor

-- * Trie-only operations

-- | @O(s)@. The map which contains all keys of which the given key is a
-- prefix. For example:
--
-- > lookupPrefix "ab" (fromList [("a",1),("ab",2),("ac",3),("abc",4)])
-- >    == fromList [("ab",2),("abc",4)]
lookupPrefix :: (VG.Vector v k, Monoid (v k), Map map k) => v k -> TrieMap map v k a -> TrieMap map v k a
lookupPrefix = Base.lookupPrefix

-- | @O(s)@. Prepends the given key to all the keys of the map. For example:
--
-- > addPrefix "xa" (fromList [("a",1),("b",2)])
-- >    == fromList [("xaa",1),("xab",2)]
addPrefix :: (VG.Vector v k, Monoid (v k), Map map k) => v k -> TrieMap map v k a -> TrieMap map v k a
addPrefix = Base.addPrefix

-- | @O(s)@. The map which contains all keys of which the given key is a
-- prefix, with the prefix removed from each key. If the given key is not a
-- prefix of any key in the map, an empty map is returned. For example:
--
-- > deletePrefix "a" (fromList [("a",1),("ab",2),("ac",3)])
-- >    == fromList [("",1),("b",2),("c",3)]
--
-- This function can be used, for instance, to reduce potentially expensive I/O
-- operations: if you need to find the value in a map associated with a string,
-- but you only have a prefix of it and retrieving the rest is an expensive
-- operation, calling 'deletePrefix' with what you have might allow you to
-- avoid the operation: if the resulting map is empty, the entire string cannot
-- be a member of the map.
deletePrefix :: (VG.Vector v k, Monoid (v k), Map map k) => v k -> TrieMap map v k a -> TrieMap map v k a
deletePrefix = Base.deletePrefix

-- | @O(s)@. Deletes all keys which are suffixes of the given key. For example:
--
-- > deleteSuffixes "ab" (fromList $ zip ["a","ab","ac","b","abc"] [1..])
-- >    == fromList [("a",1),("ac",3),("b",4)]
deleteSuffixes :: (VG.Vector v k, Monoid (v k), Map map k) => v k -> TrieMap map v k a -> TrieMap map v k a
deleteSuffixes = Base.deleteSuffixes

-- | @O(1)@. A triple containing the longest common prefix of all keys in the
-- map, the value associated with that prefix, if any, and the map with that
-- prefix removed from all the keys as well as the map itself. Examples:
--
-- > splitPrefix (fromList [("a",1),("b",2)])
-- >    == ("", Nothing, fromList [("a",1),("b",2)])
-- > splitPrefix (fromList [("a",1),("ab",2),("ac",3)])
-- >    == ("a", Just 1, fromList [("b",2),("c",3)])
splitPrefix :: (VG.Vector v k, Monoid (v k), Map map k) => TrieMap map v k a -> (v k, Maybe a, TrieMap map v k a)
splitPrefix = Base.splitPrefix

-- | @O(1)@. The children of the longest common prefix in the trie as maps,
-- associated with their distinguishing key value. If the map contains less
-- than two keys, this function will return an empty map. Examples;
--
-- > children (fromList [("a",1),("abc",2),("abcd",3)])
-- >    == Map.fromList [('b',fromList [("c",2),("cd",3)])]
-- > children (fromList [("b",1),("c",2)])
-- >    == Map.fromList [('b',fromList [("",1)]),('c',fromList [("",2)])]
children :: (VG.Vector v k, Monoid (v k), Map map k) => TrieMap map v k a -> map k (TrieMap map v k a)
children = Base.children

-- | @O(1)@. The children of the first element of the longest common prefix in
-- the trie as maps, associated with their distinguishing key value. If the map
-- contains less than two keys, this function will return an empty map.
--
-- If the longest common prefix of all keys in the trie is the empty list, this
-- function is equivalent to 'children'.
--
-- Examples:
--
-- > children1 (fromList [("abc",1),("abcd",2)])
-- >    == Map.fromList [('a',fromList [("bc",1),("bcd",2)])]
-- > children1 (fromList [("b",1),("c",2)])
-- >    == Map.fromList [('b',fromList [("",1)]),('c',fromList [("",2)])]
children1 :: (VG.Vector v k, Monoid (v k), Map map k) => TrieMap map v k a -> map k (TrieMap map v k a)
children1 = Base.children1

-- * Visualization

-- | @O(n m)@. Displays the map's internal structure in an undefined way. That
-- is to say, no program should depend on the function's results.
showTrie :: (VG.Vector v k, Monoid (v k), Show k, Show (v k), Show a, Map map k)
         => TrieMap map v k a -> ShowS
showTrie = Base.showTrieWith $ \mv -> case mv of
                                           Nothing -> showChar ' '
                                           Just v  -> showsPrec 11 v

-- | @O(n m)@. Like 'showTrie', but uses the given function to display the
-- elements of the map. Still undefined.
showTrieWith :: (VG.Vector v k, Monoid (v k), Show k, Show (v k), Map map k)
             => (Maybe a -> ShowS) -> TrieMap map v k a -> ShowS
showTrieWith = Base.showTrieWith
