-- autogenerated from https://www.unicode.org/Public/15.0.0/ucd/extracted/DerivedName.txt
-- |
-- Module      : Unicode.Internal.Char.UnicodeData.DerivedName
-- Copyright   : (c) 2022 Composewell Technologies and Contributors
-- License     : Apache-2.0
-- Maintainer  : streamly@composewell.com
-- Stability   : experimental

{-# LANGUAGE PatternSynonyms #-}
{-# OPTIONS_HADDOCK hide #-}

module Unicode.Internal.Char.UnicodeData.DerivedName
    ( name
    , pattern NoName
    , pattern CjkCompatibilityIdeograph
    , pattern CjkUnifiedIdeograph
    , pattern TangutIdeograph
    , pattern HangulSyllable
    ) where

import Data.Int (Int32)
import Foreign.C (CChar)
import GHC.Exts
    ( Addr#, Char#, Int#, Ptr(..),
      ord#, (-#), (<#),
      uncheckedIShiftRL#, andI#,
      plusAddr#, isTrue# )
import Unicode.Internal.Bits.Names (lookupInt32#)

-- | No name. Used to test length returned by 'name'.
--
-- @since 0.3.0
pattern NoName :: Int#
pattern NoName = 0#

-- | CJK compatibility ideograph. Used to test the length returned by 'name'.
--
-- @since 0.3.0
pattern CjkCompatibilityIdeograph :: Int#
pattern CjkCompatibilityIdeograph = 0xf0#

-- | CJK unified ideograph. Used to test the length returned by 'name'.
--
-- @since 0.3.0
pattern CjkUnifiedIdeograph :: Int#
pattern CjkUnifiedIdeograph = 0xf1#

-- | Tangut ideograph. Used to test the length returned by 'name'.
--
-- @since 0.3.0
pattern TangutIdeograph :: Int#
pattern TangutIdeograph = 0xf2#

-- | Hangul syllable. Used to test the length returned by 'name'.
--
-- @since 0.3.0
pattern HangulSyllable :: Int#
pattern HangulSyllable = 0x80#

-- | Name of a character, if defined.
--
-- The return value represents: (ASCII string, string length or special value).
--
-- Some characters require specific processing:
--
-- * If length = @'CjkCompatibilityIdeograph'@,
--   then the name is generated from the pattern “CJK COMPATIBILITY IDEOGRAPH-*”,
--   where * is the hexadecimal codepoint.
-- * If length = @'CjkUnifiedIdeograph'@,
--   then the name is generated from the pattern “CJK UNIFIED IDEOGRAPH-*”,
--   where * is the hexadecimal codepoint.
-- * If length = @'TangutIdeograph'@,
--   then the name is generated from the pattern “TANGUT IDEOGRAPH-*”,
--   where * is the hexadecimal codepoint.
-- * If length ≥ @'HangulSyllable'@,
--   then the name is generated by prepending “HANGUL SYLLABLE ”
--   to the returned string.
--
-- See an example of such implementation using 'String's in 'Unicode.Char.General.Names.name'.
--
-- @since 0.1.0
{-# INLINE name #-}
name :: Char# -> (# Addr#, Int# #)
name c#
    | isTrue# (cp# <# 0x323B0#) = getName cp#
    | isTrue# (cp# <# 0xE0000#) = (# "\0"#, 0# #)
    | isTrue# (cp# <# 0xE01F0#) = getName (cp# -# 0xADC50#)
    | otherwise = (# "\0"#, 0# #)

    where

    -- [NOTE] Encoding
    -- • The names are ASCII. Each name is encoded as a raw bytes literal.
    -- • The names are concatenated in names#.
    --   There are exceptions (see function’s doc).
    -- • The name of a character, if defined, is referenced by an offset in names#.
    -- • The offsets are stored in offsets#. A character entry is composed of:
    --   • a LE Word24 for the offset;
    --   • a Word8 for the length of the name or a special value.

    !cp# = ord# c#

    {-# INLINE getName #-}
    getName k# =
        let !entry# = lookupInt32# offsets# k#
            !offset# = entry# `uncheckedIShiftRL#` 8#
            !name# = names# `plusAddr#` offset#
            !len# = entry# `andI#` 0xff#
        in (# name#, len# #)

    !(Ptr names#) = namesBitmap
    !(Ptr offsets#) = offsetsBitmap

namesBitmap :: Ptr CChar
namesBitmap = Ptr
    "SPACEEXCLAMATION MARKQUOTATION MARKNUMBER SIGNDOLLAR SIGNPERCENT SIGNAMPERSANDAPOSTROPHELEFT PARENTHESISRIGHT PARENTHESISASTERISKPLUS SIGNCOMMAHYPHEN-MINUSFULL STOPSOLIDUSDIGIT ZERODIGIT ONEDIGIT TWODIGIT THREEDIGIT FOURDIGIT FIVEDIGIT SIXDIGIT SEVENDIGIT\
    \ EIGHTDIGIT NINECOLONSEMICOLONLESS-THAN SIGNEQUALS SIGNGREATER-THAN SIGNQUESTION MARKCOMMERCIAL ATLATIN CAPITAL LETTER ALATIN CAPITAL LETTER BLATIN CAPITAL LETTER CLATIN CAPITAL LETTER DLATIN CAPITAL LETTER ELATIN CAPITAL LETTER FLATIN CAPITAL LETTER GLAT\
    \IN CAPITAL LETTER HLATIN CAPITAL LETTER ILATIN CAPITAL LETTER JLATIN CAPITAL LETTER KLATIN CAPITAL LETTER LLATIN CAPITAL LETTER MLATIN CAPITAL LETTER NLATIN CAPITAL LETTER OLATIN CAPITAL LETTER PLATIN CAPITAL LETTER QLATIN CAPITAL LETTER RLATIN CAPITAL LE\
    \TTER SLATIN CAPITAL LETTER TLATIN CAPITAL LETTER ULATIN CAPITAL LETTER VLATIN CAPITAL LETTER WLATIN CAPITAL LETTER XLATIN CAPITAL LETTER YLATIN CAPITAL LETTER ZLEFT SQUARE BRACKETREVERSE SOLIDUSRIGHT SQUARE BRACKETCIRCUMFLEX ACCENTLOW LINEGRAVE ACCENTLATI\
    \N SMALL LETTER ALATIN SMALL LETTER BLATIN SMALL LETTER CLATIN SMALL LETTER DLATIN SMALL LETTER ELATIN SMALL LETTER FLATIN SMALL LETTER GLATIN SMALL LETTER HLATIN SMALL LETTER ILATIN SMALL LETTER JLATIN SMALL LETTER KLATIN SMALL LETTER LLATIN SMALL LETTER \
    \MLATIN SMALL LETTER NLATIN SMALL LETTER OLATIN SMALL LETTER PLATIN SMALL LETTER QLATIN SMALL LETTER RLATIN SMALL LETTER SLATIN SMALL LETTER TLATIN SMALL LETTER ULATIN SMALL LETTER VLATIN SMALL LETTER WLATIN SMALL LETTER XLATIN SMALL LETTER YLATIN SMALL LE\
    \TTER ZLEFT CURLY BRACKETVERTICAL LINERIGHT CURLY BRACKETTILDENO-BREAK SPACEINVERTED EXCLAMATION MARKCENT SIGNPOUND SIGNCURRENCY SIGNYEN SIGNBROKEN BARSECTION SIGNDIAERESISCOPYRIGHT SIGNFEMININE ORDINAL INDICATORLEFT-POINTING DOUBLE ANGLE QUOTATION MARKNOT\
    \ SIGNSOFT HYPHENREGISTERED SIGNMACRONDEGREE SIGNPLUS-MINUS SIGNSUPERSCRIPT TWOSUPERSCRIPT THREEACUTE ACCENTMICRO SIGNPILCROW SIGNMIDDLE DOTCEDILLASUPERSCRIPT ONEMASCULINE ORDINAL INDICATORRIGHT-POINTING DOUBLE ANGLE QUOTATION MARKVULGAR FRACTION ONE QUART\
    \ERVULGAR FRACTION ONE HALFVULGAR FRACTION THREE QUARTERSINVERTED QUESTION MARKLATIN CAPITAL LETTER A WITH GRAVELATIN CAPITAL LETTER A WITH ACUTELATIN CAPITAL LETTER A WITH CIRCUMFLEXLATIN CAPITAL LETTER A WITH TILDELATIN CAPITAL LETTER A WITH DIAERESISLAT\
    \IN CAPITAL LETTER A WITH RING ABOVELATIN CAPITAL LETTER AELATIN CAPITAL LETTER C WITH CEDILLALATIN CAPITAL LETTER E WITH GRAVELATIN CAPITAL LETTER E WITH ACUTELATIN CAPITAL LETTER E WITH CIRCUMFLEXLATIN CAPITAL LETTER E WITH DIAERESISLATIN CAPITAL LETTER \
    \I WITH GRAVELATIN CAPITAL LETTER I WITH ACUTELATIN CAPITAL LETTER I WITH CIRCUMFLEXLATIN CAPITAL LETTER I WITH DIAERESISLATIN CAPITAL LETTER ETHLATIN CAPITAL LETTER N WITH TILDELATIN CAPITAL LETTER O WITH GRAVELATIN CAPITAL LETTER O WITH ACUTELATIN CAPITA\
    \L LETTER O WITH CIRCUMFLEXLATIN CAPITAL LETTER O WITH TILDELATIN CAPITAL LETTER O WITH DIAERESISMULTIPLICATION SIGNLATIN CAPITAL LETTER O WITH STROKELATIN CAPITAL LETTER U WITH GRAVELATIN CAPITAL LETTER U WITH ACUTELATIN CAPITAL LETTER U WITH CIRCUMFLEXLA\
    \TIN CAPITAL LETTER U WITH DIAERESISLATIN CAPITAL LETTER Y WITH ACUTELATIN CAPITAL LETTER THORNLATIN SMALL LETTER SHARP SLATIN SMALL LETTER A WITH GRAVELATIN SMALL LETTER A WITH ACUTELATIN SMALL LETTER A WITH CIRCUMFLEXLATIN SMALL LETTER A WITH TILDELATIN \
    \SMALL LETTER A WITH DIAERESISLATIN SMALL LETTER A WITH RING ABOVELATIN SMALL LETTER AELATIN SMALL LETTER C WITH CEDILLALATIN SMALL LETTER E WITH GRAVELATIN SMALL LETTER E WITH ACUTELATIN SMALL LETTER E WITH CIRCUMFLEXLATIN SMALL LETTER E WITH DIAERESISLAT\
    \IN SMALL LETTER I WITH GRAVELATIN SMALL LETTER I WITH ACUTELATIN SMALL LETTER I WITH CIRCUMFLEXLATIN SMALL LETTER I WITH DIAERESISLATIN SMALL LETTER ETHLATIN SMALL LETTER N WITH TILDELATIN SMALL LETTER O WITH GRAVELATIN SMALL LETTER O WITH ACUTELATIN SMAL\
    \L LETTER O WITH CIRCUMFLEXLATIN SMALL LETTER O WITH TILDELATIN SMALL LETTER O WITH DIAERESISDIVISION SIGNLATIN SMALL LETTER O WITH STROKELATIN SMALL LETTER U WITH GRAVELATIN SMALL LETTER U WITH ACUTELATIN SMALL LETTER U WITH CIRCUMFLEXLATIN SMALL LETTER U\
    \ WITH DIAERESISLATIN SMALL LETTER Y WITH ACUTELATIN SMALL LETTER THORNLATIN SMALL LETTER Y WITH DIAERESISLATIN CAPITAL LETTER A WITH MACRONLATIN SMALL LETTER A WITH MACRONLATIN CAPITAL LETTER A WITH BREVELATIN SMALL LETTER A WITH BREVELATIN CAPITAL LETTER\
    \ A WITH OGONEKLATIN SMALL LETTER A WITH OGONEKLATIN CAPITAL LETTER C WITH ACUTELATIN SMALL LETTER C WITH ACUTELATIN CAPITAL LETTER C WITH CIRCUMFLEXLATIN SMALL LETTER C WITH CIRCUMFLEXLATIN CAPITAL LETTER C WITH DOT ABOVELATIN SMALL LETTER C WITH DOT ABOV\
    \ELATIN CAPITAL LETTER C WITH CARONLATIN SMALL LETTER C WITH CARONLATIN CAPITAL LETTER D WITH CARONLATIN SMALL LETTER D WITH CARONLATIN CAPITAL LETTER D WITH STROKELATIN SMALL LETTER D WITH STROKELATIN CAPITAL LETTER E WITH MACRONLATIN SMALL LETTER E WITH \
    \MACRONLATIN CAPITAL LETTER E WITH BREVELATIN SMALL LETTER E WITH BREVELATIN CAPITAL LETTER E WITH DOT ABOVELATIN SMALL LETTER E WITH DOT ABOVELATIN CAPITAL LETTER E WITH OGONEKLATIN SMALL LETTER E WITH OGONEKLATIN CAPITAL LETTER E WITH CARONLATIN SMALL LE\
    \TTER E WITH CARONLATIN CAPITAL LETTER G WITH CIRCUMFLEXLATIN SMALL LETTER G WITH CIRCUMFLEXLATIN CAPITAL LETTER G WITH BREVELATIN SMALL LETTER G WITH BREVELATIN CAPITAL LETTER G WITH DOT ABOVELATIN SMALL LETTER G WITH DOT ABOVELATIN CAPITAL LETTER G WITH \
    \CEDILLALATIN SMALL LETTER G WITH CEDILLALATIN CAPITAL LETTER H WITH CIRCUMFLEXLATIN SMALL LETTER H WITH CIRCUMFLEXLATIN CAPITAL LETTER H WITH STROKELATIN SMALL LETTER H WITH STROKELATIN CAPITAL LETTER I WITH TILDELATIN SMALL LETTER I WITH TILDELATIN CAPIT\
    \AL LETTER I WITH MACRONLATIN SMALL LETTER I WITH MACRONLATIN CAPITAL LETTER I WITH BREVELATIN SMALL LETTER I WITH BREVELATIN CAPITAL LETTER I WITH OGONEKLATIN SMALL LETTER I WITH OGONEKLATIN CAPITAL LETTER I WITH DOT ABOVELATIN SMALL LETTER DOTLESS ILATIN\
    \ CAPITAL LIGATURE IJLATIN SMALL LIGATURE IJLATIN CAPITAL LETTER J WITH CIRCUMFLEXLATIN SMALL LETTER J WITH CIRCUMFLEXLATIN CAPITAL LETTER K WITH CEDILLALATIN SMALL LETTER K WITH CEDILLALATIN SMALL LETTER KRALATIN CAPITAL LETTER L WITH ACUTELATIN SMALL LET\
    \TER L WITH ACUTELATIN CAPITAL LETTER L WITH CEDILLALATIN SMALL LETTER L WITH CEDILLALATIN CAPITAL LETTER L WITH CARONLATIN SMALL LETTER L WITH CARONLATIN CAPITAL LETTER L WITH MIDDLE DOTLATIN SMALL LETTER L WITH MIDDLE DOTLATIN CAPITAL LETTER L WITH STROK\
    \ELATIN SMALL LETTER L WITH STROKELATIN CAPITAL LETTER N WITH ACUTELATIN SMALL LETTER N WITH ACUTELATIN CAPITAL LETTER N WITH CEDILLALATIN SMALL LETTER N WITH CEDILLALATIN CAPITAL LETTER N WITH CARONLATIN SMALL LETTER N WITH CARONLATIN SMALL LETTER N PRECE\
    \DED BY APOSTROPHELATIN CAPITAL LETTER ENGLATIN SMALL LETTER ENGLATIN CAPITAL LETTER O WITH MACRONLATIN SMALL LETTER O WITH MACRONLATIN CAPITAL LETTER O WITH BREVELATIN SMALL LETTER O WITH BREVELATIN CAPITAL LETTER O WITH DOUBLE ACUTELATIN SMALL LETTER O W\
    \ITH DOUBLE ACUTELATIN CAPITAL LIGATURE OELATIN SMALL LIGATURE OELATIN CAPITAL LETTER R WITH ACUTELATIN SMALL LETTER R WITH ACUTELATIN CAPITAL LETTER R WITH CEDILLALATIN SMALL LETTER R WITH CEDILLALATIN CAPITAL LETTER R WITH CARONLATIN SMALL LETTER R WITH \
    \CARONLATIN CAPITAL LETTER S WITH ACUTELATIN SMALL LETTER S WITH ACUTELATIN CAPITAL LETTER S WITH CIRCUMFLEXLATIN SMALL LETTER S WITH CIRCUMFLEXLATIN CAPITAL LETTER S WITH CEDILLALATIN SMALL LETTER S WITH CEDILLALATIN CAPITAL LETTER S WITH CARONLATIN SMALL\
    \ LETTER S WITH CARONLATIN CAPITAL LETTER T WITH CEDILLALATIN SMALL LETTER T WITH CEDILLALATIN CAPITAL LETTER T WITH CARONLATIN SMALL LETTER T WITH CARONLATIN CAPITAL LETTER T WITH STROKELATIN SMALL LETTER T WITH STROKELATIN CAPITAL LETTER U WITH TILDELATI\
    \N SMALL LETTER U WITH TILDELATIN CAPITAL LETTER U WITH MACRONLATIN SMALL LETTER U WITH MACRONLATIN CAPITAL LETTER U WITH BREVELATIN SMALL LETTER U WITH BREVELATIN CAPITAL LETTER U WITH RING ABOVELATIN SMALL LETTER U WITH RING ABOVELATIN CAPITAL LETTER U W\
    \ITH DOUBLE ACUTELATIN SMALL LETTER U WITH DOUBLE ACUTELATIN CAPITAL LETTER U WITH OGONEKLATIN SMALL LETTER U WITH OGONEKLATIN CAPITAL LETTER W WITH CIRCUMFLEXLATIN SMALL LETTER W WITH CIRCUMFLEXLATIN CAPITAL LETTER Y WITH CIRCUMFLEXLATIN SMALL LETTER Y WI\
    \TH CIRCUMFLEXLATIN CAPITAL LETTER Y WITH DIAERESISLATIN CAPITAL LETTER Z WITH ACUTELATIN SMALL LETTER Z WITH ACUTELATIN CAPITAL LETTER Z WITH DOT ABOVELATIN SMALL LETTER Z WITH DOT ABOVELATIN CAPITAL LETTER Z WITH CARONLATIN SMALL LETTER Z WITH CARONLATIN\
    \ SMALL LETTER LONG SLATIN SMALL LETTER B WITH STROKELATIN CAPITAL LETTER B WITH HOOKLATIN CAPITAL LETTER B WITH TOPBARLATIN SMALL LETTER B WITH TOPBARLATIN CAPITAL LETTER TONE SIXLATIN SMALL LETTER TONE SIXLATIN CAPITAL LETTER OPEN OLATIN CAPITAL LETTER C\
    \ WITH HOOKLATIN SMALL LETTER C WITH HOOKLATIN CAPITAL LETTER AFRICAN DLATIN CAPITAL LETTER D WITH HOOKLATIN CAPITAL LETTER D WITH TOPBARLATIN SMALL LETTER D WITH TOPBARLATIN SMALL LETTER TURNED DELTALATIN CAPITAL LETTER REVERSED ELATIN CAPITAL LETTER SCHW\
    \ALATIN CAPITAL LETTER OPEN ELATIN CAPITAL LETTER F WITH HOOKLATIN SMALL LETTER F WITH HOOKLATIN CAPITAL LETTER G WITH HOOKLATIN CAPITAL LETTER GAMMALATIN SMALL LETTER HVLATIN CAPITAL LETTER IOTALATIN CAPITAL LETTER I WITH STROKELATIN CAPITAL LETTER K WITH\
    \ HOOKLATIN SMALL LETTER K WITH HOOKLATIN SMALL LETTER L WITH BARLATIN SMALL LETTER LAMBDA WITH STROKELATIN CAPITAL LETTER TURNED MLATIN CAPITAL LETTER N WITH LEFT HOOKLATIN SMALL LETTER N WITH LONG RIGHT LEGLATIN CAPITAL LETTER O WITH MIDDLE TILDELATIN CA\
    \PITAL LETTER O WITH HORNLATIN SMALL LETTER O WITH HORNLATIN CAPITAL LETTER OILATIN SMALL LETTER OILATIN CAPITAL LETTER P WITH HOOKLATIN SMALL LETTER P WITH HOOKLATIN LETTER YRLATIN CAPITAL LETTER TONE TWOLATIN SMALL LETTER TONE TWOLATIN CAPITAL LETTER ESH\
    \LATIN LETTER REVERSED ESH LOOPLATIN SMALL LETTER T WITH PALATAL HOOKLATIN CAPITAL LETTER T WITH HOOKLATIN SMALL LETTER T WITH HOOKLATIN CAPITAL LETTER T WITH RETROFLEX HOOKLATIN CAPITAL LETTER U WITH HORNLATIN SMALL LETTER U WITH HORNLATIN CAPITAL LETTER \
    \UPSILONLATIN CAPITAL LETTER V WITH HOOKLATIN CAPITAL LETTER Y WITH HOOKLATIN SMALL LETTER Y WITH HOOKLATIN CAPITAL LETTER Z WITH STROKELATIN SMALL LETTER Z WITH STROKELATIN CAPITAL LETTER EZHLATIN CAPITAL LETTER EZH REVERSEDLATIN SMALL LETTER EZH REVERSED\
    \LATIN SMALL LETTER EZH WITH TAILLATIN LETTER TWO WITH STROKELATIN CAPITAL LETTER TONE FIVELATIN SMALL LETTER TONE FIVELATIN LETTER INVERTED GLOTTAL STOP WITH STROKELATIN LETTER WYNNLATIN LETTER DENTAL CLICKLATIN LETTER LATERAL CLICKLATIN LETTER ALVEOLAR C\
    \LICKLATIN LETTER RETROFLEX CLICKLATIN CAPITAL LETTER DZ WITH CARONLATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CARONLATIN SMALL LETTER DZ WITH CARONLATIN CAPITAL LETTER LJLATIN CAPITAL LETTER L WITH SMALL LETTER JLATIN SMALL LETTER LJLATIN CAPITAL LETT\
    \ER NJLATIN CAPITAL LETTER N WITH SMALL LETTER JLATIN SMALL LETTER NJLATIN CAPITAL LETTER A WITH CARONLATIN SMALL LETTER A WITH CARONLATIN CAPITAL LETTER I WITH CARONLATIN SMALL LETTER I WITH CARONLATIN CAPITAL LETTER O WITH CARONLATIN SMALL LETTER O WITH \
    \CARONLATIN CAPITAL LETTER U WITH CARONLATIN SMALL LETTER U WITH CARONLATIN CAPITAL LETTER U WITH DIAERESIS AND MACRONLATIN SMALL LETTER U WITH DIAERESIS AND MACRONLATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTELATIN SMALL LETTER U WITH DIAERESIS AND ACUTE\
    \LATIN CAPITAL LETTER U WITH DIAERESIS AND CARONLATIN SMALL LETTER U WITH DIAERESIS AND CARONLATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVELATIN SMALL LETTER U WITH DIAERESIS AND GRAVELATIN SMALL LETTER TURNED ELATIN CAPITAL LETTER A WITH DIAERESIS AND MA\
    \CRONLATIN SMALL LETTER A WITH DIAERESIS AND MACRONLATIN CAPITAL LETTER A WITH DOT ABOVE AND MACRONLATIN SMALL LETTER A WITH DOT ABOVE AND MACRONLATIN CAPITAL LETTER AE WITH MACRONLATIN SMALL LETTER AE WITH MACRONLATIN CAPITAL LETTER G WITH STROKELATIN SMA\
    \LL LETTER G WITH STROKELATIN CAPITAL LETTER G WITH CARONLATIN SMALL LETTER G WITH CARONLATIN CAPITAL LETTER K WITH CARONLATIN SMALL LETTER K WITH CARONLATIN CAPITAL LETTER O WITH OGONEKLATIN SMALL LETTER O WITH OGONEKLATIN CAPITAL LETTER O WITH OGONEK AND\
    \ MACRONLATIN SMALL LETTER O WITH OGONEK AND MACRONLATIN CAPITAL LETTER EZH WITH CARONLATIN SMALL LETTER EZH WITH CARONLATIN SMALL LETTER J WITH CARONLATIN CAPITAL LETTER DZLATIN CAPITAL LETTER D WITH SMALL LETTER ZLATIN SMALL LETTER DZLATIN CAPITAL LETTER\
    \ G WITH ACUTELATIN SMALL LETTER G WITH ACUTELATIN CAPITAL LETTER HWAIRLATIN CAPITAL LETTER WYNNLATIN CAPITAL LETTER N WITH GRAVELATIN SMALL LETTER N WITH GRAVELATIN CAPITAL LETTER A WITH RING ABOVE AND ACUTELATIN SMALL LETTER A WITH RING ABOVE AND ACUTELA\
    \TIN CAPITAL LETTER AE WITH ACUTELATIN SMALL LETTER AE WITH ACUTELATIN CAPITAL LETTER O WITH STROKE AND ACUTELATIN SMALL LETTER O WITH STROKE AND ACUTELATIN CAPITAL LETTER A WITH DOUBLE GRAVELATIN SMALL LETTER A WITH DOUBLE GRAVELATIN CAPITAL LETTER A WITH\
    \ INVERTED BREVELATIN SMALL LETTER A WITH INVERTED BREVELATIN CAPITAL LETTER E WITH DOUBLE GRAVELATIN SMALL LETTER E WITH DOUBLE GRAVELATIN CAPITAL LETTER E WITH INVERTED BREVELATIN SMALL LETTER E WITH INVERTED BREVELATIN CAPITAL LETTER I WITH DOUBLE GRAVE\
    \LATIN SMALL LETTER I WITH DOUBLE GRAVELATIN CAPITAL LETTER I WITH INVERTED BREVELATIN SMALL LETTER I WITH INVERTED BREVELATIN CAPITAL LETTER O WITH DOUBLE GRAVELATIN SMALL LETTER O WITH DOUBLE GRAVELATIN CAPITAL LETTER O WITH INVERTED BREVELATIN SMALL LET\
    \TER O WITH INVERTED BREVELATIN CAPITAL LETTER R WITH DOUBLE GRAVELATIN SMALL LETTER R WITH DOUBLE GRAVELATIN CAPITAL LETTER R WITH INVERTED BREVELATIN SMALL LETTER R WITH INVERTED BREVELATIN CAPITAL LETTER U WITH DOUBLE GRAVELATIN SMALL LETTER U WITH DOUB\
    \LE GRAVELATIN CAPITAL LETTER U WITH INVERTED BREVELATIN SMALL LETTER U WITH INVERTED BREVELATIN CAPITAL LETTER S WITH COMMA BELOWLATIN SMALL LETTER S WITH COMMA BELOWLATIN CAPITAL LETTER T WITH COMMA BELOWLATIN SMALL LETTER T WITH COMMA BELOWLATIN CAPITAL\
    \ LETTER YOGHLATIN SMALL LETTER YOGHLATIN CAPITAL LETTER H WITH CARONLATIN SMALL LETTER H WITH CARONLATIN CAPITAL LETTER N WITH LONG RIGHT LEGLATIN SMALL LETTER D WITH CURLLATIN CAPITAL LETTER OULATIN SMALL LETTER OULATIN CAPITAL LETTER Z WITH HOOKLATIN SM\
    \ALL LETTER Z WITH HOOKLATIN CAPITAL LETTER A WITH DOT ABOVELATIN SMALL LETTER A WITH DOT ABOVELATIN CAPITAL LETTER E WITH CEDILLALATIN SMALL LETTER E WITH CEDILLALATIN CAPITAL LETTER O WITH DIAERESIS AND MACRONLATIN SMALL LETTER O WITH DIAERESIS AND MACRO\
    \NLATIN CAPITAL LETTER O WITH TILDE AND MACRONLATIN SMALL LETTER O WITH TILDE AND MACRONLATIN CAPITAL LETTER O WITH DOT ABOVELATIN SMALL LETTER O WITH DOT ABOVELATIN CAPITAL LETTER O WITH DOT ABOVE AND MACRONLATIN SMALL LETTER O WITH DOT ABOVE AND MACRONLA\
    \TIN CAPITAL LETTER Y WITH MACRONLATIN SMALL LETTER Y WITH MACRONLATIN SMALL LETTER L WITH CURLLATIN SMALL LETTER N WITH CURLLATIN SMALL LETTER T WITH CURLLATIN SMALL LETTER DOTLESS JLATIN SMALL LETTER DB DIGRAPHLATIN SMALL LETTER QP DIGRAPHLATIN CAPITAL L\
    \ETTER A WITH STROKELATIN CAPITAL LETTER C WITH STROKELATIN SMALL LETTER C WITH STROKELATIN CAPITAL LETTER L WITH BARLATIN CAPITAL LETTER T WITH DIAGONAL STROKELATIN SMALL LETTER S WITH SWASH TAILLATIN SMALL LETTER Z WITH SWASH TAILLATIN CAPITAL LETTER GLO\
    \TTAL STOPLATIN SMALL LETTER GLOTTAL STOPLATIN CAPITAL LETTER B WITH STROKELATIN CAPITAL LETTER U BARLATIN CAPITAL LETTER TURNED VLATIN CAPITAL LETTER E WITH STROKELATIN SMALL LETTER E WITH STROKELATIN CAPITAL LETTER J WITH STROKELATIN SMALL LETTER J WITH \
    \STROKELATIN CAPITAL LETTER SMALL Q WITH HOOK TAILLATIN SMALL LETTER Q WITH HOOK TAILLATIN CAPITAL LETTER R WITH STROKELATIN SMALL LETTER R WITH STROKELATIN CAPITAL LETTER Y WITH STROKELATIN SMALL LETTER Y WITH STROKELATIN SMALL LETTER TURNED ALATIN SMALL \
    \LETTER ALPHALATIN SMALL LETTER TURNED ALPHALATIN SMALL LETTER B WITH HOOKLATIN SMALL LETTER OPEN OLATIN SMALL LETTER C WITH CURLLATIN SMALL LETTER D WITH TAILLATIN SMALL LETTER D WITH HOOKLATIN SMALL LETTER REVERSED ELATIN SMALL LETTER SCHWALATIN SMALL LE\
    \TTER SCHWA WITH HOOKLATIN SMALL LETTER OPEN ELATIN SMALL LETTER REVERSED OPEN ELATIN SMALL LETTER REVERSED OPEN E WITH HOOKLATIN SMALL LETTER CLOSED REVERSED OPEN ELATIN SMALL LETTER DOTLESS J WITH STROKELATIN SMALL LETTER G WITH HOOKLATIN SMALL LETTER SC\
    \RIPT GLATIN LETTER SMALL CAPITAL GLATIN SMALL LETTER GAMMALATIN SMALL LETTER RAMS HORNLATIN SMALL LETTER TURNED HLATIN SMALL LETTER H WITH HOOKLATIN SMALL LETTER HENG WITH HOOKLATIN SMALL LETTER I WITH STROKELATIN SMALL LETTER IOTALATIN LETTER SMALL CAPIT\
    \AL ILATIN SMALL LETTER L WITH MIDDLE TILDELATIN SMALL LETTER L WITH BELTLATIN SMALL LETTER L WITH RETROFLEX HOOKLATIN SMALL LETTER LEZHLATIN SMALL LETTER TURNED MLATIN SMALL LETTER TURNED M WITH LONG LEGLATIN SMALL LETTER M WITH HOOKLATIN SMALL LETTER N W\
    \ITH LEFT HOOKLATIN SMALL LETTER N WITH RETROFLEX HOOKLATIN LETTER SMALL CAPITAL NLATIN SMALL LETTER BARRED OLATIN LETTER SMALL CAPITAL OELATIN SMALL LETTER CLOSED OMEGALATIN SMALL LETTER PHILATIN SMALL LETTER TURNED RLATIN SMALL LETTER TURNED R WITH LONG \
    \LEGLATIN SMALL LETTER TURNED R WITH HOOKLATIN SMALL LETTER R WITH LONG LEGLATIN SMALL LETTER R WITH TAILLATIN SMALL LETTER R WITH FISHHOOKLATIN SMALL LETTER REVERSED R WITH FISHHOOKLATIN LETTER SMALL CAPITAL RLATIN LETTER SMALL CAPITAL INVERTED RLATIN SMA\
    \LL LETTER S WITH HOOKLATIN SMALL LETTER ESHLATIN SMALL LETTER DOTLESS J WITH STROKE AND HOOKLATIN SMALL LETTER SQUAT REVERSED ESHLATIN SMALL LETTER ESH WITH CURLLATIN SMALL LETTER TURNED TLATIN SMALL LETTER T WITH RETROFLEX HOOKLATIN SMALL LETTER U BARLAT\
    \IN SMALL LETTER UPSILONLATIN SMALL LETTER V WITH HOOKLATIN SMALL LETTER TURNED VLATIN SMALL LETTER TURNED WLATIN SMALL LETTER TURNED YLATIN LETTER SMALL CAPITAL YLATIN SMALL LETTER Z WITH RETROFLEX HOOKLATIN SMALL LETTER Z WITH CURLLATIN SMALL LETTER EZHL\
    \ATIN SMALL LETTER EZH WITH CURLLATIN LETTER GLOTTAL STOPLATIN LETTER PHARYNGEAL VOICED FRICATIVELATIN LETTER INVERTED GLOTTAL STOPLATIN LETTER STRETCHED CLATIN LETTER BILABIAL CLICKLATIN LETTER SMALL CAPITAL BLATIN SMALL LETTER CLOSED OPEN ELATIN LETTER S\
    \MALL CAPITAL G WITH HOOKLATIN LETTER SMALL CAPITAL HLATIN SMALL LETTER J WITH CROSSED-TAILLATIN SMALL LETTER TURNED KLATIN LETTER SMALL CAPITAL LLATIN SMALL LETTER Q WITH HOOKLATIN LETTER GLOTTAL STOP WITH STROKELATIN LETTER REVERSED GLOTTAL STOP WITH STR\
    \OKELATIN SMALL LETTER DZ DIGRAPHLATIN SMALL LETTER DEZH DIGRAPHLATIN SMALL LETTER DZ DIGRAPH WITH CURLLATIN SMALL LETTER TS DIGRAPHLATIN SMALL LETTER TESH DIGRAPHLATIN SMALL LETTER TC DIGRAPH WITH CURLLATIN SMALL LETTER FENG DIGRAPHLATIN SMALL LETTER LS D\
    \IGRAPHLATIN SMALL LETTER LZ DIGRAPHLATIN LETTER BILABIAL PERCUSSIVELATIN LETTER BIDENTAL PERCUSSIVELATIN SMALL LETTER TURNED H WITH FISHHOOKLATIN SMALL LETTER TURNED H WITH FISHHOOK AND TAILMODIFIER LETTER SMALL HMODIFIER LETTER SMALL H WITH HOOKMODIFIER \
    \LETTER SMALL JMODIFIER LETTER SMALL RMODIFIER LETTER SMALL TURNED RMODIFIER LETTER SMALL TURNED R WITH HOOKMODIFIER LETTER SMALL CAPITAL INVERTED RMODIFIER LETTER SMALL WMODIFIER LETTER SMALL YMODIFIER LETTER PRIMEMODIFIER LETTER DOUBLE PRIMEMODIFIER LETT\
    \ER TURNED COMMAMODIFIER LETTER APOSTROPHEMODIFIER LETTER REVERSED COMMAMODIFIER LETTER RIGHT HALF RINGMODIFIER LETTER LEFT HALF RINGMODIFIER LETTER GLOTTAL STOPMODIFIER LETTER REVERSED GLOTTAL STOPMODIFIER LETTER LEFT ARROWHEADMODIFIER LETTER RIGHT ARROWH\
    \EADMODIFIER LETTER UP ARROWHEADMODIFIER LETTER DOWN ARROWHEADMODIFIER LETTER CIRCUMFLEX ACCENTCARONMODIFIER LETTER VERTICAL LINEMODIFIER LETTER MACRONMODIFIER LETTER ACUTE ACCENTMODIFIER LETTER GRAVE ACCENTMODIFIER LETTER LOW VERTICAL LINEMODIFIER LETTER \
    \LOW MACRONMODIFIER LETTER LOW GRAVE ACCENTMODIFIER LETTER LOW ACUTE ACCENTMODIFIER LETTER TRIANGULAR COLONMODIFIER LETTER HALF TRIANGULAR COLONMODIFIER LETTER CENTRED RIGHT HALF RINGMODIFIER LETTER CENTRED LEFT HALF RINGMODIFIER LETTER UP TACKMODIFIER LET\
    \TER DOWN TACKMODIFIER LETTER PLUS SIGNMODIFIER LETTER MINUS SIGNBREVEDOT ABOVERING ABOVEOGONEKSMALL TILDEDOUBLE ACUTE ACCENTMODIFIER LETTER RHOTIC HOOKMODIFIER LETTER CROSS ACCENTMODIFIER LETTER SMALL GAMMAMODIFIER LETTER SMALL LMODIFIER LETTER SMALL SMOD\
    \IFIER LETTER SMALL XMODIFIER LETTER SMALL REVERSED GLOTTAL STOPMODIFIER LETTER EXTRA-HIGH TONE BARMODIFIER LETTER HIGH TONE BARMODIFIER LETTER MID TONE BARMODIFIER LETTER LOW TONE BARMODIFIER LETTER EXTRA-LOW TONE BARMODIFIER LETTER YIN DEPARTING TONE MAR\
    \KMODIFIER LETTER YANG DEPARTING TONE MARKMODIFIER LETTER VOICINGMODIFIER LETTER UNASPIRATEDMODIFIER LETTER DOUBLE APOSTROPHEMODIFIER LETTER LOW DOWN ARROWHEADMODIFIER LETTER LOW UP ARROWHEADMODIFIER LETTER LOW LEFT ARROWHEADMODIFIER LETTER LOW RIGHT ARROW\
    \HEADMODIFIER LETTER LOW RINGMODIFIER LETTER MIDDLE GRAVE ACCENTMODIFIER LETTER MIDDLE DOUBLE GRAVE ACCENTMODIFIER LETTER MIDDLE DOUBLE ACUTE ACCENTMODIFIER LETTER LOW TILDEMODIFIER LETTER RAISED COLONMODIFIER LETTER BEGIN HIGH TONEMODIFIER LETTER END HIGH\
    \ TONEMODIFIER LETTER BEGIN LOW TONEMODIFIER LETTER END LOW TONEMODIFIER LETTER SHELFMODIFIER LETTER OPEN SHELFMODIFIER LETTER LOW LEFT ARROWCOMBINING GRAVE ACCENTCOMBINING ACUTE ACCENTCOMBINING CIRCUMFLEX ACCENTCOMBINING TILDECOMBINING MACRONCOMBINING OVE\
    \RLINECOMBINING BREVECOMBINING DOT ABOVECOMBINING DIAERESISCOMBINING HOOK ABOVECOMBINING RING ABOVECOMBINING DOUBLE ACUTE ACCENTCOMBINING CARONCOMBINING VERTICAL LINE ABOVECOMBINING DOUBLE VERTICAL LINE ABOVECOMBINING DOUBLE GRAVE ACCENTCOMBINING CANDRABIN\
    \DUCOMBINING INVERTED BREVECOMBINING TURNED COMMA ABOVECOMBINING COMMA ABOVECOMBINING REVERSED COMMA ABOVECOMBINING COMMA ABOVE RIGHTCOMBINING GRAVE ACCENT BELOWCOMBINING ACUTE ACCENT BELOWCOMBINING LEFT TACK BELOWCOMBINING RIGHT TACK BELOWCOMBINING LEFT A\
    \NGLE ABOVECOMBINING HORNCOMBINING LEFT HALF RING BELOWCOMBINING UP TACK BELOWCOMBINING DOWN TACK BELOWCOMBINING PLUS SIGN BELOWCOMBINING MINUS SIGN BELOWCOMBINING PALATALIZED HOOK BELOWCOMBINING RETROFLEX HOOK BELOWCOMBINING DOT BELOWCOMBINING DIAERESIS B\
    \ELOWCOMBINING RING BELOWCOMBINING COMMA BELOWCOMBINING CEDILLACOMBINING OGONEKCOMBINING VERTICAL LINE BELOWCOMBINING BRIDGE BELOWCOMBINING INVERTED DOUBLE ARCH BELOWCOMBINING CARON BELOWCOMBINING CIRCUMFLEX ACCENT BELOWCOMBINING BREVE BELOWCOMBINING INVER\
    \TED BREVE BELOWCOMBINING TILDE BELOWCOMBINING MACRON BELOWCOMBINING LOW LINECOMBINING DOUBLE LOW LINECOMBINING TILDE OVERLAYCOMBINING SHORT STROKE OVERLAYCOMBINING LONG STROKE OVERLAYCOMBINING SHORT SOLIDUS OVERLAYCOMBINING LONG SOLIDUS OVERLAYCOMBINING R\
    \IGHT HALF RING BELOWCOMBINING INVERTED BRIDGE BELOWCOMBINING SQUARE BELOWCOMBINING SEAGULL BELOWCOMBINING X ABOVECOMBINING VERTICAL TILDECOMBINING DOUBLE OVERLINECOMBINING GRAVE TONE MARKCOMBINING ACUTE TONE MARKCOMBINING GREEK PERISPOMENICOMBINING GREEK \
    \KORONISCOMBINING GREEK DIALYTIKA TONOSCOMBINING GREEK YPOGEGRAMMENICOMBINING BRIDGE ABOVECOMBINING EQUALS SIGN BELOWCOMBINING DOUBLE VERTICAL LINE BELOWCOMBINING LEFT ANGLE BELOWCOMBINING NOT TILDE ABOVECOMBINING HOMOTHETIC ABOVECOMBINING ALMOST EQUAL TO \
    \ABOVECOMBINING LEFT RIGHT ARROW BELOWCOMBINING UPWARDS ARROW BELOWCOMBINING GRAPHEME JOINERCOMBINING RIGHT ARROWHEAD ABOVECOMBINING LEFT HALF RING ABOVECOMBINING FERMATACOMBINING X BELOWCOMBINING LEFT ARROWHEAD BELOWCOMBINING RIGHT ARROWHEAD BELOWCOMBININ\
    \G RIGHT ARROWHEAD AND UP ARROWHEAD BELOWCOMBINING RIGHT HALF RING ABOVECOMBINING DOT ABOVE RIGHTCOMBINING ASTERISK BELOWCOMBINING DOUBLE RING BELOWCOMBINING ZIGZAG ABOVECOMBINING DOUBLE BREVE BELOWCOMBINING DOUBLE BREVECOMBINING DOUBLE MACRONCOMBINING DOU\
    \BLE MACRON BELOWCOMBINING DOUBLE TILDECOMBINING DOUBLE INVERTED BREVECOMBINING DOUBLE RIGHTWARDS ARROW BELOWCOMBINING LATIN SMALL LETTER ACOMBINING LATIN SMALL LETTER ECOMBINING LATIN SMALL LETTER ICOMBINING LATIN SMALL LETTER OCOMBINING LATIN SMALL LETTE\
    \R UCOMBINING LATIN SMALL LETTER CCOMBINING LATIN SMALL LETTER DCOMBINING LATIN SMALL LETTER HCOMBINING LATIN SMALL LETTER MCOMBINING LATIN SMALL LETTER RCOMBINING LATIN SMALL LETTER TCOMBINING LATIN SMALL LETTER VCOMBINING LATIN SMALL LETTER XGREEK CAPITA\
    \L LETTER HETAGREEK SMALL LETTER HETAGREEK CAPITAL LETTER ARCHAIC SAMPIGREEK SMALL LETTER ARCHAIC SAMPIGREEK NUMERAL SIGNGREEK LOWER NUMERAL SIGNGREEK CAPITAL LETTER PAMPHYLIAN DIGAMMAGREEK SMALL LETTER PAMPHYLIAN DIGAMMAGREEK YPOGEGRAMMENIGREEK SMALL REVE\
    \RSED LUNATE SIGMA SYMBOLGREEK SMALL DOTTED LUNATE SIGMA SYMBOLGREEK SMALL REVERSED DOTTED LUNATE SIGMA SYMBOLGREEK QUESTION MARKGREEK CAPITAL LETTER YOTGREEK TONOSGREEK DIALYTIKA TONOSGREEK CAPITAL LETTER ALPHA WITH TONOSGREEK ANO TELEIAGREEK CAPITAL LETT\
    \ER EPSILON WITH TONOSGREEK CAPITAL LETTER ETA WITH TONOSGREEK CAPITAL LETTER IOTA WITH TONOSGREEK CAPITAL LETTER OMICRON WITH TONOSGREEK CAPITAL LETTER UPSILON WITH TONOSGREEK CAPITAL LETTER OMEGA WITH TONOSGREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS\
    \GREEK CAPITAL LETTER ALPHAGREEK CAPITAL LETTER BETAGREEK CAPITAL LETTER GAMMAGREEK CAPITAL LETTER DELTAGREEK CAPITAL LETTER EPSILONGREEK CAPITAL LETTER ZETAGREEK CAPITAL LETTER ETAGREEK CAPITAL LETTER THETAGREEK CAPITAL LETTER IOTAGREEK CAPITAL LETTER KAP\
    \PAGREEK CAPITAL LETTER LAMDAGREEK CAPITAL LETTER MUGREEK CAPITAL LETTER NUGREEK CAPITAL LETTER XIGREEK CAPITAL LETTER OMICRONGREEK CAPITAL LETTER PIGREEK CAPITAL LETTER RHOGREEK CAPITAL LETTER SIGMAGREEK CAPITAL LETTER TAUGREEK CAPITAL LETTER UPSILONGREEK\
    \ CAPITAL LETTER PHIGREEK CAPITAL LETTER CHIGREEK CAPITAL LETTER PSIGREEK CAPITAL LETTER OMEGAGREEK CAPITAL LETTER IOTA WITH DIALYTIKAGREEK CAPITAL LETTER UPSILON WITH DIALYTIKAGREEK SMALL LETTER ALPHA WITH TONOSGREEK SMALL LETTER EPSILON WITH TONOSGREEK S\
    \MALL LETTER ETA WITH TONOSGREEK SMALL LETTER IOTA WITH TONOSGREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOSGREEK SMALL LETTER ALPHAGREEK SMALL LETTER BETAGREEK SMALL LETTER GAMMAGREEK SMALL LETTER DELTAGREEK SMALL LETTER EPSILONGREEK SMALL LETTER ZETA\
    \GREEK SMALL LETTER ETAGREEK SMALL LETTER THETAGREEK SMALL LETTER IOTAGREEK SMALL LETTER KAPPAGREEK SMALL LETTER LAMDAGREEK SMALL LETTER MUGREEK SMALL LETTER NUGREEK SMALL LETTER XIGREEK SMALL LETTER OMICRONGREEK SMALL LETTER PIGREEK SMALL LETTER RHOGREEK \
    \SMALL LETTER FINAL SIGMAGREEK SMALL LETTER SIGMAGREEK SMALL LETTER TAUGREEK SMALL LETTER UPSILONGREEK SMALL LETTER PHIGREEK SMALL LETTER CHIGREEK SMALL LETTER PSIGREEK SMALL LETTER OMEGAGREEK SMALL LETTER IOTA WITH DIALYTIKAGREEK SMALL LETTER UPSILON WITH\
    \ DIALYTIKAGREEK SMALL LETTER OMICRON WITH TONOSGREEK SMALL LETTER UPSILON WITH TONOSGREEK SMALL LETTER OMEGA WITH TONOSGREEK CAPITAL KAI SYMBOLGREEK BETA SYMBOLGREEK THETA SYMBOLGREEK UPSILON WITH HOOK SYMBOLGREEK UPSILON WITH ACUTE AND HOOK SYMBOLGREEK U\
    \PSILON WITH DIAERESIS AND HOOK SYMBOLGREEK PHI SYMBOLGREEK PI SYMBOLGREEK KAI SYMBOLGREEK LETTER ARCHAIC KOPPAGREEK SMALL LETTER ARCHAIC KOPPAGREEK LETTER STIGMAGREEK SMALL LETTER STIGMAGREEK LETTER DIGAMMAGREEK SMALL LETTER DIGAMMAGREEK LETTER KOPPAGREEK\
    \ SMALL LETTER KOPPAGREEK LETTER SAMPIGREEK SMALL LETTER SAMPICOPTIC CAPITAL LETTER SHEICOPTIC SMALL LETTER SHEICOPTIC CAPITAL LETTER FEICOPTIC SMALL LETTER FEICOPTIC CAPITAL LETTER KHEICOPTIC SMALL LETTER KHEICOPTIC CAPITAL LETTER HORICOPTIC SMALL LETTER \
    \HORICOPTIC CAPITAL LETTER GANGIACOPTIC SMALL LETTER GANGIACOPTIC CAPITAL LETTER SHIMACOPTIC SMALL LETTER SHIMACOPTIC CAPITAL LETTER DEICOPTIC SMALL LETTER DEIGREEK KAPPA SYMBOLGREEK RHO SYMBOLGREEK LUNATE SIGMA SYMBOLGREEK LETTER YOTGREEK CAPITAL THETA SY\
    \MBOLGREEK LUNATE EPSILON SYMBOLGREEK REVERSED LUNATE EPSILON SYMBOLGREEK CAPITAL LETTER SHOGREEK SMALL LETTER SHOGREEK CAPITAL LUNATE SIGMA SYMBOLGREEK CAPITAL LETTER SANGREEK SMALL LETTER SANGREEK RHO WITH STROKE SYMBOLGREEK CAPITAL REVERSED LUNATE SIGMA\
    \ SYMBOLGREEK CAPITAL DOTTED LUNATE SIGMA SYMBOLGREEK CAPITAL REVERSED DOTTED LUNATE SIGMA SYMBOLCYRILLIC CAPITAL LETTER IE WITH GRAVECYRILLIC CAPITAL LETTER IOCYRILLIC CAPITAL LETTER DJECYRILLIC CAPITAL LETTER GJECYRILLIC CAPITAL LETTER UKRAINIAN IECYRILL\
    \IC CAPITAL LETTER DZECYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN ICYRILLIC CAPITAL LETTER YICYRILLIC CAPITAL LETTER JECYRILLIC CAPITAL LETTER LJECYRILLIC CAPITAL LETTER NJECYRILLIC CAPITAL LETTER TSHECYRILLIC CAPITAL LETTER KJECYRILLIC CAPITAL LETTER I\
    \ WITH GRAVECYRILLIC CAPITAL LETTER SHORT UCYRILLIC CAPITAL LETTER DZHECYRILLIC CAPITAL LETTER ACYRILLIC CAPITAL LETTER BECYRILLIC CAPITAL LETTER VECYRILLIC CAPITAL LETTER GHECYRILLIC CAPITAL LETTER DECYRILLIC CAPITAL LETTER IECYRILLIC CAPITAL LETTER ZHECY\
    \RILLIC CAPITAL LETTER ZECYRILLIC CAPITAL LETTER ICYRILLIC CAPITAL LETTER SHORT ICYRILLIC CAPITAL LETTER KACYRILLIC CAPITAL LETTER ELCYRILLIC CAPITAL LETTER EMCYRILLIC CAPITAL LETTER ENCYRILLIC CAPITAL LETTER OCYRILLIC CAPITAL LETTER PECYRILLIC CAPITAL LET\
    \TER ERCYRILLIC CAPITAL LETTER ESCYRILLIC CAPITAL LETTER TECYRILLIC CAPITAL LETTER UCYRILLIC CAPITAL LETTER EFCYRILLIC CAPITAL LETTER HACYRILLIC CAPITAL LETTER TSECYRILLIC CAPITAL LETTER CHECYRILLIC CAPITAL LETTER SHACYRILLIC CAPITAL LETTER SHCHACYRILLIC C\
    \APITAL LETTER HARD SIGNCYRILLIC CAPITAL LETTER YERUCYRILLIC CAPITAL LETTER SOFT SIGNCYRILLIC CAPITAL LETTER ECYRILLIC CAPITAL LETTER YUCYRILLIC CAPITAL LETTER YACYRILLIC SMALL LETTER ACYRILLIC SMALL LETTER BECYRILLIC SMALL LETTER VECYRILLIC SMALL LETTER G\
    \HECYRILLIC SMALL LETTER DECYRILLIC SMALL LETTER IECYRILLIC SMALL LETTER ZHECYRILLIC SMALL LETTER ZECYRILLIC SMALL LETTER ICYRILLIC SMALL LETTER SHORT ICYRILLIC SMALL LETTER KACYRILLIC SMALL LETTER ELCYRILLIC SMALL LETTER EMCYRILLIC SMALL LETTER ENCYRILLIC\
    \ SMALL LETTER OCYRILLIC SMALL LETTER PECYRILLIC SMALL LETTER ERCYRILLIC SMALL LETTER ESCYRILLIC SMALL LETTER TECYRILLIC SMALL LETTER UCYRILLIC SMALL LETTER EFCYRILLIC SMALL LETTER HACYRILLIC SMALL LETTER TSECYRILLIC SMALL LETTER CHECYRILLIC SMALL LETTER S\
    \HACYRILLIC SMALL LETTER SHCHACYRILLIC SMALL LETTER HARD SIGNCYRILLIC SMALL LETTER YERUCYRILLIC SMALL LETTER SOFT SIGNCYRILLIC SMALL LETTER ECYRILLIC SMALL LETTER YUCYRILLIC SMALL LETTER YACYRILLIC SMALL LETTER IE WITH GRAVECYRILLIC SMALL LETTER IOCYRILLIC\
    \ SMALL LETTER DJECYRILLIC SMALL LETTER GJECYRILLIC SMALL LETTER UKRAINIAN IECYRILLIC SMALL LETTER DZECYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN ICYRILLIC SMALL LETTER YICYRILLIC SMALL LETTER JECYRILLIC SMALL LETTER LJECYRILLIC SMALL LETTER NJECYRILLIC S\
    \MALL LETTER TSHECYRILLIC SMALL LETTER KJECYRILLIC SMALL LETTER I WITH GRAVECYRILLIC SMALL LETTER SHORT UCYRILLIC SMALL LETTER DZHECYRILLIC CAPITAL LETTER OMEGACYRILLIC SMALL LETTER OMEGACYRILLIC CAPITAL LETTER YATCYRILLIC SMALL LETTER YATCYRILLIC CAPITAL \
    \LETTER IOTIFIED ECYRILLIC SMALL LETTER IOTIFIED ECYRILLIC CAPITAL LETTER LITTLE YUSCYRILLIC SMALL LETTER LITTLE YUSCYRILLIC CAPITAL LETTER IOTIFIED LITTLE YUSCYRILLIC SMALL LETTER IOTIFIED LITTLE YUSCYRILLIC CAPITAL LETTER BIG YUSCYRILLIC SMALL LETTER BIG\
    \ YUSCYRILLIC CAPITAL LETTER IOTIFIED BIG YUSCYRILLIC SMALL LETTER IOTIFIED BIG YUSCYRILLIC CAPITAL LETTER KSICYRILLIC SMALL LETTER KSICYRILLIC CAPITAL LETTER PSICYRILLIC SMALL LETTER PSICYRILLIC CAPITAL LETTER FITACYRILLIC SMALL LETTER FITACYRILLIC CAPITA\
    \L LETTER IZHITSACYRILLIC SMALL LETTER IZHITSACYRILLIC CAPITAL LETTER IZHITSA WITH DOUBLE GRAVE ACCENTCYRILLIC SMALL LETTER IZHITSA WITH DOUBLE GRAVE ACCENTCYRILLIC CAPITAL LETTER UKCYRILLIC SMALL LETTER UKCYRILLIC CAPITAL LETTER ROUND OMEGACYRILLIC SMALL \
    \LETTER ROUND OMEGACYRILLIC CAPITAL LETTER OMEGA WITH TITLOCYRILLIC SMALL LETTER OMEGA WITH TITLOCYRILLIC CAPITAL LETTER OTCYRILLIC SMALL LETTER OTCYRILLIC CAPITAL LETTER KOPPACYRILLIC SMALL LETTER KOPPACYRILLIC THOUSANDS SIGNCOMBINING CYRILLIC TITLOCOMBIN\
    \ING CYRILLIC PALATALIZATIONCOMBINING CYRILLIC DASIA PNEUMATACOMBINING CYRILLIC PSILI PNEUMATACOMBINING CYRILLIC POKRYTIECOMBINING CYRILLIC HUNDRED THOUSANDS SIGNCOMBINING CYRILLIC MILLIONS SIGNCYRILLIC CAPITAL LETTER SHORT I WITH TAILCYRILLIC SMALL LETTER\
    \ SHORT I WITH TAILCYRILLIC CAPITAL LETTER SEMISOFT SIGNCYRILLIC SMALL LETTER SEMISOFT SIGNCYRILLIC CAPITAL LETTER ER WITH TICKCYRILLIC SMALL LETTER ER WITH TICKCYRILLIC CAPITAL LETTER GHE WITH UPTURNCYRILLIC SMALL LETTER GHE WITH UPTURNCYRILLIC CAPITAL LE\
    \TTER GHE WITH STROKECYRILLIC SMALL LETTER GHE WITH STROKECYRILLIC CAPITAL LETTER GHE WITH MIDDLE HOOKCYRILLIC SMALL LETTER GHE WITH MIDDLE HOOKCYRILLIC CAPITAL LETTER ZHE WITH DESCENDERCYRILLIC SMALL LETTER ZHE WITH DESCENDERCYRILLIC CAPITAL LETTER ZE WIT\
    \H DESCENDERCYRILLIC SMALL LETTER ZE WITH DESCENDERCYRILLIC CAPITAL LETTER KA WITH DESCENDERCYRILLIC SMALL LETTER KA WITH DESCENDERCYRILLIC CAPITAL LETTER KA WITH VERTICAL STROKECYRILLIC SMALL LETTER KA WITH VERTICAL STROKECYRILLIC CAPITAL LETTER KA WITH S\
    \TROKECYRILLIC SMALL LETTER KA WITH STROKECYRILLIC CAPITAL LETTER BASHKIR KACYRILLIC SMALL LETTER BASHKIR KACYRILLIC CAPITAL LETTER EN WITH DESCENDERCYRILLIC SMALL LETTER EN WITH DESCENDERCYRILLIC CAPITAL LIGATURE EN GHECYRILLIC SMALL LIGATURE EN GHECYRILL\
    \IC CAPITAL LETTER PE WITH MIDDLE HOOKCYRILLIC SMALL LETTER PE WITH MIDDLE HOOKCYRILLIC CAPITAL LETTER ABKHASIAN HACYRILLIC SMALL LETTER ABKHASIAN HACYRILLIC CAPITAL LETTER ES WITH DESCENDERCYRILLIC SMALL LETTER ES WITH DESCENDERCYRILLIC CAPITAL LETTER TE \
    \WITH DESCENDERCYRILLIC SMALL LETTER TE WITH DESCENDERCYRILLIC CAPITAL LETTER STRAIGHT UCYRILLIC SMALL LETTER STRAIGHT UCYRILLIC CAPITAL LETTER STRAIGHT U WITH STROKECYRILLIC SMALL LETTER STRAIGHT U WITH STROKECYRILLIC CAPITAL LETTER HA WITH DESCENDERCYRIL\
    \LIC SMALL LETTER HA WITH DESCENDERCYRILLIC CAPITAL LIGATURE TE TSECYRILLIC SMALL LIGATURE TE TSECYRILLIC CAPITAL LETTER CHE WITH DESCENDERCYRILLIC SMALL LETTER CHE WITH DESCENDERCYRILLIC CAPITAL LETTER CHE WITH VERTICAL STROKECYRILLIC SMALL LETTER CHE WIT\
    \H VERTICAL STROKECYRILLIC CAPITAL LETTER SHHACYRILLIC SMALL LETTER SHHACYRILLIC CAPITAL LETTER ABKHASIAN CHECYRILLIC SMALL LETTER ABKHASIAN CHECYRILLIC CAPITAL LETTER ABKHASIAN CHE WITH DESCENDERCYRILLIC SMALL LETTER ABKHASIAN CHE WITH DESCENDERCYRILLIC L\
    \ETTER PALOCHKACYRILLIC CAPITAL LETTER ZHE WITH BREVECYRILLIC SMALL LETTER ZHE WITH BREVECYRILLIC CAPITAL LETTER KA WITH HOOKCYRILLIC SMALL LETTER KA WITH HOOKCYRILLIC CAPITAL LETTER EL WITH TAILCYRILLIC SMALL LETTER EL WITH TAILCYRILLIC CAPITAL LETTER EN \
    \WITH HOOKCYRILLIC SMALL LETTER EN WITH HOOKCYRILLIC CAPITAL LETTER EN WITH TAILCYRILLIC SMALL LETTER EN WITH TAILCYRILLIC CAPITAL LETTER KHAKASSIAN CHECYRILLIC SMALL LETTER KHAKASSIAN CHECYRILLIC CAPITAL LETTER EM WITH TAILCYRILLIC SMALL LETTER EM WITH TA\
    \ILCYRILLIC SMALL LETTER PALOCHKACYRILLIC CAPITAL LETTER A WITH BREVECYRILLIC SMALL LETTER A WITH BREVECYRILLIC CAPITAL LETTER A WITH DIAERESISCYRILLIC SMALL LETTER A WITH DIAERESISCYRILLIC CAPITAL LIGATURE A IECYRILLIC SMALL LIGATURE A IECYRILLIC CAPITAL \
    \LETTER IE WITH BREVECYRILLIC SMALL LETTER IE WITH BREVECYRILLIC CAPITAL LETTER SCHWACYRILLIC SMALL LETTER SCHWACYRILLIC CAPITAL LETTER SCHWA WITH DIAERESISCYRILLIC SMALL LETTER SCHWA WITH DIAERESISCYRILLIC CAPITAL LETTER ZHE WITH DIAERESISCYRILLIC SMALL L\
    \ETTER ZHE WITH DIAERESISCYRILLIC CAPITAL LETTER ZE WITH DIAERESISCYRILLIC SMALL LETTER ZE WITH DIAERESISCYRILLIC CAPITAL LETTER ABKHASIAN DZECYRILLIC SMALL LETTER ABKHASIAN DZECYRILLIC CAPITAL LETTER I WITH MACRONCYRILLIC SMALL LETTER I WITH MACRONCYRILLI\
    \C CAPITAL LETTER I WITH DIAERESISCYRILLIC SMALL LETTER I WITH DIAERESISCYRILLIC CAPITAL LETTER O WITH DIAERESISCYRILLIC SMALL LETTER O WITH DIAERESISCYRILLIC CAPITAL LETTER BARRED OCYRILLIC SMALL LETTER BARRED OCYRILLIC CAPITAL LETTER BARRED O WITH DIAERE\
    \SISCYRILLIC SMALL LETTER BARRED O WITH DIAERESISCYRILLIC CAPITAL LETTER E WITH DIAERESISCYRILLIC SMALL LETTER E WITH DIAERESISCYRILLIC CAPITAL LETTER U WITH MACRONCYRILLIC SMALL LETTER U WITH MACRONCYRILLIC CAPITAL LETTER U WITH DIAERESISCYRILLIC SMALL LE\
    \TTER U WITH DIAERESISCYRILLIC CAPITAL LETTER U WITH DOUBLE ACUTECYRILLIC SMALL LETTER U WITH DOUBLE ACUTECYRILLIC CAPITAL LETTER CHE WITH DIAERESISCYRILLIC SMALL LETTER CHE WITH DIAERESISCYRILLIC CAPITAL LETTER GHE WITH DESCENDERCYRILLIC SMALL LETTER GHE \
    \WITH DESCENDERCYRILLIC CAPITAL LETTER YERU WITH DIAERESISCYRILLIC SMALL LETTER YERU WITH DIAERESISCYRILLIC CAPITAL LETTER GHE WITH STROKE AND HOOKCYRILLIC SMALL LETTER GHE WITH STROKE AND HOOKCYRILLIC CAPITAL LETTER HA WITH HOOKCYRILLIC SMALL LETTER HA WI\
    \TH HOOKCYRILLIC CAPITAL LETTER HA WITH STROKECYRILLIC SMALL LETTER HA WITH STROKECYRILLIC CAPITAL LETTER KOMI DECYRILLIC SMALL LETTER KOMI DECYRILLIC CAPITAL LETTER KOMI DJECYRILLIC SMALL LETTER KOMI DJECYRILLIC CAPITAL LETTER KOMI ZJECYRILLIC SMALL LETTE\
    \R KOMI ZJECYRILLIC CAPITAL LETTER KOMI DZJECYRILLIC SMALL LETTER KOMI DZJECYRILLIC CAPITAL LETTER KOMI LJECYRILLIC SMALL LETTER KOMI LJECYRILLIC CAPITAL LETTER KOMI NJECYRILLIC SMALL LETTER KOMI NJECYRILLIC CAPITAL LETTER KOMI SJECYRILLIC SMALL LETTER KOM\
    \I SJECYRILLIC CAPITAL LETTER KOMI TJECYRILLIC SMALL LETTER KOMI TJECYRILLIC CAPITAL LETTER REVERSED ZECYRILLIC SMALL LETTER REVERSED ZECYRILLIC CAPITAL LETTER EL WITH HOOKCYRILLIC SMALL LETTER EL WITH HOOKCYRILLIC CAPITAL LETTER LHACYRILLIC SMALL LETTER L\
    \HACYRILLIC CAPITAL LETTER RHACYRILLIC SMALL LETTER RHACYRILLIC CAPITAL LETTER YAECYRILLIC SMALL LETTER YAECYRILLIC CAPITAL LETTER QACYRILLIC SMALL LETTER QACYRILLIC CAPITAL LETTER WECYRILLIC SMALL LETTER WECYRILLIC CAPITAL LETTER ALEUT KACYRILLIC SMALL LE\
    \TTER ALEUT KACYRILLIC CAPITAL LETTER EL WITH MIDDLE HOOKCYRILLIC SMALL LETTER EL WITH MIDDLE HOOKCYRILLIC CAPITAL LETTER EN WITH MIDDLE HOOKCYRILLIC SMALL LETTER EN WITH MIDDLE HOOKCYRILLIC CAPITAL LETTER PE WITH DESCENDERCYRILLIC SMALL LETTER PE WITH DES\
    \CENDERCYRILLIC CAPITAL LETTER SHHA WITH DESCENDERCYRILLIC SMALL LETTER SHHA WITH DESCENDERCYRILLIC CAPITAL LETTER EN WITH LEFT HOOKCYRILLIC SMALL LETTER EN WITH LEFT HOOKCYRILLIC CAPITAL LETTER DZZHECYRILLIC SMALL LETTER DZZHECYRILLIC CAPITAL LETTER DCHEC\
    \YRILLIC SMALL LETTER DCHECYRILLIC CAPITAL LETTER EL WITH DESCENDERCYRILLIC SMALL LETTER EL WITH DESCENDERARMENIAN CAPITAL LETTER AYBARMENIAN CAPITAL LETTER BENARMENIAN CAPITAL LETTER GIMARMENIAN CAPITAL LETTER DAARMENIAN CAPITAL LETTER ECHARMENIAN CAPITAL\
    \ LETTER ZAARMENIAN CAPITAL LETTER EHARMENIAN CAPITAL LETTER ETARMENIAN CAPITAL LETTER TOARMENIAN CAPITAL LETTER ZHEARMENIAN CAPITAL LETTER INIARMENIAN CAPITAL LETTER LIWNARMENIAN CAPITAL LETTER XEHARMENIAN CAPITAL LETTER CAARMENIAN CAPITAL LETTER KENARMEN\
    \IAN CAPITAL LETTER HOARMENIAN CAPITAL LETTER JAARMENIAN CAPITAL LETTER GHADARMENIAN CAPITAL LETTER CHEHARMENIAN CAPITAL LETTER MENARMENIAN CAPITAL LETTER YIARMENIAN CAPITAL LETTER NOWARMENIAN CAPITAL LETTER SHAARMENIAN CAPITAL LETTER VOARMENIAN CAPITAL LE\
    \TTER CHAARMENIAN CAPITAL LETTER PEHARMENIAN CAPITAL LETTER JHEHARMENIAN CAPITAL LETTER RAARMENIAN CAPITAL LETTER SEHARMENIAN CAPITAL LETTER VEWARMENIAN CAPITAL LETTER TIWNARMENIAN CAPITAL LETTER REHARMENIAN CAPITAL LETTER COARMENIAN CAPITAL LETTER YIWNARM\
    \ENIAN CAPITAL LETTER PIWRARMENIAN CAPITAL LETTER KEHARMENIAN CAPITAL LETTER OHARMENIAN CAPITAL LETTER FEHARMENIAN MODIFIER LETTER LEFT HALF RINGARMENIAN APOSTROPHEARMENIAN EMPHASIS MARKARMENIAN EXCLAMATION MARKARMENIAN COMMAARMENIAN QUESTION MARKARMENIAN \
    \ABBREVIATION MARKARMENIAN SMALL LETTER TURNED AYBARMENIAN SMALL LETTER AYBARMENIAN SMALL LETTER BENARMENIAN SMALL LETTER GIMARMENIAN SMALL LETTER DAARMENIAN SMALL LETTER ECHARMENIAN SMALL LETTER ZAARMENIAN SMALL LETTER EHARMENIAN SMALL LETTER ETARMENIAN S\
    \MALL LETTER TOARMENIAN SMALL LETTER ZHEARMENIAN SMALL LETTER INIARMENIAN SMALL LETTER LIWNARMENIAN SMALL LETTER XEHARMENIAN SMALL LETTER CAARMENIAN SMALL LETTER KENARMENIAN SMALL LETTER HOARMENIAN SMALL LETTER JAARMENIAN SMALL LETTER GHADARMENIAN SMALL LE\
    \TTER CHEHARMENIAN SMALL LETTER MENARMENIAN SMALL LETTER YIARMENIAN SMALL LETTER NOWARMENIAN SMALL LETTER SHAARMENIAN SMALL LETTER VOARMENIAN SMALL LETTER CHAARMENIAN SMALL LETTER PEHARMENIAN SMALL LETTER JHEHARMENIAN SMALL LETTER RAARMENIAN SMALL LETTER S\
    \EHARMENIAN SMALL LETTER VEWARMENIAN SMALL LETTER TIWNARMENIAN SMALL LETTER REHARMENIAN SMALL LETTER COARMENIAN SMALL LETTER YIWNARMENIAN SMALL LETTER PIWRARMENIAN SMALL LETTER KEHARMENIAN SMALL LETTER OHARMENIAN SMALL LETTER FEHARMENIAN SMALL LIGATURE ECH\
    \ YIWNARMENIAN SMALL LETTER YI WITH STROKEARMENIAN FULL STOPARMENIAN HYPHENRIGHT-FACING ARMENIAN ETERNITY SIGNLEFT-FACING ARMENIAN ETERNITY SIGNARMENIAN DRAM SIGNHEBREW ACCENT ETNAHTAHEBREW ACCENT SEGOLHEBREW ACCENT SHALSHELETHEBREW ACCENT ZAQEF QATANHEBRE\
    \W ACCENT ZAQEF GADOLHEBREW ACCENT TIPEHAHEBREW ACCENT REVIAHEBREW ACCENT ZARQAHEBREW ACCENT PASHTAHEBREW ACCENT YETIVHEBREW ACCENT TEVIRHEBREW ACCENT GERESHHEBREW ACCENT GERESH MUQDAMHEBREW ACCENT GERSHAYIMHEBREW ACCENT QARNEY PARAHEBREW ACCENT TELISHA GE\
    \DOLAHEBREW ACCENT PAZERHEBREW ACCENT ATNAH HAFUKHHEBREW ACCENT MUNAHHEBREW ACCENT MAHAPAKHHEBREW ACCENT MERKHAHEBREW ACCENT MERKHA KEFULAHEBREW ACCENT DARGAHEBREW ACCENT QADMAHEBREW ACCENT TELISHA QETANAHEBREW ACCENT YERAH BEN YOMOHEBREW ACCENT OLEHEBREW \
    \ACCENT ILUYHEBREW ACCENT DEHIHEBREW ACCENT ZINORHEBREW MARK MASORA CIRCLEHEBREW POINT SHEVAHEBREW POINT HATAF SEGOLHEBREW POINT HATAF PATAHHEBREW POINT HATAF QAMATSHEBREW POINT HIRIQHEBREW POINT TSEREHEBREW POINT SEGOLHEBREW POINT PATAHHEBREW POINT QAMATS\
    \HEBREW POINT HOLAMHEBREW POINT HOLAM HASER FOR VAVHEBREW POINT QUBUTSHEBREW POINT DAGESH OR MAPIQHEBREW POINT METEGHEBREW PUNCTUATION MAQAFHEBREW POINT RAFEHEBREW PUNCTUATION PASEQHEBREW POINT SHIN DOTHEBREW POINT SIN DOTHEBREW PUNCTUATION SOF PASUQHEBREW\
    \ MARK UPPER DOTHEBREW MARK LOWER DOTHEBREW PUNCTUATION NUN HAFUKHAHEBREW POINT QAMATS QATANHEBREW LETTER ALEFHEBREW LETTER BETHEBREW LETTER GIMELHEBREW LETTER DALETHEBREW LETTER HEHEBREW LETTER VAVHEBREW LETTER ZAYINHEBREW LETTER HETHEBREW LETTER TETHEBRE\
    \W LETTER YODHEBREW LETTER FINAL KAFHEBREW LETTER KAFHEBREW LETTER LAMEDHEBREW LETTER FINAL MEMHEBREW LETTER MEMHEBREW LETTER FINAL NUNHEBREW LETTER NUNHEBREW LETTER SAMEKHHEBREW LETTER AYINHEBREW LETTER FINAL PEHEBREW LETTER PEHEBREW LETTER FINAL TSADIHEB\
    \REW LETTER TSADIHEBREW LETTER QOFHEBREW LETTER RESHHEBREW LETTER SHINHEBREW LETTER TAVHEBREW YOD TRIANGLEHEBREW LIGATURE YIDDISH DOUBLE VAVHEBREW LIGATURE YIDDISH VAV YODHEBREW LIGATURE YIDDISH DOUBLE YODHEBREW PUNCTUATION GERESHHEBREW PUNCTUATION GERSHAY\
    \IMARABIC NUMBER SIGNARABIC SIGN SANAHARABIC FOOTNOTE MARKERARABIC SIGN SAFHAARABIC SIGN SAMVATARABIC NUMBER MARK ABOVEARABIC-INDIC CUBE ROOTARABIC-INDIC FOURTH ROOTARABIC RAYARABIC-INDIC PER MILLE SIGNARABIC-INDIC PER TEN THOUSAND SIGNAFGHANI SIGNARABIC C\
    \OMMAARABIC DATE SEPARATORARABIC POETIC VERSE SIGNARABIC SIGN MISRAARABIC SIGN SALLALLAHOU ALAYHE WASSALLAMARABIC SIGN ALAYHE ASSALLAMARABIC SIGN RAHMATULLAH ALAYHEARABIC SIGN RADI ALLAHOU ANHUARABIC SIGN TAKHALLUSARABIC SMALL HIGH TAHARABIC SMALL HIGH LIG\
    \ATURE ALEF WITH LAM WITH YEHARABIC SMALL HIGH ZAINARABIC SMALL FATHAARABIC SMALL DAMMAARABIC SMALL KASRAARABIC SEMICOLONARABIC LETTER MARKARABIC END OF TEXT MARKARABIC TRIPLE DOT PUNCTUATION MARKARABIC QUESTION MARKARABIC LETTER KASHMIRI YEHARABIC LETTER \
    \HAMZAARABIC LETTER ALEF WITH MADDA ABOVEARABIC LETTER ALEF WITH HAMZA ABOVEARABIC LETTER WAW WITH HAMZA ABOVEARABIC LETTER ALEF WITH HAMZA BELOWARABIC LETTER YEH WITH HAMZA ABOVEARABIC LETTER ALEFARABIC LETTER BEHARABIC LETTER TEH MARBUTAARABIC LETTER TEH\
    \ARABIC LETTER THEHARABIC LETTER JEEMARABIC LETTER HAHARABIC LETTER KHAHARABIC LETTER DALARABIC LETTER THALARABIC LETTER REHARABIC LETTER ZAINARABIC LETTER SEENARABIC LETTER SHEENARABIC LETTER SADARABIC LETTER DADARABIC LETTER TAHARABIC LETTER ZAHARABIC LE\
    \TTER AINARABIC LETTER GHAINARABIC LETTER KEHEH WITH TWO DOTS ABOVEARABIC LETTER KEHEH WITH THREE DOTS BELOWARABIC LETTER FARSI YEH WITH INVERTED VARABIC LETTER FARSI YEH WITH TWO DOTS ABOVEARABIC LETTER FARSI YEH WITH THREE DOTS ABOVEARABIC TATWEELARABIC \
    \LETTER FEHARABIC LETTER QAFARABIC LETTER KAFARABIC LETTER LAMARABIC LETTER MEEMARABIC LETTER NOONARABIC LETTER HEHARABIC LETTER WAWARABIC LETTER ALEF MAKSURAARABIC LETTER YEHARABIC FATHATANARABIC DAMMATANARABIC KASRATANARABIC FATHAARABIC DAMMAARABIC KASRA\
    \ARABIC SHADDAARABIC SUKUNARABIC MADDAH ABOVEARABIC HAMZA ABOVEARABIC HAMZA BELOWARABIC SUBSCRIPT ALEFARABIC INVERTED DAMMAARABIC MARK NOON GHUNNAARABIC ZWARAKAYARABIC VOWEL SIGN SMALL V ABOVEARABIC VOWEL SIGN INVERTED SMALL V ABOVEARABIC VOWEL SIGN DOT BE\
    \LOWARABIC REVERSED DAMMAARABIC FATHA WITH TWO DOTSARABIC WAVY HAMZA BELOWARABIC-INDIC DIGIT ZEROARABIC-INDIC DIGIT ONEARABIC-INDIC DIGIT TWOARABIC-INDIC DIGIT THREEARABIC-INDIC DIGIT FOURARABIC-INDIC DIGIT FIVEARABIC-INDIC DIGIT SIXARABIC-INDIC DIGIT SEVE\
    \NARABIC-INDIC DIGIT EIGHTARABIC-INDIC DIGIT NINEARABIC PERCENT SIGNARABIC DECIMAL SEPARATORARABIC THOUSANDS SEPARATORARABIC FIVE POINTED STARARABIC LETTER DOTLESS BEHARABIC LETTER DOTLESS QAFARABIC LETTER SUPERSCRIPT ALEFARABIC LETTER ALEF WASLAARABIC LET\
    \TER ALEF WITH WAVY HAMZA ABOVEARABIC LETTER ALEF WITH WAVY HAMZA BELOWARABIC LETTER HIGH HAMZAARABIC LETTER HIGH HAMZA ALEFARABIC LETTER HIGH HAMZA WAWARABIC LETTER U WITH HAMZA ABOVEARABIC LETTER HIGH HAMZA YEHARABIC LETTER TTEHARABIC LETTER TTEHEHARABIC\
    \ LETTER BEEHARABIC LETTER TEH WITH RINGARABIC LETTER TEH WITH THREE DOTS ABOVE DOWNWARDSARABIC LETTER PEHARABIC LETTER TEHEHARABIC LETTER BEHEHARABIC LETTER HAH WITH HAMZA ABOVEARABIC LETTER HAH WITH TWO DOTS VERTICAL ABOVEARABIC LETTER NYEHARABIC LETTER \
    \DYEHARABIC LETTER HAH WITH THREE DOTS ABOVEARABIC LETTER TCHEHARABIC LETTER TCHEHEHARABIC LETTER DDALARABIC LETTER DAL WITH RINGARABIC LETTER DAL WITH DOT BELOWARABIC LETTER DAL WITH DOT BELOW AND SMALL TAHARABIC LETTER DAHALARABIC LETTER DDAHALARABIC LET\
    \TER DULARABIC LETTER DAL WITH THREE DOTS ABOVE DOWNWARDSARABIC LETTER DAL WITH FOUR DOTS ABOVEARABIC LETTER RREHARABIC LETTER REH WITH SMALL VARABIC LETTER REH WITH RINGARABIC LETTER REH WITH DOT BELOWARABIC LETTER REH WITH SMALL V BELOWARABIC LETTER REH \
    \WITH DOT BELOW AND DOT ABOVEARABIC LETTER REH WITH TWO DOTS ABOVEARABIC LETTER JEHARABIC LETTER REH WITH FOUR DOTS ABOVEARABIC LETTER SEEN WITH DOT BELOW AND DOT ABOVEARABIC LETTER SEEN WITH THREE DOTS BELOWARABIC LETTER SEEN WITH THREE DOTS BELOW AND THR\
    \EE DOTS ABOVEARABIC LETTER SAD WITH TWO DOTS BELOWARABIC LETTER SAD WITH THREE DOTS ABOVEARABIC LETTER TAH WITH THREE DOTS ABOVEARABIC LETTER AIN WITH THREE DOTS ABOVEARABIC LETTER DOTLESS FEHARABIC LETTER FEH WITH DOT MOVED BELOWARABIC LETTER FEH WITH DO\
    \T BELOWARABIC LETTER VEHARABIC LETTER FEH WITH THREE DOTS BELOWARABIC LETTER PEHEHARABIC LETTER QAF WITH DOT ABOVEARABIC LETTER QAF WITH THREE DOTS ABOVEARABIC LETTER KEHEHARABIC LETTER SWASH KAFARABIC LETTER KAF WITH RINGARABIC LETTER KAF WITH DOT ABOVEA\
    \RABIC LETTER NGARABIC LETTER KAF WITH THREE DOTS BELOWARABIC LETTER GAFARABIC LETTER GAF WITH RINGARABIC LETTER NGOEHARABIC LETTER GAF WITH TWO DOTS BELOWARABIC LETTER GUEHARABIC LETTER GAF WITH THREE DOTS ABOVEARABIC LETTER LAM WITH SMALL VARABIC LETTER \
    \LAM WITH DOT ABOVEARABIC LETTER LAM WITH THREE DOTS ABOVEARABIC LETTER LAM WITH THREE DOTS BELOWARABIC LETTER NOON WITH DOT BELOWARABIC LETTER NOON GHUNNAARABIC LETTER RNOONARABIC LETTER NOON WITH RINGARABIC LETTER NOON WITH THREE DOTS ABOVEARABIC LETTER \
    \HEH DOACHASHMEEARABIC LETTER TCHEH WITH DOT ABOVEARABIC LETTER HEH WITH YEH ABOVEARABIC LETTER HEH GOALARABIC LETTER HEH GOAL WITH HAMZA ABOVEARABIC LETTER TEH MARBUTA GOALARABIC LETTER WAW WITH RINGARABIC LETTER KIRGHIZ OEARABIC LETTER OEARABIC LETTER UA\
    \RABIC LETTER YUARABIC LETTER KIRGHIZ YUARABIC LETTER WAW WITH TWO DOTS ABOVEARABIC LETTER VEARABIC LETTER FARSI YEHARABIC LETTER YEH WITH TAILARABIC LETTER YEH WITH SMALL VARABIC LETTER WAW WITH DOT ABOVEARABIC LETTER EARABIC LETTER YEH WITH THREE DOTS BE\
    \LOWARABIC LETTER YEH BARREEARABIC LETTER YEH BARREE WITH HAMZA ABOVEARABIC FULL STOPARABIC LETTER AEARABIC SMALL HIGH LIGATURE SAD WITH LAM WITH ALEF MAKSURAARABIC SMALL HIGH LIGATURE QAF WITH LAM WITH ALEF MAKSURAARABIC SMALL HIGH MEEM INITIAL FORMARABIC\
    \ SMALL HIGH LAM ALEFARABIC SMALL HIGH JEEMARABIC SMALL HIGH THREE DOTSARABIC SMALL HIGH SEENARABIC END OF AYAHARABIC START OF RUB EL HIZBARABIC SMALL HIGH ROUNDED ZEROARABIC SMALL HIGH UPRIGHT RECTANGULAR ZEROARABIC SMALL HIGH DOTLESS HEAD OF KHAHARABIC S\
    \MALL HIGH MEEM ISOLATED FORMARABIC SMALL LOW SEENARABIC SMALL HIGH MADDAARABIC SMALL WAWARABIC SMALL YEHARABIC SMALL HIGH YEHARABIC SMALL HIGH NOONARABIC PLACE OF SAJDAHARABIC EMPTY CENTRE LOW STOPARABIC EMPTY CENTRE HIGH STOPARABIC ROUNDED HIGH STOP WITH\
    \ FILLED CENTREARABIC SMALL LOW MEEMARABIC LETTER DAL WITH INVERTED VARABIC LETTER REH WITH INVERTED VEXTENDED ARABIC-INDIC DIGIT ZEROEXTENDED ARABIC-INDIC DIGIT ONEEXTENDED ARABIC-INDIC DIGIT TWOEXTENDED ARABIC-INDIC DIGIT THREEEXTENDED ARABIC-INDIC DIGIT\
    \ FOUREXTENDED ARABIC-INDIC DIGIT FIVEEXTENDED ARABIC-INDIC DIGIT SIXEXTENDED ARABIC-INDIC DIGIT SEVENEXTENDED ARABIC-INDIC DIGIT EIGHTEXTENDED ARABIC-INDIC DIGIT NINEARABIC LETTER SHEEN WITH DOT BELOWARABIC LETTER DAD WITH DOT BELOWARABIC LETTER GHAIN WIT\
    \H DOT BELOWARABIC SIGN SINDHI AMPERSANDARABIC SIGN SINDHI POSTPOSITION MENARABIC LETTER HEH WITH INVERTED VSYRIAC END OF PARAGRAPHSYRIAC SUPRALINEAR FULL STOPSYRIAC SUBLINEAR FULL STOPSYRIAC SUPRALINEAR COLONSYRIAC SUBLINEAR COLONSYRIAC HORIZONTAL COLONSY\
    \RIAC COLON SKEWED LEFTSYRIAC COLON SKEWED RIGHTSYRIAC SUPRALINEAR COLON SKEWED LEFTSYRIAC SUBLINEAR COLON SKEWED RIGHTSYRIAC CONTRACTIONSYRIAC HARKLEAN OBELUSSYRIAC HARKLEAN METOBELUSSYRIAC HARKLEAN ASTERISCUSSYRIAC ABBREVIATION MARKSYRIAC LETTER ALAPHSYR\
    \IAC LETTER SUPERSCRIPT ALAPHSYRIAC LETTER BETHSYRIAC LETTER GAMALSYRIAC LETTER GAMAL GARSHUNISYRIAC LETTER DALATHSYRIAC LETTER DOTLESS DALATH RISHSYRIAC LETTER HESYRIAC LETTER WAWSYRIAC LETTER ZAINSYRIAC LETTER HETHSYRIAC LETTER TETHSYRIAC LETTER TETH GAR\
    \SHUNISYRIAC LETTER YUDHSYRIAC LETTER YUDH HESYRIAC LETTER KAPHSYRIAC LETTER LAMADHSYRIAC LETTER MIMSYRIAC LETTER NUNSYRIAC LETTER SEMKATHSYRIAC LETTER FINAL SEMKATHSYRIAC LETTER ESYRIAC LETTER PESYRIAC LETTER REVERSED PESYRIAC LETTER SADHESYRIAC LETTER QA\
    \PHSYRIAC LETTER RISHSYRIAC LETTER SHINSYRIAC LETTER TAWSYRIAC LETTER PERSIAN BHETHSYRIAC LETTER PERSIAN GHAMALSYRIAC LETTER PERSIAN DHALATHSYRIAC PTHAHA ABOVESYRIAC PTHAHA BELOWSYRIAC PTHAHA DOTTEDSYRIAC ZQAPHA ABOVESYRIAC ZQAPHA BELOWSYRIAC ZQAPHA DOTTED\
    \SYRIAC RBASA ABOVESYRIAC RBASA BELOWSYRIAC DOTTED ZLAMA HORIZONTALSYRIAC DOTTED ZLAMA ANGULARSYRIAC HBASA ABOVESYRIAC HBASA BELOWSYRIAC HBASA-ESASA DOTTEDSYRIAC ESASA ABOVESYRIAC ESASA BELOWSYRIAC RWAHASYRIAC FEMININE DOTSYRIAC QUSHSHAYASYRIAC RUKKAKHASYR\
    \IAC TWO VERTICAL DOTS ABOVESYRIAC TWO VERTICAL DOTS BELOWSYRIAC THREE DOTS ABOVESYRIAC THREE DOTS BELOWSYRIAC OBLIQUE LINE ABOVESYRIAC OBLIQUE LINE BELOWSYRIAC MUSICSYRIAC BARREKHSYRIAC LETTER SOGDIAN ZHAINSYRIAC LETTER SOGDIAN KHAPHSYRIAC LETTER SOGDIAN \
    \FEARABIC LETTER BEH WITH THREE DOTS HORIZONTALLY BELOWARABIC LETTER BEH WITH DOT BELOW AND THREE DOTS ABOVEARABIC LETTER BEH WITH THREE DOTS POINTING UPWARDS BELOWARABIC LETTER BEH WITH THREE DOTS POINTING UPWARDS BELOW AND TWO DOTS ABOVEARABIC LETTER BEH\
    \ WITH TWO DOTS BELOW AND DOT ABOVEARABIC LETTER BEH WITH INVERTED SMALL V BELOWARABIC LETTER BEH WITH SMALL VARABIC LETTER HAH WITH TWO DOTS ABOVEARABIC LETTER HAH WITH THREE DOTS POINTING UPWARDS BELOWARABIC LETTER DAL WITH TWO DOTS VERTICALLY BELOW AND \
    \SMALL TAHARABIC LETTER DAL WITH INVERTED SMALL V BELOWARABIC LETTER REH WITH STROKEARABIC LETTER SEEN WITH FOUR DOTS ABOVEARABIC LETTER AIN WITH TWO DOTS ABOVEARABIC LETTER AIN WITH THREE DOTS POINTING DOWNWARDS ABOVEARABIC LETTER AIN WITH TWO DOTS VERTIC\
    \ALLY ABOVEARABIC LETTER FEH WITH TWO DOTS BELOWARABIC LETTER FEH WITH THREE DOTS POINTING UPWARDS BELOWARABIC LETTER KEHEH WITH DOT ABOVEARABIC LETTER KEHEH WITH THREE DOTS ABOVEARABIC LETTER KEHEH WITH THREE DOTS POINTING UPWARDS BELOWARABIC LETTER MEEM \
    \WITH DOT ABOVEARABIC LETTER MEEM WITH DOT BELOWARABIC LETTER NOON WITH TWO DOTS BELOWARABIC LETTER NOON WITH SMALL TAHARABIC LETTER NOON WITH SMALL VARABIC LETTER LAM WITH BARARABIC LETTER REH WITH TWO DOTS VERTICALLY ABOVEARABIC LETTER REH WITH HAMZA ABO\
    \VEARABIC LETTER SEEN WITH TWO DOTS VERTICALLY ABOVEARABIC LETTER HAH WITH SMALL ARABIC LETTER TAH BELOWARABIC LETTER HAH WITH SMALL ARABIC LETTER TAH AND TWO DOTSARABIC LETTER SEEN WITH SMALL ARABIC LETTER TAH AND TWO DOTSARABIC LETTER REH WITH SMALL ARAB\
    \IC LETTER TAH AND TWO DOTSARABIC LETTER HAH WITH SMALL ARABIC LETTER TAH ABOVEARABIC LETTER ALEF WITH EXTENDED ARABIC-INDIC DIGIT TWO ABOVEARABIC LETTER ALEF WITH EXTENDED ARABIC-INDIC DIGIT THREE ABOVEARABIC LETTER FARSI YEH WITH EXTENDED ARABIC-INDIC DI\
    \GIT TWO ABOVEARABIC LETTER FARSI YEH WITH EXTENDED ARABIC-INDIC DIGIT THREE ABOVEARABIC LETTER FARSI YEH WITH EXTENDED ARABIC-INDIC DIGIT FOUR BELOWARABIC LETTER WAW WITH EXTENDED ARABIC-INDIC DIGIT TWO ABOVEARABIC LETTER WAW WITH EXTENDED ARABIC-INDIC DI\
    \GIT THREE ABOVEARABIC LETTER YEH BARREE WITH EXTENDED ARABIC-INDIC DIGIT TWO ABOVEARABIC LETTER YEH BARREE WITH EXTENDED ARABIC-INDIC DIGIT THREE ABOVEARABIC LETTER HAH WITH EXTENDED ARABIC-INDIC DIGIT FOUR BELOWARABIC LETTER SEEN WITH EXTENDED ARABIC-IND\
    \IC DIGIT FOUR ABOVEARABIC LETTER SEEN WITH INVERTED VARABIC LETTER KAF WITH TWO DOTS ABOVETHAANA LETTER HAATHAANA LETTER SHAVIYANITHAANA LETTER NOONUTHAANA LETTER RAATHAANA LETTER BAATHAANA LETTER LHAVIYANITHAANA LETTER KAAFUTHAANA LETTER ALIFUTHAANA LETT\
    \ER VAAVUTHAANA LETTER MEEMUTHAANA LETTER FAAFUTHAANA LETTER DHAALUTHAANA LETTER THAATHAANA LETTER LAAMUTHAANA LETTER GAAFUTHAANA LETTER GNAVIYANITHAANA LETTER SEENUTHAANA LETTER DAVIYANITHAANA LETTER ZAVIYANITHAANA LETTER TAVIYANITHAANA LETTER YAATHAANA L\
    \ETTER PAVIYANITHAANA LETTER JAVIYANITHAANA LETTER CHAVIYANITHAANA LETTER TTAATHAANA LETTER HHAATHAANA LETTER KHAATHAANA LETTER THAALUTHAANA LETTER ZAATHAANA LETTER SHEENUTHAANA LETTER SAADHUTHAANA LETTER DAADHUTHAANA LETTER TOTHAANA LETTER ZOTHAANA LETTER\
    \ AINUTHAANA LETTER GHAINUTHAANA LETTER QAAFUTHAANA LETTER WAAVUTHAANA ABAFILITHAANA AABAAFILITHAANA IBIFILITHAANA EEBEEFILITHAANA UBUFILITHAANA OOBOOFILITHAANA EBEFILITHAANA EYBEYFILITHAANA OBOFILITHAANA OABOAFILITHAANA SUKUNTHAANA LETTER NAANKO DIGIT ZER\
    \ONKO DIGIT ONENKO DIGIT TWONKO DIGIT THREENKO DIGIT FOURNKO DIGIT FIVENKO DIGIT SIXNKO DIGIT SEVENNKO DIGIT EIGHTNKO DIGIT NINENKO LETTER ANKO LETTER EENKO LETTER INKO LETTER ENKO LETTER UNKO LETTER OONKO LETTER ONKO LETTER DAGBASINNANKO LETTER NNKO LETTE\
    \R BANKO LETTER PANKO LETTER TANKO LETTER JANKO LETTER CHANKO LETTER DANKO LETTER RANKO LETTER RRANKO LETTER SANKO LETTER GBANKO LETTER FANKO LETTER KANKO LETTER LANKO LETTER NA WOLOSONKO LETTER MANKO LETTER NYANKO LETTER NANKO LETTER HANKO LETTER WANKO LE\
    \TTER YANKO LETTER NYA WOLOSONKO LETTER JONA JANKO LETTER JONA CHANKO LETTER JONA RANKO COMBINING SHORT HIGH TONENKO COMBINING SHORT LOW TONENKO COMBINING SHORT RISING TONENKO COMBINING LONG DESCENDING TONENKO COMBINING LONG HIGH TONENKO COMBINING LONG LOW\
    \ TONENKO COMBINING LONG RISING TONENKO COMBINING NASALIZATION MARKNKO COMBINING DOUBLE DOT ABOVENKO HIGH TONE APOSTROPHENKO LOW TONE APOSTROPHENKO SYMBOL OO DENNENNKO SYMBOL GBAKURUNENNKO COMMANKO EXCLAMATION MARKNKO LAJANYALANNKO DANTAYALANNKO DOROME SIG\
    \NNKO TAMAN SIGNSAMARITAN LETTER ALAFSAMARITAN LETTER BITSAMARITAN LETTER GAMANSAMARITAN LETTER DALATSAMARITAN LETTER IYSAMARITAN LETTER BAASAMARITAN LETTER ZENSAMARITAN LETTER ITSAMARITAN LETTER TITSAMARITAN LETTER YUTSAMARITAN LETTER KAAFSAMARITAN LETTER\
    \ LABATSAMARITAN LETTER MIMSAMARITAN LETTER NUNSAMARITAN LETTER SINGAATSAMARITAN LETTER INSAMARITAN LETTER FISAMARITAN LETTER TSAADIYSAMARITAN LETTER QUFSAMARITAN LETTER RISHSAMARITAN LETTER SHANSAMARITAN LETTER TAAFSAMARITAN MARK INSAMARITAN MARK IN-ALAFS\
    \AMARITAN MARK OCCLUSIONSAMARITAN MARK DAGESHSAMARITAN MODIFIER LETTER EPENTHETIC YUTSAMARITAN MARK EPENTHETIC YUTSAMARITAN VOWEL SIGN LONG ESAMARITAN VOWEL SIGN ESAMARITAN VOWEL SIGN OVERLONG AASAMARITAN VOWEL SIGN LONG AASAMARITAN VOWEL SIGN AASAMARITAN \
    \VOWEL SIGN OVERLONG ASAMARITAN VOWEL SIGN LONG ASAMARITAN VOWEL SIGN ASAMARITAN MODIFIER LETTER SHORT ASAMARITAN VOWEL SIGN SHORT ASAMARITAN VOWEL SIGN LONG USAMARITAN VOWEL SIGN USAMARITAN MODIFIER LETTER ISAMARITAN VOWEL SIGN LONG ISAMARITAN VOWEL SIGN \
    \ISAMARITAN VOWEL SIGN OSAMARITAN VOWEL SIGN SUKUNSAMARITAN MARK NEQUDAASAMARITAN PUNCTUATION NEQUDAASAMARITAN PUNCTUATION AFSAAQSAMARITAN PUNCTUATION ANGEDSAMARITAN PUNCTUATION BAUSAMARITAN PUNCTUATION ATMAAUSAMARITAN PUNCTUATION SHIYYAALAASAMARITAN ABBRE\
    \VIATION MARKSAMARITAN PUNCTUATION MELODIC QITSASAMARITAN PUNCTUATION ZIQAASAMARITAN PUNCTUATION QITSASAMARITAN PUNCTUATION ZAEFSAMARITAN PUNCTUATION TURUSAMARITAN PUNCTUATION ARKAANUSAMARITAN PUNCTUATION SOF MASHFAATSAMARITAN PUNCTUATION ANNAAUMANDAIC LET\
    \TER HALQAMANDAIC LETTER ABMANDAIC LETTER AGMANDAIC LETTER ADMANDAIC LETTER AHMANDAIC LETTER USHENNAMANDAIC LETTER AZMANDAIC LETTER ITMANDAIC LETTER ATTMANDAIC LETTER AKSAMANDAIC LETTER AKMANDAIC LETTER ALMANDAIC LETTER AMMANDAIC LETTER ANMANDAIC LETTER AS\
    \MANDAIC LETTER INMANDAIC LETTER APMANDAIC LETTER ASZMANDAIC LETTER AQMANDAIC LETTER ARMANDAIC LETTER ASHMANDAIC LETTER ATMANDAIC LETTER DUSHENNAMANDAIC LETTER KADMANDAIC LETTER AINMANDAIC AFFRICATION MARKMANDAIC VOCALIZATION MARKMANDAIC GEMINATION MARKMAN\
    \DAIC PUNCTUATIONSYRIAC LETTER MALAYALAM NGASYRIAC LETTER MALAYALAM JASYRIAC LETTER MALAYALAM NYASYRIAC LETTER MALAYALAM TTASYRIAC LETTER MALAYALAM NNASYRIAC LETTER MALAYALAM NNNASYRIAC LETTER MALAYALAM BHASYRIAC LETTER MALAYALAM RASYRIAC LETTER MALAYALAM \
    \LLASYRIAC LETTER MALAYALAM LLLASYRIAC LETTER MALAYALAM SSAARABIC LETTER ALEF WITH ATTACHED FATHAARABIC LETTER ALEF WITH ATTACHED TOP RIGHT FATHAARABIC LETTER ALEF WITH RIGHT MIDDLE STROKEARABIC LETTER ALEF WITH LEFT MIDDLE STROKEARABIC LETTER ALEF WITH AT\
    \TACHED KASRAARABIC LETTER ALEF WITH ATTACHED BOTTOM RIGHT KASRAARABIC LETTER ALEF WITH ATTACHED ROUND DOT ABOVEARABIC LETTER ALEF WITH ATTACHED RIGHT ROUND DOTARABIC LETTER ALEF WITH ATTACHED LEFT ROUND DOTARABIC LETTER ALEF WITH ATTACHED ROUND DOT BELOWA\
    \RABIC LETTER ALEF WITH DOT ABOVEARABIC LETTER ALEF WITH ATTACHED TOP RIGHT FATHA AND DOT ABOVEARABIC LETTER ALEF WITH RIGHT MIDDLE STROKE AND DOT ABOVEARABIC LETTER ALEF WITH ATTACHED BOTTOM RIGHT KASRA AND DOT ABOVEARABIC LETTER ALEF WITH ATTACHED TOP RI\
    \GHT FATHA AND LEFT RINGARABIC LETTER ALEF WITH RIGHT MIDDLE STROKE AND LEFT RINGARABIC LETTER ALEF WITH ATTACHED BOTTOM RIGHT KASRA AND LEFT RINGARABIC LETTER ALEF WITH ATTACHED RIGHT HAMZAARABIC LETTER ALEF WITH ATTACHED LEFT HAMZAARABIC TATWEEL WITH OVE\
    \RSTRUCK HAMZAARABIC TATWEEL WITH OVERSTRUCK WAWARABIC TATWEEL WITH TWO DOTS BELOWARABIC LETTER THIN YEHARABIC BASELINE ROUND DOTARABIC RAISED ROUND DOTARABIC LETTER NOON WITH INVERTED SMALL VARABIC LETTER HAH WITH INVERTED SMALL V BELOWARABIC LETTER TAH W\
    \ITH DOT BELOWARABIC LETTER TAH WITH THREE DOTS BELOWARABIC LETTER KEHEH WITH TWO DOTS VERTICALLY BELOWARABIC VERTICAL TAILARABIC POUND MARK ABOVEARABIC PIASTRE MARK ABOVEARABIC SMALL HIGH WORD AL-JUZARABIC SMALL LOW WORD ISHMAAMARABIC SMALL LOW WORD IMAAL\
    \AARABIC SMALL LOW WORD TASHEELARABIC MADDA WAAJIBARABIC SUPERSCRIPT ALEF MOKHASSASARABIC DOUBLED MADDAARABIC HALF MADDA OVER MADDAARABIC LETTER BEH WITH SMALL V BELOWARABIC LETTER BEH WITH HAMZA ABOVEARABIC LETTER JEEM WITH TWO DOTS ABOVEARABIC LETTER TAH\
    \ WITH TWO DOTS ABOVEARABIC LETTER FEH WITH DOT BELOW AND THREE DOTS ABOVEARABIC LETTER QAF WITH DOT BELOWARABIC LETTER LAM WITH DOUBLE BARARABIC LETTER MEEM WITH THREE DOTS ABOVEARABIC LETTER YEH WITH TWO DOTS BELOW AND HAMZA ABOVEARABIC LETTER YEH WITH T\
    \WO DOTS BELOW AND DOT ABOVEARABIC LETTER REH WITH LOOPARABIC LETTER WAW WITH DOT WITHINARABIC LETTER ROHINGYA YEHARABIC LETTER LOW ALEFARABIC LETTER DAL WITH THREE DOTS BELOWARABIC LETTER SAD WITH THREE DOTS BELOWARABIC LETTER GAF WITH INVERTED STROKEARAB\
    \IC LETTER STRAIGHT WAWARABIC LETTER ZAIN WITH INVERTED V ABOVEARABIC LETTER AIN WITH THREE DOTS BELOWARABIC LETTER KAF WITH DOT BELOWARABIC LETTER QAF WITH DOT BELOW AND NO DOTS ABOVEARABIC LETTER BEH WITH SMALL MEEM ABOVEARABIC LETTER PEH WITH SMALL MEEM\
    \ ABOVEARABIC LETTER TEH WITH SMALL TEH ABOVEARABIC LETTER REH WITH SMALL NOON ABOVEARABIC LETTER YEH WITH TWO DOTS BELOW AND SMALL NOON ABOVEARABIC LETTER AFRICAN FEHARABIC LETTER AFRICAN QAFARABIC LETTER AFRICAN NOONARABIC LETTER PEH WITH SMALL VARABIC L\
    \ETTER TEH WITH SMALL VARABIC LETTER TTEH WITH SMALL VARABIC LETTER TCHEH WITH SMALL VARABIC LETTER KEHEH WITH SMALL VARABIC LETTER GHAIN WITH THREE DOTS ABOVEARABIC LETTER AFRICAN QAF WITH THREE DOTS ABOVEARABIC LETTER JEEM WITH THREE DOTS ABOVEARABIC LET\
    \TER JEEM WITH THREE DOTS BELOWARABIC LETTER LAM WITH SMALL ARABIC LETTER TAH ABOVEARABIC LETTER GRAFARABIC SMALL FARSI YEHARABIC SMALL HIGH FARSI YEHARABIC SMALL HIGH YEH BARREE WITH TWO DOTS BELOWARABIC SMALL HIGH WORD SAHARABIC SMALL HIGH ZAHARABIC LARG\
    \E ROUND DOT ABOVEARABIC LARGE ROUND DOT BELOWARABIC SUKUN BELOWARABIC LARGE CIRCLE BELOWARABIC LARGE ROUND DOT INSIDE CIRCLE BELOWARABIC SMALL LOW WAWARABIC SMALL HIGH WORD AR-RUBARABIC SMALL HIGH SADARABIC SMALL HIGH AINARABIC SMALL HIGH QAFARABIC SMALL \
    \HIGH NOON WITH KASRAARABIC SMALL LOW NOON WITH KASRAARABIC SMALL HIGH WORD ATH-THALATHAARABIC SMALL HIGH WORD AS-SAJDAARABIC SMALL HIGH WORD AN-NISFARABIC SMALL HIGH WORD SAKTAARABIC SMALL HIGH WORD QIFARABIC SMALL HIGH WORD WAQFAARABIC SMALL HIGH FOOTNOT\
    \E MARKERARABIC SMALL HIGH SIGN SAFHAARABIC DISPUTED END OF AYAHARABIC TURNED DAMMA BELOWARABIC CURLY FATHAARABIC CURLY DAMMAARABIC CURLY KASRAARABIC CURLY FATHATANARABIC CURLY DAMMATANARABIC CURLY KASRATANARABIC TONE ONE DOT ABOVEARABIC TONE TWO DOTS ABOV\
    \EARABIC TONE LOOP ABOVEARABIC TONE ONE DOT BELOWARABIC TONE TWO DOTS BELOWARABIC TONE LOOP BELOWARABIC OPEN FATHATANARABIC OPEN DAMMATANARABIC OPEN KASRATANARABIC SMALL HIGH WAWARABIC FATHA WITH RINGARABIC FATHA WITH DOT ABOVEARABIC KASRA WITH DOT BELOWAR\
    \ABIC LEFT ARROWHEAD ABOVEARABIC RIGHT ARROWHEAD ABOVEARABIC LEFT ARROWHEAD BELOWARABIC RIGHT ARROWHEAD BELOWARABIC DOUBLE RIGHT ARROWHEAD ABOVEARABIC DOUBLE RIGHT ARROWHEAD ABOVE WITH DOTARABIC RIGHT ARROWHEAD ABOVE WITH DOTARABIC DAMMA WITH DOTARABIC MAR\
    \K SIDEWAYS NOON GHUNNADEVANAGARI SIGN INVERTED CANDRABINDUDEVANAGARI SIGN CANDRABINDUDEVANAGARI SIGN ANUSVARADEVANAGARI SIGN VISARGADEVANAGARI LETTER SHORT ADEVANAGARI LETTER ADEVANAGARI LETTER AADEVANAGARI LETTER IDEVANAGARI LETTER IIDEVANAGARI LETTER UD\
    \EVANAGARI LETTER UUDEVANAGARI LETTER VOCALIC RDEVANAGARI LETTER VOCALIC LDEVANAGARI LETTER CANDRA EDEVANAGARI LETTER SHORT EDEVANAGARI LETTER EDEVANAGARI LETTER AIDEVANAGARI LETTER CANDRA ODEVANAGARI LETTER SHORT ODEVANAGARI LETTER ODEVANAGARI LETTER AUDE\
    \VANAGARI LETTER KADEVANAGARI LETTER KHADEVANAGARI LETTER GADEVANAGARI LETTER GHADEVANAGARI LETTER NGADEVANAGARI LETTER CADEVANAGARI LETTER CHADEVANAGARI LETTER JADEVANAGARI LETTER JHADEVANAGARI LETTER NYADEVANAGARI LETTER TTADEVANAGARI LETTER TTHADEVANAGA\
    \RI LETTER DDADEVANAGARI LETTER DDHADEVANAGARI LETTER NNADEVANAGARI LETTER TADEVANAGARI LETTER THADEVANAGARI LETTER DADEVANAGARI LETTER DHADEVANAGARI LETTER NADEVANAGARI LETTER NNNADEVANAGARI LETTER PADEVANAGARI LETTER PHADEVANAGARI LETTER BADEVANAGARI LET\
    \TER BHADEVANAGARI LETTER MADEVANAGARI LETTER YADEVANAGARI LETTER RADEVANAGARI LETTER RRADEVANAGARI LETTER LADEVANAGARI LETTER LLADEVANAGARI LETTER LLLADEVANAGARI LETTER VADEVANAGARI LETTER SHADEVANAGARI LETTER SSADEVANAGARI LETTER SADEVANAGARI LETTER HADE\
    \VANAGARI VOWEL SIGN OEDEVANAGARI VOWEL SIGN OOEDEVANAGARI SIGN NUKTADEVANAGARI SIGN AVAGRAHADEVANAGARI VOWEL SIGN AADEVANAGARI VOWEL SIGN IDEVANAGARI VOWEL SIGN IIDEVANAGARI VOWEL SIGN UDEVANAGARI VOWEL SIGN UUDEVANAGARI VOWEL SIGN VOCALIC RDEVANAGARI VOW\
    \EL SIGN VOCALIC RRDEVANAGARI VOWEL SIGN CANDRA EDEVANAGARI VOWEL SIGN SHORT EDEVANAGARI VOWEL SIGN EDEVANAGARI VOWEL SIGN AIDEVANAGARI VOWEL SIGN CANDRA ODEVANAGARI VOWEL SIGN SHORT ODEVANAGARI VOWEL SIGN ODEVANAGARI VOWEL SIGN AUDEVANAGARI SIGN VIRAMADEV\
    \ANAGARI VOWEL SIGN PRISHTHAMATRA EDEVANAGARI VOWEL SIGN AWDEVANAGARI OMDEVANAGARI STRESS SIGN UDATTADEVANAGARI STRESS SIGN ANUDATTADEVANAGARI GRAVE ACCENTDEVANAGARI ACUTE ACCENTDEVANAGARI VOWEL SIGN CANDRA LONG EDEVANAGARI VOWEL SIGN UEDEVANAGARI VOWEL SI\
    \GN UUEDEVANAGARI LETTER QADEVANAGARI LETTER KHHADEVANAGARI LETTER GHHADEVANAGARI LETTER ZADEVANAGARI LETTER DDDHADEVANAGARI LETTER RHADEVANAGARI LETTER FADEVANAGARI LETTER YYADEVANAGARI LETTER VOCALIC RRDEVANAGARI LETTER VOCALIC LLDEVANAGARI VOWEL SIGN VO\
    \CALIC LDEVANAGARI VOWEL SIGN VOCALIC LLDEVANAGARI DANDADEVANAGARI DOUBLE DANDADEVANAGARI DIGIT ZERODEVANAGARI DIGIT ONEDEVANAGARI DIGIT TWODEVANAGARI DIGIT THREEDEVANAGARI DIGIT FOURDEVANAGARI DIGIT FIVEDEVANAGARI DIGIT SIXDEVANAGARI DIGIT SEVENDEVANAGARI\
    \ DIGIT EIGHTDEVANAGARI DIGIT NINEDEVANAGARI ABBREVIATION SIGNDEVANAGARI SIGN HIGH SPACING DOTDEVANAGARI LETTER CANDRA ADEVANAGARI LETTER OEDEVANAGARI LETTER OOEDEVANAGARI LETTER AWDEVANAGARI LETTER UEDEVANAGARI LETTER UUEDEVANAGARI LETTER MARWARI DDADEVAN\
    \AGARI LETTER ZHADEVANAGARI LETTER HEAVY YADEVANAGARI LETTER GGADEVANAGARI LETTER JJADEVANAGARI LETTER GLOTTAL STOPDEVANAGARI LETTER DDDADEVANAGARI LETTER BBABENGALI ANJIBENGALI SIGN CANDRABINDUBENGALI SIGN ANUSVARABENGALI SIGN VISARGABENGALI LETTER ABENGA\
    \LI LETTER AABENGALI LETTER IBENGALI LETTER IIBENGALI LETTER UBENGALI LETTER UUBENGALI LETTER VOCALIC RBENGALI LETTER VOCALIC LBENGALI LETTER EBENGALI LETTER AIBENGALI LETTER OBENGALI LETTER AUBENGALI LETTER KABENGALI LETTER KHABENGALI LETTER GABENGALI LET\
    \TER GHABENGALI LETTER NGABENGALI LETTER CABENGALI LETTER CHABENGALI LETTER JABENGALI LETTER JHABENGALI LETTER NYABENGALI LETTER TTABENGALI LETTER TTHABENGALI LETTER DDABENGALI LETTER DDHABENGALI LETTER NNABENGALI LETTER TABENGALI LETTER THABENGALI LETTER \
    \DABENGALI LETTER DHABENGALI LETTER NABENGALI LETTER PABENGALI LETTER PHABENGALI LETTER BABENGALI LETTER BHABENGALI LETTER MABENGALI LETTER YABENGALI LETTER RABENGALI LETTER LABENGALI LETTER SHABENGALI LETTER SSABENGALI LETTER SABENGALI LETTER HABENGALI SI\
    \GN NUKTABENGALI SIGN AVAGRAHABENGALI VOWEL SIGN AABENGALI VOWEL SIGN IBENGALI VOWEL SIGN IIBENGALI VOWEL SIGN UBENGALI VOWEL SIGN UUBENGALI VOWEL SIGN VOCALIC RBENGALI VOWEL SIGN VOCALIC RRBENGALI VOWEL SIGN EBENGALI VOWEL SIGN AIBENGALI VOWEL SIGN OBENGA\
    \LI VOWEL SIGN AUBENGALI SIGN VIRAMABENGALI LETTER KHANDA TABENGALI AU LENGTH MARKBENGALI LETTER RRABENGALI LETTER RHABENGALI LETTER YYABENGALI LETTER VOCALIC RRBENGALI LETTER VOCALIC LLBENGALI VOWEL SIGN VOCALIC LBENGALI VOWEL SIGN VOCALIC LLBENGALI DIGIT\
    \ ZEROBENGALI DIGIT ONEBENGALI DIGIT TWOBENGALI DIGIT THREEBENGALI DIGIT FOURBENGALI DIGIT FIVEBENGALI DIGIT SIXBENGALI DIGIT SEVENBENGALI DIGIT EIGHTBENGALI DIGIT NINEBENGALI LETTER RA WITH MIDDLE DIAGONALBENGALI LETTER RA WITH LOWER DIAGONALBENGALI RUPEE\
    \ MARKBENGALI RUPEE SIGNBENGALI CURRENCY NUMERATOR ONEBENGALI CURRENCY NUMERATOR TWOBENGALI CURRENCY NUMERATOR THREEBENGALI CURRENCY NUMERATOR FOURBENGALI CURRENCY NUMERATOR ONE LESS THAN THE DENOMINATORBENGALI CURRENCY DENOMINATOR SIXTEENBENGALI ISSHARBEN\
    \GALI GANDA MARKBENGALI LETTER VEDIC ANUSVARABENGALI ABBREVIATION SIGNBENGALI SANDHI MARKGURMUKHI SIGN ADAK BINDIGURMUKHI SIGN BINDIGURMUKHI SIGN VISARGAGURMUKHI LETTER AGURMUKHI LETTER AAGURMUKHI LETTER IGURMUKHI LETTER IIGURMUKHI LETTER UGURMUKHI LETTER \
    \UUGURMUKHI LETTER EEGURMUKHI LETTER AIGURMUKHI LETTER OOGURMUKHI LETTER AUGURMUKHI LETTER KAGURMUKHI LETTER KHAGURMUKHI LETTER GAGURMUKHI LETTER GHAGURMUKHI LETTER NGAGURMUKHI LETTER CAGURMUKHI LETTER CHAGURMUKHI LETTER JAGURMUKHI LETTER JHAGURMUKHI LETTE\
    \R NYAGURMUKHI LETTER TTAGURMUKHI LETTER TTHAGURMUKHI LETTER DDAGURMUKHI LETTER DDHAGURMUKHI LETTER NNAGURMUKHI LETTER TAGURMUKHI LETTER THAGURMUKHI LETTER DAGURMUKHI LETTER DHAGURMUKHI LETTER NAGURMUKHI LETTER PAGURMUKHI LETTER PHAGURMUKHI LETTER BAGURMUK\
    \HI LETTER BHAGURMUKHI LETTER MAGURMUKHI LETTER YAGURMUKHI LETTER RAGURMUKHI LETTER LAGURMUKHI LETTER LLAGURMUKHI LETTER VAGURMUKHI LETTER SHAGURMUKHI LETTER SAGURMUKHI LETTER HAGURMUKHI SIGN NUKTAGURMUKHI VOWEL SIGN AAGURMUKHI VOWEL SIGN IGURMUKHI VOWEL S\
    \IGN IIGURMUKHI VOWEL SIGN UGURMUKHI VOWEL SIGN UUGURMUKHI VOWEL SIGN EEGURMUKHI VOWEL SIGN AIGURMUKHI VOWEL SIGN OOGURMUKHI VOWEL SIGN AUGURMUKHI SIGN VIRAMAGURMUKHI SIGN UDAATGURMUKHI LETTER KHHAGURMUKHI LETTER GHHAGURMUKHI LETTER ZAGURMUKHI LETTER RRAGU\
    \RMUKHI LETTER FAGURMUKHI DIGIT ZEROGURMUKHI DIGIT ONEGURMUKHI DIGIT TWOGURMUKHI DIGIT THREEGURMUKHI DIGIT FOURGURMUKHI DIGIT FIVEGURMUKHI DIGIT SIXGURMUKHI DIGIT SEVENGURMUKHI DIGIT EIGHTGURMUKHI DIGIT NINEGURMUKHI TIPPIGURMUKHI ADDAKGURMUKHI IRIGURMUKHI \
    \URAGURMUKHI EK ONKARGURMUKHI SIGN YAKASHGURMUKHI ABBREVIATION SIGNGUJARATI SIGN CANDRABINDUGUJARATI SIGN ANUSVARAGUJARATI SIGN VISARGAGUJARATI LETTER AGUJARATI LETTER AAGUJARATI LETTER IGUJARATI LETTER IIGUJARATI LETTER UGUJARATI LETTER UUGUJARATI LETTER \
    \VOCALIC RGUJARATI LETTER VOCALIC LGUJARATI VOWEL CANDRA EGUJARATI LETTER EGUJARATI LETTER AIGUJARATI VOWEL CANDRA OGUJARATI LETTER OGUJARATI LETTER AUGUJARATI LETTER KAGUJARATI LETTER KHAGUJARATI LETTER GAGUJARATI LETTER GHAGUJARATI LETTER NGAGUJARATI LET\
    \TER CAGUJARATI LETTER CHAGUJARATI LETTER JAGUJARATI LETTER JHAGUJARATI LETTER NYAGUJARATI LETTER TTAGUJARATI LETTER TTHAGUJARATI LETTER DDAGUJARATI LETTER DDHAGUJARATI LETTER NNAGUJARATI LETTER TAGUJARATI LETTER THAGUJARATI LETTER DAGUJARATI LETTER DHAGUJ\
    \ARATI LETTER NAGUJARATI LETTER PAGUJARATI LETTER PHAGUJARATI LETTER BAGUJARATI LETTER BHAGUJARATI LETTER MAGUJARATI LETTER YAGUJARATI LETTER RAGUJARATI LETTER LAGUJARATI LETTER LLAGUJARATI LETTER VAGUJARATI LETTER SHAGUJARATI LETTER SSAGUJARATI LETTER SAG\
    \UJARATI LETTER HAGUJARATI SIGN NUKTAGUJARATI SIGN AVAGRAHAGUJARATI VOWEL SIGN AAGUJARATI VOWEL SIGN IGUJARATI VOWEL SIGN IIGUJARATI VOWEL SIGN UGUJARATI VOWEL SIGN UUGUJARATI VOWEL SIGN VOCALIC RGUJARATI VOWEL SIGN VOCALIC RRGUJARATI VOWEL SIGN CANDRA EGU\
    \JARATI VOWEL SIGN EGUJARATI VOWEL SIGN AIGUJARATI VOWEL SIGN CANDRA OGUJARATI VOWEL SIGN OGUJARATI VOWEL SIGN AUGUJARATI SIGN VIRAMAGUJARATI OMGUJARATI LETTER VOCALIC RRGUJARATI LETTER VOCALIC LLGUJARATI VOWEL SIGN VOCALIC LGUJARATI VOWEL SIGN VOCALIC LLG\
    \UJARATI DIGIT ZEROGUJARATI DIGIT ONEGUJARATI DIGIT TWOGUJARATI DIGIT THREEGUJARATI DIGIT FOURGUJARATI DIGIT FIVEGUJARATI DIGIT SIXGUJARATI DIGIT SEVENGUJARATI DIGIT EIGHTGUJARATI DIGIT NINEGUJARATI ABBREVIATION SIGNGUJARATI RUPEE SIGNGUJARATI LETTER ZHAGU\
    \JARATI SIGN SUKUNGUJARATI SIGN SHADDAGUJARATI SIGN MADDAHGUJARATI SIGN THREE-DOT NUKTA ABOVEGUJARATI SIGN CIRCLE NUKTA ABOVEGUJARATI SIGN TWO-CIRCLE NUKTA ABOVEORIYA SIGN CANDRABINDUORIYA SIGN ANUSVARAORIYA SIGN VISARGAORIYA LETTER AORIYA LETTER AAORIYA L\
    \ETTER IORIYA LETTER IIORIYA LETTER UORIYA LETTER UUORIYA LETTER VOCALIC RORIYA LETTER VOCALIC LORIYA LETTER EORIYA LETTER AIORIYA LETTER OORIYA LETTER AUORIYA LETTER KAORIYA LETTER KHAORIYA LETTER GAORIYA LETTER GHAORIYA LETTER NGAORIYA LETTER CAORIYA LET\
    \TER CHAORIYA LETTER JAORIYA LETTER JHAORIYA LETTER NYAORIYA LETTER TTAORIYA LETTER TTHAORIYA LETTER DDAORIYA LETTER DDHAORIYA LETTER NNAORIYA LETTER TAORIYA LETTER THAORIYA LETTER DAORIYA LETTER DHAORIYA LETTER NAORIYA LETTER PAORIYA LETTER PHAORIYA LETTE\
    \R BAORIYA LETTER BHAORIYA LETTER MAORIYA LETTER YAORIYA LETTER RAORIYA LETTER LAORIYA LETTER LLAORIYA LETTER VAORIYA LETTER SHAORIYA LETTER SSAORIYA LETTER SAORIYA LETTER HAORIYA SIGN NUKTAORIYA SIGN AVAGRAHAORIYA VOWEL SIGN AAORIYA VOWEL SIGN IORIYA VOWE\
    \L SIGN IIORIYA VOWEL SIGN UORIYA VOWEL SIGN UUORIYA VOWEL SIGN VOCALIC RORIYA VOWEL SIGN VOCALIC RRORIYA VOWEL SIGN EORIYA VOWEL SIGN AIORIYA VOWEL SIGN OORIYA VOWEL SIGN AUORIYA SIGN VIRAMAORIYA SIGN OVERLINEORIYA AI LENGTH MARKORIYA AU LENGTH MARKORIYA \
    \LETTER RRAORIYA LETTER RHAORIYA LETTER YYAORIYA LETTER VOCALIC RRORIYA LETTER VOCALIC LLORIYA VOWEL SIGN VOCALIC LORIYA VOWEL SIGN VOCALIC LLORIYA DIGIT ZEROORIYA DIGIT ONEORIYA DIGIT TWOORIYA DIGIT THREEORIYA DIGIT FOURORIYA DIGIT FIVEORIYA DIGIT SIXORIY\
    \A DIGIT SEVENORIYA DIGIT EIGHTORIYA DIGIT NINEORIYA ISSHARORIYA LETTER WAORIYA FRACTION ONE QUARTERORIYA FRACTION ONE HALFORIYA FRACTION THREE QUARTERSORIYA FRACTION ONE SIXTEENTHORIYA FRACTION ONE EIGHTHORIYA FRACTION THREE SIXTEENTHSTAMIL SIGN ANUSVARAT\
    \AMIL SIGN VISARGATAMIL LETTER ATAMIL LETTER AATAMIL LETTER ITAMIL LETTER IITAMIL LETTER UTAMIL LETTER UUTAMIL LETTER ETAMIL LETTER EETAMIL LETTER AITAMIL LETTER OTAMIL LETTER OOTAMIL LETTER AUTAMIL LETTER KATAMIL LETTER NGATAMIL LETTER CATAMIL LETTER JATA\
    \MIL LETTER NYATAMIL LETTER TTATAMIL LETTER NNATAMIL LETTER TATAMIL LETTER NATAMIL LETTER NNNATAMIL LETTER PATAMIL LETTER MATAMIL LETTER YATAMIL LETTER RATAMIL LETTER RRATAMIL LETTER LATAMIL LETTER LLATAMIL LETTER LLLATAMIL LETTER VATAMIL LETTER SHATAMIL L\
    \ETTER SSATAMIL LETTER SATAMIL LETTER HATAMIL VOWEL SIGN AATAMIL VOWEL SIGN ITAMIL VOWEL SIGN IITAMIL VOWEL SIGN UTAMIL VOWEL SIGN UUTAMIL VOWEL SIGN ETAMIL VOWEL SIGN EETAMIL VOWEL SIGN AITAMIL VOWEL SIGN OTAMIL VOWEL SIGN OOTAMIL VOWEL SIGN AUTAMIL SIGN \
    \VIRAMATAMIL OMTAMIL AU LENGTH MARKTAMIL DIGIT ZEROTAMIL DIGIT ONETAMIL DIGIT TWOTAMIL DIGIT THREETAMIL DIGIT FOURTAMIL DIGIT FIVETAMIL DIGIT SIXTAMIL DIGIT SEVENTAMIL DIGIT EIGHTTAMIL DIGIT NINETAMIL NUMBER TENTAMIL NUMBER ONE HUNDREDTAMIL NUMBER ONE THOU\
    \SANDTAMIL DAY SIGNTAMIL MONTH SIGNTAMIL YEAR SIGNTAMIL DEBIT SIGNTAMIL CREDIT SIGNTAMIL AS ABOVE SIGNTAMIL RUPEE SIGNTAMIL NUMBER SIGNTELUGU SIGN COMBINING CANDRABINDU ABOVETELUGU SIGN CANDRABINDUTELUGU SIGN ANUSVARATELUGU SIGN VISARGATELUGU SIGN COMBININ\
    \G ANUSVARA ABOVETELUGU LETTER ATELUGU LETTER AATELUGU LETTER ITELUGU LETTER IITELUGU LETTER UTELUGU LETTER UUTELUGU LETTER VOCALIC RTELUGU LETTER VOCALIC LTELUGU LETTER ETELUGU LETTER EETELUGU LETTER AITELUGU LETTER OTELUGU LETTER OOTELUGU LETTER AUTELUGU\
    \ LETTER KATELUGU LETTER KHATELUGU LETTER GATELUGU LETTER GHATELUGU LETTER NGATELUGU LETTER CATELUGU LETTER CHATELUGU LETTER JATELUGU LETTER JHATELUGU LETTER NYATELUGU LETTER TTATELUGU LETTER TTHATELUGU LETTER DDATELUGU LETTER DDHATELUGU LETTER NNATELUGU L\
    \ETTER TATELUGU LETTER THATELUGU LETTER DATELUGU LETTER DHATELUGU LETTER NATELUGU LETTER PATELUGU LETTER PHATELUGU LETTER BATELUGU LETTER BHATELUGU LETTER MATELUGU LETTER YATELUGU LETTER RATELUGU LETTER RRATELUGU LETTER LATELUGU LETTER LLATELUGU LETTER LLL\
    \ATELUGU LETTER VATELUGU LETTER SHATELUGU LETTER SSATELUGU LETTER SATELUGU LETTER HATELUGU SIGN NUKTATELUGU SIGN AVAGRAHATELUGU VOWEL SIGN AATELUGU VOWEL SIGN ITELUGU VOWEL SIGN IITELUGU VOWEL SIGN UTELUGU VOWEL SIGN UUTELUGU VOWEL SIGN VOCALIC RTELUGU VOW\
    \EL SIGN VOCALIC RRTELUGU VOWEL SIGN ETELUGU VOWEL SIGN EETELUGU VOWEL SIGN AITELUGU VOWEL SIGN OTELUGU VOWEL SIGN OOTELUGU VOWEL SIGN AUTELUGU SIGN VIRAMATELUGU LENGTH MARKTELUGU AI LENGTH MARKTELUGU LETTER TSATELUGU LETTER DZATELUGU LETTER RRRATELUGU LET\
    \TER NAKAARA POLLUTELUGU LETTER VOCALIC RRTELUGU LETTER VOCALIC LLTELUGU VOWEL SIGN VOCALIC LTELUGU VOWEL SIGN VOCALIC LLTELUGU DIGIT ZEROTELUGU DIGIT ONETELUGU DIGIT TWOTELUGU DIGIT THREETELUGU DIGIT FOURTELUGU DIGIT FIVETELUGU DIGIT SIXTELUGU DIGIT SEVEN\
    \TELUGU DIGIT EIGHTTELUGU DIGIT NINETELUGU SIGN SIDDHAMTELUGU FRACTION DIGIT ZERO FOR ODD POWERS OF FOURTELUGU FRACTION DIGIT ONE FOR ODD POWERS OF FOURTELUGU FRACTION DIGIT TWO FOR ODD POWERS OF FOURTELUGU FRACTION DIGIT THREE FOR ODD POWERS OF FOURTELUGU\
    \ FRACTION DIGIT ONE FOR EVEN POWERS OF FOURTELUGU FRACTION DIGIT TWO FOR EVEN POWERS OF FOURTELUGU FRACTION DIGIT THREE FOR EVEN POWERS OF FOURTELUGU SIGN TUUMUKANNADA SIGN SPACING CANDRABINDUKANNADA SIGN CANDRABINDUKANNADA SIGN ANUSVARAKANNADA SIGN VISAR\
    \GAKANNADA SIGN SIDDHAMKANNADA LETTER AKANNADA LETTER AAKANNADA LETTER IKANNADA LETTER IIKANNADA LETTER UKANNADA LETTER UUKANNADA LETTER VOCALIC RKANNADA LETTER VOCALIC LKANNADA LETTER EKANNADA LETTER EEKANNADA LETTER AIKANNADA LETTER OKANNADA LETTER OOKAN\
    \NADA LETTER AUKANNADA LETTER KAKANNADA LETTER KHAKANNADA LETTER GAKANNADA LETTER GHAKANNADA LETTER NGAKANNADA LETTER CAKANNADA LETTER CHAKANNADA LETTER JAKANNADA LETTER JHAKANNADA LETTER NYAKANNADA LETTER TTAKANNADA LETTER TTHAKANNADA LETTER DDAKANNADA LE\
    \TTER DDHAKANNADA LETTER NNAKANNADA LETTER TAKANNADA LETTER THAKANNADA LETTER DAKANNADA LETTER DHAKANNADA LETTER NAKANNADA LETTER PAKANNADA LETTER PHAKANNADA LETTER BAKANNADA LETTER BHAKANNADA LETTER MAKANNADA LETTER YAKANNADA LETTER RAKANNADA LETTER RRAKA\
    \NNADA LETTER LAKANNADA LETTER LLAKANNADA LETTER VAKANNADA LETTER SHAKANNADA LETTER SSAKANNADA LETTER SAKANNADA LETTER HAKANNADA SIGN NUKTAKANNADA SIGN AVAGRAHAKANNADA VOWEL SIGN AAKANNADA VOWEL SIGN IKANNADA VOWEL SIGN IIKANNADA VOWEL SIGN UKANNADA VOWEL \
    \SIGN UUKANNADA VOWEL SIGN VOCALIC RKANNADA VOWEL SIGN VOCALIC RRKANNADA VOWEL SIGN EKANNADA VOWEL SIGN EEKANNADA VOWEL SIGN AIKANNADA VOWEL SIGN OKANNADA VOWEL SIGN OOKANNADA VOWEL SIGN AUKANNADA SIGN VIRAMAKANNADA LENGTH MARKKANNADA AI LENGTH MARKKANNADA\
    \ LETTER NAKAARA POLLUKANNADA LETTER FAKANNADA LETTER VOCALIC RRKANNADA LETTER VOCALIC LLKANNADA VOWEL SIGN VOCALIC LKANNADA VOWEL SIGN VOCALIC LLKANNADA DIGIT ZEROKANNADA DIGIT ONEKANNADA DIGIT TWOKANNADA DIGIT THREEKANNADA DIGIT FOURKANNADA DIGIT FIVEKAN\
    \NADA DIGIT SIXKANNADA DIGIT SEVENKANNADA DIGIT EIGHTKANNADA DIGIT NINEKANNADA SIGN JIHVAMULIYAKANNADA SIGN UPADHMANIYAKANNADA SIGN COMBINING ANUSVARA ABOVE RIGHTMALAYALAM SIGN COMBINING ANUSVARA ABOVEMALAYALAM SIGN CANDRABINDUMALAYALAM SIGN ANUSVARAMALAYA\
    \LAM SIGN VISARGAMALAYALAM LETTER VEDIC ANUSVARAMALAYALAM LETTER AMALAYALAM LETTER AAMALAYALAM LETTER IMALAYALAM LETTER IIMALAYALAM LETTER UMALAYALAM LETTER UUMALAYALAM LETTER VOCALIC RMALAYALAM LETTER VOCALIC LMALAYALAM LETTER EMALAYALAM LETTER EEMALAYALA\
    \M LETTER AIMALAYALAM LETTER OMALAYALAM LETTER OOMALAYALAM LETTER AUMALAYALAM LETTER KAMALAYALAM LETTER KHAMALAYALAM LETTER GAMALAYALAM LETTER GHAMALAYALAM LETTER NGAMALAYALAM LETTER CAMALAYALAM LETTER CHAMALAYALAM LETTER JAMALAYALAM LETTER JHAMALAYALAM LE\
    \TTER NYAMALAYALAM LETTER TTAMALAYALAM LETTER TTHAMALAYALAM LETTER DDAMALAYALAM LETTER DDHAMALAYALAM LETTER NNAMALAYALAM LETTER TAMALAYALAM LETTER THAMALAYALAM LETTER DAMALAYALAM LETTER DHAMALAYALAM LETTER NAMALAYALAM LETTER NNNAMALAYALAM LETTER PAMALAYALA\
    \M LETTER PHAMALAYALAM LETTER BAMALAYALAM LETTER BHAMALAYALAM LETTER MAMALAYALAM LETTER YAMALAYALAM LETTER RAMALAYALAM LETTER RRAMALAYALAM LETTER LAMALAYALAM LETTER LLAMALAYALAM LETTER LLLAMALAYALAM LETTER VAMALAYALAM LETTER SHAMALAYALAM LETTER SSAMALAYALA\
    \M LETTER SAMALAYALAM LETTER HAMALAYALAM LETTER TTTAMALAYALAM SIGN VERTICAL BAR VIRAMAMALAYALAM SIGN CIRCULAR VIRAMAMALAYALAM SIGN AVAGRAHAMALAYALAM VOWEL SIGN AAMALAYALAM VOWEL SIGN IMALAYALAM VOWEL SIGN IIMALAYALAM VOWEL SIGN UMALAYALAM VOWEL SIGN UUMALA\
    \YALAM VOWEL SIGN VOCALIC RMALAYALAM VOWEL SIGN VOCALIC RRMALAYALAM VOWEL SIGN EMALAYALAM VOWEL SIGN EEMALAYALAM VOWEL SIGN AIMALAYALAM VOWEL SIGN OMALAYALAM VOWEL SIGN OOMALAYALAM VOWEL SIGN AUMALAYALAM SIGN VIRAMAMALAYALAM LETTER DOT REPHMALAYALAM SIGN P\
    \ARAMALAYALAM LETTER CHILLU MMALAYALAM LETTER CHILLU YMALAYALAM LETTER CHILLU LLLMALAYALAM AU LENGTH MARKMALAYALAM FRACTION ONE ONE-HUNDRED-AND-SIXTIETHMALAYALAM FRACTION ONE FORTIETHMALAYALAM FRACTION THREE EIGHTIETHSMALAYALAM FRACTION ONE TWENTIETHMALAYA\
    \LAM FRACTION ONE TENTHMALAYALAM FRACTION THREE TWENTIETHSMALAYALAM FRACTION ONE FIFTHMALAYALAM LETTER ARCHAIC IIMALAYALAM LETTER VOCALIC RRMALAYALAM LETTER VOCALIC LLMALAYALAM VOWEL SIGN VOCALIC LMALAYALAM VOWEL SIGN VOCALIC LLMALAYALAM DIGIT ZEROMALAYALA\
    \M DIGIT ONEMALAYALAM DIGIT TWOMALAYALAM DIGIT THREEMALAYALAM DIGIT FOURMALAYALAM DIGIT FIVEMALAYALAM DIGIT SIXMALAYALAM DIGIT SEVENMALAYALAM DIGIT EIGHTMALAYALAM DIGIT NINEMALAYALAM NUMBER TENMALAYALAM NUMBER ONE HUNDREDMALAYALAM NUMBER ONE THOUSANDMALAYA\
    \LAM FRACTION ONE QUARTERMALAYALAM FRACTION ONE HALFMALAYALAM FRACTION THREE QUARTERSMALAYALAM FRACTION ONE SIXTEENTHMALAYALAM FRACTION ONE EIGHTHMALAYALAM FRACTION THREE SIXTEENTHSMALAYALAM DATE MARKMALAYALAM LETTER CHILLU NNMALAYALAM LETTER CHILLU NMALAY\
    \ALAM LETTER CHILLU RRMALAYALAM LETTER CHILLU LMALAYALAM LETTER CHILLU LLMALAYALAM LETTER CHILLU KSINHALA SIGN CANDRABINDUSINHALA SIGN ANUSVARAYASINHALA SIGN VISARGAYASINHALA LETTER AYANNASINHALA LETTER AAYANNASINHALA LETTER AEYANNASINHALA LETTER AEEYANNAS\
    \INHALA LETTER IYANNASINHALA LETTER IIYANNASINHALA LETTER UYANNASINHALA LETTER UUYANNASINHALA LETTER IRUYANNASINHALA LETTER IRUUYANNASINHALA LETTER ILUYANNASINHALA LETTER ILUUYANNASINHALA LETTER EYANNASINHALA LETTER EEYANNASINHALA LETTER AIYANNASINHALA LET\
    \TER OYANNASINHALA LETTER OOYANNASINHALA LETTER AUYANNASINHALA LETTER ALPAPRAANA KAYANNASINHALA LETTER MAHAAPRAANA KAYANNASINHALA LETTER ALPAPRAANA GAYANNASINHALA LETTER MAHAAPRAANA GAYANNASINHALA LETTER KANTAJA NAASIKYAYASINHALA LETTER SANYAKA GAYANNASINH\
    \ALA LETTER ALPAPRAANA CAYANNASINHALA LETTER MAHAAPRAANA CAYANNASINHALA LETTER ALPAPRAANA JAYANNASINHALA LETTER MAHAAPRAANA JAYANNASINHALA LETTER TAALUJA NAASIKYAYASINHALA LETTER TAALUJA SANYOOGA NAAKSIKYAYASINHALA LETTER SANYAKA JAYANNASINHALA LETTER ALPA\
    \PRAANA TTAYANNASINHALA LETTER MAHAAPRAANA TTAYANNASINHALA LETTER ALPAPRAANA DDAYANNASINHALA LETTER MAHAAPRAANA DDAYANNASINHALA LETTER MUURDHAJA NAYANNASINHALA LETTER SANYAKA DDAYANNASINHALA LETTER ALPAPRAANA TAYANNASINHALA LETTER MAHAAPRAANA TAYANNASINHAL\
    \A LETTER ALPAPRAANA DAYANNASINHALA LETTER MAHAAPRAANA DAYANNASINHALA LETTER DANTAJA NAYANNASINHALA LETTER SANYAKA DAYANNASINHALA LETTER ALPAPRAANA PAYANNASINHALA LETTER MAHAAPRAANA PAYANNASINHALA LETTER ALPAPRAANA BAYANNASINHALA LETTER MAHAAPRAANA BAYANNA\
    \SINHALA LETTER MAYANNASINHALA LETTER AMBA BAYANNASINHALA LETTER YAYANNASINHALA LETTER RAYANNASINHALA LETTER DANTAJA LAYANNASINHALA LETTER VAYANNASINHALA LETTER TAALUJA SAYANNASINHALA LETTER MUURDHAJA SAYANNASINHALA LETTER DANTAJA SAYANNASINHALA LETTER HAY\
    \ANNASINHALA LETTER MUURDHAJA LAYANNASINHALA LETTER FAYANNASINHALA SIGN AL-LAKUNASINHALA VOWEL SIGN AELA-PILLASINHALA VOWEL SIGN KETTI AEDA-PILLASINHALA VOWEL SIGN DIGA AEDA-PILLASINHALA VOWEL SIGN KETTI IS-PILLASINHALA VOWEL SIGN DIGA IS-PILLASINHALA VOWE\
    \L SIGN KETTI PAA-PILLASINHALA VOWEL SIGN DIGA PAA-PILLASINHALA VOWEL SIGN GAETTA-PILLASINHALA VOWEL SIGN KOMBUVASINHALA VOWEL SIGN DIGA KOMBUVASINHALA VOWEL SIGN KOMBU DEKASINHALA VOWEL SIGN KOMBUVA HAA AELA-PILLASINHALA VOWEL SIGN KOMBUVA HAA DIGA AELA-P\
    \ILLASINHALA VOWEL SIGN KOMBUVA HAA GAYANUKITTASINHALA VOWEL SIGN GAYANUKITTASINHALA LITH DIGIT ZEROSINHALA LITH DIGIT ONESINHALA LITH DIGIT TWOSINHALA LITH DIGIT THREESINHALA LITH DIGIT FOURSINHALA LITH DIGIT FIVESINHALA LITH DIGIT SIXSINHALA LITH DIGIT S\
    \EVENSINHALA LITH DIGIT EIGHTSINHALA LITH DIGIT NINESINHALA VOWEL SIGN DIGA GAETTA-PILLASINHALA VOWEL SIGN DIGA GAYANUKITTASINHALA PUNCTUATION KUNDDALIYATHAI CHARACTER KO KAITHAI CHARACTER KHO KHAITHAI CHARACTER KHO KHUATTHAI CHARACTER KHO KHWAITHAI CHARAC\
    \TER KHO KHONTHAI CHARACTER KHO RAKHANGTHAI CHARACTER NGO NGUTHAI CHARACTER CHO CHANTHAI CHARACTER CHO CHINGTHAI CHARACTER CHO CHANGTHAI CHARACTER SO SOTHAI CHARACTER CHO CHOETHAI CHARACTER YO YINGTHAI CHARACTER DO CHADATHAI CHARACTER TO PATAKTHAI CHARACTE\
    \R THO THANTHAI CHARACTER THO NANGMONTHOTHAI CHARACTER THO PHUTHAOTHAI CHARACTER NO NENTHAI CHARACTER DO DEKTHAI CHARACTER TO TAOTHAI CHARACTER THO THUNGTHAI CHARACTER THO THAHANTHAI CHARACTER THO THONGTHAI CHARACTER NO NUTHAI CHARACTER BO BAIMAITHAI CHARA\
    \CTER PO PLATHAI CHARACTER PHO PHUNGTHAI CHARACTER FO FATHAI CHARACTER PHO PHANTHAI CHARACTER FO FANTHAI CHARACTER PHO SAMPHAOTHAI CHARACTER MO MATHAI CHARACTER YO YAKTHAI CHARACTER RO RUATHAI CHARACTER RUTHAI CHARACTER LO LINGTHAI CHARACTER LUTHAI CHARACT\
    \ER WO WAENTHAI CHARACTER SO SALATHAI CHARACTER SO RUSITHAI CHARACTER SO SUATHAI CHARACTER HO HIPTHAI CHARACTER LO CHULATHAI CHARACTER O ANGTHAI CHARACTER HO NOKHUKTHAI CHARACTER PAIYANNOITHAI CHARACTER SARA ATHAI CHARACTER MAI HAN-AKATTHAI CHARACTER SARA \
    \AATHAI CHARACTER SARA AMTHAI CHARACTER SARA ITHAI CHARACTER SARA IITHAI CHARACTER SARA UETHAI CHARACTER SARA UEETHAI CHARACTER SARA UTHAI CHARACTER SARA UUTHAI CHARACTER PHINTHUTHAI CURRENCY SYMBOL BAHTTHAI CHARACTER SARA ETHAI CHARACTER SARA AETHAI CHARA\
    \CTER SARA OTHAI CHARACTER SARA AI MAIMUANTHAI CHARACTER SARA AI MAIMALAITHAI CHARACTER LAKKHANGYAOTHAI CHARACTER MAIYAMOKTHAI CHARACTER MAITAIKHUTHAI CHARACTER MAI EKTHAI CHARACTER MAI THOTHAI CHARACTER MAI TRITHAI CHARACTER MAI CHATTAWATHAI CHARACTER THA\
    \NTHAKHATTHAI CHARACTER NIKHAHITTHAI CHARACTER YAMAKKANTHAI CHARACTER FONGMANTHAI DIGIT ZEROTHAI DIGIT ONETHAI DIGIT TWOTHAI DIGIT THREETHAI DIGIT FOURTHAI DIGIT FIVETHAI DIGIT SIXTHAI DIGIT SEVENTHAI DIGIT EIGHTTHAI DIGIT NINETHAI CHARACTER ANGKHANKHUTHAI\
    \ CHARACTER KHOMUTLAO LETTER KOLAO LETTER KHO SUNGLAO LETTER KHO TAMLAO LETTER PALI GHALAO LETTER NGOLAO LETTER COLAO LETTER PALI CHALAO LETTER SO TAMLAO LETTER PALI JHALAO LETTER NYOLAO LETTER PALI NYALAO LETTER PALI TTALAO LETTER PALI TTHALAO LETTER PALI\
    \ DDALAO LETTER PALI DDHALAO LETTER PALI NNALAO LETTER DOLAO LETTER TOLAO LETTER THO SUNGLAO LETTER THO TAMLAO LETTER PALI DHALAO LETTER NOLAO LETTER BOLAO LETTER POLAO LETTER PHO SUNGLAO LETTER FO TAMLAO LETTER PHO TAMLAO LETTER FO SUNGLAO LETTER PALI BHA\
    \LAO LETTER MOLAO LETTER YOLAO LETTER LO LINGLAO LETTER LO LOOTLAO LETTER WOLAO LETTER SANSKRIT SHALAO LETTER SANSKRIT SSALAO LETTER SO SUNGLAO LETTER HO SUNGLAO LETTER PALI LLALAO LETTER OLAO LETTER HO TAMLAO ELLIPSISLAO VOWEL SIGN ALAO VOWEL SIGN MAI KAN\
    \LAO VOWEL SIGN AALAO VOWEL SIGN AMLAO VOWEL SIGN ILAO VOWEL SIGN IILAO VOWEL SIGN YLAO VOWEL SIGN YYLAO VOWEL SIGN ULAO VOWEL SIGN UULAO SIGN PALI VIRAMALAO VOWEL SIGN MAI KONLAO SEMIVOWEL SIGN LOLAO SEMIVOWEL SIGN NYOLAO VOWEL SIGN ELAO VOWEL SIGN EILAO \
    \VOWEL SIGN OLAO VOWEL SIGN AYLAO VOWEL SIGN AILAO KO LALAO TONE MAI EKLAO TONE MAI THOLAO TONE MAI TILAO TONE MAI CATAWALAO CANCELLATION MARKLAO NIGGAHITALAO YAMAKKANLAO DIGIT ZEROLAO DIGIT ONELAO DIGIT TWOLAO DIGIT THREELAO DIGIT FOURLAO DIGIT FIVELAO DI\
    \GIT SIXLAO DIGIT SEVENLAO DIGIT EIGHTLAO DIGIT NINELAO HO NOLAO HO MOLAO LETTER KHMU GOLAO LETTER KHMU NYOTIBETAN SYLLABLE OMTIBETAN MARK GTER YIG MGO TRUNCATED ATIBETAN MARK GTER YIG MGO -UM RNAM BCAD MATIBETAN MARK GTER YIG MGO -UM GTER TSHEG MATIBETAN \
    \MARK INITIAL YIG MGO MDUN MATIBETAN MARK CLOSING YIG MGO SGAB MATIBETAN MARK CARET YIG MGO PHUR SHAD MATIBETAN MARK YIG MGO TSHEG SHAD MATIBETAN MARK SBRUL SHADTIBETAN MARK BSKUR YIG MGOTIBETAN MARK BKA- SHOG YIG MGOTIBETAN MARK INTERSYLLABIC TSHEGTIBETAN\
    \ MARK DELIMITER TSHEG BSTARTIBETAN MARK SHADTIBETAN MARK NYIS SHADTIBETAN MARK TSHEG SHADTIBETAN MARK NYIS TSHEG SHADTIBETAN MARK RIN CHEN SPUNGS SHADTIBETAN MARK RGYA GRAM SHADTIBETAN MARK CARET -DZUD RTAGS ME LONG CANTIBETAN MARK GTER TSHEGTIBETAN LOGOT\
    \YPE SIGN CHAD RTAGSTIBETAN LOGOTYPE SIGN LHAG RTAGSTIBETAN ASTROLOGICAL SIGN SGRA GCAN -CHAR RTAGSTIBETAN ASTROLOGICAL SIGN -KHYUD PATIBETAN ASTROLOGICAL SIGN SDONG TSHUGSTIBETAN SIGN RDEL DKAR GCIGTIBETAN SIGN RDEL DKAR GNYISTIBETAN SIGN RDEL DKAR GSUMTI\
    \BETAN SIGN RDEL NAG GCIGTIBETAN SIGN RDEL NAG GNYISTIBETAN SIGN RDEL DKAR RDEL NAGTIBETAN DIGIT ZEROTIBETAN DIGIT ONETIBETAN DIGIT TWOTIBETAN DIGIT THREETIBETAN DIGIT FOURTIBETAN DIGIT FIVETIBETAN DIGIT SIXTIBETAN DIGIT SEVENTIBETAN DIGIT EIGHTTIBETAN DIG\
    \IT NINETIBETAN DIGIT HALF ONETIBETAN DIGIT HALF TWOTIBETAN DIGIT HALF THREETIBETAN DIGIT HALF FOURTIBETAN DIGIT HALF FIVETIBETAN DIGIT HALF SIXTIBETAN DIGIT HALF SEVENTIBETAN DIGIT HALF EIGHTTIBETAN DIGIT HALF NINETIBETAN DIGIT HALF ZEROTIBETAN MARK BSDUS\
    \ RTAGSTIBETAN MARK NGAS BZUNG NYI ZLATIBETAN MARK CARET -DZUD RTAGS BZHI MIG CANTIBETAN MARK NGAS BZUNG SGOR RTAGSTIBETAN MARK CHE MGOTIBETAN MARK TSA -PHRUTIBETAN MARK GUG RTAGS GYONTIBETAN MARK GUG RTAGS GYASTIBETAN MARK ANG KHANG GYONTIBETAN MARK ANG K\
    \HANG GYASTIBETAN SIGN YAR TSHESTIBETAN SIGN MAR TSHESTIBETAN LETTER KATIBETAN LETTER KHATIBETAN LETTER GATIBETAN LETTER GHATIBETAN LETTER NGATIBETAN LETTER CATIBETAN LETTER CHATIBETAN LETTER JATIBETAN LETTER NYATIBETAN LETTER TTATIBETAN LETTER TTHATIBETAN\
    \ LETTER DDATIBETAN LETTER DDHATIBETAN LETTER NNATIBETAN LETTER TATIBETAN LETTER THATIBETAN LETTER DATIBETAN LETTER DHATIBETAN LETTER NATIBETAN LETTER PATIBETAN LETTER PHATIBETAN LETTER BATIBETAN LETTER BHATIBETAN LETTER MATIBETAN LETTER TSATIBETAN LETTER \
    \TSHATIBETAN LETTER DZATIBETAN LETTER DZHATIBETAN LETTER WATIBETAN LETTER ZHATIBETAN LETTER ZATIBETAN LETTER -ATIBETAN LETTER YATIBETAN LETTER RATIBETAN LETTER LATIBETAN LETTER SHATIBETAN LETTER SSATIBETAN LETTER SATIBETAN LETTER HATIBETAN LETTER ATIBETAN \
    \LETTER KSSATIBETAN LETTER FIXED-FORM RATIBETAN LETTER KKATIBETAN LETTER RRATIBETAN VOWEL SIGN AATIBETAN VOWEL SIGN ITIBETAN VOWEL SIGN IITIBETAN VOWEL SIGN UTIBETAN VOWEL SIGN UUTIBETAN VOWEL SIGN VOCALIC RTIBETAN VOWEL SIGN VOCALIC RRTIBETAN VOWEL SIGN V\
    \OCALIC LTIBETAN VOWEL SIGN VOCALIC LLTIBETAN VOWEL SIGN ETIBETAN VOWEL SIGN EETIBETAN VOWEL SIGN OTIBETAN VOWEL SIGN OOTIBETAN SIGN RJES SU NGA ROTIBETAN SIGN RNAM BCADTIBETAN VOWEL SIGN REVERSED ITIBETAN VOWEL SIGN REVERSED IITIBETAN SIGN NYI ZLA NAA DAT\
    \IBETAN SIGN SNA LDANTIBETAN MARK HALANTATIBETAN MARK PALUTATIBETAN SIGN LCI RTAGSTIBETAN SIGN YANG RTAGSTIBETAN SIGN LCE TSA CANTIBETAN SIGN MCHU CANTIBETAN SIGN GRU CAN RGYINGSTIBETAN SIGN GRU MED RGYINGSTIBETAN SIGN INVERTED MCHU CANTIBETAN SUBJOINED SI\
    \GN LCE TSA CANTIBETAN SUBJOINED SIGN MCHU CANTIBETAN SUBJOINED SIGN INVERTED MCHU CANTIBETAN SUBJOINED LETTER KATIBETAN SUBJOINED LETTER KHATIBETAN SUBJOINED LETTER GATIBETAN SUBJOINED LETTER GHATIBETAN SUBJOINED LETTER NGATIBETAN SUBJOINED LETTER CATIBET\
    \AN SUBJOINED LETTER CHATIBETAN SUBJOINED LETTER JATIBETAN SUBJOINED LETTER NYATIBETAN SUBJOINED LETTER TTATIBETAN SUBJOINED LETTER TTHATIBETAN SUBJOINED LETTER DDATIBETAN SUBJOINED LETTER DDHATIBETAN SUBJOINED LETTER NNATIBETAN SUBJOINED LETTER TATIBETAN \
    \SUBJOINED LETTER THATIBETAN SUBJOINED LETTER DATIBETAN SUBJOINED LETTER DHATIBETAN SUBJOINED LETTER NATIBETAN SUBJOINED LETTER PATIBETAN SUBJOINED LETTER PHATIBETAN SUBJOINED LETTER BATIBETAN SUBJOINED LETTER BHATIBETAN SUBJOINED LETTER MATIBETAN SUBJOINE\
    \D LETTER TSATIBETAN SUBJOINED LETTER TSHATIBETAN SUBJOINED LETTER DZATIBETAN SUBJOINED LETTER DZHATIBETAN SUBJOINED LETTER WATIBETAN SUBJOINED LETTER ZHATIBETAN SUBJOINED LETTER ZATIBETAN SUBJOINED LETTER -ATIBETAN SUBJOINED LETTER YATIBETAN SUBJOINED LET\
    \TER RATIBETAN SUBJOINED LETTER LATIBETAN SUBJOINED LETTER SHATIBETAN SUBJOINED LETTER SSATIBETAN SUBJOINED LETTER SATIBETAN SUBJOINED LETTER HATIBETAN SUBJOINED LETTER ATIBETAN SUBJOINED LETTER KSSATIBETAN SUBJOINED LETTER FIXED-FORM WATIBETAN SUBJOINED L\
    \ETTER FIXED-FORM YATIBETAN SUBJOINED LETTER FIXED-FORM RATIBETAN KU RU KHATIBETAN KU RU KHA BZHI MIG CANTIBETAN CANTILLATION SIGN HEAVY BEATTIBETAN CANTILLATION SIGN LIGHT BEATTIBETAN CANTILLATION SIGN CANG TE-UTIBETAN CANTILLATION SIGN SBUB -CHALTIBETAN \
    \SYMBOL DRIL BUTIBETAN SYMBOL RDO RJETIBETAN SYMBOL PADMA GDANTIBETAN SYMBOL RDO RJE RGYA GRAMTIBETAN SYMBOL PHUR PATIBETAN SYMBOL NOR BUTIBETAN SYMBOL NOR BU NYIS -KHYILTIBETAN SYMBOL NOR BU GSUM -KHYILTIBETAN SYMBOL NOR BU BZHI -KHYILTIBETAN SIGN RDEL NA\
    \G RDEL DKARTIBETAN SIGN RDEL NAG GSUMTIBETAN MARK BSKA- SHOG GI MGO RGYANTIBETAN MARK MNYAM YIG GI MGO RGYANTIBETAN MARK NYIS TSHEGTIBETAN MARK INITIAL BRDA RNYING YIG MGO MDUN MATIBETAN MARK CLOSING BRDA RNYING YIG MGO SGAB MARIGHT-FACING SVASTI SIGNLEFT\
    \-FACING SVASTI SIGNRIGHT-FACING SVASTI SIGN WITH DOTSLEFT-FACING SVASTI SIGN WITH DOTSTIBETAN MARK LEADING MCHAN RTAGSTIBETAN MARK TRAILING MCHAN RTAGSMYANMAR LETTER KAMYANMAR LETTER KHAMYANMAR LETTER GAMYANMAR LETTER GHAMYANMAR LETTER NGAMYANMAR LETTER C\
    \AMYANMAR LETTER CHAMYANMAR LETTER JAMYANMAR LETTER JHAMYANMAR LETTER NYAMYANMAR LETTER NNYAMYANMAR LETTER TTAMYANMAR LETTER TTHAMYANMAR LETTER DDAMYANMAR LETTER DDHAMYANMAR LETTER NNAMYANMAR LETTER TAMYANMAR LETTER THAMYANMAR LETTER DAMYANMAR LETTER DHAMY\
    \ANMAR LETTER NAMYANMAR LETTER PAMYANMAR LETTER PHAMYANMAR LETTER BAMYANMAR LETTER BHAMYANMAR LETTER MAMYANMAR LETTER YAMYANMAR LETTER RAMYANMAR LETTER LAMYANMAR LETTER WAMYANMAR LETTER SAMYANMAR LETTER HAMYANMAR LETTER LLAMYANMAR LETTER AMYANMAR LETTER SH\
    \AN AMYANMAR LETTER IMYANMAR LETTER IIMYANMAR LETTER UMYANMAR LETTER UUMYANMAR LETTER EMYANMAR LETTER MON EMYANMAR LETTER OMYANMAR LETTER AUMYANMAR VOWEL SIGN TALL AAMYANMAR VOWEL SIGN AAMYANMAR VOWEL SIGN IMYANMAR VOWEL SIGN IIMYANMAR VOWEL SIGN UMYANMAR \
    \VOWEL SIGN UUMYANMAR VOWEL SIGN EMYANMAR VOWEL SIGN AIMYANMAR VOWEL SIGN MON IIMYANMAR VOWEL SIGN MON OMYANMAR VOWEL SIGN E ABOVEMYANMAR SIGN ANUSVARAMYANMAR SIGN DOT BELOWMYANMAR SIGN VISARGAMYANMAR SIGN VIRAMAMYANMAR SIGN ASATMYANMAR CONSONANT SIGN MEDI\
    \AL YAMYANMAR CONSONANT SIGN MEDIAL RAMYANMAR CONSONANT SIGN MEDIAL WAMYANMAR CONSONANT SIGN MEDIAL HAMYANMAR LETTER GREAT SAMYANMAR DIGIT ZEROMYANMAR DIGIT ONEMYANMAR DIGIT TWOMYANMAR DIGIT THREEMYANMAR DIGIT FOURMYANMAR DIGIT FIVEMYANMAR DIGIT SIXMYANMAR\
    \ DIGIT SEVENMYANMAR DIGIT EIGHTMYANMAR DIGIT NINEMYANMAR SIGN LITTLE SECTIONMYANMAR SIGN SECTIONMYANMAR SYMBOL LOCATIVEMYANMAR SYMBOL COMPLETEDMYANMAR SYMBOL AFOREMENTIONEDMYANMAR SYMBOL GENITIVEMYANMAR LETTER SHAMYANMAR LETTER SSAMYANMAR LETTER VOCALIC R\
    \MYANMAR LETTER VOCALIC RRMYANMAR LETTER VOCALIC LMYANMAR LETTER VOCALIC LLMYANMAR VOWEL SIGN VOCALIC RMYANMAR VOWEL SIGN VOCALIC RRMYANMAR VOWEL SIGN VOCALIC LMYANMAR VOWEL SIGN VOCALIC LLMYANMAR LETTER MON NGAMYANMAR LETTER MON JHAMYANMAR LETTER MON BBAM\
    \YANMAR LETTER MON BBEMYANMAR CONSONANT SIGN MON MEDIAL NAMYANMAR CONSONANT SIGN MON MEDIAL MAMYANMAR CONSONANT SIGN MON MEDIAL LAMYANMAR LETTER SGAW KAREN SHAMYANMAR VOWEL SIGN SGAW KAREN EUMYANMAR TONE MARK SGAW KAREN HATHIMYANMAR TONE MARK SGAW KAREN KE\
    \ PHOMYANMAR LETTER WESTERN PWO KAREN THAMYANMAR LETTER WESTERN PWO KAREN PWAMYANMAR VOWEL SIGN WESTERN PWO KAREN EUMYANMAR VOWEL SIGN WESTERN PWO KAREN UEMYANMAR SIGN WESTERN PWO KAREN TONE-1MYANMAR SIGN WESTERN PWO KAREN TONE-2MYANMAR SIGN WESTERN PWO KA\
    \REN TONE-3MYANMAR SIGN WESTERN PWO KAREN TONE-4MYANMAR SIGN WESTERN PWO KAREN TONE-5MYANMAR LETTER EASTERN PWO KAREN NNAMYANMAR LETTER EASTERN PWO KAREN YWAMYANMAR LETTER EASTERN PWO KAREN GHWAMYANMAR VOWEL SIGN GEBA KAREN IMYANMAR VOWEL SIGN KAYAH OEMYAN\
    \MAR VOWEL SIGN KAYAH UMYANMAR VOWEL SIGN KAYAH EEMYANMAR LETTER SHAN KAMYANMAR LETTER SHAN KHAMYANMAR LETTER SHAN GAMYANMAR LETTER SHAN CAMYANMAR LETTER SHAN ZAMYANMAR LETTER SHAN NYAMYANMAR LETTER SHAN DAMYANMAR LETTER SHAN NAMYANMAR LETTER SHAN PHAMYANM\
    \AR LETTER SHAN FAMYANMAR LETTER SHAN BAMYANMAR LETTER SHAN THAMYANMAR LETTER SHAN HAMYANMAR CONSONANT SIGN SHAN MEDIAL WAMYANMAR VOWEL SIGN SHAN AAMYANMAR VOWEL SIGN SHAN EMYANMAR VOWEL SIGN SHAN E ABOVEMYANMAR VOWEL SIGN SHAN FINAL YMYANMAR SIGN SHAN TON\
    \E-2MYANMAR SIGN SHAN TONE-3MYANMAR SIGN SHAN TONE-5MYANMAR SIGN SHAN TONE-6MYANMAR SIGN SHAN COUNCIL TONE-2MYANMAR SIGN SHAN COUNCIL TONE-3MYANMAR SIGN SHAN COUNCIL EMPHATIC TONEMYANMAR LETTER RUMAI PALAUNG FAMYANMAR SIGN RUMAI PALAUNG TONE-5MYANMAR SHAN \
    \DIGIT ZEROMYANMAR SHAN DIGIT ONEMYANMAR SHAN DIGIT TWOMYANMAR SHAN DIGIT THREEMYANMAR SHAN DIGIT FOURMYANMAR SHAN DIGIT FIVEMYANMAR SHAN DIGIT SIXMYANMAR SHAN DIGIT SEVENMYANMAR SHAN DIGIT EIGHTMYANMAR SHAN DIGIT NINEMYANMAR SIGN KHAMTI TONE-1MYANMAR SIGN\
    \ KHAMTI TONE-3MYANMAR VOWEL SIGN AITON AMYANMAR VOWEL SIGN AITON AIMYANMAR SYMBOL SHAN ONEMYANMAR SYMBOL SHAN EXCLAMATIONGEORGIAN CAPITAL LETTER ANGEORGIAN CAPITAL LETTER BANGEORGIAN CAPITAL LETTER GANGEORGIAN CAPITAL LETTER DONGEORGIAN CAPITAL LETTER ENG\
    \EORGIAN CAPITAL LETTER VINGEORGIAN CAPITAL LETTER ZENGEORGIAN CAPITAL LETTER TANGEORGIAN CAPITAL LETTER INGEORGIAN CAPITAL LETTER KANGEORGIAN CAPITAL LETTER LASGEORGIAN CAPITAL LETTER MANGEORGIAN CAPITAL LETTER NARGEORGIAN CAPITAL LETTER ONGEORGIAN CAPITA\
    \L LETTER PARGEORGIAN CAPITAL LETTER ZHARGEORGIAN CAPITAL LETTER RAEGEORGIAN CAPITAL LETTER SANGEORGIAN CAPITAL LETTER TARGEORGIAN CAPITAL LETTER UNGEORGIAN CAPITAL LETTER PHARGEORGIAN CAPITAL LETTER KHARGEORGIAN CAPITAL LETTER GHANGEORGIAN CAPITAL LETTER \
    \QARGEORGIAN CAPITAL LETTER SHINGEORGIAN CAPITAL LETTER CHINGEORGIAN CAPITAL LETTER CANGEORGIAN CAPITAL LETTER JILGEORGIAN CAPITAL LETTER CILGEORGIAN CAPITAL LETTER CHARGEORGIAN CAPITAL LETTER XANGEORGIAN CAPITAL LETTER JHANGEORGIAN CAPITAL LETTER HAEGEORG\
    \IAN CAPITAL LETTER HEGEORGIAN CAPITAL LETTER HIEGEORGIAN CAPITAL LETTER WEGEORGIAN CAPITAL LETTER HARGEORGIAN CAPITAL LETTER HOEGEORGIAN CAPITAL LETTER YNGEORGIAN CAPITAL LETTER AENGEORGIAN LETTER ANGEORGIAN LETTER BANGEORGIAN LETTER GANGEORGIAN LETTER DO\
    \NGEORGIAN LETTER ENGEORGIAN LETTER VINGEORGIAN LETTER ZENGEORGIAN LETTER TANGEORGIAN LETTER INGEORGIAN LETTER KANGEORGIAN LETTER LASGEORGIAN LETTER MANGEORGIAN LETTER NARGEORGIAN LETTER ONGEORGIAN LETTER PARGEORGIAN LETTER ZHARGEORGIAN LETTER RAEGEORGIAN \
    \LETTER SANGEORGIAN LETTER TARGEORGIAN LETTER UNGEORGIAN LETTER PHARGEORGIAN LETTER KHARGEORGIAN LETTER GHANGEORGIAN LETTER QARGEORGIAN LETTER SHINGEORGIAN LETTER CHINGEORGIAN LETTER CANGEORGIAN LETTER JILGEORGIAN LETTER CILGEORGIAN LETTER CHARGEORGIAN LET\
    \TER XANGEORGIAN LETTER JHANGEORGIAN LETTER HAEGEORGIAN LETTER HEGEORGIAN LETTER HIEGEORGIAN LETTER WEGEORGIAN LETTER HARGEORGIAN LETTER HOEGEORGIAN LETTER FIGEORGIAN LETTER YNGEORGIAN LETTER ELIFIGEORGIAN LETTER TURNED GANGEORGIAN LETTER AINGEORGIAN PARAG\
    \RAPH SEPARATORMODIFIER LETTER GEORGIAN NARGEORGIAN LETTER AENGEORGIAN LETTER HARD SIGNGEORGIAN LETTER LABIAL SIGNHANGUL CHOSEONG KIYEOKHANGUL CHOSEONG SSANGKIYEOKHANGUL CHOSEONG NIEUNHANGUL CHOSEONG TIKEUTHANGUL CHOSEONG SSANGTIKEUTHANGUL CHOSEONG RIEULHA\
    \NGUL CHOSEONG MIEUMHANGUL CHOSEONG PIEUPHANGUL CHOSEONG SSANGPIEUPHANGUL CHOSEONG SIOSHANGUL CHOSEONG SSANGSIOSHANGUL CHOSEONG IEUNGHANGUL CHOSEONG CIEUCHANGUL CHOSEONG SSANGCIEUCHANGUL CHOSEONG CHIEUCHHANGUL CHOSEONG KHIEUKHHANGUL CHOSEONG THIEUTHHANGUL \
    \CHOSEONG PHIEUPHHANGUL CHOSEONG HIEUHHANGUL CHOSEONG NIEUN-KIYEOKHANGUL CHOSEONG SSANGNIEUNHANGUL CHOSEONG NIEUN-TIKEUTHANGUL CHOSEONG NIEUN-PIEUPHANGUL CHOSEONG TIKEUT-KIYEOKHANGUL CHOSEONG RIEUL-NIEUNHANGUL CHOSEONG SSANGRIEULHANGUL CHOSEONG RIEUL-HIEUH\
    \HANGUL CHOSEONG KAPYEOUNRIEULHANGUL CHOSEONG MIEUM-PIEUPHANGUL CHOSEONG KAPYEOUNMIEUMHANGUL CHOSEONG PIEUP-KIYEOKHANGUL CHOSEONG PIEUP-NIEUNHANGUL CHOSEONG PIEUP-TIKEUTHANGUL CHOSEONG PIEUP-SIOSHANGUL CHOSEONG PIEUP-SIOS-KIYEOKHANGUL CHOSEONG PIEUP-SIOS-T\
    \IKEUTHANGUL CHOSEONG PIEUP-SIOS-PIEUPHANGUL CHOSEONG PIEUP-SSANGSIOSHANGUL CHOSEONG PIEUP-SIOS-CIEUCHANGUL CHOSEONG PIEUP-CIEUCHANGUL CHOSEONG PIEUP-CHIEUCHHANGUL CHOSEONG PIEUP-THIEUTHHANGUL CHOSEONG PIEUP-PHIEUPHHANGUL CHOSEONG KAPYEOUNPIEUPHANGUL CHOSE\
    \ONG KAPYEOUNSSANGPIEUPHANGUL CHOSEONG SIOS-KIYEOKHANGUL CHOSEONG SIOS-NIEUNHANGUL CHOSEONG SIOS-TIKEUTHANGUL CHOSEONG SIOS-RIEULHANGUL CHOSEONG SIOS-MIEUMHANGUL CHOSEONG SIOS-PIEUPHANGUL CHOSEONG SIOS-PIEUP-KIYEOKHANGUL CHOSEONG SIOS-SSANGSIOSHANGUL CHOSE\
    \ONG SIOS-IEUNGHANGUL CHOSEONG SIOS-CIEUCHANGUL CHOSEONG SIOS-CHIEUCHHANGUL CHOSEONG SIOS-KHIEUKHHANGUL CHOSEONG SIOS-THIEUTHHANGUL CHOSEONG SIOS-PHIEUPHHANGUL CHOSEONG SIOS-HIEUHHANGUL CHOSEONG CHITUEUMSIOSHANGUL CHOSEONG CHITUEUMSSANGSIOSHANGUL CHOSEONG \
    \CEONGCHIEUMSIOSHANGUL CHOSEONG CEONGCHIEUMSSANGSIOSHANGUL CHOSEONG PANSIOSHANGUL CHOSEONG IEUNG-KIYEOKHANGUL CHOSEONG IEUNG-TIKEUTHANGUL CHOSEONG IEUNG-MIEUMHANGUL CHOSEONG IEUNG-PIEUPHANGUL CHOSEONG IEUNG-SIOSHANGUL CHOSEONG IEUNG-PANSIOSHANGUL CHOSEONG \
    \SSANGIEUNGHANGUL CHOSEONG IEUNG-CIEUCHANGUL CHOSEONG IEUNG-CHIEUCHHANGUL CHOSEONG IEUNG-THIEUTHHANGUL CHOSEONG IEUNG-PHIEUPHHANGUL CHOSEONG YESIEUNGHANGUL CHOSEONG CIEUC-IEUNGHANGUL CHOSEONG CHITUEUMCIEUCHANGUL CHOSEONG CHITUEUMSSANGCIEUCHANGUL CHOSEONG C\
    \EONGCHIEUMCIEUCHANGUL CHOSEONG CEONGCHIEUMSSANGCIEUCHANGUL CHOSEONG CHIEUCH-KHIEUKHHANGUL CHOSEONG CHIEUCH-HIEUHHANGUL CHOSEONG CHITUEUMCHIEUCHHANGUL CHOSEONG CEONGCHIEUMCHIEUCHHANGUL CHOSEONG PHIEUPH-PIEUPHANGUL CHOSEONG KAPYEOUNPHIEUPHHANGUL CHOSEONG SS\
    \ANGHIEUHHANGUL CHOSEONG YEORINHIEUHHANGUL CHOSEONG KIYEOK-TIKEUTHANGUL CHOSEONG NIEUN-SIOSHANGUL CHOSEONG NIEUN-CIEUCHANGUL CHOSEONG NIEUN-HIEUHHANGUL CHOSEONG TIKEUT-RIEULHANGUL CHOSEONG FILLERHANGUL JUNGSEONG FILLERHANGUL JUNGSEONG AHANGUL JUNGSEONG AEH\
    \ANGUL JUNGSEONG YAHANGUL JUNGSEONG YAEHANGUL JUNGSEONG EOHANGUL JUNGSEONG EHANGUL JUNGSEONG YEOHANGUL JUNGSEONG YEHANGUL JUNGSEONG OHANGUL JUNGSEONG WAHANGUL JUNGSEONG WAEHANGUL JUNGSEONG OEHANGUL JUNGSEONG YOHANGUL JUNGSEONG UHANGUL JUNGSEONG WEOHANGUL J\
    \UNGSEONG WEHANGUL JUNGSEONG WIHANGUL JUNGSEONG YUHANGUL JUNGSEONG EUHANGUL JUNGSEONG YIHANGUL JUNGSEONG IHANGUL JUNGSEONG A-OHANGUL JUNGSEONG A-UHANGUL JUNGSEONG YA-OHANGUL JUNGSEONG YA-YOHANGUL JUNGSEONG EO-OHANGUL JUNGSEONG EO-UHANGUL JUNGSEONG EO-EUHAN\
    \GUL JUNGSEONG YEO-OHANGUL JUNGSEONG YEO-UHANGUL JUNGSEONG O-EOHANGUL JUNGSEONG O-EHANGUL JUNGSEONG O-YEHANGUL JUNGSEONG O-OHANGUL JUNGSEONG O-UHANGUL JUNGSEONG YO-YAHANGUL JUNGSEONG YO-YAEHANGUL JUNGSEONG YO-YEOHANGUL JUNGSEONG YO-OHANGUL JUNGSEONG YO-IHA\
    \NGUL JUNGSEONG U-AHANGUL JUNGSEONG U-AEHANGUL JUNGSEONG U-EO-EUHANGUL JUNGSEONG U-YEHANGUL JUNGSEONG U-UHANGUL JUNGSEONG YU-AHANGUL JUNGSEONG YU-EOHANGUL JUNGSEONG YU-EHANGUL JUNGSEONG YU-YEOHANGUL JUNGSEONG YU-YEHANGUL JUNGSEONG YU-UHANGUL JUNGSEONG YU-I\
    \HANGUL JUNGSEONG EU-UHANGUL JUNGSEONG EU-EUHANGUL JUNGSEONG YI-UHANGUL JUNGSEONG I-AHANGUL JUNGSEONG I-YAHANGUL JUNGSEONG I-OHANGUL JUNGSEONG I-UHANGUL JUNGSEONG I-EUHANGUL JUNGSEONG I-ARAEAHANGUL JUNGSEONG ARAEAHANGUL JUNGSEONG ARAEA-EOHANGUL JUNGSEONG A\
    \RAEA-UHANGUL JUNGSEONG ARAEA-IHANGUL JUNGSEONG SSANGARAEAHANGUL JUNGSEONG A-EUHANGUL JUNGSEONG YA-UHANGUL JUNGSEONG YEO-YAHANGUL JUNGSEONG O-YAHANGUL JUNGSEONG O-YAEHANGUL JONGSEONG KIYEOKHANGUL JONGSEONG SSANGKIYEOKHANGUL JONGSEONG KIYEOK-SIOSHANGUL JONG\
    \SEONG NIEUNHANGUL JONGSEONG NIEUN-CIEUCHANGUL JONGSEONG NIEUN-HIEUHHANGUL JONGSEONG TIKEUTHANGUL JONGSEONG RIEULHANGUL JONGSEONG RIEUL-KIYEOKHANGUL JONGSEONG RIEUL-MIEUMHANGUL JONGSEONG RIEUL-PIEUPHANGUL JONGSEONG RIEUL-SIOSHANGUL JONGSEONG RIEUL-THIEUTHH\
    \ANGUL JONGSEONG RIEUL-PHIEUPHHANGUL JONGSEONG RIEUL-HIEUHHANGUL JONGSEONG MIEUMHANGUL JONGSEONG PIEUPHANGUL JONGSEONG PIEUP-SIOSHANGUL JONGSEONG SIOSHANGUL JONGSEONG SSANGSIOSHANGUL JONGSEONG IEUNGHANGUL JONGSEONG CIEUCHANGUL JONGSEONG CHIEUCHHANGUL JONGS\
    \EONG KHIEUKHHANGUL JONGSEONG THIEUTHHANGUL JONGSEONG PHIEUPHHANGUL JONGSEONG HIEUHHANGUL JONGSEONG KIYEOK-RIEULHANGUL JONGSEONG KIYEOK-SIOS-KIYEOKHANGUL JONGSEONG NIEUN-KIYEOKHANGUL JONGSEONG NIEUN-TIKEUTHANGUL JONGSEONG NIEUN-SIOSHANGUL JONGSEONG NIEUN-P\
    \ANSIOSHANGUL JONGSEONG NIEUN-THIEUTHHANGUL JONGSEONG TIKEUT-KIYEOKHANGUL JONGSEONG TIKEUT-RIEULHANGUL JONGSEONG RIEUL-KIYEOK-SIOSHANGUL JONGSEONG RIEUL-NIEUNHANGUL JONGSEONG RIEUL-TIKEUTHANGUL JONGSEONG RIEUL-TIKEUT-HIEUHHANGUL JONGSEONG SSANGRIEULHANGUL \
    \JONGSEONG RIEUL-MIEUM-KIYEOKHANGUL JONGSEONG RIEUL-MIEUM-SIOSHANGUL JONGSEONG RIEUL-PIEUP-SIOSHANGUL JONGSEONG RIEUL-PIEUP-HIEUHHANGUL JONGSEONG RIEUL-KAPYEOUNPIEUPHANGUL JONGSEONG RIEUL-SSANGSIOSHANGUL JONGSEONG RIEUL-PANSIOSHANGUL JONGSEONG RIEUL-KHIEUK\
    \HHANGUL JONGSEONG RIEUL-YEORINHIEUHHANGUL JONGSEONG MIEUM-KIYEOKHANGUL JONGSEONG MIEUM-RIEULHANGUL JONGSEONG MIEUM-PIEUPHANGUL JONGSEONG MIEUM-SIOSHANGUL JONGSEONG MIEUM-SSANGSIOSHANGUL JONGSEONG MIEUM-PANSIOSHANGUL JONGSEONG MIEUM-CHIEUCHHANGUL JONGSEONG\
    \ MIEUM-HIEUHHANGUL JONGSEONG KAPYEOUNMIEUMHANGUL JONGSEONG PIEUP-RIEULHANGUL JONGSEONG PIEUP-PHIEUPHHANGUL JONGSEONG PIEUP-HIEUHHANGUL JONGSEONG KAPYEOUNPIEUPHANGUL JONGSEONG SIOS-KIYEOKHANGUL JONGSEONG SIOS-TIKEUTHANGUL JONGSEONG SIOS-RIEULHANGUL JONGSEO\
    \NG SIOS-PIEUPHANGUL JONGSEONG PANSIOSHANGUL JONGSEONG IEUNG-KIYEOKHANGUL JONGSEONG IEUNG-SSANGKIYEOKHANGUL JONGSEONG SSANGIEUNGHANGUL JONGSEONG IEUNG-KHIEUKHHANGUL JONGSEONG YESIEUNGHANGUL JONGSEONG YESIEUNG-SIOSHANGUL JONGSEONG YESIEUNG-PANSIOSHANGUL JON\
    \GSEONG PHIEUPH-PIEUPHANGUL JONGSEONG KAPYEOUNPHIEUPHHANGUL JONGSEONG HIEUH-NIEUNHANGUL JONGSEONG HIEUH-RIEULHANGUL JONGSEONG HIEUH-MIEUMHANGUL JONGSEONG HIEUH-PIEUPHANGUL JONGSEONG YEORINHIEUHHANGUL JONGSEONG KIYEOK-NIEUNHANGUL JONGSEONG KIYEOK-PIEUPHANGU\
    \L JONGSEONG KIYEOK-CHIEUCHHANGUL JONGSEONG KIYEOK-KHIEUKHHANGUL JONGSEONG KIYEOK-HIEUHHANGUL JONGSEONG SSANGNIEUNETHIOPIC SYLLABLE HAETHIOPIC SYLLABLE HUETHIOPIC SYLLABLE HIETHIOPIC SYLLABLE HAAETHIOPIC SYLLABLE HEEETHIOPIC SYLLABLE HEETHIOPIC SYLLABLE HO\
    \ETHIOPIC SYLLABLE HOAETHIOPIC SYLLABLE LAETHIOPIC SYLLABLE LUETHIOPIC SYLLABLE LIETHIOPIC SYLLABLE LAAETHIOPIC SYLLABLE LEEETHIOPIC SYLLABLE LEETHIOPIC SYLLABLE LOETHIOPIC SYLLABLE LWAETHIOPIC SYLLABLE HHAETHIOPIC SYLLABLE HHUETHIOPIC SYLLABLE HHIETHIOPIC\
    \ SYLLABLE HHAAETHIOPIC SYLLABLE HHEEETHIOPIC SYLLABLE HHEETHIOPIC SYLLABLE HHOETHIOPIC SYLLABLE HHWAETHIOPIC SYLLABLE MAETHIOPIC SYLLABLE MUETHIOPIC SYLLABLE MIETHIOPIC SYLLABLE MAAETHIOPIC SYLLABLE MEEETHIOPIC SYLLABLE MEETHIOPIC SYLLABLE MOETHIOPIC SYLL\
    \ABLE MWAETHIOPIC SYLLABLE SZAETHIOPIC SYLLABLE SZUETHIOPIC SYLLABLE SZIETHIOPIC SYLLABLE SZAAETHIOPIC SYLLABLE SZEEETHIOPIC SYLLABLE SZEETHIOPIC SYLLABLE SZOETHIOPIC SYLLABLE SZWAETHIOPIC SYLLABLE RAETHIOPIC SYLLABLE RUETHIOPIC SYLLABLE RIETHIOPIC SYLLABL\
    \E RAAETHIOPIC SYLLABLE REEETHIOPIC SYLLABLE REETHIOPIC SYLLABLE ROETHIOPIC SYLLABLE RWAETHIOPIC SYLLABLE SAETHIOPIC SYLLABLE SUETHIOPIC SYLLABLE SIETHIOPIC SYLLABLE SAAETHIOPIC SYLLABLE SEEETHIOPIC SYLLABLE SEETHIOPIC SYLLABLE SOETHIOPIC SYLLABLE SWAETHIO\
    \PIC SYLLABLE SHAETHIOPIC SYLLABLE SHUETHIOPIC SYLLABLE SHIETHIOPIC SYLLABLE SHAAETHIOPIC SYLLABLE SHEEETHIOPIC SYLLABLE SHEETHIOPIC SYLLABLE SHOETHIOPIC SYLLABLE SHWAETHIOPIC SYLLABLE QAETHIOPIC SYLLABLE QUETHIOPIC SYLLABLE QIETHIOPIC SYLLABLE QAAETHIOPIC\
    \ SYLLABLE QEEETHIOPIC SYLLABLE QEETHIOPIC SYLLABLE QOETHIOPIC SYLLABLE QOAETHIOPIC SYLLABLE QWAETHIOPIC SYLLABLE QWIETHIOPIC SYLLABLE QWAAETHIOPIC SYLLABLE QWEEETHIOPIC SYLLABLE QWEETHIOPIC SYLLABLE QHAETHIOPIC SYLLABLE QHUETHIOPIC SYLLABLE QHIETHIOPIC SY\
    \LLABLE QHAAETHIOPIC SYLLABLE QHEEETHIOPIC SYLLABLE QHEETHIOPIC SYLLABLE QHOETHIOPIC SYLLABLE QHWAETHIOPIC SYLLABLE QHWIETHIOPIC SYLLABLE QHWAAETHIOPIC SYLLABLE QHWEEETHIOPIC SYLLABLE QHWEETHIOPIC SYLLABLE BAETHIOPIC SYLLABLE BUETHIOPIC SYLLABLE BIETHIOPIC\
    \ SYLLABLE BAAETHIOPIC SYLLABLE BEEETHIOPIC SYLLABLE BEETHIOPIC SYLLABLE BOETHIOPIC SYLLABLE BWAETHIOPIC SYLLABLE VAETHIOPIC SYLLABLE VUETHIOPIC SYLLABLE VIETHIOPIC SYLLABLE VAAETHIOPIC SYLLABLE VEEETHIOPIC SYLLABLE VEETHIOPIC SYLLABLE VOETHIOPIC SYLLABLE \
    \VWAETHIOPIC SYLLABLE TAETHIOPIC SYLLABLE TUETHIOPIC SYLLABLE TIETHIOPIC SYLLABLE TAAETHIOPIC SYLLABLE TEEETHIOPIC SYLLABLE TEETHIOPIC SYLLABLE TOETHIOPIC SYLLABLE TWAETHIOPIC SYLLABLE CAETHIOPIC SYLLABLE CUETHIOPIC SYLLABLE CIETHIOPIC SYLLABLE CAAETHIOPIC\
    \ SYLLABLE CEEETHIOPIC SYLLABLE CEETHIOPIC SYLLABLE COETHIOPIC SYLLABLE CWAETHIOPIC SYLLABLE XAETHIOPIC SYLLABLE XUETHIOPIC SYLLABLE XIETHIOPIC SYLLABLE XAAETHIOPIC SYLLABLE XEEETHIOPIC SYLLABLE XEETHIOPIC SYLLABLE XOETHIOPIC SYLLABLE XOAETHIOPIC SYLLABLE \
    \XWAETHIOPIC SYLLABLE XWIETHIOPIC SYLLABLE XWAAETHIOPIC SYLLABLE XWEEETHIOPIC SYLLABLE XWEETHIOPIC SYLLABLE NAETHIOPIC SYLLABLE NUETHIOPIC SYLLABLE NIETHIOPIC SYLLABLE NAAETHIOPIC SYLLABLE NEEETHIOPIC SYLLABLE NEETHIOPIC SYLLABLE NOETHIOPIC SYLLABLE NWAETH\
    \IOPIC SYLLABLE NYAETHIOPIC SYLLABLE NYUETHIOPIC SYLLABLE NYIETHIOPIC SYLLABLE NYAAETHIOPIC SYLLABLE NYEEETHIOPIC SYLLABLE NYEETHIOPIC SYLLABLE NYOETHIOPIC SYLLABLE NYWAETHIOPIC SYLLABLE GLOTTAL AETHIOPIC SYLLABLE GLOTTAL UETHIOPIC SYLLABLE GLOTTAL IETHIOP\
    \IC SYLLABLE GLOTTAL AAETHIOPIC SYLLABLE GLOTTAL EEETHIOPIC SYLLABLE GLOTTAL EETHIOPIC SYLLABLE GLOTTAL OETHIOPIC SYLLABLE GLOTTAL WAETHIOPIC SYLLABLE KAETHIOPIC SYLLABLE KUETHIOPIC SYLLABLE KIETHIOPIC SYLLABLE KAAETHIOPIC SYLLABLE KEEETHIOPIC SYLLABLE KEE\
    \THIOPIC SYLLABLE KOETHIOPIC SYLLABLE KOAETHIOPIC SYLLABLE KWAETHIOPIC SYLLABLE KWIETHIOPIC SYLLABLE KWAAETHIOPIC SYLLABLE KWEEETHIOPIC SYLLABLE KWEETHIOPIC SYLLABLE KXAETHIOPIC SYLLABLE KXUETHIOPIC SYLLABLE KXIETHIOPIC SYLLABLE KXAAETHIOPIC SYLLABLE KXEEE\
    \THIOPIC SYLLABLE KXEETHIOPIC SYLLABLE KXOETHIOPIC SYLLABLE KXWAETHIOPIC SYLLABLE KXWIETHIOPIC SYLLABLE KXWAAETHIOPIC SYLLABLE KXWEEETHIOPIC SYLLABLE KXWEETHIOPIC SYLLABLE WAETHIOPIC SYLLABLE WUETHIOPIC SYLLABLE WIETHIOPIC SYLLABLE WAAETHIOPIC SYLLABLE WEE\
    \ETHIOPIC SYLLABLE WEETHIOPIC SYLLABLE WOETHIOPIC SYLLABLE WOAETHIOPIC SYLLABLE PHARYNGEAL AETHIOPIC SYLLABLE PHARYNGEAL UETHIOPIC SYLLABLE PHARYNGEAL IETHIOPIC SYLLABLE PHARYNGEAL AAETHIOPIC SYLLABLE PHARYNGEAL EEETHIOPIC SYLLABLE PHARYNGEAL EETHIOPIC SYL\
    \LABLE PHARYNGEAL OETHIOPIC SYLLABLE ZAETHIOPIC SYLLABLE ZUETHIOPIC SYLLABLE ZIETHIOPIC SYLLABLE ZAAETHIOPIC SYLLABLE ZEEETHIOPIC SYLLABLE ZEETHIOPIC SYLLABLE ZOETHIOPIC SYLLABLE ZWAETHIOPIC SYLLABLE ZHAETHIOPIC SYLLABLE ZHUETHIOPIC SYLLABLE ZHIETHIOPIC SY\
    \LLABLE ZHAAETHIOPIC SYLLABLE ZHEEETHIOPIC SYLLABLE ZHEETHIOPIC SYLLABLE ZHOETHIOPIC SYLLABLE ZHWAETHIOPIC SYLLABLE YAETHIOPIC SYLLABLE YUETHIOPIC SYLLABLE YIETHIOPIC SYLLABLE YAAETHIOPIC SYLLABLE YEEETHIOPIC SYLLABLE YEETHIOPIC SYLLABLE YOETHIOPIC SYLLABL\
    \E YOAETHIOPIC SYLLABLE DAETHIOPIC SYLLABLE DUETHIOPIC SYLLABLE DIETHIOPIC SYLLABLE DAAETHIOPIC SYLLABLE DEEETHIOPIC SYLLABLE DEETHIOPIC SYLLABLE DOETHIOPIC SYLLABLE DWAETHIOPIC SYLLABLE DDAETHIOPIC SYLLABLE DDUETHIOPIC SYLLABLE DDIETHIOPIC SYLLABLE DDAAET\
    \HIOPIC SYLLABLE DDEEETHIOPIC SYLLABLE DDEETHIOPIC SYLLABLE DDOETHIOPIC SYLLABLE DDWAETHIOPIC SYLLABLE JAETHIOPIC SYLLABLE JUETHIOPIC SYLLABLE JIETHIOPIC SYLLABLE JAAETHIOPIC SYLLABLE JEEETHIOPIC SYLLABLE JEETHIOPIC SYLLABLE JOETHIOPIC SYLLABLE JWAETHIOPIC\
    \ SYLLABLE GAETHIOPIC SYLLABLE GUETHIOPIC SYLLABLE GIETHIOPIC SYLLABLE GAAETHIOPIC SYLLABLE GEEETHIOPIC SYLLABLE GEETHIOPIC SYLLABLE GOETHIOPIC SYLLABLE GOAETHIOPIC SYLLABLE GWAETHIOPIC SYLLABLE GWIETHIOPIC SYLLABLE GWAAETHIOPIC SYLLABLE GWEEETHIOPIC SYLLA\
    \BLE GWEETHIOPIC SYLLABLE GGAETHIOPIC SYLLABLE GGUETHIOPIC SYLLABLE GGIETHIOPIC SYLLABLE GGAAETHIOPIC SYLLABLE GGEEETHIOPIC SYLLABLE GGEETHIOPIC SYLLABLE GGOETHIOPIC SYLLABLE GGWAAETHIOPIC SYLLABLE THAETHIOPIC SYLLABLE THUETHIOPIC SYLLABLE THIETHIOPIC SYLL\
    \ABLE THAAETHIOPIC SYLLABLE THEEETHIOPIC SYLLABLE THEETHIOPIC SYLLABLE THOETHIOPIC SYLLABLE THWAETHIOPIC SYLLABLE CHAETHIOPIC SYLLABLE CHUETHIOPIC SYLLABLE CHIETHIOPIC SYLLABLE CHAAETHIOPIC SYLLABLE CHEEETHIOPIC SYLLABLE CHEETHIOPIC SYLLABLE CHOETHIOPIC SY\
    \LLABLE CHWAETHIOPIC SYLLABLE PHAETHIOPIC SYLLABLE PHUETHIOPIC SYLLABLE PHIETHIOPIC SYLLABLE PHAAETHIOPIC SYLLABLE PHEEETHIOPIC SYLLABLE PHEETHIOPIC SYLLABLE PHOETHIOPIC SYLLABLE PHWAETHIOPIC SYLLABLE TSAETHIOPIC SYLLABLE TSUETHIOPIC SYLLABLE TSIETHIOPIC S\
    \YLLABLE TSAAETHIOPIC SYLLABLE TSEEETHIOPIC SYLLABLE TSEETHIOPIC SYLLABLE TSOETHIOPIC SYLLABLE TSWAETHIOPIC SYLLABLE TZAETHIOPIC SYLLABLE TZUETHIOPIC SYLLABLE TZIETHIOPIC SYLLABLE TZAAETHIOPIC SYLLABLE TZEEETHIOPIC SYLLABLE TZEETHIOPIC SYLLABLE TZOETHIOPIC\
    \ SYLLABLE TZOAETHIOPIC SYLLABLE FAETHIOPIC SYLLABLE FUETHIOPIC SYLLABLE FIETHIOPIC SYLLABLE FAAETHIOPIC SYLLABLE FEEETHIOPIC SYLLABLE FEETHIOPIC SYLLABLE FOETHIOPIC SYLLABLE FWAETHIOPIC SYLLABLE PAETHIOPIC SYLLABLE PUETHIOPIC SYLLABLE PIETHIOPIC SYLLABLE \
    \PAAETHIOPIC SYLLABLE PEEETHIOPIC SYLLABLE PEETHIOPIC SYLLABLE POETHIOPIC SYLLABLE PWAETHIOPIC SYLLABLE RYAETHIOPIC SYLLABLE MYAETHIOPIC SYLLABLE FYAETHIOPIC COMBINING GEMINATION AND VOWEL LENGTH MARKETHIOPIC COMBINING VOWEL LENGTH MARKETHIOPIC COMBINING G\
    \EMINATION MARKETHIOPIC SECTION MARKETHIOPIC WORDSPACEETHIOPIC FULL STOPETHIOPIC COMMAETHIOPIC SEMICOLONETHIOPIC COLONETHIOPIC PREFACE COLONETHIOPIC QUESTION MARKETHIOPIC PARAGRAPH SEPARATORETHIOPIC DIGIT ONEETHIOPIC DIGIT TWOETHIOPIC DIGIT THREEETHIOPIC D\
    \IGIT FOURETHIOPIC DIGIT FIVEETHIOPIC DIGIT SIXETHIOPIC DIGIT SEVENETHIOPIC DIGIT EIGHTETHIOPIC DIGIT NINEETHIOPIC NUMBER TENETHIOPIC NUMBER TWENTYETHIOPIC NUMBER THIRTYETHIOPIC NUMBER FORTYETHIOPIC NUMBER FIFTYETHIOPIC NUMBER SIXTYETHIOPIC NUMBER SEVENTYE\
    \THIOPIC NUMBER EIGHTYETHIOPIC NUMBER NINETYETHIOPIC NUMBER HUNDREDETHIOPIC NUMBER TEN THOUSANDETHIOPIC SYLLABLE SEBATBEIT MWAETHIOPIC SYLLABLE MWIETHIOPIC SYLLABLE MWEEETHIOPIC SYLLABLE MWEETHIOPIC SYLLABLE SEBATBEIT BWAETHIOPIC SYLLABLE BWIETHIOPIC SYLLA\
    \BLE BWEEETHIOPIC SYLLABLE BWEETHIOPIC SYLLABLE SEBATBEIT FWAETHIOPIC SYLLABLE FWIETHIOPIC SYLLABLE FWEEETHIOPIC SYLLABLE FWEETHIOPIC SYLLABLE SEBATBEIT PWAETHIOPIC SYLLABLE PWIETHIOPIC SYLLABLE PWEEETHIOPIC SYLLABLE PWEETHIOPIC TONAL MARK YIZETETHIOPIC TO\
    \NAL MARK DERETETHIOPIC TONAL MARK RIKRIKETHIOPIC TONAL MARK SHORT RIKRIKETHIOPIC TONAL MARK DIFATETHIOPIC TONAL MARK KENATETHIOPIC TONAL MARK CHIRETETHIOPIC TONAL MARK HIDETETHIOPIC TONAL MARK DERET-HIDETETHIOPIC TONAL MARK KURTCHEROKEE LETTER ACHEROKEE L\
    \ETTER ECHEROKEE LETTER ICHEROKEE LETTER OCHEROKEE LETTER UCHEROKEE LETTER VCHEROKEE LETTER GACHEROKEE LETTER KACHEROKEE LETTER GECHEROKEE LETTER GICHEROKEE LETTER GOCHEROKEE LETTER GUCHEROKEE LETTER GVCHEROKEE LETTER HACHEROKEE LETTER HECHEROKEE LETTER HI\
    \CHEROKEE LETTER HOCHEROKEE LETTER HUCHEROKEE LETTER HVCHEROKEE LETTER LACHEROKEE LETTER LECHEROKEE LETTER LICHEROKEE LETTER LOCHEROKEE LETTER LUCHEROKEE LETTER LVCHEROKEE LETTER MACHEROKEE LETTER MECHEROKEE LETTER MICHEROKEE LETTER MOCHEROKEE LETTER MUCHE\
    \ROKEE LETTER NACHEROKEE LETTER HNACHEROKEE LETTER NAHCHEROKEE LETTER NECHEROKEE LETTER NICHEROKEE LETTER NOCHEROKEE LETTER NUCHEROKEE LETTER NVCHEROKEE LETTER QUACHEROKEE LETTER QUECHEROKEE LETTER QUICHEROKEE LETTER QUOCHEROKEE LETTER QUUCHEROKEE LETTER Q\
    \UVCHEROKEE LETTER SACHEROKEE LETTER SCHEROKEE LETTER SECHEROKEE LETTER SICHEROKEE LETTER SOCHEROKEE LETTER SUCHEROKEE LETTER SVCHEROKEE LETTER DACHEROKEE LETTER TACHEROKEE LETTER DECHEROKEE LETTER TECHEROKEE LETTER DICHEROKEE LETTER TICHEROKEE LETTER DOCH\
    \EROKEE LETTER DUCHEROKEE LETTER DVCHEROKEE LETTER DLACHEROKEE LETTER TLACHEROKEE LETTER TLECHEROKEE LETTER TLICHEROKEE LETTER TLOCHEROKEE LETTER TLUCHEROKEE LETTER TLVCHEROKEE LETTER TSACHEROKEE LETTER TSECHEROKEE LETTER TSICHEROKEE LETTER TSOCHEROKEE LET\
    \TER TSUCHEROKEE LETTER TSVCHEROKEE LETTER WACHEROKEE LETTER WECHEROKEE LETTER WICHEROKEE LETTER WOCHEROKEE LETTER WUCHEROKEE LETTER WVCHEROKEE LETTER YACHEROKEE LETTER YECHEROKEE LETTER YICHEROKEE LETTER YOCHEROKEE LETTER YUCHEROKEE LETTER YVCHEROKEE LETT\
    \ER MVCHEROKEE SMALL LETTER YECHEROKEE SMALL LETTER YICHEROKEE SMALL LETTER YOCHEROKEE SMALL LETTER YUCHEROKEE SMALL LETTER YVCHEROKEE SMALL LETTER MVCANADIAN SYLLABICS HYPHENCANADIAN SYLLABICS ECANADIAN SYLLABICS AAICANADIAN SYLLABICS ICANADIAN SYLLABICS \
    \IICANADIAN SYLLABICS OCANADIAN SYLLABICS OOCANADIAN SYLLABICS Y-CREE OOCANADIAN SYLLABICS CARRIER EECANADIAN SYLLABICS CARRIER ICANADIAN SYLLABICS ACANADIAN SYLLABICS AACANADIAN SYLLABICS WECANADIAN SYLLABICS WEST-CREE WECANADIAN SYLLABICS WICANADIAN SYLL\
    \ABICS WEST-CREE WICANADIAN SYLLABICS WIICANADIAN SYLLABICS WEST-CREE WIICANADIAN SYLLABICS WOCANADIAN SYLLABICS WEST-CREE WOCANADIAN SYLLABICS WOOCANADIAN SYLLABICS WEST-CREE WOOCANADIAN SYLLABICS NASKAPI WOOCANADIAN SYLLABICS WACANADIAN SYLLABICS WEST-CR\
    \EE WACANADIAN SYLLABICS WAACANADIAN SYLLABICS WEST-CREE WAACANADIAN SYLLABICS NASKAPI WAACANADIAN SYLLABICS AICANADIAN SYLLABICS Y-CREE WCANADIAN SYLLABICS GLOTTAL STOPCANADIAN SYLLABICS FINAL ACUTECANADIAN SYLLABICS FINAL GRAVECANADIAN SYLLABICS FINAL BO\
    \TTOM HALF RINGCANADIAN SYLLABICS FINAL TOP HALF RINGCANADIAN SYLLABICS FINAL RIGHT HALF RINGCANADIAN SYLLABICS FINAL RINGCANADIAN SYLLABICS FINAL DOUBLE ACUTECANADIAN SYLLABICS FINAL DOUBLE SHORT VERTICAL STROKESCANADIAN SYLLABICS FINAL MIDDLE DOTCANADIAN\
    \ SYLLABICS FINAL SHORT HORIZONTAL STROKECANADIAN SYLLABICS FINAL PLUSCANADIAN SYLLABICS FINAL DOWN TACKCANADIAN SYLLABICS ENCANADIAN SYLLABICS INCANADIAN SYLLABICS ONCANADIAN SYLLABICS ANCANADIAN SYLLABICS PECANADIAN SYLLABICS PAAICANADIAN SYLLABICS PICAN\
    \ADIAN SYLLABICS PIICANADIAN SYLLABICS POCANADIAN SYLLABICS POOCANADIAN SYLLABICS Y-CREE POOCANADIAN SYLLABICS CARRIER HEECANADIAN SYLLABICS CARRIER HICANADIAN SYLLABICS PACANADIAN SYLLABICS PAACANADIAN SYLLABICS PWECANADIAN SYLLABICS WEST-CREE PWECANADIAN\
    \ SYLLABICS PWICANADIAN SYLLABICS WEST-CREE PWICANADIAN SYLLABICS PWIICANADIAN SYLLABICS WEST-CREE PWIICANADIAN SYLLABICS PWOCANADIAN SYLLABICS WEST-CREE PWOCANADIAN SYLLABICS PWOOCANADIAN SYLLABICS WEST-CREE PWOOCANADIAN SYLLABICS PWACANADIAN SYLLABICS WE\
    \ST-CREE PWACANADIAN SYLLABICS PWAACANADIAN SYLLABICS WEST-CREE PWAACANADIAN SYLLABICS Y-CREE PWAACANADIAN SYLLABICS PCANADIAN SYLLABICS WEST-CREE PCANADIAN SYLLABICS CARRIER HCANADIAN SYLLABICS TECANADIAN SYLLABICS TAAICANADIAN SYLLABICS TICANADIAN SYLLAB\
    \ICS TIICANADIAN SYLLABICS TOCANADIAN SYLLABICS TOOCANADIAN SYLLABICS Y-CREE TOOCANADIAN SYLLABICS CARRIER DEECANADIAN SYLLABICS CARRIER DICANADIAN SYLLABICS TACANADIAN SYLLABICS TAACANADIAN SYLLABICS TWECANADIAN SYLLABICS WEST-CREE TWECANADIAN SYLLABICS T\
    \WICANADIAN SYLLABICS WEST-CREE TWICANADIAN SYLLABICS TWIICANADIAN SYLLABICS WEST-CREE TWIICANADIAN SYLLABICS TWOCANADIAN SYLLABICS WEST-CREE TWOCANADIAN SYLLABICS TWOOCANADIAN SYLLABICS WEST-CREE TWOOCANADIAN SYLLABICS TWACANADIAN SYLLABICS WEST-CREE TWAC\
    \ANADIAN SYLLABICS TWAACANADIAN SYLLABICS WEST-CREE TWAACANADIAN SYLLABICS NASKAPI TWAACANADIAN SYLLABICS TCANADIAN SYLLABICS TTECANADIAN SYLLABICS TTICANADIAN SYLLABICS TTOCANADIAN SYLLABICS TTACANADIAN SYLLABICS KECANADIAN SYLLABICS KAAICANADIAN SYLLABIC\
    \S KICANADIAN SYLLABICS KIICANADIAN SYLLABICS KOCANADIAN SYLLABICS KOOCANADIAN SYLLABICS Y-CREE KOOCANADIAN SYLLABICS KACANADIAN SYLLABICS KAACANADIAN SYLLABICS KWECANADIAN SYLLABICS WEST-CREE KWECANADIAN SYLLABICS KWICANADIAN SYLLABICS WEST-CREE KWICANADI\
    \AN SYLLABICS KWIICANADIAN SYLLABICS WEST-CREE KWIICANADIAN SYLLABICS KWOCANADIAN SYLLABICS WEST-CREE KWOCANADIAN SYLLABICS KWOOCANADIAN SYLLABICS WEST-CREE KWOOCANADIAN SYLLABICS KWACANADIAN SYLLABICS WEST-CREE KWACANADIAN SYLLABICS KWAACANADIAN SYLLABICS\
    \ WEST-CREE KWAACANADIAN SYLLABICS NASKAPI KWAACANADIAN SYLLABICS KCANADIAN SYLLABICS KWCANADIAN SYLLABICS SOUTH-SLAVEY KEHCANADIAN SYLLABICS SOUTH-SLAVEY KIHCANADIAN SYLLABICS SOUTH-SLAVEY KOHCANADIAN SYLLABICS SOUTH-SLAVEY KAHCANADIAN SYLLABICS CECANADIA\
    \N SYLLABICS CAAICANADIAN SYLLABICS CICANADIAN SYLLABICS CIICANADIAN SYLLABICS COCANADIAN SYLLABICS COOCANADIAN SYLLABICS Y-CREE COOCANADIAN SYLLABICS CACANADIAN SYLLABICS CAACANADIAN SYLLABICS CWECANADIAN SYLLABICS WEST-CREE CWECANADIAN SYLLABICS CWICANAD\
    \IAN SYLLABICS WEST-CREE CWICANADIAN SYLLABICS CWIICANADIAN SYLLABICS WEST-CREE CWIICANADIAN SYLLABICS CWOCANADIAN SYLLABICS WEST-CREE CWOCANADIAN SYLLABICS CWOOCANADIAN SYLLABICS WEST-CREE CWOOCANADIAN SYLLABICS CWACANADIAN SYLLABICS WEST-CREE CWACANADIAN\
    \ SYLLABICS CWAACANADIAN SYLLABICS WEST-CREE CWAACANADIAN SYLLABICS NASKAPI CWAACANADIAN SYLLABICS CCANADIAN SYLLABICS SAYISI THCANADIAN SYLLABICS MECANADIAN SYLLABICS MAAICANADIAN SYLLABICS MICANADIAN SYLLABICS MIICANADIAN SYLLABICS MOCANADIAN SYLLABICS M\
    \OOCANADIAN SYLLABICS Y-CREE MOOCANADIAN SYLLABICS MACANADIAN SYLLABICS MAACANADIAN SYLLABICS MWECANADIAN SYLLABICS WEST-CREE MWECANADIAN SYLLABICS MWICANADIAN SYLLABICS WEST-CREE MWICANADIAN SYLLABICS MWIICANADIAN SYLLABICS WEST-CREE MWIICANADIAN SYLLABIC\
    \S MWOCANADIAN SYLLABICS WEST-CREE MWOCANADIAN SYLLABICS MWOOCANADIAN SYLLABICS WEST-CREE MWOOCANADIAN SYLLABICS MWACANADIAN SYLLABICS WEST-CREE MWACANADIAN SYLLABICS MWAACANADIAN SYLLABICS WEST-CREE MWAACANADIAN SYLLABICS NASKAPI MWAACANADIAN SYLLABICS MC\
    \ANADIAN SYLLABICS WEST-CREE MCANADIAN SYLLABICS MHCANADIAN SYLLABICS ATHAPASCAN MCANADIAN SYLLABICS SAYISI MCANADIAN SYLLABICS NECANADIAN SYLLABICS NAAICANADIAN SYLLABICS NICANADIAN SYLLABICS NIICANADIAN SYLLABICS NOCANADIAN SYLLABICS NOOCANADIAN SYLLABIC\
    \S Y-CREE NOOCANADIAN SYLLABICS NACANADIAN SYLLABICS NAACANADIAN SYLLABICS NWECANADIAN SYLLABICS WEST-CREE NWECANADIAN SYLLABICS NWACANADIAN SYLLABICS WEST-CREE NWACANADIAN SYLLABICS NWAACANADIAN SYLLABICS WEST-CREE NWAACANADIAN SYLLABICS NASKAPI NWAACANAD\
    \IAN SYLLABICS NCANADIAN SYLLABICS CARRIER NGCANADIAN SYLLABICS NHCANADIAN SYLLABICS LECANADIAN SYLLABICS LAAICANADIAN SYLLABICS LICANADIAN SYLLABICS LIICANADIAN SYLLABICS LOCANADIAN SYLLABICS LOOCANADIAN SYLLABICS Y-CREE LOOCANADIAN SYLLABICS LACANADIAN S\
    \YLLABICS LAACANADIAN SYLLABICS LWECANADIAN SYLLABICS WEST-CREE LWECANADIAN SYLLABICS LWICANADIAN SYLLABICS WEST-CREE LWICANADIAN SYLLABICS LWIICANADIAN SYLLABICS WEST-CREE LWIICANADIAN SYLLABICS LWOCANADIAN SYLLABICS WEST-CREE LWOCANADIAN SYLLABICS LWOOCA\
    \NADIAN SYLLABICS WEST-CREE LWOOCANADIAN SYLLABICS LWACANADIAN SYLLABICS WEST-CREE LWACANADIAN SYLLABICS LWAACANADIAN SYLLABICS WEST-CREE LWAACANADIAN SYLLABICS LCANADIAN SYLLABICS WEST-CREE LCANADIAN SYLLABICS MEDIAL LCANADIAN SYLLABICS SECANADIAN SYLLABI\
    \CS SAAICANADIAN SYLLABICS SICANADIAN SYLLABICS SIICANADIAN SYLLABICS SOCANADIAN SYLLABICS SOOCANADIAN SYLLABICS Y-CREE SOOCANADIAN SYLLABICS SACANADIAN SYLLABICS SAACANADIAN SYLLABICS SWECANADIAN SYLLABICS WEST-CREE SWECANADIAN SYLLABICS SWICANADIAN SYLLA\
    \BICS WEST-CREE SWICANADIAN SYLLABICS SWIICANADIAN SYLLABICS WEST-CREE SWIICANADIAN SYLLABICS SWOCANADIAN SYLLABICS WEST-CREE SWOCANADIAN SYLLABICS SWOOCANADIAN SYLLABICS WEST-CREE SWOOCANADIAN SYLLABICS SWACANADIAN SYLLABICS WEST-CREE SWACANADIAN SYLLABIC\
    \S SWAACANADIAN SYLLABICS WEST-CREE SWAACANADIAN SYLLABICS NASKAPI SWAACANADIAN SYLLABICS SCANADIAN SYLLABICS ATHAPASCAN SCANADIAN SYLLABICS SWCANADIAN SYLLABICS BLACKFOOT SCANADIAN SYLLABICS MOOSE-CREE SKCANADIAN SYLLABICS NASKAPI SKWCANADIAN SYLLABICS NA\
    \SKAPI S-WCANADIAN SYLLABICS NASKAPI SPWACANADIAN SYLLABICS NASKAPI STWACANADIAN SYLLABICS NASKAPI SKWACANADIAN SYLLABICS NASKAPI SCWACANADIAN SYLLABICS SHECANADIAN SYLLABICS SHICANADIAN SYLLABICS SHIICANADIAN SYLLABICS SHOCANADIAN SYLLABICS SHOOCANADIAN S\
    \YLLABICS SHACANADIAN SYLLABICS SHAACANADIAN SYLLABICS SHWECANADIAN SYLLABICS WEST-CREE SHWECANADIAN SYLLABICS SHWICANADIAN SYLLABICS WEST-CREE SHWICANADIAN SYLLABICS SHWIICANADIAN SYLLABICS WEST-CREE SHWIICANADIAN SYLLABICS SHWOCANADIAN SYLLABICS WEST-CRE\
    \E SHWOCANADIAN SYLLABICS SHWOOCANADIAN SYLLABICS WEST-CREE SHWOOCANADIAN SYLLABICS SHWACANADIAN SYLLABICS WEST-CREE SHWACANADIAN SYLLABICS SHWAACANADIAN SYLLABICS WEST-CREE SHWAACANADIAN SYLLABICS SHCANADIAN SYLLABICS YECANADIAN SYLLABICS YAAICANADIAN SYL\
    \LABICS YICANADIAN SYLLABICS YIICANADIAN SYLLABICS YOCANADIAN SYLLABICS YOOCANADIAN SYLLABICS Y-CREE YOOCANADIAN SYLLABICS YACANADIAN SYLLABICS YAACANADIAN SYLLABICS YWECANADIAN SYLLABICS WEST-CREE YWECANADIAN SYLLABICS YWICANADIAN SYLLABICS WEST-CREE YWIC\
    \ANADIAN SYLLABICS YWIICANADIAN SYLLABICS WEST-CREE YWIICANADIAN SYLLABICS YWOCANADIAN SYLLABICS WEST-CREE YWOCANADIAN SYLLABICS YWOOCANADIAN SYLLABICS WEST-CREE YWOOCANADIAN SYLLABICS YWACANADIAN SYLLABICS WEST-CREE YWACANADIAN SYLLABICS YWAACANADIAN SYLL\
    \ABICS WEST-CREE YWAACANADIAN SYLLABICS NASKAPI YWAACANADIAN SYLLABICS YCANADIAN SYLLABICS BIBLE-CREE YCANADIAN SYLLABICS WEST-CREE YCANADIAN SYLLABICS SAYISI YICANADIAN SYLLABICS RECANADIAN SYLLABICS R-CREE RECANADIAN SYLLABICS WEST-CREE LECANADIAN SYLLAB\
    \ICS RAAICANADIAN SYLLABICS RICANADIAN SYLLABICS RIICANADIAN SYLLABICS ROCANADIAN SYLLABICS ROOCANADIAN SYLLABICS WEST-CREE LOCANADIAN SYLLABICS RACANADIAN SYLLABICS RAACANADIAN SYLLABICS WEST-CREE LACANADIAN SYLLABICS RWAACANADIAN SYLLABICS WEST-CREE RWAA\
    \CANADIAN SYLLABICS RCANADIAN SYLLABICS WEST-CREE RCANADIAN SYLLABICS MEDIAL RCANADIAN SYLLABICS FECANADIAN SYLLABICS FAAICANADIAN SYLLABICS FICANADIAN SYLLABICS FIICANADIAN SYLLABICS FOCANADIAN SYLLABICS FOOCANADIAN SYLLABICS FACANADIAN SYLLABICS FAACANAD\
    \IAN SYLLABICS FWAACANADIAN SYLLABICS WEST-CREE FWAACANADIAN SYLLABICS FCANADIAN SYLLABICS THECANADIAN SYLLABICS N-CREE THECANADIAN SYLLABICS THICANADIAN SYLLABICS N-CREE THICANADIAN SYLLABICS THIICANADIAN SYLLABICS N-CREE THIICANADIAN SYLLABICS THOCANADIA\
    \N SYLLABICS THOOCANADIAN SYLLABICS THACANADIAN SYLLABICS THAACANADIAN SYLLABICS THWAACANADIAN SYLLABICS WEST-CREE THWAACANADIAN SYLLABICS THCANADIAN SYLLABICS TTHECANADIAN SYLLABICS TTHICANADIAN SYLLABICS TTHOCANADIAN SYLLABICS TTHACANADIAN SYLLABICS TTHC\
    \ANADIAN SYLLABICS TYECANADIAN SYLLABICS TYICANADIAN SYLLABICS TYOCANADIAN SYLLABICS TYACANADIAN SYLLABICS NUNAVIK HECANADIAN SYLLABICS NUNAVIK HICANADIAN SYLLABICS NUNAVIK HIICANADIAN SYLLABICS NUNAVIK HOCANADIAN SYLLABICS NUNAVIK HOOCANADIAN SYLLABICS NU\
    \NAVIK HACANADIAN SYLLABICS NUNAVIK HAACANADIAN SYLLABICS NUNAVIK HCANADIAN SYLLABICS NUNAVUT HCANADIAN SYLLABICS HKCANADIAN SYLLABICS QAAICANADIAN SYLLABICS QICANADIAN SYLLABICS QIICANADIAN SYLLABICS QOCANADIAN SYLLABICS QOOCANADIAN SYLLABICS QACANADIAN S\
    \YLLABICS QAACANADIAN SYLLABICS QCANADIAN SYLLABICS TLHECANADIAN SYLLABICS TLHICANADIAN SYLLABICS TLHOCANADIAN SYLLABICS TLHACANADIAN SYLLABICS WEST-CREE RECANADIAN SYLLABICS WEST-CREE RICANADIAN SYLLABICS WEST-CREE ROCANADIAN SYLLABICS WEST-CREE RACANADIA\
    \N SYLLABICS NGAAICANADIAN SYLLABICS NGICANADIAN SYLLABICS NGIICANADIAN SYLLABICS NGOCANADIAN SYLLABICS NGOOCANADIAN SYLLABICS NGACANADIAN SYLLABICS NGAACANADIAN SYLLABICS NGCANADIAN SYLLABICS NNGCANADIAN SYLLABICS SAYISI SHECANADIAN SYLLABICS SAYISI SHICA\
    \NADIAN SYLLABICS SAYISI SHOCANADIAN SYLLABICS SAYISI SHACANADIAN SYLLABICS WOODS-CREE THECANADIAN SYLLABICS WOODS-CREE THICANADIAN SYLLABICS WOODS-CREE THOCANADIAN SYLLABICS WOODS-CREE THACANADIAN SYLLABICS WOODS-CREE THCANADIAN SYLLABICS LHICANADIAN SYLL\
    \ABICS LHIICANADIAN SYLLABICS LHOCANADIAN SYLLABICS LHOOCANADIAN SYLLABICS LHACANADIAN SYLLABICS LHAACANADIAN SYLLABICS LHCANADIAN SYLLABICS TH-CREE THECANADIAN SYLLABICS TH-CREE THICANADIAN SYLLABICS TH-CREE THIICANADIAN SYLLABICS TH-CREE THOCANADIAN SYLL\
    \ABICS TH-CREE THOOCANADIAN SYLLABICS TH-CREE THACANADIAN SYLLABICS TH-CREE THAACANADIAN SYLLABICS TH-CREE THCANADIAN SYLLABICS AIVILIK BCANADIAN SYLLABICS BLACKFOOT ECANADIAN SYLLABICS BLACKFOOT ICANADIAN SYLLABICS BLACKFOOT OCANADIAN SYLLABICS BLACKFOOT \
    \ACANADIAN SYLLABICS BLACKFOOT WECANADIAN SYLLABICS BLACKFOOT WICANADIAN SYLLABICS BLACKFOOT WOCANADIAN SYLLABICS BLACKFOOT WACANADIAN SYLLABICS BLACKFOOT NECANADIAN SYLLABICS BLACKFOOT NICANADIAN SYLLABICS BLACKFOOT NOCANADIAN SYLLABICS BLACKFOOT NACANADI\
    \AN SYLLABICS BLACKFOOT KECANADIAN SYLLABICS BLACKFOOT KICANADIAN SYLLABICS BLACKFOOT KOCANADIAN SYLLABICS BLACKFOOT KACANADIAN SYLLABICS SAYISI HECANADIAN SYLLABICS SAYISI HICANADIAN SYLLABICS SAYISI HOCANADIAN SYLLABICS SAYISI HACANADIAN SYLLABICS CARRIE\
    \R GHUCANADIAN SYLLABICS CARRIER GHOCANADIAN SYLLABICS CARRIER GHECANADIAN SYLLABICS CARRIER GHEECANADIAN SYLLABICS CARRIER GHICANADIAN SYLLABICS CARRIER GHACANADIAN SYLLABICS CARRIER RUCANADIAN SYLLABICS CARRIER ROCANADIAN SYLLABICS CARRIER RECANADIAN SYL\
    \LABICS CARRIER REECANADIAN SYLLABICS CARRIER RICANADIAN SYLLABICS CARRIER RACANADIAN SYLLABICS CARRIER WUCANADIAN SYLLABICS CARRIER WOCANADIAN SYLLABICS CARRIER WECANADIAN SYLLABICS CARRIER WEECANADIAN SYLLABICS CARRIER WICANADIAN SYLLABICS CARRIER WACANA\
    \DIAN SYLLABICS CARRIER HWUCANADIAN SYLLABICS CARRIER HWOCANADIAN SYLLABICS CARRIER HWECANADIAN SYLLABICS CARRIER HWEECANADIAN SYLLABICS CARRIER HWICANADIAN SYLLABICS CARRIER HWACANADIAN SYLLABICS CARRIER THUCANADIAN SYLLABICS CARRIER THOCANADIAN SYLLABICS\
    \ CARRIER THECANADIAN SYLLABICS CARRIER THEECANADIAN SYLLABICS CARRIER THICANADIAN SYLLABICS CARRIER THACANADIAN SYLLABICS CARRIER TTUCANADIAN SYLLABICS CARRIER TTOCANADIAN SYLLABICS CARRIER TTECANADIAN SYLLABICS CARRIER TTEECANADIAN SYLLABICS CARRIER TTIC\
    \ANADIAN SYLLABICS CARRIER TTACANADIAN SYLLABICS CARRIER PUCANADIAN SYLLABICS CARRIER POCANADIAN SYLLABICS CARRIER PECANADIAN SYLLABICS CARRIER PEECANADIAN SYLLABICS CARRIER PICANADIAN SYLLABICS CARRIER PACANADIAN SYLLABICS CARRIER PCANADIAN SYLLABICS CARR\
    \IER GUCANADIAN SYLLABICS CARRIER GOCANADIAN SYLLABICS CARRIER GECANADIAN SYLLABICS CARRIER GEECANADIAN SYLLABICS CARRIER GICANADIAN SYLLABICS CARRIER GACANADIAN SYLLABICS CARRIER KHUCANADIAN SYLLABICS CARRIER KHOCANADIAN SYLLABICS CARRIER KHECANADIAN SYLL\
    \ABICS CARRIER KHEECANADIAN SYLLABICS CARRIER KHICANADIAN SYLLABICS CARRIER KHACANADIAN SYLLABICS CARRIER KKUCANADIAN SYLLABICS CARRIER KKOCANADIAN SYLLABICS CARRIER KKECANADIAN SYLLABICS CARRIER KKEECANADIAN SYLLABICS CARRIER KKICANADIAN SYLLABICS CARRIER\
    \ KKACANADIAN SYLLABICS CARRIER KKCANADIAN SYLLABICS CARRIER NUCANADIAN SYLLABICS CARRIER NOCANADIAN SYLLABICS CARRIER NECANADIAN SYLLABICS CARRIER NEECANADIAN SYLLABICS CARRIER NICANADIAN SYLLABICS CARRIER NACANADIAN SYLLABICS CARRIER MUCANADIAN SYLLABICS\
    \ CARRIER MOCANADIAN SYLLABICS CARRIER MECANADIAN SYLLABICS CARRIER MEECANADIAN SYLLABICS CARRIER MICANADIAN SYLLABICS CARRIER MACANADIAN SYLLABICS CARRIER YUCANADIAN SYLLABICS CARRIER YOCANADIAN SYLLABICS CARRIER YECANADIAN SYLLABICS CARRIER YEECANADIAN S\
    \YLLABICS CARRIER YICANADIAN SYLLABICS CARRIER YACANADIAN SYLLABICS CARRIER JUCANADIAN SYLLABICS SAYISI JUCANADIAN SYLLABICS CARRIER JOCANADIAN SYLLABICS CARRIER JECANADIAN SYLLABICS CARRIER JEECANADIAN SYLLABICS CARRIER JICANADIAN SYLLABICS SAYISI JICANAD\
    \IAN SYLLABICS CARRIER JACANADIAN SYLLABICS CARRIER JJUCANADIAN SYLLABICS CARRIER JJOCANADIAN SYLLABICS CARRIER JJECANADIAN SYLLABICS CARRIER JJEECANADIAN SYLLABICS CARRIER JJICANADIAN SYLLABICS CARRIER JJACANADIAN SYLLABICS CARRIER LUCANADIAN SYLLABICS CA\
    \RRIER LOCANADIAN SYLLABICS CARRIER LECANADIAN SYLLABICS CARRIER LEECANADIAN SYLLABICS CARRIER LICANADIAN SYLLABICS CARRIER LACANADIAN SYLLABICS CARRIER DLUCANADIAN SYLLABICS CARRIER DLOCANADIAN SYLLABICS CARRIER DLECANADIAN SYLLABICS CARRIER DLEECANADIAN \
    \SYLLABICS CARRIER DLICANADIAN SYLLABICS CARRIER DLACANADIAN SYLLABICS CARRIER LHUCANADIAN SYLLABICS CARRIER LHOCANADIAN SYLLABICS CARRIER LHECANADIAN SYLLABICS CARRIER LHEECANADIAN SYLLABICS CARRIER LHICANADIAN SYLLABICS CARRIER LHACANADIAN SYLLABICS CARR\
    \IER TLHUCANADIAN SYLLABICS CARRIER TLHOCANADIAN SYLLABICS CARRIER TLHECANADIAN SYLLABICS CARRIER TLHEECANADIAN SYLLABICS CARRIER TLHICANADIAN SYLLABICS CARRIER TLHACANADIAN SYLLABICS CARRIER TLUCANADIAN SYLLABICS CARRIER TLOCANADIAN SYLLABICS CARRIER TLEC\
    \ANADIAN SYLLABICS CARRIER TLEECANADIAN SYLLABICS CARRIER TLICANADIAN SYLLABICS CARRIER TLACANADIAN SYLLABICS CARRIER ZUCANADIAN SYLLABICS CARRIER ZOCANADIAN SYLLABICS CARRIER ZECANADIAN SYLLABICS CARRIER ZEECANADIAN SYLLABICS CARRIER ZICANADIAN SYLLABICS \
    \CARRIER ZACANADIAN SYLLABICS CARRIER ZCANADIAN SYLLABICS CARRIER INITIAL ZCANADIAN SYLLABICS CARRIER DZUCANADIAN SYLLABICS CARRIER DZOCANADIAN SYLLABICS CARRIER DZECANADIAN SYLLABICS CARRIER DZEECANADIAN SYLLABICS CARRIER DZICANADIAN SYLLABICS CARRIER DZA\
    \CANADIAN SYLLABICS CARRIER SUCANADIAN SYLLABICS CARRIER SOCANADIAN SYLLABICS CARRIER SECANADIAN SYLLABICS CARRIER SEECANADIAN SYLLABICS CARRIER SICANADIAN SYLLABICS CARRIER SACANADIAN SYLLABICS CARRIER SHUCANADIAN SYLLABICS CARRIER SHOCANADIAN SYLLABICS C\
    \ARRIER SHECANADIAN SYLLABICS CARRIER SHEECANADIAN SYLLABICS CARRIER SHICANADIAN SYLLABICS CARRIER SHACANADIAN SYLLABICS CARRIER SHCANADIAN SYLLABICS CARRIER TSUCANADIAN SYLLABICS CARRIER TSOCANADIAN SYLLABICS CARRIER TSECANADIAN SYLLABICS CARRIER TSEECANA\
    \DIAN SYLLABICS CARRIER TSICANADIAN SYLLABICS CARRIER TSACANADIAN SYLLABICS CARRIER CHUCANADIAN SYLLABICS CARRIER CHOCANADIAN SYLLABICS CARRIER CHECANADIAN SYLLABICS CARRIER CHEECANADIAN SYLLABICS CARRIER CHICANADIAN SYLLABICS CARRIER CHACANADIAN SYLLABICS\
    \ CARRIER TTSUCANADIAN SYLLABICS CARRIER TTSOCANADIAN SYLLABICS CARRIER TTSECANADIAN SYLLABICS CARRIER TTSEECANADIAN SYLLABICS CARRIER TTSICANADIAN SYLLABICS CARRIER TTSACANADIAN SYLLABICS CHI SIGNCANADIAN SYLLABICS FULL STOPCANADIAN SYLLABICS QAICANADIAN \
    \SYLLABICS NGAICANADIAN SYLLABICS NNGICANADIAN SYLLABICS NNGIICANADIAN SYLLABICS NNGOCANADIAN SYLLABICS NNGOOCANADIAN SYLLABICS NNGACANADIAN SYLLABICS NNGAACANADIAN SYLLABICS WOODS-CREE THWEECANADIAN SYLLABICS WOODS-CREE THWICANADIAN SYLLABICS WOODS-CREE T\
    \HWIICANADIAN SYLLABICS WOODS-CREE THWOCANADIAN SYLLABICS WOODS-CREE THWOOCANADIAN SYLLABICS WOODS-CREE THWACANADIAN SYLLABICS WOODS-CREE THWAACANADIAN SYLLABICS WOODS-CREE FINAL THCANADIAN SYLLABICS BLACKFOOT WOGHAM SPACE MARKOGHAM LETTER BEITHOGHAM LETTE\
    \R LUISOGHAM LETTER FEARNOGHAM LETTER SAILOGHAM LETTER NIONOGHAM LETTER UATHOGHAM LETTER DAIROGHAM LETTER TINNEOGHAM LETTER COLLOGHAM LETTER CEIRTOGHAM LETTER MUINOGHAM LETTER GORTOGHAM LETTER NGEADALOGHAM LETTER STRAIFOGHAM LETTER RUISOGHAM LETTER AILMOGH\
    \AM LETTER ONNOGHAM LETTER UROGHAM LETTER EADHADHOGHAM LETTER IODHADHOGHAM LETTER EABHADHOGHAM LETTER OROGHAM LETTER UILLEANNOGHAM LETTER IFINOGHAM LETTER EAMHANCHOLLOGHAM LETTER PEITHOGHAM FEATHER MARKOGHAM REVERSED FEATHER MARKRUNIC LETTER FEHU FEOH FE F\
    \RUNIC LETTER VRUNIC LETTER URUZ UR URUNIC LETTER YRRUNIC LETTER YRUNIC LETTER WRUNIC LETTER THURISAZ THURS THORNRUNIC LETTER ETHRUNIC LETTER ANSUZ ARUNIC LETTER OS ORUNIC LETTER AC ARUNIC LETTER AESCRUNIC LETTER LONG-BRANCH-OSS ORUNIC LETTER SHORT-TWIG-OS\
    \S ORUNIC LETTER ORUNIC LETTER OERUNIC LETTER ONRUNIC LETTER RAIDO RAD REID RRUNIC LETTER KAUNARUNIC LETTER CENRUNIC LETTER KAUN KRUNIC LETTER GRUNIC LETTER ENGRUNIC LETTER GEBO GYFU GRUNIC LETTER GARRUNIC LETTER WUNJO WYNN WRUNIC LETTER HAGLAZ HRUNIC LETT\
    \ER HAEGL HRUNIC LETTER LONG-BRANCH-HAGALL HRUNIC LETTER SHORT-TWIG-HAGALL HRUNIC LETTER NAUDIZ NYD NAUD NRUNIC LETTER SHORT-TWIG-NAUD NRUNIC LETTER DOTTED-NRUNIC LETTER ISAZ IS ISS IRUNIC LETTER ERUNIC LETTER JERAN JRUNIC LETTER GERRUNIC LETTER LONG-BRANC\
    \H-AR AERUNIC LETTER SHORT-TWIG-AR ARUNIC LETTER IWAZ EOHRUNIC LETTER PERTHO PEORTH PRUNIC LETTER ALGIZ EOLHXRUNIC LETTER SOWILO SRUNIC LETTER SIGEL LONG-BRANCH-SOL SRUNIC LETTER SHORT-TWIG-SOL SRUNIC LETTER CRUNIC LETTER ZRUNIC LETTER TIWAZ TIR TYR TRUNIC\
    \ LETTER SHORT-TWIG-TYR TRUNIC LETTER DRUNIC LETTER BERKANAN BEORC BJARKAN BRUNIC LETTER SHORT-TWIG-BJARKAN BRUNIC LETTER DOTTED-PRUNIC LETTER OPEN-PRUNIC LETTER EHWAZ EH ERUNIC LETTER MANNAZ MAN MRUNIC LETTER LONG-BRANCH-MADR MRUNIC LETTER SHORT-TWIG-MADR\
    \ MRUNIC LETTER LAUKAZ LAGU LOGR LRUNIC LETTER DOTTED-LRUNIC LETTER INGWAZRUNIC LETTER INGRUNIC LETTER DAGAZ DAEG DRUNIC LETTER OTHALAN ETHEL ORUNIC LETTER EARRUNIC LETTER IORRUNIC LETTER CWEORTHRUNIC LETTER CALCRUNIC LETTER CEALCRUNIC LETTER STANRUNIC LET\
    \TER LONG-BRANCH-YRRUNIC LETTER SHORT-TWIG-YRRUNIC LETTER ICELANDIC-YRRUNIC LETTER QRUNIC LETTER XRUNIC SINGLE PUNCTUATIONRUNIC MULTIPLE PUNCTUATIONRUNIC CROSS PUNCTUATIONRUNIC ARLAUG SYMBOLRUNIC TVIMADUR SYMBOLRUNIC BELGTHOR SYMBOLRUNIC LETTER KRUNIC LETT\
    \ER SHRUNIC LETTER OORUNIC LETTER FRANKS CASKET OSRUNIC LETTER FRANKS CASKET ISRUNIC LETTER FRANKS CASKET EHRUNIC LETTER FRANKS CASKET ACRUNIC LETTER FRANKS CASKET AESCTAGALOG LETTER ATAGALOG LETTER ITAGALOG LETTER UTAGALOG LETTER KATAGALOG LETTER GATAGALO\
    \G LETTER NGATAGALOG LETTER TATAGALOG LETTER DATAGALOG LETTER NATAGALOG LETTER PATAGALOG LETTER BATAGALOG LETTER MATAGALOG LETTER YATAGALOG LETTER RATAGALOG LETTER LATAGALOG LETTER WATAGALOG LETTER SATAGALOG LETTER HATAGALOG VOWEL SIGN ITAGALOG VOWEL SIGN \
    \UTAGALOG SIGN VIRAMATAGALOG SIGN PAMUDPODTAGALOG LETTER ARCHAIC RAHANUNOO LETTER AHANUNOO LETTER IHANUNOO LETTER UHANUNOO LETTER KAHANUNOO LETTER GAHANUNOO LETTER NGAHANUNOO LETTER TAHANUNOO LETTER DAHANUNOO LETTER NAHANUNOO LETTER PAHANUNOO LETTER BAHANU\
    \NOO LETTER MAHANUNOO LETTER YAHANUNOO LETTER RAHANUNOO LETTER LAHANUNOO LETTER WAHANUNOO LETTER SAHANUNOO LETTER HAHANUNOO VOWEL SIGN IHANUNOO VOWEL SIGN UHANUNOO SIGN PAMUDPODPHILIPPINE SINGLE PUNCTUATIONPHILIPPINE DOUBLE PUNCTUATIONBUHID LETTER ABUHID L\
    \ETTER IBUHID LETTER UBUHID LETTER KABUHID LETTER GABUHID LETTER NGABUHID LETTER TABUHID LETTER DABUHID LETTER NABUHID LETTER PABUHID LETTER BABUHID LETTER MABUHID LETTER YABUHID LETTER RABUHID LETTER LABUHID LETTER WABUHID LETTER SABUHID LETTER HABUHID VO\
    \WEL SIGN IBUHID VOWEL SIGN UTAGBANWA LETTER ATAGBANWA LETTER ITAGBANWA LETTER UTAGBANWA LETTER KATAGBANWA LETTER GATAGBANWA LETTER NGATAGBANWA LETTER TATAGBANWA LETTER DATAGBANWA LETTER NATAGBANWA LETTER PATAGBANWA LETTER BATAGBANWA LETTER MATAGBANWA LETT\
    \ER YATAGBANWA LETTER LATAGBANWA LETTER WATAGBANWA LETTER SATAGBANWA VOWEL SIGN ITAGBANWA VOWEL SIGN UKHMER LETTER KAKHMER LETTER KHAKHMER LETTER KOKHMER LETTER KHOKHMER LETTER NGOKHMER LETTER CAKHMER LETTER CHAKHMER LETTER COKHMER LETTER CHOKHMER LETTER N\
    \YOKHMER LETTER DAKHMER LETTER TTHAKHMER LETTER DOKHMER LETTER TTHOKHMER LETTER NNOKHMER LETTER TAKHMER LETTER THAKHMER LETTER TOKHMER LETTER THOKHMER LETTER NOKHMER LETTER BAKHMER LETTER PHAKHMER LETTER POKHMER LETTER PHOKHMER LETTER MOKHMER LETTER YOKHME\
    \R LETTER ROKHMER LETTER LOKHMER LETTER VOKHMER LETTER SHAKHMER LETTER SSOKHMER LETTER SAKHMER LETTER HAKHMER LETTER LAKHMER LETTER QAKHMER INDEPENDENT VOWEL QAQKHMER INDEPENDENT VOWEL QAAKHMER INDEPENDENT VOWEL QIKHMER INDEPENDENT VOWEL QIIKHMER INDEPENDE\
    \NT VOWEL QUKHMER INDEPENDENT VOWEL QUKKHMER INDEPENDENT VOWEL QUUKHMER INDEPENDENT VOWEL QUUVKHMER INDEPENDENT VOWEL RYKHMER INDEPENDENT VOWEL RYYKHMER INDEPENDENT VOWEL LYKHMER INDEPENDENT VOWEL LYYKHMER INDEPENDENT VOWEL QEKHMER INDEPENDENT VOWEL QAIKHM\
    \ER INDEPENDENT VOWEL QOO TYPE ONEKHMER INDEPENDENT VOWEL QOO TYPE TWOKHMER INDEPENDENT VOWEL QAUKHMER VOWEL INHERENT AQKHMER VOWEL INHERENT AAKHMER VOWEL SIGN AAKHMER VOWEL SIGN IKHMER VOWEL SIGN IIKHMER VOWEL SIGN YKHMER VOWEL SIGN YYKHMER VOWEL SIGN UKH\
    \MER VOWEL SIGN UUKHMER VOWEL SIGN UAKHMER VOWEL SIGN OEKHMER VOWEL SIGN YAKHMER VOWEL SIGN IEKHMER VOWEL SIGN EKHMER VOWEL SIGN AEKHMER VOWEL SIGN AIKHMER VOWEL SIGN OOKHMER VOWEL SIGN AUKHMER SIGN NIKAHITKHMER SIGN REAHMUKKHMER SIGN YUUKALEAPINTUKHMER SI\
    \GN MUUSIKATOANKHMER SIGN TRIISAPKHMER SIGN BANTOCKHMER SIGN ROBATKHMER SIGN TOANDAKHIATKHMER SIGN KAKABATKHMER SIGN AHSDAKHMER SIGN SAMYOK SANNYAKHMER SIGN VIRIAMKHMER SIGN COENGKHMER SIGN BATHAMASATKHMER SIGN KHANKHMER SIGN BARIYOOSANKHMER SIGN CAMNUC PI\
    \I KUUHKHMER SIGN LEK TOOKHMER SIGN BEYYALKHMER SIGN PHNAEK MUANKHMER SIGN KOOMUUTKHMER CURRENCY SYMBOL RIELKHMER SIGN AVAKRAHASANYAKHMER SIGN ATTHACANKHMER DIGIT ZEROKHMER DIGIT ONEKHMER DIGIT TWOKHMER DIGIT THREEKHMER DIGIT FOURKHMER DIGIT FIVEKHMER DIGI\
    \T SIXKHMER DIGIT SEVENKHMER DIGIT EIGHTKHMER DIGIT NINEKHMER SYMBOL LEK ATTAK SONKHMER SYMBOL LEK ATTAK MUOYKHMER SYMBOL LEK ATTAK PIIKHMER SYMBOL LEK ATTAK BEIKHMER SYMBOL LEK ATTAK BUONKHMER SYMBOL LEK ATTAK PRAMKHMER SYMBOL LEK ATTAK PRAM-MUOYKHMER SYM\
    \BOL LEK ATTAK PRAM-PIIKHMER SYMBOL LEK ATTAK PRAM-BEIKHMER SYMBOL LEK ATTAK PRAM-BUONMONGOLIAN BIRGAMONGOLIAN ELLIPSISMONGOLIAN COMMAMONGOLIAN FULL STOPMONGOLIAN COLONMONGOLIAN FOUR DOTSMONGOLIAN TODO SOFT HYPHENMONGOLIAN SIBE SYLLABLE BOUNDARY MARKERMONG\
    \OLIAN MANCHU COMMAMONGOLIAN MANCHU FULL STOPMONGOLIAN NIRUGUMONGOLIAN FREE VARIATION SELECTOR ONEMONGOLIAN FREE VARIATION SELECTOR TWOMONGOLIAN FREE VARIATION SELECTOR THREEMONGOLIAN VOWEL SEPARATORMONGOLIAN FREE VARIATION SELECTOR FOURMONGOLIAN DIGIT ZER\
    \OMONGOLIAN DIGIT ONEMONGOLIAN DIGIT TWOMONGOLIAN DIGIT THREEMONGOLIAN DIGIT FOURMONGOLIAN DIGIT FIVEMONGOLIAN DIGIT SIXMONGOLIAN DIGIT SEVENMONGOLIAN DIGIT EIGHTMONGOLIAN DIGIT NINEMONGOLIAN LETTER AMONGOLIAN LETTER EMONGOLIAN LETTER IMONGOLIAN LETTER OMO\
    \NGOLIAN LETTER UMONGOLIAN LETTER OEMONGOLIAN LETTER UEMONGOLIAN LETTER EEMONGOLIAN LETTER NAMONGOLIAN LETTER ANGMONGOLIAN LETTER BAMONGOLIAN LETTER PAMONGOLIAN LETTER QAMONGOLIAN LETTER GAMONGOLIAN LETTER MAMONGOLIAN LETTER LAMONGOLIAN LETTER SAMONGOLIAN \
    \LETTER SHAMONGOLIAN LETTER TAMONGOLIAN LETTER DAMONGOLIAN LETTER CHAMONGOLIAN LETTER JAMONGOLIAN LETTER YAMONGOLIAN LETTER RAMONGOLIAN LETTER WAMONGOLIAN LETTER FAMONGOLIAN LETTER KAMONGOLIAN LETTER KHAMONGOLIAN LETTER TSAMONGOLIAN LETTER ZAMONGOLIAN LETT\
    \ER HAAMONGOLIAN LETTER ZRAMONGOLIAN LETTER LHAMONGOLIAN LETTER ZHIMONGOLIAN LETTER CHIMONGOLIAN LETTER TODO LONG VOWEL SIGNMONGOLIAN LETTER TODO EMONGOLIAN LETTER TODO IMONGOLIAN LETTER TODO OMONGOLIAN LETTER TODO UMONGOLIAN LETTER TODO OEMONGOLIAN LETTER\
    \ TODO UEMONGOLIAN LETTER TODO ANGMONGOLIAN LETTER TODO BAMONGOLIAN LETTER TODO PAMONGOLIAN LETTER TODO QAMONGOLIAN LETTER TODO GAMONGOLIAN LETTER TODO MAMONGOLIAN LETTER TODO TAMONGOLIAN LETTER TODO DAMONGOLIAN LETTER TODO CHAMONGOLIAN LETTER TODO JAMONGO\
    \LIAN LETTER TODO TSAMONGOLIAN LETTER TODO YAMONGOLIAN LETTER TODO WAMONGOLIAN LETTER TODO KAMONGOLIAN LETTER TODO GAAMONGOLIAN LETTER TODO HAAMONGOLIAN LETTER TODO JIAMONGOLIAN LETTER TODO NIAMONGOLIAN LETTER TODO DZAMONGOLIAN LETTER SIBE EMONGOLIAN LETTE\
    \R SIBE IMONGOLIAN LETTER SIBE IYMONGOLIAN LETTER SIBE UEMONGOLIAN LETTER SIBE UMONGOLIAN LETTER SIBE ANGMONGOLIAN LETTER SIBE KAMONGOLIAN LETTER SIBE GAMONGOLIAN LETTER SIBE HAMONGOLIAN LETTER SIBE PAMONGOLIAN LETTER SIBE SHAMONGOLIAN LETTER SIBE TAMONGOL\
    \IAN LETTER SIBE DAMONGOLIAN LETTER SIBE JAMONGOLIAN LETTER SIBE FAMONGOLIAN LETTER SIBE GAAMONGOLIAN LETTER SIBE HAAMONGOLIAN LETTER SIBE TSAMONGOLIAN LETTER SIBE ZAMONGOLIAN LETTER SIBE RAAMONGOLIAN LETTER SIBE CHAMONGOLIAN LETTER SIBE ZHAMONGOLIAN LETTE\
    \R MANCHU IMONGOLIAN LETTER MANCHU KAMONGOLIAN LETTER MANCHU RAMONGOLIAN LETTER MANCHU FAMONGOLIAN LETTER MANCHU ZHAMONGOLIAN LETTER CHA WITH TWO DOTSMONGOLIAN LETTER ALI GALI ANUSVARA ONEMONGOLIAN LETTER ALI GALI VISARGA ONEMONGOLIAN LETTER ALI GALI DAMAR\
    \UMONGOLIAN LETTER ALI GALI UBADAMAMONGOLIAN LETTER ALI GALI INVERTED UBADAMAMONGOLIAN LETTER ALI GALI BALUDAMONGOLIAN LETTER ALI GALI THREE BALUDAMONGOLIAN LETTER ALI GALI AMONGOLIAN LETTER ALI GALI IMONGOLIAN LETTER ALI GALI KAMONGOLIAN LETTER ALI GALI N\
    \GAMONGOLIAN LETTER ALI GALI CAMONGOLIAN LETTER ALI GALI TTAMONGOLIAN LETTER ALI GALI TTHAMONGOLIAN LETTER ALI GALI DDAMONGOLIAN LETTER ALI GALI NNAMONGOLIAN LETTER ALI GALI TAMONGOLIAN LETTER ALI GALI DAMONGOLIAN LETTER ALI GALI PAMONGOLIAN LETTER ALI GAL\
    \I PHAMONGOLIAN LETTER ALI GALI SSAMONGOLIAN LETTER ALI GALI ZHAMONGOLIAN LETTER ALI GALI ZAMONGOLIAN LETTER ALI GALI AHMONGOLIAN LETTER TODO ALI GALI TAMONGOLIAN LETTER TODO ALI GALI ZHAMONGOLIAN LETTER MANCHU ALI GALI GHAMONGOLIAN LETTER MANCHU ALI GALI \
    \NGAMONGOLIAN LETTER MANCHU ALI GALI CAMONGOLIAN LETTER MANCHU ALI GALI JHAMONGOLIAN LETTER MANCHU ALI GALI TTAMONGOLIAN LETTER MANCHU ALI GALI DDHAMONGOLIAN LETTER MANCHU ALI GALI TAMONGOLIAN LETTER MANCHU ALI GALI DHAMONGOLIAN LETTER MANCHU ALI GALI SSAM\
    \ONGOLIAN LETTER MANCHU ALI GALI CYAMONGOLIAN LETTER MANCHU ALI GALI ZHAMONGOLIAN LETTER MANCHU ALI GALI ZAMONGOLIAN LETTER ALI GALI HALF UMONGOLIAN LETTER ALI GALI HALF YAMONGOLIAN LETTER MANCHU ALI GALI BHAMONGOLIAN LETTER ALI GALI DAGALGAMONGOLIAN LETTE\
    \R MANCHU ALI GALI LHACANADIAN SYLLABICS OYCANADIAN SYLLABICS AYCANADIAN SYLLABICS AAYCANADIAN SYLLABICS WAYCANADIAN SYLLABICS POYCANADIAN SYLLABICS PAYCANADIAN SYLLABICS PWOYCANADIAN SYLLABICS TAYCANADIAN SYLLABICS KAYCANADIAN SYLLABICS KWAYCANADIAN SYLLA\
    \BICS MAYCANADIAN SYLLABICS NOYCANADIAN SYLLABICS NAYCANADIAN SYLLABICS LAYCANADIAN SYLLABICS SOYCANADIAN SYLLABICS SAYCANADIAN SYLLABICS SHOYCANADIAN SYLLABICS SHAYCANADIAN SYLLABICS SHWOYCANADIAN SYLLABICS YOYCANADIAN SYLLABICS YAYCANADIAN SYLLABICS RAYC\
    \ANADIAN SYLLABICS NWICANADIAN SYLLABICS OJIBWAY NWICANADIAN SYLLABICS NWIICANADIAN SYLLABICS OJIBWAY NWIICANADIAN SYLLABICS NWOCANADIAN SYLLABICS OJIBWAY NWOCANADIAN SYLLABICS NWOOCANADIAN SYLLABICS OJIBWAY NWOOCANADIAN SYLLABICS RWEECANADIAN SYLLABICS RW\
    \ICANADIAN SYLLABICS RWIICANADIAN SYLLABICS RWOCANADIAN SYLLABICS RWOOCANADIAN SYLLABICS RWACANADIAN SYLLABICS OJIBWAY PCANADIAN SYLLABICS OJIBWAY TCANADIAN SYLLABICS OJIBWAY KCANADIAN SYLLABICS OJIBWAY CCANADIAN SYLLABICS OJIBWAY MCANADIAN SYLLABICS OJIBW\
    \AY NCANADIAN SYLLABICS OJIBWAY SCANADIAN SYLLABICS OJIBWAY SHCANADIAN SYLLABICS EASTERN WCANADIAN SYLLABICS WESTERN WCANADIAN SYLLABICS FINAL SMALL RINGCANADIAN SYLLABICS FINAL RAISED DOTCANADIAN SYLLABICS R-CREE RWECANADIAN SYLLABICS WEST-CREE LOOCANADIA\
    \N SYLLABICS WEST-CREE LAACANADIAN SYLLABICS THWECANADIAN SYLLABICS THWACANADIAN SYLLABICS TTHWECANADIAN SYLLABICS TTHOOCANADIAN SYLLABICS TTHAACANADIAN SYLLABICS TLHWECANADIAN SYLLABICS TLHOOCANADIAN SYLLABICS SAYISI SHWECANADIAN SYLLABICS SAYISI SHOOCANA\
    \DIAN SYLLABICS SAYISI HOOCANADIAN SYLLABICS CARRIER GWUCANADIAN SYLLABICS CARRIER DENE GEECANADIAN SYLLABICS CARRIER GAACANADIAN SYLLABICS CARRIER GWACANADIAN SYLLABICS SAYISI JUUCANADIAN SYLLABICS CARRIER JWACANADIAN SYLLABICS BEAVER DENE LCANADIAN SYLLA\
    \BICS BEAVER DENE RCANADIAN SYLLABICS CARRIER DENTAL SLIMBU VOWEL-CARRIER LETTERLIMBU LETTER KALIMBU LETTER KHALIMBU LETTER GALIMBU LETTER GHALIMBU LETTER NGALIMBU LETTER CALIMBU LETTER CHALIMBU LETTER JALIMBU LETTER JHALIMBU LETTER YANLIMBU LETTER TALIMBU\
    \ LETTER THALIMBU LETTER DALIMBU LETTER DHALIMBU LETTER NALIMBU LETTER PALIMBU LETTER PHALIMBU LETTER BALIMBU LETTER BHALIMBU LETTER MALIMBU LETTER YALIMBU LETTER RALIMBU LETTER LALIMBU LETTER WALIMBU LETTER SHALIMBU LETTER SSALIMBU LETTER SALIMBU LETTER H\
    \ALIMBU LETTER GYANLIMBU LETTER TRALIMBU VOWEL SIGN ALIMBU VOWEL SIGN ILIMBU VOWEL SIGN ULIMBU VOWEL SIGN EELIMBU VOWEL SIGN AILIMBU VOWEL SIGN OOLIMBU VOWEL SIGN AULIMBU VOWEL SIGN ELIMBU VOWEL SIGN OLIMBU SUBJOINED LETTER YALIMBU SUBJOINED LETTER RALIMBU\
    \ SUBJOINED LETTER WALIMBU SMALL LETTER KALIMBU SMALL LETTER NGALIMBU SMALL LETTER ANUSVARALIMBU SMALL LETTER TALIMBU SMALL LETTER NALIMBU SMALL LETTER PALIMBU SMALL LETTER MALIMBU SMALL LETTER RALIMBU SMALL LETTER LALIMBU SIGN MUKPHRENGLIMBU SIGN KEMPHREN\
    \GLIMBU SIGN SA-ILIMBU SIGN LOOLIMBU EXCLAMATION MARKLIMBU QUESTION MARKLIMBU DIGIT ZEROLIMBU DIGIT ONELIMBU DIGIT TWOLIMBU DIGIT THREELIMBU DIGIT FOURLIMBU DIGIT FIVELIMBU DIGIT SIXLIMBU DIGIT SEVENLIMBU DIGIT EIGHTLIMBU DIGIT NINETAI LE LETTER KATAI LE L\
    \ETTER XATAI LE LETTER NGATAI LE LETTER TSATAI LE LETTER SATAI LE LETTER YATAI LE LETTER TATAI LE LETTER THATAI LE LETTER LATAI LE LETTER PATAI LE LETTER PHATAI LE LETTER MATAI LE LETTER FATAI LE LETTER VATAI LE LETTER HATAI LE LETTER QATAI LE LETTER KHATA\
    \I LE LETTER TSHATAI LE LETTER NATAI LE LETTER ATAI LE LETTER ITAI LE LETTER EETAI LE LETTER EHTAI LE LETTER UTAI LE LETTER OOTAI LE LETTER OTAI LE LETTER UETAI LE LETTER ETAI LE LETTER AUETAI LE LETTER AITAI LE LETTER TONE-2TAI LE LETTER TONE-3TAI LE LETT\
    \ER TONE-4TAI LE LETTER TONE-5TAI LE LETTER TONE-6NEW TAI LUE LETTER HIGH QANEW TAI LUE LETTER LOW QANEW TAI LUE LETTER HIGH KANEW TAI LUE LETTER HIGH XANEW TAI LUE LETTER HIGH NGANEW TAI LUE LETTER LOW KANEW TAI LUE LETTER LOW XANEW TAI LUE LETTER LOW NGA\
    \NEW TAI LUE LETTER HIGH TSANEW TAI LUE LETTER HIGH SANEW TAI LUE LETTER HIGH YANEW TAI LUE LETTER LOW TSANEW TAI LUE LETTER LOW SANEW TAI LUE LETTER LOW YANEW TAI LUE LETTER HIGH TANEW TAI LUE LETTER HIGH THANEW TAI LUE LETTER HIGH NANEW TAI LUE LETTER LO\
    \W TANEW TAI LUE LETTER LOW THANEW TAI LUE LETTER LOW NANEW TAI LUE LETTER HIGH PANEW TAI LUE LETTER HIGH PHANEW TAI LUE LETTER HIGH MANEW TAI LUE LETTER LOW PANEW TAI LUE LETTER LOW PHANEW TAI LUE LETTER LOW MANEW TAI LUE LETTER HIGH FANEW TAI LUE LETTER \
    \HIGH VANEW TAI LUE LETTER HIGH LANEW TAI LUE LETTER LOW FANEW TAI LUE LETTER LOW VANEW TAI LUE LETTER LOW LANEW TAI LUE LETTER HIGH HANEW TAI LUE LETTER HIGH DANEW TAI LUE LETTER HIGH BANEW TAI LUE LETTER LOW HANEW TAI LUE LETTER LOW DANEW TAI LUE LETTER \
    \LOW BANEW TAI LUE LETTER HIGH KVANEW TAI LUE LETTER HIGH XVANEW TAI LUE LETTER LOW KVANEW TAI LUE LETTER LOW XVANEW TAI LUE LETTER HIGH SUANEW TAI LUE LETTER LOW SUANEW TAI LUE VOWEL SIGN VOWEL SHORTENERNEW TAI LUE VOWEL SIGN AANEW TAI LUE VOWEL SIGN IINE\
    \W TAI LUE VOWEL SIGN UNEW TAI LUE VOWEL SIGN UUNEW TAI LUE VOWEL SIGN ENEW TAI LUE VOWEL SIGN AENEW TAI LUE VOWEL SIGN ONEW TAI LUE VOWEL SIGN OANEW TAI LUE VOWEL SIGN UENEW TAI LUE VOWEL SIGN AYNEW TAI LUE VOWEL SIGN AAYNEW TAI LUE VOWEL SIGN UYNEW TAI L\
    \UE VOWEL SIGN OYNEW TAI LUE VOWEL SIGN OAYNEW TAI LUE VOWEL SIGN UEYNEW TAI LUE VOWEL SIGN IYNEW TAI LUE LETTER FINAL VNEW TAI LUE LETTER FINAL NGNEW TAI LUE LETTER FINAL NNEW TAI LUE LETTER FINAL MNEW TAI LUE LETTER FINAL KNEW TAI LUE LETTER FINAL DNEW T\
    \AI LUE LETTER FINAL BNEW TAI LUE TONE MARK-1NEW TAI LUE TONE MARK-2NEW TAI LUE DIGIT ZERONEW TAI LUE DIGIT ONENEW TAI LUE DIGIT TWONEW TAI LUE DIGIT THREENEW TAI LUE DIGIT FOURNEW TAI LUE DIGIT FIVENEW TAI LUE DIGIT SIXNEW TAI LUE DIGIT SEVENNEW TAI LUE D\
    \IGIT EIGHTNEW TAI LUE DIGIT NINENEW TAI LUE THAM DIGIT ONENEW TAI LUE SIGN LAENEW TAI LUE SIGN LAEVKHMER SYMBOL PATHAMASATKHMER SYMBOL MUOY KOETKHMER SYMBOL PII KOETKHMER SYMBOL BEI KOETKHMER SYMBOL BUON KOETKHMER SYMBOL PRAM KOETKHMER SYMBOL PRAM-MUOY KO\
    \ETKHMER SYMBOL PRAM-PII KOETKHMER SYMBOL PRAM-BEI KOETKHMER SYMBOL PRAM-BUON KOETKHMER SYMBOL DAP KOETKHMER SYMBOL DAP-MUOY KOETKHMER SYMBOL DAP-PII KOETKHMER SYMBOL DAP-BEI KOETKHMER SYMBOL DAP-BUON KOETKHMER SYMBOL DAP-PRAM KOETKHMER SYMBOL TUTEYASATKHM\
    \ER SYMBOL MUOY ROCKHMER SYMBOL PII ROCKHMER SYMBOL BEI ROCKHMER SYMBOL BUON ROCKHMER SYMBOL PRAM ROCKHMER SYMBOL PRAM-MUOY ROCKHMER SYMBOL PRAM-PII ROCKHMER SYMBOL PRAM-BEI ROCKHMER SYMBOL PRAM-BUON ROCKHMER SYMBOL DAP ROCKHMER SYMBOL DAP-MUOY ROCKHMER SY\
    \MBOL DAP-PII ROCKHMER SYMBOL DAP-BEI ROCKHMER SYMBOL DAP-BUON ROCKHMER SYMBOL DAP-PRAM ROCBUGINESE LETTER KABUGINESE LETTER GABUGINESE LETTER NGABUGINESE LETTER NGKABUGINESE LETTER PABUGINESE LETTER BABUGINESE LETTER MABUGINESE LETTER MPABUGINESE LETTER T\
    \ABUGINESE LETTER DABUGINESE LETTER NABUGINESE LETTER NRABUGINESE LETTER CABUGINESE LETTER JABUGINESE LETTER NYABUGINESE LETTER NYCABUGINESE LETTER YABUGINESE LETTER RABUGINESE LETTER LABUGINESE LETTER VABUGINESE LETTER SABUGINESE LETTER ABUGINESE LETTER H\
    \ABUGINESE VOWEL SIGN IBUGINESE VOWEL SIGN UBUGINESE VOWEL SIGN EBUGINESE VOWEL SIGN OBUGINESE VOWEL SIGN AEBUGINESE PALLAWABUGINESE END OF SECTIONTAI THAM LETTER HIGH KATAI THAM LETTER HIGH KHATAI THAM LETTER HIGH KXATAI THAM LETTER LOW KATAI THAM LETTER \
    \LOW KXATAI THAM LETTER LOW KHATAI THAM LETTER NGATAI THAM LETTER HIGH CATAI THAM LETTER HIGH CHATAI THAM LETTER LOW CATAI THAM LETTER LOW SATAI THAM LETTER LOW CHATAI THAM LETTER NYATAI THAM LETTER RATATAI THAM LETTER HIGH RATHATAI THAM LETTER DATAI THAM \
    \LETTER LOW RATHATAI THAM LETTER RANATAI THAM LETTER HIGH TATAI THAM LETTER HIGH THATAI THAM LETTER LOW TATAI THAM LETTER LOW THATAI THAM LETTER NATAI THAM LETTER BATAI THAM LETTER HIGH PATAI THAM LETTER HIGH PHATAI THAM LETTER HIGH FATAI THAM LETTER LOW P\
    \ATAI THAM LETTER LOW FATAI THAM LETTER LOW PHATAI THAM LETTER MATAI THAM LETTER LOW YATAI THAM LETTER HIGH YATAI THAM LETTER RATAI THAM LETTER RUETAI THAM LETTER LATAI THAM LETTER LUETAI THAM LETTER WATAI THAM LETTER HIGH SHATAI THAM LETTER HIGH SSATAI TH\
    \AM LETTER HIGH SATAI THAM LETTER HIGH HATAI THAM LETTER LLATAI THAM LETTER ATAI THAM LETTER LOW HATAI THAM LETTER ITAI THAM LETTER IITAI THAM LETTER UTAI THAM LETTER UUTAI THAM LETTER EETAI THAM LETTER OOTAI THAM LETTER LAETAI THAM LETTER GREAT SATAI THAM\
    \ CONSONANT SIGN MEDIAL RATAI THAM CONSONANT SIGN MEDIAL LATAI THAM CONSONANT SIGN LA TANG LAITAI THAM SIGN MAI KANG LAITAI THAM CONSONANT SIGN FINAL NGATAI THAM CONSONANT SIGN LOW PATAI THAM CONSONANT SIGN HIGH RATHA OR LOW PATAI THAM CONSONANT SIGN MATAI\
    \ THAM CONSONANT SIGN BATAI THAM CONSONANT SIGN SATAI THAM SIGN SAKOTTAI THAM VOWEL SIGN ATAI THAM VOWEL SIGN MAI SATTAI THAM VOWEL SIGN AATAI THAM VOWEL SIGN TALL AATAI THAM VOWEL SIGN ITAI THAM VOWEL SIGN IITAI THAM VOWEL SIGN UETAI THAM VOWEL SIGN UUETA\
    \I THAM VOWEL SIGN UTAI THAM VOWEL SIGN UUTAI THAM VOWEL SIGN OTAI THAM VOWEL SIGN OA BELOWTAI THAM VOWEL SIGN OYTAI THAM VOWEL SIGN ETAI THAM VOWEL SIGN AETAI THAM VOWEL SIGN OOTAI THAM VOWEL SIGN AITAI THAM VOWEL SIGN THAM AITAI THAM VOWEL SIGN OA ABOVET\
    \AI THAM SIGN MAI KANGTAI THAM SIGN TONE-1TAI THAM SIGN TONE-2TAI THAM SIGN KHUEN TONE-3TAI THAM SIGN KHUEN TONE-4TAI THAM SIGN KHUEN TONE-5TAI THAM SIGN RA HAAMTAI THAM SIGN MAI SAMTAI THAM SIGN KHUEN-LUE KARANTAI THAM COMBINING CRYPTOGRAMMIC DOTTAI THAM \
    \HORA DIGIT ZEROTAI THAM HORA DIGIT ONETAI THAM HORA DIGIT TWOTAI THAM HORA DIGIT THREETAI THAM HORA DIGIT FOURTAI THAM HORA DIGIT FIVETAI THAM HORA DIGIT SIXTAI THAM HORA DIGIT SEVENTAI THAM HORA DIGIT EIGHTTAI THAM HORA DIGIT NINETAI THAM THAM DIGIT ZERO\
    \TAI THAM THAM DIGIT ONETAI THAM THAM DIGIT TWOTAI THAM THAM DIGIT THREETAI THAM THAM DIGIT FOURTAI THAM THAM DIGIT FIVETAI THAM THAM DIGIT SIXTAI THAM THAM DIGIT SEVENTAI THAM THAM DIGIT EIGHTTAI THAM THAM DIGIT NINETAI THAM SIGN WIANGTAI THAM SIGN WIANGW\
    \AAKTAI THAM SIGN SAWANTAI THAM SIGN KEOWTAI THAM SIGN HOYTAI THAM SIGN DOKMAITAI THAM SIGN REVERSED ROTATED RANATAI THAM SIGN MAI YAMOKTAI THAM SIGN KAANTAI THAM SIGN KAANKUUTAI THAM SIGN SATKAANTAI THAM SIGN SATKAANKUUTAI THAM SIGN HANGTAI THAM SIGN CAAN\
    \GCOMBINING DOUBLED CIRCUMFLEX ACCENTCOMBINING DIAERESIS-RINGCOMBINING INFINITYCOMBINING DOWNWARDS ARROWCOMBINING TRIPLE DOTCOMBINING X-X BELOWCOMBINING WIGGLY LINE BELOWCOMBINING OPEN MARK BELOWCOMBINING DOUBLE OPEN MARK BELOWCOMBINING LIGHT CENTRALIZATIO\
    \N STROKE BELOWCOMBINING STRONG CENTRALIZATION STROKE BELOWCOMBINING PARENTHESES ABOVECOMBINING DOUBLE PARENTHESES ABOVECOMBINING PARENTHESES BELOWCOMBINING PARENTHESES OVERLAYCOMBINING LATIN SMALL LETTER W BELOWCOMBINING LATIN SMALL LETTER TURNED W BELOWC\
    \OMBINING LEFT PARENTHESIS ABOVE LEFTCOMBINING RIGHT PARENTHESIS ABOVE RIGHTCOMBINING LEFT PARENTHESIS BELOW LEFTCOMBINING RIGHT PARENTHESIS BELOW RIGHTCOMBINING SQUARE BRACKETS ABOVECOMBINING NUMBER SIGN ABOVECOMBINING INVERTED DOUBLE ARCH ABOVECOMBINING \
    \PLUS SIGN ABOVECOMBINING DOUBLE PLUS SIGN ABOVECOMBINING DOUBLE PLUS SIGN BELOWCOMBINING TRIPLE ACUTE ACCENTCOMBINING LATIN SMALL LETTER INSULAR GCOMBINING LATIN SMALL LETTER INSULAR RCOMBINING LATIN SMALL LETTER INSULAR TBALINESE SIGN ULU RICEMBALINESE S\
    \IGN ULU CANDRABALINESE SIGN CECEKBALINESE SIGN SURANGBALINESE SIGN BISAHBALINESE LETTER AKARABALINESE LETTER AKARA TEDUNGBALINESE LETTER IKARABALINESE LETTER IKARA TEDUNGBALINESE LETTER UKARABALINESE LETTER UKARA TEDUNGBALINESE LETTER RA REPABALINESE LETT\
    \ER RA REPA TEDUNGBALINESE LETTER LA LENGABALINESE LETTER LA LENGA TEDUNGBALINESE LETTER EKARABALINESE LETTER AIKARABALINESE LETTER OKARABALINESE LETTER OKARA TEDUNGBALINESE LETTER KABALINESE LETTER KA MAHAPRANABALINESE LETTER GABALINESE LETTER GA GORABALI\
    \NESE LETTER NGABALINESE LETTER CABALINESE LETTER CA LACABALINESE LETTER JABALINESE LETTER JA JERABALINESE LETTER NYABALINESE LETTER TA LATIKBALINESE LETTER TA MURDA MAHAPRANABALINESE LETTER DA MURDA ALPAPRANABALINESE LETTER DA MURDA MAHAPRANABALINESE LETT\
    \ER NA RAMBATBALINESE LETTER TABALINESE LETTER TA TAWABALINESE LETTER DABALINESE LETTER DA MADUBALINESE LETTER NABALINESE LETTER PABALINESE LETTER PA KAPALBALINESE LETTER BABALINESE LETTER BA KEMBANGBALINESE LETTER MABALINESE LETTER YABALINESE LETTER RABAL\
    \INESE LETTER LABALINESE LETTER WABALINESE LETTER SA SAGABALINESE LETTER SA SAPABALINESE LETTER SABALINESE LETTER HABALINESE SIGN REREKANBALINESE VOWEL SIGN TEDUNGBALINESE VOWEL SIGN ULUBALINESE VOWEL SIGN ULU SARIBALINESE VOWEL SIGN SUKUBALINESE VOWEL SIG\
    \N SUKU ILUTBALINESE VOWEL SIGN RA REPABALINESE VOWEL SIGN RA REPA TEDUNGBALINESE VOWEL SIGN LA LENGABALINESE VOWEL SIGN LA LENGA TEDUNGBALINESE VOWEL SIGN TALINGBALINESE VOWEL SIGN TALING REPABALINESE VOWEL SIGN TALING TEDUNGBALINESE VOWEL SIGN TALING REP\
    \A TEDUNGBALINESE VOWEL SIGN PEPETBALINESE VOWEL SIGN PEPET TEDUNGBALINESE ADEG ADEGBALINESE LETTER KAF SASAKBALINESE LETTER KHOT SASAKBALINESE LETTER TZIR SASAKBALINESE LETTER EF SASAKBALINESE LETTER VE SASAKBALINESE LETTER ZAL SASAKBALINESE LETTER ASYURA\
    \ SASAKBALINESE LETTER ARCHAIC JNYABALINESE DIGIT ZEROBALINESE DIGIT ONEBALINESE DIGIT TWOBALINESE DIGIT THREEBALINESE DIGIT FOURBALINESE DIGIT FIVEBALINESE DIGIT SIXBALINESE DIGIT SEVENBALINESE DIGIT EIGHTBALINESE DIGIT NINEBALINESE PANTIBALINESE PAMADABA\
    \LINESE WINDUBALINESE CARIK PAMUNGKAHBALINESE CARIK SIKIBALINESE CARIK PARERENBALINESE PAMENENGBALINESE MUSICAL SYMBOL DONGBALINESE MUSICAL SYMBOL DENGBALINESE MUSICAL SYMBOL DUNGBALINESE MUSICAL SYMBOL DANGBALINESE MUSICAL SYMBOL DANG SURANGBALINESE MUSIC\
    \AL SYMBOL DINGBALINESE MUSICAL SYMBOL DAENGBALINESE MUSICAL SYMBOL DEUNGBALINESE MUSICAL SYMBOL DAINGBALINESE MUSICAL SYMBOL DANG GEDEBALINESE MUSICAL SYMBOL COMBINING TEGEHBALINESE MUSICAL SYMBOL COMBINING ENDEPBALINESE MUSICAL SYMBOL COMBINING KEMPULBAL\
    \INESE MUSICAL SYMBOL COMBINING KEMPLIBALINESE MUSICAL SYMBOL COMBINING JEGOGANBALINESE MUSICAL SYMBOL COMBINING KEMPUL WITH JEGOGANBALINESE MUSICAL SYMBOL COMBINING KEMPLI WITH JEGOGANBALINESE MUSICAL SYMBOL COMBINING BENDEBALINESE MUSICAL SYMBOL COMBININ\
    \G GONGBALINESE MUSICAL SYMBOL RIGHT-HAND OPEN DUGBALINESE MUSICAL SYMBOL RIGHT-HAND OPEN DAGBALINESE MUSICAL SYMBOL RIGHT-HAND CLOSED TUKBALINESE MUSICAL SYMBOL RIGHT-HAND CLOSED TAKBALINESE MUSICAL SYMBOL LEFT-HAND OPEN PANGBALINESE MUSICAL SYMBOL LEFT-H\
    \AND OPEN PUNGBALINESE MUSICAL SYMBOL LEFT-HAND CLOSED PLAKBALINESE MUSICAL SYMBOL LEFT-HAND CLOSED PLUKBALINESE MUSICAL SYMBOL LEFT-HAND OPEN PINGBALINESE PANTI LANTANGBALINESE PAMADA LANTANGSUNDANESE SIGN PANYECEKSUNDANESE SIGN PANGLAYARSUNDANESE SIGN PA\
    \NGWISADSUNDANESE LETTER ASUNDANESE LETTER ISUNDANESE LETTER USUNDANESE LETTER AESUNDANESE LETTER OSUNDANESE LETTER ESUNDANESE LETTER EUSUNDANESE LETTER KASUNDANESE LETTER QASUNDANESE LETTER GASUNDANESE LETTER NGASUNDANESE LETTER CASUNDANESE LETTER JASUNDA\
    \NESE LETTER ZASUNDANESE LETTER NYASUNDANESE LETTER TASUNDANESE LETTER DASUNDANESE LETTER NASUNDANESE LETTER PASUNDANESE LETTER FASUNDANESE LETTER VASUNDANESE LETTER BASUNDANESE LETTER MASUNDANESE LETTER YASUNDANESE LETTER RASUNDANESE LETTER LASUNDANESE LE\
    \TTER WASUNDANESE LETTER SASUNDANESE LETTER XASUNDANESE LETTER HASUNDANESE CONSONANT SIGN PAMINGKALSUNDANESE CONSONANT SIGN PANYAKRASUNDANESE CONSONANT SIGN PANYIKUSUNDANESE VOWEL SIGN PANGHULUSUNDANESE VOWEL SIGN PANYUKUSUNDANESE VOWEL SIGN PANAELAENGSUND\
    \ANESE VOWEL SIGN PANOLONGSUNDANESE VOWEL SIGN PAMEPETSUNDANESE VOWEL SIGN PANEULEUNGSUNDANESE SIGN PAMAAEHSUNDANESE SIGN VIRAMASUNDANESE CONSONANT SIGN PASANGAN MASUNDANESE CONSONANT SIGN PASANGAN WASUNDANESE LETTER KHASUNDANESE LETTER SYASUNDANESE DIGIT \
    \ZEROSUNDANESE DIGIT ONESUNDANESE DIGIT TWOSUNDANESE DIGIT THREESUNDANESE DIGIT FOURSUNDANESE DIGIT FIVESUNDANESE DIGIT SIXSUNDANESE DIGIT SEVENSUNDANESE DIGIT EIGHTSUNDANESE DIGIT NINESUNDANESE AVAGRAHASUNDANESE LETTER REUSUNDANESE LETTER LEUSUNDANESE LET\
    \TER BHASUNDANESE LETTER FINAL KSUNDANESE LETTER FINAL MBATAK LETTER ABATAK LETTER SIMALUNGUN ABATAK LETTER HABATAK LETTER SIMALUNGUN HABATAK LETTER MANDAILING HABATAK LETTER BABATAK LETTER KARO BABATAK LETTER PABATAK LETTER SIMALUNGUN PABATAK LETTER NABAT\
    \AK LETTER MANDAILING NABATAK LETTER WABATAK LETTER SIMALUNGUN WABATAK LETTER PAKPAK WABATAK LETTER GABATAK LETTER SIMALUNGUN GABATAK LETTER JABATAK LETTER DABATAK LETTER RABATAK LETTER SIMALUNGUN RABATAK LETTER MABATAK LETTER SIMALUNGUN MABATAK LETTER SOU\
    \THERN TABATAK LETTER NORTHERN TABATAK LETTER SABATAK LETTER SIMALUNGUN SABATAK LETTER MANDAILING SABATAK LETTER YABATAK LETTER SIMALUNGUN YABATAK LETTER NGABATAK LETTER LABATAK LETTER SIMALUNGUN LABATAK LETTER NYABATAK LETTER CABATAK LETTER NDABATAK LETTE\
    \R MBABATAK LETTER IBATAK LETTER UBATAK SIGN TOMPIBATAK VOWEL SIGN EBATAK VOWEL SIGN PAKPAK EBATAK VOWEL SIGN EEBATAK VOWEL SIGN IBATAK VOWEL SIGN KARO IBATAK VOWEL SIGN OBATAK VOWEL SIGN KARO OBATAK VOWEL SIGN UBATAK VOWEL SIGN U FOR SIMALUNGUN SABATAK CO\
    \NSONANT SIGN NGBATAK CONSONANT SIGN HBATAK PANGOLATBATAK PANONGONANBATAK SYMBOL BINDU NA METEKBATAK SYMBOL BINDU PINARBORASBATAK SYMBOL BINDU JUDULBATAK SYMBOL BINDU PANGOLATLEPCHA LETTER KALEPCHA LETTER KLALEPCHA LETTER KHALEPCHA LETTER GALEPCHA LETTER G\
    \LALEPCHA LETTER NGALEPCHA LETTER CALEPCHA LETTER CHALEPCHA LETTER JALEPCHA LETTER NYALEPCHA LETTER TALEPCHA LETTER THALEPCHA LETTER DALEPCHA LETTER NALEPCHA LETTER PALEPCHA LETTER PLALEPCHA LETTER PHALEPCHA LETTER FALEPCHA LETTER FLALEPCHA LETTER BALEPCHA\
    \ LETTER BLALEPCHA LETTER MALEPCHA LETTER MLALEPCHA LETTER TSALEPCHA LETTER TSHALEPCHA LETTER DZALEPCHA LETTER YALEPCHA LETTER RALEPCHA LETTER LALEPCHA LETTER HALEPCHA LETTER HLALEPCHA LETTER VALEPCHA LETTER SALEPCHA LETTER SHALEPCHA LETTER WALEPCHA LETTER\
    \ ALEPCHA SUBJOINED LETTER YALEPCHA SUBJOINED LETTER RALEPCHA VOWEL SIGN AALEPCHA VOWEL SIGN ILEPCHA VOWEL SIGN OLEPCHA VOWEL SIGN OOLEPCHA VOWEL SIGN ULEPCHA VOWEL SIGN UULEPCHA VOWEL SIGN ELEPCHA CONSONANT SIGN KLEPCHA CONSONANT SIGN MLEPCHA CONSONANT SI\
    \GN LLEPCHA CONSONANT SIGN NLEPCHA CONSONANT SIGN PLEPCHA CONSONANT SIGN RLEPCHA CONSONANT SIGN TLEPCHA CONSONANT SIGN NYIN-DOLEPCHA CONSONANT SIGN KANGLEPCHA SIGN RANLEPCHA SIGN NUKTALEPCHA PUNCTUATION TA-ROLLEPCHA PUNCTUATION NYET THYOOM TA-ROLLEPCHA PUN\
    \CTUATION CER-WALEPCHA PUNCTUATION TSHOOK CER-WALEPCHA PUNCTUATION TSHOOKLEPCHA DIGIT ZEROLEPCHA DIGIT ONELEPCHA DIGIT TWOLEPCHA DIGIT THREELEPCHA DIGIT FOURLEPCHA DIGIT FIVELEPCHA DIGIT SIXLEPCHA DIGIT SEVENLEPCHA DIGIT EIGHTLEPCHA DIGIT NINELEPCHA LETTER\
    \ TTALEPCHA LETTER TTHALEPCHA LETTER DDAOL CHIKI DIGIT ZEROOL CHIKI DIGIT ONEOL CHIKI DIGIT TWOOL CHIKI DIGIT THREEOL CHIKI DIGIT FOUROL CHIKI DIGIT FIVEOL CHIKI DIGIT SIXOL CHIKI DIGIT SEVENOL CHIKI DIGIT EIGHTOL CHIKI DIGIT NINEOL CHIKI LETTER LAOL CHIKI\
    \ LETTER ATOL CHIKI LETTER AGOL CHIKI LETTER ANGOL CHIKI LETTER ALOL CHIKI LETTER LAAOL CHIKI LETTER AAKOL CHIKI LETTER AAJOL CHIKI LETTER AAMOL CHIKI LETTER AAWOL CHIKI LETTER LIOL CHIKI LETTER ISOL CHIKI LETTER IHOL CHIKI LETTER INYOL CHIKI LETTER IROL C\
    \HIKI LETTER LUOL CHIKI LETTER UCOL CHIKI LETTER UDOL CHIKI LETTER UNNOL CHIKI LETTER UYOL CHIKI LETTER LEOL CHIKI LETTER EPOL CHIKI LETTER EDDOL CHIKI LETTER ENOL CHIKI LETTER ERROL CHIKI LETTER LOOL CHIKI LETTER OTTOL CHIKI LETTER OBOL CHIKI LETTER OVOL \
    \CHIKI LETTER OHOL CHIKI MU TTUDDAGOL CHIKI GAAHLAA TTUDDAAGOL CHIKI MU-GAAHLAA TTUDDAAGOL CHIKI RELAAOL CHIKI PHAARKAAOL CHIKI AHADOL CHIKI PUNCTUATION MUCAADOL CHIKI PUNCTUATION DOUBLE MUCAADCYRILLIC SMALL LETTER ROUNDED VECYRILLIC SMALL LETTER LONG-LEGG\
    \ED DECYRILLIC SMALL LETTER NARROW OCYRILLIC SMALL LETTER WIDE ESCYRILLIC SMALL LETTER TALL TECYRILLIC SMALL LETTER THREE-LEGGED TECYRILLIC SMALL LETTER TALL HARD SIGNCYRILLIC SMALL LETTER TALL YATCYRILLIC SMALL LETTER UNBLENDED UKGEORGIAN MTAVRULI CAPITAL\
    \ LETTER ANGEORGIAN MTAVRULI CAPITAL LETTER BANGEORGIAN MTAVRULI CAPITAL LETTER GANGEORGIAN MTAVRULI CAPITAL LETTER DONGEORGIAN MTAVRULI CAPITAL LETTER ENGEORGIAN MTAVRULI CAPITAL LETTER VINGEORGIAN MTAVRULI CAPITAL LETTER ZENGEORGIAN MTAVRULI CAPITAL LETT\
    \ER TANGEORGIAN MTAVRULI CAPITAL LETTER INGEORGIAN MTAVRULI CAPITAL LETTER KANGEORGIAN MTAVRULI CAPITAL LETTER LASGEORGIAN MTAVRULI CAPITAL LETTER MANGEORGIAN MTAVRULI CAPITAL LETTER NARGEORGIAN MTAVRULI CAPITAL LETTER ONGEORGIAN MTAVRULI CAPITAL LETTER PA\
    \RGEORGIAN MTAVRULI CAPITAL LETTER ZHARGEORGIAN MTAVRULI CAPITAL LETTER RAEGEORGIAN MTAVRULI CAPITAL LETTER SANGEORGIAN MTAVRULI CAPITAL LETTER TARGEORGIAN MTAVRULI CAPITAL LETTER UNGEORGIAN MTAVRULI CAPITAL LETTER PHARGEORGIAN MTAVRULI CAPITAL LETTER KHAR\
    \GEORGIAN MTAVRULI CAPITAL LETTER GHANGEORGIAN MTAVRULI CAPITAL LETTER QARGEORGIAN MTAVRULI CAPITAL LETTER SHINGEORGIAN MTAVRULI CAPITAL LETTER CHINGEORGIAN MTAVRULI CAPITAL LETTER CANGEORGIAN MTAVRULI CAPITAL LETTER JILGEORGIAN MTAVRULI CAPITAL LETTER CIL\
    \GEORGIAN MTAVRULI CAPITAL LETTER CHARGEORGIAN MTAVRULI CAPITAL LETTER XANGEORGIAN MTAVRULI CAPITAL LETTER JHANGEORGIAN MTAVRULI CAPITAL LETTER HAEGEORGIAN MTAVRULI CAPITAL LETTER HEGEORGIAN MTAVRULI CAPITAL LETTER HIEGEORGIAN MTAVRULI CAPITAL LETTER WEGEO\
    \RGIAN MTAVRULI CAPITAL LETTER HARGEORGIAN MTAVRULI CAPITAL LETTER HOEGEORGIAN MTAVRULI CAPITAL LETTER FIGEORGIAN MTAVRULI CAPITAL LETTER YNGEORGIAN MTAVRULI CAPITAL LETTER ELIFIGEORGIAN MTAVRULI CAPITAL LETTER TURNED GANGEORGIAN MTAVRULI CAPITAL LETTER AI\
    \NGEORGIAN MTAVRULI CAPITAL LETTER AENGEORGIAN MTAVRULI CAPITAL LETTER HARD SIGNGEORGIAN MTAVRULI CAPITAL LETTER LABIAL SIGNSUNDANESE PUNCTUATION BINDU SURYASUNDANESE PUNCTUATION BINDU PANGLONGSUNDANESE PUNCTUATION BINDU PURNAMASUNDANESE PUNCTUATION BINDU \
    \CAKRASUNDANESE PUNCTUATION BINDU LEU SATANGASUNDANESE PUNCTUATION BINDU KA SATANGASUNDANESE PUNCTUATION BINDU DA SATANGASUNDANESE PUNCTUATION BINDU BA SATANGAVEDIC TONE KARSHANAVEDIC TONE SHARAVEDIC TONE PRENKHAVEDIC SIGN NIHSHVASAVEDIC SIGN YAJURVEDIC MI\
    \DLINE SVARITAVEDIC TONE YAJURVEDIC AGGRAVATED INDEPENDENT SVARITAVEDIC TONE YAJURVEDIC INDEPENDENT SVARITAVEDIC TONE YAJURVEDIC KATHAKA INDEPENDENT SVARITAVEDIC TONE CANDRA BELOWVEDIC TONE YAJURVEDIC KATHAKA INDEPENDENT SVARITA SCHROEDERVEDIC TONE DOUBLE \
    \SVARITAVEDIC TONE TRIPLE SVARITAVEDIC TONE KATHAKA ANUDATTAVEDIC TONE DOT BELOWVEDIC TONE TWO DOTS BELOWVEDIC TONE THREE DOTS BELOWVEDIC TONE RIGVEDIC KASHMIRI INDEPENDENT SVARITAVEDIC TONE ATHARVAVEDIC INDEPENDENT SVARITAVEDIC SIGN VISARGA SVARITAVEDIC S\
    \IGN VISARGA UDATTAVEDIC SIGN REVERSED VISARGA UDATTAVEDIC SIGN VISARGA ANUDATTAVEDIC SIGN REVERSED VISARGA ANUDATTAVEDIC SIGN VISARGA UDATTA WITH TAILVEDIC SIGN VISARGA ANUDATTA WITH TAILVEDIC SIGN ANUSVARA ANTARGOMUKHAVEDIC SIGN ANUSVARA BAHIRGOMUKHAVEDI\
    \C SIGN ANUSVARA VAMAGOMUKHAVEDIC SIGN ANUSVARA VAMAGOMUKHA WITH TAILVEDIC SIGN TIRYAKVEDIC SIGN HEXIFORM LONG ANUSVARAVEDIC SIGN LONG ANUSVARAVEDIC SIGN RTHANG LONG ANUSVARAVEDIC SIGN ANUSVARA UBHAYATO MUKHAVEDIC SIGN ARDHAVISARGAVEDIC SIGN ROTATED ARDHAV\
    \ISARGAVEDIC TONE CANDRA ABOVEVEDIC SIGN JIHVAMULIYAVEDIC SIGN UPADHMANIYAVEDIC SIGN ATIKRAMAVEDIC TONE RING ABOVEVEDIC TONE DOUBLE RING ABOVEVEDIC SIGN DOUBLE ANUSVARA ANTARGOMUKHALATIN LETTER SMALL CAPITAL ALATIN LETTER SMALL CAPITAL AELATIN SMALL LETTER\
    \ TURNED AELATIN LETTER SMALL CAPITAL BARRED BLATIN LETTER SMALL CAPITAL CLATIN LETTER SMALL CAPITAL DLATIN LETTER SMALL CAPITAL ETHLATIN LETTER SMALL CAPITAL ELATIN SMALL LETTER TURNED OPEN ELATIN SMALL LETTER TURNED ILATIN LETTER SMALL CAPITAL JLATIN LET\
    \TER SMALL CAPITAL KLATIN LETTER SMALL CAPITAL L WITH STROKELATIN LETTER SMALL CAPITAL MLATIN LETTER SMALL CAPITAL REVERSED NLATIN LETTER SMALL CAPITAL OLATIN LETTER SMALL CAPITAL OPEN OLATIN SMALL LETTER SIDEWAYS OLATIN SMALL LETTER SIDEWAYS OPEN OLATIN S\
    \MALL LETTER SIDEWAYS O WITH STROKELATIN SMALL LETTER TURNED OELATIN LETTER SMALL CAPITAL OULATIN SMALL LETTER TOP HALF OLATIN SMALL LETTER BOTTOM HALF OLATIN LETTER SMALL CAPITAL PLATIN LETTER SMALL CAPITAL REVERSED RLATIN LETTER SMALL CAPITAL TURNED RLAT\
    \IN LETTER SMALL CAPITAL TLATIN LETTER SMALL CAPITAL ULATIN SMALL LETTER SIDEWAYS ULATIN SMALL LETTER SIDEWAYS DIAERESIZED ULATIN SMALL LETTER SIDEWAYS TURNED MLATIN LETTER SMALL CAPITAL VLATIN LETTER SMALL CAPITAL WLATIN LETTER SMALL CAPITAL ZLATIN LETTER\
    \ SMALL CAPITAL EZHLATIN LETTER VOICED LARYNGEAL SPIRANTLATIN LETTER AINGREEK LETTER SMALL CAPITAL GAMMAGREEK LETTER SMALL CAPITAL LAMDAGREEK LETTER SMALL CAPITAL PIGREEK LETTER SMALL CAPITAL RHOGREEK LETTER SMALL CAPITAL PSICYRILLIC LETTER SMALL CAPITAL E\
    \LMODIFIER LETTER CAPITAL AMODIFIER LETTER CAPITAL AEMODIFIER LETTER CAPITAL BMODIFIER LETTER CAPITAL BARRED BMODIFIER LETTER CAPITAL DMODIFIER LETTER CAPITAL EMODIFIER LETTER CAPITAL REVERSED EMODIFIER LETTER CAPITAL GMODIFIER LETTER CAPITAL HMODIFIER LET\
    \TER CAPITAL IMODIFIER LETTER CAPITAL JMODIFIER LETTER CAPITAL KMODIFIER LETTER CAPITAL LMODIFIER LETTER CAPITAL MMODIFIER LETTER CAPITAL NMODIFIER LETTER CAPITAL REVERSED NMODIFIER LETTER CAPITAL OMODIFIER LETTER CAPITAL OUMODIFIER LETTER CAPITAL PMODIFIE\
    \R LETTER CAPITAL RMODIFIER LETTER CAPITAL TMODIFIER LETTER CAPITAL UMODIFIER LETTER CAPITAL WMODIFIER LETTER SMALL AMODIFIER LETTER SMALL TURNED AMODIFIER LETTER SMALL ALPHAMODIFIER LETTER SMALL TURNED AEMODIFIER LETTER SMALL BMODIFIER LETTER SMALL DMODIF\
    \IER LETTER SMALL EMODIFIER LETTER SMALL SCHWAMODIFIER LETTER SMALL OPEN EMODIFIER LETTER SMALL TURNED OPEN EMODIFIER LETTER SMALL GMODIFIER LETTER SMALL TURNED IMODIFIER LETTER SMALL KMODIFIER LETTER SMALL MMODIFIER LETTER SMALL ENGMODIFIER LETTER SMALL O\
    \MODIFIER LETTER SMALL OPEN OMODIFIER LETTER SMALL TOP HALF OMODIFIER LETTER SMALL BOTTOM HALF OMODIFIER LETTER SMALL PMODIFIER LETTER SMALL TMODIFIER LETTER SMALL UMODIFIER LETTER SMALL SIDEWAYS UMODIFIER LETTER SMALL TURNED MMODIFIER LETTER SMALL VMODIFI\
    \ER LETTER SMALL AINMODIFIER LETTER SMALL BETAMODIFIER LETTER SMALL GREEK GAMMAMODIFIER LETTER SMALL DELTAMODIFIER LETTER SMALL GREEK PHIMODIFIER LETTER SMALL CHILATIN SUBSCRIPT SMALL LETTER ILATIN SUBSCRIPT SMALL LETTER RLATIN SUBSCRIPT SMALL LETTER ULATI\
    \N SUBSCRIPT SMALL LETTER VGREEK SUBSCRIPT SMALL LETTER BETAGREEK SUBSCRIPT SMALL LETTER GAMMAGREEK SUBSCRIPT SMALL LETTER RHOGREEK SUBSCRIPT SMALL LETTER PHIGREEK SUBSCRIPT SMALL LETTER CHILATIN SMALL LETTER UELATIN SMALL LETTER B WITH MIDDLE TILDELATIN S\
    \MALL LETTER D WITH MIDDLE TILDELATIN SMALL LETTER F WITH MIDDLE TILDELATIN SMALL LETTER M WITH MIDDLE TILDELATIN SMALL LETTER N WITH MIDDLE TILDELATIN SMALL LETTER P WITH MIDDLE TILDELATIN SMALL LETTER R WITH MIDDLE TILDELATIN SMALL LETTER R WITH FISHHOOK\
    \ AND MIDDLE TILDELATIN SMALL LETTER S WITH MIDDLE TILDELATIN SMALL LETTER T WITH MIDDLE TILDELATIN SMALL LETTER Z WITH MIDDLE TILDELATIN SMALL LETTER TURNED GMODIFIER LETTER CYRILLIC ENLATIN SMALL LETTER INSULAR GLATIN SMALL LETTER TH WITH STRIKETHROUGHLA\
    \TIN SMALL CAPITAL LETTER I WITH STROKELATIN SMALL LETTER IOTA WITH STROKELATIN SMALL LETTER P WITH STROKELATIN SMALL CAPITAL LETTER U WITH STROKELATIN SMALL LETTER UPSILON WITH STROKELATIN SMALL LETTER B WITH PALATAL HOOKLATIN SMALL LETTER D WITH PALATAL \
    \HOOKLATIN SMALL LETTER F WITH PALATAL HOOKLATIN SMALL LETTER G WITH PALATAL HOOKLATIN SMALL LETTER K WITH PALATAL HOOKLATIN SMALL LETTER L WITH PALATAL HOOKLATIN SMALL LETTER M WITH PALATAL HOOKLATIN SMALL LETTER N WITH PALATAL HOOKLATIN SMALL LETTER P WI\
    \TH PALATAL HOOKLATIN SMALL LETTER R WITH PALATAL HOOKLATIN SMALL LETTER S WITH PALATAL HOOKLATIN SMALL LETTER ESH WITH PALATAL HOOKLATIN SMALL LETTER V WITH PALATAL HOOKLATIN SMALL LETTER X WITH PALATAL HOOKLATIN SMALL LETTER Z WITH PALATAL HOOKLATIN SMAL\
    \L LETTER A WITH RETROFLEX HOOKLATIN SMALL LETTER ALPHA WITH RETROFLEX HOOKLATIN SMALL LETTER D WITH HOOK AND TAILLATIN SMALL LETTER E WITH RETROFLEX HOOKLATIN SMALL LETTER OPEN E WITH RETROFLEX HOOKLATIN SMALL LETTER REVERSED OPEN E WITH RETROFLEX HOOKLAT\
    \IN SMALL LETTER SCHWA WITH RETROFLEX HOOKLATIN SMALL LETTER I WITH RETROFLEX HOOKLATIN SMALL LETTER OPEN O WITH RETROFLEX HOOKLATIN SMALL LETTER ESH WITH RETROFLEX HOOKLATIN SMALL LETTER U WITH RETROFLEX HOOKLATIN SMALL LETTER EZH WITH RETROFLEX HOOKMODIF\
    \IER LETTER SMALL TURNED ALPHAMODIFIER LETTER SMALL CMODIFIER LETTER SMALL C WITH CURLMODIFIER LETTER SMALL ETHMODIFIER LETTER SMALL REVERSED OPEN EMODIFIER LETTER SMALL FMODIFIER LETTER SMALL DOTLESS J WITH STROKEMODIFIER LETTER SMALL SCRIPT GMODIFIER LET\
    \TER SMALL TURNED HMODIFIER LETTER SMALL I WITH STROKEMODIFIER LETTER SMALL IOTAMODIFIER LETTER SMALL CAPITAL IMODIFIER LETTER SMALL CAPITAL I WITH STROKEMODIFIER LETTER SMALL J WITH CROSSED-TAILMODIFIER LETTER SMALL L WITH RETROFLEX HOOKMODIFIER LETTER SM\
    \ALL L WITH PALATAL HOOKMODIFIER LETTER SMALL CAPITAL LMODIFIER LETTER SMALL M WITH HOOKMODIFIER LETTER SMALL TURNED M WITH LONG LEGMODIFIER LETTER SMALL N WITH LEFT HOOKMODIFIER LETTER SMALL N WITH RETROFLEX HOOKMODIFIER LETTER SMALL CAPITAL NMODIFIER LET\
    \TER SMALL BARRED OMODIFIER LETTER SMALL PHIMODIFIER LETTER SMALL S WITH HOOKMODIFIER LETTER SMALL ESHMODIFIER LETTER SMALL T WITH PALATAL HOOKMODIFIER LETTER SMALL U BARMODIFIER LETTER SMALL UPSILONMODIFIER LETTER SMALL CAPITAL UMODIFIER LETTER SMALL V WI\
    \TH HOOKMODIFIER LETTER SMALL TURNED VMODIFIER LETTER SMALL ZMODIFIER LETTER SMALL Z WITH RETROFLEX HOOKMODIFIER LETTER SMALL Z WITH CURLMODIFIER LETTER SMALL EZHMODIFIER LETTER SMALL THETACOMBINING DOTTED GRAVE ACCENTCOMBINING DOTTED ACUTE ACCENTCOMBINING\
    \ SNAKE BELOWCOMBINING SUSPENSION MARKCOMBINING MACRON-ACUTECOMBINING GRAVE-MACRONCOMBINING MACRON-GRAVECOMBINING ACUTE-MACRONCOMBINING GRAVE-ACUTE-GRAVECOMBINING ACUTE-GRAVE-ACUTECOMBINING LATIN SMALL LETTER R BELOWCOMBINING BREVE-MACRONCOMBINING MACRON-B\
    \REVECOMBINING DOUBLE CIRCUMFLEX ABOVECOMBINING OGONEK ABOVECOMBINING ZIGZAG BELOWCOMBINING IS BELOWCOMBINING UR ABOVECOMBINING US ABOVECOMBINING LATIN SMALL LETTER FLATTENED OPEN A ABOVECOMBINING LATIN SMALL LETTER AECOMBINING LATIN SMALL LETTER AOCOMBINI\
    \NG LATIN SMALL LETTER AVCOMBINING LATIN SMALL LETTER C CEDILLACOMBINING LATIN SMALL LETTER INSULAR DCOMBINING LATIN SMALL LETTER ETHCOMBINING LATIN SMALL LETTER GCOMBINING LATIN LETTER SMALL CAPITAL GCOMBINING LATIN SMALL LETTER KCOMBINING LATIN SMALL LET\
    \TER LCOMBINING LATIN LETTER SMALL CAPITAL LCOMBINING LATIN LETTER SMALL CAPITAL MCOMBINING LATIN SMALL LETTER NCOMBINING LATIN LETTER SMALL CAPITAL NCOMBINING LATIN LETTER SMALL CAPITAL RCOMBINING LATIN SMALL LETTER R ROTUNDACOMBINING LATIN SMALL LETTER S\
    \COMBINING LATIN SMALL LETTER LONG SCOMBINING LATIN SMALL LETTER ZCOMBINING LATIN SMALL LETTER ALPHACOMBINING LATIN SMALL LETTER BCOMBINING LATIN SMALL LETTER BETACOMBINING LATIN SMALL LETTER SCHWACOMBINING LATIN SMALL LETTER FCOMBINING LATIN SMALL LETTER \
    \L WITH DOUBLE MIDDLE TILDECOMBINING LATIN SMALL LETTER O WITH LIGHT CENTRALIZATION STROKECOMBINING LATIN SMALL LETTER PCOMBINING LATIN SMALL LETTER ESHCOMBINING LATIN SMALL LETTER U WITH LIGHT CENTRALIZATION STROKECOMBINING LATIN SMALL LETTER WCOMBINING L\
    \ATIN SMALL LETTER A WITH DIAERESISCOMBINING LATIN SMALL LETTER O WITH DIAERESISCOMBINING LATIN SMALL LETTER U WITH DIAERESISCOMBINING UP TACK ABOVECOMBINING KAVYKA ABOVE RIGHTCOMBINING KAVYKA ABOVE LEFTCOMBINING DOT ABOVE LEFTCOMBINING WIDE INVERTED BRIDG\
    \E BELOWCOMBINING DOT BELOW LEFTCOMBINING DELETION MARKCOMBINING DOUBLE INVERTED BREVE BELOWCOMBINING ALMOST EQUAL TO BELOWCOMBINING LEFT ARROWHEAD ABOVECOMBINING RIGHT ARROWHEAD AND DOWN ARROWHEAD BELOWLATIN CAPITAL LETTER A WITH RING BELOWLATIN SMALL LET\
    \TER A WITH RING BELOWLATIN CAPITAL LETTER B WITH DOT ABOVELATIN SMALL LETTER B WITH DOT ABOVELATIN CAPITAL LETTER B WITH DOT BELOWLATIN SMALL LETTER B WITH DOT BELOWLATIN CAPITAL LETTER B WITH LINE BELOWLATIN SMALL LETTER B WITH LINE BELOWLATIN CAPITAL LE\
    \TTER C WITH CEDILLA AND ACUTELATIN SMALL LETTER C WITH CEDILLA AND ACUTELATIN CAPITAL LETTER D WITH DOT ABOVELATIN SMALL LETTER D WITH DOT ABOVELATIN CAPITAL LETTER D WITH DOT BELOWLATIN SMALL LETTER D WITH DOT BELOWLATIN CAPITAL LETTER D WITH LINE BELOWL\
    \ATIN SMALL LETTER D WITH LINE BELOWLATIN CAPITAL LETTER D WITH CEDILLALATIN SMALL LETTER D WITH CEDILLALATIN CAPITAL LETTER D WITH CIRCUMFLEX BELOWLATIN SMALL LETTER D WITH CIRCUMFLEX BELOWLATIN CAPITAL LETTER E WITH MACRON AND GRAVELATIN SMALL LETTER E W\
    \ITH MACRON AND GRAVELATIN CAPITAL LETTER E WITH MACRON AND ACUTELATIN SMALL LETTER E WITH MACRON AND ACUTELATIN CAPITAL LETTER E WITH CIRCUMFLEX BELOWLATIN SMALL LETTER E WITH CIRCUMFLEX BELOWLATIN CAPITAL LETTER E WITH TILDE BELOWLATIN SMALL LETTER E WIT\
    \H TILDE BELOWLATIN CAPITAL LETTER E WITH CEDILLA AND BREVELATIN SMALL LETTER E WITH CEDILLA AND BREVELATIN CAPITAL LETTER F WITH DOT ABOVELATIN SMALL LETTER F WITH DOT ABOVELATIN CAPITAL LETTER G WITH MACRONLATIN SMALL LETTER G WITH MACRONLATIN CAPITAL LE\
    \TTER H WITH DOT ABOVELATIN SMALL LETTER H WITH DOT ABOVELATIN CAPITAL LETTER H WITH DOT BELOWLATIN SMALL LETTER H WITH DOT BELOWLATIN CAPITAL LETTER H WITH DIAERESISLATIN SMALL LETTER H WITH DIAERESISLATIN CAPITAL LETTER H WITH CEDILLALATIN SMALL LETTER H\
    \ WITH CEDILLALATIN CAPITAL LETTER H WITH BREVE BELOWLATIN SMALL LETTER H WITH BREVE BELOWLATIN CAPITAL LETTER I WITH TILDE BELOWLATIN SMALL LETTER I WITH TILDE BELOWLATIN CAPITAL LETTER I WITH DIAERESIS AND ACUTELATIN SMALL LETTER I WITH DIAERESIS AND ACU\
    \TELATIN CAPITAL LETTER K WITH ACUTELATIN SMALL LETTER K WITH ACUTELATIN CAPITAL LETTER K WITH DOT BELOWLATIN SMALL LETTER K WITH DOT BELOWLATIN CAPITAL LETTER K WITH LINE BELOWLATIN SMALL LETTER K WITH LINE BELOWLATIN CAPITAL LETTER L WITH DOT BELOWLATIN \
    \SMALL LETTER L WITH DOT BELOWLATIN CAPITAL LETTER L WITH DOT BELOW AND MACRONLATIN SMALL LETTER L WITH DOT BELOW AND MACRONLATIN CAPITAL LETTER L WITH LINE BELOWLATIN SMALL LETTER L WITH LINE BELOWLATIN CAPITAL LETTER L WITH CIRCUMFLEX BELOWLATIN SMALL LE\
    \TTER L WITH CIRCUMFLEX BELOWLATIN CAPITAL LETTER M WITH ACUTELATIN SMALL LETTER M WITH ACUTELATIN CAPITAL LETTER M WITH DOT ABOVELATIN SMALL LETTER M WITH DOT ABOVELATIN CAPITAL LETTER M WITH DOT BELOWLATIN SMALL LETTER M WITH DOT BELOWLATIN CAPITAL LETTE\
    \R N WITH DOT ABOVELATIN SMALL LETTER N WITH DOT ABOVELATIN CAPITAL LETTER N WITH DOT BELOWLATIN SMALL LETTER N WITH DOT BELOWLATIN CAPITAL LETTER N WITH LINE BELOWLATIN SMALL LETTER N WITH LINE BELOWLATIN CAPITAL LETTER N WITH CIRCUMFLEX BELOWLATIN SMALL \
    \LETTER N WITH CIRCUMFLEX BELOWLATIN CAPITAL LETTER O WITH TILDE AND ACUTELATIN SMALL LETTER O WITH TILDE AND ACUTELATIN CAPITAL LETTER O WITH TILDE AND DIAERESISLATIN SMALL LETTER O WITH TILDE AND DIAERESISLATIN CAPITAL LETTER O WITH MACRON AND GRAVELATIN\
    \ SMALL LETTER O WITH MACRON AND GRAVELATIN CAPITAL LETTER O WITH MACRON AND ACUTELATIN SMALL LETTER O WITH MACRON AND ACUTELATIN CAPITAL LETTER P WITH ACUTELATIN SMALL LETTER P WITH ACUTELATIN CAPITAL LETTER P WITH DOT ABOVELATIN SMALL LETTER P WITH DOT A\
    \BOVELATIN CAPITAL LETTER R WITH DOT ABOVELATIN SMALL LETTER R WITH DOT ABOVELATIN CAPITAL LETTER R WITH DOT BELOWLATIN SMALL LETTER R WITH DOT BELOWLATIN CAPITAL LETTER R WITH DOT BELOW AND MACRONLATIN SMALL LETTER R WITH DOT BELOW AND MACRONLATIN CAPITAL\
    \ LETTER R WITH LINE BELOWLATIN SMALL LETTER R WITH LINE BELOWLATIN CAPITAL LETTER S WITH DOT ABOVELATIN SMALL LETTER S WITH DOT ABOVELATIN CAPITAL LETTER S WITH DOT BELOWLATIN SMALL LETTER S WITH DOT BELOWLATIN CAPITAL LETTER S WITH ACUTE AND DOT ABOVELAT\
    \IN SMALL LETTER S WITH ACUTE AND DOT ABOVELATIN CAPITAL LETTER S WITH CARON AND DOT ABOVELATIN SMALL LETTER S WITH CARON AND DOT ABOVELATIN CAPITAL LETTER S WITH DOT BELOW AND DOT ABOVELATIN SMALL LETTER S WITH DOT BELOW AND DOT ABOVELATIN CAPITAL LETTER \
    \T WITH DOT ABOVELATIN SMALL LETTER T WITH DOT ABOVELATIN CAPITAL LETTER T WITH DOT BELOWLATIN SMALL LETTER T WITH DOT BELOWLATIN CAPITAL LETTER T WITH LINE BELOWLATIN SMALL LETTER T WITH LINE BELOWLATIN CAPITAL LETTER T WITH CIRCUMFLEX BELOWLATIN SMALL LE\
    \TTER T WITH CIRCUMFLEX BELOWLATIN CAPITAL LETTER U WITH DIAERESIS BELOWLATIN SMALL LETTER U WITH DIAERESIS BELOWLATIN CAPITAL LETTER U WITH TILDE BELOWLATIN SMALL LETTER U WITH TILDE BELOWLATIN CAPITAL LETTER U WITH CIRCUMFLEX BELOWLATIN SMALL LETTER U WI\
    \TH CIRCUMFLEX BELOWLATIN CAPITAL LETTER U WITH TILDE AND ACUTELATIN SMALL LETTER U WITH TILDE AND ACUTELATIN CAPITAL LETTER U WITH MACRON AND DIAERESISLATIN SMALL LETTER U WITH MACRON AND DIAERESISLATIN CAPITAL LETTER V WITH TILDELATIN SMALL LETTER V WITH\
    \ TILDELATIN CAPITAL LETTER V WITH DOT BELOWLATIN SMALL LETTER V WITH DOT BELOWLATIN CAPITAL LETTER W WITH GRAVELATIN SMALL LETTER W WITH GRAVELATIN CAPITAL LETTER W WITH ACUTELATIN SMALL LETTER W WITH ACUTELATIN CAPITAL LETTER W WITH DIAERESISLATIN SMALL \
    \LETTER W WITH DIAERESISLATIN CAPITAL LETTER W WITH DOT ABOVELATIN SMALL LETTER W WITH DOT ABOVELATIN CAPITAL LETTER W WITH DOT BELOWLATIN SMALL LETTER W WITH DOT BELOWLATIN CAPITAL LETTER X WITH DOT ABOVELATIN SMALL LETTER X WITH DOT ABOVELATIN CAPITAL LE\
    \TTER X WITH DIAERESISLATIN SMALL LETTER X WITH DIAERESISLATIN CAPITAL LETTER Y WITH DOT ABOVELATIN SMALL LETTER Y WITH DOT ABOVELATIN CAPITAL LETTER Z WITH CIRCUMFLEXLATIN SMALL LETTER Z WITH CIRCUMFLEXLATIN CAPITAL LETTER Z WITH DOT BELOWLATIN SMALL LETT\
    \ER Z WITH DOT BELOWLATIN CAPITAL LETTER Z WITH LINE BELOWLATIN SMALL LETTER Z WITH LINE BELOWLATIN SMALL LETTER H WITH LINE BELOWLATIN SMALL LETTER T WITH DIAERESISLATIN SMALL LETTER W WITH RING ABOVELATIN SMALL LETTER Y WITH RING ABOVELATIN SMALL LETTER \
    \A WITH RIGHT HALF RINGLATIN SMALL LETTER LONG S WITH DOT ABOVELATIN SMALL LETTER LONG S WITH DIAGONAL STROKELATIN SMALL LETTER LONG S WITH HIGH STROKELATIN CAPITAL LETTER SHARP SLATIN SMALL LETTER DELTALATIN CAPITAL LETTER A WITH DOT BELOWLATIN SMALL LETT\
    \ER A WITH DOT BELOWLATIN CAPITAL LETTER A WITH HOOK ABOVELATIN SMALL LETTER A WITH HOOK ABOVELATIN CAPITAL LETTER A WITH CIRCUMFLEX AND ACUTELATIN SMALL LETTER A WITH CIRCUMFLEX AND ACUTELATIN CAPITAL LETTER A WITH CIRCUMFLEX AND GRAVELATIN SMALL LETTER A\
    \ WITH CIRCUMFLEX AND GRAVELATIN CAPITAL LETTER A WITH CIRCUMFLEX AND HOOK ABOVELATIN SMALL LETTER A WITH CIRCUMFLEX AND HOOK ABOVELATIN CAPITAL LETTER A WITH CIRCUMFLEX AND TILDELATIN SMALL LETTER A WITH CIRCUMFLEX AND TILDELATIN CAPITAL LETTER A WITH CIR\
    \CUMFLEX AND DOT BELOWLATIN SMALL LETTER A WITH CIRCUMFLEX AND DOT BELOWLATIN CAPITAL LETTER A WITH BREVE AND ACUTELATIN SMALL LETTER A WITH BREVE AND ACUTELATIN CAPITAL LETTER A WITH BREVE AND GRAVELATIN SMALL LETTER A WITH BREVE AND GRAVELATIN CAPITAL LE\
    \TTER A WITH BREVE AND HOOK ABOVELATIN SMALL LETTER A WITH BREVE AND HOOK ABOVELATIN CAPITAL LETTER A WITH BREVE AND TILDELATIN SMALL LETTER A WITH BREVE AND TILDELATIN CAPITAL LETTER A WITH BREVE AND DOT BELOWLATIN SMALL LETTER A WITH BREVE AND DOT BELOWL\
    \ATIN CAPITAL LETTER E WITH DOT BELOWLATIN SMALL LETTER E WITH DOT BELOWLATIN CAPITAL LETTER E WITH HOOK ABOVELATIN SMALL LETTER E WITH HOOK ABOVELATIN CAPITAL LETTER E WITH TILDELATIN SMALL LETTER E WITH TILDELATIN CAPITAL LETTER E WITH CIRCUMFLEX AND ACU\
    \TELATIN SMALL LETTER E WITH CIRCUMFLEX AND ACUTELATIN CAPITAL LETTER E WITH CIRCUMFLEX AND GRAVELATIN SMALL LETTER E WITH CIRCUMFLEX AND GRAVELATIN CAPITAL LETTER E WITH CIRCUMFLEX AND HOOK ABOVELATIN SMALL LETTER E WITH CIRCUMFLEX AND HOOK ABOVELATIN CAP\
    \ITAL LETTER E WITH CIRCUMFLEX AND TILDELATIN SMALL LETTER E WITH CIRCUMFLEX AND TILDELATIN CAPITAL LETTER E WITH CIRCUMFLEX AND DOT BELOWLATIN SMALL LETTER E WITH CIRCUMFLEX AND DOT BELOWLATIN CAPITAL LETTER I WITH HOOK ABOVELATIN SMALL LETTER I WITH HOOK\
    \ ABOVELATIN CAPITAL LETTER I WITH DOT BELOWLATIN SMALL LETTER I WITH DOT BELOWLATIN CAPITAL LETTER O WITH DOT BELOWLATIN SMALL LETTER O WITH DOT BELOWLATIN CAPITAL LETTER O WITH HOOK ABOVELATIN SMALL LETTER O WITH HOOK ABOVELATIN CAPITAL LETTER O WITH CIR\
    \CUMFLEX AND ACUTELATIN SMALL LETTER O WITH CIRCUMFLEX AND ACUTELATIN CAPITAL LETTER O WITH CIRCUMFLEX AND GRAVELATIN SMALL LETTER O WITH CIRCUMFLEX AND GRAVELATIN CAPITAL LETTER O WITH CIRCUMFLEX AND HOOK ABOVELATIN SMALL LETTER O WITH CIRCUMFLEX AND HOOK\
    \ ABOVELATIN CAPITAL LETTER O WITH CIRCUMFLEX AND TILDELATIN SMALL LETTER O WITH CIRCUMFLEX AND TILDELATIN CAPITAL LETTER O WITH CIRCUMFLEX AND DOT BELOWLATIN SMALL LETTER O WITH CIRCUMFLEX AND DOT BELOWLATIN CAPITAL LETTER O WITH HORN AND ACUTELATIN SMALL\
    \ LETTER O WITH HORN AND ACUTELATIN CAPITAL LETTER O WITH HORN AND GRAVELATIN SMALL LETTER O WITH HORN AND GRAVELATIN CAPITAL LETTER O WITH HORN AND HOOK ABOVELATIN SMALL LETTER O WITH HORN AND HOOK ABOVELATIN CAPITAL LETTER O WITH HORN AND TILDELATIN SMAL\
    \L LETTER O WITH HORN AND TILDELATIN CAPITAL LETTER O WITH HORN AND DOT BELOWLATIN SMALL LETTER O WITH HORN AND DOT BELOWLATIN CAPITAL LETTER U WITH DOT BELOWLATIN SMALL LETTER U WITH DOT BELOWLATIN CAPITAL LETTER U WITH HOOK ABOVELATIN SMALL LETTER U WITH\
    \ HOOK ABOVELATIN CAPITAL LETTER U WITH HORN AND ACUTELATIN SMALL LETTER U WITH HORN AND ACUTELATIN CAPITAL LETTER U WITH HORN AND GRAVELATIN SMALL LETTER U WITH HORN AND GRAVELATIN CAPITAL LETTER U WITH HORN AND HOOK ABOVELATIN SMALL LETTER U WITH HORN AN\
    \D HOOK ABOVELATIN CAPITAL LETTER U WITH HORN AND TILDELATIN SMALL LETTER U WITH HORN AND TILDELATIN CAPITAL LETTER U WITH HORN AND DOT BELOWLATIN SMALL LETTER U WITH HORN AND DOT BELOWLATIN CAPITAL LETTER Y WITH GRAVELATIN SMALL LETTER Y WITH GRAVELATIN C\
    \APITAL LETTER Y WITH DOT BELOWLATIN SMALL LETTER Y WITH DOT BELOWLATIN CAPITAL LETTER Y WITH HOOK ABOVELATIN SMALL LETTER Y WITH HOOK ABOVELATIN CAPITAL LETTER Y WITH TILDELATIN SMALL LETTER Y WITH TILDELATIN CAPITAL LETTER MIDDLE-WELSH LLLATIN SMALL LETT\
    \ER MIDDLE-WELSH LLLATIN CAPITAL LETTER MIDDLE-WELSH VLATIN SMALL LETTER MIDDLE-WELSH VLATIN CAPITAL LETTER Y WITH LOOPLATIN SMALL LETTER Y WITH LOOPGREEK SMALL LETTER ALPHA WITH PSILIGREEK SMALL LETTER ALPHA WITH DASIAGREEK SMALL LETTER ALPHA WITH PSILI A\
    \ND VARIAGREEK SMALL LETTER ALPHA WITH DASIA AND VARIAGREEK SMALL LETTER ALPHA WITH PSILI AND OXIAGREEK SMALL LETTER ALPHA WITH DASIA AND OXIAGREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENIGREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENIGREEK CAPITA\
    \L LETTER ALPHA WITH PSILIGREEK CAPITAL LETTER ALPHA WITH DASIAGREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIAGREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIAGREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIAGREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIAGREEK C\
    \APITAL LETTER ALPHA WITH PSILI AND PERISPOMENIGREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENIGREEK SMALL LETTER EPSILON WITH PSILIGREEK SMALL LETTER EPSILON WITH DASIAGREEK SMALL LETTER EPSILON WITH PSILI AND VARIAGREEK SMALL LETTER EPSILON WITH DAS\
    \IA AND VARIAGREEK SMALL LETTER EPSILON WITH PSILI AND OXIAGREEK SMALL LETTER EPSILON WITH DASIA AND OXIAGREEK CAPITAL LETTER EPSILON WITH PSILIGREEK CAPITAL LETTER EPSILON WITH DASIAGREEK CAPITAL LETTER EPSILON WITH PSILI AND VARIAGREEK CAPITAL LETTER EPS\
    \ILON WITH DASIA AND VARIAGREEK CAPITAL LETTER EPSILON WITH PSILI AND OXIAGREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIAGREEK SMALL LETTER ETA WITH PSILIGREEK SMALL LETTER ETA WITH DASIAGREEK SMALL LETTER ETA WITH PSILI AND VARIAGREEK SMALL LETTER ETA WI\
    \TH DASIA AND VARIAGREEK SMALL LETTER ETA WITH PSILI AND OXIAGREEK SMALL LETTER ETA WITH DASIA AND OXIAGREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENIGREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENIGREEK CAPITAL LETTER ETA WITH PSILIGREEK CAPITAL LETTER\
    \ ETA WITH DASIAGREEK CAPITAL LETTER ETA WITH PSILI AND VARIAGREEK CAPITAL LETTER ETA WITH DASIA AND VARIAGREEK CAPITAL LETTER ETA WITH PSILI AND OXIAGREEK CAPITAL LETTER ETA WITH DASIA AND OXIAGREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENIGREEK CAPIT\
    \AL LETTER ETA WITH DASIA AND PERISPOMENIGREEK SMALL LETTER IOTA WITH PSILIGREEK SMALL LETTER IOTA WITH DASIAGREEK SMALL LETTER IOTA WITH PSILI AND VARIAGREEK SMALL LETTER IOTA WITH DASIA AND VARIAGREEK SMALL LETTER IOTA WITH PSILI AND OXIAGREEK SMALL LETT\
    \ER IOTA WITH DASIA AND OXIAGREEK SMALL LETTER IOTA WITH PSILI AND PERISPOMENIGREEK SMALL LETTER IOTA WITH DASIA AND PERISPOMENIGREEK CAPITAL LETTER IOTA WITH PSILIGREEK CAPITAL LETTER IOTA WITH DASIAGREEK CAPITAL LETTER IOTA WITH PSILI AND VARIAGREEK CAPI\
    \TAL LETTER IOTA WITH DASIA AND VARIAGREEK CAPITAL LETTER IOTA WITH PSILI AND OXIAGREEK CAPITAL LETTER IOTA WITH DASIA AND OXIAGREEK CAPITAL LETTER IOTA WITH PSILI AND PERISPOMENIGREEK CAPITAL LETTER IOTA WITH DASIA AND PERISPOMENIGREEK SMALL LETTER OMICRO\
    \N WITH PSILIGREEK SMALL LETTER OMICRON WITH DASIAGREEK SMALL LETTER OMICRON WITH PSILI AND VARIAGREEK SMALL LETTER OMICRON WITH DASIA AND VARIAGREEK SMALL LETTER OMICRON WITH PSILI AND OXIAGREEK SMALL LETTER OMICRON WITH DASIA AND OXIAGREEK CAPITAL LETTER\
    \ OMICRON WITH PSILIGREEK CAPITAL LETTER OMICRON WITH DASIAGREEK CAPITAL LETTER OMICRON WITH PSILI AND VARIAGREEK CAPITAL LETTER OMICRON WITH DASIA AND VARIAGREEK CAPITAL LETTER OMICRON WITH PSILI AND OXIAGREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIAGRE\
    \EK SMALL LETTER UPSILON WITH PSILIGREEK SMALL LETTER UPSILON WITH DASIAGREEK SMALL LETTER UPSILON WITH PSILI AND VARIAGREEK SMALL LETTER UPSILON WITH DASIA AND VARIAGREEK SMALL LETTER UPSILON WITH PSILI AND OXIAGREEK SMALL LETTER UPSILON WITH DASIA AND OX\
    \IAGREEK SMALL LETTER UPSILON WITH PSILI AND PERISPOMENIGREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENIGREEK CAPITAL LETTER UPSILON WITH DASIAGREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIAGREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIAGREEK CAPIT\
    \AL LETTER UPSILON WITH DASIA AND PERISPOMENIGREEK SMALL LETTER OMEGA WITH PSILIGREEK SMALL LETTER OMEGA WITH DASIAGREEK SMALL LETTER OMEGA WITH PSILI AND VARIAGREEK SMALL LETTER OMEGA WITH DASIA AND VARIAGREEK SMALL LETTER OMEGA WITH PSILI AND OXIAGREEK S\
    \MALL LETTER OMEGA WITH DASIA AND OXIAGREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENIGREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENIGREEK CAPITAL LETTER OMEGA WITH PSILIGREEK CAPITAL LETTER OMEGA WITH DASIAGREEK CAPITAL LETTER OMEGA WITH PSILI AND \
    \VARIAGREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIAGREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIAGREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIAGREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENIGREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENIGREEK\
    \ SMALL LETTER ALPHA WITH VARIAGREEK SMALL LETTER ALPHA WITH OXIAGREEK SMALL LETTER EPSILON WITH VARIAGREEK SMALL LETTER EPSILON WITH OXIAGREEK SMALL LETTER ETA WITH VARIAGREEK SMALL LETTER ETA WITH OXIAGREEK SMALL LETTER IOTA WITH VARIAGREEK SMALL LETTER \
    \IOTA WITH OXIAGREEK SMALL LETTER OMICRON WITH VARIAGREEK SMALL LETTER OMICRON WITH OXIAGREEK SMALL LETTER UPSILON WITH VARIAGREEK SMALL LETTER UPSILON WITH OXIAGREEK SMALL LETTER OMEGA WITH VARIAGREEK SMALL LETTER OMEGA WITH OXIAGREEK SMALL LETTER ALPHA W\
    \ITH PSILI AND YPOGEGRAMMENIGREEK SMALL LETTER ALPHA WITH DASIA AND YPOGEGRAMMENIGREEK SMALL LETTER ALPHA WITH PSILI AND VARIA AND YPOGEGRAMMENIGREEK SMALL LETTER ALPHA WITH DASIA AND VARIA AND YPOGEGRAMMENIGREEK SMALL LETTER ALPHA WITH PSILI AND OXIA AND \
    \YPOGEGRAMMENIGREEK SMALL LETTER ALPHA WITH DASIA AND OXIA AND YPOGEGRAMMENIGREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENIGREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENIGREEK CAPITAL LETTER ALPHA WITH PSILI AND \
    \PROSGEGRAMMENIGREEK CAPITAL LETTER ALPHA WITH DASIA AND PROSGEGRAMMENIGREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA AND PROSGEGRAMMENIGREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA AND PROSGEGRAMMENIGREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA AND PR\
    \OSGEGRAMMENIGREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA AND PROSGEGRAMMENIGREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENIGREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENIGREEK SMALL LETTER ETA WITH PSILI \
    \AND YPOGEGRAMMENIGREEK SMALL LETTER ETA WITH DASIA AND YPOGEGRAMMENIGREEK SMALL LETTER ETA WITH PSILI AND VARIA AND YPOGEGRAMMENIGREEK SMALL LETTER ETA WITH DASIA AND VARIA AND YPOGEGRAMMENIGREEK SMALL LETTER ETA WITH PSILI AND OXIA AND YPOGEGRAMMENIGREEK\
    \ SMALL LETTER ETA WITH DASIA AND OXIA AND YPOGEGRAMMENIGREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENIGREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENIGREEK CAPITAL LETTER ETA WITH PSILI AND PROSGEGRAMMENIGREEK CAPITA\
    \L LETTER ETA WITH DASIA AND PROSGEGRAMMENIGREEK CAPITAL LETTER ETA WITH PSILI AND VARIA AND PROSGEGRAMMENIGREEK CAPITAL LETTER ETA WITH DASIA AND VARIA AND PROSGEGRAMMENIGREEK CAPITAL LETTER ETA WITH PSILI AND OXIA AND PROSGEGRAMMENIGREEK CAPITAL LETTER E\
    \TA WITH DASIA AND OXIA AND PROSGEGRAMMENIGREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENIGREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENIGREEK SMALL LETTER OMEGA WITH PSILI AND YPOGEGRAMMENIGREEK SMALL LETTER OM\
    \EGA WITH DASIA AND YPOGEGRAMMENIGREEK SMALL LETTER OMEGA WITH PSILI AND VARIA AND YPOGEGRAMMENIGREEK SMALL LETTER OMEGA WITH DASIA AND VARIA AND YPOGEGRAMMENIGREEK SMALL LETTER OMEGA WITH PSILI AND OXIA AND YPOGEGRAMMENIGREEK SMALL LETTER OMEGA WITH DASIA\
    \ AND OXIA AND YPOGEGRAMMENIGREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENIGREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENIGREEK CAPITAL LETTER OMEGA WITH PSILI AND PROSGEGRAMMENIGREEK CAPITAL LETTER OMEGA WITH DA\
    \SIA AND PROSGEGRAMMENIGREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA AND PROSGEGRAMMENIGREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA AND PROSGEGRAMMENIGREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA AND PROSGEGRAMMENIGREEK CAPITAL LETTER OMEGA WITH DASI\
    \A AND OXIA AND PROSGEGRAMMENIGREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENIGREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENIGREEK SMALL LETTER ALPHA WITH VRACHYGREEK SMALL LETTER ALPHA WITH MACRONGREEK SMAL\
    \L LETTER ALPHA WITH VARIA AND YPOGEGRAMMENIGREEK SMALL LETTER ALPHA WITH YPOGEGRAMMENIGREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENIGREEK SMALL LETTER ALPHA WITH PERISPOMENIGREEK SMALL LETTER ALPHA WITH PERISPOMENI AND YPOGEGRAMMENIGREEK CAPITAL LET\
    \TER ALPHA WITH VRACHYGREEK CAPITAL LETTER ALPHA WITH MACRONGREEK CAPITAL LETTER ALPHA WITH VARIAGREEK CAPITAL LETTER ALPHA WITH OXIAGREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENIGREEK KORONISGREEK PROSGEGRAMMENIGREEK PSILIGREEK PERISPOMENIGREEK DIALYTIKA \
    \AND PERISPOMENIGREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENIGREEK SMALL LETTER ETA WITH YPOGEGRAMMENIGREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENIGREEK SMALL LETTER ETA WITH PERISPOMENIGREEK SMALL LETTER ETA WITH PERISPOMENI AND YPOGEGRAMMENIGR\
    \EEK CAPITAL LETTER EPSILON WITH VARIAGREEK CAPITAL LETTER EPSILON WITH OXIAGREEK CAPITAL LETTER ETA WITH VARIAGREEK CAPITAL LETTER ETA WITH OXIAGREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENIGREEK PSILI AND VARIAGREEK PSILI AND OXIAGREEK PSILI AND PERISPOMEN\
    \IGREEK SMALL LETTER IOTA WITH VRACHYGREEK SMALL LETTER IOTA WITH MACRONGREEK SMALL LETTER IOTA WITH DIALYTIKA AND VARIAGREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIAGREEK SMALL LETTER IOTA WITH PERISPOMENIGREEK SMALL LETTER IOTA WITH DIALYTIKA AND PERISP\
    \OMENIGREEK CAPITAL LETTER IOTA WITH VRACHYGREEK CAPITAL LETTER IOTA WITH MACRONGREEK CAPITAL LETTER IOTA WITH VARIAGREEK CAPITAL LETTER IOTA WITH OXIAGREEK DASIA AND VARIAGREEK DASIA AND OXIAGREEK DASIA AND PERISPOMENIGREEK SMALL LETTER UPSILON WITH VRACH\
    \YGREEK SMALL LETTER UPSILON WITH MACRONGREEK SMALL LETTER UPSILON WITH DIALYTIKA AND VARIAGREEK SMALL LETTER UPSILON WITH DIALYTIKA AND OXIAGREEK SMALL LETTER RHO WITH PSILIGREEK SMALL LETTER RHO WITH DASIAGREEK SMALL LETTER UPSILON WITH PERISPOMENIGREEK \
    \SMALL LETTER UPSILON WITH DIALYTIKA AND PERISPOMENIGREEK CAPITAL LETTER UPSILON WITH VRACHYGREEK CAPITAL LETTER UPSILON WITH MACRONGREEK CAPITAL LETTER UPSILON WITH VARIAGREEK CAPITAL LETTER UPSILON WITH OXIAGREEK CAPITAL LETTER RHO WITH DASIAGREEK DIALYT\
    \IKA AND VARIAGREEK DIALYTIKA AND OXIAGREEK VARIAGREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENIGREEK SMALL LETTER OMEGA WITH YPOGEGRAMMENIGREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENIGREEK SMALL LETTER OMEGA WITH PERISPOMENIGREEK SMALL LETTER\
    \ OMEGA WITH PERISPOMENI AND YPOGEGRAMMENIGREEK CAPITAL LETTER OMICRON WITH VARIAGREEK CAPITAL LETTER OMICRON WITH OXIAGREEK CAPITAL LETTER OMEGA WITH VARIAGREEK CAPITAL LETTER OMEGA WITH OXIAGREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENIGREEK OXIAGREEK DA\
    \SIAEN QUADEM QUADEN SPACEEM SPACETHREE-PER-EM SPACEFOUR-PER-EM SPACESIX-PER-EM SPACEFIGURE SPACEPUNCTUATION SPACETHIN SPACEHAIR SPACEZERO WIDTH SPACEZERO WIDTH NON-JOINERZERO WIDTH JOINERLEFT-TO-RIGHT MARKRIGHT-TO-LEFT MARKHYPHENNON-BREAKING HYPHENFIGURE \
    \DASHEN DASHEM DASHHORIZONTAL BARDOUBLE VERTICAL LINEDOUBLE LOW LINELEFT SINGLE QUOTATION MARKRIGHT SINGLE QUOTATION MARKSINGLE LOW-9 QUOTATION MARKSINGLE HIGH-REVERSED-9 QUOTATION MARKLEFT DOUBLE QUOTATION MARKRIGHT DOUBLE QUOTATION MARKDOUBLE LOW-9 QUOTA\
    \TION MARKDOUBLE HIGH-REVERSED-9 QUOTATION MARKDAGGERDOUBLE DAGGERBULLETTRIANGULAR BULLETONE DOT LEADERTWO DOT LEADERHORIZONTAL ELLIPSISHYPHENATION POINTLINE SEPARATORPARAGRAPH SEPARATORLEFT-TO-RIGHT EMBEDDINGRIGHT-TO-LEFT EMBEDDINGPOP DIRECTIONAL FORMATTI\
    \NGLEFT-TO-RIGHT OVERRIDERIGHT-TO-LEFT OVERRIDENARROW NO-BREAK SPACEPER MILLE SIGNPER TEN THOUSAND SIGNPRIMEDOUBLE PRIMETRIPLE PRIMEREVERSED PRIMEREVERSED DOUBLE PRIMEREVERSED TRIPLE PRIMECARETSINGLE LEFT-POINTING ANGLE QUOTATION MARKSINGLE RIGHT-POINTING \
    \ANGLE QUOTATION MARKREFERENCE MARKDOUBLE EXCLAMATION MARKINTERROBANGOVERLINEUNDERTIECHARACTER TIECARET INSERTION POINTASTERISMHYPHEN BULLETFRACTION SLASHLEFT SQUARE BRACKET WITH QUILLRIGHT SQUARE BRACKET WITH QUILLDOUBLE QUESTION MARKQUESTION EXCLAMATION \
    \MARKEXCLAMATION QUESTION MARKTIRONIAN SIGN ETREVERSED PILCROW SIGNBLACK LEFTWARDS BULLETBLACK RIGHTWARDS BULLETLOW ASTERISKREVERSED SEMICOLONCLOSE UPTWO ASTERISKS ALIGNED VERTICALLYCOMMERCIAL MINUS SIGNSWUNG DASHINVERTED UNDERTIEFLOWER PUNCTUATION MARKTHR\
    \EE DOT PUNCTUATIONQUADRUPLE PRIMEFOUR DOT PUNCTUATIONFIVE DOT PUNCTUATIONTWO DOT PUNCTUATIONFOUR DOT MARKDOTTED CROSSTRICOLONVERTICAL FOUR DOTSMEDIUM MATHEMATICAL SPACEWORD JOINERFUNCTION APPLICATIONINVISIBLE TIMESINVISIBLE SEPARATORINVISIBLE PLUSLEFT-TO-\
    \RIGHT ISOLATERIGHT-TO-LEFT ISOLATEFIRST STRONG ISOLATEPOP DIRECTIONAL ISOLATEINHIBIT SYMMETRIC SWAPPINGACTIVATE SYMMETRIC SWAPPINGINHIBIT ARABIC FORM SHAPINGACTIVATE ARABIC FORM SHAPINGNATIONAL DIGIT SHAPESNOMINAL DIGIT SHAPESSUPERSCRIPT ZEROSUPERSCRIPT L\
    \ATIN SMALL LETTER ISUPERSCRIPT FOURSUPERSCRIPT FIVESUPERSCRIPT SIXSUPERSCRIPT SEVENSUPERSCRIPT EIGHTSUPERSCRIPT NINESUPERSCRIPT PLUS SIGNSUPERSCRIPT MINUSSUPERSCRIPT EQUALS SIGNSUPERSCRIPT LEFT PARENTHESISSUPERSCRIPT RIGHT PARENTHESISSUPERSCRIPT LATIN SMA\
    \LL LETTER NSUBSCRIPT ZEROSUBSCRIPT ONESUBSCRIPT TWOSUBSCRIPT THREESUBSCRIPT FOURSUBSCRIPT FIVESUBSCRIPT SIXSUBSCRIPT SEVENSUBSCRIPT EIGHTSUBSCRIPT NINESUBSCRIPT PLUS SIGNSUBSCRIPT MINUSSUBSCRIPT EQUALS SIGNSUBSCRIPT LEFT PARENTHESISSUBSCRIPT RIGHT PARENTH\
    \ESISLATIN SUBSCRIPT SMALL LETTER ALATIN SUBSCRIPT SMALL LETTER ELATIN SUBSCRIPT SMALL LETTER OLATIN SUBSCRIPT SMALL LETTER XLATIN SUBSCRIPT SMALL LETTER SCHWALATIN SUBSCRIPT SMALL LETTER HLATIN SUBSCRIPT SMALL LETTER KLATIN SUBSCRIPT SMALL LETTER LLATIN S\
    \UBSCRIPT SMALL LETTER MLATIN SUBSCRIPT SMALL LETTER NLATIN SUBSCRIPT SMALL LETTER PLATIN SUBSCRIPT SMALL LETTER SLATIN SUBSCRIPT SMALL LETTER TEURO-CURRENCY SIGNCOLON SIGNCRUZEIRO SIGNFRENCH FRANC SIGNLIRA SIGNMILL SIGNNAIRA SIGNPESETA SIGNRUPEE SIGNWON S\
    \IGNNEW SHEQEL SIGNDONG SIGNEURO SIGNKIP SIGNTUGRIK SIGNDRACHMA SIGNGERMAN PENNY SIGNPESO SIGNGUARANI SIGNAUSTRAL SIGNHRYVNIA SIGNCEDI SIGNLIVRE TOURNOIS SIGNSPESMILO SIGNTENGE SIGNINDIAN RUPEE SIGNTURKISH LIRA SIGNNORDIC MARK SIGNMANAT SIGNRUBLE SIGNLARI \
    \SIGNBITCOIN SIGNSOM SIGNCOMBINING LEFT HARPOON ABOVECOMBINING RIGHT HARPOON ABOVECOMBINING LONG VERTICAL LINE OVERLAYCOMBINING SHORT VERTICAL LINE OVERLAYCOMBINING ANTICLOCKWISE ARROW ABOVECOMBINING CLOCKWISE ARROW ABOVECOMBINING LEFT ARROW ABOVECOMBINING\
    \ RIGHT ARROW ABOVECOMBINING RING OVERLAYCOMBINING CLOCKWISE RING OVERLAYCOMBINING ANTICLOCKWISE RING OVERLAYCOMBINING THREE DOTS ABOVECOMBINING FOUR DOTS ABOVECOMBINING ENCLOSING CIRCLECOMBINING ENCLOSING SQUARECOMBINING ENCLOSING DIAMONDCOMBINING ENCLOSI\
    \NG CIRCLE BACKSLASHCOMBINING LEFT RIGHT ARROW ABOVECOMBINING ENCLOSING SCREENCOMBINING ENCLOSING KEYCAPCOMBINING ENCLOSING UPWARD POINTING TRIANGLECOMBINING REVERSE SOLIDUS OVERLAYCOMBINING DOUBLE VERTICAL STROKE OVERLAYCOMBINING ANNUITY SYMBOLCOMBINING T\
    \RIPLE UNDERDOTCOMBINING WIDE BRIDGE ABOVECOMBINING LEFTWARDS ARROW OVERLAYCOMBINING LONG DOUBLE SOLIDUS OVERLAYCOMBINING RIGHTWARDS HARPOON WITH BARB DOWNWARDSCOMBINING LEFTWARDS HARPOON WITH BARB DOWNWARDSCOMBINING LEFT ARROW BELOWCOMBINING RIGHT ARROW B\
    \ELOWCOMBINING ASTERISK ABOVEACCOUNT OFADDRESSED TO THE SUBJECTDOUBLE-STRUCK CAPITAL CDEGREE CELSIUSCENTRE LINE SYMBOLCARE OFCADA UNAEULER CONSTANTSCRUPLEDEGREE FAHRENHEITSCRIPT SMALL GSCRIPT CAPITAL HBLACK-LETTER CAPITAL HDOUBLE-STRUCK CAPITAL HPLANCK CON\
    \STANTPLANCK CONSTANT OVER TWO PISCRIPT CAPITAL IBLACK-LETTER CAPITAL ISCRIPT CAPITAL LSCRIPT SMALL LL B BAR SYMBOLDOUBLE-STRUCK CAPITAL NNUMERO SIGNSOUND RECORDING COPYRIGHTSCRIPT CAPITAL PDOUBLE-STRUCK CAPITAL PDOUBLE-STRUCK CAPITAL QSCRIPT CAPITAL RBLAC\
    \K-LETTER CAPITAL RDOUBLE-STRUCK CAPITAL RPRESCRIPTION TAKERESPONSESERVICE MARKTELEPHONE SIGNTRADE MARK SIGNVERSICLEDOUBLE-STRUCK CAPITAL ZOUNCE SIGNOHM SIGNINVERTED OHM SIGNBLACK-LETTER CAPITAL ZTURNED GREEK SMALL LETTER IOTAKELVIN SIGNANGSTROM SIGNSCRIPT\
    \ CAPITAL BBLACK-LETTER CAPITAL CESTIMATED SYMBOLSCRIPT SMALL ESCRIPT CAPITAL ESCRIPT CAPITAL FTURNED CAPITAL FSCRIPT CAPITAL MSCRIPT SMALL OALEF SYMBOLBET SYMBOLGIMEL SYMBOLDALET SYMBOLINFORMATION SOURCEROTATED CAPITAL QFACSIMILE SIGNDOUBLE-STRUCK SMALL P\
    \IDOUBLE-STRUCK SMALL GAMMADOUBLE-STRUCK CAPITAL GAMMADOUBLE-STRUCK CAPITAL PIDOUBLE-STRUCK N-ARY SUMMATIONTURNED SANS-SERIF CAPITAL GTURNED SANS-SERIF CAPITAL LREVERSED SANS-SERIF CAPITAL LTURNED SANS-SERIF CAPITAL YDOUBLE-STRUCK ITALIC CAPITAL DDOUBLE-ST\
    \RUCK ITALIC SMALL DDOUBLE-STRUCK ITALIC SMALL EDOUBLE-STRUCK ITALIC SMALL IDOUBLE-STRUCK ITALIC SMALL JPROPERTY LINETURNED AMPERSANDPER SIGNAKTIESELSKABTURNED SMALL FSYMBOL FOR SAMARITAN SOURCEVULGAR FRACTION ONE SEVENTHVULGAR FRACTION ONE NINTHVULGAR FRA\
    \CTION ONE TENTHVULGAR FRACTION ONE THIRDVULGAR FRACTION TWO THIRDSVULGAR FRACTION ONE FIFTHVULGAR FRACTION TWO FIFTHSVULGAR FRACTION THREE FIFTHSVULGAR FRACTION FOUR FIFTHSVULGAR FRACTION ONE SIXTHVULGAR FRACTION FIVE SIXTHSVULGAR FRACTION ONE EIGHTHVULGA\
    \R FRACTION THREE EIGHTHSVULGAR FRACTION FIVE EIGHTHSVULGAR FRACTION SEVEN EIGHTHSFRACTION NUMERATOR ONEROMAN NUMERAL ONEROMAN NUMERAL TWOROMAN NUMERAL THREEROMAN NUMERAL FOURROMAN NUMERAL FIVEROMAN NUMERAL SIXROMAN NUMERAL SEVENROMAN NUMERAL EIGHTROMAN NU\
    \MERAL NINEROMAN NUMERAL TENROMAN NUMERAL ELEVENROMAN NUMERAL TWELVEROMAN NUMERAL FIFTYROMAN NUMERAL ONE HUNDREDROMAN NUMERAL FIVE HUNDREDROMAN NUMERAL ONE THOUSANDSMALL ROMAN NUMERAL ONESMALL ROMAN NUMERAL TWOSMALL ROMAN NUMERAL THREESMALL ROMAN NUMERAL F\
    \OURSMALL ROMAN NUMERAL FIVESMALL ROMAN NUMERAL SIXSMALL ROMAN NUMERAL SEVENSMALL ROMAN NUMERAL EIGHTSMALL ROMAN NUMERAL NINESMALL ROMAN NUMERAL TENSMALL ROMAN NUMERAL ELEVENSMALL ROMAN NUMERAL TWELVESMALL ROMAN NUMERAL FIFTYSMALL ROMAN NUMERAL ONE HUNDRED\
    \SMALL ROMAN NUMERAL FIVE HUNDREDSMALL ROMAN NUMERAL ONE THOUSANDROMAN NUMERAL ONE THOUSAND C DROMAN NUMERAL FIVE THOUSANDROMAN NUMERAL TEN THOUSANDROMAN NUMERAL REVERSED ONE HUNDREDLATIN SMALL LETTER REVERSED CROMAN NUMERAL SIX LATE FORMROMAN NUMERAL FIFT\
    \Y EARLY FORMROMAN NUMERAL FIFTY THOUSANDROMAN NUMERAL ONE HUNDRED THOUSANDVULGAR FRACTION ZERO THIRDSTURNED DIGIT TWOTURNED DIGIT THREELEFTWARDS ARROWUPWARDS ARROWRIGHTWARDS ARROWDOWNWARDS ARROWLEFT RIGHT ARROWUP DOWN ARROWNORTH WEST ARROWNORTH EAST ARROW\
    \SOUTH EAST ARROWSOUTH WEST ARROWLEFTWARDS ARROW WITH STROKERIGHTWARDS ARROW WITH STROKELEFTWARDS WAVE ARROWRIGHTWARDS WAVE ARROWLEFTWARDS TWO HEADED ARROWUPWARDS TWO HEADED ARROWRIGHTWARDS TWO HEADED ARROWDOWNWARDS TWO HEADED ARROWLEFTWARDS ARROW WITH TAI\
    \LRIGHTWARDS ARROW WITH TAILLEFTWARDS ARROW FROM BARUPWARDS ARROW FROM BARRIGHTWARDS ARROW FROM BARDOWNWARDS ARROW FROM BARUP DOWN ARROW WITH BASELEFTWARDS ARROW WITH HOOKRIGHTWARDS ARROW WITH HOOKLEFTWARDS ARROW WITH LOOPRIGHTWARDS ARROW WITH LOOPLEFT RIG\
    \HT WAVE ARROWLEFT RIGHT ARROW WITH STROKEDOWNWARDS ZIGZAG ARROWUPWARDS ARROW WITH TIP LEFTWARDSUPWARDS ARROW WITH TIP RIGHTWARDSDOWNWARDS ARROW WITH TIP LEFTWARDSDOWNWARDS ARROW WITH TIP RIGHTWARDSRIGHTWARDS ARROW WITH CORNER DOWNWARDSDOWNWARDS ARROW WITH\
    \ CORNER LEFTWARDSANTICLOCKWISE TOP SEMICIRCLE ARROWCLOCKWISE TOP SEMICIRCLE ARROWNORTH WEST ARROW TO LONG BARLEFTWARDS ARROW TO BAR OVER RIGHTWARDS ARROW TO BARANTICLOCKWISE OPEN CIRCLE ARROWCLOCKWISE OPEN CIRCLE ARROWLEFTWARDS HARPOON WITH BARB UPWARDSLE\
    \FTWARDS HARPOON WITH BARB DOWNWARDSUPWARDS HARPOON WITH BARB RIGHTWARDSUPWARDS HARPOON WITH BARB LEFTWARDSRIGHTWARDS HARPOON WITH BARB UPWARDSRIGHTWARDS HARPOON WITH BARB DOWNWARDSDOWNWARDS HARPOON WITH BARB RIGHTWARDSDOWNWARDS HARPOON WITH BARB LEFTWARDS\
    \RIGHTWARDS ARROW OVER LEFTWARDS ARROWUPWARDS ARROW LEFTWARDS OF DOWNWARDS ARROWLEFTWARDS ARROW OVER RIGHTWARDS ARROWLEFTWARDS PAIRED ARROWSUPWARDS PAIRED ARROWSRIGHTWARDS PAIRED ARROWSDOWNWARDS PAIRED ARROWSLEFTWARDS HARPOON OVER RIGHTWARDS HARPOONRIGHTWA\
    \RDS HARPOON OVER LEFTWARDS HARPOONLEFTWARDS DOUBLE ARROW WITH STROKELEFT RIGHT DOUBLE ARROW WITH STROKERIGHTWARDS DOUBLE ARROW WITH STROKELEFTWARDS DOUBLE ARROWUPWARDS DOUBLE ARROWRIGHTWARDS DOUBLE ARROWDOWNWARDS DOUBLE ARROWLEFT RIGHT DOUBLE ARROWUP DOWN\
    \ DOUBLE ARROWNORTH WEST DOUBLE ARROWNORTH EAST DOUBLE ARROWSOUTH EAST DOUBLE ARROWSOUTH WEST DOUBLE ARROWLEFTWARDS TRIPLE ARROWRIGHTWARDS TRIPLE ARROWLEFTWARDS SQUIGGLE ARROWRIGHTWARDS SQUIGGLE ARROWUPWARDS ARROW WITH DOUBLE STROKEDOWNWARDS ARROW WITH DOU\
    \BLE STROKELEFTWARDS DASHED ARROWUPWARDS DASHED ARROWRIGHTWARDS DASHED ARROWDOWNWARDS DASHED ARROWLEFTWARDS ARROW TO BARRIGHTWARDS ARROW TO BARLEFTWARDS WHITE ARROWUPWARDS WHITE ARROWRIGHTWARDS WHITE ARROWDOWNWARDS WHITE ARROWUPWARDS WHITE ARROW FROM BARUP\
    \WARDS WHITE ARROW ON PEDESTALUPWARDS WHITE ARROW ON PEDESTAL WITH HORIZONTAL BARUPWARDS WHITE ARROW ON PEDESTAL WITH VERTICAL BARUPWARDS WHITE DOUBLE ARROWUPWARDS WHITE DOUBLE ARROW ON PEDESTALRIGHTWARDS WHITE ARROW FROM WALLNORTH WEST ARROW TO CORNERSOUT\
    \H EAST ARROW TO CORNERUP DOWN WHITE ARROWRIGHT ARROW WITH SMALL CIRCLEDOWNWARDS ARROW LEFTWARDS OF UPWARDS ARROWTHREE RIGHTWARDS ARROWSLEFTWARDS ARROW WITH VERTICAL STROKERIGHTWARDS ARROW WITH VERTICAL STROKELEFT RIGHT ARROW WITH VERTICAL STROKELEFTWARDS \
    \ARROW WITH DOUBLE VERTICAL STROKERIGHTWARDS ARROW WITH DOUBLE VERTICAL STROKELEFT RIGHT ARROW WITH DOUBLE VERTICAL STROKELEFTWARDS OPEN-HEADED ARROWRIGHTWARDS OPEN-HEADED ARROWLEFT RIGHT OPEN-HEADED ARROWFOR ALLCOMPLEMENTPARTIAL DIFFERENTIALTHERE EXISTSTH\
    \ERE DOES NOT EXISTEMPTY SETINCREMENTNABLAELEMENT OFNOT AN ELEMENT OFSMALL ELEMENT OFCONTAINS AS MEMBERDOES NOT CONTAIN AS MEMBERSMALL CONTAINS AS MEMBEREND OF PROOFN-ARY PRODUCTN-ARY COPRODUCTN-ARY SUMMATIONMINUS SIGNMINUS-OR-PLUS SIGNDOT PLUSDIVISION SLA\
    \SHSET MINUSASTERISK OPERATORRING OPERATORBULLET OPERATORSQUARE ROOTCUBE ROOTFOURTH ROOTPROPORTIONAL TOINFINITYRIGHT ANGLEANGLEMEASURED ANGLESPHERICAL ANGLEDIVIDESDOES NOT DIVIDEPARALLEL TONOT PARALLEL TOLOGICAL ANDLOGICAL ORINTERSECTIONUNIONINTEGRALDOUBLE\
    \ INTEGRALTRIPLE INTEGRALCONTOUR INTEGRALSURFACE INTEGRALVOLUME INTEGRALCLOCKWISE INTEGRALCLOCKWISE CONTOUR INTEGRALANTICLOCKWISE CONTOUR INTEGRALTHEREFOREBECAUSERATIOPROPORTIONDOT MINUSEXCESSGEOMETRIC PROPORTIONHOMOTHETICTILDE OPERATORREVERSED TILDEINVERT\
    \ED LAZY SSINE WAVEWREATH PRODUCTNOT TILDEMINUS TILDEASYMPTOTICALLY EQUAL TONOT ASYMPTOTICALLY EQUAL TOAPPROXIMATELY EQUAL TOAPPROXIMATELY BUT NOT ACTUALLY EQUAL TONEITHER APPROXIMATELY NOR ACTUALLY EQUAL TOALMOST EQUAL TONOT ALMOST EQUAL TOALMOST EQUAL OR\
    \ EQUAL TOTRIPLE TILDEALL EQUAL TOEQUIVALENT TOGEOMETRICALLY EQUIVALENT TODIFFERENCE BETWEENAPPROACHES THE LIMITGEOMETRICALLY EQUAL TOAPPROXIMATELY EQUAL TO OR THE IMAGE OFIMAGE OF OR APPROXIMATELY EQUAL TOCOLON EQUALSEQUALS COLONRING IN EQUAL TORING EQUAL\
    \ TOCORRESPONDS TOESTIMATESEQUIANGULAR TOSTAR EQUALSDELTA EQUAL TOEQUAL TO BY DEFINITIONMEASURED BYQUESTIONED EQUAL TONOT EQUAL TOIDENTICAL TONOT IDENTICAL TOSTRICTLY EQUIVALENT TOLESS-THAN OR EQUAL TOGREATER-THAN OR EQUAL TOLESS-THAN OVER EQUAL TOGREATER-\
    \THAN OVER EQUAL TOLESS-THAN BUT NOT EQUAL TOGREATER-THAN BUT NOT EQUAL TOMUCH LESS-THANMUCH GREATER-THANBETWEENNOT EQUIVALENT TONOT LESS-THANNOT GREATER-THANNEITHER LESS-THAN NOR EQUAL TONEITHER GREATER-THAN NOR EQUAL TOLESS-THAN OR EQUIVALENT TOGREATER-T\
    \HAN OR EQUIVALENT TONEITHER LESS-THAN NOR EQUIVALENT TONEITHER GREATER-THAN NOR EQUIVALENT TOLESS-THAN OR GREATER-THANGREATER-THAN OR LESS-THANNEITHER LESS-THAN NOR GREATER-THANNEITHER GREATER-THAN NOR LESS-THANPRECEDESSUCCEEDSPRECEDES OR EQUAL TOSUCCEEDS\
    \ OR EQUAL TOPRECEDES OR EQUIVALENT TOSUCCEEDS OR EQUIVALENT TODOES NOT PRECEDEDOES NOT SUCCEEDSUBSET OFSUPERSET OFNOT A SUBSET OFNOT A SUPERSET OFSUBSET OF OR EQUAL TOSUPERSET OF OR EQUAL TONEITHER A SUBSET OF NOR EQUAL TONEITHER A SUPERSET OF NOR EQUAL T\
    \OSUBSET OF WITH NOT EQUAL TOSUPERSET OF WITH NOT EQUAL TOMULTISETMULTISET MULTIPLICATIONMULTISET UNIONSQUARE IMAGE OFSQUARE ORIGINAL OFSQUARE IMAGE OF OR EQUAL TOSQUARE ORIGINAL OF OR EQUAL TOSQUARE CAPSQUARE CUPCIRCLED PLUSCIRCLED MINUSCIRCLED TIMESCIRCL\
    \ED DIVISION SLASHCIRCLED DOT OPERATORCIRCLED RING OPERATORCIRCLED ASTERISK OPERATORCIRCLED EQUALSCIRCLED DASHSQUARED PLUSSQUARED MINUSSQUARED TIMESSQUARED DOT OPERATORRIGHT TACKLEFT TACKDOWN TACKUP TACKASSERTIONMODELSTRUEFORCESTRIPLE VERTICAL BAR RIGHT TU\
    \RNSTILEDOUBLE VERTICAL BAR DOUBLE RIGHT TURNSTILEDOES NOT PROVENOT TRUEDOES NOT FORCENEGATED DOUBLE VERTICAL BAR DOUBLE RIGHT TURNSTILEPRECEDES UNDER RELATIONSUCCEEDS UNDER RELATIONNORMAL SUBGROUP OFCONTAINS AS NORMAL SUBGROUPNORMAL SUBGROUP OF OR EQUAL T\
    \OCONTAINS AS NORMAL SUBGROUP OR EQUAL TOORIGINAL OFIMAGE OFMULTIMAPHERMITIAN CONJUGATE MATRIXINTERCALATEXORNANDNORRIGHT ANGLE WITH ARCRIGHT TRIANGLEN-ARY LOGICAL ANDN-ARY LOGICAL ORN-ARY INTERSECTIONN-ARY UNIONDIAMOND OPERATORDOT OPERATORSTAR OPERATORDIVI\
    \SION TIMESBOWTIELEFT NORMAL FACTOR SEMIDIRECT PRODUCTRIGHT NORMAL FACTOR SEMIDIRECT PRODUCTLEFT SEMIDIRECT PRODUCTRIGHT SEMIDIRECT PRODUCTREVERSED TILDE EQUALSCURLY LOGICAL ORCURLY LOGICAL ANDDOUBLE SUBSETDOUBLE SUPERSETDOUBLE INTERSECTIONDOUBLE UNIONPITC\
    \HFORKEQUAL AND PARALLEL TOLESS-THAN WITH DOTGREATER-THAN WITH DOTVERY MUCH LESS-THANVERY MUCH GREATER-THANLESS-THAN EQUAL TO OR GREATER-THANGREATER-THAN EQUAL TO OR LESS-THANEQUAL TO OR LESS-THANEQUAL TO OR GREATER-THANEQUAL TO OR PRECEDESEQUAL TO OR SUCC\
    \EEDSDOES NOT PRECEDE OR EQUALDOES NOT SUCCEED OR EQUALNOT SQUARE IMAGE OF OR EQUAL TONOT SQUARE ORIGINAL OF OR EQUAL TOSQUARE IMAGE OF OR NOT EQUAL TOSQUARE ORIGINAL OF OR NOT EQUAL TOLESS-THAN BUT NOT EQUIVALENT TOGREATER-THAN BUT NOT EQUIVALENT TOPRECED\
    \ES BUT NOT EQUIVALENT TOSUCCEEDS BUT NOT EQUIVALENT TONOT NORMAL SUBGROUP OFDOES NOT CONTAIN AS NORMAL SUBGROUPNOT NORMAL SUBGROUP OF OR EQUAL TODOES NOT CONTAIN AS NORMAL SUBGROUP OR EQUALVERTICAL ELLIPSISMIDLINE HORIZONTAL ELLIPSISUP RIGHT DIAGONAL ELLI\
    \PSISDOWN RIGHT DIAGONAL ELLIPSISELEMENT OF WITH LONG HORIZONTAL STROKEELEMENT OF WITH VERTICAL BAR AT END OF HORIZONTAL STROKESMALL ELEMENT OF WITH VERTICAL BAR AT END OF HORIZONTAL STROKEELEMENT OF WITH DOT ABOVEELEMENT OF WITH OVERBARSMALL ELEMENT OF WI\
    \TH OVERBARELEMENT OF WITH UNDERBARELEMENT OF WITH TWO HORIZONTAL STROKESCONTAINS WITH LONG HORIZONTAL STROKECONTAINS WITH VERTICAL BAR AT END OF HORIZONTAL STROKESMALL CONTAINS WITH VERTICAL BAR AT END OF HORIZONTAL STROKECONTAINS WITH OVERBARSMALL CONTAI\
    \NS WITH OVERBARZ NOTATION BAG MEMBERSHIPDIAMETER SIGNELECTRIC ARROWHOUSEUP ARROWHEADDOWN ARROWHEADPROJECTIVEPERSPECTIVEWAVY LINELEFT CEILINGRIGHT CEILINGLEFT FLOORRIGHT FLOORBOTTOM RIGHT CROPBOTTOM LEFT CROPTOP RIGHT CROPTOP LEFT CROPREVERSED NOT SIGNSQUA\
    \RE LOZENGEARCSEGMENTSECTORTELEPHONE RECORDERPOSITION INDICATORVIEWDATA SQUAREPLACE OF INTEREST SIGNTURNED NOT SIGNWATCHHOURGLASSTOP LEFT CORNERTOP RIGHT CORNERBOTTOM LEFT CORNERBOTTOM RIGHT CORNERTOP HALF INTEGRALBOTTOM HALF INTEGRALFROWNSMILEUP ARROWHEAD\
    \ BETWEEN TWO HORIZONTAL BARSOPTION KEYERASE TO THE RIGHTX IN A RECTANGLE BOXKEYBOARDLEFT-POINTING ANGLE BRACKETRIGHT-POINTING ANGLE BRACKETERASE TO THE LEFTBENZENE RINGCYLINDRICITYALL AROUND-PROFILESYMMETRYTOTAL RUNOUTDIMENSION ORIGINCONICAL TAPERSLOPECOU\
    \NTERBORECOUNTERSINKAPL FUNCTIONAL SYMBOL I-BEAMAPL FUNCTIONAL SYMBOL SQUISH QUADAPL FUNCTIONAL SYMBOL QUAD EQUALAPL FUNCTIONAL SYMBOL QUAD DIVIDEAPL FUNCTIONAL SYMBOL QUAD DIAMONDAPL FUNCTIONAL SYMBOL QUAD JOTAPL FUNCTIONAL SYMBOL QUAD CIRCLEAPL FUNCTIONA\
    \L SYMBOL CIRCLE STILEAPL FUNCTIONAL SYMBOL CIRCLE JOTAPL FUNCTIONAL SYMBOL SLASH BARAPL FUNCTIONAL SYMBOL BACKSLASH BARAPL FUNCTIONAL SYMBOL QUAD SLASHAPL FUNCTIONAL SYMBOL QUAD BACKSLASHAPL FUNCTIONAL SYMBOL QUAD LESS-THANAPL FUNCTIONAL SYMBOL QUAD GREAT\
    \ER-THANAPL FUNCTIONAL SYMBOL LEFTWARDS VANEAPL FUNCTIONAL SYMBOL RIGHTWARDS VANEAPL FUNCTIONAL SYMBOL QUAD LEFTWARDS ARROWAPL FUNCTIONAL SYMBOL QUAD RIGHTWARDS ARROWAPL FUNCTIONAL SYMBOL CIRCLE BACKSLASHAPL FUNCTIONAL SYMBOL DOWN TACK UNDERBARAPL FUNCTION\
    \AL SYMBOL DELTA STILEAPL FUNCTIONAL SYMBOL QUAD DOWN CARETAPL FUNCTIONAL SYMBOL QUAD DELTAAPL FUNCTIONAL SYMBOL DOWN TACK JOTAPL FUNCTIONAL SYMBOL UPWARDS VANEAPL FUNCTIONAL SYMBOL QUAD UPWARDS ARROWAPL FUNCTIONAL SYMBOL UP TACK OVERBARAPL FUNCTIONAL SYMB\
    \OL DEL STILEAPL FUNCTIONAL SYMBOL QUAD UP CARETAPL FUNCTIONAL SYMBOL QUAD DELAPL FUNCTIONAL SYMBOL UP TACK JOTAPL FUNCTIONAL SYMBOL DOWNWARDS VANEAPL FUNCTIONAL SYMBOL QUAD DOWNWARDS ARROWAPL FUNCTIONAL SYMBOL QUOTE UNDERBARAPL FUNCTIONAL SYMBOL DELTA UND\
    \ERBARAPL FUNCTIONAL SYMBOL DIAMOND UNDERBARAPL FUNCTIONAL SYMBOL JOT UNDERBARAPL FUNCTIONAL SYMBOL CIRCLE UNDERBARAPL FUNCTIONAL SYMBOL UP SHOE JOTAPL FUNCTIONAL SYMBOL QUOTE QUADAPL FUNCTIONAL SYMBOL CIRCLE STARAPL FUNCTIONAL SYMBOL QUAD COLONAPL FUNCTIO\
    \NAL SYMBOL UP TACK DIAERESISAPL FUNCTIONAL SYMBOL DEL DIAERESISAPL FUNCTIONAL SYMBOL STAR DIAERESISAPL FUNCTIONAL SYMBOL JOT DIAERESISAPL FUNCTIONAL SYMBOL CIRCLE DIAERESISAPL FUNCTIONAL SYMBOL DOWN SHOE STILEAPL FUNCTIONAL SYMBOL LEFT SHOE STILEAPL FUNCT\
    \IONAL SYMBOL TILDE DIAERESISAPL FUNCTIONAL SYMBOL GREATER-THAN DIAERESISAPL FUNCTIONAL SYMBOL COMMA BARAPL FUNCTIONAL SYMBOL DEL TILDEAPL FUNCTIONAL SYMBOL ZILDEAPL FUNCTIONAL SYMBOL STILE TILDEAPL FUNCTIONAL SYMBOL SEMICOLON UNDERBARAPL FUNCTIONAL SYMBOL\
    \ QUAD NOT EQUALAPL FUNCTIONAL SYMBOL QUAD QUESTIONAPL FUNCTIONAL SYMBOL DOWN CARET TILDEAPL FUNCTIONAL SYMBOL UP CARET TILDEAPL FUNCTIONAL SYMBOL IOTAAPL FUNCTIONAL SYMBOL RHOAPL FUNCTIONAL SYMBOL OMEGAAPL FUNCTIONAL SYMBOL ALPHA UNDERBARAPL FUNCTIONAL SY\
    \MBOL EPSILON UNDERBARAPL FUNCTIONAL SYMBOL IOTA UNDERBARAPL FUNCTIONAL SYMBOL OMEGA UNDERBARAPL FUNCTIONAL SYMBOL ALPHANOT CHECK MARKRIGHT ANGLE WITH DOWNWARDS ZIGZAG ARROWSHOULDERED OPEN BOXBELL SYMBOLVERTICAL LINE WITH MIDDLE DOTINSERTION SYMBOLCONTINUO\
    \US UNDERLINE SYMBOLDISCONTINUOUS UNDERLINE SYMBOLEMPHASIS SYMBOLCOMPOSITION SYMBOLWHITE SQUARE WITH CENTRE VERTICAL LINEENTER SYMBOLALTERNATIVE KEY SYMBOLHELM SYMBOLCIRCLED HORIZONTAL BAR WITH NOTCHCIRCLED TRIANGLE DOWNBROKEN CIRCLE WITH NORTHWEST ARROWUN\
    \DO SYMBOLMONOSTABLE SYMBOLHYSTERESIS SYMBOLOPEN-CIRCUIT-OUTPUT H-TYPE SYMBOLOPEN-CIRCUIT-OUTPUT L-TYPE SYMBOLPASSIVE-PULL-DOWN-OUTPUT SYMBOLPASSIVE-PULL-UP-OUTPUT SYMBOLDIRECT CURRENT SYMBOL FORM TWOSOFTWARE-FUNCTION SYMBOLAPL FUNCTIONAL SYMBOL QUADDECIMA\
    \L SEPARATOR KEY SYMBOLPREVIOUS PAGENEXT PAGEPRINT SCREEN SYMBOLCLEAR SCREEN SYMBOLLEFT PARENTHESIS UPPER HOOKLEFT PARENTHESIS EXTENSIONLEFT PARENTHESIS LOWER HOOKRIGHT PARENTHESIS UPPER HOOKRIGHT PARENTHESIS EXTENSIONRIGHT PARENTHESIS LOWER HOOKLEFT SQUAR\
    \E BRACKET UPPER CORNERLEFT SQUARE BRACKET EXTENSIONLEFT SQUARE BRACKET LOWER CORNERRIGHT SQUARE BRACKET UPPER CORNERRIGHT SQUARE BRACKET EXTENSIONRIGHT SQUARE BRACKET LOWER CORNERLEFT CURLY BRACKET UPPER HOOKLEFT CURLY BRACKET MIDDLE PIECELEFT CURLY BRACK\
    \ET LOWER HOOKCURLY BRACKET EXTENSIONRIGHT CURLY BRACKET UPPER HOOKRIGHT CURLY BRACKET MIDDLE PIECERIGHT CURLY BRACKET LOWER HOOKINTEGRAL EXTENSIONHORIZONTAL LINE EXTENSIONUPPER LEFT OR LOWER RIGHT CURLY BRACKET SECTIONUPPER RIGHT OR LOWER LEFT CURLY BRACK\
    \ET SECTIONSUMMATION TOPSUMMATION BOTTOMTOP SQUARE BRACKETBOTTOM SQUARE BRACKETBOTTOM SQUARE BRACKET OVER TOP SQUARE BRACKETRADICAL SYMBOL BOTTOMLEFT VERTICAL BOX LINERIGHT VERTICAL BOX LINEHORIZONTAL SCAN LINE-1HORIZONTAL SCAN LINE-3HORIZONTAL SCAN LINE-7\
    \HORIZONTAL SCAN LINE-9DENTISTRY SYMBOL LIGHT VERTICAL AND TOP RIGHTDENTISTRY SYMBOL LIGHT VERTICAL AND BOTTOM RIGHTDENTISTRY SYMBOL LIGHT VERTICAL WITH CIRCLEDENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL WITH CIRCLEDENTISTRY SYMBOL LIGHT UP AND HORIZONTAL WI\
    \TH CIRCLEDENTISTRY SYMBOL LIGHT VERTICAL WITH TRIANGLEDENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL WITH TRIANGLEDENTISTRY SYMBOL LIGHT UP AND HORIZONTAL WITH TRIANGLEDENTISTRY SYMBOL LIGHT VERTICAL AND WAVEDENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL WITH WAV\
    \EDENTISTRY SYMBOL LIGHT UP AND HORIZONTAL WITH WAVEDENTISTRY SYMBOL LIGHT DOWN AND HORIZONTALDENTISTRY SYMBOL LIGHT UP AND HORIZONTALDENTISTRY SYMBOL LIGHT VERTICAL AND TOP LEFTDENTISTRY SYMBOL LIGHT VERTICAL AND BOTTOM LEFTSQUARE FOOTRETURN SYMBOLEJECT S\
    \YMBOLVERTICAL LINE EXTENSIONMETRICAL BREVEMETRICAL LONG OVER SHORTMETRICAL SHORT OVER LONGMETRICAL LONG OVER TWO SHORTSMETRICAL TWO SHORTS OVER LONGMETRICAL TWO SHORTS JOINEDMETRICAL TRISEMEMETRICAL TETRASEMEMETRICAL PENTASEMEEARTH GROUNDFUSETOP PARENTHES\
    \ISBOTTOM PARENTHESISTOP CURLY BRACKETBOTTOM CURLY BRACKETTOP TORTOISE SHELL BRACKETBOTTOM TORTOISE SHELL BRACKETWHITE TRAPEZIUMBENZENE RING WITH CIRCLESTRAIGHTNESSFLATNESSAC CURRENTELECTRICAL INTERSECTIONDECIMAL EXPONENT SYMBOLBLACK RIGHT-POINTING DOUBLE \
    \TRIANGLEBLACK LEFT-POINTING DOUBLE TRIANGLEBLACK UP-POINTING DOUBLE TRIANGLEBLACK DOWN-POINTING DOUBLE TRIANGLEBLACK RIGHT-POINTING DOUBLE TRIANGLE WITH VERTICAL BARBLACK LEFT-POINTING DOUBLE TRIANGLE WITH VERTICAL BARBLACK RIGHT-POINTING TRIANGLE WITH DO\
    \UBLE VERTICAL BARALARM CLOCKSTOPWATCHTIMER CLOCKHOURGLASS WITH FLOWING SANDBLACK MEDIUM LEFT-POINTING TRIANGLEBLACK MEDIUM RIGHT-POINTING TRIANGLEBLACK MEDIUM UP-POINTING TRIANGLEBLACK MEDIUM DOWN-POINTING TRIANGLEDOUBLE VERTICAL BARBLACK SQUARE FOR STOPB\
    \LACK CIRCLE FOR RECORDPOWER SYMBOLPOWER ON-OFF SYMBOLPOWER ON SYMBOLPOWER SLEEP SYMBOLOBSERVER EYE SYMBOLSYMBOL FOR NULLSYMBOL FOR START OF HEADINGSYMBOL FOR START OF TEXTSYMBOL FOR END OF TEXTSYMBOL FOR END OF TRANSMISSIONSYMBOL FOR ENQUIRYSYMBOL FOR ACK\
    \NOWLEDGESYMBOL FOR BELLSYMBOL FOR BACKSPACESYMBOL FOR HORIZONTAL TABULATIONSYMBOL FOR LINE FEEDSYMBOL FOR VERTICAL TABULATIONSYMBOL FOR FORM FEEDSYMBOL FOR CARRIAGE RETURNSYMBOL FOR SHIFT OUTSYMBOL FOR SHIFT INSYMBOL FOR DATA LINK ESCAPESYMBOL FOR DEVICE \
    \CONTROL ONESYMBOL FOR DEVICE CONTROL TWOSYMBOL FOR DEVICE CONTROL THREESYMBOL FOR DEVICE CONTROL FOURSYMBOL FOR NEGATIVE ACKNOWLEDGESYMBOL FOR SYNCHRONOUS IDLESYMBOL FOR END OF TRANSMISSION BLOCKSYMBOL FOR CANCELSYMBOL FOR END OF MEDIUMSYMBOL FOR SUBSTITU\
    \TESYMBOL FOR ESCAPESYMBOL FOR FILE SEPARATORSYMBOL FOR GROUP SEPARATORSYMBOL FOR RECORD SEPARATORSYMBOL FOR UNIT SEPARATORSYMBOL FOR SPACESYMBOL FOR DELETEBLANK SYMBOLOPEN BOXSYMBOL FOR NEWLINESYMBOL FOR DELETE FORM TWOSYMBOL FOR SUBSTITUTE FORM TWOOCR HO\
    \OKOCR CHAIROCR FORKOCR INVERTED FORKOCR BELT BUCKLEOCR BOW TIEOCR BRANCH BANK IDENTIFICATIONOCR AMOUNT OF CHECKOCR DASHOCR CUSTOMER ACCOUNT NUMBEROCR DOUBLE BACKSLASHCIRCLED DIGIT ONECIRCLED DIGIT TWOCIRCLED DIGIT THREECIRCLED DIGIT FOURCIRCLED DIGIT FIVE\
    \CIRCLED DIGIT SIXCIRCLED DIGIT SEVENCIRCLED DIGIT EIGHTCIRCLED DIGIT NINECIRCLED NUMBER TENCIRCLED NUMBER ELEVENCIRCLED NUMBER TWELVECIRCLED NUMBER THIRTEENCIRCLED NUMBER FOURTEENCIRCLED NUMBER FIFTEENCIRCLED NUMBER SIXTEENCIRCLED NUMBER SEVENTEENCIRCLED \
    \NUMBER EIGHTEENCIRCLED NUMBER NINETEENCIRCLED NUMBER TWENTYPARENTHESIZED DIGIT ONEPARENTHESIZED DIGIT TWOPARENTHESIZED DIGIT THREEPARENTHESIZED DIGIT FOURPARENTHESIZED DIGIT FIVEPARENTHESIZED DIGIT SIXPARENTHESIZED DIGIT SEVENPARENTHESIZED DIGIT EIGHTPARE\
    \NTHESIZED DIGIT NINEPARENTHESIZED NUMBER TENPARENTHESIZED NUMBER ELEVENPARENTHESIZED NUMBER TWELVEPARENTHESIZED NUMBER THIRTEENPARENTHESIZED NUMBER FOURTEENPARENTHESIZED NUMBER FIFTEENPARENTHESIZED NUMBER SIXTEENPARENTHESIZED NUMBER SEVENTEENPARENTHESIZED\
    \ NUMBER EIGHTEENPARENTHESIZED NUMBER NINETEENPARENTHESIZED NUMBER TWENTYDIGIT ONE FULL STOPDIGIT TWO FULL STOPDIGIT THREE FULL STOPDIGIT FOUR FULL STOPDIGIT FIVE FULL STOPDIGIT SIX FULL STOPDIGIT SEVEN FULL STOPDIGIT EIGHT FULL STOPDIGIT NINE FULL STOPNUM\
    \BER TEN FULL STOPNUMBER ELEVEN FULL STOPNUMBER TWELVE FULL STOPNUMBER THIRTEEN FULL STOPNUMBER FOURTEEN FULL STOPNUMBER FIFTEEN FULL STOPNUMBER SIXTEEN FULL STOPNUMBER SEVENTEEN FULL STOPNUMBER EIGHTEEN FULL STOPNUMBER NINETEEN FULL STOPNUMBER TWENTY FULL\
    \ STOPPARENTHESIZED LATIN SMALL LETTER APARENTHESIZED LATIN SMALL LETTER BPARENTHESIZED LATIN SMALL LETTER CPARENTHESIZED LATIN SMALL LETTER DPARENTHESIZED LATIN SMALL LETTER EPARENTHESIZED LATIN SMALL LETTER FPARENTHESIZED LATIN SMALL LETTER GPARENTHESIZE\
    \D LATIN SMALL LETTER HPARENTHESIZED LATIN SMALL LETTER IPARENTHESIZED LATIN SMALL LETTER JPARENTHESIZED LATIN SMALL LETTER KPARENTHESIZED LATIN SMALL LETTER LPARENTHESIZED LATIN SMALL LETTER MPARENTHESIZED LATIN SMALL LETTER NPARENTHESIZED LATIN SMALL LET\
    \TER OPARENTHESIZED LATIN SMALL LETTER PPARENTHESIZED LATIN SMALL LETTER QPARENTHESIZED LATIN SMALL LETTER RPARENTHESIZED LATIN SMALL LETTER SPARENTHESIZED LATIN SMALL LETTER TPARENTHESIZED LATIN SMALL LETTER UPARENTHESIZED LATIN SMALL LETTER VPARENTHESIZE\
    \D LATIN SMALL LETTER WPARENTHESIZED LATIN SMALL LETTER XPARENTHESIZED LATIN SMALL LETTER YPARENTHESIZED LATIN SMALL LETTER ZCIRCLED LATIN CAPITAL LETTER ACIRCLED LATIN CAPITAL LETTER BCIRCLED LATIN CAPITAL LETTER CCIRCLED LATIN CAPITAL LETTER DCIRCLED LAT\
    \IN CAPITAL LETTER ECIRCLED LATIN CAPITAL LETTER FCIRCLED LATIN CAPITAL LETTER GCIRCLED LATIN CAPITAL LETTER HCIRCLED LATIN CAPITAL LETTER ICIRCLED LATIN CAPITAL LETTER JCIRCLED LATIN CAPITAL LETTER KCIRCLED LATIN CAPITAL LETTER LCIRCLED LATIN CAPITAL LETT\
    \ER MCIRCLED LATIN CAPITAL LETTER NCIRCLED LATIN CAPITAL LETTER OCIRCLED LATIN CAPITAL LETTER PCIRCLED LATIN CAPITAL LETTER QCIRCLED LATIN CAPITAL LETTER RCIRCLED LATIN CAPITAL LETTER SCIRCLED LATIN CAPITAL LETTER TCIRCLED LATIN CAPITAL LETTER UCIRCLED LAT\
    \IN CAPITAL LETTER VCIRCLED LATIN CAPITAL LETTER WCIRCLED LATIN CAPITAL LETTER XCIRCLED LATIN CAPITAL LETTER YCIRCLED LATIN CAPITAL LETTER ZCIRCLED LATIN SMALL LETTER ACIRCLED LATIN SMALL LETTER BCIRCLED LATIN SMALL LETTER CCIRCLED LATIN SMALL LETTER DCIRC\
    \LED LATIN SMALL LETTER ECIRCLED LATIN SMALL LETTER FCIRCLED LATIN SMALL LETTER GCIRCLED LATIN SMALL LETTER HCIRCLED LATIN SMALL LETTER ICIRCLED LATIN SMALL LETTER JCIRCLED LATIN SMALL LETTER KCIRCLED LATIN SMALL LETTER LCIRCLED LATIN SMALL LETTER MCIRCLED\
    \ LATIN SMALL LETTER NCIRCLED LATIN SMALL LETTER OCIRCLED LATIN SMALL LETTER PCIRCLED LATIN SMALL LETTER QCIRCLED LATIN SMALL LETTER RCIRCLED LATIN SMALL LETTER SCIRCLED LATIN SMALL LETTER TCIRCLED LATIN SMALL LETTER UCIRCLED LATIN SMALL LETTER VCIRCLED LA\
    \TIN SMALL LETTER WCIRCLED LATIN SMALL LETTER XCIRCLED LATIN SMALL LETTER YCIRCLED LATIN SMALL LETTER ZCIRCLED DIGIT ZERONEGATIVE CIRCLED NUMBER ELEVENNEGATIVE CIRCLED NUMBER TWELVENEGATIVE CIRCLED NUMBER THIRTEENNEGATIVE CIRCLED NUMBER FOURTEENNEGATIVE CI\
    \RCLED NUMBER FIFTEENNEGATIVE CIRCLED NUMBER SIXTEENNEGATIVE CIRCLED NUMBER SEVENTEENNEGATIVE CIRCLED NUMBER EIGHTEENNEGATIVE CIRCLED NUMBER NINETEENNEGATIVE CIRCLED NUMBER TWENTYDOUBLE CIRCLED DIGIT ONEDOUBLE CIRCLED DIGIT TWODOUBLE CIRCLED DIGIT THREEDOU\
    \BLE CIRCLED DIGIT FOURDOUBLE CIRCLED DIGIT FIVEDOUBLE CIRCLED DIGIT SIXDOUBLE CIRCLED DIGIT SEVENDOUBLE CIRCLED DIGIT EIGHTDOUBLE CIRCLED DIGIT NINEDOUBLE CIRCLED NUMBER TENNEGATIVE CIRCLED DIGIT ZEROBOX DRAWINGS LIGHT HORIZONTALBOX DRAWINGS HEAVY HORIZON\
    \TALBOX DRAWINGS LIGHT VERTICALBOX DRAWINGS HEAVY VERTICALBOX DRAWINGS LIGHT TRIPLE DASH HORIZONTALBOX DRAWINGS HEAVY TRIPLE DASH HORIZONTALBOX DRAWINGS LIGHT TRIPLE DASH VERTICALBOX DRAWINGS HEAVY TRIPLE DASH VERTICALBOX DRAWINGS LIGHT QUADRUPLE DASH HORI\
    \ZONTALBOX DRAWINGS HEAVY QUADRUPLE DASH HORIZONTALBOX DRAWINGS LIGHT QUADRUPLE DASH VERTICALBOX DRAWINGS HEAVY QUADRUPLE DASH VERTICALBOX DRAWINGS LIGHT DOWN AND RIGHTBOX DRAWINGS DOWN LIGHT AND RIGHT HEAVYBOX DRAWINGS DOWN HEAVY AND RIGHT LIGHTBOX DRAWIN\
    \GS HEAVY DOWN AND RIGHTBOX DRAWINGS LIGHT DOWN AND LEFTBOX DRAWINGS DOWN LIGHT AND LEFT HEAVYBOX DRAWINGS DOWN HEAVY AND LEFT LIGHTBOX DRAWINGS HEAVY DOWN AND LEFTBOX DRAWINGS LIGHT UP AND RIGHTBOX DRAWINGS UP LIGHT AND RIGHT HEAVYBOX DRAWINGS UP HEAVY AN\
    \D RIGHT LIGHTBOX DRAWINGS HEAVY UP AND RIGHTBOX DRAWINGS LIGHT UP AND LEFTBOX DRAWINGS UP LIGHT AND LEFT HEAVYBOX DRAWINGS UP HEAVY AND LEFT LIGHTBOX DRAWINGS HEAVY UP AND LEFTBOX DRAWINGS LIGHT VERTICAL AND RIGHTBOX DRAWINGS VERTICAL LIGHT AND RIGHT HEAV\
    \YBOX DRAWINGS UP HEAVY AND RIGHT DOWN LIGHTBOX DRAWINGS DOWN HEAVY AND RIGHT UP LIGHTBOX DRAWINGS VERTICAL HEAVY AND RIGHT LIGHTBOX DRAWINGS DOWN LIGHT AND RIGHT UP HEAVYBOX DRAWINGS UP LIGHT AND RIGHT DOWN HEAVYBOX DRAWINGS HEAVY VERTICAL AND RIGHTBOX DR\
    \AWINGS LIGHT VERTICAL AND LEFTBOX DRAWINGS VERTICAL LIGHT AND LEFT HEAVYBOX DRAWINGS UP HEAVY AND LEFT DOWN LIGHTBOX DRAWINGS DOWN HEAVY AND LEFT UP LIGHTBOX DRAWINGS VERTICAL HEAVY AND LEFT LIGHTBOX DRAWINGS DOWN LIGHT AND LEFT UP HEAVYBOX DRAWINGS UP LI\
    \GHT AND LEFT DOWN HEAVYBOX DRAWINGS HEAVY VERTICAL AND LEFTBOX DRAWINGS LIGHT DOWN AND HORIZONTALBOX DRAWINGS LEFT HEAVY AND RIGHT DOWN LIGHTBOX DRAWINGS RIGHT HEAVY AND LEFT DOWN LIGHTBOX DRAWINGS DOWN LIGHT AND HORIZONTAL HEAVYBOX DRAWINGS DOWN HEAVY AN\
    \D HORIZONTAL LIGHTBOX DRAWINGS RIGHT LIGHT AND LEFT DOWN HEAVYBOX DRAWINGS LEFT LIGHT AND RIGHT DOWN HEAVYBOX DRAWINGS HEAVY DOWN AND HORIZONTALBOX DRAWINGS LIGHT UP AND HORIZONTALBOX DRAWINGS LEFT HEAVY AND RIGHT UP LIGHTBOX DRAWINGS RIGHT HEAVY AND LEFT\
    \ UP LIGHTBOX DRAWINGS UP LIGHT AND HORIZONTAL HEAVYBOX DRAWINGS UP HEAVY AND HORIZONTAL LIGHTBOX DRAWINGS RIGHT LIGHT AND LEFT UP HEAVYBOX DRAWINGS LEFT LIGHT AND RIGHT UP HEAVYBOX DRAWINGS HEAVY UP AND HORIZONTALBOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL\
    \BOX DRAWINGS LEFT HEAVY AND RIGHT VERTICAL LIGHTBOX DRAWINGS RIGHT HEAVY AND LEFT VERTICAL LIGHTBOX DRAWINGS VERTICAL LIGHT AND HORIZONTAL HEAVYBOX DRAWINGS UP HEAVY AND DOWN HORIZONTAL LIGHTBOX DRAWINGS DOWN HEAVY AND UP HORIZONTAL LIGHTBOX DRAWINGS VERT\
    \ICAL HEAVY AND HORIZONTAL LIGHTBOX DRAWINGS LEFT UP HEAVY AND RIGHT DOWN LIGHTBOX DRAWINGS RIGHT UP HEAVY AND LEFT DOWN LIGHTBOX DRAWINGS LEFT DOWN HEAVY AND RIGHT UP LIGHTBOX DRAWINGS RIGHT DOWN HEAVY AND LEFT UP LIGHTBOX DRAWINGS DOWN LIGHT AND UP HORIZ\
    \ONTAL HEAVYBOX DRAWINGS UP LIGHT AND DOWN HORIZONTAL HEAVYBOX DRAWINGS RIGHT LIGHT AND LEFT VERTICAL HEAVYBOX DRAWINGS LEFT LIGHT AND RIGHT VERTICAL HEAVYBOX DRAWINGS HEAVY VERTICAL AND HORIZONTALBOX DRAWINGS LIGHT DOUBLE DASH HORIZONTALBOX DRAWINGS HEAVY\
    \ DOUBLE DASH HORIZONTALBOX DRAWINGS LIGHT DOUBLE DASH VERTICALBOX DRAWINGS HEAVY DOUBLE DASH VERTICALBOX DRAWINGS DOUBLE HORIZONTALBOX DRAWINGS DOUBLE VERTICALBOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLEBOX DRAWINGS DOWN DOUBLE AND RIGHT SINGLEBOX DRAWINGS D\
    \OUBLE DOWN AND RIGHTBOX DRAWINGS DOWN SINGLE AND LEFT DOUBLEBOX DRAWINGS DOWN DOUBLE AND LEFT SINGLEBOX DRAWINGS DOUBLE DOWN AND LEFTBOX DRAWINGS UP SINGLE AND RIGHT DOUBLEBOX DRAWINGS UP DOUBLE AND RIGHT SINGLEBOX DRAWINGS DOUBLE UP AND RIGHTBOX DRAWINGS\
    \ UP SINGLE AND LEFT DOUBLEBOX DRAWINGS UP DOUBLE AND LEFT SINGLEBOX DRAWINGS DOUBLE UP AND LEFTBOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLEBOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLEBOX DRAWINGS DOUBLE VERTICAL AND RIGHTBOX DRAWINGS VERTICAL SINGLE AND\
    \ LEFT DOUBLEBOX DRAWINGS VERTICAL DOUBLE AND LEFT SINGLEBOX DRAWINGS DOUBLE VERTICAL AND LEFTBOX DRAWINGS DOWN SINGLE AND HORIZONTAL DOUBLEBOX DRAWINGS DOWN DOUBLE AND HORIZONTAL SINGLEBOX DRAWINGS DOUBLE DOWN AND HORIZONTALBOX DRAWINGS UP SINGLE AND HORI\
    \ZONTAL DOUBLEBOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLEBOX DRAWINGS DOUBLE UP AND HORIZONTALBOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLEBOX DRAWINGS VERTICAL DOUBLE AND HORIZONTAL SINGLEBOX DRAWINGS DOUBLE VERTICAL AND HORIZONTALBOX DRAWINGS LIGHT\
    \ ARC DOWN AND RIGHTBOX DRAWINGS LIGHT ARC DOWN AND LEFTBOX DRAWINGS LIGHT ARC UP AND LEFTBOX DRAWINGS LIGHT ARC UP AND RIGHTBOX DRAWINGS LIGHT DIAGONAL UPPER RIGHT TO LOWER LEFTBOX DRAWINGS LIGHT DIAGONAL UPPER LEFT TO LOWER RIGHTBOX DRAWINGS LIGHT DIAGON\
    \AL CROSSBOX DRAWINGS LIGHT LEFTBOX DRAWINGS LIGHT UPBOX DRAWINGS LIGHT RIGHTBOX DRAWINGS LIGHT DOWNBOX DRAWINGS HEAVY LEFTBOX DRAWINGS HEAVY UPBOX DRAWINGS HEAVY RIGHTBOX DRAWINGS HEAVY DOWNBOX DRAWINGS LIGHT LEFT AND HEAVY RIGHTBOX DRAWINGS LIGHT UP AND \
    \HEAVY DOWNBOX DRAWINGS HEAVY LEFT AND LIGHT RIGHTBOX DRAWINGS HEAVY UP AND LIGHT DOWNUPPER HALF BLOCKLOWER ONE EIGHTH BLOCKLOWER ONE QUARTER BLOCKLOWER THREE EIGHTHS BLOCKLOWER HALF BLOCKLOWER FIVE EIGHTHS BLOCKLOWER THREE QUARTERS BLOCKLOWER SEVEN EIGHTH\
    \S BLOCKFULL BLOCKLEFT SEVEN EIGHTHS BLOCKLEFT THREE QUARTERS BLOCKLEFT FIVE EIGHTHS BLOCKLEFT HALF BLOCKLEFT THREE EIGHTHS BLOCKLEFT ONE QUARTER BLOCKLEFT ONE EIGHTH BLOCKRIGHT HALF BLOCKLIGHT SHADEMEDIUM SHADEDARK SHADEUPPER ONE EIGHTH BLOCKRIGHT ONE EIG\
    \HTH BLOCKQUADRANT LOWER LEFTQUADRANT LOWER RIGHTQUADRANT UPPER LEFTQUADRANT UPPER LEFT AND LOWER LEFT AND LOWER RIGHTQUADRANT UPPER LEFT AND LOWER RIGHTQUADRANT UPPER LEFT AND UPPER RIGHT AND LOWER LEFTQUADRANT UPPER LEFT AND UPPER RIGHT AND LOWER RIGHTQU\
    \ADRANT UPPER RIGHTQUADRANT UPPER RIGHT AND LOWER LEFTQUADRANT UPPER RIGHT AND LOWER LEFT AND LOWER RIGHTBLACK SQUAREWHITE SQUAREWHITE SQUARE WITH ROUNDED CORNERSWHITE SQUARE CONTAINING BLACK SMALL SQUARESQUARE WITH HORIZONTAL FILLSQUARE WITH VERTICAL FILL\
    \SQUARE WITH ORTHOGONAL CROSSHATCH FILLSQUARE WITH UPPER LEFT TO LOWER RIGHT FILLSQUARE WITH UPPER RIGHT TO LOWER LEFT FILLSQUARE WITH DIAGONAL CROSSHATCH FILLBLACK SMALL SQUAREWHITE SMALL SQUAREBLACK RECTANGLEWHITE RECTANGLEBLACK VERTICAL RECTANGLEWHITE V\
    \ERTICAL RECTANGLEBLACK PARALLELOGRAMWHITE PARALLELOGRAMBLACK UP-POINTING TRIANGLEWHITE UP-POINTING TRIANGLEBLACK UP-POINTING SMALL TRIANGLEWHITE UP-POINTING SMALL TRIANGLEBLACK RIGHT-POINTING TRIANGLEWHITE RIGHT-POINTING TRIANGLEBLACK RIGHT-POINTING SMALL\
    \ TRIANGLEWHITE RIGHT-POINTING SMALL TRIANGLEBLACK RIGHT-POINTING POINTERWHITE RIGHT-POINTING POINTERBLACK DOWN-POINTING TRIANGLEWHITE DOWN-POINTING TRIANGLEBLACK DOWN-POINTING SMALL TRIANGLEWHITE DOWN-POINTING SMALL TRIANGLEBLACK LEFT-POINTING TRIANGLEWHI\
    \TE LEFT-POINTING TRIANGLEBLACK LEFT-POINTING SMALL TRIANGLEWHITE LEFT-POINTING SMALL TRIANGLEBLACK LEFT-POINTING POINTERWHITE LEFT-POINTING POINTERBLACK DIAMONDWHITE DIAMONDWHITE DIAMOND CONTAINING BLACK SMALL DIAMONDFISHEYELOZENGEWHITE CIRCLEDOTTED CIRCL\
    \ECIRCLE WITH VERTICAL FILLBULLSEYEBLACK CIRCLECIRCLE WITH LEFT HALF BLACKCIRCLE WITH RIGHT HALF BLACKCIRCLE WITH LOWER HALF BLACKCIRCLE WITH UPPER HALF BLACKCIRCLE WITH UPPER RIGHT QUADRANT BLACKCIRCLE WITH ALL BUT UPPER LEFT QUADRANT BLACKLEFT HALF BLACK\
    \ CIRCLERIGHT HALF BLACK CIRCLEINVERSE BULLETINVERSE WHITE CIRCLEUPPER HALF INVERSE WHITE CIRCLELOWER HALF INVERSE WHITE CIRCLEUPPER LEFT QUADRANT CIRCULAR ARCUPPER RIGHT QUADRANT CIRCULAR ARCLOWER RIGHT QUADRANT CIRCULAR ARCLOWER LEFT QUADRANT CIRCULAR AR\
    \CUPPER HALF CIRCLELOWER HALF CIRCLEBLACK LOWER RIGHT TRIANGLEBLACK LOWER LEFT TRIANGLEBLACK UPPER LEFT TRIANGLEBLACK UPPER RIGHT TRIANGLEWHITE BULLETSQUARE WITH LEFT HALF BLACKSQUARE WITH RIGHT HALF BLACKSQUARE WITH UPPER LEFT DIAGONAL HALF BLACKSQUARE WI\
    \TH LOWER RIGHT DIAGONAL HALF BLACKWHITE SQUARE WITH VERTICAL BISECTING LINEWHITE UP-POINTING TRIANGLE WITH DOTUP-POINTING TRIANGLE WITH LEFT HALF BLACKUP-POINTING TRIANGLE WITH RIGHT HALF BLACKLARGE CIRCLEWHITE SQUARE WITH UPPER LEFT QUADRANTWHITE SQUARE \
    \WITH LOWER LEFT QUADRANTWHITE SQUARE WITH LOWER RIGHT QUADRANTWHITE SQUARE WITH UPPER RIGHT QUADRANTWHITE CIRCLE WITH UPPER LEFT QUADRANTWHITE CIRCLE WITH LOWER LEFT QUADRANTWHITE CIRCLE WITH LOWER RIGHT QUADRANTWHITE CIRCLE WITH UPPER RIGHT QUADRANTUPPER\
    \ LEFT TRIANGLEUPPER RIGHT TRIANGLELOWER LEFT TRIANGLEWHITE MEDIUM SQUAREBLACK MEDIUM SQUAREWHITE MEDIUM SMALL SQUAREBLACK MEDIUM SMALL SQUARELOWER RIGHT TRIANGLEBLACK SUN WITH RAYSCLOUDUMBRELLASNOWMANCOMETBLACK STARWHITE STARLIGHTNINGTHUNDERSTORMSUNASCEND\
    \ING NODEDESCENDING NODECONJUNCTIONOPPOSITIONBLACK TELEPHONEWHITE TELEPHONEBALLOT BOXBALLOT BOX WITH CHECKBALLOT BOX WITH XSALTIREUMBRELLA WITH RAIN DROPSHOT BEVERAGEWHITE SHOGI PIECEBLACK SHOGI PIECESHAMROCKREVERSED ROTATED FLORAL HEART BULLETBLACK LEFT P\
    \OINTING INDEXBLACK RIGHT POINTING INDEXWHITE LEFT POINTING INDEXWHITE UP POINTING INDEXWHITE RIGHT POINTING INDEXWHITE DOWN POINTING INDEXSKULL AND CROSSBONESCAUTION SIGNRADIOACTIVE SIGNBIOHAZARD SIGNCADUCEUSANKHORTHODOX CROSSCHI RHOCROSS OF LORRAINECROSS\
    \ OF JERUSALEMSTAR AND CRESCENTFARSI SYMBOLADI SHAKTIHAMMER AND SICKLEPEACE SYMBOLYIN YANGTRIGRAM FOR HEAVENTRIGRAM FOR LAKETRIGRAM FOR FIRETRIGRAM FOR THUNDERTRIGRAM FOR WINDTRIGRAM FOR WATERTRIGRAM FOR MOUNTAINTRIGRAM FOR EARTHWHEEL OF DHARMAWHITE FROWNI\
    \NG FACEWHITE SMILING FACEBLACK SMILING FACEWHITE SUN WITH RAYSFIRST QUARTER MOONLAST QUARTER MOONMERCURYFEMALE SIGNEARTHMALE SIGNJUPITERSATURNURANUSNEPTUNEPLUTOARIESTAURUSGEMINICANCERLEOVIRGOLIBRASCORPIUSSAGITTARIUSCAPRICORNAQUARIUSPISCESWHITE CHESS KINGW\
    \HITE CHESS QUEENWHITE CHESS ROOKWHITE CHESS BISHOPWHITE CHESS KNIGHTWHITE CHESS PAWNBLACK CHESS KINGBLACK CHESS QUEENBLACK CHESS ROOKBLACK CHESS BISHOPBLACK CHESS KNIGHTBLACK CHESS PAWNBLACK SPADE SUITWHITE HEART SUITWHITE DIAMOND SUITBLACK CLUB SUITWHITE\
    \ SPADE SUITBLACK HEART SUITBLACK DIAMOND SUITWHITE CLUB SUITHOT SPRINGSQUARTER NOTEEIGHTH NOTEBEAMED EIGHTH NOTESBEAMED SIXTEENTH NOTESMUSIC FLAT SIGNMUSIC NATURAL SIGNMUSIC SHARP SIGNWEST SYRIAC CROSSEAST SYRIAC CROSSUNIVERSAL RECYCLING SYMBOLRECYCLING S\
    \YMBOL FOR TYPE-1 PLASTICSRECYCLING SYMBOL FOR TYPE-2 PLASTICSRECYCLING SYMBOL FOR TYPE-3 PLASTICSRECYCLING SYMBOL FOR TYPE-4 PLASTICSRECYCLING SYMBOL FOR TYPE-5 PLASTICSRECYCLING SYMBOL FOR TYPE-6 PLASTICSRECYCLING SYMBOL FOR TYPE-7 PLASTICSRECYCLING SYMB\
    \OL FOR GENERIC MATERIALSBLACK UNIVERSAL RECYCLING SYMBOLRECYCLED PAPER SYMBOLPARTIALLY-RECYCLED PAPER SYMBOLPERMANENT PAPER SIGNWHEELCHAIR SYMBOLDIE FACE-1DIE FACE-2DIE FACE-3DIE FACE-4DIE FACE-5DIE FACE-6WHITE CIRCLE WITH DOT RIGHTWHITE CIRCLE WITH TWO D\
    \OTSBLACK CIRCLE WITH WHITE DOT RIGHTBLACK CIRCLE WITH TWO WHITE DOTSMONOGRAM FOR YANGMONOGRAM FOR YINDIGRAM FOR GREATER YANGDIGRAM FOR LESSER YINDIGRAM FOR LESSER YANGDIGRAM FOR GREATER YINWHITE FLAGBLACK FLAGHAMMER AND PICKANCHORCROSSED SWORDSSTAFF OF AE\
    \SCULAPIUSSCALESALEMBICFLOWERGEARSTAFF OF HERMESATOM SYMBOLFLEUR-DE-LISOUTLINED WHITE STARTHREE LINES CONVERGING RIGHTTHREE LINES CONVERGING LEFTWARNING SIGNHIGH VOLTAGE SIGNDOUBLED FEMALE SIGNDOUBLED MALE SIGNINTERLOCKED FEMALE AND MALE SIGNMALE AND FEMAL\
    \E SIGNMALE WITH STROKE SIGNMALE WITH STROKE AND MALE AND FEMALE SIGNVERTICAL MALE WITH STROKE SIGNHORIZONTAL MALE WITH STROKE SIGNMEDIUM WHITE CIRCLEMEDIUM BLACK CIRCLEMEDIUM SMALL WHITE CIRCLEMARRIAGE SYMBOLDIVORCE SYMBOLUNMARRIED PARTNERSHIP SYMBOLCOFFI\
    \NFUNERAL URNNEUTERCERESPALLASJUNOVESTACHIRONBLACK MOON LILITHSEXTILESEMISEXTILEQUINCUNXSESQUIQUADRATESOCCER BALLBASEBALLSQUARED KEYWHITE DRAUGHTS MANWHITE DRAUGHTS KINGBLACK DRAUGHTS MANBLACK DRAUGHTS KINGSNOWMAN WITHOUT SNOWSUN BEHIND CLOUDRAINBLACK SNOW\
    \MANTHUNDER CLOUD AND RAINTURNED WHITE SHOGI PIECETURNED BLACK SHOGI PIECEWHITE DIAMOND IN SQUARECROSSING LANESDISABLED CAROPHIUCHUSPICKCAR SLIDINGHELMET WITH WHITE CROSSCIRCLED CROSSING LANESCHAINSNO ENTRYALTERNATE ONE-WAY LEFT WAY TRAFFICBLACK TWO-WAY LE\
    \FT WAY TRAFFICWHITE TWO-WAY LEFT WAY TRAFFICBLACK LEFT LANE MERGEWHITE LEFT LANE MERGEDRIVE SLOW SIGNHEAVY WHITE DOWN-POINTING TRIANGLELEFT CLOSED ENTRYSQUARED SALTIREFALLING DIAGONAL IN WHITE CIRCLE IN BLACK SQUAREBLACK TRUCKRESTRICTED LEFT ENTRY-1RESTRI\
    \CTED LEFT ENTRY-2ASTRONOMICAL SYMBOL FOR URANUSHEAVY CIRCLE WITH STROKE AND TWO DOTS ABOVEPENTAGRAMRIGHT-HANDED INTERLACED PENTAGRAMLEFT-HANDED INTERLACED PENTAGRAMINVERTED PENTAGRAMBLACK CROSS ON SHIELDSHINTO SHRINECHURCHCASTLEHISTORIC SITEGEAR WITHOUT H\
    \UBGEAR WITH HANDLESMAP SYMBOL FOR LIGHTHOUSEMOUNTAINUMBRELLA ON GROUNDFOUNTAINFLAG IN HOLEFERRYSAILBOATSQUARE FOUR CORNERSSKIERICE SKATEPERSON WITH BALLTENTJAPANESE BANK SYMBOLHEADSTONE GRAVEYARD SYMBOLFUEL PUMPCUP ON BLACK SQUAREWHITE FLAG WITH HORIZONTA\
    \L MIDDLE BLACK STRIPEBLACK SAFETY SCISSORSUPPER BLADE SCISSORSBLACK SCISSORSLOWER BLADE SCISSORSWHITE SCISSORSWHITE HEAVY CHECK MARKTELEPHONE LOCATION SIGNTAPE DRIVEAIRPLANEENVELOPERAISED FISTRAISED HANDVICTORY HANDWRITING HANDLOWER RIGHT PENCILPENCILUPPE\
    \R RIGHT PENCILWHITE NIBBLACK NIBCHECK MARKHEAVY CHECK MARKMULTIPLICATION XHEAVY MULTIPLICATION XBALLOT XHEAVY BALLOT XOUTLINED GREEK CROSSHEAVY GREEK CROSSOPEN CENTRE CROSSHEAVY OPEN CENTRE CROSSLATIN CROSSSHADOWED WHITE LATIN CROSSOUTLINED LATIN CROSSMAL\
    \TESE CROSSSTAR OF DAVIDFOUR TEARDROP-SPOKED ASTERISKFOUR BALLOON-SPOKED ASTERISKHEAVY FOUR BALLOON-SPOKED ASTERISKFOUR CLUB-SPOKED ASTERISKBLACK FOUR POINTED STARWHITE FOUR POINTED STARSPARKLESSTRESS OUTLINED WHITE STARCIRCLED WHITE STAROPEN CENTRE BLACK \
    \STARBLACK CENTRE WHITE STAROUTLINED BLACK STARHEAVY OUTLINED BLACK STARPINWHEEL STARSHADOWED WHITE STARHEAVY ASTERISKOPEN CENTRE ASTERISKEIGHT SPOKED ASTERISKEIGHT POINTED BLACK STAREIGHT POINTED PINWHEEL STARSIX POINTED BLACK STAREIGHT POINTED RECTILINEA\
    \R BLACK STARHEAVY EIGHT POINTED RECTILINEAR BLACK STARTWELVE POINTED BLACK STARSIXTEEN POINTED ASTERISKTEARDROP-SPOKED ASTERISKOPEN CENTRE TEARDROP-SPOKED ASTERISKHEAVY TEARDROP-SPOKED ASTERISKSIX PETALLED BLACK AND WHITE FLORETTEBLACK FLORETTEWHITE FLORE\
    \TTEEIGHT PETALLED OUTLINED BLACK FLORETTECIRCLED OPEN CENTRE EIGHT POINTED STARHEAVY TEARDROP-SPOKED PINWHEEL ASTERISKSNOWFLAKETIGHT TRIFOLIATE SNOWFLAKEHEAVY CHEVRON SNOWFLAKESPARKLEHEAVY SPARKLEBALLOON-SPOKED ASTERISKEIGHT TEARDROP-SPOKED PROPELLER ASTE\
    \RISKHEAVY EIGHT TEARDROP-SPOKED PROPELLER ASTERISKCROSS MARKSHADOWED WHITE CIRCLENEGATIVE SQUARED CROSS MARKLOWER RIGHT DROP-SHADOWED WHITE SQUAREUPPER RIGHT DROP-SHADOWED WHITE SQUARELOWER RIGHT SHADOWED WHITE SQUAREUPPER RIGHT SHADOWED WHITE SQUAREBLACK\
    \ QUESTION MARK ORNAMENTWHITE QUESTION MARK ORNAMENTWHITE EXCLAMATION MARK ORNAMENTBLACK DIAMOND MINUS WHITE XHEAVY EXCLAMATION MARK SYMBOLLIGHT VERTICAL BARMEDIUM VERTICAL BARHEAVY VERTICAL BARHEAVY SINGLE TURNED COMMA QUOTATION MARK ORNAMENTHEAVY SINGLE \
    \COMMA QUOTATION MARK ORNAMENTHEAVY DOUBLE TURNED COMMA QUOTATION MARK ORNAMENTHEAVY DOUBLE COMMA QUOTATION MARK ORNAMENTHEAVY LOW SINGLE COMMA QUOTATION MARK ORNAMENTHEAVY LOW DOUBLE COMMA QUOTATION MARK ORNAMENTCURVED STEM PARAGRAPH SIGN ORNAMENTHEAVY EX\
    \CLAMATION MARK ORNAMENTHEAVY HEART EXCLAMATION MARK ORNAMENTHEAVY BLACK HEARTROTATED HEAVY BLACK HEART BULLETFLORAL HEARTROTATED FLORAL HEART BULLETMEDIUM LEFT PARENTHESIS ORNAMENTMEDIUM RIGHT PARENTHESIS ORNAMENTMEDIUM FLATTENED LEFT PARENTHESIS ORNAMENT\
    \MEDIUM FLATTENED RIGHT PARENTHESIS ORNAMENTMEDIUM LEFT-POINTING ANGLE BRACKET ORNAMENTMEDIUM RIGHT-POINTING ANGLE BRACKET ORNAMENTHEAVY LEFT-POINTING ANGLE QUOTATION MARK ORNAMENTHEAVY RIGHT-POINTING ANGLE QUOTATION MARK ORNAMENTHEAVY LEFT-POINTING ANGLE \
    \BRACKET ORNAMENTHEAVY RIGHT-POINTING ANGLE BRACKET ORNAMENTLIGHT LEFT TORTOISE SHELL BRACKET ORNAMENTLIGHT RIGHT TORTOISE SHELL BRACKET ORNAMENTMEDIUM LEFT CURLY BRACKET ORNAMENTMEDIUM RIGHT CURLY BRACKET ORNAMENTDINGBAT NEGATIVE CIRCLED DIGIT ONEDINGBAT \
    \NEGATIVE CIRCLED DIGIT TWODINGBAT NEGATIVE CIRCLED DIGIT THREEDINGBAT NEGATIVE CIRCLED DIGIT FOURDINGBAT NEGATIVE CIRCLED DIGIT FIVEDINGBAT NEGATIVE CIRCLED DIGIT SIXDINGBAT NEGATIVE CIRCLED DIGIT SEVENDINGBAT NEGATIVE CIRCLED DIGIT EIGHTDINGBAT NEGATIVE \
    \CIRCLED DIGIT NINEDINGBAT NEGATIVE CIRCLED NUMBER TENDINGBAT CIRCLED SANS-SERIF DIGIT ONEDINGBAT CIRCLED SANS-SERIF DIGIT TWODINGBAT CIRCLED SANS-SERIF DIGIT THREEDINGBAT CIRCLED SANS-SERIF DIGIT FOURDINGBAT CIRCLED SANS-SERIF DIGIT FIVEDINGBAT CIRCLED SA\
    \NS-SERIF DIGIT SIXDINGBAT CIRCLED SANS-SERIF DIGIT SEVENDINGBAT CIRCLED SANS-SERIF DIGIT EIGHTDINGBAT CIRCLED SANS-SERIF DIGIT NINEDINGBAT CIRCLED SANS-SERIF NUMBER TENDINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT ONEDINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT \
    \TWODINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT THREEDINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT FOURDINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT FIVEDINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT SIXDINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT SEVENDINGBAT NEGATIVE CIRC\
    \LED SANS-SERIF DIGIT EIGHTDINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT NINEDINGBAT NEGATIVE CIRCLED SANS-SERIF NUMBER TENHEAVY WIDE-HEADED RIGHTWARDS ARROWHEAVY PLUS SIGNHEAVY MINUS SIGNHEAVY DIVISION SIGNHEAVY SOUTH EAST ARROWHEAVY RIGHTWARDS ARROWHEAVY NOR\
    \TH EAST ARROWDRAFTING POINT RIGHTWARDS ARROWHEAVY ROUND-TIPPED RIGHTWARDS ARROWTRIANGLE-HEADED RIGHTWARDS ARROWHEAVY TRIANGLE-HEADED RIGHTWARDS ARROWDASHED TRIANGLE-HEADED RIGHTWARDS ARROWHEAVY DASHED TRIANGLE-HEADED RIGHTWARDS ARROWBLACK RIGHTWARDS ARROW\
    \THREE-D TOP-LIGHTED RIGHTWARDS ARROWHEADTHREE-D BOTTOM-LIGHTED RIGHTWARDS ARROWHEADBLACK RIGHTWARDS ARROWHEADHEAVY BLACK CURVED DOWNWARDS AND RIGHTWARDS ARROWHEAVY BLACK CURVED UPWARDS AND RIGHTWARDS ARROWSQUAT BLACK RIGHTWARDS ARROWHEAVY CONCAVE-POINTED \
    \BLACK RIGHTWARDS ARROWRIGHT-SHADED WHITE RIGHTWARDS ARROWLEFT-SHADED WHITE RIGHTWARDS ARROWBACK-TILTED SHADOWED WHITE RIGHTWARDS ARROWFRONT-TILTED SHADOWED WHITE RIGHTWARDS ARROWHEAVY LOWER RIGHT-SHADOWED WHITE RIGHTWARDS ARROWHEAVY UPPER RIGHT-SHADOWED W\
    \HITE RIGHTWARDS ARROWNOTCHED LOWER RIGHT-SHADOWED WHITE RIGHTWARDS ARROWCURLY LOOPNOTCHED UPPER RIGHT-SHADOWED WHITE RIGHTWARDS ARROWCIRCLED HEAVY WHITE RIGHTWARDS ARROWWHITE-FEATHERED RIGHTWARDS ARROWBLACK-FEATHERED SOUTH EAST ARROWBLACK-FEATHERED RIGHTW\
    \ARDS ARROWBLACK-FEATHERED NORTH EAST ARROWHEAVY BLACK-FEATHERED SOUTH EAST ARROWHEAVY BLACK-FEATHERED RIGHTWARDS ARROWHEAVY BLACK-FEATHERED NORTH EAST ARROWTEARDROP-BARBED RIGHTWARDS ARROWHEAVY TEARDROP-SHANKED RIGHTWARDS ARROWWEDGE-TAILED RIGHTWARDS ARRO\
    \WHEAVY WEDGE-TAILED RIGHTWARDS ARROWOPEN-OUTLINED RIGHTWARDS ARROWDOUBLE CURLY LOOPTHREE DIMENSIONAL ANGLEWHITE TRIANGLE CONTAINING SMALL WHITE TRIANGLEPERPENDICULAROPEN SUBSETOPEN SUPERSETLEFT S-SHAPED BAG DELIMITERRIGHT S-SHAPED BAG DELIMITEROR WITH DOT\
    \ INSIDEREVERSE SOLIDUS PRECEDING SUBSETSUPERSET PRECEDING SOLIDUSVERTICAL BAR WITH HORIZONTAL STROKEMATHEMATICAL RISING DIAGONALLONG DIVISIONMATHEMATICAL FALLING DIAGONALSQUARED LOGICAL ANDSQUARED LOGICAL ORWHITE DIAMOND WITH CENTRED DOTAND WITH DOTELEMEN\
    \T OF OPENING UPWARDSLOWER RIGHT CORNER WITH DOTUPPER LEFT CORNER WITH DOTLEFT OUTER JOINRIGHT OUTER JOINFULL OUTER JOINLARGE UP TACKLARGE DOWN TACKLEFT AND RIGHT DOUBLE TURNSTILELEFT AND RIGHT TACKLEFT MULTIMAPLONG RIGHT TACKLONG LEFT TACKUP TACK WITH CIR\
    \CLE ABOVELOZENGE DIVIDED BY HORIZONTAL RULEWHITE CONCAVE-SIDED DIAMONDWHITE CONCAVE-SIDED DIAMOND WITH LEFTWARDS TICKWHITE CONCAVE-SIDED DIAMOND WITH RIGHTWARDS TICKWHITE SQUARE WITH LEFTWARDS TICKWHITE SQUARE WITH RIGHTWARDS TICKMATHEMATICAL LEFT WHITE S\
    \QUARE BRACKETMATHEMATICAL RIGHT WHITE SQUARE BRACKETMATHEMATICAL LEFT ANGLE BRACKETMATHEMATICAL RIGHT ANGLE BRACKETMATHEMATICAL LEFT DOUBLE ANGLE BRACKETMATHEMATICAL RIGHT DOUBLE ANGLE BRACKETMATHEMATICAL LEFT WHITE TORTOISE SHELL BRACKETMATHEMATICAL RIGH\
    \T WHITE TORTOISE SHELL BRACKETMATHEMATICAL LEFT FLATTENED PARENTHESISMATHEMATICAL RIGHT FLATTENED PARENTHESISUPWARDS QUADRUPLE ARROWDOWNWARDS QUADRUPLE ARROWANTICLOCKWISE GAPPED CIRCLE ARROWCLOCKWISE GAPPED CIRCLE ARROWRIGHT ARROW WITH CIRCLED PLUSLONG LE\
    \FTWARDS ARROWLONG RIGHTWARDS ARROWLONG LEFT RIGHT ARROWLONG LEFTWARDS DOUBLE ARROWLONG RIGHTWARDS DOUBLE ARROWLONG LEFT RIGHT DOUBLE ARROWLONG LEFTWARDS ARROW FROM BARLONG RIGHTWARDS ARROW FROM BARLONG LEFTWARDS DOUBLE ARROW FROM BARLONG RIGHTWARDS DOUBLE\
    \ ARROW FROM BARLONG RIGHTWARDS SQUIGGLE ARROWBRAILLE PATTERN BLANKBRAILLE PATTERN DOTS-1BRAILLE PATTERN DOTS-2BRAILLE PATTERN DOTS-12BRAILLE PATTERN DOTS-3BRAILLE PATTERN DOTS-13BRAILLE PATTERN DOTS-23BRAILLE PATTERN DOTS-123BRAILLE PATTERN DOTS-4BRAILLE \
    \PATTERN DOTS-14BRAILLE PATTERN DOTS-24BRAILLE PATTERN DOTS-124BRAILLE PATTERN DOTS-34BRAILLE PATTERN DOTS-134BRAILLE PATTERN DOTS-234BRAILLE PATTERN DOTS-1234BRAILLE PATTERN DOTS-5BRAILLE PATTERN DOTS-15BRAILLE PATTERN DOTS-25BRAILLE PATTERN DOTS-125BRAIL\
    \LE PATTERN DOTS-35BRAILLE PATTERN DOTS-135BRAILLE PATTERN DOTS-235BRAILLE PATTERN DOTS-1235BRAILLE PATTERN DOTS-45BRAILLE PATTERN DOTS-145BRAILLE PATTERN DOTS-245BRAILLE PATTERN DOTS-1245BRAILLE PATTERN DOTS-345BRAILLE PATTERN DOTS-1345BRAILLE PATTERN DOT\
    \S-2345BRAILLE PATTERN DOTS-12345BRAILLE PATTERN DOTS-6BRAILLE PATTERN DOTS-16BRAILLE PATTERN DOTS-26BRAILLE PATTERN DOTS-126BRAILLE PATTERN DOTS-36BRAILLE PATTERN DOTS-136BRAILLE PATTERN DOTS-236BRAILLE PATTERN DOTS-1236BRAILLE PATTERN DOTS-46BRAILLE PATT\
    \ERN DOTS-146BRAILLE PATTERN DOTS-246BRAILLE PATTERN DOTS-1246BRAILLE PATTERN DOTS-346BRAILLE PATTERN DOTS-1346BRAILLE PATTERN DOTS-2346BRAILLE PATTERN DOTS-12346BRAILLE PATTERN DOTS-56BRAILLE PATTERN DOTS-156BRAILLE PATTERN DOTS-256BRAILLE PATTERN DOTS-12\
    \56BRAILLE PATTERN DOTS-356BRAILLE PATTERN DOTS-1356BRAILLE PATTERN DOTS-2356BRAILLE PATTERN DOTS-12356BRAILLE PATTERN DOTS-456BRAILLE PATTERN DOTS-1456BRAILLE PATTERN DOTS-2456BRAILLE PATTERN DOTS-12456BRAILLE PATTERN DOTS-3456BRAILLE PATTERN DOTS-13456BR\
    \AILLE PATTERN DOTS-23456BRAILLE PATTERN DOTS-123456BRAILLE PATTERN DOTS-7BRAILLE PATTERN DOTS-17BRAILLE PATTERN DOTS-27BRAILLE PATTERN DOTS-127BRAILLE PATTERN DOTS-37BRAILLE PATTERN DOTS-137BRAILLE PATTERN DOTS-237BRAILLE PATTERN DOTS-1237BRAILLE PATTERN \
    \DOTS-47BRAILLE PATTERN DOTS-147BRAILLE PATTERN DOTS-247BRAILLE PATTERN DOTS-1247BRAILLE PATTERN DOTS-347BRAILLE PATTERN DOTS-1347BRAILLE PATTERN DOTS-2347BRAILLE PATTERN DOTS-12347BRAILLE PATTERN DOTS-57BRAILLE PATTERN DOTS-157BRAILLE PATTERN DOTS-257BRAI\
    \LLE PATTERN DOTS-1257BRAILLE PATTERN DOTS-357BRAILLE PATTERN DOTS-1357BRAILLE PATTERN DOTS-2357BRAILLE PATTERN DOTS-12357BRAILLE PATTERN DOTS-457BRAILLE PATTERN DOTS-1457BRAILLE PATTERN DOTS-2457BRAILLE PATTERN DOTS-12457BRAILLE PATTERN DOTS-3457BRAILLE P\
    \ATTERN DOTS-13457BRAILLE PATTERN DOTS-23457BRAILLE PATTERN DOTS-123457BRAILLE PATTERN DOTS-67BRAILLE PATTERN DOTS-167BRAILLE PATTERN DOTS-267BRAILLE PATTERN DOTS-1267BRAILLE PATTERN DOTS-367BRAILLE PATTERN DOTS-1367BRAILLE PATTERN DOTS-2367BRAILLE PATTERN\
    \ DOTS-12367BRAILLE PATTERN DOTS-467BRAILLE PATTERN DOTS-1467BRAILLE PATTERN DOTS-2467BRAILLE PATTERN DOTS-12467BRAILLE PATTERN DOTS-3467BRAILLE PATTERN DOTS-13467BRAILLE PATTERN DOTS-23467BRAILLE PATTERN DOTS-123467BRAILLE PATTERN DOTS-567BRAILLE PATTERN \
    \DOTS-1567BRAILLE PATTERN DOTS-2567BRAILLE PATTERN DOTS-12567BRAILLE PATTERN DOTS-3567BRAILLE PATTERN DOTS-13567BRAILLE PATTERN DOTS-23567BRAILLE PATTERN DOTS-123567BRAILLE PATTERN DOTS-4567BRAILLE PATTERN DOTS-14567BRAILLE PATTERN DOTS-24567BRAILLE PATTER\
    \N DOTS-124567BRAILLE PATTERN DOTS-34567BRAILLE PATTERN DOTS-134567BRAILLE PATTERN DOTS-234567BRAILLE PATTERN DOTS-1234567BRAILLE PATTERN DOTS-8BRAILLE PATTERN DOTS-18BRAILLE PATTERN DOTS-28BRAILLE PATTERN DOTS-128BRAILLE PATTERN DOTS-38BRAILLE PATTERN DOT\
    \S-138BRAILLE PATTERN DOTS-238BRAILLE PATTERN DOTS-1238BRAILLE PATTERN DOTS-48BRAILLE PATTERN DOTS-148BRAILLE PATTERN DOTS-248BRAILLE PATTERN DOTS-1248BRAILLE PATTERN DOTS-348BRAILLE PATTERN DOTS-1348BRAILLE PATTERN DOTS-2348BRAILLE PATTERN DOTS-12348BRAIL\
    \LE PATTERN DOTS-58BRAILLE PATTERN DOTS-158BRAILLE PATTERN DOTS-258BRAILLE PATTERN DOTS-1258BRAILLE PATTERN DOTS-358BRAILLE PATTERN DOTS-1358BRAILLE PATTERN DOTS-2358BRAILLE PATTERN DOTS-12358BRAILLE PATTERN DOTS-458BRAILLE PATTERN DOTS-1458BRAILLE PATTERN\
    \ DOTS-2458BRAILLE PATTERN DOTS-12458BRAILLE PATTERN DOTS-3458BRAILLE PATTERN DOTS-13458BRAILLE PATTERN DOTS-23458BRAILLE PATTERN DOTS-123458BRAILLE PATTERN DOTS-68BRAILLE PATTERN DOTS-168BRAILLE PATTERN DOTS-268BRAILLE PATTERN DOTS-1268BRAILLE PATTERN DOT\
    \S-368BRAILLE PATTERN DOTS-1368BRAILLE PATTERN DOTS-2368BRAILLE PATTERN DOTS-12368BRAILLE PATTERN DOTS-468BRAILLE PATTERN DOTS-1468BRAILLE PATTERN DOTS-2468BRAILLE PATTERN DOTS-12468BRAILLE PATTERN DOTS-3468BRAILLE PATTERN DOTS-13468BRAILLE PATTERN DOTS-23\
    \468BRAILLE PATTERN DOTS-123468BRAILLE PATTERN DOTS-568BRAILLE PATTERN DOTS-1568BRAILLE PATTERN DOTS-2568BRAILLE PATTERN DOTS-12568BRAILLE PATTERN DOTS-3568BRAILLE PATTERN DOTS-13568BRAILLE PATTERN DOTS-23568BRAILLE PATTERN DOTS-123568BRAILLE PATTERN DOTS-\
    \4568BRAILLE PATTERN DOTS-14568BRAILLE PATTERN DOTS-24568BRAILLE PATTERN DOTS-124568BRAILLE PATTERN DOTS-34568BRAILLE PATTERN DOTS-134568BRAILLE PATTERN DOTS-234568BRAILLE PATTERN DOTS-1234568BRAILLE PATTERN DOTS-78BRAILLE PATTERN DOTS-178BRAILLE PATTERN D\
    \OTS-278BRAILLE PATTERN DOTS-1278BRAILLE PATTERN DOTS-378BRAILLE PATTERN DOTS-1378BRAILLE PATTERN DOTS-2378BRAILLE PATTERN DOTS-12378BRAILLE PATTERN DOTS-478BRAILLE PATTERN DOTS-1478BRAILLE PATTERN DOTS-2478BRAILLE PATTERN DOTS-12478BRAILLE PATTERN DOTS-34\
    \78BRAILLE PATTERN DOTS-13478BRAILLE PATTERN DOTS-23478BRAILLE PATTERN DOTS-123478BRAILLE PATTERN DOTS-578BRAILLE PATTERN DOTS-1578BRAILLE PATTERN DOTS-2578BRAILLE PATTERN DOTS-12578BRAILLE PATTERN DOTS-3578BRAILLE PATTERN DOTS-13578BRAILLE PATTERN DOTS-23\
    \578BRAILLE PATTERN DOTS-123578BRAILLE PATTERN DOTS-4578BRAILLE PATTERN DOTS-14578BRAILLE PATTERN DOTS-24578BRAILLE PATTERN DOTS-124578BRAILLE PATTERN DOTS-34578BRAILLE PATTERN DOTS-134578BRAILLE PATTERN DOTS-234578BRAILLE PATTERN DOTS-1234578BRAILLE PATTE\
    \RN DOTS-678BRAILLE PATTERN DOTS-1678BRAILLE PATTERN DOTS-2678BRAILLE PATTERN DOTS-12678BRAILLE PATTERN DOTS-3678BRAILLE PATTERN DOTS-13678BRAILLE PATTERN DOTS-23678BRAILLE PATTERN DOTS-123678BRAILLE PATTERN DOTS-4678BRAILLE PATTERN DOTS-14678BRAILLE PATTE\
    \RN DOTS-24678BRAILLE PATTERN DOTS-124678BRAILLE PATTERN DOTS-34678BRAILLE PATTERN DOTS-134678BRAILLE PATTERN DOTS-234678BRAILLE PATTERN DOTS-1234678BRAILLE PATTERN DOTS-5678BRAILLE PATTERN DOTS-15678BRAILLE PATTERN DOTS-25678BRAILLE PATTERN DOTS-125678BRA\
    \ILLE PATTERN DOTS-35678BRAILLE PATTERN DOTS-135678BRAILLE PATTERN DOTS-235678BRAILLE PATTERN DOTS-1235678BRAILLE PATTERN DOTS-45678BRAILLE PATTERN DOTS-145678BRAILLE PATTERN DOTS-245678BRAILLE PATTERN DOTS-1245678BRAILLE PATTERN DOTS-345678BRAILLE PATTERN\
    \ DOTS-1345678BRAILLE PATTERN DOTS-2345678BRAILLE PATTERN DOTS-12345678RIGHTWARDS TWO-HEADED ARROW WITH VERTICAL STROKERIGHTWARDS TWO-HEADED ARROW WITH DOUBLE VERTICAL STROKELEFTWARDS DOUBLE ARROW WITH VERTICAL STROKERIGHTWARDS DOUBLE ARROW WITH VERTICAL S\
    \TROKELEFT RIGHT DOUBLE ARROW WITH VERTICAL STROKERIGHTWARDS TWO-HEADED ARROW FROM BARLEFTWARDS DOUBLE ARROW FROM BARRIGHTWARDS DOUBLE ARROW FROM BARDOWNWARDS ARROW WITH HORIZONTAL STROKEUPWARDS ARROW WITH HORIZONTAL STROKEUPWARDS TRIPLE ARROWDOWNWARDS TRI\
    \PLE ARROWLEFTWARDS DOUBLE DASH ARROWRIGHTWARDS DOUBLE DASH ARROWLEFTWARDS TRIPLE DASH ARROWRIGHTWARDS TRIPLE DASH ARROWRIGHTWARDS TWO-HEADED TRIPLE DASH ARROWRIGHTWARDS ARROW WITH DOTTED STEMUPWARDS ARROW TO BARDOWNWARDS ARROW TO BARRIGHTWARDS ARROW WITH \
    \TAIL WITH VERTICAL STROKERIGHTWARDS ARROW WITH TAIL WITH DOUBLE VERTICAL STROKERIGHTWARDS TWO-HEADED ARROW WITH TAILRIGHTWARDS TWO-HEADED ARROW WITH TAIL WITH VERTICAL STROKERIGHTWARDS TWO-HEADED ARROW WITH TAIL WITH DOUBLE VERTICAL STROKELEFTWARDS ARROW-\
    \TAILRIGHTWARDS ARROW-TAILLEFTWARDS DOUBLE ARROW-TAILRIGHTWARDS DOUBLE ARROW-TAILLEFTWARDS ARROW TO BLACK DIAMONDRIGHTWARDS ARROW TO BLACK DIAMONDLEFTWARDS ARROW FROM BAR TO BLACK DIAMONDRIGHTWARDS ARROW FROM BAR TO BLACK DIAMONDNORTH WEST AND SOUTH EAST A\
    \RROWNORTH EAST AND SOUTH WEST ARROWNORTH WEST ARROW WITH HOOKNORTH EAST ARROW WITH HOOKSOUTH EAST ARROW WITH HOOKSOUTH WEST ARROW WITH HOOKNORTH WEST ARROW AND NORTH EAST ARROWNORTH EAST ARROW AND SOUTH EAST ARROWSOUTH EAST ARROW AND SOUTH WEST ARROWSOUTH\
    \ WEST ARROW AND NORTH WEST ARROWRISING DIAGONAL CROSSING FALLING DIAGONALFALLING DIAGONAL CROSSING RISING DIAGONALSOUTH EAST ARROW CROSSING NORTH EAST ARROWNORTH EAST ARROW CROSSING SOUTH EAST ARROWFALLING DIAGONAL CROSSING NORTH EAST ARROWRISING DIAGONAL\
    \ CROSSING SOUTH EAST ARROWNORTH EAST ARROW CROSSING NORTH WEST ARROWNORTH WEST ARROW CROSSING NORTH EAST ARROWWAVE ARROW POINTING DIRECTLY RIGHTARROW POINTING RIGHTWARDS THEN CURVING UPWARDSARROW POINTING RIGHTWARDS THEN CURVING DOWNWARDSARROW POINTING DO\
    \WNWARDS THEN CURVING LEFTWARDSARROW POINTING DOWNWARDS THEN CURVING RIGHTWARDSRIGHT-SIDE ARC CLOCKWISE ARROWLEFT-SIDE ARC ANTICLOCKWISE ARROWTOP ARC ANTICLOCKWISE ARROWBOTTOM ARC ANTICLOCKWISE ARROWTOP ARC CLOCKWISE ARROW WITH MINUSTOP ARC ANTICLOCKWISE A\
    \RROW WITH PLUSLOWER RIGHT SEMICIRCULAR CLOCKWISE ARROWLOWER LEFT SEMICIRCULAR ANTICLOCKWISE ARROWANTICLOCKWISE CLOSED CIRCLE ARROWCLOCKWISE CLOSED CIRCLE ARROWRIGHTWARDS ARROW ABOVE SHORT LEFTWARDS ARROWLEFTWARDS ARROW ABOVE SHORT RIGHTWARDS ARROWSHORT RI\
    \GHTWARDS ARROW ABOVE LEFTWARDS ARROWRIGHTWARDS ARROW WITH PLUS BELOWLEFTWARDS ARROW WITH PLUS BELOWRIGHTWARDS ARROW THROUGH XLEFT RIGHT ARROW THROUGH SMALL CIRCLEUPWARDS TWO-HEADED ARROW FROM SMALL CIRCLELEFT BARB UP RIGHT BARB DOWN HARPOONLEFT BARB DOWN \
    \RIGHT BARB UP HARPOONUP BARB RIGHT DOWN BARB LEFT HARPOONUP BARB LEFT DOWN BARB RIGHT HARPOONLEFT BARB UP RIGHT BARB UP HARPOONUP BARB RIGHT DOWN BARB RIGHT HARPOONLEFT BARB DOWN RIGHT BARB DOWN HARPOONUP BARB LEFT DOWN BARB LEFT HARPOONLEFTWARDS HARPOON \
    \WITH BARB UP TO BARRIGHTWARDS HARPOON WITH BARB UP TO BARUPWARDS HARPOON WITH BARB RIGHT TO BARDOWNWARDS HARPOON WITH BARB RIGHT TO BARLEFTWARDS HARPOON WITH BARB DOWN TO BARRIGHTWARDS HARPOON WITH BARB DOWN TO BARUPWARDS HARPOON WITH BARB LEFT TO BARDOWN\
    \WARDS HARPOON WITH BARB LEFT TO BARLEFTWARDS HARPOON WITH BARB UP FROM BARRIGHTWARDS HARPOON WITH BARB UP FROM BARUPWARDS HARPOON WITH BARB RIGHT FROM BARDOWNWARDS HARPOON WITH BARB RIGHT FROM BARLEFTWARDS HARPOON WITH BARB DOWN FROM BARRIGHTWARDS HARPOON\
    \ WITH BARB DOWN FROM BARUPWARDS HARPOON WITH BARB LEFT FROM BARDOWNWARDS HARPOON WITH BARB LEFT FROM BARLEFTWARDS HARPOON WITH BARB UP ABOVE LEFTWARDS HARPOON WITH BARB DOWNUPWARDS HARPOON WITH BARB LEFT BESIDE UPWARDS HARPOON WITH BARB RIGHTRIGHTWARDS HA\
    \RPOON WITH BARB UP ABOVE RIGHTWARDS HARPOON WITH BARB DOWNDOWNWARDS HARPOON WITH BARB LEFT BESIDE DOWNWARDS HARPOON WITH BARB RIGHTLEFTWARDS HARPOON WITH BARB UP ABOVE RIGHTWARDS HARPOON WITH BARB UPLEFTWARDS HARPOON WITH BARB DOWN ABOVE RIGHTWARDS HARPOO\
    \N WITH BARB DOWNRIGHTWARDS HARPOON WITH BARB UP ABOVE LEFTWARDS HARPOON WITH BARB UPRIGHTWARDS HARPOON WITH BARB DOWN ABOVE LEFTWARDS HARPOON WITH BARB DOWNLEFTWARDS HARPOON WITH BARB UP ABOVE LONG DASHLEFTWARDS HARPOON WITH BARB DOWN BELOW LONG DASHRIGHT\
    \WARDS HARPOON WITH BARB UP ABOVE LONG DASHRIGHTWARDS HARPOON WITH BARB DOWN BELOW LONG DASHUPWARDS HARPOON WITH BARB LEFT BESIDE DOWNWARDS HARPOON WITH BARB RIGHTDOWNWARDS HARPOON WITH BARB LEFT BESIDE UPWARDS HARPOON WITH BARB RIGHTRIGHT DOUBLE ARROW WIT\
    \H ROUNDED HEADEQUALS SIGN ABOVE RIGHTWARDS ARROWTILDE OPERATOR ABOVE RIGHTWARDS ARROWLEFTWARDS ARROW ABOVE TILDE OPERATORRIGHTWARDS ARROW ABOVE TILDE OPERATORRIGHTWARDS ARROW ABOVE ALMOST EQUAL TOLESS-THAN ABOVE LEFTWARDS ARROWLEFTWARDS ARROW THROUGH LESS\
    \-THANGREATER-THAN ABOVE RIGHTWARDS ARROWSUBSET ABOVE RIGHTWARDS ARROWLEFTWARDS ARROW THROUGH SUBSETSUPERSET ABOVE LEFTWARDS ARROWLEFT FISH TAILRIGHT FISH TAILUP FISH TAILDOWN FISH TAILTRIPLE VERTICAL BAR DELIMITERZ NOTATION SPOTZ NOTATION TYPE COLONLEFT W\
    \HITE CURLY BRACKETRIGHT WHITE CURLY BRACKETLEFT WHITE PARENTHESISRIGHT WHITE PARENTHESISZ NOTATION LEFT IMAGE BRACKETZ NOTATION RIGHT IMAGE BRACKETZ NOTATION LEFT BINDING BRACKETZ NOTATION RIGHT BINDING BRACKETLEFT SQUARE BRACKET WITH UNDERBARRIGHT SQUARE\
    \ BRACKET WITH UNDERBARLEFT SQUARE BRACKET WITH TICK IN TOP CORNERRIGHT SQUARE BRACKET WITH TICK IN BOTTOM CORNERLEFT SQUARE BRACKET WITH TICK IN BOTTOM CORNERRIGHT SQUARE BRACKET WITH TICK IN TOP CORNERLEFT ANGLE BRACKET WITH DOTRIGHT ANGLE BRACKET WITH D\
    \OTLEFT ARC LESS-THAN BRACKETRIGHT ARC GREATER-THAN BRACKETDOUBLE LEFT ARC GREATER-THAN BRACKETDOUBLE RIGHT ARC LESS-THAN BRACKETLEFT BLACK TORTOISE SHELL BRACKETRIGHT BLACK TORTOISE SHELL BRACKETDOTTED FENCEVERTICAL ZIGZAG LINEMEASURED ANGLE OPENING LEFTR\
    \IGHT ANGLE VARIANT WITH SQUAREMEASURED RIGHT ANGLE WITH DOTANGLE WITH S INSIDEACUTE ANGLESPHERICAL ANGLE OPENING LEFTSPHERICAL ANGLE OPENING UPTURNED ANGLEREVERSED ANGLEANGLE WITH UNDERBARREVERSED ANGLE WITH UNDERBAROBLIQUE ANGLE OPENING UPOBLIQUE ANGLE O\
    \PENING DOWNMEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING UP AND RIGHTMEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING UP AND LEFTMEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING DOWN AND RIGHTMEASURED ANGLE WITH OPEN ARM ENDING IN ARROW \
    \POINTING DOWN AND LEFTMEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING RIGHT AND UPMEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING LEFT AND UPMEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING RIGHT AND DOWNMEASURED ANGLE WITH OPEN ARM ENDIN\
    \G IN ARROW POINTING LEFT AND DOWNREVERSED EMPTY SETEMPTY SET WITH OVERBAREMPTY SET WITH SMALL CIRCLE ABOVEEMPTY SET WITH RIGHT ARROW ABOVEEMPTY SET WITH LEFT ARROW ABOVECIRCLE WITH HORIZONTAL BARCIRCLED VERTICAL BARCIRCLED PARALLELCIRCLED REVERSE SOLIDUSC\
    \IRCLED PERPENDICULARCIRCLE DIVIDED BY HORIZONTAL BAR AND TOP HALF DIVIDED BY VERTICAL BARCIRCLE WITH SUPERIMPOSED XCIRCLED ANTICLOCKWISE-ROTATED DIVISION SIGNUP ARROW THROUGH CIRCLECIRCLED WHITE BULLETCIRCLED BULLETCIRCLED LESS-THANCIRCLED GREATER-THANCIR\
    \CLE WITH SMALL CIRCLE TO THE RIGHTCIRCLE WITH TWO HORIZONTAL STROKES TO THE RIGHTSQUARED RISING DIAGONAL SLASHSQUARED FALLING DIAGONAL SLASHSQUARED ASTERISKSQUARED SMALL CIRCLESQUARED SQUARETWO JOINED SQUARESTRIANGLE WITH DOT ABOVETRIANGLE WITH UNDERBARS \
    \IN TRIANGLETRIANGLE WITH SERIFS AT BOTTOMRIGHT TRIANGLE ABOVE LEFT TRIANGLELEFT TRIANGLE BESIDE VERTICAL BARVERTICAL BAR BESIDE RIGHT TRIANGLEBOWTIE WITH LEFT HALF BLACKBOWTIE WITH RIGHT HALF BLACKBLACK BOWTIETIMES WITH LEFT HALF BLACKTIMES WITH RIGHT HAL\
    \F BLACKWHITE HOURGLASSBLACK HOURGLASSLEFT WIGGLY FENCERIGHT WIGGLY FENCELEFT DOUBLE WIGGLY FENCERIGHT DOUBLE WIGGLY FENCEINCOMPLETE INFINITYTIE OVER INFINITYINFINITY NEGATED WITH VERTICAL BARDOUBLE-ENDED MULTIMAPSQUARE WITH CONTOURED OUTLINEINCREASES ASSH\
    \UFFLE PRODUCTEQUALS SIGN AND SLANTED PARALLELEQUALS SIGN AND SLANTED PARALLEL WITH TILDE ABOVEIDENTICAL TO AND SLANTED PARALLELGLEICH STARKTHERMODYNAMICDOWN-POINTING TRIANGLE WITH LEFT HALF BLACKDOWN-POINTING TRIANGLE WITH RIGHT HALF BLACKBLACK DIAMOND WI\
    \TH DOWN ARROWBLACK LOZENGEWHITE CIRCLE WITH DOWN ARROWBLACK CIRCLE WITH DOWN ARROWERROR-BARRED WHITE SQUAREERROR-BARRED BLACK SQUAREERROR-BARRED WHITE DIAMONDERROR-BARRED BLACK DIAMONDERROR-BARRED WHITE CIRCLEERROR-BARRED BLACK CIRCLERULE-DELAYEDREVERSE S\
    \OLIDUS OPERATORSOLIDUS WITH OVERBARREVERSE SOLIDUS WITH HORIZONTAL STROKEBIG SOLIDUSBIG REVERSE SOLIDUSDOUBLE PLUSTRIPLE PLUSLEFT-POINTING CURVED ANGLE BRACKETRIGHT-POINTING CURVED ANGLE BRACKETTINYMINYN-ARY CIRCLED DOT OPERATORN-ARY CIRCLED PLUS OPERATOR\
    \N-ARY CIRCLED TIMES OPERATORN-ARY UNION OPERATOR WITH DOTN-ARY UNION OPERATOR WITH PLUSN-ARY SQUARE INTERSECTION OPERATORN-ARY SQUARE UNION OPERATORTWO LOGICAL AND OPERATORTWO LOGICAL OR OPERATORN-ARY TIMES OPERATORMODULO TWO SUMSUMMATION WITH INTEGRALQUA\
    \DRUPLE INTEGRAL OPERATORFINITE PART INTEGRALINTEGRAL WITH DOUBLE STROKEINTEGRAL AVERAGE WITH SLASHCIRCULATION FUNCTIONANTICLOCKWISE INTEGRATIONLINE INTEGRATION WITH RECTANGULAR PATH AROUND POLELINE INTEGRATION WITH SEMICIRCULAR PATH AROUND POLELINE INTEGR\
    \ATION NOT INCLUDING THE POLEINTEGRAL AROUND A POINT OPERATORQUATERNION INTEGRAL OPERATORINTEGRAL WITH LEFTWARDS ARROW WITH HOOKINTEGRAL WITH TIMES SIGNINTEGRAL WITH INTERSECTIONINTEGRAL WITH UNIONINTEGRAL WITH OVERBARINTEGRAL WITH UNDERBARJOINLARGE LEFT T\
    \RIANGLE OPERATORZ NOTATION SCHEMA COMPOSITIONZ NOTATION SCHEMA PIPINGZ NOTATION SCHEMA PROJECTIONPLUS SIGN WITH SMALL CIRCLE ABOVEPLUS SIGN WITH CIRCUMFLEX ACCENT ABOVEPLUS SIGN WITH TILDE ABOVEPLUS SIGN WITH DOT BELOWPLUS SIGN WITH TILDE BELOWPLUS SIGN W\
    \ITH SUBSCRIPT TWOPLUS SIGN WITH BLACK TRIANGLEMINUS SIGN WITH COMMA ABOVEMINUS SIGN WITH DOT BELOWMINUS SIGN WITH FALLING DOTSMINUS SIGN WITH RISING DOTSPLUS SIGN IN LEFT HALF CIRCLEPLUS SIGN IN RIGHT HALF CIRCLEVECTOR OR CROSS PRODUCTMULTIPLICATION SIGN \
    \WITH DOT ABOVEMULTIPLICATION SIGN WITH UNDERBARSEMIDIRECT PRODUCT WITH BOTTOM CLOSEDSMASH PRODUCTMULTIPLICATION SIGN IN LEFT HALF CIRCLEMULTIPLICATION SIGN IN RIGHT HALF CIRCLECIRCLED MULTIPLICATION SIGN WITH CIRCUMFLEX ACCENTMULTIPLICATION SIGN IN DOUBLE\
    \ CIRCLECIRCLED DIVISION SIGNPLUS SIGN IN TRIANGLEMINUS SIGN IN TRIANGLEMULTIPLICATION SIGN IN TRIANGLEINTERIOR PRODUCTRIGHTHAND INTERIOR PRODUCTZ NOTATION RELATIONAL COMPOSITIONAMALGAMATION OR COPRODUCTINTERSECTION WITH DOTUNION WITH MINUS SIGNUNION WITH \
    \OVERBARINTERSECTION WITH OVERBARINTERSECTION WITH LOGICAL ANDUNION WITH LOGICAL ORUNION ABOVE INTERSECTIONINTERSECTION ABOVE UNIONUNION ABOVE BAR ABOVE INTERSECTIONINTERSECTION ABOVE BAR ABOVE UNIONUNION BESIDE AND JOINED WITH UNIONINTERSECTION BESIDE AND\
    \ JOINED WITH INTERSECTIONCLOSED UNION WITH SERIFSCLOSED INTERSECTION WITH SERIFSDOUBLE SQUARE INTERSECTIONDOUBLE SQUARE UNIONCLOSED UNION WITH SERIFS AND SMASH PRODUCTLOGICAL AND WITH DOT ABOVELOGICAL OR WITH DOT ABOVEDOUBLE LOGICAL ANDDOUBLE LOGICAL ORTW\
    \O INTERSECTING LOGICAL ANDTWO INTERSECTING LOGICAL ORSLOPING LARGE ORSLOPING LARGE ANDLOGICAL OR OVERLAPPING LOGICAL ANDLOGICAL AND WITH MIDDLE STEMLOGICAL OR WITH MIDDLE STEMLOGICAL AND WITH HORIZONTAL DASHLOGICAL OR WITH HORIZONTAL DASHLOGICAL AND WITH \
    \DOUBLE OVERBARLOGICAL AND WITH UNDERBARLOGICAL AND WITH DOUBLE UNDERBARSMALL VEE WITH UNDERBARLOGICAL OR WITH DOUBLE OVERBARLOGICAL OR WITH DOUBLE UNDERBARZ NOTATION DOMAIN ANTIRESTRICTIONZ NOTATION RANGE ANTIRESTRICTIONEQUALS SIGN WITH DOT BELOWIDENTICAL\
    \ WITH DOT ABOVETRIPLE HORIZONTAL BAR WITH DOUBLE VERTICAL STROKETRIPLE HORIZONTAL BAR WITH TRIPLE VERTICAL STROKETILDE OPERATOR WITH DOT ABOVETILDE OPERATOR WITH RISING DOTSSIMILAR MINUS SIMILARCONGRUENT WITH DOT ABOVEEQUALS WITH ASTERISKALMOST EQUAL TO W\
    \ITH CIRCUMFLEX ACCENTAPPROXIMATELY EQUAL OR EQUAL TOEQUALS SIGN ABOVE PLUS SIGNPLUS SIGN ABOVE EQUALS SIGNEQUALS SIGN ABOVE TILDE OPERATORDOUBLE COLON EQUALTWO CONSECUTIVE EQUALS SIGNSTHREE CONSECUTIVE EQUALS SIGNSEQUALS SIGN WITH TWO DOTS ABOVE AND TWO D\
    \OTS BELOWEQUIVALENT WITH FOUR DOTS ABOVELESS-THAN WITH CIRCLE INSIDEGREATER-THAN WITH CIRCLE INSIDELESS-THAN WITH QUESTION MARK ABOVEGREATER-THAN WITH QUESTION MARK ABOVELESS-THAN OR SLANTED EQUAL TOGREATER-THAN OR SLANTED EQUAL TOLESS-THAN OR SLANTED EQU\
    \AL TO WITH DOT INSIDEGREATER-THAN OR SLANTED EQUAL TO WITH DOT INSIDELESS-THAN OR SLANTED EQUAL TO WITH DOT ABOVEGREATER-THAN OR SLANTED EQUAL TO WITH DOT ABOVELESS-THAN OR SLANTED EQUAL TO WITH DOT ABOVE RIGHTGREATER-THAN OR SLANTED EQUAL TO WITH DOT ABO\
    \VE LEFTLESS-THAN OR APPROXIMATEGREATER-THAN OR APPROXIMATELESS-THAN AND SINGLE-LINE NOT EQUAL TOGREATER-THAN AND SINGLE-LINE NOT EQUAL TOLESS-THAN AND NOT APPROXIMATEGREATER-THAN AND NOT APPROXIMATELESS-THAN ABOVE DOUBLE-LINE EQUAL ABOVE GREATER-THANGREAT\
    \ER-THAN ABOVE DOUBLE-LINE EQUAL ABOVE LESS-THANLESS-THAN ABOVE SIMILAR OR EQUALGREATER-THAN ABOVE SIMILAR OR EQUALLESS-THAN ABOVE SIMILAR ABOVE GREATER-THANGREATER-THAN ABOVE SIMILAR ABOVE LESS-THANLESS-THAN ABOVE GREATER-THAN ABOVE DOUBLE-LINE EQUALGREAT\
    \ER-THAN ABOVE LESS-THAN ABOVE DOUBLE-LINE EQUALLESS-THAN ABOVE SLANTED EQUAL ABOVE GREATER-THAN ABOVE SLANTED EQUALGREATER-THAN ABOVE SLANTED EQUAL ABOVE LESS-THAN ABOVE SLANTED EQUALSLANTED EQUAL TO OR LESS-THANSLANTED EQUAL TO OR GREATER-THANSLANTED EQU\
    \AL TO OR LESS-THAN WITH DOT INSIDESLANTED EQUAL TO OR GREATER-THAN WITH DOT INSIDEDOUBLE-LINE EQUAL TO OR LESS-THANDOUBLE-LINE EQUAL TO OR GREATER-THANDOUBLE-LINE SLANTED EQUAL TO OR LESS-THANDOUBLE-LINE SLANTED EQUAL TO OR GREATER-THANSIMILAR OR LESS-THA\
    \NSIMILAR OR GREATER-THANSIMILAR ABOVE LESS-THAN ABOVE EQUALS SIGNSIMILAR ABOVE GREATER-THAN ABOVE EQUALS SIGNDOUBLE NESTED LESS-THANDOUBLE NESTED GREATER-THANDOUBLE NESTED LESS-THAN WITH UNDERBARGREATER-THAN OVERLAPPING LESS-THANGREATER-THAN BESIDE LESS-T\
    \HANLESS-THAN CLOSED BY CURVEGREATER-THAN CLOSED BY CURVELESS-THAN CLOSED BY CURVE ABOVE SLANTED EQUALGREATER-THAN CLOSED BY CURVE ABOVE SLANTED EQUALSMALLER THANLARGER THANSMALLER THAN OR EQUAL TOLARGER THAN OR EQUAL TOEQUALS SIGN WITH BUMPY ABOVEPRECEDES\
    \ ABOVE SINGLE-LINE EQUALS SIGNSUCCEEDS ABOVE SINGLE-LINE EQUALS SIGNPRECEDES ABOVE SINGLE-LINE NOT EQUAL TOSUCCEEDS ABOVE SINGLE-LINE NOT EQUAL TOPRECEDES ABOVE EQUALS SIGNSUCCEEDS ABOVE EQUALS SIGNPRECEDES ABOVE NOT EQUAL TOSUCCEEDS ABOVE NOT EQUAL TOPRE\
    \CEDES ABOVE ALMOST EQUAL TOSUCCEEDS ABOVE ALMOST EQUAL TOPRECEDES ABOVE NOT ALMOST EQUAL TOSUCCEEDS ABOVE NOT ALMOST EQUAL TODOUBLE PRECEDESDOUBLE SUCCEEDSSUBSET WITH DOTSUPERSET WITH DOTSUBSET WITH PLUS SIGN BELOWSUPERSET WITH PLUS SIGN BELOWSUBSET WITH \
    \MULTIPLICATION SIGN BELOWSUPERSET WITH MULTIPLICATION SIGN BELOWSUBSET OF OR EQUAL TO WITH DOT ABOVESUPERSET OF OR EQUAL TO WITH DOT ABOVESUBSET OF ABOVE EQUALS SIGNSUPERSET OF ABOVE EQUALS SIGNSUBSET OF ABOVE TILDE OPERATORSUPERSET OF ABOVE TILDE OPERATO\
    \RSUBSET OF ABOVE ALMOST EQUAL TOSUPERSET OF ABOVE ALMOST EQUAL TOSUBSET OF ABOVE NOT EQUAL TOSUPERSET OF ABOVE NOT EQUAL TOSQUARE LEFT OPEN BOX OPERATORSQUARE RIGHT OPEN BOX OPERATORCLOSED SUBSETCLOSED SUPERSETCLOSED SUBSET OR EQUAL TOCLOSED SUPERSET OR E\
    \QUAL TOSUBSET ABOVE SUPERSETSUPERSET ABOVE SUBSETSUBSET ABOVE SUBSETSUPERSET ABOVE SUPERSETSUPERSET BESIDE SUBSETSUPERSET BESIDE AND JOINED BY DASH WITH SUBSETELEMENT OF OPENING DOWNWARDSPITCHFORK WITH TEE TOPTRANSVERSAL INTERSECTIONFORKINGNONFORKINGSHORT\
    \ LEFT TACKSHORT DOWN TACKSHORT UP TACKPERPENDICULAR WITH SVERTICAL BAR TRIPLE RIGHT TURNSTILEDOUBLE VERTICAL BAR LEFT TURNSTILEVERTICAL BAR DOUBLE LEFT TURNSTILEDOUBLE VERTICAL BAR DOUBLE LEFT TURNSTILELONG DASH FROM LEFT MEMBER OF DOUBLE VERTICALSHORT DO\
    \WN TACK WITH OVERBARSHORT UP TACK WITH UNDERBARSHORT UP TACK ABOVE SHORT DOWN TACKDOUBLE DOWN TACKDOUBLE UP TACKDOUBLE STROKE NOT SIGNREVERSED DOUBLE STROKE NOT SIGNDOES NOT DIVIDE WITH REVERSED NEGATION SLASHVERTICAL LINE WITH CIRCLE ABOVEVERTICAL LINE W\
    \ITH CIRCLE BELOWDOWN TACK WITH CIRCLE BELOWPARALLEL WITH HORIZONTAL STROKEPARALLEL WITH TILDE OPERATORTRIPLE VERTICAL BAR BINARY RELATIONTRIPLE VERTICAL BAR WITH HORIZONTAL STROKETRIPLE COLON OPERATORTRIPLE NESTED LESS-THANTRIPLE NESTED GREATER-THANDOUBLE\
    \-LINE SLANTED LESS-THAN OR EQUAL TODOUBLE-LINE SLANTED GREATER-THAN OR EQUAL TOTRIPLE SOLIDUS BINARY RELATIONLARGE TRIPLE VERTICAL BAR OPERATORDOUBLE SOLIDUS OPERATORWHITE VERTICAL BARN-ARY WHITE VERTICAL BARNORTH EAST WHITE ARROWNORTH WEST WHITE ARROWSOU\
    \TH EAST WHITE ARROWSOUTH WEST WHITE ARROWLEFT RIGHT WHITE ARROWLEFTWARDS BLACK ARROWUPWARDS BLACK ARROWDOWNWARDS BLACK ARROWNORTH EAST BLACK ARROWNORTH WEST BLACK ARROWSOUTH EAST BLACK ARROWSOUTH WEST BLACK ARROWLEFT RIGHT BLACK ARROWUP DOWN BLACK ARROWRI\
    \GHTWARDS ARROW WITH TIP DOWNWARDSRIGHTWARDS ARROW WITH TIP UPWARDSLEFTWARDS ARROW WITH TIP DOWNWARDSLEFTWARDS ARROW WITH TIP UPWARDSSQUARE WITH TOP HALF BLACKSQUARE WITH BOTTOM HALF BLACKSQUARE WITH UPPER RIGHT DIAGONAL HALF BLACKSQUARE WITH LOWER LEFT DI\
    \AGONAL HALF BLACKDIAMOND WITH LEFT HALF BLACKDIAMOND WITH RIGHT HALF BLACKDIAMOND WITH TOP HALF BLACKDIAMOND WITH BOTTOM HALF BLACKDOTTED SQUAREBLACK LARGE SQUAREWHITE LARGE SQUAREBLACK VERY SMALL SQUAREWHITE VERY SMALL SQUAREBLACK PENTAGONWHITE PENTAGONW\
    \HITE HEXAGONBLACK HEXAGONHORIZONTAL BLACK HEXAGONBLACK LARGE CIRCLEBLACK MEDIUM DIAMONDWHITE MEDIUM DIAMONDBLACK MEDIUM LOZENGEWHITE MEDIUM LOZENGEBLACK SMALL DIAMONDBLACK SMALL LOZENGEWHITE SMALL LOZENGEBLACK HORIZONTAL ELLIPSEWHITE HORIZONTAL ELLIPSEBLA\
    \CK VERTICAL ELLIPSEWHITE VERTICAL ELLIPSELEFT ARROW WITH SMALL CIRCLETHREE LEFTWARDS ARROWSLEFT ARROW WITH CIRCLED PLUSLONG LEFTWARDS SQUIGGLE ARROWLEFTWARDS TWO-HEADED ARROW WITH VERTICAL STROKELEFTWARDS TWO-HEADED ARROW WITH DOUBLE VERTICAL STROKELEFTWA\
    \RDS TWO-HEADED ARROW FROM BARLEFTWARDS TWO-HEADED TRIPLE DASH ARROWLEFTWARDS ARROW WITH DOTTED STEMLEFTWARDS ARROW WITH TAIL WITH VERTICAL STROKELEFTWARDS ARROW WITH TAIL WITH DOUBLE VERTICAL STROKELEFTWARDS TWO-HEADED ARROW WITH TAILLEFTWARDS TWO-HEADED \
    \ARROW WITH TAIL WITH VERTICAL STROKELEFTWARDS TWO-HEADED ARROW WITH TAIL WITH DOUBLE VERTICAL STROKELEFTWARDS ARROW THROUGH XWAVE ARROW POINTING DIRECTLY LEFTEQUALS SIGN ABOVE LEFTWARDS ARROWREVERSE TILDE OPERATOR ABOVE LEFTWARDS ARROWLEFTWARDS ARROW ABOV\
    \E REVERSE ALMOST EQUAL TORIGHTWARDS ARROW THROUGH GREATER-THANRIGHTWARDS ARROW THROUGH SUPERSETLEFTWARDS QUADRUPLE ARROWRIGHTWARDS QUADRUPLE ARROWREVERSE TILDE OPERATOR ABOVE RIGHTWARDS ARROWRIGHTWARDS ARROW ABOVE REVERSE ALMOST EQUAL TOTILDE OPERATOR ABO\
    \VE LEFTWARDS ARROWLEFTWARDS ARROW ABOVE ALMOST EQUAL TOLEFTWARDS ARROW ABOVE REVERSE TILDE OPERATORRIGHTWARDS ARROW ABOVE REVERSE TILDE OPERATORDOWNWARDS TRIANGLE-HEADED ZIGZAG ARROWSHORT SLANTED NORTH ARROWSHORT BACKSLANTED SOUTH ARROWWHITE MEDIUM STARBL\
    \ACK SMALL STARWHITE SMALL STARBLACK RIGHT-POINTING PENTAGONWHITE RIGHT-POINTING PENTAGONHEAVY LARGE CIRCLEHEAVY OVAL WITH OVAL INSIDEHEAVY CIRCLE WITH CIRCLE INSIDEHEAVY CIRCLEHEAVY CIRCLED SALTIRESLANTED NORTH ARROW WITH HOOKED HEADBACKSLANTED SOUTH ARRO\
    \W WITH HOOKED TAILSLANTED NORTH ARROW WITH HORIZONTAL TAILBACKSLANTED SOUTH ARROW WITH HORIZONTAL TAILBENT ARROW POINTING DOWNWARDS THEN NORTH EASTSHORT BENT ARROW POINTING DOWNWARDS THEN NORTH EASTLEFTWARDS TRIANGLE-HEADED ARROWUPWARDS TRIANGLE-HEADED AR\
    \ROWRIGHTWARDS TRIANGLE-HEADED ARROWDOWNWARDS TRIANGLE-HEADED ARROWLEFT RIGHT TRIANGLE-HEADED ARROWUP DOWN TRIANGLE-HEADED ARROWNORTH WEST TRIANGLE-HEADED ARROWNORTH EAST TRIANGLE-HEADED ARROWSOUTH EAST TRIANGLE-HEADED ARROWSOUTH WEST TRIANGLE-HEADED ARROW\
    \LEFTWARDS TRIANGLE-HEADED DASHED ARROWUPWARDS TRIANGLE-HEADED DASHED ARROWRIGHTWARDS TRIANGLE-HEADED DASHED ARROWDOWNWARDS TRIANGLE-HEADED DASHED ARROWCLOCKWISE TRIANGLE-HEADED OPEN CIRCLE ARROWANTICLOCKWISE TRIANGLE-HEADED OPEN CIRCLE ARROWLEFTWARDS TRIA\
    \NGLE-HEADED ARROW TO BARUPWARDS TRIANGLE-HEADED ARROW TO BARRIGHTWARDS TRIANGLE-HEADED ARROW TO BARDOWNWARDS TRIANGLE-HEADED ARROW TO BARNORTH WEST TRIANGLE-HEADED ARROW TO BARNORTH EAST TRIANGLE-HEADED ARROW TO BARSOUTH EAST TRIANGLE-HEADED ARROW TO BARS\
    \OUTH WEST TRIANGLE-HEADED ARROW TO BARLEFTWARDS TRIANGLE-HEADED ARROW WITH DOUBLE HORIZONTAL STROKEUPWARDS TRIANGLE-HEADED ARROW WITH DOUBLE HORIZONTAL STROKERIGHTWARDS TRIANGLE-HEADED ARROW WITH DOUBLE HORIZONTAL STROKEDOWNWARDS TRIANGLE-HEADED ARROW WIT\
    \H DOUBLE HORIZONTAL STROKEHORIZONTAL TAB KEYVERTICAL TAB KEYLEFTWARDS TRIANGLE-HEADED ARROW OVER RIGHTWARDS TRIANGLE-HEADED ARROWUPWARDS TRIANGLE-HEADED ARROW LEFTWARDS OF DOWNWARDS TRIANGLE-HEADED ARROWRIGHTWARDS TRIANGLE-HEADED ARROW OVER LEFTWARDS TRIA\
    \NGLE-HEADED ARROWDOWNWARDS TRIANGLE-HEADED ARROW LEFTWARDS OF UPWARDS TRIANGLE-HEADED ARROWLEFTWARDS TRIANGLE-HEADED PAIRED ARROWSUPWARDS TRIANGLE-HEADED PAIRED ARROWSRIGHTWARDS TRIANGLE-HEADED PAIRED ARROWSDOWNWARDS TRIANGLE-HEADED PAIRED ARROWSLEFTWARDS\
    \ BLACK CIRCLED WHITE ARROWUPWARDS BLACK CIRCLED WHITE ARROWRIGHTWARDS BLACK CIRCLED WHITE ARROWDOWNWARDS BLACK CIRCLED WHITE ARROWANTICLOCKWISE TRIANGLE-HEADED RIGHT U-SHAPED ARROWANTICLOCKWISE TRIANGLE-HEADED BOTTOM U-SHAPED ARROWANTICLOCKWISE TRIANGLE-H\
    \EADED LEFT U-SHAPED ARROWANTICLOCKWISE TRIANGLE-HEADED TOP U-SHAPED ARROWRETURN LEFTRETURN RIGHTNEWLINE LEFTNEWLINE RIGHTFOUR CORNER ARROWS CIRCLING ANTICLOCKWISERIGHTWARDS BLACK ARROWSYMBOL FOR TYPE A ELECTRONICSTHREE-D TOP-LIGHTED LEFTWARDS EQUILATERAL \
    \ARROWHEADTHREE-D RIGHT-LIGHTED UPWARDS EQUILATERAL ARROWHEADTHREE-D TOP-LIGHTED RIGHTWARDS EQUILATERAL ARROWHEADTHREE-D LEFT-LIGHTED DOWNWARDS EQUILATERAL ARROWHEADBLACK LEFTWARDS EQUILATERAL ARROWHEADBLACK UPWARDS EQUILATERAL ARROWHEADBLACK RIGHTWARDS EQ\
    \UILATERAL ARROWHEADBLACK DOWNWARDS EQUILATERAL ARROWHEADDOWNWARDS TRIANGLE-HEADED ARROW WITH LONG TIP LEFTWARDSDOWNWARDS TRIANGLE-HEADED ARROW WITH LONG TIP RIGHTWARDSUPWARDS TRIANGLE-HEADED ARROW WITH LONG TIP LEFTWARDSUPWARDS TRIANGLE-HEADED ARROW WITH \
    \LONG TIP RIGHTWARDSLEFTWARDS TRIANGLE-HEADED ARROW WITH LONG TIP UPWARDSRIGHTWARDS TRIANGLE-HEADED ARROW WITH LONG TIP UPWARDSLEFTWARDS TRIANGLE-HEADED ARROW WITH LONG TIP DOWNWARDSRIGHTWARDS TRIANGLE-HEADED ARROW WITH LONG TIP DOWNWARDSBLACK CURVED DOWNW\
    \ARDS AND LEFTWARDS ARROWBLACK CURVED DOWNWARDS AND RIGHTWARDS ARROWBLACK CURVED UPWARDS AND LEFTWARDS ARROWBLACK CURVED UPWARDS AND RIGHTWARDS ARROWBLACK CURVED LEFTWARDS AND UPWARDS ARROWBLACK CURVED RIGHTWARDS AND UPWARDS ARROWBLACK CURVED LEFTWARDS AND\
    \ DOWNWARDS ARROWBLACK CURVED RIGHTWARDS AND DOWNWARDS ARROWRIBBON ARROW DOWN LEFTRIBBON ARROW DOWN RIGHTRIBBON ARROW UP LEFTRIBBON ARROW UP RIGHTRIBBON ARROW LEFT UPRIBBON ARROW RIGHT UPRIBBON ARROW LEFT DOWNRIBBON ARROW RIGHT DOWNUPWARDS WHITE ARROW FROM\
    \ BAR WITH HORIZONTAL BARUP ARROWHEAD IN A RECTANGLE BOXOVERLAPPING WHITE SQUARESOVERLAPPING WHITE AND BLACK SQUARESOVERLAPPING BLACK SQUARESBALLOT BOX WITH LIGHT XCIRCLED XCIRCLED BOLD XBLACK SQUARE CENTREDBLACK DIAMOND CENTREDTURNED BLACK PENTAGONHORIZON\
    \TAL BLACK OCTAGONBLACK OCTAGONBLACK MEDIUM UP-POINTING TRIANGLE CENTREDBLACK MEDIUM DOWN-POINTING TRIANGLE CENTREDBLACK MEDIUM LEFT-POINTING TRIANGLE CENTREDBLACK MEDIUM RIGHT-POINTING TRIANGLE CENTREDNEPTUNE FORM TWOTOP HALF BLACK CIRCLEBOTTOM HALF BLACK\
    \ CIRCLELIGHT FOUR POINTED BLACK CUSPROTATED LIGHT FOUR POINTED BLACK CUSPWHITE FOUR POINTED CUSPROTATED WHITE FOUR POINTED CUSPSQUARE POSITION INDICATORUNCERTAINTY SIGNGROUP MARKPLUTO FORM TWOPLUTO FORM THREEPLUTO FORM FOURPLUTO FORM FIVETRANSPLUTOPROSERP\
    \INAASTRAEAHYGIEAPHOLUSNESSUSWHITE MOON SELENABLACK DIAMOND ON CROSSTRUE LIGHT MOON ARTACUPIDOHADESZEUSKRONOSAPOLLONADMETOSVULCANUSPOSEIDONLEFT HALF BLACK STARRIGHT HALF BLACK STARSTAR WITH LEFT HALF BLACKSTAR WITH RIGHT HALF BLACKLEFTWARDS TWO-HEADED ARRO\
    \W WITH TRIANGLE ARROWHEADSUPWARDS TWO-HEADED ARROW WITH TRIANGLE ARROWHEADSRIGHTWARDS TWO-HEADED ARROW WITH TRIANGLE ARROWHEADSDOWNWARDS TWO-HEADED ARROW WITH TRIANGLE ARROWHEADSERIS FORM ONEERIS FORM TWOSEDNARUSSIAN ASTROLOGICAL SYMBOL VIGINTILERUSSIAN A\
    \STROLOGICAL SYMBOL NOVILERUSSIAN ASTROLOGICAL SYMBOL QUINTILERUSSIAN ASTROLOGICAL SYMBOL BINOVILERUSSIAN ASTROLOGICAL SYMBOL SENTAGONRUSSIAN ASTROLOGICAL SYMBOL TREDECILEEQUALS SIGN WITH INFINITY BELOWUNITED SYMBOLSEPARATED SYMBOLDOUBLED SYMBOLPASSED SYMB\
    \OLREVERSED RIGHT ANGLEHELLSCHREIBER PAUSE SYMBOLGLAGOLITIC CAPITAL LETTER AZUGLAGOLITIC CAPITAL LETTER BUKYGLAGOLITIC CAPITAL LETTER VEDEGLAGOLITIC CAPITAL LETTER GLAGOLIGLAGOLITIC CAPITAL LETTER DOBROGLAGOLITIC CAPITAL LETTER YESTUGLAGOLITIC CAPITAL LETT\
    \ER ZHIVETEGLAGOLITIC CAPITAL LETTER DZELOGLAGOLITIC CAPITAL LETTER ZEMLJAGLAGOLITIC CAPITAL LETTER IZHEGLAGOLITIC CAPITAL LETTER INITIAL IZHEGLAGOLITIC CAPITAL LETTER IGLAGOLITIC CAPITAL LETTER DJERVIGLAGOLITIC CAPITAL LETTER KAKOGLAGOLITIC CAPITAL LETTER\
    \ LJUDIJEGLAGOLITIC CAPITAL LETTER MYSLITEGLAGOLITIC CAPITAL LETTER NASHIGLAGOLITIC CAPITAL LETTER ONUGLAGOLITIC CAPITAL LETTER POKOJIGLAGOLITIC CAPITAL LETTER RITSIGLAGOLITIC CAPITAL LETTER SLOVOGLAGOLITIC CAPITAL LETTER TVRIDOGLAGOLITIC CAPITAL LETTER UK\
    \UGLAGOLITIC CAPITAL LETTER FRITUGLAGOLITIC CAPITAL LETTER HERUGLAGOLITIC CAPITAL LETTER OTUGLAGOLITIC CAPITAL LETTER PEGLAGOLITIC CAPITAL LETTER SHTAGLAGOLITIC CAPITAL LETTER TSIGLAGOLITIC CAPITAL LETTER CHRIVIGLAGOLITIC CAPITAL LETTER SHAGLAGOLITIC CAPIT\
    \AL LETTER YERUGLAGOLITIC CAPITAL LETTER YERIGLAGOLITIC CAPITAL LETTER YATIGLAGOLITIC CAPITAL LETTER SPIDERY HAGLAGOLITIC CAPITAL LETTER YUGLAGOLITIC CAPITAL LETTER SMALL YUSGLAGOLITIC CAPITAL LETTER SMALL YUS WITH TAILGLAGOLITIC CAPITAL LETTER YOGLAGOLITI\
    \C CAPITAL LETTER IOTATED SMALL YUSGLAGOLITIC CAPITAL LETTER BIG YUSGLAGOLITIC CAPITAL LETTER IOTATED BIG YUSGLAGOLITIC CAPITAL LETTER FITAGLAGOLITIC CAPITAL LETTER IZHITSAGLAGOLITIC CAPITAL LETTER SHTAPICGLAGOLITIC CAPITAL LETTER TROKUTASTI AGLAGOLITIC CA\
    \PITAL LETTER LATINATE MYSLITEGLAGOLITIC CAPITAL LETTER CAUDATE CHRIVIGLAGOLITIC SMALL LETTER AZUGLAGOLITIC SMALL LETTER BUKYGLAGOLITIC SMALL LETTER VEDEGLAGOLITIC SMALL LETTER GLAGOLIGLAGOLITIC SMALL LETTER DOBROGLAGOLITIC SMALL LETTER YESTUGLAGOLITIC SMA\
    \LL LETTER ZHIVETEGLAGOLITIC SMALL LETTER DZELOGLAGOLITIC SMALL LETTER ZEMLJAGLAGOLITIC SMALL LETTER IZHEGLAGOLITIC SMALL LETTER INITIAL IZHEGLAGOLITIC SMALL LETTER IGLAGOLITIC SMALL LETTER DJERVIGLAGOLITIC SMALL LETTER KAKOGLAGOLITIC SMALL LETTER LJUDIJEG\
    \LAGOLITIC SMALL LETTER MYSLITEGLAGOLITIC SMALL LETTER NASHIGLAGOLITIC SMALL LETTER ONUGLAGOLITIC SMALL LETTER POKOJIGLAGOLITIC SMALL LETTER RITSIGLAGOLITIC SMALL LETTER SLOVOGLAGOLITIC SMALL LETTER TVRIDOGLAGOLITIC SMALL LETTER UKUGLAGOLITIC SMALL LETTER \
    \FRITUGLAGOLITIC SMALL LETTER HERUGLAGOLITIC SMALL LETTER OTUGLAGOLITIC SMALL LETTER PEGLAGOLITIC SMALL LETTER SHTAGLAGOLITIC SMALL LETTER TSIGLAGOLITIC SMALL LETTER CHRIVIGLAGOLITIC SMALL LETTER SHAGLAGOLITIC SMALL LETTER YERUGLAGOLITIC SMALL LETTER YERIG\
    \LAGOLITIC SMALL LETTER YATIGLAGOLITIC SMALL LETTER SPIDERY HAGLAGOLITIC SMALL LETTER YUGLAGOLITIC SMALL LETTER SMALL YUSGLAGOLITIC SMALL LETTER SMALL YUS WITH TAILGLAGOLITIC SMALL LETTER YOGLAGOLITIC SMALL LETTER IOTATED SMALL YUSGLAGOLITIC SMALL LETTER B\
    \IG YUSGLAGOLITIC SMALL LETTER IOTATED BIG YUSGLAGOLITIC SMALL LETTER FITAGLAGOLITIC SMALL LETTER IZHITSAGLAGOLITIC SMALL LETTER SHTAPICGLAGOLITIC SMALL LETTER TROKUTASTI AGLAGOLITIC SMALL LETTER LATINATE MYSLITEGLAGOLITIC SMALL LETTER CAUDATE CHRIVILATIN \
    \CAPITAL LETTER L WITH DOUBLE BARLATIN SMALL LETTER L WITH DOUBLE BARLATIN CAPITAL LETTER L WITH MIDDLE TILDELATIN CAPITAL LETTER P WITH STROKELATIN CAPITAL LETTER R WITH TAILLATIN SMALL LETTER A WITH STROKELATIN SMALL LETTER T WITH DIAGONAL STROKELATIN CA\
    \PITAL LETTER H WITH DESCENDERLATIN SMALL LETTER H WITH DESCENDERLATIN CAPITAL LETTER K WITH DESCENDERLATIN SMALL LETTER K WITH DESCENDERLATIN CAPITAL LETTER Z WITH DESCENDERLATIN SMALL LETTER Z WITH DESCENDERLATIN CAPITAL LETTER ALPHALATIN CAPITAL LETTER \
    \M WITH HOOKLATIN CAPITAL LETTER TURNED ALATIN CAPITAL LETTER TURNED ALPHALATIN SMALL LETTER V WITH RIGHT HOOKLATIN CAPITAL LETTER W WITH HOOKLATIN SMALL LETTER W WITH HOOKLATIN SMALL LETTER V WITH CURLLATIN CAPITAL LETTER HALF HLATIN SMALL LETTER HALF HLA\
    \TIN SMALL LETTER TAILLESS PHILATIN SMALL LETTER E WITH NOTCHLATIN SMALL LETTER TURNED R WITH TAILLATIN SMALL LETTER O WITH LOW RING INSIDELATIN LETTER SMALL CAPITAL TURNED ELATIN SUBSCRIPT SMALL LETTER JMODIFIER LETTER CAPITAL VLATIN CAPITAL LETTER S WITH\
    \ SWASH TAILLATIN CAPITAL LETTER Z WITH SWASH TAILCOPTIC CAPITAL LETTER ALFACOPTIC SMALL LETTER ALFACOPTIC CAPITAL LETTER VIDACOPTIC SMALL LETTER VIDACOPTIC CAPITAL LETTER GAMMACOPTIC SMALL LETTER GAMMACOPTIC CAPITAL LETTER DALDACOPTIC SMALL LETTER DALDACO\
    \PTIC CAPITAL LETTER EIECOPTIC SMALL LETTER EIECOPTIC CAPITAL LETTER SOUCOPTIC SMALL LETTER SOUCOPTIC CAPITAL LETTER ZATACOPTIC SMALL LETTER ZATACOPTIC CAPITAL LETTER HATECOPTIC SMALL LETTER HATECOPTIC CAPITAL LETTER THETHECOPTIC SMALL LETTER THETHECOPTIC \
    \CAPITAL LETTER IAUDACOPTIC SMALL LETTER IAUDACOPTIC CAPITAL LETTER KAPACOPTIC SMALL LETTER KAPACOPTIC CAPITAL LETTER LAULACOPTIC SMALL LETTER LAULACOPTIC CAPITAL LETTER MICOPTIC SMALL LETTER MICOPTIC CAPITAL LETTER NICOPTIC SMALL LETTER NICOPTIC CAPITAL L\
    \ETTER KSICOPTIC SMALL LETTER KSICOPTIC CAPITAL LETTER OCOPTIC SMALL LETTER OCOPTIC CAPITAL LETTER PICOPTIC SMALL LETTER PICOPTIC CAPITAL LETTER ROCOPTIC SMALL LETTER ROCOPTIC CAPITAL LETTER SIMACOPTIC SMALL LETTER SIMACOPTIC CAPITAL LETTER TAUCOPTIC SMALL\
    \ LETTER TAUCOPTIC CAPITAL LETTER UACOPTIC SMALL LETTER UACOPTIC CAPITAL LETTER FICOPTIC SMALL LETTER FICOPTIC CAPITAL LETTER KHICOPTIC SMALL LETTER KHICOPTIC CAPITAL LETTER PSICOPTIC SMALL LETTER PSICOPTIC CAPITAL LETTER OOUCOPTIC SMALL LETTER OOUCOPTIC C\
    \APITAL LETTER DIALECT-P ALEFCOPTIC SMALL LETTER DIALECT-P ALEFCOPTIC CAPITAL LETTER OLD COPTIC AINCOPTIC SMALL LETTER OLD COPTIC AINCOPTIC CAPITAL LETTER CRYPTOGRAMMIC EIECOPTIC SMALL LETTER CRYPTOGRAMMIC EIECOPTIC CAPITAL LETTER DIALECT-P KAPACOPTIC SMAL\
    \L LETTER DIALECT-P KAPACOPTIC CAPITAL LETTER DIALECT-P NICOPTIC SMALL LETTER DIALECT-P NICOPTIC CAPITAL LETTER CRYPTOGRAMMIC NICOPTIC SMALL LETTER CRYPTOGRAMMIC NICOPTIC CAPITAL LETTER OLD COPTIC OOUCOPTIC SMALL LETTER OLD COPTIC OOUCOPTIC CAPITAL LETTER \
    \SAMPICOPTIC SMALL LETTER SAMPICOPTIC CAPITAL LETTER CROSSED SHEICOPTIC SMALL LETTER CROSSED SHEICOPTIC CAPITAL LETTER OLD COPTIC SHEICOPTIC SMALL LETTER OLD COPTIC SHEICOPTIC CAPITAL LETTER OLD COPTIC ESHCOPTIC SMALL LETTER OLD COPTIC ESHCOPTIC CAPITAL LE\
    \TTER AKHMIMIC KHEICOPTIC SMALL LETTER AKHMIMIC KHEICOPTIC CAPITAL LETTER DIALECT-P HORICOPTIC SMALL LETTER DIALECT-P HORICOPTIC CAPITAL LETTER OLD COPTIC HORICOPTIC SMALL LETTER OLD COPTIC HORICOPTIC CAPITAL LETTER OLD COPTIC HACOPTIC SMALL LETTER OLD COP\
    \TIC HACOPTIC CAPITAL LETTER L-SHAPED HACOPTIC SMALL LETTER L-SHAPED HACOPTIC CAPITAL LETTER OLD COPTIC HEICOPTIC SMALL LETTER OLD COPTIC HEICOPTIC CAPITAL LETTER OLD COPTIC HATCOPTIC SMALL LETTER OLD COPTIC HATCOPTIC CAPITAL LETTER OLD COPTIC GANGIACOPTIC\
    \ SMALL LETTER OLD COPTIC GANGIACOPTIC CAPITAL LETTER OLD COPTIC DJACOPTIC SMALL LETTER OLD COPTIC DJACOPTIC CAPITAL LETTER OLD COPTIC SHIMACOPTIC SMALL LETTER OLD COPTIC SHIMACOPTIC CAPITAL LETTER OLD NUBIAN SHIMACOPTIC SMALL LETTER OLD NUBIAN SHIMACOPTIC\
    \ CAPITAL LETTER OLD NUBIAN NGICOPTIC SMALL LETTER OLD NUBIAN NGICOPTIC CAPITAL LETTER OLD NUBIAN NYICOPTIC SMALL LETTER OLD NUBIAN NYICOPTIC CAPITAL LETTER OLD NUBIAN WAUCOPTIC SMALL LETTER OLD NUBIAN WAUCOPTIC SYMBOL KAICOPTIC SYMBOL MI ROCOPTIC SYMBOL P\
    \I ROCOPTIC SYMBOL STAUROSCOPTIC SYMBOL TAU ROCOPTIC SYMBOL KHI ROCOPTIC SYMBOL SHIMA SIMACOPTIC CAPITAL LETTER CRYPTOGRAMMIC SHEICOPTIC SMALL LETTER CRYPTOGRAMMIC SHEICOPTIC CAPITAL LETTER CRYPTOGRAMMIC GANGIACOPTIC SMALL LETTER CRYPTOGRAMMIC GANGIACOPTIC\
    \ COMBINING NI ABOVECOPTIC COMBINING SPIRITUS ASPERCOPTIC COMBINING SPIRITUS LENISCOPTIC CAPITAL LETTER BOHAIRIC KHEICOPTIC SMALL LETTER BOHAIRIC KHEICOPTIC OLD NUBIAN FULL STOPCOPTIC OLD NUBIAN DIRECT QUESTION MARKCOPTIC OLD NUBIAN INDIRECT QUESTION MARKC\
    \OPTIC OLD NUBIAN VERSE DIVIDERCOPTIC FRACTION ONE HALFCOPTIC FULL STOPCOPTIC MORPHOLOGICAL DIVIDERGEORGIAN SMALL LETTER ANGEORGIAN SMALL LETTER BANGEORGIAN SMALL LETTER GANGEORGIAN SMALL LETTER DONGEORGIAN SMALL LETTER ENGEORGIAN SMALL LETTER VINGEORGIAN \
    \SMALL LETTER ZENGEORGIAN SMALL LETTER TANGEORGIAN SMALL LETTER INGEORGIAN SMALL LETTER KANGEORGIAN SMALL LETTER LASGEORGIAN SMALL LETTER MANGEORGIAN SMALL LETTER NARGEORGIAN SMALL LETTER ONGEORGIAN SMALL LETTER PARGEORGIAN SMALL LETTER ZHARGEORGIAN SMALL \
    \LETTER RAEGEORGIAN SMALL LETTER SANGEORGIAN SMALL LETTER TARGEORGIAN SMALL LETTER UNGEORGIAN SMALL LETTER PHARGEORGIAN SMALL LETTER KHARGEORGIAN SMALL LETTER GHANGEORGIAN SMALL LETTER QARGEORGIAN SMALL LETTER SHINGEORGIAN SMALL LETTER CHINGEORGIAN SMALL L\
    \ETTER CANGEORGIAN SMALL LETTER JILGEORGIAN SMALL LETTER CILGEORGIAN SMALL LETTER CHARGEORGIAN SMALL LETTER XANGEORGIAN SMALL LETTER JHANGEORGIAN SMALL LETTER HAEGEORGIAN SMALL LETTER HEGEORGIAN SMALL LETTER HIEGEORGIAN SMALL LETTER WEGEORGIAN SMALL LETTER\
    \ HARGEORGIAN SMALL LETTER HOEGEORGIAN SMALL LETTER YNGEORGIAN SMALL LETTER AENTIFINAGH LETTER YATIFINAGH LETTER YABTIFINAGH LETTER YABHTIFINAGH LETTER YAGTIFINAGH LETTER YAGHHTIFINAGH LETTER BERBER ACADEMY YAJTIFINAGH LETTER YAJTIFINAGH LETTER YADTIFINAGH\
    \ LETTER YADHTIFINAGH LETTER YADDTIFINAGH LETTER YADDHTIFINAGH LETTER YEYTIFINAGH LETTER YAFTIFINAGH LETTER YAKTIFINAGH LETTER TUAREG YAKTIFINAGH LETTER YAKHHTIFINAGH LETTER YAHTIFINAGH LETTER BERBER ACADEMY YAHTIFINAGH LETTER TUAREG YAHTIFINAGH LETTER YAH\
    \HTIFINAGH LETTER YAATIFINAGH LETTER YAKHTIFINAGH LETTER TUAREG YAKHTIFINAGH LETTER YAQTIFINAGH LETTER TUAREG YAQTIFINAGH LETTER YITIFINAGH LETTER YAZHTIFINAGH LETTER AHAGGAR YAZHTIFINAGH LETTER TUAREG YAZHTIFINAGH LETTER YALTIFINAGH LETTER YAMTIFINAGH LET\
    \TER YANTIFINAGH LETTER TUAREG YAGNTIFINAGH LETTER TUAREG YANGTIFINAGH LETTER YAPTIFINAGH LETTER YUTIFINAGH LETTER YARTIFINAGH LETTER YARRTIFINAGH LETTER YAGHTIFINAGH LETTER TUAREG YAGHTIFINAGH LETTER AYER YAGHTIFINAGH LETTER YASTIFINAGH LETTER YASSTIFINAG\
    \H LETTER YASHTIFINAGH LETTER YATTIFINAGH LETTER YATHTIFINAGH LETTER YACHTIFINAGH LETTER YATTTIFINAGH LETTER YAVTIFINAGH LETTER YAWTIFINAGH LETTER YAYTIFINAGH LETTER YAZTIFINAGH LETTER TAWELLEMET YAZTIFINAGH LETTER YAZZTIFINAGH LETTER YETIFINAGH LETTER YOT\
    \IFINAGH MODIFIER LETTER LABIALIZATION MARKTIFINAGH SEPARATOR MARKTIFINAGH CONSONANT JOINERETHIOPIC SYLLABLE LOAETHIOPIC SYLLABLE MOAETHIOPIC SYLLABLE ROAETHIOPIC SYLLABLE SOAETHIOPIC SYLLABLE SHOAETHIOPIC SYLLABLE BOAETHIOPIC SYLLABLE TOAETHIOPIC SYLLABLE\
    \ COAETHIOPIC SYLLABLE NOAETHIOPIC SYLLABLE NYOAETHIOPIC SYLLABLE GLOTTAL OAETHIOPIC SYLLABLE ZOAETHIOPIC SYLLABLE DOAETHIOPIC SYLLABLE DDOAETHIOPIC SYLLABLE JOAETHIOPIC SYLLABLE THOAETHIOPIC SYLLABLE CHOAETHIOPIC SYLLABLE PHOAETHIOPIC SYLLABLE POAETHIOPIC\
    \ SYLLABLE GGWAETHIOPIC SYLLABLE GGWIETHIOPIC SYLLABLE GGWEEETHIOPIC SYLLABLE GGWEETHIOPIC SYLLABLE SSAETHIOPIC SYLLABLE SSUETHIOPIC SYLLABLE SSIETHIOPIC SYLLABLE SSAAETHIOPIC SYLLABLE SSEEETHIOPIC SYLLABLE SSEETHIOPIC SYLLABLE SSOETHIOPIC SYLLABLE CCAETHI\
    \OPIC SYLLABLE CCUETHIOPIC SYLLABLE CCIETHIOPIC SYLLABLE CCAAETHIOPIC SYLLABLE CCEEETHIOPIC SYLLABLE CCEETHIOPIC SYLLABLE CCOETHIOPIC SYLLABLE ZZAETHIOPIC SYLLABLE ZZUETHIOPIC SYLLABLE ZZIETHIOPIC SYLLABLE ZZAAETHIOPIC SYLLABLE ZZEEETHIOPIC SYLLABLE ZZEETH\
    \IOPIC SYLLABLE ZZOETHIOPIC SYLLABLE CCHAETHIOPIC SYLLABLE CCHUETHIOPIC SYLLABLE CCHIETHIOPIC SYLLABLE CCHAAETHIOPIC SYLLABLE CCHEEETHIOPIC SYLLABLE CCHEETHIOPIC SYLLABLE CCHOETHIOPIC SYLLABLE QYAETHIOPIC SYLLABLE QYUETHIOPIC SYLLABLE QYIETHIOPIC SYLLABLE \
    \QYAAETHIOPIC SYLLABLE QYEEETHIOPIC SYLLABLE QYEETHIOPIC SYLLABLE QYOETHIOPIC SYLLABLE KYAETHIOPIC SYLLABLE KYUETHIOPIC SYLLABLE KYIETHIOPIC SYLLABLE KYAAETHIOPIC SYLLABLE KYEEETHIOPIC SYLLABLE KYEETHIOPIC SYLLABLE KYOETHIOPIC SYLLABLE XYAETHIOPIC SYLLABLE\
    \ XYUETHIOPIC SYLLABLE XYIETHIOPIC SYLLABLE XYAAETHIOPIC SYLLABLE XYEEETHIOPIC SYLLABLE XYEETHIOPIC SYLLABLE XYOETHIOPIC SYLLABLE GYAETHIOPIC SYLLABLE GYUETHIOPIC SYLLABLE GYIETHIOPIC SYLLABLE GYAAETHIOPIC SYLLABLE GYEEETHIOPIC SYLLABLE GYEETHIOPIC SYLLABL\
    \E GYOCOMBINING CYRILLIC LETTER BECOMBINING CYRILLIC LETTER VECOMBINING CYRILLIC LETTER GHECOMBINING CYRILLIC LETTER DECOMBINING CYRILLIC LETTER ZHECOMBINING CYRILLIC LETTER ZECOMBINING CYRILLIC LETTER KACOMBINING CYRILLIC LETTER ELCOMBINING CYRILLIC LETTE\
    \R EMCOMBINING CYRILLIC LETTER ENCOMBINING CYRILLIC LETTER OCOMBINING CYRILLIC LETTER PECOMBINING CYRILLIC LETTER ERCOMBINING CYRILLIC LETTER ESCOMBINING CYRILLIC LETTER TECOMBINING CYRILLIC LETTER HACOMBINING CYRILLIC LETTER TSECOMBINING CYRILLIC LETTER C\
    \HECOMBINING CYRILLIC LETTER SHACOMBINING CYRILLIC LETTER SHCHACOMBINING CYRILLIC LETTER FITACOMBINING CYRILLIC LETTER ES-TECOMBINING CYRILLIC LETTER ACOMBINING CYRILLIC LETTER IECOMBINING CYRILLIC LETTER DJERVCOMBINING CYRILLIC LETTER MONOGRAPH UKCOMBININ\
    \G CYRILLIC LETTER YATCOMBINING CYRILLIC LETTER YUCOMBINING CYRILLIC LETTER IOTIFIED ACOMBINING CYRILLIC LETTER LITTLE YUSCOMBINING CYRILLIC LETTER BIG YUSCOMBINING CYRILLIC LETTER IOTIFIED BIG YUSRIGHT ANGLE SUBSTITUTION MARKERRIGHT ANGLE DOTTED SUBSTITUT\
    \ION MARKERLEFT SUBSTITUTION BRACKETRIGHT SUBSTITUTION BRACKETLEFT DOTTED SUBSTITUTION BRACKETRIGHT DOTTED SUBSTITUTION BRACKETRAISED INTERPOLATION MARKERRAISED DOTTED INTERPOLATION MARKERDOTTED TRANSPOSITION MARKERLEFT TRANSPOSITION BRACKETRIGHT TRANSPOSI\
    \TION BRACKETRAISED SQUARELEFT RAISED OMISSION BRACKETRIGHT RAISED OMISSION BRACKETEDITORIAL CORONISPARAGRAPHOSFORKED PARAGRAPHOSREVERSED FORKED PARAGRAPHOSHYPODIASTOLEDOTTED OBELOSDOWNWARDS ANCORAUPWARDS ANCORADOTTED RIGHT-POINTING ANGLEDOUBLE OBLIQUE HYP\
    \HENINVERTED INTERROBANGPALM BRANCHHYPHEN WITH DIAERESISTILDE WITH RING ABOVELEFT LOW PARAPHRASE BRACKETRIGHT LOW PARAPHRASE BRACKETTILDE WITH DOT ABOVETILDE WITH DOT BELOWLEFT VERTICAL BAR WITH QUILLRIGHT VERTICAL BAR WITH QUILLTOP LEFT HALF BRACKETTOP RI\
    \GHT HALF BRACKETBOTTOM LEFT HALF BRACKETBOTTOM RIGHT HALF BRACKETLEFT SIDEWAYS U BRACKETRIGHT SIDEWAYS U BRACKETLEFT DOUBLE PARENTHESISRIGHT DOUBLE PARENTHESISTWO DOTS OVER ONE DOT PUNCTUATIONONE DOT OVER TWO DOTS PUNCTUATIONSQUARED FOUR DOT PUNCTUATIONFI\
    \VE DOT MARKREVERSED QUESTION MARKVERTICAL TILDERING POINTWORD SEPARATOR MIDDLE DOTTURNED COMMARAISED DOTRAISED COMMATURNED SEMICOLONDAGGER WITH LEFT GUARDDAGGER WITH RIGHT GUARDTURNED DAGGERTOP HALF SECTION SIGNTWO-EM DASHTHREE-EM DASHSTENOGRAPHIC FULL ST\
    \OPVERTICAL SIX DOTSWIGGLY VERTICAL LINECAPITULUMDOUBLE HYPHENREVERSED COMMADOUBLE LOW-REVERSED-9 QUOTATION MARKDASH WITH LEFT UPTURNDOUBLE SUSPENSION MARKINVERTED LOW KAVYKAINVERTED LOW KAVYKA WITH KAVYKA ABOVELOW KAVYKALOW KAVYKA WITH DOTDOUBLE STACKED C\
    \OMMADOTTED SOLIDUSTRIPLE DAGGERMEDIEVAL COMMAPARAGRAPHUS MARKPUNCTUS ELEVATUS MARKCORNISH VERSE DIVIDERCROSS PATTY WITH RIGHT CROSSBARCROSS PATTY WITH LEFT CROSSBARTIRONIAN SIGN CAPITAL ETMEDIEVAL EXCLAMATION MARKMEDIEVAL QUESTION MARKLEFT SQUARE BRACKET \
    \WITH STROKERIGHT SQUARE BRACKET WITH STROKELEFT SQUARE BRACKET WITH DOUBLE STROKERIGHT SQUARE BRACKET WITH DOUBLE STROKETOP HALF LEFT PARENTHESISTOP HALF RIGHT PARENTHESISBOTTOM HALF LEFT PARENTHESISBOTTOM HALF RIGHT PARENTHESISOBLIQUE HYPHENCJK RADICAL R\
    \EPEATCJK RADICAL CLIFFCJK RADICAL SECOND ONECJK RADICAL SECOND TWOCJK RADICAL SECOND THREECJK RADICAL PERSONCJK RADICAL BOXCJK RADICAL TABLECJK RADICAL KNIFE ONECJK RADICAL KNIFE TWOCJK RADICAL DIVINATIONCJK RADICAL SEALCJK RADICAL SMALL ONECJK RADICAL SM\
    \ALL TWOCJK RADICAL LAME ONECJK RADICAL LAME TWOCJK RADICAL LAME THREECJK RADICAL LAME FOURCJK RADICAL SNAKECJK RADICAL THREADCJK RADICAL SNOUT ONECJK RADICAL SNOUT TWOCJK RADICAL HEART ONECJK RADICAL HEART TWOCJK RADICAL HANDCJK RADICAL RAPCJK RADICAL CHO\
    \KECJK RADICAL SUNCJK RADICAL MOONCJK RADICAL DEATHCJK RADICAL MOTHERCJK RADICAL CIVILIANCJK RADICAL WATER ONECJK RADICAL WATER TWOCJK RADICAL FIRECJK RADICAL PAW ONECJK RADICAL PAW TWOCJK RADICAL SIMPLIFIED HALF TREE TRUNKCJK RADICAL COWCJK RADICAL DOGCJK\
    \ RADICAL JADECJK RADICAL BOLT OF CLOTHCJK RADICAL EYECJK RADICAL SPIRIT ONECJK RADICAL SPIRIT TWOCJK RADICAL BAMBOOCJK RADICAL SILKCJK RADICAL C-SIMPLIFIED SILKCJK RADICAL NET ONECJK RADICAL NET TWOCJK RADICAL NET THREECJK RADICAL NET FOURCJK RADICAL MESH\
    \CJK RADICAL SHEEPCJK RADICAL RAMCJK RADICAL EWECJK RADICAL OLDCJK RADICAL BRUSH ONECJK RADICAL BRUSH TWOCJK RADICAL MEATCJK RADICAL MORTARCJK RADICAL GRASS ONECJK RADICAL GRASS TWOCJK RADICAL GRASS THREECJK RADICAL TIGERCJK RADICAL CLOTHESCJK RADICAL WEST\
    \ ONECJK RADICAL WEST TWOCJK RADICAL C-SIMPLIFIED SEECJK RADICAL SIMPLIFIED HORNCJK RADICAL HORNCJK RADICAL C-SIMPLIFIED SPEECHCJK RADICAL C-SIMPLIFIED SHELLCJK RADICAL FOOTCJK RADICAL C-SIMPLIFIED CARTCJK RADICAL SIMPLIFIED WALKCJK RADICAL WALK ONECJK RAD\
    \ICAL WALK TWOCJK RADICAL CITYCJK RADICAL C-SIMPLIFIED GOLDCJK RADICAL LONG ONECJK RADICAL LONG TWOCJK RADICAL C-SIMPLIFIED LONGCJK RADICAL C-SIMPLIFIED GATECJK RADICAL MOUND ONECJK RADICAL MOUND TWOCJK RADICAL RAINCJK RADICAL BLUECJK RADICAL C-SIMPLIFIED \
    \TANNED LEATHERCJK RADICAL C-SIMPLIFIED LEAFCJK RADICAL C-SIMPLIFIED WINDCJK RADICAL C-SIMPLIFIED FLYCJK RADICAL EAT ONECJK RADICAL EAT TWOCJK RADICAL EAT THREECJK RADICAL C-SIMPLIFIED EATCJK RADICAL HEADCJK RADICAL C-SIMPLIFIED HORSECJK RADICAL BONECJK RA\
    \DICAL GHOSTCJK RADICAL C-SIMPLIFIED FISHCJK RADICAL C-SIMPLIFIED BIRDCJK RADICAL C-SIMPLIFIED SALTCJK RADICAL SIMPLIFIED WHEATCJK RADICAL SIMPLIFIED YELLOWCJK RADICAL C-SIMPLIFIED FROGCJK RADICAL J-SIMPLIFIED EVENCJK RADICAL C-SIMPLIFIED EVENCJK RADICAL J\
    \-SIMPLIFIED TOOTHCJK RADICAL C-SIMPLIFIED TOOTHCJK RADICAL J-SIMPLIFIED DRAGONCJK RADICAL C-SIMPLIFIED DRAGONCJK RADICAL TURTLECJK RADICAL J-SIMPLIFIED TURTLECJK RADICAL C-SIMPLIFIED TURTLEKANGXI RADICAL ONEKANGXI RADICAL LINEKANGXI RADICAL DOTKANGXI RADI\
    \CAL SLASHKANGXI RADICAL SECONDKANGXI RADICAL HOOKKANGXI RADICAL TWOKANGXI RADICAL LIDKANGXI RADICAL MANKANGXI RADICAL LEGSKANGXI RADICAL ENTERKANGXI RADICAL EIGHTKANGXI RADICAL DOWN BOXKANGXI RADICAL COVERKANGXI RADICAL ICEKANGXI RADICAL TABLEKANGXI RADIC\
    \AL OPEN BOXKANGXI RADICAL KNIFEKANGXI RADICAL POWERKANGXI RADICAL WRAPKANGXI RADICAL SPOONKANGXI RADICAL RIGHT OPEN BOXKANGXI RADICAL HIDING ENCLOSUREKANGXI RADICAL TENKANGXI RADICAL DIVINATIONKANGXI RADICAL SEALKANGXI RADICAL CLIFFKANGXI RADICAL PRIVATEK\
    \ANGXI RADICAL AGAINKANGXI RADICAL MOUTHKANGXI RADICAL ENCLOSUREKANGXI RADICAL EARTHKANGXI RADICAL SCHOLARKANGXI RADICAL GOKANGXI RADICAL GO SLOWLYKANGXI RADICAL EVENINGKANGXI RADICAL BIGKANGXI RADICAL WOMANKANGXI RADICAL CHILDKANGXI RADICAL ROOFKANGXI RAD\
    \ICAL INCHKANGXI RADICAL SMALLKANGXI RADICAL LAMEKANGXI RADICAL CORPSEKANGXI RADICAL SPROUTKANGXI RADICAL MOUNTAINKANGXI RADICAL RIVERKANGXI RADICAL WORKKANGXI RADICAL ONESELFKANGXI RADICAL TURBANKANGXI RADICAL DRYKANGXI RADICAL SHORT THREADKANGXI RADICAL \
    \DOTTED CLIFFKANGXI RADICAL LONG STRIDEKANGXI RADICAL TWO HANDSKANGXI RADICAL SHOOTKANGXI RADICAL BOWKANGXI RADICAL SNOUTKANGXI RADICAL BRISTLEKANGXI RADICAL STEPKANGXI RADICAL HEARTKANGXI RADICAL HALBERDKANGXI RADICAL DOORKANGXI RADICAL HANDKANGXI RADICAL\
    \ BRANCHKANGXI RADICAL RAPKANGXI RADICAL SCRIPTKANGXI RADICAL DIPPERKANGXI RADICAL AXEKANGXI RADICAL SQUAREKANGXI RADICAL NOTKANGXI RADICAL SUNKANGXI RADICAL SAYKANGXI RADICAL MOONKANGXI RADICAL TREEKANGXI RADICAL LACKKANGXI RADICAL STOPKANGXI RADICAL DEAT\
    \HKANGXI RADICAL WEAPONKANGXI RADICAL DO NOTKANGXI RADICAL COMPAREKANGXI RADICAL FURKANGXI RADICAL CLANKANGXI RADICAL STEAMKANGXI RADICAL WATERKANGXI RADICAL FIREKANGXI RADICAL CLAWKANGXI RADICAL FATHERKANGXI RADICAL DOUBLE XKANGXI RADICAL HALF TREE TRUNKK\
    \ANGXI RADICAL SLICEKANGXI RADICAL FANGKANGXI RADICAL COWKANGXI RADICAL DOGKANGXI RADICAL PROFOUNDKANGXI RADICAL JADEKANGXI RADICAL MELONKANGXI RADICAL TILEKANGXI RADICAL SWEETKANGXI RADICAL LIFEKANGXI RADICAL USEKANGXI RADICAL FIELDKANGXI RADICAL BOLT OF \
    \CLOTHKANGXI RADICAL SICKNESSKANGXI RADICAL DOTTED TENTKANGXI RADICAL WHITEKANGXI RADICAL SKINKANGXI RADICAL DISHKANGXI RADICAL EYEKANGXI RADICAL SPEARKANGXI RADICAL ARROWKANGXI RADICAL STONEKANGXI RADICAL SPIRITKANGXI RADICAL TRACKKANGXI RADICAL GRAINKANG\
    \XI RADICAL CAVEKANGXI RADICAL STANDKANGXI RADICAL BAMBOOKANGXI RADICAL RICEKANGXI RADICAL SILKKANGXI RADICAL JARKANGXI RADICAL NETKANGXI RADICAL SHEEPKANGXI RADICAL FEATHERKANGXI RADICAL OLDKANGXI RADICAL ANDKANGXI RADICAL PLOWKANGXI RADICAL EARKANGXI RAD\
    \ICAL BRUSHKANGXI RADICAL MEATKANGXI RADICAL MINISTERKANGXI RADICAL SELFKANGXI RADICAL ARRIVEKANGXI RADICAL MORTARKANGXI RADICAL TONGUEKANGXI RADICAL OPPOSEKANGXI RADICAL BOATKANGXI RADICAL STOPPINGKANGXI RADICAL COLORKANGXI RADICAL GRASSKANGXI RADICAL TIG\
    \ERKANGXI RADICAL INSECTKANGXI RADICAL BLOODKANGXI RADICAL WALK ENCLOSUREKANGXI RADICAL CLOTHESKANGXI RADICAL WESTKANGXI RADICAL SEEKANGXI RADICAL HORNKANGXI RADICAL SPEECHKANGXI RADICAL VALLEYKANGXI RADICAL BEANKANGXI RADICAL PIGKANGXI RADICAL BADGERKANGX\
    \I RADICAL SHELLKANGXI RADICAL REDKANGXI RADICAL RUNKANGXI RADICAL FOOTKANGXI RADICAL BODYKANGXI RADICAL CARTKANGXI RADICAL BITTERKANGXI RADICAL MORNINGKANGXI RADICAL WALKKANGXI RADICAL CITYKANGXI RADICAL WINEKANGXI RADICAL DISTINGUISHKANGXI RADICAL VILLAG\
    \EKANGXI RADICAL GOLDKANGXI RADICAL LONGKANGXI RADICAL GATEKANGXI RADICAL MOUNDKANGXI RADICAL SLAVEKANGXI RADICAL SHORT TAILED BIRDKANGXI RADICAL RAINKANGXI RADICAL BLUEKANGXI RADICAL WRONGKANGXI RADICAL FACEKANGXI RADICAL LEATHERKANGXI RADICAL TANNED LEAT\
    \HERKANGXI RADICAL LEEKKANGXI RADICAL SOUNDKANGXI RADICAL LEAFKANGXI RADICAL WINDKANGXI RADICAL FLYKANGXI RADICAL EATKANGXI RADICAL HEADKANGXI RADICAL FRAGRANTKANGXI RADICAL HORSEKANGXI RADICAL BONEKANGXI RADICAL TALLKANGXI RADICAL HAIRKANGXI RADICAL FIGHT\
    \KANGXI RADICAL SACRIFICIAL WINEKANGXI RADICAL CAULDRONKANGXI RADICAL GHOSTKANGXI RADICAL FISHKANGXI RADICAL BIRDKANGXI RADICAL SALTKANGXI RADICAL DEERKANGXI RADICAL WHEATKANGXI RADICAL HEMPKANGXI RADICAL YELLOWKANGXI RADICAL MILLETKANGXI RADICAL BLACKKANG\
    \XI RADICAL EMBROIDERYKANGXI RADICAL FROGKANGXI RADICAL TRIPODKANGXI RADICAL DRUMKANGXI RADICAL RATKANGXI RADICAL NOSEKANGXI RADICAL EVENKANGXI RADICAL TOOTHKANGXI RADICAL DRAGONKANGXI RADICAL TURTLEKANGXI RADICAL FLUTEIDEOGRAPHIC DESCRIPTION CHARACTER LEF\
    \T TO RIGHTIDEOGRAPHIC DESCRIPTION CHARACTER ABOVE TO BELOWIDEOGRAPHIC DESCRIPTION CHARACTER LEFT TO MIDDLE AND RIGHTIDEOGRAPHIC DESCRIPTION CHARACTER ABOVE TO MIDDLE AND BELOWIDEOGRAPHIC DESCRIPTION CHARACTER FULL SURROUNDIDEOGRAPHIC DESCRIPTION CHARACTER\
    \ SURROUND FROM ABOVEIDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM BELOWIDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM LEFTIDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM UPPER LEFTIDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM UPPER RIGHTIDEOGRAPHIC D\
    \ESCRIPTION CHARACTER SURROUND FROM LOWER LEFTIDEOGRAPHIC DESCRIPTION CHARACTER OVERLAIDIDEOGRAPHIC SPACEIDEOGRAPHIC COMMAIDEOGRAPHIC FULL STOPDITTO MARKJAPANESE INDUSTRIAL STANDARD SYMBOLIDEOGRAPHIC ITERATION MARKIDEOGRAPHIC CLOSING MARKIDEOGRAPHIC NUMBER\
    \ ZEROLEFT ANGLE BRACKETRIGHT ANGLE BRACKETLEFT DOUBLE ANGLE BRACKETRIGHT DOUBLE ANGLE BRACKETLEFT CORNER BRACKETRIGHT CORNER BRACKETLEFT WHITE CORNER BRACKETRIGHT WHITE CORNER BRACKETLEFT BLACK LENTICULAR BRACKETRIGHT BLACK LENTICULAR BRACKETPOSTAL MARKGE\
    \TA MARKLEFT TORTOISE SHELL BRACKETRIGHT TORTOISE SHELL BRACKETLEFT WHITE LENTICULAR BRACKETRIGHT WHITE LENTICULAR BRACKETLEFT WHITE TORTOISE SHELL BRACKETRIGHT WHITE TORTOISE SHELL BRACKETLEFT WHITE SQUARE BRACKETRIGHT WHITE SQUARE BRACKETWAVE DASHREVERSE\
    \D DOUBLE PRIME QUOTATION MARKDOUBLE PRIME QUOTATION MARKLOW DOUBLE PRIME QUOTATION MARKPOSTAL MARK FACEHANGZHOU NUMERAL ONEHANGZHOU NUMERAL TWOHANGZHOU NUMERAL THREEHANGZHOU NUMERAL FOURHANGZHOU NUMERAL FIVEHANGZHOU NUMERAL SIXHANGZHOU NUMERAL SEVENHANGZH\
    \OU NUMERAL EIGHTHANGZHOU NUMERAL NINEIDEOGRAPHIC LEVEL TONE MARKIDEOGRAPHIC RISING TONE MARKIDEOGRAPHIC DEPARTING TONE MARKIDEOGRAPHIC ENTERING TONE MARKHANGUL SINGLE DOT TONE MARKHANGUL DOUBLE DOT TONE MARKWAVY DASHVERTICAL KANA REPEAT MARKVERTICAL KANA \
    \REPEAT WITH VOICED SOUND MARKVERTICAL KANA REPEAT MARK UPPER HALFVERTICAL KANA REPEAT WITH VOICED SOUND MARK UPPER HALFVERTICAL KANA REPEAT MARK LOWER HALFCIRCLED POSTAL MARKIDEOGRAPHIC TELEGRAPH LINE FEED SEPARATOR SYMBOLHANGZHOU NUMERAL TENHANGZHOU NUME\
    \RAL TWENTYHANGZHOU NUMERAL THIRTYVERTICAL IDEOGRAPHIC ITERATION MARKMASU MARKPART ALTERNATION MARKIDEOGRAPHIC VARIATION INDICATORIDEOGRAPHIC HALF FILL SPACEHIRAGANA LETTER SMALL AHIRAGANA LETTER AHIRAGANA LETTER SMALL IHIRAGANA LETTER IHIRAGANA LETTER SMA\
    \LL UHIRAGANA LETTER UHIRAGANA LETTER SMALL EHIRAGANA LETTER EHIRAGANA LETTER SMALL OHIRAGANA LETTER OHIRAGANA LETTER KAHIRAGANA LETTER GAHIRAGANA LETTER KIHIRAGANA LETTER GIHIRAGANA LETTER KUHIRAGANA LETTER GUHIRAGANA LETTER KEHIRAGANA LETTER GEHIRAGANA L\
    \ETTER KOHIRAGANA LETTER GOHIRAGANA LETTER SAHIRAGANA LETTER ZAHIRAGANA LETTER SIHIRAGANA LETTER ZIHIRAGANA LETTER SUHIRAGANA LETTER ZUHIRAGANA LETTER SEHIRAGANA LETTER ZEHIRAGANA LETTER SOHIRAGANA LETTER ZOHIRAGANA LETTER TAHIRAGANA LETTER DAHIRAGANA LETT\
    \ER TIHIRAGANA LETTER DIHIRAGANA LETTER SMALL TUHIRAGANA LETTER TUHIRAGANA LETTER DUHIRAGANA LETTER TEHIRAGANA LETTER DEHIRAGANA LETTER TOHIRAGANA LETTER DOHIRAGANA LETTER NAHIRAGANA LETTER NIHIRAGANA LETTER NUHIRAGANA LETTER NEHIRAGANA LETTER NOHIRAGANA L\
    \ETTER HAHIRAGANA LETTER BAHIRAGANA LETTER PAHIRAGANA LETTER HIHIRAGANA LETTER BIHIRAGANA LETTER PIHIRAGANA LETTER HUHIRAGANA LETTER BUHIRAGANA LETTER PUHIRAGANA LETTER HEHIRAGANA LETTER BEHIRAGANA LETTER PEHIRAGANA LETTER HOHIRAGANA LETTER BOHIRAGANA LETT\
    \ER POHIRAGANA LETTER MAHIRAGANA LETTER MIHIRAGANA LETTER MUHIRAGANA LETTER MEHIRAGANA LETTER MOHIRAGANA LETTER SMALL YAHIRAGANA LETTER YAHIRAGANA LETTER SMALL YUHIRAGANA LETTER YUHIRAGANA LETTER SMALL YOHIRAGANA LETTER YOHIRAGANA LETTER RAHIRAGANA LETTER \
    \RIHIRAGANA LETTER RUHIRAGANA LETTER REHIRAGANA LETTER ROHIRAGANA LETTER SMALL WAHIRAGANA LETTER WAHIRAGANA LETTER WIHIRAGANA LETTER WEHIRAGANA LETTER WOHIRAGANA LETTER NHIRAGANA LETTER VUHIRAGANA LETTER SMALL KAHIRAGANA LETTER SMALL KECOMBINING KATAKANA-H\
    \IRAGANA VOICED SOUND MARKCOMBINING KATAKANA-HIRAGANA SEMI-VOICED SOUND MARKKATAKANA-HIRAGANA VOICED SOUND MARKKATAKANA-HIRAGANA SEMI-VOICED SOUND MARKHIRAGANA ITERATION MARKHIRAGANA VOICED ITERATION MARKHIRAGANA DIGRAPH YORIKATAKANA-HIRAGANA DOUBLE HYPHEN\
    \KATAKANA LETTER SMALL AKATAKANA LETTER AKATAKANA LETTER SMALL IKATAKANA LETTER IKATAKANA LETTER SMALL UKATAKANA LETTER UKATAKANA LETTER SMALL EKATAKANA LETTER EKATAKANA LETTER SMALL OKATAKANA LETTER OKATAKANA LETTER KAKATAKANA LETTER GAKATAKANA LETTER KIK\
    \ATAKANA LETTER GIKATAKANA LETTER KUKATAKANA LETTER GUKATAKANA LETTER KEKATAKANA LETTER GEKATAKANA LETTER KOKATAKANA LETTER GOKATAKANA LETTER SAKATAKANA LETTER ZAKATAKANA LETTER SIKATAKANA LETTER ZIKATAKANA LETTER SUKATAKANA LETTER ZUKATAKANA LETTER SEKATA\
    \KANA LETTER ZEKATAKANA LETTER SOKATAKANA LETTER ZOKATAKANA LETTER TAKATAKANA LETTER DAKATAKANA LETTER TIKATAKANA LETTER DIKATAKANA LETTER SMALL TUKATAKANA LETTER TUKATAKANA LETTER DUKATAKANA LETTER TEKATAKANA LETTER DEKATAKANA LETTER TOKATAKANA LETTER DOK\
    \ATAKANA LETTER NAKATAKANA LETTER NIKATAKANA LETTER NUKATAKANA LETTER NEKATAKANA LETTER NOKATAKANA LETTER HAKATAKANA LETTER BAKATAKANA LETTER PAKATAKANA LETTER HIKATAKANA LETTER BIKATAKANA LETTER PIKATAKANA LETTER HUKATAKANA LETTER BUKATAKANA LETTER PUKATA\
    \KANA LETTER HEKATAKANA LETTER BEKATAKANA LETTER PEKATAKANA LETTER HOKATAKANA LETTER BOKATAKANA LETTER POKATAKANA LETTER MAKATAKANA LETTER MIKATAKANA LETTER MUKATAKANA LETTER MEKATAKANA LETTER MOKATAKANA LETTER SMALL YAKATAKANA LETTER YAKATAKANA LETTER SMA\
    \LL YUKATAKANA LETTER YUKATAKANA LETTER SMALL YOKATAKANA LETTER YOKATAKANA LETTER RAKATAKANA LETTER RIKATAKANA LETTER RUKATAKANA LETTER REKATAKANA LETTER ROKATAKANA LETTER SMALL WAKATAKANA LETTER WAKATAKANA LETTER WIKATAKANA LETTER WEKATAKANA LETTER WOKATA\
    \KANA LETTER NKATAKANA LETTER VUKATAKANA LETTER SMALL KAKATAKANA LETTER SMALL KEKATAKANA LETTER VAKATAKANA LETTER VIKATAKANA LETTER VEKATAKANA LETTER VOKATAKANA MIDDLE DOTKATAKANA-HIRAGANA PROLONGED SOUND MARKKATAKANA ITERATION MARKKATAKANA VOICED ITERATIO\
    \N MARKKATAKANA DIGRAPH KOTOBOPOMOFO LETTER BBOPOMOFO LETTER PBOPOMOFO LETTER MBOPOMOFO LETTER FBOPOMOFO LETTER DBOPOMOFO LETTER TBOPOMOFO LETTER NBOPOMOFO LETTER LBOPOMOFO LETTER GBOPOMOFO LETTER KBOPOMOFO LETTER HBOPOMOFO LETTER JBOPOMOFO LETTER QBOPOMOF\
    \O LETTER XBOPOMOFO LETTER ZHBOPOMOFO LETTER CHBOPOMOFO LETTER SHBOPOMOFO LETTER RBOPOMOFO LETTER ZBOPOMOFO LETTER CBOPOMOFO LETTER SBOPOMOFO LETTER ABOPOMOFO LETTER OBOPOMOFO LETTER EBOPOMOFO LETTER EHBOPOMOFO LETTER AIBOPOMOFO LETTER EIBOPOMOFO LETTER AU\
    \BOPOMOFO LETTER OUBOPOMOFO LETTER ANBOPOMOFO LETTER ENBOPOMOFO LETTER ANGBOPOMOFO LETTER ENGBOPOMOFO LETTER ERBOPOMOFO LETTER IBOPOMOFO LETTER UBOPOMOFO LETTER IUBOPOMOFO LETTER VBOPOMOFO LETTER NGBOPOMOFO LETTER GNBOPOMOFO LETTER IHBOPOMOFO LETTER O WITH\
    \ DOT ABOVEBOPOMOFO LETTER NNHANGUL LETTER KIYEOKHANGUL LETTER SSANGKIYEOKHANGUL LETTER KIYEOK-SIOSHANGUL LETTER NIEUNHANGUL LETTER NIEUN-CIEUCHANGUL LETTER NIEUN-HIEUHHANGUL LETTER TIKEUTHANGUL LETTER SSANGTIKEUTHANGUL LETTER RIEULHANGUL LETTER RIEUL-KIYE\
    \OKHANGUL LETTER RIEUL-MIEUMHANGUL LETTER RIEUL-PIEUPHANGUL LETTER RIEUL-SIOSHANGUL LETTER RIEUL-THIEUTHHANGUL LETTER RIEUL-PHIEUPHHANGUL LETTER RIEUL-HIEUHHANGUL LETTER MIEUMHANGUL LETTER PIEUPHANGUL LETTER SSANGPIEUPHANGUL LETTER PIEUP-SIOSHANGUL LETTER \
    \SIOSHANGUL LETTER SSANGSIOSHANGUL LETTER IEUNGHANGUL LETTER CIEUCHANGUL LETTER SSANGCIEUCHANGUL LETTER CHIEUCHHANGUL LETTER KHIEUKHHANGUL LETTER THIEUTHHANGUL LETTER PHIEUPHHANGUL LETTER HIEUHHANGUL LETTER AHANGUL LETTER AEHANGUL LETTER YAHANGUL LETTER YA\
    \EHANGUL LETTER EOHANGUL LETTER EHANGUL LETTER YEOHANGUL LETTER YEHANGUL LETTER OHANGUL LETTER WAHANGUL LETTER WAEHANGUL LETTER OEHANGUL LETTER YOHANGUL LETTER UHANGUL LETTER WEOHANGUL LETTER WEHANGUL LETTER WIHANGUL LETTER YUHANGUL LETTER EUHANGUL LETTER \
    \YIHANGUL LETTER IHANGUL FILLERHANGUL LETTER SSANGNIEUNHANGUL LETTER NIEUN-TIKEUTHANGUL LETTER NIEUN-SIOSHANGUL LETTER NIEUN-PANSIOSHANGUL LETTER RIEUL-KIYEOK-SIOSHANGUL LETTER RIEUL-TIKEUTHANGUL LETTER RIEUL-PIEUP-SIOSHANGUL LETTER RIEUL-PANSIOSHANGUL LET\
    \TER RIEUL-YEORINHIEUHHANGUL LETTER MIEUM-PIEUPHANGUL LETTER MIEUM-SIOSHANGUL LETTER MIEUM-PANSIOSHANGUL LETTER KAPYEOUNMIEUMHANGUL LETTER PIEUP-KIYEOKHANGUL LETTER PIEUP-TIKEUTHANGUL LETTER PIEUP-SIOS-KIYEOKHANGUL LETTER PIEUP-SIOS-TIKEUTHANGUL LETTER PIE\
    \UP-CIEUCHANGUL LETTER PIEUP-THIEUTHHANGUL LETTER KAPYEOUNPIEUPHANGUL LETTER KAPYEOUNSSANGPIEUPHANGUL LETTER SIOS-KIYEOKHANGUL LETTER SIOS-NIEUNHANGUL LETTER SIOS-TIKEUTHANGUL LETTER SIOS-PIEUPHANGUL LETTER SIOS-CIEUCHANGUL LETTER PANSIOSHANGUL LETTER SSAN\
    \GIEUNGHANGUL LETTER YESIEUNGHANGUL LETTER YESIEUNG-SIOSHANGUL LETTER YESIEUNG-PANSIOSHANGUL LETTER KAPYEOUNPHIEUPHHANGUL LETTER SSANGHIEUHHANGUL LETTER YEORINHIEUHHANGUL LETTER YO-YAHANGUL LETTER YO-YAEHANGUL LETTER YO-IHANGUL LETTER YU-YEOHANGUL LETTER Y\
    \U-YEHANGUL LETTER YU-IHANGUL LETTER ARAEAHANGUL LETTER ARAEAEIDEOGRAPHIC ANNOTATION LINKING MARKIDEOGRAPHIC ANNOTATION REVERSE MARKIDEOGRAPHIC ANNOTATION ONE MARKIDEOGRAPHIC ANNOTATION TWO MARKIDEOGRAPHIC ANNOTATION THREE MARKIDEOGRAPHIC ANNOTATION FOUR M\
    \ARKIDEOGRAPHIC ANNOTATION TOP MARKIDEOGRAPHIC ANNOTATION MIDDLE MARKIDEOGRAPHIC ANNOTATION BOTTOM MARKIDEOGRAPHIC ANNOTATION FIRST MARKIDEOGRAPHIC ANNOTATION SECOND MARKIDEOGRAPHIC ANNOTATION THIRD MARKIDEOGRAPHIC ANNOTATION FOURTH MARKIDEOGRAPHIC ANNOTAT\
    \ION HEAVEN MARKIDEOGRAPHIC ANNOTATION EARTH MARKIDEOGRAPHIC ANNOTATION MAN MARKBOPOMOFO LETTER BUBOPOMOFO LETTER ZIBOPOMOFO LETTER JIBOPOMOFO LETTER GUBOPOMOFO LETTER EEBOPOMOFO LETTER ENNBOPOMOFO LETTER OOBOPOMOFO LETTER ONNBOPOMOFO LETTER IRBOPOMOFO LET\
    \TER ANNBOPOMOFO LETTER INNBOPOMOFO LETTER UNNBOPOMOFO LETTER IMBOPOMOFO LETTER NGGBOPOMOFO LETTER AINNBOPOMOFO LETTER AUNNBOPOMOFO LETTER AMBOPOMOFO LETTER OMBOPOMOFO LETTER ONGBOPOMOFO LETTER INNNBOPOMOFO FINAL LETTER PBOPOMOFO FINAL LETTER TBOPOMOFO FIN\
    \AL LETTER KBOPOMOFO FINAL LETTER HBOPOMOFO LETTER GHBOPOMOFO LETTER LHBOPOMOFO LETTER ZYBOPOMOFO FINAL LETTER GBOPOMOFO LETTER GWBOPOMOFO LETTER KWBOPOMOFO LETTER OEBOPOMOFO LETTER AHCJK STROKE TCJK STROKE WGCJK STROKE XGCJK STROKE BXGCJK STROKE SWCJK STR\
    \OKE HZZCJK STROKE HZGCJK STROKE HPCJK STROKE HZWGCJK STROKE SZWGCJK STROKE HZTCJK STROKE HZZPCJK STROKE HPWGCJK STROKE HZWCJK STROKE HZZZCJK STROKE NCJK STROKE HCJK STROKE SCJK STROKE PCJK STROKE SPCJK STROKE DCJK STROKE HZCJK STROKE HGCJK STROKE SZCJK ST\
    \ROKE SWZCJK STROKE STCJK STROKE SGCJK STROKE PDCJK STROKE PZCJK STROKE TNCJK STROKE SZZCJK STROKE SWGCJK STROKE HXWGCJK STROKE HZZZGCJK STROKE PGCJK STROKE QKATAKANA LETTER SMALL KUKATAKANA LETTER SMALL SIKATAKANA LETTER SMALL SUKATAKANA LETTER SMALL TOKA\
    \TAKANA LETTER SMALL NUKATAKANA LETTER SMALL HAKATAKANA LETTER SMALL HIKATAKANA LETTER SMALL HUKATAKANA LETTER SMALL HEKATAKANA LETTER SMALL HOKATAKANA LETTER SMALL MUKATAKANA LETTER SMALL RAKATAKANA LETTER SMALL RIKATAKANA LETTER SMALL RUKATAKANA LETTER S\
    \MALL REKATAKANA LETTER SMALL ROPARENTHESIZED HANGUL KIYEOKPARENTHESIZED HANGUL NIEUNPARENTHESIZED HANGUL TIKEUTPARENTHESIZED HANGUL RIEULPARENTHESIZED HANGUL MIEUMPARENTHESIZED HANGUL PIEUPPARENTHESIZED HANGUL SIOSPARENTHESIZED HANGUL IEUNGPARENTHESIZED H\
    \ANGUL CIEUCPARENTHESIZED HANGUL CHIEUCHPARENTHESIZED HANGUL KHIEUKHPARENTHESIZED HANGUL THIEUTHPARENTHESIZED HANGUL PHIEUPHPARENTHESIZED HANGUL HIEUHPARENTHESIZED HANGUL KIYEOK APARENTHESIZED HANGUL NIEUN APARENTHESIZED HANGUL TIKEUT APARENTHESIZED HANGUL\
    \ RIEUL APARENTHESIZED HANGUL MIEUM APARENTHESIZED HANGUL PIEUP APARENTHESIZED HANGUL SIOS APARENTHESIZED HANGUL IEUNG APARENTHESIZED HANGUL CIEUC APARENTHESIZED HANGUL CHIEUCH APARENTHESIZED HANGUL KHIEUKH APARENTHESIZED HANGUL THIEUTH APARENTHESIZED HANG\
    \UL PHIEUPH APARENTHESIZED HANGUL HIEUH APARENTHESIZED HANGUL CIEUC UPARENTHESIZED KOREAN CHARACTER OJEONPARENTHESIZED KOREAN CHARACTER O HUPARENTHESIZED IDEOGRAPH ONEPARENTHESIZED IDEOGRAPH TWOPARENTHESIZED IDEOGRAPH THREEPARENTHESIZED IDEOGRAPH FOURPAREN\
    \THESIZED IDEOGRAPH FIVEPARENTHESIZED IDEOGRAPH SIXPARENTHESIZED IDEOGRAPH SEVENPARENTHESIZED IDEOGRAPH EIGHTPARENTHESIZED IDEOGRAPH NINEPARENTHESIZED IDEOGRAPH TENPARENTHESIZED IDEOGRAPH MOONPARENTHESIZED IDEOGRAPH FIREPARENTHESIZED IDEOGRAPH WATERPARENTH\
    \ESIZED IDEOGRAPH WOODPARENTHESIZED IDEOGRAPH METALPARENTHESIZED IDEOGRAPH EARTHPARENTHESIZED IDEOGRAPH SUNPARENTHESIZED IDEOGRAPH STOCKPARENTHESIZED IDEOGRAPH HAVEPARENTHESIZED IDEOGRAPH SOCIETYPARENTHESIZED IDEOGRAPH NAMEPARENTHESIZED IDEOGRAPH SPECIALPA\
    \RENTHESIZED IDEOGRAPH FINANCIALPARENTHESIZED IDEOGRAPH CONGRATULATIONPARENTHESIZED IDEOGRAPH LABORPARENTHESIZED IDEOGRAPH REPRESENTPARENTHESIZED IDEOGRAPH CALLPARENTHESIZED IDEOGRAPH STUDYPARENTHESIZED IDEOGRAPH SUPERVISEPARENTHESIZED IDEOGRAPH ENTERPRISE\
    \PARENTHESIZED IDEOGRAPH RESOURCEPARENTHESIZED IDEOGRAPH ALLIANCEPARENTHESIZED IDEOGRAPH FESTIVALPARENTHESIZED IDEOGRAPH RESTPARENTHESIZED IDEOGRAPH SELFPARENTHESIZED IDEOGRAPH REACHCIRCLED IDEOGRAPH QUESTIONCIRCLED IDEOGRAPH KINDERGARTENCIRCLED IDEOGRAPH \
    \SCHOOLCIRCLED IDEOGRAPH KOTOCIRCLED NUMBER TEN ON BLACK SQUARECIRCLED NUMBER TWENTY ON BLACK SQUARECIRCLED NUMBER THIRTY ON BLACK SQUARECIRCLED NUMBER FORTY ON BLACK SQUARECIRCLED NUMBER FIFTY ON BLACK SQUARECIRCLED NUMBER SIXTY ON BLACK SQUARECIRCLED NUM\
    \BER SEVENTY ON BLACK SQUARECIRCLED NUMBER EIGHTY ON BLACK SQUAREPARTNERSHIP SIGNCIRCLED NUMBER TWENTY ONECIRCLED NUMBER TWENTY TWOCIRCLED NUMBER TWENTY THREECIRCLED NUMBER TWENTY FOURCIRCLED NUMBER TWENTY FIVECIRCLED NUMBER TWENTY SIXCIRCLED NUMBER TWENTY\
    \ SEVENCIRCLED NUMBER TWENTY EIGHTCIRCLED NUMBER TWENTY NINECIRCLED NUMBER THIRTYCIRCLED NUMBER THIRTY ONECIRCLED NUMBER THIRTY TWOCIRCLED NUMBER THIRTY THREECIRCLED NUMBER THIRTY FOURCIRCLED NUMBER THIRTY FIVECIRCLED HANGUL KIYEOKCIRCLED HANGUL NIEUNCIRCL\
    \ED HANGUL TIKEUTCIRCLED HANGUL RIEULCIRCLED HANGUL MIEUMCIRCLED HANGUL PIEUPCIRCLED HANGUL SIOSCIRCLED HANGUL IEUNGCIRCLED HANGUL CIEUCCIRCLED HANGUL CHIEUCHCIRCLED HANGUL KHIEUKHCIRCLED HANGUL THIEUTHCIRCLED HANGUL PHIEUPHCIRCLED HANGUL HIEUHCIRCLED HANG\
    \UL KIYEOK ACIRCLED HANGUL NIEUN ACIRCLED HANGUL TIKEUT ACIRCLED HANGUL RIEUL ACIRCLED HANGUL MIEUM ACIRCLED HANGUL PIEUP ACIRCLED HANGUL SIOS ACIRCLED HANGUL IEUNG ACIRCLED HANGUL CIEUC ACIRCLED HANGUL CHIEUCH ACIRCLED HANGUL KHIEUKH ACIRCLED HANGUL THIEU\
    \TH ACIRCLED HANGUL PHIEUPH ACIRCLED HANGUL HIEUH ACIRCLED KOREAN CHARACTER CHAMKOCIRCLED KOREAN CHARACTER JUEUICIRCLED HANGUL IEUNG UKOREAN STANDARD SYMBOLCIRCLED IDEOGRAPH ONECIRCLED IDEOGRAPH TWOCIRCLED IDEOGRAPH THREECIRCLED IDEOGRAPH FOURCIRCLED IDEOG\
    \RAPH FIVECIRCLED IDEOGRAPH SIXCIRCLED IDEOGRAPH SEVENCIRCLED IDEOGRAPH EIGHTCIRCLED IDEOGRAPH NINECIRCLED IDEOGRAPH TENCIRCLED IDEOGRAPH MOONCIRCLED IDEOGRAPH FIRECIRCLED IDEOGRAPH WATERCIRCLED IDEOGRAPH WOODCIRCLED IDEOGRAPH METALCIRCLED IDEOGRAPH EARTHC\
    \IRCLED IDEOGRAPH SUNCIRCLED IDEOGRAPH STOCKCIRCLED IDEOGRAPH HAVECIRCLED IDEOGRAPH SOCIETYCIRCLED IDEOGRAPH NAMECIRCLED IDEOGRAPH SPECIALCIRCLED IDEOGRAPH FINANCIALCIRCLED IDEOGRAPH CONGRATULATIONCIRCLED IDEOGRAPH LABORCIRCLED IDEOGRAPH SECRETCIRCLED IDEO\
    \GRAPH MALECIRCLED IDEOGRAPH FEMALECIRCLED IDEOGRAPH SUITABLECIRCLED IDEOGRAPH EXCELLENTCIRCLED IDEOGRAPH PRINTCIRCLED IDEOGRAPH ATTENTIONCIRCLED IDEOGRAPH ITEMCIRCLED IDEOGRAPH RESTCIRCLED IDEOGRAPH COPYCIRCLED IDEOGRAPH CORRECTCIRCLED IDEOGRAPH HIGHCIRCL\
    \ED IDEOGRAPH CENTRECIRCLED IDEOGRAPH LOWCIRCLED IDEOGRAPH LEFTCIRCLED IDEOGRAPH RIGHTCIRCLED IDEOGRAPH MEDICINECIRCLED IDEOGRAPH RELIGIONCIRCLED IDEOGRAPH STUDYCIRCLED IDEOGRAPH SUPERVISECIRCLED IDEOGRAPH ENTERPRISECIRCLED IDEOGRAPH RESOURCECIRCLED IDEOGR\
    \APH ALLIANCECIRCLED IDEOGRAPH NIGHTCIRCLED NUMBER THIRTY SIXCIRCLED NUMBER THIRTY SEVENCIRCLED NUMBER THIRTY EIGHTCIRCLED NUMBER THIRTY NINECIRCLED NUMBER FORTYCIRCLED NUMBER FORTY ONECIRCLED NUMBER FORTY TWOCIRCLED NUMBER FORTY THREECIRCLED NUMBER FORTY \
    \FOURCIRCLED NUMBER FORTY FIVECIRCLED NUMBER FORTY SIXCIRCLED NUMBER FORTY SEVENCIRCLED NUMBER FORTY EIGHTCIRCLED NUMBER FORTY NINECIRCLED NUMBER FIFTYIDEOGRAPHIC TELEGRAPH SYMBOL FOR JANUARYIDEOGRAPHIC TELEGRAPH SYMBOL FOR FEBRUARYIDEOGRAPHIC TELEGRAPH SY\
    \MBOL FOR MARCHIDEOGRAPHIC TELEGRAPH SYMBOL FOR APRILIDEOGRAPHIC TELEGRAPH SYMBOL FOR MAYIDEOGRAPHIC TELEGRAPH SYMBOL FOR JUNEIDEOGRAPHIC TELEGRAPH SYMBOL FOR JULYIDEOGRAPHIC TELEGRAPH SYMBOL FOR AUGUSTIDEOGRAPHIC TELEGRAPH SYMBOL FOR SEPTEMBERIDEOGRAPHIC \
    \TELEGRAPH SYMBOL FOR OCTOBERIDEOGRAPHIC TELEGRAPH SYMBOL FOR NOVEMBERIDEOGRAPHIC TELEGRAPH SYMBOL FOR DECEMBERSQUARE HGSQUARE ERGSQUARE EVLIMITED LIABILITY SIGNCIRCLED KATAKANA ACIRCLED KATAKANA ICIRCLED KATAKANA UCIRCLED KATAKANA ECIRCLED KATAKANA OCIRCL\
    \ED KATAKANA KACIRCLED KATAKANA KICIRCLED KATAKANA KUCIRCLED KATAKANA KECIRCLED KATAKANA KOCIRCLED KATAKANA SACIRCLED KATAKANA SICIRCLED KATAKANA SUCIRCLED KATAKANA SECIRCLED KATAKANA SOCIRCLED KATAKANA TACIRCLED KATAKANA TICIRCLED KATAKANA TUCIRCLED KATAK\
    \ANA TECIRCLED KATAKANA TOCIRCLED KATAKANA NACIRCLED KATAKANA NICIRCLED KATAKANA NUCIRCLED KATAKANA NECIRCLED KATAKANA NOCIRCLED KATAKANA HACIRCLED KATAKANA HICIRCLED KATAKANA HUCIRCLED KATAKANA HECIRCLED KATAKANA HOCIRCLED KATAKANA MACIRCLED KATAKANA MICI\
    \RCLED KATAKANA MUCIRCLED KATAKANA MECIRCLED KATAKANA MOCIRCLED KATAKANA YACIRCLED KATAKANA YUCIRCLED KATAKANA YOCIRCLED KATAKANA RACIRCLED KATAKANA RICIRCLED KATAKANA RUCIRCLED KATAKANA RECIRCLED KATAKANA ROCIRCLED KATAKANA WACIRCLED KATAKANA WICIRCLED KA\
    \TAKANA WECIRCLED KATAKANA WOSQUARE ERA NAME REIWASQUARE APAATOSQUARE ARUHUASQUARE ANPEASQUARE AARUSQUARE ININGUSQUARE INTISQUARE UONSQUARE ESUKUUDOSQUARE EEKAASQUARE ONSUSQUARE OOMUSQUARE KAIRISQUARE KARATTOSQUARE KARORIISQUARE GARONSQUARE GANMASQUARE GIG\
    \ASQUARE GINIISQUARE KYURIISQUARE GIRUDAASQUARE KIROSQUARE KIROGURAMUSQUARE KIROMEETORUSQUARE KIROWATTOSQUARE GURAMUSQUARE GURAMUTONSQUARE KURUZEIROSQUARE KUROONESQUARE KEESUSQUARE KORUNASQUARE KOOPOSQUARE SAIKURUSQUARE SANTIIMUSQUARE SIRINGUSQUARE SENTISQ\
    \UARE SENTOSQUARE DAASUSQUARE DESISQUARE DORUSQUARE TONSQUARE NANOSQUARE NOTTOSQUARE HAITUSQUARE PAASENTOSQUARE PAATUSQUARE BAARERUSQUARE PIASUTORUSQUARE PIKURUSQUARE PIKOSQUARE BIRUSQUARE HUARADDOSQUARE HUIITOSQUARE BUSSYERUSQUARE HURANSQUARE HEKUTAARUSQU\
    \ARE PESOSQUARE PENIHISQUARE HERUTUSQUARE PENSUSQUARE PEEZISQUARE BEETASQUARE POINTOSQUARE BORUTOSQUARE HONSQUARE PONDOSQUARE HOORUSQUARE HOONSQUARE MAIKUROSQUARE MAIRUSQUARE MAHHASQUARE MARUKUSQUARE MANSYONSQUARE MIKURONSQUARE MIRISQUARE MIRIBAARUSQUARE M\
    \EGASQUARE MEGATONSQUARE MEETORUSQUARE YAADOSQUARE YAARUSQUARE YUANSQUARE RITTORUSQUARE RIRASQUARE RUPIISQUARE RUUBURUSQUARE REMUSQUARE RENTOGENSQUARE WATTOIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ZEROIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ONEIDEOGRAPHIC TELEG\
    \RAPH SYMBOL FOR HOUR TWOIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR THREEIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FOURIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FIVEIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SIXIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SEVENIDEOGRAPHIC TELEGRAP\
    \H SYMBOL FOR HOUR EIGHTIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR NINEIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TENIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ELEVENIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWELVEIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR THIRTEENIDEOGRAPHIC TEL\
    \EGRAPH SYMBOL FOR HOUR FOURTEENIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FIFTEENIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SIXTEENIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SEVENTEENIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR EIGHTEENIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR NIN\
    \ETEENIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTYIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-ONEIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-TWOIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-THREEIDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-FOURSQUARE HPAS\
    \QUARE DASQUARE AUSQUARE BARSQUARE OVSQUARE PCSQUARE DMSQUARE DM SQUAREDSQUARE DM CUBEDSQUARE IUSQUARE ERA NAME HEISEISQUARE ERA NAME SYOUWASQUARE ERA NAME TAISYOUSQUARE ERA NAME MEIZISQUARE CORPORATIONSQUARE PA AMPSSQUARE NASQUARE MU ASQUARE MASQUARE KASQ\
    \UARE KBSQUARE MBSQUARE GBSQUARE CALSQUARE KCALSQUARE PFSQUARE NFSQUARE MU FSQUARE MU GSQUARE MGSQUARE KGSQUARE HZSQUARE KHZSQUARE MHZSQUARE GHZSQUARE THZSQUARE MU LSQUARE MLSQUARE DLSQUARE KLSQUARE FMSQUARE NMSQUARE MU MSQUARE MMSQUARE CMSQUARE KMSQUARE M\
    \M SQUAREDSQUARE CM SQUAREDSQUARE M SQUAREDSQUARE KM SQUAREDSQUARE MM CUBEDSQUARE CM CUBEDSQUARE M CUBEDSQUARE KM CUBEDSQUARE M OVER SSQUARE M OVER S SQUAREDSQUARE PASQUARE KPASQUARE MPASQUARE GPASQUARE RADSQUARE RAD OVER SSQUARE RAD OVER S SQUAREDSQUARE P\
    \SSQUARE NSSQUARE MU SSQUARE MSSQUARE PVSQUARE NVSQUARE MU VSQUARE MVSQUARE KVSQUARE MV MEGASQUARE PWSQUARE NWSQUARE MU WSQUARE MWSQUARE KWSQUARE MW MEGASQUARE K OHMSQUARE M OHMSQUARE AMSQUARE BQSQUARE CCSQUARE CDSQUARE C OVER KGSQUARE COSQUARE DBSQUARE GY\
    \SQUARE HASQUARE HPSQUARE INSQUARE KKSQUARE KM CAPITALSQUARE KTSQUARE LMSQUARE LNSQUARE LOGSQUARE LXSQUARE MB SMALLSQUARE MILSQUARE MOLSQUARE PHSQUARE PMSQUARE PPMSQUARE PRSQUARE SRSQUARE SVSQUARE WBSQUARE V OVER MSQUARE A OVER MIDEOGRAPHIC TELEGRAPH SYMBO\
    \L FOR DAY ONEIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWOIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THREEIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FOURIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FIVEIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SIXIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY S\
    \EVENIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY EIGHTIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY NINEIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TENIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY ELEVENIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWELVEIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRT\
    \EENIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FOURTEENIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FIFTEENIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SIXTEENIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SEVENTEENIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY EIGHTEENIDEOGRAPHIC TELEGRAPH SYMBOL\
    \ FOR DAY NINETEENIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTYIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-ONEIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-TWOIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-THREEIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-FOURIDEO\
    \GRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-FIVEIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-SIXIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-SEVENIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-EIGHTIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-NINEIDEOGRAPHIC TELEGR\
    \APH SYMBOL FOR DAY THIRTYIDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTY-ONESQUARE GALHEXAGRAM FOR THE CREATIVE HEAVENHEXAGRAM FOR THE RECEPTIVE EARTHHEXAGRAM FOR DIFFICULTY AT THE BEGINNINGHEXAGRAM FOR YOUTHFUL FOLLYHEXAGRAM FOR WAITINGHEXAGRAM FOR CONFLICTH\
    \EXAGRAM FOR THE ARMYHEXAGRAM FOR HOLDING TOGETHERHEXAGRAM FOR SMALL TAMINGHEXAGRAM FOR TREADINGHEXAGRAM FOR PEACEHEXAGRAM FOR STANDSTILLHEXAGRAM FOR FELLOWSHIPHEXAGRAM FOR GREAT POSSESSIONHEXAGRAM FOR MODESTYHEXAGRAM FOR ENTHUSIASMHEXAGRAM FOR FOLLOWINGHE\
    \XAGRAM FOR WORK ON THE DECAYEDHEXAGRAM FOR APPROACHHEXAGRAM FOR CONTEMPLATIONHEXAGRAM FOR BITING THROUGHHEXAGRAM FOR GRACEHEXAGRAM FOR SPLITTING APARTHEXAGRAM FOR RETURNHEXAGRAM FOR INNOCENCEHEXAGRAM FOR GREAT TAMINGHEXAGRAM FOR MOUTH CORNERSHEXAGRAM FOR \
    \GREAT PREPONDERANCEHEXAGRAM FOR THE ABYSMAL WATERHEXAGRAM FOR THE CLINGING FIREHEXAGRAM FOR INFLUENCEHEXAGRAM FOR DURATIONHEXAGRAM FOR RETREATHEXAGRAM FOR GREAT POWERHEXAGRAM FOR PROGRESSHEXAGRAM FOR DARKENING OF THE LIGHTHEXAGRAM FOR THE FAMILYHEXAGRAM F\
    \OR OPPOSITIONHEXAGRAM FOR OBSTRUCTIONHEXAGRAM FOR DELIVERANCEHEXAGRAM FOR DECREASEHEXAGRAM FOR INCREASEHEXAGRAM FOR BREAKTHROUGHHEXAGRAM FOR COMING TO MEETHEXAGRAM FOR GATHERING TOGETHERHEXAGRAM FOR PUSHING UPWARDHEXAGRAM FOR OPPRESSIONHEXAGRAM FOR THE WE\
    \LLHEXAGRAM FOR REVOLUTIONHEXAGRAM FOR THE CAULDRONHEXAGRAM FOR THE AROUSING THUNDERHEXAGRAM FOR THE KEEPING STILL MOUNTAINHEXAGRAM FOR DEVELOPMENTHEXAGRAM FOR THE MARRYING MAIDENHEXAGRAM FOR ABUNDANCEHEXAGRAM FOR THE WANDERERHEXAGRAM FOR THE GENTLE WINDHE\
    \XAGRAM FOR THE JOYOUS LAKEHEXAGRAM FOR DISPERSIONHEXAGRAM FOR LIMITATIONHEXAGRAM FOR INNER TRUTHHEXAGRAM FOR SMALL PREPONDERANCEHEXAGRAM FOR AFTER COMPLETIONHEXAGRAM FOR BEFORE COMPLETIONYI SYLLABLE ITYI SYLLABLE IXYI SYLLABLE IYI SYLLABLE IPYI SYLLABLE I\
    \ETYI SYLLABLE IEXYI SYLLABLE IEYI SYLLABLE IEPYI SYLLABLE ATYI SYLLABLE AXYI SYLLABLE AYI SYLLABLE APYI SYLLABLE UOXYI SYLLABLE UOYI SYLLABLE UOPYI SYLLABLE OTYI SYLLABLE OXYI SYLLABLE OYI SYLLABLE OPYI SYLLABLE EXYI SYLLABLE EYI SYLLABLE WUYI SYLLABLE BI\
    \TYI SYLLABLE BIXYI SYLLABLE BIYI SYLLABLE BIPYI SYLLABLE BIETYI SYLLABLE BIEXYI SYLLABLE BIEYI SYLLABLE BIEPYI SYLLABLE BATYI SYLLABLE BAXYI SYLLABLE BAYI SYLLABLE BAPYI SYLLABLE BUOXYI SYLLABLE BUOYI SYLLABLE BUOPYI SYLLABLE BOTYI SYLLABLE BOXYI SYLLABLE\
    \ BOYI SYLLABLE BOPYI SYLLABLE BEXYI SYLLABLE BEYI SYLLABLE BEPYI SYLLABLE BUTYI SYLLABLE BUXYI SYLLABLE BUYI SYLLABLE BUPYI SYLLABLE BURXYI SYLLABLE BURYI SYLLABLE BYTYI SYLLABLE BYXYI SYLLABLE BYYI SYLLABLE BYPYI SYLLABLE BYRXYI SYLLABLE BYRYI SYLLABLE P\
    \ITYI SYLLABLE PIXYI SYLLABLE PIYI SYLLABLE PIPYI SYLLABLE PIEXYI SYLLABLE PIEYI SYLLABLE PIEPYI SYLLABLE PATYI SYLLABLE PAXYI SYLLABLE PAYI SYLLABLE PAPYI SYLLABLE PUOXYI SYLLABLE PUOYI SYLLABLE PUOPYI SYLLABLE POTYI SYLLABLE POXYI SYLLABLE POYI SYLLABLE \
    \POPYI SYLLABLE PUTYI SYLLABLE PUXYI SYLLABLE PUYI SYLLABLE PUPYI SYLLABLE PURXYI SYLLABLE PURYI SYLLABLE PYTYI SYLLABLE PYXYI SYLLABLE PYYI SYLLABLE PYPYI SYLLABLE PYRXYI SYLLABLE PYRYI SYLLABLE BBITYI SYLLABLE BBIXYI SYLLABLE BBIYI SYLLABLE BBIPYI SYLLAB\
    \LE BBIETYI SYLLABLE BBIEXYI SYLLABLE BBIEYI SYLLABLE BBIEPYI SYLLABLE BBATYI SYLLABLE BBAXYI SYLLABLE BBAYI SYLLABLE BBAPYI SYLLABLE BBUOXYI SYLLABLE BBUOYI SYLLABLE BBUOPYI SYLLABLE BBOTYI SYLLABLE BBOXYI SYLLABLE BBOYI SYLLABLE BBOPYI SYLLABLE BBEXYI SY\
    \LLABLE BBEYI SYLLABLE BBEPYI SYLLABLE BBUTYI SYLLABLE BBUXYI SYLLABLE BBUYI SYLLABLE BBUPYI SYLLABLE BBURXYI SYLLABLE BBURYI SYLLABLE BBYTYI SYLLABLE BBYXYI SYLLABLE BBYYI SYLLABLE BBYPYI SYLLABLE NBITYI SYLLABLE NBIXYI SYLLABLE NBIYI SYLLABLE NBIPYI SYLL\
    \ABLE NBIEXYI SYLLABLE NBIEYI SYLLABLE NBIEPYI SYLLABLE NBATYI SYLLABLE NBAXYI SYLLABLE NBAYI SYLLABLE NBAPYI SYLLABLE NBOTYI SYLLABLE NBOXYI SYLLABLE NBOYI SYLLABLE NBOPYI SYLLABLE NBUTYI SYLLABLE NBUXYI SYLLABLE NBUYI SYLLABLE NBUPYI SYLLABLE NBURXYI SYL\
    \LABLE NBURYI SYLLABLE NBYTYI SYLLABLE NBYXYI SYLLABLE NBYYI SYLLABLE NBYPYI SYLLABLE NBYRXYI SYLLABLE NBYRYI SYLLABLE HMITYI SYLLABLE HMIXYI SYLLABLE HMIYI SYLLABLE HMIPYI SYLLABLE HMIEXYI SYLLABLE HMIEYI SYLLABLE HMIEPYI SYLLABLE HMATYI SYLLABLE HMAXYI S\
    \YLLABLE HMAYI SYLLABLE HMAPYI SYLLABLE HMUOXYI SYLLABLE HMUOYI SYLLABLE HMUOPYI SYLLABLE HMOTYI SYLLABLE HMOXYI SYLLABLE HMOYI SYLLABLE HMOPYI SYLLABLE HMUTYI SYLLABLE HMUXYI SYLLABLE HMUYI SYLLABLE HMUPYI SYLLABLE HMURXYI SYLLABLE HMURYI SYLLABLE HMYXYI \
    \SYLLABLE HMYYI SYLLABLE HMYPYI SYLLABLE HMYRXYI SYLLABLE HMYRYI SYLLABLE MITYI SYLLABLE MIXYI SYLLABLE MIYI SYLLABLE MIPYI SYLLABLE MIEXYI SYLLABLE MIEYI SYLLABLE MIEPYI SYLLABLE MATYI SYLLABLE MAXYI SYLLABLE MAYI SYLLABLE MAPYI SYLLABLE MUOTYI SYLLABLE M\
    \UOXYI SYLLABLE MUOYI SYLLABLE MUOPYI SYLLABLE MOTYI SYLLABLE MOXYI SYLLABLE MOYI SYLLABLE MOPYI SYLLABLE MEXYI SYLLABLE MEYI SYLLABLE MUTYI SYLLABLE MUXYI SYLLABLE MUYI SYLLABLE MUPYI SYLLABLE MURXYI SYLLABLE MURYI SYLLABLE MYTYI SYLLABLE MYXYI SYLLABLE M\
    \YYI SYLLABLE MYPYI SYLLABLE FITYI SYLLABLE FIXYI SYLLABLE FIYI SYLLABLE FIPYI SYLLABLE FATYI SYLLABLE FAXYI SYLLABLE FAYI SYLLABLE FAPYI SYLLABLE FOXYI SYLLABLE FOYI SYLLABLE FOPYI SYLLABLE FUTYI SYLLABLE FUXYI SYLLABLE FUYI SYLLABLE FUPYI SYLLABLE FURXYI\
    \ SYLLABLE FURYI SYLLABLE FYTYI SYLLABLE FYXYI SYLLABLE FYYI SYLLABLE FYPYI SYLLABLE VITYI SYLLABLE VIXYI SYLLABLE VIYI SYLLABLE VIPYI SYLLABLE VIETYI SYLLABLE VIEXYI SYLLABLE VIEYI SYLLABLE VIEPYI SYLLABLE VATYI SYLLABLE VAXYI SYLLABLE VAYI SYLLABLE VAPYI\
    \ SYLLABLE VOTYI SYLLABLE VOXYI SYLLABLE VOYI SYLLABLE VOPYI SYLLABLE VEXYI SYLLABLE VEPYI SYLLABLE VUTYI SYLLABLE VUXYI SYLLABLE VUYI SYLLABLE VUPYI SYLLABLE VURXYI SYLLABLE VURYI SYLLABLE VYTYI SYLLABLE VYXYI SYLLABLE VYYI SYLLABLE VYPYI SYLLABLE VYRXYI \
    \SYLLABLE VYRYI SYLLABLE DITYI SYLLABLE DIXYI SYLLABLE DIYI SYLLABLE DIPYI SYLLABLE DIEXYI SYLLABLE DIEYI SYLLABLE DIEPYI SYLLABLE DATYI SYLLABLE DAXYI SYLLABLE DAYI SYLLABLE DAPYI SYLLABLE DUOXYI SYLLABLE DUOYI SYLLABLE DOTYI SYLLABLE DOXYI SYLLABLE DOYI \
    \SYLLABLE DOPYI SYLLABLE DEXYI SYLLABLE DEYI SYLLABLE DEPYI SYLLABLE DUTYI SYLLABLE DUXYI SYLLABLE DUYI SYLLABLE DUPYI SYLLABLE DURXYI SYLLABLE DURYI SYLLABLE TITYI SYLLABLE TIXYI SYLLABLE TIYI SYLLABLE TIPYI SYLLABLE TIEXYI SYLLABLE TIEYI SYLLABLE TIEPYI \
    \SYLLABLE TATYI SYLLABLE TAXYI SYLLABLE TAYI SYLLABLE TAPYI SYLLABLE TUOTYI SYLLABLE TUOXYI SYLLABLE TUOYI SYLLABLE TUOPYI SYLLABLE TOTYI SYLLABLE TOXYI SYLLABLE TOYI SYLLABLE TOPYI SYLLABLE TEXYI SYLLABLE TEYI SYLLABLE TEPYI SYLLABLE TUTYI SYLLABLE TUXYI \
    \SYLLABLE TUYI SYLLABLE TUPYI SYLLABLE TURXYI SYLLABLE TURYI SYLLABLE DDITYI SYLLABLE DDIXYI SYLLABLE DDIYI SYLLABLE DDIPYI SYLLABLE DDIEXYI SYLLABLE DDIEYI SYLLABLE DDIEPYI SYLLABLE DDATYI SYLLABLE DDAXYI SYLLABLE DDAYI SYLLABLE DDAPYI SYLLABLE DDUOXYI SY\
    \LLABLE DDUOYI SYLLABLE DDUOPYI SYLLABLE DDOTYI SYLLABLE DDOXYI SYLLABLE DDOYI SYLLABLE DDOPYI SYLLABLE DDEXYI SYLLABLE DDEYI SYLLABLE DDEPYI SYLLABLE DDUTYI SYLLABLE DDUXYI SYLLABLE DDUYI SYLLABLE DDUPYI SYLLABLE DDURXYI SYLLABLE DDURYI SYLLABLE NDITYI SY\
    \LLABLE NDIXYI SYLLABLE NDIYI SYLLABLE NDIPYI SYLLABLE NDIEXYI SYLLABLE NDIEYI SYLLABLE NDATYI SYLLABLE NDAXYI SYLLABLE NDAYI SYLLABLE NDAPYI SYLLABLE NDOTYI SYLLABLE NDOXYI SYLLABLE NDOYI SYLLABLE NDOPYI SYLLABLE NDEXYI SYLLABLE NDEYI SYLLABLE NDEPYI SYLL\
    \ABLE NDUTYI SYLLABLE NDUXYI SYLLABLE NDUYI SYLLABLE NDUPYI SYLLABLE NDURXYI SYLLABLE NDURYI SYLLABLE HNITYI SYLLABLE HNIXYI SYLLABLE HNIYI SYLLABLE HNIPYI SYLLABLE HNIETYI SYLLABLE HNIEXYI SYLLABLE HNIEYI SYLLABLE HNIEPYI SYLLABLE HNATYI SYLLABLE HNAXYI S\
    \YLLABLE HNAYI SYLLABLE HNAPYI SYLLABLE HNUOXYI SYLLABLE HNUOYI SYLLABLE HNOTYI SYLLABLE HNOXYI SYLLABLE HNOPYI SYLLABLE HNEXYI SYLLABLE HNEYI SYLLABLE HNEPYI SYLLABLE HNUTYI SYLLABLE NITYI SYLLABLE NIXYI SYLLABLE NIYI SYLLABLE NIPYI SYLLABLE NIEXYI SYLLAB\
    \LE NIEYI SYLLABLE NIEPYI SYLLABLE NAXYI SYLLABLE NAYI SYLLABLE NAPYI SYLLABLE NUOXYI SYLLABLE NUOYI SYLLABLE NUOPYI SYLLABLE NOTYI SYLLABLE NOXYI SYLLABLE NOYI SYLLABLE NOPYI SYLLABLE NEXYI SYLLABLE NEYI SYLLABLE NEPYI SYLLABLE NUTYI SYLLABLE NUXYI SYLLAB\
    \LE NUYI SYLLABLE NUPYI SYLLABLE NURXYI SYLLABLE NURYI SYLLABLE HLITYI SYLLABLE HLIXYI SYLLABLE HLIYI SYLLABLE HLIPYI SYLLABLE HLIEXYI SYLLABLE HLIEYI SYLLABLE HLIEPYI SYLLABLE HLATYI SYLLABLE HLAXYI SYLLABLE HLAYI SYLLABLE HLAPYI SYLLABLE HLUOXYI SYLLABLE\
    \ HLUOYI SYLLABLE HLUOPYI SYLLABLE HLOXYI SYLLABLE HLOYI SYLLABLE HLOPYI SYLLABLE HLEXYI SYLLABLE HLEYI SYLLABLE HLEPYI SYLLABLE HLUTYI SYLLABLE HLUXYI SYLLABLE HLUYI SYLLABLE HLUPYI SYLLABLE HLURXYI SYLLABLE HLURYI SYLLABLE HLYTYI SYLLABLE HLYXYI SYLLABLE\
    \ HLYYI SYLLABLE HLYPYI SYLLABLE HLYRXYI SYLLABLE HLYRYI SYLLABLE LITYI SYLLABLE LIXYI SYLLABLE LIYI SYLLABLE LIPYI SYLLABLE LIETYI SYLLABLE LIEXYI SYLLABLE LIEYI SYLLABLE LIEPYI SYLLABLE LATYI SYLLABLE LAXYI SYLLABLE LAYI SYLLABLE LAPYI SYLLABLE LUOTYI SY\
    \LLABLE LUOXYI SYLLABLE LUOYI SYLLABLE LUOPYI SYLLABLE LOTYI SYLLABLE LOXYI SYLLABLE LOYI SYLLABLE LOPYI SYLLABLE LEXYI SYLLABLE LEYI SYLLABLE LEPYI SYLLABLE LUTYI SYLLABLE LUXYI SYLLABLE LUYI SYLLABLE LUPYI SYLLABLE LURXYI SYLLABLE LURYI SYLLABLE LYTYI SY\
    \LLABLE LYXYI SYLLABLE LYYI SYLLABLE LYPYI SYLLABLE LYRXYI SYLLABLE LYRYI SYLLABLE GITYI SYLLABLE GIXYI SYLLABLE GIYI SYLLABLE GIPYI SYLLABLE GIETYI SYLLABLE GIEXYI SYLLABLE GIEYI SYLLABLE GIEPYI SYLLABLE GATYI SYLLABLE GAXYI SYLLABLE GAYI SYLLABLE GAPYI S\
    \YLLABLE GUOTYI SYLLABLE GUOXYI SYLLABLE GUOYI SYLLABLE GUOPYI SYLLABLE GOTYI SYLLABLE GOXYI SYLLABLE GOYI SYLLABLE GOPYI SYLLABLE GETYI SYLLABLE GEXYI SYLLABLE GEYI SYLLABLE GEPYI SYLLABLE GUTYI SYLLABLE GUXYI SYLLABLE GUYI SYLLABLE GUPYI SYLLABLE GURXYI \
    \SYLLABLE GURYI SYLLABLE KITYI SYLLABLE KIXYI SYLLABLE KIYI SYLLABLE KIPYI SYLLABLE KIEXYI SYLLABLE KIEYI SYLLABLE KIEPYI SYLLABLE KATYI SYLLABLE KAXYI SYLLABLE KAYI SYLLABLE KAPYI SYLLABLE KUOXYI SYLLABLE KUOYI SYLLABLE KUOPYI SYLLABLE KOTYI SYLLABLE KOXY\
    \I SYLLABLE KOYI SYLLABLE KOPYI SYLLABLE KETYI SYLLABLE KEXYI SYLLABLE KEYI SYLLABLE KEPYI SYLLABLE KUTYI SYLLABLE KUXYI SYLLABLE KUYI SYLLABLE KUPYI SYLLABLE KURXYI SYLLABLE KURYI SYLLABLE GGITYI SYLLABLE GGIXYI SYLLABLE GGIYI SYLLABLE GGIEXYI SYLLABLE GG\
    \IEYI SYLLABLE GGIEPYI SYLLABLE GGATYI SYLLABLE GGAXYI SYLLABLE GGAYI SYLLABLE GGAPYI SYLLABLE GGUOTYI SYLLABLE GGUOXYI SYLLABLE GGUOYI SYLLABLE GGUOPYI SYLLABLE GGOTYI SYLLABLE GGOXYI SYLLABLE GGOYI SYLLABLE GGOPYI SYLLABLE GGETYI SYLLABLE GGEXYI SYLLABLE\
    \ GGEYI SYLLABLE GGEPYI SYLLABLE GGUTYI SYLLABLE GGUXYI SYLLABLE GGUYI SYLLABLE GGUPYI SYLLABLE GGURXYI SYLLABLE GGURYI SYLLABLE MGIEXYI SYLLABLE MGIEYI SYLLABLE MGATYI SYLLABLE MGAXYI SYLLABLE MGAYI SYLLABLE MGAPYI SYLLABLE MGUOXYI SYLLABLE MGUOYI SYLLABL\
    \E MGUOPYI SYLLABLE MGOTYI SYLLABLE MGOXYI SYLLABLE MGOYI SYLLABLE MGOPYI SYLLABLE MGEXYI SYLLABLE MGEYI SYLLABLE MGEPYI SYLLABLE MGUTYI SYLLABLE MGUXYI SYLLABLE MGUYI SYLLABLE MGUPYI SYLLABLE MGURXYI SYLLABLE MGURYI SYLLABLE HXITYI SYLLABLE HXIXYI SYLLABL\
    \E HXIYI SYLLABLE HXIPYI SYLLABLE HXIETYI SYLLABLE HXIEXYI SYLLABLE HXIEYI SYLLABLE HXIEPYI SYLLABLE HXATYI SYLLABLE HXAXYI SYLLABLE HXAYI SYLLABLE HXAPYI SYLLABLE HXUOTYI SYLLABLE HXUOXYI SYLLABLE HXUOYI SYLLABLE HXUOPYI SYLLABLE HXOTYI SYLLABLE HXOXYI SY\
    \LLABLE HXOYI SYLLABLE HXOPYI SYLLABLE HXEXYI SYLLABLE HXEYI SYLLABLE HXEPYI SYLLABLE NGIEXYI SYLLABLE NGIEYI SYLLABLE NGIEPYI SYLLABLE NGATYI SYLLABLE NGAXYI SYLLABLE NGAYI SYLLABLE NGAPYI SYLLABLE NGUOTYI SYLLABLE NGUOXYI SYLLABLE NGUOYI SYLLABLE NGOTYI \
    \SYLLABLE NGOXYI SYLLABLE NGOYI SYLLABLE NGOPYI SYLLABLE NGEXYI SYLLABLE NGEYI SYLLABLE NGEPYI SYLLABLE HITYI SYLLABLE HIEXYI SYLLABLE HIEYI SYLLABLE HATYI SYLLABLE HAXYI SYLLABLE HAYI SYLLABLE HAPYI SYLLABLE HUOTYI SYLLABLE HUOXYI SYLLABLE HUOYI SYLLABLE \
    \HUOPYI SYLLABLE HOTYI SYLLABLE HOXYI SYLLABLE HOYI SYLLABLE HOPYI SYLLABLE HEXYI SYLLABLE HEYI SYLLABLE HEPYI SYLLABLE WATYI SYLLABLE WAXYI SYLLABLE WAYI SYLLABLE WAPYI SYLLABLE WUOXYI SYLLABLE WUOYI SYLLABLE WUOPYI SYLLABLE WOXYI SYLLABLE WOYI SYLLABLE W\
    \OPYI SYLLABLE WEXYI SYLLABLE WEYI SYLLABLE WEPYI SYLLABLE ZITYI SYLLABLE ZIXYI SYLLABLE ZIYI SYLLABLE ZIPYI SYLLABLE ZIEXYI SYLLABLE ZIEYI SYLLABLE ZIEPYI SYLLABLE ZATYI SYLLABLE ZAXYI SYLLABLE ZAYI SYLLABLE ZAPYI SYLLABLE ZUOXYI SYLLABLE ZUOYI SYLLABLE Z\
    \UOPYI SYLLABLE ZOTYI SYLLABLE ZOXYI SYLLABLE ZOYI SYLLABLE ZOPYI SYLLABLE ZEXYI SYLLABLE ZEYI SYLLABLE ZEPYI SYLLABLE ZUTYI SYLLABLE ZUXYI SYLLABLE ZUYI SYLLABLE ZUPYI SYLLABLE ZURXYI SYLLABLE ZURYI SYLLABLE ZYTYI SYLLABLE ZYXYI SYLLABLE ZYYI SYLLABLE ZYP\
    \YI SYLLABLE ZYRXYI SYLLABLE ZYRYI SYLLABLE CITYI SYLLABLE CIXYI SYLLABLE CIYI SYLLABLE CIPYI SYLLABLE CIETYI SYLLABLE CIEXYI SYLLABLE CIEYI SYLLABLE CIEPYI SYLLABLE CATYI SYLLABLE CAXYI SYLLABLE CAYI SYLLABLE CAPYI SYLLABLE CUOXYI SYLLABLE CUOYI SYLLABLE \
    \CUOPYI SYLLABLE COTYI SYLLABLE COXYI SYLLABLE COYI SYLLABLE COPYI SYLLABLE CEXYI SYLLABLE CEYI SYLLABLE CEPYI SYLLABLE CUTYI SYLLABLE CUXYI SYLLABLE CUYI SYLLABLE CUPYI SYLLABLE CURXYI SYLLABLE CURYI SYLLABLE CYTYI SYLLABLE CYXYI SYLLABLE CYYI SYLLABLE CY\
    \PYI SYLLABLE CYRXYI SYLLABLE CYRYI SYLLABLE ZZITYI SYLLABLE ZZIXYI SYLLABLE ZZIYI SYLLABLE ZZIPYI SYLLABLE ZZIETYI SYLLABLE ZZIEXYI SYLLABLE ZZIEYI SYLLABLE ZZIEPYI SYLLABLE ZZATYI SYLLABLE ZZAXYI SYLLABLE ZZAYI SYLLABLE ZZAPYI SYLLABLE ZZOXYI SYLLABLE ZZ\
    \OYI SYLLABLE ZZOPYI SYLLABLE ZZEXYI SYLLABLE ZZEYI SYLLABLE ZZEPYI SYLLABLE ZZUXYI SYLLABLE ZZUYI SYLLABLE ZZUPYI SYLLABLE ZZURXYI SYLLABLE ZZURYI SYLLABLE ZZYTYI SYLLABLE ZZYXYI SYLLABLE ZZYYI SYLLABLE ZZYPYI SYLLABLE ZZYRXYI SYLLABLE ZZYRYI SYLLABLE NZI\
    \TYI SYLLABLE NZIXYI SYLLABLE NZIYI SYLLABLE NZIPYI SYLLABLE NZIEXYI SYLLABLE NZIEYI SYLLABLE NZIEPYI SYLLABLE NZATYI SYLLABLE NZAXYI SYLLABLE NZAYI SYLLABLE NZAPYI SYLLABLE NZUOXYI SYLLABLE NZUOYI SYLLABLE NZOXYI SYLLABLE NZOPYI SYLLABLE NZEXYI SYLLABLE N\
    \ZEYI SYLLABLE NZUXYI SYLLABLE NZUYI SYLLABLE NZUPYI SYLLABLE NZURXYI SYLLABLE NZURYI SYLLABLE NZYTYI SYLLABLE NZYXYI SYLLABLE NZYYI SYLLABLE NZYPYI SYLLABLE NZYRXYI SYLLABLE NZYRYI SYLLABLE SITYI SYLLABLE SIXYI SYLLABLE SIYI SYLLABLE SIPYI SYLLABLE SIEXYI\
    \ SYLLABLE SIEYI SYLLABLE SIEPYI SYLLABLE SATYI SYLLABLE SAXYI SYLLABLE SAYI SYLLABLE SAPYI SYLLABLE SUOXYI SYLLABLE SUOYI SYLLABLE SUOPYI SYLLABLE SOTYI SYLLABLE SOXYI SYLLABLE SOYI SYLLABLE SOPYI SYLLABLE SEXYI SYLLABLE SEYI SYLLABLE SEPYI SYLLABLE SUTYI\
    \ SYLLABLE SUXYI SYLLABLE SUYI SYLLABLE SUPYI SYLLABLE SURXYI SYLLABLE SURYI SYLLABLE SYTYI SYLLABLE SYXYI SYLLABLE SYYI SYLLABLE SYPYI SYLLABLE SYRXYI SYLLABLE SYRYI SYLLABLE SSITYI SYLLABLE SSIXYI SYLLABLE SSIYI SYLLABLE SSIPYI SYLLABLE SSIEXYI SYLLABLE \
    \SSIEYI SYLLABLE SSIEPYI SYLLABLE SSATYI SYLLABLE SSAXYI SYLLABLE SSAYI SYLLABLE SSAPYI SYLLABLE SSOTYI SYLLABLE SSOXYI SYLLABLE SSOYI SYLLABLE SSOPYI SYLLABLE SSEXYI SYLLABLE SSEYI SYLLABLE SSEPYI SYLLABLE SSUTYI SYLLABLE SSUXYI SYLLABLE SSUYI SYLLABLE SS\
    \UPYI SYLLABLE SSYTYI SYLLABLE SSYXYI SYLLABLE SSYYI SYLLABLE SSYPYI SYLLABLE SSYRXYI SYLLABLE SSYRYI SYLLABLE ZHATYI SYLLABLE ZHAXYI SYLLABLE ZHAYI SYLLABLE ZHAPYI SYLLABLE ZHUOXYI SYLLABLE ZHUOYI SYLLABLE ZHUOPYI SYLLABLE ZHOTYI SYLLABLE ZHOXYI SYLLABLE \
    \ZHOYI SYLLABLE ZHOPYI SYLLABLE ZHETYI SYLLABLE ZHEXYI SYLLABLE ZHEYI SYLLABLE ZHEPYI SYLLABLE ZHUTYI SYLLABLE ZHUXYI SYLLABLE ZHUYI SYLLABLE ZHUPYI SYLLABLE ZHURXYI SYLLABLE ZHURYI SYLLABLE ZHYTYI SYLLABLE ZHYXYI SYLLABLE ZHYYI SYLLABLE ZHYPYI SYLLABLE ZH\
    \YRXYI SYLLABLE ZHYRYI SYLLABLE CHATYI SYLLABLE CHAXYI SYLLABLE CHAYI SYLLABLE CHAPYI SYLLABLE CHUOTYI SYLLABLE CHUOXYI SYLLABLE CHUOYI SYLLABLE CHUOPYI SYLLABLE CHOTYI SYLLABLE CHOXYI SYLLABLE CHOYI SYLLABLE CHOPYI SYLLABLE CHETYI SYLLABLE CHEXYI SYLLABLE\
    \ CHEYI SYLLABLE CHEPYI SYLLABLE CHUXYI SYLLABLE CHUYI SYLLABLE CHUPYI SYLLABLE CHURXYI SYLLABLE CHURYI SYLLABLE CHYTYI SYLLABLE CHYXYI SYLLABLE CHYYI SYLLABLE CHYPYI SYLLABLE CHYRXYI SYLLABLE CHYRYI SYLLABLE RRAXYI SYLLABLE RRAYI SYLLABLE RRUOXYI SYLLABLE\
    \ RRUOYI SYLLABLE RROTYI SYLLABLE RROXYI SYLLABLE RROYI SYLLABLE RROPYI SYLLABLE RRETYI SYLLABLE RREXYI SYLLABLE RREYI SYLLABLE RREPYI SYLLABLE RRUTYI SYLLABLE RRUXYI SYLLABLE RRUYI SYLLABLE RRUPYI SYLLABLE RRURXYI SYLLABLE RRURYI SYLLABLE RRYTYI SYLLABLE \
    \RRYXYI SYLLABLE RRYYI SYLLABLE RRYPYI SYLLABLE RRYRXYI SYLLABLE RRYRYI SYLLABLE NRATYI SYLLABLE NRAXYI SYLLABLE NRAYI SYLLABLE NRAPYI SYLLABLE NROXYI SYLLABLE NROYI SYLLABLE NROPYI SYLLABLE NRETYI SYLLABLE NREXYI SYLLABLE NREYI SYLLABLE NREPYI SYLLABLE NR\
    \UTYI SYLLABLE NRUXYI SYLLABLE NRUYI SYLLABLE NRUPYI SYLLABLE NRURXYI SYLLABLE NRURYI SYLLABLE NRYTYI SYLLABLE NRYXYI SYLLABLE NRYYI SYLLABLE NRYPYI SYLLABLE NRYRXYI SYLLABLE NRYRYI SYLLABLE SHATYI SYLLABLE SHAXYI SYLLABLE SHAYI SYLLABLE SHAPYI SYLLABLE SH\
    \UOXYI SYLLABLE SHUOYI SYLLABLE SHUOPYI SYLLABLE SHOTYI SYLLABLE SHOXYI SYLLABLE SHOYI SYLLABLE SHOPYI SYLLABLE SHETYI SYLLABLE SHEXYI SYLLABLE SHEYI SYLLABLE SHEPYI SYLLABLE SHUTYI SYLLABLE SHUXYI SYLLABLE SHUYI SYLLABLE SHUPYI SYLLABLE SHURXYI SYLLABLE S\
    \HURYI SYLLABLE SHYTYI SYLLABLE SHYXYI SYLLABLE SHYYI SYLLABLE SHYPYI SYLLABLE SHYRXYI SYLLABLE SHYRYI SYLLABLE RATYI SYLLABLE RAXYI SYLLABLE RAYI SYLLABLE RAPYI SYLLABLE RUOXYI SYLLABLE RUOYI SYLLABLE RUOPYI SYLLABLE ROTYI SYLLABLE ROXYI SYLLABLE ROYI SYL\
    \LABLE ROPYI SYLLABLE REXYI SYLLABLE REYI SYLLABLE REPYI SYLLABLE RUTYI SYLLABLE RUXYI SYLLABLE RUYI SYLLABLE RUPYI SYLLABLE RURXYI SYLLABLE RURYI SYLLABLE RYTYI SYLLABLE RYXYI SYLLABLE RYYI SYLLABLE RYPYI SYLLABLE RYRXYI SYLLABLE RYRYI SYLLABLE JITYI SYLL\
    \ABLE JIXYI SYLLABLE JIYI SYLLABLE JIPYI SYLLABLE JIETYI SYLLABLE JIEXYI SYLLABLE JIEYI SYLLABLE JIEPYI SYLLABLE JUOTYI SYLLABLE JUOXYI SYLLABLE JUOYI SYLLABLE JUOPYI SYLLABLE JOTYI SYLLABLE JOXYI SYLLABLE JOYI SYLLABLE JOPYI SYLLABLE JUTYI SYLLABLE JUXYI \
    \SYLLABLE JUYI SYLLABLE JUPYI SYLLABLE JURXYI SYLLABLE JURYI SYLLABLE JYTYI SYLLABLE JYXYI SYLLABLE JYYI SYLLABLE JYPYI SYLLABLE JYRXYI SYLLABLE JYRYI SYLLABLE QITYI SYLLABLE QIXYI SYLLABLE QIYI SYLLABLE QIPYI SYLLABLE QIETYI SYLLABLE QIEXYI SYLLABLE QIEYI\
    \ SYLLABLE QIEPYI SYLLABLE QUOTYI SYLLABLE QUOXYI SYLLABLE QUOYI SYLLABLE QUOPYI SYLLABLE QOTYI SYLLABLE QOXYI SYLLABLE QOYI SYLLABLE QOPYI SYLLABLE QUTYI SYLLABLE QUXYI SYLLABLE QUYI SYLLABLE QUPYI SYLLABLE QURXYI SYLLABLE QURYI SYLLABLE QYTYI SYLLABLE QY\
    \XYI SYLLABLE QYYI SYLLABLE QYPYI SYLLABLE QYRXYI SYLLABLE QYRYI SYLLABLE JJITYI SYLLABLE JJIXYI SYLLABLE JJIYI SYLLABLE JJIPYI SYLLABLE JJIETYI SYLLABLE JJIEXYI SYLLABLE JJIEYI SYLLABLE JJIEPYI SYLLABLE JJUOXYI SYLLABLE JJUOYI SYLLABLE JJUOPYI SYLLABLE JJ\
    \OTYI SYLLABLE JJOXYI SYLLABLE JJOYI SYLLABLE JJOPYI SYLLABLE JJUTYI SYLLABLE JJUXYI SYLLABLE JJUYI SYLLABLE JJUPYI SYLLABLE JJURXYI SYLLABLE JJURYI SYLLABLE JJYTYI SYLLABLE JJYXYI SYLLABLE JJYYI SYLLABLE JJYPYI SYLLABLE NJITYI SYLLABLE NJIXYI SYLLABLE NJI\
    \YI SYLLABLE NJIPYI SYLLABLE NJIETYI SYLLABLE NJIEXYI SYLLABLE NJIEYI SYLLABLE NJIEPYI SYLLABLE NJUOXYI SYLLABLE NJUOYI SYLLABLE NJOTYI SYLLABLE NJOXYI SYLLABLE NJOYI SYLLABLE NJOPYI SYLLABLE NJUXYI SYLLABLE NJUYI SYLLABLE NJUPYI SYLLABLE NJURXYI SYLLABLE \
    \NJURYI SYLLABLE NJYTYI SYLLABLE NJYXYI SYLLABLE NJYYI SYLLABLE NJYPYI SYLLABLE NJYRXYI SYLLABLE NJYRYI SYLLABLE NYITYI SYLLABLE NYIXYI SYLLABLE NYIYI SYLLABLE NYIPYI SYLLABLE NYIETYI SYLLABLE NYIEXYI SYLLABLE NYIEYI SYLLABLE NYIEPYI SYLLABLE NYUOXYI SYLLA\
    \BLE NYUOYI SYLLABLE NYUOPYI SYLLABLE NYOTYI SYLLABLE NYOXYI SYLLABLE NYOYI SYLLABLE NYOPYI SYLLABLE NYUTYI SYLLABLE NYUXYI SYLLABLE NYUYI SYLLABLE NYUPYI SYLLABLE XITYI SYLLABLE XIXYI SYLLABLE XIYI SYLLABLE XIPYI SYLLABLE XIETYI SYLLABLE XIEXYI SYLLABLE X\
    \IEYI SYLLABLE XIEPYI SYLLABLE XUOXYI SYLLABLE XUOYI SYLLABLE XOTYI SYLLABLE XOXYI SYLLABLE XOYI SYLLABLE XOPYI SYLLABLE XYTYI SYLLABLE XYXYI SYLLABLE XYYI SYLLABLE XYPYI SYLLABLE XYRXYI SYLLABLE XYRYI SYLLABLE YITYI SYLLABLE YIXYI SYLLABLE YIYI SYLLABLE Y\
    \IPYI SYLLABLE YIETYI SYLLABLE YIEXYI SYLLABLE YIEYI SYLLABLE YIEPYI SYLLABLE YUOTYI SYLLABLE YUOXYI SYLLABLE YUOYI SYLLABLE YUOPYI SYLLABLE YOTYI SYLLABLE YOXYI SYLLABLE YOYI SYLLABLE YOPYI SYLLABLE YUTYI SYLLABLE YUXYI SYLLABLE YUYI SYLLABLE YUPYI SYLLAB\
    \LE YURXYI SYLLABLE YURYI SYLLABLE YYTYI SYLLABLE YYXYI SYLLABLE YYYI SYLLABLE YYPYI SYLLABLE YYRXYI SYLLABLE YYRYI RADICAL QOTYI RADICAL LIYI RADICAL KITYI RADICAL NYIPYI RADICAL CYPYI RADICAL SSIYI RADICAL GGOPYI RADICAL GEPYI RADICAL MIYI RADICAL HXITYI\
    \ RADICAL LYRYI RADICAL BBUTYI RADICAL MOPYI RADICAL YOYI RADICAL PUTYI RADICAL HXUOYI RADICAL TATYI RADICAL GAYI RADICAL ZUPYI RADICAL CYTYI RADICAL DDURYI RADICAL BURYI RADICAL GGUOYI RADICAL NYOPYI RADICAL TUYI RADICAL OPYI RADICAL JJUTYI RADICAL ZOTYI \
    \RADICAL PYTYI RADICAL HMOYI RADICAL YITYI RADICAL VURYI RADICAL SHYYI RADICAL VEPYI RADICAL ZAYI RADICAL JOYI RADICAL NZUPYI RADICAL JJYYI RADICAL GOTYI RADICAL JJIEYI RADICAL WOYI RADICAL DUYI RADICAL SHURYI RADICAL LIEYI RADICAL CYYI RADICAL CUOPYI RADI\
    \CAL CIPYI RADICAL HXOPYI RADICAL SHATYI RADICAL ZURYI RADICAL SHOPYI RADICAL CHEYI RADICAL ZZIETYI RADICAL NBIEYI RADICAL KELISU LETTER BALISU LETTER PALISU LETTER PHALISU LETTER DALISU LETTER TALISU LETTER THALISU LETTER GALISU LETTER KALISU LETTER KHALI\
    \SU LETTER JALISU LETTER CALISU LETTER CHALISU LETTER DZALISU LETTER TSALISU LETTER TSHALISU LETTER MALISU LETTER NALISU LETTER LALISU LETTER SALISU LETTER ZHALISU LETTER ZALISU LETTER NGALISU LETTER HALISU LETTER XALISU LETTER HHALISU LETTER FALISU LETTER\
    \ WALISU LETTER SHALISU LETTER YALISU LETTER GHALISU LETTER ALISU LETTER AELISU LETTER ELISU LETTER EULISU LETTER ILISU LETTER OLISU LETTER ULISU LETTER UELISU LETTER UHLISU LETTER OELISU LETTER TONE MYA TILISU LETTER TONE NA POLISU LETTER TONE MYA CYALISU\
    \ LETTER TONE MYA BOLISU LETTER TONE MYA NALISU LETTER TONE MYA JEULISU PUNCTUATION COMMALISU PUNCTUATION FULL STOPVAI SYLLABLE EEVAI SYLLABLE EENVAI SYLLABLE HEEVAI SYLLABLE WEEVAI SYLLABLE WEENVAI SYLLABLE PEEVAI SYLLABLE BHEEVAI SYLLABLE BEEVAI SYLLABLE\
    \ MBEEVAI SYLLABLE KPEEVAI SYLLABLE MGBEEVAI SYLLABLE GBEEVAI SYLLABLE FEEVAI SYLLABLE VEEVAI SYLLABLE TEEVAI SYLLABLE THEEVAI SYLLABLE DHEEVAI SYLLABLE DHHEEVAI SYLLABLE LEEVAI SYLLABLE REEVAI SYLLABLE DEEVAI SYLLABLE NDEEVAI SYLLABLE SEEVAI SYLLABLE SHEE\
    \VAI SYLLABLE ZEEVAI SYLLABLE ZHEEVAI SYLLABLE CEEVAI SYLLABLE JEEVAI SYLLABLE NJEEVAI SYLLABLE YEEVAI SYLLABLE KEEVAI SYLLABLE NGGEEVAI SYLLABLE GEEVAI SYLLABLE MEEVAI SYLLABLE NEEVAI SYLLABLE NYEEVAI SYLLABLE IVAI SYLLABLE INVAI SYLLABLE HIVAI SYLLABLE H\
    \INVAI SYLLABLE WIVAI SYLLABLE WINVAI SYLLABLE PIVAI SYLLABLE BHIVAI SYLLABLE BIVAI SYLLABLE MBIVAI SYLLABLE KPIVAI SYLLABLE MGBIVAI SYLLABLE GBIVAI SYLLABLE FIVAI SYLLABLE VIVAI SYLLABLE TIVAI SYLLABLE THIVAI SYLLABLE DHIVAI SYLLABLE DHHIVAI SYLLABLE LIVA\
    \I SYLLABLE RIVAI SYLLABLE DIVAI SYLLABLE NDIVAI SYLLABLE SIVAI SYLLABLE SHIVAI SYLLABLE ZIVAI SYLLABLE ZHIVAI SYLLABLE CIVAI SYLLABLE JIVAI SYLLABLE NJIVAI SYLLABLE YIVAI SYLLABLE KIVAI SYLLABLE NGGIVAI SYLLABLE GIVAI SYLLABLE MIVAI SYLLABLE NIVAI SYLLABL\
    \E NYIVAI SYLLABLE AVAI SYLLABLE ANVAI SYLLABLE NGANVAI SYLLABLE HAVAI SYLLABLE HANVAI SYLLABLE WAVAI SYLLABLE WANVAI SYLLABLE PAVAI SYLLABLE BHAVAI SYLLABLE BAVAI SYLLABLE MBAVAI SYLLABLE KPAVAI SYLLABLE KPANVAI SYLLABLE MGBAVAI SYLLABLE GBAVAI SYLLABLE F\
    \AVAI SYLLABLE VAVAI SYLLABLE TAVAI SYLLABLE THAVAI SYLLABLE DHAVAI SYLLABLE DHHAVAI SYLLABLE LAVAI SYLLABLE RAVAI SYLLABLE DAVAI SYLLABLE NDAVAI SYLLABLE SAVAI SYLLABLE SHAVAI SYLLABLE ZAVAI SYLLABLE ZHAVAI SYLLABLE CAVAI SYLLABLE JAVAI SYLLABLE NJAVAI SY\
    \LLABLE YAVAI SYLLABLE KAVAI SYLLABLE KANVAI SYLLABLE NGGAVAI SYLLABLE GAVAI SYLLABLE MAVAI SYLLABLE NAVAI SYLLABLE NYAVAI SYLLABLE OOVAI SYLLABLE OONVAI SYLLABLE HOOVAI SYLLABLE WOOVAI SYLLABLE WOONVAI SYLLABLE POOVAI SYLLABLE BHOOVAI SYLLABLE BOOVAI SYLL\
    \ABLE MBOOVAI SYLLABLE KPOOVAI SYLLABLE MGBOOVAI SYLLABLE GBOOVAI SYLLABLE FOOVAI SYLLABLE VOOVAI SYLLABLE TOOVAI SYLLABLE THOOVAI SYLLABLE DHOOVAI SYLLABLE DHHOOVAI SYLLABLE LOOVAI SYLLABLE ROOVAI SYLLABLE DOOVAI SYLLABLE NDOOVAI SYLLABLE SOOVAI SYLLABLE \
    \SHOOVAI SYLLABLE ZOOVAI SYLLABLE ZHOOVAI SYLLABLE COOVAI SYLLABLE JOOVAI SYLLABLE NJOOVAI SYLLABLE YOOVAI SYLLABLE KOOVAI SYLLABLE NGGOOVAI SYLLABLE GOOVAI SYLLABLE MOOVAI SYLLABLE NOOVAI SYLLABLE NYOOVAI SYLLABLE UVAI SYLLABLE UNVAI SYLLABLE HUVAI SYLLAB\
    \LE HUNVAI SYLLABLE WUVAI SYLLABLE WUNVAI SYLLABLE PUVAI SYLLABLE BHUVAI SYLLABLE BUVAI SYLLABLE MBUVAI SYLLABLE KPUVAI SYLLABLE MGBUVAI SYLLABLE GBUVAI SYLLABLE FUVAI SYLLABLE VUVAI SYLLABLE TUVAI SYLLABLE THUVAI SYLLABLE DHUVAI SYLLABLE DHHUVAI SYLLABLE \
    \LUVAI SYLLABLE RUVAI SYLLABLE DUVAI SYLLABLE NDUVAI SYLLABLE SUVAI SYLLABLE SHUVAI SYLLABLE ZUVAI SYLLABLE ZHUVAI SYLLABLE CUVAI SYLLABLE JUVAI SYLLABLE NJUVAI SYLLABLE YUVAI SYLLABLE KUVAI SYLLABLE NGGUVAI SYLLABLE GUVAI SYLLABLE MUVAI SYLLABLE NUVAI SYL\
    \LABLE NYUVAI SYLLABLE OVAI SYLLABLE ONVAI SYLLABLE NGONVAI SYLLABLE HOVAI SYLLABLE HONVAI SYLLABLE WOVAI SYLLABLE WONVAI SYLLABLE POVAI SYLLABLE BHOVAI SYLLABLE BOVAI SYLLABLE MBOVAI SYLLABLE KPOVAI SYLLABLE MGBOVAI SYLLABLE GBOVAI SYLLABLE GBONVAI SYLLAB\
    \LE FOVAI SYLLABLE VOVAI SYLLABLE TOVAI SYLLABLE THOVAI SYLLABLE DHOVAI SYLLABLE DHHOVAI SYLLABLE LOVAI SYLLABLE ROVAI SYLLABLE DOVAI SYLLABLE NDOVAI SYLLABLE SOVAI SYLLABLE SHOVAI SYLLABLE ZOVAI SYLLABLE ZHOVAI SYLLABLE COVAI SYLLABLE JOVAI SYLLABLE NJOVA\
    \I SYLLABLE YOVAI SYLLABLE KOVAI SYLLABLE NGGOVAI SYLLABLE GOVAI SYLLABLE MOVAI SYLLABLE NOVAI SYLLABLE NYOVAI SYLLABLE EVAI SYLLABLE ENVAI SYLLABLE NGENVAI SYLLABLE HEVAI SYLLABLE HENVAI SYLLABLE WEVAI SYLLABLE WENVAI SYLLABLE PEVAI SYLLABLE BHEVAI SYLLAB\
    \LE BEVAI SYLLABLE MBEVAI SYLLABLE KPEVAI SYLLABLE KPENVAI SYLLABLE MGBEVAI SYLLABLE GBEVAI SYLLABLE GBENVAI SYLLABLE FEVAI SYLLABLE VEVAI SYLLABLE TEVAI SYLLABLE THEVAI SYLLABLE DHEVAI SYLLABLE DHHEVAI SYLLABLE LEVAI SYLLABLE REVAI SYLLABLE DEVAI SYLLABLE\
    \ NDEVAI SYLLABLE SEVAI SYLLABLE SHEVAI SYLLABLE ZEVAI SYLLABLE ZHEVAI SYLLABLE CEVAI SYLLABLE JEVAI SYLLABLE NJEVAI SYLLABLE YEVAI SYLLABLE KEVAI SYLLABLE NGGEVAI SYLLABLE NGGENVAI SYLLABLE GEVAI SYLLABLE GENVAI SYLLABLE MEVAI SYLLABLE NEVAI SYLLABLE NYEV\
    \AI SYLLABLE NGVAI SYLLABLE LENGTHENERVAI COMMAVAI FULL STOPVAI QUESTION MARKVAI SYLLABLE NDOLE FAVAI SYLLABLE NDOLE KAVAI SYLLABLE NDOLE SOOVAI SYMBOL FEENGVAI SYMBOL KEENGVAI SYMBOL TINGVAI SYMBOL NIIVAI SYMBOL BANGVAI SYMBOL FAAVAI SYMBOL TAAVAI SYMBOL \
    \DANGVAI SYMBOL DOONGVAI SYMBOL KUNGVAI SYMBOL TONGVAI SYMBOL DO-OVAI SYMBOL JONGVAI DIGIT ZEROVAI DIGIT ONEVAI DIGIT TWOVAI DIGIT THREEVAI DIGIT FOURVAI DIGIT FIVEVAI DIGIT SIXVAI DIGIT SEVENVAI DIGIT EIGHTVAI DIGIT NINEVAI SYLLABLE NDOLE MAVAI SYLLABLE N\
    \DOLE DOCYRILLIC CAPITAL LETTER ZEMLYACYRILLIC SMALL LETTER ZEMLYACYRILLIC CAPITAL LETTER DZELOCYRILLIC SMALL LETTER DZELOCYRILLIC CAPITAL LETTER REVERSED DZECYRILLIC SMALL LETTER REVERSED DZECYRILLIC CAPITAL LETTER IOTACYRILLIC SMALL LETTER IOTACYRILLIC C\
    \APITAL LETTER DJERVCYRILLIC SMALL LETTER DJERVCYRILLIC CAPITAL LETTER MONOGRAPH UKCYRILLIC SMALL LETTER MONOGRAPH UKCYRILLIC CAPITAL LETTER BROAD OMEGACYRILLIC SMALL LETTER BROAD OMEGACYRILLIC CAPITAL LETTER NEUTRAL YERCYRILLIC SMALL LETTER NEUTRAL YERCYR\
    \ILLIC CAPITAL LETTER YERU WITH BACK YERCYRILLIC SMALL LETTER YERU WITH BACK YERCYRILLIC CAPITAL LETTER IOTIFIED YATCYRILLIC SMALL LETTER IOTIFIED YATCYRILLIC CAPITAL LETTER REVERSED YUCYRILLIC SMALL LETTER REVERSED YUCYRILLIC CAPITAL LETTER IOTIFIED ACYRI\
    \LLIC SMALL LETTER IOTIFIED ACYRILLIC CAPITAL LETTER CLOSED LITTLE YUSCYRILLIC SMALL LETTER CLOSED LITTLE YUSCYRILLIC CAPITAL LETTER BLENDED YUSCYRILLIC SMALL LETTER BLENDED YUSCYRILLIC CAPITAL LETTER IOTIFIED CLOSED LITTLE YUSCYRILLIC SMALL LETTER IOTIFIE\
    \D CLOSED LITTLE YUSCYRILLIC CAPITAL LETTER YNCYRILLIC SMALL LETTER YNCYRILLIC CAPITAL LETTER REVERSED TSECYRILLIC SMALL LETTER REVERSED TSECYRILLIC CAPITAL LETTER SOFT DECYRILLIC SMALL LETTER SOFT DECYRILLIC CAPITAL LETTER SOFT ELCYRILLIC SMALL LETTER SOF\
    \T ELCYRILLIC CAPITAL LETTER SOFT EMCYRILLIC SMALL LETTER SOFT EMCYRILLIC CAPITAL LETTER MONOCULAR OCYRILLIC SMALL LETTER MONOCULAR OCYRILLIC CAPITAL LETTER BINOCULAR OCYRILLIC SMALL LETTER BINOCULAR OCYRILLIC CAPITAL LETTER DOUBLE MONOCULAR OCYRILLIC SMAL\
    \L LETTER DOUBLE MONOCULAR OCYRILLIC LETTER MULTIOCULAR OCOMBINING CYRILLIC VZMETCOMBINING CYRILLIC TEN MILLIONS SIGNCOMBINING CYRILLIC HUNDRED MILLIONS SIGNCOMBINING CYRILLIC THOUSAND MILLIONS SIGNSLAVONIC ASTERISKCOMBINING CYRILLIC LETTER UKRAINIAN IECOM\
    \BINING CYRILLIC LETTER ICOMBINING CYRILLIC LETTER YICOMBINING CYRILLIC LETTER UCOMBINING CYRILLIC LETTER HARD SIGNCOMBINING CYRILLIC LETTER YERUCOMBINING CYRILLIC LETTER SOFT SIGNCOMBINING CYRILLIC LETTER OMEGACOMBINING CYRILLIC KAVYKACOMBINING CYRILLIC P\
    \AYEROKCYRILLIC KAVYKACYRILLIC PAYEROKCYRILLIC CAPITAL LETTER DWECYRILLIC SMALL LETTER DWECYRILLIC CAPITAL LETTER DZWECYRILLIC SMALL LETTER DZWECYRILLIC CAPITAL LETTER ZHWECYRILLIC SMALL LETTER ZHWECYRILLIC CAPITAL LETTER CCHECYRILLIC SMALL LETTER CCHECYRI\
    \LLIC CAPITAL LETTER DZZECYRILLIC SMALL LETTER DZZECYRILLIC CAPITAL LETTER TE WITH MIDDLE HOOKCYRILLIC SMALL LETTER TE WITH MIDDLE HOOKCYRILLIC CAPITAL LETTER TWECYRILLIC SMALL LETTER TWECYRILLIC CAPITAL LETTER TSWECYRILLIC SMALL LETTER TSWECYRILLIC CAPITA\
    \L LETTER TSSECYRILLIC SMALL LETTER TSSECYRILLIC CAPITAL LETTER TCHECYRILLIC SMALL LETTER TCHECYRILLIC CAPITAL LETTER HWECYRILLIC SMALL LETTER HWECYRILLIC CAPITAL LETTER SHWECYRILLIC SMALL LETTER SHWECYRILLIC CAPITAL LETTER DOUBLE OCYRILLIC SMALL LETTER DO\
    \UBLE OCYRILLIC CAPITAL LETTER CROSSED OCYRILLIC SMALL LETTER CROSSED OMODIFIER LETTER CYRILLIC HARD SIGNMODIFIER LETTER CYRILLIC SOFT SIGNCOMBINING CYRILLIC LETTER EFCOMBINING CYRILLIC LETTER IOTIFIED EBAMUM LETTER ABAMUM LETTER KABAMUM LETTER UBAMUM LETT\
    \ER KUBAMUM LETTER EEBAMUM LETTER REEBAMUM LETTER TAEBAMUM LETTER OBAMUM LETTER NYIBAMUM LETTER IBAMUM LETTER LABAMUM LETTER PABAMUM LETTER RIIBAMUM LETTER RIEEBAMUM LETTER LEEEEBAMUM LETTER MEEEEBAMUM LETTER TAABAMUM LETTER NDAABAMUM LETTER NJAEMBAMUM LET\
    \TER MBAMUM LETTER SUUBAMUM LETTER MUBAMUM LETTER SHIIBAMUM LETTER SIBAMUM LETTER SHEUXBAMUM LETTER SEUXBAMUM LETTER KYEEBAMUM LETTER KETBAMUM LETTER NUAEBAMUM LETTER NUBAMUM LETTER NJUAEBAMUM LETTER YOQBAMUM LETTER SHUBAMUM LETTER YUQBAMUM LETTER YABAMUM \
    \LETTER NSHABAMUM LETTER KEUXBAMUM LETTER PEUXBAMUM LETTER NJEEBAMUM LETTER NTEEBAMUM LETTER PUEBAMUM LETTER WUEBAMUM LETTER PEEBAMUM LETTER FEEBAMUM LETTER RUBAMUM LETTER LUBAMUM LETTER MIBAMUM LETTER NIBAMUM LETTER REUXBAMUM LETTER RAEBAMUM LETTER KENBAM\
    \UM LETTER NGKWAENBAMUM LETTER NGGABAMUM LETTER NGABAMUM LETTER SHOBAMUM LETTER PUAEBAMUM LETTER FUBAMUM LETTER FOMBAMUM LETTER WABAMUM LETTER NABAMUM LETTER LIBAMUM LETTER PIBAMUM LETTER LOQBAMUM LETTER KOBAMUM LETTER MBENBAMUM LETTER RENBAMUM LETTER MENB\
    \AMUM LETTER MABAMUM LETTER TIBAMUM LETTER KIBAMUM LETTER MOBAMUM LETTER MBAABAMUM LETTER TETBAMUM LETTER KPABAMUM LETTER TENBAMUM LETTER NTUUBAMUM LETTER SAMBABAMUM LETTER FAAMAEBAMUM LETTER KOVUUBAMUM LETTER KOGHOMBAMUM COMBINING MARK KOQNDONBAMUM COMBIN\
    \ING MARK TUKWENTISBAMUM NJAEMLIBAMUM FULL STOPBAMUM COLONBAMUM COMMABAMUM SEMICOLONBAMUM QUESTION MARKMODIFIER LETTER CHINESE TONE YIN PINGMODIFIER LETTER CHINESE TONE YANG PINGMODIFIER LETTER CHINESE TONE YIN SHANGMODIFIER LETTER CHINESE TONE YANG SHANGM\
    \ODIFIER LETTER CHINESE TONE YIN QUMODIFIER LETTER CHINESE TONE YANG QUMODIFIER LETTER CHINESE TONE YIN RUMODIFIER LETTER CHINESE TONE YANG RUMODIFIER LETTER EXTRA-HIGH DOTTED TONE BARMODIFIER LETTER HIGH DOTTED TONE BARMODIFIER LETTER MID DOTTED TONE BARM\
    \ODIFIER LETTER LOW DOTTED TONE BARMODIFIER LETTER EXTRA-LOW DOTTED TONE BARMODIFIER LETTER EXTRA-HIGH DOTTED LEFT-STEM TONE BARMODIFIER LETTER HIGH DOTTED LEFT-STEM TONE BARMODIFIER LETTER MID DOTTED LEFT-STEM TONE BARMODIFIER LETTER LOW DOTTED LEFT-STEM \
    \TONE BARMODIFIER LETTER EXTRA-LOW DOTTED LEFT-STEM TONE BARMODIFIER LETTER EXTRA-HIGH LEFT-STEM TONE BARMODIFIER LETTER HIGH LEFT-STEM TONE BARMODIFIER LETTER MID LEFT-STEM TONE BARMODIFIER LETTER LOW LEFT-STEM TONE BARMODIFIER LETTER EXTRA-LOW LEFT-STEM \
    \TONE BARMODIFIER LETTER DOT VERTICAL BARMODIFIER LETTER DOT SLASHMODIFIER LETTER DOT HORIZONTAL BARMODIFIER LETTER LOWER RIGHT CORNER ANGLEMODIFIER LETTER RAISED UP ARROWMODIFIER LETTER RAISED DOWN ARROWMODIFIER LETTER RAISED EXCLAMATION MARKMODIFIER LETT\
    \ER RAISED INVERTED EXCLAMATION MARKMODIFIER LETTER LOW INVERTED EXCLAMATION MARKMODIFIER LETTER STRESS AND HIGH TONEMODIFIER LETTER STRESS AND LOW TONELATIN CAPITAL LETTER EGYPTOLOGICAL ALEFLATIN SMALL LETTER EGYPTOLOGICAL ALEFLATIN CAPITAL LETTER EGYPTOL\
    \OGICAL AINLATIN SMALL LETTER EGYPTOLOGICAL AINLATIN CAPITAL LETTER HENGLATIN SMALL LETTER HENGLATIN CAPITAL LETTER TZLATIN SMALL LETTER TZLATIN CAPITAL LETTER TRESILLOLATIN SMALL LETTER TRESILLOLATIN CAPITAL LETTER CUATRILLOLATIN SMALL LETTER CUATRILLOLAT\
    \IN CAPITAL LETTER CUATRILLO WITH COMMALATIN SMALL LETTER CUATRILLO WITH COMMALATIN LETTER SMALL CAPITAL FLATIN LETTER SMALL CAPITAL SLATIN CAPITAL LETTER AALATIN SMALL LETTER AALATIN CAPITAL LETTER AOLATIN SMALL LETTER AOLATIN CAPITAL LETTER AULATIN SMALL\
    \ LETTER AULATIN CAPITAL LETTER AVLATIN SMALL LETTER AVLATIN CAPITAL LETTER AV WITH HORIZONTAL BARLATIN SMALL LETTER AV WITH HORIZONTAL BARLATIN CAPITAL LETTER AYLATIN SMALL LETTER AYLATIN CAPITAL LETTER REVERSED C WITH DOTLATIN SMALL LETTER REVERSED C WIT\
    \H DOTLATIN CAPITAL LETTER K WITH STROKELATIN SMALL LETTER K WITH STROKELATIN CAPITAL LETTER K WITH DIAGONAL STROKELATIN SMALL LETTER K WITH DIAGONAL STROKELATIN CAPITAL LETTER K WITH STROKE AND DIAGONAL STROKELATIN SMALL LETTER K WITH STROKE AND DIAGONAL \
    \STROKELATIN CAPITAL LETTER BROKEN LLATIN SMALL LETTER BROKEN LLATIN CAPITAL LETTER L WITH HIGH STROKELATIN SMALL LETTER L WITH HIGH STROKELATIN CAPITAL LETTER O WITH LONG STROKE OVERLAYLATIN SMALL LETTER O WITH LONG STROKE OVERLAYLATIN CAPITAL LETTER O WI\
    \TH LOOPLATIN SMALL LETTER O WITH LOOPLATIN CAPITAL LETTER OOLATIN SMALL LETTER OOLATIN CAPITAL LETTER P WITH STROKE THROUGH DESCENDERLATIN SMALL LETTER P WITH STROKE THROUGH DESCENDERLATIN CAPITAL LETTER P WITH FLOURISHLATIN SMALL LETTER P WITH FLOURISHLA\
    \TIN CAPITAL LETTER P WITH SQUIRREL TAILLATIN SMALL LETTER P WITH SQUIRREL TAILLATIN CAPITAL LETTER Q WITH STROKE THROUGH DESCENDERLATIN SMALL LETTER Q WITH STROKE THROUGH DESCENDERLATIN CAPITAL LETTER Q WITH DIAGONAL STROKELATIN SMALL LETTER Q WITH DIAGON\
    \AL STROKELATIN CAPITAL LETTER R ROTUNDALATIN SMALL LETTER R ROTUNDALATIN CAPITAL LETTER RUM ROTUNDALATIN SMALL LETTER RUM ROTUNDALATIN CAPITAL LETTER V WITH DIAGONAL STROKELATIN SMALL LETTER V WITH DIAGONAL STROKELATIN CAPITAL LETTER VYLATIN SMALL LETTER \
    \VYLATIN CAPITAL LETTER VISIGOTHIC ZLATIN SMALL LETTER VISIGOTHIC ZLATIN CAPITAL LETTER THORN WITH STROKELATIN SMALL LETTER THORN WITH STROKELATIN CAPITAL LETTER THORN WITH STROKE THROUGH DESCENDERLATIN SMALL LETTER THORN WITH STROKE THROUGH DESCENDERLATIN\
    \ CAPITAL LETTER VENDLATIN SMALL LETTER VENDLATIN CAPITAL LETTER ETLATIN SMALL LETTER ETLATIN CAPITAL LETTER ISLATIN SMALL LETTER ISLATIN CAPITAL LETTER CONLATIN SMALL LETTER CONMODIFIER LETTER USLATIN SMALL LETTER DUMLATIN SMALL LETTER LUMLATIN SMALL LETT\
    \ER MUMLATIN SMALL LETTER NUMLATIN SMALL LETTER RUMLATIN LETTER SMALL CAPITAL RUMLATIN SMALL LETTER TUMLATIN SMALL LETTER UMLATIN CAPITAL LETTER INSULAR DLATIN SMALL LETTER INSULAR DLATIN CAPITAL LETTER INSULAR FLATIN SMALL LETTER INSULAR FLATIN CAPITAL LE\
    \TTER INSULAR GLATIN CAPITAL LETTER TURNED INSULAR GLATIN SMALL LETTER TURNED INSULAR GLATIN CAPITAL LETTER TURNED LLATIN SMALL LETTER TURNED LLATIN CAPITAL LETTER INSULAR RLATIN SMALL LETTER INSULAR RLATIN CAPITAL LETTER INSULAR SLATIN SMALL LETTER INSULA\
    \R SLATIN CAPITAL LETTER INSULAR TLATIN SMALL LETTER INSULAR TMODIFIER LETTER LOW CIRCUMFLEX ACCENTMODIFIER LETTER COLONMODIFIER LETTER SHORT EQUALS SIGNLATIN CAPITAL LETTER SALTILLOLATIN SMALL LETTER SALTILLOLATIN CAPITAL LETTER TURNED HLATIN SMALL LETTER\
    \ L WITH RETROFLEX HOOK AND BELTLATIN LETTER SINOLOGICAL DOTLATIN CAPITAL LETTER N WITH DESCENDERLATIN SMALL LETTER N WITH DESCENDERLATIN CAPITAL LETTER C WITH BARLATIN SMALL LETTER C WITH BARLATIN SMALL LETTER C WITH PALATAL HOOKLATIN SMALL LETTER H WITH \
    \PALATAL HOOKLATIN CAPITAL LETTER B WITH FLOURISHLATIN SMALL LETTER B WITH FLOURISHLATIN CAPITAL LETTER F WITH STROKELATIN SMALL LETTER F WITH STROKELATIN CAPITAL LETTER VOLAPUK AELATIN SMALL LETTER VOLAPUK AELATIN CAPITAL LETTER VOLAPUK OELATIN SMALL LETT\
    \ER VOLAPUK OELATIN CAPITAL LETTER VOLAPUK UELATIN SMALL LETTER VOLAPUK UELATIN CAPITAL LETTER G WITH OBLIQUE STROKELATIN SMALL LETTER G WITH OBLIQUE STROKELATIN CAPITAL LETTER K WITH OBLIQUE STROKELATIN SMALL LETTER K WITH OBLIQUE STROKELATIN CAPITAL LETT\
    \ER N WITH OBLIQUE STROKELATIN SMALL LETTER N WITH OBLIQUE STROKELATIN CAPITAL LETTER R WITH OBLIQUE STROKELATIN SMALL LETTER R WITH OBLIQUE STROKELATIN CAPITAL LETTER S WITH OBLIQUE STROKELATIN SMALL LETTER S WITH OBLIQUE STROKELATIN CAPITAL LETTER H WITH\
    \ HOOKLATIN CAPITAL LETTER REVERSED OPEN ELATIN CAPITAL LETTER SCRIPT GLATIN CAPITAL LETTER L WITH BELTLATIN CAPITAL LETTER SMALL CAPITAL ILATIN LETTER SMALL CAPITAL QLATIN CAPITAL LETTER TURNED KLATIN CAPITAL LETTER TURNED TLATIN CAPITAL LETTER J WITH CRO\
    \SSED-TAILLATIN CAPITAL LETTER CHILATIN CAPITAL LETTER BETALATIN SMALL LETTER BETALATIN CAPITAL LETTER OMEGALATIN SMALL LETTER OMEGALATIN CAPITAL LETTER U WITH STROKELATIN SMALL LETTER U WITH STROKELATIN CAPITAL LETTER GLOTTAL ALATIN SMALL LETTER GLOTTAL A\
    \LATIN CAPITAL LETTER GLOTTAL ILATIN SMALL LETTER GLOTTAL ILATIN CAPITAL LETTER GLOTTAL ULATIN SMALL LETTER GLOTTAL ULATIN CAPITAL LETTER OLD POLISH OLATIN SMALL LETTER OLD POLISH OLATIN CAPITAL LETTER ANGLICANA WLATIN SMALL LETTER ANGLICANA WLATIN CAPITAL\
    \ LETTER C WITH PALATAL HOOKLATIN CAPITAL LETTER S WITH HOOKLATIN CAPITAL LETTER Z WITH PALATAL HOOKLATIN CAPITAL LETTER D WITH SHORT STROKE OVERLAYLATIN SMALL LETTER D WITH SHORT STROKE OVERLAYLATIN CAPITAL LETTER S WITH SHORT STROKE OVERLAYLATIN SMALL LE\
    \TTER S WITH SHORT STROKE OVERLAYLATIN CAPITAL LETTER CLOSED INSULAR GLATIN SMALL LETTER CLOSED INSULAR GLATIN SMALL LETTER DOUBLE THORNLATIN SMALL LETTER DOUBLE WYNNLATIN CAPITAL LETTER MIDDLE SCOTS SLATIN SMALL LETTER MIDDLE SCOTS SLATIN CAPITAL LETTER S\
    \IGMOID SLATIN SMALL LETTER SIGMOID SMODIFIER LETTER CAPITAL CMODIFIER LETTER CAPITAL FMODIFIER LETTER CAPITAL QLATIN CAPITAL LETTER REVERSED HALF HLATIN SMALL LETTER REVERSED HALF HLATIN EPIGRAPHIC LETTER SIDEWAYS IMODIFIER LETTER CAPITAL H WITH STROKEMOD\
    \IFIER LETTER SMALL LIGATURE OELATIN LETTER SMALL CAPITAL TURNED MLATIN EPIGRAPHIC LETTER REVERSED FLATIN EPIGRAPHIC LETTER REVERSED PLATIN EPIGRAPHIC LETTER INVERTED MLATIN EPIGRAPHIC LETTER I LONGALATIN EPIGRAPHIC LETTER ARCHAIC MSYLOTI NAGRI LETTER ASYL\
    \OTI NAGRI LETTER ISYLOTI NAGRI SIGN DVISVARASYLOTI NAGRI LETTER USYLOTI NAGRI LETTER ESYLOTI NAGRI LETTER OSYLOTI NAGRI SIGN HASANTASYLOTI NAGRI LETTER KOSYLOTI NAGRI LETTER KHOSYLOTI NAGRI LETTER GOSYLOTI NAGRI LETTER GHOSYLOTI NAGRI SIGN ANUSVARASYLOTI \
    \NAGRI LETTER COSYLOTI NAGRI LETTER CHOSYLOTI NAGRI LETTER JOSYLOTI NAGRI LETTER JHOSYLOTI NAGRI LETTER TTOSYLOTI NAGRI LETTER TTHOSYLOTI NAGRI LETTER DDOSYLOTI NAGRI LETTER DDHOSYLOTI NAGRI LETTER TOSYLOTI NAGRI LETTER THOSYLOTI NAGRI LETTER DOSYLOTI NAGR\
    \I LETTER DHOSYLOTI NAGRI LETTER NOSYLOTI NAGRI LETTER POSYLOTI NAGRI LETTER PHOSYLOTI NAGRI LETTER BOSYLOTI NAGRI LETTER BHOSYLOTI NAGRI LETTER MOSYLOTI NAGRI LETTER ROSYLOTI NAGRI LETTER LOSYLOTI NAGRI LETTER RROSYLOTI NAGRI LETTER SOSYLOTI NAGRI LETTER \
    \HOSYLOTI NAGRI VOWEL SIGN ASYLOTI NAGRI VOWEL SIGN ISYLOTI NAGRI VOWEL SIGN USYLOTI NAGRI VOWEL SIGN ESYLOTI NAGRI VOWEL SIGN OOSYLOTI NAGRI POETRY MARK-1SYLOTI NAGRI POETRY MARK-2SYLOTI NAGRI POETRY MARK-3SYLOTI NAGRI POETRY MARK-4SYLOTI NAGRI SIGN ALTER\
    \NATE HASANTANORTH INDIC FRACTION ONE QUARTERNORTH INDIC FRACTION ONE HALFNORTH INDIC FRACTION THREE QUARTERSNORTH INDIC FRACTION ONE SIXTEENTHNORTH INDIC FRACTION ONE EIGHTHNORTH INDIC FRACTION THREE SIXTEENTHSNORTH INDIC QUARTER MARKNORTH INDIC PLACEHOLD\
    \ER MARKNORTH INDIC RUPEE MARKNORTH INDIC QUANTITY MARKPHAGS-PA LETTER KAPHAGS-PA LETTER KHAPHAGS-PA LETTER GAPHAGS-PA LETTER NGAPHAGS-PA LETTER CAPHAGS-PA LETTER CHAPHAGS-PA LETTER JAPHAGS-PA LETTER NYAPHAGS-PA LETTER TAPHAGS-PA LETTER THAPHAGS-PA LETTER \
    \DAPHAGS-PA LETTER NAPHAGS-PA LETTER PAPHAGS-PA LETTER PHAPHAGS-PA LETTER BAPHAGS-PA LETTER MAPHAGS-PA LETTER TSAPHAGS-PA LETTER TSHAPHAGS-PA LETTER DZAPHAGS-PA LETTER WAPHAGS-PA LETTER ZHAPHAGS-PA LETTER ZAPHAGS-PA LETTER SMALL APHAGS-PA LETTER YAPHAGS-PA\
    \ LETTER RAPHAGS-PA LETTER LAPHAGS-PA LETTER SHAPHAGS-PA LETTER SAPHAGS-PA LETTER HAPHAGS-PA LETTER APHAGS-PA LETTER IPHAGS-PA LETTER UPHAGS-PA LETTER EPHAGS-PA LETTER OPHAGS-PA LETTER QAPHAGS-PA LETTER XAPHAGS-PA LETTER FAPHAGS-PA LETTER GGAPHAGS-PA LETTE\
    \R EEPHAGS-PA SUBJOINED LETTER WAPHAGS-PA SUBJOINED LETTER YAPHAGS-PA LETTER TTAPHAGS-PA LETTER TTHAPHAGS-PA LETTER DDAPHAGS-PA LETTER NNAPHAGS-PA LETTER ALTERNATE YAPHAGS-PA LETTER VOICELESS SHAPHAGS-PA LETTER VOICED HAPHAGS-PA LETTER ASPIRATED FAPHAGS-PA\
    \ SUBJOINED LETTER RAPHAGS-PA SUPERFIXED LETTER RAPHAGS-PA LETTER CANDRABINDUPHAGS-PA SINGLE HEAD MARKPHAGS-PA DOUBLE HEAD MARKPHAGS-PA MARK SHADPHAGS-PA MARK DOUBLE SHADSAURASHTRA SIGN ANUSVARASAURASHTRA SIGN VISARGASAURASHTRA LETTER ASAURASHTRA LETTER AA\
    \SAURASHTRA LETTER ISAURASHTRA LETTER IISAURASHTRA LETTER USAURASHTRA LETTER UUSAURASHTRA LETTER VOCALIC RSAURASHTRA LETTER VOCALIC RRSAURASHTRA LETTER VOCALIC LSAURASHTRA LETTER VOCALIC LLSAURASHTRA LETTER ESAURASHTRA LETTER EESAURASHTRA LETTER AISAURASHT\
    \RA LETTER OSAURASHTRA LETTER OOSAURASHTRA LETTER AUSAURASHTRA LETTER KASAURASHTRA LETTER KHASAURASHTRA LETTER GASAURASHTRA LETTER GHASAURASHTRA LETTER NGASAURASHTRA LETTER CASAURASHTRA LETTER CHASAURASHTRA LETTER JASAURASHTRA LETTER JHASAURASHTRA LETTER N\
    \YASAURASHTRA LETTER TTASAURASHTRA LETTER TTHASAURASHTRA LETTER DDASAURASHTRA LETTER DDHASAURASHTRA LETTER NNASAURASHTRA LETTER TASAURASHTRA LETTER THASAURASHTRA LETTER DASAURASHTRA LETTER DHASAURASHTRA LETTER NASAURASHTRA LETTER PASAURASHTRA LETTER PHASAU\
    \RASHTRA LETTER BASAURASHTRA LETTER BHASAURASHTRA LETTER MASAURASHTRA LETTER YASAURASHTRA LETTER RASAURASHTRA LETTER LASAURASHTRA LETTER VASAURASHTRA LETTER SHASAURASHTRA LETTER SSASAURASHTRA LETTER SASAURASHTRA LETTER HASAURASHTRA LETTER LLASAURASHTRA CON\
    \SONANT SIGN HAARUSAURASHTRA VOWEL SIGN AASAURASHTRA VOWEL SIGN ISAURASHTRA VOWEL SIGN IISAURASHTRA VOWEL SIGN USAURASHTRA VOWEL SIGN UUSAURASHTRA VOWEL SIGN VOCALIC RSAURASHTRA VOWEL SIGN VOCALIC RRSAURASHTRA VOWEL SIGN VOCALIC LSAURASHTRA VOWEL SIGN VOCA\
    \LIC LLSAURASHTRA VOWEL SIGN ESAURASHTRA VOWEL SIGN EESAURASHTRA VOWEL SIGN AISAURASHTRA VOWEL SIGN OSAURASHTRA VOWEL SIGN OOSAURASHTRA VOWEL SIGN AUSAURASHTRA SIGN VIRAMASAURASHTRA SIGN CANDRABINDUSAURASHTRA DANDASAURASHTRA DOUBLE DANDASAURASHTRA DIGIT ZE\
    \ROSAURASHTRA DIGIT ONESAURASHTRA DIGIT TWOSAURASHTRA DIGIT THREESAURASHTRA DIGIT FOURSAURASHTRA DIGIT FIVESAURASHTRA DIGIT SIXSAURASHTRA DIGIT SEVENSAURASHTRA DIGIT EIGHTSAURASHTRA DIGIT NINECOMBINING DEVANAGARI DIGIT ZEROCOMBINING DEVANAGARI DIGIT ONECOM\
    \BINING DEVANAGARI DIGIT TWOCOMBINING DEVANAGARI DIGIT THREECOMBINING DEVANAGARI DIGIT FOURCOMBINING DEVANAGARI DIGIT FIVECOMBINING DEVANAGARI DIGIT SIXCOMBINING DEVANAGARI DIGIT SEVENCOMBINING DEVANAGARI DIGIT EIGHTCOMBINING DEVANAGARI DIGIT NINECOMBINING\
    \ DEVANAGARI LETTER ACOMBINING DEVANAGARI LETTER UCOMBINING DEVANAGARI LETTER KACOMBINING DEVANAGARI LETTER NACOMBINING DEVANAGARI LETTER PACOMBINING DEVANAGARI LETTER RACOMBINING DEVANAGARI LETTER VICOMBINING DEVANAGARI SIGN AVAGRAHADEVANAGARI SIGN SPACIN\
    \G CANDRABINDUDEVANAGARI SIGN CANDRABINDU VIRAMADEVANAGARI SIGN DOUBLE CANDRABINDU VIRAMADEVANAGARI SIGN CANDRABINDU TWODEVANAGARI SIGN CANDRABINDU THREEDEVANAGARI SIGN CANDRABINDU AVAGRAHADEVANAGARI SIGN PUSHPIKADEVANAGARI GAP FILLERDEVANAGARI CARETDEVANA\
    \GARI HEADSTROKEDEVANAGARI SIGN SIDDHAMDEVANAGARI JAIN OMDEVANAGARI LETTER AYDEVANAGARI VOWEL SIGN AYKAYAH LI DIGIT ZEROKAYAH LI DIGIT ONEKAYAH LI DIGIT TWOKAYAH LI DIGIT THREEKAYAH LI DIGIT FOURKAYAH LI DIGIT FIVEKAYAH LI DIGIT SIXKAYAH LI DIGIT SEVENKAYA\
    \H LI DIGIT EIGHTKAYAH LI DIGIT NINEKAYAH LI LETTER KAKAYAH LI LETTER KHAKAYAH LI LETTER GAKAYAH LI LETTER NGAKAYAH LI LETTER SAKAYAH LI LETTER SHAKAYAH LI LETTER ZAKAYAH LI LETTER NYAKAYAH LI LETTER TAKAYAH LI LETTER HTAKAYAH LI LETTER NAKAYAH LI LETTER P\
    \AKAYAH LI LETTER PHAKAYAH LI LETTER MAKAYAH LI LETTER DAKAYAH LI LETTER BAKAYAH LI LETTER RAKAYAH LI LETTER YAKAYAH LI LETTER LAKAYAH LI LETTER WAKAYAH LI LETTER THAKAYAH LI LETTER HAKAYAH LI LETTER VAKAYAH LI LETTER CAKAYAH LI LETTER AKAYAH LI LETTER OEK\
    \AYAH LI LETTER IKAYAH LI LETTER OOKAYAH LI VOWEL UEKAYAH LI VOWEL EKAYAH LI VOWEL UKAYAH LI VOWEL EEKAYAH LI VOWEL OKAYAH LI TONE PLOPHUKAYAH LI TONE CALYAKAYAH LI TONE CALYA PLOPHUKAYAH LI SIGN CWIKAYAH LI SIGN SHYAREJANG LETTER KAREJANG LETTER GAREJANG \
    \LETTER NGAREJANG LETTER TAREJANG LETTER DAREJANG LETTER NAREJANG LETTER PAREJANG LETTER BAREJANG LETTER MAREJANG LETTER CAREJANG LETTER JAREJANG LETTER NYAREJANG LETTER SAREJANG LETTER RAREJANG LETTER LAREJANG LETTER YAREJANG LETTER WAREJANG LETTER HAREJA\
    \NG LETTER MBAREJANG LETTER NGGAREJANG LETTER NDAREJANG LETTER NYJAREJANG LETTER AREJANG VOWEL SIGN IREJANG VOWEL SIGN UREJANG VOWEL SIGN EREJANG VOWEL SIGN AIREJANG VOWEL SIGN OREJANG VOWEL SIGN AUREJANG VOWEL SIGN EUREJANG VOWEL SIGN EAREJANG CONSONANT S\
    \IGN NGREJANG CONSONANT SIGN NREJANG CONSONANT SIGN RREJANG CONSONANT SIGN HREJANG VIRAMAREJANG SECTION MARKHANGUL CHOSEONG TIKEUT-MIEUMHANGUL CHOSEONG TIKEUT-PIEUPHANGUL CHOSEONG TIKEUT-SIOSHANGUL CHOSEONG TIKEUT-CIEUCHANGUL CHOSEONG RIEUL-KIYEOKHANGUL CH\
    \OSEONG RIEUL-SSANGKIYEOKHANGUL CHOSEONG RIEUL-TIKEUTHANGUL CHOSEONG RIEUL-SSANGTIKEUTHANGUL CHOSEONG RIEUL-MIEUMHANGUL CHOSEONG RIEUL-PIEUPHANGUL CHOSEONG RIEUL-SSANGPIEUPHANGUL CHOSEONG RIEUL-KAPYEOUNPIEUPHANGUL CHOSEONG RIEUL-SIOSHANGUL CHOSEONG RIEUL-C\
    \IEUCHANGUL CHOSEONG RIEUL-KHIEUKHHANGUL CHOSEONG MIEUM-KIYEOKHANGUL CHOSEONG MIEUM-TIKEUTHANGUL CHOSEONG MIEUM-SIOSHANGUL CHOSEONG PIEUP-SIOS-THIEUTHHANGUL CHOSEONG PIEUP-KHIEUKHHANGUL CHOSEONG PIEUP-HIEUHHANGUL CHOSEONG SSANGSIOS-PIEUPHANGUL CHOSEONG IEU\
    \NG-RIEULHANGUL CHOSEONG IEUNG-HIEUHHANGUL CHOSEONG SSANGCIEUC-HIEUHHANGUL CHOSEONG SSANGTHIEUTHHANGUL CHOSEONG PHIEUPH-HIEUHHANGUL CHOSEONG HIEUH-SIOSHANGUL CHOSEONG SSANGYEORINHIEUHJAVANESE SIGN PANYANGGAJAVANESE SIGN CECAKJAVANESE SIGN LAYARJAVANESE SIG\
    \N WIGNYANJAVANESE LETTER AJAVANESE LETTER I KAWIJAVANESE LETTER IJAVANESE LETTER IIJAVANESE LETTER UJAVANESE LETTER PA CEREKJAVANESE LETTER NGA LELETJAVANESE LETTER NGA LELET RASWADIJAVANESE LETTER EJAVANESE LETTER AIJAVANESE LETTER OJAVANESE LETTER KAJAV\
    \ANESE LETTER KA SASAKJAVANESE LETTER KA MURDAJAVANESE LETTER GAJAVANESE LETTER GA MURDAJAVANESE LETTER NGAJAVANESE LETTER CAJAVANESE LETTER CA MURDAJAVANESE LETTER JAJAVANESE LETTER NYA MURDAJAVANESE LETTER JA MAHAPRANAJAVANESE LETTER NYAJAVANESE LETTER T\
    \TAJAVANESE LETTER TTA MAHAPRANAJAVANESE LETTER DDAJAVANESE LETTER DDA MAHAPRANAJAVANESE LETTER NA MURDAJAVANESE LETTER TAJAVANESE LETTER TA MURDAJAVANESE LETTER DAJAVANESE LETTER DA MAHAPRANAJAVANESE LETTER NAJAVANESE LETTER PAJAVANESE LETTER PA MURDAJAVA\
    \NESE LETTER BAJAVANESE LETTER BA MURDAJAVANESE LETTER MAJAVANESE LETTER YAJAVANESE LETTER RAJAVANESE LETTER RA AGUNGJAVANESE LETTER LAJAVANESE LETTER WAJAVANESE LETTER SA MURDAJAVANESE LETTER SA MAHAPRANAJAVANESE LETTER SAJAVANESE LETTER HAJAVANESE SIGN C\
    \ECAK TELUJAVANESE VOWEL SIGN TARUNGJAVANESE VOWEL SIGN TOLONGJAVANESE VOWEL SIGN WULUJAVANESE VOWEL SIGN WULU MELIKJAVANESE VOWEL SIGN SUKUJAVANESE VOWEL SIGN SUKU MENDUTJAVANESE VOWEL SIGN TALINGJAVANESE VOWEL SIGN DIRGA MUREJAVANESE VOWEL SIGN PEPETJAVA\
    \NESE CONSONANT SIGN KERETJAVANESE CONSONANT SIGN PENGKALJAVANESE CONSONANT SIGN CAKRAJAVANESE PANGKONJAVANESE LEFT RERENGGANJAVANESE RIGHT RERENGGANJAVANESE PADA ANDAPJAVANESE PADA MADYAJAVANESE PADA LUHURJAVANESE PADA WINDUJAVANESE PADA PANGKATJAVANESE P\
    \ADA LINGSAJAVANESE PADA LUNGSIJAVANESE PADA ADEGJAVANESE PADA ADEG ADEGJAVANESE PADA PISELEHJAVANESE TURNED PADA PISELEHJAVANESE PANGRANGKEPJAVANESE DIGIT ZEROJAVANESE DIGIT ONEJAVANESE DIGIT TWOJAVANESE DIGIT THREEJAVANESE DIGIT FOURJAVANESE DIGIT FIVEJA\
    \VANESE DIGIT SIXJAVANESE DIGIT SEVENJAVANESE DIGIT EIGHTJAVANESE DIGIT NINEJAVANESE PADA TIRTA TUMETESJAVANESE PADA ISEN-ISENMYANMAR LETTER SHAN GHAMYANMAR LETTER SHAN CHAMYANMAR LETTER SHAN JHAMYANMAR LETTER SHAN NNAMYANMAR LETTER SHAN BHAMYANMAR SIGN SH\
    \AN SAWMYANMAR MODIFIER LETTER SHAN REDUPLICATIONMYANMAR LETTER TAI LAING NYAMYANMAR LETTER TAI LAING FAMYANMAR LETTER TAI LAING GAMYANMAR LETTER TAI LAING GHAMYANMAR LETTER TAI LAING JAMYANMAR LETTER TAI LAING JHAMYANMAR LETTER TAI LAING DDAMYANMAR LETTER\
    \ TAI LAING DDHAMYANMAR LETTER TAI LAING NNAMYANMAR TAI LAING DIGIT ZEROMYANMAR TAI LAING DIGIT ONEMYANMAR TAI LAING DIGIT TWOMYANMAR TAI LAING DIGIT THREEMYANMAR TAI LAING DIGIT FOURMYANMAR TAI LAING DIGIT FIVEMYANMAR TAI LAING DIGIT SIXMYANMAR TAI LAING \
    \DIGIT SEVENMYANMAR TAI LAING DIGIT EIGHTMYANMAR TAI LAING DIGIT NINEMYANMAR LETTER TAI LAING LLAMYANMAR LETTER TAI LAING DAMYANMAR LETTER TAI LAING DHAMYANMAR LETTER TAI LAING BAMYANMAR LETTER TAI LAING BHACHAM LETTER ACHAM LETTER ICHAM LETTER UCHAM LETTE\
    \R ECHAM LETTER AICHAM LETTER OCHAM LETTER KACHAM LETTER KHACHAM LETTER GACHAM LETTER GHACHAM LETTER NGUECHAM LETTER NGACHAM LETTER CHACHAM LETTER CHHACHAM LETTER JACHAM LETTER JHACHAM LETTER NHUECHAM LETTER NHACHAM LETTER NHJACHAM LETTER TACHAM LETTER THA\
    \CHAM LETTER DACHAM LETTER DHACHAM LETTER NUECHAM LETTER NACHAM LETTER DDACHAM LETTER PACHAM LETTER PPACHAM LETTER PHACHAM LETTER BACHAM LETTER BHACHAM LETTER MUECHAM LETTER MACHAM LETTER BBACHAM LETTER YACHAM LETTER RACHAM LETTER LACHAM LETTER VACHAM LETT\
    \ER SSACHAM LETTER SACHAM LETTER HACHAM VOWEL SIGN AACHAM VOWEL SIGN ICHAM VOWEL SIGN IICHAM VOWEL SIGN EICHAM VOWEL SIGN UCHAM VOWEL SIGN OECHAM VOWEL SIGN OCHAM VOWEL SIGN AICHAM VOWEL SIGN AUCHAM VOWEL SIGN UECHAM CONSONANT SIGN YACHAM CONSONANT SIGN RA\
    \CHAM CONSONANT SIGN LACHAM CONSONANT SIGN WACHAM LETTER FINAL KCHAM LETTER FINAL GCHAM LETTER FINAL NGCHAM CONSONANT SIGN FINAL NGCHAM LETTER FINAL CHCHAM LETTER FINAL TCHAM LETTER FINAL NCHAM LETTER FINAL PCHAM LETTER FINAL YCHAM LETTER FINAL RCHAM LETTE\
    \R FINAL LCHAM LETTER FINAL SSCHAM CONSONANT SIGN FINAL MCHAM CONSONANT SIGN FINAL HCHAM DIGIT ZEROCHAM DIGIT ONECHAM DIGIT TWOCHAM DIGIT THREECHAM DIGIT FOURCHAM DIGIT FIVECHAM DIGIT SIXCHAM DIGIT SEVENCHAM DIGIT EIGHTCHAM DIGIT NINECHAM PUNCTUATION SPIRA\
    \LCHAM PUNCTUATION DANDACHAM PUNCTUATION DOUBLE DANDACHAM PUNCTUATION TRIPLE DANDAMYANMAR LETTER KHAMTI GAMYANMAR LETTER KHAMTI CAMYANMAR LETTER KHAMTI CHAMYANMAR LETTER KHAMTI JAMYANMAR LETTER KHAMTI JHAMYANMAR LETTER KHAMTI NYAMYANMAR LETTER KHAMTI TTAMY\
    \ANMAR LETTER KHAMTI TTHAMYANMAR LETTER KHAMTI DDAMYANMAR LETTER KHAMTI DDHAMYANMAR LETTER KHAMTI DHAMYANMAR LETTER KHAMTI NAMYANMAR LETTER KHAMTI SAMYANMAR LETTER KHAMTI HAMYANMAR LETTER KHAMTI HHAMYANMAR LETTER KHAMTI FAMYANMAR MODIFIER LETTER KHAMTI RED\
    \UPLICATIONMYANMAR LETTER KHAMTI XAMYANMAR LETTER KHAMTI ZAMYANMAR LETTER KHAMTI RAMYANMAR LOGOGRAM KHAMTI OAYMYANMAR LOGOGRAM KHAMTI QNMYANMAR LOGOGRAM KHAMTI HMMYANMAR SYMBOL AITON EXCLAMATIONMYANMAR SYMBOL AITON ONEMYANMAR SYMBOL AITON TWOMYANMAR LETTER\
    \ AITON RAMYANMAR SIGN PAO KAREN TONEMYANMAR SIGN TAI LAING TONE-2MYANMAR SIGN TAI LAING TONE-5MYANMAR LETTER SHWE PALAUNG CHAMYANMAR LETTER SHWE PALAUNG SHATAI VIET LETTER LOW KOTAI VIET LETTER HIGH KOTAI VIET LETTER LOW KHOTAI VIET LETTER HIGH KHOTAI VIE\
    \T LETTER LOW KHHOTAI VIET LETTER HIGH KHHOTAI VIET LETTER LOW GOTAI VIET LETTER HIGH GOTAI VIET LETTER LOW NGOTAI VIET LETTER HIGH NGOTAI VIET LETTER LOW COTAI VIET LETTER HIGH COTAI VIET LETTER LOW CHOTAI VIET LETTER HIGH CHOTAI VIET LETTER LOW SOTAI VIE\
    \T LETTER HIGH SOTAI VIET LETTER LOW NYOTAI VIET LETTER HIGH NYOTAI VIET LETTER LOW DOTAI VIET LETTER HIGH DOTAI VIET LETTER LOW TOTAI VIET LETTER HIGH TOTAI VIET LETTER LOW THOTAI VIET LETTER HIGH THOTAI VIET LETTER LOW NOTAI VIET LETTER HIGH NOTAI VIET L\
    \ETTER LOW BOTAI VIET LETTER HIGH BOTAI VIET LETTER LOW POTAI VIET LETTER HIGH POTAI VIET LETTER LOW PHOTAI VIET LETTER HIGH PHOTAI VIET LETTER LOW FOTAI VIET LETTER HIGH FOTAI VIET LETTER LOW MOTAI VIET LETTER HIGH MOTAI VIET LETTER LOW YOTAI VIET LETTER \
    \HIGH YOTAI VIET LETTER LOW ROTAI VIET LETTER HIGH ROTAI VIET LETTER LOW LOTAI VIET LETTER HIGH LOTAI VIET LETTER LOW VOTAI VIET LETTER HIGH VOTAI VIET LETTER LOW HOTAI VIET LETTER HIGH HOTAI VIET LETTER LOW OTAI VIET LETTER HIGH OTAI VIET MAI KANGTAI VIET\
    \ VOWEL AATAI VIET VOWEL ITAI VIET VOWEL UETAI VIET VOWEL UTAI VIET VOWEL ETAI VIET VOWEL OTAI VIET MAI KHITTAI VIET VOWEL IATAI VIET VOWEL UEATAI VIET VOWEL UATAI VIET VOWEL AUETAI VIET VOWEL AYTAI VIET VOWEL ANTAI VIET VOWEL AMTAI VIET TONE MAI EKTAI VIE\
    \T TONE MAI NUENGTAI VIET TONE MAI THOTAI VIET TONE MAI SONGTAI VIET SYMBOL KONTAI VIET SYMBOL NUENGTAI VIET SYMBOL SAMTAI VIET SYMBOL HO HOITAI VIET SYMBOL KOI KOIMEETEI MAYEK LETTER EMEETEI MAYEK LETTER OMEETEI MAYEK LETTER CHAMEETEI MAYEK LETTER NYAMEET\
    \EI MAYEK LETTER TTAMEETEI MAYEK LETTER TTHAMEETEI MAYEK LETTER DDAMEETEI MAYEK LETTER DDHAMEETEI MAYEK LETTER NNAMEETEI MAYEK LETTER SHAMEETEI MAYEK LETTER SSAMEETEI MAYEK VOWEL SIGN IIMEETEI MAYEK VOWEL SIGN UUMEETEI MAYEK VOWEL SIGN AAIMEETEI MAYEK VOWE\
    \L SIGN AUMEETEI MAYEK VOWEL SIGN AAUMEETEI MAYEK CHEIKHANMEETEI MAYEK AHANG KHUDAMMEETEI MAYEK ANJIMEETEI MAYEK SYLLABLE REPETITION MARKMEETEI MAYEK WORD REPETITION MARKMEETEI MAYEK VOWEL SIGN VISARGAMEETEI MAYEK VIRAMAETHIOPIC SYLLABLE TTHUETHIOPIC SYLLA\
    \BLE TTHIETHIOPIC SYLLABLE TTHAAETHIOPIC SYLLABLE TTHEEETHIOPIC SYLLABLE TTHEETHIOPIC SYLLABLE TTHOETHIOPIC SYLLABLE DDHUETHIOPIC SYLLABLE DDHIETHIOPIC SYLLABLE DDHAAETHIOPIC SYLLABLE DDHEEETHIOPIC SYLLABLE DDHEETHIOPIC SYLLABLE DDHOETHIOPIC SYLLABLE DZUET\
    \HIOPIC SYLLABLE DZIETHIOPIC SYLLABLE DZAAETHIOPIC SYLLABLE DZEEETHIOPIC SYLLABLE DZEETHIOPIC SYLLABLE DZOETHIOPIC SYLLABLE CCHHAETHIOPIC SYLLABLE CCHHUETHIOPIC SYLLABLE CCHHIETHIOPIC SYLLABLE CCHHAAETHIOPIC SYLLABLE CCHHEEETHIOPIC SYLLABLE CCHHEETHIOPIC S\
    \YLLABLE CCHHOETHIOPIC SYLLABLE BBAETHIOPIC SYLLABLE BBUETHIOPIC SYLLABLE BBIETHIOPIC SYLLABLE BBAAETHIOPIC SYLLABLE BBEEETHIOPIC SYLLABLE BBEETHIOPIC SYLLABLE BBOLATIN SMALL LETTER BARRED ALPHALATIN SMALL LETTER A REVERSED-SCHWALATIN SMALL LETTER BLACKLET\
    \TER ELATIN SMALL LETTER BARRED ELATIN SMALL LETTER E WITH FLOURISHLATIN SMALL LETTER LENIS FLATIN SMALL LETTER SCRIPT G WITH CROSSED-TAILLATIN SMALL LETTER L WITH INVERTED LAZY SLATIN SMALL LETTER L WITH DOUBLE MIDDLE TILDELATIN SMALL LETTER L WITH MIDDLE\
    \ RINGLATIN SMALL LETTER M WITH CROSSED-TAILLATIN SMALL LETTER N WITH CROSSED-TAILLATIN SMALL LETTER ENG WITH CROSSED-TAILLATIN SMALL LETTER BLACKLETTER OLATIN SMALL LETTER BLACKLETTER O WITH STROKELATIN SMALL LETTER OPEN O WITH STROKELATIN SMALL LETTER IN\
    \VERTED OELATIN SMALL LETTER TURNED OE WITH STROKELATIN SMALL LETTER TURNED OE WITH HORIZONTAL STROKELATIN SMALL LETTER TURNED O OPEN-OLATIN SMALL LETTER TURNED O OPEN-O WITH STROKELATIN SMALL LETTER STIRRUP RLATIN LETTER SMALL CAPITAL R WITH RIGHT LEGLATI\
    \N SMALL LETTER R WITHOUT HANDLELATIN SMALL LETTER DOUBLE RLATIN SMALL LETTER R WITH CROSSED-TAILLATIN SMALL LETTER DOUBLE R WITH CROSSED-TAILLATIN SMALL LETTER SCRIPT RLATIN SMALL LETTER SCRIPT R WITH RINGLATIN SMALL LETTER BASELINE ESHLATIN SMALL LETTER \
    \U WITH SHORT RIGHT LEGLATIN SMALL LETTER U BAR WITH SHORT RIGHT LEGLATIN SMALL LETTER UILATIN SMALL LETTER TURNED UILATIN SMALL LETTER U WITH LEFT HOOKLATIN SMALL LETTER CHILATIN SMALL LETTER CHI WITH LOW RIGHT RINGLATIN SMALL LETTER CHI WITH LOW LEFT SER\
    \IFLATIN SMALL LETTER X WITH LOW RIGHT RINGLATIN SMALL LETTER X WITH LONG LEFT LEGLATIN SMALL LETTER X WITH LONG LEFT LEG AND LOW RIGHT RINGLATIN SMALL LETTER X WITH LONG LEFT LEG WITH SERIFLATIN SMALL LETTER Y WITH SHORT RIGHT LEGMODIFIER BREVE WITH INVER\
    \TED BREVEMODIFIER LETTER SMALL HENGMODIFIER LETTER SMALL L WITH INVERTED LAZY SMODIFIER LETTER SMALL L WITH MIDDLE TILDEMODIFIER LETTER SMALL U WITH LEFT HOOKLATIN SMALL LETTER SAKHA YATLATIN SMALL LETTER IOTIFIED ELATIN SMALL LETTER OPEN OELATIN SMALL LE\
    \TTER UOLATIN SMALL LETTER INVERTED ALPHAGREEK LETTER SMALL CAPITAL OMEGALATIN SMALL LETTER DZ DIGRAPH WITH RETROFLEX HOOKLATIN SMALL LETTER TS DIGRAPH WITH RETROFLEX HOOKLATIN SMALL LETTER TURNED R WITH MIDDLE TILDEMODIFIER LETTER SMALL TURNED WMODIFIER L\
    \ETTER LEFT TACKMODIFIER LETTER RIGHT TACKCHEROKEE SMALL LETTER ACHEROKEE SMALL LETTER ECHEROKEE SMALL LETTER ICHEROKEE SMALL LETTER OCHEROKEE SMALL LETTER UCHEROKEE SMALL LETTER VCHEROKEE SMALL LETTER GACHEROKEE SMALL LETTER KACHEROKEE SMALL LETTER GECHER\
    \OKEE SMALL LETTER GICHEROKEE SMALL LETTER GOCHEROKEE SMALL LETTER GUCHEROKEE SMALL LETTER GVCHEROKEE SMALL LETTER HACHEROKEE SMALL LETTER HECHEROKEE SMALL LETTER HICHEROKEE SMALL LETTER HOCHEROKEE SMALL LETTER HUCHEROKEE SMALL LETTER HVCHEROKEE SMALL LETT\
    \ER LACHEROKEE SMALL LETTER LECHEROKEE SMALL LETTER LICHEROKEE SMALL LETTER LOCHEROKEE SMALL LETTER LUCHEROKEE SMALL LETTER LVCHEROKEE SMALL LETTER MACHEROKEE SMALL LETTER MECHEROKEE SMALL LETTER MICHEROKEE SMALL LETTER MOCHEROKEE SMALL LETTER MUCHEROKEE S\
    \MALL LETTER NACHEROKEE SMALL LETTER HNACHEROKEE SMALL LETTER NAHCHEROKEE SMALL LETTER NECHEROKEE SMALL LETTER NICHEROKEE SMALL LETTER NOCHEROKEE SMALL LETTER NUCHEROKEE SMALL LETTER NVCHEROKEE SMALL LETTER QUACHEROKEE SMALL LETTER QUECHEROKEE SMALL LETTER\
    \ QUICHEROKEE SMALL LETTER QUOCHEROKEE SMALL LETTER QUUCHEROKEE SMALL LETTER QUVCHEROKEE SMALL LETTER SACHEROKEE SMALL LETTER SCHEROKEE SMALL LETTER SECHEROKEE SMALL LETTER SICHEROKEE SMALL LETTER SOCHEROKEE SMALL LETTER SUCHEROKEE SMALL LETTER SVCHEROKEE \
    \SMALL LETTER DACHEROKEE SMALL LETTER TACHEROKEE SMALL LETTER DECHEROKEE SMALL LETTER TECHEROKEE SMALL LETTER DICHEROKEE SMALL LETTER TICHEROKEE SMALL LETTER DOCHEROKEE SMALL LETTER DUCHEROKEE SMALL LETTER DVCHEROKEE SMALL LETTER DLACHEROKEE SMALL LETTER T\
    \LACHEROKEE SMALL LETTER TLECHEROKEE SMALL LETTER TLICHEROKEE SMALL LETTER TLOCHEROKEE SMALL LETTER TLUCHEROKEE SMALL LETTER TLVCHEROKEE SMALL LETTER TSACHEROKEE SMALL LETTER TSECHEROKEE SMALL LETTER TSICHEROKEE SMALL LETTER TSOCHEROKEE SMALL LETTER TSUCHE\
    \ROKEE SMALL LETTER TSVCHEROKEE SMALL LETTER WACHEROKEE SMALL LETTER WECHEROKEE SMALL LETTER WICHEROKEE SMALL LETTER WOCHEROKEE SMALL LETTER WUCHEROKEE SMALL LETTER WVCHEROKEE SMALL LETTER YAMEETEI MAYEK LETTER KOKMEETEI MAYEK LETTER SAMMEETEI MAYEK LETTER\
    \ LAIMEETEI MAYEK LETTER MITMEETEI MAYEK LETTER PAMEETEI MAYEK LETTER NAMEETEI MAYEK LETTER CHILMEETEI MAYEK LETTER TILMEETEI MAYEK LETTER KHOUMEETEI MAYEK LETTER NGOUMEETEI MAYEK LETTER THOUMEETEI MAYEK LETTER WAIMEETEI MAYEK LETTER YANGMEETEI MAYEK LETTE\
    \R HUKMEETEI MAYEK LETTER UNMEETEI MAYEK LETTER IMEETEI MAYEK LETTER PHAMMEETEI MAYEK LETTER ATIYAMEETEI MAYEK LETTER GOKMEETEI MAYEK LETTER JHAMMEETEI MAYEK LETTER RAIMEETEI MAYEK LETTER BAMEETEI MAYEK LETTER JILMEETEI MAYEK LETTER DILMEETEI MAYEK LETTER \
    \GHOUMEETEI MAYEK LETTER DHOUMEETEI MAYEK LETTER BHAMMEETEI MAYEK LETTER KOK LONSUMMEETEI MAYEK LETTER LAI LONSUMMEETEI MAYEK LETTER MIT LONSUMMEETEI MAYEK LETTER PA LONSUMMEETEI MAYEK LETTER NA LONSUMMEETEI MAYEK LETTER TIL LONSUMMEETEI MAYEK LETTER NGOU \
    \LONSUMMEETEI MAYEK LETTER I LONSUMMEETEI MAYEK VOWEL SIGN ONAPMEETEI MAYEK VOWEL SIGN INAPMEETEI MAYEK VOWEL SIGN ANAPMEETEI MAYEK VOWEL SIGN YENAPMEETEI MAYEK VOWEL SIGN SOUNAPMEETEI MAYEK VOWEL SIGN UNAPMEETEI MAYEK VOWEL SIGN CHEINAPMEETEI MAYEK VOWEL \
    \SIGN NUNGMEETEI MAYEK CHEIKHEIMEETEI MAYEK LUM IYEKMEETEI MAYEK APUN IYEKMEETEI MAYEK DIGIT ZEROMEETEI MAYEK DIGIT ONEMEETEI MAYEK DIGIT TWOMEETEI MAYEK DIGIT THREEMEETEI MAYEK DIGIT FOURMEETEI MAYEK DIGIT FIVEMEETEI MAYEK DIGIT SIXMEETEI MAYEK DIGIT SEVE\
    \NMEETEI MAYEK DIGIT EIGHTMEETEI MAYEK DIGIT NINEGAGAGGAGGGAGSGANGANJGANHGADGALGALGGALMGALBGALSGALTGALPGALHGAMGABGABSGASGASSGANGGAJGACGAKGATGAPGAHGAEGAEGGAEGGGAEGSGAENGAENJGAENHGAEDGAELGAELGGAELMGAELBGAELSGAELTGAELPGAELHGAEMGAEBGAEBSGAESGAESSGAENGGAEJGAECG\
    \AEKGAETGAEPGAEHGYAGYAGGYAGGGYAGSGYANGYANJGYANHGYADGYALGYALGGYALMGYALBGYALSGYALTGYALPGYALHGYAMGYABGYABSGYASGYASSGYANGGYAJGYACGYAKGYATGYAPGYAHGYAEGYAEGGYAEGGGYAEGSGYAENGYAENJGYAENHGYAEDGYAELGYAELGGYAELMGYAELBGYAELSGYAELTGYAELPGYAELHGYAEMGYAEBGYAEBSGYAESGYAE\
    \SSGYAENGGYAEJGYAECGYAEKGYAETGYAEPGYAEHGEOGEOGGEOGGGEOGSGEONGEONJGEONHGEODGEOLGEOLGGEOLMGEOLBGEOLSGEOLTGEOLPGEOLHGEOMGEOBGEOBSGEOSGEOSSGEONGGEOJGEOCGEOKGEOTGEOPGEOHGEGEGGEGGGEGSGENGENJGENHGEDGELGELGGELMGELBGELSGELTGELPGELHGEMGEBGEBSGESGESSGENGGEJGECGEKGETG\
    \EPGEHGYEOGYEOGGYEOGGGYEOGSGYEONGYEONJGYEONHGYEODGYEOLGYEOLGGYEOLMGYEOLBGYEOLSGYEOLTGYEOLPGYEOLHGYEOMGYEOBGYEOBSGYEOSGYEOSSGYEONGGYEOJGYEOCGYEOKGYEOTGYEOPGYEOHGYEGYEGGYEGGGYEGSGYENGYENJGYENHGYEDGYELGYELGGYELMGYELBGYELSGYELTGYELPGYELHGYEMGYEBGYEBSGYESGYESSG\
    \YENGGYEJGYECGYEKGYETGYEPGYEHGOGOGGOGGGOGSGONGONJGONHGODGOLGOLGGOLMGOLBGOLSGOLTGOLPGOLHGOMGOBGOBSGOSGOSSGONGGOJGOCGOKGOTGOPGOHGWAGWAGGWAGGGWAGSGWANGWANJGWANHGWADGWALGWALGGWALMGWALBGWALSGWALTGWALPGWALHGWAMGWABGWABSGWASGWASSGWANGGWAJGWACGWAKGWATGWAPGWAHGWAEG\
    \WAEGGWAEGGGWAEGSGWAENGWAENJGWAENHGWAEDGWAELGWAELGGWAELMGWAELBGWAELSGWAELTGWAELPGWAELHGWAEMGWAEBGWAEBSGWAESGWAESSGWAENGGWAEJGWAECGWAEKGWAETGWAEPGWAEHGOEGOEGGOEGGGOEGSGOENGOENJGOENHGOEDGOELGOELGGOELMGOELBGOELSGOELTGOELPGOELHGOEMGOEBGOEBSGOESGOESSGOENGGOEJGO\
    \ECGOEKGOETGOEPGOEHGYOGYOGGYOGGGYOGSGYONGYONJGYONHGYODGYOLGYOLGGYOLMGYOLBGYOLSGYOLTGYOLPGYOLHGYOMGYOBGYOBSGYOSGYOSSGYONGGYOJGYOCGYOKGYOTGYOPGYOHGUGUGGUGGGUGSGUNGUNJGUNHGUDGULGULGGULMGULBGULSGULTGULPGULHGUMGUBGUBSGUSGUSSGUNGGUJGUCGUKGUTGUPGUHGWEOGWEOGGWEOGG\
    \GWEOGSGWEONGWEONJGWEONHGWEODGWEOLGWEOLGGWEOLMGWEOLBGWEOLSGWEOLTGWEOLPGWEOLHGWEOMGWEOBGWEOBSGWEOSGWEOSSGWEONGGWEOJGWEOCGWEOKGWEOTGWEOPGWEOHGWEGWEGGWEGGGWEGSGWENGWENJGWENHGWEDGWELGWELGGWELMGWELBGWELSGWELTGWELPGWELHGWEMGWEBGWEBSGWESGWESSGWENGGWEJGWECGWEKGWET\
    \GWEPGWEHGWIGWIGGWIGGGWIGSGWINGWINJGWINHGWIDGWILGWILGGWILMGWILBGWILSGWILTGWILPGWILHGWIMGWIBGWIBSGWISGWISSGWINGGWIJGWICGWIKGWITGWIPGWIHGYUGYUGGYUGGGYUGSGYUNGYUNJGYUNHGYUDGYULGYULGGYULMGYULBGYULSGYULTGYULPGYULHGYUMGYUBGYUBSGYUSGYUSSGYUNGGYUJGYUCGYUKGYUTGYUPG\
    \YUHGEUGEUGGEUGGGEUGSGEUNGEUNJGEUNHGEUDGEULGEULGGEULMGEULBGEULSGEULTGEULPGEULHGEUMGEUBGEUBSGEUSGEUSSGEUNGGEUJGEUCGEUKGEUTGEUPGEUHGYIGYIGGYIGGGYIGSGYINGYINJGYINHGYIDGYILGYILGGYILMGYILBGYILSGYILTGYILPGYILHGYIMGYIBGYIBSGYISGYISSGYINGGYIJGYICGYIKGYITGYIPGYIHGI\
    \GIGGIGGGIGSGINGINJGINHGIDGILGILGGILMGILBGILSGILTGILPGILHGIMGIBGIBSGISGISSGINGGIJGICGIKGITGIPGIHGGAGGAGGGAGGGGAGSGGANGGANJGGANHGGADGGALGGALGGGALMGGALBGGALSGGALTGGALPGGALHGGAMGGABGGABSGGASGGASSGGANGGGAJGGACGGAKGGATGGAPGGAHGGAEGGAEGGGAEGGGGAEGSGGAENGGAENJGGA\
    \ENHGGAEDGGAELGGAELGGGAELMGGAELBGGAELSGGAELTGGAELPGGAELHGGAEMGGAEBGGAEBSGGAESGGAESSGGAENGGGAEJGGAECGGAEKGGAETGGAEPGGAEHGGYAGGYAGGGYAGGGGYAGSGGYANGGYANJGGYANHGGYADGGYALGGYALGGGYALMGGYALBGGYALSGGYALTGGYALPGGYALHGGYAMGGYABGGYABSGGYASGGYASSGGYANGGGYAJGGYACGGYA\
    \KGGYATGGYAPGGYAHGGYAEGGYAEGGGYAEGGGGYAEGSGGYAENGGYAENJGGYAENHGGYAEDGGYAELGGYAELGGGYAELMGGYAELBGGYAELSGGYAELTGGYAELPGGYAELHGGYAEMGGYAEBGGYAEBSGGYAESGGYAESSGGYAENGGGYAEJGGYAECGGYAEKGGYAETGGYAEPGGYAEHGGEOGGEOGGGEOGGGGEOGSGGEONGGEONJGGEONHGGEODGGEOLGGEOLGGGEO\
    \LMGGEOLBGGEOLSGGEOLTGGEOLPGGEOLHGGEOMGGEOBGGEOBSGGEOSGGEOSSGGEONGGGEOJGGEOCGGEOKGGEOTGGEOPGGEOHGGEGGEGGGEGGGGEGSGGENGGENJGGENHGGEDGGELGGELGGGELMGGELBGGELSGGELTGGELPGGELHGGEMGGEBGGEBSGGESGGESSGGENGGGEJGGECGGEKGGETGGEPGGEHGGYEOGGYEOGGGYEOGGGGYEOGSGGYEONGGYE\
    \ONJGGYEONHGGYEODGGYEOLGGYEOLGGGYEOLMGGYEOLBGGYEOLSGGYEOLTGGYEOLPGGYEOLHGGYEOMGGYEOBGGYEOBSGGYEOSGGYEOSSGGYEONGGGYEOJGGYEOCGGYEOKGGYEOTGGYEOPGGYEOHGGYEGGYEGGGYEGGGGYEGSGGYENGGYENJGGYENHGGYEDGGYELGGYELGGGYELMGGYELBGGYELSGGYELTGGYELPGGYELHGGYEMGGYEBGGYEBSGGY\
    \ESGGYESSGGYENGGGYEJGGYECGGYEKGGYETGGYEPGGYEHGGOGGOGGGOGGGGOGSGGONGGONJGGONHGGODGGOLGGOLGGGOLMGGOLBGGOLSGGOLTGGOLPGGOLHGGOMGGOBGGOBSGGOSGGOSSGGONGGGOJGGOCGGOKGGOTGGOPGGOHGGWAGGWAGGGWAGGGGWAGSGGWANGGWANJGGWANHGGWADGGWALGGWALGGGWALMGGWALBGGWALSGGWALTGGWALPGG\
    \WALHGGWAMGGWABGGWABSGGWASGGWASSGGWANGGGWAJGGWACGGWAKGGWATGGWAPGGWAHGGWAEGGWAEGGGWAEGGGGWAEGSGGWAENGGWAENJGGWAENHGGWAEDGGWAELGGWAELGGGWAELMGGWAELBGGWAELSGGWAELTGGWAELPGGWAELHGGWAEMGGWAEBGGWAEBSGGWAESGGWAESSGGWAENGGGWAEJGGWAECGGWAEKGGWAETGGWAEPGGWAEHGGOEGGO\
    \EGGGOEGGGGOEGSGGOENGGOENJGGOENHGGOEDGGOELGGOELGGGOELMGGOELBGGOELSGGOELTGGOELPGGOELHGGOEMGGOEBGGOEBSGGOESGGOESSGGOENGGGOEJGGOECGGOEKGGOETGGOEPGGOEHGGYOGGYOGGGYOGGGGYOGSGGYONGGYONJGGYONHGGYODGGYOLGGYOLGGGYOLMGGYOLBGGYOLSGGYOLTGGYOLPGGYOLHGGYOMGGYOBGGYOBSGGY\
    \OSGGYOSSGGYONGGGYOJGGYOCGGYOKGGYOTGGYOPGGYOHGGUGGUGGGUGGGGUGSGGUNGGUNJGGUNHGGUDGGULGGULGGGULMGGULBGGULSGGULTGGULPGGULHGGUMGGUBGGUBSGGUSGGUSSGGUNGGGUJGGUCGGUKGGUTGGUPGGUHGGWEOGGWEOGGGWEOGGGGWEOGSGGWEONGGWEONJGGWEONHGGWEODGGWEOLGGWEOLGGGWEOLMGGWEOLBGGWEOLSG\
    \GWEOLTGGWEOLPGGWEOLHGGWEOMGGWEOBGGWEOBSGGWEOSGGWEOSSGGWEONGGGWEOJGGWEOCGGWEOKGGWEOTGGWEOPGGWEOHGGWEGGWEGGGWEGGGGWEGSGGWENGGWENJGGWENHGGWEDGGWELGGWELGGGWELMGGWELBGGWELSGGWELTGGWELPGGWELHGGWEMGGWEBGGWEBSGGWESGGWESSGGWENGGGWEJGGWECGGWEKGGWETGGWEPGGWEHGGWIGGW\
    \IGGGWIGGGGWIGSGGWINGGWINJGGWINHGGWIDGGWILGGWILGGGWILMGGWILBGGWILSGGWILTGGWILPGGWILHGGWIMGGWIBGGWIBSGGWISGGWISSGGWINGGGWIJGGWICGGWIKGGWITGGWIPGGWIHGGYUGGYUGGGYUGGGGYUGSGGYUNGGYUNJGGYUNHGGYUDGGYULGGYULGGGYULMGGYULBGGYULSGGYULTGGYULPGGYULHGGYUMGGYUBGGYUBSGGY\
    \USGGYUSSGGYUNGGGYUJGGYUCGGYUKGGYUTGGYUPGGYUHGGEUGGEUGGGEUGGGGEUGSGGEUNGGEUNJGGEUNHGGEUDGGEULGGEULGGGEULMGGEULBGGEULSGGEULTGGEULPGGEULHGGEUMGGEUBGGEUBSGGEUSGGEUSSGGEUNGGGEUJGGEUCGGEUKGGEUTGGEUPGGEUHGGYIGGYIGGGYIGGGGYIGSGGYINGGYINJGGYINHGGYIDGGYILGGYILGGGYI\
    \LMGGYILBGGYILSGGYILTGGYILPGGYILHGGYIMGGYIBGGYIBSGGYISGGYISSGGYINGGGYIJGGYICGGYIKGGYITGGYIPGGYIHGGIGGIGGGIGGGGIGSGGINGGINJGGINHGGIDGGILGGILGGGILMGGILBGGILSGGILTGGILPGGILHGGIMGGIBGGIBSGGISGGISSGGINGGGIJGGICGGIKGGITGGIPGGIHNANAGNAGGNAGSNANNANJNANHNADNALNALGN\
    \ALMNALBNALSNALTNALPNALHNAMNABNABSNASNASSNANGNAJNACNAKNATNAPNAHNAENAEGNAEGGNAEGSNAENNAENJNAENHNAEDNAELNAELGNAELMNAELBNAELSNAELTNAELPNAELHNAEMNAEBNAEBSNAESNAESSNAENGNAEJNAECNAEKNAETNAEPNAEHNYANYAGNYAGGNYAGSNYANNYANJNYANHNYADNYALNYALGNYALMNYALBNYALSNYALTNYAL\
    \PNYALHNYAMNYABNYABSNYASNYASSNYANGNYAJNYACNYAKNYATNYAPNYAHNYAENYAEGNYAEGGNYAEGSNYAENNYAENJNYAENHNYAEDNYAELNYAELGNYAELMNYAELBNYAELSNYAELTNYAELPNYAELHNYAEMNYAEBNYAEBSNYAESNYAESSNYAENGNYAEJNYAECNYAEKNYAETNYAEPNYAEHNEONEOGNEOGGNEOGSNEONNEONJNEONHNEODNEOLNEOLGN\
    \EOLMNEOLBNEOLSNEOLTNEOLPNEOLHNEOMNEOBNEOBSNEOSNEOSSNEONGNEOJNEOCNEOKNEOTNEOPNEOHNENEGNEGGNEGSNENNENJNENHNEDNELNELGNELMNELBNELSNELTNELPNELHNEMNEBNEBSNESNESSNENGNEJNECNEKNETNEPNEHNYEONYEOGNYEOGGNYEOGSNYEONNYEONJNYEONHNYEODNYEOLNYEOLGNYEOLMNYEOLBNYEOLSNYEOLT\
    \NYEOLPNYEOLHNYEOMNYEOBNYEOBSNYEOSNYEOSSNYEONGNYEOJNYEOCNYEOKNYEOTNYEOPNYEOHNYENYEGNYEGGNYEGSNYENNYENJNYENHNYEDNYELNYELGNYELMNYELBNYELSNYELTNYELPNYELHNYEMNYEBNYEBSNYESNYESSNYENGNYEJNYECNYEKNYETNYEPNYEHNONOGNOGGNOGSNONNONJNONHNODNOLNOLGNOLMNOLBNOLSNOLTNOLPN\
    \OLHNOMNOBNOBSNOSNOSSNONGNOJNOCNOKNOTNOPNOHNWANWAGNWAGGNWAGSNWANNWANJNWANHNWADNWALNWALGNWALMNWALBNWALSNWALTNWALPNWALHNWAMNWABNWABSNWASNWASSNWANGNWAJNWACNWAKNWATNWAPNWAHNWAENWAEGNWAEGGNWAEGSNWAENNWAENJNWAENHNWAEDNWAELNWAELGNWAELMNWAELBNWAELSNWAELTNWAELPNWAE\
    \LHNWAEMNWAEBNWAEBSNWAESNWAESSNWAENGNWAEJNWAECNWAEKNWAETNWAEPNWAEHNOENOEGNOEGGNOEGSNOENNOENJNOENHNOEDNOELNOELGNOELMNOELBNOELSNOELTNOELPNOELHNOEMNOEBNOEBSNOESNOESSNOENGNOEJNOECNOEKNOETNOEPNOEHNYONYOGNYOGGNYOGSNYONNYONJNYONHNYODNYOLNYOLGNYOLMNYOLBNYOLSNYOLTN\
    \YOLPNYOLHNYOMNYOBNYOBSNYOSNYOSSNYONGNYOJNYOCNYOKNYOTNYOPNYOHNUNUGNUGGNUGSNUNNUNJNUNHNUDNULNULGNULMNULBNULSNULTNULPNULHNUMNUBNUBSNUSNUSSNUNGNUJNUCNUKNUTNUPNUHNWEONWEOGNWEOGGNWEOGSNWEONNWEONJNWEONHNWEODNWEOLNWEOLGNWEOLMNWEOLBNWEOLSNWEOLTNWEOLPNWEOLHNWEOMNWE\
    \OBNWEOBSNWEOSNWEOSSNWEONGNWEOJNWEOCNWEOKNWEOTNWEOPNWEOHNWENWEGNWEGGNWEGSNWENNWENJNWENHNWEDNWELNWELGNWELMNWELBNWELSNWELTNWELPNWELHNWEMNWEBNWEBSNWESNWESSNWENGNWEJNWECNWEKNWETNWEPNWEHNWINWIGNWIGGNWIGSNWINNWINJNWINHNWIDNWILNWILGNWILMNWILBNWILSNWILTNWILPNWILHN\
    \WIMNWIBNWIBSNWISNWISSNWINGNWIJNWICNWIKNWITNWIPNWIHNYUNYUGNYUGGNYUGSNYUNNYUNJNYUNHNYUDNYULNYULGNYULMNYULBNYULSNYULTNYULPNYULHNYUMNYUBNYUBSNYUSNYUSSNYUNGNYUJNYUCNYUKNYUTNYUPNYUHNEUNEUGNEUGGNEUGSNEUNNEUNJNEUNHNEUDNEULNEULGNEULMNEULBNEULSNEULTNEULPNEULHNEUMNE\
    \UBNEUBSNEUSNEUSSNEUNGNEUJNEUCNEUKNEUTNEUPNEUHNYINYIGNYIGGNYIGSNYINNYINJNYINHNYIDNYILNYILGNYILMNYILBNYILSNYILTNYILPNYILHNYIMNYIBNYIBSNYISNYISSNYINGNYIJNYICNYIKNYITNYIPNYIHNINIGNIGGNIGSNINNINJNINHNIDNILNILGNILMNILBNILSNILTNILPNILHNIMNIBNIBSNISNISSNINGNIJNIC\
    \NIKNITNIPNIHDADAGDAGGDAGSDANDANJDANHDADDALDALGDALMDALBDALSDALTDALPDALHDAMDABDABSDASDASSDANGDAJDACDAKDATDAPDAHDAEDAEGDAEGGDAEGSDAENDAENJDAENHDAEDDAELDAELGDAELMDAELBDAELSDAELTDAELPDAELHDAEMDAEBDAEBSDAESDAESSDAENGDAEJDAECDAEKDAETDAEPDAEHDYADYAGDYAGGDYAGSDYAN\
    \DYANJDYANHDYADDYALDYALGDYALMDYALBDYALSDYALTDYALPDYALHDYAMDYABDYABSDYASDYASSDYANGDYAJDYACDYAKDYATDYAPDYAHDYAEDYAEGDYAEGGDYAEGSDYAENDYAENJDYAENHDYAEDDYAELDYAELGDYAELMDYAELBDYAELSDYAELTDYAELPDYAELHDYAEMDYAEBDYAEBSDYAESDYAESSDYAENGDYAEJDYAECDYAEKDYAETDYAEPDYA\
    \EHDEODEOGDEOGGDEOGSDEONDEONJDEONHDEODDEOLDEOLGDEOLMDEOLBDEOLSDEOLTDEOLPDEOLHDEOMDEOBDEOBSDEOSDEOSSDEONGDEOJDEOCDEOKDEOTDEOPDEOHDEDEGDEGGDEGSDENDENJDENHDEDDELDELGDELMDELBDELSDELTDELPDELHDEMDEBDEBSDESDESSDENGDEJDECDEKDETDEPDEHDYEODYEOGDYEOGGDYEOGSDYEONDYEON\
    \JDYEONHDYEODDYEOLDYEOLGDYEOLMDYEOLBDYEOLSDYEOLTDYEOLPDYEOLHDYEOMDYEOBDYEOBSDYEOSDYEOSSDYEONGDYEOJDYEOCDYEOKDYEOTDYEOPDYEOHDYEDYEGDYEGGDYEGSDYENDYENJDYENHDYEDDYELDYELGDYELMDYELBDYELSDYELTDYELPDYELHDYEMDYEBDYEBSDYESDYESSDYENGDYEJDYECDYEKDYETDYEPDYEHDODOGDOG\
    \GDOGSDONDONJDONHDODDOLDOLGDOLMDOLBDOLSDOLTDOLPDOLHDOMDOBDOBSDOSDOSSDONGDOJDOCDOKDOTDOPDOHDWADWAGDWAGGDWAGSDWANDWANJDWANHDWADDWALDWALGDWALMDWALBDWALSDWALTDWALPDWALHDWAMDWABDWABSDWASDWASSDWANGDWAJDWACDWAKDWATDWAPDWAHDWAEDWAEGDWAEGGDWAEGSDWAENDWAENJDWAENHDWA\
    \EDDWAELDWAELGDWAELMDWAELBDWAELSDWAELTDWAELPDWAELHDWAEMDWAEBDWAEBSDWAESDWAESSDWAENGDWAEJDWAECDWAEKDWAETDWAEPDWAEHDOEDOEGDOEGGDOEGSDOENDOENJDOENHDOEDDOELDOELGDOELMDOELBDOELSDOELTDOELPDOELHDOEMDOEBDOEBSDOESDOESSDOENGDOEJDOECDOEKDOETDOEPDOEHDYODYOGDYOGGDYOGSD\
    \YONDYONJDYONHDYODDYOLDYOLGDYOLMDYOLBDYOLSDYOLTDYOLPDYOLHDYOMDYOBDYOBSDYOSDYOSSDYONGDYOJDYOCDYOKDYOTDYOPDYOHDUDUGDUGGDUGSDUNDUNJDUNHDUDDULDULGDULMDULBDULSDULTDULPDULHDUMDUBDUBSDUSDUSSDUNGDUJDUCDUKDUTDUPDUHDWEODWEOGDWEOGGDWEOGSDWEONDWEONJDWEONHDWEODDWEOLDWE\
    \OLGDWEOLMDWEOLBDWEOLSDWEOLTDWEOLPDWEOLHDWEOMDWEOBDWEOBSDWEOSDWEOSSDWEONGDWEOJDWEOCDWEOKDWEOTDWEOPDWEOHDWEDWEGDWEGGDWEGSDWENDWENJDWENHDWEDDWELDWELGDWELMDWELBDWELSDWELTDWELPDWELHDWEMDWEBDWEBSDWESDWESSDWENGDWEJDWECDWEKDWETDWEPDWEHDWIDWIGDWIGGDWIGSDWINDWINJDW\
    \INHDWIDDWILDWILGDWILMDWILBDWILSDWILTDWILPDWILHDWIMDWIBDWIBSDWISDWISSDWINGDWIJDWICDWIKDWITDWIPDWIHDYUDYUGDYUGGDYUGSDYUNDYUNJDYUNHDYUDDYULDYULGDYULMDYULBDYULSDYULTDYULPDYULHDYUMDYUBDYUBSDYUSDYUSSDYUNGDYUJDYUCDYUKDYUTDYUPDYUHDEUDEUGDEUGGDEUGSDEUNDEUNJDEUNHDE\
    \UDDEULDEULGDEULMDEULBDEULSDEULTDEULPDEULHDEUMDEUBDEUBSDEUSDEUSSDEUNGDEUJDEUCDEUKDEUTDEUPDEUHDYIDYIGDYIGGDYIGSDYINDYINJDYINHDYIDDYILDYILGDYILMDYILBDYILSDYILTDYILPDYILHDYIMDYIBDYIBSDYISDYISSDYINGDYIJDYICDYIKDYITDYIPDYIHDIDIGDIGGDIGSDINDINJDINHDIDDILDILGDILM\
    \DILBDILSDILTDILPDILHDIMDIBDIBSDISDISSDINGDIJDICDIKDITDIPDIHDDADDAGDDAGGDDAGSDDANDDANJDDANHDDADDDALDDALGDDALMDDALBDDALSDDALTDDALPDDALHDDAMDDABDDABSDDASDDASSDDANGDDAJDDACDDAKDDATDDAPDDAHDDAEDDAEGDDAEGGDDAEGSDDAENDDAENJDDAENHDDAEDDDAELDDAELGDDAELMDDAELBDDAEL\
    \SDDAELTDDAELPDDAELHDDAEMDDAEBDDAEBSDDAESDDAESSDDAENGDDAEJDDAECDDAEKDDAETDDAEPDDAEHDDYADDYAGDDYAGGDDYAGSDDYANDDYANJDDYANHDDYADDDYALDDYALGDDYALMDDYALBDDYALSDDYALTDDYALPDDYALHDDYAMDDYABDDYABSDDYASDDYASSDDYANGDDYAJDDYACDDYAKDDYATDDYAPDDYAHDDYAEDDYAEGDDYAEGGDD\
    \YAEGSDDYAENDDYAENJDDYAENHDDYAEDDDYAELDDYAELGDDYAELMDDYAELBDDYAELSDDYAELTDDYAELPDDYAELHDDYAEMDDYAEBDDYAEBSDDYAESDDYAESSDDYAENGDDYAEJDDYAECDDYAEKDDYAETDDYAEPDDYAEHDDEODDEOGDDEOGGDDEOGSDDEONDDEONJDDEONHDDEODDDEOLDDEOLGDDEOLMDDEOLBDDEOLSDDEOLTDDEOLPDDEOLHDDEO\
    \MDDEOBDDEOBSDDEOSDDEOSSDDEONGDDEOJDDEOCDDEOKDDEOTDDEOPDDEOHDDEDDEGDDEGGDDEGSDDENDDENJDDENHDDEDDDELDDELGDDELMDDELBDDELSDDELTDDELPDDELHDDEMDDEBDDEBSDDESDDESSDDENGDDEJDDECDDEKDDETDDEPDDEHDDYEODDYEOGDDYEOGGDDYEOGSDDYEONDDYEONJDDYEONHDDYEODDDYEOLDDYEOLGDDYEOLM\
    \DDYEOLBDDYEOLSDDYEOLTDDYEOLPDDYEOLHDDYEOMDDYEOBDDYEOBSDDYEOSDDYEOSSDDYEONGDDYEOJDDYEOCDDYEOKDDYEOTDDYEOPDDYEOHDDYEDDYEGDDYEGGDDYEGSDDYENDDYENJDDYENHDDYEDDDYELDDYELGDDYELMDDYELBDDYELSDDYELTDDYELPDDYELHDDYEMDDYEBDDYEBSDDYESDDYESSDDYENGDDYEJDDYECDDYEKDDYETDD\
    \YEPDDYEHDDODDOGDDOGGDDOGSDDONDDONJDDONHDDODDDOLDDOLGDDOLMDDOLBDDOLSDDOLTDDOLPDDOLHDDOMDDOBDDOBSDDOSDDOSSDDONGDDOJDDOCDDOKDDOTDDOPDDOHDDWADDWAGDDWAGGDDWAGSDDWANDDWANJDDWANHDDWADDDWALDDWALGDDWALMDDWALBDDWALSDDWALTDDWALPDDWALHDDWAMDDWABDDWABSDDWASDDWASSDDWAN\
    \GDDWAJDDWACDDWAKDDWATDDWAPDDWAHDDWAEDDWAEGDDWAEGGDDWAEGSDDWAENDDWAENJDDWAENHDDWAEDDDWAELDDWAELGDDWAELMDDWAELBDDWAELSDDWAELTDDWAELPDDWAELHDDWAEMDDWAEBDDWAEBSDDWAESDDWAESSDDWAENGDDWAEJDDWAECDDWAEKDDWAETDDWAEPDDWAEHDDOEDDOEGDDOEGGDDOEGSDDOENDDOENJDDOENHDDOED\
    \DDOELDDOELGDDOELMDDOELBDDOELSDDOELTDDOELPDDOELHDDOEMDDOEBDDOEBSDDOESDDOESSDDOENGDDOEJDDOECDDOEKDDOETDDOEPDDOEHDDYODDYOGDDYOGGDDYOGSDDYONDDYONJDDYONHDDYODDDYOLDDYOLGDDYOLMDDYOLBDDYOLSDDYOLTDDYOLPDDYOLHDDYOMDDYOBDDYOBSDDYOSDDYOSSDDYONGDDYOJDDYOCDDYOKDDYOTDD\
    \YOPDDYOHDDUDDUGDDUGGDDUGSDDUNDDUNJDDUNHDDUDDDULDDULGDDULMDDULBDDULSDDULTDDULPDDULHDDUMDDUBDDUBSDDUSDDUSSDDUNGDDUJDDUCDDUKDDUTDDUPDDUHDDWEODDWEOGDDWEOGGDDWEOGSDDWEONDDWEONJDDWEONHDDWEODDDWEOLDDWEOLGDDWEOLMDDWEOLBDDWEOLSDDWEOLTDDWEOLPDDWEOLHDDWEOMDDWEOBDDWE\
    \OBSDDWEOSDDWEOSSDDWEONGDDWEOJDDWEOCDDWEOKDDWEOTDDWEOPDDWEOHDDWEDDWEGDDWEGGDDWEGSDDWENDDWENJDDWENHDDWEDDDWELDDWELGDDWELMDDWELBDDWELSDDWELTDDWELPDDWELHDDWEMDDWEBDDWEBSDDWESDDWESSDDWENGDDWEJDDWECDDWEKDDWETDDWEPDDWEHDDWIDDWIGDDWIGGDDWIGSDDWINDDWINJDDWINHDDWID\
    \DDWILDDWILGDDWILMDDWILBDDWILSDDWILTDDWILPDDWILHDDWIMDDWIBDDWIBSDDWISDDWISSDDWINGDDWIJDDWICDDWIKDDWITDDWIPDDWIHDDYUDDYUGDDYUGGDDYUGSDDYUNDDYUNJDDYUNHDDYUDDDYULDDYULGDDYULMDDYULBDDYULSDDYULTDDYULPDDYULHDDYUMDDYUBDDYUBSDDYUSDDYUSSDDYUNGDDYUJDDYUCDDYUKDDYUTDD\
    \YUPDDYUHDDEUDDEUGDDEUGGDDEUGSDDEUNDDEUNJDDEUNHDDEUDDDEULDDEULGDDEULMDDEULBDDEULSDDEULTDDEULPDDEULHDDEUMDDEUBDDEUBSDDEUSDDEUSSDDEUNGDDEUJDDEUCDDEUKDDEUTDDEUPDDEUHDDYIDDYIGDDYIGGDDYIGSDDYINDDYINJDDYINHDDYIDDDYILDDYILGDDYILMDDYILBDDYILSDDYILTDDYILPDDYILHDDYI\
    \MDDYIBDDYIBSDDYISDDYISSDDYINGDDYIJDDYICDDYIKDDYITDDYIPDDYIHDDIDDIGDDIGGDDIGSDDINDDINJDDINHDDIDDDILDDILGDDILMDDILBDDILSDDILTDDILPDDILHDDIMDDIBDDIBSDDISDDISSDDINGDDIJDDICDDIKDDITDDIPDDIHRARAGRAGGRAGSRANRANJRANHRADRALRALGRALMRALBRALSRALTRALPRALHRAMRABRABSRAS\
    \RASSRANGRAJRACRAKRATRAPRAHRAERAEGRAEGGRAEGSRAENRAENJRAENHRAEDRAELRAELGRAELMRAELBRAELSRAELTRAELPRAELHRAEMRAEBRAEBSRAESRAESSRAENGRAEJRAECRAEKRAETRAEPRAEHRYARYAGRYAGGRYAGSRYANRYANJRYANHRYADRYALRYALGRYALMRYALBRYALSRYALTRYALPRYALHRYAMRYABRYABSRYASRYASSRYANGRYA\
    \JRYACRYAKRYATRYAPRYAHRYAERYAEGRYAEGGRYAEGSRYAENRYAENJRYAENHRYAEDRYAELRYAELGRYAELMRYAELBRYAELSRYAELTRYAELPRYAELHRYAEMRYAEBRYAEBSRYAESRYAESSRYAENGRYAEJRYAECRYAEKRYAETRYAEPRYAEHREOREOGREOGGREOGSREONREONJREONHREODREOLREOLGREOLMREOLBREOLSREOLTREOLPREOLHREOMREO\
    \BREOBSREOSREOSSREONGREOJREOCREOKREOTREOPREOHREREGREGGREGSRENRENJRENHREDRELRELGRELMRELBRELSRELTRELPRELHREMREBREBSRESRESSRENGREJRECREKRETREPREHRYEORYEOGRYEOGGRYEOGSRYEONRYEONJRYEONHRYEODRYEOLRYEOLGRYEOLMRYEOLBRYEOLSRYEOLTRYEOLPRYEOLHRYEOMRYEOBRYEOBSRYEOSRYE\
    \OSSRYEONGRYEOJRYEOCRYEOKRYEOTRYEOPRYEOHRYERYEGRYEGGRYEGSRYENRYENJRYENHRYEDRYELRYELGRYELMRYELBRYELSRYELTRYELPRYELHRYEMRYEBRYEBSRYESRYESSRYENGRYEJRYECRYEKRYETRYEPRYEHROROGROGGROGSRONRONJRONHRODROLROLGROLMROLBROLSROLTROLPROLHROMROBROBSROSROSSRONGROJROCROKROT\
    \ROPROHRWARWAGRWAGGRWAGSRWANRWANJRWANHRWADRWALRWALGRWALMRWALBRWALSRWALTRWALPRWALHRWAMRWABRWABSRWASRWASSRWANGRWAJRWACRWAKRWATRWAPRWAHRWAERWAEGRWAEGGRWAEGSRWAENRWAENJRWAENHRWAEDRWAELRWAELGRWAELMRWAELBRWAELSRWAELTRWAELPRWAELHRWAEMRWAEBRWAEBSRWAESRWAESSRWAENGR\
    \WAEJRWAECRWAEKRWAETRWAEPRWAEHROEROEGROEGGROEGSROENROENJROENHROEDROELROELGROELMROELBROELSROELTROELPROELHROEMROEBROEBSROESROESSROENGROEJROECROEKROETROEPROEHRYORYOGRYOGGRYOGSRYONRYONJRYONHRYODRYOLRYOLGRYOLMRYOLBRYOLSRYOLTRYOLPRYOLHRYOMRYOBRYOBSRYOSRYOSSRYONG\
    \RYOJRYOCRYOKRYOTRYOPRYOHRURUGRUGGRUGSRUNRUNJRUNHRUDRULRULGRULMRULBRULSRULTRULPRULHRUMRUBRUBSRUSRUSSRUNGRUJRUCRUKRUTRUPRUHRWEORWEOGRWEOGGRWEOGSRWEONRWEONJRWEONHRWEODRWEOLRWEOLGRWEOLMRWEOLBRWEOLSRWEOLTRWEOLPRWEOLHRWEOMRWEOBRWEOBSRWEOSRWEOSSRWEONGRWEOJRWEOCR\
    \WEOKRWEOTRWEOPRWEOHRWERWEGRWEGGRWEGSRWENRWENJRWENHRWEDRWELRWELGRWELMRWELBRWELSRWELTRWELPRWELHRWEMRWEBRWEBSRWESRWESSRWENGRWEJRWECRWEKRWETRWEPRWEHRWIRWIGRWIGGRWIGSRWINRWINJRWINHRWIDRWILRWILGRWILMRWILBRWILSRWILTRWILPRWILHRWIMRWIBRWIBSRWISRWISSRWINGRWIJRWICRW\
    \IKRWITRWIPRWIHRYURYUGRYUGGRYUGSRYUNRYUNJRYUNHRYUDRYULRYULGRYULMRYULBRYULSRYULTRYULPRYULHRYUMRYUBRYUBSRYUSRYUSSRYUNGRYUJRYUCRYUKRYUTRYUPRYUHREUREUGREUGGREUGSREUNREUNJREUNHREUDREULREULGREULMREULBREULSREULTREULPREULHREUMREUBREUBSREUSREUSSREUNGREUJREUCREUKREU\
    \TREUPREUHRYIRYIGRYIGGRYIGSRYINRYINJRYINHRYIDRYILRYILGRYILMRYILBRYILSRYILTRYILPRYILHRYIMRYIBRYIBSRYISRYISSRYINGRYIJRYICRYIKRYITRYIPRYIHRIRIGRIGGRIGSRINRINJRINHRIDRILRILGRILMRILBRILSRILTRILPRILHRIMRIBRIBSRISRISSRINGRIJRICRIKRITRIPRIHMAMAGMAGGMAGSMANMANJMANH\
    \MADMALMALGMALMMALBMALSMALTMALPMALHMAMMABMABSMASMASSMANGMAJMACMAKMATMAPMAHMAEMAEGMAEGGMAEGSMAENMAENJMAENHMAEDMAELMAELGMAELMMAELBMAELSMAELTMAELPMAELHMAEMMAEBMAEBSMAESMAESSMAENGMAEJMAECMAEKMAETMAEPMAEHMYAMYAGMYAGGMYAGSMYANMYANJMYANHMYADMYALMYALGMYALMMYALBMYA\
    \LSMYALTMYALPMYALHMYAMMYABMYABSMYASMYASSMYANGMYAJMYACMYAKMYATMYAPMYAHMYAEMYAEGMYAEGGMYAEGSMYAENMYAENJMYAENHMYAEDMYAELMYAELGMYAELMMYAELBMYAELSMYAELTMYAELPMYAELHMYAEMMYAEBMYAEBSMYAESMYAESSMYAENGMYAEJMYAECMYAEKMYAETMYAEPMYAEHMEOMEOGMEOGGMEOGSMEONMEONJMEONHMEO\
    \DMEOLMEOLGMEOLMMEOLBMEOLSMEOLTMEOLPMEOLHMEOMMEOBMEOBSMEOSMEOSSMEONGMEOJMEOCMEOKMEOTMEOPMEOHMEMEGMEGGMEGSMENMENJMENHMEDMELMELGMELMMELBMELSMELTMELPMELHMEMMEBMEBSMESMESSMENGMEJMECMEKMETMEPMEHMYEOMYEOGMYEOGGMYEOGSMYEONMYEONJMYEONHMYEODMYEOLMYEOLGMYEOLMMYEOLBM\
    \YEOLSMYEOLTMYEOLPMYEOLHMYEOMMYEOBMYEOBSMYEOSMYEOSSMYEONGMYEOJMYEOCMYEOKMYEOTMYEOPMYEOHMYEMYEGMYEGGMYEGSMYENMYENJMYENHMYEDMYELMYELGMYELMMYELBMYELSMYELTMYELPMYELHMYEMMYEBMYEBSMYESMYESSMYENGMYEJMYECMYEKMYETMYEPMYEHMOMOGMOGGMOGSMONMONJMONHMODMOLMOLGMOLMMOLBMO\
    \LSMOLTMOLPMOLHMOMMOBMOBSMOSMOSSMONGMOJMOCMOKMOTMOPMOHMWAMWAGMWAGGMWAGSMWANMWANJMWANHMWADMWALMWALGMWALMMWALBMWALSMWALTMWALPMWALHMWAMMWABMWABSMWASMWASSMWANGMWAJMWACMWAKMWATMWAPMWAHMWAEMWAEGMWAEGGMWAEGSMWAENMWAENJMWAENHMWAEDMWAELMWAELGMWAELMMWAELBMWAELSMWAEL\
    \TMWAELPMWAELHMWAEMMWAEBMWAEBSMWAESMWAESSMWAENGMWAEJMWAECMWAEKMWAETMWAEPMWAEHMOEMOEGMOEGGMOEGSMOENMOENJMOENHMOEDMOELMOELGMOELMMOELBMOELSMOELTMOELPMOELHMOEMMOEBMOEBSMOESMOESSMOENGMOEJMOECMOEKMOETMOEPMOEHMYOMYOGMYOGGMYOGSMYONMYONJMYONHMYODMYOLMYOLGMYOLMMYOLB\
    \MYOLSMYOLTMYOLPMYOLHMYOMMYOBMYOBSMYOSMYOSSMYONGMYOJMYOCMYOKMYOTMYOPMYOHMUMUGMUGGMUGSMUNMUNJMUNHMUDMULMULGMULMMULBMULSMULTMULPMULHMUMMUBMUBSMUSMUSSMUNGMUJMUCMUKMUTMUPMUHMWEOMWEOGMWEOGGMWEOGSMWEONMWEONJMWEONHMWEODMWEOLMWEOLGMWEOLMMWEOLBMWEOLSMWEOLTMWEOLPMWE\
    \OLHMWEOMMWEOBMWEOBSMWEOSMWEOSSMWEONGMWEOJMWEOCMWEOKMWEOTMWEOPMWEOHMWEMWEGMWEGGMWEGSMWENMWENJMWENHMWEDMWELMWELGMWELMMWELBMWELSMWELTMWELPMWELHMWEMMWEBMWEBSMWESMWESSMWENGMWEJMWECMWEKMWETMWEPMWEHMWIMWIGMWIGGMWIGSMWINMWINJMWINHMWIDMWILMWILGMWILMMWILBMWILSMWILT\
    \MWILPMWILHMWIMMWIBMWIBSMWISMWISSMWINGMWIJMWICMWIKMWITMWIPMWIHMYUMYUGMYUGGMYUGSMYUNMYUNJMYUNHMYUDMYULMYULGMYULMMYULBMYULSMYULTMYULPMYULHMYUMMYUBMYUBSMYUSMYUSSMYUNGMYUJMYUCMYUKMYUTMYUPMYUHMEUMEUGMEUGGMEUGSMEUNMEUNJMEUNHMEUDMEULMEULGMEULMMEULBMEULSMEULTMEULP\
    \MEULHMEUMMEUBMEUBSMEUSMEUSSMEUNGMEUJMEUCMEUKMEUTMEUPMEUHMYIMYIGMYIGGMYIGSMYINMYINJMYINHMYIDMYILMYILGMYILMMYILBMYILSMYILTMYILPMYILHMYIMMYIBMYIBSMYISMYISSMYINGMYIJMYICMYIKMYITMYIPMYIHMIMIGMIGGMIGSMINMINJMINHMIDMILMILGMILMMILBMILSMILTMILPMILHMIMMIBMIBSMISMIS\
    \SMINGMIJMICMIKMITMIPMIHBABAGBAGGBAGSBANBANJBANHBADBALBALGBALMBALBBALSBALTBALPBALHBAMBABBABSBASBASSBANGBAJBACBAKBATBAPBAHBAEBAEGBAEGGBAEGSBAENBAENJBAENHBAEDBAELBAELGBAELMBAELBBAELSBAELTBAELPBAELHBAEMBAEBBAEBSBAESBAESSBAENGBAEJBAECBAEKBAETBAEPBAEHBYABYAGBYA\
    \GGBYAGSBYANBYANJBYANHBYADBYALBYALGBYALMBYALBBYALSBYALTBYALPBYALHBYAMBYABBYABSBYASBYASSBYANGBYAJBYACBYAKBYATBYAPBYAHBYAEBYAEGBYAEGGBYAEGSBYAENBYAENJBYAENHBYAEDBYAELBYAELGBYAELMBYAELBBYAELSBYAELTBYAELPBYAELHBYAEMBYAEBBYAEBSBYAESBYAESSBYAENGBYAEJBYAECBYAEKBY\
    \AETBYAEPBYAEHBEOBEOGBEOGGBEOGSBEONBEONJBEONHBEODBEOLBEOLGBEOLMBEOLBBEOLSBEOLTBEOLPBEOLHBEOMBEOBBEOBSBEOSBEOSSBEONGBEOJBEOCBEOKBEOTBEOPBEOHBEBEGBEGGBEGSBENBENJBENHBEDBELBELGBELMBELBBELSBELTBELPBELHBEMBEBBEBSBESBESSBENGBEJBECBEKBETBEPBEHBYEOBYEOGBYEOGGBYEOG\
    \SBYEONBYEONJBYEONHBYEODBYEOLBYEOLGBYEOLMBYEOLBBYEOLSBYEOLTBYEOLPBYEOLHBYEOMBYEOBBYEOBSBYEOSBYEOSSBYEONGBYEOJBYEOCBYEOKBYEOTBYEOPBYEOHBYEBYEGBYEGGBYEGSBYENBYENJBYENHBYEDBYELBYELGBYELMBYELBBYELSBYELTBYELPBYELHBYEMBYEBBYEBSBYESBYESSBYENGBYEJBYECBYEKBYETBYEPB\
    \YEHBOBOGBOGGBOGSBONBONJBONHBODBOLBOLGBOLMBOLBBOLSBOLTBOLPBOLHBOMBOBBOBSBOSBOSSBONGBOJBOCBOKBOTBOPBOHBWABWAGBWAGGBWAGSBWANBWANJBWANHBWADBWALBWALGBWALMBWALBBWALSBWALTBWALPBWALHBWAMBWABBWABSBWASBWASSBWANGBWAJBWACBWAKBWATBWAPBWAHBWAEBWAEGBWAEGGBWAEGSBWAENBWAE\
    \NJBWAENHBWAEDBWAELBWAELGBWAELMBWAELBBWAELSBWAELTBWAELPBWAELHBWAEMBWAEBBWAEBSBWAESBWAESSBWAENGBWAEJBWAECBWAEKBWAETBWAEPBWAEHBOEBOEGBOEGGBOEGSBOENBOENJBOENHBOEDBOELBOELGBOELMBOELBBOELSBOELTBOELPBOELHBOEMBOEBBOEBSBOESBOESSBOENGBOEJBOECBOEKBOETBOEPBOEHBYOBYOG\
    \BYOGGBYOGSBYONBYONJBYONHBYODBYOLBYOLGBYOLMBYOLBBYOLSBYOLTBYOLPBYOLHBYOMBYOBBYOBSBYOSBYOSSBYONGBYOJBYOCBYOKBYOTBYOPBYOHBUBUGBUGGBUGSBUNBUNJBUNHBUDBULBULGBULMBULBBULSBULTBULPBULHBUMBUBBUBSBUSBUSSBUNGBUJBUCBUKBUTBUPBUHBWEOBWEOGBWEOGGBWEOGSBWEONBWEONJBWEONHBW\
    \EODBWEOLBWEOLGBWEOLMBWEOLBBWEOLSBWEOLTBWEOLPBWEOLHBWEOMBWEOBBWEOBSBWEOSBWEOSSBWEONGBWEOJBWEOCBWEOKBWEOTBWEOPBWEOHBWEBWEGBWEGGBWEGSBWENBWENJBWENHBWEDBWELBWELGBWELMBWELBBWELSBWELTBWELPBWELHBWEMBWEBBWEBSBWESBWESSBWENGBWEJBWECBWEKBWETBWEPBWEHBWIBWIGBWIGGBWIGS\
    \BWINBWINJBWINHBWIDBWILBWILGBWILMBWILBBWILSBWILTBWILPBWILHBWIMBWIBBWIBSBWISBWISSBWINGBWIJBWICBWIKBWITBWIPBWIHBYUBYUGBYUGGBYUGSBYUNBYUNJBYUNHBYUDBYULBYULGBYULMBYULBBYULSBYULTBYULPBYULHBYUMBYUBBYUBSBYUSBYUSSBYUNGBYUJBYUCBYUKBYUTBYUPBYUHBEUBEUGBEUGGBEUGSBEUNB\
    \EUNJBEUNHBEUDBEULBEULGBEULMBEULBBEULSBEULTBEULPBEULHBEUMBEUBBEUBSBEUSBEUSSBEUNGBEUJBEUCBEUKBEUTBEUPBEUHBYIBYIGBYIGGBYIGSBYINBYINJBYINHBYIDBYILBYILGBYILMBYILBBYILSBYILTBYILPBYILHBYIMBYIBBYIBSBYISBYISSBYINGBYIJBYICBYIKBYITBYIPBYIHBIBIGBIGGBIGSBINBINJBINHBID\
    \BILBILGBILMBILBBILSBILTBILPBILHBIMBIBBIBSBISBISSBINGBIJBICBIKBITBIPBIHBBABBAGBBAGGBBAGSBBANBBANJBBANHBBADBBALBBALGBBALMBBALBBBALSBBALTBBALPBBALHBBAMBBABBBABSBBASBBASSBBANGBBAJBBACBBAKBBATBBAPBBAHBBAEBBAEGBBAEGGBBAEGSBBAENBBAENJBBAENHBBAEDBBAELBBAELGBBAELM\
    \BBAELBBBAELSBBAELTBBAELPBBAELHBBAEMBBAEBBBAEBSBBAESBBAESSBBAENGBBAEJBBAECBBAEKBBAETBBAEPBBAEHBBYABBYAGBBYAGGBBYAGSBBYANBBYANJBBYANHBBYADBBYALBBYALGBBYALMBBYALBBBYALSBBYALTBBYALPBBYALHBBYAMBBYABBBYABSBBYASBBYASSBBYANGBBYAJBBYACBBYAKBBYATBBYAPBBYAHBBYAEBBYA\
    \EGBBYAEGGBBYAEGSBBYAENBBYAENJBBYAENHBBYAEDBBYAELBBYAELGBBYAELMBBYAELBBBYAELSBBYAELTBBYAELPBBYAELHBBYAEMBBYAEBBBYAEBSBBYAESBBYAESSBBYAENGBBYAEJBBYAECBBYAEKBBYAETBBYAEPBBYAEHBBEOBBEOGBBEOGGBBEOGSBBEONBBEONJBBEONHBBEODBBEOLBBEOLGBBEOLMBBEOLBBBEOLSBBEOLTBBEOL\
    \PBBEOLHBBEOMBBEOBBBEOBSBBEOSBBEOSSBBEONGBBEOJBBEOCBBEOKBBEOTBBEOPBBEOHBBEBBEGBBEGGBBEGSBBENBBENJBBENHBBEDBBELBBELGBBELMBBELBBBELSBBELTBBELPBBELHBBEMBBEBBBEBSBBESBBESSBBENGBBEJBBECBBEKBBETBBEPBBEHBBYEOBBYEOGBBYEOGGBBYEOGSBBYEONBBYEONJBBYEONHBBYEODBBYEOLBBY\
    \EOLGBBYEOLMBBYEOLBBBYEOLSBBYEOLTBBYEOLPBBYEOLHBBYEOMBBYEOBBBYEOBSBBYEOSBBYEOSSBBYEONGBBYEOJBBYEOCBBYEOKBBYEOTBBYEOPBBYEOHBBYEBBYEGBBYEGGBBYEGSBBYENBBYENJBBYENHBBYEDBBYELBBYELGBBYELMBBYELBBBYELSBBYELTBBYELPBBYELHBBYEMBBYEBBBYEBSBBYESBBYESSBBYENGBBYEJBBYECB\
    \BYEKBBYETBBYEPBBYEHBBOBBOGBBOGGBBOGSBBONBBONJBBONHBBODBBOLBBOLGBBOLMBBOLBBBOLSBBOLTBBOLPBBOLHBBOMBBOBBBOBSBBOSBBOSSBBONGBBOJBBOCBBOKBBOTBBOPBBOHBBWABBWAGBBWAGGBBWAGSBBWANBBWANJBBWANHBBWADBBWALBBWALGBBWALMBBWALBBBWALSBBWALTBBWALPBBWALHBBWAMBBWABBBWABSBBWAS\
    \BBWASSBBWANGBBWAJBBWACBBWAKBBWATBBWAPBBWAHBBWAEBBWAEGBBWAEGGBBWAEGSBBWAENBBWAENJBBWAENHBBWAEDBBWAELBBWAELGBBWAELMBBWAELBBBWAELSBBWAELTBBWAELPBBWAELHBBWAEMBBWAEBBBWAEBSBBWAESBBWAESSBBWAENGBBWAEJBBWAECBBWAEKBBWAETBBWAEPBBWAEHBBOEBBOEGBBOEGGBBOEGSBBOENBBOENJ\
    \BBOENHBBOEDBBOELBBOELGBBOELMBBOELBBBOELSBBOELTBBOELPBBOELHBBOEMBBOEBBBOEBSBBOESBBOESSBBOENGBBOEJBBOECBBOEKBBOETBBOEPBBOEHBBYOBBYOGBBYOGGBBYOGSBBYONBBYONJBBYONHBBYODBBYOLBBYOLGBBYOLMBBYOLBBBYOLSBBYOLTBBYOLPBBYOLHBBYOMBBYOBBBYOBSBBYOSBBYOSSBBYONGBBYOJBBYOCB\
    \BYOKBBYOTBBYOPBBYOHBBUBBUGBBUGGBBUGSBBUNBBUNJBBUNHBBUDBBULBBULGBBULMBBULBBBULSBBULTBBULPBBULHBBUMBBUBBBUBSBBUSBBUSSBBUNGBBUJBBUCBBUKBBUTBBUPBBUHBBWEOBBWEOGBBWEOGGBBWEOGSBBWEONBBWEONJBBWEONHBBWEODBBWEOLBBWEOLGBBWEOLMBBWEOLBBBWEOLSBBWEOLTBBWEOLPBBWEOLHBBWEO\
    \MBBWEOBBBWEOBSBBWEOSBBWEOSSBBWEONGBBWEOJBBWEOCBBWEOKBBWEOTBBWEOPBBWEOHBBWEBBWEGBBWEGGBBWEGSBBWENBBWENJBBWENHBBWEDBBWELBBWELGBBWELMBBWELBBBWELSBBWELTBBWELPBBWELHBBWEMBBWEBBBWEBSBBWESBBWESSBBWENGBBWEJBBWECBBWEKBBWETBBWEPBBWEHBBWIBBWIGBBWIGGBBWIGSBBWINBBWINJ\
    \BBWINHBBWIDBBWILBBWILGBBWILMBBWILBBBWILSBBWILTBBWILPBBWILHBBWIMBBWIBBBWIBSBBWISBBWISSBBWINGBBWIJBBWICBBWIKBBWITBBWIPBBWIHBBYUBBYUGBBYUGGBBYUGSBBYUNBBYUNJBBYUNHBBYUDBBYULBBYULGBBYULMBBYULBBBYULSBBYULTBBYULPBBYULHBBYUMBBYUBBBYUBSBBYUSBBYUSSBBYUNGBBYUJBBYUCB\
    \BYUKBBYUTBBYUPBBYUHBBEUBBEUGBBEUGGBBEUGSBBEUNBBEUNJBBEUNHBBEUDBBEULBBEULGBBEULMBBEULBBBEULSBBEULTBBEULPBBEULHBBEUMBBEUBBBEUBSBBEUSBBEUSSBBEUNGBBEUJBBEUCBBEUKBBEUTBBEUPBBEUHBBYIBBYIGBBYIGGBBYIGSBBYINBBYINJBBYINHBBYIDBBYILBBYILGBBYILMBBYILBBBYILSBBYILTBBYIL\
    \PBBYILHBBYIMBBYIBBBYIBSBBYISBBYISSBBYINGBBYIJBBYICBBYIKBBYITBBYIPBBYIHBBIBBIGBBIGGBBIGSBBINBBINJBBINHBBIDBBILBBILGBBILMBBILBBBILSBBILTBBILPBBILHBBIMBBIBBBIBSBBISBBISSBBINGBBIJBBICBBIKBBITBBIPBBIHSASAGSAGGSAGSSANSANJSANHSADSALSALGSALMSALBSALSSALTSALPSALHSA\
    \MSABSABSSASSASSSANGSAJSACSAKSATSAPSAHSAESAEGSAEGGSAEGSSAENSAENJSAENHSAEDSAELSAELGSAELMSAELBSAELSSAELTSAELPSAELHSAEMSAEBSAEBSSAESSAESSSAENGSAEJSAECSAEKSAETSAEPSAEHSYASYAGSYAGGSYAGSSYANSYANJSYANHSYADSYALSYALGSYALMSYALBSYALSSYALTSYALPSYALHSYAMSYABSYABSSYASSY\
    \ASSSYANGSYAJSYACSYAKSYATSYAPSYAHSYAESYAEGSYAEGGSYAEGSSYAENSYAENJSYAENHSYAEDSYAELSYAELGSYAELMSYAELBSYAELSSYAELTSYAELPSYAELHSYAEMSYAEBSYAEBSSYAESSYAESSSYAENGSYAEJSYAECSYAEKSYAETSYAEPSYAEHSEOSEOGSEOGGSEOGSSEONSEONJSEONHSEODSEOLSEOLGSEOLMSEOLBSEOLSSEOLTSEOLPS\
    \EOLHSEOMSEOBSEOBSSEOSSEOSSSEONGSEOJSEOCSEOKSEOTSEOPSEOHSESEGSEGGSEGSSENSENJSENHSEDSELSELGSELMSELBSELSSELTSELPSELHSEMSEBSEBSSESSESSSENGSEJSECSEKSETSEPSEHSYEOSYEOGSYEOGGSYEOGSSYEONSYEONJSYEONHSYEODSYEOLSYEOLGSYEOLMSYEOLBSYEOLSSYEOLTSYEOLPSYEOLHSYEOMSYEOBSYE\
    \OBSSYEOSSYEOSSSYEONGSYEOJSYEOCSYEOKSYEOTSYEOPSYEOHSYESYEGSYEGGSYEGSSYENSYENJSYENHSYEDSYELSYELGSYELMSYELBSYELSSYELTSYELPSYELHSYEMSYEBSYEBSSYESSYESSSYENGSYEJSYECSYEKSYETSYEPSYEHSOSOGSOGGSOGSSONSONJSONHSODSOLSOLGSOLMSOLBSOLSSOLTSOLPSOLHSOMSOBSOBSSOSSOSSSONGS\
    \OJSOCSOKSOTSOPSOHSWASWAGSWAGGSWAGSSWANSWANJSWANHSWADSWALSWALGSWALMSWALBSWALSSWALTSWALPSWALHSWAMSWABSWABSSWASSWASSSWANGSWAJSWACSWAKSWATSWAPSWAHSWAESWAEGSWAEGGSWAEGSSWAENSWAENJSWAENHSWAEDSWAELSWAELGSWAELMSWAELBSWAELSSWAELTSWAELPSWAELHSWAEMSWAEBSWAEBSSWAESSW\
    \AESSSWAENGSWAEJSWAECSWAEKSWAETSWAEPSWAEHSOESOEGSOEGGSOEGSSOENSOENJSOENHSOEDSOELSOELGSOELMSOELBSOELSSOELTSOELPSOELHSOEMSOEBSOEBSSOESSOESSSOENGSOEJSOECSOEKSOETSOEPSOEHSYOSYOGSYOGGSYOGSSYONSYONJSYONHSYODSYOLSYOLGSYOLMSYOLBSYOLSSYOLTSYOLPSYOLHSYOMSYOBSYOBSSYO\
    \SSYOSSSYONGSYOJSYOCSYOKSYOTSYOPSYOHSUSUGSUGGSUGSSUNSUNJSUNHSUDSULSULGSULMSULBSULSSULTSULPSULHSUMSUBSUBSSUSSUSSSUNGSUJSUCSUKSUTSUPSUHSWEOSWEOGSWEOGGSWEOGSSWEONSWEONJSWEONHSWEODSWEOLSWEOLGSWEOLMSWEOLBSWEOLSSWEOLTSWEOLPSWEOLHSWEOMSWEOBSWEOBSSWEOSSWEOSSSWEONG\
    \SWEOJSWEOCSWEOKSWEOTSWEOPSWEOHSWESWEGSWEGGSWEGSSWENSWENJSWENHSWEDSWELSWELGSWELMSWELBSWELSSWELTSWELPSWELHSWEMSWEBSWEBSSWESSWESSSWENGSWEJSWECSWEKSWETSWEPSWEHSWISWIGSWIGGSWIGSSWINSWINJSWINHSWIDSWILSWILGSWILMSWILBSWILSSWILTSWILPSWILHSWIMSWIBSWIBSSWISSWISSSWIN\
    \GSWIJSWICSWIKSWITSWIPSWIHSYUSYUGSYUGGSYUGSSYUNSYUNJSYUNHSYUDSYULSYULGSYULMSYULBSYULSSYULTSYULPSYULHSYUMSYUBSYUBSSYUSSYUSSSYUNGSYUJSYUCSYUKSYUTSYUPSYUHSEUSEUGSEUGGSEUGSSEUNSEUNJSEUNHSEUDSEULSEULGSEULMSEULBSEULSSEULTSEULPSEULHSEUMSEUBSEUBSSEUSSEUSSSEUNGSEUJ\
    \SEUCSEUKSEUTSEUPSEUHSYISYIGSYIGGSYIGSSYINSYINJSYINHSYIDSYILSYILGSYILMSYILBSYILSSYILTSYILPSYILHSYIMSYIBSYIBSSYISSYISSSYINGSYIJSYICSYIKSYITSYIPSYIHSISIGSIGGSIGSSINSINJSINHSIDSILSILGSILMSILBSILSSILTSILPSILHSIMSIBSIBSSISSISSSINGSIJSICSIKSITSIPSIHSSASSAGSSAGGS\
    \SAGSSSANSSANJSSANHSSADSSALSSALGSSALMSSALBSSALSSSALTSSALPSSALHSSAMSSABSSABSSSASSSASSSSANGSSAJSSACSSAKSSATSSAPSSAHSSAESSAEGSSAEGGSSAEGSSSAENSSAENJSSAENHSSAEDSSAELSSAELGSSAELMSSAELBSSAELSSSAELTSSAELPSSAELHSSAEMSSAEBSSAEBSSSAESSSAESSSSAENGSSAEJSSAECSSAEKSSAET\
    \SSAEPSSAEHSSYASSYAGSSYAGGSSYAGSSSYANSSYANJSSYANHSSYADSSYALSSYALGSSYALMSSYALBSSYALSSSYALTSSYALPSSYALHSSYAMSSYABSSYABSSSYASSSYASSSSYANGSSYAJSSYACSSYAKSSYATSSYAPSSYAHSSYAESSYAEGSSYAEGGSSYAEGSSSYAENSSYAENJSSYAENHSSYAEDSSYAELSSYAELGSSYAELMSSYAELBSSYAELSSSYAELT\
    \SSYAELPSSYAELHSSYAEMSSYAEBSSYAEBSSSYAESSSYAESSSSYAENGSSYAEJSSYAECSSYAEKSSYAETSSYAEPSSYAEHSSEOSSEOGSSEOGGSSEOGSSSEONSSEONJSSEONHSSEODSSEOLSSEOLGSSEOLMSSEOLBSSEOLSSSEOLTSSEOLPSSEOLHSSEOMSSEOBSSEOBSSSEOSSSEOSSSSEONGSSEOJSSEOCSSEOKSSEOTSSEOPSSEOHSSESSEGSSEGGS\
    \SEGSSSENSSENJSSENHSSEDSSELSSELGSSELMSSELBSSELSSSELTSSELPSSELHSSEMSSEBSSEBSSSESSSESSSSENGSSEJSSECSSEKSSETSSEPSSEHSSYEOSSYEOGSSYEOGGSSYEOGSSSYEONSSYEONJSSYEONHSSYEODSSYEOLSSYEOLGSSYEOLMSSYEOLBSSYEOLSSSYEOLTSSYEOLPSSYEOLHSSYEOMSSYEOBSSYEOBSSSYEOSSSYEOSSSSYEO\
    \NGSSYEOJSSYEOCSSYEOKSSYEOTSSYEOPSSYEOHSSYESSYEGSSYEGGSSYEGSSSYENSSYENJSSYENHSSYEDSSYELSSYELGSSYELMSSYELBSSYELSSSYELTSSYELPSSYELHSSYEMSSYEBSSYEBSSSYESSSYESSSSYENGSSYEJSSYECSSYEKSSYETSSYEPSSYEHSSOSSOGSSOGGSSOGSSSONSSONJSSONHSSODSSOLSSOLGSSOLMSSOLBSSOLSSSOLT\
    \SSOLPSSOLHSSOMSSOBSSOBSSSOSSSOSSSSONGSSOJSSOCSSOKSSOTSSOPSSOHSSWASSWAGSSWAGGSSWAGSSSWANSSWANJSSWANHSSWADSSWALSSWALGSSWALMSSWALBSSWALSSSWALTSSWALPSSWALHSSWAMSSWABSSWABSSSWASSSWASSSSWANGSSWAJSSWACSSWAKSSWATSSWAPSSWAHSSWAESSWAEGSSWAEGGSSWAEGSSSWAENSSWAENJSSW\
    \AENHSSWAEDSSWAELSSWAELGSSWAELMSSWAELBSSWAELSSSWAELTSSWAELPSSWAELHSSWAEMSSWAEBSSWAEBSSSWAESSSWAESSSSWAENGSSWAEJSSWAECSSWAEKSSWAETSSWAEPSSWAEHSSOESSOEGSSOEGGSSOEGSSSOENSSOENJSSOENHSSOEDSSOELSSOELGSSOELMSSOELBSSOELSSSOELTSSOELPSSOELHSSOEMSSOEBSSOEBSSSOESSSOE\
    \SSSSOENGSSOEJSSOECSSOEKSSOETSSOEPSSOEHSSYOSSYOGSSYOGGSSYOGSSSYONSSYONJSSYONHSSYODSSYOLSSYOLGSSYOLMSSYOLBSSYOLSSSYOLTSSYOLPSSYOLHSSYOMSSYOBSSYOBSSSYOSSSYOSSSSYONGSSYOJSSYOCSSYOKSSYOTSSYOPSSYOHSSUSSUGSSUGGSSUGSSSUNSSUNJSSUNHSSUDSSULSSULGSSULMSSULBSSULSSSULT\
    \SSULPSSULHSSUMSSUBSSUBSSSUSSSUSSSSUNGSSUJSSUCSSUKSSUTSSUPSSUHSSWEOSSWEOGSSWEOGGSSWEOGSSSWEONSSWEONJSSWEONHSSWEODSSWEOLSSWEOLGSSWEOLMSSWEOLBSSWEOLSSSWEOLTSSWEOLPSSWEOLHSSWEOMSSWEOBSSWEOBSSSWEOSSSWEOSSSSWEONGSSWEOJSSWEOCSSWEOKSSWEOTSSWEOPSSWEOHSSWESSWEGSSWE\
    \GGSSWEGSSSWENSSWENJSSWENHSSWEDSSWELSSWELGSSWELMSSWELBSSWELSSSWELTSSWELPSSWELHSSWEMSSWEBSSWEBSSSWESSSWESSSSWENGSSWEJSSWECSSWEKSSWETSSWEPSSWEHSSWISSWIGSSWIGGSSWIGSSSWINSSWINJSSWINHSSWIDSSWILSSWILGSSWILMSSWILBSSWILSSSWILTSSWILPSSWILHSSWIMSSWIBSSWIBSSSWISSSWI\
    \SSSSWINGSSWIJSSWICSSWIKSSWITSSWIPSSWIHSSYUSSYUGSSYUGGSSYUGSSSYUNSSYUNJSSYUNHSSYUDSSYULSSYULGSSYULMSSYULBSSYULSSSYULTSSYULPSSYULHSSYUMSSYUBSSYUBSSSYUSSSYUSSSSYUNGSSYUJSSYUCSSYUKSSYUTSSYUPSSYUHSSEUSSEUGSSEUGGSSEUGSSSEUNSSEUNJSSEUNHSSEUDSSEULSSEULGSSEULMSSEU\
    \LBSSEULSSSEULTSSEULPSSEULHSSEUMSSEUBSSEUBSSSEUSSSEUSSSSEUNGSSEUJSSEUCSSEUKSSEUTSSEUPSSEUHSSYISSYIGSSYIGGSSYIGSSSYINSSYINJSSYINHSSYIDSSYILSSYILGSSYILMSSYILBSSYILSSSYILTSSYILPSSYILHSSYIMSSYIBSSYIBSSSYISSSYISSSSYINGSSYIJSSYICSSYIKSSYITSSYIPSSYIHSSISSIGSSIGGS\
    \SIGSSSINSSINJSSINHSSIDSSILSSILGSSILMSSILBSSILSSSILTSSILPSSILHSSIMSSIBSSIBSSSISSSISSSSINGSSIJSSICSSIKSSITSSIPSSIHAAGAGGAGSANANJANHADALALGALMALBALSALTALPALHAMABABSASASSANGAJACAKATAPAHAEAEGAEGGAEGSAENAENJAENHAEDAELAELGAELMAELBAELSAELTAELPAELHAEMAEBAEBSAESAES\
    \SAENGAEJAECAEKAETAEPAEHYAYAGYAGGYAGSYANYANJYANHYADYALYALGYALMYALBYALSYALTYALPYALHYAMYABYABSYASYASSYANGYAJYACYAKYATYAPYAHYAEYAEGYAEGGYAEGSYAENYAENJYAENHYAEDYAELYAELGYAELMYAELBYAELSYAELTYAELPYAELHYAEMYAEBYAEBSYAESYAESSYAENGYAEJYAECYAEKYAETYAEPYAEHEOEOGEOGGE\
    \OGSEONEONJEONHEODEOLEOLGEOLMEOLBEOLSEOLTEOLPEOLHEOMEOBEOBSEOSEOSSEONGEOJEOCEOKEOTEOPEOHEEGEGGEGSENENJENHEDELELGELMELBELSELTELPELHEMEBEBSESESSENGEJECEKETEPEHYEOYEOGYEOGGYEOGSYEONYEONJYEONHYEODYEOLYEOLGYEOLMYEOLBYEOLSYEOLTYEOLPYEOLHYEOMYEOBYEOBSYEOSYEOSSYEO\
    \NGYEOJYEOCYEOKYEOTYEOPYEOHYEYEGYEGGYEGSYENYENJYENHYEDYELYELGYELMYELBYELSYELTYELPYELHYEMYEBYEBSYESYESSYENGYEJYECYEKYETYEPYEHOOGOGGOGSONONJONHODOLOLGOLMOLBOLSOLTOLPOLHOMOBOBSOSOSSONGOJOCOKOTOPOHWAWAGWAGGWAGSWANWANJWANHWADWALWALGWALMWALBWALSWALTWALPWALHWAMWA\
    \BWABSWASWASSWANGWAJWACWAKWATWAPWAHWAEWAEGWAEGGWAEGSWAENWAENJWAENHWAEDWAELWAELGWAELMWAELBWAELSWAELTWAELPWAELHWAEMWAEBWAEBSWAESWAESSWAENGWAEJWAECWAEKWAETWAEPWAEHOEOEGOEGGOEGSOENOENJOENHOEDOELOELGOELMOELBOELSOELTOELPOELHOEMOEBOEBSOESOESSOENGOEJOECOEKOETOEPOE\
    \HYOYOGYOGGYOGSYONYONJYONHYODYOLYOLGYOLMYOLBYOLSYOLTYOLPYOLHYOMYOBYOBSYOSYOSSYONGYOJYOCYOKYOTYOPYOHUUGUGGUGSUNUNJUNHUDULULGULMULBULSULTULPULHUMUBUBSUSUSSUNGUJUCUKUTUPUHWEOWEOGWEOGGWEOGSWEONWEONJWEONHWEODWEOLWEOLGWEOLMWEOLBWEOLSWEOLTWEOLPWEOLHWEOMWEOBWEOBSW\
    \EOSWEOSSWEONGWEOJWEOCWEOKWEOTWEOPWEOHWEWEGWEGGWEGSWENWENJWENHWEDWELWELGWELMWELBWELSWELTWELPWELHWEMWEBWEBSWESWESSWENGWEJWECWEKWETWEPWEHWIWIGWIGGWIGSWINWINJWINHWIDWILWILGWILMWILBWILSWILTWILPWILHWIMWIBWIBSWISWISSWINGWIJWICWIKWITWIPWIHYUYUGYUGGYUGSYUNYUNJYUNH\
    \YUDYULYULGYULMYULBYULSYULTYULPYULHYUMYUBYUBSYUSYUSSYUNGYUJYUCYUKYUTYUPYUHEUEUGEUGGEUGSEUNEUNJEUNHEUDEULEULGEULMEULBEULSEULTEULPEULHEUMEUBEUBSEUSEUSSEUNGEUJEUCEUKEUTEUPEUHYIYIGYIGGYIGSYINYINJYINHYIDYILYILGYILMYILBYILSYILTYILPYILHYIMYIBYIBSYISYISSYINGYIJYIC\
    \YIKYITYIPYIHIIGIGGIGSININJINHIDILILGILMILBILSILTILPILHIMIBIBSISISSINGIJICIKITIPIHJAJAGJAGGJAGSJANJANJJANHJADJALJALGJALMJALBJALSJALTJALPJALHJAMJABJABSJASJASSJANGJAJJACJAKJATJAPJAHJAEJAEGJAEGGJAEGSJAENJAENJJAENHJAEDJAELJAELGJAELMJAELBJAELSJAELTJAELPJAELHJAE\
    \MJAEBJAEBSJAESJAESSJAENGJAEJJAECJAEKJAETJAEPJAEHJYAJYAGJYAGGJYAGSJYANJYANJJYANHJYADJYALJYALGJYALMJYALBJYALSJYALTJYALPJYALHJYAMJYABJYABSJYASJYASSJYANGJYAJJYACJYAKJYATJYAPJYAHJYAEJYAEGJYAEGGJYAEGSJYAENJYAENJJYAENHJYAEDJYAELJYAELGJYAELMJYAELBJYAELSJYAELTJYAE\
    \LPJYAELHJYAEMJYAEBJYAEBSJYAESJYAESSJYAENGJYAEJJYAECJYAEKJYAETJYAEPJYAEHJEOJEOGJEOGGJEOGSJEONJEONJJEONHJEODJEOLJEOLGJEOLMJEOLBJEOLSJEOLTJEOLPJEOLHJEOMJEOBJEOBSJEOSJEOSSJEONGJEOJJEOCJEOKJEOTJEOPJEOHJEJEGJEGGJEGSJENJENJJENHJEDJELJELGJELMJELBJELSJELTJELPJELHJ\
    \EMJEBJEBSJESJESSJENGJEJJECJEKJETJEPJEHJYEOJYEOGJYEOGGJYEOGSJYEONJYEONJJYEONHJYEODJYEOLJYEOLGJYEOLMJYEOLBJYEOLSJYEOLTJYEOLPJYEOLHJYEOMJYEOBJYEOBSJYEOSJYEOSSJYEONGJYEOJJYEOCJYEOKJYEOTJYEOPJYEOHJYEJYEGJYEGGJYEGSJYENJYENJJYENHJYEDJYELJYELGJYELMJYELBJYELSJYELT\
    \JYELPJYELHJYEMJYEBJYEBSJYESJYESSJYENGJYEJJYECJYEKJYETJYEPJYEHJOJOGJOGGJOGSJONJONJJONHJODJOLJOLGJOLMJOLBJOLSJOLTJOLPJOLHJOMJOBJOBSJOSJOSSJONGJOJJOCJOKJOTJOPJOHJWAJWAGJWAGGJWAGSJWANJWANJJWANHJWADJWALJWALGJWALMJWALBJWALSJWALTJWALPJWALHJWAMJWABJWABSJWASJWASSJ\
    \WANGJWAJJWACJWAKJWATJWAPJWAHJWAEJWAEGJWAEGGJWAEGSJWAENJWAENJJWAENHJWAEDJWAELJWAELGJWAELMJWAELBJWAELSJWAELTJWAELPJWAELHJWAEMJWAEBJWAEBSJWAESJWAESSJWAENGJWAEJJWAECJWAEKJWAETJWAEPJWAEHJOEJOEGJOEGGJOEGSJOENJOENJJOENHJOEDJOELJOELGJOELMJOELBJOELSJOELTJOELPJOELH\
    \JOEMJOEBJOEBSJOESJOESSJOENGJOEJJOECJOEKJOETJOEPJOEHJYOJYOGJYOGGJYOGSJYONJYONJJYONHJYODJYOLJYOLGJYOLMJYOLBJYOLSJYOLTJYOLPJYOLHJYOMJYOBJYOBSJYOSJYOSSJYONGJYOJJYOCJYOKJYOTJYOPJYOHJUJUGJUGGJUGSJUNJUNJJUNHJUDJULJULGJULMJULBJULSJULTJULPJULHJUMJUBJUBSJUSJUSSJUNG\
    \JUJJUCJUKJUTJUPJUHJWEOJWEOGJWEOGGJWEOGSJWEONJWEONJJWEONHJWEODJWEOLJWEOLGJWEOLMJWEOLBJWEOLSJWEOLTJWEOLPJWEOLHJWEOMJWEOBJWEOBSJWEOSJWEOSSJWEONGJWEOJJWEOCJWEOKJWEOTJWEOPJWEOHJWEJWEGJWEGGJWEGSJWENJWENJJWENHJWEDJWELJWELGJWELMJWELBJWELSJWELTJWELPJWELHJWEMJWEBJW\
    \EBSJWESJWESSJWENGJWEJJWECJWEKJWETJWEPJWEHJWIJWIGJWIGGJWIGSJWINJWINJJWINHJWIDJWILJWILGJWILMJWILBJWILSJWILTJWILPJWILHJWIMJWIBJWIBSJWISJWISSJWINGJWIJJWICJWIKJWITJWIPJWIHJYUJYUGJYUGGJYUGSJYUNJYUNJJYUNHJYUDJYULJYULGJYULMJYULBJYULSJYULTJYULPJYULHJYUMJYUBJYUBSJY\
    \USJYUSSJYUNGJYUJJYUCJYUKJYUTJYUPJYUHJEUJEUGJEUGGJEUGSJEUNJEUNJJEUNHJEUDJEULJEULGJEULMJEULBJEULSJEULTJEULPJEULHJEUMJEUBJEUBSJEUSJEUSSJEUNGJEUJJEUCJEUKJEUTJEUPJEUHJYIJYIGJYIGGJYIGSJYINJYINJJYINHJYIDJYILJYILGJYILMJYILBJYILSJYILTJYILPJYILHJYIMJYIBJYIBSJYISJYI\
    \SSJYINGJYIJJYICJYIKJYITJYIPJYIHJIJIGJIGGJIGSJINJINJJINHJIDJILJILGJILMJILBJILSJILTJILPJILHJIMJIBJIBSJISJISSJINGJIJJICJIKJITJIPJIHJJAJJAGJJAGGJJAGSJJANJJANJJJANHJJADJJALJJALGJJALMJJALBJJALSJJALTJJALPJJALHJJAMJJABJJABSJJASJJASSJJANGJJAJJJACJJAKJJATJJAPJJAHJJ\
    \AEJJAEGJJAEGGJJAEGSJJAENJJAENJJJAENHJJAEDJJAELJJAELGJJAELMJJAELBJJAELSJJAELTJJAELPJJAELHJJAEMJJAEBJJAEBSJJAESJJAESSJJAENGJJAEJJJAECJJAEKJJAETJJAEPJJAEHJJYAJJYAGJJYAGGJJYAGSJJYANJJYANJJJYANHJJYADJJYALJJYALGJJYALMJJYALBJJYALSJJYALTJJYALPJJYALHJJYAMJJYABJJYA\
    \BSJJYASJJYASSJJYANGJJYAJJJYACJJYAKJJYATJJYAPJJYAHJJYAEJJYAEGJJYAEGGJJYAEGSJJYAENJJYAENJJJYAENHJJYAEDJJYAELJJYAELGJJYAELMJJYAELBJJYAELSJJYAELTJJYAELPJJYAELHJJYAEMJJYAEBJJYAEBSJJYAESJJYAESSJJYAENGJJYAEJJJYAECJJYAEKJJYAETJJYAEPJJYAEHJJEOJJEOGJJEOGGJJEOGSJJEO\
    \NJJEONJJJEONHJJEODJJEOLJJEOLGJJEOLMJJEOLBJJEOLSJJEOLTJJEOLPJJEOLHJJEOMJJEOBJJEOBSJJEOSJJEOSSJJEONGJJEOJJJEOCJJEOKJJEOTJJEOPJJEOHJJEJJEGJJEGGJJEGSJJENJJENJJJENHJJEDJJELJJELGJJELMJJELBJJELSJJELTJJELPJJELHJJEMJJEBJJEBSJJESJJESSJJENGJJEJJJECJJEKJJETJJEPJJEHJJ\
    \YEOJJYEOGJJYEOGGJJYEOGSJJYEONJJYEONJJJYEONHJJYEODJJYEOLJJYEOLGJJYEOLMJJYEOLBJJYEOLSJJYEOLTJJYEOLPJJYEOLHJJYEOMJJYEOBJJYEOBSJJYEOSJJYEOSSJJYEONGJJYEOJJJYEOCJJYEOKJJYEOTJJYEOPJJYEOHJJYEJJYEGJJYEGGJJYEGSJJYENJJYENJJJYENHJJYEDJJYELJJYELGJJYELMJJYELBJJYELSJJYE\
    \LTJJYELPJJYELHJJYEMJJYEBJJYEBSJJYESJJYESSJJYENGJJYEJJJYECJJYEKJJYETJJYEPJJYEHJJOJJOGJJOGGJJOGSJJONJJONJJJONHJJODJJOLJJOLGJJOLMJJOLBJJOLSJJOLTJJOLPJJOLHJJOMJJOBJJOBSJJOSJJOSSJJONGJJOJJJOCJJOKJJOTJJOPJJOHJJWAJJWAGJJWAGGJJWAGSJJWANJJWANJJJWANHJJWADJJWALJJWAL\
    \GJJWALMJJWALBJJWALSJJWALTJJWALPJJWALHJJWAMJJWABJJWABSJJWASJJWASSJJWANGJJWAJJJWACJJWAKJJWATJJWAPJJWAHJJWAEJJWAEGJJWAEGGJJWAEGSJJWAENJJWAENJJJWAENHJJWAEDJJWAELJJWAELGJJWAELMJJWAELBJJWAELSJJWAELTJJWAELPJJWAELHJJWAEMJJWAEBJJWAEBSJJWAESJJWAESSJJWAENGJJWAEJJJWA\
    \ECJJWAEKJJWAETJJWAEPJJWAEHJJOEJJOEGJJOEGGJJOEGSJJOENJJOENJJJOENHJJOEDJJOELJJOELGJJOELMJJOELBJJOELSJJOELTJJOELPJJOELHJJOEMJJOEBJJOEBSJJOESJJOESSJJOENGJJOEJJJOECJJOEKJJOETJJOEPJJOEHJJYOJJYOGJJYOGGJJYOGSJJYONJJYONJJJYONHJJYODJJYOLJJYOLGJJYOLMJJYOLBJJYOLSJJYO\
    \LTJJYOLPJJYOLHJJYOMJJYOBJJYOBSJJYOSJJYOSSJJYONGJJYOJJJYOCJJYOKJJYOTJJYOPJJYOHJJUJJUGJJUGGJJUGSJJUNJJUNJJJUNHJJUDJJULJJULGJJULMJJULBJJULSJJULTJJULPJJULHJJUMJJUBJJUBSJJUSJJUSSJJUNGJJUJJJUCJJUKJJUTJJUPJJUHJJWEOJJWEOGJJWEOGGJJWEOGSJJWEONJJWEONJJJWEONHJJWEODJJ\
    \WEOLJJWEOLGJJWEOLMJJWEOLBJJWEOLSJJWEOLTJJWEOLPJJWEOLHJJWEOMJJWEOBJJWEOBSJJWEOSJJWEOSSJJWEONGJJWEOJJJWEOCJJWEOKJJWEOTJJWEOPJJWEOHJJWEJJWEGJJWEGGJJWEGSJJWENJJWENJJJWENHJJWEDJJWELJJWELGJJWELMJJWELBJJWELSJJWELTJJWELPJJWELHJJWEMJJWEBJJWEBSJJWESJJWESSJJWENGJJWE\
    \JJJWECJJWEKJJWETJJWEPJJWEHJJWIJJWIGJJWIGGJJWIGSJJWINJJWINJJJWINHJJWIDJJWILJJWILGJJWILMJJWILBJJWILSJJWILTJJWILPJJWILHJJWIMJJWIBJJWIBSJJWISJJWISSJJWINGJJWIJJJWICJJWIKJJWITJJWIPJJWIHJJYUJJYUGJJYUGGJJYUGSJJYUNJJYUNJJJYUNHJJYUDJJYULJJYULGJJYULMJJYULBJJYULSJJYU\
    \LTJJYULPJJYULHJJYUMJJYUBJJYUBSJJYUSJJYUSSJJYUNGJJYUJJJYUCJJYUKJJYUTJJYUPJJYUHJJEUJJEUGJJEUGGJJEUGSJJEUNJJEUNJJJEUNHJJEUDJJEULJJEULGJJEULMJJEULBJJEULSJJEULTJJEULPJJEULHJJEUMJJEUBJJEUBSJJEUSJJEUSSJJEUNGJJEUJJJEUCJJEUKJJEUTJJEUPJJEUHJJYIJJYIGJJYIGGJJYIGSJJYI\
    \NJJYINJJJYINHJJYIDJJYILJJYILGJJYILMJJYILBJJYILSJJYILTJJYILPJJYILHJJYIMJJYIBJJYIBSJJYISJJYISSJJYINGJJYIJJJYICJJYIKJJYITJJYIPJJYIHJJIJJIGJJIGGJJIGSJJINJJINJJJINHJJIDJJILJJILGJJILMJJILBJJILSJJILTJJILPJJILHJJIMJJIBJJIBSJJISJJISSJJINGJJIJJJICJJIKJJITJJIPJJIHCA\
    \CAGCAGGCAGSCANCANJCANHCADCALCALGCALMCALBCALSCALTCALPCALHCAMCABCABSCASCASSCANGCAJCACCAKCATCAPCAHCAECAEGCAEGGCAEGSCAENCAENJCAENHCAEDCAELCAELGCAELMCAELBCAELSCAELTCAELPCAELHCAEMCAEBCAEBSCAESCAESSCAENGCAEJCAECCAEKCAETCAEPCAEHCYACYAGCYAGGCYAGSCYANCYANJCYANHCYAD\
    \CYALCYALGCYALMCYALBCYALSCYALTCYALPCYALHCYAMCYABCYABSCYASCYASSCYANGCYAJCYACCYAKCYATCYAPCYAHCYAECYAEGCYAEGGCYAEGSCYAENCYAENJCYAENHCYAEDCYAELCYAELGCYAELMCYAELBCYAELSCYAELTCYAELPCYAELHCYAEMCYAEBCYAEBSCYAESCYAESSCYAENGCYAEJCYAECCYAEKCYAETCYAEPCYAEHCEOCEOGCEOGG\
    \CEOGSCEONCEONJCEONHCEODCEOLCEOLGCEOLMCEOLBCEOLSCEOLTCEOLPCEOLHCEOMCEOBCEOBSCEOSCEOSSCEONGCEOJCEOCCEOKCEOTCEOPCEOHCECEGCEGGCEGSCENCENJCENHCEDCELCELGCELMCELBCELSCELTCELPCELHCEMCEBCEBSCESCESSCENGCEJCECCEKCETCEPCEHCYEOCYEOGCYEOGGCYEOGSCYEONCYEONJCYEONHCYEODCY\
    \EOLCYEOLGCYEOLMCYEOLBCYEOLSCYEOLTCYEOLPCYEOLHCYEOMCYEOBCYEOBSCYEOSCYEOSSCYEONGCYEOJCYEOCCYEOKCYEOTCYEOPCYEOHCYECYEGCYEGGCYEGSCYENCYENJCYENHCYEDCYELCYELGCYELMCYELBCYELSCYELTCYELPCYELHCYEMCYEBCYEBSCYESCYESSCYENGCYEJCYECCYEKCYETCYEPCYEHCOCOGCOGGCOGSCONCONJCO\
    \NHCODCOLCOLGCOLMCOLBCOLSCOLTCOLPCOLHCOMCOBCOBSCOSCOSSCONGCOJCOCCOKCOTCOPCOHCWACWAGCWAGGCWAGSCWANCWANJCWANHCWADCWALCWALGCWALMCWALBCWALSCWALTCWALPCWALHCWAMCWABCWABSCWASCWASSCWANGCWAJCWACCWAKCWATCWAPCWAHCWAECWAEGCWAEGGCWAEGSCWAENCWAENJCWAENHCWAEDCWAELCWAELGC\
    \WAELMCWAELBCWAELSCWAELTCWAELPCWAELHCWAEMCWAEBCWAEBSCWAESCWAESSCWAENGCWAEJCWAECCWAEKCWAETCWAEPCWAEHCOECOEGCOEGGCOEGSCOENCOENJCOENHCOEDCOELCOELGCOELMCOELBCOELSCOELTCOELPCOELHCOEMCOEBCOEBSCOESCOESSCOENGCOEJCOECCOEKCOETCOEPCOEHCYOCYOGCYOGGCYOGSCYONCYONJCYONHC\
    \YODCYOLCYOLGCYOLMCYOLBCYOLSCYOLTCYOLPCYOLHCYOMCYOBCYOBSCYOSCYOSSCYONGCYOJCYOCCYOKCYOTCYOPCYOHCUCUGCUGGCUGSCUNCUNJCUNHCUDCULCULGCULMCULBCULSCULTCULPCULHCUMCUBCUBSCUSCUSSCUNGCUJCUCCUKCUTCUPCUHCWEOCWEOGCWEOGGCWEOGSCWEONCWEONJCWEONHCWEODCWEOLCWEOLGCWEOLMCWEOL\
    \BCWEOLSCWEOLTCWEOLPCWEOLHCWEOMCWEOBCWEOBSCWEOSCWEOSSCWEONGCWEOJCWEOCCWEOKCWEOTCWEOPCWEOHCWECWEGCWEGGCWEGSCWENCWENJCWENHCWEDCWELCWELGCWELMCWELBCWELSCWELTCWELPCWELHCWEMCWEBCWEBSCWESCWESSCWENGCWEJCWECCWEKCWETCWEPCWEHCWICWIGCWIGGCWIGSCWINCWINJCWINHCWIDCWILCWI\
    \LGCWILMCWILBCWILSCWILTCWILPCWILHCWIMCWIBCWIBSCWISCWISSCWINGCWIJCWICCWIKCWITCWIPCWIHCYUCYUGCYUGGCYUGSCYUNCYUNJCYUNHCYUDCYULCYULGCYULMCYULBCYULSCYULTCYULPCYULHCYUMCYUBCYUBSCYUSCYUSSCYUNGCYUJCYUCCYUKCYUTCYUPCYUHCEUCEUGCEUGGCEUGSCEUNCEUNJCEUNHCEUDCEULCEULGCEU\
    \LMCEULBCEULSCEULTCEULPCEULHCEUMCEUBCEUBSCEUSCEUSSCEUNGCEUJCEUCCEUKCEUTCEUPCEUHCYICYIGCYIGGCYIGSCYINCYINJCYINHCYIDCYILCYILGCYILMCYILBCYILSCYILTCYILPCYILHCYIMCYIBCYIBSCYISCYISSCYINGCYIJCYICCYIKCYITCYIPCYIHCICIGCIGGCIGSCINCINJCINHCIDCILCILGCILMCILBCILSCILTCI\
    \LPCILHCIMCIBCIBSCISCISSCINGCIJCICCIKCITCIPCIHKAKAGKAGGKAGSKANKANJKANHKADKALKALGKALMKALBKALSKALTKALPKALHKAMKABKABSKASKASSKANGKAJKACKAKKATKAPKAHKAEKAEGKAEGGKAEGSKAENKAENJKAENHKAEDKAELKAELGKAELMKAELBKAELSKAELTKAELPKAELHKAEMKAEBKAEBSKAESKAESSKAENGKAEJKAECKAEK\
    \KAETKAEPKAEHKYAKYAGKYAGGKYAGSKYANKYANJKYANHKYADKYALKYALGKYALMKYALBKYALSKYALTKYALPKYALHKYAMKYABKYABSKYASKYASSKYANGKYAJKYACKYAKKYATKYAPKYAHKYAEKYAEGKYAEGGKYAEGSKYAENKYAENJKYAENHKYAEDKYAELKYAELGKYAELMKYAELBKYAELSKYAELTKYAELPKYAELHKYAEMKYAEBKYAEBSKYAESKYAESSK\
    \YAENGKYAEJKYAECKYAEKKYAETKYAEPKYAEHKEOKEOGKEOGGKEOGSKEONKEONJKEONHKEODKEOLKEOLGKEOLMKEOLBKEOLSKEOLTKEOLPKEOLHKEOMKEOBKEOBSKEOSKEOSSKEONGKEOJKEOCKEOKKEOTKEOPKEOHKEKEGKEGGKEGSKENKENJKENHKEDKELKELGKELMKELBKELSKELTKELPKELHKEMKEBKEBSKESKESSKENGKEJKECKEKKETKEPK\
    \EHKYEOKYEOGKYEOGGKYEOGSKYEONKYEONJKYEONHKYEODKYEOLKYEOLGKYEOLMKYEOLBKYEOLSKYEOLTKYEOLPKYEOLHKYEOMKYEOBKYEOBSKYEOSKYEOSSKYEONGKYEOJKYEOCKYEOKKYEOTKYEOPKYEOHKYEKYEGKYEGGKYEGSKYENKYENJKYENHKYEDKYELKYELGKYELMKYELBKYELSKYELTKYELPKYELHKYEMKYEBKYEBSKYESKYESSKYEN\
    \GKYEJKYECKYEKKYETKYEPKYEHKOKOGKOGGKOGSKONKONJKONHKODKOLKOLGKOLMKOLBKOLSKOLTKOLPKOLHKOMKOBKOBSKOSKOSSKONGKOJKOCKOKKOTKOPKOHKWAKWAGKWAGGKWAGSKWANKWANJKWANHKWADKWALKWALGKWALMKWALBKWALSKWALTKWALPKWALHKWAMKWABKWABSKWASKWASSKWANGKWAJKWACKWAKKWATKWAPKWAHKWAEKWAE\
    \GKWAEGGKWAEGSKWAENKWAENJKWAENHKWAEDKWAELKWAELGKWAELMKWAELBKWAELSKWAELTKWAELPKWAELHKWAEMKWAEBKWAEBSKWAESKWAESSKWAENGKWAEJKWAECKWAEKKWAETKWAEPKWAEHKOEKOEGKOEGGKOEGSKOENKOENJKOENHKOEDKOELKOELGKOELMKOELBKOELSKOELTKOELPKOELHKOEMKOEBKOEBSKOESKOESSKOENGKOEJKOECK\
    \OEKKOETKOEPKOEHKYOKYOGKYOGGKYOGSKYONKYONJKYONHKYODKYOLKYOLGKYOLMKYOLBKYOLSKYOLTKYOLPKYOLHKYOMKYOBKYOBSKYOSKYOSSKYONGKYOJKYOCKYOKKYOTKYOPKYOHKUKUGKUGGKUGSKUNKUNJKUNHKUDKULKULGKULMKULBKULSKULTKULPKULHKUMKUBKUBSKUSKUSSKUNGKUJKUCKUKKUTKUPKUHKWEOKWEOGKWEOGGKWE\
    \OGSKWEONKWEONJKWEONHKWEODKWEOLKWEOLGKWEOLMKWEOLBKWEOLSKWEOLTKWEOLPKWEOLHKWEOMKWEOBKWEOBSKWEOSKWEOSSKWEONGKWEOJKWEOCKWEOKKWEOTKWEOPKWEOHKWEKWEGKWEGGKWEGSKWENKWENJKWENHKWEDKWELKWELGKWELMKWELBKWELSKWELTKWELPKWELHKWEMKWEBKWEBSKWESKWESSKWENGKWEJKWECKWEKKWETKWE\
    \PKWEHKWIKWIGKWIGGKWIGSKWINKWINJKWINHKWIDKWILKWILGKWILMKWILBKWILSKWILTKWILPKWILHKWIMKWIBKWIBSKWISKWISSKWINGKWIJKWICKWIKKWITKWIPKWIHKYUKYUGKYUGGKYUGSKYUNKYUNJKYUNHKYUDKYULKYULGKYULMKYULBKYULSKYULTKYULPKYULHKYUMKYUBKYUBSKYUSKYUSSKYUNGKYUJKYUCKYUKKYUTKYUPKYUH\
    \KEUKEUGKEUGGKEUGSKEUNKEUNJKEUNHKEUDKEULKEULGKEULMKEULBKEULSKEULTKEULPKEULHKEUMKEUBKEUBSKEUSKEUSSKEUNGKEUJKEUCKEUKKEUTKEUPKEUHKYIKYIGKYIGGKYIGSKYINKYINJKYINHKYIDKYILKYILGKYILMKYILBKYILSKYILTKYILPKYILHKYIMKYIBKYIBSKYISKYISSKYINGKYIJKYICKYIKKYITKYIPKYIHKIKIG\
    \KIGGKIGSKINKINJKINHKIDKILKILGKILMKILBKILSKILTKILPKILHKIMKIBKIBSKISKISSKINGKIJKICKIKKITKIPKIHTATAGTAGGTAGSTANTANJTANHTADTALTALGTALMTALBTALSTALTTALPTALHTAMTABTABSTASTASSTANGTAJTACTAKTATTAPTAHTAETAEGTAEGGTAEGSTAENTAENJTAENHTAEDTAELTAELGTAELMTAELBTAELSTAELTTA\
    \ELPTAELHTAEMTAEBTAEBSTAESTAESSTAENGTAEJTAECTAEKTAETTAEPTAEHTYATYAGTYAGGTYAGSTYANTYANJTYANHTYADTYALTYALGTYALMTYALBTYALSTYALTTYALPTYALHTYAMTYABTYABSTYASTYASSTYANGTYAJTYACTYAKTYATTYAPTYAHTYAETYAEGTYAEGGTYAEGSTYAENTYAENJTYAENHTYAEDTYAELTYAELGTYAELMTYAELBTYAEL\
    \STYAELTTYAELPTYAELHTYAEMTYAEBTYAEBSTYAESTYAESSTYAENGTYAEJTYAECTYAEKTYAETTYAEPTYAEHTEOTEOGTEOGGTEOGSTEONTEONJTEONHTEODTEOLTEOLGTEOLMTEOLBTEOLSTEOLTTEOLPTEOLHTEOMTEOBTEOBSTEOSTEOSSTEONGTEOJTEOCTEOKTEOTTEOPTEOHTETEGTEGGTEGSTENTENJTENHTEDTELTELGTELMTELBTELSTE\
    \LTTELPTELHTEMTEBTEBSTESTESSTENGTEJTECTEKTETTEPTEHTYEOTYEOGTYEOGGTYEOGSTYEONTYEONJTYEONHTYEODTYEOLTYEOLGTYEOLMTYEOLBTYEOLSTYEOLTTYEOLPTYEOLHTYEOMTYEOBTYEOBSTYEOSTYEOSSTYEONGTYEOJTYEOCTYEOKTYEOTTYEOPTYEOHTYETYEGTYEGGTYEGSTYENTYENJTYENHTYEDTYELTYELGTYELMTYEL\
    \BTYELSTYELTTYELPTYELHTYEMTYEBTYEBSTYESTYESSTYENGTYEJTYECTYEKTYETTYEPTYEHTOTOGTOGGTOGSTONTONJTONHTODTOLTOLGTOLMTOLBTOLSTOLTTOLPTOLHTOMTOBTOBSTOSTOSSTONGTOJTOCTOKTOTTOPTOHTWATWAGTWAGGTWAGSTWANTWANJTWANHTWADTWALTWALGTWALMTWALBTWALSTWALTTWALPTWALHTWAMTWABTWAB\
    \STWASTWASSTWANGTWAJTWACTWAKTWATTWAPTWAHTWAETWAEGTWAEGGTWAEGSTWAENTWAENJTWAENHTWAEDTWAELTWAELGTWAELMTWAELBTWAELSTWAELTTWAELPTWAELHTWAEMTWAEBTWAEBSTWAESTWAESSTWAENGTWAEJTWAECTWAEKTWAETTWAEPTWAEHTOETOEGTOEGGTOEGSTOENTOENJTOENHTOEDTOELTOELGTOELMTOELBTOELSTOEL\
    \TTOELPTOELHTOEMTOEBTOEBSTOESTOESSTOENGTOEJTOECTOEKTOETTOEPTOEHTYOTYOGTYOGGTYOGSTYONTYONJTYONHTYODTYOLTYOLGTYOLMTYOLBTYOLSTYOLTTYOLPTYOLHTYOMTYOBTYOBSTYOSTYOSSTYONGTYOJTYOCTYOKTYOTTYOPTYOHTUTUGTUGGTUGSTUNTUNJTUNHTUDTULTULGTULMTULBTULSTULTTULPTULHTUMTUBTUBS\
    \TUSTUSSTUNGTUJTUCTUKTUTTUPTUHTWEOTWEOGTWEOGGTWEOGSTWEONTWEONJTWEONHTWEODTWEOLTWEOLGTWEOLMTWEOLBTWEOLSTWEOLTTWEOLPTWEOLHTWEOMTWEOBTWEOBSTWEOSTWEOSSTWEONGTWEOJTWEOCTWEOKTWEOTTWEOPTWEOHTWETWEGTWEGGTWEGSTWENTWENJTWENHTWEDTWELTWELGTWELMTWELBTWELSTWELTTWELPTWEL\
    \HTWEMTWEBTWEBSTWESTWESSTWENGTWEJTWECTWEKTWETTWEPTWEHTWITWIGTWIGGTWIGSTWINTWINJTWINHTWIDTWILTWILGTWILMTWILBTWILSTWILTTWILPTWILHTWIMTWIBTWIBSTWISTWISSTWINGTWIJTWICTWIKTWITTWIPTWIHTYUTYUGTYUGGTYUGSTYUNTYUNJTYUNHTYUDTYULTYULGTYULMTYULBTYULSTYULTTYULPTYULHTYUM\
    \TYUBTYUBSTYUSTYUSSTYUNGTYUJTYUCTYUKTYUTTYUPTYUHTEUTEUGTEUGGTEUGSTEUNTEUNJTEUNHTEUDTEULTEULGTEULMTEULBTEULSTEULTTEULPTEULHTEUMTEUBTEUBSTEUSTEUSSTEUNGTEUJTEUCTEUKTEUTTEUPTEUHTYITYIGTYIGGTYIGSTYINTYINJTYINHTYIDTYILTYILGTYILMTYILBTYILSTYILTTYILPTYILHTYIMTYIBT\
    \YIBSTYISTYISSTYINGTYIJTYICTYIKTYITTYIPTYIHTITIGTIGGTIGSTINTINJTINHTIDTILTILGTILMTILBTILSTILTTILPTILHTIMTIBTIBSTISTISSTINGTIJTICTIKTITTIPTIHPAPAGPAGGPAGSPANPANJPANHPADPALPALGPALMPALBPALSPALTPALPPALHPAMPABPABSPASPASSPANGPAJPACPAKPATPAPPAHPAEPAEGPAEGGPAEGSPA\
    \ENPAENJPAENHPAEDPAELPAELGPAELMPAELBPAELSPAELTPAELPPAELHPAEMPAEBPAEBSPAESPAESSPAENGPAEJPAECPAEKPAETPAEPPAEHPYAPYAGPYAGGPYAGSPYANPYANJPYANHPYADPYALPYALGPYALMPYALBPYALSPYALTPYALPPYALHPYAMPYABPYABSPYASPYASSPYANGPYAJPYACPYAKPYATPYAPPYAHPYAEPYAEGPYAEGGPYAEGSPYA\
    \ENPYAENJPYAENHPYAEDPYAELPYAELGPYAELMPYAELBPYAELSPYAELTPYAELPPYAELHPYAEMPYAEBPYAEBSPYAESPYAESSPYAENGPYAEJPYAECPYAEKPYAETPYAEPPYAEHPEOPEOGPEOGGPEOGSPEONPEONJPEONHPEODPEOLPEOLGPEOLMPEOLBPEOLSPEOLTPEOLPPEOLHPEOMPEOBPEOBSPEOSPEOSSPEONGPEOJPEOCPEOKPEOTPEOPPEOHP\
    \EPEGPEGGPEGSPENPENJPENHPEDPELPELGPELMPELBPELSPELTPELPPELHPEMPEBPEBSPESPESSPENGPEJPECPEKPETPEPPEHPYEOPYEOGPYEOGGPYEOGSPYEONPYEONJPYEONHPYEODPYEOLPYEOLGPYEOLMPYEOLBPYEOLSPYEOLTPYEOLPPYEOLHPYEOMPYEOBPYEOBSPYEOSPYEOSSPYEONGPYEOJPYEOCPYEOKPYEOTPYEOPPYEOHPYEPYE\
    \GPYEGGPYEGSPYENPYENJPYENHPYEDPYELPYELGPYELMPYELBPYELSPYELTPYELPPYELHPYEMPYEBPYEBSPYESPYESSPYENGPYEJPYECPYEKPYETPYEPPYEHPOPOGPOGGPOGSPONPONJPONHPODPOLPOLGPOLMPOLBPOLSPOLTPOLPPOLHPOMPOBPOBSPOSPOSSPONGPOJPOCPOKPOTPOPPOHPWAPWAGPWAGGPWAGSPWANPWANJPWANHPWADPWAL\
    \PWALGPWALMPWALBPWALSPWALTPWALPPWALHPWAMPWABPWABSPWASPWASSPWANGPWAJPWACPWAKPWATPWAPPWAHPWAEPWAEGPWAEGGPWAEGSPWAENPWAENJPWAENHPWAEDPWAELPWAELGPWAELMPWAELBPWAELSPWAELTPWAELPPWAELHPWAEMPWAEBPWAEBSPWAESPWAESSPWAENGPWAEJPWAECPWAEKPWAETPWAEPPWAEHPOEPOEGPOEGGPOEG\
    \SPOENPOENJPOENHPOEDPOELPOELGPOELMPOELBPOELSPOELTPOELPPOELHPOEMPOEBPOEBSPOESPOESSPOENGPOEJPOECPOEKPOETPOEPPOEHPYOPYOGPYOGGPYOGSPYONPYONJPYONHPYODPYOLPYOLGPYOLMPYOLBPYOLSPYOLTPYOLPPYOLHPYOMPYOBPYOBSPYOSPYOSSPYONGPYOJPYOCPYOKPYOTPYOPPYOHPUPUGPUGGPUGSPUNPUNJP\
    \UNHPUDPULPULGPULMPULBPULSPULTPULPPULHPUMPUBPUBSPUSPUSSPUNGPUJPUCPUKPUTPUPPUHPWEOPWEOGPWEOGGPWEOGSPWEONPWEONJPWEONHPWEODPWEOLPWEOLGPWEOLMPWEOLBPWEOLSPWEOLTPWEOLPPWEOLHPWEOMPWEOBPWEOBSPWEOSPWEOSSPWEONGPWEOJPWEOCPWEOKPWEOTPWEOPPWEOHPWEPWEGPWEGGPWEGSPWENPWENJ\
    \PWENHPWEDPWELPWELGPWELMPWELBPWELSPWELTPWELPPWELHPWEMPWEBPWEBSPWESPWESSPWENGPWEJPWECPWEKPWETPWEPPWEHPWIPWIGPWIGGPWIGSPWINPWINJPWINHPWIDPWILPWILGPWILMPWILBPWILSPWILTPWILPPWILHPWIMPWIBPWIBSPWISPWISSPWINGPWIJPWICPWIKPWITPWIPPWIHPYUPYUGPYUGGPYUGSPYUNPYUNJPYUNH\
    \PYUDPYULPYULGPYULMPYULBPYULSPYULTPYULPPYULHPYUMPYUBPYUBSPYUSPYUSSPYUNGPYUJPYUCPYUKPYUTPYUPPYUHPEUPEUGPEUGGPEUGSPEUNPEUNJPEUNHPEUDPEULPEULGPEULMPEULBPEULSPEULTPEULPPEULHPEUMPEUBPEUBSPEUSPEUSSPEUNGPEUJPEUCPEUKPEUTPEUPPEUHPYIPYIGPYIGGPYIGSPYINPYINJPYINHPYIDP\
    \YILPYILGPYILMPYILBPYILSPYILTPYILPPYILHPYIMPYIBPYIBSPYISPYISSPYINGPYIJPYICPYIKPYITPYIPPYIHPIPIGPIGGPIGSPINPINJPINHPIDPILPILGPILMPILBPILSPILTPILPPILHPIMPIBPIBSPISPISSPINGPIJPICPIKPITPIPPIHHAHAGHAGGHAGSHANHANJHANHHADHALHALGHALMHALBHALSHALTHALPHALHHAMHABHABSH\
    \ASHASSHANGHAJHACHAKHATHAPHAHHAEHAEGHAEGGHAEGSHAENHAENJHAENHHAEDHAELHAELGHAELMHAELBHAELSHAELTHAELPHAELHHAEMHAEBHAEBSHAESHAESSHAENGHAEJHAECHAEKHAETHAEPHAEHHYAHYAGHYAGGHYAGSHYANHYANJHYANHHYADHYALHYALGHYALMHYALBHYALSHYALTHYALPHYALHHYAMHYABHYABSHYASHYASSHYANGH\
    \YAJHYACHYAKHYATHYAPHYAHHYAEHYAEGHYAEGGHYAEGSHYAENHYAENJHYAENHHYAEDHYAELHYAELGHYAELMHYAELBHYAELSHYAELTHYAELPHYAELHHYAEMHYAEBHYAEBSHYAESHYAESSHYAENGHYAEJHYAECHYAEKHYAETHYAEPHYAEHHEOHEOGHEOGGHEOGSHEONHEONJHEONHHEODHEOLHEOLGHEOLMHEOLBHEOLSHEOLTHEOLPHEOLHHEOMH\
    \EOBHEOBSHEOSHEOSSHEONGHEOJHEOCHEOKHEOTHEOPHEOHHEHEGHEGGHEGSHENHENJHENHHEDHELHELGHELMHELBHELSHELTHELPHELHHEMHEBHEBSHESHESSHENGHEJHECHEKHETHEPHEHHYEOHYEOGHYEOGGHYEOGSHYEONHYEONJHYEONHHYEODHYEOLHYEOLGHYEOLMHYEOLBHYEOLSHYEOLTHYEOLPHYEOLHHYEOMHYEOBHYEOBSHYEOSH\
    \YEOSSHYEONGHYEOJHYEOCHYEOKHYEOTHYEOPHYEOHHYEHYEGHYEGGHYEGSHYENHYENJHYENHHYEDHYELHYELGHYELMHYELBHYELSHYELTHYELPHYELHHYEMHYEBHYEBSHYESHYESSHYENGHYEJHYECHYEKHYETHYEPHYEHHOHOGHOGGHOGSHONHONJHONHHODHOLHOLGHOLMHOLBHOLSHOLTHOLPHOLHHOMHOBHOBSHOSHOSSHONGHOJHOCHOKH\
    \OTHOPHOHHWAHWAGHWAGGHWAGSHWANHWANJHWANHHWADHWALHWALGHWALMHWALBHWALSHWALTHWALPHWALHHWAMHWABHWABSHWASHWASSHWANGHWAJHWACHWAKHWATHWAPHWAHHWAEHWAEGHWAEGGHWAEGSHWAENHWAENJHWAENHHWAEDHWAELHWAELGHWAELMHWAELBHWAELSHWAELTHWAELPHWAELHHWAEMHWAEBHWAEBSHWAESHWAESSHWAEN\
    \GHWAEJHWAECHWAEKHWAETHWAEPHWAEHHOEHOEGHOEGGHOEGSHOENHOENJHOENHHOEDHOELHOELGHOELMHOELBHOELSHOELTHOELPHOELHHOEMHOEBHOEBSHOESHOESSHOENGHOEJHOECHOEKHOETHOEPHOEHHYOHYOGHYOGGHYOGSHYONHYONJHYONHHYODHYOLHYOLGHYOLMHYOLBHYOLSHYOLTHYOLPHYOLHHYOMHYOBHYOBSHYOSHYOSSHYO\
    \NGHYOJHYOCHYOKHYOTHYOPHYOHHUHUGHUGGHUGSHUNHUNJHUNHHUDHULHULGHULMHULBHULSHULTHULPHULHHUMHUBHUBSHUSHUSSHUNGHUJHUCHUKHUTHUPHUHHWEOHWEOGHWEOGGHWEOGSHWEONHWEONJHWEONHHWEODHWEOLHWEOLGHWEOLMHWEOLBHWEOLSHWEOLTHWEOLPHWEOLHHWEOMHWEOBHWEOBSHWEOSHWEOSSHWEONGHWEOJHWEO\
    \CHWEOKHWEOTHWEOPHWEOHHWEHWEGHWEGGHWEGSHWENHWENJHWENHHWEDHWELHWELGHWELMHWELBHWELSHWELTHWELPHWELHHWEMHWEBHWEBSHWESHWESSHWENGHWEJHWECHWEKHWETHWEPHWEHHWIHWIGHWIGGHWIGSHWINHWINJHWINHHWIDHWILHWILGHWILMHWILBHWILSHWILTHWILPHWILHHWIMHWIBHWIBSHWISHWISSHWINGHWIJHWIC\
    \HWIKHWITHWIPHWIHHYUHYUGHYUGGHYUGSHYUNHYUNJHYUNHHYUDHYULHYULGHYULMHYULBHYULSHYULTHYULPHYULHHYUMHYUBHYUBSHYUSHYUSSHYUNGHYUJHYUCHYUKHYUTHYUPHYUHHEUHEUGHEUGGHEUGSHEUNHEUNJHEUNHHEUDHEULHEULGHEULMHEULBHEULSHEULTHEULPHEULHHEUMHEUBHEUBSHEUSHEUSSHEUNGHEUJHEUCHEUKH\
    \EUTHEUPHEUHHYIHYIGHYIGGHYIGSHYINHYINJHYINHHYIDHYILHYILGHYILMHYILBHYILSHYILTHYILPHYILHHYIMHYIBHYIBSHYISHYISSHYINGHYIJHYICHYIKHYITHYIPHYIHHIHIGHIGGHIGSHINHINJHINHHIDHILHILGHILMHILBHILSHILTHILPHILHHIMHIBHIBSHISHISSHINGHIJHICHIKHITHIPHIHHANGUL JUNGSEONG O-YEO\
    \HANGUL JUNGSEONG O-O-IHANGUL JUNGSEONG YO-AHANGUL JUNGSEONG YO-AEHANGUL JUNGSEONG YO-EOHANGUL JUNGSEONG U-YEOHANGUL JUNGSEONG U-I-IHANGUL JUNGSEONG YU-AEHANGUL JUNGSEONG YU-OHANGUL JUNGSEONG EU-AHANGUL JUNGSEONG EU-EOHANGUL JUNGSEONG EU-EHANGUL JUNGSEONG \
    \EU-OHANGUL JUNGSEONG I-YA-OHANGUL JUNGSEONG I-YAEHANGUL JUNGSEONG I-YEOHANGUL JUNGSEONG I-YEHANGUL JUNGSEONG I-O-IHANGUL JUNGSEONG I-YOHANGUL JUNGSEONG I-YUHANGUL JUNGSEONG I-IHANGUL JUNGSEONG ARAEA-AHANGUL JUNGSEONG ARAEA-EHANGUL JONGSEONG NIEUN-RIEULHAN\
    \GUL JONGSEONG NIEUN-CHIEUCHHANGUL JONGSEONG SSANGTIKEUTHANGUL JONGSEONG SSANGTIKEUT-PIEUPHANGUL JONGSEONG TIKEUT-PIEUPHANGUL JONGSEONG TIKEUT-SIOSHANGUL JONGSEONG TIKEUT-SIOS-KIYEOKHANGUL JONGSEONG TIKEUT-CIEUCHANGUL JONGSEONG TIKEUT-CHIEUCHHANGUL JONGSEO\
    \NG TIKEUT-THIEUTHHANGUL JONGSEONG RIEUL-SSANGKIYEOKHANGUL JONGSEONG RIEUL-KIYEOK-HIEUHHANGUL JONGSEONG SSANGRIEUL-KHIEUKHHANGUL JONGSEONG RIEUL-MIEUM-HIEUHHANGUL JONGSEONG RIEUL-PIEUP-TIKEUTHANGUL JONGSEONG RIEUL-PIEUP-PHIEUPHHANGUL JONGSEONG RIEUL-YESIEU\
    \NGHANGUL JONGSEONG RIEUL-YEORINHIEUH-HIEUHHANGUL JONGSEONG KAPYEOUNRIEULHANGUL JONGSEONG MIEUM-NIEUNHANGUL JONGSEONG MIEUM-SSANGNIEUNHANGUL JONGSEONG SSANGMIEUMHANGUL JONGSEONG MIEUM-PIEUP-SIOSHANGUL JONGSEONG MIEUM-CIEUCHANGUL JONGSEONG PIEUP-TIKEUTHANGU\
    \L JONGSEONG PIEUP-RIEUL-PHIEUPHHANGUL JONGSEONG PIEUP-MIEUMHANGUL JONGSEONG SSANGPIEUPHANGUL JONGSEONG PIEUP-SIOS-TIKEUTHANGUL JONGSEONG PIEUP-CIEUCHANGUL JONGSEONG PIEUP-CHIEUCHHANGUL JONGSEONG SIOS-MIEUMHANGUL JONGSEONG SIOS-KAPYEOUNPIEUPHANGUL JONGSEON\
    \G SSANGSIOS-KIYEOKHANGUL JONGSEONG SSANGSIOS-TIKEUTHANGUL JONGSEONG SIOS-PANSIOSHANGUL JONGSEONG SIOS-CIEUCHANGUL JONGSEONG SIOS-CHIEUCHHANGUL JONGSEONG SIOS-THIEUTHHANGUL JONGSEONG SIOS-HIEUHHANGUL JONGSEONG PANSIOS-PIEUPHANGUL JONGSEONG PANSIOS-KAPYEOUN\
    \PIEUPHANGUL JONGSEONG YESIEUNG-MIEUMHANGUL JONGSEONG YESIEUNG-HIEUHHANGUL JONGSEONG CIEUC-PIEUPHANGUL JONGSEONG CIEUC-SSANGPIEUPHANGUL JONGSEONG SSANGCIEUCHANGUL JONGSEONG PHIEUPH-SIOSHANGUL JONGSEONG PHIEUPH-THIEUTHLATIN SMALL LIGATURE FFLATIN SMALL LIGA\
    \TURE FILATIN SMALL LIGATURE FLLATIN SMALL LIGATURE FFILATIN SMALL LIGATURE FFLLATIN SMALL LIGATURE LONG S TLATIN SMALL LIGATURE STARMENIAN SMALL LIGATURE MEN NOWARMENIAN SMALL LIGATURE MEN ECHARMENIAN SMALL LIGATURE MEN INIARMENIAN SMALL LIGATURE VEW NOWA\
    \RMENIAN SMALL LIGATURE MEN XEHHEBREW LETTER YOD WITH HIRIQHEBREW POINT JUDEO-SPANISH VARIKAHEBREW LIGATURE YIDDISH YOD YOD PATAHHEBREW LETTER ALTERNATIVE AYINHEBREW LETTER WIDE ALEFHEBREW LETTER WIDE DALETHEBREW LETTER WIDE HEHEBREW LETTER WIDE KAFHEBREW \
    \LETTER WIDE LAMEDHEBREW LETTER WIDE FINAL MEMHEBREW LETTER WIDE RESHHEBREW LETTER WIDE TAVHEBREW LETTER ALTERNATIVE PLUS SIGNHEBREW LETTER SHIN WITH SHIN DOTHEBREW LETTER SHIN WITH SIN DOTHEBREW LETTER SHIN WITH DAGESH AND SHIN DOTHEBREW LETTER SHIN WITH \
    \DAGESH AND SIN DOTHEBREW LETTER ALEF WITH PATAHHEBREW LETTER ALEF WITH QAMATSHEBREW LETTER ALEF WITH MAPIQHEBREW LETTER BET WITH DAGESHHEBREW LETTER GIMEL WITH DAGESHHEBREW LETTER DALET WITH DAGESHHEBREW LETTER HE WITH MAPIQHEBREW LETTER VAV WITH DAGESHHE\
    \BREW LETTER ZAYIN WITH DAGESHHEBREW LETTER TET WITH DAGESHHEBREW LETTER YOD WITH DAGESHHEBREW LETTER FINAL KAF WITH DAGESHHEBREW LETTER KAF WITH DAGESHHEBREW LETTER LAMED WITH DAGESHHEBREW LETTER MEM WITH DAGESHHEBREW LETTER NUN WITH DAGESHHEBREW LETTER S\
    \AMEKH WITH DAGESHHEBREW LETTER FINAL PE WITH DAGESHHEBREW LETTER PE WITH DAGESHHEBREW LETTER TSADI WITH DAGESHHEBREW LETTER QOF WITH DAGESHHEBREW LETTER RESH WITH DAGESHHEBREW LETTER SHIN WITH DAGESHHEBREW LETTER TAV WITH DAGESHHEBREW LETTER VAV WITH HOLA\
    \MHEBREW LETTER BET WITH RAFEHEBREW LETTER KAF WITH RAFEHEBREW LETTER PE WITH RAFEHEBREW LIGATURE ALEF LAMEDARABIC LETTER ALEF WASLA ISOLATED FORMARABIC LETTER ALEF WASLA FINAL FORMARABIC LETTER BEEH ISOLATED FORMARABIC LETTER BEEH FINAL FORMARABIC LETTER \
    \BEEH INITIAL FORMARABIC LETTER BEEH MEDIAL FORMARABIC LETTER PEH ISOLATED FORMARABIC LETTER PEH FINAL FORMARABIC LETTER PEH INITIAL FORMARABIC LETTER PEH MEDIAL FORMARABIC LETTER BEHEH ISOLATED FORMARABIC LETTER BEHEH FINAL FORMARABIC LETTER BEHEH INITIAL\
    \ FORMARABIC LETTER BEHEH MEDIAL FORMARABIC LETTER TTEHEH ISOLATED FORMARABIC LETTER TTEHEH FINAL FORMARABIC LETTER TTEHEH INITIAL FORMARABIC LETTER TTEHEH MEDIAL FORMARABIC LETTER TEHEH ISOLATED FORMARABIC LETTER TEHEH FINAL FORMARABIC LETTER TEHEH INITIA\
    \L FORMARABIC LETTER TEHEH MEDIAL FORMARABIC LETTER TTEH ISOLATED FORMARABIC LETTER TTEH FINAL FORMARABIC LETTER TTEH INITIAL FORMARABIC LETTER TTEH MEDIAL FORMARABIC LETTER VEH ISOLATED FORMARABIC LETTER VEH FINAL FORMARABIC LETTER VEH INITIAL FORMARABIC \
    \LETTER VEH MEDIAL FORMARABIC LETTER PEHEH ISOLATED FORMARABIC LETTER PEHEH FINAL FORMARABIC LETTER PEHEH INITIAL FORMARABIC LETTER PEHEH MEDIAL FORMARABIC LETTER DYEH ISOLATED FORMARABIC LETTER DYEH FINAL FORMARABIC LETTER DYEH INITIAL FORMARABIC LETTER D\
    \YEH MEDIAL FORMARABIC LETTER NYEH ISOLATED FORMARABIC LETTER NYEH FINAL FORMARABIC LETTER NYEH INITIAL FORMARABIC LETTER NYEH MEDIAL FORMARABIC LETTER TCHEH ISOLATED FORMARABIC LETTER TCHEH FINAL FORMARABIC LETTER TCHEH INITIAL FORMARABIC LETTER TCHEH MED\
    \IAL FORMARABIC LETTER TCHEHEH ISOLATED FORMARABIC LETTER TCHEHEH FINAL FORMARABIC LETTER TCHEHEH INITIAL FORMARABIC LETTER TCHEHEH MEDIAL FORMARABIC LETTER DDAHAL ISOLATED FORMARABIC LETTER DDAHAL FINAL FORMARABIC LETTER DAHAL ISOLATED FORMARABIC LETTER D\
    \AHAL FINAL FORMARABIC LETTER DUL ISOLATED FORMARABIC LETTER DUL FINAL FORMARABIC LETTER DDAL ISOLATED FORMARABIC LETTER DDAL FINAL FORMARABIC LETTER JEH ISOLATED FORMARABIC LETTER JEH FINAL FORMARABIC LETTER RREH ISOLATED FORMARABIC LETTER RREH FINAL FORM\
    \ARABIC LETTER KEHEH ISOLATED FORMARABIC LETTER KEHEH FINAL FORMARABIC LETTER KEHEH INITIAL FORMARABIC LETTER KEHEH MEDIAL FORMARABIC LETTER GAF ISOLATED FORMARABIC LETTER GAF FINAL FORMARABIC LETTER GAF INITIAL FORMARABIC LETTER GAF MEDIAL FORMARABIC LETT\
    \ER GUEH ISOLATED FORMARABIC LETTER GUEH FINAL FORMARABIC LETTER GUEH INITIAL FORMARABIC LETTER GUEH MEDIAL FORMARABIC LETTER NGOEH ISOLATED FORMARABIC LETTER NGOEH FINAL FORMARABIC LETTER NGOEH INITIAL FORMARABIC LETTER NGOEH MEDIAL FORMARABIC LETTER NOON\
    \ GHUNNA ISOLATED FORMARABIC LETTER NOON GHUNNA FINAL FORMARABIC LETTER RNOON ISOLATED FORMARABIC LETTER RNOON FINAL FORMARABIC LETTER RNOON INITIAL FORMARABIC LETTER RNOON MEDIAL FORMARABIC LETTER HEH WITH YEH ABOVE ISOLATED FORMARABIC LETTER HEH WITH YEH\
    \ ABOVE FINAL FORMARABIC LETTER HEH GOAL ISOLATED FORMARABIC LETTER HEH GOAL FINAL FORMARABIC LETTER HEH GOAL INITIAL FORMARABIC LETTER HEH GOAL MEDIAL FORMARABIC LETTER HEH DOACHASHMEE ISOLATED FORMARABIC LETTER HEH DOACHASHMEE FINAL FORMARABIC LETTER HEH\
    \ DOACHASHMEE INITIAL FORMARABIC LETTER HEH DOACHASHMEE MEDIAL FORMARABIC LETTER YEH BARREE ISOLATED FORMARABIC LETTER YEH BARREE FINAL FORMARABIC LETTER YEH BARREE WITH HAMZA ABOVE ISOLATED FORMARABIC LETTER YEH BARREE WITH HAMZA ABOVE FINAL FORMARABIC SY\
    \MBOL DOT ABOVEARABIC SYMBOL DOT BELOWARABIC SYMBOL TWO DOTS ABOVEARABIC SYMBOL TWO DOTS BELOWARABIC SYMBOL THREE DOTS ABOVEARABIC SYMBOL THREE DOTS BELOWARABIC SYMBOL THREE DOTS POINTING DOWNWARDS ABOVEARABIC SYMBOL THREE DOTS POINTING DOWNWARDS BELOWARAB\
    \IC SYMBOL FOUR DOTS ABOVEARABIC SYMBOL FOUR DOTS BELOWARABIC SYMBOL DOUBLE VERTICAL BAR BELOWARABIC SYMBOL TWO DOTS VERTICALLY ABOVEARABIC SYMBOL TWO DOTS VERTICALLY BELOWARABIC SYMBOL RINGARABIC SYMBOL SMALL TAH ABOVEARABIC SYMBOL SMALL TAH BELOWARABIC S\
    \YMBOL WASLA ABOVEARABIC LETTER NG ISOLATED FORMARABIC LETTER NG FINAL FORMARABIC LETTER NG INITIAL FORMARABIC LETTER NG MEDIAL FORMARABIC LETTER U ISOLATED FORMARABIC LETTER U FINAL FORMARABIC LETTER OE ISOLATED FORMARABIC LETTER OE FINAL FORMARABIC LETTE\
    \R YU ISOLATED FORMARABIC LETTER YU FINAL FORMARABIC LETTER U WITH HAMZA ABOVE ISOLATED FORMARABIC LETTER VE ISOLATED FORMARABIC LETTER VE FINAL FORMARABIC LETTER KIRGHIZ OE ISOLATED FORMARABIC LETTER KIRGHIZ OE FINAL FORMARABIC LETTER KIRGHIZ YU ISOLATED \
    \FORMARABIC LETTER KIRGHIZ YU FINAL FORMARABIC LETTER E ISOLATED FORMARABIC LETTER E FINAL FORMARABIC LETTER E INITIAL FORMARABIC LETTER E MEDIAL FORMARABIC LETTER UIGHUR KAZAKH KIRGHIZ ALEF MAKSURA INITIAL FORMARABIC LETTER UIGHUR KAZAKH KIRGHIZ ALEF MAKS\
    \URA MEDIAL FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF ISOLATED FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF FINAL FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH AE ISOLATED FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH AE FINAL FORMARABIC LIG\
    \ATURE YEH WITH HAMZA ABOVE WITH WAW ISOLATED FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH WAW FINAL FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH U ISOLATED FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH U FINAL FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WI\
    \TH OE ISOLATED FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH OE FINAL FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YU ISOLATED FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YU FINAL FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH E ISOLATED FORMARABIC LIGA\
    \TURE YEH WITH HAMZA ABOVE WITH E FINAL FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH E INITIAL FORMARABIC LIGATURE UIGHUR KIRGHIZ YEH WITH HAMZA ABOVE WITH ALEF MAKSURA ISOLATED FORMARABIC LIGATURE UIGHUR KIRGHIZ YEH WITH HAMZA ABOVE WITH ALEF MAKSURA FIN\
    \AL FORMARABIC LIGATURE UIGHUR KIRGHIZ YEH WITH HAMZA ABOVE WITH ALEF MAKSURA INITIAL FORMARABIC LETTER FARSI YEH ISOLATED FORMARABIC LETTER FARSI YEH FINAL FORMARABIC LETTER FARSI YEH INITIAL FORMARABIC LETTER FARSI YEH MEDIAL FORMARABIC LIGATURE YEH WITH\
    \ HAMZA ABOVE WITH JEEM ISOLATED FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HAH ISOLATED FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM ISOLATED FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF MAKSURA ISOLATED FORMARABIC LIGATURE YEH WITH HAMZA A\
    \BOVE WITH YEH ISOLATED FORMARABIC LIGATURE BEH WITH JEEM ISOLATED FORMARABIC LIGATURE BEH WITH HAH ISOLATED FORMARABIC LIGATURE BEH WITH KHAH ISOLATED FORMARABIC LIGATURE BEH WITH MEEM ISOLATED FORMARABIC LIGATURE BEH WITH ALEF MAKSURA ISOLATED FORMARABIC\
    \ LIGATURE BEH WITH YEH ISOLATED FORMARABIC LIGATURE TEH WITH JEEM ISOLATED FORMARABIC LIGATURE TEH WITH HAH ISOLATED FORMARABIC LIGATURE TEH WITH KHAH ISOLATED FORMARABIC LIGATURE TEH WITH MEEM ISOLATED FORMARABIC LIGATURE TEH WITH ALEF MAKSURA ISOLATED F\
    \ORMARABIC LIGATURE TEH WITH YEH ISOLATED FORMARABIC LIGATURE THEH WITH JEEM ISOLATED FORMARABIC LIGATURE THEH WITH MEEM ISOLATED FORMARABIC LIGATURE THEH WITH ALEF MAKSURA ISOLATED FORMARABIC LIGATURE THEH WITH YEH ISOLATED FORMARABIC LIGATURE JEEM WITH H\
    \AH ISOLATED FORMARABIC LIGATURE JEEM WITH MEEM ISOLATED FORMARABIC LIGATURE HAH WITH JEEM ISOLATED FORMARABIC LIGATURE HAH WITH MEEM ISOLATED FORMARABIC LIGATURE KHAH WITH JEEM ISOLATED FORMARABIC LIGATURE KHAH WITH HAH ISOLATED FORMARABIC LIGATURE KHAH W\
    \ITH MEEM ISOLATED FORMARABIC LIGATURE SEEN WITH JEEM ISOLATED FORMARABIC LIGATURE SEEN WITH HAH ISOLATED FORMARABIC LIGATURE SEEN WITH KHAH ISOLATED FORMARABIC LIGATURE SEEN WITH MEEM ISOLATED FORMARABIC LIGATURE SAD WITH HAH ISOLATED FORMARABIC LIGATURE \
    \SAD WITH MEEM ISOLATED FORMARABIC LIGATURE DAD WITH JEEM ISOLATED FORMARABIC LIGATURE DAD WITH HAH ISOLATED FORMARABIC LIGATURE DAD WITH KHAH ISOLATED FORMARABIC LIGATURE DAD WITH MEEM ISOLATED FORMARABIC LIGATURE TAH WITH HAH ISOLATED FORMARABIC LIGATURE\
    \ TAH WITH MEEM ISOLATED FORMARABIC LIGATURE ZAH WITH MEEM ISOLATED FORMARABIC LIGATURE AIN WITH JEEM ISOLATED FORMARABIC LIGATURE AIN WITH MEEM ISOLATED FORMARABIC LIGATURE GHAIN WITH JEEM ISOLATED FORMARABIC LIGATURE GHAIN WITH MEEM ISOLATED FORMARABIC L\
    \IGATURE FEH WITH JEEM ISOLATED FORMARABIC LIGATURE FEH WITH HAH ISOLATED FORMARABIC LIGATURE FEH WITH KHAH ISOLATED FORMARABIC LIGATURE FEH WITH MEEM ISOLATED FORMARABIC LIGATURE FEH WITH ALEF MAKSURA ISOLATED FORMARABIC LIGATURE FEH WITH YEH ISOLATED FOR\
    \MARABIC LIGATURE QAF WITH HAH ISOLATED FORMARABIC LIGATURE QAF WITH MEEM ISOLATED FORMARABIC LIGATURE QAF WITH ALEF MAKSURA ISOLATED FORMARABIC LIGATURE QAF WITH YEH ISOLATED FORMARABIC LIGATURE KAF WITH ALEF ISOLATED FORMARABIC LIGATURE KAF WITH JEEM ISO\
    \LATED FORMARABIC LIGATURE KAF WITH HAH ISOLATED FORMARABIC LIGATURE KAF WITH KHAH ISOLATED FORMARABIC LIGATURE KAF WITH LAM ISOLATED FORMARABIC LIGATURE KAF WITH MEEM ISOLATED FORMARABIC LIGATURE KAF WITH ALEF MAKSURA ISOLATED FORMARABIC LIGATURE KAF WITH\
    \ YEH ISOLATED FORMARABIC LIGATURE LAM WITH JEEM ISOLATED FORMARABIC LIGATURE LAM WITH HAH ISOLATED FORMARABIC LIGATURE LAM WITH KHAH ISOLATED FORMARABIC LIGATURE LAM WITH MEEM ISOLATED FORMARABIC LIGATURE LAM WITH ALEF MAKSURA ISOLATED FORMARABIC LIGATURE\
    \ LAM WITH YEH ISOLATED FORMARABIC LIGATURE MEEM WITH JEEM ISOLATED FORMARABIC LIGATURE MEEM WITH HAH ISOLATED FORMARABIC LIGATURE MEEM WITH KHAH ISOLATED FORMARABIC LIGATURE MEEM WITH MEEM ISOLATED FORMARABIC LIGATURE MEEM WITH ALEF MAKSURA ISOLATED FORMA\
    \RABIC LIGATURE MEEM WITH YEH ISOLATED FORMARABIC LIGATURE NOON WITH JEEM ISOLATED FORMARABIC LIGATURE NOON WITH HAH ISOLATED FORMARABIC LIGATURE NOON WITH KHAH ISOLATED FORMARABIC LIGATURE NOON WITH MEEM ISOLATED FORMARABIC LIGATURE NOON WITH ALEF MAKSURA\
    \ ISOLATED FORMARABIC LIGATURE NOON WITH YEH ISOLATED FORMARABIC LIGATURE HEH WITH JEEM ISOLATED FORMARABIC LIGATURE HEH WITH MEEM ISOLATED FORMARABIC LIGATURE HEH WITH ALEF MAKSURA ISOLATED FORMARABIC LIGATURE HEH WITH YEH ISOLATED FORMARABIC LIGATURE YEH\
    \ WITH JEEM ISOLATED FORMARABIC LIGATURE YEH WITH HAH ISOLATED FORMARABIC LIGATURE YEH WITH KHAH ISOLATED FORMARABIC LIGATURE YEH WITH MEEM ISOLATED FORMARABIC LIGATURE YEH WITH ALEF MAKSURA ISOLATED FORMARABIC LIGATURE YEH WITH YEH ISOLATED FORMARABIC LIG\
    \ATURE THAL WITH SUPERSCRIPT ALEF ISOLATED FORMARABIC LIGATURE REH WITH SUPERSCRIPT ALEF ISOLATED FORMARABIC LIGATURE ALEF MAKSURA WITH SUPERSCRIPT ALEF ISOLATED FORMARABIC LIGATURE SHADDA WITH DAMMATAN ISOLATED FORMARABIC LIGATURE SHADDA WITH KASRATAN ISO\
    \LATED FORMARABIC LIGATURE SHADDA WITH FATHA ISOLATED FORMARABIC LIGATURE SHADDA WITH DAMMA ISOLATED FORMARABIC LIGATURE SHADDA WITH KASRA ISOLATED FORMARABIC LIGATURE SHADDA WITH SUPERSCRIPT ALEF ISOLATED FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH REH \
    \FINAL FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ZAIN FINAL FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM FINAL FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH NOON FINAL FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF MAKSURA FINAL FORMARABIC LI\
    \GATURE YEH WITH HAMZA ABOVE WITH YEH FINAL FORMARABIC LIGATURE BEH WITH REH FINAL FORMARABIC LIGATURE BEH WITH ZAIN FINAL FORMARABIC LIGATURE BEH WITH MEEM FINAL FORMARABIC LIGATURE BEH WITH NOON FINAL FORMARABIC LIGATURE BEH WITH ALEF MAKSURA FINAL FORMA\
    \RABIC LIGATURE BEH WITH YEH FINAL FORMARABIC LIGATURE TEH WITH REH FINAL FORMARABIC LIGATURE TEH WITH ZAIN FINAL FORMARABIC LIGATURE TEH WITH MEEM FINAL FORMARABIC LIGATURE TEH WITH NOON FINAL FORMARABIC LIGATURE TEH WITH ALEF MAKSURA FINAL FORMARABIC LIG\
    \ATURE TEH WITH YEH FINAL FORMARABIC LIGATURE THEH WITH REH FINAL FORMARABIC LIGATURE THEH WITH ZAIN FINAL FORMARABIC LIGATURE THEH WITH MEEM FINAL FORMARABIC LIGATURE THEH WITH NOON FINAL FORMARABIC LIGATURE THEH WITH ALEF MAKSURA FINAL FORMARABIC LIGATUR\
    \E THEH WITH YEH FINAL FORMARABIC LIGATURE FEH WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE FEH WITH YEH FINAL FORMARABIC LIGATURE QAF WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE QAF WITH YEH FINAL FORMARABIC LIGATURE KAF WITH ALEF FINAL FORMARABIC LIGATURE\
    \ KAF WITH LAM FINAL FORMARABIC LIGATURE KAF WITH MEEM FINAL FORMARABIC LIGATURE KAF WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE KAF WITH YEH FINAL FORMARABIC LIGATURE LAM WITH MEEM FINAL FORMARABIC LIGATURE LAM WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE \
    \LAM WITH YEH FINAL FORMARABIC LIGATURE MEEM WITH ALEF FINAL FORMARABIC LIGATURE MEEM WITH MEEM FINAL FORMARABIC LIGATURE NOON WITH REH FINAL FORMARABIC LIGATURE NOON WITH ZAIN FINAL FORMARABIC LIGATURE NOON WITH MEEM FINAL FORMARABIC LIGATURE NOON WITH NO\
    \ON FINAL FORMARABIC LIGATURE NOON WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE NOON WITH YEH FINAL FORMARABIC LIGATURE ALEF MAKSURA WITH SUPERSCRIPT ALEF FINAL FORMARABIC LIGATURE YEH WITH REH FINAL FORMARABIC LIGATURE YEH WITH ZAIN FINAL FORMARABIC LIGATU\
    \RE YEH WITH MEEM FINAL FORMARABIC LIGATURE YEH WITH NOON FINAL FORMARABIC LIGATURE YEH WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE YEH WITH YEH FINAL FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH JEEM INITIAL FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH \
    \HAH INITIAL FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH KHAH INITIAL FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM INITIAL FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HEH INITIAL FORMARABIC LIGATURE BEH WITH JEEM INITIAL FORMARABIC LIGATURE BEH W\
    \ITH HAH INITIAL FORMARABIC LIGATURE BEH WITH KHAH INITIAL FORMARABIC LIGATURE BEH WITH MEEM INITIAL FORMARABIC LIGATURE BEH WITH HEH INITIAL FORMARABIC LIGATURE TEH WITH JEEM INITIAL FORMARABIC LIGATURE TEH WITH HAH INITIAL FORMARABIC LIGATURE TEH WITH KH\
    \AH INITIAL FORMARABIC LIGATURE TEH WITH MEEM INITIAL FORMARABIC LIGATURE TEH WITH HEH INITIAL FORMARABIC LIGATURE THEH WITH MEEM INITIAL FORMARABIC LIGATURE JEEM WITH HAH INITIAL FORMARABIC LIGATURE JEEM WITH MEEM INITIAL FORMARABIC LIGATURE HAH WITH JEEM\
    \ INITIAL FORMARABIC LIGATURE HAH WITH MEEM INITIAL FORMARABIC LIGATURE KHAH WITH JEEM INITIAL FORMARABIC LIGATURE KHAH WITH MEEM INITIAL FORMARABIC LIGATURE SEEN WITH JEEM INITIAL FORMARABIC LIGATURE SEEN WITH HAH INITIAL FORMARABIC LIGATURE SEEN WITH KHA\
    \H INITIAL FORMARABIC LIGATURE SEEN WITH MEEM INITIAL FORMARABIC LIGATURE SAD WITH HAH INITIAL FORMARABIC LIGATURE SAD WITH KHAH INITIAL FORMARABIC LIGATURE SAD WITH MEEM INITIAL FORMARABIC LIGATURE DAD WITH JEEM INITIAL FORMARABIC LIGATURE DAD WITH HAH IN\
    \ITIAL FORMARABIC LIGATURE DAD WITH KHAH INITIAL FORMARABIC LIGATURE DAD WITH MEEM INITIAL FORMARABIC LIGATURE TAH WITH HAH INITIAL FORMARABIC LIGATURE ZAH WITH MEEM INITIAL FORMARABIC LIGATURE AIN WITH JEEM INITIAL FORMARABIC LIGATURE AIN WITH MEEM INITIA\
    \L FORMARABIC LIGATURE GHAIN WITH JEEM INITIAL FORMARABIC LIGATURE GHAIN WITH MEEM INITIAL FORMARABIC LIGATURE FEH WITH JEEM INITIAL FORMARABIC LIGATURE FEH WITH HAH INITIAL FORMARABIC LIGATURE FEH WITH KHAH INITIAL FORMARABIC LIGATURE FEH WITH MEEM INITIA\
    \L FORMARABIC LIGATURE QAF WITH HAH INITIAL FORMARABIC LIGATURE QAF WITH MEEM INITIAL FORMARABIC LIGATURE KAF WITH JEEM INITIAL FORMARABIC LIGATURE KAF WITH HAH INITIAL FORMARABIC LIGATURE KAF WITH KHAH INITIAL FORMARABIC LIGATURE KAF WITH LAM INITIAL FORM\
    \ARABIC LIGATURE KAF WITH MEEM INITIAL FORMARABIC LIGATURE LAM WITH JEEM INITIAL FORMARABIC LIGATURE LAM WITH HAH INITIAL FORMARABIC LIGATURE LAM WITH KHAH INITIAL FORMARABIC LIGATURE LAM WITH MEEM INITIAL FORMARABIC LIGATURE LAM WITH HEH INITIAL FORMARABI\
    \C LIGATURE MEEM WITH JEEM INITIAL FORMARABIC LIGATURE MEEM WITH HAH INITIAL FORMARABIC LIGATURE MEEM WITH KHAH INITIAL FORMARABIC LIGATURE MEEM WITH MEEM INITIAL FORMARABIC LIGATURE NOON WITH JEEM INITIAL FORMARABIC LIGATURE NOON WITH HAH INITIAL FORMARAB\
    \IC LIGATURE NOON WITH KHAH INITIAL FORMARABIC LIGATURE NOON WITH MEEM INITIAL FORMARABIC LIGATURE NOON WITH HEH INITIAL FORMARABIC LIGATURE HEH WITH JEEM INITIAL FORMARABIC LIGATURE HEH WITH MEEM INITIAL FORMARABIC LIGATURE HEH WITH SUPERSCRIPT ALEF INITI\
    \AL FORMARABIC LIGATURE YEH WITH JEEM INITIAL FORMARABIC LIGATURE YEH WITH HAH INITIAL FORMARABIC LIGATURE YEH WITH KHAH INITIAL FORMARABIC LIGATURE YEH WITH MEEM INITIAL FORMARABIC LIGATURE YEH WITH HEH INITIAL FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WIT\
    \H MEEM MEDIAL FORMARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HEH MEDIAL FORMARABIC LIGATURE BEH WITH MEEM MEDIAL FORMARABIC LIGATURE BEH WITH HEH MEDIAL FORMARABIC LIGATURE TEH WITH MEEM MEDIAL FORMARABIC LIGATURE TEH WITH HEH MEDIAL FORMARABIC LIGATURE TH\
    \EH WITH MEEM MEDIAL FORMARABIC LIGATURE THEH WITH HEH MEDIAL FORMARABIC LIGATURE SEEN WITH MEEM MEDIAL FORMARABIC LIGATURE SEEN WITH HEH MEDIAL FORMARABIC LIGATURE SHEEN WITH MEEM MEDIAL FORMARABIC LIGATURE SHEEN WITH HEH MEDIAL FORMARABIC LIGATURE KAF WI\
    \TH LAM MEDIAL FORMARABIC LIGATURE KAF WITH MEEM MEDIAL FORMARABIC LIGATURE LAM WITH MEEM MEDIAL FORMARABIC LIGATURE NOON WITH MEEM MEDIAL FORMARABIC LIGATURE NOON WITH HEH MEDIAL FORMARABIC LIGATURE YEH WITH MEEM MEDIAL FORMARABIC LIGATURE YEH WITH HEH ME\
    \DIAL FORMARABIC LIGATURE SHADDA WITH FATHA MEDIAL FORMARABIC LIGATURE SHADDA WITH DAMMA MEDIAL FORMARABIC LIGATURE SHADDA WITH KASRA MEDIAL FORMARABIC LIGATURE TAH WITH ALEF MAKSURA ISOLATED FORMARABIC LIGATURE TAH WITH YEH ISOLATED FORMARABIC LIGATURE AI\
    \N WITH ALEF MAKSURA ISOLATED FORMARABIC LIGATURE AIN WITH YEH ISOLATED FORMARABIC LIGATURE GHAIN WITH ALEF MAKSURA ISOLATED FORMARABIC LIGATURE GHAIN WITH YEH ISOLATED FORMARABIC LIGATURE SEEN WITH ALEF MAKSURA ISOLATED FORMARABIC LIGATURE SEEN WITH YEH I\
    \SOLATED FORMARABIC LIGATURE SHEEN WITH ALEF MAKSURA ISOLATED FORMARABIC LIGATURE SHEEN WITH YEH ISOLATED FORMARABIC LIGATURE HAH WITH ALEF MAKSURA ISOLATED FORMARABIC LIGATURE HAH WITH YEH ISOLATED FORMARABIC LIGATURE JEEM WITH ALEF MAKSURA ISOLATED FORMA\
    \RABIC LIGATURE JEEM WITH YEH ISOLATED FORMARABIC LIGATURE KHAH WITH ALEF MAKSURA ISOLATED FORMARABIC LIGATURE KHAH WITH YEH ISOLATED FORMARABIC LIGATURE SAD WITH ALEF MAKSURA ISOLATED FORMARABIC LIGATURE SAD WITH YEH ISOLATED FORMARABIC LIGATURE DAD WITH \
    \ALEF MAKSURA ISOLATED FORMARABIC LIGATURE DAD WITH YEH ISOLATED FORMARABIC LIGATURE SHEEN WITH JEEM ISOLATED FORMARABIC LIGATURE SHEEN WITH HAH ISOLATED FORMARABIC LIGATURE SHEEN WITH KHAH ISOLATED FORMARABIC LIGATURE SHEEN WITH MEEM ISOLATED FORMARABIC L\
    \IGATURE SHEEN WITH REH ISOLATED FORMARABIC LIGATURE SEEN WITH REH ISOLATED FORMARABIC LIGATURE SAD WITH REH ISOLATED FORMARABIC LIGATURE DAD WITH REH ISOLATED FORMARABIC LIGATURE TAH WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE TAH WITH YEH FINAL FORMARABI\
    \C LIGATURE AIN WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE AIN WITH YEH FINAL FORMARABIC LIGATURE GHAIN WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE GHAIN WITH YEH FINAL FORMARABIC LIGATURE SEEN WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE SEEN WITH YEH FIN\
    \AL FORMARABIC LIGATURE SHEEN WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE SHEEN WITH YEH FINAL FORMARABIC LIGATURE HAH WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE HAH WITH YEH FINAL FORMARABIC LIGATURE JEEM WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE JEEM \
    \WITH YEH FINAL FORMARABIC LIGATURE KHAH WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE KHAH WITH YEH FINAL FORMARABIC LIGATURE SAD WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE SAD WITH YEH FINAL FORMARABIC LIGATURE DAD WITH ALEF MAKSURA FINAL FORMARABIC LIGAT\
    \URE DAD WITH YEH FINAL FORMARABIC LIGATURE SHEEN WITH JEEM FINAL FORMARABIC LIGATURE SHEEN WITH HAH FINAL FORMARABIC LIGATURE SHEEN WITH KHAH FINAL FORMARABIC LIGATURE SHEEN WITH MEEM FINAL FORMARABIC LIGATURE SHEEN WITH REH FINAL FORMARABIC LIGATURE SEEN\
    \ WITH REH FINAL FORMARABIC LIGATURE SAD WITH REH FINAL FORMARABIC LIGATURE DAD WITH REH FINAL FORMARABIC LIGATURE SHEEN WITH JEEM INITIAL FORMARABIC LIGATURE SHEEN WITH HAH INITIAL FORMARABIC LIGATURE SHEEN WITH KHAH INITIAL FORMARABIC LIGATURE SHEEN WITH\
    \ MEEM INITIAL FORMARABIC LIGATURE SEEN WITH HEH INITIAL FORMARABIC LIGATURE SHEEN WITH HEH INITIAL FORMARABIC LIGATURE TAH WITH MEEM INITIAL FORMARABIC LIGATURE SEEN WITH JEEM MEDIAL FORMARABIC LIGATURE SEEN WITH HAH MEDIAL FORMARABIC LIGATURE SEEN WITH K\
    \HAH MEDIAL FORMARABIC LIGATURE SHEEN WITH JEEM MEDIAL FORMARABIC LIGATURE SHEEN WITH HAH MEDIAL FORMARABIC LIGATURE SHEEN WITH KHAH MEDIAL FORMARABIC LIGATURE TAH WITH MEEM MEDIAL FORMARABIC LIGATURE ZAH WITH MEEM MEDIAL FORMARABIC LIGATURE ALEF WITH FATH\
    \ATAN FINAL FORMARABIC LIGATURE ALEF WITH FATHATAN ISOLATED FORMORNATE LEFT PARENTHESISORNATE RIGHT PARENTHESISARABIC LIGATURE RAHIMAHU ALLAAHARABIC LIGATURE RADI ALLAAHU ANHARABIC LIGATURE RADI ALLAAHU ANHAAARABIC LIGATURE RADI ALLAAHU ANHUMARABIC LIGATUR\
    \E RADI ALLAAHU ANHUMAAARABIC LIGATURE RADI ALLAAHU ANHUNNAARABIC LIGATURE SALLALLAAHU ALAYHI WA-AALIHARABIC LIGATURE ALAYHI AS-SALAAMARABIC LIGATURE ALAYHIM AS-SALAAMARABIC LIGATURE ALAYHIMAA AS-SALAAMARABIC LIGATURE ALAYHI AS-SALAATU WAS-SALAAMARABIC LIG\
    \ATURE QUDDISA SIRRAHARABIC LIGATURE SALLALLAHU ALAYHI WAAALIHEE WA-SALLAMARABIC LIGATURE ALAYHAA AS-SALAAMARABIC LIGATURE TABAARAKA WA-TAAALAAARABIC LIGATURE RAHIMAHUM ALLAAHARABIC LIGATURE TEH WITH JEEM WITH MEEM INITIAL FORMARABIC LIGATURE TEH WITH HAH \
    \WITH JEEM FINAL FORMARABIC LIGATURE TEH WITH HAH WITH JEEM INITIAL FORMARABIC LIGATURE TEH WITH HAH WITH MEEM INITIAL FORMARABIC LIGATURE TEH WITH KHAH WITH MEEM INITIAL FORMARABIC LIGATURE TEH WITH MEEM WITH JEEM INITIAL FORMARABIC LIGATURE TEH WITH MEEM\
    \ WITH HAH INITIAL FORMARABIC LIGATURE TEH WITH MEEM WITH KHAH INITIAL FORMARABIC LIGATURE JEEM WITH MEEM WITH HAH FINAL FORMARABIC LIGATURE JEEM WITH MEEM WITH HAH INITIAL FORMARABIC LIGATURE HAH WITH MEEM WITH YEH FINAL FORMARABIC LIGATURE HAH WITH MEEM \
    \WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE SEEN WITH HAH WITH JEEM INITIAL FORMARABIC LIGATURE SEEN WITH JEEM WITH HAH INITIAL FORMARABIC LIGATURE SEEN WITH JEEM WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE SEEN WITH MEEM WITH HAH FINAL FORMARABIC LIGATUR\
    \E SEEN WITH MEEM WITH HAH INITIAL FORMARABIC LIGATURE SEEN WITH MEEM WITH JEEM INITIAL FORMARABIC LIGATURE SEEN WITH MEEM WITH MEEM FINAL FORMARABIC LIGATURE SEEN WITH MEEM WITH MEEM INITIAL FORMARABIC LIGATURE SAD WITH HAH WITH HAH FINAL FORMARABIC LIGAT\
    \URE SAD WITH HAH WITH HAH INITIAL FORMARABIC LIGATURE SAD WITH MEEM WITH MEEM FINAL FORMARABIC LIGATURE SHEEN WITH HAH WITH MEEM FINAL FORMARABIC LIGATURE SHEEN WITH HAH WITH MEEM INITIAL FORMARABIC LIGATURE SHEEN WITH JEEM WITH YEH FINAL FORMARABIC LIGAT\
    \URE SHEEN WITH MEEM WITH KHAH FINAL FORMARABIC LIGATURE SHEEN WITH MEEM WITH KHAH INITIAL FORMARABIC LIGATURE SHEEN WITH MEEM WITH MEEM FINAL FORMARABIC LIGATURE SHEEN WITH MEEM WITH MEEM INITIAL FORMARABIC LIGATURE DAD WITH HAH WITH ALEF MAKSURA FINAL FO\
    \RMARABIC LIGATURE DAD WITH KHAH WITH MEEM FINAL FORMARABIC LIGATURE DAD WITH KHAH WITH MEEM INITIAL FORMARABIC LIGATURE TAH WITH MEEM WITH HAH FINAL FORMARABIC LIGATURE TAH WITH MEEM WITH HAH INITIAL FORMARABIC LIGATURE TAH WITH MEEM WITH MEEM INITIAL FOR\
    \MARABIC LIGATURE TAH WITH MEEM WITH YEH FINAL FORMARABIC LIGATURE AIN WITH JEEM WITH MEEM FINAL FORMARABIC LIGATURE AIN WITH MEEM WITH MEEM FINAL FORMARABIC LIGATURE AIN WITH MEEM WITH MEEM INITIAL FORMARABIC LIGATURE AIN WITH MEEM WITH ALEF MAKSURA FINAL\
    \ FORMARABIC LIGATURE GHAIN WITH MEEM WITH MEEM FINAL FORMARABIC LIGATURE GHAIN WITH MEEM WITH YEH FINAL FORMARABIC LIGATURE GHAIN WITH MEEM WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE FEH WITH KHAH WITH MEEM FINAL FORMARABIC LIGATURE FEH WITH KHAH WITH ME\
    \EM INITIAL FORMARABIC LIGATURE QAF WITH MEEM WITH HAH FINAL FORMARABIC LIGATURE QAF WITH MEEM WITH MEEM FINAL FORMARABIC LIGATURE LAM WITH HAH WITH MEEM FINAL FORMARABIC LIGATURE LAM WITH HAH WITH YEH FINAL FORMARABIC LIGATURE LAM WITH HAH WITH ALEF MAKSU\
    \RA FINAL FORMARABIC LIGATURE LAM WITH JEEM WITH JEEM INITIAL FORMARABIC LIGATURE LAM WITH JEEM WITH JEEM FINAL FORMARABIC LIGATURE LAM WITH KHAH WITH MEEM FINAL FORMARABIC LIGATURE LAM WITH KHAH WITH MEEM INITIAL FORMARABIC LIGATURE LAM WITH MEEM WITH HAH\
    \ FINAL FORMARABIC LIGATURE LAM WITH MEEM WITH HAH INITIAL FORMARABIC LIGATURE MEEM WITH HAH WITH JEEM INITIAL FORMARABIC LIGATURE MEEM WITH HAH WITH MEEM INITIAL FORMARABIC LIGATURE MEEM WITH HAH WITH YEH FINAL FORMARABIC LIGATURE MEEM WITH JEEM WITH HAH \
    \INITIAL FORMARABIC LIGATURE MEEM WITH JEEM WITH MEEM INITIAL FORMARABIC LIGATURE MEEM WITH KHAH WITH JEEM INITIAL FORMARABIC LIGATURE MEEM WITH KHAH WITH MEEM INITIAL FORMARABIC LIGATURE MEEM WITH JEEM WITH KHAH INITIAL FORMARABIC LIGATURE HEH WITH MEEM W\
    \ITH JEEM INITIAL FORMARABIC LIGATURE HEH WITH MEEM WITH MEEM INITIAL FORMARABIC LIGATURE NOON WITH HAH WITH MEEM INITIAL FORMARABIC LIGATURE NOON WITH HAH WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE NOON WITH JEEM WITH MEEM FINAL FORMARABIC LIGATURE NOON \
    \WITH JEEM WITH MEEM INITIAL FORMARABIC LIGATURE NOON WITH JEEM WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE NOON WITH MEEM WITH YEH FINAL FORMARABIC LIGATURE NOON WITH MEEM WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE YEH WITH MEEM WITH MEEM FINAL FORMARABI\
    \C LIGATURE YEH WITH MEEM WITH MEEM INITIAL FORMARABIC LIGATURE BEH WITH KHAH WITH YEH FINAL FORMARABIC LIGATURE TEH WITH JEEM WITH YEH FINAL FORMARABIC LIGATURE TEH WITH JEEM WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE TEH WITH KHAH WITH YEH FINAL FORMARA\
    \BIC LIGATURE TEH WITH KHAH WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE TEH WITH MEEM WITH YEH FINAL FORMARABIC LIGATURE TEH WITH MEEM WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE JEEM WITH MEEM WITH YEH FINAL FORMARABIC LIGATURE JEEM WITH HAH WITH ALEF MAK\
    \SURA FINAL FORMARABIC LIGATURE JEEM WITH MEEM WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE SEEN WITH KHAH WITH ALEF MAKSURA FINAL FORMARABIC LIGATURE SAD WITH HAH WITH YEH FINAL FORMARABIC LIGATURE SHEEN WITH HAH WITH YEH FINAL FORMARABIC LIGATURE DAD WITH\
    \ HAH WITH YEH FINAL FORMARABIC LIGATURE LAM WITH JEEM WITH YEH FINAL FORMARABIC LIGATURE LAM WITH MEEM WITH YEH FINAL FORMARABIC LIGATURE YEH WITH HAH WITH YEH FINAL FORMARABIC LIGATURE YEH WITH JEEM WITH YEH FINAL FORMARABIC LIGATURE YEH WITH MEEM WITH Y\
    \EH FINAL FORMARABIC LIGATURE MEEM WITH MEEM WITH YEH FINAL FORMARABIC LIGATURE QAF WITH MEEM WITH YEH FINAL FORMARABIC LIGATURE NOON WITH HAH WITH YEH FINAL FORMARABIC LIGATURE QAF WITH MEEM WITH HAH INITIAL FORMARABIC LIGATURE LAM WITH HAH WITH MEEM INIT\
    \IAL FORMARABIC LIGATURE AIN WITH MEEM WITH YEH FINAL FORMARABIC LIGATURE KAF WITH MEEM WITH YEH FINAL FORMARABIC LIGATURE NOON WITH JEEM WITH HAH INITIAL FORMARABIC LIGATURE MEEM WITH KHAH WITH YEH FINAL FORMARABIC LIGATURE LAM WITH JEEM WITH MEEM INITIAL\
    \ FORMARABIC LIGATURE KAF WITH MEEM WITH MEEM FINAL FORMARABIC LIGATURE LAM WITH JEEM WITH MEEM FINAL FORMARABIC LIGATURE NOON WITH JEEM WITH HAH FINAL FORMARABIC LIGATURE JEEM WITH HAH WITH YEH FINAL FORMARABIC LIGATURE HAH WITH JEEM WITH YEH FINAL FORMAR\
    \ABIC LIGATURE MEEM WITH JEEM WITH YEH FINAL FORMARABIC LIGATURE FEH WITH MEEM WITH YEH FINAL FORMARABIC LIGATURE BEH WITH HAH WITH YEH FINAL FORMARABIC LIGATURE KAF WITH MEEM WITH MEEM INITIAL FORMARABIC LIGATURE AIN WITH JEEM WITH MEEM INITIAL FORMARABIC\
    \ LIGATURE SAD WITH MEEM WITH MEEM INITIAL FORMARABIC LIGATURE SEEN WITH KHAH WITH YEH FINAL FORMARABIC LIGATURE NOON WITH JEEM WITH YEH FINAL FORMARABIC LIGATURE SALAAMUHU ALAYNAAARABIC LIGATURE SALLA USED AS KORANIC STOP SIGN ISOLATED FORMARABIC LIGATURE\
    \ QALA USED AS KORANIC STOP SIGN ISOLATED FORMARABIC LIGATURE ALLAH ISOLATED FORMARABIC LIGATURE AKBAR ISOLATED FORMARABIC LIGATURE MOHAMMAD ISOLATED FORMARABIC LIGATURE SALAM ISOLATED FORMARABIC LIGATURE RASOUL ISOLATED FORMARABIC LIGATURE ALAYHE ISOLATED\
    \ FORMARABIC LIGATURE WASALLAM ISOLATED FORMARABIC LIGATURE SALLA ISOLATED FORMARABIC LIGATURE SALLALLAHOU ALAYHE WASALLAMARABIC LIGATURE JALLAJALALOUHOURIAL SIGNARABIC LIGATURE BISMILLAH AR-RAHMAN AR-RAHEEMARABIC LIGATURE SUBHAANAHU WA TAAALAAARABIC LIGAT\
    \URE AZZA WA JALLVARIATION SELECTOR-1VARIATION SELECTOR-2VARIATION SELECTOR-3VARIATION SELECTOR-4VARIATION SELECTOR-5VARIATION SELECTOR-6VARIATION SELECTOR-7VARIATION SELECTOR-8VARIATION SELECTOR-9VARIATION SELECTOR-10VARIATION SELECTOR-11VARIATION SELECTO\
    \R-12VARIATION SELECTOR-13VARIATION SELECTOR-14VARIATION SELECTOR-15VARIATION SELECTOR-16PRESENTATION FORM FOR VERTICAL COMMAPRESENTATION FORM FOR VERTICAL IDEOGRAPHIC COMMAPRESENTATION FORM FOR VERTICAL IDEOGRAPHIC FULL STOPPRESENTATION FORM FOR VERTICAL \
    \COLONPRESENTATION FORM FOR VERTICAL SEMICOLONPRESENTATION FORM FOR VERTICAL EXCLAMATION MARKPRESENTATION FORM FOR VERTICAL QUESTION MARKPRESENTATION FORM FOR VERTICAL LEFT WHITE LENTICULAR BRACKETPRESENTATION FORM FOR VERTICAL RIGHT WHITE LENTICULAR BRAKC\
    \ETPRESENTATION FORM FOR VERTICAL HORIZONTAL ELLIPSISCOMBINING LIGATURE LEFT HALFCOMBINING LIGATURE RIGHT HALFCOMBINING DOUBLE TILDE LEFT HALFCOMBINING DOUBLE TILDE RIGHT HALFCOMBINING MACRON LEFT HALFCOMBINING MACRON RIGHT HALFCOMBINING CONJOINING MACRONC\
    \OMBINING LIGATURE LEFT HALF BELOWCOMBINING LIGATURE RIGHT HALF BELOWCOMBINING TILDE LEFT HALF BELOWCOMBINING TILDE RIGHT HALF BELOWCOMBINING MACRON LEFT HALF BELOWCOMBINING MACRON RIGHT HALF BELOWCOMBINING CONJOINING MACRON BELOWCOMBINING CYRILLIC TITLO L\
    \EFT HALFCOMBINING CYRILLIC TITLO RIGHT HALFPRESENTATION FORM FOR VERTICAL TWO DOT LEADERPRESENTATION FORM FOR VERTICAL EM DASHPRESENTATION FORM FOR VERTICAL EN DASHPRESENTATION FORM FOR VERTICAL LOW LINEPRESENTATION FORM FOR VERTICAL WAVY LOW LINEPRESENTA\
    \TION FORM FOR VERTICAL LEFT PARENTHESISPRESENTATION FORM FOR VERTICAL RIGHT PARENTHESISPRESENTATION FORM FOR VERTICAL LEFT CURLY BRACKETPRESENTATION FORM FOR VERTICAL RIGHT CURLY BRACKETPRESENTATION FORM FOR VERTICAL LEFT TORTOISE SHELL BRACKETPRESENTATIO\
    \N FORM FOR VERTICAL RIGHT TORTOISE SHELL BRACKETPRESENTATION FORM FOR VERTICAL LEFT BLACK LENTICULAR BRACKETPRESENTATION FORM FOR VERTICAL RIGHT BLACK LENTICULAR BRACKETPRESENTATION FORM FOR VERTICAL LEFT DOUBLE ANGLE BRACKETPRESENTATION FORM FOR VERTICAL\
    \ RIGHT DOUBLE ANGLE BRACKETPRESENTATION FORM FOR VERTICAL LEFT ANGLE BRACKETPRESENTATION FORM FOR VERTICAL RIGHT ANGLE BRACKETPRESENTATION FORM FOR VERTICAL LEFT CORNER BRACKETPRESENTATION FORM FOR VERTICAL RIGHT CORNER BRACKETPRESENTATION FORM FOR VERTIC\
    \AL LEFT WHITE CORNER BRACKETPRESENTATION FORM FOR VERTICAL RIGHT WHITE CORNER BRACKETSESAME DOTWHITE SESAME DOTPRESENTATION FORM FOR VERTICAL LEFT SQUARE BRACKETPRESENTATION FORM FOR VERTICAL RIGHT SQUARE BRACKETDASHED OVERLINECENTRELINE OVERLINEWAVY OVER\
    \LINEDOUBLE WAVY OVERLINEDASHED LOW LINECENTRELINE LOW LINEWAVY LOW LINESMALL COMMASMALL IDEOGRAPHIC COMMASMALL FULL STOPSMALL SEMICOLONSMALL COLONSMALL QUESTION MARKSMALL EXCLAMATION MARKSMALL EM DASHSMALL LEFT PARENTHESISSMALL RIGHT PARENTHESISSMALL LEFT\
    \ CURLY BRACKETSMALL RIGHT CURLY BRACKETSMALL LEFT TORTOISE SHELL BRACKETSMALL RIGHT TORTOISE SHELL BRACKETSMALL NUMBER SIGNSMALL AMPERSANDSMALL ASTERISKSMALL PLUS SIGNSMALL HYPHEN-MINUSSMALL LESS-THAN SIGNSMALL GREATER-THAN SIGNSMALL EQUALS SIGNSMALL REVE\
    \RSE SOLIDUSSMALL DOLLAR SIGNSMALL PERCENT SIGNSMALL COMMERCIAL ATARABIC FATHATAN ISOLATED FORMARABIC TATWEEL WITH FATHATAN ABOVEARABIC DAMMATAN ISOLATED FORMARABIC TAIL FRAGMENTARABIC KASRATAN ISOLATED FORMARABIC FATHA ISOLATED FORMARABIC FATHA MEDIAL FOR\
    \MARABIC DAMMA ISOLATED FORMARABIC DAMMA MEDIAL FORMARABIC KASRA ISOLATED FORMARABIC KASRA MEDIAL FORMARABIC SHADDA ISOLATED FORMARABIC SHADDA MEDIAL FORMARABIC SUKUN ISOLATED FORMARABIC SUKUN MEDIAL FORMARABIC LETTER HAMZA ISOLATED FORMARABIC LETTER ALEF \
    \WITH MADDA ABOVE ISOLATED FORMARABIC LETTER ALEF WITH MADDA ABOVE FINAL FORMARABIC LETTER ALEF WITH HAMZA ABOVE ISOLATED FORMARABIC LETTER ALEF WITH HAMZA ABOVE FINAL FORMARABIC LETTER WAW WITH HAMZA ABOVE ISOLATED FORMARABIC LETTER WAW WITH HAMZA ABOVE F\
    \INAL FORMARABIC LETTER ALEF WITH HAMZA BELOW ISOLATED FORMARABIC LETTER ALEF WITH HAMZA BELOW FINAL FORMARABIC LETTER YEH WITH HAMZA ABOVE ISOLATED FORMARABIC LETTER YEH WITH HAMZA ABOVE FINAL FORMARABIC LETTER YEH WITH HAMZA ABOVE INITIAL FORMARABIC LETT\
    \ER YEH WITH HAMZA ABOVE MEDIAL FORMARABIC LETTER ALEF ISOLATED FORMARABIC LETTER ALEF FINAL FORMARABIC LETTER BEH ISOLATED FORMARABIC LETTER BEH FINAL FORMARABIC LETTER BEH INITIAL FORMARABIC LETTER BEH MEDIAL FORMARABIC LETTER TEH MARBUTA ISOLATED FORMAR\
    \ABIC LETTER TEH MARBUTA FINAL FORMARABIC LETTER TEH ISOLATED FORMARABIC LETTER TEH FINAL FORMARABIC LETTER TEH INITIAL FORMARABIC LETTER TEH MEDIAL FORMARABIC LETTER THEH ISOLATED FORMARABIC LETTER THEH FINAL FORMARABIC LETTER THEH INITIAL FORMARABIC LETT\
    \ER THEH MEDIAL FORMARABIC LETTER JEEM ISOLATED FORMARABIC LETTER JEEM FINAL FORMARABIC LETTER JEEM INITIAL FORMARABIC LETTER JEEM MEDIAL FORMARABIC LETTER HAH ISOLATED FORMARABIC LETTER HAH FINAL FORMARABIC LETTER HAH INITIAL FORMARABIC LETTER HAH MEDIAL \
    \FORMARABIC LETTER KHAH ISOLATED FORMARABIC LETTER KHAH FINAL FORMARABIC LETTER KHAH INITIAL FORMARABIC LETTER KHAH MEDIAL FORMARABIC LETTER DAL ISOLATED FORMARABIC LETTER DAL FINAL FORMARABIC LETTER THAL ISOLATED FORMARABIC LETTER THAL FINAL FORMARABIC LE\
    \TTER REH ISOLATED FORMARABIC LETTER REH FINAL FORMARABIC LETTER ZAIN ISOLATED FORMARABIC LETTER ZAIN FINAL FORMARABIC LETTER SEEN ISOLATED FORMARABIC LETTER SEEN FINAL FORMARABIC LETTER SEEN INITIAL FORMARABIC LETTER SEEN MEDIAL FORMARABIC LETTER SHEEN IS\
    \OLATED FORMARABIC LETTER SHEEN FINAL FORMARABIC LETTER SHEEN INITIAL FORMARABIC LETTER SHEEN MEDIAL FORMARABIC LETTER SAD ISOLATED FORMARABIC LETTER SAD FINAL FORMARABIC LETTER SAD INITIAL FORMARABIC LETTER SAD MEDIAL FORMARABIC LETTER DAD ISOLATED FORMAR\
    \ABIC LETTER DAD FINAL FORMARABIC LETTER DAD INITIAL FORMARABIC LETTER DAD MEDIAL FORMARABIC LETTER TAH ISOLATED FORMARABIC LETTER TAH FINAL FORMARABIC LETTER TAH INITIAL FORMARABIC LETTER TAH MEDIAL FORMARABIC LETTER ZAH ISOLATED FORMARABIC LETTER ZAH FIN\
    \AL FORMARABIC LETTER ZAH INITIAL FORMARABIC LETTER ZAH MEDIAL FORMARABIC LETTER AIN ISOLATED FORMARABIC LETTER AIN FINAL FORMARABIC LETTER AIN INITIAL FORMARABIC LETTER AIN MEDIAL FORMARABIC LETTER GHAIN ISOLATED FORMARABIC LETTER GHAIN FINAL FORMARABIC L\
    \ETTER GHAIN INITIAL FORMARABIC LETTER GHAIN MEDIAL FORMARABIC LETTER FEH ISOLATED FORMARABIC LETTER FEH FINAL FORMARABIC LETTER FEH INITIAL FORMARABIC LETTER FEH MEDIAL FORMARABIC LETTER QAF ISOLATED FORMARABIC LETTER QAF FINAL FORMARABIC LETTER QAF INITI\
    \AL FORMARABIC LETTER QAF MEDIAL FORMARABIC LETTER KAF ISOLATED FORMARABIC LETTER KAF FINAL FORMARABIC LETTER KAF INITIAL FORMARABIC LETTER KAF MEDIAL FORMARABIC LETTER LAM ISOLATED FORMARABIC LETTER LAM FINAL FORMARABIC LETTER LAM INITIAL FORMARABIC LETTE\
    \R LAM MEDIAL FORMARABIC LETTER MEEM ISOLATED FORMARABIC LETTER MEEM FINAL FORMARABIC LETTER MEEM INITIAL FORMARABIC LETTER MEEM MEDIAL FORMARABIC LETTER NOON ISOLATED FORMARABIC LETTER NOON FINAL FORMARABIC LETTER NOON INITIAL FORMARABIC LETTER NOON MEDIA\
    \L FORMARABIC LETTER HEH ISOLATED FORMARABIC LETTER HEH FINAL FORMARABIC LETTER HEH INITIAL FORMARABIC LETTER HEH MEDIAL FORMARABIC LETTER WAW ISOLATED FORMARABIC LETTER WAW FINAL FORMARABIC LETTER ALEF MAKSURA ISOLATED FORMARABIC LETTER ALEF MAKSURA FINAL\
    \ FORMARABIC LETTER YEH ISOLATED FORMARABIC LETTER YEH FINAL FORMARABIC LETTER YEH INITIAL FORMARABIC LETTER YEH MEDIAL FORMARABIC LIGATURE LAM WITH ALEF WITH MADDA ABOVE ISOLATED FORMARABIC LIGATURE LAM WITH ALEF WITH MADDA ABOVE FINAL FORMARABIC LIGATURE\
    \ LAM WITH ALEF WITH HAMZA ABOVE ISOLATED FORMARABIC LIGATURE LAM WITH ALEF WITH HAMZA ABOVE FINAL FORMARABIC LIGATURE LAM WITH ALEF WITH HAMZA BELOW ISOLATED FORMARABIC LIGATURE LAM WITH ALEF WITH HAMZA BELOW FINAL FORMARABIC LIGATURE LAM WITH ALEF ISOLAT\
    \ED FORMARABIC LIGATURE LAM WITH ALEF FINAL FORMZERO WIDTH NO-BREAK SPACEFULLWIDTH EXCLAMATION MARKFULLWIDTH QUOTATION MARKFULLWIDTH NUMBER SIGNFULLWIDTH DOLLAR SIGNFULLWIDTH PERCENT SIGNFULLWIDTH AMPERSANDFULLWIDTH APOSTROPHEFULLWIDTH LEFT PARENTHESISFULL\
    \WIDTH RIGHT PARENTHESISFULLWIDTH ASTERISKFULLWIDTH PLUS SIGNFULLWIDTH COMMAFULLWIDTH HYPHEN-MINUSFULLWIDTH FULL STOPFULLWIDTH SOLIDUSFULLWIDTH DIGIT ZEROFULLWIDTH DIGIT ONEFULLWIDTH DIGIT TWOFULLWIDTH DIGIT THREEFULLWIDTH DIGIT FOURFULLWIDTH DIGIT FIVEFUL\
    \LWIDTH DIGIT SIXFULLWIDTH DIGIT SEVENFULLWIDTH DIGIT EIGHTFULLWIDTH DIGIT NINEFULLWIDTH COLONFULLWIDTH SEMICOLONFULLWIDTH LESS-THAN SIGNFULLWIDTH EQUALS SIGNFULLWIDTH GREATER-THAN SIGNFULLWIDTH QUESTION MARKFULLWIDTH COMMERCIAL ATFULLWIDTH LATIN CAPITAL L\
    \ETTER AFULLWIDTH LATIN CAPITAL LETTER BFULLWIDTH LATIN CAPITAL LETTER CFULLWIDTH LATIN CAPITAL LETTER DFULLWIDTH LATIN CAPITAL LETTER EFULLWIDTH LATIN CAPITAL LETTER FFULLWIDTH LATIN CAPITAL LETTER GFULLWIDTH LATIN CAPITAL LETTER HFULLWIDTH LATIN CAPITAL \
    \LETTER IFULLWIDTH LATIN CAPITAL LETTER JFULLWIDTH LATIN CAPITAL LETTER KFULLWIDTH LATIN CAPITAL LETTER LFULLWIDTH LATIN CAPITAL LETTER MFULLWIDTH LATIN CAPITAL LETTER NFULLWIDTH LATIN CAPITAL LETTER OFULLWIDTH LATIN CAPITAL LETTER PFULLWIDTH LATIN CAPITAL\
    \ LETTER QFULLWIDTH LATIN CAPITAL LETTER RFULLWIDTH LATIN CAPITAL LETTER SFULLWIDTH LATIN CAPITAL LETTER TFULLWIDTH LATIN CAPITAL LETTER UFULLWIDTH LATIN CAPITAL LETTER VFULLWIDTH LATIN CAPITAL LETTER WFULLWIDTH LATIN CAPITAL LETTER XFULLWIDTH LATIN CAPITA\
    \L LETTER YFULLWIDTH LATIN CAPITAL LETTER ZFULLWIDTH LEFT SQUARE BRACKETFULLWIDTH REVERSE SOLIDUSFULLWIDTH RIGHT SQUARE BRACKETFULLWIDTH CIRCUMFLEX ACCENTFULLWIDTH LOW LINEFULLWIDTH GRAVE ACCENTFULLWIDTH LATIN SMALL LETTER AFULLWIDTH LATIN SMALL LETTER BFU\
    \LLWIDTH LATIN SMALL LETTER CFULLWIDTH LATIN SMALL LETTER DFULLWIDTH LATIN SMALL LETTER EFULLWIDTH LATIN SMALL LETTER FFULLWIDTH LATIN SMALL LETTER GFULLWIDTH LATIN SMALL LETTER HFULLWIDTH LATIN SMALL LETTER IFULLWIDTH LATIN SMALL LETTER JFULLWIDTH LATIN S\
    \MALL LETTER KFULLWIDTH LATIN SMALL LETTER LFULLWIDTH LATIN SMALL LETTER MFULLWIDTH LATIN SMALL LETTER NFULLWIDTH LATIN SMALL LETTER OFULLWIDTH LATIN SMALL LETTER PFULLWIDTH LATIN SMALL LETTER QFULLWIDTH LATIN SMALL LETTER RFULLWIDTH LATIN SMALL LETTER SFU\
    \LLWIDTH LATIN SMALL LETTER TFULLWIDTH LATIN SMALL LETTER UFULLWIDTH LATIN SMALL LETTER VFULLWIDTH LATIN SMALL LETTER WFULLWIDTH LATIN SMALL LETTER XFULLWIDTH LATIN SMALL LETTER YFULLWIDTH LATIN SMALL LETTER ZFULLWIDTH LEFT CURLY BRACKETFULLWIDTH VERTICAL \
    \LINEFULLWIDTH RIGHT CURLY BRACKETFULLWIDTH TILDEFULLWIDTH LEFT WHITE PARENTHESISFULLWIDTH RIGHT WHITE PARENTHESISHALFWIDTH IDEOGRAPHIC FULL STOPHALFWIDTH LEFT CORNER BRACKETHALFWIDTH RIGHT CORNER BRACKETHALFWIDTH IDEOGRAPHIC COMMAHALFWIDTH KATAKANA MIDDLE\
    \ DOTHALFWIDTH KATAKANA LETTER WOHALFWIDTH KATAKANA LETTER SMALL AHALFWIDTH KATAKANA LETTER SMALL IHALFWIDTH KATAKANA LETTER SMALL UHALFWIDTH KATAKANA LETTER SMALL EHALFWIDTH KATAKANA LETTER SMALL OHALFWIDTH KATAKANA LETTER SMALL YAHALFWIDTH KATAKANA LETTE\
    \R SMALL YUHALFWIDTH KATAKANA LETTER SMALL YOHALFWIDTH KATAKANA LETTER SMALL TUHALFWIDTH KATAKANA-HIRAGANA PROLONGED SOUND MARKHALFWIDTH KATAKANA LETTER AHALFWIDTH KATAKANA LETTER IHALFWIDTH KATAKANA LETTER UHALFWIDTH KATAKANA LETTER EHALFWIDTH KATAKANA LE\
    \TTER OHALFWIDTH KATAKANA LETTER KAHALFWIDTH KATAKANA LETTER KIHALFWIDTH KATAKANA LETTER KUHALFWIDTH KATAKANA LETTER KEHALFWIDTH KATAKANA LETTER KOHALFWIDTH KATAKANA LETTER SAHALFWIDTH KATAKANA LETTER SIHALFWIDTH KATAKANA LETTER SUHALFWIDTH KATAKANA LETTER\
    \ SEHALFWIDTH KATAKANA LETTER SOHALFWIDTH KATAKANA LETTER TAHALFWIDTH KATAKANA LETTER TIHALFWIDTH KATAKANA LETTER TUHALFWIDTH KATAKANA LETTER TEHALFWIDTH KATAKANA LETTER TOHALFWIDTH KATAKANA LETTER NAHALFWIDTH KATAKANA LETTER NIHALFWIDTH KATAKANA LETTER NU\
    \HALFWIDTH KATAKANA LETTER NEHALFWIDTH KATAKANA LETTER NOHALFWIDTH KATAKANA LETTER HAHALFWIDTH KATAKANA LETTER HIHALFWIDTH KATAKANA LETTER HUHALFWIDTH KATAKANA LETTER HEHALFWIDTH KATAKANA LETTER HOHALFWIDTH KATAKANA LETTER MAHALFWIDTH KATAKANA LETTER MIHAL\
    \FWIDTH KATAKANA LETTER MUHALFWIDTH KATAKANA LETTER MEHALFWIDTH KATAKANA LETTER MOHALFWIDTH KATAKANA LETTER YAHALFWIDTH KATAKANA LETTER YUHALFWIDTH KATAKANA LETTER YOHALFWIDTH KATAKANA LETTER RAHALFWIDTH KATAKANA LETTER RIHALFWIDTH KATAKANA LETTER RUHALFWI\
    \DTH KATAKANA LETTER REHALFWIDTH KATAKANA LETTER ROHALFWIDTH KATAKANA LETTER WAHALFWIDTH KATAKANA LETTER NHALFWIDTH KATAKANA VOICED SOUND MARKHALFWIDTH KATAKANA SEMI-VOICED SOUND MARKHALFWIDTH HANGUL FILLERHALFWIDTH HANGUL LETTER KIYEOKHALFWIDTH HANGUL LET\
    \TER SSANGKIYEOKHALFWIDTH HANGUL LETTER KIYEOK-SIOSHALFWIDTH HANGUL LETTER NIEUNHALFWIDTH HANGUL LETTER NIEUN-CIEUCHALFWIDTH HANGUL LETTER NIEUN-HIEUHHALFWIDTH HANGUL LETTER TIKEUTHALFWIDTH HANGUL LETTER SSANGTIKEUTHALFWIDTH HANGUL LETTER RIEULHALFWIDTH HA\
    \NGUL LETTER RIEUL-KIYEOKHALFWIDTH HANGUL LETTER RIEUL-MIEUMHALFWIDTH HANGUL LETTER RIEUL-PIEUPHALFWIDTH HANGUL LETTER RIEUL-SIOSHALFWIDTH HANGUL LETTER RIEUL-THIEUTHHALFWIDTH HANGUL LETTER RIEUL-PHIEUPHHALFWIDTH HANGUL LETTER RIEUL-HIEUHHALFWIDTH HANGUL L\
    \ETTER MIEUMHALFWIDTH HANGUL LETTER PIEUPHALFWIDTH HANGUL LETTER SSANGPIEUPHALFWIDTH HANGUL LETTER PIEUP-SIOSHALFWIDTH HANGUL LETTER SIOSHALFWIDTH HANGUL LETTER SSANGSIOSHALFWIDTH HANGUL LETTER IEUNGHALFWIDTH HANGUL LETTER CIEUCHALFWIDTH HANGUL LETTER SSAN\
    \GCIEUCHALFWIDTH HANGUL LETTER CHIEUCHHALFWIDTH HANGUL LETTER KHIEUKHHALFWIDTH HANGUL LETTER THIEUTHHALFWIDTH HANGUL LETTER PHIEUPHHALFWIDTH HANGUL LETTER HIEUHHALFWIDTH HANGUL LETTER AHALFWIDTH HANGUL LETTER AEHALFWIDTH HANGUL LETTER YAHALFWIDTH HANGUL LE\
    \TTER YAEHALFWIDTH HANGUL LETTER EOHALFWIDTH HANGUL LETTER EHALFWIDTH HANGUL LETTER YEOHALFWIDTH HANGUL LETTER YEHALFWIDTH HANGUL LETTER OHALFWIDTH HANGUL LETTER WAHALFWIDTH HANGUL LETTER WAEHALFWIDTH HANGUL LETTER OEHALFWIDTH HANGUL LETTER YOHALFWIDTH HAN\
    \GUL LETTER UHALFWIDTH HANGUL LETTER WEOHALFWIDTH HANGUL LETTER WEHALFWIDTH HANGUL LETTER WIHALFWIDTH HANGUL LETTER YUHALFWIDTH HANGUL LETTER EUHALFWIDTH HANGUL LETTER YIHALFWIDTH HANGUL LETTER IFULLWIDTH CENT SIGNFULLWIDTH POUND SIGNFULLWIDTH NOT SIGNFULL\
    \WIDTH MACRONFULLWIDTH BROKEN BARFULLWIDTH YEN SIGNFULLWIDTH WON SIGNHALFWIDTH FORMS LIGHT VERTICALHALFWIDTH LEFTWARDS ARROWHALFWIDTH UPWARDS ARROWHALFWIDTH RIGHTWARDS ARROWHALFWIDTH DOWNWARDS ARROWHALFWIDTH BLACK SQUAREHALFWIDTH WHITE CIRCLEINTERLINEAR AN\
    \NOTATION ANCHORINTERLINEAR ANNOTATION SEPARATORINTERLINEAR ANNOTATION TERMINATOROBJECT REPLACEMENT CHARACTERREPLACEMENT CHARACTERLINEAR B SYLLABLE B008 ALINEAR B SYLLABLE B038 ELINEAR B SYLLABLE B028 ILINEAR B SYLLABLE B061 OLINEAR B SYLLABLE B010 ULINEAR\
    \ B SYLLABLE B001 DALINEAR B SYLLABLE B045 DELINEAR B SYLLABLE B007 DILINEAR B SYLLABLE B014 DOLINEAR B SYLLABLE B051 DULINEAR B SYLLABLE B057 JALINEAR B SYLLABLE B046 JELINEAR B SYLLABLE B036 JOLINEAR B SYLLABLE B065 JULINEAR B SYLLABLE B077 KALINEAR B SY\
    \LLABLE B044 KELINEAR B SYLLABLE B067 KILINEAR B SYLLABLE B070 KOLINEAR B SYLLABLE B081 KULINEAR B SYLLABLE B080 MALINEAR B SYLLABLE B013 MELINEAR B SYLLABLE B073 MILINEAR B SYLLABLE B015 MOLINEAR B SYLLABLE B023 MULINEAR B SYLLABLE B006 NALINEAR B SYLLABL\
    \E B024 NELINEAR B SYLLABLE B030 NILINEAR B SYLLABLE B052 NOLINEAR B SYLLABLE B055 NULINEAR B SYLLABLE B003 PALINEAR B SYLLABLE B072 PELINEAR B SYLLABLE B039 PILINEAR B SYLLABLE B011 POLINEAR B SYLLABLE B050 PULINEAR B SYLLABLE B016 QALINEAR B SYLLABLE B07\
    \8 QELINEAR B SYLLABLE B021 QILINEAR B SYLLABLE B032 QOLINEAR B SYLLABLE B060 RALINEAR B SYLLABLE B027 RELINEAR B SYLLABLE B053 RILINEAR B SYLLABLE B002 ROLINEAR B SYLLABLE B026 RULINEAR B SYLLABLE B031 SALINEAR B SYLLABLE B009 SELINEAR B SYLLABLE B041 SIL\
    \INEAR B SYLLABLE B012 SOLINEAR B SYLLABLE B058 SULINEAR B SYLLABLE B059 TALINEAR B SYLLABLE B004 TELINEAR B SYLLABLE B037 TILINEAR B SYLLABLE B005 TOLINEAR B SYLLABLE B069 TULINEAR B SYLLABLE B054 WALINEAR B SYLLABLE B075 WELINEAR B SYLLABLE B040 WILINEAR\
    \ B SYLLABLE B042 WOLINEAR B SYLLABLE B017 ZALINEAR B SYLLABLE B074 ZELINEAR B SYLLABLE B020 ZOLINEAR B SYLLABLE B025 A2LINEAR B SYLLABLE B043 A3LINEAR B SYLLABLE B085 AULINEAR B SYLLABLE B071 DWELINEAR B SYLLABLE B090 DWOLINEAR B SYLLABLE B048 NWALINEAR B\
    \ SYLLABLE B029 PU2LINEAR B SYLLABLE B062 PTELINEAR B SYLLABLE B076 RA2LINEAR B SYLLABLE B033 RA3LINEAR B SYLLABLE B068 RO2LINEAR B SYLLABLE B066 TA2LINEAR B SYLLABLE B087 TWELINEAR B SYLLABLE B091 TWOLINEAR B SYMBOL B018LINEAR B SYMBOL B019LINEAR B SYMBOL\
    \ B022LINEAR B SYMBOL B034LINEAR B SYMBOL B047LINEAR B SYMBOL B049LINEAR B SYMBOL B056LINEAR B SYMBOL B063LINEAR B SYMBOL B064LINEAR B SYMBOL B079LINEAR B SYMBOL B082LINEAR B SYMBOL B083LINEAR B SYMBOL B086LINEAR B SYMBOL B089LINEAR B IDEOGRAM B100 MANLINE\
    \AR B IDEOGRAM B102 WOMANLINEAR B IDEOGRAM B104 DEERLINEAR B IDEOGRAM B105 EQUIDLINEAR B IDEOGRAM B105F MARELINEAR B IDEOGRAM B105M STALLIONLINEAR B IDEOGRAM B106F EWELINEAR B IDEOGRAM B106M RAMLINEAR B IDEOGRAM B107F SHE-GOATLINEAR B IDEOGRAM B107M HE-GOA\
    \TLINEAR B IDEOGRAM B108F SOWLINEAR B IDEOGRAM B108M BOARLINEAR B IDEOGRAM B109F COWLINEAR B IDEOGRAM B109M BULLLINEAR B IDEOGRAM B120 WHEATLINEAR B IDEOGRAM B121 BARLEYLINEAR B IDEOGRAM B122 OLIVELINEAR B IDEOGRAM B123 SPICELINEAR B IDEOGRAM B125 CYPERUSL\
    \INEAR B MONOGRAM B127 KAPOLINEAR B MONOGRAM B128 KANAKOLINEAR B IDEOGRAM B130 OILLINEAR B IDEOGRAM B131 WINELINEAR B IDEOGRAM B132LINEAR B MONOGRAM B133 AREPALINEAR B MONOGRAM B135 MERILINEAR B IDEOGRAM B140 BRONZELINEAR B IDEOGRAM B141 GOLDLINEAR B IDEOG\
    \RAM B142LINEAR B IDEOGRAM B145 WOOLLINEAR B IDEOGRAM B146LINEAR B IDEOGRAM B150LINEAR B IDEOGRAM B151 HORNLINEAR B IDEOGRAM B152LINEAR B IDEOGRAM B153LINEAR B IDEOGRAM B154LINEAR B MONOGRAM B156 TURO2LINEAR B IDEOGRAM B157LINEAR B IDEOGRAM B158LINEAR B ID\
    \EOGRAM B159 CLOTHLINEAR B IDEOGRAM B160LINEAR B IDEOGRAM B161LINEAR B IDEOGRAM B162 GARMENTLINEAR B IDEOGRAM B163 ARMOURLINEAR B IDEOGRAM B164LINEAR B IDEOGRAM B165LINEAR B IDEOGRAM B166LINEAR B IDEOGRAM B167LINEAR B IDEOGRAM B168LINEAR B IDEOGRAM B169LIN\
    \EAR B IDEOGRAM B170LINEAR B IDEOGRAM B171LINEAR B IDEOGRAM B172LINEAR B IDEOGRAM B173 MONTHLINEAR B IDEOGRAM B174LINEAR B IDEOGRAM B176 TREELINEAR B IDEOGRAM B177LINEAR B IDEOGRAM B178LINEAR B IDEOGRAM B179LINEAR B IDEOGRAM B180LINEAR B IDEOGRAM B181LINEA\
    \R B IDEOGRAM B182LINEAR B IDEOGRAM B183LINEAR B IDEOGRAM B184LINEAR B IDEOGRAM B185LINEAR B IDEOGRAM B189LINEAR B IDEOGRAM B190LINEAR B IDEOGRAM B191 HELMETLINEAR B IDEOGRAM B220 FOOTSTOOLLINEAR B IDEOGRAM B225 BATHTUBLINEAR B IDEOGRAM B230 SPEARLINEAR B \
    \IDEOGRAM B231 ARROWLINEAR B IDEOGRAM B232LINEAR B IDEOGRAM B233 SWORDLINEAR B IDEOGRAM B234LINEAR B IDEOGRAM B236LINEAR B IDEOGRAM B240 WHEELED CHARIOTLINEAR B IDEOGRAM B241 CHARIOTLINEAR B IDEOGRAM B242 CHARIOT FRAMELINEAR B IDEOGRAM B243 WHEELLINEAR B I\
    \DEOGRAM B245LINEAR B IDEOGRAM B246LINEAR B MONOGRAM B247 DIPTELINEAR B IDEOGRAM B248LINEAR B IDEOGRAM B249LINEAR B IDEOGRAM B251LINEAR B IDEOGRAM B252LINEAR B IDEOGRAM B253LINEAR B IDEOGRAM B254 DARTLINEAR B IDEOGRAM B255LINEAR B IDEOGRAM B256LINEAR B IDE\
    \OGRAM B257LINEAR B IDEOGRAM B258LINEAR B IDEOGRAM B259LINEAR B IDEOGRAM VESSEL B155LINEAR B IDEOGRAM VESSEL B200LINEAR B IDEOGRAM VESSEL B201LINEAR B IDEOGRAM VESSEL B202LINEAR B IDEOGRAM VESSEL B203LINEAR B IDEOGRAM VESSEL B204LINEAR B IDEOGRAM VESSEL B2\
    \05LINEAR B IDEOGRAM VESSEL B206LINEAR B IDEOGRAM VESSEL B207LINEAR B IDEOGRAM VESSEL B208LINEAR B IDEOGRAM VESSEL B209LINEAR B IDEOGRAM VESSEL B210LINEAR B IDEOGRAM VESSEL B211LINEAR B IDEOGRAM VESSEL B212LINEAR B IDEOGRAM VESSEL B213LINEAR B IDEOGRAM VES\
    \SEL B214LINEAR B IDEOGRAM VESSEL B215LINEAR B IDEOGRAM VESSEL B216LINEAR B IDEOGRAM VESSEL B217LINEAR B IDEOGRAM VESSEL B218LINEAR B IDEOGRAM VESSEL B219LINEAR B IDEOGRAM VESSEL B221LINEAR B IDEOGRAM VESSEL B222LINEAR B IDEOGRAM VESSEL B226LINEAR B IDEOGR\
    \AM VESSEL B227LINEAR B IDEOGRAM VESSEL B228LINEAR B IDEOGRAM VESSEL B229LINEAR B IDEOGRAM VESSEL B250LINEAR B IDEOGRAM VESSEL B305AEGEAN WORD SEPARATOR LINEAEGEAN WORD SEPARATOR DOTAEGEAN CHECK MARKAEGEAN NUMBER ONEAEGEAN NUMBER TWOAEGEAN NUMBER THREEAEGE\
    \AN NUMBER FOURAEGEAN NUMBER FIVEAEGEAN NUMBER SIXAEGEAN NUMBER SEVENAEGEAN NUMBER EIGHTAEGEAN NUMBER NINEAEGEAN NUMBER TENAEGEAN NUMBER TWENTYAEGEAN NUMBER THIRTYAEGEAN NUMBER FORTYAEGEAN NUMBER FIFTYAEGEAN NUMBER SIXTYAEGEAN NUMBER SEVENTYAEGEAN NUMBER E\
    \IGHTYAEGEAN NUMBER NINETYAEGEAN NUMBER ONE HUNDREDAEGEAN NUMBER TWO HUNDREDAEGEAN NUMBER THREE HUNDREDAEGEAN NUMBER FOUR HUNDREDAEGEAN NUMBER FIVE HUNDREDAEGEAN NUMBER SIX HUNDREDAEGEAN NUMBER SEVEN HUNDREDAEGEAN NUMBER EIGHT HUNDREDAEGEAN NUMBER NINE HUN\
    \DREDAEGEAN NUMBER ONE THOUSANDAEGEAN NUMBER TWO THOUSANDAEGEAN NUMBER THREE THOUSANDAEGEAN NUMBER FOUR THOUSANDAEGEAN NUMBER FIVE THOUSANDAEGEAN NUMBER SIX THOUSANDAEGEAN NUMBER SEVEN THOUSANDAEGEAN NUMBER EIGHT THOUSANDAEGEAN NUMBER NINE THOUSANDAEGEAN N\
    \UMBER TEN THOUSANDAEGEAN NUMBER TWENTY THOUSANDAEGEAN NUMBER THIRTY THOUSANDAEGEAN NUMBER FORTY THOUSANDAEGEAN NUMBER FIFTY THOUSANDAEGEAN NUMBER SIXTY THOUSANDAEGEAN NUMBER SEVENTY THOUSANDAEGEAN NUMBER EIGHTY THOUSANDAEGEAN NUMBER NINETY THOUSANDAEGEAN \
    \WEIGHT BASE UNITAEGEAN WEIGHT FIRST SUBUNITAEGEAN WEIGHT SECOND SUBUNITAEGEAN WEIGHT THIRD SUBUNITAEGEAN WEIGHT FOURTH SUBUNITAEGEAN DRY MEASURE FIRST SUBUNITAEGEAN LIQUID MEASURE FIRST SUBUNITAEGEAN MEASURE SECOND SUBUNITAEGEAN MEASURE THIRD SUBUNITGREEK\
    \ ACROPHONIC ATTIC ONE QUARTERGREEK ACROPHONIC ATTIC ONE HALFGREEK ACROPHONIC ATTIC ONE DRACHMAGREEK ACROPHONIC ATTIC FIVEGREEK ACROPHONIC ATTIC FIFTYGREEK ACROPHONIC ATTIC FIVE HUNDREDGREEK ACROPHONIC ATTIC FIVE THOUSANDGREEK ACROPHONIC ATTIC FIFTY THOUSA\
    \NDGREEK ACROPHONIC ATTIC FIVE TALENTSGREEK ACROPHONIC ATTIC TEN TALENTSGREEK ACROPHONIC ATTIC FIFTY TALENTSGREEK ACROPHONIC ATTIC ONE HUNDRED TALENTSGREEK ACROPHONIC ATTIC FIVE HUNDRED TALENTSGREEK ACROPHONIC ATTIC ONE THOUSAND TALENTSGREEK ACROPHONIC ATT\
    \IC FIVE THOUSAND TALENTSGREEK ACROPHONIC ATTIC FIVE STATERSGREEK ACROPHONIC ATTIC TEN STATERSGREEK ACROPHONIC ATTIC FIFTY STATERSGREEK ACROPHONIC ATTIC ONE HUNDRED STATERSGREEK ACROPHONIC ATTIC FIVE HUNDRED STATERSGREEK ACROPHONIC ATTIC ONE THOUSAND STATE\
    \RSGREEK ACROPHONIC ATTIC TEN THOUSAND STATERSGREEK ACROPHONIC ATTIC FIFTY THOUSAND STATERSGREEK ACROPHONIC ATTIC TEN MNASGREEK ACROPHONIC HERAEUM ONE PLETHRONGREEK ACROPHONIC THESPIAN ONEGREEK ACROPHONIC HERMIONIAN ONEGREEK ACROPHONIC EPIDAUREAN TWOGREEK \
    \ACROPHONIC THESPIAN TWOGREEK ACROPHONIC CYRENAIC TWO DRACHMASGREEK ACROPHONIC EPIDAUREAN TWO DRACHMASGREEK ACROPHONIC TROEZENIAN FIVEGREEK ACROPHONIC TROEZENIAN TENGREEK ACROPHONIC TROEZENIAN TEN ALTERNATE FORMGREEK ACROPHONIC HERMIONIAN TENGREEK ACROPHON\
    \IC MESSENIAN TENGREEK ACROPHONIC THESPIAN TENGREEK ACROPHONIC THESPIAN THIRTYGREEK ACROPHONIC TROEZENIAN FIFTYGREEK ACROPHONIC TROEZENIAN FIFTY ALTERNATE FORMGREEK ACROPHONIC HERMIONIAN FIFTYGREEK ACROPHONIC THESPIAN FIFTYGREEK ACROPHONIC THESPIAN ONE HUN\
    \DREDGREEK ACROPHONIC THESPIAN THREE HUNDREDGREEK ACROPHONIC EPIDAUREAN FIVE HUNDREDGREEK ACROPHONIC TROEZENIAN FIVE HUNDREDGREEK ACROPHONIC THESPIAN FIVE HUNDREDGREEK ACROPHONIC CARYSTIAN FIVE HUNDREDGREEK ACROPHONIC NAXIAN FIVE HUNDREDGREEK ACROPHONIC TH\
    \ESPIAN ONE THOUSANDGREEK ACROPHONIC THESPIAN FIVE THOUSANDGREEK ACROPHONIC DELPHIC FIVE MNASGREEK ACROPHONIC STRATIAN FIFTY MNASGREEK ONE HALF SIGNGREEK ONE HALF SIGN ALTERNATE FORMGREEK TWO THIRDS SIGNGREEK THREE QUARTERS SIGNGREEK YEAR SIGNGREEK TALENT \
    \SIGNGREEK DRACHMA SIGNGREEK OBOL SIGNGREEK TWO OBOLS SIGNGREEK THREE OBOLS SIGNGREEK FOUR OBOLS SIGNGREEK FIVE OBOLS SIGNGREEK METRETES SIGNGREEK KYATHOS BASE SIGNGREEK LITRA SIGNGREEK OUNKIA SIGNGREEK XESTES SIGNGREEK ARTABE SIGNGREEK AROURA SIGNGREEK GR\
    \AMMA SIGNGREEK TRYBLION BASE SIGNGREEK ZERO SIGNGREEK ONE QUARTER SIGNGREEK SINUSOID SIGNGREEK INDICTION SIGNNOMISMA SIGNROMAN SEXTANS SIGNROMAN UNCIA SIGNROMAN SEMUNCIA SIGNROMAN SEXTULA SIGNROMAN DIMIDIA SEXTULA SIGNROMAN SILIQUA SIGNROMAN DENARIUS SIGN\
    \ROMAN QUINARIUS SIGNROMAN SESTERTIUS SIGNROMAN DUPONDIUS SIGNROMAN AS SIGNROMAN CENTURIAL SIGNASCIA SYMBOLGREEK SYMBOL TAU RHOPHAISTOS DISC SIGN PEDESTRIANPHAISTOS DISC SIGN PLUMED HEADPHAISTOS DISC SIGN TATTOOED HEADPHAISTOS DISC SIGN CAPTIVEPHAISTOS DIS\
    \C SIGN CHILDPHAISTOS DISC SIGN WOMANPHAISTOS DISC SIGN HELMETPHAISTOS DISC SIGN GAUNTLETPHAISTOS DISC SIGN TIARAPHAISTOS DISC SIGN ARROWPHAISTOS DISC SIGN BOWPHAISTOS DISC SIGN SHIELDPHAISTOS DISC SIGN CLUBPHAISTOS DISC SIGN MANACLESPHAISTOS DISC SIGN MAT\
    \TOCKPHAISTOS DISC SIGN SAWPHAISTOS DISC SIGN LIDPHAISTOS DISC SIGN BOOMERANGPHAISTOS DISC SIGN CARPENTRY PLANEPHAISTOS DISC SIGN DOLIUMPHAISTOS DISC SIGN COMBPHAISTOS DISC SIGN SLINGPHAISTOS DISC SIGN COLUMNPHAISTOS DISC SIGN BEEHIVEPHAISTOS DISC SIGN SHI\
    \PPHAISTOS DISC SIGN HORNPHAISTOS DISC SIGN HIDEPHAISTOS DISC SIGN BULLS LEGPHAISTOS DISC SIGN CATPHAISTOS DISC SIGN RAMPHAISTOS DISC SIGN EAGLEPHAISTOS DISC SIGN DOVEPHAISTOS DISC SIGN TUNNYPHAISTOS DISC SIGN BEEPHAISTOS DISC SIGN PLANE TREEPHAISTOS DISC \
    \SIGN VINEPHAISTOS DISC SIGN PAPYRUSPHAISTOS DISC SIGN ROSETTEPHAISTOS DISC SIGN LILYPHAISTOS DISC SIGN OX BACKPHAISTOS DISC SIGN FLUTEPHAISTOS DISC SIGN GRATERPHAISTOS DISC SIGN STRAINERPHAISTOS DISC SIGN SMALL AXEPHAISTOS DISC SIGN WAVY BANDPHAISTOS DISC\
    \ SIGN COMBINING OBLIQUE STROKELYCIAN LETTER ALYCIAN LETTER ELYCIAN LETTER BLYCIAN LETTER BHLYCIAN LETTER GLYCIAN LETTER DLYCIAN LETTER ILYCIAN LETTER WLYCIAN LETTER ZLYCIAN LETTER THLYCIAN LETTER JLYCIAN LETTER KLYCIAN LETTER QLYCIAN LETTER LLYCIAN LETTER\
    \ MLYCIAN LETTER NLYCIAN LETTER MMLYCIAN LETTER NNLYCIAN LETTER ULYCIAN LETTER PLYCIAN LETTER KKLYCIAN LETTER RLYCIAN LETTER SLYCIAN LETTER TLYCIAN LETTER TTLYCIAN LETTER ANLYCIAN LETTER ENLYCIAN LETTER HLYCIAN LETTER XCARIAN LETTER ACARIAN LETTER P2CARIAN\
    \ LETTER DCARIAN LETTER LCARIAN LETTER UUUCARIAN LETTER RCARIAN LETTER LDCARIAN LETTER A2CARIAN LETTER QCARIAN LETTER BCARIAN LETTER MCARIAN LETTER OCARIAN LETTER D2CARIAN LETTER TCARIAN LETTER SHCARIAN LETTER SH2CARIAN LETTER SCARIAN LETTER C-18CARIAN LET\
    \TER UCARIAN LETTER NNCARIAN LETTER XCARIAN LETTER NCARIAN LETTER TT2CARIAN LETTER PCARIAN LETTER SSCARIAN LETTER ICARIAN LETTER ECARIAN LETTER UUUUCARIAN LETTER KCARIAN LETTER K2CARIAN LETTER NDCARIAN LETTER UUCARIAN LETTER GCARIAN LETTER G2CARIAN LETTER \
    \STCARIAN LETTER ST2CARIAN LETTER NGCARIAN LETTER IICARIAN LETTER C-39CARIAN LETTER TTCARIAN LETTER UUU2CARIAN LETTER RRCARIAN LETTER MBCARIAN LETTER MB2CARIAN LETTER MB3CARIAN LETTER MB4CARIAN LETTER LD2CARIAN LETTER E2CARIAN LETTER UUU3COPTIC EPACT THOUS\
    \ANDS MARKCOPTIC EPACT DIGIT ONECOPTIC EPACT DIGIT TWOCOPTIC EPACT DIGIT THREECOPTIC EPACT DIGIT FOURCOPTIC EPACT DIGIT FIVECOPTIC EPACT DIGIT SIXCOPTIC EPACT DIGIT SEVENCOPTIC EPACT DIGIT EIGHTCOPTIC EPACT DIGIT NINECOPTIC EPACT NUMBER TENCOPTIC EPACT NUM\
    \BER TWENTYCOPTIC EPACT NUMBER THIRTYCOPTIC EPACT NUMBER FORTYCOPTIC EPACT NUMBER FIFTYCOPTIC EPACT NUMBER SIXTYCOPTIC EPACT NUMBER SEVENTYCOPTIC EPACT NUMBER EIGHTYCOPTIC EPACT NUMBER NINETYCOPTIC EPACT NUMBER ONE HUNDREDCOPTIC EPACT NUMBER TWO HUNDREDCOP\
    \TIC EPACT NUMBER THREE HUNDREDCOPTIC EPACT NUMBER FOUR HUNDREDCOPTIC EPACT NUMBER FIVE HUNDREDCOPTIC EPACT NUMBER SIX HUNDREDCOPTIC EPACT NUMBER SEVEN HUNDREDCOPTIC EPACT NUMBER EIGHT HUNDREDCOPTIC EPACT NUMBER NINE HUNDREDOLD ITALIC LETTER AOLD ITALIC LE\
    \TTER BEOLD ITALIC LETTER KEOLD ITALIC LETTER DEOLD ITALIC LETTER EOLD ITALIC LETTER VEOLD ITALIC LETTER ZEOLD ITALIC LETTER HEOLD ITALIC LETTER THEOLD ITALIC LETTER IOLD ITALIC LETTER KAOLD ITALIC LETTER ELOLD ITALIC LETTER EMOLD ITALIC LETTER ENOLD ITALI\
    \C LETTER ESHOLD ITALIC LETTER OOLD ITALIC LETTER PEOLD ITALIC LETTER SHEOLD ITALIC LETTER KUOLD ITALIC LETTER EROLD ITALIC LETTER ESOLD ITALIC LETTER TEOLD ITALIC LETTER UOLD ITALIC LETTER EKSOLD ITALIC LETTER PHEOLD ITALIC LETTER KHEOLD ITALIC LETTER EFO\
    \LD ITALIC LETTER ERSOLD ITALIC LETTER CHEOLD ITALIC LETTER IIOLD ITALIC LETTER UUOLD ITALIC LETTER ESSOLD ITALIC NUMERAL ONEOLD ITALIC NUMERAL FIVEOLD ITALIC NUMERAL TENOLD ITALIC NUMERAL FIFTYOLD ITALIC LETTER YEOLD ITALIC LETTER NORTHERN TSEOLD ITALIC L\
    \ETTER SOUTHERN TSEGOTHIC LETTER AHSAGOTHIC LETTER BAIRKANGOTHIC LETTER GIBAGOTHIC LETTER DAGSGOTHIC LETTER AIHVUSGOTHIC LETTER QAIRTHRAGOTHIC LETTER IUJAGOTHIC LETTER HAGLGOTHIC LETTER THIUTHGOTHIC LETTER EISGOTHIC LETTER KUSMAGOTHIC LETTER LAGUSGOTHIC LE\
    \TTER MANNAGOTHIC LETTER NAUTHSGOTHIC LETTER JERGOTHIC LETTER URUSGOTHIC LETTER PAIRTHRAGOTHIC LETTER NINETYGOTHIC LETTER RAIDAGOTHIC LETTER SAUILGOTHIC LETTER TEIWSGOTHIC LETTER WINJAGOTHIC LETTER FAIHUGOTHIC LETTER IGGWSGOTHIC LETTER HWAIRGOTHIC LETTER O\
    \THALGOTHIC LETTER NINE HUNDREDOLD PERMIC LETTER ANOLD PERMIC LETTER BUROLD PERMIC LETTER GAIOLD PERMIC LETTER DOIOLD PERMIC LETTER EOLD PERMIC LETTER ZHOIOLD PERMIC LETTER DZHOIOLD PERMIC LETTER ZATAOLD PERMIC LETTER DZITAOLD PERMIC LETTER IOLD PERMIC LET\
    \TER KOKEOLD PERMIC LETTER LEIOLD PERMIC LETTER MENOEOLD PERMIC LETTER NENOEOLD PERMIC LETTER VOOIOLD PERMIC LETTER PEEIOLD PERMIC LETTER REIOLD PERMIC LETTER SIIOLD PERMIC LETTER TAIOLD PERMIC LETTER UOLD PERMIC LETTER CHERYOLD PERMIC LETTER SHOOIOLD PERM\
    \IC LETTER SHCHOOIOLD PERMIC LETTER YRYOLD PERMIC LETTER YERUOLD PERMIC LETTER OOLD PERMIC LETTER OOOLD PERMIC LETTER EFOLD PERMIC LETTER HAOLD PERMIC LETTER TSIUOLD PERMIC LETTER VEROLD PERMIC LETTER YEROLD PERMIC LETTER YERIOLD PERMIC LETTER YATOLD PERMI\
    \C LETTER IEOLD PERMIC LETTER YUOLD PERMIC LETTER YAOLD PERMIC LETTER IACOMBINING OLD PERMIC LETTER ANCOMBINING OLD PERMIC LETTER DOICOMBINING OLD PERMIC LETTER ZATACOMBINING OLD PERMIC LETTER NENOECOMBINING OLD PERMIC LETTER SIIUGARITIC LETTER ALPAUGARITI\
    \C LETTER BETAUGARITIC LETTER GAMLAUGARITIC LETTER KHAUGARITIC LETTER DELTAUGARITIC LETTER HOUGARITIC LETTER WOUGARITIC LETTER ZETAUGARITIC LETTER HOTAUGARITIC LETTER TETUGARITIC LETTER YODUGARITIC LETTER KAFUGARITIC LETTER SHINUGARITIC LETTER LAMDAUGARITI\
    \C LETTER MEMUGARITIC LETTER DHALUGARITIC LETTER NUNUGARITIC LETTER ZUUGARITIC LETTER SAMKAUGARITIC LETTER AINUGARITIC LETTER PUUGARITIC LETTER SADEUGARITIC LETTER QOPAUGARITIC LETTER RASHAUGARITIC LETTER THANNAUGARITIC LETTER GHAINUGARITIC LETTER TOUGARIT\
    \IC LETTER IUGARITIC LETTER UUGARITIC LETTER SSUUGARITIC WORD DIVIDEROLD PERSIAN SIGN AOLD PERSIAN SIGN IOLD PERSIAN SIGN UOLD PERSIAN SIGN KAOLD PERSIAN SIGN KUOLD PERSIAN SIGN GAOLD PERSIAN SIGN GUOLD PERSIAN SIGN XAOLD PERSIAN SIGN CAOLD PERSIAN SIGN JA\
    \OLD PERSIAN SIGN JIOLD PERSIAN SIGN TAOLD PERSIAN SIGN TUOLD PERSIAN SIGN DAOLD PERSIAN SIGN DIOLD PERSIAN SIGN DUOLD PERSIAN SIGN THAOLD PERSIAN SIGN PAOLD PERSIAN SIGN BAOLD PERSIAN SIGN FAOLD PERSIAN SIGN NAOLD PERSIAN SIGN NUOLD PERSIAN SIGN MAOLD PER\
    \SIAN SIGN MIOLD PERSIAN SIGN MUOLD PERSIAN SIGN YAOLD PERSIAN SIGN VAOLD PERSIAN SIGN VIOLD PERSIAN SIGN RAOLD PERSIAN SIGN RUOLD PERSIAN SIGN LAOLD PERSIAN SIGN SAOLD PERSIAN SIGN ZAOLD PERSIAN SIGN SHAOLD PERSIAN SIGN SSAOLD PERSIAN SIGN HAOLD PERSIAN S\
    \IGN AURAMAZDAAOLD PERSIAN SIGN AURAMAZDAA-2OLD PERSIAN SIGN AURAMAZDAAHAOLD PERSIAN SIGN XSHAAYATHIYAOLD PERSIAN SIGN DAHYAAUSHOLD PERSIAN SIGN DAHYAAUSH-2OLD PERSIAN SIGN BAGAOLD PERSIAN SIGN BUUMISHOLD PERSIAN WORD DIVIDEROLD PERSIAN NUMBER ONEOLD PERSI\
    \AN NUMBER TWOOLD PERSIAN NUMBER TENOLD PERSIAN NUMBER TWENTYOLD PERSIAN NUMBER HUNDREDDESERET CAPITAL LETTER LONG IDESERET CAPITAL LETTER LONG EDESERET CAPITAL LETTER LONG ADESERET CAPITAL LETTER LONG AHDESERET CAPITAL LETTER LONG ODESERET CAPITAL LETTER \
    \LONG OODESERET CAPITAL LETTER SHORT IDESERET CAPITAL LETTER SHORT EDESERET CAPITAL LETTER SHORT ADESERET CAPITAL LETTER SHORT AHDESERET CAPITAL LETTER SHORT ODESERET CAPITAL LETTER SHORT OODESERET CAPITAL LETTER AYDESERET CAPITAL LETTER OWDESERET CAPITAL \
    \LETTER WUDESERET CAPITAL LETTER YEEDESERET CAPITAL LETTER HDESERET CAPITAL LETTER PEEDESERET CAPITAL LETTER BEEDESERET CAPITAL LETTER TEEDESERET CAPITAL LETTER DEEDESERET CAPITAL LETTER CHEEDESERET CAPITAL LETTER JEEDESERET CAPITAL LETTER KAYDESERET CAPIT\
    \AL LETTER GAYDESERET CAPITAL LETTER EFDESERET CAPITAL LETTER VEEDESERET CAPITAL LETTER ETHDESERET CAPITAL LETTER THEEDESERET CAPITAL LETTER ESDESERET CAPITAL LETTER ZEEDESERET CAPITAL LETTER ESHDESERET CAPITAL LETTER ZHEEDESERET CAPITAL LETTER ERDESERET C\
    \APITAL LETTER ELDESERET CAPITAL LETTER EMDESERET CAPITAL LETTER ENDESERET CAPITAL LETTER ENGDESERET CAPITAL LETTER OIDESERET CAPITAL LETTER EWDESERET SMALL LETTER LONG IDESERET SMALL LETTER LONG EDESERET SMALL LETTER LONG ADESERET SMALL LETTER LONG AHDESE\
    \RET SMALL LETTER LONG ODESERET SMALL LETTER LONG OODESERET SMALL LETTER SHORT IDESERET SMALL LETTER SHORT EDESERET SMALL LETTER SHORT ADESERET SMALL LETTER SHORT AHDESERET SMALL LETTER SHORT ODESERET SMALL LETTER SHORT OODESERET SMALL LETTER AYDESERET SMA\
    \LL LETTER OWDESERET SMALL LETTER WUDESERET SMALL LETTER YEEDESERET SMALL LETTER HDESERET SMALL LETTER PEEDESERET SMALL LETTER BEEDESERET SMALL LETTER TEEDESERET SMALL LETTER DEEDESERET SMALL LETTER CHEEDESERET SMALL LETTER JEEDESERET SMALL LETTER KAYDESER\
    \ET SMALL LETTER GAYDESERET SMALL LETTER EFDESERET SMALL LETTER VEEDESERET SMALL LETTER ETHDESERET SMALL LETTER THEEDESERET SMALL LETTER ESDESERET SMALL LETTER ZEEDESERET SMALL LETTER ESHDESERET SMALL LETTER ZHEEDESERET SMALL LETTER ERDESERET SMALL LETTER \
    \ELDESERET SMALL LETTER EMDESERET SMALL LETTER ENDESERET SMALL LETTER ENGDESERET SMALL LETTER OIDESERET SMALL LETTER EWSHAVIAN LETTER PEEPSHAVIAN LETTER TOTSHAVIAN LETTER KICKSHAVIAN LETTER FEESHAVIAN LETTER THIGHSHAVIAN LETTER SOSHAVIAN LETTER SURESHAVIAN\
    \ LETTER CHURCHSHAVIAN LETTER YEASHAVIAN LETTER HUNGSHAVIAN LETTER BIBSHAVIAN LETTER DEADSHAVIAN LETTER GAGSHAVIAN LETTER VOWSHAVIAN LETTER THEYSHAVIAN LETTER ZOOSHAVIAN LETTER MEASURESHAVIAN LETTER JUDGESHAVIAN LETTER WOESHAVIAN LETTER HA-HASHAVIAN LETTER\
    \ LOLLSHAVIAN LETTER MIMESHAVIAN LETTER IFSHAVIAN LETTER EGGSHAVIAN LETTER ASHSHAVIAN LETTER ADOSHAVIAN LETTER ONSHAVIAN LETTER WOOLSHAVIAN LETTER OUTSHAVIAN LETTER AHSHAVIAN LETTER ROARSHAVIAN LETTER NUNSHAVIAN LETTER EATSHAVIAN LETTER AGESHAVIAN LETTER I\
    \CESHAVIAN LETTER UPSHAVIAN LETTER OAKSHAVIAN LETTER OOZESHAVIAN LETTER OILSHAVIAN LETTER AWESHAVIAN LETTER ARESHAVIAN LETTER ORSHAVIAN LETTER AIRSHAVIAN LETTER ERRSHAVIAN LETTER ARRAYSHAVIAN LETTER EARSHAVIAN LETTER IANSHAVIAN LETTER YEWOSMANYA LETTER ALE\
    \FOSMANYA LETTER BAOSMANYA LETTER TAOSMANYA LETTER JAOSMANYA LETTER XAOSMANYA LETTER KHAOSMANYA LETTER DEELOSMANYA LETTER RAOSMANYA LETTER SAOSMANYA LETTER SHIINOSMANYA LETTER DHAOSMANYA LETTER CAYNOSMANYA LETTER GAOSMANYA LETTER FAOSMANYA LETTER QAAFOSMAN\
    \YA LETTER KAAFOSMANYA LETTER LAANOSMANYA LETTER MIINOSMANYA LETTER NUUNOSMANYA LETTER WAWOSMANYA LETTER HAOSMANYA LETTER YAOSMANYA LETTER AOSMANYA LETTER EOSMANYA LETTER IOSMANYA LETTER OOSMANYA LETTER UOSMANYA LETTER AAOSMANYA LETTER EEOSMANYA LETTER OOO\
    \SMANYA DIGIT ZEROOSMANYA DIGIT ONEOSMANYA DIGIT TWOOSMANYA DIGIT THREEOSMANYA DIGIT FOUROSMANYA DIGIT FIVEOSMANYA DIGIT SIXOSMANYA DIGIT SEVENOSMANYA DIGIT EIGHTOSMANYA DIGIT NINEOSAGE CAPITAL LETTER AOSAGE CAPITAL LETTER AIOSAGE CAPITAL LETTER AINOSAGE C\
    \APITAL LETTER AHOSAGE CAPITAL LETTER BRAOSAGE CAPITAL LETTER CHAOSAGE CAPITAL LETTER EHCHAOSAGE CAPITAL LETTER EOSAGE CAPITAL LETTER EINOSAGE CAPITAL LETTER HAOSAGE CAPITAL LETTER HYAOSAGE CAPITAL LETTER IOSAGE CAPITAL LETTER KAOSAGE CAPITAL LETTER EHKAOS\
    \AGE CAPITAL LETTER KYAOSAGE CAPITAL LETTER LAOSAGE CAPITAL LETTER MAOSAGE CAPITAL LETTER NAOSAGE CAPITAL LETTER OOSAGE CAPITAL LETTER OINOSAGE CAPITAL LETTER PAOSAGE CAPITAL LETTER EHPAOSAGE CAPITAL LETTER SAOSAGE CAPITAL LETTER SHAOSAGE CAPITAL LETTER TA\
    \OSAGE CAPITAL LETTER EHTAOSAGE CAPITAL LETTER TSAOSAGE CAPITAL LETTER EHTSAOSAGE CAPITAL LETTER TSHAOSAGE CAPITAL LETTER DHAOSAGE CAPITAL LETTER UOSAGE CAPITAL LETTER WAOSAGE CAPITAL LETTER KHAOSAGE CAPITAL LETTER GHAOSAGE CAPITAL LETTER ZAOSAGE CAPITAL L\
    \ETTER ZHAOSAGE SMALL LETTER AOSAGE SMALL LETTER AIOSAGE SMALL LETTER AINOSAGE SMALL LETTER AHOSAGE SMALL LETTER BRAOSAGE SMALL LETTER CHAOSAGE SMALL LETTER EHCHAOSAGE SMALL LETTER EOSAGE SMALL LETTER EINOSAGE SMALL LETTER HAOSAGE SMALL LETTER HYAOSAGE SMA\
    \LL LETTER IOSAGE SMALL LETTER KAOSAGE SMALL LETTER EHKAOSAGE SMALL LETTER KYAOSAGE SMALL LETTER LAOSAGE SMALL LETTER MAOSAGE SMALL LETTER NAOSAGE SMALL LETTER OOSAGE SMALL LETTER OINOSAGE SMALL LETTER PAOSAGE SMALL LETTER EHPAOSAGE SMALL LETTER SAOSAGE SM\
    \ALL LETTER SHAOSAGE SMALL LETTER TAOSAGE SMALL LETTER EHTAOSAGE SMALL LETTER TSAOSAGE SMALL LETTER EHTSAOSAGE SMALL LETTER TSHAOSAGE SMALL LETTER DHAOSAGE SMALL LETTER UOSAGE SMALL LETTER WAOSAGE SMALL LETTER KHAOSAGE SMALL LETTER GHAOSAGE SMALL LETTER ZA\
    \OSAGE SMALL LETTER ZHAELBASAN LETTER AELBASAN LETTER BEELBASAN LETTER CEELBASAN LETTER CHEELBASAN LETTER DEELBASAN LETTER NDEELBASAN LETTER DHEELBASAN LETTER EIELBASAN LETTER EELBASAN LETTER FEELBASAN LETTER GEELBASAN LETTER GJEELBASAN LETTER HEELBASAN LE\
    \TTER IELBASAN LETTER JEELBASAN LETTER KEELBASAN LETTER LEELBASAN LETTER LLEELBASAN LETTER MEELBASAN LETTER NEELBASAN LETTER NAELBASAN LETTER NJEELBASAN LETTER OELBASAN LETTER PEELBASAN LETTER QEELBASAN LETTER REELBASAN LETTER RREELBASAN LETTER SEELBASAN L\
    \ETTER SHEELBASAN LETTER TEELBASAN LETTER THEELBASAN LETTER UELBASAN LETTER VEELBASAN LETTER XEELBASAN LETTER YELBASAN LETTER ZEELBASAN LETTER ZHEELBASAN LETTER GHEELBASAN LETTER GHAMMAELBASAN LETTER KHECAUCASIAN ALBANIAN LETTER ALTCAUCASIAN ALBANIAN LETTE\
    \R BETCAUCASIAN ALBANIAN LETTER GIMCAUCASIAN ALBANIAN LETTER DATCAUCASIAN ALBANIAN LETTER EBCAUCASIAN ALBANIAN LETTER ZARLCAUCASIAN ALBANIAN LETTER EYNCAUCASIAN ALBANIAN LETTER ZHILCAUCASIAN ALBANIAN LETTER TASCAUCASIAN ALBANIAN LETTER CHACAUCASIAN ALBANIA\
    \N LETTER YOWDCAUCASIAN ALBANIAN LETTER ZHACAUCASIAN ALBANIAN LETTER IRBCAUCASIAN ALBANIAN LETTER SHACAUCASIAN ALBANIAN LETTER LANCAUCASIAN ALBANIAN LETTER INYACAUCASIAN ALBANIAN LETTER XEYNCAUCASIAN ALBANIAN LETTER DYANCAUCASIAN ALBANIAN LETTER CARCAUCASI\
    \AN ALBANIAN LETTER JHOXCAUCASIAN ALBANIAN LETTER KARCAUCASIAN ALBANIAN LETTER LYITCAUCASIAN ALBANIAN LETTER HEYTCAUCASIAN ALBANIAN LETTER QAYCAUCASIAN ALBANIAN LETTER AORCAUCASIAN ALBANIAN LETTER CHOYCAUCASIAN ALBANIAN LETTER CHICAUCASIAN ALBANIAN LETTER \
    \CYAYCAUCASIAN ALBANIAN LETTER MAQCAUCASIAN ALBANIAN LETTER QARCAUCASIAN ALBANIAN LETTER NOWCCAUCASIAN ALBANIAN LETTER DZYAYCAUCASIAN ALBANIAN LETTER SHAKCAUCASIAN ALBANIAN LETTER JAYNCAUCASIAN ALBANIAN LETTER ONCAUCASIAN ALBANIAN LETTER TYAYCAUCASIAN ALBA\
    \NIAN LETTER FAMCAUCASIAN ALBANIAN LETTER DZAYCAUCASIAN ALBANIAN LETTER CHATCAUCASIAN ALBANIAN LETTER PENCAUCASIAN ALBANIAN LETTER GHEYSCAUCASIAN ALBANIAN LETTER RATCAUCASIAN ALBANIAN LETTER SEYKCAUCASIAN ALBANIAN LETTER VEYZCAUCASIAN ALBANIAN LETTER TIWRC\
    \AUCASIAN ALBANIAN LETTER SHOYCAUCASIAN ALBANIAN LETTER IWNCAUCASIAN ALBANIAN LETTER CYAWCAUCASIAN ALBANIAN LETTER CAYNCAUCASIAN ALBANIAN LETTER YAYDCAUCASIAN ALBANIAN LETTER PIWRCAUCASIAN ALBANIAN LETTER KIWCAUCASIAN ALBANIAN CITATION MARKVITHKUQI CAPITAL\
    \ LETTER AVITHKUQI CAPITAL LETTER BBEVITHKUQI CAPITAL LETTER BEVITHKUQI CAPITAL LETTER CEVITHKUQI CAPITAL LETTER CHEVITHKUQI CAPITAL LETTER DEVITHKUQI CAPITAL LETTER DHEVITHKUQI CAPITAL LETTER EIVITHKUQI CAPITAL LETTER EVITHKUQI CAPITAL LETTER FEVITHKUQI C\
    \APITAL LETTER GAVITHKUQI CAPITAL LETTER HAVITHKUQI CAPITAL LETTER HHAVITHKUQI CAPITAL LETTER IVITHKUQI CAPITAL LETTER IJEVITHKUQI CAPITAL LETTER JEVITHKUQI CAPITAL LETTER KAVITHKUQI CAPITAL LETTER LAVITHKUQI CAPITAL LETTER LLAVITHKUQI CAPITAL LETTER MEVIT\
    \HKUQI CAPITAL LETTER NEVITHKUQI CAPITAL LETTER NJEVITHKUQI CAPITAL LETTER OVITHKUQI CAPITAL LETTER PEVITHKUQI CAPITAL LETTER QAVITHKUQI CAPITAL LETTER REVITHKUQI CAPITAL LETTER SEVITHKUQI CAPITAL LETTER SHEVITHKUQI CAPITAL LETTER TEVITHKUQI CAPITAL LETTER\
    \ THEVITHKUQI CAPITAL LETTER UVITHKUQI CAPITAL LETTER VEVITHKUQI CAPITAL LETTER XEVITHKUQI CAPITAL LETTER YVITHKUQI CAPITAL LETTER ZEVITHKUQI SMALL LETTER AVITHKUQI SMALL LETTER BBEVITHKUQI SMALL LETTER BEVITHKUQI SMALL LETTER CEVITHKUQI SMALL LETTER CHEVI\
    \THKUQI SMALL LETTER DEVITHKUQI SMALL LETTER DHEVITHKUQI SMALL LETTER EIVITHKUQI SMALL LETTER EVITHKUQI SMALL LETTER FEVITHKUQI SMALL LETTER GAVITHKUQI SMALL LETTER HAVITHKUQI SMALL LETTER HHAVITHKUQI SMALL LETTER IVITHKUQI SMALL LETTER IJEVITHKUQI SMALL L\
    \ETTER JEVITHKUQI SMALL LETTER KAVITHKUQI SMALL LETTER LAVITHKUQI SMALL LETTER LLAVITHKUQI SMALL LETTER MEVITHKUQI SMALL LETTER NEVITHKUQI SMALL LETTER NJEVITHKUQI SMALL LETTER OVITHKUQI SMALL LETTER PEVITHKUQI SMALL LETTER QAVITHKUQI SMALL LETTER REVITHKU\
    \QI SMALL LETTER SEVITHKUQI SMALL LETTER SHEVITHKUQI SMALL LETTER TEVITHKUQI SMALL LETTER THEVITHKUQI SMALL LETTER UVITHKUQI SMALL LETTER VEVITHKUQI SMALL LETTER XEVITHKUQI SMALL LETTER YVITHKUQI SMALL LETTER ZELINEAR A SIGN AB001LINEAR A SIGN AB002LINEAR \
    \A SIGN AB003LINEAR A SIGN AB004LINEAR A SIGN AB005LINEAR A SIGN AB006LINEAR A SIGN AB007LINEAR A SIGN AB008LINEAR A SIGN AB009LINEAR A SIGN AB010LINEAR A SIGN AB011LINEAR A SIGN AB013LINEAR A SIGN AB016LINEAR A SIGN AB017LINEAR A SIGN AB020LINEAR A SIGN A\
    \B021LINEAR A SIGN AB021FLINEAR A SIGN AB021MLINEAR A SIGN AB022LINEAR A SIGN AB022FLINEAR A SIGN AB022MLINEAR A SIGN AB023LINEAR A SIGN AB023MLINEAR A SIGN AB024LINEAR A SIGN AB026LINEAR A SIGN AB027LINEAR A SIGN AB028LINEAR A SIGN A028BLINEAR A SIGN AB02\
    \9LINEAR A SIGN AB030LINEAR A SIGN AB031LINEAR A SIGN AB034LINEAR A SIGN AB037LINEAR A SIGN AB038LINEAR A SIGN AB039LINEAR A SIGN AB040LINEAR A SIGN AB041LINEAR A SIGN AB044LINEAR A SIGN AB045LINEAR A SIGN AB046LINEAR A SIGN AB047LINEAR A SIGN AB048LINEAR \
    \A SIGN AB049LINEAR A SIGN AB050LINEAR A SIGN AB051LINEAR A SIGN AB053LINEAR A SIGN AB054LINEAR A SIGN AB055LINEAR A SIGN AB056LINEAR A SIGN AB057LINEAR A SIGN AB058LINEAR A SIGN AB059LINEAR A SIGN AB060LINEAR A SIGN AB061LINEAR A SIGN AB065LINEAR A SIGN A\
    \B066LINEAR A SIGN AB067LINEAR A SIGN AB069LINEAR A SIGN AB070LINEAR A SIGN AB073LINEAR A SIGN AB074LINEAR A SIGN AB076LINEAR A SIGN AB077LINEAR A SIGN AB078LINEAR A SIGN AB079LINEAR A SIGN AB080LINEAR A SIGN AB081LINEAR A SIGN AB082LINEAR A SIGN AB085LINE\
    \AR A SIGN AB086LINEAR A SIGN AB087LINEAR A SIGN A100-102LINEAR A SIGN AB118LINEAR A SIGN AB120LINEAR A SIGN A120BLINEAR A SIGN AB122LINEAR A SIGN AB123LINEAR A SIGN AB131ALINEAR A SIGN AB131BLINEAR A SIGN A131CLINEAR A SIGN AB164LINEAR A SIGN AB171LINEAR \
    \A SIGN AB180LINEAR A SIGN AB188LINEAR A SIGN AB191LINEAR A SIGN A301LINEAR A SIGN A302LINEAR A SIGN A303LINEAR A SIGN A304LINEAR A SIGN A305LINEAR A SIGN A306LINEAR A SIGN A307LINEAR A SIGN A308LINEAR A SIGN A309ALINEAR A SIGN A309BLINEAR A SIGN A309CLINE\
    \AR A SIGN A310LINEAR A SIGN A311LINEAR A SIGN A312LINEAR A SIGN A313ALINEAR A SIGN A313BLINEAR A SIGN A313CLINEAR A SIGN A314LINEAR A SIGN A315LINEAR A SIGN A316LINEAR A SIGN A317LINEAR A SIGN A318LINEAR A SIGN A319LINEAR A SIGN A320LINEAR A SIGN A321LINE\
    \AR A SIGN A322LINEAR A SIGN A323LINEAR A SIGN A324LINEAR A SIGN A325LINEAR A SIGN A326LINEAR A SIGN A327LINEAR A SIGN A328LINEAR A SIGN A329LINEAR A SIGN A330LINEAR A SIGN A331LINEAR A SIGN A332LINEAR A SIGN A333LINEAR A SIGN A334LINEAR A SIGN A335LINEAR \
    \A SIGN A336LINEAR A SIGN A337LINEAR A SIGN A338LINEAR A SIGN A339LINEAR A SIGN A340LINEAR A SIGN A341LINEAR A SIGN A342LINEAR A SIGN A343LINEAR A SIGN A344LINEAR A SIGN A345LINEAR A SIGN A346LINEAR A SIGN A347LINEAR A SIGN A348LINEAR A SIGN A349LINEAR A S\
    \IGN A350LINEAR A SIGN A351LINEAR A SIGN A352LINEAR A SIGN A353LINEAR A SIGN A354LINEAR A SIGN A355LINEAR A SIGN A356LINEAR A SIGN A357LINEAR A SIGN A358LINEAR A SIGN A359LINEAR A SIGN A360LINEAR A SIGN A361LINEAR A SIGN A362LINEAR A SIGN A363LINEAR A SIGN\
    \ A364LINEAR A SIGN A365LINEAR A SIGN A366LINEAR A SIGN A367LINEAR A SIGN A368LINEAR A SIGN A369LINEAR A SIGN A370LINEAR A SIGN A371LINEAR A SIGN A400-VASLINEAR A SIGN A401-VASLINEAR A SIGN A402-VASLINEAR A SIGN A403-VASLINEAR A SIGN A404-VASLINEAR A SIGN \
    \A405-VASLINEAR A SIGN A406-VASLINEAR A SIGN A407-VASLINEAR A SIGN A408-VASLINEAR A SIGN A409-VASLINEAR A SIGN A410-VASLINEAR A SIGN A411-VASLINEAR A SIGN A412-VASLINEAR A SIGN A413-VASLINEAR A SIGN A414-VASLINEAR A SIGN A415-VASLINEAR A SIGN A416-VASLINEA\
    \R A SIGN A417-VASLINEAR A SIGN A418-VASLINEAR A SIGN A501LINEAR A SIGN A502LINEAR A SIGN A503LINEAR A SIGN A504LINEAR A SIGN A505LINEAR A SIGN A506LINEAR A SIGN A508LINEAR A SIGN A509LINEAR A SIGN A510LINEAR A SIGN A511LINEAR A SIGN A512LINEAR A SIGN A513\
    \LINEAR A SIGN A515LINEAR A SIGN A516LINEAR A SIGN A520LINEAR A SIGN A521LINEAR A SIGN A523LINEAR A SIGN A524LINEAR A SIGN A525LINEAR A SIGN A526LINEAR A SIGN A527LINEAR A SIGN A528LINEAR A SIGN A529LINEAR A SIGN A530LINEAR A SIGN A531LINEAR A SIGN A532LIN\
    \EAR A SIGN A534LINEAR A SIGN A535LINEAR A SIGN A536LINEAR A SIGN A537LINEAR A SIGN A538LINEAR A SIGN A539LINEAR A SIGN A540LINEAR A SIGN A541LINEAR A SIGN A542LINEAR A SIGN A545LINEAR A SIGN A547LINEAR A SIGN A548LINEAR A SIGN A549LINEAR A SIGN A550LINEAR\
    \ A SIGN A551LINEAR A SIGN A552LINEAR A SIGN A553LINEAR A SIGN A554LINEAR A SIGN A555LINEAR A SIGN A556LINEAR A SIGN A557LINEAR A SIGN A559LINEAR A SIGN A563LINEAR A SIGN A564LINEAR A SIGN A565LINEAR A SIGN A566LINEAR A SIGN A568LINEAR A SIGN A569LINEAR A \
    \SIGN A570LINEAR A SIGN A571LINEAR A SIGN A572LINEAR A SIGN A573LINEAR A SIGN A574LINEAR A SIGN A575LINEAR A SIGN A576LINEAR A SIGN A577LINEAR A SIGN A578LINEAR A SIGN A579LINEAR A SIGN A580LINEAR A SIGN A581LINEAR A SIGN A582LINEAR A SIGN A583LINEAR A SIG\
    \N A584LINEAR A SIGN A585LINEAR A SIGN A586LINEAR A SIGN A587LINEAR A SIGN A588LINEAR A SIGN A589LINEAR A SIGN A591LINEAR A SIGN A592LINEAR A SIGN A594LINEAR A SIGN A595LINEAR A SIGN A596LINEAR A SIGN A598LINEAR A SIGN A600LINEAR A SIGN A601LINEAR A SIGN A\
    \602LINEAR A SIGN A603LINEAR A SIGN A604LINEAR A SIGN A606LINEAR A SIGN A608LINEAR A SIGN A609LINEAR A SIGN A610LINEAR A SIGN A611LINEAR A SIGN A612LINEAR A SIGN A613LINEAR A SIGN A614LINEAR A SIGN A615LINEAR A SIGN A616LINEAR A SIGN A617LINEAR A SIGN A618\
    \LINEAR A SIGN A619LINEAR A SIGN A620LINEAR A SIGN A621LINEAR A SIGN A622LINEAR A SIGN A623LINEAR A SIGN A624LINEAR A SIGN A626LINEAR A SIGN A627LINEAR A SIGN A628LINEAR A SIGN A629LINEAR A SIGN A634LINEAR A SIGN A637LINEAR A SIGN A638LINEAR A SIGN A640LIN\
    \EAR A SIGN A642LINEAR A SIGN A643LINEAR A SIGN A644LINEAR A SIGN A645LINEAR A SIGN A646LINEAR A SIGN A648LINEAR A SIGN A649LINEAR A SIGN A651LINEAR A SIGN A652LINEAR A SIGN A653LINEAR A SIGN A654LINEAR A SIGN A655LINEAR A SIGN A656LINEAR A SIGN A657LINEAR\
    \ A SIGN A658LINEAR A SIGN A659LINEAR A SIGN A660LINEAR A SIGN A661LINEAR A SIGN A662LINEAR A SIGN A663LINEAR A SIGN A664LINEAR A SIGN A701 ALINEAR A SIGN A702 BLINEAR A SIGN A703 DLINEAR A SIGN A704 ELINEAR A SIGN A705 FLINEAR A SIGN A706 HLINEAR A SIGN A\
    \707 JLINEAR A SIGN A708 KLINEAR A SIGN A709 LLINEAR A SIGN A709-2 L2LINEAR A SIGN A709-3 L3LINEAR A SIGN A709-4 L4LINEAR A SIGN A709-6 L6LINEAR A SIGN A710 WLINEAR A SIGN A711 XLINEAR A SIGN A712 YLINEAR A SIGN A713 OMEGALINEAR A SIGN A714 ABBLINEAR A SIG\
    \N A715 BBLINEAR A SIGN A717 DDLINEAR A SIGN A726 EYYYLINEAR A SIGN A732 JELINEAR A SIGN A800LINEAR A SIGN A801LINEAR A SIGN A802LINEAR A SIGN A803LINEAR A SIGN A804LINEAR A SIGN A805LINEAR A SIGN A806LINEAR A SIGN A807MODIFIER LETTER SMALL CAPITAL AAMODIF\
    \IER LETTER SUPERSCRIPT TRIANGULAR COLONMODIFIER LETTER SUPERSCRIPT HALF TRIANGULAR COLONMODIFIER LETTER SMALL AEMODIFIER LETTER SMALL CAPITAL BMODIFIER LETTER SMALL B WITH HOOKMODIFIER LETTER SMALL DZ DIGRAPHMODIFIER LETTER SMALL DZ DIGRAPH WITH RETROFLEX\
    \ HOOKMODIFIER LETTER SMALL DZ DIGRAPH WITH CURLMODIFIER LETTER SMALL DEZH DIGRAPHMODIFIER LETTER SMALL D WITH TAILMODIFIER LETTER SMALL D WITH HOOKMODIFIER LETTER SMALL D WITH HOOK AND TAILMODIFIER LETTER SMALL REVERSED EMODIFIER LETTER SMALL CLOSED REVER\
    \SED OPEN EMODIFIER LETTER SMALL FENG DIGRAPHMODIFIER LETTER SMALL RAMS HORNMODIFIER LETTER SMALL CAPITAL GMODIFIER LETTER SMALL G WITH HOOKMODIFIER LETTER SMALL CAPITAL G WITH HOOKMODIFIER LETTER SMALL H WITH STROKEMODIFIER LETTER SMALL CAPITAL HMODIFIER \
    \LETTER SMALL HENG WITH HOOKMODIFIER LETTER SMALL DOTLESS J WITH STROKE AND HOOKMODIFIER LETTER SMALL LS DIGRAPHMODIFIER LETTER SMALL LZ DIGRAPHMODIFIER LETTER SMALL L WITH BELTMODIFIER LETTER SMALL CAPITAL L WITH BELTMODIFIER LETTER SMALL L WITH RETROFLEX\
    \ HOOK AND BELTMODIFIER LETTER SMALL LEZHMODIFIER LETTER SMALL LEZH WITH RETROFLEX HOOKMODIFIER LETTER SMALL TURNED YMODIFIER LETTER SMALL TURNED Y WITH BELTMODIFIER LETTER SMALL O WITH STROKEMODIFIER LETTER SMALL CAPITAL OEMODIFIER LETTER SMALL CLOSED OME\
    \GAMODIFIER LETTER SMALL QMODIFIER LETTER SMALL TURNED R WITH LONG LEGMODIFIER LETTER SMALL TURNED R WITH LONG LEG AND RETROFLEX HOOKMODIFIER LETTER SMALL R WITH TAILMODIFIER LETTER SMALL R WITH FISHHOOKMODIFIER LETTER SMALL CAPITAL RMODIFIER LETTER SMALL \
    \TC DIGRAPH WITH CURLMODIFIER LETTER SMALL TS DIGRAPHMODIFIER LETTER SMALL TS DIGRAPH WITH RETROFLEX HOOKMODIFIER LETTER SMALL TESH DIGRAPHMODIFIER LETTER SMALL T WITH RETROFLEX HOOKMODIFIER LETTER SMALL V WITH RIGHT HOOKMODIFIER LETTER SMALL CAPITAL YMODI\
    \FIER LETTER GLOTTAL STOP WITH STROKEMODIFIER LETTER REVERSED GLOTTAL STOP WITH STROKEMODIFIER LETTER BILABIAL CLICKMODIFIER LETTER DENTAL CLICKMODIFIER LETTER LATERAL CLICKMODIFIER LETTER ALVEOLAR CLICKMODIFIER LETTER RETROFLEX CLICK WITH RETROFLEX HOOKMO\
    \DIFIER LETTER SMALL S WITH CURLCYPRIOT SYLLABLE ACYPRIOT SYLLABLE ECYPRIOT SYLLABLE ICYPRIOT SYLLABLE OCYPRIOT SYLLABLE UCYPRIOT SYLLABLE JACYPRIOT SYLLABLE JOCYPRIOT SYLLABLE KACYPRIOT SYLLABLE KECYPRIOT SYLLABLE KICYPRIOT SYLLABLE KOCYPRIOT SYLLABLE KUC\
    \YPRIOT SYLLABLE LACYPRIOT SYLLABLE LECYPRIOT SYLLABLE LICYPRIOT SYLLABLE LOCYPRIOT SYLLABLE LUCYPRIOT SYLLABLE MACYPRIOT SYLLABLE MECYPRIOT SYLLABLE MICYPRIOT SYLLABLE MOCYPRIOT SYLLABLE MUCYPRIOT SYLLABLE NACYPRIOT SYLLABLE NECYPRIOT SYLLABLE NICYPRIOT S\
    \YLLABLE NOCYPRIOT SYLLABLE NUCYPRIOT SYLLABLE PACYPRIOT SYLLABLE PECYPRIOT SYLLABLE PICYPRIOT SYLLABLE POCYPRIOT SYLLABLE PUCYPRIOT SYLLABLE RACYPRIOT SYLLABLE RECYPRIOT SYLLABLE RICYPRIOT SYLLABLE ROCYPRIOT SYLLABLE RUCYPRIOT SYLLABLE SACYPRIOT SYLLABLE \
    \SECYPRIOT SYLLABLE SICYPRIOT SYLLABLE SOCYPRIOT SYLLABLE SUCYPRIOT SYLLABLE TACYPRIOT SYLLABLE TECYPRIOT SYLLABLE TICYPRIOT SYLLABLE TOCYPRIOT SYLLABLE TUCYPRIOT SYLLABLE WACYPRIOT SYLLABLE WECYPRIOT SYLLABLE WICYPRIOT SYLLABLE WOCYPRIOT SYLLABLE XACYPRIO\
    \T SYLLABLE XECYPRIOT SYLLABLE ZACYPRIOT SYLLABLE ZOIMPERIAL ARAMAIC LETTER ALEPHIMPERIAL ARAMAIC LETTER BETHIMPERIAL ARAMAIC LETTER GIMELIMPERIAL ARAMAIC LETTER DALETHIMPERIAL ARAMAIC LETTER HEIMPERIAL ARAMAIC LETTER WAWIMPERIAL ARAMAIC LETTER ZAYINIMPERI\
    \AL ARAMAIC LETTER HETHIMPERIAL ARAMAIC LETTER TETHIMPERIAL ARAMAIC LETTER YODHIMPERIAL ARAMAIC LETTER KAPHIMPERIAL ARAMAIC LETTER LAMEDHIMPERIAL ARAMAIC LETTER MEMIMPERIAL ARAMAIC LETTER NUNIMPERIAL ARAMAIC LETTER SAMEKHIMPERIAL ARAMAIC LETTER AYINIMPERIA\
    \L ARAMAIC LETTER PEIMPERIAL ARAMAIC LETTER SADHEIMPERIAL ARAMAIC LETTER QOPHIMPERIAL ARAMAIC LETTER RESHIMPERIAL ARAMAIC LETTER SHINIMPERIAL ARAMAIC LETTER TAWIMPERIAL ARAMAIC SECTION SIGNIMPERIAL ARAMAIC NUMBER ONEIMPERIAL ARAMAIC NUMBER TWOIMPERIAL ARAM\
    \AIC NUMBER THREEIMPERIAL ARAMAIC NUMBER TENIMPERIAL ARAMAIC NUMBER TWENTYIMPERIAL ARAMAIC NUMBER ONE HUNDREDIMPERIAL ARAMAIC NUMBER ONE THOUSANDIMPERIAL ARAMAIC NUMBER TEN THOUSANDPALMYRENE LETTER ALEPHPALMYRENE LETTER BETHPALMYRENE LETTER GIMELPALMYRENE \
    \LETTER DALETHPALMYRENE LETTER HEPALMYRENE LETTER WAWPALMYRENE LETTER ZAYINPALMYRENE LETTER HETHPALMYRENE LETTER TETHPALMYRENE LETTER YODHPALMYRENE LETTER KAPHPALMYRENE LETTER LAMEDHPALMYRENE LETTER MEMPALMYRENE LETTER FINAL NUNPALMYRENE LETTER NUNPALMYREN\
    \E LETTER SAMEKHPALMYRENE LETTER AYINPALMYRENE LETTER PEPALMYRENE LETTER SADHEPALMYRENE LETTER QOPHPALMYRENE LETTER RESHPALMYRENE LETTER SHINPALMYRENE LETTER TAWPALMYRENE LEFT-POINTING FLEURONPALMYRENE RIGHT-POINTING FLEURONPALMYRENE NUMBER ONEPALMYRENE NU\
    \MBER TWOPALMYRENE NUMBER THREEPALMYRENE NUMBER FOURPALMYRENE NUMBER FIVEPALMYRENE NUMBER TENPALMYRENE NUMBER TWENTYNABATAEAN LETTER FINAL ALEPHNABATAEAN LETTER ALEPHNABATAEAN LETTER FINAL BETHNABATAEAN LETTER BETHNABATAEAN LETTER GIMELNABATAEAN LETTER DAL\
    \ETHNABATAEAN LETTER FINAL HENABATAEAN LETTER HENABATAEAN LETTER WAWNABATAEAN LETTER ZAYINNABATAEAN LETTER HETHNABATAEAN LETTER TETHNABATAEAN LETTER FINAL YODHNABATAEAN LETTER YODHNABATAEAN LETTER FINAL KAPHNABATAEAN LETTER KAPHNABATAEAN LETTER FINAL LAMED\
    \HNABATAEAN LETTER LAMEDHNABATAEAN LETTER FINAL MEMNABATAEAN LETTER MEMNABATAEAN LETTER FINAL NUNNABATAEAN LETTER NUNNABATAEAN LETTER SAMEKHNABATAEAN LETTER AYINNABATAEAN LETTER PENABATAEAN LETTER SADHENABATAEAN LETTER QOPHNABATAEAN LETTER RESHNABATAEAN LE\
    \TTER FINAL SHINNABATAEAN LETTER SHINNABATAEAN LETTER TAWNABATAEAN NUMBER ONENABATAEAN NUMBER TWONABATAEAN NUMBER THREENABATAEAN NUMBER FOURNABATAEAN CRUCIFORM NUMBER FOURNABATAEAN NUMBER FIVENABATAEAN NUMBER TENNABATAEAN NUMBER TWENTYNABATAEAN NUMBER ONE \
    \HUNDREDHATRAN LETTER ALEPHHATRAN LETTER BETHHATRAN LETTER GIMELHATRAN LETTER DALETH-RESHHATRAN LETTER HEHATRAN LETTER WAWHATRAN LETTER ZAYNHATRAN LETTER HETHHATRAN LETTER TETHHATRAN LETTER YODHHATRAN LETTER KAPHHATRAN LETTER LAMEDHHATRAN LETTER MEMHATRAN \
    \LETTER NUNHATRAN LETTER SAMEKHHATRAN LETTER AYNHATRAN LETTER PEHATRAN LETTER SADHEHATRAN LETTER QOPHHATRAN LETTER SHINHATRAN LETTER TAWHATRAN NUMBER ONEHATRAN NUMBER FIVEHATRAN NUMBER TENHATRAN NUMBER TWENTYHATRAN NUMBER ONE HUNDREDPHOENICIAN LETTER ALFPH\
    \OENICIAN LETTER BETPHOENICIAN LETTER GAMLPHOENICIAN LETTER DELTPHOENICIAN LETTER HEPHOENICIAN LETTER WAUPHOENICIAN LETTER ZAIPHOENICIAN LETTER HETPHOENICIAN LETTER TETPHOENICIAN LETTER YODPHOENICIAN LETTER KAFPHOENICIAN LETTER LAMDPHOENICIAN LETTER MEMPHO\
    \ENICIAN LETTER NUNPHOENICIAN LETTER SEMKPHOENICIAN LETTER AINPHOENICIAN LETTER PEPHOENICIAN LETTER SADEPHOENICIAN LETTER QOFPHOENICIAN LETTER ROSHPHOENICIAN LETTER SHINPHOENICIAN LETTER TAUPHOENICIAN NUMBER ONEPHOENICIAN NUMBER TENPHOENICIAN NUMBER TWENTY\
    \PHOENICIAN NUMBER ONE HUNDREDPHOENICIAN NUMBER TWOPHOENICIAN NUMBER THREEPHOENICIAN WORD SEPARATORLYDIAN LETTER ALYDIAN LETTER BLYDIAN LETTER GLYDIAN LETTER DLYDIAN LETTER ELYDIAN LETTER VLYDIAN LETTER ILYDIAN LETTER YLYDIAN LETTER KLYDIAN LETTER LLYDIAN \
    \LETTER MLYDIAN LETTER NLYDIAN LETTER OLYDIAN LETTER RLYDIAN LETTER SSLYDIAN LETTER TLYDIAN LETTER ULYDIAN LETTER FLYDIAN LETTER QLYDIAN LETTER SLYDIAN LETTER TTLYDIAN LETTER ANLYDIAN LETTER ENLYDIAN LETTER LYLYDIAN LETTER NNLYDIAN LETTER CLYDIAN TRIANGULA\
    \R MARKMEROITIC HIEROGLYPHIC LETTER AMEROITIC HIEROGLYPHIC LETTER EMEROITIC HIEROGLYPHIC LETTER IMEROITIC HIEROGLYPHIC LETTER OMEROITIC HIEROGLYPHIC LETTER YAMEROITIC HIEROGLYPHIC LETTER WAMEROITIC HIEROGLYPHIC LETTER BAMEROITIC HIEROGLYPHIC LETTER BA-2MER\
    \OITIC HIEROGLYPHIC LETTER PAMEROITIC HIEROGLYPHIC LETTER MAMEROITIC HIEROGLYPHIC LETTER NAMEROITIC HIEROGLYPHIC LETTER NA-2MEROITIC HIEROGLYPHIC LETTER NEMEROITIC HIEROGLYPHIC LETTER NE-2MEROITIC HIEROGLYPHIC LETTER RAMEROITIC HIEROGLYPHIC LETTER RA-2MERO\
    \ITIC HIEROGLYPHIC LETTER LAMEROITIC HIEROGLYPHIC LETTER KHAMEROITIC HIEROGLYPHIC LETTER HHAMEROITIC HIEROGLYPHIC LETTER SAMEROITIC HIEROGLYPHIC LETTER SA-2MEROITIC HIEROGLYPHIC LETTER SEMEROITIC HIEROGLYPHIC LETTER KAMEROITIC HIEROGLYPHIC LETTER QAMEROITI\
    \C HIEROGLYPHIC LETTER TAMEROITIC HIEROGLYPHIC LETTER TA-2MEROITIC HIEROGLYPHIC LETTER TEMEROITIC HIEROGLYPHIC LETTER TE-2MEROITIC HIEROGLYPHIC LETTER TOMEROITIC HIEROGLYPHIC LETTER DAMEROITIC HIEROGLYPHIC SYMBOL VIDJMEROITIC HIEROGLYPHIC SYMBOL VIDJ-2MERO\
    \ITIC CURSIVE LETTER AMEROITIC CURSIVE LETTER EMEROITIC CURSIVE LETTER IMEROITIC CURSIVE LETTER OMEROITIC CURSIVE LETTER YAMEROITIC CURSIVE LETTER WAMEROITIC CURSIVE LETTER BAMEROITIC CURSIVE LETTER PAMEROITIC CURSIVE LETTER MAMEROITIC CURSIVE LETTER NAMER\
    \OITIC CURSIVE LETTER NEMEROITIC CURSIVE LETTER RAMEROITIC CURSIVE LETTER LAMEROITIC CURSIVE LETTER KHAMEROITIC CURSIVE LETTER HHAMEROITIC CURSIVE LETTER SAMEROITIC CURSIVE LETTER ARCHAIC SAMEROITIC CURSIVE LETTER SEMEROITIC CURSIVE LETTER KAMEROITIC CURSI\
    \VE LETTER QAMEROITIC CURSIVE LETTER TAMEROITIC CURSIVE LETTER TEMEROITIC CURSIVE LETTER TOMEROITIC CURSIVE LETTER DAMEROITIC CURSIVE FRACTION ELEVEN TWELFTHSMEROITIC CURSIVE FRACTION ONE HALFMEROITIC CURSIVE LOGOGRAM RMTMEROITIC CURSIVE LOGOGRAM IMNMEROIT\
    \IC CURSIVE NUMBER ONEMEROITIC CURSIVE NUMBER TWOMEROITIC CURSIVE NUMBER THREEMEROITIC CURSIVE NUMBER FOURMEROITIC CURSIVE NUMBER FIVEMEROITIC CURSIVE NUMBER SIXMEROITIC CURSIVE NUMBER SEVENMEROITIC CURSIVE NUMBER EIGHTMEROITIC CURSIVE NUMBER NINEMEROITIC \
    \CURSIVE NUMBER TENMEROITIC CURSIVE NUMBER TWENTYMEROITIC CURSIVE NUMBER THIRTYMEROITIC CURSIVE NUMBER FORTYMEROITIC CURSIVE NUMBER FIFTYMEROITIC CURSIVE NUMBER SIXTYMEROITIC CURSIVE NUMBER SEVENTYMEROITIC CURSIVE NUMBER ONE HUNDREDMEROITIC CURSIVE NUMBER \
    \TWO HUNDREDMEROITIC CURSIVE NUMBER THREE HUNDREDMEROITIC CURSIVE NUMBER FOUR HUNDREDMEROITIC CURSIVE NUMBER FIVE HUNDREDMEROITIC CURSIVE NUMBER SIX HUNDREDMEROITIC CURSIVE NUMBER SEVEN HUNDREDMEROITIC CURSIVE NUMBER EIGHT HUNDREDMEROITIC CURSIVE NUMBER NI\
    \NE HUNDREDMEROITIC CURSIVE NUMBER ONE THOUSANDMEROITIC CURSIVE NUMBER TWO THOUSANDMEROITIC CURSIVE NUMBER THREE THOUSANDMEROITIC CURSIVE NUMBER FOUR THOUSANDMEROITIC CURSIVE NUMBER FIVE THOUSANDMEROITIC CURSIVE NUMBER SIX THOUSANDMEROITIC CURSIVE NUMBER S\
    \EVEN THOUSANDMEROITIC CURSIVE NUMBER EIGHT THOUSANDMEROITIC CURSIVE NUMBER NINE THOUSANDMEROITIC CURSIVE NUMBER TEN THOUSANDMEROITIC CURSIVE NUMBER TWENTY THOUSANDMEROITIC CURSIVE NUMBER THIRTY THOUSANDMEROITIC CURSIVE NUMBER FORTY THOUSANDMEROITIC CURSIV\
    \E NUMBER FIFTY THOUSANDMEROITIC CURSIVE NUMBER SIXTY THOUSANDMEROITIC CURSIVE NUMBER SEVENTY THOUSANDMEROITIC CURSIVE NUMBER EIGHTY THOUSANDMEROITIC CURSIVE NUMBER NINETY THOUSANDMEROITIC CURSIVE NUMBER ONE HUNDRED THOUSANDMEROITIC CURSIVE NUMBER TWO HUND\
    \RED THOUSANDMEROITIC CURSIVE NUMBER THREE HUNDRED THOUSANDMEROITIC CURSIVE NUMBER FOUR HUNDRED THOUSANDMEROITIC CURSIVE NUMBER FIVE HUNDRED THOUSANDMEROITIC CURSIVE NUMBER SIX HUNDRED THOUSANDMEROITIC CURSIVE NUMBER SEVEN HUNDRED THOUSANDMEROITIC CURSIVE \
    \NUMBER EIGHT HUNDRED THOUSANDMEROITIC CURSIVE NUMBER NINE HUNDRED THOUSANDMEROITIC CURSIVE FRACTION ONE TWELFTHMEROITIC CURSIVE FRACTION TWO TWELFTHSMEROITIC CURSIVE FRACTION THREE TWELFTHSMEROITIC CURSIVE FRACTION FOUR TWELFTHSMEROITIC CURSIVE FRACTION F\
    \IVE TWELFTHSMEROITIC CURSIVE FRACTION SIX TWELFTHSMEROITIC CURSIVE FRACTION SEVEN TWELFTHSMEROITIC CURSIVE FRACTION EIGHT TWELFTHSMEROITIC CURSIVE FRACTION NINE TWELFTHSMEROITIC CURSIVE FRACTION TEN TWELFTHSKHAROSHTHI LETTER AKHAROSHTHI VOWEL SIGN IKHAROS\
    \HTHI VOWEL SIGN UKHAROSHTHI VOWEL SIGN VOCALIC RKHAROSHTHI VOWEL SIGN EKHAROSHTHI VOWEL SIGN OKHAROSHTHI VOWEL LENGTH MARKKHAROSHTHI SIGN DOUBLE RING BELOWKHAROSHTHI SIGN ANUSVARAKHAROSHTHI SIGN VISARGAKHAROSHTHI LETTER KAKHAROSHTHI LETTER KHAKHAROSHTHI L\
    \ETTER GAKHAROSHTHI LETTER GHAKHAROSHTHI LETTER CAKHAROSHTHI LETTER CHAKHAROSHTHI LETTER JAKHAROSHTHI LETTER NYAKHAROSHTHI LETTER TTAKHAROSHTHI LETTER TTHAKHAROSHTHI LETTER DDAKHAROSHTHI LETTER DDHAKHAROSHTHI LETTER NNAKHAROSHTHI LETTER TAKHAROSHTHI LETTER\
    \ THAKHAROSHTHI LETTER DAKHAROSHTHI LETTER DHAKHAROSHTHI LETTER NAKHAROSHTHI LETTER PAKHAROSHTHI LETTER PHAKHAROSHTHI LETTER BAKHAROSHTHI LETTER BHAKHAROSHTHI LETTER MAKHAROSHTHI LETTER YAKHAROSHTHI LETTER RAKHAROSHTHI LETTER LAKHAROSHTHI LETTER VAKHAROSHT\
    \HI LETTER SHAKHAROSHTHI LETTER SSAKHAROSHTHI LETTER SAKHAROSHTHI LETTER ZAKHAROSHTHI LETTER HAKHAROSHTHI LETTER KKAKHAROSHTHI LETTER TTTHAKHAROSHTHI LETTER TTTAKHAROSHTHI LETTER VHAKHAROSHTHI SIGN BAR ABOVEKHAROSHTHI SIGN CAUDAKHAROSHTHI SIGN DOT BELOWKHA\
    \ROSHTHI VIRAMAKHAROSHTHI DIGIT ONEKHAROSHTHI DIGIT TWOKHAROSHTHI DIGIT THREEKHAROSHTHI DIGIT FOURKHAROSHTHI NUMBER TENKHAROSHTHI NUMBER TWENTYKHAROSHTHI NUMBER ONE HUNDREDKHAROSHTHI NUMBER ONE THOUSANDKHAROSHTHI FRACTION ONE HALFKHAROSHTHI PUNCTUATION DOT\
    \KHAROSHTHI PUNCTUATION SMALL CIRCLEKHAROSHTHI PUNCTUATION CIRCLEKHAROSHTHI PUNCTUATION CRESCENT BARKHAROSHTHI PUNCTUATION MANGALAMKHAROSHTHI PUNCTUATION LOTUSKHAROSHTHI PUNCTUATION DANDAKHAROSHTHI PUNCTUATION DOUBLE DANDAKHAROSHTHI PUNCTUATION LINESOLD SO\
    \UTH ARABIAN LETTER HEOLD SOUTH ARABIAN LETTER LAMEDHOLD SOUTH ARABIAN LETTER HETHOLD SOUTH ARABIAN LETTER MEMOLD SOUTH ARABIAN LETTER QOPHOLD SOUTH ARABIAN LETTER WAWOLD SOUTH ARABIAN LETTER SHINOLD SOUTH ARABIAN LETTER RESHOLD SOUTH ARABIAN LETTER BETHOL\
    \D SOUTH ARABIAN LETTER TAWOLD SOUTH ARABIAN LETTER SATOLD SOUTH ARABIAN LETTER KAPHOLD SOUTH ARABIAN LETTER NUNOLD SOUTH ARABIAN LETTER KHETHOLD SOUTH ARABIAN LETTER SADHEOLD SOUTH ARABIAN LETTER SAMEKHOLD SOUTH ARABIAN LETTER FEOLD SOUTH ARABIAN LETTER A\
    \LEFOLD SOUTH ARABIAN LETTER AYNOLD SOUTH ARABIAN LETTER DHADHEOLD SOUTH ARABIAN LETTER GIMELOLD SOUTH ARABIAN LETTER DALETHOLD SOUTH ARABIAN LETTER GHAYNOLD SOUTH ARABIAN LETTER TETHOLD SOUTH ARABIAN LETTER ZAYNOLD SOUTH ARABIAN LETTER DHALETHOLD SOUTH AR\
    \ABIAN LETTER YODHOLD SOUTH ARABIAN LETTER THAWOLD SOUTH ARABIAN LETTER THETHOLD SOUTH ARABIAN NUMBER ONEOLD SOUTH ARABIAN NUMBER FIFTYOLD SOUTH ARABIAN NUMERIC INDICATOROLD NORTH ARABIAN LETTER HEHOLD NORTH ARABIAN LETTER LAMOLD NORTH ARABIAN LETTER HAHOL\
    \D NORTH ARABIAN LETTER MEEMOLD NORTH ARABIAN LETTER QAFOLD NORTH ARABIAN LETTER WAWOLD NORTH ARABIAN LETTER ES-2OLD NORTH ARABIAN LETTER REHOLD NORTH ARABIAN LETTER BEHOLD NORTH ARABIAN LETTER TEHOLD NORTH ARABIAN LETTER ES-1OLD NORTH ARABIAN LETTER KAFOL\
    \D NORTH ARABIAN LETTER NOONOLD NORTH ARABIAN LETTER KHAHOLD NORTH ARABIAN LETTER SADOLD NORTH ARABIAN LETTER ES-3OLD NORTH ARABIAN LETTER FEHOLD NORTH ARABIAN LETTER ALEFOLD NORTH ARABIAN LETTER AINOLD NORTH ARABIAN LETTER DADOLD NORTH ARABIAN LETTER GEEM\
    \OLD NORTH ARABIAN LETTER DALOLD NORTH ARABIAN LETTER GHAINOLD NORTH ARABIAN LETTER TAHOLD NORTH ARABIAN LETTER ZAINOLD NORTH ARABIAN LETTER THALOLD NORTH ARABIAN LETTER YEHOLD NORTH ARABIAN LETTER THEHOLD NORTH ARABIAN LETTER ZAHOLD NORTH ARABIAN NUMBER O\
    \NEOLD NORTH ARABIAN NUMBER TENOLD NORTH ARABIAN NUMBER TWENTYMANICHAEAN LETTER ALEPHMANICHAEAN LETTER BETHMANICHAEAN LETTER BHETHMANICHAEAN LETTER GIMELMANICHAEAN LETTER GHIMELMANICHAEAN LETTER DALETHMANICHAEAN LETTER HEMANICHAEAN LETTER WAWMANICHAEAN SIG\
    \N UDMANICHAEAN LETTER ZAYINMANICHAEAN LETTER ZHAYINMANICHAEAN LETTER JAYINMANICHAEAN LETTER JHAYINMANICHAEAN LETTER HETHMANICHAEAN LETTER TETHMANICHAEAN LETTER YODHMANICHAEAN LETTER KAPHMANICHAEAN LETTER XAPHMANICHAEAN LETTER KHAPHMANICHAEAN LETTER LAMEDH\
    \MANICHAEAN LETTER DHAMEDHMANICHAEAN LETTER THAMEDHMANICHAEAN LETTER MEMMANICHAEAN LETTER NUNMANICHAEAN LETTER SAMEKHMANICHAEAN LETTER AYINMANICHAEAN LETTER AAYINMANICHAEAN LETTER PEMANICHAEAN LETTER FEMANICHAEAN LETTER SADHEMANICHAEAN LETTER QOPHMANICHAEA\
    \N LETTER XOPHMANICHAEAN LETTER QHOPHMANICHAEAN LETTER RESHMANICHAEAN LETTER SHINMANICHAEAN LETTER SSHINMANICHAEAN LETTER TAWMANICHAEAN ABBREVIATION MARK ABOVEMANICHAEAN ABBREVIATION MARK BELOWMANICHAEAN NUMBER ONEMANICHAEAN NUMBER FIVEMANICHAEAN NUMBER TE\
    \NMANICHAEAN NUMBER TWENTYMANICHAEAN NUMBER ONE HUNDREDMANICHAEAN PUNCTUATION STARMANICHAEAN PUNCTUATION FLEURONMANICHAEAN PUNCTUATION DOUBLE DOT WITHIN DOTMANICHAEAN PUNCTUATION DOT WITHIN DOTMANICHAEAN PUNCTUATION DOTMANICHAEAN PUNCTUATION TWO DOTSMANICH\
    \AEAN PUNCTUATION LINE FILLERAVESTAN LETTER AAVESTAN LETTER AAAVESTAN LETTER AOAVESTAN LETTER AAOAVESTAN LETTER ANAVESTAN LETTER AANAVESTAN LETTER AEAVESTAN LETTER AEEAVESTAN LETTER EAVESTAN LETTER EEAVESTAN LETTER OAVESTAN LETTER OOAVESTAN LETTER IAVESTAN\
    \ LETTER IIAVESTAN LETTER UAVESTAN LETTER UUAVESTAN LETTER KEAVESTAN LETTER XEAVESTAN LETTER XYEAVESTAN LETTER XVEAVESTAN LETTER GEAVESTAN LETTER GGEAVESTAN LETTER GHEAVESTAN LETTER CEAVESTAN LETTER JEAVESTAN LETTER TEAVESTAN LETTER THEAVESTAN LETTER DEAVE\
    \STAN LETTER DHEAVESTAN LETTER TTEAVESTAN LETTER PEAVESTAN LETTER FEAVESTAN LETTER BEAVESTAN LETTER BHEAVESTAN LETTER NGEAVESTAN LETTER NGYEAVESTAN LETTER NGVEAVESTAN LETTER NEAVESTAN LETTER NYEAVESTAN LETTER NNEAVESTAN LETTER MEAVESTAN LETTER HMEAVESTAN L\
    \ETTER YYEAVESTAN LETTER YEAVESTAN LETTER VEAVESTAN LETTER REAVESTAN LETTER LEAVESTAN LETTER SEAVESTAN LETTER ZEAVESTAN LETTER SHEAVESTAN LETTER ZHEAVESTAN LETTER SHYEAVESTAN LETTER SSHEAVESTAN LETTER HEAVESTAN ABBREVIATION MARKTINY TWO DOTS OVER ONE DOT P\
    \UNCTUATIONSMALL TWO DOTS OVER ONE DOT PUNCTUATIONLARGE TWO DOTS OVER ONE DOT PUNCTUATIONLARGE ONE DOT OVER TWO DOTS PUNCTUATIONLARGE TWO RINGS OVER ONE RING PUNCTUATIONLARGE ONE RING OVER TWO RINGS PUNCTUATIONINSCRIPTIONAL PARTHIAN LETTER ALEPHINSCRIPTION\
    \AL PARTHIAN LETTER BETHINSCRIPTIONAL PARTHIAN LETTER GIMELINSCRIPTIONAL PARTHIAN LETTER DALETHINSCRIPTIONAL PARTHIAN LETTER HEINSCRIPTIONAL PARTHIAN LETTER WAWINSCRIPTIONAL PARTHIAN LETTER ZAYININSCRIPTIONAL PARTHIAN LETTER HETHINSCRIPTIONAL PARTHIAN LETT\
    \ER TETHINSCRIPTIONAL PARTHIAN LETTER YODHINSCRIPTIONAL PARTHIAN LETTER KAPHINSCRIPTIONAL PARTHIAN LETTER LAMEDHINSCRIPTIONAL PARTHIAN LETTER MEMINSCRIPTIONAL PARTHIAN LETTER NUNINSCRIPTIONAL PARTHIAN LETTER SAMEKHINSCRIPTIONAL PARTHIAN LETTER AYININSCRIPT\
    \IONAL PARTHIAN LETTER PEINSCRIPTIONAL PARTHIAN LETTER SADHEINSCRIPTIONAL PARTHIAN LETTER QOPHINSCRIPTIONAL PARTHIAN LETTER RESHINSCRIPTIONAL PARTHIAN LETTER SHININSCRIPTIONAL PARTHIAN LETTER TAWINSCRIPTIONAL PARTHIAN NUMBER ONEINSCRIPTIONAL PARTHIAN NUMBE\
    \R TWOINSCRIPTIONAL PARTHIAN NUMBER THREEINSCRIPTIONAL PARTHIAN NUMBER FOURINSCRIPTIONAL PARTHIAN NUMBER TENINSCRIPTIONAL PARTHIAN NUMBER TWENTYINSCRIPTIONAL PARTHIAN NUMBER ONE HUNDREDINSCRIPTIONAL PARTHIAN NUMBER ONE THOUSANDINSCRIPTIONAL PAHLAVI LETTER \
    \ALEPHINSCRIPTIONAL PAHLAVI LETTER BETHINSCRIPTIONAL PAHLAVI LETTER GIMELINSCRIPTIONAL PAHLAVI LETTER DALETHINSCRIPTIONAL PAHLAVI LETTER HEINSCRIPTIONAL PAHLAVI LETTER WAW-AYIN-RESHINSCRIPTIONAL PAHLAVI LETTER ZAYININSCRIPTIONAL PAHLAVI LETTER HETHINSCRIPT\
    \IONAL PAHLAVI LETTER TETHINSCRIPTIONAL PAHLAVI LETTER YODHINSCRIPTIONAL PAHLAVI LETTER KAPHINSCRIPTIONAL PAHLAVI LETTER LAMEDHINSCRIPTIONAL PAHLAVI LETTER MEM-QOPHINSCRIPTIONAL PAHLAVI LETTER NUNINSCRIPTIONAL PAHLAVI LETTER SAMEKHINSCRIPTIONAL PAHLAVI LET\
    \TER PEINSCRIPTIONAL PAHLAVI LETTER SADHEINSCRIPTIONAL PAHLAVI LETTER SHININSCRIPTIONAL PAHLAVI LETTER TAWINSCRIPTIONAL PAHLAVI NUMBER ONEINSCRIPTIONAL PAHLAVI NUMBER TWOINSCRIPTIONAL PAHLAVI NUMBER THREEINSCRIPTIONAL PAHLAVI NUMBER FOURINSCRIPTIONAL PAHLA\
    \VI NUMBER TENINSCRIPTIONAL PAHLAVI NUMBER TWENTYINSCRIPTIONAL PAHLAVI NUMBER ONE HUNDREDINSCRIPTIONAL PAHLAVI NUMBER ONE THOUSANDPSALTER PAHLAVI LETTER ALEPHPSALTER PAHLAVI LETTER BETHPSALTER PAHLAVI LETTER GIMELPSALTER PAHLAVI LETTER DALETHPSALTER PAHLAV\
    \I LETTER HEPSALTER PAHLAVI LETTER WAW-AYIN-RESHPSALTER PAHLAVI LETTER ZAYINPSALTER PAHLAVI LETTER HETHPSALTER PAHLAVI LETTER YODHPSALTER PAHLAVI LETTER KAPHPSALTER PAHLAVI LETTER LAMEDHPSALTER PAHLAVI LETTER MEM-QOPHPSALTER PAHLAVI LETTER NUNPSALTER PAHLA\
    \VI LETTER SAMEKHPSALTER PAHLAVI LETTER PEPSALTER PAHLAVI LETTER SADHEPSALTER PAHLAVI LETTER SHINPSALTER PAHLAVI LETTER TAWPSALTER PAHLAVI SECTION MARKPSALTER PAHLAVI TURNED SECTION MARKPSALTER PAHLAVI FOUR DOTS WITH CROSSPSALTER PAHLAVI FOUR DOTS WITH DOT\
    \PSALTER PAHLAVI NUMBER ONEPSALTER PAHLAVI NUMBER TWOPSALTER PAHLAVI NUMBER THREEPSALTER PAHLAVI NUMBER FOURPSALTER PAHLAVI NUMBER TENPSALTER PAHLAVI NUMBER TWENTYPSALTER PAHLAVI NUMBER ONE HUNDREDOLD TURKIC LETTER ORKHON AOLD TURKIC LETTER YENISEI AOLD TU\
    \RKIC LETTER YENISEI AEOLD TURKIC LETTER ORKHON IOLD TURKIC LETTER YENISEI IOLD TURKIC LETTER YENISEI EOLD TURKIC LETTER ORKHON OOLD TURKIC LETTER ORKHON OEOLD TURKIC LETTER YENISEI OEOLD TURKIC LETTER ORKHON ABOLD TURKIC LETTER YENISEI ABOLD TURKIC LETTER\
    \ ORKHON AEBOLD TURKIC LETTER YENISEI AEBOLD TURKIC LETTER ORKHON AGOLD TURKIC LETTER YENISEI AGOLD TURKIC LETTER ORKHON AEGOLD TURKIC LETTER YENISEI AEGOLD TURKIC LETTER ORKHON ADOLD TURKIC LETTER YENISEI ADOLD TURKIC LETTER ORKHON AEDOLD TURKIC LETTER OR\
    \KHON EZOLD TURKIC LETTER YENISEI EZOLD TURKIC LETTER ORKHON AYOLD TURKIC LETTER YENISEI AYOLD TURKIC LETTER ORKHON AEYOLD TURKIC LETTER YENISEI AEYOLD TURKIC LETTER ORKHON AEKOLD TURKIC LETTER YENISEI AEKOLD TURKIC LETTER ORKHON OEKOLD TURKIC LETTER YENIS\
    \EI OEKOLD TURKIC LETTER ORKHON ALOLD TURKIC LETTER YENISEI ALOLD TURKIC LETTER ORKHON AELOLD TURKIC LETTER ORKHON ELTOLD TURKIC LETTER ORKHON EMOLD TURKIC LETTER ORKHON ANOLD TURKIC LETTER ORKHON AENOLD TURKIC LETTER YENISEI AENOLD TURKIC LETTER ORKHON EN\
    \TOLD TURKIC LETTER YENISEI ENTOLD TURKIC LETTER ORKHON ENCOLD TURKIC LETTER YENISEI ENCOLD TURKIC LETTER ORKHON ENYOLD TURKIC LETTER YENISEI ENYOLD TURKIC LETTER YENISEI ANGOLD TURKIC LETTER ORKHON ENGOLD TURKIC LETTER YENISEI AENGOLD TURKIC LETTER ORKHON\
    \ EPOLD TURKIC LETTER ORKHON OPOLD TURKIC LETTER ORKHON ICOLD TURKIC LETTER ORKHON ECOLD TURKIC LETTER YENISEI ECOLD TURKIC LETTER ORKHON AQOLD TURKIC LETTER YENISEI AQOLD TURKIC LETTER ORKHON IQOLD TURKIC LETTER YENISEI IQOLD TURKIC LETTER ORKHON OQOLD TU\
    \RKIC LETTER YENISEI OQOLD TURKIC LETTER ORKHON AROLD TURKIC LETTER YENISEI AROLD TURKIC LETTER ORKHON AEROLD TURKIC LETTER ORKHON ASOLD TURKIC LETTER ORKHON AESOLD TURKIC LETTER ORKHON ASHOLD TURKIC LETTER YENISEI ASHOLD TURKIC LETTER ORKHON ESHOLD TURKIC\
    \ LETTER YENISEI ESHOLD TURKIC LETTER ORKHON ATOLD TURKIC LETTER YENISEI ATOLD TURKIC LETTER ORKHON AETOLD TURKIC LETTER YENISEI AETOLD TURKIC LETTER ORKHON OTOLD TURKIC LETTER ORKHON BASHOLD HUNGARIAN CAPITAL LETTER AOLD HUNGARIAN CAPITAL LETTER AAOLD HUN\
    \GARIAN CAPITAL LETTER EBOLD HUNGARIAN CAPITAL LETTER AMBOLD HUNGARIAN CAPITAL LETTER ECOLD HUNGARIAN CAPITAL LETTER ENCOLD HUNGARIAN CAPITAL LETTER ECSOLD HUNGARIAN CAPITAL LETTER EDOLD HUNGARIAN CAPITAL LETTER ANDOLD HUNGARIAN CAPITAL LETTER EOLD HUNGARI\
    \AN CAPITAL LETTER CLOSE EOLD HUNGARIAN CAPITAL LETTER EEOLD HUNGARIAN CAPITAL LETTER EFOLD HUNGARIAN CAPITAL LETTER EGOLD HUNGARIAN CAPITAL LETTER EGYOLD HUNGARIAN CAPITAL LETTER EHOLD HUNGARIAN CAPITAL LETTER IOLD HUNGARIAN CAPITAL LETTER IIOLD HUNGARIAN\
    \ CAPITAL LETTER EJOLD HUNGARIAN CAPITAL LETTER EKOLD HUNGARIAN CAPITAL LETTER AKOLD HUNGARIAN CAPITAL LETTER UNKOLD HUNGARIAN CAPITAL LETTER ELOLD HUNGARIAN CAPITAL LETTER ELYOLD HUNGARIAN CAPITAL LETTER EMOLD HUNGARIAN CAPITAL LETTER ENOLD HUNGARIAN CAPI\
    \TAL LETTER ENYOLD HUNGARIAN CAPITAL LETTER OOLD HUNGARIAN CAPITAL LETTER OOOLD HUNGARIAN CAPITAL LETTER NIKOLSBURG OEOLD HUNGARIAN CAPITAL LETTER RUDIMENTA OEOLD HUNGARIAN CAPITAL LETTER OEEOLD HUNGARIAN CAPITAL LETTER EPOLD HUNGARIAN CAPITAL LETTER EMPOL\
    \D HUNGARIAN CAPITAL LETTER EROLD HUNGARIAN CAPITAL LETTER SHORT EROLD HUNGARIAN CAPITAL LETTER ESOLD HUNGARIAN CAPITAL LETTER ESZOLD HUNGARIAN CAPITAL LETTER ETOLD HUNGARIAN CAPITAL LETTER ENTOLD HUNGARIAN CAPITAL LETTER ETYOLD HUNGARIAN CAPITAL LETTER EC\
    \HOLD HUNGARIAN CAPITAL LETTER UOLD HUNGARIAN CAPITAL LETTER UUOLD HUNGARIAN CAPITAL LETTER NIKOLSBURG UEOLD HUNGARIAN CAPITAL LETTER RUDIMENTA UEOLD HUNGARIAN CAPITAL LETTER EVOLD HUNGARIAN CAPITAL LETTER EZOLD HUNGARIAN CAPITAL LETTER EZSOLD HUNGARIAN CA\
    \PITAL LETTER ENT-SHAPED SIGNOLD HUNGARIAN CAPITAL LETTER USOLD HUNGARIAN SMALL LETTER AOLD HUNGARIAN SMALL LETTER AAOLD HUNGARIAN SMALL LETTER EBOLD HUNGARIAN SMALL LETTER AMBOLD HUNGARIAN SMALL LETTER ECOLD HUNGARIAN SMALL LETTER ENCOLD HUNGARIAN SMALL L\
    \ETTER ECSOLD HUNGARIAN SMALL LETTER EDOLD HUNGARIAN SMALL LETTER ANDOLD HUNGARIAN SMALL LETTER EOLD HUNGARIAN SMALL LETTER CLOSE EOLD HUNGARIAN SMALL LETTER EEOLD HUNGARIAN SMALL LETTER EFOLD HUNGARIAN SMALL LETTER EGOLD HUNGARIAN SMALL LETTER EGYOLD HUNG\
    \ARIAN SMALL LETTER EHOLD HUNGARIAN SMALL LETTER IOLD HUNGARIAN SMALL LETTER IIOLD HUNGARIAN SMALL LETTER EJOLD HUNGARIAN SMALL LETTER EKOLD HUNGARIAN SMALL LETTER AKOLD HUNGARIAN SMALL LETTER UNKOLD HUNGARIAN SMALL LETTER ELOLD HUNGARIAN SMALL LETTER ELYO\
    \LD HUNGARIAN SMALL LETTER EMOLD HUNGARIAN SMALL LETTER ENOLD HUNGARIAN SMALL LETTER ENYOLD HUNGARIAN SMALL LETTER OOLD HUNGARIAN SMALL LETTER OOOLD HUNGARIAN SMALL LETTER NIKOLSBURG OEOLD HUNGARIAN SMALL LETTER RUDIMENTA OEOLD HUNGARIAN SMALL LETTER OEEOL\
    \D HUNGARIAN SMALL LETTER EPOLD HUNGARIAN SMALL LETTER EMPOLD HUNGARIAN SMALL LETTER EROLD HUNGARIAN SMALL LETTER SHORT EROLD HUNGARIAN SMALL LETTER ESOLD HUNGARIAN SMALL LETTER ESZOLD HUNGARIAN SMALL LETTER ETOLD HUNGARIAN SMALL LETTER ENTOLD HUNGARIAN SM\
    \ALL LETTER ETYOLD HUNGARIAN SMALL LETTER ECHOLD HUNGARIAN SMALL LETTER UOLD HUNGARIAN SMALL LETTER UUOLD HUNGARIAN SMALL LETTER NIKOLSBURG UEOLD HUNGARIAN SMALL LETTER RUDIMENTA UEOLD HUNGARIAN SMALL LETTER EVOLD HUNGARIAN SMALL LETTER EZOLD HUNGARIAN SMA\
    \LL LETTER EZSOLD HUNGARIAN SMALL LETTER ENT-SHAPED SIGNOLD HUNGARIAN SMALL LETTER USOLD HUNGARIAN NUMBER ONEOLD HUNGARIAN NUMBER FIVEOLD HUNGARIAN NUMBER TENOLD HUNGARIAN NUMBER FIFTYOLD HUNGARIAN NUMBER ONE HUNDREDOLD HUNGARIAN NUMBER ONE THOUSANDHANIFI \
    \ROHINGYA LETTER AHANIFI ROHINGYA LETTER BAHANIFI ROHINGYA LETTER PAHANIFI ROHINGYA LETTER TAHANIFI ROHINGYA LETTER TTAHANIFI ROHINGYA LETTER JAHANIFI ROHINGYA LETTER CAHANIFI ROHINGYA LETTER HAHANIFI ROHINGYA LETTER KHAHANIFI ROHINGYA LETTER FAHANIFI ROHI\
    \NGYA LETTER DAHANIFI ROHINGYA LETTER DDAHANIFI ROHINGYA LETTER RAHANIFI ROHINGYA LETTER RRAHANIFI ROHINGYA LETTER ZAHANIFI ROHINGYA LETTER SAHANIFI ROHINGYA LETTER SHAHANIFI ROHINGYA LETTER KAHANIFI ROHINGYA LETTER GAHANIFI ROHINGYA LETTER LAHANIFI ROHING\
    \YA LETTER MAHANIFI ROHINGYA LETTER NAHANIFI ROHINGYA LETTER WAHANIFI ROHINGYA LETTER KINNA WAHANIFI ROHINGYA LETTER YAHANIFI ROHINGYA LETTER KINNA YAHANIFI ROHINGYA LETTER NGAHANIFI ROHINGYA LETTER NYAHANIFI ROHINGYA LETTER VAHANIFI ROHINGYA VOWEL AHANIFI\
    \ ROHINGYA VOWEL IHANIFI ROHINGYA VOWEL UHANIFI ROHINGYA VOWEL EHANIFI ROHINGYA VOWEL OHANIFI ROHINGYA MARK SAKINHANIFI ROHINGYA MARK NA KHONNAHANIFI ROHINGYA SIGN HARBAHAYHANIFI ROHINGYA SIGN TAHALAHANIFI ROHINGYA SIGN TANAHANIFI ROHINGYA SIGN TASSIHANIFI\
    \ ROHINGYA DIGIT ZEROHANIFI ROHINGYA DIGIT ONEHANIFI ROHINGYA DIGIT TWOHANIFI ROHINGYA DIGIT THREEHANIFI ROHINGYA DIGIT FOURHANIFI ROHINGYA DIGIT FIVEHANIFI ROHINGYA DIGIT SIXHANIFI ROHINGYA DIGIT SEVENHANIFI ROHINGYA DIGIT EIGHTHANIFI ROHINGYA DIGIT NINER\
    \UMI DIGIT ONERUMI DIGIT TWORUMI DIGIT THREERUMI DIGIT FOURRUMI DIGIT FIVERUMI DIGIT SIXRUMI DIGIT SEVENRUMI DIGIT EIGHTRUMI DIGIT NINERUMI NUMBER TENRUMI NUMBER TWENTYRUMI NUMBER THIRTYRUMI NUMBER FORTYRUMI NUMBER FIFTYRUMI NUMBER SIXTYRUMI NUMBER SEVENTY\
    \RUMI NUMBER EIGHTYRUMI NUMBER NINETYRUMI NUMBER ONE HUNDREDRUMI NUMBER TWO HUNDREDRUMI NUMBER THREE HUNDREDRUMI NUMBER FOUR HUNDREDRUMI NUMBER FIVE HUNDREDRUMI NUMBER SIX HUNDREDRUMI NUMBER SEVEN HUNDREDRUMI NUMBER EIGHT HUNDREDRUMI NUMBER NINE HUNDREDRUM\
    \I FRACTION ONE HALFRUMI FRACTION ONE QUARTERRUMI FRACTION ONE THIRDRUMI FRACTION TWO THIRDSYEZIDI LETTER ELIFYEZIDI LETTER BEYEZIDI LETTER PEYEZIDI LETTER PHEYEZIDI LETTER THEYEZIDI LETTER SEYEZIDI LETTER CIMYEZIDI LETTER CHIMYEZIDI LETTER CHHIMYEZIDI LET\
    \TER HHAYEZIDI LETTER XAYEZIDI LETTER DALYEZIDI LETTER ZALYEZIDI LETTER RAYEZIDI LETTER RHAYEZIDI LETTER ZAYEZIDI LETTER JAYEZIDI LETTER SINYEZIDI LETTER SHINYEZIDI LETTER SADYEZIDI LETTER DADYEZIDI LETTER TAYEZIDI LETTER ZEYEZIDI LETTER EYNYEZIDI LETTER X\
    \HEYNYEZIDI LETTER FAYEZIDI LETTER VAYEZIDI LETTER VA ALTERNATE FORMYEZIDI LETTER QAFYEZIDI LETTER KAFYEZIDI LETTER KHAFYEZIDI LETTER GAFYEZIDI LETTER LAMYEZIDI LETTER MIMYEZIDI LETTER NUNYEZIDI LETTER UMYEZIDI LETTER WAWYEZIDI LETTER OWYEZIDI LETTER EWYEZ\
    \IDI LETTER HAYYEZIDI LETTER YOTYEZIDI LETTER ETYEZIDI COMBINING HAMZA MARKYEZIDI COMBINING MADDA MARKYEZIDI HYPHENATION MARKYEZIDI LETTER LAM WITH DOT ABOVEYEZIDI LETTER YOT WITH CIRCUMFLEX ABOVEARABIC SMALL LOW WORD SAKTAARABIC SMALL LOW WORD QASRARABIC \
    \SMALL LOW WORD MADDAOLD SOGDIAN LETTER ALEPHOLD SOGDIAN LETTER FINAL ALEPHOLD SOGDIAN LETTER BETHOLD SOGDIAN LETTER FINAL BETHOLD SOGDIAN LETTER GIMELOLD SOGDIAN LETTER HEOLD SOGDIAN LETTER FINAL HEOLD SOGDIAN LETTER WAWOLD SOGDIAN LETTER ZAYINOLD SOGDIAN\
    \ LETTER HETHOLD SOGDIAN LETTER YODHOLD SOGDIAN LETTER KAPHOLD SOGDIAN LETTER LAMEDHOLD SOGDIAN LETTER MEMOLD SOGDIAN LETTER NUNOLD SOGDIAN LETTER FINAL NUNOLD SOGDIAN LETTER FINAL NUN WITH VERTICAL TAILOLD SOGDIAN LETTER SAMEKHOLD SOGDIAN LETTER AYINOLD S\
    \OGDIAN LETTER ALTERNATE AYINOLD SOGDIAN LETTER PEOLD SOGDIAN LETTER SADHEOLD SOGDIAN LETTER FINAL SADHEOLD SOGDIAN LETTER FINAL SADHE WITH VERTICAL TAILOLD SOGDIAN LETTER RESH-AYIN-DALETHOLD SOGDIAN LETTER SHINOLD SOGDIAN LETTER TAWOLD SOGDIAN LETTER FINA\
    \L TAWOLD SOGDIAN LETTER FINAL TAW WITH VERTICAL TAILOLD SOGDIAN NUMBER ONEOLD SOGDIAN NUMBER TWOOLD SOGDIAN NUMBER THREEOLD SOGDIAN NUMBER FOUROLD SOGDIAN NUMBER FIVEOLD SOGDIAN NUMBER TENOLD SOGDIAN NUMBER TWENTYOLD SOGDIAN NUMBER THIRTYOLD SOGDIAN NUMBE\
    \R ONE HUNDREDOLD SOGDIAN FRACTION ONE HALFOLD SOGDIAN LIGATURE AYIN-DALETHSOGDIAN LETTER ALEPHSOGDIAN LETTER BETHSOGDIAN LETTER GIMELSOGDIAN LETTER HESOGDIAN LETTER WAWSOGDIAN LETTER ZAYINSOGDIAN LETTER HETHSOGDIAN LETTER YODHSOGDIAN LETTER KAPHSOGDIAN LE\
    \TTER LAMEDHSOGDIAN LETTER MEMSOGDIAN LETTER NUNSOGDIAN LETTER SAMEKHSOGDIAN LETTER AYINSOGDIAN LETTER PESOGDIAN LETTER SADHESOGDIAN LETTER RESH-AYINSOGDIAN LETTER SHINSOGDIAN LETTER TAWSOGDIAN LETTER FETHSOGDIAN LETTER LESHSOGDIAN INDEPENDENT SHINSOGDIAN \
    \COMBINING DOT BELOWSOGDIAN COMBINING TWO DOTS BELOWSOGDIAN COMBINING DOT ABOVESOGDIAN COMBINING TWO DOTS ABOVESOGDIAN COMBINING CURVE ABOVESOGDIAN COMBINING CURVE BELOWSOGDIAN COMBINING HOOK ABOVESOGDIAN COMBINING HOOK BELOWSOGDIAN COMBINING LONG HOOK BEL\
    \OWSOGDIAN COMBINING RESH BELOWSOGDIAN COMBINING STROKE BELOWSOGDIAN NUMBER ONESOGDIAN NUMBER TENSOGDIAN NUMBER TWENTYSOGDIAN NUMBER ONE HUNDREDSOGDIAN PUNCTUATION TWO VERTICAL BARSSOGDIAN PUNCTUATION TWO VERTICAL BARS WITH DOTSSOGDIAN PUNCTUATION CIRCLE W\
    \ITH DOTSOGDIAN PUNCTUATION TWO CIRCLES WITH DOTSSOGDIAN PUNCTUATION HALF CIRCLE WITH DOTOLD UYGHUR LETTER ALEPHOLD UYGHUR LETTER BETHOLD UYGHUR LETTER GIMEL-HETHOLD UYGHUR LETTER WAWOLD UYGHUR LETTER ZAYINOLD UYGHUR LETTER FINAL HETHOLD UYGHUR LETTER YODH\
    \OLD UYGHUR LETTER KAPHOLD UYGHUR LETTER LAMEDHOLD UYGHUR LETTER MEMOLD UYGHUR LETTER NUNOLD UYGHUR LETTER SAMEKHOLD UYGHUR LETTER PEOLD UYGHUR LETTER SADHEOLD UYGHUR LETTER RESHOLD UYGHUR LETTER SHINOLD UYGHUR LETTER TAWOLD UYGHUR LETTER LESHOLD UYGHUR CO\
    \MBINING DOT ABOVEOLD UYGHUR COMBINING DOT BELOWOLD UYGHUR COMBINING TWO DOTS ABOVEOLD UYGHUR COMBINING TWO DOTS BELOWOLD UYGHUR PUNCTUATION BAROLD UYGHUR PUNCTUATION TWO BARSOLD UYGHUR PUNCTUATION TWO DOTSOLD UYGHUR PUNCTUATION FOUR DOTSCHORASMIAN LETTER \
    \ALEPHCHORASMIAN LETTER SMALL ALEPHCHORASMIAN LETTER BETHCHORASMIAN LETTER GIMELCHORASMIAN LETTER DALETHCHORASMIAN LETTER HECHORASMIAN LETTER WAWCHORASMIAN LETTER CURLED WAWCHORASMIAN LETTER ZAYINCHORASMIAN LETTER HETHCHORASMIAN LETTER YODHCHORASMIAN LETTE\
    \R KAPHCHORASMIAN LETTER LAMEDHCHORASMIAN LETTER MEMCHORASMIAN LETTER NUNCHORASMIAN LETTER SAMEKHCHORASMIAN LETTER AYINCHORASMIAN LETTER PECHORASMIAN LETTER RESHCHORASMIAN LETTER SHINCHORASMIAN LETTER TAWCHORASMIAN NUMBER ONECHORASMIAN NUMBER TWOCHORASMIAN\
    \ NUMBER THREECHORASMIAN NUMBER FOURCHORASMIAN NUMBER TENCHORASMIAN NUMBER TWENTYCHORASMIAN NUMBER ONE HUNDREDELYMAIC LETTER ALEPHELYMAIC LETTER BETHELYMAIC LETTER GIMELELYMAIC LETTER DALETHELYMAIC LETTER HEELYMAIC LETTER WAWELYMAIC LETTER ZAYINELYMAIC LET\
    \TER HETHELYMAIC LETTER TETHELYMAIC LETTER YODHELYMAIC LETTER KAPHELYMAIC LETTER LAMEDHELYMAIC LETTER MEMELYMAIC LETTER NUNELYMAIC LETTER SAMEKHELYMAIC LETTER AYINELYMAIC LETTER PEELYMAIC LETTER SADHEELYMAIC LETTER QOPHELYMAIC LETTER RESHELYMAIC LETTER SHI\
    \NELYMAIC LETTER TAWELYMAIC LIGATURE ZAYIN-YODHBRAHMI SIGN CANDRABINDUBRAHMI SIGN ANUSVARABRAHMI SIGN VISARGABRAHMI SIGN JIHVAMULIYABRAHMI SIGN UPADHMANIYABRAHMI LETTER ABRAHMI LETTER AABRAHMI LETTER IBRAHMI LETTER IIBRAHMI LETTER UBRAHMI LETTER UUBRAHMI L\
    \ETTER VOCALIC RBRAHMI LETTER VOCALIC RRBRAHMI LETTER VOCALIC LBRAHMI LETTER VOCALIC LLBRAHMI LETTER EBRAHMI LETTER AIBRAHMI LETTER OBRAHMI LETTER AUBRAHMI LETTER KABRAHMI LETTER KHABRAHMI LETTER GABRAHMI LETTER GHABRAHMI LETTER NGABRAHMI LETTER CABRAHMI L\
    \ETTER CHABRAHMI LETTER JABRAHMI LETTER JHABRAHMI LETTER NYABRAHMI LETTER TTABRAHMI LETTER TTHABRAHMI LETTER DDABRAHMI LETTER DDHABRAHMI LETTER NNABRAHMI LETTER TABRAHMI LETTER THABRAHMI LETTER DABRAHMI LETTER DHABRAHMI LETTER NABRAHMI LETTER PABRAHMI LETT\
    \ER PHABRAHMI LETTER BABRAHMI LETTER BHABRAHMI LETTER MABRAHMI LETTER YABRAHMI LETTER RABRAHMI LETTER LABRAHMI LETTER VABRAHMI LETTER SHABRAHMI LETTER SSABRAHMI LETTER SABRAHMI LETTER HABRAHMI LETTER LLABRAHMI LETTER OLD TAMIL LLLABRAHMI LETTER OLD TAMIL R\
    \RABRAHMI LETTER OLD TAMIL NNNABRAHMI VOWEL SIGN AABRAHMI VOWEL SIGN BHATTIPROLU AABRAHMI VOWEL SIGN IBRAHMI VOWEL SIGN IIBRAHMI VOWEL SIGN UBRAHMI VOWEL SIGN UUBRAHMI VOWEL SIGN VOCALIC RBRAHMI VOWEL SIGN VOCALIC RRBRAHMI VOWEL SIGN VOCALIC LBRAHMI VOWEL \
    \SIGN VOCALIC LLBRAHMI VOWEL SIGN EBRAHMI VOWEL SIGN AIBRAHMI VOWEL SIGN OBRAHMI VOWEL SIGN AUBRAHMI VIRAMABRAHMI DANDABRAHMI DOUBLE DANDABRAHMI PUNCTUATION DOTBRAHMI PUNCTUATION DOUBLE DOTBRAHMI PUNCTUATION LINEBRAHMI PUNCTUATION CRESCENT BARBRAHMI PUNCTU\
    \ATION LOTUSBRAHMI NUMBER ONEBRAHMI NUMBER TWOBRAHMI NUMBER THREEBRAHMI NUMBER FOURBRAHMI NUMBER FIVEBRAHMI NUMBER SIXBRAHMI NUMBER SEVENBRAHMI NUMBER EIGHTBRAHMI NUMBER NINEBRAHMI NUMBER TENBRAHMI NUMBER TWENTYBRAHMI NUMBER THIRTYBRAHMI NUMBER FORTYBRAHMI\
    \ NUMBER FIFTYBRAHMI NUMBER SIXTYBRAHMI NUMBER SEVENTYBRAHMI NUMBER EIGHTYBRAHMI NUMBER NINETYBRAHMI NUMBER ONE HUNDREDBRAHMI NUMBER ONE THOUSANDBRAHMI DIGIT ZEROBRAHMI DIGIT ONEBRAHMI DIGIT TWOBRAHMI DIGIT THREEBRAHMI DIGIT FOURBRAHMI DIGIT FIVEBRAHMI DIG\
    \IT SIXBRAHMI DIGIT SEVENBRAHMI DIGIT EIGHTBRAHMI DIGIT NINEBRAHMI SIGN OLD TAMIL VIRAMABRAHMI LETTER OLD TAMIL SHORT EBRAHMI LETTER OLD TAMIL SHORT OBRAHMI VOWEL SIGN OLD TAMIL SHORT EBRAHMI VOWEL SIGN OLD TAMIL SHORT OBRAHMI LETTER OLD TAMIL LLABRAHMI NU\
    \MBER JOINERKAITHI SIGN CANDRABINDUKAITHI SIGN ANUSVARAKAITHI SIGN VISARGAKAITHI LETTER AKAITHI LETTER AAKAITHI LETTER IKAITHI LETTER IIKAITHI LETTER UKAITHI LETTER UUKAITHI LETTER EKAITHI LETTER AIKAITHI LETTER OKAITHI LETTER AUKAITHI LETTER KAKAITHI LETT\
    \ER KHAKAITHI LETTER GAKAITHI LETTER GHAKAITHI LETTER NGAKAITHI LETTER CAKAITHI LETTER CHAKAITHI LETTER JAKAITHI LETTER JHAKAITHI LETTER NYAKAITHI LETTER TTAKAITHI LETTER TTHAKAITHI LETTER DDAKAITHI LETTER DDDHAKAITHI LETTER DDHAKAITHI LETTER RHAKAITHI LET\
    \TER NNAKAITHI LETTER TAKAITHI LETTER THAKAITHI LETTER DAKAITHI LETTER DHAKAITHI LETTER NAKAITHI LETTER PAKAITHI LETTER PHAKAITHI LETTER BAKAITHI LETTER BHAKAITHI LETTER MAKAITHI LETTER YAKAITHI LETTER RAKAITHI LETTER LAKAITHI LETTER VAKAITHI LETTER SHAKAI\
    \THI LETTER SSAKAITHI LETTER SAKAITHI LETTER HAKAITHI VOWEL SIGN AAKAITHI VOWEL SIGN IKAITHI VOWEL SIGN IIKAITHI VOWEL SIGN UKAITHI VOWEL SIGN UUKAITHI VOWEL SIGN EKAITHI VOWEL SIGN AIKAITHI VOWEL SIGN OKAITHI VOWEL SIGN AUKAITHI SIGN VIRAMAKAITHI SIGN NUK\
    \TAKAITHI ABBREVIATION SIGNKAITHI ENUMERATION SIGNKAITHI NUMBER SIGNKAITHI SECTION MARKKAITHI DOUBLE SECTION MARKKAITHI DANDAKAITHI DOUBLE DANDAKAITHI VOWEL SIGN VOCALIC RKAITHI NUMBER SIGN ABOVESORA SOMPENG LETTER SAHSORA SOMPENG LETTER TAHSORA SOMPENG LE\
    \TTER BAHSORA SOMPENG LETTER CAHSORA SOMPENG LETTER DAHSORA SOMPENG LETTER GAHSORA SOMPENG LETTER MAHSORA SOMPENG LETTER NGAHSORA SOMPENG LETTER LAHSORA SOMPENG LETTER NAHSORA SOMPENG LETTER VAHSORA SOMPENG LETTER PAHSORA SOMPENG LETTER YAHSORA SOMPENG LET\
    \TER RAHSORA SOMPENG LETTER HAHSORA SOMPENG LETTER KAHSORA SOMPENG LETTER JAHSORA SOMPENG LETTER NYAHSORA SOMPENG LETTER AHSORA SOMPENG LETTER EEHSORA SOMPENG LETTER IHSORA SOMPENG LETTER UHSORA SOMPENG LETTER OHSORA SOMPENG LETTER EHSORA SOMPENG LETTER MA\
    \ESORA SOMPENG DIGIT ZEROSORA SOMPENG DIGIT ONESORA SOMPENG DIGIT TWOSORA SOMPENG DIGIT THREESORA SOMPENG DIGIT FOURSORA SOMPENG DIGIT FIVESORA SOMPENG DIGIT SIXSORA SOMPENG DIGIT SEVENSORA SOMPENG DIGIT EIGHTSORA SOMPENG DIGIT NINECHAKMA SIGN CANDRABINDUC\
    \HAKMA SIGN ANUSVARACHAKMA SIGN VISARGACHAKMA LETTER AACHAKMA LETTER ICHAKMA LETTER UCHAKMA LETTER ECHAKMA LETTER KAACHAKMA LETTER KHAACHAKMA LETTER GAACHAKMA LETTER GHAACHAKMA LETTER NGAACHAKMA LETTER CAACHAKMA LETTER CHAACHAKMA LETTER JAACHAKMA LETTER JH\
    \AACHAKMA LETTER NYAACHAKMA LETTER TTAACHAKMA LETTER TTHAACHAKMA LETTER DDAACHAKMA LETTER DDHAACHAKMA LETTER NNAACHAKMA LETTER TAACHAKMA LETTER THAACHAKMA LETTER DAACHAKMA LETTER DHAACHAKMA LETTER NAACHAKMA LETTER PAACHAKMA LETTER PHAACHAKMA LETTER BAACHAK\
    \MA LETTER BHAACHAKMA LETTER MAACHAKMA LETTER YYAACHAKMA LETTER YAACHAKMA LETTER RAACHAKMA LETTER LAACHAKMA LETTER WAACHAKMA LETTER SAACHAKMA LETTER HAACHAKMA VOWEL SIGN ACHAKMA VOWEL SIGN ICHAKMA VOWEL SIGN IICHAKMA VOWEL SIGN UCHAKMA VOWEL SIGN UUCHAKMA \
    \VOWEL SIGN ECHAKMA VOWEL SIGN AICHAKMA VOWEL SIGN OCHAKMA VOWEL SIGN AUCHAKMA VOWEL SIGN OICHAKMA O MARKCHAKMA AU MARKCHAKMA VIRAMACHAKMA MAAYYAACHAKMA DIGIT ZEROCHAKMA DIGIT ONECHAKMA DIGIT TWOCHAKMA DIGIT THREECHAKMA DIGIT FOURCHAKMA DIGIT FIVECHAKMA DI\
    \GIT SIXCHAKMA DIGIT SEVENCHAKMA DIGIT EIGHTCHAKMA DIGIT NINECHAKMA SECTION MARKCHAKMA DANDACHAKMA DOUBLE DANDACHAKMA QUESTION MARKCHAKMA LETTER LHAACHAKMA VOWEL SIGN AACHAKMA VOWEL SIGN EICHAKMA LETTER VAAMAHAJANI LETTER AMAHAJANI LETTER IMAHAJANI LETTER \
    \UMAHAJANI LETTER EMAHAJANI LETTER OMAHAJANI LETTER KAMAHAJANI LETTER KHAMAHAJANI LETTER GAMAHAJANI LETTER GHAMAHAJANI LETTER CAMAHAJANI LETTER CHAMAHAJANI LETTER JAMAHAJANI LETTER JHAMAHAJANI LETTER NYAMAHAJANI LETTER TTAMAHAJANI LETTER TTHAMAHAJANI LETTE\
    \R DDAMAHAJANI LETTER DDHAMAHAJANI LETTER NNAMAHAJANI LETTER TAMAHAJANI LETTER THAMAHAJANI LETTER DAMAHAJANI LETTER DHAMAHAJANI LETTER NAMAHAJANI LETTER PAMAHAJANI LETTER PHAMAHAJANI LETTER BAMAHAJANI LETTER BHAMAHAJANI LETTER MAMAHAJANI LETTER RAMAHAJANI \
    \LETTER LAMAHAJANI LETTER VAMAHAJANI LETTER SAMAHAJANI LETTER HAMAHAJANI LETTER RRAMAHAJANI SIGN NUKTAMAHAJANI ABBREVIATION SIGNMAHAJANI SECTION MARKMAHAJANI LIGATURE SHRISHARADA SIGN CANDRABINDUSHARADA SIGN ANUSVARASHARADA SIGN VISARGASHARADA LETTER ASHAR\
    \ADA LETTER AASHARADA LETTER ISHARADA LETTER IISHARADA LETTER USHARADA LETTER UUSHARADA LETTER VOCALIC RSHARADA LETTER VOCALIC RRSHARADA LETTER VOCALIC LSHARADA LETTER VOCALIC LLSHARADA LETTER ESHARADA LETTER AISHARADA LETTER OSHARADA LETTER AUSHARADA LETT\
    \ER KASHARADA LETTER KHASHARADA LETTER GASHARADA LETTER GHASHARADA LETTER NGASHARADA LETTER CASHARADA LETTER CHASHARADA LETTER JASHARADA LETTER JHASHARADA LETTER NYASHARADA LETTER TTASHARADA LETTER TTHASHARADA LETTER DDASHARADA LETTER DDHASHARADA LETTER NN\
    \ASHARADA LETTER TASHARADA LETTER THASHARADA LETTER DASHARADA LETTER DHASHARADA LETTER NASHARADA LETTER PASHARADA LETTER PHASHARADA LETTER BASHARADA LETTER BHASHARADA LETTER MASHARADA LETTER YASHARADA LETTER RASHARADA LETTER LASHARADA LETTER LLASHARADA LET\
    \TER VASHARADA LETTER SHASHARADA LETTER SSASHARADA LETTER SASHARADA LETTER HASHARADA VOWEL SIGN AASHARADA VOWEL SIGN ISHARADA VOWEL SIGN IISHARADA VOWEL SIGN USHARADA VOWEL SIGN UUSHARADA VOWEL SIGN VOCALIC RSHARADA VOWEL SIGN VOCALIC RRSHARADA VOWEL SIGN \
    \VOCALIC LSHARADA VOWEL SIGN VOCALIC LLSHARADA VOWEL SIGN ESHARADA VOWEL SIGN AISHARADA VOWEL SIGN OSHARADA VOWEL SIGN AUSHARADA SIGN VIRAMASHARADA SIGN AVAGRAHASHARADA SIGN JIHVAMULIYASHARADA SIGN UPADHMANIYASHARADA OMSHARADA DANDASHARADA DOUBLE DANDASHAR\
    \ADA ABBREVIATION SIGNSHARADA SEPARATORSHARADA SANDHI MARKSHARADA SIGN NUKTASHARADA VOWEL MODIFIER MARKSHARADA EXTRA SHORT VOWEL MARKSHARADA SUTRA MARKSHARADA VOWEL SIGN PRISHTHAMATRA ESHARADA SIGN INVERTED CANDRABINDUSHARADA DIGIT ZEROSHARADA DIGIT ONESHA\
    \RADA DIGIT TWOSHARADA DIGIT THREESHARADA DIGIT FOURSHARADA DIGIT FIVESHARADA DIGIT SIXSHARADA DIGIT SEVENSHARADA DIGIT EIGHTSHARADA DIGIT NINESHARADA EKAMSHARADA SIGN SIDDHAMSHARADA HEADSTROKESHARADA CONTINUATION SIGNSHARADA SECTION MARK-1SHARADA SECTION \
    \MARK-2SINHALA ARCHAIC DIGIT ONESINHALA ARCHAIC DIGIT TWOSINHALA ARCHAIC DIGIT THREESINHALA ARCHAIC DIGIT FOURSINHALA ARCHAIC DIGIT FIVESINHALA ARCHAIC DIGIT SIXSINHALA ARCHAIC DIGIT SEVENSINHALA ARCHAIC DIGIT EIGHTSINHALA ARCHAIC DIGIT NINESINHALA ARCHAIC\
    \ NUMBER TENSINHALA ARCHAIC NUMBER TWENTYSINHALA ARCHAIC NUMBER THIRTYSINHALA ARCHAIC NUMBER FORTYSINHALA ARCHAIC NUMBER FIFTYSINHALA ARCHAIC NUMBER SIXTYSINHALA ARCHAIC NUMBER SEVENTYSINHALA ARCHAIC NUMBER EIGHTYSINHALA ARCHAIC NUMBER NINETYSINHALA ARCHAI\
    \C NUMBER ONE HUNDREDSINHALA ARCHAIC NUMBER ONE THOUSANDKHOJKI LETTER AKHOJKI LETTER AAKHOJKI LETTER IKHOJKI LETTER UKHOJKI LETTER EKHOJKI LETTER AIKHOJKI LETTER OKHOJKI LETTER AUKHOJKI LETTER KAKHOJKI LETTER KHAKHOJKI LETTER GAKHOJKI LETTER GGAKHOJKI LETT\
    \ER GHAKHOJKI LETTER NGAKHOJKI LETTER CAKHOJKI LETTER CHAKHOJKI LETTER JAKHOJKI LETTER JJAKHOJKI LETTER NYAKHOJKI LETTER TTAKHOJKI LETTER TTHAKHOJKI LETTER DDAKHOJKI LETTER DDHAKHOJKI LETTER NNAKHOJKI LETTER TAKHOJKI LETTER THAKHOJKI LETTER DAKHOJKI LETTER\
    \ DDDAKHOJKI LETTER DHAKHOJKI LETTER NAKHOJKI LETTER PAKHOJKI LETTER PHAKHOJKI LETTER BAKHOJKI LETTER BBAKHOJKI LETTER BHAKHOJKI LETTER MAKHOJKI LETTER YAKHOJKI LETTER RAKHOJKI LETTER LAKHOJKI LETTER VAKHOJKI LETTER SAKHOJKI LETTER HAKHOJKI LETTER LLAKHOJK\
    \I VOWEL SIGN AAKHOJKI VOWEL SIGN IKHOJKI VOWEL SIGN IIKHOJKI VOWEL SIGN UKHOJKI VOWEL SIGN EKHOJKI VOWEL SIGN AIKHOJKI VOWEL SIGN OKHOJKI VOWEL SIGN AUKHOJKI SIGN ANUSVARAKHOJKI SIGN VIRAMAKHOJKI SIGN NUKTAKHOJKI SIGN SHADDAKHOJKI DANDAKHOJKI DOUBLE DANDA\
    \KHOJKI WORD SEPARATORKHOJKI SECTION MARKKHOJKI DOUBLE SECTION MARKKHOJKI ABBREVIATION SIGNKHOJKI SIGN SUKUNKHOJKI LETTER QAKHOJKI LETTER SHORT IKHOJKI VOWEL SIGN VOCALIC RMULTANI LETTER AMULTANI LETTER IMULTANI LETTER UMULTANI LETTER EMULTANI LETTER KAMUL\
    \TANI LETTER KHAMULTANI LETTER GAMULTANI LETTER GHAMULTANI LETTER CAMULTANI LETTER CHAMULTANI LETTER JAMULTANI LETTER JJAMULTANI LETTER NYAMULTANI LETTER TTAMULTANI LETTER TTHAMULTANI LETTER DDAMULTANI LETTER DDDAMULTANI LETTER DDHAMULTANI LETTER NNAMULTAN\
    \I LETTER TAMULTANI LETTER THAMULTANI LETTER DAMULTANI LETTER DHAMULTANI LETTER NAMULTANI LETTER PAMULTANI LETTER PHAMULTANI LETTER BAMULTANI LETTER BHAMULTANI LETTER MAMULTANI LETTER YAMULTANI LETTER RAMULTANI LETTER LAMULTANI LETTER VAMULTANI LETTER SAMU\
    \LTANI LETTER HAMULTANI LETTER RRAMULTANI LETTER RHAMULTANI SECTION MARKKHUDAWADI LETTER AKHUDAWADI LETTER AAKHUDAWADI LETTER IKHUDAWADI LETTER IIKHUDAWADI LETTER UKHUDAWADI LETTER UUKHUDAWADI LETTER EKHUDAWADI LETTER AIKHUDAWADI LETTER OKHUDAWADI LETTER A\
    \UKHUDAWADI LETTER KAKHUDAWADI LETTER KHAKHUDAWADI LETTER GAKHUDAWADI LETTER GGAKHUDAWADI LETTER GHAKHUDAWADI LETTER NGAKHUDAWADI LETTER CAKHUDAWADI LETTER CHAKHUDAWADI LETTER JAKHUDAWADI LETTER JJAKHUDAWADI LETTER JHAKHUDAWADI LETTER NYAKHUDAWADI LETTER T\
    \TAKHUDAWADI LETTER TTHAKHUDAWADI LETTER DDAKHUDAWADI LETTER DDDAKHUDAWADI LETTER RRAKHUDAWADI LETTER DDHAKHUDAWADI LETTER NNAKHUDAWADI LETTER TAKHUDAWADI LETTER THAKHUDAWADI LETTER DAKHUDAWADI LETTER DHAKHUDAWADI LETTER NAKHUDAWADI LETTER PAKHUDAWADI LETT\
    \ER PHAKHUDAWADI LETTER BAKHUDAWADI LETTER BBAKHUDAWADI LETTER BHAKHUDAWADI LETTER MAKHUDAWADI LETTER YAKHUDAWADI LETTER RAKHUDAWADI LETTER LAKHUDAWADI LETTER VAKHUDAWADI LETTER SHAKHUDAWADI LETTER SAKHUDAWADI LETTER HAKHUDAWADI SIGN ANUSVARAKHUDAWADI VOWE\
    \L SIGN AAKHUDAWADI VOWEL SIGN IKHUDAWADI VOWEL SIGN IIKHUDAWADI VOWEL SIGN UKHUDAWADI VOWEL SIGN UUKHUDAWADI VOWEL SIGN EKHUDAWADI VOWEL SIGN AIKHUDAWADI VOWEL SIGN OKHUDAWADI VOWEL SIGN AUKHUDAWADI SIGN NUKTAKHUDAWADI SIGN VIRAMAKHUDAWADI DIGIT ZEROKHUDA\
    \WADI DIGIT ONEKHUDAWADI DIGIT TWOKHUDAWADI DIGIT THREEKHUDAWADI DIGIT FOURKHUDAWADI DIGIT FIVEKHUDAWADI DIGIT SIXKHUDAWADI DIGIT SEVENKHUDAWADI DIGIT EIGHTKHUDAWADI DIGIT NINEGRANTHA SIGN COMBINING ANUSVARA ABOVEGRANTHA SIGN CANDRABINDUGRANTHA SIGN ANUSVA\
    \RAGRANTHA SIGN VISARGAGRANTHA LETTER AGRANTHA LETTER AAGRANTHA LETTER IGRANTHA LETTER IIGRANTHA LETTER UGRANTHA LETTER UUGRANTHA LETTER VOCALIC RGRANTHA LETTER VOCALIC LGRANTHA LETTER EEGRANTHA LETTER AIGRANTHA LETTER OOGRANTHA LETTER AUGRANTHA LETTER KAG\
    \RANTHA LETTER KHAGRANTHA LETTER GAGRANTHA LETTER GHAGRANTHA LETTER NGAGRANTHA LETTER CAGRANTHA LETTER CHAGRANTHA LETTER JAGRANTHA LETTER JHAGRANTHA LETTER NYAGRANTHA LETTER TTAGRANTHA LETTER TTHAGRANTHA LETTER DDAGRANTHA LETTER DDHAGRANTHA LETTER NNAGRANT\
    \HA LETTER TAGRANTHA LETTER THAGRANTHA LETTER DAGRANTHA LETTER DHAGRANTHA LETTER NAGRANTHA LETTER PAGRANTHA LETTER PHAGRANTHA LETTER BAGRANTHA LETTER BHAGRANTHA LETTER MAGRANTHA LETTER YAGRANTHA LETTER RAGRANTHA LETTER LAGRANTHA LETTER LLAGRANTHA LETTER VA\
    \GRANTHA LETTER SHAGRANTHA LETTER SSAGRANTHA LETTER SAGRANTHA LETTER HACOMBINING BINDU BELOWGRANTHA SIGN NUKTAGRANTHA SIGN AVAGRAHAGRANTHA VOWEL SIGN AAGRANTHA VOWEL SIGN IGRANTHA VOWEL SIGN IIGRANTHA VOWEL SIGN UGRANTHA VOWEL SIGN UUGRANTHA VOWEL SIGN VOC\
    \ALIC RGRANTHA VOWEL SIGN VOCALIC RRGRANTHA VOWEL SIGN EEGRANTHA VOWEL SIGN AIGRANTHA VOWEL SIGN OOGRANTHA VOWEL SIGN AUGRANTHA SIGN VIRAMAGRANTHA OMGRANTHA AU LENGTH MARKGRANTHA SIGN PLUTAGRANTHA LETTER VEDIC ANUSVARAGRANTHA LETTER VEDIC DOUBLE ANUSVARAGR\
    \ANTHA LETTER VOCALIC RRGRANTHA LETTER VOCALIC LLGRANTHA VOWEL SIGN VOCALIC LGRANTHA VOWEL SIGN VOCALIC LLCOMBINING GRANTHA DIGIT ZEROCOMBINING GRANTHA DIGIT ONECOMBINING GRANTHA DIGIT TWOCOMBINING GRANTHA DIGIT THREECOMBINING GRANTHA DIGIT FOURCOMBINING G\
    \RANTHA DIGIT FIVECOMBINING GRANTHA DIGIT SIXCOMBINING GRANTHA LETTER ACOMBINING GRANTHA LETTER KACOMBINING GRANTHA LETTER NACOMBINING GRANTHA LETTER VICOMBINING GRANTHA LETTER PANEWA LETTER ANEWA LETTER AANEWA LETTER INEWA LETTER IINEWA LETTER UNEWA LETTE\
    \R UUNEWA LETTER VOCALIC RNEWA LETTER VOCALIC RRNEWA LETTER VOCALIC LNEWA LETTER VOCALIC LLNEWA LETTER ENEWA LETTER AINEWA LETTER ONEWA LETTER AUNEWA LETTER KANEWA LETTER KHANEWA LETTER GANEWA LETTER GHANEWA LETTER NGANEWA LETTER NGHANEWA LETTER CANEWA LET\
    \TER CHANEWA LETTER JANEWA LETTER JHANEWA LETTER NYANEWA LETTER NYHANEWA LETTER TTANEWA LETTER TTHANEWA LETTER DDANEWA LETTER DDHANEWA LETTER NNANEWA LETTER TANEWA LETTER THANEWA LETTER DANEWA LETTER DHANEWA LETTER NANEWA LETTER NHANEWA LETTER PANEWA LETTE\
    \R PHANEWA LETTER BANEWA LETTER BHANEWA LETTER MANEWA LETTER MHANEWA LETTER YANEWA LETTER RANEWA LETTER RHANEWA LETTER LANEWA LETTER LHANEWA LETTER WANEWA LETTER SHANEWA LETTER SSANEWA LETTER SANEWA LETTER HANEWA VOWEL SIGN AANEWA VOWEL SIGN INEWA VOWEL SI\
    \GN IINEWA VOWEL SIGN UNEWA VOWEL SIGN UUNEWA VOWEL SIGN VOCALIC RNEWA VOWEL SIGN VOCALIC RRNEWA VOWEL SIGN VOCALIC LNEWA VOWEL SIGN VOCALIC LLNEWA VOWEL SIGN ENEWA VOWEL SIGN AINEWA VOWEL SIGN ONEWA VOWEL SIGN AUNEWA SIGN VIRAMANEWA SIGN CANDRABINDUNEWA S\
    \IGN ANUSVARANEWA SIGN VISARGANEWA SIGN NUKTANEWA SIGN AVAGRAHANEWA SIGN FINAL ANUSVARANEWA OMNEWA SIDDHINEWA DANDANEWA DOUBLE DANDANEWA COMMANEWA GAP FILLERNEWA ABBREVIATION SIGNNEWA DIGIT ZERONEWA DIGIT ONENEWA DIGIT TWONEWA DIGIT THREENEWA DIGIT FOURNEW\
    \A DIGIT FIVENEWA DIGIT SIXNEWA DIGIT SEVENNEWA DIGIT EIGHTNEWA DIGIT NINENEWA DOUBLE COMMANEWA PLACEHOLDER MARKNEWA INSERTION SIGNNEWA SANDHI MARKNEWA LETTER VEDIC ANUSVARANEWA SIGN JIHVAMULIYANEWA SIGN UPADHMANIYATIRHUTA ANJITIRHUTA LETTER ATIRHUTA LETTE\
    \R AATIRHUTA LETTER ITIRHUTA LETTER IITIRHUTA LETTER UTIRHUTA LETTER UUTIRHUTA LETTER VOCALIC RTIRHUTA LETTER VOCALIC RRTIRHUTA LETTER VOCALIC LTIRHUTA LETTER VOCALIC LLTIRHUTA LETTER ETIRHUTA LETTER AITIRHUTA LETTER OTIRHUTA LETTER AUTIRHUTA LETTER KATIRH\
    \UTA LETTER KHATIRHUTA LETTER GATIRHUTA LETTER GHATIRHUTA LETTER NGATIRHUTA LETTER CATIRHUTA LETTER CHATIRHUTA LETTER JATIRHUTA LETTER JHATIRHUTA LETTER NYATIRHUTA LETTER TTATIRHUTA LETTER TTHATIRHUTA LETTER DDATIRHUTA LETTER DDHATIRHUTA LETTER NNATIRHUTA \
    \LETTER TATIRHUTA LETTER THATIRHUTA LETTER DATIRHUTA LETTER DHATIRHUTA LETTER NATIRHUTA LETTER PATIRHUTA LETTER PHATIRHUTA LETTER BATIRHUTA LETTER BHATIRHUTA LETTER MATIRHUTA LETTER YATIRHUTA LETTER RATIRHUTA LETTER LATIRHUTA LETTER VATIRHUTA LETTER SHATIR\
    \HUTA LETTER SSATIRHUTA LETTER SATIRHUTA LETTER HATIRHUTA VOWEL SIGN AATIRHUTA VOWEL SIGN ITIRHUTA VOWEL SIGN IITIRHUTA VOWEL SIGN UTIRHUTA VOWEL SIGN UUTIRHUTA VOWEL SIGN VOCALIC RTIRHUTA VOWEL SIGN VOCALIC RRTIRHUTA VOWEL SIGN VOCALIC LTIRHUTA VOWEL SIGN\
    \ VOCALIC LLTIRHUTA VOWEL SIGN ETIRHUTA VOWEL SIGN SHORT ETIRHUTA VOWEL SIGN AITIRHUTA VOWEL SIGN OTIRHUTA VOWEL SIGN SHORT OTIRHUTA VOWEL SIGN AUTIRHUTA SIGN CANDRABINDUTIRHUTA SIGN ANUSVARATIRHUTA SIGN VISARGATIRHUTA SIGN VIRAMATIRHUTA SIGN NUKTATIRHUTA \
    \SIGN AVAGRAHATIRHUTA GVANGTIRHUTA ABBREVIATION SIGNTIRHUTA OMTIRHUTA DIGIT ZEROTIRHUTA DIGIT ONETIRHUTA DIGIT TWOTIRHUTA DIGIT THREETIRHUTA DIGIT FOURTIRHUTA DIGIT FIVETIRHUTA DIGIT SIXTIRHUTA DIGIT SEVENTIRHUTA DIGIT EIGHTTIRHUTA DIGIT NINESIDDHAM LETTER\
    \ ASIDDHAM LETTER AASIDDHAM LETTER ISIDDHAM LETTER IISIDDHAM LETTER USIDDHAM LETTER UUSIDDHAM LETTER VOCALIC RSIDDHAM LETTER VOCALIC RRSIDDHAM LETTER VOCALIC LSIDDHAM LETTER VOCALIC LLSIDDHAM LETTER ESIDDHAM LETTER AISIDDHAM LETTER OSIDDHAM LETTER AUSIDDHA\
    \M LETTER KASIDDHAM LETTER KHASIDDHAM LETTER GASIDDHAM LETTER GHASIDDHAM LETTER NGASIDDHAM LETTER CASIDDHAM LETTER CHASIDDHAM LETTER JASIDDHAM LETTER JHASIDDHAM LETTER NYASIDDHAM LETTER TTASIDDHAM LETTER TTHASIDDHAM LETTER DDASIDDHAM LETTER DDHASIDDHAM LET\
    \TER NNASIDDHAM LETTER TASIDDHAM LETTER THASIDDHAM LETTER DASIDDHAM LETTER DHASIDDHAM LETTER NASIDDHAM LETTER PASIDDHAM LETTER PHASIDDHAM LETTER BASIDDHAM LETTER BHASIDDHAM LETTER MASIDDHAM LETTER YASIDDHAM LETTER RASIDDHAM LETTER LASIDDHAM LETTER VASIDDHA\
    \M LETTER SHASIDDHAM LETTER SSASIDDHAM LETTER SASIDDHAM LETTER HASIDDHAM VOWEL SIGN AASIDDHAM VOWEL SIGN ISIDDHAM VOWEL SIGN IISIDDHAM VOWEL SIGN USIDDHAM VOWEL SIGN UUSIDDHAM VOWEL SIGN VOCALIC RSIDDHAM VOWEL SIGN VOCALIC RRSIDDHAM VOWEL SIGN ESIDDHAM VOW\
    \EL SIGN AISIDDHAM VOWEL SIGN OSIDDHAM VOWEL SIGN AUSIDDHAM SIGN CANDRABINDUSIDDHAM SIGN ANUSVARASIDDHAM SIGN VISARGASIDDHAM SIGN VIRAMASIDDHAM SIGN NUKTASIDDHAM SIGN SIDDHAMSIDDHAM DANDASIDDHAM DOUBLE DANDASIDDHAM SEPARATOR DOTSIDDHAM SEPARATOR BARSIDDHAM\
    \ REPETITION MARK-1SIDDHAM REPETITION MARK-2SIDDHAM REPETITION MARK-3SIDDHAM END OF TEXT MARKSIDDHAM SECTION MARK WITH TRIDENT AND U-SHAPED ORNAMENTSSIDDHAM SECTION MARK WITH TRIDENT AND DOTTED CRESCENTSSIDDHAM SECTION MARK WITH RAYS AND DOTTED CRESCENTSSI\
    \DDHAM SECTION MARK WITH RAYS AND DOTTED DOUBLE CRESCENTSSIDDHAM SECTION MARK WITH RAYS AND DOTTED TRIPLE CRESCENTSSIDDHAM SECTION MARK DOUBLE RINGSIDDHAM SECTION MARK DOUBLE RING WITH RAYSSIDDHAM SECTION MARK WITH DOUBLE CRESCENTSSIDDHAM SECTION MARK WITH\
    \ TRIPLE CRESCENTSSIDDHAM SECTION MARK WITH QUADRUPLE CRESCENTSSIDDHAM SECTION MARK WITH SEPTUPLE CRESCENTSSIDDHAM SECTION MARK WITH CIRCLES AND RAYSSIDDHAM SECTION MARK WITH CIRCLES AND TWO ENCLOSURESSIDDHAM SECTION MARK WITH CIRCLES AND FOUR ENCLOSURESSI\
    \DDHAM LETTER THREE-CIRCLE ALTERNATE ISIDDHAM LETTER TWO-CIRCLE ALTERNATE ISIDDHAM LETTER TWO-CIRCLE ALTERNATE IISIDDHAM LETTER ALTERNATE USIDDHAM VOWEL SIGN ALTERNATE USIDDHAM VOWEL SIGN ALTERNATE UUMODI LETTER AMODI LETTER AAMODI LETTER IMODI LETTER IIMO\
    \DI LETTER UMODI LETTER UUMODI LETTER VOCALIC RMODI LETTER VOCALIC RRMODI LETTER VOCALIC LMODI LETTER VOCALIC LLMODI LETTER EMODI LETTER AIMODI LETTER OMODI LETTER AUMODI LETTER KAMODI LETTER KHAMODI LETTER GAMODI LETTER GHAMODI LETTER NGAMODI LETTER CAMOD\
    \I LETTER CHAMODI LETTER JAMODI LETTER JHAMODI LETTER NYAMODI LETTER TTAMODI LETTER TTHAMODI LETTER DDAMODI LETTER DDHAMODI LETTER NNAMODI LETTER TAMODI LETTER THAMODI LETTER DAMODI LETTER DHAMODI LETTER NAMODI LETTER PAMODI LETTER PHAMODI LETTER BAMODI LE\
    \TTER BHAMODI LETTER MAMODI LETTER YAMODI LETTER RAMODI LETTER LAMODI LETTER VAMODI LETTER SHAMODI LETTER SSAMODI LETTER SAMODI LETTER HAMODI LETTER LLAMODI VOWEL SIGN AAMODI VOWEL SIGN IMODI VOWEL SIGN IIMODI VOWEL SIGN UMODI VOWEL SIGN UUMODI VOWEL SIGN \
    \VOCALIC RMODI VOWEL SIGN VOCALIC RRMODI VOWEL SIGN VOCALIC LMODI VOWEL SIGN VOCALIC LLMODI VOWEL SIGN EMODI VOWEL SIGN AIMODI VOWEL SIGN OMODI VOWEL SIGN AUMODI SIGN ANUSVARAMODI SIGN VISARGAMODI SIGN VIRAMAMODI SIGN ARDHACANDRAMODI DANDAMODI DOUBLE DANDA\
    \MODI ABBREVIATION SIGNMODI SIGN HUVAMODI DIGIT ZEROMODI DIGIT ONEMODI DIGIT TWOMODI DIGIT THREEMODI DIGIT FOURMODI DIGIT FIVEMODI DIGIT SIXMODI DIGIT SEVENMODI DIGIT EIGHTMODI DIGIT NINEMONGOLIAN BIRGA WITH ORNAMENTMONGOLIAN ROTATED BIRGAMONGOLIAN DOUBLE \
    \BIRGA WITH ORNAMENTMONGOLIAN TRIPLE BIRGA WITH ORNAMENTMONGOLIAN BIRGA WITH DOUBLE ORNAMENTMONGOLIAN ROTATED BIRGA WITH ORNAMENTMONGOLIAN ROTATED BIRGA WITH DOUBLE ORNAMENTMONGOLIAN INVERTED BIRGAMONGOLIAN INVERTED BIRGA WITH DOUBLE ORNAMENTMONGOLIAN SWIR\
    \L BIRGAMONGOLIAN SWIRL BIRGA WITH ORNAMENTMONGOLIAN SWIRL BIRGA WITH DOUBLE ORNAMENTMONGOLIAN TURNED SWIRL BIRGA WITH DOUBLE ORNAMENTTAKRI LETTER ATAKRI LETTER AATAKRI LETTER ITAKRI LETTER IITAKRI LETTER UTAKRI LETTER UUTAKRI LETTER ETAKRI LETTER AITAKRI \
    \LETTER OTAKRI LETTER AUTAKRI LETTER KATAKRI LETTER KHATAKRI LETTER GATAKRI LETTER GHATAKRI LETTER NGATAKRI LETTER CATAKRI LETTER CHATAKRI LETTER JATAKRI LETTER JHATAKRI LETTER NYATAKRI LETTER TTATAKRI LETTER TTHATAKRI LETTER DDATAKRI LETTER DDHATAKRI LETT\
    \ER NNATAKRI LETTER TATAKRI LETTER THATAKRI LETTER DATAKRI LETTER DHATAKRI LETTER NATAKRI LETTER PATAKRI LETTER PHATAKRI LETTER BATAKRI LETTER BHATAKRI LETTER MATAKRI LETTER YATAKRI LETTER RATAKRI LETTER LATAKRI LETTER VATAKRI LETTER SHATAKRI LETTER SATAKR\
    \I LETTER HATAKRI LETTER RRATAKRI SIGN ANUSVARATAKRI SIGN VISARGATAKRI VOWEL SIGN AATAKRI VOWEL SIGN ITAKRI VOWEL SIGN IITAKRI VOWEL SIGN UTAKRI VOWEL SIGN UUTAKRI VOWEL SIGN ETAKRI VOWEL SIGN AITAKRI VOWEL SIGN OTAKRI VOWEL SIGN AUTAKRI SIGN VIRAMATAKRI S\
    \IGN NUKTATAKRI LETTER ARCHAIC KHATAKRI ABBREVIATION SIGNTAKRI DIGIT ZEROTAKRI DIGIT ONETAKRI DIGIT TWOTAKRI DIGIT THREETAKRI DIGIT FOURTAKRI DIGIT FIVETAKRI DIGIT SIXTAKRI DIGIT SEVENTAKRI DIGIT EIGHTTAKRI DIGIT NINEAHOM LETTER KAAHOM LETTER KHAAHOM LETTE\
    \R NGAAHOM LETTER NAAHOM LETTER TAAHOM LETTER ALTERNATE TAAHOM LETTER PAAHOM LETTER PHAAHOM LETTER BAAHOM LETTER MAAHOM LETTER JAAHOM LETTER CHAAHOM LETTER THAAHOM LETTER RAAHOM LETTER LAAHOM LETTER SAAHOM LETTER NYAAHOM LETTER HAAHOM LETTER AAHOM LETTER D\
    \AAHOM LETTER DHAAHOM LETTER GAAHOM LETTER ALTERNATE GAAHOM LETTER GHAAHOM LETTER BHAAHOM LETTER JHAAHOM LETTER ALTERNATE BAAHOM CONSONANT SIGN MEDIAL LAAHOM CONSONANT SIGN MEDIAL RAAHOM CONSONANT SIGN MEDIAL LIGATING RAAHOM VOWEL SIGN AAHOM VOWEL SIGN AAA\
    \HOM VOWEL SIGN IAHOM VOWEL SIGN IIAHOM VOWEL SIGN UAHOM VOWEL SIGN UUAHOM VOWEL SIGN EAHOM VOWEL SIGN AWAHOM VOWEL SIGN OAHOM VOWEL SIGN AIAHOM VOWEL SIGN AMAHOM SIGN KILLERAHOM DIGIT ZEROAHOM DIGIT ONEAHOM DIGIT TWOAHOM DIGIT THREEAHOM DIGIT FOURAHOM DIG\
    \IT FIVEAHOM DIGIT SIXAHOM DIGIT SEVENAHOM DIGIT EIGHTAHOM DIGIT NINEAHOM NUMBER TENAHOM NUMBER TWENTYAHOM SIGN SMALL SECTIONAHOM SIGN SECTIONAHOM SIGN RULAIAHOM SYMBOL VIAHOM LETTER CAAHOM LETTER TTAAHOM LETTER TTHAAHOM LETTER DDAAHOM LETTER DDHAAHOM LETT\
    \ER NNAAHOM LETTER LLADOGRA LETTER ADOGRA LETTER AADOGRA LETTER IDOGRA LETTER IIDOGRA LETTER UDOGRA LETTER UUDOGRA LETTER EDOGRA LETTER AIDOGRA LETTER ODOGRA LETTER AUDOGRA LETTER KADOGRA LETTER KHADOGRA LETTER GADOGRA LETTER GHADOGRA LETTER NGADOGRA LETTE\
    \R CADOGRA LETTER CHADOGRA LETTER JADOGRA LETTER JHADOGRA LETTER NYADOGRA LETTER TTADOGRA LETTER TTHADOGRA LETTER DDADOGRA LETTER DDHADOGRA LETTER NNADOGRA LETTER TADOGRA LETTER THADOGRA LETTER DADOGRA LETTER DHADOGRA LETTER NADOGRA LETTER PADOGRA LETTER P\
    \HADOGRA LETTER BADOGRA LETTER BHADOGRA LETTER MADOGRA LETTER YADOGRA LETTER RADOGRA LETTER LADOGRA LETTER VADOGRA LETTER SHADOGRA LETTER SSADOGRA LETTER SADOGRA LETTER HADOGRA LETTER RRADOGRA VOWEL SIGN AADOGRA VOWEL SIGN IDOGRA VOWEL SIGN IIDOGRA VOWEL S\
    \IGN UDOGRA VOWEL SIGN UUDOGRA VOWEL SIGN VOCALIC RDOGRA VOWEL SIGN VOCALIC RRDOGRA VOWEL SIGN EDOGRA VOWEL SIGN AIDOGRA VOWEL SIGN ODOGRA VOWEL SIGN AUDOGRA SIGN ANUSVARADOGRA SIGN VISARGADOGRA SIGN VIRAMADOGRA SIGN NUKTADOGRA ABBREVIATION SIGNWARANG CITI\
    \ CAPITAL LETTER NGAAWARANG CITI CAPITAL LETTER AWARANG CITI CAPITAL LETTER WIWARANG CITI CAPITAL LETTER YUWARANG CITI CAPITAL LETTER YAWARANG CITI CAPITAL LETTER YOWARANG CITI CAPITAL LETTER IIWARANG CITI CAPITAL LETTER UUWARANG CITI CAPITAL LETTER EWARAN\
    \G CITI CAPITAL LETTER OWARANG CITI CAPITAL LETTER ANGWARANG CITI CAPITAL LETTER GAWARANG CITI CAPITAL LETTER KOWARANG CITI CAPITAL LETTER ENYWARANG CITI CAPITAL LETTER YUJWARANG CITI CAPITAL LETTER UCWARANG CITI CAPITAL LETTER ENNWARANG CITI CAPITAL LETTE\
    \R ODDWARANG CITI CAPITAL LETTER TTEWARANG CITI CAPITAL LETTER NUNGWARANG CITI CAPITAL LETTER DAWARANG CITI CAPITAL LETTER ATWARANG CITI CAPITAL LETTER AMWARANG CITI CAPITAL LETTER BUWARANG CITI CAPITAL LETTER PUWARANG CITI CAPITAL LETTER HIYOWARANG CITI C\
    \APITAL LETTER HOLOWARANG CITI CAPITAL LETTER HORRWARANG CITI CAPITAL LETTER HARWARANG CITI CAPITAL LETTER SSUUWARANG CITI CAPITAL LETTER SIIWARANG CITI CAPITAL LETTER VIYOWARANG CITI SMALL LETTER NGAAWARANG CITI SMALL LETTER AWARANG CITI SMALL LETTER WIWA\
    \RANG CITI SMALL LETTER YUWARANG CITI SMALL LETTER YAWARANG CITI SMALL LETTER YOWARANG CITI SMALL LETTER IIWARANG CITI SMALL LETTER UUWARANG CITI SMALL LETTER EWARANG CITI SMALL LETTER OWARANG CITI SMALL LETTER ANGWARANG CITI SMALL LETTER GAWARANG CITI SMA\
    \LL LETTER KOWARANG CITI SMALL LETTER ENYWARANG CITI SMALL LETTER YUJWARANG CITI SMALL LETTER UCWARANG CITI SMALL LETTER ENNWARANG CITI SMALL LETTER ODDWARANG CITI SMALL LETTER TTEWARANG CITI SMALL LETTER NUNGWARANG CITI SMALL LETTER DAWARANG CITI SMALL LE\
    \TTER ATWARANG CITI SMALL LETTER AMWARANG CITI SMALL LETTER BUWARANG CITI SMALL LETTER PUWARANG CITI SMALL LETTER HIYOWARANG CITI SMALL LETTER HOLOWARANG CITI SMALL LETTER HORRWARANG CITI SMALL LETTER HARWARANG CITI SMALL LETTER SSUUWARANG CITI SMALL LETTE\
    \R SIIWARANG CITI SMALL LETTER VIYOWARANG CITI DIGIT ZEROWARANG CITI DIGIT ONEWARANG CITI DIGIT TWOWARANG CITI DIGIT THREEWARANG CITI DIGIT FOURWARANG CITI DIGIT FIVEWARANG CITI DIGIT SIXWARANG CITI DIGIT SEVENWARANG CITI DIGIT EIGHTWARANG CITI DIGIT NINEW\
    \ARANG CITI NUMBER TENWARANG CITI NUMBER TWENTYWARANG CITI NUMBER THIRTYWARANG CITI NUMBER FORTYWARANG CITI NUMBER FIFTYWARANG CITI NUMBER SIXTYWARANG CITI NUMBER SEVENTYWARANG CITI NUMBER EIGHTYWARANG CITI NUMBER NINETYWARANG CITI OMDIVES AKURU LETTER ADI\
    \VES AKURU LETTER AADIVES AKURU LETTER IDIVES AKURU LETTER IIDIVES AKURU LETTER UDIVES AKURU LETTER UUDIVES AKURU LETTER EDIVES AKURU LETTER ODIVES AKURU LETTER KADIVES AKURU LETTER KHADIVES AKURU LETTER GADIVES AKURU LETTER GHADIVES AKURU LETTER NGADIVES \
    \AKURU LETTER CADIVES AKURU LETTER CHADIVES AKURU LETTER JADIVES AKURU LETTER NYADIVES AKURU LETTER TTADIVES AKURU LETTER DDADIVES AKURU LETTER DDHADIVES AKURU LETTER NNADIVES AKURU LETTER TADIVES AKURU LETTER THADIVES AKURU LETTER DADIVES AKURU LETTER DHA\
    \DIVES AKURU LETTER NADIVES AKURU LETTER PADIVES AKURU LETTER PHADIVES AKURU LETTER BADIVES AKURU LETTER BHADIVES AKURU LETTER MADIVES AKURU LETTER YADIVES AKURU LETTER YYADIVES AKURU LETTER RADIVES AKURU LETTER LADIVES AKURU LETTER VADIVES AKURU LETTER SH\
    \ADIVES AKURU LETTER SSADIVES AKURU LETTER SADIVES AKURU LETTER HADIVES AKURU LETTER LLADIVES AKURU LETTER ZADIVES AKURU VOWEL SIGN AADIVES AKURU VOWEL SIGN IDIVES AKURU VOWEL SIGN IIDIVES AKURU VOWEL SIGN UDIVES AKURU VOWEL SIGN UUDIVES AKURU VOWEL SIGN E\
    \DIVES AKURU VOWEL SIGN AIDIVES AKURU VOWEL SIGN ODIVES AKURU SIGN ANUSVARADIVES AKURU SIGN CANDRABINDUDIVES AKURU SIGN HALANTADIVES AKURU VIRAMADIVES AKURU PREFIXED NASAL SIGNDIVES AKURU MEDIAL YADIVES AKURU INITIAL RADIVES AKURU MEDIAL RADIVES AKURU SIGN\
    \ NUKTADIVES AKURU DOUBLE DANDADIVES AKURU GAP FILLERDIVES AKURU END OF TEXT MARKDIVES AKURU DIGIT ZERODIVES AKURU DIGIT ONEDIVES AKURU DIGIT TWODIVES AKURU DIGIT THREEDIVES AKURU DIGIT FOURDIVES AKURU DIGIT FIVEDIVES AKURU DIGIT SIXDIVES AKURU DIGIT SEVEN\
    \DIVES AKURU DIGIT EIGHTDIVES AKURU DIGIT NINENANDINAGARI LETTER ANANDINAGARI LETTER AANANDINAGARI LETTER INANDINAGARI LETTER IINANDINAGARI LETTER UNANDINAGARI LETTER UUNANDINAGARI LETTER VOCALIC RNANDINAGARI LETTER VOCALIC RRNANDINAGARI LETTER ENANDINAGAR\
    \I LETTER AINANDINAGARI LETTER ONANDINAGARI LETTER AUNANDINAGARI LETTER KANANDINAGARI LETTER KHANANDINAGARI LETTER GANANDINAGARI LETTER GHANANDINAGARI LETTER NGANANDINAGARI LETTER CANANDINAGARI LETTER CHANANDINAGARI LETTER JANANDINAGARI LETTER JHANANDINAGA\
    \RI LETTER NYANANDINAGARI LETTER TTANANDINAGARI LETTER TTHANANDINAGARI LETTER DDANANDINAGARI LETTER DDHANANDINAGARI LETTER NNANANDINAGARI LETTER TANANDINAGARI LETTER THANANDINAGARI LETTER DANANDINAGARI LETTER DHANANDINAGARI LETTER NANANDINAGARI LETTER PANA\
    \NDINAGARI LETTER PHANANDINAGARI LETTER BANANDINAGARI LETTER BHANANDINAGARI LETTER MANANDINAGARI LETTER YANANDINAGARI LETTER RANANDINAGARI LETTER LANANDINAGARI LETTER VANANDINAGARI LETTER SHANANDINAGARI LETTER SSANANDINAGARI LETTER SANANDINAGARI LETTER HAN\
    \ANDINAGARI LETTER LLANANDINAGARI LETTER RRANANDINAGARI VOWEL SIGN AANANDINAGARI VOWEL SIGN INANDINAGARI VOWEL SIGN IINANDINAGARI VOWEL SIGN UNANDINAGARI VOWEL SIGN UUNANDINAGARI VOWEL SIGN VOCALIC RNANDINAGARI VOWEL SIGN VOCALIC RRNANDINAGARI VOWEL SIGN E\
    \NANDINAGARI VOWEL SIGN AINANDINAGARI VOWEL SIGN ONANDINAGARI VOWEL SIGN AUNANDINAGARI SIGN ANUSVARANANDINAGARI SIGN VISARGANANDINAGARI SIGN VIRAMANANDINAGARI SIGN AVAGRAHANANDINAGARI SIGN SIDDHAMNANDINAGARI HEADSTROKENANDINAGARI VOWEL SIGN PRISHTHAMATRA E\
    \ZANABAZAR SQUARE LETTER AZANABAZAR SQUARE VOWEL SIGN IZANABAZAR SQUARE VOWEL SIGN UEZANABAZAR SQUARE VOWEL SIGN UZANABAZAR SQUARE VOWEL SIGN EZANABAZAR SQUARE VOWEL SIGN OEZANABAZAR SQUARE VOWEL SIGN OZANABAZAR SQUARE VOWEL SIGN AIZANABAZAR SQUARE VOWEL S\
    \IGN AUZANABAZAR SQUARE VOWEL SIGN REVERSED IZANABAZAR SQUARE VOWEL LENGTH MARKZANABAZAR SQUARE LETTER KAZANABAZAR SQUARE LETTER KHAZANABAZAR SQUARE LETTER GAZANABAZAR SQUARE LETTER GHAZANABAZAR SQUARE LETTER NGAZANABAZAR SQUARE LETTER CAZANABAZAR SQUARE L\
    \ETTER CHAZANABAZAR SQUARE LETTER JAZANABAZAR SQUARE LETTER NYAZANABAZAR SQUARE LETTER TTAZANABAZAR SQUARE LETTER TTHAZANABAZAR SQUARE LETTER DDAZANABAZAR SQUARE LETTER DDHAZANABAZAR SQUARE LETTER NNAZANABAZAR SQUARE LETTER TAZANABAZAR SQUARE LETTER THAZAN\
    \ABAZAR SQUARE LETTER DAZANABAZAR SQUARE LETTER DHAZANABAZAR SQUARE LETTER NAZANABAZAR SQUARE LETTER PAZANABAZAR SQUARE LETTER PHAZANABAZAR SQUARE LETTER BAZANABAZAR SQUARE LETTER BHAZANABAZAR SQUARE LETTER MAZANABAZAR SQUARE LETTER TSAZANABAZAR SQUARE LET\
    \TER TSHAZANABAZAR SQUARE LETTER DZAZANABAZAR SQUARE LETTER DZHAZANABAZAR SQUARE LETTER ZHAZANABAZAR SQUARE LETTER ZAZANABAZAR SQUARE LETTER -AZANABAZAR SQUARE LETTER YAZANABAZAR SQUARE LETTER RAZANABAZAR SQUARE LETTER LAZANABAZAR SQUARE LETTER VAZANABAZAR\
    \ SQUARE LETTER SHAZANABAZAR SQUARE LETTER SSAZANABAZAR SQUARE LETTER SAZANABAZAR SQUARE LETTER HAZANABAZAR SQUARE LETTER KSSAZANABAZAR SQUARE FINAL CONSONANT MARKZANABAZAR SQUARE SIGN VIRAMAZANABAZAR SQUARE SIGN CANDRABINDUZANABAZAR SQUARE SIGN CANDRABIND\
    \U WITH ORNAMENTZANABAZAR SQUARE SIGN CANDRA WITH ORNAMENTZANABAZAR SQUARE SIGN ANUSVARAZANABAZAR SQUARE SIGN VISARGAZANABAZAR SQUARE CLUSTER-INITIAL LETTER RAZANABAZAR SQUARE CLUSTER-FINAL LETTER YAZANABAZAR SQUARE CLUSTER-FINAL LETTER RAZANABAZAR SQUARE \
    \CLUSTER-FINAL LETTER LAZANABAZAR SQUARE CLUSTER-FINAL LETTER VAZANABAZAR SQUARE INITIAL HEAD MARKZANABAZAR SQUARE CLOSING HEAD MARKZANABAZAR SQUARE MARK TSHEGZANABAZAR SQUARE MARK SHADZANABAZAR SQUARE MARK DOUBLE SHADZANABAZAR SQUARE MARK LONG TSHEGZANABA\
    \ZAR SQUARE INITIAL DOUBLE-LINED HEAD MARKZANABAZAR SQUARE CLOSING DOUBLE-LINED HEAD MARKZANABAZAR SQUARE SUBJOINERSOYOMBO LETTER ASOYOMBO VOWEL SIGN ISOYOMBO VOWEL SIGN UESOYOMBO VOWEL SIGN USOYOMBO VOWEL SIGN ESOYOMBO VOWEL SIGN OSOYOMBO VOWEL SIGN OESOY\
    \OMBO VOWEL SIGN AISOYOMBO VOWEL SIGN AUSOYOMBO VOWEL SIGN VOCALIC RSOYOMBO VOWEL SIGN VOCALIC LSOYOMBO VOWEL LENGTH MARKSOYOMBO LETTER KASOYOMBO LETTER KHASOYOMBO LETTER GASOYOMBO LETTER GHASOYOMBO LETTER NGASOYOMBO LETTER CASOYOMBO LETTER CHASOYOMBO LETT\
    \ER JASOYOMBO LETTER JHASOYOMBO LETTER NYASOYOMBO LETTER TTASOYOMBO LETTER TTHASOYOMBO LETTER DDASOYOMBO LETTER DDHASOYOMBO LETTER NNASOYOMBO LETTER TASOYOMBO LETTER THASOYOMBO LETTER DASOYOMBO LETTER DHASOYOMBO LETTER NASOYOMBO LETTER PASOYOMBO LETTER PHA\
    \SOYOMBO LETTER BASOYOMBO LETTER BHASOYOMBO LETTER MASOYOMBO LETTER TSASOYOMBO LETTER TSHASOYOMBO LETTER DZASOYOMBO LETTER ZHASOYOMBO LETTER ZASOYOMBO LETTER -ASOYOMBO LETTER YASOYOMBO LETTER RASOYOMBO LETTER LASOYOMBO LETTER VASOYOMBO LETTER SHASOYOMBO LE\
    \TTER SSASOYOMBO LETTER SASOYOMBO LETTER HASOYOMBO LETTER KSSASOYOMBO SIGN JIHVAMULIYASOYOMBO SIGN UPADHMANIYASOYOMBO CLUSTER-INITIAL LETTER RASOYOMBO CLUSTER-INITIAL LETTER LASOYOMBO CLUSTER-INITIAL LETTER SHASOYOMBO CLUSTER-INITIAL LETTER SASOYOMBO FINAL\
    \ CONSONANT SIGN GSOYOMBO FINAL CONSONANT SIGN KSOYOMBO FINAL CONSONANT SIGN NGSOYOMBO FINAL CONSONANT SIGN DSOYOMBO FINAL CONSONANT SIGN NSOYOMBO FINAL CONSONANT SIGN BSOYOMBO FINAL CONSONANT SIGN MSOYOMBO FINAL CONSONANT SIGN RSOYOMBO FINAL CONSONANT SIG\
    \N LSOYOMBO FINAL CONSONANT SIGN SHSOYOMBO FINAL CONSONANT SIGN SSOYOMBO FINAL CONSONANT SIGN -ASOYOMBO SIGN ANUSVARASOYOMBO SIGN VISARGASOYOMBO GEMINATION MARKSOYOMBO SUBJOINERSOYOMBO MARK TSHEGSOYOMBO MARK SHADSOYOMBO MARK DOUBLE SHADSOYOMBO MARK PLUTASO\
    \YOMBO HEAD MARK WITH MOON AND SUN AND TRIPLE FLAMESOYOMBO HEAD MARK WITH MOON AND SUN AND FLAMESOYOMBO HEAD MARK WITH MOON AND SUNSOYOMBO TERMINAL MARK-1SOYOMBO TERMINAL MARK-2CANADIAN SYLLABICS NATTILIK HICANADIAN SYLLABICS NATTILIK HIICANADIAN SYLLABICS\
    \ NATTILIK HOCANADIAN SYLLABICS NATTILIK HOOCANADIAN SYLLABICS NATTILIK HACANADIAN SYLLABICS NATTILIK HAACANADIAN SYLLABICS NATTILIK SHRICANADIAN SYLLABICS NATTILIK SHRIICANADIAN SYLLABICS NATTILIK SHROCANADIAN SYLLABICS NATTILIK SHROOCANADIAN SYLLABICS NA\
    \TTILIK SHRACANADIAN SYLLABICS NATTILIK SHRAACANADIAN SYLLABICS SPECANADIAN SYLLABICS SPICANADIAN SYLLABICS SPOCANADIAN SYLLABICS SPAPAU CIN HAU LETTER PAPAU CIN HAU LETTER KAPAU CIN HAU LETTER LAPAU CIN HAU LETTER MAPAU CIN HAU LETTER DAPAU CIN HAU LETTER\
    \ ZAPAU CIN HAU LETTER VAPAU CIN HAU LETTER NGAPAU CIN HAU LETTER HAPAU CIN HAU LETTER GAPAU CIN HAU LETTER KHAPAU CIN HAU LETTER SAPAU CIN HAU LETTER BAPAU CIN HAU LETTER CAPAU CIN HAU LETTER TAPAU CIN HAU LETTER THAPAU CIN HAU LETTER NAPAU CIN HAU LETTER\
    \ PHAPAU CIN HAU LETTER RAPAU CIN HAU LETTER FAPAU CIN HAU LETTER CHAPAU CIN HAU LETTER APAU CIN HAU LETTER EPAU CIN HAU LETTER IPAU CIN HAU LETTER OPAU CIN HAU LETTER UPAU CIN HAU LETTER UAPAU CIN HAU LETTER IAPAU CIN HAU LETTER FINAL PPAU CIN HAU LETTER \
    \FINAL KPAU CIN HAU LETTER FINAL TPAU CIN HAU LETTER FINAL MPAU CIN HAU LETTER FINAL NPAU CIN HAU LETTER FINAL LPAU CIN HAU LETTER FINAL WPAU CIN HAU LETTER FINAL NGPAU CIN HAU LETTER FINAL YPAU CIN HAU RISING TONE LONGPAU CIN HAU RISING TONEPAU CIN HAU SA\
    \NDHI GLOTTAL STOPPAU CIN HAU RISING TONE LONG FINALPAU CIN HAU RISING TONE FINALPAU CIN HAU SANDHI GLOTTAL STOP FINALPAU CIN HAU SANDHI TONE LONGPAU CIN HAU SANDHI TONEPAU CIN HAU SANDHI TONE LONG FINALPAU CIN HAU SANDHI TONE FINALPAU CIN HAU MID-LEVEL TO\
    \NEPAU CIN HAU GLOTTAL STOP VARIANTPAU CIN HAU MID-LEVEL TONE LONG FINALPAU CIN HAU MID-LEVEL TONE FINALPAU CIN HAU LOW-FALLING TONE LONGPAU CIN HAU LOW-FALLING TONEPAU CIN HAU GLOTTAL STOPPAU CIN HAU LOW-FALLING TONE LONG FINALPAU CIN HAU LOW-FALLING TONE\
    \ FINALPAU CIN HAU GLOTTAL STOP FINALDEVANAGARI HEAD MARKDEVANAGARI HEAD MARK WITH HEADSTROKEDEVANAGARI SIGN BHALEDEVANAGARI SIGN BHALE WITH HOOKDEVANAGARI SIGN EXTENDED BHALEDEVANAGARI SIGN EXTENDED BHALE WITH HOOKDEVANAGARI SIGN WESTERN FIVE-LIKE BHALEDE\
    \VANAGARI SIGN WESTERN NINE-LIKE BHALEDEVANAGARI SIGN REVERSED NINE-LIKE BHALEDEVANAGARI SIGN MINDUBHAIKSUKI LETTER ABHAIKSUKI LETTER AABHAIKSUKI LETTER IBHAIKSUKI LETTER IIBHAIKSUKI LETTER UBHAIKSUKI LETTER UUBHAIKSUKI LETTER VOCALIC RBHAIKSUKI LETTER VOC\
    \ALIC RRBHAIKSUKI LETTER VOCALIC LBHAIKSUKI LETTER EBHAIKSUKI LETTER AIBHAIKSUKI LETTER OBHAIKSUKI LETTER AUBHAIKSUKI LETTER KABHAIKSUKI LETTER KHABHAIKSUKI LETTER GABHAIKSUKI LETTER GHABHAIKSUKI LETTER NGABHAIKSUKI LETTER CABHAIKSUKI LETTER CHABHAIKSUKI L\
    \ETTER JABHAIKSUKI LETTER JHABHAIKSUKI LETTER NYABHAIKSUKI LETTER TTABHAIKSUKI LETTER TTHABHAIKSUKI LETTER DDABHAIKSUKI LETTER DDHABHAIKSUKI LETTER NNABHAIKSUKI LETTER TABHAIKSUKI LETTER THABHAIKSUKI LETTER DABHAIKSUKI LETTER DHABHAIKSUKI LETTER NABHAIKSUK\
    \I LETTER PABHAIKSUKI LETTER PHABHAIKSUKI LETTER BABHAIKSUKI LETTER BHABHAIKSUKI LETTER MABHAIKSUKI LETTER YABHAIKSUKI LETTER RABHAIKSUKI LETTER LABHAIKSUKI LETTER VABHAIKSUKI LETTER SHABHAIKSUKI LETTER SSABHAIKSUKI LETTER SABHAIKSUKI LETTER HABHAIKSUKI VO\
    \WEL SIGN AABHAIKSUKI VOWEL SIGN IBHAIKSUKI VOWEL SIGN IIBHAIKSUKI VOWEL SIGN UBHAIKSUKI VOWEL SIGN UUBHAIKSUKI VOWEL SIGN VOCALIC RBHAIKSUKI VOWEL SIGN VOCALIC RRBHAIKSUKI VOWEL SIGN VOCALIC LBHAIKSUKI VOWEL SIGN EBHAIKSUKI VOWEL SIGN AIBHAIKSUKI VOWEL SI\
    \GN OBHAIKSUKI VOWEL SIGN AUBHAIKSUKI SIGN CANDRABINDUBHAIKSUKI SIGN ANUSVARABHAIKSUKI SIGN VISARGABHAIKSUKI SIGN VIRAMABHAIKSUKI SIGN AVAGRAHABHAIKSUKI DANDABHAIKSUKI DOUBLE DANDABHAIKSUKI WORD SEPARATORBHAIKSUKI GAP FILLER-1BHAIKSUKI GAP FILLER-2BHAIKSUK\
    \I DIGIT ZEROBHAIKSUKI DIGIT ONEBHAIKSUKI DIGIT TWOBHAIKSUKI DIGIT THREEBHAIKSUKI DIGIT FOURBHAIKSUKI DIGIT FIVEBHAIKSUKI DIGIT SIXBHAIKSUKI DIGIT SEVENBHAIKSUKI DIGIT EIGHTBHAIKSUKI DIGIT NINEBHAIKSUKI NUMBER ONEBHAIKSUKI NUMBER TWOBHAIKSUKI NUMBER THREEB\
    \HAIKSUKI NUMBER FOURBHAIKSUKI NUMBER FIVEBHAIKSUKI NUMBER SIXBHAIKSUKI NUMBER SEVENBHAIKSUKI NUMBER EIGHTBHAIKSUKI NUMBER NINEBHAIKSUKI NUMBER TENBHAIKSUKI NUMBER TWENTYBHAIKSUKI NUMBER THIRTYBHAIKSUKI NUMBER FORTYBHAIKSUKI NUMBER FIFTYBHAIKSUKI NUMBER SI\
    \XTYBHAIKSUKI NUMBER SEVENTYBHAIKSUKI NUMBER EIGHTYBHAIKSUKI NUMBER NINETYBHAIKSUKI HUNDREDS UNIT MARKMARCHEN HEAD MARKMARCHEN MARK SHADMARCHEN LETTER KAMARCHEN LETTER KHAMARCHEN LETTER GAMARCHEN LETTER NGAMARCHEN LETTER CAMARCHEN LETTER CHAMARCHEN LETTER \
    \JAMARCHEN LETTER NYAMARCHEN LETTER TAMARCHEN LETTER THAMARCHEN LETTER DAMARCHEN LETTER NAMARCHEN LETTER PAMARCHEN LETTER PHAMARCHEN LETTER BAMARCHEN LETTER MAMARCHEN LETTER TSAMARCHEN LETTER TSHAMARCHEN LETTER DZAMARCHEN LETTER WAMARCHEN LETTER ZHAMARCHEN\
    \ LETTER ZAMARCHEN LETTER -AMARCHEN LETTER YAMARCHEN LETTER RAMARCHEN LETTER LAMARCHEN LETTER SHAMARCHEN LETTER SAMARCHEN LETTER HAMARCHEN LETTER AMARCHEN SUBJOINED LETTER KAMARCHEN SUBJOINED LETTER KHAMARCHEN SUBJOINED LETTER GAMARCHEN SUBJOINED LETTER NG\
    \AMARCHEN SUBJOINED LETTER CAMARCHEN SUBJOINED LETTER CHAMARCHEN SUBJOINED LETTER JAMARCHEN SUBJOINED LETTER NYAMARCHEN SUBJOINED LETTER TAMARCHEN SUBJOINED LETTER THAMARCHEN SUBJOINED LETTER DAMARCHEN SUBJOINED LETTER NAMARCHEN SUBJOINED LETTER PAMARCHEN \
    \SUBJOINED LETTER PHAMARCHEN SUBJOINED LETTER BAMARCHEN SUBJOINED LETTER MAMARCHEN SUBJOINED LETTER TSAMARCHEN SUBJOINED LETTER TSHAMARCHEN SUBJOINED LETTER DZAMARCHEN SUBJOINED LETTER WAMARCHEN SUBJOINED LETTER ZHAMARCHEN SUBJOINED LETTER ZAMARCHEN SUBJOI\
    \NED LETTER YAMARCHEN SUBJOINED LETTER RAMARCHEN SUBJOINED LETTER LAMARCHEN SUBJOINED LETTER SHAMARCHEN SUBJOINED LETTER SAMARCHEN SUBJOINED LETTER HAMARCHEN SUBJOINED LETTER AMARCHEN VOWEL SIGN AAMARCHEN VOWEL SIGN IMARCHEN VOWEL SIGN UMARCHEN VOWEL SIGN \
    \EMARCHEN VOWEL SIGN OMARCHEN SIGN ANUSVARAMARCHEN SIGN CANDRABINDUMASARAM GONDI LETTER AMASARAM GONDI LETTER AAMASARAM GONDI LETTER IMASARAM GONDI LETTER IIMASARAM GONDI LETTER UMASARAM GONDI LETTER UUMASARAM GONDI LETTER EMASARAM GONDI LETTER AIMASARAM G\
    \ONDI LETTER OMASARAM GONDI LETTER AUMASARAM GONDI LETTER KAMASARAM GONDI LETTER KHAMASARAM GONDI LETTER GAMASARAM GONDI LETTER GHAMASARAM GONDI LETTER NGAMASARAM GONDI LETTER CAMASARAM GONDI LETTER CHAMASARAM GONDI LETTER JAMASARAM GONDI LETTER JHAMASARAM\
    \ GONDI LETTER NYAMASARAM GONDI LETTER TTAMASARAM GONDI LETTER TTHAMASARAM GONDI LETTER DDAMASARAM GONDI LETTER DDHAMASARAM GONDI LETTER NNAMASARAM GONDI LETTER TAMASARAM GONDI LETTER THAMASARAM GONDI LETTER DAMASARAM GONDI LETTER DHAMASARAM GONDI LETTER N\
    \AMASARAM GONDI LETTER PAMASARAM GONDI LETTER PHAMASARAM GONDI LETTER BAMASARAM GONDI LETTER BHAMASARAM GONDI LETTER MAMASARAM GONDI LETTER YAMASARAM GONDI LETTER RAMASARAM GONDI LETTER LAMASARAM GONDI LETTER VAMASARAM GONDI LETTER SHAMASARAM GONDI LETTER \
    \SSAMASARAM GONDI LETTER SAMASARAM GONDI LETTER HAMASARAM GONDI LETTER LLAMASARAM GONDI LETTER KSSAMASARAM GONDI LETTER JNYAMASARAM GONDI LETTER TRAMASARAM GONDI VOWEL SIGN AAMASARAM GONDI VOWEL SIGN IMASARAM GONDI VOWEL SIGN IIMASARAM GONDI VOWEL SIGN UMA\
    \SARAM GONDI VOWEL SIGN UUMASARAM GONDI VOWEL SIGN VOCALIC RMASARAM GONDI VOWEL SIGN EMASARAM GONDI VOWEL SIGN AIMASARAM GONDI VOWEL SIGN OMASARAM GONDI VOWEL SIGN AUMASARAM GONDI SIGN ANUSVARAMASARAM GONDI SIGN VISARGAMASARAM GONDI SIGN NUKTAMASARAM GONDI\
    \ SIGN CANDRAMASARAM GONDI SIGN HALANTAMASARAM GONDI VIRAMAMASARAM GONDI REPHAMASARAM GONDI RA-KARAMASARAM GONDI DIGIT ZEROMASARAM GONDI DIGIT ONEMASARAM GONDI DIGIT TWOMASARAM GONDI DIGIT THREEMASARAM GONDI DIGIT FOURMASARAM GONDI DIGIT FIVEMASARAM GONDI \
    \DIGIT SIXMASARAM GONDI DIGIT SEVENMASARAM GONDI DIGIT EIGHTMASARAM GONDI DIGIT NINEGUNJALA GONDI LETTER AGUNJALA GONDI LETTER AAGUNJALA GONDI LETTER IGUNJALA GONDI LETTER IIGUNJALA GONDI LETTER UGUNJALA GONDI LETTER UUGUNJALA GONDI LETTER EEGUNJALA GONDI \
    \LETTER AIGUNJALA GONDI LETTER OOGUNJALA GONDI LETTER AUGUNJALA GONDI LETTER YAGUNJALA GONDI LETTER VAGUNJALA GONDI LETTER BAGUNJALA GONDI LETTER BHAGUNJALA GONDI LETTER MAGUNJALA GONDI LETTER KAGUNJALA GONDI LETTER KHAGUNJALA GONDI LETTER TAGUNJALA GONDI \
    \LETTER THAGUNJALA GONDI LETTER LAGUNJALA GONDI LETTER GAGUNJALA GONDI LETTER GHAGUNJALA GONDI LETTER DAGUNJALA GONDI LETTER DHAGUNJALA GONDI LETTER NAGUNJALA GONDI LETTER CAGUNJALA GONDI LETTER CHAGUNJALA GONDI LETTER TTAGUNJALA GONDI LETTER TTHAGUNJALA G\
    \ONDI LETTER LLAGUNJALA GONDI LETTER JAGUNJALA GONDI LETTER JHAGUNJALA GONDI LETTER DDAGUNJALA GONDI LETTER DDHAGUNJALA GONDI LETTER NGAGUNJALA GONDI LETTER PAGUNJALA GONDI LETTER PHAGUNJALA GONDI LETTER HAGUNJALA GONDI LETTER RAGUNJALA GONDI LETTER SAGUNJ\
    \ALA GONDI VOWEL SIGN AAGUNJALA GONDI VOWEL SIGN IGUNJALA GONDI VOWEL SIGN IIGUNJALA GONDI VOWEL SIGN UGUNJALA GONDI VOWEL SIGN UUGUNJALA GONDI VOWEL SIGN EEGUNJALA GONDI VOWEL SIGN AIGUNJALA GONDI VOWEL SIGN OOGUNJALA GONDI VOWEL SIGN AUGUNJALA GONDI SIGN\
    \ ANUSVARAGUNJALA GONDI SIGN VISARGAGUNJALA GONDI VIRAMAGUNJALA GONDI OMGUNJALA GONDI DIGIT ZEROGUNJALA GONDI DIGIT ONEGUNJALA GONDI DIGIT TWOGUNJALA GONDI DIGIT THREEGUNJALA GONDI DIGIT FOURGUNJALA GONDI DIGIT FIVEGUNJALA GONDI DIGIT SIXGUNJALA GONDI DIGI\
    \T SEVENGUNJALA GONDI DIGIT EIGHTGUNJALA GONDI DIGIT NINEMAKASAR LETTER KAMAKASAR LETTER GAMAKASAR LETTER NGAMAKASAR LETTER PAMAKASAR LETTER BAMAKASAR LETTER MAMAKASAR LETTER TAMAKASAR LETTER DAMAKASAR LETTER NAMAKASAR LETTER CAMAKASAR LETTER JAMAKASAR LET\
    \TER NYAMAKASAR LETTER YAMAKASAR LETTER RAMAKASAR LETTER LAMAKASAR LETTER VAMAKASAR LETTER SAMAKASAR LETTER AMAKASAR ANGKAMAKASAR VOWEL SIGN IMAKASAR VOWEL SIGN UMAKASAR VOWEL SIGN EMAKASAR VOWEL SIGN OMAKASAR PASSIMBANGMAKASAR END OF SECTIONKAWI SIGN CAND\
    \RABINDUKAWI SIGN ANUSVARAKAWI SIGN REPHAKAWI SIGN VISARGAKAWI LETTER AKAWI LETTER AAKAWI LETTER IKAWI LETTER IIKAWI LETTER UKAWI LETTER UUKAWI LETTER VOCALIC RKAWI LETTER VOCALIC RRKAWI LETTER VOCALIC LKAWI LETTER VOCALIC LLKAWI LETTER EKAWI LETTER AIKAWI\
    \ LETTER OKAWI LETTER KAKAWI LETTER KHAKAWI LETTER GAKAWI LETTER GHAKAWI LETTER NGAKAWI LETTER CAKAWI LETTER CHAKAWI LETTER JAKAWI LETTER JHAKAWI LETTER NYAKAWI LETTER TTAKAWI LETTER TTHAKAWI LETTER DDAKAWI LETTER DDHAKAWI LETTER NNAKAWI LETTER TAKAWI LETT\
    \ER THAKAWI LETTER DAKAWI LETTER DHAKAWI LETTER NAKAWI LETTER PAKAWI LETTER PHAKAWI LETTER BAKAWI LETTER BHAKAWI LETTER MAKAWI LETTER YAKAWI LETTER RAKAWI LETTER LAKAWI LETTER WAKAWI LETTER SHAKAWI LETTER SSAKAWI LETTER SAKAWI LETTER HAKAWI LETTER JNYAKAWI\
    \ VOWEL SIGN AAKAWI VOWEL SIGN ALTERNATE AAKAWI VOWEL SIGN IKAWI VOWEL SIGN IIKAWI VOWEL SIGN UKAWI VOWEL SIGN UUKAWI VOWEL SIGN VOCALIC RKAWI VOWEL SIGN EKAWI VOWEL SIGN AIKAWI VOWEL SIGN EUKAWI SIGN KILLERKAWI CONJOINERKAWI DANDAKAWI DOUBLE DANDAKAWI PUN\
    \CTUATION SECTION MARKERKAWI PUNCTUATION ALTERNATE SECTION MARKERKAWI PUNCTUATION FLOWERKAWI PUNCTUATION SPACE FILLERKAWI PUNCTUATION DOTKAWI PUNCTUATION DOUBLE DOTKAWI PUNCTUATION TRIPLE DOTKAWI PUNCTUATION CIRCLEKAWI PUNCTUATION FILLED CIRCLEKAWI PUNCTUA\
    \TION SPIRALKAWI PUNCTUATION CLOSING SPIRALKAWI DIGIT ZEROKAWI DIGIT ONEKAWI DIGIT TWOKAWI DIGIT THREEKAWI DIGIT FOURKAWI DIGIT FIVEKAWI DIGIT SIXKAWI DIGIT SEVENKAWI DIGIT EIGHTKAWI DIGIT NINELISU LETTER YHATAMIL FRACTION ONE THREE-HUNDRED-AND-TWENTIETHTA\
    \MIL FRACTION ONE ONE-HUNDRED-AND-SIXTIETHTAMIL FRACTION ONE EIGHTIETHTAMIL FRACTION ONE SIXTY-FOURTHTAMIL FRACTION ONE FORTIETHTAMIL FRACTION ONE THIRTY-SECONDTAMIL FRACTION THREE EIGHTIETHSTAMIL FRACTION THREE SIXTY-FOURTHSTAMIL FRACTION ONE TWENTIETHTAM\
    \IL FRACTION ONE SIXTEENTH-1TAMIL FRACTION ONE SIXTEENTH-2TAMIL FRACTION ONE TENTHTAMIL FRACTION ONE EIGHTHTAMIL FRACTION THREE TWENTIETHSTAMIL FRACTION THREE SIXTEENTHSTAMIL FRACTION ONE FIFTHTAMIL FRACTION ONE QUARTERTAMIL FRACTION ONE HALF-1TAMIL FRACTI\
    \ON ONE HALF-2TAMIL FRACTION THREE QUARTERSTAMIL FRACTION DOWNSCALING FACTOR KIIZHTAMIL SIGN NELTAMIL SIGN CEVITUTAMIL SIGN AAZHAAKKUTAMIL SIGN UZHAKKUTAMIL SIGN MUUVUZHAKKUTAMIL SIGN KURUNITAMIL SIGN PATHAKKUTAMIL SIGN MUKKURUNITAMIL SIGN KAACUTAMIL SIGN \
    \PANAMTAMIL SIGN PONTAMIL SIGN VARAAKANTAMIL SIGN PAARAMTAMIL SIGN KUZHITAMIL SIGN VELITAMIL WET CULTIVATION SIGNTAMIL DRY CULTIVATION SIGNTAMIL LAND SIGNTAMIL SALT PAN SIGNTAMIL TRADITIONAL CREDIT SIGNTAMIL TRADITIONAL NUMBER SIGNTAMIL CURRENT SIGNTAMIL A\
    \ND ODD SIGNTAMIL SPENT SIGNTAMIL TOTAL SIGNTAMIL IN POSSESSION SIGNTAMIL STARTING FROM SIGNTAMIL SIGN MUTHALIYATAMIL SIGN VAKAIYARAATAMIL PUNCTUATION END OF TEXTCUNEIFORM SIGN ACUNEIFORM SIGN A TIMES ACUNEIFORM SIGN A TIMES BADCUNEIFORM SIGN A TIMES GAN2 \
    \TENUCUNEIFORM SIGN A TIMES HACUNEIFORM SIGN A TIMES IGICUNEIFORM SIGN A TIMES LAGAR GUNUCUNEIFORM SIGN A TIMES MUSHCUNEIFORM SIGN A TIMES SAGCUNEIFORM SIGN A2CUNEIFORM SIGN ABCUNEIFORM SIGN AB TIMES ASH2CUNEIFORM SIGN AB TIMES DUN3 GUNUCUNEIFORM SIGN AB T\
    \IMES GALCUNEIFORM SIGN AB TIMES GAN2 TENUCUNEIFORM SIGN AB TIMES HACUNEIFORM SIGN AB TIMES IGI GUNUCUNEIFORM SIGN AB TIMES IMINCUNEIFORM SIGN AB TIMES LAGABCUNEIFORM SIGN AB TIMES SHESHCUNEIFORM SIGN AB TIMES U PLUS U PLUS UCUNEIFORM SIGN AB GUNUCUNEIFORM\
    \ SIGN AB2CUNEIFORM SIGN AB2 TIMES BALAGCUNEIFORM SIGN AB2 TIMES GAN2 TENUCUNEIFORM SIGN AB2 TIMES ME PLUS ENCUNEIFORM SIGN AB2 TIMES SHA3CUNEIFORM SIGN AB2 TIMES TAK4CUNEIFORM SIGN ADCUNEIFORM SIGN AKCUNEIFORM SIGN AK TIMES ERIN2CUNEIFORM SIGN AK TIMES SH\
    \ITA PLUS GISHCUNEIFORM SIGN ALCUNEIFORM SIGN AL TIMES ALCUNEIFORM SIGN AL TIMES DIM2CUNEIFORM SIGN AL TIMES GISHCUNEIFORM SIGN AL TIMES HACUNEIFORM SIGN AL TIMES KAD3CUNEIFORM SIGN AL TIMES KICUNEIFORM SIGN AL TIMES SHECUNEIFORM SIGN AL TIMES USHCUNEIFORM\
    \ SIGN ALANCUNEIFORM SIGN ALEPHCUNEIFORM SIGN AMARCUNEIFORM SIGN AMAR TIMES SHECUNEIFORM SIGN ANCUNEIFORM SIGN AN OVER ANCUNEIFORM SIGN AN THREE TIMESCUNEIFORM SIGN AN PLUS NAGA OPPOSING AN PLUS NAGACUNEIFORM SIGN AN PLUS NAGA SQUAREDCUNEIFORM SIGN ANSHECU\
    \NEIFORM SIGN APINCUNEIFORM SIGN ARADCUNEIFORM SIGN ARAD TIMES KURCUNEIFORM SIGN ARKABCUNEIFORM SIGN ASAL2CUNEIFORM SIGN ASHCUNEIFORM SIGN ASH ZIDA TENUCUNEIFORM SIGN ASH KABA TENUCUNEIFORM SIGN ASH OVER ASH TUG2 OVER TUG2 TUG2 OVER TUG2 PAPCUNEIFORM SIGN \
    \ASH OVER ASH OVER ASHCUNEIFORM SIGN ASH OVER ASH OVER ASH CROSSING ASH OVER ASH OVER ASHCUNEIFORM SIGN ASH2CUNEIFORM SIGN ASHGABCUNEIFORM SIGN BACUNEIFORM SIGN BADCUNEIFORM SIGN BAG3CUNEIFORM SIGN BAHAR2CUNEIFORM SIGN BALCUNEIFORM SIGN BAL OVER BALCUNEIFO\
    \RM SIGN BALAGCUNEIFORM SIGN BARCUNEIFORM SIGN BARA2CUNEIFORM SIGN BICUNEIFORM SIGN BI TIMES ACUNEIFORM SIGN BI TIMES GARCUNEIFORM SIGN BI TIMES IGI GUNUCUNEIFORM SIGN BUCUNEIFORM SIGN BU OVER BU ABCUNEIFORM SIGN BU OVER BU UNCUNEIFORM SIGN BU CROSSING BUC\
    \UNEIFORM SIGN BULUGCUNEIFORM SIGN BULUG OVER BULUGCUNEIFORM SIGN BURCUNEIFORM SIGN BUR2CUNEIFORM SIGN DACUNEIFORM SIGN DAGCUNEIFORM SIGN DAG KISIM5 TIMES A PLUS MASHCUNEIFORM SIGN DAG KISIM5 TIMES AMARCUNEIFORM SIGN DAG KISIM5 TIMES BALAGCUNEIFORM SIGN DA\
    \G KISIM5 TIMES BICUNEIFORM SIGN DAG KISIM5 TIMES GACUNEIFORM SIGN DAG KISIM5 TIMES GA PLUS MASHCUNEIFORM SIGN DAG KISIM5 TIMES GICUNEIFORM SIGN DAG KISIM5 TIMES GIR2CUNEIFORM SIGN DAG KISIM5 TIMES GUDCUNEIFORM SIGN DAG KISIM5 TIMES HACUNEIFORM SIGN DAG KI\
    \SIM5 TIMES IRCUNEIFORM SIGN DAG KISIM5 TIMES IR PLUS LUCUNEIFORM SIGN DAG KISIM5 TIMES KAKCUNEIFORM SIGN DAG KISIM5 TIMES LACUNEIFORM SIGN DAG KISIM5 TIMES LUCUNEIFORM SIGN DAG KISIM5 TIMES LU PLUS MASH2CUNEIFORM SIGN DAG KISIM5 TIMES LUMCUNEIFORM SIGN DA\
    \G KISIM5 TIMES NECUNEIFORM SIGN DAG KISIM5 TIMES PAP PLUS PAPCUNEIFORM SIGN DAG KISIM5 TIMES SICUNEIFORM SIGN DAG KISIM5 TIMES TAK4CUNEIFORM SIGN DAG KISIM5 TIMES U2 PLUS GIR2CUNEIFORM SIGN DAG KISIM5 TIMES USHCUNEIFORM SIGN DAMCUNEIFORM SIGN DARCUNEIFORM\
    \ SIGN DARA3CUNEIFORM SIGN DARA4CUNEIFORM SIGN DICUNEIFORM SIGN DIBCUNEIFORM SIGN DIMCUNEIFORM SIGN DIM TIMES SHECUNEIFORM SIGN DIM2CUNEIFORM SIGN DINCUNEIFORM SIGN DIN KASKAL U GUNU DISHCUNEIFORM SIGN DISHCUNEIFORM SIGN DUCUNEIFORM SIGN DU OVER DUCUNEIFOR\
    \M SIGN DU GUNUCUNEIFORM SIGN DU SHESHIGCUNEIFORM SIGN DUBCUNEIFORM SIGN DUB TIMES ESH2CUNEIFORM SIGN DUB2CUNEIFORM SIGN DUGCUNEIFORM SIGN DUGUDCUNEIFORM SIGN DUHCUNEIFORM SIGN DUNCUNEIFORM SIGN DUN3CUNEIFORM SIGN DUN3 GUNUCUNEIFORM SIGN DUN3 GUNU GUNUCUNE\
    \IFORM SIGN DUN4CUNEIFORM SIGN DUR2CUNEIFORM SIGN ECUNEIFORM SIGN E TIMES PAPCUNEIFORM SIGN E OVER E NUN OVER NUNCUNEIFORM SIGN E2CUNEIFORM SIGN E2 TIMES A PLUS HA PLUS DACUNEIFORM SIGN E2 TIMES GARCUNEIFORM SIGN E2 TIMES MICUNEIFORM SIGN E2 TIMES SALCUNEI\
    \FORM SIGN E2 TIMES SHECUNEIFORM SIGN E2 TIMES UCUNEIFORM SIGN EDINCUNEIFORM SIGN EGIRCUNEIFORM SIGN ELCUNEIFORM SIGN ENCUNEIFORM SIGN EN TIMES GAN2CUNEIFORM SIGN EN TIMES GAN2 TENUCUNEIFORM SIGN EN TIMES MECUNEIFORM SIGN EN CROSSING ENCUNEIFORM SIGN EN OP\
    \POSING ENCUNEIFORM SIGN EN SQUAREDCUNEIFORM SIGN ERENCUNEIFORM SIGN ERIN2CUNEIFORM SIGN ESH2CUNEIFORM SIGN EZENCUNEIFORM SIGN EZEN TIMES ACUNEIFORM SIGN EZEN TIMES A PLUS LALCUNEIFORM SIGN EZEN TIMES A PLUS LAL TIMES LALCUNEIFORM SIGN EZEN TIMES ANCUNEIFO\
    \RM SIGN EZEN TIMES BADCUNEIFORM SIGN EZEN TIMES DUN3 GUNUCUNEIFORM SIGN EZEN TIMES DUN3 GUNU GUNUCUNEIFORM SIGN EZEN TIMES HACUNEIFORM SIGN EZEN TIMES HA GUNUCUNEIFORM SIGN EZEN TIMES IGI GUNUCUNEIFORM SIGN EZEN TIMES KASKALCUNEIFORM SIGN EZEN TIMES KASKA\
    \L SQUAREDCUNEIFORM SIGN EZEN TIMES KU3CUNEIFORM SIGN EZEN TIMES LACUNEIFORM SIGN EZEN TIMES LAL TIMES LALCUNEIFORM SIGN EZEN TIMES LICUNEIFORM SIGN EZEN TIMES LUCUNEIFORM SIGN EZEN TIMES U2CUNEIFORM SIGN EZEN TIMES UDCUNEIFORM SIGN GACUNEIFORM SIGN GA GUN\
    \UCUNEIFORM SIGN GA2CUNEIFORM SIGN GA2 TIMES A PLUS DA PLUS HACUNEIFORM SIGN GA2 TIMES A PLUS HACUNEIFORM SIGN GA2 TIMES A PLUS IGICUNEIFORM SIGN GA2 TIMES AB2 TENU PLUS TABCUNEIFORM SIGN GA2 TIMES ANCUNEIFORM SIGN GA2 TIMES ASHCUNEIFORM SIGN GA2 TIMES ASH\
    \2 PLUS GALCUNEIFORM SIGN GA2 TIMES BADCUNEIFORM SIGN GA2 TIMES BAR PLUS RACUNEIFORM SIGN GA2 TIMES BURCUNEIFORM SIGN GA2 TIMES BUR PLUS RACUNEIFORM SIGN GA2 TIMES DACUNEIFORM SIGN GA2 TIMES DICUNEIFORM SIGN GA2 TIMES DIM TIMES SHECUNEIFORM SIGN GA2 TIMES \
    \DUBCUNEIFORM SIGN GA2 TIMES ELCUNEIFORM SIGN GA2 TIMES EL PLUS LACUNEIFORM SIGN GA2 TIMES ENCUNEIFORM SIGN GA2 TIMES EN TIMES GAN2 TENUCUNEIFORM SIGN GA2 TIMES GAN2 TENUCUNEIFORM SIGN GA2 TIMES GARCUNEIFORM SIGN GA2 TIMES GICUNEIFORM SIGN GA2 TIMES GI4CUN\
    \EIFORM SIGN GA2 TIMES GI4 PLUS ACUNEIFORM SIGN GA2 TIMES GIR2 PLUS SUCUNEIFORM SIGN GA2 TIMES HA PLUS LU PLUS ESH2CUNEIFORM SIGN GA2 TIMES HALCUNEIFORM SIGN GA2 TIMES HAL PLUS LACUNEIFORM SIGN GA2 TIMES HI PLUS LICUNEIFORM SIGN GA2 TIMES HUB2CUNEIFORM SIG\
    \N GA2 TIMES IGI GUNUCUNEIFORM SIGN GA2 TIMES ISH PLUS HU PLUS ASHCUNEIFORM SIGN GA2 TIMES KAKCUNEIFORM SIGN GA2 TIMES KASKALCUNEIFORM SIGN GA2 TIMES KIDCUNEIFORM SIGN GA2 TIMES KID PLUS LALCUNEIFORM SIGN GA2 TIMES KU3 PLUS ANCUNEIFORM SIGN GA2 TIMES LACUN\
    \EIFORM SIGN GA2 TIMES ME PLUS ENCUNEIFORM SIGN GA2 TIMES MICUNEIFORM SIGN GA2 TIMES NUNCUNEIFORM SIGN GA2 TIMES NUN OVER NUNCUNEIFORM SIGN GA2 TIMES PACUNEIFORM SIGN GA2 TIMES SALCUNEIFORM SIGN GA2 TIMES SARCUNEIFORM SIGN GA2 TIMES SHECUNEIFORM SIGN GA2 T\
    \IMES SHE PLUS TURCUNEIFORM SIGN GA2 TIMES SHIDCUNEIFORM SIGN GA2 TIMES SUMCUNEIFORM SIGN GA2 TIMES TAK4CUNEIFORM SIGN GA2 TIMES UCUNEIFORM SIGN GA2 TIMES UDCUNEIFORM SIGN GA2 TIMES UD PLUS DUCUNEIFORM SIGN GA2 OVER GA2CUNEIFORM SIGN GABACUNEIFORM SIGN GAB\
    \A CROSSING GABACUNEIFORM SIGN GADCUNEIFORM SIGN GAD OVER GAD GAR OVER GARCUNEIFORM SIGN GALCUNEIFORM SIGN GAL GAD OVER GAD GAR OVER GARCUNEIFORM SIGN GALAMCUNEIFORM SIGN GAMCUNEIFORM SIGN GANCUNEIFORM SIGN GAN2CUNEIFORM SIGN GAN2 TENUCUNEIFORM SIGN GAN2 O\
    \VER GAN2CUNEIFORM SIGN GAN2 CROSSING GAN2CUNEIFORM SIGN GARCUNEIFORM SIGN GAR3CUNEIFORM SIGN GASHANCUNEIFORM SIGN GESHTINCUNEIFORM SIGN GESHTIN TIMES KURCUNEIFORM SIGN GICUNEIFORM SIGN GI TIMES ECUNEIFORM SIGN GI TIMES UCUNEIFORM SIGN GI CROSSING GICUNEIF\
    \ORM SIGN GI4CUNEIFORM SIGN GI4 OVER GI4CUNEIFORM SIGN GI4 CROSSING GI4CUNEIFORM SIGN GIDIMCUNEIFORM SIGN GIR2CUNEIFORM SIGN GIR2 GUNUCUNEIFORM SIGN GIR3CUNEIFORM SIGN GIR3 TIMES A PLUS IGICUNEIFORM SIGN GIR3 TIMES GAN2 TENUCUNEIFORM SIGN GIR3 TIMES IGICUN\
    \EIFORM SIGN GIR3 TIMES LU PLUS IGICUNEIFORM SIGN GIR3 TIMES PACUNEIFORM SIGN GISALCUNEIFORM SIGN GISHCUNEIFORM SIGN GISH CROSSING GISHCUNEIFORM SIGN GISH TIMES BADCUNEIFORM SIGN GISH TIMES TAK4CUNEIFORM SIGN GISH TENUCUNEIFORM SIGN GUCUNEIFORM SIGN GU CRO\
    \SSING GUCUNEIFORM SIGN GU2CUNEIFORM SIGN GU2 TIMES KAKCUNEIFORM SIGN GU2 TIMES KAK TIMES IGI GUNUCUNEIFORM SIGN GU2 TIMES NUNCUNEIFORM SIGN GU2 TIMES SAL PLUS TUG2CUNEIFORM SIGN GU2 GUNUCUNEIFORM SIGN GUDCUNEIFORM SIGN GUD TIMES A PLUS KURCUNEIFORM SIGN G\
    \UD TIMES KURCUNEIFORM SIGN GUD OVER GUD LUGALCUNEIFORM SIGN GULCUNEIFORM SIGN GUMCUNEIFORM SIGN GUM TIMES SHECUNEIFORM SIGN GURCUNEIFORM SIGN GUR7CUNEIFORM SIGN GURUNCUNEIFORM SIGN GURUSHCUNEIFORM SIGN HACUNEIFORM SIGN HA TENUCUNEIFORM SIGN HA GUNUCUNEIFO\
    \RM SIGN HALCUNEIFORM SIGN HICUNEIFORM SIGN HI TIMES ASHCUNEIFORM SIGN HI TIMES ASH2CUNEIFORM SIGN HI TIMES BADCUNEIFORM SIGN HI TIMES DISHCUNEIFORM SIGN HI TIMES GADCUNEIFORM SIGN HI TIMES KINCUNEIFORM SIGN HI TIMES NUNCUNEIFORM SIGN HI TIMES SHECUNEIFORM\
    \ SIGN HI TIMES UCUNEIFORM SIGN HUCUNEIFORM SIGN HUB2CUNEIFORM SIGN HUB2 TIMES ANCUNEIFORM SIGN HUB2 TIMES HALCUNEIFORM SIGN HUB2 TIMES KASKALCUNEIFORM SIGN HUB2 TIMES LISHCUNEIFORM SIGN HUB2 TIMES UDCUNEIFORM SIGN HUL2CUNEIFORM SIGN ICUNEIFORM SIGN I ACUN\
    \EIFORM SIGN IBCUNEIFORM SIGN IDIMCUNEIFORM SIGN IDIM OVER IDIM BURCUNEIFORM SIGN IDIM OVER IDIM SQUAREDCUNEIFORM SIGN IGCUNEIFORM SIGN IGICUNEIFORM SIGN IGI DIBCUNEIFORM SIGN IGI RICUNEIFORM SIGN IGI OVER IGI SHIR OVER SHIR UD OVER UDCUNEIFORM SIGN IGI GU\
    \NUCUNEIFORM SIGN ILCUNEIFORM SIGN IL TIMES GAN2 TENUCUNEIFORM SIGN IL2CUNEIFORM SIGN IMCUNEIFORM SIGN IM TIMES TAK4CUNEIFORM SIGN IM CROSSING IMCUNEIFORM SIGN IM OPPOSING IMCUNEIFORM SIGN IM SQUAREDCUNEIFORM SIGN IMINCUNEIFORM SIGN INCUNEIFORM SIGN IRCUNE\
    \IFORM SIGN ISHCUNEIFORM SIGN KACUNEIFORM SIGN KA TIMES ACUNEIFORM SIGN KA TIMES ADCUNEIFORM SIGN KA TIMES AD PLUS KU3CUNEIFORM SIGN KA TIMES ASH2CUNEIFORM SIGN KA TIMES BADCUNEIFORM SIGN KA TIMES BALAGCUNEIFORM SIGN KA TIMES BARCUNEIFORM SIGN KA TIMES BIC\
    \UNEIFORM SIGN KA TIMES ERIN2CUNEIFORM SIGN KA TIMES ESH2CUNEIFORM SIGN KA TIMES GACUNEIFORM SIGN KA TIMES GALCUNEIFORM SIGN KA TIMES GAN2 TENUCUNEIFORM SIGN KA TIMES GARCUNEIFORM SIGN KA TIMES GAR PLUS SHA3 PLUS ACUNEIFORM SIGN KA TIMES GICUNEIFORM SIGN K\
    \A TIMES GIR2CUNEIFORM SIGN KA TIMES GISH PLUS SARCUNEIFORM SIGN KA TIMES GISH CROSSING GISHCUNEIFORM SIGN KA TIMES GUCUNEIFORM SIGN KA TIMES GUR7CUNEIFORM SIGN KA TIMES IGICUNEIFORM SIGN KA TIMES IMCUNEIFORM SIGN KA TIMES KAKCUNEIFORM SIGN KA TIMES KICUNE\
    \IFORM SIGN KA TIMES KIDCUNEIFORM SIGN KA TIMES LICUNEIFORM SIGN KA TIMES LUCUNEIFORM SIGN KA TIMES MECUNEIFORM SIGN KA TIMES ME PLUS DUCUNEIFORM SIGN KA TIMES ME PLUS GICUNEIFORM SIGN KA TIMES ME PLUS TECUNEIFORM SIGN KA TIMES MICUNEIFORM SIGN KA TIMES MI\
    \ PLUS NUNUZCUNEIFORM SIGN KA TIMES NECUNEIFORM SIGN KA TIMES NUNCUNEIFORM SIGN KA TIMES PICUNEIFORM SIGN KA TIMES RUCUNEIFORM SIGN KA TIMES SACUNEIFORM SIGN KA TIMES SARCUNEIFORM SIGN KA TIMES SHACUNEIFORM SIGN KA TIMES SHECUNEIFORM SIGN KA TIMES SHIDCUNE\
    \IFORM SIGN KA TIMES SHUCUNEIFORM SIGN KA TIMES SIGCUNEIFORM SIGN KA TIMES SUHURCUNEIFORM SIGN KA TIMES TARCUNEIFORM SIGN KA TIMES UCUNEIFORM SIGN KA TIMES U2CUNEIFORM SIGN KA TIMES UDCUNEIFORM SIGN KA TIMES UMUM TIMES PACUNEIFORM SIGN KA TIMES USHCUNEIFOR\
    \M SIGN KA TIMES ZICUNEIFORM SIGN KA2CUNEIFORM SIGN KA2 CROSSING KA2CUNEIFORM SIGN KABCUNEIFORM SIGN KAD2CUNEIFORM SIGN KAD3CUNEIFORM SIGN KAD4CUNEIFORM SIGN KAD5CUNEIFORM SIGN KAD5 OVER KAD5CUNEIFORM SIGN KAKCUNEIFORM SIGN KAK TIMES IGI GUNUCUNEIFORM SIGN\
    \ KALCUNEIFORM SIGN KAL TIMES BADCUNEIFORM SIGN KAL CROSSING KALCUNEIFORM SIGN KAM2CUNEIFORM SIGN KAM4CUNEIFORM SIGN KASKALCUNEIFORM SIGN KASKAL LAGAB TIMES U OVER LAGAB TIMES UCUNEIFORM SIGN KASKAL OVER KASKAL LAGAB TIMES U OVER LAGAB TIMES UCUNEIFORM SIG\
    \N KESH2CUNEIFORM SIGN KICUNEIFORM SIGN KI TIMES BADCUNEIFORM SIGN KI TIMES UCUNEIFORM SIGN KI TIMES UDCUNEIFORM SIGN KIDCUNEIFORM SIGN KINCUNEIFORM SIGN KISALCUNEIFORM SIGN KISHCUNEIFORM SIGN KISIM5CUNEIFORM SIGN KISIM5 OVER KISIM5CUNEIFORM SIGN KUCUNEIFO\
    \RM SIGN KU OVER HI TIMES ASH2 KU OVER HI TIMES ASH2CUNEIFORM SIGN KU3CUNEIFORM SIGN KU4CUNEIFORM SIGN KU4 VARIANT FORMCUNEIFORM SIGN KU7CUNEIFORM SIGN KULCUNEIFORM SIGN KUL GUNUCUNEIFORM SIGN KUNCUNEIFORM SIGN KURCUNEIFORM SIGN KUR OPPOSING KURCUNEIFORM S\
    \IGN KUSHU2CUNEIFORM SIGN KWU318CUNEIFORM SIGN LACUNEIFORM SIGN LAGABCUNEIFORM SIGN LAGAB TIMES ACUNEIFORM SIGN LAGAB TIMES A PLUS DA PLUS HACUNEIFORM SIGN LAGAB TIMES A PLUS GARCUNEIFORM SIGN LAGAB TIMES A PLUS LALCUNEIFORM SIGN LAGAB TIMES ALCUNEIFORM SI\
    \GN LAGAB TIMES ANCUNEIFORM SIGN LAGAB TIMES ASH ZIDA TENUCUNEIFORM SIGN LAGAB TIMES BADCUNEIFORM SIGN LAGAB TIMES BICUNEIFORM SIGN LAGAB TIMES DARCUNEIFORM SIGN LAGAB TIMES ENCUNEIFORM SIGN LAGAB TIMES GACUNEIFORM SIGN LAGAB TIMES GARCUNEIFORM SIGN LAGAB \
    \TIMES GUDCUNEIFORM SIGN LAGAB TIMES GUD PLUS GUDCUNEIFORM SIGN LAGAB TIMES HACUNEIFORM SIGN LAGAB TIMES HALCUNEIFORM SIGN LAGAB TIMES HI TIMES NUNCUNEIFORM SIGN LAGAB TIMES IGI GUNUCUNEIFORM SIGN LAGAB TIMES IMCUNEIFORM SIGN LAGAB TIMES IM PLUS HACUNEIFOR\
    \M SIGN LAGAB TIMES IM PLUS LUCUNEIFORM SIGN LAGAB TIMES KICUNEIFORM SIGN LAGAB TIMES KINCUNEIFORM SIGN LAGAB TIMES KU3CUNEIFORM SIGN LAGAB TIMES KULCUNEIFORM SIGN LAGAB TIMES KUL PLUS HI PLUS ACUNEIFORM SIGN LAGAB TIMES LAGABCUNEIFORM SIGN LAGAB TIMES LIS\
    \HCUNEIFORM SIGN LAGAB TIMES LUCUNEIFORM SIGN LAGAB TIMES LULCUNEIFORM SIGN LAGAB TIMES MECUNEIFORM SIGN LAGAB TIMES ME PLUS ENCUNEIFORM SIGN LAGAB TIMES MUSHCUNEIFORM SIGN LAGAB TIMES NECUNEIFORM SIGN LAGAB TIMES SHE PLUS SUMCUNEIFORM SIGN LAGAB TIMES SHI\
    \TA PLUS GISH PLUS ERIN2CUNEIFORM SIGN LAGAB TIMES SHITA PLUS GISH TENUCUNEIFORM SIGN LAGAB TIMES SHU2CUNEIFORM SIGN LAGAB TIMES SHU2 PLUS SHU2CUNEIFORM SIGN LAGAB TIMES SUMCUNEIFORM SIGN LAGAB TIMES TAGCUNEIFORM SIGN LAGAB TIMES TAK4CUNEIFORM SIGN LAGAB T\
    \IMES TE PLUS A PLUS SU PLUS NACUNEIFORM SIGN LAGAB TIMES UCUNEIFORM SIGN LAGAB TIMES U PLUS ACUNEIFORM SIGN LAGAB TIMES U PLUS U PLUS UCUNEIFORM SIGN LAGAB TIMES U2 PLUS ASHCUNEIFORM SIGN LAGAB TIMES UDCUNEIFORM SIGN LAGAB TIMES USHCUNEIFORM SIGN LAGAB SQ\
    \UAREDCUNEIFORM SIGN LAGARCUNEIFORM SIGN LAGAR TIMES SHECUNEIFORM SIGN LAGAR TIMES SHE PLUS SUMCUNEIFORM SIGN LAGAR GUNUCUNEIFORM SIGN LAGAR GUNU OVER LAGAR GUNU SHECUNEIFORM SIGN LAHSHUCUNEIFORM SIGN LALCUNEIFORM SIGN LAL TIMES LALCUNEIFORM SIGN LAMCUNEIF\
    \ORM SIGN LAM TIMES KURCUNEIFORM SIGN LAM TIMES KUR PLUS RUCUNEIFORM SIGN LICUNEIFORM SIGN LILCUNEIFORM SIGN LIMMU2CUNEIFORM SIGN LISHCUNEIFORM SIGN LUCUNEIFORM SIGN LU TIMES BADCUNEIFORM SIGN LU2CUNEIFORM SIGN LU2 TIMES ALCUNEIFORM SIGN LU2 TIMES BADCUNEI\
    \FORM SIGN LU2 TIMES ESH2CUNEIFORM SIGN LU2 TIMES ESH2 TENUCUNEIFORM SIGN LU2 TIMES GAN2 TENUCUNEIFORM SIGN LU2 TIMES HI TIMES BADCUNEIFORM SIGN LU2 TIMES IMCUNEIFORM SIGN LU2 TIMES KAD2CUNEIFORM SIGN LU2 TIMES KAD3CUNEIFORM SIGN LU2 TIMES KAD3 PLUS ASHCUN\
    \EIFORM SIGN LU2 TIMES KICUNEIFORM SIGN LU2 TIMES LA PLUS ASHCUNEIFORM SIGN LU2 TIMES LAGABCUNEIFORM SIGN LU2 TIMES ME PLUS ENCUNEIFORM SIGN LU2 TIMES NECUNEIFORM SIGN LU2 TIMES NUCUNEIFORM SIGN LU2 TIMES SI PLUS ASHCUNEIFORM SIGN LU2 TIMES SIK2 PLUS BUCUN\
    \EIFORM SIGN LU2 TIMES TUG2CUNEIFORM SIGN LU2 TENUCUNEIFORM SIGN LU2 CROSSING LU2CUNEIFORM SIGN LU2 OPPOSING LU2CUNEIFORM SIGN LU2 SQUAREDCUNEIFORM SIGN LU2 SHESHIGCUNEIFORM SIGN LU3CUNEIFORM SIGN LUGALCUNEIFORM SIGN LUGAL OVER LUGALCUNEIFORM SIGN LUGAL OP\
    \POSING LUGALCUNEIFORM SIGN LUGAL SHESHIGCUNEIFORM SIGN LUHCUNEIFORM SIGN LULCUNEIFORM SIGN LUMCUNEIFORM SIGN LUM OVER LUMCUNEIFORM SIGN LUM OVER LUM GAR OVER GARCUNEIFORM SIGN MACUNEIFORM SIGN MA TIMES TAK4CUNEIFORM SIGN MA GUNUCUNEIFORM SIGN MA2CUNEIFORM\
    \ SIGN MAHCUNEIFORM SIGN MARCUNEIFORM SIGN MASHCUNEIFORM SIGN MASH2CUNEIFORM SIGN MECUNEIFORM SIGN MESCUNEIFORM SIGN MICUNEIFORM SIGN MINCUNEIFORM SIGN MUCUNEIFORM SIGN MU OVER MUCUNEIFORM SIGN MUGCUNEIFORM SIGN MUG GUNUCUNEIFORM SIGN MUNSUBCUNEIFORM SIGN \
    \MURGU2CUNEIFORM SIGN MUSHCUNEIFORM SIGN MUSH TIMES ACUNEIFORM SIGN MUSH TIMES KURCUNEIFORM SIGN MUSH TIMES ZACUNEIFORM SIGN MUSH OVER MUSHCUNEIFORM SIGN MUSH OVER MUSH TIMES A PLUS NACUNEIFORM SIGN MUSH CROSSING MUSHCUNEIFORM SIGN MUSH3CUNEIFORM SIGN MUSH\
    \3 TIMES ACUNEIFORM SIGN MUSH3 TIMES A PLUS DICUNEIFORM SIGN MUSH3 TIMES DICUNEIFORM SIGN MUSH3 GUNUCUNEIFORM SIGN NACUNEIFORM SIGN NA2CUNEIFORM SIGN NAGACUNEIFORM SIGN NAGA INVERTEDCUNEIFORM SIGN NAGA TIMES SHU TENUCUNEIFORM SIGN NAGA OPPOSING NAGACUNEIFO\
    \RM SIGN NAGARCUNEIFORM SIGN NAM NUTILLUCUNEIFORM SIGN NAMCUNEIFORM SIGN NAM2CUNEIFORM SIGN NECUNEIFORM SIGN NE TIMES ACUNEIFORM SIGN NE TIMES UDCUNEIFORM SIGN NE SHESHIGCUNEIFORM SIGN NICUNEIFORM SIGN NI TIMES ECUNEIFORM SIGN NI2CUNEIFORM SIGN NIMCUNEIFOR\
    \M SIGN NIM TIMES GAN2 TENUCUNEIFORM SIGN NIM TIMES GAR PLUS GAN2 TENUCUNEIFORM SIGN NINDA2CUNEIFORM SIGN NINDA2 TIMES ANCUNEIFORM SIGN NINDA2 TIMES ASHCUNEIFORM SIGN NINDA2 TIMES ASH PLUS ASHCUNEIFORM SIGN NINDA2 TIMES GUDCUNEIFORM SIGN NINDA2 TIMES ME PL\
    \US GAN2 TENUCUNEIFORM SIGN NINDA2 TIMES NECUNEIFORM SIGN NINDA2 TIMES NUNCUNEIFORM SIGN NINDA2 TIMES SHECUNEIFORM SIGN NINDA2 TIMES SHE PLUS A ANCUNEIFORM SIGN NINDA2 TIMES SHE PLUS ASHCUNEIFORM SIGN NINDA2 TIMES SHE PLUS ASH PLUS ASHCUNEIFORM SIGN NINDA2\
    \ TIMES U2 PLUS ASHCUNEIFORM SIGN NINDA2 TIMES USHCUNEIFORM SIGN NISAGCUNEIFORM SIGN NUCUNEIFORM SIGN NU11CUNEIFORM SIGN NUNCUNEIFORM SIGN NUN LAGAR TIMES GARCUNEIFORM SIGN NUN LAGAR TIMES MASHCUNEIFORM SIGN NUN LAGAR TIMES SALCUNEIFORM SIGN NUN LAGAR TIME\
    \S SAL OVER NUN LAGAR TIMES SALCUNEIFORM SIGN NUN LAGAR TIMES USHCUNEIFORM SIGN NUN TENUCUNEIFORM SIGN NUN OVER NUNCUNEIFORM SIGN NUN CROSSING NUNCUNEIFORM SIGN NUN CROSSING NUN LAGAR OVER LAGARCUNEIFORM SIGN NUNUZCUNEIFORM SIGN NUNUZ AB2 TIMES ASHGABCUNEI\
    \FORM SIGN NUNUZ AB2 TIMES BICUNEIFORM SIGN NUNUZ AB2 TIMES DUGCUNEIFORM SIGN NUNUZ AB2 TIMES GUDCUNEIFORM SIGN NUNUZ AB2 TIMES IGI GUNUCUNEIFORM SIGN NUNUZ AB2 TIMES KAD3CUNEIFORM SIGN NUNUZ AB2 TIMES LACUNEIFORM SIGN NUNUZ AB2 TIMES NECUNEIFORM SIGN NUNU\
    \Z AB2 TIMES SILA3CUNEIFORM SIGN NUNUZ AB2 TIMES U2CUNEIFORM SIGN NUNUZ KISIM5 TIMES BICUNEIFORM SIGN NUNUZ KISIM5 TIMES BI UCUNEIFORM SIGN PACUNEIFORM SIGN PADCUNEIFORM SIGN PANCUNEIFORM SIGN PAPCUNEIFORM SIGN PESH2CUNEIFORM SIGN PICUNEIFORM SIGN PI TIMES\
    \ ACUNEIFORM SIGN PI TIMES ABCUNEIFORM SIGN PI TIMES BICUNEIFORM SIGN PI TIMES BUCUNEIFORM SIGN PI TIMES ECUNEIFORM SIGN PI TIMES ICUNEIFORM SIGN PI TIMES IBCUNEIFORM SIGN PI TIMES UCUNEIFORM SIGN PI TIMES U2CUNEIFORM SIGN PI CROSSING PICUNEIFORM SIGN PIRI\
    \GCUNEIFORM SIGN PIRIG TIMES KALCUNEIFORM SIGN PIRIG TIMES UDCUNEIFORM SIGN PIRIG TIMES ZACUNEIFORM SIGN PIRIG OPPOSING PIRIGCUNEIFORM SIGN RACUNEIFORM SIGN RABCUNEIFORM SIGN RICUNEIFORM SIGN RUCUNEIFORM SIGN SACUNEIFORM SIGN SAG NUTILLUCUNEIFORM SIGN SAGC\
    \UNEIFORM SIGN SAG TIMES ACUNEIFORM SIGN SAG TIMES DUCUNEIFORM SIGN SAG TIMES DUBCUNEIFORM SIGN SAG TIMES HACUNEIFORM SIGN SAG TIMES KAKCUNEIFORM SIGN SAG TIMES KURCUNEIFORM SIGN SAG TIMES LUMCUNEIFORM SIGN SAG TIMES MICUNEIFORM SIGN SAG TIMES NUNCUNEIFORM\
    \ SIGN SAG TIMES SALCUNEIFORM SIGN SAG TIMES SHIDCUNEIFORM SIGN SAG TIMES TABCUNEIFORM SIGN SAG TIMES U2CUNEIFORM SIGN SAG TIMES UBCUNEIFORM SIGN SAG TIMES UMCUNEIFORM SIGN SAG TIMES URCUNEIFORM SIGN SAG TIMES USHCUNEIFORM SIGN SAG OVER SAGCUNEIFORM SIGN S\
    \AG GUNUCUNEIFORM SIGN SALCUNEIFORM SIGN SAL LAGAB TIMES ASH2CUNEIFORM SIGN SANGA2CUNEIFORM SIGN SARCUNEIFORM SIGN SHACUNEIFORM SIGN SHA3CUNEIFORM SIGN SHA3 TIMES ACUNEIFORM SIGN SHA3 TIMES BADCUNEIFORM SIGN SHA3 TIMES GISHCUNEIFORM SIGN SHA3 TIMES NECUNEI\
    \FORM SIGN SHA3 TIMES SHU2CUNEIFORM SIGN SHA3 TIMES TURCUNEIFORM SIGN SHA3 TIMES UCUNEIFORM SIGN SHA3 TIMES U PLUS ACUNEIFORM SIGN SHA6CUNEIFORM SIGN SHAB6CUNEIFORM SIGN SHAR2CUNEIFORM SIGN SHECUNEIFORM SIGN SHE HUCUNEIFORM SIGN SHE OVER SHE GAD OVER GAD G\
    \AR OVER GARCUNEIFORM SIGN SHE OVER SHE TAB OVER TAB GAR OVER GARCUNEIFORM SIGN SHEG9CUNEIFORM SIGN SHENCUNEIFORM SIGN SHESHCUNEIFORM SIGN SHESH2CUNEIFORM SIGN SHESHLAMCUNEIFORM SIGN SHIDCUNEIFORM SIGN SHID TIMES ACUNEIFORM SIGN SHID TIMES IMCUNEIFORM SIGN\
    \ SHIMCUNEIFORM SIGN SHIM TIMES ACUNEIFORM SIGN SHIM TIMES BALCUNEIFORM SIGN SHIM TIMES BULUGCUNEIFORM SIGN SHIM TIMES DINCUNEIFORM SIGN SHIM TIMES GARCUNEIFORM SIGN SHIM TIMES IGICUNEIFORM SIGN SHIM TIMES IGI GUNUCUNEIFORM SIGN SHIM TIMES KUSHU2CUNEIFORM \
    \SIGN SHIM TIMES LULCUNEIFORM SIGN SHIM TIMES MUGCUNEIFORM SIGN SHIM TIMES SALCUNEIFORM SIGN SHINIGCUNEIFORM SIGN SHIRCUNEIFORM SIGN SHIR TENUCUNEIFORM SIGN SHIR OVER SHIR BUR OVER BURCUNEIFORM SIGN SHITACUNEIFORM SIGN SHUCUNEIFORM SIGN SHU OVER INVERTED S\
    \HUCUNEIFORM SIGN SHU2CUNEIFORM SIGN SHUBURCUNEIFORM SIGN SICUNEIFORM SIGN SI GUNUCUNEIFORM SIGN SIGCUNEIFORM SIGN SIG4CUNEIFORM SIGN SIG4 OVER SIG4 SHU2CUNEIFORM SIGN SIK2CUNEIFORM SIGN SILA3CUNEIFORM SIGN SUCUNEIFORM SIGN SU OVER SUCUNEIFORM SIGN SUDCUNE\
    \IFORM SIGN SUD2CUNEIFORM SIGN SUHURCUNEIFORM SIGN SUMCUNEIFORM SIGN SUMASHCUNEIFORM SIGN SURCUNEIFORM SIGN SUR9CUNEIFORM SIGN TACUNEIFORM SIGN TA ASTERISKCUNEIFORM SIGN TA TIMES HICUNEIFORM SIGN TA TIMES MICUNEIFORM SIGN TA GUNUCUNEIFORM SIGN TABCUNEIFORM\
    \ SIGN TAB OVER TAB NI OVER NI DISH OVER DISHCUNEIFORM SIGN TAB SQUAREDCUNEIFORM SIGN TAGCUNEIFORM SIGN TAG TIMES BICUNEIFORM SIGN TAG TIMES GUDCUNEIFORM SIGN TAG TIMES SHECUNEIFORM SIGN TAG TIMES SHUCUNEIFORM SIGN TAG TIMES TUG2CUNEIFORM SIGN TAG TIMES UD\
    \CUNEIFORM SIGN TAK4CUNEIFORM SIGN TARCUNEIFORM SIGN TECUNEIFORM SIGN TE GUNUCUNEIFORM SIGN TICUNEIFORM SIGN TI TENUCUNEIFORM SIGN TILCUNEIFORM SIGN TIRCUNEIFORM SIGN TIR TIMES TAK4CUNEIFORM SIGN TIR OVER TIRCUNEIFORM SIGN TIR OVER TIR GAD OVER GAD GAR OVE\
    \R GARCUNEIFORM SIGN TUCUNEIFORM SIGN TUG2CUNEIFORM SIGN TUKCUNEIFORM SIGN TUMCUNEIFORM SIGN TURCUNEIFORM SIGN TUR OVER TUR ZA OVER ZACUNEIFORM SIGN UCUNEIFORM SIGN U GUDCUNEIFORM SIGN U U UCUNEIFORM SIGN U OVER U PA OVER PA GAR OVER GARCUNEIFORM SIGN U OV\
    \ER U SUR OVER SURCUNEIFORM SIGN U OVER U U REVERSED OVER U REVERSEDCUNEIFORM SIGN U2CUNEIFORM SIGN UBCUNEIFORM SIGN UDCUNEIFORM SIGN UD KUSHU2CUNEIFORM SIGN UD TIMES BADCUNEIFORM SIGN UD TIMES MICUNEIFORM SIGN UD TIMES U PLUS U PLUS UCUNEIFORM SIGN UD TIM\
    \ES U PLUS U PLUS U GUNUCUNEIFORM SIGN UD GUNUCUNEIFORM SIGN UD SHESHIGCUNEIFORM SIGN UD SHESHIG TIMES BADCUNEIFORM SIGN UDUGCUNEIFORM SIGN UMCUNEIFORM SIGN UM TIMES LAGABCUNEIFORM SIGN UM TIMES ME PLUS DACUNEIFORM SIGN UM TIMES SHA3CUNEIFORM SIGN UM TIMES\
    \ UCUNEIFORM SIGN UMBINCUNEIFORM SIGN UMUMCUNEIFORM SIGN UMUM TIMES KASKALCUNEIFORM SIGN UMUM TIMES PACUNEIFORM SIGN UNCUNEIFORM SIGN UN GUNUCUNEIFORM SIGN URCUNEIFORM SIGN UR CROSSING URCUNEIFORM SIGN UR SHESHIGCUNEIFORM SIGN UR2CUNEIFORM SIGN UR2 TIMES A\
    \ PLUS HACUNEIFORM SIGN UR2 TIMES A PLUS NACUNEIFORM SIGN UR2 TIMES ALCUNEIFORM SIGN UR2 TIMES HACUNEIFORM SIGN UR2 TIMES NUNCUNEIFORM SIGN UR2 TIMES U2CUNEIFORM SIGN UR2 TIMES U2 PLUS ASHCUNEIFORM SIGN UR2 TIMES U2 PLUS BICUNEIFORM SIGN UR4CUNEIFORM SIGN \
    \URICUNEIFORM SIGN URI3CUNEIFORM SIGN URUCUNEIFORM SIGN URU TIMES ACUNEIFORM SIGN URU TIMES ASHGABCUNEIFORM SIGN URU TIMES BARCUNEIFORM SIGN URU TIMES DUNCUNEIFORM SIGN URU TIMES GACUNEIFORM SIGN URU TIMES GALCUNEIFORM SIGN URU TIMES GAN2 TENUCUNEIFORM SIG\
    \N URU TIMES GARCUNEIFORM SIGN URU TIMES GUCUNEIFORM SIGN URU TIMES HACUNEIFORM SIGN URU TIMES IGICUNEIFORM SIGN URU TIMES IMCUNEIFORM SIGN URU TIMES ISHCUNEIFORM SIGN URU TIMES KICUNEIFORM SIGN URU TIMES LUMCUNEIFORM SIGN URU TIMES MINCUNEIFORM SIGN URU T\
    \IMES PACUNEIFORM SIGN URU TIMES SHECUNEIFORM SIGN URU TIMES SIG4CUNEIFORM SIGN URU TIMES TUCUNEIFORM SIGN URU TIMES U PLUS GUDCUNEIFORM SIGN URU TIMES UDCUNEIFORM SIGN URU TIMES URUDACUNEIFORM SIGN URUDACUNEIFORM SIGN URUDA TIMES UCUNEIFORM SIGN USHCUNEIF\
    \ORM SIGN USH TIMES ACUNEIFORM SIGN USH TIMES KUCUNEIFORM SIGN USH TIMES KURCUNEIFORM SIGN USH TIMES TAK4CUNEIFORM SIGN USHXCUNEIFORM SIGN USH2CUNEIFORM SIGN USHUMXCUNEIFORM SIGN UTUKICUNEIFORM SIGN UZ3CUNEIFORM SIGN UZ3 TIMES KASKALCUNEIFORM SIGN UZUCUNEI\
    \FORM SIGN ZACUNEIFORM SIGN ZA TENUCUNEIFORM SIGN ZA SQUARED TIMES KURCUNEIFORM SIGN ZAGCUNEIFORM SIGN ZAMXCUNEIFORM SIGN ZE2CUNEIFORM SIGN ZICUNEIFORM SIGN ZI OVER ZICUNEIFORM SIGN ZI3CUNEIFORM SIGN ZIBCUNEIFORM SIGN ZIB KABA TENUCUNEIFORM SIGN ZIGCUNEIFO\
    \RM SIGN ZIZ2CUNEIFORM SIGN ZUCUNEIFORM SIGN ZU5CUNEIFORM SIGN ZU5 TIMES ACUNEIFORM SIGN ZUBURCUNEIFORM SIGN ZUMCUNEIFORM SIGN KAP ELAMITECUNEIFORM SIGN AB TIMES NUNCUNEIFORM SIGN AB2 TIMES ACUNEIFORM SIGN AMAR TIMES KUGCUNEIFORM SIGN DAG KISIM5 TIMES U2 P\
    \LUS MASHCUNEIFORM SIGN DAG3CUNEIFORM SIGN DISH PLUS SHUCUNEIFORM SIGN DUB TIMES SHECUNEIFORM SIGN EZEN TIMES GUDCUNEIFORM SIGN EZEN TIMES SHECUNEIFORM SIGN GA2 TIMES AN PLUS KAK PLUS ACUNEIFORM SIGN GA2 TIMES ASH2CUNEIFORM SIGN GE22CUNEIFORM SIGN GIGCUNEI\
    \FORM SIGN HUSHCUNEIFORM SIGN KA TIMES ANSHECUNEIFORM SIGN KA TIMES ASH3CUNEIFORM SIGN KA TIMES GISHCUNEIFORM SIGN KA TIMES GUDCUNEIFORM SIGN KA TIMES HI TIMES ASH2CUNEIFORM SIGN KA TIMES LUMCUNEIFORM SIGN KA TIMES PACUNEIFORM SIGN KA TIMES SHULCUNEIFORM S\
    \IGN KA TIMES TUCUNEIFORM SIGN KA TIMES UR2CUNEIFORM SIGN LAGAB TIMES GICUNEIFORM SIGN LU2 SHESHIG TIMES BADCUNEIFORM SIGN LU2 TIMES ESH2 PLUS LALCUNEIFORM SIGN LU2 TIMES SHUCUNEIFORM SIGN MESHCUNEIFORM SIGN MUSH3 TIMES ZACUNEIFORM SIGN NA4CUNEIFORM SIGN N\
    \INCUNEIFORM SIGN NIN9CUNEIFORM SIGN NINDA2 TIMES BALCUNEIFORM SIGN NINDA2 TIMES GICUNEIFORM SIGN NU11 ROTATED NINETY DEGREESCUNEIFORM SIGN PESH2 ASTERISKCUNEIFORM SIGN PIR2CUNEIFORM SIGN SAG TIMES IGI GUNUCUNEIFORM SIGN TI2CUNEIFORM SIGN UM TIMES MECUNEIF\
    \ORM SIGN U UCUNEIFORM NUMERIC SIGN TWO ASHCUNEIFORM NUMERIC SIGN THREE ASHCUNEIFORM NUMERIC SIGN FOUR ASHCUNEIFORM NUMERIC SIGN FIVE ASHCUNEIFORM NUMERIC SIGN SIX ASHCUNEIFORM NUMERIC SIGN SEVEN ASHCUNEIFORM NUMERIC SIGN EIGHT ASHCUNEIFORM NUMERIC SIGN NI\
    \NE ASHCUNEIFORM NUMERIC SIGN THREE DISHCUNEIFORM NUMERIC SIGN FOUR DISHCUNEIFORM NUMERIC SIGN FIVE DISHCUNEIFORM NUMERIC SIGN SIX DISHCUNEIFORM NUMERIC SIGN SEVEN DISHCUNEIFORM NUMERIC SIGN EIGHT DISHCUNEIFORM NUMERIC SIGN NINE DISHCUNEIFORM NUMERIC SIGN \
    \FOUR UCUNEIFORM NUMERIC SIGN FIVE UCUNEIFORM NUMERIC SIGN SIX UCUNEIFORM NUMERIC SIGN SEVEN UCUNEIFORM NUMERIC SIGN EIGHT UCUNEIFORM NUMERIC SIGN NINE UCUNEIFORM NUMERIC SIGN ONE GESH2CUNEIFORM NUMERIC SIGN TWO GESH2CUNEIFORM NUMERIC SIGN THREE GESH2CUNEI\
    \FORM NUMERIC SIGN FOUR GESH2CUNEIFORM NUMERIC SIGN FIVE GESH2CUNEIFORM NUMERIC SIGN SIX GESH2CUNEIFORM NUMERIC SIGN SEVEN GESH2CUNEIFORM NUMERIC SIGN EIGHT GESH2CUNEIFORM NUMERIC SIGN NINE GESH2CUNEIFORM NUMERIC SIGN ONE GESHUCUNEIFORM NUMERIC SIGN TWO GE\
    \SHUCUNEIFORM NUMERIC SIGN THREE GESHUCUNEIFORM NUMERIC SIGN FOUR GESHUCUNEIFORM NUMERIC SIGN FIVE GESHUCUNEIFORM NUMERIC SIGN TWO SHAR2CUNEIFORM NUMERIC SIGN THREE SHAR2CUNEIFORM NUMERIC SIGN THREE SHAR2 VARIANT FORMCUNEIFORM NUMERIC SIGN FOUR SHAR2CUNEIF\
    \ORM NUMERIC SIGN FIVE SHAR2CUNEIFORM NUMERIC SIGN SIX SHAR2CUNEIFORM NUMERIC SIGN SEVEN SHAR2CUNEIFORM NUMERIC SIGN EIGHT SHAR2CUNEIFORM NUMERIC SIGN NINE SHAR2CUNEIFORM NUMERIC SIGN ONE SHARUCUNEIFORM NUMERIC SIGN TWO SHARUCUNEIFORM NUMERIC SIGN THREE SH\
    \ARUCUNEIFORM NUMERIC SIGN THREE SHARU VARIANT FORMCUNEIFORM NUMERIC SIGN FOUR SHARUCUNEIFORM NUMERIC SIGN FIVE SHARUCUNEIFORM NUMERIC SIGN SHAR2 TIMES GAL PLUS DISHCUNEIFORM NUMERIC SIGN SHAR2 TIMES GAL PLUS MINCUNEIFORM NUMERIC SIGN ONE BURUCUNEIFORM NUM\
    \ERIC SIGN TWO BURUCUNEIFORM NUMERIC SIGN THREE BURUCUNEIFORM NUMERIC SIGN THREE BURU VARIANT FORMCUNEIFORM NUMERIC SIGN FOUR BURUCUNEIFORM NUMERIC SIGN FIVE BURUCUNEIFORM NUMERIC SIGN THREE VARIANT FORM ESH16CUNEIFORM NUMERIC SIGN THREE VARIANT FORM ESH21\
    \CUNEIFORM NUMERIC SIGN FOUR VARIANT FORM LIMMUCUNEIFORM NUMERIC SIGN FOUR VARIANT FORM LIMMU4CUNEIFORM NUMERIC SIGN FOUR VARIANT FORM LIMMU ACUNEIFORM NUMERIC SIGN FOUR VARIANT FORM LIMMU BCUNEIFORM NUMERIC SIGN SIX VARIANT FORM ASH9CUNEIFORM NUMERIC SIGN\
    \ SEVEN VARIANT FORM IMIN3CUNEIFORM NUMERIC SIGN SEVEN VARIANT FORM IMIN ACUNEIFORM NUMERIC SIGN SEVEN VARIANT FORM IMIN BCUNEIFORM NUMERIC SIGN EIGHT VARIANT FORM USSUCUNEIFORM NUMERIC SIGN EIGHT VARIANT FORM USSU3CUNEIFORM NUMERIC SIGN NINE VARIANT FORM \
    \ILIMMUCUNEIFORM NUMERIC SIGN NINE VARIANT FORM ILIMMU3CUNEIFORM NUMERIC SIGN NINE VARIANT FORM ILIMMU4CUNEIFORM NUMERIC SIGN NINE VARIANT FORM ILIMMU ACUNEIFORM NUMERIC SIGN TWO ASH TENUCUNEIFORM NUMERIC SIGN THREE ASH TENUCUNEIFORM NUMERIC SIGN FOUR ASH \
    \TENUCUNEIFORM NUMERIC SIGN FIVE ASH TENUCUNEIFORM NUMERIC SIGN SIX ASH TENUCUNEIFORM NUMERIC SIGN ONE BAN2CUNEIFORM NUMERIC SIGN TWO BAN2CUNEIFORM NUMERIC SIGN THREE BAN2CUNEIFORM NUMERIC SIGN FOUR BAN2CUNEIFORM NUMERIC SIGN FOUR BAN2 VARIANT FORMCUNEIFOR\
    \M NUMERIC SIGN FIVE BAN2CUNEIFORM NUMERIC SIGN FIVE BAN2 VARIANT FORMCUNEIFORM NUMERIC SIGN NIGIDAMINCUNEIFORM NUMERIC SIGN NIGIDAESHCUNEIFORM NUMERIC SIGN ONE ESHE3CUNEIFORM NUMERIC SIGN TWO ESHE3CUNEIFORM NUMERIC SIGN ONE THIRD DISHCUNEIFORM NUMERIC SIG\
    \N TWO THIRDS DISHCUNEIFORM NUMERIC SIGN FIVE SIXTHS DISHCUNEIFORM NUMERIC SIGN ONE THIRD VARIANT FORM ACUNEIFORM NUMERIC SIGN TWO THIRDS VARIANT FORM ACUNEIFORM NUMERIC SIGN ONE EIGHTH ASHCUNEIFORM NUMERIC SIGN ONE QUARTER ASHCUNEIFORM NUMERIC SIGN OLD AS\
    \SYRIAN ONE SIXTHCUNEIFORM NUMERIC SIGN OLD ASSYRIAN ONE QUARTERCUNEIFORM NUMERIC SIGN ONE QUARTER GURCUNEIFORM NUMERIC SIGN ONE HALF GURCUNEIFORM NUMERIC SIGN ELAMITE ONE THIRDCUNEIFORM NUMERIC SIGN ELAMITE TWO THIRDSCUNEIFORM NUMERIC SIGN ELAMITE FORTYCU\
    \NEIFORM NUMERIC SIGN ELAMITE FIFTYCUNEIFORM NUMERIC SIGN FOUR U VARIANT FORMCUNEIFORM NUMERIC SIGN FIVE U VARIANT FORMCUNEIFORM NUMERIC SIGN SIX U VARIANT FORMCUNEIFORM NUMERIC SIGN SEVEN U VARIANT FORMCUNEIFORM NUMERIC SIGN EIGHT U VARIANT FORMCUNEIFORM \
    \NUMERIC SIGN NINE U VARIANT FORMCUNEIFORM PUNCTUATION SIGN OLD ASSYRIAN WORD DIVIDERCUNEIFORM PUNCTUATION SIGN VERTICAL COLONCUNEIFORM PUNCTUATION SIGN DIAGONAL COLONCUNEIFORM PUNCTUATION SIGN DIAGONAL TRICOLONCUNEIFORM PUNCTUATION SIGN DIAGONAL QUADCOLON\
    \CUNEIFORM SIGN AB TIMES NUN TENUCUNEIFORM SIGN AB TIMES SHU2CUNEIFORM SIGN AD TIMES ESH2CUNEIFORM SIGN BAD TIMES DISH TENUCUNEIFORM SIGN BAHAR2 TIMES AB2CUNEIFORM SIGN BAHAR2 TIMES NICUNEIFORM SIGN BAHAR2 TIMES ZACUNEIFORM SIGN BU OVER BU TIMES NA2CUNEIFO\
    \RM SIGN DA TIMES TAK4CUNEIFORM SIGN DAG TIMES KURCUNEIFORM SIGN DIM TIMES IGICUNEIFORM SIGN DIM TIMES U U UCUNEIFORM SIGN DIM2 TIMES UDCUNEIFORM SIGN DUG TIMES ANSHECUNEIFORM SIGN DUG TIMES ASHCUNEIFORM SIGN DUG TIMES ASH AT LEFTCUNEIFORM SIGN DUG TIMES D\
    \INCUNEIFORM SIGN DUG TIMES DUNCUNEIFORM SIGN DUG TIMES ERIN2CUNEIFORM SIGN DUG TIMES GACUNEIFORM SIGN DUG TIMES GICUNEIFORM SIGN DUG TIMES GIR2 GUNUCUNEIFORM SIGN DUG TIMES GISHCUNEIFORM SIGN DUG TIMES HACUNEIFORM SIGN DUG TIMES HICUNEIFORM SIGN DUG TIMES\
    \ IGI GUNUCUNEIFORM SIGN DUG TIMES KASKALCUNEIFORM SIGN DUG TIMES KURCUNEIFORM SIGN DUG TIMES KUSHU2CUNEIFORM SIGN DUG TIMES KUSHU2 PLUS KASKALCUNEIFORM SIGN DUG TIMES LAK-020CUNEIFORM SIGN DUG TIMES LAMCUNEIFORM SIGN DUG TIMES LAM TIMES KURCUNEIFORM SIGN \
    \DUG TIMES LUH PLUS GISHCUNEIFORM SIGN DUG TIMES MASHCUNEIFORM SIGN DUG TIMES MESCUNEIFORM SIGN DUG TIMES MICUNEIFORM SIGN DUG TIMES NICUNEIFORM SIGN DUG TIMES PICUNEIFORM SIGN DUG TIMES SHECUNEIFORM SIGN DUG TIMES SI GUNUCUNEIFORM SIGN E2 TIMES KURCUNEIFO\
    \RM SIGN E2 TIMES PAPCUNEIFORM SIGN ERIN2 XCUNEIFORM SIGN ESH2 CROSSING ESH2CUNEIFORM SIGN EZEN SHESHIG TIMES ASHCUNEIFORM SIGN EZEN SHESHIG TIMES HICUNEIFORM SIGN EZEN SHESHIG TIMES IGI GUNUCUNEIFORM SIGN EZEN SHESHIG TIMES LACUNEIFORM SIGN EZEN SHESHIG T\
    \IMES LALCUNEIFORM SIGN EZEN SHESHIG TIMES MECUNEIFORM SIGN EZEN SHESHIG TIMES MESCUNEIFORM SIGN EZEN SHESHIG TIMES SUCUNEIFORM SIGN EZEN TIMES SUCUNEIFORM SIGN GA2 TIMES BAHAR2CUNEIFORM SIGN GA2 TIMES DIM GUNUCUNEIFORM SIGN GA2 TIMES DUG TIMES IGI GUNUCUN\
    \EIFORM SIGN GA2 TIMES DUG TIMES KASKALCUNEIFORM SIGN GA2 TIMES ERENCUNEIFORM SIGN GA2 TIMES GACUNEIFORM SIGN GA2 TIMES GAR PLUS DICUNEIFORM SIGN GA2 TIMES GAR PLUS NECUNEIFORM SIGN GA2 TIMES HA PLUS ACUNEIFORM SIGN GA2 TIMES KUSHU2 PLUS KASKALCUNEIFORM SI\
    \GN GA2 TIMES LAMCUNEIFORM SIGN GA2 TIMES LAM TIMES KURCUNEIFORM SIGN GA2 TIMES LUHCUNEIFORM SIGN GA2 TIMES MUSHCUNEIFORM SIGN GA2 TIMES NECUNEIFORM SIGN GA2 TIMES NE PLUS E2CUNEIFORM SIGN GA2 TIMES NE PLUS GICUNEIFORM SIGN GA2 TIMES SHIMCUNEIFORM SIGN GA2\
    \ TIMES ZIZ2CUNEIFORM SIGN GABA ROTATED NINETY DEGREESCUNEIFORM SIGN GESHTIN TIMES UCUNEIFORM SIGN GISH TIMES GISH CROSSING GISHCUNEIFORM SIGN GU2 TIMES IGI GUNUCUNEIFORM SIGN GUD PLUS GISH TIMES TAK4CUNEIFORM SIGN HA TENU GUNUCUNEIFORM SIGN HI TIMES ASH O\
    \VER HI TIMES ASHCUNEIFORM SIGN KA TIMES BUCUNEIFORM SIGN KA TIMES KACUNEIFORM SIGN KA TIMES U U UCUNEIFORM SIGN KA TIMES URCUNEIFORM SIGN LAGAB TIMES ZU OVER ZUCUNEIFORM SIGN LAK-003CUNEIFORM SIGN LAK-021CUNEIFORM SIGN LAK-025CUNEIFORM SIGN LAK-030CUNEIFO\
    \RM SIGN LAK-050CUNEIFORM SIGN LAK-051CUNEIFORM SIGN LAK-062CUNEIFORM SIGN LAK-079 OVER LAK-079 GUNUCUNEIFORM SIGN LAK-080CUNEIFORM SIGN LAK-081 OVER LAK-081CUNEIFORM SIGN LAK-092CUNEIFORM SIGN LAK-130CUNEIFORM SIGN LAK-142CUNEIFORM SIGN LAK-210CUNEIFORM S\
    \IGN LAK-219CUNEIFORM SIGN LAK-220CUNEIFORM SIGN LAK-225CUNEIFORM SIGN LAK-228CUNEIFORM SIGN LAK-238CUNEIFORM SIGN LAK-265CUNEIFORM SIGN LAK-266CUNEIFORM SIGN LAK-343CUNEIFORM SIGN LAK-347CUNEIFORM SIGN LAK-348CUNEIFORM SIGN LAK-383CUNEIFORM SIGN LAK-384CU\
    \NEIFORM SIGN LAK-390CUNEIFORM SIGN LAK-441CUNEIFORM SIGN LAK-449CUNEIFORM SIGN LAK-449 TIMES GUCUNEIFORM SIGN LAK-449 TIMES IGICUNEIFORM SIGN LAK-449 TIMES PAP PLUS LU3CUNEIFORM SIGN LAK-449 TIMES PAP PLUS PAP PLUS LU3CUNEIFORM SIGN LAK-449 TIMES U2 PLUS \
    \BACUNEIFORM SIGN LAK-450CUNEIFORM SIGN LAK-457CUNEIFORM SIGN LAK-470CUNEIFORM SIGN LAK-483CUNEIFORM SIGN LAK-490CUNEIFORM SIGN LAK-492CUNEIFORM SIGN LAK-493CUNEIFORM SIGN LAK-495CUNEIFORM SIGN LAK-550CUNEIFORM SIGN LAK-608CUNEIFORM SIGN LAK-617CUNEIFORM S\
    \IGN LAK-617 TIMES ASHCUNEIFORM SIGN LAK-617 TIMES BADCUNEIFORM SIGN LAK-617 TIMES DUN3 GUNU GUNUCUNEIFORM SIGN LAK-617 TIMES KU3CUNEIFORM SIGN LAK-617 TIMES LACUNEIFORM SIGN LAK-617 TIMES TARCUNEIFORM SIGN LAK-617 TIMES TECUNEIFORM SIGN LAK-617 TIMES U2CU\
    \NEIFORM SIGN LAK-617 TIMES UDCUNEIFORM SIGN LAK-617 TIMES URUDACUNEIFORM SIGN LAK-636CUNEIFORM SIGN LAK-648CUNEIFORM SIGN LAK-648 TIMES DUBCUNEIFORM SIGN LAK-648 TIMES GACUNEIFORM SIGN LAK-648 TIMES IGICUNEIFORM SIGN LAK-648 TIMES IGI GUNUCUNEIFORM SIGN L\
    \AK-648 TIMES NICUNEIFORM SIGN LAK-648 TIMES PAP PLUS PAP PLUS LU3CUNEIFORM SIGN LAK-648 TIMES SHESH PLUS KICUNEIFORM SIGN LAK-648 TIMES UDCUNEIFORM SIGN LAK-648 TIMES URUDACUNEIFORM SIGN LAK-724CUNEIFORM SIGN LAK-749CUNEIFORM SIGN LU2 GUNU TIMES ASHCUNEIF\
    \ORM SIGN LU2 TIMES DISHCUNEIFORM SIGN LU2 TIMES HALCUNEIFORM SIGN LU2 TIMES PAPCUNEIFORM SIGN LU2 TIMES PAP PLUS PAP PLUS LU3CUNEIFORM SIGN LU2 TIMES TAK4CUNEIFORM SIGN MI PLUS ZA7CUNEIFORM SIGN MUSH OVER MUSH TIMES GACUNEIFORM SIGN MUSH OVER MUSH TIMES K\
    \AKCUNEIFORM SIGN NINDA2 TIMES DIM GUNUCUNEIFORM SIGN NINDA2 TIMES GISHCUNEIFORM SIGN NINDA2 TIMES GULCUNEIFORM SIGN NINDA2 TIMES HICUNEIFORM SIGN NINDA2 TIMES KESH2CUNEIFORM SIGN NINDA2 TIMES LAK-050CUNEIFORM SIGN NINDA2 TIMES MASHCUNEIFORM SIGN NINDA2 TI\
    \MES PAP PLUS PAPCUNEIFORM SIGN NINDA2 TIMES UCUNEIFORM SIGN NINDA2 TIMES U PLUS UCUNEIFORM SIGN NINDA2 TIMES URUDACUNEIFORM SIGN SAG GUNU TIMES HACUNEIFORM SIGN SAG TIMES ENCUNEIFORM SIGN SAG TIMES SHE AT LEFTCUNEIFORM SIGN SAG TIMES TAK4CUNEIFORM SIGN SH\
    \A6 TENUCUNEIFORM SIGN SHE OVER SHECUNEIFORM SIGN SHE PLUS HUB2CUNEIFORM SIGN SHE PLUS NAM2CUNEIFORM SIGN SHE PLUS SARCUNEIFORM SIGN SHU2 PLUS DUG TIMES NICUNEIFORM SIGN SHU2 PLUS E2 TIMES ANCUNEIFORM SIGN SI TIMES TAK4CUNEIFORM SIGN TAK4 PLUS SAGCUNEIFORM\
    \ SIGN TUM TIMES GAN2 TENUCUNEIFORM SIGN TUM TIMES THREE DISHCUNEIFORM SIGN UR2 INVERTEDCUNEIFORM SIGN UR2 TIMES UDCUNEIFORM SIGN URU TIMES DARA3CUNEIFORM SIGN URU TIMES LAK-668CUNEIFORM SIGN URU TIMES LU3CUNEIFORM SIGN ZA7CUNEIFORM SIGN ZU OVER ZU PLUS SA\
    \RCUNEIFORM SIGN ZU5 TIMES THREE DISH TENUCYPRO-MINOAN SIGN CM001CYPRO-MINOAN SIGN CM002CYPRO-MINOAN SIGN CM004CYPRO-MINOAN SIGN CM005CYPRO-MINOAN SIGN CM006CYPRO-MINOAN SIGN CM007CYPRO-MINOAN SIGN CM008CYPRO-MINOAN SIGN CM009CYPRO-MINOAN SIGN CM010CYPRO-M\
    \INOAN SIGN CM011CYPRO-MINOAN SIGN CM012CYPRO-MINOAN SIGN CM012BCYPRO-MINOAN SIGN CM013CYPRO-MINOAN SIGN CM015CYPRO-MINOAN SIGN CM017CYPRO-MINOAN SIGN CM019CYPRO-MINOAN SIGN CM021CYPRO-MINOAN SIGN CM023CYPRO-MINOAN SIGN CM024CYPRO-MINOAN SIGN CM025CYPRO-MI\
    \NOAN SIGN CM026CYPRO-MINOAN SIGN CM027CYPRO-MINOAN SIGN CM028CYPRO-MINOAN SIGN CM029CYPRO-MINOAN SIGN CM030CYPRO-MINOAN SIGN CM033CYPRO-MINOAN SIGN CM034CYPRO-MINOAN SIGN CM035CYPRO-MINOAN SIGN CM036CYPRO-MINOAN SIGN CM037CYPRO-MINOAN SIGN CM038CYPRO-MINO\
    \AN SIGN CM039CYPRO-MINOAN SIGN CM040CYPRO-MINOAN SIGN CM041CYPRO-MINOAN SIGN CM044CYPRO-MINOAN SIGN CM046CYPRO-MINOAN SIGN CM047CYPRO-MINOAN SIGN CM049CYPRO-MINOAN SIGN CM050CYPRO-MINOAN SIGN CM051CYPRO-MINOAN SIGN CM052CYPRO-MINOAN SIGN CM053CYPRO-MINOAN\
    \ SIGN CM054CYPRO-MINOAN SIGN CM055CYPRO-MINOAN SIGN CM056CYPRO-MINOAN SIGN CM058CYPRO-MINOAN SIGN CM059CYPRO-MINOAN SIGN CM060CYPRO-MINOAN SIGN CM061CYPRO-MINOAN SIGN CM062CYPRO-MINOAN SIGN CM063CYPRO-MINOAN SIGN CM064CYPRO-MINOAN SIGN CM066CYPRO-MINOAN S\
    \IGN CM067CYPRO-MINOAN SIGN CM068CYPRO-MINOAN SIGN CM069CYPRO-MINOAN SIGN CM070CYPRO-MINOAN SIGN CM071CYPRO-MINOAN SIGN CM072CYPRO-MINOAN SIGN CM073CYPRO-MINOAN SIGN CM074CYPRO-MINOAN SIGN CM075CYPRO-MINOAN SIGN CM075BCYPRO-MINOAN SIGN CM076CYPRO-MINOAN SI\
    \GN CM078CYPRO-MINOAN SIGN CM079CYPRO-MINOAN SIGN CM080CYPRO-MINOAN SIGN CM081CYPRO-MINOAN SIGN CM082CYPRO-MINOAN SIGN CM083CYPRO-MINOAN SIGN CM084CYPRO-MINOAN SIGN CM085CYPRO-MINOAN SIGN CM086CYPRO-MINOAN SIGN CM087CYPRO-MINOAN SIGN CM088CYPRO-MINOAN SIGN\
    \ CM089CYPRO-MINOAN SIGN CM090CYPRO-MINOAN SIGN CM091CYPRO-MINOAN SIGN CM092CYPRO-MINOAN SIGN CM094CYPRO-MINOAN SIGN CM095CYPRO-MINOAN SIGN CM096CYPRO-MINOAN SIGN CM097CYPRO-MINOAN SIGN CM098CYPRO-MINOAN SIGN CM099CYPRO-MINOAN SIGN CM100CYPRO-MINOAN SIGN C\
    \M101CYPRO-MINOAN SIGN CM102CYPRO-MINOAN SIGN CM103CYPRO-MINOAN SIGN CM104CYPRO-MINOAN SIGN CM105CYPRO-MINOAN SIGN CM107CYPRO-MINOAN SIGN CM108CYPRO-MINOAN SIGN CM109CYPRO-MINOAN SIGN CM110CYPRO-MINOAN SIGN CM112CYPRO-MINOAN SIGN CM114CYPRO-MINOAN SIGN CM3\
    \01CYPRO-MINOAN SIGN CM302EGYPTIAN HIEROGLYPH A001EGYPTIAN HIEROGLYPH A002EGYPTIAN HIEROGLYPH A003EGYPTIAN HIEROGLYPH A004EGYPTIAN HIEROGLYPH A005EGYPTIAN HIEROGLYPH A005AEGYPTIAN HIEROGLYPH A006EGYPTIAN HIEROGLYPH A006AEGYPTIAN HIEROGLYPH A006BEGYPTIAN HI\
    \EROGLYPH A007EGYPTIAN HIEROGLYPH A008EGYPTIAN HIEROGLYPH A009EGYPTIAN HIEROGLYPH A010EGYPTIAN HIEROGLYPH A011EGYPTIAN HIEROGLYPH A012EGYPTIAN HIEROGLYPH A013EGYPTIAN HIEROGLYPH A014EGYPTIAN HIEROGLYPH A014AEGYPTIAN HIEROGLYPH A015EGYPTIAN HIEROGLYPH A016E\
    \GYPTIAN HIEROGLYPH A017EGYPTIAN HIEROGLYPH A017AEGYPTIAN HIEROGLYPH A018EGYPTIAN HIEROGLYPH A019EGYPTIAN HIEROGLYPH A020EGYPTIAN HIEROGLYPH A021EGYPTIAN HIEROGLYPH A022EGYPTIAN HIEROGLYPH A023EGYPTIAN HIEROGLYPH A024EGYPTIAN HIEROGLYPH A025EGYPTIAN HIEROG\
    \LYPH A026EGYPTIAN HIEROGLYPH A027EGYPTIAN HIEROGLYPH A028EGYPTIAN HIEROGLYPH A029EGYPTIAN HIEROGLYPH A030EGYPTIAN HIEROGLYPH A031EGYPTIAN HIEROGLYPH A032EGYPTIAN HIEROGLYPH A032AEGYPTIAN HIEROGLYPH A033EGYPTIAN HIEROGLYPH A034EGYPTIAN HIEROGLYPH A035EGYPT\
    \IAN HIEROGLYPH A036EGYPTIAN HIEROGLYPH A037EGYPTIAN HIEROGLYPH A038EGYPTIAN HIEROGLYPH A039EGYPTIAN HIEROGLYPH A040EGYPTIAN HIEROGLYPH A040AEGYPTIAN HIEROGLYPH A041EGYPTIAN HIEROGLYPH A042EGYPTIAN HIEROGLYPH A042AEGYPTIAN HIEROGLYPH A043EGYPTIAN HIEROGLYP\
    \H A043AEGYPTIAN HIEROGLYPH A044EGYPTIAN HIEROGLYPH A045EGYPTIAN HIEROGLYPH A045AEGYPTIAN HIEROGLYPH A046EGYPTIAN HIEROGLYPH A047EGYPTIAN HIEROGLYPH A048EGYPTIAN HIEROGLYPH A049EGYPTIAN HIEROGLYPH A050EGYPTIAN HIEROGLYPH A051EGYPTIAN HIEROGLYPH A052EGYPTIA\
    \N HIEROGLYPH A053EGYPTIAN HIEROGLYPH A054EGYPTIAN HIEROGLYPH A055EGYPTIAN HIEROGLYPH A056EGYPTIAN HIEROGLYPH A057EGYPTIAN HIEROGLYPH A058EGYPTIAN HIEROGLYPH A059EGYPTIAN HIEROGLYPH A060EGYPTIAN HIEROGLYPH A061EGYPTIAN HIEROGLYPH A062EGYPTIAN HIEROGLYPH A0\
    \63EGYPTIAN HIEROGLYPH A064EGYPTIAN HIEROGLYPH A065EGYPTIAN HIEROGLYPH A066EGYPTIAN HIEROGLYPH A067EGYPTIAN HIEROGLYPH A068EGYPTIAN HIEROGLYPH A069EGYPTIAN HIEROGLYPH A070EGYPTIAN HIEROGLYPH B001EGYPTIAN HIEROGLYPH B002EGYPTIAN HIEROGLYPH B003EGYPTIAN HIER\
    \OGLYPH B004EGYPTIAN HIEROGLYPH B005EGYPTIAN HIEROGLYPH B005AEGYPTIAN HIEROGLYPH B006EGYPTIAN HIEROGLYPH B007EGYPTIAN HIEROGLYPH B008EGYPTIAN HIEROGLYPH B009EGYPTIAN HIEROGLYPH C001EGYPTIAN HIEROGLYPH C002EGYPTIAN HIEROGLYPH C002AEGYPTIAN HIEROGLYPH C002BE\
    \GYPTIAN HIEROGLYPH C002CEGYPTIAN HIEROGLYPH C003EGYPTIAN HIEROGLYPH C004EGYPTIAN HIEROGLYPH C005EGYPTIAN HIEROGLYPH C006EGYPTIAN HIEROGLYPH C007EGYPTIAN HIEROGLYPH C008EGYPTIAN HIEROGLYPH C009EGYPTIAN HIEROGLYPH C010EGYPTIAN HIEROGLYPH C010AEGYPTIAN HIERO\
    \GLYPH C011EGYPTIAN HIEROGLYPH C012EGYPTIAN HIEROGLYPH C013EGYPTIAN HIEROGLYPH C014EGYPTIAN HIEROGLYPH C015EGYPTIAN HIEROGLYPH C016EGYPTIAN HIEROGLYPH C017EGYPTIAN HIEROGLYPH C018EGYPTIAN HIEROGLYPH C019EGYPTIAN HIEROGLYPH C020EGYPTIAN HIEROGLYPH C021EGYPT\
    \IAN HIEROGLYPH C022EGYPTIAN HIEROGLYPH C023EGYPTIAN HIEROGLYPH C024EGYPTIAN HIEROGLYPH D001EGYPTIAN HIEROGLYPH D002EGYPTIAN HIEROGLYPH D003EGYPTIAN HIEROGLYPH D004EGYPTIAN HIEROGLYPH D005EGYPTIAN HIEROGLYPH D006EGYPTIAN HIEROGLYPH D007EGYPTIAN HIEROGLYPH \
    \D008EGYPTIAN HIEROGLYPH D008AEGYPTIAN HIEROGLYPH D009EGYPTIAN HIEROGLYPH D010EGYPTIAN HIEROGLYPH D011EGYPTIAN HIEROGLYPH D012EGYPTIAN HIEROGLYPH D013EGYPTIAN HIEROGLYPH D014EGYPTIAN HIEROGLYPH D015EGYPTIAN HIEROGLYPH D016EGYPTIAN HIEROGLYPH D017EGYPTIAN H\
    \IEROGLYPH D018EGYPTIAN HIEROGLYPH D019EGYPTIAN HIEROGLYPH D020EGYPTIAN HIEROGLYPH D021EGYPTIAN HIEROGLYPH D022EGYPTIAN HIEROGLYPH D023EGYPTIAN HIEROGLYPH D024EGYPTIAN HIEROGLYPH D025EGYPTIAN HIEROGLYPH D026EGYPTIAN HIEROGLYPH D027EGYPTIAN HIEROGLYPH D027A\
    \EGYPTIAN HIEROGLYPH D028EGYPTIAN HIEROGLYPH D029EGYPTIAN HIEROGLYPH D030EGYPTIAN HIEROGLYPH D031EGYPTIAN HIEROGLYPH D031AEGYPTIAN HIEROGLYPH D032EGYPTIAN HIEROGLYPH D033EGYPTIAN HIEROGLYPH D034EGYPTIAN HIEROGLYPH D034AEGYPTIAN HIEROGLYPH D035EGYPTIAN HIER\
    \OGLYPH D036EGYPTIAN HIEROGLYPH D037EGYPTIAN HIEROGLYPH D038EGYPTIAN HIEROGLYPH D039EGYPTIAN HIEROGLYPH D040EGYPTIAN HIEROGLYPH D041EGYPTIAN HIEROGLYPH D042EGYPTIAN HIEROGLYPH D043EGYPTIAN HIEROGLYPH D044EGYPTIAN HIEROGLYPH D045EGYPTIAN HIEROGLYPH D046EGYP\
    \TIAN HIEROGLYPH D046AEGYPTIAN HIEROGLYPH D047EGYPTIAN HIEROGLYPH D048EGYPTIAN HIEROGLYPH D048AEGYPTIAN HIEROGLYPH D049EGYPTIAN HIEROGLYPH D050EGYPTIAN HIEROGLYPH D050AEGYPTIAN HIEROGLYPH D050BEGYPTIAN HIEROGLYPH D050CEGYPTIAN HIEROGLYPH D050DEGYPTIAN HIER\
    \OGLYPH D050EEGYPTIAN HIEROGLYPH D050FEGYPTIAN HIEROGLYPH D050GEGYPTIAN HIEROGLYPH D050HEGYPTIAN HIEROGLYPH D050IEGYPTIAN HIEROGLYPH D051EGYPTIAN HIEROGLYPH D052EGYPTIAN HIEROGLYPH D052AEGYPTIAN HIEROGLYPH D053EGYPTIAN HIEROGLYPH D054EGYPTIAN HIEROGLYPH D0\
    \54AEGYPTIAN HIEROGLYPH D055EGYPTIAN HIEROGLYPH D056EGYPTIAN HIEROGLYPH D057EGYPTIAN HIEROGLYPH D058EGYPTIAN HIEROGLYPH D059EGYPTIAN HIEROGLYPH D060EGYPTIAN HIEROGLYPH D061EGYPTIAN HIEROGLYPH D062EGYPTIAN HIEROGLYPH D063EGYPTIAN HIEROGLYPH D064EGYPTIAN HIE\
    \ROGLYPH D065EGYPTIAN HIEROGLYPH D066EGYPTIAN HIEROGLYPH D067EGYPTIAN HIEROGLYPH D067AEGYPTIAN HIEROGLYPH D067BEGYPTIAN HIEROGLYPH D067CEGYPTIAN HIEROGLYPH D067DEGYPTIAN HIEROGLYPH D067EEGYPTIAN HIEROGLYPH D067FEGYPTIAN HIEROGLYPH D067GEGYPTIAN HIEROGLYPH \
    \D067HEGYPTIAN HIEROGLYPH E001EGYPTIAN HIEROGLYPH E002EGYPTIAN HIEROGLYPH E003EGYPTIAN HIEROGLYPH E004EGYPTIAN HIEROGLYPH E005EGYPTIAN HIEROGLYPH E006EGYPTIAN HIEROGLYPH E007EGYPTIAN HIEROGLYPH E008EGYPTIAN HIEROGLYPH E008AEGYPTIAN HIEROGLYPH E009EGYPTIAN \
    \HIEROGLYPH E009AEGYPTIAN HIEROGLYPH E010EGYPTIAN HIEROGLYPH E011EGYPTIAN HIEROGLYPH E012EGYPTIAN HIEROGLYPH E013EGYPTIAN HIEROGLYPH E014EGYPTIAN HIEROGLYPH E015EGYPTIAN HIEROGLYPH E016EGYPTIAN HIEROGLYPH E016AEGYPTIAN HIEROGLYPH E017EGYPTIAN HIEROGLYPH E0\
    \17AEGYPTIAN HIEROGLYPH E018EGYPTIAN HIEROGLYPH E019EGYPTIAN HIEROGLYPH E020EGYPTIAN HIEROGLYPH E020AEGYPTIAN HIEROGLYPH E021EGYPTIAN HIEROGLYPH E022EGYPTIAN HIEROGLYPH E023EGYPTIAN HIEROGLYPH E024EGYPTIAN HIEROGLYPH E025EGYPTIAN HIEROGLYPH E026EGYPTIAN HI\
    \EROGLYPH E027EGYPTIAN HIEROGLYPH E028EGYPTIAN HIEROGLYPH E028AEGYPTIAN HIEROGLYPH E029EGYPTIAN HIEROGLYPH E030EGYPTIAN HIEROGLYPH E031EGYPTIAN HIEROGLYPH E032EGYPTIAN HIEROGLYPH E033EGYPTIAN HIEROGLYPH E034EGYPTIAN HIEROGLYPH E034AEGYPTIAN HIEROGLYPH E036\
    \EGYPTIAN HIEROGLYPH E037EGYPTIAN HIEROGLYPH E038EGYPTIAN HIEROGLYPH F001EGYPTIAN HIEROGLYPH F001AEGYPTIAN HIEROGLYPH F002EGYPTIAN HIEROGLYPH F003EGYPTIAN HIEROGLYPH F004EGYPTIAN HIEROGLYPH F005EGYPTIAN HIEROGLYPH F006EGYPTIAN HIEROGLYPH F007EGYPTIAN HIERO\
    \GLYPH F008EGYPTIAN HIEROGLYPH F009EGYPTIAN HIEROGLYPH F010EGYPTIAN HIEROGLYPH F011EGYPTIAN HIEROGLYPH F012EGYPTIAN HIEROGLYPH F013EGYPTIAN HIEROGLYPH F013AEGYPTIAN HIEROGLYPH F014EGYPTIAN HIEROGLYPH F015EGYPTIAN HIEROGLYPH F016EGYPTIAN HIEROGLYPH F017EGYP\
    \TIAN HIEROGLYPH F018EGYPTIAN HIEROGLYPH F019EGYPTIAN HIEROGLYPH F020EGYPTIAN HIEROGLYPH F021EGYPTIAN HIEROGLYPH F021AEGYPTIAN HIEROGLYPH F022EGYPTIAN HIEROGLYPH F023EGYPTIAN HIEROGLYPH F024EGYPTIAN HIEROGLYPH F025EGYPTIAN HIEROGLYPH F026EGYPTIAN HIEROGLYP\
    \H F027EGYPTIAN HIEROGLYPH F028EGYPTIAN HIEROGLYPH F029EGYPTIAN HIEROGLYPH F030EGYPTIAN HIEROGLYPH F031EGYPTIAN HIEROGLYPH F031AEGYPTIAN HIEROGLYPH F032EGYPTIAN HIEROGLYPH F033EGYPTIAN HIEROGLYPH F034EGYPTIAN HIEROGLYPH F035EGYPTIAN HIEROGLYPH F036EGYPTIAN\
    \ HIEROGLYPH F037EGYPTIAN HIEROGLYPH F037AEGYPTIAN HIEROGLYPH F038EGYPTIAN HIEROGLYPH F038AEGYPTIAN HIEROGLYPH F039EGYPTIAN HIEROGLYPH F040EGYPTIAN HIEROGLYPH F041EGYPTIAN HIEROGLYPH F042EGYPTIAN HIEROGLYPH F043EGYPTIAN HIEROGLYPH F044EGYPTIAN HIEROGLYPH F\
    \045EGYPTIAN HIEROGLYPH F045AEGYPTIAN HIEROGLYPH F046EGYPTIAN HIEROGLYPH F046AEGYPTIAN HIEROGLYPH F047EGYPTIAN HIEROGLYPH F047AEGYPTIAN HIEROGLYPH F048EGYPTIAN HIEROGLYPH F049EGYPTIAN HIEROGLYPH F050EGYPTIAN HIEROGLYPH F051EGYPTIAN HIEROGLYPH F051AEGYPTIAN\
    \ HIEROGLYPH F051BEGYPTIAN HIEROGLYPH F051CEGYPTIAN HIEROGLYPH F052EGYPTIAN HIEROGLYPH F053EGYPTIAN HIEROGLYPH G001EGYPTIAN HIEROGLYPH G002EGYPTIAN HIEROGLYPH G003EGYPTIAN HIEROGLYPH G004EGYPTIAN HIEROGLYPH G005EGYPTIAN HIEROGLYPH G006EGYPTIAN HIEROGLYPH G\
    \006AEGYPTIAN HIEROGLYPH G007EGYPTIAN HIEROGLYPH G007AEGYPTIAN HIEROGLYPH G007BEGYPTIAN HIEROGLYPH G008EGYPTIAN HIEROGLYPH G009EGYPTIAN HIEROGLYPH G010EGYPTIAN HIEROGLYPH G011EGYPTIAN HIEROGLYPH G011AEGYPTIAN HIEROGLYPH G012EGYPTIAN HIEROGLYPH G013EGYPTIAN\
    \ HIEROGLYPH G014EGYPTIAN HIEROGLYPH G015EGYPTIAN HIEROGLYPH G016EGYPTIAN HIEROGLYPH G017EGYPTIAN HIEROGLYPH G018EGYPTIAN HIEROGLYPH G019EGYPTIAN HIEROGLYPH G020EGYPTIAN HIEROGLYPH G020AEGYPTIAN HIEROGLYPH G021EGYPTIAN HIEROGLYPH G022EGYPTIAN HIEROGLYPH G0\
    \23EGYPTIAN HIEROGLYPH G024EGYPTIAN HIEROGLYPH G025EGYPTIAN HIEROGLYPH G026EGYPTIAN HIEROGLYPH G026AEGYPTIAN HIEROGLYPH G027EGYPTIAN HIEROGLYPH G028EGYPTIAN HIEROGLYPH G029EGYPTIAN HIEROGLYPH G030EGYPTIAN HIEROGLYPH G031EGYPTIAN HIEROGLYPH G032EGYPTIAN HIE\
    \ROGLYPH G033EGYPTIAN HIEROGLYPH G034EGYPTIAN HIEROGLYPH G035EGYPTIAN HIEROGLYPH G036EGYPTIAN HIEROGLYPH G036AEGYPTIAN HIEROGLYPH G037EGYPTIAN HIEROGLYPH G037AEGYPTIAN HIEROGLYPH G038EGYPTIAN HIEROGLYPH G039EGYPTIAN HIEROGLYPH G040EGYPTIAN HIEROGLYPH G041E\
    \GYPTIAN HIEROGLYPH G042EGYPTIAN HIEROGLYPH G043EGYPTIAN HIEROGLYPH G043AEGYPTIAN HIEROGLYPH G044EGYPTIAN HIEROGLYPH G045EGYPTIAN HIEROGLYPH G045AEGYPTIAN HIEROGLYPH G046EGYPTIAN HIEROGLYPH G047EGYPTIAN HIEROGLYPH G048EGYPTIAN HIEROGLYPH G049EGYPTIAN HIERO\
    \GLYPH G050EGYPTIAN HIEROGLYPH G051EGYPTIAN HIEROGLYPH G052EGYPTIAN HIEROGLYPH G053EGYPTIAN HIEROGLYPH G054EGYPTIAN HIEROGLYPH H001EGYPTIAN HIEROGLYPH H002EGYPTIAN HIEROGLYPH H003EGYPTIAN HIEROGLYPH H004EGYPTIAN HIEROGLYPH H005EGYPTIAN HIEROGLYPH H006EGYPT\
    \IAN HIEROGLYPH H006AEGYPTIAN HIEROGLYPH H007EGYPTIAN HIEROGLYPH H008EGYPTIAN HIEROGLYPH I001EGYPTIAN HIEROGLYPH I002EGYPTIAN HIEROGLYPH I003EGYPTIAN HIEROGLYPH I004EGYPTIAN HIEROGLYPH I005EGYPTIAN HIEROGLYPH I005AEGYPTIAN HIEROGLYPH I006EGYPTIAN HIEROGLYP\
    \H I007EGYPTIAN HIEROGLYPH I008EGYPTIAN HIEROGLYPH I009EGYPTIAN HIEROGLYPH I009AEGYPTIAN HIEROGLYPH I010EGYPTIAN HIEROGLYPH I010AEGYPTIAN HIEROGLYPH I011EGYPTIAN HIEROGLYPH I011AEGYPTIAN HIEROGLYPH I012EGYPTIAN HIEROGLYPH I013EGYPTIAN HIEROGLYPH I014EGYPTI\
    \AN HIEROGLYPH I015EGYPTIAN HIEROGLYPH K001EGYPTIAN HIEROGLYPH K002EGYPTIAN HIEROGLYPH K003EGYPTIAN HIEROGLYPH K004EGYPTIAN HIEROGLYPH K005EGYPTIAN HIEROGLYPH K006EGYPTIAN HIEROGLYPH K007EGYPTIAN HIEROGLYPH K008EGYPTIAN HIEROGLYPH L001EGYPTIAN HIEROGLYPH L\
    \002EGYPTIAN HIEROGLYPH L002AEGYPTIAN HIEROGLYPH L003EGYPTIAN HIEROGLYPH L004EGYPTIAN HIEROGLYPH L005EGYPTIAN HIEROGLYPH L006EGYPTIAN HIEROGLYPH L006AEGYPTIAN HIEROGLYPH L007EGYPTIAN HIEROGLYPH L008EGYPTIAN HIEROGLYPH M001EGYPTIAN HIEROGLYPH M001AEGYPTIAN \
    \HIEROGLYPH M001BEGYPTIAN HIEROGLYPH M002EGYPTIAN HIEROGLYPH M003EGYPTIAN HIEROGLYPH M003AEGYPTIAN HIEROGLYPH M004EGYPTIAN HIEROGLYPH M005EGYPTIAN HIEROGLYPH M006EGYPTIAN HIEROGLYPH M007EGYPTIAN HIEROGLYPH M008EGYPTIAN HIEROGLYPH M009EGYPTIAN HIEROGLYPH M0\
    \10EGYPTIAN HIEROGLYPH M010AEGYPTIAN HIEROGLYPH M011EGYPTIAN HIEROGLYPH M012EGYPTIAN HIEROGLYPH M012AEGYPTIAN HIEROGLYPH M012BEGYPTIAN HIEROGLYPH M012CEGYPTIAN HIEROGLYPH M012DEGYPTIAN HIEROGLYPH M012EEGYPTIAN HIEROGLYPH M012FEGYPTIAN HIEROGLYPH M012GEGYPT\
    \IAN HIEROGLYPH M012HEGYPTIAN HIEROGLYPH M013EGYPTIAN HIEROGLYPH M014EGYPTIAN HIEROGLYPH M015EGYPTIAN HIEROGLYPH M015AEGYPTIAN HIEROGLYPH M016EGYPTIAN HIEROGLYPH M016AEGYPTIAN HIEROGLYPH M017EGYPTIAN HIEROGLYPH M017AEGYPTIAN HIEROGLYPH M018EGYPTIAN HIEROGL\
    \YPH M019EGYPTIAN HIEROGLYPH M020EGYPTIAN HIEROGLYPH M021EGYPTIAN HIEROGLYPH M022EGYPTIAN HIEROGLYPH M022AEGYPTIAN HIEROGLYPH M023EGYPTIAN HIEROGLYPH M024EGYPTIAN HIEROGLYPH M024AEGYPTIAN HIEROGLYPH M025EGYPTIAN HIEROGLYPH M026EGYPTIAN HIEROGLYPH M027EGYPT\
    \IAN HIEROGLYPH M028EGYPTIAN HIEROGLYPH M028AEGYPTIAN HIEROGLYPH M029EGYPTIAN HIEROGLYPH M030EGYPTIAN HIEROGLYPH M031EGYPTIAN HIEROGLYPH M031AEGYPTIAN HIEROGLYPH M032EGYPTIAN HIEROGLYPH M033EGYPTIAN HIEROGLYPH M033AEGYPTIAN HIEROGLYPH M033BEGYPTIAN HIEROGL\
    \YPH M034EGYPTIAN HIEROGLYPH M035EGYPTIAN HIEROGLYPH M036EGYPTIAN HIEROGLYPH M037EGYPTIAN HIEROGLYPH M038EGYPTIAN HIEROGLYPH M039EGYPTIAN HIEROGLYPH M040EGYPTIAN HIEROGLYPH M040AEGYPTIAN HIEROGLYPH M041EGYPTIAN HIEROGLYPH M042EGYPTIAN HIEROGLYPH M043EGYPTI\
    \AN HIEROGLYPH M044EGYPTIAN HIEROGLYPH N001EGYPTIAN HIEROGLYPH N002EGYPTIAN HIEROGLYPH N003EGYPTIAN HIEROGLYPH N004EGYPTIAN HIEROGLYPH N005EGYPTIAN HIEROGLYPH N006EGYPTIAN HIEROGLYPH N007EGYPTIAN HIEROGLYPH N008EGYPTIAN HIEROGLYPH N009EGYPTIAN HIEROGLYPH N\
    \010EGYPTIAN HIEROGLYPH N011EGYPTIAN HIEROGLYPH N012EGYPTIAN HIEROGLYPH N013EGYPTIAN HIEROGLYPH N014EGYPTIAN HIEROGLYPH N015EGYPTIAN HIEROGLYPH N016EGYPTIAN HIEROGLYPH N017EGYPTIAN HIEROGLYPH N018EGYPTIAN HIEROGLYPH N018AEGYPTIAN HIEROGLYPH N018BEGYPTIAN H\
    \IEROGLYPH N019EGYPTIAN HIEROGLYPH N020EGYPTIAN HIEROGLYPH N021EGYPTIAN HIEROGLYPH N022EGYPTIAN HIEROGLYPH N023EGYPTIAN HIEROGLYPH N024EGYPTIAN HIEROGLYPH N025EGYPTIAN HIEROGLYPH N025AEGYPTIAN HIEROGLYPH N026EGYPTIAN HIEROGLYPH N027EGYPTIAN HIEROGLYPH N028\
    \EGYPTIAN HIEROGLYPH N029EGYPTIAN HIEROGLYPH N030EGYPTIAN HIEROGLYPH N031EGYPTIAN HIEROGLYPH N032EGYPTIAN HIEROGLYPH N033EGYPTIAN HIEROGLYPH N033AEGYPTIAN HIEROGLYPH N034EGYPTIAN HIEROGLYPH N034AEGYPTIAN HIEROGLYPH N035EGYPTIAN HIEROGLYPH N035AEGYPTIAN HIE\
    \ROGLYPH N036EGYPTIAN HIEROGLYPH N037EGYPTIAN HIEROGLYPH N037AEGYPTIAN HIEROGLYPH N038EGYPTIAN HIEROGLYPH N039EGYPTIAN HIEROGLYPH N040EGYPTIAN HIEROGLYPH N041EGYPTIAN HIEROGLYPH N042EGYPTIAN HIEROGLYPH NL001EGYPTIAN HIEROGLYPH NL002EGYPTIAN HIEROGLYPH NL00\
    \3EGYPTIAN HIEROGLYPH NL004EGYPTIAN HIEROGLYPH NL005EGYPTIAN HIEROGLYPH NL005AEGYPTIAN HIEROGLYPH NL006EGYPTIAN HIEROGLYPH NL007EGYPTIAN HIEROGLYPH NL008EGYPTIAN HIEROGLYPH NL009EGYPTIAN HIEROGLYPH NL010EGYPTIAN HIEROGLYPH NL011EGYPTIAN HIEROGLYPH NL012EGY\
    \PTIAN HIEROGLYPH NL013EGYPTIAN HIEROGLYPH NL014EGYPTIAN HIEROGLYPH NL015EGYPTIAN HIEROGLYPH NL016EGYPTIAN HIEROGLYPH NL017EGYPTIAN HIEROGLYPH NL017AEGYPTIAN HIEROGLYPH NL018EGYPTIAN HIEROGLYPH NL019EGYPTIAN HIEROGLYPH NL020EGYPTIAN HIEROGLYPH NU001EGYPTIA\
    \N HIEROGLYPH NU002EGYPTIAN HIEROGLYPH NU003EGYPTIAN HIEROGLYPH NU004EGYPTIAN HIEROGLYPH NU005EGYPTIAN HIEROGLYPH NU006EGYPTIAN HIEROGLYPH NU007EGYPTIAN HIEROGLYPH NU008EGYPTIAN HIEROGLYPH NU009EGYPTIAN HIEROGLYPH NU010EGYPTIAN HIEROGLYPH NU010AEGYPTIAN HI\
    \EROGLYPH NU011EGYPTIAN HIEROGLYPH NU011AEGYPTIAN HIEROGLYPH NU012EGYPTIAN HIEROGLYPH NU013EGYPTIAN HIEROGLYPH NU014EGYPTIAN HIEROGLYPH NU015EGYPTIAN HIEROGLYPH NU016EGYPTIAN HIEROGLYPH NU017EGYPTIAN HIEROGLYPH NU018EGYPTIAN HIEROGLYPH NU018AEGYPTIAN HIERO\
    \GLYPH NU019EGYPTIAN HIEROGLYPH NU020EGYPTIAN HIEROGLYPH NU021EGYPTIAN HIEROGLYPH NU022EGYPTIAN HIEROGLYPH NU022AEGYPTIAN HIEROGLYPH O001EGYPTIAN HIEROGLYPH O001AEGYPTIAN HIEROGLYPH O002EGYPTIAN HIEROGLYPH O003EGYPTIAN HIEROGLYPH O004EGYPTIAN HIEROGLYPH O0\
    \05EGYPTIAN HIEROGLYPH O005AEGYPTIAN HIEROGLYPH O006EGYPTIAN HIEROGLYPH O006AEGYPTIAN HIEROGLYPH O006BEGYPTIAN HIEROGLYPH O006CEGYPTIAN HIEROGLYPH O006DEGYPTIAN HIEROGLYPH O006EEGYPTIAN HIEROGLYPH O006FEGYPTIAN HIEROGLYPH O007EGYPTIAN HIEROGLYPH O008EGYPTI\
    \AN HIEROGLYPH O009EGYPTIAN HIEROGLYPH O010EGYPTIAN HIEROGLYPH O010AEGYPTIAN HIEROGLYPH O010BEGYPTIAN HIEROGLYPH O010CEGYPTIAN HIEROGLYPH O011EGYPTIAN HIEROGLYPH O012EGYPTIAN HIEROGLYPH O013EGYPTIAN HIEROGLYPH O014EGYPTIAN HIEROGLYPH O015EGYPTIAN HIEROGLYP\
    \H O016EGYPTIAN HIEROGLYPH O017EGYPTIAN HIEROGLYPH O018EGYPTIAN HIEROGLYPH O019EGYPTIAN HIEROGLYPH O019AEGYPTIAN HIEROGLYPH O020EGYPTIAN HIEROGLYPH O020AEGYPTIAN HIEROGLYPH O021EGYPTIAN HIEROGLYPH O022EGYPTIAN HIEROGLYPH O023EGYPTIAN HIEROGLYPH O024EGYPTIA\
    \N HIEROGLYPH O024AEGYPTIAN HIEROGLYPH O025EGYPTIAN HIEROGLYPH O025AEGYPTIAN HIEROGLYPH O026EGYPTIAN HIEROGLYPH O027EGYPTIAN HIEROGLYPH O028EGYPTIAN HIEROGLYPH O029EGYPTIAN HIEROGLYPH O029AEGYPTIAN HIEROGLYPH O030EGYPTIAN HIEROGLYPH O030AEGYPTIAN HIEROGLYP\
    \H O031EGYPTIAN HIEROGLYPH O032EGYPTIAN HIEROGLYPH O033EGYPTIAN HIEROGLYPH O033AEGYPTIAN HIEROGLYPH O034EGYPTIAN HIEROGLYPH O035EGYPTIAN HIEROGLYPH O036EGYPTIAN HIEROGLYPH O036AEGYPTIAN HIEROGLYPH O036BEGYPTIAN HIEROGLYPH O036CEGYPTIAN HIEROGLYPH O036DEGYP\
    \TIAN HIEROGLYPH O037EGYPTIAN HIEROGLYPH O038EGYPTIAN HIEROGLYPH O039EGYPTIAN HIEROGLYPH O040EGYPTIAN HIEROGLYPH O041EGYPTIAN HIEROGLYPH O042EGYPTIAN HIEROGLYPH O043EGYPTIAN HIEROGLYPH O044EGYPTIAN HIEROGLYPH O045EGYPTIAN HIEROGLYPH O046EGYPTIAN HIEROGLYPH\
    \ O047EGYPTIAN HIEROGLYPH O048EGYPTIAN HIEROGLYPH O049EGYPTIAN HIEROGLYPH O050EGYPTIAN HIEROGLYPH O050AEGYPTIAN HIEROGLYPH O050BEGYPTIAN HIEROGLYPH O051EGYPTIAN HIEROGLYPH P001EGYPTIAN HIEROGLYPH P001AEGYPTIAN HIEROGLYPH P002EGYPTIAN HIEROGLYPH P003EGYPTIA\
    \N HIEROGLYPH P003AEGYPTIAN HIEROGLYPH P004EGYPTIAN HIEROGLYPH P005EGYPTIAN HIEROGLYPH P006EGYPTIAN HIEROGLYPH P007EGYPTIAN HIEROGLYPH P008EGYPTIAN HIEROGLYPH P009EGYPTIAN HIEROGLYPH P010EGYPTIAN HIEROGLYPH P011EGYPTIAN HIEROGLYPH Q001EGYPTIAN HIEROGLYPH Q\
    \002EGYPTIAN HIEROGLYPH Q003EGYPTIAN HIEROGLYPH Q004EGYPTIAN HIEROGLYPH Q005EGYPTIAN HIEROGLYPH Q006EGYPTIAN HIEROGLYPH Q007EGYPTIAN HIEROGLYPH R001EGYPTIAN HIEROGLYPH R002EGYPTIAN HIEROGLYPH R002AEGYPTIAN HIEROGLYPH R003EGYPTIAN HIEROGLYPH R003AEGYPTIAN H\
    \IEROGLYPH R003BEGYPTIAN HIEROGLYPH R004EGYPTIAN HIEROGLYPH R005EGYPTIAN HIEROGLYPH R006EGYPTIAN HIEROGLYPH R007EGYPTIAN HIEROGLYPH R008EGYPTIAN HIEROGLYPH R009EGYPTIAN HIEROGLYPH R010EGYPTIAN HIEROGLYPH R010AEGYPTIAN HIEROGLYPH R011EGYPTIAN HIEROGLYPH R01\
    \2EGYPTIAN HIEROGLYPH R013EGYPTIAN HIEROGLYPH R014EGYPTIAN HIEROGLYPH R015EGYPTIAN HIEROGLYPH R016EGYPTIAN HIEROGLYPH R016AEGYPTIAN HIEROGLYPH R017EGYPTIAN HIEROGLYPH R018EGYPTIAN HIEROGLYPH R019EGYPTIAN HIEROGLYPH R020EGYPTIAN HIEROGLYPH R021EGYPTIAN HIER\
    \OGLYPH R022EGYPTIAN HIEROGLYPH R023EGYPTIAN HIEROGLYPH R024EGYPTIAN HIEROGLYPH R025EGYPTIAN HIEROGLYPH R026EGYPTIAN HIEROGLYPH R027EGYPTIAN HIEROGLYPH R028EGYPTIAN HIEROGLYPH R029EGYPTIAN HIEROGLYPH S001EGYPTIAN HIEROGLYPH S002EGYPTIAN HIEROGLYPH S002AEGY\
    \PTIAN HIEROGLYPH S003EGYPTIAN HIEROGLYPH S004EGYPTIAN HIEROGLYPH S005EGYPTIAN HIEROGLYPH S006EGYPTIAN HIEROGLYPH S006AEGYPTIAN HIEROGLYPH S007EGYPTIAN HIEROGLYPH S008EGYPTIAN HIEROGLYPH S009EGYPTIAN HIEROGLYPH S010EGYPTIAN HIEROGLYPH S011EGYPTIAN HIEROGLY\
    \PH S012EGYPTIAN HIEROGLYPH S013EGYPTIAN HIEROGLYPH S014EGYPTIAN HIEROGLYPH S014AEGYPTIAN HIEROGLYPH S014BEGYPTIAN HIEROGLYPH S015EGYPTIAN HIEROGLYPH S016EGYPTIAN HIEROGLYPH S017EGYPTIAN HIEROGLYPH S017AEGYPTIAN HIEROGLYPH S018EGYPTIAN HIEROGLYPH S019EGYPT\
    \IAN HIEROGLYPH S020EGYPTIAN HIEROGLYPH S021EGYPTIAN HIEROGLYPH S022EGYPTIAN HIEROGLYPH S023EGYPTIAN HIEROGLYPH S024EGYPTIAN HIEROGLYPH S025EGYPTIAN HIEROGLYPH S026EGYPTIAN HIEROGLYPH S026AEGYPTIAN HIEROGLYPH S026BEGYPTIAN HIEROGLYPH S027EGYPTIAN HIEROGLYP\
    \H S028EGYPTIAN HIEROGLYPH S029EGYPTIAN HIEROGLYPH S030EGYPTIAN HIEROGLYPH S031EGYPTIAN HIEROGLYPH S032EGYPTIAN HIEROGLYPH S033EGYPTIAN HIEROGLYPH S034EGYPTIAN HIEROGLYPH S035EGYPTIAN HIEROGLYPH S035AEGYPTIAN HIEROGLYPH S036EGYPTIAN HIEROGLYPH S037EGYPTIAN\
    \ HIEROGLYPH S038EGYPTIAN HIEROGLYPH S039EGYPTIAN HIEROGLYPH S040EGYPTIAN HIEROGLYPH S041EGYPTIAN HIEROGLYPH S042EGYPTIAN HIEROGLYPH S043EGYPTIAN HIEROGLYPH S044EGYPTIAN HIEROGLYPH S045EGYPTIAN HIEROGLYPH S046EGYPTIAN HIEROGLYPH T001EGYPTIAN HIEROGLYPH T00\
    \2EGYPTIAN HIEROGLYPH T003EGYPTIAN HIEROGLYPH T003AEGYPTIAN HIEROGLYPH T004EGYPTIAN HIEROGLYPH T005EGYPTIAN HIEROGLYPH T006EGYPTIAN HIEROGLYPH T007EGYPTIAN HIEROGLYPH T007AEGYPTIAN HIEROGLYPH T008EGYPTIAN HIEROGLYPH T008AEGYPTIAN HIEROGLYPH T009EGYPTIAN HI\
    \EROGLYPH T009AEGYPTIAN HIEROGLYPH T010EGYPTIAN HIEROGLYPH T011EGYPTIAN HIEROGLYPH T011AEGYPTIAN HIEROGLYPH T012EGYPTIAN HIEROGLYPH T013EGYPTIAN HIEROGLYPH T014EGYPTIAN HIEROGLYPH T015EGYPTIAN HIEROGLYPH T016EGYPTIAN HIEROGLYPH T016AEGYPTIAN HIEROGLYPH T01\
    \7EGYPTIAN HIEROGLYPH T018EGYPTIAN HIEROGLYPH T019EGYPTIAN HIEROGLYPH T020EGYPTIAN HIEROGLYPH T021EGYPTIAN HIEROGLYPH T022EGYPTIAN HIEROGLYPH T023EGYPTIAN HIEROGLYPH T024EGYPTIAN HIEROGLYPH T025EGYPTIAN HIEROGLYPH T026EGYPTIAN HIEROGLYPH T027EGYPTIAN HIERO\
    \GLYPH T028EGYPTIAN HIEROGLYPH T029EGYPTIAN HIEROGLYPH T030EGYPTIAN HIEROGLYPH T031EGYPTIAN HIEROGLYPH T032EGYPTIAN HIEROGLYPH T032AEGYPTIAN HIEROGLYPH T033EGYPTIAN HIEROGLYPH T033AEGYPTIAN HIEROGLYPH T034EGYPTIAN HIEROGLYPH T035EGYPTIAN HIEROGLYPH T036EGY\
    \PTIAN HIEROGLYPH U001EGYPTIAN HIEROGLYPH U002EGYPTIAN HIEROGLYPH U003EGYPTIAN HIEROGLYPH U004EGYPTIAN HIEROGLYPH U005EGYPTIAN HIEROGLYPH U006EGYPTIAN HIEROGLYPH U006AEGYPTIAN HIEROGLYPH U006BEGYPTIAN HIEROGLYPH U007EGYPTIAN HIEROGLYPH U008EGYPTIAN HIEROGL\
    \YPH U009EGYPTIAN HIEROGLYPH U010EGYPTIAN HIEROGLYPH U011EGYPTIAN HIEROGLYPH U012EGYPTIAN HIEROGLYPH U013EGYPTIAN HIEROGLYPH U014EGYPTIAN HIEROGLYPH U015EGYPTIAN HIEROGLYPH U016EGYPTIAN HIEROGLYPH U017EGYPTIAN HIEROGLYPH U018EGYPTIAN HIEROGLYPH U019EGYPTIA\
    \N HIEROGLYPH U020EGYPTIAN HIEROGLYPH U021EGYPTIAN HIEROGLYPH U022EGYPTIAN HIEROGLYPH U023EGYPTIAN HIEROGLYPH U023AEGYPTIAN HIEROGLYPH U024EGYPTIAN HIEROGLYPH U025EGYPTIAN HIEROGLYPH U026EGYPTIAN HIEROGLYPH U027EGYPTIAN HIEROGLYPH U028EGYPTIAN HIEROGLYPH U\
    \029EGYPTIAN HIEROGLYPH U029AEGYPTIAN HIEROGLYPH U030EGYPTIAN HIEROGLYPH U031EGYPTIAN HIEROGLYPH U032EGYPTIAN HIEROGLYPH U032AEGYPTIAN HIEROGLYPH U033EGYPTIAN HIEROGLYPH U034EGYPTIAN HIEROGLYPH U035EGYPTIAN HIEROGLYPH U036EGYPTIAN HIEROGLYPH U037EGYPTIAN H\
    \IEROGLYPH U038EGYPTIAN HIEROGLYPH U039EGYPTIAN HIEROGLYPH U040EGYPTIAN HIEROGLYPH U041EGYPTIAN HIEROGLYPH U042EGYPTIAN HIEROGLYPH V001EGYPTIAN HIEROGLYPH V001AEGYPTIAN HIEROGLYPH V001BEGYPTIAN HIEROGLYPH V001CEGYPTIAN HIEROGLYPH V001DEGYPTIAN HIEROGLYPH V\
    \001EEGYPTIAN HIEROGLYPH V001FEGYPTIAN HIEROGLYPH V001GEGYPTIAN HIEROGLYPH V001HEGYPTIAN HIEROGLYPH V001IEGYPTIAN HIEROGLYPH V002EGYPTIAN HIEROGLYPH V002AEGYPTIAN HIEROGLYPH V003EGYPTIAN HIEROGLYPH V004EGYPTIAN HIEROGLYPH V005EGYPTIAN HIEROGLYPH V006EGYPTI\
    \AN HIEROGLYPH V007EGYPTIAN HIEROGLYPH V007AEGYPTIAN HIEROGLYPH V007BEGYPTIAN HIEROGLYPH V008EGYPTIAN HIEROGLYPH V009EGYPTIAN HIEROGLYPH V010EGYPTIAN HIEROGLYPH V011EGYPTIAN HIEROGLYPH V011AEGYPTIAN HIEROGLYPH V011BEGYPTIAN HIEROGLYPH V011CEGYPTIAN HIEROGL\
    \YPH V012EGYPTIAN HIEROGLYPH V012AEGYPTIAN HIEROGLYPH V012BEGYPTIAN HIEROGLYPH V013EGYPTIAN HIEROGLYPH V014EGYPTIAN HIEROGLYPH V015EGYPTIAN HIEROGLYPH V016EGYPTIAN HIEROGLYPH V017EGYPTIAN HIEROGLYPH V018EGYPTIAN HIEROGLYPH V019EGYPTIAN HIEROGLYPH V020EGYPT\
    \IAN HIEROGLYPH V020AEGYPTIAN HIEROGLYPH V020BEGYPTIAN HIEROGLYPH V020CEGYPTIAN HIEROGLYPH V020DEGYPTIAN HIEROGLYPH V020EEGYPTIAN HIEROGLYPH V020FEGYPTIAN HIEROGLYPH V020GEGYPTIAN HIEROGLYPH V020HEGYPTIAN HIEROGLYPH V020IEGYPTIAN HIEROGLYPH V020JEGYPTIAN H\
    \IEROGLYPH V020KEGYPTIAN HIEROGLYPH V020LEGYPTIAN HIEROGLYPH V021EGYPTIAN HIEROGLYPH V022EGYPTIAN HIEROGLYPH V023EGYPTIAN HIEROGLYPH V023AEGYPTIAN HIEROGLYPH V024EGYPTIAN HIEROGLYPH V025EGYPTIAN HIEROGLYPH V026EGYPTIAN HIEROGLYPH V027EGYPTIAN HIEROGLYPH V0\
    \28EGYPTIAN HIEROGLYPH V028AEGYPTIAN HIEROGLYPH V029EGYPTIAN HIEROGLYPH V029AEGYPTIAN HIEROGLYPH V030EGYPTIAN HIEROGLYPH V030AEGYPTIAN HIEROGLYPH V031EGYPTIAN HIEROGLYPH V031AEGYPTIAN HIEROGLYPH V032EGYPTIAN HIEROGLYPH V033EGYPTIAN HIEROGLYPH V033AEGYPTIAN\
    \ HIEROGLYPH V034EGYPTIAN HIEROGLYPH V035EGYPTIAN HIEROGLYPH V036EGYPTIAN HIEROGLYPH V037EGYPTIAN HIEROGLYPH V037AEGYPTIAN HIEROGLYPH V038EGYPTIAN HIEROGLYPH V039EGYPTIAN HIEROGLYPH V040EGYPTIAN HIEROGLYPH V040AEGYPTIAN HIEROGLYPH W001EGYPTIAN HIEROGLYPH W\
    \002EGYPTIAN HIEROGLYPH W003EGYPTIAN HIEROGLYPH W003AEGYPTIAN HIEROGLYPH W004EGYPTIAN HIEROGLYPH W005EGYPTIAN HIEROGLYPH W006EGYPTIAN HIEROGLYPH W007EGYPTIAN HIEROGLYPH W008EGYPTIAN HIEROGLYPH W009EGYPTIAN HIEROGLYPH W009AEGYPTIAN HIEROGLYPH W010EGYPTIAN H\
    \IEROGLYPH W010AEGYPTIAN HIEROGLYPH W011EGYPTIAN HIEROGLYPH W012EGYPTIAN HIEROGLYPH W013EGYPTIAN HIEROGLYPH W014EGYPTIAN HIEROGLYPH W014AEGYPTIAN HIEROGLYPH W015EGYPTIAN HIEROGLYPH W016EGYPTIAN HIEROGLYPH W017EGYPTIAN HIEROGLYPH W017AEGYPTIAN HIEROGLYPH W0\
    \18EGYPTIAN HIEROGLYPH W018AEGYPTIAN HIEROGLYPH W019EGYPTIAN HIEROGLYPH W020EGYPTIAN HIEROGLYPH W021EGYPTIAN HIEROGLYPH W022EGYPTIAN HIEROGLYPH W023EGYPTIAN HIEROGLYPH W024EGYPTIAN HIEROGLYPH W024AEGYPTIAN HIEROGLYPH W025EGYPTIAN HIEROGLYPH X001EGYPTIAN HI\
    \EROGLYPH X002EGYPTIAN HIEROGLYPH X003EGYPTIAN HIEROGLYPH X004EGYPTIAN HIEROGLYPH X004AEGYPTIAN HIEROGLYPH X004BEGYPTIAN HIEROGLYPH X005EGYPTIAN HIEROGLYPH X006EGYPTIAN HIEROGLYPH X006AEGYPTIAN HIEROGLYPH X007EGYPTIAN HIEROGLYPH X008EGYPTIAN HIEROGLYPH X00\
    \8AEGYPTIAN HIEROGLYPH Y001EGYPTIAN HIEROGLYPH Y001AEGYPTIAN HIEROGLYPH Y002EGYPTIAN HIEROGLYPH Y003EGYPTIAN HIEROGLYPH Y004EGYPTIAN HIEROGLYPH Y005EGYPTIAN HIEROGLYPH Y006EGYPTIAN HIEROGLYPH Y007EGYPTIAN HIEROGLYPH Y008EGYPTIAN HIEROGLYPH Z001EGYPTIAN HIE\
    \ROGLYPH Z002EGYPTIAN HIEROGLYPH Z002AEGYPTIAN HIEROGLYPH Z002BEGYPTIAN HIEROGLYPH Z002CEGYPTIAN HIEROGLYPH Z002DEGYPTIAN HIEROGLYPH Z003EGYPTIAN HIEROGLYPH Z003AEGYPTIAN HIEROGLYPH Z003BEGYPTIAN HIEROGLYPH Z004EGYPTIAN HIEROGLYPH Z004AEGYPTIAN HIEROGLYPH \
    \Z005EGYPTIAN HIEROGLYPH Z005AEGYPTIAN HIEROGLYPH Z006EGYPTIAN HIEROGLYPH Z007EGYPTIAN HIEROGLYPH Z008EGYPTIAN HIEROGLYPH Z009EGYPTIAN HIEROGLYPH Z010EGYPTIAN HIEROGLYPH Z011EGYPTIAN HIEROGLYPH Z012EGYPTIAN HIEROGLYPH Z013EGYPTIAN HIEROGLYPH Z014EGYPTIAN H\
    \IEROGLYPH Z015EGYPTIAN HIEROGLYPH Z015AEGYPTIAN HIEROGLYPH Z015BEGYPTIAN HIEROGLYPH Z015CEGYPTIAN HIEROGLYPH Z015DEGYPTIAN HIEROGLYPH Z015EEGYPTIAN HIEROGLYPH Z015FEGYPTIAN HIEROGLYPH Z015GEGYPTIAN HIEROGLYPH Z015HEGYPTIAN HIEROGLYPH Z015IEGYPTIAN HIEROGL\
    \YPH Z016EGYPTIAN HIEROGLYPH Z016AEGYPTIAN HIEROGLYPH Z016BEGYPTIAN HIEROGLYPH Z016CEGYPTIAN HIEROGLYPH Z016DEGYPTIAN HIEROGLYPH Z016EEGYPTIAN HIEROGLYPH Z016FEGYPTIAN HIEROGLYPH Z016GEGYPTIAN HIEROGLYPH Z016HEGYPTIAN HIEROGLYPH AA001EGYPTIAN HIEROGLYPH AA\
    \002EGYPTIAN HIEROGLYPH AA003EGYPTIAN HIEROGLYPH AA004EGYPTIAN HIEROGLYPH AA005EGYPTIAN HIEROGLYPH AA006EGYPTIAN HIEROGLYPH AA007EGYPTIAN HIEROGLYPH AA007AEGYPTIAN HIEROGLYPH AA007BEGYPTIAN HIEROGLYPH AA008EGYPTIAN HIEROGLYPH AA009EGYPTIAN HIEROGLYPH AA010\
    \EGYPTIAN HIEROGLYPH AA011EGYPTIAN HIEROGLYPH AA012EGYPTIAN HIEROGLYPH AA013EGYPTIAN HIEROGLYPH AA014EGYPTIAN HIEROGLYPH AA015EGYPTIAN HIEROGLYPH AA016EGYPTIAN HIEROGLYPH AA017EGYPTIAN HIEROGLYPH AA018EGYPTIAN HIEROGLYPH AA019EGYPTIAN HIEROGLYPH AA020EGYPT\
    \IAN HIEROGLYPH AA021EGYPTIAN HIEROGLYPH AA022EGYPTIAN HIEROGLYPH AA023EGYPTIAN HIEROGLYPH AA024EGYPTIAN HIEROGLYPH AA025EGYPTIAN HIEROGLYPH AA026EGYPTIAN HIEROGLYPH AA027EGYPTIAN HIEROGLYPH AA028EGYPTIAN HIEROGLYPH AA029EGYPTIAN HIEROGLYPH AA030EGYPTIAN H\
    \IEROGLYPH AA031EGYPTIAN HIEROGLYPH AA032EGYPTIAN HIEROGLYPH V011DEGYPTIAN HIEROGLYPH VERTICAL JOINEREGYPTIAN HIEROGLYPH HORIZONTAL JOINEREGYPTIAN HIEROGLYPH INSERT AT TOP STARTEGYPTIAN HIEROGLYPH INSERT AT BOTTOM STARTEGYPTIAN HIEROGLYPH INSERT AT TOP END\
    \EGYPTIAN HIEROGLYPH INSERT AT BOTTOM ENDEGYPTIAN HIEROGLYPH OVERLAY MIDDLEEGYPTIAN HIEROGLYPH BEGIN SEGMENTEGYPTIAN HIEROGLYPH END SEGMENTEGYPTIAN HIEROGLYPH INSERT AT MIDDLEEGYPTIAN HIEROGLYPH INSERT AT TOPEGYPTIAN HIEROGLYPH INSERT AT BOTTOMEGYPTIAN HIE\
    \ROGLYPH BEGIN ENCLOSUREEGYPTIAN HIEROGLYPH END ENCLOSUREEGYPTIAN HIEROGLYPH BEGIN WALLED ENCLOSUREEGYPTIAN HIEROGLYPH END WALLED ENCLOSUREEGYPTIAN HIEROGLYPH MIRROR HORIZONTALLYEGYPTIAN HIEROGLYPH FULL BLANKEGYPTIAN HIEROGLYPH HALF BLANKEGYPTIAN HIEROGLYP\
    \H LOST SIGNEGYPTIAN HIEROGLYPH HALF LOST SIGNEGYPTIAN HIEROGLYPH TALL LOST SIGNEGYPTIAN HIEROGLYPH WIDE LOST SIGNEGYPTIAN HIEROGLYPH MODIFIER DAMAGED AT TOP STARTEGYPTIAN HIEROGLYPH MODIFIER DAMAGED AT BOTTOM STARTEGYPTIAN HIEROGLYPH MODIFIER DAMAGED AT S\
    \TARTEGYPTIAN HIEROGLYPH MODIFIER DAMAGED AT TOP ENDEGYPTIAN HIEROGLYPH MODIFIER DAMAGED AT TOPEGYPTIAN HIEROGLYPH MODIFIER DAMAGED AT BOTTOM START AND TOP ENDEGYPTIAN HIEROGLYPH MODIFIER DAMAGED AT START AND TOPEGYPTIAN HIEROGLYPH MODIFIER DAMAGED AT BOTT\
    \OM ENDEGYPTIAN HIEROGLYPH MODIFIER DAMAGED AT TOP START AND BOTTOM ENDEGYPTIAN HIEROGLYPH MODIFIER DAMAGED AT BOTTOMEGYPTIAN HIEROGLYPH MODIFIER DAMAGED AT START AND BOTTOMEGYPTIAN HIEROGLYPH MODIFIER DAMAGED AT ENDEGYPTIAN HIEROGLYPH MODIFIER DAMAGED AT \
    \TOP AND ENDEGYPTIAN HIEROGLYPH MODIFIER DAMAGED AT BOTTOM AND ENDEGYPTIAN HIEROGLYPH MODIFIER DAMAGEDANATOLIAN HIEROGLYPH A001ANATOLIAN HIEROGLYPH A002ANATOLIAN HIEROGLYPH A003ANATOLIAN HIEROGLYPH A004ANATOLIAN HIEROGLYPH A005ANATOLIAN HIEROGLYPH A006ANAT\
    \OLIAN HIEROGLYPH A007ANATOLIAN HIEROGLYPH A008ANATOLIAN HIEROGLYPH A009ANATOLIAN HIEROGLYPH A010ANATOLIAN HIEROGLYPH A010AANATOLIAN HIEROGLYPH A011ANATOLIAN HIEROGLYPH A012ANATOLIAN HIEROGLYPH A013ANATOLIAN HIEROGLYPH A014ANATOLIAN HIEROGLYPH A015ANATOLIA\
    \N HIEROGLYPH A016ANATOLIAN HIEROGLYPH A017ANATOLIAN HIEROGLYPH A018ANATOLIAN HIEROGLYPH A019ANATOLIAN HIEROGLYPH A020ANATOLIAN HIEROGLYPH A021ANATOLIAN HIEROGLYPH A022ANATOLIAN HIEROGLYPH A023ANATOLIAN HIEROGLYPH A024ANATOLIAN HIEROGLYPH A025ANATOLIAN HIE\
    \ROGLYPH A026ANATOLIAN HIEROGLYPH A026AANATOLIAN HIEROGLYPH A027ANATOLIAN HIEROGLYPH A028ANATOLIAN HIEROGLYPH A029ANATOLIAN HIEROGLYPH A030ANATOLIAN HIEROGLYPH A031ANATOLIAN HIEROGLYPH A032ANATOLIAN HIEROGLYPH A033ANATOLIAN HIEROGLYPH A034ANATOLIAN HIEROGL\
    \YPH A035ANATOLIAN HIEROGLYPH A036ANATOLIAN HIEROGLYPH A037ANATOLIAN HIEROGLYPH A038ANATOLIAN HIEROGLYPH A039ANATOLIAN HIEROGLYPH A039AANATOLIAN HIEROGLYPH A040ANATOLIAN HIEROGLYPH A041ANATOLIAN HIEROGLYPH A041AANATOLIAN HIEROGLYPH A042ANATOLIAN HIEROGLYPH\
    \ A043ANATOLIAN HIEROGLYPH A044ANATOLIAN HIEROGLYPH A045ANATOLIAN HIEROGLYPH A045AANATOLIAN HIEROGLYPH A046ANATOLIAN HIEROGLYPH A046AANATOLIAN HIEROGLYPH A046BANATOLIAN HIEROGLYPH A047ANATOLIAN HIEROGLYPH A048ANATOLIAN HIEROGLYPH A049ANATOLIAN HIEROGLYPH A\
    \050ANATOLIAN HIEROGLYPH A051ANATOLIAN HIEROGLYPH A052ANATOLIAN HIEROGLYPH A053ANATOLIAN HIEROGLYPH A054ANATOLIAN HIEROGLYPH A055ANATOLIAN HIEROGLYPH A056ANATOLIAN HIEROGLYPH A057ANATOLIAN HIEROGLYPH A058ANATOLIAN HIEROGLYPH A059ANATOLIAN HIEROGLYPH A060AN\
    \ATOLIAN HIEROGLYPH A061ANATOLIAN HIEROGLYPH A062ANATOLIAN HIEROGLYPH A063ANATOLIAN HIEROGLYPH A064ANATOLIAN HIEROGLYPH A065ANATOLIAN HIEROGLYPH A066ANATOLIAN HIEROGLYPH A066AANATOLIAN HIEROGLYPH A066BANATOLIAN HIEROGLYPH A066CANATOLIAN HIEROGLYPH A067ANAT\
    \OLIAN HIEROGLYPH A068ANATOLIAN HIEROGLYPH A069ANATOLIAN HIEROGLYPH A070ANATOLIAN HIEROGLYPH A071ANATOLIAN HIEROGLYPH A072ANATOLIAN HIEROGLYPH A073ANATOLIAN HIEROGLYPH A074ANATOLIAN HIEROGLYPH A075ANATOLIAN HIEROGLYPH A076ANATOLIAN HIEROGLYPH A077ANATOLIAN\
    \ HIEROGLYPH A078ANATOLIAN HIEROGLYPH A079ANATOLIAN HIEROGLYPH A080ANATOLIAN HIEROGLYPH A081ANATOLIAN HIEROGLYPH A082ANATOLIAN HIEROGLYPH A083ANATOLIAN HIEROGLYPH A084ANATOLIAN HIEROGLYPH A085ANATOLIAN HIEROGLYPH A086ANATOLIAN HIEROGLYPH A087ANATOLIAN HIER\
    \OGLYPH A088ANATOLIAN HIEROGLYPH A089ANATOLIAN HIEROGLYPH A090ANATOLIAN HIEROGLYPH A091ANATOLIAN HIEROGLYPH A092ANATOLIAN HIEROGLYPH A093ANATOLIAN HIEROGLYPH A094ANATOLIAN HIEROGLYPH A095ANATOLIAN HIEROGLYPH A096ANATOLIAN HIEROGLYPH A097ANATOLIAN HIEROGLYP\
    \H A097AANATOLIAN HIEROGLYPH A098ANATOLIAN HIEROGLYPH A098AANATOLIAN HIEROGLYPH A099ANATOLIAN HIEROGLYPH A100ANATOLIAN HIEROGLYPH A100AANATOLIAN HIEROGLYPH A101ANATOLIAN HIEROGLYPH A101AANATOLIAN HIEROGLYPH A102ANATOLIAN HIEROGLYPH A102AANATOLIAN HIEROGLYP\
    \H A103ANATOLIAN HIEROGLYPH A104ANATOLIAN HIEROGLYPH A104AANATOLIAN HIEROGLYPH A104BANATOLIAN HIEROGLYPH A104CANATOLIAN HIEROGLYPH A105ANATOLIAN HIEROGLYPH A105AANATOLIAN HIEROGLYPH A105BANATOLIAN HIEROGLYPH A106ANATOLIAN HIEROGLYPH A107ANATOLIAN HIEROGLYP\
    \H A107AANATOLIAN HIEROGLYPH A107BANATOLIAN HIEROGLYPH A107CANATOLIAN HIEROGLYPH A108ANATOLIAN HIEROGLYPH A109ANATOLIAN HIEROGLYPH A110ANATOLIAN HIEROGLYPH A110AANATOLIAN HIEROGLYPH A110BANATOLIAN HIEROGLYPH A111ANATOLIAN HIEROGLYPH A112ANATOLIAN HIEROGLYP\
    \H A113ANATOLIAN HIEROGLYPH A114ANATOLIAN HIEROGLYPH A115ANATOLIAN HIEROGLYPH A115AANATOLIAN HIEROGLYPH A116ANATOLIAN HIEROGLYPH A117ANATOLIAN HIEROGLYPH A118ANATOLIAN HIEROGLYPH A119ANATOLIAN HIEROGLYPH A120ANATOLIAN HIEROGLYPH A121ANATOLIAN HIEROGLYPH A1\
    \22ANATOLIAN HIEROGLYPH A123ANATOLIAN HIEROGLYPH A124ANATOLIAN HIEROGLYPH A125ANATOLIAN HIEROGLYPH A125AANATOLIAN HIEROGLYPH A126ANATOLIAN HIEROGLYPH A127ANATOLIAN HIEROGLYPH A128ANATOLIAN HIEROGLYPH A129ANATOLIAN HIEROGLYPH A130ANATOLIAN HIEROGLYPH A131AN\
    \ATOLIAN HIEROGLYPH A132ANATOLIAN HIEROGLYPH A133ANATOLIAN HIEROGLYPH A134ANATOLIAN HIEROGLYPH A135ANATOLIAN HIEROGLYPH A135AANATOLIAN HIEROGLYPH A136ANATOLIAN HIEROGLYPH A137ANATOLIAN HIEROGLYPH A138ANATOLIAN HIEROGLYPH A139ANATOLIAN HIEROGLYPH A140ANATOL\
    \IAN HIEROGLYPH A141ANATOLIAN HIEROGLYPH A142ANATOLIAN HIEROGLYPH A143ANATOLIAN HIEROGLYPH A144ANATOLIAN HIEROGLYPH A145ANATOLIAN HIEROGLYPH A146ANATOLIAN HIEROGLYPH A147ANATOLIAN HIEROGLYPH A148ANATOLIAN HIEROGLYPH A149ANATOLIAN HIEROGLYPH A150ANATOLIAN H\
    \IEROGLYPH A151ANATOLIAN HIEROGLYPH A152ANATOLIAN HIEROGLYPH A153ANATOLIAN HIEROGLYPH A154ANATOLIAN HIEROGLYPH A155ANATOLIAN HIEROGLYPH A156ANATOLIAN HIEROGLYPH A157ANATOLIAN HIEROGLYPH A158ANATOLIAN HIEROGLYPH A159ANATOLIAN HIEROGLYPH A160ANATOLIAN HIEROG\
    \LYPH A161ANATOLIAN HIEROGLYPH A162ANATOLIAN HIEROGLYPH A163ANATOLIAN HIEROGLYPH A164ANATOLIAN HIEROGLYPH A165ANATOLIAN HIEROGLYPH A166ANATOLIAN HIEROGLYPH A167ANATOLIAN HIEROGLYPH A168ANATOLIAN HIEROGLYPH A169ANATOLIAN HIEROGLYPH A170ANATOLIAN HIEROGLYPH \
    \A171ANATOLIAN HIEROGLYPH A172ANATOLIAN HIEROGLYPH A173ANATOLIAN HIEROGLYPH A174ANATOLIAN HIEROGLYPH A175ANATOLIAN HIEROGLYPH A176ANATOLIAN HIEROGLYPH A177ANATOLIAN HIEROGLYPH A178ANATOLIAN HIEROGLYPH A179ANATOLIAN HIEROGLYPH A180ANATOLIAN HIEROGLYPH A181A\
    \NATOLIAN HIEROGLYPH A182ANATOLIAN HIEROGLYPH A183ANATOLIAN HIEROGLYPH A184ANATOLIAN HIEROGLYPH A185ANATOLIAN HIEROGLYPH A186ANATOLIAN HIEROGLYPH A187ANATOLIAN HIEROGLYPH A188ANATOLIAN HIEROGLYPH A189ANATOLIAN HIEROGLYPH A190ANATOLIAN HIEROGLYPH A191ANATOL\
    \IAN HIEROGLYPH A192ANATOLIAN HIEROGLYPH A193ANATOLIAN HIEROGLYPH A194ANATOLIAN HIEROGLYPH A195ANATOLIAN HIEROGLYPH A196ANATOLIAN HIEROGLYPH A197ANATOLIAN HIEROGLYPH A198ANATOLIAN HIEROGLYPH A199ANATOLIAN HIEROGLYPH A200ANATOLIAN HIEROGLYPH A201ANATOLIAN H\
    \IEROGLYPH A202ANATOLIAN HIEROGLYPH A202AANATOLIAN HIEROGLYPH A202BANATOLIAN HIEROGLYPH A203ANATOLIAN HIEROGLYPH A204ANATOLIAN HIEROGLYPH A205ANATOLIAN HIEROGLYPH A206ANATOLIAN HIEROGLYPH A207ANATOLIAN HIEROGLYPH A207AANATOLIAN HIEROGLYPH A208ANATOLIAN HIE\
    \ROGLYPH A209ANATOLIAN HIEROGLYPH A209AANATOLIAN HIEROGLYPH A210ANATOLIAN HIEROGLYPH A211ANATOLIAN HIEROGLYPH A212ANATOLIAN HIEROGLYPH A213ANATOLIAN HIEROGLYPH A214ANATOLIAN HIEROGLYPH A215ANATOLIAN HIEROGLYPH A215AANATOLIAN HIEROGLYPH A216ANATOLIAN HIEROG\
    \LYPH A216AANATOLIAN HIEROGLYPH A217ANATOLIAN HIEROGLYPH A218ANATOLIAN HIEROGLYPH A219ANATOLIAN HIEROGLYPH A220ANATOLIAN HIEROGLYPH A221ANATOLIAN HIEROGLYPH A222ANATOLIAN HIEROGLYPH A223ANATOLIAN HIEROGLYPH A224ANATOLIAN HIEROGLYPH A225ANATOLIAN HIEROGLYPH\
    \ A226ANATOLIAN HIEROGLYPH A227ANATOLIAN HIEROGLYPH A227AANATOLIAN HIEROGLYPH A228ANATOLIAN HIEROGLYPH A229ANATOLIAN HIEROGLYPH A230ANATOLIAN HIEROGLYPH A231ANATOLIAN HIEROGLYPH A232ANATOLIAN HIEROGLYPH A233ANATOLIAN HIEROGLYPH A234ANATOLIAN HIEROGLYPH A23\
    \5ANATOLIAN HIEROGLYPH A236ANATOLIAN HIEROGLYPH A237ANATOLIAN HIEROGLYPH A238ANATOLIAN HIEROGLYPH A239ANATOLIAN HIEROGLYPH A240ANATOLIAN HIEROGLYPH A241ANATOLIAN HIEROGLYPH A242ANATOLIAN HIEROGLYPH A243ANATOLIAN HIEROGLYPH A244ANATOLIAN HIEROGLYPH A245ANAT\
    \OLIAN HIEROGLYPH A246ANATOLIAN HIEROGLYPH A247ANATOLIAN HIEROGLYPH A248ANATOLIAN HIEROGLYPH A249ANATOLIAN HIEROGLYPH A250ANATOLIAN HIEROGLYPH A251ANATOLIAN HIEROGLYPH A252ANATOLIAN HIEROGLYPH A253ANATOLIAN HIEROGLYPH A254ANATOLIAN HIEROGLYPH A255ANATOLIAN\
    \ HIEROGLYPH A256ANATOLIAN HIEROGLYPH A257ANATOLIAN HIEROGLYPH A258ANATOLIAN HIEROGLYPH A259ANATOLIAN HIEROGLYPH A260ANATOLIAN HIEROGLYPH A261ANATOLIAN HIEROGLYPH A262ANATOLIAN HIEROGLYPH A263ANATOLIAN HIEROGLYPH A264ANATOLIAN HIEROGLYPH A265ANATOLIAN HIER\
    \OGLYPH A266ANATOLIAN HIEROGLYPH A267ANATOLIAN HIEROGLYPH A267AANATOLIAN HIEROGLYPH A268ANATOLIAN HIEROGLYPH A269ANATOLIAN HIEROGLYPH A270ANATOLIAN HIEROGLYPH A271ANATOLIAN HIEROGLYPH A272ANATOLIAN HIEROGLYPH A273ANATOLIAN HIEROGLYPH A274ANATOLIAN HIEROGLY\
    \PH A275ANATOLIAN HIEROGLYPH A276ANATOLIAN HIEROGLYPH A277ANATOLIAN HIEROGLYPH A278ANATOLIAN HIEROGLYPH A279ANATOLIAN HIEROGLYPH A280ANATOLIAN HIEROGLYPH A281ANATOLIAN HIEROGLYPH A282ANATOLIAN HIEROGLYPH A283ANATOLIAN HIEROGLYPH A284ANATOLIAN HIEROGLYPH A2\
    \85ANATOLIAN HIEROGLYPH A286ANATOLIAN HIEROGLYPH A287ANATOLIAN HIEROGLYPH A288ANATOLIAN HIEROGLYPH A289ANATOLIAN HIEROGLYPH A289AANATOLIAN HIEROGLYPH A290ANATOLIAN HIEROGLYPH A291ANATOLIAN HIEROGLYPH A292ANATOLIAN HIEROGLYPH A293ANATOLIAN HIEROGLYPH A294AN\
    \ATOLIAN HIEROGLYPH A294AANATOLIAN HIEROGLYPH A295ANATOLIAN HIEROGLYPH A296ANATOLIAN HIEROGLYPH A297ANATOLIAN HIEROGLYPH A298ANATOLIAN HIEROGLYPH A299ANATOLIAN HIEROGLYPH A299AANATOLIAN HIEROGLYPH A300ANATOLIAN HIEROGLYPH A301ANATOLIAN HIEROGLYPH A302ANATO\
    \LIAN HIEROGLYPH A303ANATOLIAN HIEROGLYPH A304ANATOLIAN HIEROGLYPH A305ANATOLIAN HIEROGLYPH A306ANATOLIAN HIEROGLYPH A307ANATOLIAN HIEROGLYPH A308ANATOLIAN HIEROGLYPH A309ANATOLIAN HIEROGLYPH A309AANATOLIAN HIEROGLYPH A310ANATOLIAN HIEROGLYPH A311ANATOLIAN\
    \ HIEROGLYPH A312ANATOLIAN HIEROGLYPH A313ANATOLIAN HIEROGLYPH A314ANATOLIAN HIEROGLYPH A315ANATOLIAN HIEROGLYPH A316ANATOLIAN HIEROGLYPH A317ANATOLIAN HIEROGLYPH A318ANATOLIAN HIEROGLYPH A319ANATOLIAN HIEROGLYPH A320ANATOLIAN HIEROGLYPH A321ANATOLIAN HIER\
    \OGLYPH A322ANATOLIAN HIEROGLYPH A323ANATOLIAN HIEROGLYPH A324ANATOLIAN HIEROGLYPH A325ANATOLIAN HIEROGLYPH A326ANATOLIAN HIEROGLYPH A327ANATOLIAN HIEROGLYPH A328ANATOLIAN HIEROGLYPH A329ANATOLIAN HIEROGLYPH A329AANATOLIAN HIEROGLYPH A330ANATOLIAN HIEROGLY\
    \PH A331ANATOLIAN HIEROGLYPH A332AANATOLIAN HIEROGLYPH A332BANATOLIAN HIEROGLYPH A332CANATOLIAN HIEROGLYPH A333ANATOLIAN HIEROGLYPH A334ANATOLIAN HIEROGLYPH A335ANATOLIAN HIEROGLYPH A336ANATOLIAN HIEROGLYPH A336AANATOLIAN HIEROGLYPH A336BANATOLIAN HIEROGLY\
    \PH A336CANATOLIAN HIEROGLYPH A337ANATOLIAN HIEROGLYPH A338ANATOLIAN HIEROGLYPH A339ANATOLIAN HIEROGLYPH A340ANATOLIAN HIEROGLYPH A341ANATOLIAN HIEROGLYPH A342ANATOLIAN HIEROGLYPH A343ANATOLIAN HIEROGLYPH A344ANATOLIAN HIEROGLYPH A345ANATOLIAN HIEROGLYPH A\
    \346ANATOLIAN HIEROGLYPH A347ANATOLIAN HIEROGLYPH A348ANATOLIAN HIEROGLYPH A349ANATOLIAN HIEROGLYPH A350ANATOLIAN HIEROGLYPH A351ANATOLIAN HIEROGLYPH A352ANATOLIAN HIEROGLYPH A353ANATOLIAN HIEROGLYPH A354ANATOLIAN HIEROGLYPH A355ANATOLIAN HIEROGLYPH A356AN\
    \ATOLIAN HIEROGLYPH A357ANATOLIAN HIEROGLYPH A358ANATOLIAN HIEROGLYPH A359ANATOLIAN HIEROGLYPH A359AANATOLIAN HIEROGLYPH A360ANATOLIAN HIEROGLYPH A361ANATOLIAN HIEROGLYPH A362ANATOLIAN HIEROGLYPH A363ANATOLIAN HIEROGLYPH A364ANATOLIAN HIEROGLYPH A364AANATO\
    \LIAN HIEROGLYPH A365ANATOLIAN HIEROGLYPH A366ANATOLIAN HIEROGLYPH A367ANATOLIAN HIEROGLYPH A368ANATOLIAN HIEROGLYPH A368AANATOLIAN HIEROGLYPH A369ANATOLIAN HIEROGLYPH A370ANATOLIAN HIEROGLYPH A371ANATOLIAN HIEROGLYPH A371AANATOLIAN HIEROGLYPH A372ANATOLIA\
    \N HIEROGLYPH A373ANATOLIAN HIEROGLYPH A374ANATOLIAN HIEROGLYPH A375ANATOLIAN HIEROGLYPH A376ANATOLIAN HIEROGLYPH A377ANATOLIAN HIEROGLYPH A378ANATOLIAN HIEROGLYPH A379ANATOLIAN HIEROGLYPH A380ANATOLIAN HIEROGLYPH A381ANATOLIAN HIEROGLYPH A381AANATOLIAN HI\
    \EROGLYPH A382ANATOLIAN HIEROGLYPH A383 RA OR RIANATOLIAN HIEROGLYPH A383AANATOLIAN HIEROGLYPH A384ANATOLIAN HIEROGLYPH A385ANATOLIAN HIEROGLYPH A386ANATOLIAN HIEROGLYPH A386AANATOLIAN HIEROGLYPH A387ANATOLIAN HIEROGLYPH A388ANATOLIAN HIEROGLYPH A389ANATOL\
    \IAN HIEROGLYPH A390ANATOLIAN HIEROGLYPH A391ANATOLIAN HIEROGLYPH A392ANATOLIAN HIEROGLYPH A393 EIGHTANATOLIAN HIEROGLYPH A394ANATOLIAN HIEROGLYPH A395ANATOLIAN HIEROGLYPH A396ANATOLIAN HIEROGLYPH A397ANATOLIAN HIEROGLYPH A398ANATOLIAN HIEROGLYPH A399ANATO\
    \LIAN HIEROGLYPH A400ANATOLIAN HIEROGLYPH A401ANATOLIAN HIEROGLYPH A402ANATOLIAN HIEROGLYPH A403ANATOLIAN HIEROGLYPH A404ANATOLIAN HIEROGLYPH A405ANATOLIAN HIEROGLYPH A406ANATOLIAN HIEROGLYPH A407ANATOLIAN HIEROGLYPH A408ANATOLIAN HIEROGLYPH A409ANATOLIAN \
    \HIEROGLYPH A410 BEGIN LOGOGRAM MARKANATOLIAN HIEROGLYPH A410A END LOGOGRAM MARKANATOLIAN HIEROGLYPH A411ANATOLIAN HIEROGLYPH A412ANATOLIAN HIEROGLYPH A413ANATOLIAN HIEROGLYPH A414ANATOLIAN HIEROGLYPH A415ANATOLIAN HIEROGLYPH A416ANATOLIAN HIEROGLYPH A417A\
    \NATOLIAN HIEROGLYPH A418ANATOLIAN HIEROGLYPH A419ANATOLIAN HIEROGLYPH A420ANATOLIAN HIEROGLYPH A421ANATOLIAN HIEROGLYPH A422ANATOLIAN HIEROGLYPH A423ANATOLIAN HIEROGLYPH A424ANATOLIAN HIEROGLYPH A425ANATOLIAN HIEROGLYPH A426ANATOLIAN HIEROGLYPH A427ANATOL\
    \IAN HIEROGLYPH A428ANATOLIAN HIEROGLYPH A429ANATOLIAN HIEROGLYPH A430ANATOLIAN HIEROGLYPH A431ANATOLIAN HIEROGLYPH A432ANATOLIAN HIEROGLYPH A433ANATOLIAN HIEROGLYPH A434ANATOLIAN HIEROGLYPH A435ANATOLIAN HIEROGLYPH A436ANATOLIAN HIEROGLYPH A437ANATOLIAN H\
    \IEROGLYPH A438ANATOLIAN HIEROGLYPH A439ANATOLIAN HIEROGLYPH A440ANATOLIAN HIEROGLYPH A441ANATOLIAN HIEROGLYPH A442ANATOLIAN HIEROGLYPH A443ANATOLIAN HIEROGLYPH A444ANATOLIAN HIEROGLYPH A445ANATOLIAN HIEROGLYPH A446ANATOLIAN HIEROGLYPH A447ANATOLIAN HIEROG\
    \LYPH A448ANATOLIAN HIEROGLYPH A449ANATOLIAN HIEROGLYPH A450ANATOLIAN HIEROGLYPH A450AANATOLIAN HIEROGLYPH A451ANATOLIAN HIEROGLYPH A452ANATOLIAN HIEROGLYPH A453ANATOLIAN HIEROGLYPH A454ANATOLIAN HIEROGLYPH A455ANATOLIAN HIEROGLYPH A456ANATOLIAN HIEROGLYPH\
    \ A457ANATOLIAN HIEROGLYPH A457AANATOLIAN HIEROGLYPH A458ANATOLIAN HIEROGLYPH A459ANATOLIAN HIEROGLYPH A460ANATOLIAN HIEROGLYPH A461ANATOLIAN HIEROGLYPH A462ANATOLIAN HIEROGLYPH A463ANATOLIAN HIEROGLYPH A464ANATOLIAN HIEROGLYPH A465ANATOLIAN HIEROGLYPH A46\
    \6ANATOLIAN HIEROGLYPH A467ANATOLIAN HIEROGLYPH A468ANATOLIAN HIEROGLYPH A469ANATOLIAN HIEROGLYPH A470ANATOLIAN HIEROGLYPH A471ANATOLIAN HIEROGLYPH A472ANATOLIAN HIEROGLYPH A473ANATOLIAN HIEROGLYPH A474ANATOLIAN HIEROGLYPH A475ANATOLIAN HIEROGLYPH A476ANAT\
    \OLIAN HIEROGLYPH A477ANATOLIAN HIEROGLYPH A478ANATOLIAN HIEROGLYPH A479ANATOLIAN HIEROGLYPH A480ANATOLIAN HIEROGLYPH A481ANATOLIAN HIEROGLYPH A482ANATOLIAN HIEROGLYPH A483ANATOLIAN HIEROGLYPH A484ANATOLIAN HIEROGLYPH A485ANATOLIAN HIEROGLYPH A486ANATOLIAN\
    \ HIEROGLYPH A487ANATOLIAN HIEROGLYPH A488ANATOLIAN HIEROGLYPH A489ANATOLIAN HIEROGLYPH A490ANATOLIAN HIEROGLYPH A491ANATOLIAN HIEROGLYPH A492ANATOLIAN HIEROGLYPH A493ANATOLIAN HIEROGLYPH A494ANATOLIAN HIEROGLYPH A495ANATOLIAN HIEROGLYPH A496ANATOLIAN HIER\
    \OGLYPH A497ANATOLIAN HIEROGLYPH A501ANATOLIAN HIEROGLYPH A502ANATOLIAN HIEROGLYPH A503ANATOLIAN HIEROGLYPH A504ANATOLIAN HIEROGLYPH A505ANATOLIAN HIEROGLYPH A506ANATOLIAN HIEROGLYPH A507ANATOLIAN HIEROGLYPH A508ANATOLIAN HIEROGLYPH A509ANATOLIAN HIEROGLYP\
    \H A510ANATOLIAN HIEROGLYPH A511ANATOLIAN HIEROGLYPH A512ANATOLIAN HIEROGLYPH A513ANATOLIAN HIEROGLYPH A514ANATOLIAN HIEROGLYPH A515ANATOLIAN HIEROGLYPH A516ANATOLIAN HIEROGLYPH A517ANATOLIAN HIEROGLYPH A518ANATOLIAN HIEROGLYPH A519ANATOLIAN HIEROGLYPH A52\
    \0ANATOLIAN HIEROGLYPH A521ANATOLIAN HIEROGLYPH A522ANATOLIAN HIEROGLYPH A523ANATOLIAN HIEROGLYPH A524ANATOLIAN HIEROGLYPH A525ANATOLIAN HIEROGLYPH A526ANATOLIAN HIEROGLYPH A527ANATOLIAN HIEROGLYPH A528ANATOLIAN HIEROGLYPH A529ANATOLIAN HIEROGLYPH A530BAMU\
    \M LETTER PHASE-A NGKUE MFONBAMUM LETTER PHASE-A GBIEE FONBAMUM LETTER PHASE-A PON MFON PIPAEMGBIEEBAMUM LETTER PHASE-A PON MFON PIPAEMBABAMUM LETTER PHASE-A NAA MFONBAMUM LETTER PHASE-A SHUENSHUETBAMUM LETTER PHASE-A TITA MFONBAMUM LETTER PHASE-A NZA MFON\
    \BAMUM LETTER PHASE-A SHINDA PA NJIBAMUM LETTER PHASE-A PON PA NJI PIPAEMGBIEEBAMUM LETTER PHASE-A PON PA NJI PIPAEMBABAMUM LETTER PHASE-A MAEMBGBIEEBAMUM LETTER PHASE-A TU MAEMBABAMUM LETTER PHASE-A NGANGUBAMUM LETTER PHASE-A MAEMVEUXBAMUM LETTER PHASE-A \
    \MANSUAEBAMUM LETTER PHASE-A MVEUAENGAMBAMUM LETTER PHASE-A SEUNYAMBAMUM LETTER PHASE-A NTOQPENBAMUM LETTER PHASE-A KEUKEUTNDABAMUM LETTER PHASE-A NKINDIBAMUM LETTER PHASE-A SUUBAMUM LETTER PHASE-A NGKUENZEUMBAMUM LETTER PHASE-A LAPAQBAMUM LETTER PHASE-A L\
    \ET KUTBAMUM LETTER PHASE-A NTAP MFAABAMUM LETTER PHASE-A MAEKEUPBAMUM LETTER PHASE-A PASHAEBAMUM LETTER PHASE-A GHEUAERAEBAMUM LETTER PHASE-A PAMSHAEBAMUM LETTER PHASE-A MON NGGEUAETBAMUM LETTER PHASE-A NZUN MEUTBAMUM LETTER PHASE-A U YUQ NAEBAMUM LETTER \
    \PHASE-A GHEUAEGHEUAEBAMUM LETTER PHASE-A NTAP NTAABAMUM LETTER PHASE-A SISABAMUM LETTER PHASE-A MGBASABAMUM LETTER PHASE-A MEUNJOMNDEUQBAMUM LETTER PHASE-A MOOMPUQBAMUM LETTER PHASE-A KAFABAMUM LETTER PHASE-A PA LEERAEWABAMUM LETTER PHASE-A NDA LEERAEWABA\
    \MUM LETTER PHASE-A PETBAMUM LETTER PHASE-A MAEMKPENBAMUM LETTER PHASE-A NIKABAMUM LETTER PHASE-A PUPBAMUM LETTER PHASE-A TUAEPBAMUM LETTER PHASE-A LUAEPBAMUM LETTER PHASE-A SONJAMBAMUM LETTER PHASE-A TEUTEUWENBAMUM LETTER PHASE-A MAENYIBAMUM LETTER PHASE-\
    \A KETBAMUM LETTER PHASE-A NDAANGGEUAETBAMUM LETTER PHASE-A KUOQBAMUM LETTER PHASE-A MOOMEUTBAMUM LETTER PHASE-A SHUMBAMUM LETTER PHASE-A LOMMAEBAMUM LETTER PHASE-A FIRIBAMUM LETTER PHASE-A ROMBAMUM LETTER PHASE-A KPOQBAMUM LETTER PHASE-A SOQBAMUM LETTER P\
    \HASE-A MAP PIEETBAMUM LETTER PHASE-A SHIRAEBAMUM LETTER PHASE-A NTAPBAMUM LETTER PHASE-A SHOQ NSHUT YUMBAMUM LETTER PHASE-A NYIT MONGKEUAEQBAMUM LETTER PHASE-A PAARAEBAMUM LETTER PHASE-A NKAARAEBAMUM LETTER PHASE-A UNKNOWNBAMUM LETTER PHASE-A NGGENBAMUM L\
    \ETTER PHASE-A MAESIBAMUM LETTER PHASE-A NJAMBAMUM LETTER PHASE-A MBANYIBAMUM LETTER PHASE-A NYETBAMUM LETTER PHASE-A TEUAENBAMUM LETTER PHASE-A SOTBAMUM LETTER PHASE-A PAAMBAMUM LETTER PHASE-A NSHIEEBAMUM LETTER PHASE-A MAEMBAMUM LETTER PHASE-A NYIBAMUM L\
    \ETTER PHASE-A KAQBAMUM LETTER PHASE-A NSHABAMUM LETTER PHASE-A VEEBAMUM LETTER PHASE-A LUBAMUM LETTER PHASE-A NENBAMUM LETTER PHASE-A NAQBAMUM LETTER PHASE-A MBAQBAMUM LETTER PHASE-B NSHUETBAMUM LETTER PHASE-B TU MAEMGBIEEBAMUM LETTER PHASE-B SIEEBAMUM LE\
    \TTER PHASE-B SET TUBAMUM LETTER PHASE-B LOM NTEUMBAMUM LETTER PHASE-B MBA MAELEEBAMUM LETTER PHASE-B KIEEMBAMUM LETTER PHASE-B YEURAEBAMUM LETTER PHASE-B MBAARAEBAMUM LETTER PHASE-B KAMBAMUM LETTER PHASE-B PEESHIBAMUM LETTER PHASE-B YAFU LEERAEWABAMUM LET\
    \TER PHASE-B LAM NSHUT NYAMBAMUM LETTER PHASE-B NTIEE SHEUOQBAMUM LETTER PHASE-B NDU NJAABAMUM LETTER PHASE-B GHEUGHEUAEMBAMUM LETTER PHASE-B PITBAMUM LETTER PHASE-B TU NSIEEBAMUM LETTER PHASE-B SHET NJAQBAMUM LETTER PHASE-B SHEUAEQTUBAMUM LETTER PHASE-B M\
    \FON TEUAEQBAMUM LETTER PHASE-B MBIT MBAAKETBAMUM LETTER PHASE-B NYI NTEUMBAMUM LETTER PHASE-B KEUPUQBAMUM LETTER PHASE-B GHEUGHENBAMUM LETTER PHASE-B KEUYEUXBAMUM LETTER PHASE-B LAANAEBAMUM LETTER PHASE-B PARUMBAMUM LETTER PHASE-B VEUMBAMUM LETTER PHASE-B\
    \ NGKINDI MVOPBAMUM LETTER PHASE-B NGGEU MBUBAMUM LETTER PHASE-B WUAETBAMUM LETTER PHASE-B SAKEUAEBAMUM LETTER PHASE-B TAAMBAMUM LETTER PHASE-B MEUQBAMUM LETTER PHASE-B NGGUOQBAMUM LETTER PHASE-B NGGUOQ LARGEBAMUM LETTER PHASE-B MFIYAQBAMUM LETTER PHASE-B \
    \SUEBAMUM LETTER PHASE-B MBEURIBAMUM LETTER PHASE-B MONTIEENBAMUM LETTER PHASE-B NYAEMAEBAMUM LETTER PHASE-B PUNGAAMBAMUM LETTER PHASE-B MEUT NGGEETBAMUM LETTER PHASE-B FEUXBAMUM LETTER PHASE-B MBUOQBAMUM LETTER PHASE-B FEEBAMUM LETTER PHASE-B KEUAEMBAMUM \
    \LETTER PHASE-B MA NJEUAENABAMUM LETTER PHASE-B MA NJUQABAMUM LETTER PHASE-B LETBAMUM LETTER PHASE-B NGGAAMBAMUM LETTER PHASE-B NSENBAMUM LETTER PHASE-B MABAMUM LETTER PHASE-B KIQBAMUM LETTER PHASE-B NGOMBAMUM LETTER PHASE-C NGKUE MAEMBABAMUM LETTER PHASE-\
    \C NZABAMUM LETTER PHASE-C YUMBAMUM LETTER PHASE-C WANGKUOQBAMUM LETTER PHASE-C NGGENBAMUM LETTER PHASE-C NDEUAEREEBAMUM LETTER PHASE-C NGKAQBAMUM LETTER PHASE-C GHARAEBAMUM LETTER PHASE-C MBEEKEETBAMUM LETTER PHASE-C GBAYIBAMUM LETTER PHASE-C NYIR MKPARAQ\
    \ MEUNBAMUM LETTER PHASE-C NTU MBITBAMUM LETTER PHASE-C MBEUMBAMUM LETTER PHASE-C PIRIEENBAMUM LETTER PHASE-C NDOMBUBAMUM LETTER PHASE-C MBAA CABBAGE-TREEBAMUM LETTER PHASE-C KEUSHEUAEPBAMUM LETTER PHASE-C GHAPBAMUM LETTER PHASE-C KEUKAQBAMUM LETTER PHASE-\
    \C YU MUOMAEBAMUM LETTER PHASE-C NZEUMBAMUM LETTER PHASE-C MBUEBAMUM LETTER PHASE-C NSEUAENBAMUM LETTER PHASE-C MBITBAMUM LETTER PHASE-C YEUQBAMUM LETTER PHASE-C KPARAQBAMUM LETTER PHASE-C KAABAMUM LETTER PHASE-C SEUXBAMUM LETTER PHASE-C NDIDABAMUM LETTER \
    \PHASE-C TAASHAEBAMUM LETTER PHASE-C NJUEQBAMUM LETTER PHASE-C TITA YUEBAMUM LETTER PHASE-C SUAETBAMUM LETTER PHASE-C NGGUAEN NYAMBAMUM LETTER PHASE-C VEUXBAMUM LETTER PHASE-C NANSANAQBAMUM LETTER PHASE-C MA KEUAERIBAMUM LETTER PHASE-C NTAABAMUM LETTER PHA\
    \SE-C NGGUONBAMUM LETTER PHASE-C LAPBAMUM LETTER PHASE-C MBIRIEENBAMUM LETTER PHASE-C MGBASAQBAMUM LETTER PHASE-C NTEUNGBABAMUM LETTER PHASE-C TEUTEUXBAMUM LETTER PHASE-C NGGUMBAMUM LETTER PHASE-C FUEBAMUM LETTER PHASE-C NDEUTBAMUM LETTER PHASE-C NSABAMUM \
    \LETTER PHASE-C NSHAQBAMUM LETTER PHASE-C BUNGBAMUM LETTER PHASE-C VEUAEPENBAMUM LETTER PHASE-C MBERAEBAMUM LETTER PHASE-C RUBAMUM LETTER PHASE-C NJAEMBAMUM LETTER PHASE-C LAMBAMUM LETTER PHASE-C TITUAEPBAMUM LETTER PHASE-C NSUOT NGOMBAMUM LETTER PHASE-C N\
    \JEEEEBAMUM LETTER PHASE-C KETBAMUM LETTER PHASE-C NGGUBAMUM LETTER PHASE-C MAESIBAMUM LETTER PHASE-C MBUAEMBAMUM LETTER PHASE-C LUBAMUM LETTER PHASE-C KUTBAMUM LETTER PHASE-C NJAMBAMUM LETTER PHASE-C NGOMBAMUM LETTER PHASE-C WUPBAMUM LETTER PHASE-C NGGUEE\
    \TBAMUM LETTER PHASE-C NSOMBAMUM LETTER PHASE-C NTENBAMUM LETTER PHASE-C KUOP NKAARAEBAMUM LETTER PHASE-C NSUNBAMUM LETTER PHASE-C NDAMBAMUM LETTER PHASE-C MA NSIEEBAMUM LETTER PHASE-C YAABAMUM LETTER PHASE-C NDAPBAMUM LETTER PHASE-C SHUEQBAMUM LETTER PHAS\
    \E-C SETFONBAMUM LETTER PHASE-C MBIBAMUM LETTER PHASE-C MAEMBABAMUM LETTER PHASE-C MBANYIBAMUM LETTER PHASE-C KEUSEUXBAMUM LETTER PHASE-C MBEUXBAMUM LETTER PHASE-C KEUMBAMUM LETTER PHASE-C MBAA PICKETBAMUM LETTER PHASE-C YUWOQBAMUM LETTER PHASE-C NJEUXBAMU\
    \M LETTER PHASE-C MIEEBAMUM LETTER PHASE-C MUAEBAMUM LETTER PHASE-C SHIQBAMUM LETTER PHASE-C KEN LAWBAMUM LETTER PHASE-C KEN FATIGUEBAMUM LETTER PHASE-C NGAQBAMUM LETTER PHASE-C NAQBAMUM LETTER PHASE-C LIQBAMUM LETTER PHASE-C PINBAMUM LETTER PHASE-C PENBAM\
    \UM LETTER PHASE-C TETBAMUM LETTER PHASE-D MBUOBAMUM LETTER PHASE-D WAPBAMUM LETTER PHASE-D NJIBAMUM LETTER PHASE-D MFONBAMUM LETTER PHASE-D NJIEEBAMUM LETTER PHASE-D LIEEBAMUM LETTER PHASE-D NJEUTBAMUM LETTER PHASE-D NSHEEBAMUM LETTER PHASE-D NGGAAMAEBAMU\
    \M LETTER PHASE-D NYAMBAMUM LETTER PHASE-D WUAENBAMUM LETTER PHASE-D NGKUNBAMUM LETTER PHASE-D SHEEBAMUM LETTER PHASE-D NGKAPBAMUM LETTER PHASE-D KEUAETMEUNBAMUM LETTER PHASE-D TEUTBAMUM LETTER PHASE-D SHEUAEBAMUM LETTER PHASE-D NJAPBAMUM LETTER PHASE-D SU\
    \EBAMUM LETTER PHASE-D KETBAMUM LETTER PHASE-D YAEMMAEBAMUM LETTER PHASE-D KUOMBAMUM LETTER PHASE-D SAPBAMUM LETTER PHASE-D MFEUTBAMUM LETTER PHASE-D NDEUXBAMUM LETTER PHASE-D MALEERIBAMUM LETTER PHASE-D MEUTBAMUM LETTER PHASE-D SEUAEQBAMUM LETTER PHASE-D \
    \YENBAMUM LETTER PHASE-D NJEUAEMBAMUM LETTER PHASE-D KEUOT MBUAEBAMUM LETTER PHASE-D NGKEURIBAMUM LETTER PHASE-D TUBAMUM LETTER PHASE-D GHAABAMUM LETTER PHASE-D NGKYEEBAMUM LETTER PHASE-D FEUFEUAETBAMUM LETTER PHASE-D NDEEBAMUM LETTER PHASE-D MGBOFUMBAMUM \
    \LETTER PHASE-D LEUAEPBAMUM LETTER PHASE-D NDONBAMUM LETTER PHASE-D MONIBAMUM LETTER PHASE-D MGBEUNBAMUM LETTER PHASE-D PUUTBAMUM LETTER PHASE-D MGBIEEBAMUM LETTER PHASE-D MFOBAMUM LETTER PHASE-D LUMBAMUM LETTER PHASE-D NSIEEPBAMUM LETTER PHASE-D MBAABAMUM\
    \ LETTER PHASE-D KWAETBAMUM LETTER PHASE-D NYETBAMUM LETTER PHASE-D TEUAENBAMUM LETTER PHASE-D SOTBAMUM LETTER PHASE-D YUWOQBAMUM LETTER PHASE-D KEUMBAMUM LETTER PHASE-D RAEMBAMUM LETTER PHASE-D TEEEEBAMUM LETTER PHASE-D NGKEUAEQBAMUM LETTER PHASE-D MFEUAE\
    \BAMUM LETTER PHASE-D NSIEETBAMUM LETTER PHASE-D KEUPBAMUM LETTER PHASE-D PIPBAMUM LETTER PHASE-D PEUTAEBAMUM LETTER PHASE-D NYUEBAMUM LETTER PHASE-D LETBAMUM LETTER PHASE-D NGGAAMBAMUM LETTER PHASE-D MFIEEBAMUM LETTER PHASE-D NGGWAENBAMUM LETTER PHASE-D Y\
    \UOMBAMUM LETTER PHASE-D PAPBAMUM LETTER PHASE-D YUOPBAMUM LETTER PHASE-D NDAMBAMUM LETTER PHASE-D NTEUMBAMUM LETTER PHASE-D SUAEBAMUM LETTER PHASE-D KUNBAMUM LETTER PHASE-D NGGEUXBAMUM LETTER PHASE-D NGKIEEBAMUM LETTER PHASE-D TUOTBAMUM LETTER PHASE-D MEU\
    \NBAMUM LETTER PHASE-D KUQBAMUM LETTER PHASE-D NSUMBAMUM LETTER PHASE-D TEUNBAMUM LETTER PHASE-D MAENJETBAMUM LETTER PHASE-D NGGAPBAMUM LETTER PHASE-D LEUMBAMUM LETTER PHASE-D NGGUOMBAMUM LETTER PHASE-D NSHUTBAMUM LETTER PHASE-D NJUEQBAMUM LETTER PHASE-D G\
    \HEUAEBAMUM LETTER PHASE-D KUBAMUM LETTER PHASE-D REN OLDBAMUM LETTER PHASE-D TAEBAMUM LETTER PHASE-D TOQBAMUM LETTER PHASE-D NYIBAMUM LETTER PHASE-D RIIBAMUM LETTER PHASE-D LEEEEBAMUM LETTER PHASE-D MEEEEBAMUM LETTER PHASE-D MBAMUM LETTER PHASE-D SUUBAMUM\
    \ LETTER PHASE-D MUBAMUM LETTER PHASE-D SHIIBAMUM LETTER PHASE-D SHEUXBAMUM LETTER PHASE-D KYEEBAMUM LETTER PHASE-D NUBAMUM LETTER PHASE-D SHUBAMUM LETTER PHASE-D NTEEBAMUM LETTER PHASE-D PEEBAMUM LETTER PHASE-D NIBAMUM LETTER PHASE-D SHOQBAMUM LETTER PHAS\
    \E-D PUQBAMUM LETTER PHASE-D MVOPBAMUM LETTER PHASE-D LOQBAMUM LETTER PHASE-D REN MUCHBAMUM LETTER PHASE-D TIBAMUM LETTER PHASE-D NTUUBAMUM LETTER PHASE-D MBAA SEVENBAMUM LETTER PHASE-D SAQBAMUM LETTER PHASE-D FAABAMUM LETTER PHASE-E NDAPBAMUM LETTER PHASE\
    \-E TOONBAMUM LETTER PHASE-E MBEUMBAMUM LETTER PHASE-E LAPBAMUM LETTER PHASE-E VOMBAMUM LETTER PHASE-E LOONBAMUM LETTER PHASE-E PAABAMUM LETTER PHASE-E SOMBAMUM LETTER PHASE-E RAQBAMUM LETTER PHASE-E NSHUOPBAMUM LETTER PHASE-E NDUNBAMUM LETTER PHASE-E PUAE\
    \BAMUM LETTER PHASE-E TAMBAMUM LETTER PHASE-E NGKABAMUM LETTER PHASE-E KPEUXBAMUM LETTER PHASE-E WUOBAMUM LETTER PHASE-E SEEBAMUM LETTER PHASE-E NGGEUAETBAMUM LETTER PHASE-E PAAMBAMUM LETTER PHASE-E TOOBAMUM LETTER PHASE-E KUOPBAMUM LETTER PHASE-E LOMBAMUM\
    \ LETTER PHASE-E NSHIEEBAMUM LETTER PHASE-E NGOPBAMUM LETTER PHASE-E MAEMBAMUM LETTER PHASE-E NGKEUXBAMUM LETTER PHASE-E NGOQBAMUM LETTER PHASE-E NSHUEBAMUM LETTER PHASE-E RIMGBABAMUM LETTER PHASE-E NJEUXBAMUM LETTER PHASE-E PEEMBAMUM LETTER PHASE-E SAABAM\
    \UM LETTER PHASE-E NGGURAEBAMUM LETTER PHASE-E MGBABAMUM LETTER PHASE-E GHEUXBAMUM LETTER PHASE-E NGKEUAEMBAMUM LETTER PHASE-E NJAEMLIBAMUM LETTER PHASE-E MAPBAMUM LETTER PHASE-E LOOTBAMUM LETTER PHASE-E NGGEEEEBAMUM LETTER PHASE-E NDIQBAMUM LETTER PHASE-E\
    \ TAEN NTEUMBAMUM LETTER PHASE-E SETBAMUM LETTER PHASE-E PUMBAMUM LETTER PHASE-E NDAA SOFTNESSBAMUM LETTER PHASE-E NGGUAESHAE NYAMBAMUM LETTER PHASE-E YIEEBAMUM LETTER PHASE-E GHEUNBAMUM LETTER PHASE-E TUAEBAMUM LETTER PHASE-E YEUAEBAMUM LETTER PHASE-E POB\
    \AMUM LETTER PHASE-E TUMAEBAMUM LETTER PHASE-E KEUAEBAMUM LETTER PHASE-E SUAENBAMUM LETTER PHASE-E TEUAEQBAMUM LETTER PHASE-E VEUAEBAMUM LETTER PHASE-E WEUXBAMUM LETTER PHASE-E LAAMBAMUM LETTER PHASE-E PUBAMUM LETTER PHASE-E TAAQBAMUM LETTER PHASE-E GHAAMA\
    \EBAMUM LETTER PHASE-E NGEUREUTBAMUM LETTER PHASE-E SHEUAEQBAMUM LETTER PHASE-E MGBENBAMUM LETTER PHASE-E MBEEBAMUM LETTER PHASE-E NZAQBAMUM LETTER PHASE-E NKOMBAMUM LETTER PHASE-E GBETBAMUM LETTER PHASE-E TUMBAMUM LETTER PHASE-E KUETBAMUM LETTER PHASE-E Y\
    \APBAMUM LETTER PHASE-E NYI CLEAVERBAMUM LETTER PHASE-E YITBAMUM LETTER PHASE-E MFEUQBAMUM LETTER PHASE-E NDIAQBAMUM LETTER PHASE-E PIEEQBAMUM LETTER PHASE-E YUEQBAMUM LETTER PHASE-E LEUAEMBAMUM LETTER PHASE-E FUEBAMUM LETTER PHASE-E GBEUXBAMUM LETTER PHAS\
    \E-E NGKUPBAMUM LETTER PHASE-E KETBAMUM LETTER PHASE-E MAEBAMUM LETTER PHASE-E NGKAAMIBAMUM LETTER PHASE-E GHETBAMUM LETTER PHASE-E FABAMUM LETTER PHASE-E NTUMBAMUM LETTER PHASE-E PEUTBAMUM LETTER PHASE-E YEUMBAMUM LETTER PHASE-E NGGEUAEBAMUM LETTER PHASE-\
    \E NYI BETWEENBAMUM LETTER PHASE-E NZUQBAMUM LETTER PHASE-E POONBAMUM LETTER PHASE-E MIEEBAMUM LETTER PHASE-E FUETBAMUM LETTER PHASE-E NAEBAMUM LETTER PHASE-E MUAEBAMUM LETTER PHASE-E GHEUAEBAMUM LETTER PHASE-E FU IBAMUM LETTER PHASE-E MVIBAMUM LETTER PHAS\
    \E-E PUAQBAMUM LETTER PHASE-E NGKUMBAMUM LETTER PHASE-E KUTBAMUM LETTER PHASE-E PIETBAMUM LETTER PHASE-E NTAPBAMUM LETTER PHASE-E YEUAETBAMUM LETTER PHASE-E NGGUPBAMUM LETTER PHASE-E PA PEOPLEBAMUM LETTER PHASE-E FU CALLBAMUM LETTER PHASE-E FOMBAMUM LETTER\
    \ PHASE-E NJEEBAMUM LETTER PHASE-E ABAMUM LETTER PHASE-E TOQBAMUM LETTER PHASE-E OBAMUM LETTER PHASE-E IBAMUM LETTER PHASE-E LAQBAMUM LETTER PHASE-E PA PLURALBAMUM LETTER PHASE-E TAABAMUM LETTER PHASE-E TAQBAMUM LETTER PHASE-E NDAA MY HOUSEBAMUM LETTER PHA\
    \SE-E SHIQBAMUM LETTER PHASE-E YEUXBAMUM LETTER PHASE-E NGUAEBAMUM LETTER PHASE-E YUAENBAMUM LETTER PHASE-E YOQ SWIMMINGBAMUM LETTER PHASE-E YOQ COVERBAMUM LETTER PHASE-E YUQBAMUM LETTER PHASE-E YUNBAMUM LETTER PHASE-E KEUXBAMUM LETTER PHASE-E PEUXBAMUM LE\
    \TTER PHASE-E NJEE EPOCHBAMUM LETTER PHASE-E PUEBAMUM LETTER PHASE-E WUEBAMUM LETTER PHASE-E FEEBAMUM LETTER PHASE-E VEEBAMUM LETTER PHASE-E LUBAMUM LETTER PHASE-E MIBAMUM LETTER PHASE-E REUXBAMUM LETTER PHASE-E RAEBAMUM LETTER PHASE-E NGUAETBAMUM LETTER P\
    \HASE-E NGABAMUM LETTER PHASE-E SHOBAMUM LETTER PHASE-E SHOQBAMUM LETTER PHASE-E FU REMEDYBAMUM LETTER PHASE-E NABAMUM LETTER PHASE-E PIBAMUM LETTER PHASE-E LOQBAMUM LETTER PHASE-E KOBAMUM LETTER PHASE-E MENBAMUM LETTER PHASE-E MABAMUM LETTER PHASE-E MAQBA\
    \MUM LETTER PHASE-E TEUBAMUM LETTER PHASE-E KIBAMUM LETTER PHASE-E MONBAMUM LETTER PHASE-E TENBAMUM LETTER PHASE-E FAQBAMUM LETTER PHASE-E GHOMBAMUM LETTER PHASE-F KABAMUM LETTER PHASE-F UBAMUM LETTER PHASE-F KUBAMUM LETTER PHASE-F EEBAMUM LETTER PHASE-F R\
    \EEBAMUM LETTER PHASE-F TAEBAMUM LETTER PHASE-F NYIBAMUM LETTER PHASE-F LABAMUM LETTER PHASE-F RIIBAMUM LETTER PHASE-F RIEEBAMUM LETTER PHASE-F MEEEEBAMUM LETTER PHASE-F TAABAMUM LETTER PHASE-F NDAABAMUM LETTER PHASE-F NJAEMBAMUM LETTER PHASE-F MBAMUM LETT\
    \ER PHASE-F SUUBAMUM LETTER PHASE-F SHIIBAMUM LETTER PHASE-F SIBAMUM LETTER PHASE-F SEUXBAMUM LETTER PHASE-F KYEEBAMUM LETTER PHASE-F KETBAMUM LETTER PHASE-F NUAEBAMUM LETTER PHASE-F NUBAMUM LETTER PHASE-F NJUAEBAMUM LETTER PHASE-F YOQBAMUM LETTER PHASE-F \
    \SHUBAMUM LETTER PHASE-F YABAMUM LETTER PHASE-F NSHABAMUM LETTER PHASE-F PEUXBAMUM LETTER PHASE-F NTEEBAMUM LETTER PHASE-F WUEBAMUM LETTER PHASE-F PEEBAMUM LETTER PHASE-F RUBAMUM LETTER PHASE-F NIBAMUM LETTER PHASE-F REUXBAMUM LETTER PHASE-F KENBAMUM LETTE\
    \R PHASE-F NGKWAENBAMUM LETTER PHASE-F NGGABAMUM LETTER PHASE-F SHOBAMUM LETTER PHASE-F PUAEBAMUM LETTER PHASE-F FOMBAMUM LETTER PHASE-F WABAMUM LETTER PHASE-F LIBAMUM LETTER PHASE-F LOQBAMUM LETTER PHASE-F KOBAMUM LETTER PHASE-F MBENBAMUM LETTER PHASE-F R\
    \ENBAMUM LETTER PHASE-F MABAMUM LETTER PHASE-F MOBAMUM LETTER PHASE-F MBAABAMUM LETTER PHASE-F TETBAMUM LETTER PHASE-F KPABAMUM LETTER PHASE-F SAMBABAMUM LETTER PHASE-F VUEQMRO LETTER TAMRO LETTER NGIMRO LETTER YOMRO LETTER MIMMRO LETTER BAMRO LETTER DAMRO\
    \ LETTER AMRO LETTER PHIMRO LETTER KHAIMRO LETTER HAOMRO LETTER DAIMRO LETTER CHUMRO LETTER KEAAEMRO LETTER OLMRO LETTER MAEMMRO LETTER NINMRO LETTER PAMRO LETTER OOMRO LETTER OMRO LETTER ROMRO LETTER SHIMRO LETTER THEAMRO LETTER EAMRO LETTER WAMRO LETTER \
    \EMRO LETTER KOMRO LETTER LANMRO LETTER LAMRO LETTER HAIMRO LETTER RIMRO LETTER TEKMRO DIGIT ZEROMRO DIGIT ONEMRO DIGIT TWOMRO DIGIT THREEMRO DIGIT FOURMRO DIGIT FIVEMRO DIGIT SIXMRO DIGIT SEVENMRO DIGIT EIGHTMRO DIGIT NINEMRO DANDAMRO DOUBLE DANDATANGSA L\
    \ETTER OZTANGSA LETTER OCTANGSA LETTER OQTANGSA LETTER OXTANGSA LETTER AZTANGSA LETTER ACTANGSA LETTER AQTANGSA LETTER AXTANGSA LETTER VZTANGSA LETTER VCTANGSA LETTER VQTANGSA LETTER VXTANGSA LETTER EZTANGSA LETTER ECTANGSA LETTER EQTANGSA LETTER EXTANGSA \
    \LETTER IZTANGSA LETTER ICTANGSA LETTER IQTANGSA LETTER IXTANGSA LETTER UZTANGSA LETTER UCTANGSA LETTER UQTANGSA LETTER UXTANGSA LETTER AWZTANGSA LETTER AWCTANGSA LETTER AWQTANGSA LETTER AWXTANGSA LETTER UIZTANGSA LETTER UICTANGSA LETTER UIQTANGSA LETTER U\
    \IXTANGSA LETTER FINAL NGTANGSA LETTER LONG UEXTANGSA LETTER SHORT UEZTANGSA LETTER SHORT AWXTANGSA LETTER UECTANGSA LETTER UEZTANGSA LETTER UEQTANGSA LETTER UEXTANGSA LETTER UIUZTANGSA LETTER UIUCTANGSA LETTER UIUQTANGSA LETTER UIUXTANGSA LETTER MZTANGSA \
    \LETTER MCTANGSA LETTER MQTANGSA LETTER MXTANGSA LETTER KATANGSA LETTER KHATANGSA LETTER GATANGSA LETTER NGATANGSA LETTER SATANGSA LETTER YATANGSA LETTER WATANGSA LETTER PATANGSA LETTER NYATANGSA LETTER PHATANGSA LETTER BATANGSA LETTER MATANGSA LETTER NATA\
    \NGSA LETTER HATANGSA LETTER LATANGSA LETTER HTATANGSA LETTER TATANGSA LETTER DATANGSA LETTER RATANGSA LETTER NHATANGSA LETTER SHATANGSA LETTER CATANGSA LETTER TSATANGSA LETTER GHATANGSA LETTER HTTATANGSA LETTER THATANGSA LETTER XATANGSA LETTER FATANGSA LE\
    \TTER DHATANGSA LETTER CHATANGSA LETTER ZATANGSA DIGIT ZEROTANGSA DIGIT ONETANGSA DIGIT TWOTANGSA DIGIT THREETANGSA DIGIT FOURTANGSA DIGIT FIVETANGSA DIGIT SIXTANGSA DIGIT SEVENTANGSA DIGIT EIGHTTANGSA DIGIT NINEBASSA VAH LETTER ENNIBASSA VAH LETTER KABASS\
    \A VAH LETTER SEBASSA VAH LETTER FABASSA VAH LETTER MBEBASSA VAH LETTER YIEBASSA VAH LETTER GAHBASSA VAH LETTER DHIIBASSA VAH LETTER KPAHBASSA VAH LETTER JOBASSA VAH LETTER HWAHBASSA VAH LETTER WABASSA VAH LETTER ZOBASSA VAH LETTER GBUBASSA VAH LETTER DOBA\
    \SSA VAH LETTER CEBASSA VAH LETTER UWUBASSA VAH LETTER TOBASSA VAH LETTER BABASSA VAH LETTER VUBASSA VAH LETTER YEINBASSA VAH LETTER PABASSA VAH LETTER WADDABASSA VAH LETTER ABASSA VAH LETTER OBASSA VAH LETTER OOBASSA VAH LETTER UBASSA VAH LETTER EEBASSA V\
    \AH LETTER EBASSA VAH LETTER IBASSA VAH COMBINING HIGH TONEBASSA VAH COMBINING LOW TONEBASSA VAH COMBINING MID TONEBASSA VAH COMBINING LOW-MID TONEBASSA VAH COMBINING HIGH-LOW TONEBASSA VAH FULL STOPPAHAWH HMONG VOWEL KEEBPAHAWH HMONG VOWEL KEEVPAHAWH HMON\
    \G VOWEL KIBPAHAWH HMONG VOWEL KIVPAHAWH HMONG VOWEL KAUBPAHAWH HMONG VOWEL KAUVPAHAWH HMONG VOWEL KUBPAHAWH HMONG VOWEL KUVPAHAWH HMONG VOWEL KEBPAHAWH HMONG VOWEL KEVPAHAWH HMONG VOWEL KAIBPAHAWH HMONG VOWEL KAIVPAHAWH HMONG VOWEL KOOBPAHAWH HMONG VOWEL \
    \KOOVPAHAWH HMONG VOWEL KAWBPAHAWH HMONG VOWEL KAWVPAHAWH HMONG VOWEL KUABPAHAWH HMONG VOWEL KUAVPAHAWH HMONG VOWEL KOBPAHAWH HMONG VOWEL KOVPAHAWH HMONG VOWEL KIABPAHAWH HMONG VOWEL KIAVPAHAWH HMONG VOWEL KABPAHAWH HMONG VOWEL KAVPAHAWH HMONG VOWEL KWBPAH\
    \AWH HMONG VOWEL KWVPAHAWH HMONG VOWEL KAABPAHAWH HMONG VOWEL KAAVPAHAWH HMONG CONSONANT VAUPAHAWH HMONG CONSONANT NTSAUPAHAWH HMONG CONSONANT LAUPAHAWH HMONG CONSONANT HAUPAHAWH HMONG CONSONANT NLAUPAHAWH HMONG CONSONANT RAUPAHAWH HMONG CONSONANT NKAUPAHA\
    \WH HMONG CONSONANT QHAUPAHAWH HMONG CONSONANT YAUPAHAWH HMONG CONSONANT HLAUPAHAWH HMONG CONSONANT MAUPAHAWH HMONG CONSONANT CHAUPAHAWH HMONG CONSONANT NCHAUPAHAWH HMONG CONSONANT HNAUPAHAWH HMONG CONSONANT PLHAUPAHAWH HMONG CONSONANT NTHAUPAHAWH HMONG CO\
    \NSONANT NAUPAHAWH HMONG CONSONANT AUPAHAWH HMONG CONSONANT XAUPAHAWH HMONG CONSONANT CAUPAHAWH HMONG MARK CIM TUBPAHAWH HMONG MARK CIM SOPAHAWH HMONG MARK CIM KESPAHAWH HMONG MARK CIM KHAVPAHAWH HMONG MARK CIM SUAMPAHAWH HMONG MARK CIM HOMPAHAWH HMONG MAR\
    \K CIM TAUMPAHAWH HMONG SIGN VOS THOMPAHAWH HMONG SIGN VOS TSHAB CEEBPAHAWH HMONG SIGN CIM CHEEMPAHAWH HMONG SIGN VOS THIABPAHAWH HMONG SIGN VOS FEEMPAHAWH HMONG SIGN XYEEM NTXIVPAHAWH HMONG SIGN XYEEM RHOPAHAWH HMONG SIGN XYEEM TOVPAHAWH HMONG SIGN XYEEM \
    \FAIBPAHAWH HMONG SIGN VOS SEEVPAHAWH HMONG SIGN MEEJ SUABPAHAWH HMONG SIGN VOS NRUAPAHAWH HMONG SIGN IB YAMPAHAWH HMONG SIGN XAUSPAHAWH HMONG SIGN CIM TSOV ROGPAHAWH HMONG DIGIT ZEROPAHAWH HMONG DIGIT ONEPAHAWH HMONG DIGIT TWOPAHAWH HMONG DIGIT THREEPAHAW\
    \H HMONG DIGIT FOURPAHAWH HMONG DIGIT FIVEPAHAWH HMONG DIGIT SIXPAHAWH HMONG DIGIT SEVENPAHAWH HMONG DIGIT EIGHTPAHAWH HMONG DIGIT NINEPAHAWH HMONG NUMBER TENSPAHAWH HMONG NUMBER HUNDREDSPAHAWH HMONG NUMBER TEN THOUSANDSPAHAWH HMONG NUMBER MILLIONSPAHAWH H\
    \MONG NUMBER HUNDRED MILLIONSPAHAWH HMONG NUMBER TEN BILLIONSPAHAWH HMONG NUMBER TRILLIONSPAHAWH HMONG SIGN VOS LUBPAHAWH HMONG SIGN XYOOPAHAWH HMONG SIGN HLIPAHAWH HMONG SIGN THIRD-STAGE HLIPAHAWH HMONG SIGN ZWJ THAJPAHAWH HMONG SIGN HNUBPAHAWH HMONG SIGN\
    \ NQIGPAHAWH HMONG SIGN XIABPAHAWH HMONG SIGN NTUJPAHAWH HMONG SIGN AVPAHAWH HMONG SIGN TXHEEJ CEEVPAHAWH HMONG SIGN MEEJ TSEEBPAHAWH HMONG SIGN TAUPAHAWH HMONG SIGN LOSPAHAWH HMONG SIGN MUSPAHAWH HMONG SIGN CIM HAIS LUS NTOG NTOGPAHAWH HMONG SIGN CIM CUAM\
    \ TSHOOJPAHAWH HMONG SIGN CIM TXWVPAHAWH HMONG SIGN CIM TXWV CHWVPAHAWH HMONG SIGN CIM PUB DAWBPAHAWH HMONG SIGN CIM NRES TOSPAHAWH HMONG CLAN SIGN TSHEEJPAHAWH HMONG CLAN SIGN YEEGPAHAWH HMONG CLAN SIGN LISPAHAWH HMONG CLAN SIGN LAUJPAHAWH HMONG CLAN SIGN\
    \ XYOOJPAHAWH HMONG CLAN SIGN KOOPAHAWH HMONG CLAN SIGN HAWJPAHAWH HMONG CLAN SIGN MUASPAHAWH HMONG CLAN SIGN THOJPAHAWH HMONG CLAN SIGN TSABPAHAWH HMONG CLAN SIGN PHABPAHAWH HMONG CLAN SIGN KHABPAHAWH HMONG CLAN SIGN HAMPAHAWH HMONG CLAN SIGN VAJPAHAWH HM\
    \ONG CLAN SIGN FAJPAHAWH HMONG CLAN SIGN YAJPAHAWH HMONG CLAN SIGN TSWBPAHAWH HMONG CLAN SIGN KWMPAHAWH HMONG CLAN SIGN VWJMEDEFAIDRIN CAPITAL LETTER MMEDEFAIDRIN CAPITAL LETTER SMEDEFAIDRIN CAPITAL LETTER VMEDEFAIDRIN CAPITAL LETTER WMEDEFAIDRIN CAPITAL L\
    \ETTER ATIUMEDEFAIDRIN CAPITAL LETTER ZMEDEFAIDRIN CAPITAL LETTER KPMEDEFAIDRIN CAPITAL LETTER PMEDEFAIDRIN CAPITAL LETTER TMEDEFAIDRIN CAPITAL LETTER GMEDEFAIDRIN CAPITAL LETTER FMEDEFAIDRIN CAPITAL LETTER IMEDEFAIDRIN CAPITAL LETTER KMEDEFAIDRIN CAPITAL \
    \LETTER AMEDEFAIDRIN CAPITAL LETTER JMEDEFAIDRIN CAPITAL LETTER EMEDEFAIDRIN CAPITAL LETTER BMEDEFAIDRIN CAPITAL LETTER CMEDEFAIDRIN CAPITAL LETTER UMEDEFAIDRIN CAPITAL LETTER YUMEDEFAIDRIN CAPITAL LETTER LMEDEFAIDRIN CAPITAL LETTER QMEDEFAIDRIN CAPITAL LE\
    \TTER HPMEDEFAIDRIN CAPITAL LETTER NYMEDEFAIDRIN CAPITAL LETTER XMEDEFAIDRIN CAPITAL LETTER DMEDEFAIDRIN CAPITAL LETTER OEMEDEFAIDRIN CAPITAL LETTER NMEDEFAIDRIN CAPITAL LETTER RMEDEFAIDRIN CAPITAL LETTER OMEDEFAIDRIN CAPITAL LETTER AIMEDEFAIDRIN CAPITAL L\
    \ETTER YMEDEFAIDRIN SMALL LETTER MMEDEFAIDRIN SMALL LETTER SMEDEFAIDRIN SMALL LETTER VMEDEFAIDRIN SMALL LETTER WMEDEFAIDRIN SMALL LETTER ATIUMEDEFAIDRIN SMALL LETTER ZMEDEFAIDRIN SMALL LETTER KPMEDEFAIDRIN SMALL LETTER PMEDEFAIDRIN SMALL LETTER TMEDEFAIDRI\
    \N SMALL LETTER GMEDEFAIDRIN SMALL LETTER FMEDEFAIDRIN SMALL LETTER IMEDEFAIDRIN SMALL LETTER KMEDEFAIDRIN SMALL LETTER AMEDEFAIDRIN SMALL LETTER JMEDEFAIDRIN SMALL LETTER EMEDEFAIDRIN SMALL LETTER BMEDEFAIDRIN SMALL LETTER CMEDEFAIDRIN SMALL LETTER UMEDEF\
    \AIDRIN SMALL LETTER YUMEDEFAIDRIN SMALL LETTER LMEDEFAIDRIN SMALL LETTER QMEDEFAIDRIN SMALL LETTER HPMEDEFAIDRIN SMALL LETTER NYMEDEFAIDRIN SMALL LETTER XMEDEFAIDRIN SMALL LETTER DMEDEFAIDRIN SMALL LETTER OEMEDEFAIDRIN SMALL LETTER NMEDEFAIDRIN SMALL LETT\
    \ER RMEDEFAIDRIN SMALL LETTER OMEDEFAIDRIN SMALL LETTER AIMEDEFAIDRIN SMALL LETTER YMEDEFAIDRIN DIGIT ZEROMEDEFAIDRIN DIGIT ONEMEDEFAIDRIN DIGIT TWOMEDEFAIDRIN DIGIT THREEMEDEFAIDRIN DIGIT FOURMEDEFAIDRIN DIGIT FIVEMEDEFAIDRIN DIGIT SIXMEDEFAIDRIN DIGIT SE\
    \VENMEDEFAIDRIN DIGIT EIGHTMEDEFAIDRIN DIGIT NINEMEDEFAIDRIN NUMBER TENMEDEFAIDRIN NUMBER ELEVENMEDEFAIDRIN NUMBER TWELVEMEDEFAIDRIN NUMBER THIRTEENMEDEFAIDRIN NUMBER FOURTEENMEDEFAIDRIN NUMBER FIFTEENMEDEFAIDRIN NUMBER SIXTEENMEDEFAIDRIN NUMBER SEVENTEENM\
    \EDEFAIDRIN NUMBER EIGHTEENMEDEFAIDRIN NUMBER NINETEENMEDEFAIDRIN DIGIT ONE ALTERNATE FORMMEDEFAIDRIN DIGIT TWO ALTERNATE FORMMEDEFAIDRIN DIGIT THREE ALTERNATE FORMMEDEFAIDRIN COMMAMEDEFAIDRIN FULL STOPMEDEFAIDRIN SYMBOL AIVAMEDEFAIDRIN EXCLAMATION OHMIAO \
    \LETTER PAMIAO LETTER BAMIAO LETTER YI PAMIAO LETTER PLAMIAO LETTER MAMIAO LETTER MHAMIAO LETTER ARCHAIC MAMIAO LETTER FAMIAO LETTER VAMIAO LETTER VFAMIAO LETTER TAMIAO LETTER DAMIAO LETTER YI TTAMIAO LETTER YI TAMIAO LETTER TTAMIAO LETTER DDAMIAO LETTER N\
    \AMIAO LETTER NHAMIAO LETTER YI NNAMIAO LETTER ARCHAIC NAMIAO LETTER NNAMIAO LETTER NNHAMIAO LETTER LAMIAO LETTER LYAMIAO LETTER LHAMIAO LETTER LHYAMIAO LETTER TLHAMIAO LETTER DLHAMIAO LETTER TLHYAMIAO LETTER DLHYAMIAO LETTER KAMIAO LETTER GAMIAO LETTER YI\
    \ KAMIAO LETTER QAMIAO LETTER QGAMIAO LETTER NGAMIAO LETTER NGHAMIAO LETTER ARCHAIC NGAMIAO LETTER HAMIAO LETTER XAMIAO LETTER GHAMIAO LETTER GHHAMIAO LETTER TSSAMIAO LETTER DZZAMIAO LETTER NYAMIAO LETTER NYHAMIAO LETTER TSHAMIAO LETTER DZHAMIAO LETTER YI \
    \TSHAMIAO LETTER YI DZHAMIAO LETTER REFORMED TSHAMIAO LETTER SHAMIAO LETTER SSAMIAO LETTER ZHAMIAO LETTER ZSHAMIAO LETTER TSAMIAO LETTER DZAMIAO LETTER YI TSAMIAO LETTER SAMIAO LETTER ZAMIAO LETTER ZSAMIAO LETTER ZZAMIAO LETTER ZZSAMIAO LETTER ARCHAIC ZZAM\
    \IAO LETTER ZZYAMIAO LETTER ZZSYAMIAO LETTER WAMIAO LETTER AHMIAO LETTER HHAMIAO LETTER BRIMIAO LETTER SYIMIAO LETTER DZYIMIAO LETTER TEMIAO LETTER TSEMIAO LETTER RTEMIAO SIGN CONSONANT MODIFIER BARMIAO LETTER NASALIZATIONMIAO SIGN ASPIRATIONMIAO SIGN REFO\
    \RMED VOICINGMIAO SIGN REFORMED ASPIRATIONMIAO VOWEL SIGN AMIAO VOWEL SIGN AAMIAO VOWEL SIGN AHHMIAO VOWEL SIGN ANMIAO VOWEL SIGN ANGMIAO VOWEL SIGN OMIAO VOWEL SIGN OOMIAO VOWEL SIGN WOMIAO VOWEL SIGN WMIAO VOWEL SIGN EMIAO VOWEL SIGN ENMIAO VOWEL SIGN EN\
    \GMIAO VOWEL SIGN OEYMIAO VOWEL SIGN IMIAO VOWEL SIGN IAMIAO VOWEL SIGN IANMIAO VOWEL SIGN IANGMIAO VOWEL SIGN IOMIAO VOWEL SIGN IEMIAO VOWEL SIGN IIMIAO VOWEL SIGN IUMIAO VOWEL SIGN INGMIAO VOWEL SIGN UMIAO VOWEL SIGN UAMIAO VOWEL SIGN UANMIAO VOWEL SIGN \
    \UANGMIAO VOWEL SIGN UUMIAO VOWEL SIGN UEIMIAO VOWEL SIGN UNGMIAO VOWEL SIGN YMIAO VOWEL SIGN YIMIAO VOWEL SIGN AEMIAO VOWEL SIGN AEEMIAO VOWEL SIGN ERRMIAO VOWEL SIGN ROUNDED ERRMIAO VOWEL SIGN ERMIAO VOWEL SIGN ROUNDED ERMIAO VOWEL SIGN AIMIAO VOWEL SIGN\
    \ EIMIAO VOWEL SIGN AUMIAO VOWEL SIGN OUMIAO VOWEL SIGN NMIAO VOWEL SIGN NGMIAO VOWEL SIGN UOGMIAO VOWEL SIGN YUIMIAO VOWEL SIGN OGMIAO VOWEL SIGN OERMIAO VOWEL SIGN VWMIAO VOWEL SIGN IGMIAO VOWEL SIGN EAMIAO VOWEL SIGN IONGMIAO VOWEL SIGN UIMIAO TONE RIGH\
    \TMIAO TONE TOP RIGHTMIAO TONE ABOVEMIAO TONE BELOWMIAO LETTER TONE-2MIAO LETTER TONE-3MIAO LETTER TONE-4MIAO LETTER TONE-5MIAO LETTER TONE-6MIAO LETTER TONE-7MIAO LETTER TONE-8MIAO LETTER REFORMED TONE-1MIAO LETTER REFORMED TONE-2MIAO LETTER REFORMED TONE\
    \-4MIAO LETTER REFORMED TONE-5MIAO LETTER REFORMED TONE-6MIAO LETTER REFORMED TONE-8TANGUT ITERATION MARKNUSHU ITERATION MARKOLD CHINESE HOOK MARKOLD CHINESE ITERATION MARKKHITAN SMALL SCRIPT FILLERVIETNAMESE ALTERNATE READING MARK CAVIETNAMESE ALTERNATE R\
    \EADING MARK NHAYTANGUT COMPONENT-001TANGUT COMPONENT-002TANGUT COMPONENT-003TANGUT COMPONENT-004TANGUT COMPONENT-005TANGUT COMPONENT-006TANGUT COMPONENT-007TANGUT COMPONENT-008TANGUT COMPONENT-009TANGUT COMPONENT-010TANGUT COMPONENT-011TANGUT COMPONENT-01\
    \2TANGUT COMPONENT-013TANGUT COMPONENT-014TANGUT COMPONENT-015TANGUT COMPONENT-016TANGUT COMPONENT-017TANGUT COMPONENT-018TANGUT COMPONENT-019TANGUT COMPONENT-020TANGUT COMPONENT-021TANGUT COMPONENT-022TANGUT COMPONENT-023TANGUT COMPONENT-024TANGUT COMPONE\
    \NT-025TANGUT COMPONENT-026TANGUT COMPONENT-027TANGUT COMPONENT-028TANGUT COMPONENT-029TANGUT COMPONENT-030TANGUT COMPONENT-031TANGUT COMPONENT-032TANGUT COMPONENT-033TANGUT COMPONENT-034TANGUT COMPONENT-035TANGUT COMPONENT-036TANGUT COMPONENT-037TANGUT CO\
    \MPONENT-038TANGUT COMPONENT-039TANGUT COMPONENT-040TANGUT COMPONENT-041TANGUT COMPONENT-042TANGUT COMPONENT-043TANGUT COMPONENT-044TANGUT COMPONENT-045TANGUT COMPONENT-046TANGUT COMPONENT-047TANGUT COMPONENT-048TANGUT COMPONENT-049TANGUT COMPONENT-050TANG\
    \UT COMPONENT-051TANGUT COMPONENT-052TANGUT COMPONENT-053TANGUT COMPONENT-054TANGUT COMPONENT-055TANGUT COMPONENT-056TANGUT COMPONENT-057TANGUT COMPONENT-058TANGUT COMPONENT-059TANGUT COMPONENT-060TANGUT COMPONENT-061TANGUT COMPONENT-062TANGUT COMPONENT-06\
    \3TANGUT COMPONENT-064TANGUT COMPONENT-065TANGUT COMPONENT-066TANGUT COMPONENT-067TANGUT COMPONENT-068TANGUT COMPONENT-069TANGUT COMPONENT-070TANGUT COMPONENT-071TANGUT COMPONENT-072TANGUT COMPONENT-073TANGUT COMPONENT-074TANGUT COMPONENT-075TANGUT COMPONE\
    \NT-076TANGUT COMPONENT-077TANGUT COMPONENT-078TANGUT COMPONENT-079TANGUT COMPONENT-080TANGUT COMPONENT-081TANGUT COMPONENT-082TANGUT COMPONENT-083TANGUT COMPONENT-084TANGUT COMPONENT-085TANGUT COMPONENT-086TANGUT COMPONENT-087TANGUT COMPONENT-088TANGUT CO\
    \MPONENT-089TANGUT COMPONENT-090TANGUT COMPONENT-091TANGUT COMPONENT-092TANGUT COMPONENT-093TANGUT COMPONENT-094TANGUT COMPONENT-095TANGUT COMPONENT-096TANGUT COMPONENT-097TANGUT COMPONENT-098TANGUT COMPONENT-099TANGUT COMPONENT-100TANGUT COMPONENT-101TANG\
    \UT COMPONENT-102TANGUT COMPONENT-103TANGUT COMPONENT-104TANGUT COMPONENT-105TANGUT COMPONENT-106TANGUT COMPONENT-107TANGUT COMPONENT-108TANGUT COMPONENT-109TANGUT COMPONENT-110TANGUT COMPONENT-111TANGUT COMPONENT-112TANGUT COMPONENT-113TANGUT COMPONENT-11\
    \4TANGUT COMPONENT-115TANGUT COMPONENT-116TANGUT COMPONENT-117TANGUT COMPONENT-118TANGUT COMPONENT-119TANGUT COMPONENT-120TANGUT COMPONENT-121TANGUT COMPONENT-122TANGUT COMPONENT-123TANGUT COMPONENT-124TANGUT COMPONENT-125TANGUT COMPONENT-126TANGUT COMPONE\
    \NT-127TANGUT COMPONENT-128TANGUT COMPONENT-129TANGUT COMPONENT-130TANGUT COMPONENT-131TANGUT COMPONENT-132TANGUT COMPONENT-133TANGUT COMPONENT-134TANGUT COMPONENT-135TANGUT COMPONENT-136TANGUT COMPONENT-137TANGUT COMPONENT-138TANGUT COMPONENT-139TANGUT CO\
    \MPONENT-140TANGUT COMPONENT-141TANGUT COMPONENT-142TANGUT COMPONENT-143TANGUT COMPONENT-144TANGUT COMPONENT-145TANGUT COMPONENT-146TANGUT COMPONENT-147TANGUT COMPONENT-148TANGUT COMPONENT-149TANGUT COMPONENT-150TANGUT COMPONENT-151TANGUT COMPONENT-152TANG\
    \UT COMPONENT-153TANGUT COMPONENT-154TANGUT COMPONENT-155TANGUT COMPONENT-156TANGUT COMPONENT-157TANGUT COMPONENT-158TANGUT COMPONENT-159TANGUT COMPONENT-160TANGUT COMPONENT-161TANGUT COMPONENT-162TANGUT COMPONENT-163TANGUT COMPONENT-164TANGUT COMPONENT-16\
    \5TANGUT COMPONENT-166TANGUT COMPONENT-167TANGUT COMPONENT-168TANGUT COMPONENT-169TANGUT COMPONENT-170TANGUT COMPONENT-171TANGUT COMPONENT-172TANGUT COMPONENT-173TANGUT COMPONENT-174TANGUT COMPONENT-175TANGUT COMPONENT-176TANGUT COMPONENT-177TANGUT COMPONE\
    \NT-178TANGUT COMPONENT-179TANGUT COMPONENT-180TANGUT COMPONENT-181TANGUT COMPONENT-182TANGUT COMPONENT-183TANGUT COMPONENT-184TANGUT COMPONENT-185TANGUT COMPONENT-186TANGUT COMPONENT-187TANGUT COMPONENT-188TANGUT COMPONENT-189TANGUT COMPONENT-190TANGUT CO\
    \MPONENT-191TANGUT COMPONENT-192TANGUT COMPONENT-193TANGUT COMPONENT-194TANGUT COMPONENT-195TANGUT COMPONENT-196TANGUT COMPONENT-197TANGUT COMPONENT-198TANGUT COMPONENT-199TANGUT COMPONENT-200TANGUT COMPONENT-201TANGUT COMPONENT-202TANGUT COMPONENT-203TANG\
    \UT COMPONENT-204TANGUT COMPONENT-205TANGUT COMPONENT-206TANGUT COMPONENT-207TANGUT COMPONENT-208TANGUT COMPONENT-209TANGUT COMPONENT-210TANGUT COMPONENT-211TANGUT COMPONENT-212TANGUT COMPONENT-213TANGUT COMPONENT-214TANGUT COMPONENT-215TANGUT COMPONENT-21\
    \6TANGUT COMPONENT-217TANGUT COMPONENT-218TANGUT COMPONENT-219TANGUT COMPONENT-220TANGUT COMPONENT-221TANGUT COMPONENT-222TANGUT COMPONENT-223TANGUT COMPONENT-224TANGUT COMPONENT-225TANGUT COMPONENT-226TANGUT COMPONENT-227TANGUT COMPONENT-228TANGUT COMPONE\
    \NT-229TANGUT COMPONENT-230TANGUT COMPONENT-231TANGUT COMPONENT-232TANGUT COMPONENT-233TANGUT COMPONENT-234TANGUT COMPONENT-235TANGUT COMPONENT-236TANGUT COMPONENT-237TANGUT COMPONENT-238TANGUT COMPONENT-239TANGUT COMPONENT-240TANGUT COMPONENT-241TANGUT CO\
    \MPONENT-242TANGUT COMPONENT-243TANGUT COMPONENT-244TANGUT COMPONENT-245TANGUT COMPONENT-246TANGUT COMPONENT-247TANGUT COMPONENT-248TANGUT COMPONENT-249TANGUT COMPONENT-250TANGUT COMPONENT-251TANGUT COMPONENT-252TANGUT COMPONENT-253TANGUT COMPONENT-254TANG\
    \UT COMPONENT-255TANGUT COMPONENT-256TANGUT COMPONENT-257TANGUT COMPONENT-258TANGUT COMPONENT-259TANGUT COMPONENT-260TANGUT COMPONENT-261TANGUT COMPONENT-262TANGUT COMPONENT-263TANGUT COMPONENT-264TANGUT COMPONENT-265TANGUT COMPONENT-266TANGUT COMPONENT-26\
    \7TANGUT COMPONENT-268TANGUT COMPONENT-269TANGUT COMPONENT-270TANGUT COMPONENT-271TANGUT COMPONENT-272TANGUT COMPONENT-273TANGUT COMPONENT-274TANGUT COMPONENT-275TANGUT COMPONENT-276TANGUT COMPONENT-277TANGUT COMPONENT-278TANGUT COMPONENT-279TANGUT COMPONE\
    \NT-280TANGUT COMPONENT-281TANGUT COMPONENT-282TANGUT COMPONENT-283TANGUT COMPONENT-284TANGUT COMPONENT-285TANGUT COMPONENT-286TANGUT COMPONENT-287TANGUT COMPONENT-288TANGUT COMPONENT-289TANGUT COMPONENT-290TANGUT COMPONENT-291TANGUT COMPONENT-292TANGUT CO\
    \MPONENT-293TANGUT COMPONENT-294TANGUT COMPONENT-295TANGUT COMPONENT-296TANGUT COMPONENT-297TANGUT COMPONENT-298TANGUT COMPONENT-299TANGUT COMPONENT-300TANGUT COMPONENT-301TANGUT COMPONENT-302TANGUT COMPONENT-303TANGUT COMPONENT-304TANGUT COMPONENT-305TANG\
    \UT COMPONENT-306TANGUT COMPONENT-307TANGUT COMPONENT-308TANGUT COMPONENT-309TANGUT COMPONENT-310TANGUT COMPONENT-311TANGUT COMPONENT-312TANGUT COMPONENT-313TANGUT COMPONENT-314TANGUT COMPONENT-315TANGUT COMPONENT-316TANGUT COMPONENT-317TANGUT COMPONENT-31\
    \8TANGUT COMPONENT-319TANGUT COMPONENT-320TANGUT COMPONENT-321TANGUT COMPONENT-322TANGUT COMPONENT-323TANGUT COMPONENT-324TANGUT COMPONENT-325TANGUT COMPONENT-326TANGUT COMPONENT-327TANGUT COMPONENT-328TANGUT COMPONENT-329TANGUT COMPONENT-330TANGUT COMPONE\
    \NT-331TANGUT COMPONENT-332TANGUT COMPONENT-333TANGUT COMPONENT-334TANGUT COMPONENT-335TANGUT COMPONENT-336TANGUT COMPONENT-337TANGUT COMPONENT-338TANGUT COMPONENT-339TANGUT COMPONENT-340TANGUT COMPONENT-341TANGUT COMPONENT-342TANGUT COMPONENT-343TANGUT CO\
    \MPONENT-344TANGUT COMPONENT-345TANGUT COMPONENT-346TANGUT COMPONENT-347TANGUT COMPONENT-348TANGUT COMPONENT-349TANGUT COMPONENT-350TANGUT COMPONENT-351TANGUT COMPONENT-352TANGUT COMPONENT-353TANGUT COMPONENT-354TANGUT COMPONENT-355TANGUT COMPONENT-356TANG\
    \UT COMPONENT-357TANGUT COMPONENT-358TANGUT COMPONENT-359TANGUT COMPONENT-360TANGUT COMPONENT-361TANGUT COMPONENT-362TANGUT COMPONENT-363TANGUT COMPONENT-364TANGUT COMPONENT-365TANGUT COMPONENT-366TANGUT COMPONENT-367TANGUT COMPONENT-368TANGUT COMPONENT-36\
    \9TANGUT COMPONENT-370TANGUT COMPONENT-371TANGUT COMPONENT-372TANGUT COMPONENT-373TANGUT COMPONENT-374TANGUT COMPONENT-375TANGUT COMPONENT-376TANGUT COMPONENT-377TANGUT COMPONENT-378TANGUT COMPONENT-379TANGUT COMPONENT-380TANGUT COMPONENT-381TANGUT COMPONE\
    \NT-382TANGUT COMPONENT-383TANGUT COMPONENT-384TANGUT COMPONENT-385TANGUT COMPONENT-386TANGUT COMPONENT-387TANGUT COMPONENT-388TANGUT COMPONENT-389TANGUT COMPONENT-390TANGUT COMPONENT-391TANGUT COMPONENT-392TANGUT COMPONENT-393TANGUT COMPONENT-394TANGUT CO\
    \MPONENT-395TANGUT COMPONENT-396TANGUT COMPONENT-397TANGUT COMPONENT-398TANGUT COMPONENT-399TANGUT COMPONENT-400TANGUT COMPONENT-401TANGUT COMPONENT-402TANGUT COMPONENT-403TANGUT COMPONENT-404TANGUT COMPONENT-405TANGUT COMPONENT-406TANGUT COMPONENT-407TANG\
    \UT COMPONENT-408TANGUT COMPONENT-409TANGUT COMPONENT-410TANGUT COMPONENT-411TANGUT COMPONENT-412TANGUT COMPONENT-413TANGUT COMPONENT-414TANGUT COMPONENT-415TANGUT COMPONENT-416TANGUT COMPONENT-417TANGUT COMPONENT-418TANGUT COMPONENT-419TANGUT COMPONENT-42\
    \0TANGUT COMPONENT-421TANGUT COMPONENT-422TANGUT COMPONENT-423TANGUT COMPONENT-424TANGUT COMPONENT-425TANGUT COMPONENT-426TANGUT COMPONENT-427TANGUT COMPONENT-428TANGUT COMPONENT-429TANGUT COMPONENT-430TANGUT COMPONENT-431TANGUT COMPONENT-432TANGUT COMPONE\
    \NT-433TANGUT COMPONENT-434TANGUT COMPONENT-435TANGUT COMPONENT-436TANGUT COMPONENT-437TANGUT COMPONENT-438TANGUT COMPONENT-439TANGUT COMPONENT-440TANGUT COMPONENT-441TANGUT COMPONENT-442TANGUT COMPONENT-443TANGUT COMPONENT-444TANGUT COMPONENT-445TANGUT CO\
    \MPONENT-446TANGUT COMPONENT-447TANGUT COMPONENT-448TANGUT COMPONENT-449TANGUT COMPONENT-450TANGUT COMPONENT-451TANGUT COMPONENT-452TANGUT COMPONENT-453TANGUT COMPONENT-454TANGUT COMPONENT-455TANGUT COMPONENT-456TANGUT COMPONENT-457TANGUT COMPONENT-458TANG\
    \UT COMPONENT-459TANGUT COMPONENT-460TANGUT COMPONENT-461TANGUT COMPONENT-462TANGUT COMPONENT-463TANGUT COMPONENT-464TANGUT COMPONENT-465TANGUT COMPONENT-466TANGUT COMPONENT-467TANGUT COMPONENT-468TANGUT COMPONENT-469TANGUT COMPONENT-470TANGUT COMPONENT-47\
    \1TANGUT COMPONENT-472TANGUT COMPONENT-473TANGUT COMPONENT-474TANGUT COMPONENT-475TANGUT COMPONENT-476TANGUT COMPONENT-477TANGUT COMPONENT-478TANGUT COMPONENT-479TANGUT COMPONENT-480TANGUT COMPONENT-481TANGUT COMPONENT-482TANGUT COMPONENT-483TANGUT COMPONE\
    \NT-484TANGUT COMPONENT-485TANGUT COMPONENT-486TANGUT COMPONENT-487TANGUT COMPONENT-488TANGUT COMPONENT-489TANGUT COMPONENT-490TANGUT COMPONENT-491TANGUT COMPONENT-492TANGUT COMPONENT-493TANGUT COMPONENT-494TANGUT COMPONENT-495TANGUT COMPONENT-496TANGUT CO\
    \MPONENT-497TANGUT COMPONENT-498TANGUT COMPONENT-499TANGUT COMPONENT-500TANGUT COMPONENT-501TANGUT COMPONENT-502TANGUT COMPONENT-503TANGUT COMPONENT-504TANGUT COMPONENT-505TANGUT COMPONENT-506TANGUT COMPONENT-507TANGUT COMPONENT-508TANGUT COMPONENT-509TANG\
    \UT COMPONENT-510TANGUT COMPONENT-511TANGUT COMPONENT-512TANGUT COMPONENT-513TANGUT COMPONENT-514TANGUT COMPONENT-515TANGUT COMPONENT-516TANGUT COMPONENT-517TANGUT COMPONENT-518TANGUT COMPONENT-519TANGUT COMPONENT-520TANGUT COMPONENT-521TANGUT COMPONENT-52\
    \2TANGUT COMPONENT-523TANGUT COMPONENT-524TANGUT COMPONENT-525TANGUT COMPONENT-526TANGUT COMPONENT-527TANGUT COMPONENT-528TANGUT COMPONENT-529TANGUT COMPONENT-530TANGUT COMPONENT-531TANGUT COMPONENT-532TANGUT COMPONENT-533TANGUT COMPONENT-534TANGUT COMPONE\
    \NT-535TANGUT COMPONENT-536TANGUT COMPONENT-537TANGUT COMPONENT-538TANGUT COMPONENT-539TANGUT COMPONENT-540TANGUT COMPONENT-541TANGUT COMPONENT-542TANGUT COMPONENT-543TANGUT COMPONENT-544TANGUT COMPONENT-545TANGUT COMPONENT-546TANGUT COMPONENT-547TANGUT CO\
    \MPONENT-548TANGUT COMPONENT-549TANGUT COMPONENT-550TANGUT COMPONENT-551TANGUT COMPONENT-552TANGUT COMPONENT-553TANGUT COMPONENT-554TANGUT COMPONENT-555TANGUT COMPONENT-556TANGUT COMPONENT-557TANGUT COMPONENT-558TANGUT COMPONENT-559TANGUT COMPONENT-560TANG\
    \UT COMPONENT-561TANGUT COMPONENT-562TANGUT COMPONENT-563TANGUT COMPONENT-564TANGUT COMPONENT-565TANGUT COMPONENT-566TANGUT COMPONENT-567TANGUT COMPONENT-568TANGUT COMPONENT-569TANGUT COMPONENT-570TANGUT COMPONENT-571TANGUT COMPONENT-572TANGUT COMPONENT-57\
    \3TANGUT COMPONENT-574TANGUT COMPONENT-575TANGUT COMPONENT-576TANGUT COMPONENT-577TANGUT COMPONENT-578TANGUT COMPONENT-579TANGUT COMPONENT-580TANGUT COMPONENT-581TANGUT COMPONENT-582TANGUT COMPONENT-583TANGUT COMPONENT-584TANGUT COMPONENT-585TANGUT COMPONE\
    \NT-586TANGUT COMPONENT-587TANGUT COMPONENT-588TANGUT COMPONENT-589TANGUT COMPONENT-590TANGUT COMPONENT-591TANGUT COMPONENT-592TANGUT COMPONENT-593TANGUT COMPONENT-594TANGUT COMPONENT-595TANGUT COMPONENT-596TANGUT COMPONENT-597TANGUT COMPONENT-598TANGUT CO\
    \MPONENT-599TANGUT COMPONENT-600TANGUT COMPONENT-601TANGUT COMPONENT-602TANGUT COMPONENT-603TANGUT COMPONENT-604TANGUT COMPONENT-605TANGUT COMPONENT-606TANGUT COMPONENT-607TANGUT COMPONENT-608TANGUT COMPONENT-609TANGUT COMPONENT-610TANGUT COMPONENT-611TANG\
    \UT COMPONENT-612TANGUT COMPONENT-613TANGUT COMPONENT-614TANGUT COMPONENT-615TANGUT COMPONENT-616TANGUT COMPONENT-617TANGUT COMPONENT-618TANGUT COMPONENT-619TANGUT COMPONENT-620TANGUT COMPONENT-621TANGUT COMPONENT-622TANGUT COMPONENT-623TANGUT COMPONENT-62\
    \4TANGUT COMPONENT-625TANGUT COMPONENT-626TANGUT COMPONENT-627TANGUT COMPONENT-628TANGUT COMPONENT-629TANGUT COMPONENT-630TANGUT COMPONENT-631TANGUT COMPONENT-632TANGUT COMPONENT-633TANGUT COMPONENT-634TANGUT COMPONENT-635TANGUT COMPONENT-636TANGUT COMPONE\
    \NT-637TANGUT COMPONENT-638TANGUT COMPONENT-639TANGUT COMPONENT-640TANGUT COMPONENT-641TANGUT COMPONENT-642TANGUT COMPONENT-643TANGUT COMPONENT-644TANGUT COMPONENT-645TANGUT COMPONENT-646TANGUT COMPONENT-647TANGUT COMPONENT-648TANGUT COMPONENT-649TANGUT CO\
    \MPONENT-650TANGUT COMPONENT-651TANGUT COMPONENT-652TANGUT COMPONENT-653TANGUT COMPONENT-654TANGUT COMPONENT-655TANGUT COMPONENT-656TANGUT COMPONENT-657TANGUT COMPONENT-658TANGUT COMPONENT-659TANGUT COMPONENT-660TANGUT COMPONENT-661TANGUT COMPONENT-662TANG\
    \UT COMPONENT-663TANGUT COMPONENT-664TANGUT COMPONENT-665TANGUT COMPONENT-666TANGUT COMPONENT-667TANGUT COMPONENT-668TANGUT COMPONENT-669TANGUT COMPONENT-670TANGUT COMPONENT-671TANGUT COMPONENT-672TANGUT COMPONENT-673TANGUT COMPONENT-674TANGUT COMPONENT-67\
    \5TANGUT COMPONENT-676TANGUT COMPONENT-677TANGUT COMPONENT-678TANGUT COMPONENT-679TANGUT COMPONENT-680TANGUT COMPONENT-681TANGUT COMPONENT-682TANGUT COMPONENT-683TANGUT COMPONENT-684TANGUT COMPONENT-685TANGUT COMPONENT-686TANGUT COMPONENT-687TANGUT COMPONE\
    \NT-688TANGUT COMPONENT-689TANGUT COMPONENT-690TANGUT COMPONENT-691TANGUT COMPONENT-692TANGUT COMPONENT-693TANGUT COMPONENT-694TANGUT COMPONENT-695TANGUT COMPONENT-696TANGUT COMPONENT-697TANGUT COMPONENT-698TANGUT COMPONENT-699TANGUT COMPONENT-700TANGUT CO\
    \MPONENT-701TANGUT COMPONENT-702TANGUT COMPONENT-703TANGUT COMPONENT-704TANGUT COMPONENT-705TANGUT COMPONENT-706TANGUT COMPONENT-707TANGUT COMPONENT-708TANGUT COMPONENT-709TANGUT COMPONENT-710TANGUT COMPONENT-711TANGUT COMPONENT-712TANGUT COMPONENT-713TANG\
    \UT COMPONENT-714TANGUT COMPONENT-715TANGUT COMPONENT-716TANGUT COMPONENT-717TANGUT COMPONENT-718TANGUT COMPONENT-719TANGUT COMPONENT-720TANGUT COMPONENT-721TANGUT COMPONENT-722TANGUT COMPONENT-723TANGUT COMPONENT-724TANGUT COMPONENT-725TANGUT COMPONENT-72\
    \6TANGUT COMPONENT-727TANGUT COMPONENT-728TANGUT COMPONENT-729TANGUT COMPONENT-730TANGUT COMPONENT-731TANGUT COMPONENT-732TANGUT COMPONENT-733TANGUT COMPONENT-734TANGUT COMPONENT-735TANGUT COMPONENT-736TANGUT COMPONENT-737TANGUT COMPONENT-738TANGUT COMPONE\
    \NT-739TANGUT COMPONENT-740TANGUT COMPONENT-741TANGUT COMPONENT-742TANGUT COMPONENT-743TANGUT COMPONENT-744TANGUT COMPONENT-745TANGUT COMPONENT-746TANGUT COMPONENT-747TANGUT COMPONENT-748TANGUT COMPONENT-749TANGUT COMPONENT-750TANGUT COMPONENT-751TANGUT CO\
    \MPONENT-752TANGUT COMPONENT-753TANGUT COMPONENT-754TANGUT COMPONENT-755TANGUT COMPONENT-756TANGUT COMPONENT-757TANGUT COMPONENT-758TANGUT COMPONENT-759TANGUT COMPONENT-760TANGUT COMPONENT-761TANGUT COMPONENT-762TANGUT COMPONENT-763TANGUT COMPONENT-764TANG\
    \UT COMPONENT-765TANGUT COMPONENT-766TANGUT COMPONENT-767TANGUT COMPONENT-768KHITAN SMALL SCRIPT CHARACTER-18B00KHITAN SMALL SCRIPT CHARACTER-18B01KHITAN SMALL SCRIPT CHARACTER-18B02KHITAN SMALL SCRIPT CHARACTER-18B03KHITAN SMALL SCRIPT CHARACTER-18B04KHIT\
    \AN SMALL SCRIPT CHARACTER-18B05KHITAN SMALL SCRIPT CHARACTER-18B06KHITAN SMALL SCRIPT CHARACTER-18B07KHITAN SMALL SCRIPT CHARACTER-18B08KHITAN SMALL SCRIPT CHARACTER-18B09KHITAN SMALL SCRIPT CHARACTER-18B0AKHITAN SMALL SCRIPT CHARACTER-18B0BKHITAN SMALL S\
    \CRIPT CHARACTER-18B0CKHITAN SMALL SCRIPT CHARACTER-18B0DKHITAN SMALL SCRIPT CHARACTER-18B0EKHITAN SMALL SCRIPT CHARACTER-18B0FKHITAN SMALL SCRIPT CHARACTER-18B10KHITAN SMALL SCRIPT CHARACTER-18B11KHITAN SMALL SCRIPT CHARACTER-18B12KHITAN SMALL SCRIPT CHAR\
    \ACTER-18B13KHITAN SMALL SCRIPT CHARACTER-18B14KHITAN SMALL SCRIPT CHARACTER-18B15KHITAN SMALL SCRIPT CHARACTER-18B16KHITAN SMALL SCRIPT CHARACTER-18B17KHITAN SMALL SCRIPT CHARACTER-18B18KHITAN SMALL SCRIPT CHARACTER-18B19KHITAN SMALL SCRIPT CHARACTER-18B1\
    \AKHITAN SMALL SCRIPT CHARACTER-18B1BKHITAN SMALL SCRIPT CHARACTER-18B1CKHITAN SMALL SCRIPT CHARACTER-18B1DKHITAN SMALL SCRIPT CHARACTER-18B1EKHITAN SMALL SCRIPT CHARACTER-18B1FKHITAN SMALL SCRIPT CHARACTER-18B20KHITAN SMALL SCRIPT CHARACTER-18B21KHITAN SM\
    \ALL SCRIPT CHARACTER-18B22KHITAN SMALL SCRIPT CHARACTER-18B23KHITAN SMALL SCRIPT CHARACTER-18B24KHITAN SMALL SCRIPT CHARACTER-18B25KHITAN SMALL SCRIPT CHARACTER-18B26KHITAN SMALL SCRIPT CHARACTER-18B27KHITAN SMALL SCRIPT CHARACTER-18B28KHITAN SMALL SCRIPT\
    \ CHARACTER-18B29KHITAN SMALL SCRIPT CHARACTER-18B2AKHITAN SMALL SCRIPT CHARACTER-18B2BKHITAN SMALL SCRIPT CHARACTER-18B2CKHITAN SMALL SCRIPT CHARACTER-18B2DKHITAN SMALL SCRIPT CHARACTER-18B2EKHITAN SMALL SCRIPT CHARACTER-18B2FKHITAN SMALL SCRIPT CHARACTER\
    \-18B30KHITAN SMALL SCRIPT CHARACTER-18B31KHITAN SMALL SCRIPT CHARACTER-18B32KHITAN SMALL SCRIPT CHARACTER-18B33KHITAN SMALL SCRIPT CHARACTER-18B34KHITAN SMALL SCRIPT CHARACTER-18B35KHITAN SMALL SCRIPT CHARACTER-18B36KHITAN SMALL SCRIPT CHARACTER-18B37KHIT\
    \AN SMALL SCRIPT CHARACTER-18B38KHITAN SMALL SCRIPT CHARACTER-18B39KHITAN SMALL SCRIPT CHARACTER-18B3AKHITAN SMALL SCRIPT CHARACTER-18B3BKHITAN SMALL SCRIPT CHARACTER-18B3CKHITAN SMALL SCRIPT CHARACTER-18B3DKHITAN SMALL SCRIPT CHARACTER-18B3EKHITAN SMALL S\
    \CRIPT CHARACTER-18B3FKHITAN SMALL SCRIPT CHARACTER-18B40KHITAN SMALL SCRIPT CHARACTER-18B41KHITAN SMALL SCRIPT CHARACTER-18B42KHITAN SMALL SCRIPT CHARACTER-18B43KHITAN SMALL SCRIPT CHARACTER-18B44KHITAN SMALL SCRIPT CHARACTER-18B45KHITAN SMALL SCRIPT CHAR\
    \ACTER-18B46KHITAN SMALL SCRIPT CHARACTER-18B47KHITAN SMALL SCRIPT CHARACTER-18B48KHITAN SMALL SCRIPT CHARACTER-18B49KHITAN SMALL SCRIPT CHARACTER-18B4AKHITAN SMALL SCRIPT CHARACTER-18B4BKHITAN SMALL SCRIPT CHARACTER-18B4CKHITAN SMALL SCRIPT CHARACTER-18B4\
    \DKHITAN SMALL SCRIPT CHARACTER-18B4EKHITAN SMALL SCRIPT CHARACTER-18B4FKHITAN SMALL SCRIPT CHARACTER-18B50KHITAN SMALL SCRIPT CHARACTER-18B51KHITAN SMALL SCRIPT CHARACTER-18B52KHITAN SMALL SCRIPT CHARACTER-18B53KHITAN SMALL SCRIPT CHARACTER-18B54KHITAN SM\
    \ALL SCRIPT CHARACTER-18B55KHITAN SMALL SCRIPT CHARACTER-18B56KHITAN SMALL SCRIPT CHARACTER-18B57KHITAN SMALL SCRIPT CHARACTER-18B58KHITAN SMALL SCRIPT CHARACTER-18B59KHITAN SMALL SCRIPT CHARACTER-18B5AKHITAN SMALL SCRIPT CHARACTER-18B5BKHITAN SMALL SCRIPT\
    \ CHARACTER-18B5CKHITAN SMALL SCRIPT CHARACTER-18B5DKHITAN SMALL SCRIPT CHARACTER-18B5EKHITAN SMALL SCRIPT CHARACTER-18B5FKHITAN SMALL SCRIPT CHARACTER-18B60KHITAN SMALL SCRIPT CHARACTER-18B61KHITAN SMALL SCRIPT CHARACTER-18B62KHITAN SMALL SCRIPT CHARACTER\
    \-18B63KHITAN SMALL SCRIPT CHARACTER-18B64KHITAN SMALL SCRIPT CHARACTER-18B65KHITAN SMALL SCRIPT CHARACTER-18B66KHITAN SMALL SCRIPT CHARACTER-18B67KHITAN SMALL SCRIPT CHARACTER-18B68KHITAN SMALL SCRIPT CHARACTER-18B69KHITAN SMALL SCRIPT CHARACTER-18B6AKHIT\
    \AN SMALL SCRIPT CHARACTER-18B6BKHITAN SMALL SCRIPT CHARACTER-18B6CKHITAN SMALL SCRIPT CHARACTER-18B6DKHITAN SMALL SCRIPT CHARACTER-18B6EKHITAN SMALL SCRIPT CHARACTER-18B6FKHITAN SMALL SCRIPT CHARACTER-18B70KHITAN SMALL SCRIPT CHARACTER-18B71KHITAN SMALL S\
    \CRIPT CHARACTER-18B72KHITAN SMALL SCRIPT CHARACTER-18B73KHITAN SMALL SCRIPT CHARACTER-18B74KHITAN SMALL SCRIPT CHARACTER-18B75KHITAN SMALL SCRIPT CHARACTER-18B76KHITAN SMALL SCRIPT CHARACTER-18B77KHITAN SMALL SCRIPT CHARACTER-18B78KHITAN SMALL SCRIPT CHAR\
    \ACTER-18B79KHITAN SMALL SCRIPT CHARACTER-18B7AKHITAN SMALL SCRIPT CHARACTER-18B7BKHITAN SMALL SCRIPT CHARACTER-18B7CKHITAN SMALL SCRIPT CHARACTER-18B7DKHITAN SMALL SCRIPT CHARACTER-18B7EKHITAN SMALL SCRIPT CHARACTER-18B7FKHITAN SMALL SCRIPT CHARACTER-18B8\
    \0KHITAN SMALL SCRIPT CHARACTER-18B81KHITAN SMALL SCRIPT CHARACTER-18B82KHITAN SMALL SCRIPT CHARACTER-18B83KHITAN SMALL SCRIPT CHARACTER-18B84KHITAN SMALL SCRIPT CHARACTER-18B85KHITAN SMALL SCRIPT CHARACTER-18B86KHITAN SMALL SCRIPT CHARACTER-18B87KHITAN SM\
    \ALL SCRIPT CHARACTER-18B88KHITAN SMALL SCRIPT CHARACTER-18B89KHITAN SMALL SCRIPT CHARACTER-18B8AKHITAN SMALL SCRIPT CHARACTER-18B8BKHITAN SMALL SCRIPT CHARACTER-18B8CKHITAN SMALL SCRIPT CHARACTER-18B8DKHITAN SMALL SCRIPT CHARACTER-18B8EKHITAN SMALL SCRIPT\
    \ CHARACTER-18B8FKHITAN SMALL SCRIPT CHARACTER-18B90KHITAN SMALL SCRIPT CHARACTER-18B91KHITAN SMALL SCRIPT CHARACTER-18B92KHITAN SMALL SCRIPT CHARACTER-18B93KHITAN SMALL SCRIPT CHARACTER-18B94KHITAN SMALL SCRIPT CHARACTER-18B95KHITAN SMALL SCRIPT CHARACTER\
    \-18B96KHITAN SMALL SCRIPT CHARACTER-18B97KHITAN SMALL SCRIPT CHARACTER-18B98KHITAN SMALL SCRIPT CHARACTER-18B99KHITAN SMALL SCRIPT CHARACTER-18B9AKHITAN SMALL SCRIPT CHARACTER-18B9BKHITAN SMALL SCRIPT CHARACTER-18B9CKHITAN SMALL SCRIPT CHARACTER-18B9DKHIT\
    \AN SMALL SCRIPT CHARACTER-18B9EKHITAN SMALL SCRIPT CHARACTER-18B9FKHITAN SMALL SCRIPT CHARACTER-18BA0KHITAN SMALL SCRIPT CHARACTER-18BA1KHITAN SMALL SCRIPT CHARACTER-18BA2KHITAN SMALL SCRIPT CHARACTER-18BA3KHITAN SMALL SCRIPT CHARACTER-18BA4KHITAN SMALL S\
    \CRIPT CHARACTER-18BA5KHITAN SMALL SCRIPT CHARACTER-18BA6KHITAN SMALL SCRIPT CHARACTER-18BA7KHITAN SMALL SCRIPT CHARACTER-18BA8KHITAN SMALL SCRIPT CHARACTER-18BA9KHITAN SMALL SCRIPT CHARACTER-18BAAKHITAN SMALL SCRIPT CHARACTER-18BABKHITAN SMALL SCRIPT CHAR\
    \ACTER-18BACKHITAN SMALL SCRIPT CHARACTER-18BADKHITAN SMALL SCRIPT CHARACTER-18BAEKHITAN SMALL SCRIPT CHARACTER-18BAFKHITAN SMALL SCRIPT CHARACTER-18BB0KHITAN SMALL SCRIPT CHARACTER-18BB1KHITAN SMALL SCRIPT CHARACTER-18BB2KHITAN SMALL SCRIPT CHARACTER-18BB\
    \3KHITAN SMALL SCRIPT CHARACTER-18BB4KHITAN SMALL SCRIPT CHARACTER-18BB5KHITAN SMALL SCRIPT CHARACTER-18BB6KHITAN SMALL SCRIPT CHARACTER-18BB7KHITAN SMALL SCRIPT CHARACTER-18BB8KHITAN SMALL SCRIPT CHARACTER-18BB9KHITAN SMALL SCRIPT CHARACTER-18BBAKHITAN SM\
    \ALL SCRIPT CHARACTER-18BBBKHITAN SMALL SCRIPT CHARACTER-18BBCKHITAN SMALL SCRIPT CHARACTER-18BBDKHITAN SMALL SCRIPT CHARACTER-18BBEKHITAN SMALL SCRIPT CHARACTER-18BBFKHITAN SMALL SCRIPT CHARACTER-18BC0KHITAN SMALL SCRIPT CHARACTER-18BC1KHITAN SMALL SCRIPT\
    \ CHARACTER-18BC2KHITAN SMALL SCRIPT CHARACTER-18BC3KHITAN SMALL SCRIPT CHARACTER-18BC4KHITAN SMALL SCRIPT CHARACTER-18BC5KHITAN SMALL SCRIPT CHARACTER-18BC6KHITAN SMALL SCRIPT CHARACTER-18BC7KHITAN SMALL SCRIPT CHARACTER-18BC8KHITAN SMALL SCRIPT CHARACTER\
    \-18BC9KHITAN SMALL SCRIPT CHARACTER-18BCAKHITAN SMALL SCRIPT CHARACTER-18BCBKHITAN SMALL SCRIPT CHARACTER-18BCCKHITAN SMALL SCRIPT CHARACTER-18BCDKHITAN SMALL SCRIPT CHARACTER-18BCEKHITAN SMALL SCRIPT CHARACTER-18BCFKHITAN SMALL SCRIPT CHARACTER-18BD0KHIT\
    \AN SMALL SCRIPT CHARACTER-18BD1KHITAN SMALL SCRIPT CHARACTER-18BD2KHITAN SMALL SCRIPT CHARACTER-18BD3KHITAN SMALL SCRIPT CHARACTER-18BD4KHITAN SMALL SCRIPT CHARACTER-18BD5KHITAN SMALL SCRIPT CHARACTER-18BD6KHITAN SMALL SCRIPT CHARACTER-18BD7KHITAN SMALL S\
    \CRIPT CHARACTER-18BD8KHITAN SMALL SCRIPT CHARACTER-18BD9KHITAN SMALL SCRIPT CHARACTER-18BDAKHITAN SMALL SCRIPT CHARACTER-18BDBKHITAN SMALL SCRIPT CHARACTER-18BDCKHITAN SMALL SCRIPT CHARACTER-18BDDKHITAN SMALL SCRIPT CHARACTER-18BDEKHITAN SMALL SCRIPT CHAR\
    \ACTER-18BDFKHITAN SMALL SCRIPT CHARACTER-18BE0KHITAN SMALL SCRIPT CHARACTER-18BE1KHITAN SMALL SCRIPT CHARACTER-18BE2KHITAN SMALL SCRIPT CHARACTER-18BE3KHITAN SMALL SCRIPT CHARACTER-18BE4KHITAN SMALL SCRIPT CHARACTER-18BE5KHITAN SMALL SCRIPT CHARACTER-18BE\
    \6KHITAN SMALL SCRIPT CHARACTER-18BE7KHITAN SMALL SCRIPT CHARACTER-18BE8KHITAN SMALL SCRIPT CHARACTER-18BE9KHITAN SMALL SCRIPT CHARACTER-18BEAKHITAN SMALL SCRIPT CHARACTER-18BEBKHITAN SMALL SCRIPT CHARACTER-18BECKHITAN SMALL SCRIPT CHARACTER-18BEDKHITAN SM\
    \ALL SCRIPT CHARACTER-18BEEKHITAN SMALL SCRIPT CHARACTER-18BEFKHITAN SMALL SCRIPT CHARACTER-18BF0KHITAN SMALL SCRIPT CHARACTER-18BF1KHITAN SMALL SCRIPT CHARACTER-18BF2KHITAN SMALL SCRIPT CHARACTER-18BF3KHITAN SMALL SCRIPT CHARACTER-18BF4KHITAN SMALL SCRIPT\
    \ CHARACTER-18BF5KHITAN SMALL SCRIPT CHARACTER-18BF6KHITAN SMALL SCRIPT CHARACTER-18BF7KHITAN SMALL SCRIPT CHARACTER-18BF8KHITAN SMALL SCRIPT CHARACTER-18BF9KHITAN SMALL SCRIPT CHARACTER-18BFAKHITAN SMALL SCRIPT CHARACTER-18BFBKHITAN SMALL SCRIPT CHARACTER\
    \-18BFCKHITAN SMALL SCRIPT CHARACTER-18BFDKHITAN SMALL SCRIPT CHARACTER-18BFEKHITAN SMALL SCRIPT CHARACTER-18BFFKHITAN SMALL SCRIPT CHARACTER-18C00KHITAN SMALL SCRIPT CHARACTER-18C01KHITAN SMALL SCRIPT CHARACTER-18C02KHITAN SMALL SCRIPT CHARACTER-18C03KHIT\
    \AN SMALL SCRIPT CHARACTER-18C04KHITAN SMALL SCRIPT CHARACTER-18C05KHITAN SMALL SCRIPT CHARACTER-18C06KHITAN SMALL SCRIPT CHARACTER-18C07KHITAN SMALL SCRIPT CHARACTER-18C08KHITAN SMALL SCRIPT CHARACTER-18C09KHITAN SMALL SCRIPT CHARACTER-18C0AKHITAN SMALL S\
    \CRIPT CHARACTER-18C0BKHITAN SMALL SCRIPT CHARACTER-18C0CKHITAN SMALL SCRIPT CHARACTER-18C0DKHITAN SMALL SCRIPT CHARACTER-18C0EKHITAN SMALL SCRIPT CHARACTER-18C0FKHITAN SMALL SCRIPT CHARACTER-18C10KHITAN SMALL SCRIPT CHARACTER-18C11KHITAN SMALL SCRIPT CHAR\
    \ACTER-18C12KHITAN SMALL SCRIPT CHARACTER-18C13KHITAN SMALL SCRIPT CHARACTER-18C14KHITAN SMALL SCRIPT CHARACTER-18C15KHITAN SMALL SCRIPT CHARACTER-18C16KHITAN SMALL SCRIPT CHARACTER-18C17KHITAN SMALL SCRIPT CHARACTER-18C18KHITAN SMALL SCRIPT CHARACTER-18C1\
    \9KHITAN SMALL SCRIPT CHARACTER-18C1AKHITAN SMALL SCRIPT CHARACTER-18C1BKHITAN SMALL SCRIPT CHARACTER-18C1CKHITAN SMALL SCRIPT CHARACTER-18C1DKHITAN SMALL SCRIPT CHARACTER-18C1EKHITAN SMALL SCRIPT CHARACTER-18C1FKHITAN SMALL SCRIPT CHARACTER-18C20KHITAN SM\
    \ALL SCRIPT CHARACTER-18C21KHITAN SMALL SCRIPT CHARACTER-18C22KHITAN SMALL SCRIPT CHARACTER-18C23KHITAN SMALL SCRIPT CHARACTER-18C24KHITAN SMALL SCRIPT CHARACTER-18C25KHITAN SMALL SCRIPT CHARACTER-18C26KHITAN SMALL SCRIPT CHARACTER-18C27KHITAN SMALL SCRIPT\
    \ CHARACTER-18C28KHITAN SMALL SCRIPT CHARACTER-18C29KHITAN SMALL SCRIPT CHARACTER-18C2AKHITAN SMALL SCRIPT CHARACTER-18C2BKHITAN SMALL SCRIPT CHARACTER-18C2CKHITAN SMALL SCRIPT CHARACTER-18C2DKHITAN SMALL SCRIPT CHARACTER-18C2EKHITAN SMALL SCRIPT CHARACTER\
    \-18C2FKHITAN SMALL SCRIPT CHARACTER-18C30KHITAN SMALL SCRIPT CHARACTER-18C31KHITAN SMALL SCRIPT CHARACTER-18C32KHITAN SMALL SCRIPT CHARACTER-18C33KHITAN SMALL SCRIPT CHARACTER-18C34KHITAN SMALL SCRIPT CHARACTER-18C35KHITAN SMALL SCRIPT CHARACTER-18C36KHIT\
    \AN SMALL SCRIPT CHARACTER-18C37KHITAN SMALL SCRIPT CHARACTER-18C38KHITAN SMALL SCRIPT CHARACTER-18C39KHITAN SMALL SCRIPT CHARACTER-18C3AKHITAN SMALL SCRIPT CHARACTER-18C3BKHITAN SMALL SCRIPT CHARACTER-18C3CKHITAN SMALL SCRIPT CHARACTER-18C3DKHITAN SMALL S\
    \CRIPT CHARACTER-18C3EKHITAN SMALL SCRIPT CHARACTER-18C3FKHITAN SMALL SCRIPT CHARACTER-18C40KHITAN SMALL SCRIPT CHARACTER-18C41KHITAN SMALL SCRIPT CHARACTER-18C42KHITAN SMALL SCRIPT CHARACTER-18C43KHITAN SMALL SCRIPT CHARACTER-18C44KHITAN SMALL SCRIPT CHAR\
    \ACTER-18C45KHITAN SMALL SCRIPT CHARACTER-18C46KHITAN SMALL SCRIPT CHARACTER-18C47KHITAN SMALL SCRIPT CHARACTER-18C48KHITAN SMALL SCRIPT CHARACTER-18C49KHITAN SMALL SCRIPT CHARACTER-18C4AKHITAN SMALL SCRIPT CHARACTER-18C4BKHITAN SMALL SCRIPT CHARACTER-18C4\
    \CKHITAN SMALL SCRIPT CHARACTER-18C4DKHITAN SMALL SCRIPT CHARACTER-18C4EKHITAN SMALL SCRIPT CHARACTER-18C4FKHITAN SMALL SCRIPT CHARACTER-18C50KHITAN SMALL SCRIPT CHARACTER-18C51KHITAN SMALL SCRIPT CHARACTER-18C52KHITAN SMALL SCRIPT CHARACTER-18C53KHITAN SM\
    \ALL SCRIPT CHARACTER-18C54KHITAN SMALL SCRIPT CHARACTER-18C55KHITAN SMALL SCRIPT CHARACTER-18C56KHITAN SMALL SCRIPT CHARACTER-18C57KHITAN SMALL SCRIPT CHARACTER-18C58KHITAN SMALL SCRIPT CHARACTER-18C59KHITAN SMALL SCRIPT CHARACTER-18C5AKHITAN SMALL SCRIPT\
    \ CHARACTER-18C5BKHITAN SMALL SCRIPT CHARACTER-18C5CKHITAN SMALL SCRIPT CHARACTER-18C5DKHITAN SMALL SCRIPT CHARACTER-18C5EKHITAN SMALL SCRIPT CHARACTER-18C5FKHITAN SMALL SCRIPT CHARACTER-18C60KHITAN SMALL SCRIPT CHARACTER-18C61KHITAN SMALL SCRIPT CHARACTER\
    \-18C62KHITAN SMALL SCRIPT CHARACTER-18C63KHITAN SMALL SCRIPT CHARACTER-18C64KHITAN SMALL SCRIPT CHARACTER-18C65KHITAN SMALL SCRIPT CHARACTER-18C66KHITAN SMALL SCRIPT CHARACTER-18C67KHITAN SMALL SCRIPT CHARACTER-18C68KHITAN SMALL SCRIPT CHARACTER-18C69KHIT\
    \AN SMALL SCRIPT CHARACTER-18C6AKHITAN SMALL SCRIPT CHARACTER-18C6BKHITAN SMALL SCRIPT CHARACTER-18C6CKHITAN SMALL SCRIPT CHARACTER-18C6DKHITAN SMALL SCRIPT CHARACTER-18C6EKHITAN SMALL SCRIPT CHARACTER-18C6FKHITAN SMALL SCRIPT CHARACTER-18C70KHITAN SMALL S\
    \CRIPT CHARACTER-18C71KHITAN SMALL SCRIPT CHARACTER-18C72KHITAN SMALL SCRIPT CHARACTER-18C73KHITAN SMALL SCRIPT CHARACTER-18C74KHITAN SMALL SCRIPT CHARACTER-18C75KHITAN SMALL SCRIPT CHARACTER-18C76KHITAN SMALL SCRIPT CHARACTER-18C77KHITAN SMALL SCRIPT CHAR\
    \ACTER-18C78KHITAN SMALL SCRIPT CHARACTER-18C79KHITAN SMALL SCRIPT CHARACTER-18C7AKHITAN SMALL SCRIPT CHARACTER-18C7BKHITAN SMALL SCRIPT CHARACTER-18C7CKHITAN SMALL SCRIPT CHARACTER-18C7DKHITAN SMALL SCRIPT CHARACTER-18C7EKHITAN SMALL SCRIPT CHARACTER-18C7\
    \FKHITAN SMALL SCRIPT CHARACTER-18C80KHITAN SMALL SCRIPT CHARACTER-18C81KHITAN SMALL SCRIPT CHARACTER-18C82KHITAN SMALL SCRIPT CHARACTER-18C83KHITAN SMALL SCRIPT CHARACTER-18C84KHITAN SMALL SCRIPT CHARACTER-18C85KHITAN SMALL SCRIPT CHARACTER-18C86KHITAN SM\
    \ALL SCRIPT CHARACTER-18C87KHITAN SMALL SCRIPT CHARACTER-18C88KHITAN SMALL SCRIPT CHARACTER-18C89KHITAN SMALL SCRIPT CHARACTER-18C8AKHITAN SMALL SCRIPT CHARACTER-18C8BKHITAN SMALL SCRIPT CHARACTER-18C8CKHITAN SMALL SCRIPT CHARACTER-18C8DKHITAN SMALL SCRIPT\
    \ CHARACTER-18C8EKHITAN SMALL SCRIPT CHARACTER-18C8FKHITAN SMALL SCRIPT CHARACTER-18C90KHITAN SMALL SCRIPT CHARACTER-18C91KHITAN SMALL SCRIPT CHARACTER-18C92KHITAN SMALL SCRIPT CHARACTER-18C93KHITAN SMALL SCRIPT CHARACTER-18C94KHITAN SMALL SCRIPT CHARACTER\
    \-18C95KHITAN SMALL SCRIPT CHARACTER-18C96KHITAN SMALL SCRIPT CHARACTER-18C97KHITAN SMALL SCRIPT CHARACTER-18C98KHITAN SMALL SCRIPT CHARACTER-18C99KHITAN SMALL SCRIPT CHARACTER-18C9AKHITAN SMALL SCRIPT CHARACTER-18C9BKHITAN SMALL SCRIPT CHARACTER-18C9CKHIT\
    \AN SMALL SCRIPT CHARACTER-18C9DKHITAN SMALL SCRIPT CHARACTER-18C9EKHITAN SMALL SCRIPT CHARACTER-18C9FKHITAN SMALL SCRIPT CHARACTER-18CA0KHITAN SMALL SCRIPT CHARACTER-18CA1KHITAN SMALL SCRIPT CHARACTER-18CA2KHITAN SMALL SCRIPT CHARACTER-18CA3KHITAN SMALL S\
    \CRIPT CHARACTER-18CA4KHITAN SMALL SCRIPT CHARACTER-18CA5KHITAN SMALL SCRIPT CHARACTER-18CA6KHITAN SMALL SCRIPT CHARACTER-18CA7KHITAN SMALL SCRIPT CHARACTER-18CA8KHITAN SMALL SCRIPT CHARACTER-18CA9KHITAN SMALL SCRIPT CHARACTER-18CAAKHITAN SMALL SCRIPT CHAR\
    \ACTER-18CABKHITAN SMALL SCRIPT CHARACTER-18CACKHITAN SMALL SCRIPT CHARACTER-18CADKHITAN SMALL SCRIPT CHARACTER-18CAEKHITAN SMALL SCRIPT CHARACTER-18CAFKHITAN SMALL SCRIPT CHARACTER-18CB0KHITAN SMALL SCRIPT CHARACTER-18CB1KHITAN SMALL SCRIPT CHARACTER-18CB\
    \2KHITAN SMALL SCRIPT CHARACTER-18CB3KHITAN SMALL SCRIPT CHARACTER-18CB4KHITAN SMALL SCRIPT CHARACTER-18CB5KHITAN SMALL SCRIPT CHARACTER-18CB6KHITAN SMALL SCRIPT CHARACTER-18CB7KHITAN SMALL SCRIPT CHARACTER-18CB8KHITAN SMALL SCRIPT CHARACTER-18CB9KHITAN SM\
    \ALL SCRIPT CHARACTER-18CBAKHITAN SMALL SCRIPT CHARACTER-18CBBKHITAN SMALL SCRIPT CHARACTER-18CBCKHITAN SMALL SCRIPT CHARACTER-18CBDKHITAN SMALL SCRIPT CHARACTER-18CBEKHITAN SMALL SCRIPT CHARACTER-18CBFKHITAN SMALL SCRIPT CHARACTER-18CC0KHITAN SMALL SCRIPT\
    \ CHARACTER-18CC1KHITAN SMALL SCRIPT CHARACTER-18CC2KHITAN SMALL SCRIPT CHARACTER-18CC3KHITAN SMALL SCRIPT CHARACTER-18CC4KHITAN SMALL SCRIPT CHARACTER-18CC5KHITAN SMALL SCRIPT CHARACTER-18CC6KHITAN SMALL SCRIPT CHARACTER-18CC7KHITAN SMALL SCRIPT CHARACTER\
    \-18CC8KHITAN SMALL SCRIPT CHARACTER-18CC9KHITAN SMALL SCRIPT CHARACTER-18CCAKHITAN SMALL SCRIPT CHARACTER-18CCBKHITAN SMALL SCRIPT CHARACTER-18CCCKHITAN SMALL SCRIPT CHARACTER-18CCDKHITAN SMALL SCRIPT CHARACTER-18CCEKHITAN SMALL SCRIPT CHARACTER-18CCFKHIT\
    \AN SMALL SCRIPT CHARACTER-18CD0KHITAN SMALL SCRIPT CHARACTER-18CD1KHITAN SMALL SCRIPT CHARACTER-18CD2KHITAN SMALL SCRIPT CHARACTER-18CD3KHITAN SMALL SCRIPT CHARACTER-18CD4KHITAN SMALL SCRIPT CHARACTER-18CD5KATAKANA LETTER MINNAN TONE-2KATAKANA LETTER MINN\
    \AN TONE-3KATAKANA LETTER MINNAN TONE-4KATAKANA LETTER MINNAN TONE-5KATAKANA LETTER MINNAN TONE-7KATAKANA LETTER MINNAN TONE-8KATAKANA LETTER MINNAN NASALIZED TONE-1KATAKANA LETTER MINNAN NASALIZED TONE-2KATAKANA LETTER MINNAN NASALIZED TONE-3KATAKANA LETT\
    \ER MINNAN NASALIZED TONE-4KATAKANA LETTER MINNAN NASALIZED TONE-5KATAKANA LETTER MINNAN NASALIZED TONE-7KATAKANA LETTER MINNAN NASALIZED TONE-8KATAKANA LETTER ARCHAIC EHIRAGANA LETTER ARCHAIC YEHENTAIGANA LETTER A-1HENTAIGANA LETTER A-2HENTAIGANA LETTER A\
    \-3HENTAIGANA LETTER A-WOHENTAIGANA LETTER I-1HENTAIGANA LETTER I-2HENTAIGANA LETTER I-3HENTAIGANA LETTER I-4HENTAIGANA LETTER U-1HENTAIGANA LETTER U-2HENTAIGANA LETTER U-3HENTAIGANA LETTER U-4HENTAIGANA LETTER U-5HENTAIGANA LETTER E-2HENTAIGANA LETTER E-3\
    \HENTAIGANA LETTER E-4HENTAIGANA LETTER E-5HENTAIGANA LETTER E-6HENTAIGANA LETTER O-1HENTAIGANA LETTER O-2HENTAIGANA LETTER O-3HENTAIGANA LETTER KA-1HENTAIGANA LETTER KA-2HENTAIGANA LETTER KA-3HENTAIGANA LETTER KA-4HENTAIGANA LETTER KA-5HENTAIGANA LETTER K\
    \A-6HENTAIGANA LETTER KA-7HENTAIGANA LETTER KA-8HENTAIGANA LETTER KA-9HENTAIGANA LETTER KA-10HENTAIGANA LETTER KA-11HENTAIGANA LETTER KA-KEHENTAIGANA LETTER KI-1HENTAIGANA LETTER KI-2HENTAIGANA LETTER KI-3HENTAIGANA LETTER KI-4HENTAIGANA LETTER KI-5HENTAIG\
    \ANA LETTER KI-6HENTAIGANA LETTER KI-7HENTAIGANA LETTER KI-8HENTAIGANA LETTER KU-1HENTAIGANA LETTER KU-2HENTAIGANA LETTER KU-3HENTAIGANA LETTER KU-4HENTAIGANA LETTER KU-5HENTAIGANA LETTER KU-6HENTAIGANA LETTER KU-7HENTAIGANA LETTER KE-1HENTAIGANA LETTER KE\
    \-2HENTAIGANA LETTER KE-3HENTAIGANA LETTER KE-4HENTAIGANA LETTER KE-5HENTAIGANA LETTER KE-6HENTAIGANA LETTER KO-1HENTAIGANA LETTER KO-2HENTAIGANA LETTER KO-3HENTAIGANA LETTER KO-KIHENTAIGANA LETTER SA-1HENTAIGANA LETTER SA-2HENTAIGANA LETTER SA-3HENTAIGANA\
    \ LETTER SA-4HENTAIGANA LETTER SA-5HENTAIGANA LETTER SA-6HENTAIGANA LETTER SA-7HENTAIGANA LETTER SA-8HENTAIGANA LETTER SI-1HENTAIGANA LETTER SI-2HENTAIGANA LETTER SI-3HENTAIGANA LETTER SI-4HENTAIGANA LETTER SI-5HENTAIGANA LETTER SI-6HENTAIGANA LETTER SU-1H\
    \ENTAIGANA LETTER SU-2HENTAIGANA LETTER SU-3HENTAIGANA LETTER SU-4HENTAIGANA LETTER SU-5HENTAIGANA LETTER SU-6HENTAIGANA LETTER SU-7HENTAIGANA LETTER SU-8HENTAIGANA LETTER SE-1HENTAIGANA LETTER SE-2HENTAIGANA LETTER SE-3HENTAIGANA LETTER SE-4HENTAIGANA LET\
    \TER SE-5HENTAIGANA LETTER SO-1HENTAIGANA LETTER SO-2HENTAIGANA LETTER SO-3HENTAIGANA LETTER SO-4HENTAIGANA LETTER SO-5HENTAIGANA LETTER SO-6HENTAIGANA LETTER SO-7HENTAIGANA LETTER TA-1HENTAIGANA LETTER TA-2HENTAIGANA LETTER TA-3HENTAIGANA LETTER TA-4HENTA\
    \IGANA LETTER TI-1HENTAIGANA LETTER TI-2HENTAIGANA LETTER TI-3HENTAIGANA LETTER TI-4HENTAIGANA LETTER TI-5HENTAIGANA LETTER TI-6HENTAIGANA LETTER TI-7HENTAIGANA LETTER TU-1HENTAIGANA LETTER TU-2HENTAIGANA LETTER TU-3HENTAIGANA LETTER TU-4HENTAIGANA LETTER \
    \TU-TOHENTAIGANA LETTER TE-1HENTAIGANA LETTER TE-2HENTAIGANA LETTER TE-3HENTAIGANA LETTER TE-4HENTAIGANA LETTER TE-5HENTAIGANA LETTER TE-6HENTAIGANA LETTER TE-7HENTAIGANA LETTER TE-8HENTAIGANA LETTER TE-9HENTAIGANA LETTER TO-1HENTAIGANA LETTER TO-2HENTAIGA\
    \NA LETTER TO-3HENTAIGANA LETTER TO-4HENTAIGANA LETTER TO-5HENTAIGANA LETTER TO-6HENTAIGANA LETTER TO-RAHENTAIGANA LETTER NA-1HENTAIGANA LETTER NA-2HENTAIGANA LETTER NA-3HENTAIGANA LETTER NA-4HENTAIGANA LETTER NA-5HENTAIGANA LETTER NA-6HENTAIGANA LETTER NA\
    \-7HENTAIGANA LETTER NA-8HENTAIGANA LETTER NA-9HENTAIGANA LETTER NI-1HENTAIGANA LETTER NI-2HENTAIGANA LETTER NI-3HENTAIGANA LETTER NI-4HENTAIGANA LETTER NI-5HENTAIGANA LETTER NI-6HENTAIGANA LETTER NI-7HENTAIGANA LETTER NI-TEHENTAIGANA LETTER NU-1HENTAIGANA\
    \ LETTER NU-2HENTAIGANA LETTER NU-3HENTAIGANA LETTER NE-1HENTAIGANA LETTER NE-2HENTAIGANA LETTER NE-3HENTAIGANA LETTER NE-4HENTAIGANA LETTER NE-5HENTAIGANA LETTER NE-6HENTAIGANA LETTER NE-KOHENTAIGANA LETTER NO-1HENTAIGANA LETTER NO-2HENTAIGANA LETTER NO-3\
    \HENTAIGANA LETTER NO-4HENTAIGANA LETTER NO-5HENTAIGANA LETTER HA-1HENTAIGANA LETTER HA-2HENTAIGANA LETTER HA-3HENTAIGANA LETTER HA-4HENTAIGANA LETTER HA-5HENTAIGANA LETTER HA-6HENTAIGANA LETTER HA-7HENTAIGANA LETTER HA-8HENTAIGANA LETTER HA-9HENTAIGANA LE\
    \TTER HA-10HENTAIGANA LETTER HA-11HENTAIGANA LETTER HI-1HENTAIGANA LETTER HI-2HENTAIGANA LETTER HI-3HENTAIGANA LETTER HI-4HENTAIGANA LETTER HI-5HENTAIGANA LETTER HI-6HENTAIGANA LETTER HI-7HENTAIGANA LETTER HU-1HENTAIGANA LETTER HU-2HENTAIGANA LETTER HU-3HE\
    \NTAIGANA LETTER HE-1HENTAIGANA LETTER HE-2HENTAIGANA LETTER HE-3HENTAIGANA LETTER HE-4HENTAIGANA LETTER HE-5HENTAIGANA LETTER HE-6HENTAIGANA LETTER HE-7HENTAIGANA LETTER HO-1HENTAIGANA LETTER HO-2HENTAIGANA LETTER HO-3HENTAIGANA LETTER HO-4HENTAIGANA LETT\
    \ER HO-5HENTAIGANA LETTER HO-6HENTAIGANA LETTER HO-7HENTAIGANA LETTER HO-8HENTAIGANA LETTER MA-1HENTAIGANA LETTER MA-2HENTAIGANA LETTER MA-3HENTAIGANA LETTER MA-4HENTAIGANA LETTER MA-5HENTAIGANA LETTER MA-6HENTAIGANA LETTER MA-7HENTAIGANA LETTER MI-1HENTAI\
    \GANA LETTER MI-2HENTAIGANA LETTER MI-3HENTAIGANA LETTER MI-4HENTAIGANA LETTER MI-5HENTAIGANA LETTER MI-6HENTAIGANA LETTER MI-7HENTAIGANA LETTER MU-1HENTAIGANA LETTER MU-2HENTAIGANA LETTER MU-3HENTAIGANA LETTER MU-4HENTAIGANA LETTER ME-1HENTAIGANA LETTER M\
    \E-2HENTAIGANA LETTER ME-MAHENTAIGANA LETTER MO-1HENTAIGANA LETTER MO-2HENTAIGANA LETTER MO-3HENTAIGANA LETTER MO-4HENTAIGANA LETTER MO-5HENTAIGANA LETTER MO-6HENTAIGANA LETTER YA-1HENTAIGANA LETTER YA-2HENTAIGANA LETTER YA-3HENTAIGANA LETTER YA-4HENTAIGAN\
    \A LETTER YA-5HENTAIGANA LETTER YA-YOHENTAIGANA LETTER YU-1HENTAIGANA LETTER YU-2HENTAIGANA LETTER YU-3HENTAIGANA LETTER YU-4HENTAIGANA LETTER YO-1HENTAIGANA LETTER YO-2HENTAIGANA LETTER YO-3HENTAIGANA LETTER YO-4HENTAIGANA LETTER YO-5HENTAIGANA LETTER YO-\
    \6HENTAIGANA LETTER RA-1HENTAIGANA LETTER RA-2HENTAIGANA LETTER RA-3HENTAIGANA LETTER RA-4HENTAIGANA LETTER RI-1HENTAIGANA LETTER RI-2HENTAIGANA LETTER RI-3HENTAIGANA LETTER RI-4HENTAIGANA LETTER RI-5HENTAIGANA LETTER RI-6HENTAIGANA LETTER RI-7HENTAIGANA L\
    \ETTER RU-1HENTAIGANA LETTER RU-2HENTAIGANA LETTER RU-3HENTAIGANA LETTER RU-4HENTAIGANA LETTER RU-5HENTAIGANA LETTER RU-6HENTAIGANA LETTER RE-1HENTAIGANA LETTER RE-2HENTAIGANA LETTER RE-3HENTAIGANA LETTER RE-4HENTAIGANA LETTER RO-1HENTAIGANA LETTER RO-2HEN\
    \TAIGANA LETTER RO-3HENTAIGANA LETTER RO-4HENTAIGANA LETTER RO-5HENTAIGANA LETTER RO-6HENTAIGANA LETTER WA-1HENTAIGANA LETTER WA-2HENTAIGANA LETTER WA-3HENTAIGANA LETTER WA-4HENTAIGANA LETTER WA-5HENTAIGANA LETTER WI-1HENTAIGANA LETTER WI-2HENTAIGANA LETTE\
    \R WI-3HENTAIGANA LETTER WI-4HENTAIGANA LETTER WI-5HENTAIGANA LETTER WE-1HENTAIGANA LETTER WE-2HENTAIGANA LETTER WE-3HENTAIGANA LETTER WE-4HENTAIGANA LETTER WO-1HENTAIGANA LETTER WO-2HENTAIGANA LETTER WO-3HENTAIGANA LETTER WO-4HENTAIGANA LETTER WO-5HENTAIG\
    \ANA LETTER WO-6HENTAIGANA LETTER WO-7HENTAIGANA LETTER N-MU-MO-1HENTAIGANA LETTER N-MU-MO-2HIRAGANA LETTER ARCHAIC WUKATAKANA LETTER ARCHAIC YIKATAKANA LETTER ARCHAIC YEKATAKANA LETTER ARCHAIC WUHIRAGANA LETTER SMALL KOHIRAGANA LETTER SMALL WIHIRAGANA LET\
    \TER SMALL WEHIRAGANA LETTER SMALL WOKATAKANA LETTER SMALL KOKATAKANA LETTER SMALL WIKATAKANA LETTER SMALL WEKATAKANA LETTER SMALL WOKATAKANA LETTER SMALL NNUSHU CHARACTER-1B170NUSHU CHARACTER-1B171NUSHU CHARACTER-1B172NUSHU CHARACTER-1B173NUSHU CHARACTER-\
    \1B174NUSHU CHARACTER-1B175NUSHU CHARACTER-1B176NUSHU CHARACTER-1B177NUSHU CHARACTER-1B178NUSHU CHARACTER-1B179NUSHU CHARACTER-1B17ANUSHU CHARACTER-1B17BNUSHU CHARACTER-1B17CNUSHU CHARACTER-1B17DNUSHU CHARACTER-1B17ENUSHU CHARACTER-1B17FNUSHU CHARACTER-1B1\
    \80NUSHU CHARACTER-1B181NUSHU CHARACTER-1B182NUSHU CHARACTER-1B183NUSHU CHARACTER-1B184NUSHU CHARACTER-1B185NUSHU CHARACTER-1B186NUSHU CHARACTER-1B187NUSHU CHARACTER-1B188NUSHU CHARACTER-1B189NUSHU CHARACTER-1B18ANUSHU CHARACTER-1B18BNUSHU CHARACTER-1B18CN\
    \USHU CHARACTER-1B18DNUSHU CHARACTER-1B18ENUSHU CHARACTER-1B18FNUSHU CHARACTER-1B190NUSHU CHARACTER-1B191NUSHU CHARACTER-1B192NUSHU CHARACTER-1B193NUSHU CHARACTER-1B194NUSHU CHARACTER-1B195NUSHU CHARACTER-1B196NUSHU CHARACTER-1B197NUSHU CHARACTER-1B198NUSH\
    \U CHARACTER-1B199NUSHU CHARACTER-1B19ANUSHU CHARACTER-1B19BNUSHU CHARACTER-1B19CNUSHU CHARACTER-1B19DNUSHU CHARACTER-1B19ENUSHU CHARACTER-1B19FNUSHU CHARACTER-1B1A0NUSHU CHARACTER-1B1A1NUSHU CHARACTER-1B1A2NUSHU CHARACTER-1B1A3NUSHU CHARACTER-1B1A4NUSHU C\
    \HARACTER-1B1A5NUSHU CHARACTER-1B1A6NUSHU CHARACTER-1B1A7NUSHU CHARACTER-1B1A8NUSHU CHARACTER-1B1A9NUSHU CHARACTER-1B1AANUSHU CHARACTER-1B1ABNUSHU CHARACTER-1B1ACNUSHU CHARACTER-1B1ADNUSHU CHARACTER-1B1AENUSHU CHARACTER-1B1AFNUSHU CHARACTER-1B1B0NUSHU CHAR\
    \ACTER-1B1B1NUSHU CHARACTER-1B1B2NUSHU CHARACTER-1B1B3NUSHU CHARACTER-1B1B4NUSHU CHARACTER-1B1B5NUSHU CHARACTER-1B1B6NUSHU CHARACTER-1B1B7NUSHU CHARACTER-1B1B8NUSHU CHARACTER-1B1B9NUSHU CHARACTER-1B1BANUSHU CHARACTER-1B1BBNUSHU CHARACTER-1B1BCNUSHU CHARACT\
    \ER-1B1BDNUSHU CHARACTER-1B1BENUSHU CHARACTER-1B1BFNUSHU CHARACTER-1B1C0NUSHU CHARACTER-1B1C1NUSHU CHARACTER-1B1C2NUSHU CHARACTER-1B1C3NUSHU CHARACTER-1B1C4NUSHU CHARACTER-1B1C5NUSHU CHARACTER-1B1C6NUSHU CHARACTER-1B1C7NUSHU CHARACTER-1B1C8NUSHU CHARACTER-\
    \1B1C9NUSHU CHARACTER-1B1CANUSHU CHARACTER-1B1CBNUSHU CHARACTER-1B1CCNUSHU CHARACTER-1B1CDNUSHU CHARACTER-1B1CENUSHU CHARACTER-1B1CFNUSHU CHARACTER-1B1D0NUSHU CHARACTER-1B1D1NUSHU CHARACTER-1B1D2NUSHU CHARACTER-1B1D3NUSHU CHARACTER-1B1D4NUSHU CHARACTER-1B1\
    \D5NUSHU CHARACTER-1B1D6NUSHU CHARACTER-1B1D7NUSHU CHARACTER-1B1D8NUSHU CHARACTER-1B1D9NUSHU CHARACTER-1B1DANUSHU CHARACTER-1B1DBNUSHU CHARACTER-1B1DCNUSHU CHARACTER-1B1DDNUSHU CHARACTER-1B1DENUSHU CHARACTER-1B1DFNUSHU CHARACTER-1B1E0NUSHU CHARACTER-1B1E1N\
    \USHU CHARACTER-1B1E2NUSHU CHARACTER-1B1E3NUSHU CHARACTER-1B1E4NUSHU CHARACTER-1B1E5NUSHU CHARACTER-1B1E6NUSHU CHARACTER-1B1E7NUSHU CHARACTER-1B1E8NUSHU CHARACTER-1B1E9NUSHU CHARACTER-1B1EANUSHU CHARACTER-1B1EBNUSHU CHARACTER-1B1ECNUSHU CHARACTER-1B1EDNUSH\
    \U CHARACTER-1B1EENUSHU CHARACTER-1B1EFNUSHU CHARACTER-1B1F0NUSHU CHARACTER-1B1F1NUSHU CHARACTER-1B1F2NUSHU CHARACTER-1B1F3NUSHU CHARACTER-1B1F4NUSHU CHARACTER-1B1F5NUSHU CHARACTER-1B1F6NUSHU CHARACTER-1B1F7NUSHU CHARACTER-1B1F8NUSHU CHARACTER-1B1F9NUSHU C\
    \HARACTER-1B1FANUSHU CHARACTER-1B1FBNUSHU CHARACTER-1B1FCNUSHU CHARACTER-1B1FDNUSHU CHARACTER-1B1FENUSHU CHARACTER-1B1FFNUSHU CHARACTER-1B200NUSHU CHARACTER-1B201NUSHU CHARACTER-1B202NUSHU CHARACTER-1B203NUSHU CHARACTER-1B204NUSHU CHARACTER-1B205NUSHU CHAR\
    \ACTER-1B206NUSHU CHARACTER-1B207NUSHU CHARACTER-1B208NUSHU CHARACTER-1B209NUSHU CHARACTER-1B20ANUSHU CHARACTER-1B20BNUSHU CHARACTER-1B20CNUSHU CHARACTER-1B20DNUSHU CHARACTER-1B20ENUSHU CHARACTER-1B20FNUSHU CHARACTER-1B210NUSHU CHARACTER-1B211NUSHU CHARACT\
    \ER-1B212NUSHU CHARACTER-1B213NUSHU CHARACTER-1B214NUSHU CHARACTER-1B215NUSHU CHARACTER-1B216NUSHU CHARACTER-1B217NUSHU CHARACTER-1B218NUSHU CHARACTER-1B219NUSHU CHARACTER-1B21ANUSHU CHARACTER-1B21BNUSHU CHARACTER-1B21CNUSHU CHARACTER-1B21DNUSHU CHARACTER-\
    \1B21ENUSHU CHARACTER-1B21FNUSHU CHARACTER-1B220NUSHU CHARACTER-1B221NUSHU CHARACTER-1B222NUSHU CHARACTER-1B223NUSHU CHARACTER-1B224NUSHU CHARACTER-1B225NUSHU CHARACTER-1B226NUSHU CHARACTER-1B227NUSHU CHARACTER-1B228NUSHU CHARACTER-1B229NUSHU CHARACTER-1B2\
    \2ANUSHU CHARACTER-1B22BNUSHU CHARACTER-1B22CNUSHU CHARACTER-1B22DNUSHU CHARACTER-1B22ENUSHU CHARACTER-1B22FNUSHU CHARACTER-1B230NUSHU CHARACTER-1B231NUSHU CHARACTER-1B232NUSHU CHARACTER-1B233NUSHU CHARACTER-1B234NUSHU CHARACTER-1B235NUSHU CHARACTER-1B236N\
    \USHU CHARACTER-1B237NUSHU CHARACTER-1B238NUSHU CHARACTER-1B239NUSHU CHARACTER-1B23ANUSHU CHARACTER-1B23BNUSHU CHARACTER-1B23CNUSHU CHARACTER-1B23DNUSHU CHARACTER-1B23ENUSHU CHARACTER-1B23FNUSHU CHARACTER-1B240NUSHU CHARACTER-1B241NUSHU CHARACTER-1B242NUSH\
    \U CHARACTER-1B243NUSHU CHARACTER-1B244NUSHU CHARACTER-1B245NUSHU CHARACTER-1B246NUSHU CHARACTER-1B247NUSHU CHARACTER-1B248NUSHU CHARACTER-1B249NUSHU CHARACTER-1B24ANUSHU CHARACTER-1B24BNUSHU CHARACTER-1B24CNUSHU CHARACTER-1B24DNUSHU CHARACTER-1B24ENUSHU C\
    \HARACTER-1B24FNUSHU CHARACTER-1B250NUSHU CHARACTER-1B251NUSHU CHARACTER-1B252NUSHU CHARACTER-1B253NUSHU CHARACTER-1B254NUSHU CHARACTER-1B255NUSHU CHARACTER-1B256NUSHU CHARACTER-1B257NUSHU CHARACTER-1B258NUSHU CHARACTER-1B259NUSHU CHARACTER-1B25ANUSHU CHAR\
    \ACTER-1B25BNUSHU CHARACTER-1B25CNUSHU CHARACTER-1B25DNUSHU CHARACTER-1B25ENUSHU CHARACTER-1B25FNUSHU CHARACTER-1B260NUSHU CHARACTER-1B261NUSHU CHARACTER-1B262NUSHU CHARACTER-1B263NUSHU CHARACTER-1B264NUSHU CHARACTER-1B265NUSHU CHARACTER-1B266NUSHU CHARACT\
    \ER-1B267NUSHU CHARACTER-1B268NUSHU CHARACTER-1B269NUSHU CHARACTER-1B26ANUSHU CHARACTER-1B26BNUSHU CHARACTER-1B26CNUSHU CHARACTER-1B26DNUSHU CHARACTER-1B26ENUSHU CHARACTER-1B26FNUSHU CHARACTER-1B270NUSHU CHARACTER-1B271NUSHU CHARACTER-1B272NUSHU CHARACTER-\
    \1B273NUSHU CHARACTER-1B274NUSHU CHARACTER-1B275NUSHU CHARACTER-1B276NUSHU CHARACTER-1B277NUSHU CHARACTER-1B278NUSHU CHARACTER-1B279NUSHU CHARACTER-1B27ANUSHU CHARACTER-1B27BNUSHU CHARACTER-1B27CNUSHU CHARACTER-1B27DNUSHU CHARACTER-1B27ENUSHU CHARACTER-1B2\
    \7FNUSHU CHARACTER-1B280NUSHU CHARACTER-1B281NUSHU CHARACTER-1B282NUSHU CHARACTER-1B283NUSHU CHARACTER-1B284NUSHU CHARACTER-1B285NUSHU CHARACTER-1B286NUSHU CHARACTER-1B287NUSHU CHARACTER-1B288NUSHU CHARACTER-1B289NUSHU CHARACTER-1B28ANUSHU CHARACTER-1B28BN\
    \USHU CHARACTER-1B28CNUSHU CHARACTER-1B28DNUSHU CHARACTER-1B28ENUSHU CHARACTER-1B28FNUSHU CHARACTER-1B290NUSHU CHARACTER-1B291NUSHU CHARACTER-1B292NUSHU CHARACTER-1B293NUSHU CHARACTER-1B294NUSHU CHARACTER-1B295NUSHU CHARACTER-1B296NUSHU CHARACTER-1B297NUSH\
    \U CHARACTER-1B298NUSHU CHARACTER-1B299NUSHU CHARACTER-1B29ANUSHU CHARACTER-1B29BNUSHU CHARACTER-1B29CNUSHU CHARACTER-1B29DNUSHU CHARACTER-1B29ENUSHU CHARACTER-1B29FNUSHU CHARACTER-1B2A0NUSHU CHARACTER-1B2A1NUSHU CHARACTER-1B2A2NUSHU CHARACTER-1B2A3NUSHU C\
    \HARACTER-1B2A4NUSHU CHARACTER-1B2A5NUSHU CHARACTER-1B2A6NUSHU CHARACTER-1B2A7NUSHU CHARACTER-1B2A8NUSHU CHARACTER-1B2A9NUSHU CHARACTER-1B2AANUSHU CHARACTER-1B2ABNUSHU CHARACTER-1B2ACNUSHU CHARACTER-1B2ADNUSHU CHARACTER-1B2AENUSHU CHARACTER-1B2AFNUSHU CHAR\
    \ACTER-1B2B0NUSHU CHARACTER-1B2B1NUSHU CHARACTER-1B2B2NUSHU CHARACTER-1B2B3NUSHU CHARACTER-1B2B4NUSHU CHARACTER-1B2B5NUSHU CHARACTER-1B2B6NUSHU CHARACTER-1B2B7NUSHU CHARACTER-1B2B8NUSHU CHARACTER-1B2B9NUSHU CHARACTER-1B2BANUSHU CHARACTER-1B2BBNUSHU CHARACT\
    \ER-1B2BCNUSHU CHARACTER-1B2BDNUSHU CHARACTER-1B2BENUSHU CHARACTER-1B2BFNUSHU CHARACTER-1B2C0NUSHU CHARACTER-1B2C1NUSHU CHARACTER-1B2C2NUSHU CHARACTER-1B2C3NUSHU CHARACTER-1B2C4NUSHU CHARACTER-1B2C5NUSHU CHARACTER-1B2C6NUSHU CHARACTER-1B2C7NUSHU CHARACTER-\
    \1B2C8NUSHU CHARACTER-1B2C9NUSHU CHARACTER-1B2CANUSHU CHARACTER-1B2CBNUSHU CHARACTER-1B2CCNUSHU CHARACTER-1B2CDNUSHU CHARACTER-1B2CENUSHU CHARACTER-1B2CFNUSHU CHARACTER-1B2D0NUSHU CHARACTER-1B2D1NUSHU CHARACTER-1B2D2NUSHU CHARACTER-1B2D3NUSHU CHARACTER-1B2\
    \D4NUSHU CHARACTER-1B2D5NUSHU CHARACTER-1B2D6NUSHU CHARACTER-1B2D7NUSHU CHARACTER-1B2D8NUSHU CHARACTER-1B2D9NUSHU CHARACTER-1B2DANUSHU CHARACTER-1B2DBNUSHU CHARACTER-1B2DCNUSHU CHARACTER-1B2DDNUSHU CHARACTER-1B2DENUSHU CHARACTER-1B2DFNUSHU CHARACTER-1B2E0N\
    \USHU CHARACTER-1B2E1NUSHU CHARACTER-1B2E2NUSHU CHARACTER-1B2E3NUSHU CHARACTER-1B2E4NUSHU CHARACTER-1B2E5NUSHU CHARACTER-1B2E6NUSHU CHARACTER-1B2E7NUSHU CHARACTER-1B2E8NUSHU CHARACTER-1B2E9NUSHU CHARACTER-1B2EANUSHU CHARACTER-1B2EBNUSHU CHARACTER-1B2ECNUSH\
    \U CHARACTER-1B2EDNUSHU CHARACTER-1B2EENUSHU CHARACTER-1B2EFNUSHU CHARACTER-1B2F0NUSHU CHARACTER-1B2F1NUSHU CHARACTER-1B2F2NUSHU CHARACTER-1B2F3NUSHU CHARACTER-1B2F4NUSHU CHARACTER-1B2F5NUSHU CHARACTER-1B2F6NUSHU CHARACTER-1B2F7NUSHU CHARACTER-1B2F8NUSHU C\
    \HARACTER-1B2F9NUSHU CHARACTER-1B2FANUSHU CHARACTER-1B2FBDUPLOYAN LETTER HDUPLOYAN LETTER XDUPLOYAN LETTER PDUPLOYAN LETTER TDUPLOYAN LETTER FDUPLOYAN LETTER KDUPLOYAN LETTER LDUPLOYAN LETTER BDUPLOYAN LETTER DDUPLOYAN LETTER VDUPLOYAN LETTER GDUPLOYAN LET\
    \TER RDUPLOYAN LETTER P NDUPLOYAN LETTER D SDUPLOYAN LETTER F NDUPLOYAN LETTER K MDUPLOYAN LETTER R SDUPLOYAN LETTER THDUPLOYAN LETTER SLOAN DHDUPLOYAN LETTER DHDUPLOYAN LETTER KKDUPLOYAN LETTER SLOAN JDUPLOYAN LETTER HLDUPLOYAN LETTER LHDUPLOYAN LETTER RH\
    \DUPLOYAN LETTER MDUPLOYAN LETTER NDUPLOYAN LETTER JDUPLOYAN LETTER SDUPLOYAN LETTER M NDUPLOYAN LETTER N MDUPLOYAN LETTER J MDUPLOYAN LETTER S JDUPLOYAN LETTER M WITH DOTDUPLOYAN LETTER N WITH DOTDUPLOYAN LETTER J WITH DOTDUPLOYAN LETTER J WITH DOTS INSID\
    \E AND ABOVEDUPLOYAN LETTER S WITH DOTDUPLOYAN LETTER S WITH DOT BELOWDUPLOYAN LETTER M SDUPLOYAN LETTER N SDUPLOYAN LETTER J SDUPLOYAN LETTER S SDUPLOYAN LETTER M N SDUPLOYAN LETTER N M SDUPLOYAN LETTER J M SDUPLOYAN LETTER S J SDUPLOYAN LETTER J S WITH D\
    \OTDUPLOYAN LETTER J NDUPLOYAN LETTER J N SDUPLOYAN LETTER S TDUPLOYAN LETTER S T RDUPLOYAN LETTER S PDUPLOYAN LETTER S P RDUPLOYAN LETTER T SDUPLOYAN LETTER T R SDUPLOYAN LETTER WDUPLOYAN LETTER WHDUPLOYAN LETTER W RDUPLOYAN LETTER S NDUPLOYAN LETTER S MD\
    \UPLOYAN LETTER K R SDUPLOYAN LETTER G R SDUPLOYAN LETTER S KDUPLOYAN LETTER S K RDUPLOYAN LETTER ADUPLOYAN LETTER SLOAN OWDUPLOYAN LETTER OADUPLOYAN LETTER ODUPLOYAN LETTER AOUDUPLOYAN LETTER IDUPLOYAN LETTER EDUPLOYAN LETTER IEDUPLOYAN LETTER SHORT IDUPL\
    \OYAN LETTER UIDUPLOYAN LETTER EEDUPLOYAN LETTER SLOAN EHDUPLOYAN LETTER ROMANIAN IDUPLOYAN LETTER SLOAN EEDUPLOYAN LETTER LONG IDUPLOYAN LETTER YEDUPLOYAN LETTER UDUPLOYAN LETTER EUDUPLOYAN LETTER XWDUPLOYAN LETTER U NDUPLOYAN LETTER LONG UDUPLOYAN LETTER\
    \ ROMANIAN UDUPLOYAN LETTER UHDUPLOYAN LETTER SLOAN UDUPLOYAN LETTER OOHDUPLOYAN LETTER OWDUPLOYAN LETTER OUDUPLOYAN LETTER WADUPLOYAN LETTER WODUPLOYAN LETTER WIDUPLOYAN LETTER WEIDUPLOYAN LETTER WOWDUPLOYAN LETTER NASAL UDUPLOYAN LETTER NASAL ODUPLOYAN L\
    \ETTER NASAL IDUPLOYAN LETTER NASAL ADUPLOYAN LETTER PERNIN ANDUPLOYAN LETTER PERNIN AMDUPLOYAN LETTER SLOAN ENDUPLOYAN LETTER SLOAN ANDUPLOYAN LETTER SLOAN ONDUPLOYAN LETTER VOCALIC MDUPLOYAN AFFIX LEFT HORIZONTAL SECANTDUPLOYAN AFFIX MID HORIZONTAL SECAN\
    \TDUPLOYAN AFFIX RIGHT HORIZONTAL SECANTDUPLOYAN AFFIX LOW VERTICAL SECANTDUPLOYAN AFFIX MID VERTICAL SECANTDUPLOYAN AFFIX HIGH VERTICAL SECANTDUPLOYAN AFFIX ATTACHED SECANTDUPLOYAN AFFIX ATTACHED LEFT-TO-RIGHT SECANTDUPLOYAN AFFIX ATTACHED TANGENTDUPLOYAN\
    \ AFFIX ATTACHED TAILDUPLOYAN AFFIX ATTACHED E HOOKDUPLOYAN AFFIX ATTACHED I HOOKDUPLOYAN AFFIX ATTACHED TANGENT HOOKDUPLOYAN AFFIX HIGH ACUTEDUPLOYAN AFFIX HIGH TIGHT ACUTEDUPLOYAN AFFIX HIGH GRAVEDUPLOYAN AFFIX HIGH LONG GRAVEDUPLOYAN AFFIX HIGH DOTDUPLO\
    \YAN AFFIX HIGH CIRCLEDUPLOYAN AFFIX HIGH LINEDUPLOYAN AFFIX HIGH WAVEDUPLOYAN AFFIX HIGH VERTICALDUPLOYAN AFFIX LOW ACUTEDUPLOYAN AFFIX LOW TIGHT ACUTEDUPLOYAN AFFIX LOW GRAVEDUPLOYAN AFFIX LOW LONG GRAVEDUPLOYAN AFFIX LOW DOTDUPLOYAN AFFIX LOW CIRCLEDUPL\
    \OYAN AFFIX LOW LINEDUPLOYAN AFFIX LOW WAVEDUPLOYAN AFFIX LOW VERTICALDUPLOYAN AFFIX LOW ARROWDUPLOYAN SIGN O WITH CROSSDUPLOYAN THICK LETTER SELECTORDUPLOYAN DOUBLE MARKDUPLOYAN PUNCTUATION CHINOOK FULL STOPSHORTHAND FORMAT LETTER OVERLAPSHORTHAND FORMAT \
    \CONTINUING OVERLAPSHORTHAND FORMAT DOWN STEPSHORTHAND FORMAT UP STEPZNAMENNY COMBINING MARK GORAZDO NIZKO S KRYZHEM ON LEFTZNAMENNY COMBINING MARK NIZKO S KRYZHEM ON LEFTZNAMENNY COMBINING MARK TSATA ON LEFTZNAMENNY COMBINING MARK GORAZDO NIZKO ON LEFTZNA\
    \MENNY COMBINING MARK NIZKO ON LEFTZNAMENNY COMBINING MARK SREDNE ON LEFTZNAMENNY COMBINING MARK MALO POVYSHE ON LEFTZNAMENNY COMBINING MARK POVYSHE ON LEFTZNAMENNY COMBINING MARK VYSOKO ON LEFTZNAMENNY COMBINING MARK MALO POVYSHE S KHOKHLOM ON LEFTZNAMENN\
    \Y COMBINING MARK POVYSHE S KHOKHLOM ON LEFTZNAMENNY COMBINING MARK VYSOKO S KHOKHLOM ON LEFTZNAMENNY COMBINING MARK GORAZDO NIZKO S KRYZHEM ON RIGHTZNAMENNY COMBINING MARK NIZKO S KRYZHEM ON RIGHTZNAMENNY COMBINING MARK TSATA ON RIGHTZNAMENNY COMBINING MA\
    \RK GORAZDO NIZKO ON RIGHTZNAMENNY COMBINING MARK NIZKO ON RIGHTZNAMENNY COMBINING MARK SREDNE ON RIGHTZNAMENNY COMBINING MARK MALO POVYSHE ON RIGHTZNAMENNY COMBINING MARK POVYSHE ON RIGHTZNAMENNY COMBINING MARK VYSOKO ON RIGHTZNAMENNY COMBINING MARK MALO \
    \POVYSHE S KHOKHLOM ON RIGHTZNAMENNY COMBINING MARK POVYSHE S KHOKHLOM ON RIGHTZNAMENNY COMBINING MARK VYSOKO S KHOKHLOM ON RIGHTZNAMENNY COMBINING MARK TSATA S KRYZHEMZNAMENNY COMBINING MARK MALO POVYSHE S KRYZHEMZNAMENNY COMBINING MARK STRANNO MALO POVYS\
    \HEZNAMENNY COMBINING MARK POVYSHE S KRYZHEMZNAMENNY COMBINING MARK POVYSHE STRANNOZNAMENNY COMBINING MARK VYSOKO S KRYZHEMZNAMENNY COMBINING MARK MALO POVYSHE STRANNOZNAMENNY COMBINING MARK GORAZDO VYSOKOZNAMENNY COMBINING MARK ZELOZNAMENNY COMBINING MARK\
    \ ONZNAMENNY COMBINING MARK RAVNOZNAMENNY COMBINING MARK TIKHAYAZNAMENNY COMBINING MARK BORZAYAZNAMENNY COMBINING MARK UDARKAZNAMENNY COMBINING MARK PODVERTKAZNAMENNY COMBINING MARK LOMKAZNAMENNY COMBINING MARK KUPNAYAZNAMENNY COMBINING MARK KACHKAZNAMENNY\
    \ COMBINING MARK ZEVOKZNAMENNY COMBINING MARK SKOBAZNAMENNY COMBINING MARK RAZSEKAZNAMENNY COMBINING MARK KRYZH ON LEFTZNAMENNY COMBINING TONAL RANGE MARK MRACHNOZNAMENNY COMBINING TONAL RANGE MARK SVETLOZNAMENNY COMBINING TONAL RANGE MARK TRESVETLOZNAMENN\
    \Y COMBINING MARK ZADERZHKAZNAMENNY COMBINING MARK DEMESTVENNY ZADERZHKAZNAMENNY COMBINING MARK OTSECHKAZNAMENNY COMBINING MARK PODCHASHIEZNAMENNY COMBINING MARK PODCHASHIE WITH VERTICAL STROKEZNAMENNY COMBINING MARK CHASHKAZNAMENNY COMBINING MARK CHASHKA \
    \POLNAYAZNAMENNY COMBINING MARK OBLACHKOZNAMENNY COMBINING MARK SOROCHYA NOZHKAZNAMENNY COMBINING MARK TOCHKAZNAMENNY COMBINING MARK DVOETOCHIEZNAMENNY COMBINING ATTACHING VERTICAL OMETZNAMENNY COMBINING MARK CURVED OMETZNAMENNY COMBINING MARK KRYZHZNAMENN\
    \Y COMBINING LOWER TONAL RANGE INDICATORZNAMENNY PRIZNAK MODIFIER LEVEL-2ZNAMENNY PRIZNAK MODIFIER LEVEL-3ZNAMENNY PRIZNAK MODIFIER DIRECTION FLIPZNAMENNY PRIZNAK MODIFIER KRYZHZNAMENNY PRIZNAK MODIFIER ROGZNAMENNY NEUME KRYUKZNAMENNY NEUME KRYUK TIKHYZNAM\
    \ENNY NEUME PARAKLITZNAMENNY NEUME DVA V CHELNUZNAMENNY NEUME KLYUCHZNAMENNY NEUME ZANOZHEKZNAMENNY NEUME STOPITSAZNAMENNY NEUME STOPITSA S OCHKOMZNAMENNY NEUME PEREVODKAZNAMENNY NEUME PEREVODKA NEPOSTOYANNAYAZNAMENNY NEUME STOPITSA WITH SOROCHYA NOZHKAZNA\
    \MENNY NEUME CHELYUSTKAZNAMENNY NEUME PALKAZNAMENNY NEUME ZAPYATAYAZNAMENNY NEUME GOLUBCHIK BORZYZNAMENNY NEUME GOLUBCHIK TIKHYZNAMENNY NEUME GOLUBCHIK MRACHNYZNAMENNY NEUME GOLUBCHIK SVETLYZNAMENNY NEUME GOLUBCHIK TRESVETLYZNAMENNY NEUME VRAKHIYA PROSTAYA\
    \ZNAMENNY NEUME VRAKHIYA MRACHNAYAZNAMENNY NEUME VRAKHIYA SVETLAYAZNAMENNY NEUME VRAKHIYA TRESVETLAYAZNAMENNY NEUME VRAKHIYA KLYUCHEVAYA PROSTAYAZNAMENNY NEUME VRAKHIYA KLYUCHEVAYA MRACHNAYAZNAMENNY NEUME VRAKHIYA KLYUCHEVAYA SVETLAYAZNAMENNY NEUME VRAKHIY\
    \A KLYUCHEVAYA TRESVETLAYAZNAMENNY NEUME DOUBLE ZAPYATAYAZNAMENNY NEUME REVERSED CHELYUSTKAZNAMENNY NEUME DERBITSAZNAMENNY NEUME KHAMILOZNAMENNY NEUME CHASHKAZNAMENNY NEUME PODCHASHIEZNAMENNY NEUME SKAMEYTSA MRACHNAYAZNAMENNY NEUME SKAMEYTSA SVETLAYAZNAMEN\
    \NY NEUME SKAMEYTSA TRESVETLAYAZNAMENNY NEUME SKAMEYTSA TIKHAYAZNAMENNY NEUME DEMESTVENNY KLYUCHZNAMENNY NEUME SKAMEYTSA KLYUCHEVAYA SVETLAYAZNAMENNY NEUME SKAMEYTSA KLYUCHENEPOSTOYANNAYAZNAMENNY NEUME SKAMEYTSA KLYUCHEVAYA TIKHAYAZNAMENNY NEUME SKAMEYTSA \
    \DVOECHELNAYA PROSTAYAZNAMENNY NEUME SKAMEYTSA DVOECHELNAYA SVETLAYAZNAMENNY NEUME SKAMEYTSA DVOECHELNAYA NEPOSTOYANNAYAZNAMENNY NEUME SKAMEYTSA DVOECHELNAYA KLYUCHEVAYAZNAMENNY NEUME SLOZHITIEZNAMENNY NEUME SLOZHITIE S ZAPYATOYZNAMENNY NEUME SLOZHITIE ZAK\
    \RYTOEZNAMENNY NEUME SLOZHITIE S KRYZHEMZNAMENNY NEUME KRYZHZNAMENNY NEUME ROGZNAMENNY NEUME FITAZNAMENNY NEUME KOBYLAZNAMENNY NEUME ZMEYTSAZNAMENNY NEUME STATYAZNAMENNY NEUME STATYA S ZAPYATOYZNAMENNY NEUME STATYA S KRYZHEMZNAMENNY NEUME STATYA S ZAPYATOY\
    \ I KRYZHEMZNAMENNY NEUME STATYA S KRYZHEM I ZAPYATOYZNAMENNY NEUME STATYA ZAKRYTAYAZNAMENNY NEUME STATYA ZAKRYTAYA S ZAPYATOYZNAMENNY NEUME STATYA S ROGOMZNAMENNY NEUME STATYA S DVUMYA ZAPYATYMIZNAMENNY NEUME STATYA S ZAPYATOY I PODCHASHIEMZNAMENNY NEUME \
    \POLKULIZMYZNAMENNY NEUME STATYA NEPOSTOYANNAYAZNAMENNY NEUME STRELA PROSTAYAZNAMENNY NEUME STRELA MRACHNOTIKHAYAZNAMENNY NEUME STRELA KRYZHEVAYAZNAMENNY NEUME STRELA POLUPOVODNAYAZNAMENNY NEUME STRELA POVODNAYAZNAMENNY NEUME STRELA NEPOSTOYANNAYAZNAMENNY \
    \NEUME STRELA KLYUCHEPOVODNAYAZNAMENNY NEUME STRELA KLYUCHENEPOSTOYANNAYAZNAMENNY NEUME STRELA TIKHAYA PUTNAYAZNAMENNY NEUME STRELA DVOECHELNAYAZNAMENNY NEUME STRELA DVOECHELNOKRYZHEVAYAZNAMENNY NEUME STRELA DVOECHELNOPOVODNAYAZNAMENNY NEUME STRELA DVOECHE\
    \LNAYA KLYUCHEVAYAZNAMENNY NEUME STRELA DVOECHELNOPOVODNAYA KLYUCHEVAYAZNAMENNY NEUME STRELA GROMNAYA WITH SINGLE ZAPYATAYAZNAMENNY NEUME STRELA GROMOPOVODNAYA WITH SINGLE ZAPYATAYAZNAMENNY NEUME STRELA GROMNAYAZNAMENNY NEUME STRELA GROMOPOVODNAYAZNAMENNY \
    \NEUME STRELA GROMOPOVODNAYA WITH DOUBLE ZAPYATAYAZNAMENNY NEUME STRELA GROMOKRYZHEVAYAZNAMENNY NEUME STRELA GROMOKRYZHEVAYA POVODNAYAZNAMENNY NEUME MECHIKZNAMENNY NEUME MECHIK POVODNYZNAMENNY NEUME MECHIK KLYUCHEVOYZNAMENNY NEUME MECHIK KLYUCHEPOVODNYZNAM\
    \ENNY NEUME MECHIK KLYUCHENEPOSTOYANNYZNAMENNY NEUME STRELA TRYASOGLASNAYAZNAMENNY NEUME STRELA TRYASOPOVODNAYAZNAMENNY NEUME STRELA TRYASOSTRELNAYAZNAMENNY NEUME OSOKAZNAMENNY NEUME OSOKA SVETLAYAZNAMENNY NEUME OSOKA TRESVETLAYAZNAMENNY NEUME OSOKA KRYUKO\
    \VAYA SVETLAYAZNAMENNY NEUME OSOKA KLYUCHEVAYA SVETLAYAZNAMENNY NEUME OSOKA KLYUCHEVAYA NEPOSTOYANNAYAZNAMENNY NEUME STRELA KRYUKOVAYAZNAMENNY NEUME STRELA KRYUKOVAYA POVODNAYAZNAMENNY NEUME STRELA KRYUKOVAYA GROMNAYA WITH SINGLE ZAPYATAYAZNAMENNY NEUME ST\
    \RELA KRYUKOVAYA GROMOPOVODNAYA WITH SINGLE ZAPYATAYAZNAMENNY NEUME STRELA KRYUKOVAYA GROMNAYAZNAMENNY NEUME STRELA KRYUKOVAYA GROMOPOVODNAYAZNAMENNY NEUME STRELA KRYUKOVAYA GROMOPOVODNAYA WITH DOUBLE ZAPYATAYAZNAMENNY NEUME STRELA KRYUKOVAYA GROMOKRYZHEVA\
    \YAZNAMENNY NEUME STRELA KRYUKOVAYA GROMOKRYZHEVAYA POVODNAYAZNAMENNY NEUME STRELA KRYUKOVAYA TRYASKAZNAMENNY NEUME KUFISMAZNAMENNY NEUME OBLAKOZNAMENNY NEUME DUDAZNAMENNY NEUME NEMKAZNAMENNY NEUME PAUKBYZANTINE MUSICAL SYMBOL PSILIBYZANTINE MUSICAL SYMBOL\
    \ DASEIABYZANTINE MUSICAL SYMBOL PERISPOMENIBYZANTINE MUSICAL SYMBOL OXEIA EKFONITIKONBYZANTINE MUSICAL SYMBOL OXEIA DIPLIBYZANTINE MUSICAL SYMBOL VAREIA EKFONITIKONBYZANTINE MUSICAL SYMBOL VAREIA DIPLIBYZANTINE MUSICAL SYMBOL KATHISTIBYZANTINE MUSICAL SYM\
    \BOL SYRMATIKIBYZANTINE MUSICAL SYMBOL PARAKLITIKIBYZANTINE MUSICAL SYMBOL YPOKRISISBYZANTINE MUSICAL SYMBOL YPOKRISIS DIPLIBYZANTINE MUSICAL SYMBOL KREMASTIBYZANTINE MUSICAL SYMBOL APESO EKFONITIKONBYZANTINE MUSICAL SYMBOL EXO EKFONITIKONBYZANTINE MUSICAL\
    \ SYMBOL TELEIABYZANTINE MUSICAL SYMBOL KENTIMATABYZANTINE MUSICAL SYMBOL APOSTROFOSBYZANTINE MUSICAL SYMBOL APOSTROFOS DIPLIBYZANTINE MUSICAL SYMBOL SYNEVMABYZANTINE MUSICAL SYMBOL THITABYZANTINE MUSICAL SYMBOL OLIGON ARCHAIONBYZANTINE MUSICAL SYMBOL GORG\
    \ON ARCHAIONBYZANTINE MUSICAL SYMBOL PSILONBYZANTINE MUSICAL SYMBOL CHAMILONBYZANTINE MUSICAL SYMBOL VATHYBYZANTINE MUSICAL SYMBOL ISON ARCHAIONBYZANTINE MUSICAL SYMBOL KENTIMA ARCHAIONBYZANTINE MUSICAL SYMBOL KENTIMATA ARCHAIONBYZANTINE MUSICAL SYMBOL SAX\
    \IMATABYZANTINE MUSICAL SYMBOL PARICHONBYZANTINE MUSICAL SYMBOL STAVROS APODEXIABYZANTINE MUSICAL SYMBOL OXEIAI ARCHAIONBYZANTINE MUSICAL SYMBOL VAREIAI ARCHAIONBYZANTINE MUSICAL SYMBOL APODERMA ARCHAIONBYZANTINE MUSICAL SYMBOL APOTHEMABYZANTINE MUSICAL SY\
    \MBOL KLASMABYZANTINE MUSICAL SYMBOL REVMABYZANTINE MUSICAL SYMBOL PIASMA ARCHAIONBYZANTINE MUSICAL SYMBOL TINAGMABYZANTINE MUSICAL SYMBOL ANATRICHISMABYZANTINE MUSICAL SYMBOL SEISMABYZANTINE MUSICAL SYMBOL SYNAGMA ARCHAIONBYZANTINE MUSICAL SYMBOL SYNAGMA \
    \META STAVROUBYZANTINE MUSICAL SYMBOL OYRANISMA ARCHAIONBYZANTINE MUSICAL SYMBOL THEMABYZANTINE MUSICAL SYMBOL LEMOIBYZANTINE MUSICAL SYMBOL DYOBYZANTINE MUSICAL SYMBOL TRIABYZANTINE MUSICAL SYMBOL TESSERABYZANTINE MUSICAL SYMBOL KRATIMATABYZANTINE MUSICAL\
    \ SYMBOL APESO EXO NEOBYZANTINE MUSICAL SYMBOL FTHORA ARCHAIONBYZANTINE MUSICAL SYMBOL IMIFTHORABYZANTINE MUSICAL SYMBOL TROMIKON ARCHAIONBYZANTINE MUSICAL SYMBOL KATAVA TROMIKONBYZANTINE MUSICAL SYMBOL PELASTONBYZANTINE MUSICAL SYMBOL PSIFISTONBYZANTINE M\
    \USICAL SYMBOL KONTEVMABYZANTINE MUSICAL SYMBOL CHOREVMA ARCHAIONBYZANTINE MUSICAL SYMBOL RAPISMABYZANTINE MUSICAL SYMBOL PARAKALESMA ARCHAIONBYZANTINE MUSICAL SYMBOL PARAKLITIKI ARCHAIONBYZANTINE MUSICAL SYMBOL ICHADINBYZANTINE MUSICAL SYMBOL NANABYZANTIN\
    \E MUSICAL SYMBOL PETASMABYZANTINE MUSICAL SYMBOL KONTEVMA ALLOBYZANTINE MUSICAL SYMBOL TROMIKON ALLOBYZANTINE MUSICAL SYMBOL STRAGGISMATABYZANTINE MUSICAL SYMBOL GRONTHISMATABYZANTINE MUSICAL SYMBOL ISON NEOBYZANTINE MUSICAL SYMBOL OLIGON NEOBYZANTINE MUS\
    \ICAL SYMBOL OXEIA NEOBYZANTINE MUSICAL SYMBOL PETASTIBYZANTINE MUSICAL SYMBOL KOUFISMABYZANTINE MUSICAL SYMBOL PETASTOKOUFISMABYZANTINE MUSICAL SYMBOL KRATIMOKOUFISMABYZANTINE MUSICAL SYMBOL PELASTON NEOBYZANTINE MUSICAL SYMBOL KENTIMATA NEO ANOBYZANTINE \
    \MUSICAL SYMBOL KENTIMA NEO ANOBYZANTINE MUSICAL SYMBOL YPSILIBYZANTINE MUSICAL SYMBOL APOSTROFOS NEOBYZANTINE MUSICAL SYMBOL APOSTROFOI SYNDESMOS NEOBYZANTINE MUSICAL SYMBOL YPORROIBYZANTINE MUSICAL SYMBOL KRATIMOYPORROONBYZANTINE MUSICAL SYMBOL ELAFRONBY\
    \ZANTINE MUSICAL SYMBOL CHAMILIBYZANTINE MUSICAL SYMBOL MIKRON ISONBYZANTINE MUSICAL SYMBOL VAREIA NEOBYZANTINE MUSICAL SYMBOL PIASMA NEOBYZANTINE MUSICAL SYMBOL PSIFISTON NEOBYZANTINE MUSICAL SYMBOL OMALONBYZANTINE MUSICAL SYMBOL ANTIKENOMABYZANTINE MUSIC\
    \AL SYMBOL LYGISMABYZANTINE MUSICAL SYMBOL PARAKLITIKI NEOBYZANTINE MUSICAL SYMBOL PARAKALESMA NEOBYZANTINE MUSICAL SYMBOL ETERON PARAKALESMABYZANTINE MUSICAL SYMBOL KYLISMABYZANTINE MUSICAL SYMBOL ANTIKENOKYLISMABYZANTINE MUSICAL SYMBOL TROMIKON NEOBYZANT\
    \INE MUSICAL SYMBOL EKSTREPTONBYZANTINE MUSICAL SYMBOL SYNAGMA NEOBYZANTINE MUSICAL SYMBOL SYRMABYZANTINE MUSICAL SYMBOL CHOREVMA NEOBYZANTINE MUSICAL SYMBOL EPEGERMABYZANTINE MUSICAL SYMBOL SEISMA NEOBYZANTINE MUSICAL SYMBOL XIRON KLASMABYZANTINE MUSICAL \
    \SYMBOL TROMIKOPSIFISTONBYZANTINE MUSICAL SYMBOL PSIFISTOLYGISMABYZANTINE MUSICAL SYMBOL TROMIKOLYGISMABYZANTINE MUSICAL SYMBOL TROMIKOPARAKALESMABYZANTINE MUSICAL SYMBOL PSIFISTOPARAKALESMABYZANTINE MUSICAL SYMBOL TROMIKOSYNAGMABYZANTINE MUSICAL SYMBOL PS\
    \IFISTOSYNAGMABYZANTINE MUSICAL SYMBOL GORGOSYNTHETONBYZANTINE MUSICAL SYMBOL ARGOSYNTHETONBYZANTINE MUSICAL SYMBOL ETERON ARGOSYNTHETONBYZANTINE MUSICAL SYMBOL OYRANISMA NEOBYZANTINE MUSICAL SYMBOL THEMATISMOS ESOBYZANTINE MUSICAL SYMBOL THEMATISMOS EXOBY\
    \ZANTINE MUSICAL SYMBOL THEMA APLOUNBYZANTINE MUSICAL SYMBOL THES KAI APOTHESBYZANTINE MUSICAL SYMBOL KATAVASMABYZANTINE MUSICAL SYMBOL ENDOFONONBYZANTINE MUSICAL SYMBOL YFEN KATOBYZANTINE MUSICAL SYMBOL YFEN ANOBYZANTINE MUSICAL SYMBOL STAVROSBYZANTINE MU\
    \SICAL SYMBOL KLASMA ANOBYZANTINE MUSICAL SYMBOL DIPLI ARCHAIONBYZANTINE MUSICAL SYMBOL KRATIMA ARCHAIONBYZANTINE MUSICAL SYMBOL KRATIMA ALLOBYZANTINE MUSICAL SYMBOL KRATIMA NEOBYZANTINE MUSICAL SYMBOL APODERMA NEOBYZANTINE MUSICAL SYMBOL APLIBYZANTINE MUS\
    \ICAL SYMBOL DIPLIBYZANTINE MUSICAL SYMBOL TRIPLIBYZANTINE MUSICAL SYMBOL TETRAPLIBYZANTINE MUSICAL SYMBOL KORONISBYZANTINE MUSICAL SYMBOL LEIMMA ENOS CHRONOUBYZANTINE MUSICAL SYMBOL LEIMMA DYO CHRONONBYZANTINE MUSICAL SYMBOL LEIMMA TRION CHRONONBYZANTINE \
    \MUSICAL SYMBOL LEIMMA TESSARON CHRONONBYZANTINE MUSICAL SYMBOL LEIMMA IMISEOS CHRONOUBYZANTINE MUSICAL SYMBOL GORGON NEO ANOBYZANTINE MUSICAL SYMBOL GORGON PARESTIGMENON ARISTERABYZANTINE MUSICAL SYMBOL GORGON PARESTIGMENON DEXIABYZANTINE MUSICAL SYMBOL D\
    \IGORGONBYZANTINE MUSICAL SYMBOL DIGORGON PARESTIGMENON ARISTERA KATOBYZANTINE MUSICAL SYMBOL DIGORGON PARESTIGMENON ARISTERA ANOBYZANTINE MUSICAL SYMBOL DIGORGON PARESTIGMENON DEXIABYZANTINE MUSICAL SYMBOL TRIGORGONBYZANTINE MUSICAL SYMBOL ARGONBYZANTINE \
    \MUSICAL SYMBOL IMIDIARGONBYZANTINE MUSICAL SYMBOL DIARGONBYZANTINE MUSICAL SYMBOL AGOGI POLI ARGIBYZANTINE MUSICAL SYMBOL AGOGI ARGOTERIBYZANTINE MUSICAL SYMBOL AGOGI ARGIBYZANTINE MUSICAL SYMBOL AGOGI METRIABYZANTINE MUSICAL SYMBOL AGOGI MESIBYZANTINE MU\
    \SICAL SYMBOL AGOGI GORGIBYZANTINE MUSICAL SYMBOL AGOGI GORGOTERIBYZANTINE MUSICAL SYMBOL AGOGI POLI GORGIBYZANTINE MUSICAL SYMBOL MARTYRIA PROTOS ICHOSBYZANTINE MUSICAL SYMBOL MARTYRIA ALLI PROTOS ICHOSBYZANTINE MUSICAL SYMBOL MARTYRIA DEYTEROS ICHOSBYZAN\
    \TINE MUSICAL SYMBOL MARTYRIA ALLI DEYTEROS ICHOSBYZANTINE MUSICAL SYMBOL MARTYRIA TRITOS ICHOSBYZANTINE MUSICAL SYMBOL MARTYRIA TRIFONIASBYZANTINE MUSICAL SYMBOL MARTYRIA TETARTOS ICHOSBYZANTINE MUSICAL SYMBOL MARTYRIA TETARTOS LEGETOS ICHOSBYZANTINE MUSI\
    \CAL SYMBOL MARTYRIA LEGETOS ICHOSBYZANTINE MUSICAL SYMBOL MARTYRIA PLAGIOS ICHOSBYZANTINE MUSICAL SYMBOL ISAKIA TELOUS ICHIMATOSBYZANTINE MUSICAL SYMBOL APOSTROFOI TELOUS ICHIMATOSBYZANTINE MUSICAL SYMBOL FANEROSIS TETRAFONIASBYZANTINE MUSICAL SYMBOL FANE\
    \ROSIS MONOFONIASBYZANTINE MUSICAL SYMBOL FANEROSIS DIFONIASBYZANTINE MUSICAL SYMBOL MARTYRIA VARYS ICHOSBYZANTINE MUSICAL SYMBOL MARTYRIA PROTOVARYS ICHOSBYZANTINE MUSICAL SYMBOL MARTYRIA PLAGIOS TETARTOS ICHOSBYZANTINE MUSICAL SYMBOL GORTHMIKON N APLOUNB\
    \YZANTINE MUSICAL SYMBOL GORTHMIKON N DIPLOUNBYZANTINE MUSICAL SYMBOL ENARXIS KAI FTHORA VOUBYZANTINE MUSICAL SYMBOL IMIFONONBYZANTINE MUSICAL SYMBOL IMIFTHORONBYZANTINE MUSICAL SYMBOL FTHORA ARCHAION DEYTEROU ICHOUBYZANTINE MUSICAL SYMBOL FTHORA DIATONIKI\
    \ PABYZANTINE MUSICAL SYMBOL FTHORA DIATONIKI NANABYZANTINE MUSICAL SYMBOL FTHORA NAOS ICHOSBYZANTINE MUSICAL SYMBOL FTHORA DIATONIKI DIBYZANTINE MUSICAL SYMBOL FTHORA SKLIRON DIATONON DIBYZANTINE MUSICAL SYMBOL FTHORA DIATONIKI KEBYZANTINE MUSICAL SYMBOL \
    \FTHORA DIATONIKI ZOBYZANTINE MUSICAL SYMBOL FTHORA DIATONIKI NI KATOBYZANTINE MUSICAL SYMBOL FTHORA DIATONIKI NI ANOBYZANTINE MUSICAL SYMBOL FTHORA MALAKON CHROMA DIFONIASBYZANTINE MUSICAL SYMBOL FTHORA MALAKON CHROMA MONOFONIASBYZANTINE MUSICAL SYMBOL FH\
    \TORA SKLIRON CHROMA VASISBYZANTINE MUSICAL SYMBOL FTHORA SKLIRON CHROMA SYNAFIBYZANTINE MUSICAL SYMBOL FTHORA NENANOBYZANTINE MUSICAL SYMBOL CHROA ZYGOSBYZANTINE MUSICAL SYMBOL CHROA KLITONBYZANTINE MUSICAL SYMBOL CHROA SPATHIBYZANTINE MUSICAL SYMBOL FTHO\
    \RA I YFESIS TETARTIMORIONBYZANTINE MUSICAL SYMBOL FTHORA ENARMONIOS ANTIFONIABYZANTINE MUSICAL SYMBOL YFESIS TRITIMORIONBYZANTINE MUSICAL SYMBOL DIESIS TRITIMORIONBYZANTINE MUSICAL SYMBOL DIESIS TETARTIMORIONBYZANTINE MUSICAL SYMBOL DIESIS APLI DYO DODEKA\
    \TABYZANTINE MUSICAL SYMBOL DIESIS MONOGRAMMOS TESSERA DODEKATABYZANTINE MUSICAL SYMBOL DIESIS DIGRAMMOS EX DODEKATABYZANTINE MUSICAL SYMBOL DIESIS TRIGRAMMOS OKTO DODEKATABYZANTINE MUSICAL SYMBOL YFESIS APLI DYO DODEKATABYZANTINE MUSICAL SYMBOL YFESIS MON\
    \OGRAMMOS TESSERA DODEKATABYZANTINE MUSICAL SYMBOL YFESIS DIGRAMMOS EX DODEKATABYZANTINE MUSICAL SYMBOL YFESIS TRIGRAMMOS OKTO DODEKATABYZANTINE MUSICAL SYMBOL GENIKI DIESISBYZANTINE MUSICAL SYMBOL GENIKI YFESISBYZANTINE MUSICAL SYMBOL DIASTOLI APLI MIKRIB\
    \YZANTINE MUSICAL SYMBOL DIASTOLI APLI MEGALIBYZANTINE MUSICAL SYMBOL DIASTOLI DIPLIBYZANTINE MUSICAL SYMBOL DIASTOLI THESEOSBYZANTINE MUSICAL SYMBOL SIMANSIS THESEOSBYZANTINE MUSICAL SYMBOL SIMANSIS THESEOS DISIMOUBYZANTINE MUSICAL SYMBOL SIMANSIS THESEOS\
    \ TRISIMOUBYZANTINE MUSICAL SYMBOL SIMANSIS THESEOS TETRASIMOUBYZANTINE MUSICAL SYMBOL SIMANSIS ARSEOSBYZANTINE MUSICAL SYMBOL SIMANSIS ARSEOS DISIMOUBYZANTINE MUSICAL SYMBOL SIMANSIS ARSEOS TRISIMOUBYZANTINE MUSICAL SYMBOL SIMANSIS ARSEOS TETRASIMOUBYZANT\
    \INE MUSICAL SYMBOL DIGRAMMA GGBYZANTINE MUSICAL SYMBOL DIFTOGGOS OUBYZANTINE MUSICAL SYMBOL STIGMABYZANTINE MUSICAL SYMBOL ARKTIKO PABYZANTINE MUSICAL SYMBOL ARKTIKO VOUBYZANTINE MUSICAL SYMBOL ARKTIKO GABYZANTINE MUSICAL SYMBOL ARKTIKO DIBYZANTINE MUSICA\
    \L SYMBOL ARKTIKO KEBYZANTINE MUSICAL SYMBOL ARKTIKO ZOBYZANTINE MUSICAL SYMBOL ARKTIKO NIBYZANTINE MUSICAL SYMBOL KENTIMATA NEO MESOBYZANTINE MUSICAL SYMBOL KENTIMA NEO MESOBYZANTINE MUSICAL SYMBOL KENTIMATA NEO KATOBYZANTINE MUSICAL SYMBOL KENTIMA NEO KA\
    \TOBYZANTINE MUSICAL SYMBOL KLASMA KATOBYZANTINE MUSICAL SYMBOL GORGON NEO KATOMUSICAL SYMBOL SINGLE BARLINEMUSICAL SYMBOL DOUBLE BARLINEMUSICAL SYMBOL FINAL BARLINEMUSICAL SYMBOL REVERSE FINAL BARLINEMUSICAL SYMBOL DASHED BARLINEMUSICAL SYMBOL SHORT BARLI\
    \NEMUSICAL SYMBOL LEFT REPEAT SIGNMUSICAL SYMBOL RIGHT REPEAT SIGNMUSICAL SYMBOL REPEAT DOTSMUSICAL SYMBOL DAL SEGNOMUSICAL SYMBOL DA CAPOMUSICAL SYMBOL SEGNOMUSICAL SYMBOL CODAMUSICAL SYMBOL REPEATED FIGURE-1MUSICAL SYMBOL REPEATED FIGURE-2MUSICAL SYMBOL \
    \REPEATED FIGURE-3MUSICAL SYMBOL FERMATAMUSICAL SYMBOL FERMATA BELOWMUSICAL SYMBOL BREATH MARKMUSICAL SYMBOL CAESURAMUSICAL SYMBOL BRACEMUSICAL SYMBOL BRACKETMUSICAL SYMBOL ONE-LINE STAFFMUSICAL SYMBOL TWO-LINE STAFFMUSICAL SYMBOL THREE-LINE STAFFMUSICAL S\
    \YMBOL FOUR-LINE STAFFMUSICAL SYMBOL FIVE-LINE STAFFMUSICAL SYMBOL SIX-LINE STAFFMUSICAL SYMBOL SIX-STRING FRETBOARDMUSICAL SYMBOL FOUR-STRING FRETBOARDMUSICAL SYMBOL G CLEFMUSICAL SYMBOL G CLEF OTTAVA ALTAMUSICAL SYMBOL G CLEF OTTAVA BASSAMUSICAL SYMBOL C\
    \ CLEFMUSICAL SYMBOL F CLEFMUSICAL SYMBOL F CLEF OTTAVA ALTAMUSICAL SYMBOL F CLEF OTTAVA BASSAMUSICAL SYMBOL DRUM CLEF-1MUSICAL SYMBOL DRUM CLEF-2MUSICAL SYMBOL MULTIPLE MEASURE RESTMUSICAL SYMBOL DOUBLE SHARPMUSICAL SYMBOL DOUBLE FLATMUSICAL SYMBOL FLAT U\
    \PMUSICAL SYMBOL FLAT DOWNMUSICAL SYMBOL NATURAL UPMUSICAL SYMBOL NATURAL DOWNMUSICAL SYMBOL SHARP UPMUSICAL SYMBOL SHARP DOWNMUSICAL SYMBOL QUARTER TONE SHARPMUSICAL SYMBOL QUARTER TONE FLATMUSICAL SYMBOL COMMON TIMEMUSICAL SYMBOL CUT TIMEMUSICAL SYMBOL O\
    \TTAVA ALTAMUSICAL SYMBOL OTTAVA BASSAMUSICAL SYMBOL QUINDICESIMA ALTAMUSICAL SYMBOL QUINDICESIMA BASSAMUSICAL SYMBOL MULTI RESTMUSICAL SYMBOL WHOLE RESTMUSICAL SYMBOL HALF RESTMUSICAL SYMBOL QUARTER RESTMUSICAL SYMBOL EIGHTH RESTMUSICAL SYMBOL SIXTEENTH R\
    \ESTMUSICAL SYMBOL THIRTY-SECOND RESTMUSICAL SYMBOL SIXTY-FOURTH RESTMUSICAL SYMBOL ONE HUNDRED TWENTY-EIGHTH RESTMUSICAL SYMBOL X NOTEHEADMUSICAL SYMBOL PLUS NOTEHEADMUSICAL SYMBOL CIRCLE X NOTEHEADMUSICAL SYMBOL SQUARE NOTEHEAD WHITEMUSICAL SYMBOL SQUARE\
    \ NOTEHEAD BLACKMUSICAL SYMBOL TRIANGLE NOTEHEAD UP WHITEMUSICAL SYMBOL TRIANGLE NOTEHEAD UP BLACKMUSICAL SYMBOL TRIANGLE NOTEHEAD LEFT WHITEMUSICAL SYMBOL TRIANGLE NOTEHEAD LEFT BLACKMUSICAL SYMBOL TRIANGLE NOTEHEAD RIGHT WHITEMUSICAL SYMBOL TRIANGLE NOTE\
    \HEAD RIGHT BLACKMUSICAL SYMBOL TRIANGLE NOTEHEAD DOWN WHITEMUSICAL SYMBOL TRIANGLE NOTEHEAD DOWN BLACKMUSICAL SYMBOL TRIANGLE NOTEHEAD UP RIGHT WHITEMUSICAL SYMBOL TRIANGLE NOTEHEAD UP RIGHT BLACKMUSICAL SYMBOL MOON NOTEHEAD WHITEMUSICAL SYMBOL MOON NOTEH\
    \EAD BLACKMUSICAL SYMBOL TRIANGLE-ROUND NOTEHEAD DOWN WHITEMUSICAL SYMBOL TRIANGLE-ROUND NOTEHEAD DOWN BLACKMUSICAL SYMBOL PARENTHESIS NOTEHEADMUSICAL SYMBOL VOID NOTEHEADMUSICAL SYMBOL NOTEHEAD BLACKMUSICAL SYMBOL NULL NOTEHEADMUSICAL SYMBOL CLUSTER NOTEH\
    \EAD WHITEMUSICAL SYMBOL CLUSTER NOTEHEAD BLACKMUSICAL SYMBOL BREVEMUSICAL SYMBOL WHOLE NOTEMUSICAL SYMBOL HALF NOTEMUSICAL SYMBOL QUARTER NOTEMUSICAL SYMBOL EIGHTH NOTEMUSICAL SYMBOL SIXTEENTH NOTEMUSICAL SYMBOL THIRTY-SECOND NOTEMUSICAL SYMBOL SIXTY-FOUR\
    \TH NOTEMUSICAL SYMBOL ONE HUNDRED TWENTY-EIGHTH NOTEMUSICAL SYMBOL COMBINING STEMMUSICAL SYMBOL COMBINING SPRECHGESANG STEMMUSICAL SYMBOL COMBINING TREMOLO-1MUSICAL SYMBOL COMBINING TREMOLO-2MUSICAL SYMBOL COMBINING TREMOLO-3MUSICAL SYMBOL FINGERED TREMOL\
    \O-1MUSICAL SYMBOL FINGERED TREMOLO-2MUSICAL SYMBOL FINGERED TREMOLO-3MUSICAL SYMBOL COMBINING AUGMENTATION DOTMUSICAL SYMBOL COMBINING FLAG-1MUSICAL SYMBOL COMBINING FLAG-2MUSICAL SYMBOL COMBINING FLAG-3MUSICAL SYMBOL COMBINING FLAG-4MUSICAL SYMBOL COMBIN\
    \ING FLAG-5MUSICAL SYMBOL BEGIN BEAMMUSICAL SYMBOL END BEAMMUSICAL SYMBOL BEGIN TIEMUSICAL SYMBOL END TIEMUSICAL SYMBOL BEGIN SLURMUSICAL SYMBOL END SLURMUSICAL SYMBOL BEGIN PHRASEMUSICAL SYMBOL END PHRASEMUSICAL SYMBOL COMBINING ACCENTMUSICAL SYMBOL COMBI\
    \NING STACCATOMUSICAL SYMBOL COMBINING TENUTOMUSICAL SYMBOL COMBINING STACCATISSIMOMUSICAL SYMBOL COMBINING MARCATOMUSICAL SYMBOL COMBINING MARCATO-STACCATOMUSICAL SYMBOL COMBINING ACCENT-STACCATOMUSICAL SYMBOL COMBINING LOUREMUSICAL SYMBOL ARPEGGIATO UPMU\
    \SICAL SYMBOL ARPEGGIATO DOWNMUSICAL SYMBOL COMBINING DOITMUSICAL SYMBOL COMBINING RIPMUSICAL SYMBOL COMBINING FLIPMUSICAL SYMBOL COMBINING SMEARMUSICAL SYMBOL COMBINING BENDMUSICAL SYMBOL COMBINING DOUBLE TONGUEMUSICAL SYMBOL COMBINING TRIPLE TONGUEMUSICA\
    \L SYMBOL RINFORZANDOMUSICAL SYMBOL SUBITOMUSICAL SYMBOL ZMUSICAL SYMBOL PIANOMUSICAL SYMBOL MEZZOMUSICAL SYMBOL FORTEMUSICAL SYMBOL CRESCENDOMUSICAL SYMBOL DECRESCENDOMUSICAL SYMBOL GRACE NOTE SLASHMUSICAL SYMBOL GRACE NOTE NO SLASHMUSICAL SYMBOL TRMUSICA\
    \L SYMBOL TURNMUSICAL SYMBOL INVERTED TURNMUSICAL SYMBOL TURN SLASHMUSICAL SYMBOL TURN UPMUSICAL SYMBOL ORNAMENT STROKE-1MUSICAL SYMBOL ORNAMENT STROKE-2MUSICAL SYMBOL ORNAMENT STROKE-3MUSICAL SYMBOL ORNAMENT STROKE-4MUSICAL SYMBOL ORNAMENT STROKE-5MUSICAL\
    \ SYMBOL ORNAMENT STROKE-6MUSICAL SYMBOL ORNAMENT STROKE-7MUSICAL SYMBOL ORNAMENT STROKE-8MUSICAL SYMBOL ORNAMENT STROKE-9MUSICAL SYMBOL ORNAMENT STROKE-10MUSICAL SYMBOL ORNAMENT STROKE-11MUSICAL SYMBOL HAUPTSTIMMEMUSICAL SYMBOL NEBENSTIMMEMUSICAL SYMBOL E\
    \ND OF STIMMEMUSICAL SYMBOL DEGREE SLASHMUSICAL SYMBOL COMBINING DOWN BOWMUSICAL SYMBOL COMBINING UP BOWMUSICAL SYMBOL COMBINING HARMONICMUSICAL SYMBOL COMBINING SNAP PIZZICATOMUSICAL SYMBOL PEDAL MARKMUSICAL SYMBOL PEDAL UP MARKMUSICAL SYMBOL HALF PEDAL M\
    \ARKMUSICAL SYMBOL GLISSANDO UPMUSICAL SYMBOL GLISSANDO DOWNMUSICAL SYMBOL WITH FINGERNAILSMUSICAL SYMBOL DAMPMUSICAL SYMBOL DAMP ALLMUSICAL SYMBOL MAXIMAMUSICAL SYMBOL LONGAMUSICAL SYMBOL BREVISMUSICAL SYMBOL SEMIBREVIS WHITEMUSICAL SYMBOL SEMIBREVIS BLAC\
    \KMUSICAL SYMBOL MINIMAMUSICAL SYMBOL MINIMA BLACKMUSICAL SYMBOL SEMIMINIMA WHITEMUSICAL SYMBOL SEMIMINIMA BLACKMUSICAL SYMBOL FUSA WHITEMUSICAL SYMBOL FUSA BLACKMUSICAL SYMBOL LONGA PERFECTA RESTMUSICAL SYMBOL LONGA IMPERFECTA RESTMUSICAL SYMBOL BREVIS RE\
    \STMUSICAL SYMBOL SEMIBREVIS RESTMUSICAL SYMBOL MINIMA RESTMUSICAL SYMBOL SEMIMINIMA RESTMUSICAL SYMBOL TEMPUS PERFECTUM CUM PROLATIONE PERFECTAMUSICAL SYMBOL TEMPUS PERFECTUM CUM PROLATIONE IMPERFECTAMUSICAL SYMBOL TEMPUS PERFECTUM CUM PROLATIONE PERFECTA\
    \ DIMINUTION-1MUSICAL SYMBOL TEMPUS IMPERFECTUM CUM PROLATIONE PERFECTAMUSICAL SYMBOL TEMPUS IMPERFECTUM CUM PROLATIONE IMPERFECTAMUSICAL SYMBOL TEMPUS IMPERFECTUM CUM PROLATIONE IMPERFECTA DIMINUTION-1MUSICAL SYMBOL TEMPUS IMPERFECTUM CUM PROLATIONE IMPER\
    \FECTA DIMINUTION-2MUSICAL SYMBOL TEMPUS IMPERFECTUM CUM PROLATIONE IMPERFECTA DIMINUTION-3MUSICAL SYMBOL CROIXMUSICAL SYMBOL GREGORIAN C CLEFMUSICAL SYMBOL GREGORIAN F CLEFMUSICAL SYMBOL SQUARE BMUSICAL SYMBOL VIRGAMUSICAL SYMBOL PODATUSMUSICAL SYMBOL CLI\
    \VISMUSICAL SYMBOL SCANDICUSMUSICAL SYMBOL CLIMACUSMUSICAL SYMBOL TORCULUSMUSICAL SYMBOL PORRECTUSMUSICAL SYMBOL PORRECTUS FLEXUSMUSICAL SYMBOL SCANDICUS FLEXUSMUSICAL SYMBOL TORCULUS RESUPINUSMUSICAL SYMBOL PES SUBPUNCTISMUSICAL SYMBOL KIEVAN C CLEFMUSICA\
    \L SYMBOL KIEVAN END OF PIECEMUSICAL SYMBOL KIEVAN FINAL NOTEMUSICAL SYMBOL KIEVAN RECITATIVE MARKMUSICAL SYMBOL KIEVAN WHOLE NOTEMUSICAL SYMBOL KIEVAN HALF NOTEMUSICAL SYMBOL KIEVAN QUARTER NOTE STEM DOWNMUSICAL SYMBOL KIEVAN QUARTER NOTE STEM UPMUSICAL S\
    \YMBOL KIEVAN EIGHTH NOTE STEM DOWNMUSICAL SYMBOL KIEVAN EIGHTH NOTE STEM UPMUSICAL SYMBOL KIEVAN FLAT SIGNMUSICAL SYMBOL SORIMUSICAL SYMBOL KORONGREEK VOCAL NOTATION SYMBOL-1GREEK VOCAL NOTATION SYMBOL-2GREEK VOCAL NOTATION SYMBOL-3GREEK VOCAL NOTATION SY\
    \MBOL-4GREEK VOCAL NOTATION SYMBOL-5GREEK VOCAL NOTATION SYMBOL-6GREEK VOCAL NOTATION SYMBOL-7GREEK VOCAL NOTATION SYMBOL-8GREEK VOCAL NOTATION SYMBOL-9GREEK VOCAL NOTATION SYMBOL-10GREEK VOCAL NOTATION SYMBOL-11GREEK VOCAL NOTATION SYMBOL-12GREEK VOCAL NO\
    \TATION SYMBOL-13GREEK VOCAL NOTATION SYMBOL-14GREEK VOCAL NOTATION SYMBOL-15GREEK VOCAL NOTATION SYMBOL-16GREEK VOCAL NOTATION SYMBOL-17GREEK VOCAL NOTATION SYMBOL-18GREEK VOCAL NOTATION SYMBOL-19GREEK VOCAL NOTATION SYMBOL-20GREEK VOCAL NOTATION SYMBOL-2\
    \1GREEK VOCAL NOTATION SYMBOL-22GREEK VOCAL NOTATION SYMBOL-23GREEK VOCAL NOTATION SYMBOL-24GREEK VOCAL NOTATION SYMBOL-50GREEK VOCAL NOTATION SYMBOL-51GREEK VOCAL NOTATION SYMBOL-52GREEK VOCAL NOTATION SYMBOL-53GREEK VOCAL NOTATION SYMBOL-54GREEK INSTRUME\
    \NTAL NOTATION SYMBOL-1GREEK INSTRUMENTAL NOTATION SYMBOL-2GREEK INSTRUMENTAL NOTATION SYMBOL-4GREEK INSTRUMENTAL NOTATION SYMBOL-5GREEK INSTRUMENTAL NOTATION SYMBOL-7GREEK INSTRUMENTAL NOTATION SYMBOL-8GREEK INSTRUMENTAL NOTATION SYMBOL-11GREEK INSTRUMENT\
    \AL NOTATION SYMBOL-12GREEK INSTRUMENTAL NOTATION SYMBOL-13GREEK INSTRUMENTAL NOTATION SYMBOL-14GREEK INSTRUMENTAL NOTATION SYMBOL-17GREEK INSTRUMENTAL NOTATION SYMBOL-18GREEK INSTRUMENTAL NOTATION SYMBOL-19GREEK INSTRUMENTAL NOTATION SYMBOL-23GREEK INSTRU\
    \MENTAL NOTATION SYMBOL-24GREEK INSTRUMENTAL NOTATION SYMBOL-25GREEK INSTRUMENTAL NOTATION SYMBOL-26GREEK INSTRUMENTAL NOTATION SYMBOL-27GREEK INSTRUMENTAL NOTATION SYMBOL-29GREEK INSTRUMENTAL NOTATION SYMBOL-30GREEK INSTRUMENTAL NOTATION SYMBOL-32GREEK IN\
    \STRUMENTAL NOTATION SYMBOL-36GREEK INSTRUMENTAL NOTATION SYMBOL-37GREEK INSTRUMENTAL NOTATION SYMBOL-38GREEK INSTRUMENTAL NOTATION SYMBOL-39GREEK INSTRUMENTAL NOTATION SYMBOL-40GREEK INSTRUMENTAL NOTATION SYMBOL-42GREEK INSTRUMENTAL NOTATION SYMBOL-43GREE\
    \K INSTRUMENTAL NOTATION SYMBOL-45GREEK INSTRUMENTAL NOTATION SYMBOL-47GREEK INSTRUMENTAL NOTATION SYMBOL-48GREEK INSTRUMENTAL NOTATION SYMBOL-49GREEK INSTRUMENTAL NOTATION SYMBOL-50GREEK INSTRUMENTAL NOTATION SYMBOL-51GREEK INSTRUMENTAL NOTATION SYMBOL-52\
    \GREEK INSTRUMENTAL NOTATION SYMBOL-53GREEK INSTRUMENTAL NOTATION SYMBOL-54COMBINING GREEK MUSICAL TRISEMECOMBINING GREEK MUSICAL TETRASEMECOMBINING GREEK MUSICAL PENTASEMEGREEK MUSICAL LEIMMAKAKTOVIK NUMERAL ZEROKAKTOVIK NUMERAL ONEKAKTOVIK NUMERAL TWOKAK\
    \TOVIK NUMERAL THREEKAKTOVIK NUMERAL FOURKAKTOVIK NUMERAL FIVEKAKTOVIK NUMERAL SIXKAKTOVIK NUMERAL SEVENKAKTOVIK NUMERAL EIGHTKAKTOVIK NUMERAL NINEKAKTOVIK NUMERAL TENKAKTOVIK NUMERAL ELEVENKAKTOVIK NUMERAL TWELVEKAKTOVIK NUMERAL THIRTEENKAKTOVIK NUMERAL F\
    \OURTEENKAKTOVIK NUMERAL FIFTEENKAKTOVIK NUMERAL SIXTEENKAKTOVIK NUMERAL SEVENTEENKAKTOVIK NUMERAL EIGHTEENKAKTOVIK NUMERAL NINETEENMAYAN NUMERAL ZEROMAYAN NUMERAL ONEMAYAN NUMERAL TWOMAYAN NUMERAL THREEMAYAN NUMERAL FOURMAYAN NUMERAL FIVEMAYAN NUMERAL SIX\
    \MAYAN NUMERAL SEVENMAYAN NUMERAL EIGHTMAYAN NUMERAL NINEMAYAN NUMERAL TENMAYAN NUMERAL ELEVENMAYAN NUMERAL TWELVEMAYAN NUMERAL THIRTEENMAYAN NUMERAL FOURTEENMAYAN NUMERAL FIFTEENMAYAN NUMERAL SIXTEENMAYAN NUMERAL SEVENTEENMAYAN NUMERAL EIGHTEENMAYAN NUMER\
    \AL NINETEENMONOGRAM FOR EARTHDIGRAM FOR HEAVENLY EARTHDIGRAM FOR HUMAN EARTHDIGRAM FOR EARTHLY HEAVENDIGRAM FOR EARTHLY HUMANDIGRAM FOR EARTHTETRAGRAM FOR CENTRETETRAGRAM FOR FULL CIRCLETETRAGRAM FOR MIREDTETRAGRAM FOR BARRIERTETRAGRAM FOR KEEPING SMALLTE\
    \TRAGRAM FOR CONTRARIETYTETRAGRAM FOR ASCENTTETRAGRAM FOR OPPOSITIONTETRAGRAM FOR BRANCHING OUTTETRAGRAM FOR DEFECTIVENESS OR DISTORTIONTETRAGRAM FOR DIVERGENCETETRAGRAM FOR YOUTHFULNESSTETRAGRAM FOR INCREASETETRAGRAM FOR PENETRATIONTETRAGRAM FOR REACHTETR\
    \AGRAM FOR CONTACTTETRAGRAM FOR HOLDING BACKTETRAGRAM FOR WAITINGTETRAGRAM FOR FOLLOWINGTETRAGRAM FOR ADVANCETETRAGRAM FOR RELEASETETRAGRAM FOR RESISTANCETETRAGRAM FOR EASETETRAGRAM FOR JOYTETRAGRAM FOR CONTENTIONTETRAGRAM FOR ENDEAVOURTETRAGRAM FOR DUTIES\
    \TETRAGRAM FOR CHANGETETRAGRAM FOR DECISIVENESSTETRAGRAM FOR BOLD RESOLUTIONTETRAGRAM FOR PACKINGTETRAGRAM FOR LEGIONTETRAGRAM FOR CLOSENESSTETRAGRAM FOR KINSHIPTETRAGRAM FOR GATHERINGTETRAGRAM FOR STRENGTHTETRAGRAM FOR PURITYTETRAGRAM FOR FULLNESSTETRAGRA\
    \M FOR RESIDENCETETRAGRAM FOR LAW OR MODELTETRAGRAM FOR RESPONSETETRAGRAM FOR GOING TO MEETTETRAGRAM FOR ENCOUNTERSTETRAGRAM FOR STOVETETRAGRAM FOR GREATNESSTETRAGRAM FOR ENLARGEMENTTETRAGRAM FOR PATTERNTETRAGRAM FOR RITUALTETRAGRAM FOR FLIGHTTETRAGRAM FOR\
    \ VASTNESS OR WASTINGTETRAGRAM FOR CONSTANCYTETRAGRAM FOR MEASURETETRAGRAM FOR ETERNITYTETRAGRAM FOR UNITYTETRAGRAM FOR DIMINISHMENTTETRAGRAM FOR CLOSED MOUTHTETRAGRAM FOR GUARDEDNESSTETRAGRAM FOR GATHERING INTETRAGRAM FOR MASSINGTETRAGRAM FOR ACCUMULATION\
    \TETRAGRAM FOR EMBELLISHMENTTETRAGRAM FOR DOUBTTETRAGRAM FOR WATCHTETRAGRAM FOR SINKINGTETRAGRAM FOR INNERTETRAGRAM FOR DEPARTURETETRAGRAM FOR DARKENINGTETRAGRAM FOR DIMMINGTETRAGRAM FOR EXHAUSTIONTETRAGRAM FOR SEVERANCETETRAGRAM FOR STOPPAGETETRAGRAM FOR \
    \HARDNESSTETRAGRAM FOR COMPLETIONTETRAGRAM FOR CLOSURETETRAGRAM FOR FAILURETETRAGRAM FOR AGGRAVATIONTETRAGRAM FOR COMPLIANCETETRAGRAM FOR ON THE VERGETETRAGRAM FOR DIFFICULTIESTETRAGRAM FOR LABOURINGTETRAGRAM FOR FOSTERINGCOUNTING ROD UNIT DIGIT ONECOUNTIN\
    \G ROD UNIT DIGIT TWOCOUNTING ROD UNIT DIGIT THREECOUNTING ROD UNIT DIGIT FOURCOUNTING ROD UNIT DIGIT FIVECOUNTING ROD UNIT DIGIT SIXCOUNTING ROD UNIT DIGIT SEVENCOUNTING ROD UNIT DIGIT EIGHTCOUNTING ROD UNIT DIGIT NINECOUNTING ROD TENS DIGIT ONECOUNTING R\
    \OD TENS DIGIT TWOCOUNTING ROD TENS DIGIT THREECOUNTING ROD TENS DIGIT FOURCOUNTING ROD TENS DIGIT FIVECOUNTING ROD TENS DIGIT SIXCOUNTING ROD TENS DIGIT SEVENCOUNTING ROD TENS DIGIT EIGHTCOUNTING ROD TENS DIGIT NINEIDEOGRAPHIC TALLY MARK ONEIDEOGRAPHIC TA\
    \LLY MARK TWOIDEOGRAPHIC TALLY MARK THREEIDEOGRAPHIC TALLY MARK FOURIDEOGRAPHIC TALLY MARK FIVETALLY MARK ONETALLY MARK FIVEMATHEMATICAL BOLD CAPITAL AMATHEMATICAL BOLD CAPITAL BMATHEMATICAL BOLD CAPITAL CMATHEMATICAL BOLD CAPITAL DMATHEMATICAL BOLD CAPITA\
    \L EMATHEMATICAL BOLD CAPITAL FMATHEMATICAL BOLD CAPITAL GMATHEMATICAL BOLD CAPITAL HMATHEMATICAL BOLD CAPITAL IMATHEMATICAL BOLD CAPITAL JMATHEMATICAL BOLD CAPITAL KMATHEMATICAL BOLD CAPITAL LMATHEMATICAL BOLD CAPITAL MMATHEMATICAL BOLD CAPITAL NMATHEMATI\
    \CAL BOLD CAPITAL OMATHEMATICAL BOLD CAPITAL PMATHEMATICAL BOLD CAPITAL QMATHEMATICAL BOLD CAPITAL RMATHEMATICAL BOLD CAPITAL SMATHEMATICAL BOLD CAPITAL TMATHEMATICAL BOLD CAPITAL UMATHEMATICAL BOLD CAPITAL VMATHEMATICAL BOLD CAPITAL WMATHEMATICAL BOLD CAP\
    \ITAL XMATHEMATICAL BOLD CAPITAL YMATHEMATICAL BOLD CAPITAL ZMATHEMATICAL BOLD SMALL AMATHEMATICAL BOLD SMALL BMATHEMATICAL BOLD SMALL CMATHEMATICAL BOLD SMALL DMATHEMATICAL BOLD SMALL EMATHEMATICAL BOLD SMALL FMATHEMATICAL BOLD SMALL GMATHEMATICAL BOLD SM\
    \ALL HMATHEMATICAL BOLD SMALL IMATHEMATICAL BOLD SMALL JMATHEMATICAL BOLD SMALL KMATHEMATICAL BOLD SMALL LMATHEMATICAL BOLD SMALL MMATHEMATICAL BOLD SMALL NMATHEMATICAL BOLD SMALL OMATHEMATICAL BOLD SMALL PMATHEMATICAL BOLD SMALL QMATHEMATICAL BOLD SMALL R\
    \MATHEMATICAL BOLD SMALL SMATHEMATICAL BOLD SMALL TMATHEMATICAL BOLD SMALL UMATHEMATICAL BOLD SMALL VMATHEMATICAL BOLD SMALL WMATHEMATICAL BOLD SMALL XMATHEMATICAL BOLD SMALL YMATHEMATICAL BOLD SMALL ZMATHEMATICAL ITALIC CAPITAL AMATHEMATICAL ITALIC CAPITA\
    \L BMATHEMATICAL ITALIC CAPITAL CMATHEMATICAL ITALIC CAPITAL DMATHEMATICAL ITALIC CAPITAL EMATHEMATICAL ITALIC CAPITAL FMATHEMATICAL ITALIC CAPITAL GMATHEMATICAL ITALIC CAPITAL HMATHEMATICAL ITALIC CAPITAL IMATHEMATICAL ITALIC CAPITAL JMATHEMATICAL ITALIC \
    \CAPITAL KMATHEMATICAL ITALIC CAPITAL LMATHEMATICAL ITALIC CAPITAL MMATHEMATICAL ITALIC CAPITAL NMATHEMATICAL ITALIC CAPITAL OMATHEMATICAL ITALIC CAPITAL PMATHEMATICAL ITALIC CAPITAL QMATHEMATICAL ITALIC CAPITAL RMATHEMATICAL ITALIC CAPITAL SMATHEMATICAL I\
    \TALIC CAPITAL TMATHEMATICAL ITALIC CAPITAL UMATHEMATICAL ITALIC CAPITAL VMATHEMATICAL ITALIC CAPITAL WMATHEMATICAL ITALIC CAPITAL XMATHEMATICAL ITALIC CAPITAL YMATHEMATICAL ITALIC CAPITAL ZMATHEMATICAL ITALIC SMALL AMATHEMATICAL ITALIC SMALL BMATHEMATICAL\
    \ ITALIC SMALL CMATHEMATICAL ITALIC SMALL DMATHEMATICAL ITALIC SMALL EMATHEMATICAL ITALIC SMALL FMATHEMATICAL ITALIC SMALL GMATHEMATICAL ITALIC SMALL IMATHEMATICAL ITALIC SMALL JMATHEMATICAL ITALIC SMALL KMATHEMATICAL ITALIC SMALL LMATHEMATICAL ITALIC SMAL\
    \L MMATHEMATICAL ITALIC SMALL NMATHEMATICAL ITALIC SMALL OMATHEMATICAL ITALIC SMALL PMATHEMATICAL ITALIC SMALL QMATHEMATICAL ITALIC SMALL RMATHEMATICAL ITALIC SMALL SMATHEMATICAL ITALIC SMALL TMATHEMATICAL ITALIC SMALL UMATHEMATICAL ITALIC SMALL VMATHEMATI\
    \CAL ITALIC SMALL WMATHEMATICAL ITALIC SMALL XMATHEMATICAL ITALIC SMALL YMATHEMATICAL ITALIC SMALL ZMATHEMATICAL BOLD ITALIC CAPITAL AMATHEMATICAL BOLD ITALIC CAPITAL BMATHEMATICAL BOLD ITALIC CAPITAL CMATHEMATICAL BOLD ITALIC CAPITAL DMATHEMATICAL BOLD IT\
    \ALIC CAPITAL EMATHEMATICAL BOLD ITALIC CAPITAL FMATHEMATICAL BOLD ITALIC CAPITAL GMATHEMATICAL BOLD ITALIC CAPITAL HMATHEMATICAL BOLD ITALIC CAPITAL IMATHEMATICAL BOLD ITALIC CAPITAL JMATHEMATICAL BOLD ITALIC CAPITAL KMATHEMATICAL BOLD ITALIC CAPITAL LMAT\
    \HEMATICAL BOLD ITALIC CAPITAL MMATHEMATICAL BOLD ITALIC CAPITAL NMATHEMATICAL BOLD ITALIC CAPITAL OMATHEMATICAL BOLD ITALIC CAPITAL PMATHEMATICAL BOLD ITALIC CAPITAL QMATHEMATICAL BOLD ITALIC CAPITAL RMATHEMATICAL BOLD ITALIC CAPITAL SMATHEMATICAL BOLD IT\
    \ALIC CAPITAL TMATHEMATICAL BOLD ITALIC CAPITAL UMATHEMATICAL BOLD ITALIC CAPITAL VMATHEMATICAL BOLD ITALIC CAPITAL WMATHEMATICAL BOLD ITALIC CAPITAL XMATHEMATICAL BOLD ITALIC CAPITAL YMATHEMATICAL BOLD ITALIC CAPITAL ZMATHEMATICAL BOLD ITALIC SMALL AMATHE\
    \MATICAL BOLD ITALIC SMALL BMATHEMATICAL BOLD ITALIC SMALL CMATHEMATICAL BOLD ITALIC SMALL DMATHEMATICAL BOLD ITALIC SMALL EMATHEMATICAL BOLD ITALIC SMALL FMATHEMATICAL BOLD ITALIC SMALL GMATHEMATICAL BOLD ITALIC SMALL HMATHEMATICAL BOLD ITALIC SMALL IMATH\
    \EMATICAL BOLD ITALIC SMALL JMATHEMATICAL BOLD ITALIC SMALL KMATHEMATICAL BOLD ITALIC SMALL LMATHEMATICAL BOLD ITALIC SMALL MMATHEMATICAL BOLD ITALIC SMALL NMATHEMATICAL BOLD ITALIC SMALL OMATHEMATICAL BOLD ITALIC SMALL PMATHEMATICAL BOLD ITALIC SMALL QMAT\
    \HEMATICAL BOLD ITALIC SMALL RMATHEMATICAL BOLD ITALIC SMALL SMATHEMATICAL BOLD ITALIC SMALL TMATHEMATICAL BOLD ITALIC SMALL UMATHEMATICAL BOLD ITALIC SMALL VMATHEMATICAL BOLD ITALIC SMALL WMATHEMATICAL BOLD ITALIC SMALL XMATHEMATICAL BOLD ITALIC SMALL YMA\
    \THEMATICAL BOLD ITALIC SMALL ZMATHEMATICAL SCRIPT CAPITAL AMATHEMATICAL SCRIPT CAPITAL CMATHEMATICAL SCRIPT CAPITAL DMATHEMATICAL SCRIPT CAPITAL GMATHEMATICAL SCRIPT CAPITAL JMATHEMATICAL SCRIPT CAPITAL KMATHEMATICAL SCRIPT CAPITAL NMATHEMATICAL SCRIPT CA\
    \PITAL OMATHEMATICAL SCRIPT CAPITAL PMATHEMATICAL SCRIPT CAPITAL QMATHEMATICAL SCRIPT CAPITAL SMATHEMATICAL SCRIPT CAPITAL TMATHEMATICAL SCRIPT CAPITAL UMATHEMATICAL SCRIPT CAPITAL VMATHEMATICAL SCRIPT CAPITAL WMATHEMATICAL SCRIPT CAPITAL XMATHEMATICAL SCR\
    \IPT CAPITAL YMATHEMATICAL SCRIPT CAPITAL ZMATHEMATICAL SCRIPT SMALL AMATHEMATICAL SCRIPT SMALL BMATHEMATICAL SCRIPT SMALL CMATHEMATICAL SCRIPT SMALL DMATHEMATICAL SCRIPT SMALL FMATHEMATICAL SCRIPT SMALL HMATHEMATICAL SCRIPT SMALL IMATHEMATICAL SCRIPT SMAL\
    \L JMATHEMATICAL SCRIPT SMALL KMATHEMATICAL SCRIPT SMALL LMATHEMATICAL SCRIPT SMALL MMATHEMATICAL SCRIPT SMALL NMATHEMATICAL SCRIPT SMALL PMATHEMATICAL SCRIPT SMALL QMATHEMATICAL SCRIPT SMALL RMATHEMATICAL SCRIPT SMALL SMATHEMATICAL SCRIPT SMALL TMATHEMATI\
    \CAL SCRIPT SMALL UMATHEMATICAL SCRIPT SMALL VMATHEMATICAL SCRIPT SMALL WMATHEMATICAL SCRIPT SMALL XMATHEMATICAL SCRIPT SMALL YMATHEMATICAL SCRIPT SMALL ZMATHEMATICAL BOLD SCRIPT CAPITAL AMATHEMATICAL BOLD SCRIPT CAPITAL BMATHEMATICAL BOLD SCRIPT CAPITAL C\
    \MATHEMATICAL BOLD SCRIPT CAPITAL DMATHEMATICAL BOLD SCRIPT CAPITAL EMATHEMATICAL BOLD SCRIPT CAPITAL FMATHEMATICAL BOLD SCRIPT CAPITAL GMATHEMATICAL BOLD SCRIPT CAPITAL HMATHEMATICAL BOLD SCRIPT CAPITAL IMATHEMATICAL BOLD SCRIPT CAPITAL JMATHEMATICAL BOLD\
    \ SCRIPT CAPITAL KMATHEMATICAL BOLD SCRIPT CAPITAL LMATHEMATICAL BOLD SCRIPT CAPITAL MMATHEMATICAL BOLD SCRIPT CAPITAL NMATHEMATICAL BOLD SCRIPT CAPITAL OMATHEMATICAL BOLD SCRIPT CAPITAL PMATHEMATICAL BOLD SCRIPT CAPITAL QMATHEMATICAL BOLD SCRIPT CAPITAL R\
    \MATHEMATICAL BOLD SCRIPT CAPITAL SMATHEMATICAL BOLD SCRIPT CAPITAL TMATHEMATICAL BOLD SCRIPT CAPITAL UMATHEMATICAL BOLD SCRIPT CAPITAL VMATHEMATICAL BOLD SCRIPT CAPITAL WMATHEMATICAL BOLD SCRIPT CAPITAL XMATHEMATICAL BOLD SCRIPT CAPITAL YMATHEMATICAL BOLD\
    \ SCRIPT CAPITAL ZMATHEMATICAL BOLD SCRIPT SMALL AMATHEMATICAL BOLD SCRIPT SMALL BMATHEMATICAL BOLD SCRIPT SMALL CMATHEMATICAL BOLD SCRIPT SMALL DMATHEMATICAL BOLD SCRIPT SMALL EMATHEMATICAL BOLD SCRIPT SMALL FMATHEMATICAL BOLD SCRIPT SMALL GMATHEMATICAL B\
    \OLD SCRIPT SMALL HMATHEMATICAL BOLD SCRIPT SMALL IMATHEMATICAL BOLD SCRIPT SMALL JMATHEMATICAL BOLD SCRIPT SMALL KMATHEMATICAL BOLD SCRIPT SMALL LMATHEMATICAL BOLD SCRIPT SMALL MMATHEMATICAL BOLD SCRIPT SMALL NMATHEMATICAL BOLD SCRIPT SMALL OMATHEMATICAL \
    \BOLD SCRIPT SMALL PMATHEMATICAL BOLD SCRIPT SMALL QMATHEMATICAL BOLD SCRIPT SMALL RMATHEMATICAL BOLD SCRIPT SMALL SMATHEMATICAL BOLD SCRIPT SMALL TMATHEMATICAL BOLD SCRIPT SMALL UMATHEMATICAL BOLD SCRIPT SMALL VMATHEMATICAL BOLD SCRIPT SMALL WMATHEMATICAL\
    \ BOLD SCRIPT SMALL XMATHEMATICAL BOLD SCRIPT SMALL YMATHEMATICAL BOLD SCRIPT SMALL ZMATHEMATICAL FRAKTUR CAPITAL AMATHEMATICAL FRAKTUR CAPITAL BMATHEMATICAL FRAKTUR CAPITAL DMATHEMATICAL FRAKTUR CAPITAL EMATHEMATICAL FRAKTUR CAPITAL FMATHEMATICAL FRAKTUR \
    \CAPITAL GMATHEMATICAL FRAKTUR CAPITAL JMATHEMATICAL FRAKTUR CAPITAL KMATHEMATICAL FRAKTUR CAPITAL LMATHEMATICAL FRAKTUR CAPITAL MMATHEMATICAL FRAKTUR CAPITAL NMATHEMATICAL FRAKTUR CAPITAL OMATHEMATICAL FRAKTUR CAPITAL PMATHEMATICAL FRAKTUR CAPITAL QMATHEM\
    \ATICAL FRAKTUR CAPITAL SMATHEMATICAL FRAKTUR CAPITAL TMATHEMATICAL FRAKTUR CAPITAL UMATHEMATICAL FRAKTUR CAPITAL VMATHEMATICAL FRAKTUR CAPITAL WMATHEMATICAL FRAKTUR CAPITAL XMATHEMATICAL FRAKTUR CAPITAL YMATHEMATICAL FRAKTUR SMALL AMATHEMATICAL FRAKTUR SM\
    \ALL BMATHEMATICAL FRAKTUR SMALL CMATHEMATICAL FRAKTUR SMALL DMATHEMATICAL FRAKTUR SMALL EMATHEMATICAL FRAKTUR SMALL FMATHEMATICAL FRAKTUR SMALL GMATHEMATICAL FRAKTUR SMALL HMATHEMATICAL FRAKTUR SMALL IMATHEMATICAL FRAKTUR SMALL JMATHEMATICAL FRAKTUR SMALL\
    \ KMATHEMATICAL FRAKTUR SMALL LMATHEMATICAL FRAKTUR SMALL MMATHEMATICAL FRAKTUR SMALL NMATHEMATICAL FRAKTUR SMALL OMATHEMATICAL FRAKTUR SMALL PMATHEMATICAL FRAKTUR SMALL QMATHEMATICAL FRAKTUR SMALL RMATHEMATICAL FRAKTUR SMALL SMATHEMATICAL FRAKTUR SMALL TM\
    \ATHEMATICAL FRAKTUR SMALL UMATHEMATICAL FRAKTUR SMALL VMATHEMATICAL FRAKTUR SMALL WMATHEMATICAL FRAKTUR SMALL XMATHEMATICAL FRAKTUR SMALL YMATHEMATICAL FRAKTUR SMALL ZMATHEMATICAL DOUBLE-STRUCK CAPITAL AMATHEMATICAL DOUBLE-STRUCK CAPITAL BMATHEMATICAL DOU\
    \BLE-STRUCK CAPITAL DMATHEMATICAL DOUBLE-STRUCK CAPITAL EMATHEMATICAL DOUBLE-STRUCK CAPITAL FMATHEMATICAL DOUBLE-STRUCK CAPITAL GMATHEMATICAL DOUBLE-STRUCK CAPITAL IMATHEMATICAL DOUBLE-STRUCK CAPITAL JMATHEMATICAL DOUBLE-STRUCK CAPITAL KMATHEMATICAL DOUBLE\
    \-STRUCK CAPITAL LMATHEMATICAL DOUBLE-STRUCK CAPITAL MMATHEMATICAL DOUBLE-STRUCK CAPITAL OMATHEMATICAL DOUBLE-STRUCK CAPITAL SMATHEMATICAL DOUBLE-STRUCK CAPITAL TMATHEMATICAL DOUBLE-STRUCK CAPITAL UMATHEMATICAL DOUBLE-STRUCK CAPITAL VMATHEMATICAL DOUBLE-ST\
    \RUCK CAPITAL WMATHEMATICAL DOUBLE-STRUCK CAPITAL XMATHEMATICAL DOUBLE-STRUCK CAPITAL YMATHEMATICAL DOUBLE-STRUCK SMALL AMATHEMATICAL DOUBLE-STRUCK SMALL BMATHEMATICAL DOUBLE-STRUCK SMALL CMATHEMATICAL DOUBLE-STRUCK SMALL DMATHEMATICAL DOUBLE-STRUCK SMALL \
    \EMATHEMATICAL DOUBLE-STRUCK SMALL FMATHEMATICAL DOUBLE-STRUCK SMALL GMATHEMATICAL DOUBLE-STRUCK SMALL HMATHEMATICAL DOUBLE-STRUCK SMALL IMATHEMATICAL DOUBLE-STRUCK SMALL JMATHEMATICAL DOUBLE-STRUCK SMALL KMATHEMATICAL DOUBLE-STRUCK SMALL LMATHEMATICAL DOU\
    \BLE-STRUCK SMALL MMATHEMATICAL DOUBLE-STRUCK SMALL NMATHEMATICAL DOUBLE-STRUCK SMALL OMATHEMATICAL DOUBLE-STRUCK SMALL PMATHEMATICAL DOUBLE-STRUCK SMALL QMATHEMATICAL DOUBLE-STRUCK SMALL RMATHEMATICAL DOUBLE-STRUCK SMALL SMATHEMATICAL DOUBLE-STRUCK SMALL \
    \TMATHEMATICAL DOUBLE-STRUCK SMALL UMATHEMATICAL DOUBLE-STRUCK SMALL VMATHEMATICAL DOUBLE-STRUCK SMALL WMATHEMATICAL DOUBLE-STRUCK SMALL XMATHEMATICAL DOUBLE-STRUCK SMALL YMATHEMATICAL DOUBLE-STRUCK SMALL ZMATHEMATICAL BOLD FRAKTUR CAPITAL AMATHEMATICAL BO\
    \LD FRAKTUR CAPITAL BMATHEMATICAL BOLD FRAKTUR CAPITAL CMATHEMATICAL BOLD FRAKTUR CAPITAL DMATHEMATICAL BOLD FRAKTUR CAPITAL EMATHEMATICAL BOLD FRAKTUR CAPITAL FMATHEMATICAL BOLD FRAKTUR CAPITAL GMATHEMATICAL BOLD FRAKTUR CAPITAL HMATHEMATICAL BOLD FRAKTUR\
    \ CAPITAL IMATHEMATICAL BOLD FRAKTUR CAPITAL JMATHEMATICAL BOLD FRAKTUR CAPITAL KMATHEMATICAL BOLD FRAKTUR CAPITAL LMATHEMATICAL BOLD FRAKTUR CAPITAL MMATHEMATICAL BOLD FRAKTUR CAPITAL NMATHEMATICAL BOLD FRAKTUR CAPITAL OMATHEMATICAL BOLD FRAKTUR CAPITAL P\
    \MATHEMATICAL BOLD FRAKTUR CAPITAL QMATHEMATICAL BOLD FRAKTUR CAPITAL RMATHEMATICAL BOLD FRAKTUR CAPITAL SMATHEMATICAL BOLD FRAKTUR CAPITAL TMATHEMATICAL BOLD FRAKTUR CAPITAL UMATHEMATICAL BOLD FRAKTUR CAPITAL VMATHEMATICAL BOLD FRAKTUR CAPITAL WMATHEMATIC\
    \AL BOLD FRAKTUR CAPITAL XMATHEMATICAL BOLD FRAKTUR CAPITAL YMATHEMATICAL BOLD FRAKTUR CAPITAL ZMATHEMATICAL BOLD FRAKTUR SMALL AMATHEMATICAL BOLD FRAKTUR SMALL BMATHEMATICAL BOLD FRAKTUR SMALL CMATHEMATICAL BOLD FRAKTUR SMALL DMATHEMATICAL BOLD FRAKTUR SM\
    \ALL EMATHEMATICAL BOLD FRAKTUR SMALL FMATHEMATICAL BOLD FRAKTUR SMALL GMATHEMATICAL BOLD FRAKTUR SMALL HMATHEMATICAL BOLD FRAKTUR SMALL IMATHEMATICAL BOLD FRAKTUR SMALL JMATHEMATICAL BOLD FRAKTUR SMALL KMATHEMATICAL BOLD FRAKTUR SMALL LMATHEMATICAL BOLD F\
    \RAKTUR SMALL MMATHEMATICAL BOLD FRAKTUR SMALL NMATHEMATICAL BOLD FRAKTUR SMALL OMATHEMATICAL BOLD FRAKTUR SMALL PMATHEMATICAL BOLD FRAKTUR SMALL QMATHEMATICAL BOLD FRAKTUR SMALL RMATHEMATICAL BOLD FRAKTUR SMALL SMATHEMATICAL BOLD FRAKTUR SMALL TMATHEMATIC\
    \AL BOLD FRAKTUR SMALL UMATHEMATICAL BOLD FRAKTUR SMALL VMATHEMATICAL BOLD FRAKTUR SMALL WMATHEMATICAL BOLD FRAKTUR SMALL XMATHEMATICAL BOLD FRAKTUR SMALL YMATHEMATICAL BOLD FRAKTUR SMALL ZMATHEMATICAL SANS-SERIF CAPITAL AMATHEMATICAL SANS-SERIF CAPITAL BM\
    \ATHEMATICAL SANS-SERIF CAPITAL CMATHEMATICAL SANS-SERIF CAPITAL DMATHEMATICAL SANS-SERIF CAPITAL EMATHEMATICAL SANS-SERIF CAPITAL FMATHEMATICAL SANS-SERIF CAPITAL GMATHEMATICAL SANS-SERIF CAPITAL HMATHEMATICAL SANS-SERIF CAPITAL IMATHEMATICAL SANS-SERIF C\
    \APITAL JMATHEMATICAL SANS-SERIF CAPITAL KMATHEMATICAL SANS-SERIF CAPITAL LMATHEMATICAL SANS-SERIF CAPITAL MMATHEMATICAL SANS-SERIF CAPITAL NMATHEMATICAL SANS-SERIF CAPITAL OMATHEMATICAL SANS-SERIF CAPITAL PMATHEMATICAL SANS-SERIF CAPITAL QMATHEMATICAL SAN\
    \S-SERIF CAPITAL RMATHEMATICAL SANS-SERIF CAPITAL SMATHEMATICAL SANS-SERIF CAPITAL TMATHEMATICAL SANS-SERIF CAPITAL UMATHEMATICAL SANS-SERIF CAPITAL VMATHEMATICAL SANS-SERIF CAPITAL WMATHEMATICAL SANS-SERIF CAPITAL XMATHEMATICAL SANS-SERIF CAPITAL YMATHEMA\
    \TICAL SANS-SERIF CAPITAL ZMATHEMATICAL SANS-SERIF SMALL AMATHEMATICAL SANS-SERIF SMALL BMATHEMATICAL SANS-SERIF SMALL CMATHEMATICAL SANS-SERIF SMALL DMATHEMATICAL SANS-SERIF SMALL EMATHEMATICAL SANS-SERIF SMALL FMATHEMATICAL SANS-SERIF SMALL GMATHEMATICAL\
    \ SANS-SERIF SMALL HMATHEMATICAL SANS-SERIF SMALL IMATHEMATICAL SANS-SERIF SMALL JMATHEMATICAL SANS-SERIF SMALL KMATHEMATICAL SANS-SERIF SMALL LMATHEMATICAL SANS-SERIF SMALL MMATHEMATICAL SANS-SERIF SMALL NMATHEMATICAL SANS-SERIF SMALL OMATHEMATICAL SANS-S\
    \ERIF SMALL PMATHEMATICAL SANS-SERIF SMALL QMATHEMATICAL SANS-SERIF SMALL RMATHEMATICAL SANS-SERIF SMALL SMATHEMATICAL SANS-SERIF SMALL TMATHEMATICAL SANS-SERIF SMALL UMATHEMATICAL SANS-SERIF SMALL VMATHEMATICAL SANS-SERIF SMALL WMATHEMATICAL SANS-SERIF SM\
    \ALL XMATHEMATICAL SANS-SERIF SMALL YMATHEMATICAL SANS-SERIF SMALL ZMATHEMATICAL SANS-SERIF BOLD CAPITAL AMATHEMATICAL SANS-SERIF BOLD CAPITAL BMATHEMATICAL SANS-SERIF BOLD CAPITAL CMATHEMATICAL SANS-SERIF BOLD CAPITAL DMATHEMATICAL SANS-SERIF BOLD CAPITAL\
    \ EMATHEMATICAL SANS-SERIF BOLD CAPITAL FMATHEMATICAL SANS-SERIF BOLD CAPITAL GMATHEMATICAL SANS-SERIF BOLD CAPITAL HMATHEMATICAL SANS-SERIF BOLD CAPITAL IMATHEMATICAL SANS-SERIF BOLD CAPITAL JMATHEMATICAL SANS-SERIF BOLD CAPITAL KMATHEMATICAL SANS-SERIF B\
    \OLD CAPITAL LMATHEMATICAL SANS-SERIF BOLD CAPITAL MMATHEMATICAL SANS-SERIF BOLD CAPITAL NMATHEMATICAL SANS-SERIF BOLD CAPITAL OMATHEMATICAL SANS-SERIF BOLD CAPITAL PMATHEMATICAL SANS-SERIF BOLD CAPITAL QMATHEMATICAL SANS-SERIF BOLD CAPITAL RMATHEMATICAL S\
    \ANS-SERIF BOLD CAPITAL SMATHEMATICAL SANS-SERIF BOLD CAPITAL TMATHEMATICAL SANS-SERIF BOLD CAPITAL UMATHEMATICAL SANS-SERIF BOLD CAPITAL VMATHEMATICAL SANS-SERIF BOLD CAPITAL WMATHEMATICAL SANS-SERIF BOLD CAPITAL XMATHEMATICAL SANS-SERIF BOLD CAPITAL YMAT\
    \HEMATICAL SANS-SERIF BOLD CAPITAL ZMATHEMATICAL SANS-SERIF BOLD SMALL AMATHEMATICAL SANS-SERIF BOLD SMALL BMATHEMATICAL SANS-SERIF BOLD SMALL CMATHEMATICAL SANS-SERIF BOLD SMALL DMATHEMATICAL SANS-SERIF BOLD SMALL EMATHEMATICAL SANS-SERIF BOLD SMALL FMATH\
    \EMATICAL SANS-SERIF BOLD SMALL GMATHEMATICAL SANS-SERIF BOLD SMALL HMATHEMATICAL SANS-SERIF BOLD SMALL IMATHEMATICAL SANS-SERIF BOLD SMALL JMATHEMATICAL SANS-SERIF BOLD SMALL KMATHEMATICAL SANS-SERIF BOLD SMALL LMATHEMATICAL SANS-SERIF BOLD SMALL MMATHEMA\
    \TICAL SANS-SERIF BOLD SMALL NMATHEMATICAL SANS-SERIF BOLD SMALL OMATHEMATICAL SANS-SERIF BOLD SMALL PMATHEMATICAL SANS-SERIF BOLD SMALL QMATHEMATICAL SANS-SERIF BOLD SMALL RMATHEMATICAL SANS-SERIF BOLD SMALL SMATHEMATICAL SANS-SERIF BOLD SMALL TMATHEMATIC\
    \AL SANS-SERIF BOLD SMALL UMATHEMATICAL SANS-SERIF BOLD SMALL VMATHEMATICAL SANS-SERIF BOLD SMALL WMATHEMATICAL SANS-SERIF BOLD SMALL XMATHEMATICAL SANS-SERIF BOLD SMALL YMATHEMATICAL SANS-SERIF BOLD SMALL ZMATHEMATICAL SANS-SERIF ITALIC CAPITAL AMATHEMATI\
    \CAL SANS-SERIF ITALIC CAPITAL BMATHEMATICAL SANS-SERIF ITALIC CAPITAL CMATHEMATICAL SANS-SERIF ITALIC CAPITAL DMATHEMATICAL SANS-SERIF ITALIC CAPITAL EMATHEMATICAL SANS-SERIF ITALIC CAPITAL FMATHEMATICAL SANS-SERIF ITALIC CAPITAL GMATHEMATICAL SANS-SERIF \
    \ITALIC CAPITAL HMATHEMATICAL SANS-SERIF ITALIC CAPITAL IMATHEMATICAL SANS-SERIF ITALIC CAPITAL JMATHEMATICAL SANS-SERIF ITALIC CAPITAL KMATHEMATICAL SANS-SERIF ITALIC CAPITAL LMATHEMATICAL SANS-SERIF ITALIC CAPITAL MMATHEMATICAL SANS-SERIF ITALIC CAPITAL \
    \NMATHEMATICAL SANS-SERIF ITALIC CAPITAL OMATHEMATICAL SANS-SERIF ITALIC CAPITAL PMATHEMATICAL SANS-SERIF ITALIC CAPITAL QMATHEMATICAL SANS-SERIF ITALIC CAPITAL RMATHEMATICAL SANS-SERIF ITALIC CAPITAL SMATHEMATICAL SANS-SERIF ITALIC CAPITAL TMATHEMATICAL S\
    \ANS-SERIF ITALIC CAPITAL UMATHEMATICAL SANS-SERIF ITALIC CAPITAL VMATHEMATICAL SANS-SERIF ITALIC CAPITAL WMATHEMATICAL SANS-SERIF ITALIC CAPITAL XMATHEMATICAL SANS-SERIF ITALIC CAPITAL YMATHEMATICAL SANS-SERIF ITALIC CAPITAL ZMATHEMATICAL SANS-SERIF ITALI\
    \C SMALL AMATHEMATICAL SANS-SERIF ITALIC SMALL BMATHEMATICAL SANS-SERIF ITALIC SMALL CMATHEMATICAL SANS-SERIF ITALIC SMALL DMATHEMATICAL SANS-SERIF ITALIC SMALL EMATHEMATICAL SANS-SERIF ITALIC SMALL FMATHEMATICAL SANS-SERIF ITALIC SMALL GMATHEMATICAL SANS-\
    \SERIF ITALIC SMALL HMATHEMATICAL SANS-SERIF ITALIC SMALL IMATHEMATICAL SANS-SERIF ITALIC SMALL JMATHEMATICAL SANS-SERIF ITALIC SMALL KMATHEMATICAL SANS-SERIF ITALIC SMALL LMATHEMATICAL SANS-SERIF ITALIC SMALL MMATHEMATICAL SANS-SERIF ITALIC SMALL NMATHEMA\
    \TICAL SANS-SERIF ITALIC SMALL OMATHEMATICAL SANS-SERIF ITALIC SMALL PMATHEMATICAL SANS-SERIF ITALIC SMALL QMATHEMATICAL SANS-SERIF ITALIC SMALL RMATHEMATICAL SANS-SERIF ITALIC SMALL SMATHEMATICAL SANS-SERIF ITALIC SMALL TMATHEMATICAL SANS-SERIF ITALIC SMA\
    \LL UMATHEMATICAL SANS-SERIF ITALIC SMALL VMATHEMATICAL SANS-SERIF ITALIC SMALL WMATHEMATICAL SANS-SERIF ITALIC SMALL XMATHEMATICAL SANS-SERIF ITALIC SMALL YMATHEMATICAL SANS-SERIF ITALIC SMALL ZMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL AMATHEMATICAL SAN\
    \S-SERIF BOLD ITALIC CAPITAL BMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL CMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL DMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL EMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL FMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL GM\
    \ATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL HMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL IMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL JMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL KMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL LMATHEMATICAL SANS-SERIF BOLD IT\
    \ALIC CAPITAL MMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL NMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL OMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL PMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL QMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL RMATHEMATICAL SAN\
    \S-SERIF BOLD ITALIC CAPITAL SMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL TMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL UMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL VMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL WMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL XM\
    \ATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL YMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL ZMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL AMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL BMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL CMATHEMATICAL SANS-SERIF BOLD ITALIC S\
    \MALL DMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL EMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL FMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL GMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL HMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL IMATHEMATICAL SANS-SERIF BOLD ITALI\
    \C SMALL JMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL KMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL LMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL MMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL NMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL OMATHEMATICAL SANS-SERIF BOLD IT\
    \ALIC SMALL PMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL QMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL RMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL SMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL TMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL UMATHEMATICAL SANS-SERIF BOLD\
    \ ITALIC SMALL VMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL WMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL XMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL YMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL ZMATHEMATICAL MONOSPACE CAPITAL AMATHEMATICAL MONOSPACE CAPITAL BMATH\
    \EMATICAL MONOSPACE CAPITAL CMATHEMATICAL MONOSPACE CAPITAL DMATHEMATICAL MONOSPACE CAPITAL EMATHEMATICAL MONOSPACE CAPITAL FMATHEMATICAL MONOSPACE CAPITAL GMATHEMATICAL MONOSPACE CAPITAL HMATHEMATICAL MONOSPACE CAPITAL IMATHEMATICAL MONOSPACE CAPITAL JMAT\
    \HEMATICAL MONOSPACE CAPITAL KMATHEMATICAL MONOSPACE CAPITAL LMATHEMATICAL MONOSPACE CAPITAL MMATHEMATICAL MONOSPACE CAPITAL NMATHEMATICAL MONOSPACE CAPITAL OMATHEMATICAL MONOSPACE CAPITAL PMATHEMATICAL MONOSPACE CAPITAL QMATHEMATICAL MONOSPACE CAPITAL RMA\
    \THEMATICAL MONOSPACE CAPITAL SMATHEMATICAL MONOSPACE CAPITAL TMATHEMATICAL MONOSPACE CAPITAL UMATHEMATICAL MONOSPACE CAPITAL VMATHEMATICAL MONOSPACE CAPITAL WMATHEMATICAL MONOSPACE CAPITAL XMATHEMATICAL MONOSPACE CAPITAL YMATHEMATICAL MONOSPACE CAPITAL ZM\
    \ATHEMATICAL MONOSPACE SMALL AMATHEMATICAL MONOSPACE SMALL BMATHEMATICAL MONOSPACE SMALL CMATHEMATICAL MONOSPACE SMALL DMATHEMATICAL MONOSPACE SMALL EMATHEMATICAL MONOSPACE SMALL FMATHEMATICAL MONOSPACE SMALL GMATHEMATICAL MONOSPACE SMALL HMATHEMATICAL MON\
    \OSPACE SMALL IMATHEMATICAL MONOSPACE SMALL JMATHEMATICAL MONOSPACE SMALL KMATHEMATICAL MONOSPACE SMALL LMATHEMATICAL MONOSPACE SMALL MMATHEMATICAL MONOSPACE SMALL NMATHEMATICAL MONOSPACE SMALL OMATHEMATICAL MONOSPACE SMALL PMATHEMATICAL MONOSPACE SMALL QM\
    \ATHEMATICAL MONOSPACE SMALL RMATHEMATICAL MONOSPACE SMALL SMATHEMATICAL MONOSPACE SMALL TMATHEMATICAL MONOSPACE SMALL UMATHEMATICAL MONOSPACE SMALL VMATHEMATICAL MONOSPACE SMALL WMATHEMATICAL MONOSPACE SMALL XMATHEMATICAL MONOSPACE SMALL YMATHEMATICAL MON\
    \OSPACE SMALL ZMATHEMATICAL ITALIC SMALL DOTLESS IMATHEMATICAL ITALIC SMALL DOTLESS JMATHEMATICAL BOLD CAPITAL ALPHAMATHEMATICAL BOLD CAPITAL BETAMATHEMATICAL BOLD CAPITAL GAMMAMATHEMATICAL BOLD CAPITAL DELTAMATHEMATICAL BOLD CAPITAL EPSILONMATHEMATICAL BO\
    \LD CAPITAL ZETAMATHEMATICAL BOLD CAPITAL ETAMATHEMATICAL BOLD CAPITAL THETAMATHEMATICAL BOLD CAPITAL IOTAMATHEMATICAL BOLD CAPITAL KAPPAMATHEMATICAL BOLD CAPITAL LAMDAMATHEMATICAL BOLD CAPITAL MUMATHEMATICAL BOLD CAPITAL NUMATHEMATICAL BOLD CAPITAL XIMATH\
    \EMATICAL BOLD CAPITAL OMICRONMATHEMATICAL BOLD CAPITAL PIMATHEMATICAL BOLD CAPITAL RHOMATHEMATICAL BOLD CAPITAL THETA SYMBOLMATHEMATICAL BOLD CAPITAL SIGMAMATHEMATICAL BOLD CAPITAL TAUMATHEMATICAL BOLD CAPITAL UPSILONMATHEMATICAL BOLD CAPITAL PHIMATHEMATI\
    \CAL BOLD CAPITAL CHIMATHEMATICAL BOLD CAPITAL PSIMATHEMATICAL BOLD CAPITAL OMEGAMATHEMATICAL BOLD NABLAMATHEMATICAL BOLD SMALL ALPHAMATHEMATICAL BOLD SMALL BETAMATHEMATICAL BOLD SMALL GAMMAMATHEMATICAL BOLD SMALL DELTAMATHEMATICAL BOLD SMALL EPSILONMATHEM\
    \ATICAL BOLD SMALL ZETAMATHEMATICAL BOLD SMALL ETAMATHEMATICAL BOLD SMALL THETAMATHEMATICAL BOLD SMALL IOTAMATHEMATICAL BOLD SMALL KAPPAMATHEMATICAL BOLD SMALL LAMDAMATHEMATICAL BOLD SMALL MUMATHEMATICAL BOLD SMALL NUMATHEMATICAL BOLD SMALL XIMATHEMATICAL \
    \BOLD SMALL OMICRONMATHEMATICAL BOLD SMALL PIMATHEMATICAL BOLD SMALL RHOMATHEMATICAL BOLD SMALL FINAL SIGMAMATHEMATICAL BOLD SMALL SIGMAMATHEMATICAL BOLD SMALL TAUMATHEMATICAL BOLD SMALL UPSILONMATHEMATICAL BOLD SMALL PHIMATHEMATICAL BOLD SMALL CHIMATHEMAT\
    \ICAL BOLD SMALL PSIMATHEMATICAL BOLD SMALL OMEGAMATHEMATICAL BOLD PARTIAL DIFFERENTIALMATHEMATICAL BOLD EPSILON SYMBOLMATHEMATICAL BOLD THETA SYMBOLMATHEMATICAL BOLD KAPPA SYMBOLMATHEMATICAL BOLD PHI SYMBOLMATHEMATICAL BOLD RHO SYMBOLMATHEMATICAL BOLD PI \
    \SYMBOLMATHEMATICAL ITALIC CAPITAL ALPHAMATHEMATICAL ITALIC CAPITAL BETAMATHEMATICAL ITALIC CAPITAL GAMMAMATHEMATICAL ITALIC CAPITAL DELTAMATHEMATICAL ITALIC CAPITAL EPSILONMATHEMATICAL ITALIC CAPITAL ZETAMATHEMATICAL ITALIC CAPITAL ETAMATHEMATICAL ITALIC \
    \CAPITAL THETAMATHEMATICAL ITALIC CAPITAL IOTAMATHEMATICAL ITALIC CAPITAL KAPPAMATHEMATICAL ITALIC CAPITAL LAMDAMATHEMATICAL ITALIC CAPITAL MUMATHEMATICAL ITALIC CAPITAL NUMATHEMATICAL ITALIC CAPITAL XIMATHEMATICAL ITALIC CAPITAL OMICRONMATHEMATICAL ITALIC\
    \ CAPITAL PIMATHEMATICAL ITALIC CAPITAL RHOMATHEMATICAL ITALIC CAPITAL THETA SYMBOLMATHEMATICAL ITALIC CAPITAL SIGMAMATHEMATICAL ITALIC CAPITAL TAUMATHEMATICAL ITALIC CAPITAL UPSILONMATHEMATICAL ITALIC CAPITAL PHIMATHEMATICAL ITALIC CAPITAL CHIMATHEMATICAL\
    \ ITALIC CAPITAL PSIMATHEMATICAL ITALIC CAPITAL OMEGAMATHEMATICAL ITALIC NABLAMATHEMATICAL ITALIC SMALL ALPHAMATHEMATICAL ITALIC SMALL BETAMATHEMATICAL ITALIC SMALL GAMMAMATHEMATICAL ITALIC SMALL DELTAMATHEMATICAL ITALIC SMALL EPSILONMATHEMATICAL ITALIC SM\
    \ALL ZETAMATHEMATICAL ITALIC SMALL ETAMATHEMATICAL ITALIC SMALL THETAMATHEMATICAL ITALIC SMALL IOTAMATHEMATICAL ITALIC SMALL KAPPAMATHEMATICAL ITALIC SMALL LAMDAMATHEMATICAL ITALIC SMALL MUMATHEMATICAL ITALIC SMALL NUMATHEMATICAL ITALIC SMALL XIMATHEMATICA\
    \L ITALIC SMALL OMICRONMATHEMATICAL ITALIC SMALL PIMATHEMATICAL ITALIC SMALL RHOMATHEMATICAL ITALIC SMALL FINAL SIGMAMATHEMATICAL ITALIC SMALL SIGMAMATHEMATICAL ITALIC SMALL TAUMATHEMATICAL ITALIC SMALL UPSILONMATHEMATICAL ITALIC SMALL PHIMATHEMATICAL ITAL\
    \IC SMALL CHIMATHEMATICAL ITALIC SMALL PSIMATHEMATICAL ITALIC SMALL OMEGAMATHEMATICAL ITALIC PARTIAL DIFFERENTIALMATHEMATICAL ITALIC EPSILON SYMBOLMATHEMATICAL ITALIC THETA SYMBOLMATHEMATICAL ITALIC KAPPA SYMBOLMATHEMATICAL ITALIC PHI SYMBOLMATHEMATICAL IT\
    \ALIC RHO SYMBOLMATHEMATICAL ITALIC PI SYMBOLMATHEMATICAL BOLD ITALIC CAPITAL ALPHAMATHEMATICAL BOLD ITALIC CAPITAL BETAMATHEMATICAL BOLD ITALIC CAPITAL GAMMAMATHEMATICAL BOLD ITALIC CAPITAL DELTAMATHEMATICAL BOLD ITALIC CAPITAL EPSILONMATHEMATICAL BOLD IT\
    \ALIC CAPITAL ZETAMATHEMATICAL BOLD ITALIC CAPITAL ETAMATHEMATICAL BOLD ITALIC CAPITAL THETAMATHEMATICAL BOLD ITALIC CAPITAL IOTAMATHEMATICAL BOLD ITALIC CAPITAL KAPPAMATHEMATICAL BOLD ITALIC CAPITAL LAMDAMATHEMATICAL BOLD ITALIC CAPITAL MUMATHEMATICAL BOL\
    \D ITALIC CAPITAL NUMATHEMATICAL BOLD ITALIC CAPITAL XIMATHEMATICAL BOLD ITALIC CAPITAL OMICRONMATHEMATICAL BOLD ITALIC CAPITAL PIMATHEMATICAL BOLD ITALIC CAPITAL RHOMATHEMATICAL BOLD ITALIC CAPITAL THETA SYMBOLMATHEMATICAL BOLD ITALIC CAPITAL SIGMAMATHEMA\
    \TICAL BOLD ITALIC CAPITAL TAUMATHEMATICAL BOLD ITALIC CAPITAL UPSILONMATHEMATICAL BOLD ITALIC CAPITAL PHIMATHEMATICAL BOLD ITALIC CAPITAL CHIMATHEMATICAL BOLD ITALIC CAPITAL PSIMATHEMATICAL BOLD ITALIC CAPITAL OMEGAMATHEMATICAL BOLD ITALIC NABLAMATHEMATIC\
    \AL BOLD ITALIC SMALL ALPHAMATHEMATICAL BOLD ITALIC SMALL BETAMATHEMATICAL BOLD ITALIC SMALL GAMMAMATHEMATICAL BOLD ITALIC SMALL DELTAMATHEMATICAL BOLD ITALIC SMALL EPSILONMATHEMATICAL BOLD ITALIC SMALL ZETAMATHEMATICAL BOLD ITALIC SMALL ETAMATHEMATICAL BO\
    \LD ITALIC SMALL THETAMATHEMATICAL BOLD ITALIC SMALL IOTAMATHEMATICAL BOLD ITALIC SMALL KAPPAMATHEMATICAL BOLD ITALIC SMALL LAMDAMATHEMATICAL BOLD ITALIC SMALL MUMATHEMATICAL BOLD ITALIC SMALL NUMATHEMATICAL BOLD ITALIC SMALL XIMATHEMATICAL BOLD ITALIC SMA\
    \LL OMICRONMATHEMATICAL BOLD ITALIC SMALL PIMATHEMATICAL BOLD ITALIC SMALL RHOMATHEMATICAL BOLD ITALIC SMALL FINAL SIGMAMATHEMATICAL BOLD ITALIC SMALL SIGMAMATHEMATICAL BOLD ITALIC SMALL TAUMATHEMATICAL BOLD ITALIC SMALL UPSILONMATHEMATICAL BOLD ITALIC SMA\
    \LL PHIMATHEMATICAL BOLD ITALIC SMALL CHIMATHEMATICAL BOLD ITALIC SMALL PSIMATHEMATICAL BOLD ITALIC SMALL OMEGAMATHEMATICAL BOLD ITALIC PARTIAL DIFFERENTIALMATHEMATICAL BOLD ITALIC EPSILON SYMBOLMATHEMATICAL BOLD ITALIC THETA SYMBOLMATHEMATICAL BOLD ITALIC\
    \ KAPPA SYMBOLMATHEMATICAL BOLD ITALIC PHI SYMBOLMATHEMATICAL BOLD ITALIC RHO SYMBOLMATHEMATICAL BOLD ITALIC PI SYMBOLMATHEMATICAL SANS-SERIF BOLD CAPITAL ALPHAMATHEMATICAL SANS-SERIF BOLD CAPITAL BETAMATHEMATICAL SANS-SERIF BOLD CAPITAL GAMMAMATHEMATICAL \
    \SANS-SERIF BOLD CAPITAL DELTAMATHEMATICAL SANS-SERIF BOLD CAPITAL EPSILONMATHEMATICAL SANS-SERIF BOLD CAPITAL ZETAMATHEMATICAL SANS-SERIF BOLD CAPITAL ETAMATHEMATICAL SANS-SERIF BOLD CAPITAL THETAMATHEMATICAL SANS-SERIF BOLD CAPITAL IOTAMATHEMATICAL SANS-\
    \SERIF BOLD CAPITAL KAPPAMATHEMATICAL SANS-SERIF BOLD CAPITAL LAMDAMATHEMATICAL SANS-SERIF BOLD CAPITAL MUMATHEMATICAL SANS-SERIF BOLD CAPITAL NUMATHEMATICAL SANS-SERIF BOLD CAPITAL XIMATHEMATICAL SANS-SERIF BOLD CAPITAL OMICRONMATHEMATICAL SANS-SERIF BOLD\
    \ CAPITAL PIMATHEMATICAL SANS-SERIF BOLD CAPITAL RHOMATHEMATICAL SANS-SERIF BOLD CAPITAL THETA SYMBOLMATHEMATICAL SANS-SERIF BOLD CAPITAL SIGMAMATHEMATICAL SANS-SERIF BOLD CAPITAL TAUMATHEMATICAL SANS-SERIF BOLD CAPITAL UPSILONMATHEMATICAL SANS-SERIF BOLD \
    \CAPITAL PHIMATHEMATICAL SANS-SERIF BOLD CAPITAL CHIMATHEMATICAL SANS-SERIF BOLD CAPITAL PSIMATHEMATICAL SANS-SERIF BOLD CAPITAL OMEGAMATHEMATICAL SANS-SERIF BOLD NABLAMATHEMATICAL SANS-SERIF BOLD SMALL ALPHAMATHEMATICAL SANS-SERIF BOLD SMALL BETAMATHEMATI\
    \CAL SANS-SERIF BOLD SMALL GAMMAMATHEMATICAL SANS-SERIF BOLD SMALL DELTAMATHEMATICAL SANS-SERIF BOLD SMALL EPSILONMATHEMATICAL SANS-SERIF BOLD SMALL ZETAMATHEMATICAL SANS-SERIF BOLD SMALL ETAMATHEMATICAL SANS-SERIF BOLD SMALL THETAMATHEMATICAL SANS-SERIF B\
    \OLD SMALL IOTAMATHEMATICAL SANS-SERIF BOLD SMALL KAPPAMATHEMATICAL SANS-SERIF BOLD SMALL LAMDAMATHEMATICAL SANS-SERIF BOLD SMALL MUMATHEMATICAL SANS-SERIF BOLD SMALL NUMATHEMATICAL SANS-SERIF BOLD SMALL XIMATHEMATICAL SANS-SERIF BOLD SMALL OMICRONMATHEMAT\
    \ICAL SANS-SERIF BOLD SMALL PIMATHEMATICAL SANS-SERIF BOLD SMALL RHOMATHEMATICAL SANS-SERIF BOLD SMALL FINAL SIGMAMATHEMATICAL SANS-SERIF BOLD SMALL SIGMAMATHEMATICAL SANS-SERIF BOLD SMALL TAUMATHEMATICAL SANS-SERIF BOLD SMALL UPSILONMATHEMATICAL SANS-SERI\
    \F BOLD SMALL PHIMATHEMATICAL SANS-SERIF BOLD SMALL CHIMATHEMATICAL SANS-SERIF BOLD SMALL PSIMATHEMATICAL SANS-SERIF BOLD SMALL OMEGAMATHEMATICAL SANS-SERIF BOLD PARTIAL DIFFERENTIALMATHEMATICAL SANS-SERIF BOLD EPSILON SYMBOLMATHEMATICAL SANS-SERIF BOLD TH\
    \ETA SYMBOLMATHEMATICAL SANS-SERIF BOLD KAPPA SYMBOLMATHEMATICAL SANS-SERIF BOLD PHI SYMBOLMATHEMATICAL SANS-SERIF BOLD RHO SYMBOLMATHEMATICAL SANS-SERIF BOLD PI SYMBOLMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL ALPHAMATHEMATICAL SANS-SERIF BOLD ITALIC CAP\
    \ITAL BETAMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL GAMMAMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL DELTAMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL EPSILONMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL ZETAMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL ETAMA\
    \THEMATICAL SANS-SERIF BOLD ITALIC CAPITAL THETAMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL IOTAMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL KAPPAMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL LAMDAMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL MUMATHEMATICAL SAN\
    \S-SERIF BOLD ITALIC CAPITAL NUMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL XIMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL OMICRONMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL PIMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL RHOMATHEMATICAL SANS-SERIF BOLD ITALIC\
    \ CAPITAL THETA SYMBOLMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL SIGMAMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL TAUMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL UPSILONMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL PHIMATHEMATICAL SANS-SERIF BOLD ITALIC CAPI\
    \TAL CHIMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL PSIMATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL OMEGAMATHEMATICAL SANS-SERIF BOLD ITALIC NABLAMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL ALPHAMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL BETAMATHEMATICAL SANS-\
    \SERIF BOLD ITALIC SMALL GAMMAMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL DELTAMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL EPSILONMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL ZETAMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL ETAMATHEMATICAL SANS-SERIF BOLD ITALIC SMA\
    \LL THETAMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL IOTAMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL KAPPAMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL LAMDAMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL MUMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL NUMATHEMATICAL SANS-S\
    \ERIF BOLD ITALIC SMALL XIMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL OMICRONMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL PIMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL RHOMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL FINAL SIGMAMATHEMATICAL SANS-SERIF BOLD ITALIC SMA\
    \LL SIGMAMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL TAUMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL UPSILONMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL PHIMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL CHIMATHEMATICAL SANS-SERIF BOLD ITALIC SMALL PSIMATHEMATICAL SANS-\
    \SERIF BOLD ITALIC SMALL OMEGAMATHEMATICAL SANS-SERIF BOLD ITALIC PARTIAL DIFFERENTIALMATHEMATICAL SANS-SERIF BOLD ITALIC EPSILON SYMBOLMATHEMATICAL SANS-SERIF BOLD ITALIC THETA SYMBOLMATHEMATICAL SANS-SERIF BOLD ITALIC KAPPA SYMBOLMATHEMATICAL SANS-SERIF \
    \BOLD ITALIC PHI SYMBOLMATHEMATICAL SANS-SERIF BOLD ITALIC RHO SYMBOLMATHEMATICAL SANS-SERIF BOLD ITALIC PI SYMBOLMATHEMATICAL BOLD CAPITAL DIGAMMAMATHEMATICAL BOLD SMALL DIGAMMAMATHEMATICAL BOLD DIGIT ZEROMATHEMATICAL BOLD DIGIT ONEMATHEMATICAL BOLD DIGIT\
    \ TWOMATHEMATICAL BOLD DIGIT THREEMATHEMATICAL BOLD DIGIT FOURMATHEMATICAL BOLD DIGIT FIVEMATHEMATICAL BOLD DIGIT SIXMATHEMATICAL BOLD DIGIT SEVENMATHEMATICAL BOLD DIGIT EIGHTMATHEMATICAL BOLD DIGIT NINEMATHEMATICAL DOUBLE-STRUCK DIGIT ZEROMATHEMATICAL DOU\
    \BLE-STRUCK DIGIT ONEMATHEMATICAL DOUBLE-STRUCK DIGIT TWOMATHEMATICAL DOUBLE-STRUCK DIGIT THREEMATHEMATICAL DOUBLE-STRUCK DIGIT FOURMATHEMATICAL DOUBLE-STRUCK DIGIT FIVEMATHEMATICAL DOUBLE-STRUCK DIGIT SIXMATHEMATICAL DOUBLE-STRUCK DIGIT SEVENMATHEMATICAL \
    \DOUBLE-STRUCK DIGIT EIGHTMATHEMATICAL DOUBLE-STRUCK DIGIT NINEMATHEMATICAL SANS-SERIF DIGIT ZEROMATHEMATICAL SANS-SERIF DIGIT ONEMATHEMATICAL SANS-SERIF DIGIT TWOMATHEMATICAL SANS-SERIF DIGIT THREEMATHEMATICAL SANS-SERIF DIGIT FOURMATHEMATICAL SANS-SERIF \
    \DIGIT FIVEMATHEMATICAL SANS-SERIF DIGIT SIXMATHEMATICAL SANS-SERIF DIGIT SEVENMATHEMATICAL SANS-SERIF DIGIT EIGHTMATHEMATICAL SANS-SERIF DIGIT NINEMATHEMATICAL SANS-SERIF BOLD DIGIT ZEROMATHEMATICAL SANS-SERIF BOLD DIGIT ONEMATHEMATICAL SANS-SERIF BOLD DI\
    \GIT TWOMATHEMATICAL SANS-SERIF BOLD DIGIT THREEMATHEMATICAL SANS-SERIF BOLD DIGIT FOURMATHEMATICAL SANS-SERIF BOLD DIGIT FIVEMATHEMATICAL SANS-SERIF BOLD DIGIT SIXMATHEMATICAL SANS-SERIF BOLD DIGIT SEVENMATHEMATICAL SANS-SERIF BOLD DIGIT EIGHTMATHEMATICAL\
    \ SANS-SERIF BOLD DIGIT NINEMATHEMATICAL MONOSPACE DIGIT ZEROMATHEMATICAL MONOSPACE DIGIT ONEMATHEMATICAL MONOSPACE DIGIT TWOMATHEMATICAL MONOSPACE DIGIT THREEMATHEMATICAL MONOSPACE DIGIT FOURMATHEMATICAL MONOSPACE DIGIT FIVEMATHEMATICAL MONOSPACE DIGIT SI\
    \XMATHEMATICAL MONOSPACE DIGIT SEVENMATHEMATICAL MONOSPACE DIGIT EIGHTMATHEMATICAL MONOSPACE DIGIT NINESIGNWRITING HAND-FIST INDEXSIGNWRITING HAND-CIRCLE INDEXSIGNWRITING HAND-CUP INDEXSIGNWRITING HAND-OVAL INDEXSIGNWRITING HAND-HINGE INDEXSIGNWRITING HAND\
    \-ANGLE INDEXSIGNWRITING HAND-FIST INDEX BENTSIGNWRITING HAND-CIRCLE INDEX BENTSIGNWRITING HAND-FIST THUMB UNDER INDEX BENTSIGNWRITING HAND-FIST INDEX RAISED KNUCKLESIGNWRITING HAND-FIST INDEX CUPPEDSIGNWRITING HAND-FIST INDEX HINGEDSIGNWRITING HAND-FIST I\
    \NDEX HINGED LOWSIGNWRITING HAND-CIRCLE INDEX HINGESIGNWRITING HAND-FIST INDEX MIDDLESIGNWRITING HAND-CIRCLE INDEX MIDDLESIGNWRITING HAND-FIST INDEX MIDDLE BENTSIGNWRITING HAND-FIST INDEX MIDDLE RAISED KNUCKLESSIGNWRITING HAND-FIST INDEX MIDDLE HINGEDSIGNW\
    \RITING HAND-FIST INDEX UP MIDDLE HINGEDSIGNWRITING HAND-FIST INDEX HINGED MIDDLE UPSIGNWRITING HAND-FIST INDEX MIDDLE CONJOINEDSIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED INDEX BENTSIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED MIDDLE BENTSIGNWRITING HAND-\
    \FIST INDEX MIDDLE CONJOINED CUPPEDSIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED HINGEDSIGNWRITING HAND-FIST INDEX MIDDLE CROSSEDSIGNWRITING HAND-CIRCLE INDEX MIDDLE CROSSEDSIGNWRITING HAND-FIST MIDDLE BENT OVER INDEXSIGNWRITING HAND-FIST INDEX BENT OVER MI\
    \DDLESIGNWRITING HAND-FIST INDEX MIDDLE THUMBSIGNWRITING HAND-CIRCLE INDEX MIDDLE THUMBSIGNWRITING HAND-FIST INDEX MIDDLE STRAIGHT THUMB BENTSIGNWRITING HAND-FIST INDEX MIDDLE BENT THUMB STRAIGHTSIGNWRITING HAND-FIST INDEX MIDDLE THUMB BENTSIGNWRITING HAND\
    \-FIST INDEX MIDDLE HINGED SPREAD THUMB SIDESIGNWRITING HAND-FIST INDEX UP MIDDLE HINGED THUMB SIDESIGNWRITING HAND-FIST INDEX UP MIDDLE HINGED THUMB CONJOINEDSIGNWRITING HAND-FIST INDEX HINGED MIDDLE UP THUMB SIDESIGNWRITING HAND-FIST INDEX MIDDLE UP SPRE\
    \AD THUMB FORWARDSIGNWRITING HAND-FIST INDEX MIDDLE THUMB CUPPEDSIGNWRITING HAND-FIST INDEX MIDDLE THUMB CIRCLEDSIGNWRITING HAND-FIST INDEX MIDDLE THUMB HOOKEDSIGNWRITING HAND-FIST INDEX MIDDLE THUMB HINGEDSIGNWRITING HAND-FIST THUMB BETWEEN INDEX MIDDLE S\
    \TRAIGHTSIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED THUMB SIDESIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED THUMB SIDE CONJOINEDSIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED THUMB SIDE BENTSIGNWRITING HAND-FIST MIDDLE THUMB HOOKED INDEX UPSIGNWRITING HAND-F\
    \IST INDEX THUMB HOOKED MIDDLE UPSIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED HINGED THUMB SIDESIGNWRITING HAND-FIST INDEX MIDDLE CROSSED THUMB SIDESIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED THUMB FORWARDSIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED CUPPE\
    \D THUMB FORWARDSIGNWRITING HAND-FIST MIDDLE THUMB CUPPED INDEX UPSIGNWRITING HAND-FIST INDEX THUMB CUPPED MIDDLE UPSIGNWRITING HAND-FIST MIDDLE THUMB CIRCLED INDEX UPSIGNWRITING HAND-FIST MIDDLE THUMB CIRCLED INDEX HINGEDSIGNWRITING HAND-FIST INDEX THUMB \
    \ANGLED OUT MIDDLE UPSIGNWRITING HAND-FIST INDEX THUMB ANGLED IN MIDDLE UPSIGNWRITING HAND-FIST INDEX THUMB CIRCLED MIDDLE UPSIGNWRITING HAND-FIST INDEX MIDDLE THUMB CONJOINED HINGEDSIGNWRITING HAND-FIST INDEX MIDDLE THUMB ANGLED OUTSIGNWRITING HAND-FIST I\
    \NDEX MIDDLE THUMB ANGLEDSIGNWRITING HAND-FIST MIDDLE THUMB ANGLED OUT INDEX UPSIGNWRITING HAND-FIST MIDDLE THUMB ANGLED OUT INDEX CROSSEDSIGNWRITING HAND-FIST MIDDLE THUMB ANGLED INDEX UPSIGNWRITING HAND-FIST INDEX THUMB HOOKED MIDDLE HINGEDSIGNWRITING HA\
    \ND-FLAT FOUR FINGERSSIGNWRITING HAND-FLAT FOUR FINGERS BENTSIGNWRITING HAND-FLAT FOUR FINGERS HINGEDSIGNWRITING HAND-FLAT FOUR FINGERS CONJOINEDSIGNWRITING HAND-FLAT FOUR FINGERS CONJOINED SPLITSIGNWRITING HAND-CLAW FOUR FINGERS CONJOINEDSIGNWRITING HAND-\
    \FIST FOUR FINGERS CONJOINED BENTSIGNWRITING HAND-HINGE FOUR FINGERS CONJOINEDSIGNWRITING HAND-FLAT FIVE FINGERS SPREADSIGNWRITING HAND-FLAT HEEL FIVE FINGERS SPREADSIGNWRITING HAND-FLAT FIVE FINGERS SPREAD FOUR BENTSIGNWRITING HAND-FLAT HEEL FIVE FINGERS \
    \SPREAD FOUR BENTSIGNWRITING HAND-FLAT FIVE FINGERS SPREAD BENTSIGNWRITING HAND-FLAT HEEL FIVE FINGERS SPREAD BENTSIGNWRITING HAND-FLAT FIVE FINGERS SPREAD THUMB FORWARDSIGNWRITING HAND-CUP FIVE FINGERS SPREADSIGNWRITING HAND-CUP FIVE FINGERS SPREAD OPENSI\
    \GNWRITING HAND-HINGE FIVE FINGERS SPREAD OPENSIGNWRITING HAND-OVAL FIVE FINGERS SPREADSIGNWRITING HAND-FLAT FIVE FINGERS SPREAD HINGEDSIGNWRITING HAND-FLAT FIVE FINGERS SPREAD HINGED THUMB SIDESIGNWRITING HAND-FLAT FIVE FINGERS SPREAD HINGED NO THUMBSIGNW\
    \RITING HAND-FLATSIGNWRITING HAND-FLAT BETWEEN PALM FACINGSSIGNWRITING HAND-FLAT HEELSIGNWRITING HAND-FLAT THUMB SIDESIGNWRITING HAND-FLAT HEEL THUMB SIDESIGNWRITING HAND-FLAT THUMB BENTSIGNWRITING HAND-FLAT THUMB FORWARDSIGNWRITING HAND-FLAT SPLIT INDEX T\
    \HUMB SIDESIGNWRITING HAND-FLAT SPLIT CENTRESIGNWRITING HAND-FLAT SPLIT CENTRE THUMB SIDESIGNWRITING HAND-FLAT SPLIT CENTRE THUMB SIDE BENTSIGNWRITING HAND-FLAT SPLIT LITTLESIGNWRITING HAND-CLAWSIGNWRITING HAND-CLAW THUMB SIDESIGNWRITING HAND-CLAW NO THUMB\
    \SIGNWRITING HAND-CLAW THUMB FORWARDSIGNWRITING HAND-HOOK CURLICUESIGNWRITING HAND-HOOKSIGNWRITING HAND-CUP OPENSIGNWRITING HAND-CUPSIGNWRITING HAND-CUP OPEN THUMB SIDESIGNWRITING HAND-CUP THUMB SIDESIGNWRITING HAND-CUP OPEN NO THUMBSIGNWRITING HAND-CUP NO\
    \ THUMBSIGNWRITING HAND-CUP OPEN THUMB FORWARDSIGNWRITING HAND-CUP THUMB FORWARDSIGNWRITING HAND-CURLICUE OPENSIGNWRITING HAND-CURLICUESIGNWRITING HAND-CIRCLESIGNWRITING HAND-OVALSIGNWRITING HAND-OVAL THUMB SIDESIGNWRITING HAND-OVAL NO THUMBSIGNWRITING HAN\
    \D-OVAL THUMB FORWARDSIGNWRITING HAND-HINGE OPENSIGNWRITING HAND-HINGE OPEN THUMB FORWARDSIGNWRITING HAND-HINGESIGNWRITING HAND-HINGE SMALLSIGNWRITING HAND-HINGE OPEN THUMB SIDESIGNWRITING HAND-HINGE THUMB SIDESIGNWRITING HAND-HINGE OPEN NO THUMBSIGNWRITIN\
    \G HAND-HINGE NO THUMBSIGNWRITING HAND-HINGE THUMB SIDE TOUCHING INDEXSIGNWRITING HAND-HINGE THUMB BETWEEN MIDDLE RINGSIGNWRITING HAND-ANGLESIGNWRITING HAND-FIST INDEX MIDDLE RINGSIGNWRITING HAND-CIRCLE INDEX MIDDLE RINGSIGNWRITING HAND-HINGE INDEX MIDDLE \
    \RINGSIGNWRITING HAND-ANGLE INDEX MIDDLE RINGSIGNWRITING HAND-HINGE LITTLESIGNWRITING HAND-FIST INDEX MIDDLE RING BENTSIGNWRITING HAND-FIST INDEX MIDDLE RING CONJOINEDSIGNWRITING HAND-HINGE INDEX MIDDLE RING CONJOINEDSIGNWRITING HAND-FIST LITTLE DOWNSIGNWR\
    \ITING HAND-FIST LITTLE DOWN RIPPLE STRAIGHTSIGNWRITING HAND-FIST LITTLE DOWN RIPPLE CURVEDSIGNWRITING HAND-FIST LITTLE DOWN OTHERS CIRCLEDSIGNWRITING HAND-FIST LITTLE UPSIGNWRITING HAND-FIST THUMB UNDER LITTLE UPSIGNWRITING HAND-CIRCLE LITTLE UPSIGNWRITIN\
    \G HAND-OVAL LITTLE UPSIGNWRITING HAND-ANGLE LITTLE UPSIGNWRITING HAND-FIST LITTLE RAISED KNUCKLESIGNWRITING HAND-FIST LITTLE BENTSIGNWRITING HAND-FIST LITTLE TOUCHES THUMBSIGNWRITING HAND-FIST LITTLE THUMBSIGNWRITING HAND-HINGE LITTLE THUMBSIGNWRITING HAN\
    \D-FIST LITTLE INDEX THUMBSIGNWRITING HAND-HINGE LITTLE INDEX THUMBSIGNWRITING HAND-ANGLE LITTLE INDEX THUMB INDEX THUMB OUTSIGNWRITING HAND-ANGLE LITTLE INDEX THUMB INDEX THUMBSIGNWRITING HAND-FIST LITTLE INDEXSIGNWRITING HAND-CIRCLE LITTLE INDEXSIGNWRITI\
    \NG HAND-HINGE LITTLE INDEXSIGNWRITING HAND-ANGLE LITTLE INDEXSIGNWRITING HAND-FIST INDEX MIDDLE LITTLESIGNWRITING HAND-CIRCLE INDEX MIDDLE LITTLESIGNWRITING HAND-HINGE INDEX MIDDLE LITTLESIGNWRITING HAND-HINGE RINGSIGNWRITING HAND-ANGLE INDEX MIDDLE LITTL\
    \ESIGNWRITING HAND-FIST INDEX MIDDLE CROSS LITTLESIGNWRITING HAND-CIRCLE INDEX MIDDLE CROSS LITTLESIGNWRITING HAND-FIST RING DOWNSIGNWRITING HAND-HINGE RING DOWN INDEX THUMB HOOK MIDDLESIGNWRITING HAND-ANGLE RING DOWN MIDDLE THUMB INDEX CROSSSIGNWRITING HA\
    \ND-FIST RING UPSIGNWRITING HAND-FIST RING RAISED KNUCKLESIGNWRITING HAND-FIST RING LITTLESIGNWRITING HAND-CIRCLE RING LITTLESIGNWRITING HAND-OVAL RING LITTLESIGNWRITING HAND-ANGLE RING LITTLESIGNWRITING HAND-FIST RING MIDDLESIGNWRITING HAND-FIST RING MIDD\
    \LE CONJOINEDSIGNWRITING HAND-FIST RING MIDDLE RAISED KNUCKLESSIGNWRITING HAND-FIST RING INDEXSIGNWRITING HAND-FIST RING THUMBSIGNWRITING HAND-HOOK RING THUMBSIGNWRITING HAND-FIST INDEX RING LITTLESIGNWRITING HAND-CIRCLE INDEX RING LITTLESIGNWRITING HAND-C\
    \URLICUE INDEX RING LITTLE ONSIGNWRITING HAND-HOOK INDEX RING LITTLE OUTSIGNWRITING HAND-HOOK INDEX RING LITTLE INSIGNWRITING HAND-HOOK INDEX RING LITTLE UNDERSIGNWRITING HAND-CUP INDEX RING LITTLESIGNWRITING HAND-HINGE INDEX RING LITTLESIGNWRITING HAND-AN\
    \GLE INDEX RING LITTLE OUTSIGNWRITING HAND-ANGLE INDEX RING LITTLESIGNWRITING HAND-FIST MIDDLE DOWNSIGNWRITING HAND-HINGE MIDDLESIGNWRITING HAND-FIST MIDDLE UPSIGNWRITING HAND-CIRCLE MIDDLE UPSIGNWRITING HAND-FIST MIDDLE RAISED KNUCKLESIGNWRITING HAND-FIST\
    \ MIDDLE UP THUMB SIDESIGNWRITING HAND-HOOK MIDDLE THUMBSIGNWRITING HAND-FIST MIDDLE THUMB LITTLESIGNWRITING HAND-FIST MIDDLE LITTLESIGNWRITING HAND-FIST MIDDLE RING LITTLESIGNWRITING HAND-CIRCLE MIDDLE RING LITTLESIGNWRITING HAND-CURLICUE MIDDLE RING LITT\
    \LE ONSIGNWRITING HAND-CUP MIDDLE RING LITTLESIGNWRITING HAND-HINGE MIDDLE RING LITTLESIGNWRITING HAND-ANGLE MIDDLE RING LITTLE OUTSIGNWRITING HAND-ANGLE MIDDLE RING LITTLE INSIGNWRITING HAND-ANGLE MIDDLE RING LITTLESIGNWRITING HAND-CIRCLE MIDDLE RING LITT\
    \LE BENTSIGNWRITING HAND-CLAW MIDDLE RING LITTLE CONJOINEDSIGNWRITING HAND-CLAW MIDDLE RING LITTLE CONJOINED SIDESIGNWRITING HAND-HOOK MIDDLE RING LITTLE CONJOINED OUTSIGNWRITING HAND-HOOK MIDDLE RING LITTLE CONJOINED INSIGNWRITING HAND-HOOK MIDDLE RING LI\
    \TTLE CONJOINEDSIGNWRITING HAND-HINGE INDEX HINGEDSIGNWRITING HAND-FIST INDEX THUMB SIDESIGNWRITING HAND-HINGE INDEX THUMB SIDESIGNWRITING HAND-FIST INDEX THUMB SIDE THUMB DIAGONALSIGNWRITING HAND-FIST INDEX THUMB SIDE THUMB CONJOINEDSIGNWRITING HAND-FIST \
    \INDEX THUMB SIDE THUMB BENTSIGNWRITING HAND-FIST INDEX THUMB SIDE INDEX BENTSIGNWRITING HAND-FIST INDEX THUMB SIDE BOTH BENTSIGNWRITING HAND-FIST INDEX THUMB SIDE INDEX HINGESIGNWRITING HAND-FIST INDEX THUMB FORWARD INDEX STRAIGHTSIGNWRITING HAND-FIST IND\
    \EX THUMB FORWARD INDEX BENTSIGNWRITING HAND-FIST INDEX THUMB HOOKSIGNWRITING HAND-FIST INDEX THUMB CURLICUESIGNWRITING HAND-FIST INDEX THUMB CURVE THUMB INSIDESIGNWRITING HAND-CLAW INDEX THUMB CURVE THUMB INSIDESIGNWRITING HAND-FIST INDEX THUMB CURVE THUM\
    \B UNDERSIGNWRITING HAND-FIST INDEX THUMB CIRCLESIGNWRITING HAND-CUP INDEX THUMBSIGNWRITING HAND-CUP INDEX THUMB OPENSIGNWRITING HAND-HINGE INDEX THUMB OPENSIGNWRITING HAND-HINGE INDEX THUMB LARGESIGNWRITING HAND-HINGE INDEX THUMBSIGNWRITING HAND-HINGE IND\
    \EX THUMB SMALLSIGNWRITING HAND-ANGLE INDEX THUMB OUTSIGNWRITING HAND-ANGLE INDEX THUMB INSIGNWRITING HAND-ANGLE INDEX THUMBSIGNWRITING HAND-FIST THUMBSIGNWRITING HAND-FIST THUMB HEELSIGNWRITING HAND-FIST THUMB SIDE DIAGONALSIGNWRITING HAND-FIST THUMB SIDE\
    \ CONJOINEDSIGNWRITING HAND-FIST THUMB SIDE BENTSIGNWRITING HAND-FIST THUMB FORWARDSIGNWRITING HAND-FIST THUMB BETWEEN INDEX MIDDLESIGNWRITING HAND-FIST THUMB BETWEEN MIDDLE RINGSIGNWRITING HAND-FIST THUMB BETWEEN RING LITTLESIGNWRITING HAND-FIST THUMB UND\
    \ER TWO FINGERSSIGNWRITING HAND-FIST THUMB OVER TWO FINGERSSIGNWRITING HAND-FIST THUMB UNDER THREE FINGERSSIGNWRITING HAND-FIST THUMB UNDER FOUR FINGERSSIGNWRITING HAND-FIST THUMB OVER FOUR RAISED KNUCKLESSIGNWRITING HAND-FISTSIGNWRITING HAND-FIST HEELSIGN\
    \WRITING TOUCH SINGLESIGNWRITING TOUCH MULTIPLESIGNWRITING TOUCH BETWEENSIGNWRITING GRASP SINGLESIGNWRITING GRASP MULTIPLESIGNWRITING GRASP BETWEENSIGNWRITING STRIKE SINGLESIGNWRITING STRIKE MULTIPLESIGNWRITING STRIKE BETWEENSIGNWRITING BRUSH SINGLESIGNWRI\
    \TING BRUSH MULTIPLESIGNWRITING BRUSH BETWEENSIGNWRITING RUB SINGLESIGNWRITING RUB MULTIPLESIGNWRITING RUB BETWEENSIGNWRITING SURFACE SYMBOLSSIGNWRITING SURFACE BETWEENSIGNWRITING SQUEEZE LARGE SINGLESIGNWRITING SQUEEZE SMALL SINGLESIGNWRITING SQUEEZE LARG\
    \E MULTIPLESIGNWRITING SQUEEZE SMALL MULTIPLESIGNWRITING SQUEEZE SEQUENTIALSIGNWRITING FLICK LARGE SINGLESIGNWRITING FLICK SMALL SINGLESIGNWRITING FLICK LARGE MULTIPLESIGNWRITING FLICK SMALL MULTIPLESIGNWRITING FLICK SEQUENTIALSIGNWRITING SQUEEZE FLICK ALT\
    \ERNATINGSIGNWRITING MOVEMENT-HINGE UP DOWN LARGESIGNWRITING MOVEMENT-HINGE UP DOWN SMALLSIGNWRITING MOVEMENT-HINGE UP SEQUENTIALSIGNWRITING MOVEMENT-HINGE DOWN SEQUENTIALSIGNWRITING MOVEMENT-HINGE UP DOWN ALTERNATING LARGESIGNWRITING MOVEMENT-HINGE UP DOW\
    \N ALTERNATING SMALLSIGNWRITING MOVEMENT-HINGE SIDE TO SIDE SCISSORSSIGNWRITING MOVEMENT-WALLPLANE FINGER CONTACTSIGNWRITING MOVEMENT-FLOORPLANE FINGER CONTACTSIGNWRITING MOVEMENT-WALLPLANE SINGLE STRAIGHT SMALLSIGNWRITING MOVEMENT-WALLPLANE SINGLE STRAIGH\
    \T MEDIUMSIGNWRITING MOVEMENT-WALLPLANE SINGLE STRAIGHT LARGESIGNWRITING MOVEMENT-WALLPLANE SINGLE STRAIGHT LARGESTSIGNWRITING MOVEMENT-WALLPLANE SINGLE WRIST FLEXSIGNWRITING MOVEMENT-WALLPLANE DOUBLE STRAIGHTSIGNWRITING MOVEMENT-WALLPLANE DOUBLE WRIST FLE\
    \XSIGNWRITING MOVEMENT-WALLPLANE DOUBLE ALTERNATINGSIGNWRITING MOVEMENT-WALLPLANE DOUBLE ALTERNATING WRIST FLEXSIGNWRITING MOVEMENT-WALLPLANE CROSSSIGNWRITING MOVEMENT-WALLPLANE TRIPLE STRAIGHT MOVEMENTSIGNWRITING MOVEMENT-WALLPLANE TRIPLE WRIST FLEXSIGNWR\
    \ITING MOVEMENT-WALLPLANE TRIPLE ALTERNATINGSIGNWRITING MOVEMENT-WALLPLANE TRIPLE ALTERNATING WRIST FLEXSIGNWRITING MOVEMENT-WALLPLANE BEND SMALLSIGNWRITING MOVEMENT-WALLPLANE BEND MEDIUMSIGNWRITING MOVEMENT-WALLPLANE BEND LARGESIGNWRITING MOVEMENT-WALLPLA\
    \NE CORNER SMALLSIGNWRITING MOVEMENT-WALLPLANE CORNER MEDIUMSIGNWRITING MOVEMENT-WALLPLANE CORNER LARGESIGNWRITING MOVEMENT-WALLPLANE CORNER ROTATIONSIGNWRITING MOVEMENT-WALLPLANE CHECK SMALLSIGNWRITING MOVEMENT-WALLPLANE CHECK MEDIUMSIGNWRITING MOVEMENT-W\
    \ALLPLANE CHECK LARGESIGNWRITING MOVEMENT-WALLPLANE BOX SMALLSIGNWRITING MOVEMENT-WALLPLANE BOX MEDIUMSIGNWRITING MOVEMENT-WALLPLANE BOX LARGESIGNWRITING MOVEMENT-WALLPLANE ZIGZAG SMALLSIGNWRITING MOVEMENT-WALLPLANE ZIGZAG MEDIUMSIGNWRITING MOVEMENT-WALLPL\
    \ANE ZIGZAG LARGESIGNWRITING MOVEMENT-WALLPLANE PEAKS SMALLSIGNWRITING MOVEMENT-WALLPLANE PEAKS MEDIUMSIGNWRITING MOVEMENT-WALLPLANE PEAKS LARGESIGNWRITING TRAVEL-WALLPLANE ROTATION-WALLPLANE SINGLESIGNWRITING TRAVEL-WALLPLANE ROTATION-WALLPLANE DOUBLESIGN\
    \WRITING TRAVEL-WALLPLANE ROTATION-WALLPLANE ALTERNATINGSIGNWRITING TRAVEL-WALLPLANE ROTATION-FLOORPLANE SINGLESIGNWRITING TRAVEL-WALLPLANE ROTATION-FLOORPLANE DOUBLESIGNWRITING TRAVEL-WALLPLANE ROTATION-FLOORPLANE ALTERNATINGSIGNWRITING TRAVEL-WALLPLANE S\
    \HAKINGSIGNWRITING TRAVEL-WALLPLANE ARM SPIRAL SINGLESIGNWRITING TRAVEL-WALLPLANE ARM SPIRAL DOUBLESIGNWRITING TRAVEL-WALLPLANE ARM SPIRAL TRIPLESIGNWRITING MOVEMENT-DIAGONAL AWAY SMALLSIGNWRITING MOVEMENT-DIAGONAL AWAY MEDIUMSIGNWRITING MOVEMENT-DIAGONAL \
    \AWAY LARGESIGNWRITING MOVEMENT-DIAGONAL AWAY LARGESTSIGNWRITING MOVEMENT-DIAGONAL TOWARDS SMALLSIGNWRITING MOVEMENT-DIAGONAL TOWARDS MEDIUMSIGNWRITING MOVEMENT-DIAGONAL TOWARDS LARGESIGNWRITING MOVEMENT-DIAGONAL TOWARDS LARGESTSIGNWRITING MOVEMENT-DIAGONA\
    \L BETWEEN AWAY SMALLSIGNWRITING MOVEMENT-DIAGONAL BETWEEN AWAY MEDIUMSIGNWRITING MOVEMENT-DIAGONAL BETWEEN AWAY LARGESIGNWRITING MOVEMENT-DIAGONAL BETWEEN AWAY LARGESTSIGNWRITING MOVEMENT-DIAGONAL BETWEEN TOWARDS SMALLSIGNWRITING MOVEMENT-DIAGONAL BETWEEN\
    \ TOWARDS MEDIUMSIGNWRITING MOVEMENT-DIAGONAL BETWEEN TOWARDS LARGESIGNWRITING MOVEMENT-DIAGONAL BETWEEN TOWARDS LARGESTSIGNWRITING MOVEMENT-FLOORPLANE SINGLE STRAIGHT SMALLSIGNWRITING MOVEMENT-FLOORPLANE SINGLE STRAIGHT MEDIUMSIGNWRITING MOVEMENT-FLOORPLA\
    \NE SINGLE STRAIGHT LARGESIGNWRITING MOVEMENT-FLOORPLANE SINGLE STRAIGHT LARGESTSIGNWRITING MOVEMENT-FLOORPLANE SINGLE WRIST FLEXSIGNWRITING MOVEMENT-FLOORPLANE DOUBLE STRAIGHTSIGNWRITING MOVEMENT-FLOORPLANE DOUBLE WRIST FLEXSIGNWRITING MOVEMENT-FLOORPLANE\
    \ DOUBLE ALTERNATINGSIGNWRITING MOVEMENT-FLOORPLANE DOUBLE ALTERNATING WRIST FLEXSIGNWRITING MOVEMENT-FLOORPLANE CROSSSIGNWRITING MOVEMENT-FLOORPLANE TRIPLE STRAIGHT MOVEMENTSIGNWRITING MOVEMENT-FLOORPLANE TRIPLE WRIST FLEXSIGNWRITING MOVEMENT-FLOORPLANE T\
    \RIPLE ALTERNATING MOVEMENTSIGNWRITING MOVEMENT-FLOORPLANE TRIPLE ALTERNATING WRIST FLEXSIGNWRITING MOVEMENT-FLOORPLANE BENDSIGNWRITING MOVEMENT-FLOORPLANE CORNER SMALLSIGNWRITING MOVEMENT-FLOORPLANE CORNER MEDIUMSIGNWRITING MOVEMENT-FLOORPLANE CORNER LARG\
    \ESIGNWRITING MOVEMENT-FLOORPLANE CHECKSIGNWRITING MOVEMENT-FLOORPLANE BOX SMALLSIGNWRITING MOVEMENT-FLOORPLANE BOX MEDIUMSIGNWRITING MOVEMENT-FLOORPLANE BOX LARGESIGNWRITING MOVEMENT-FLOORPLANE ZIGZAG SMALLSIGNWRITING MOVEMENT-FLOORPLANE ZIGZAG MEDIUMSIGN\
    \WRITING MOVEMENT-FLOORPLANE ZIGZAG LARGESIGNWRITING MOVEMENT-FLOORPLANE PEAKS SMALLSIGNWRITING MOVEMENT-FLOORPLANE PEAKS MEDIUMSIGNWRITING MOVEMENT-FLOORPLANE PEAKS LARGESIGNWRITING TRAVEL-FLOORPLANE ROTATION-FLOORPLANE SINGLESIGNWRITING TRAVEL-FLOORPLANE\
    \ ROTATION-FLOORPLANE DOUBLESIGNWRITING TRAVEL-FLOORPLANE ROTATION-FLOORPLANE ALTERNATINGSIGNWRITING TRAVEL-FLOORPLANE ROTATION-WALLPLANE SINGLESIGNWRITING TRAVEL-FLOORPLANE ROTATION-WALLPLANE DOUBLESIGNWRITING TRAVEL-FLOORPLANE ROTATION-WALLPLANE ALTERNAT\
    \INGSIGNWRITING TRAVEL-FLOORPLANE SHAKINGSIGNWRITING MOVEMENT-WALLPLANE CURVE QUARTER SMALLSIGNWRITING MOVEMENT-WALLPLANE CURVE QUARTER MEDIUMSIGNWRITING MOVEMENT-WALLPLANE CURVE QUARTER LARGESIGNWRITING MOVEMENT-WALLPLANE CURVE QUARTER LARGESTSIGNWRITING \
    \MOVEMENT-WALLPLANE CURVE HALF-CIRCLE SMALLSIGNWRITING MOVEMENT-WALLPLANE CURVE HALF-CIRCLE MEDIUMSIGNWRITING MOVEMENT-WALLPLANE CURVE HALF-CIRCLE LARGESIGNWRITING MOVEMENT-WALLPLANE CURVE HALF-CIRCLE LARGESTSIGNWRITING MOVEMENT-WALLPLANE CURVE THREE-QUART\
    \ER CIRCLE SMALLSIGNWRITING MOVEMENT-WALLPLANE CURVE THREE-QUARTER CIRCLE MEDIUMSIGNWRITING MOVEMENT-WALLPLANE HUMP SMALLSIGNWRITING MOVEMENT-WALLPLANE HUMP MEDIUMSIGNWRITING MOVEMENT-WALLPLANE HUMP LARGESIGNWRITING MOVEMENT-WALLPLANE LOOP SMALLSIGNWRITING\
    \ MOVEMENT-WALLPLANE LOOP MEDIUMSIGNWRITING MOVEMENT-WALLPLANE LOOP LARGESIGNWRITING MOVEMENT-WALLPLANE LOOP SMALL DOUBLESIGNWRITING MOVEMENT-WALLPLANE WAVE CURVE DOUBLE SMALLSIGNWRITING MOVEMENT-WALLPLANE WAVE CURVE DOUBLE MEDIUMSIGNWRITING MOVEMENT-WALLP\
    \LANE WAVE CURVE DOUBLE LARGESIGNWRITING MOVEMENT-WALLPLANE WAVE CURVE TRIPLE SMALLSIGNWRITING MOVEMENT-WALLPLANE WAVE CURVE TRIPLE MEDIUMSIGNWRITING MOVEMENT-WALLPLANE WAVE CURVE TRIPLE LARGESIGNWRITING MOVEMENT-WALLPLANE CURVE THEN STRAIGHTSIGNWRITING MO\
    \VEMENT-WALLPLANE CURVED CROSS SMALLSIGNWRITING MOVEMENT-WALLPLANE CURVED CROSS MEDIUMSIGNWRITING ROTATION-WALLPLANE SINGLESIGNWRITING ROTATION-WALLPLANE DOUBLESIGNWRITING ROTATION-WALLPLANE ALTERNATESIGNWRITING MOVEMENT-WALLPLANE SHAKINGSIGNWRITING MOVEME\
    \NT-WALLPLANE CURVE HITTING FRONT WALLSIGNWRITING MOVEMENT-WALLPLANE HUMP HITTING FRONT WALLSIGNWRITING MOVEMENT-WALLPLANE LOOP HITTING FRONT WALLSIGNWRITING MOVEMENT-WALLPLANE WAVE HITTING FRONT WALLSIGNWRITING ROTATION-WALLPLANE SINGLE HITTING FRONT WALL\
    \SIGNWRITING ROTATION-WALLPLANE DOUBLE HITTING FRONT WALLSIGNWRITING ROTATION-WALLPLANE ALTERNATING HITTING FRONT WALLSIGNWRITING MOVEMENT-WALLPLANE CURVE HITTING CHESTSIGNWRITING MOVEMENT-WALLPLANE HUMP HITTING CHESTSIGNWRITING MOVEMENT-WALLPLANE LOOP HIT\
    \TING CHESTSIGNWRITING MOVEMENT-WALLPLANE WAVE HITTING CHESTSIGNWRITING ROTATION-WALLPLANE SINGLE HITTING CHESTSIGNWRITING ROTATION-WALLPLANE DOUBLE HITTING CHESTSIGNWRITING ROTATION-WALLPLANE ALTERNATING HITTING CHESTSIGNWRITING MOVEMENT-WALLPLANE WAVE DI\
    \AGONAL PATH SMALLSIGNWRITING MOVEMENT-WALLPLANE WAVE DIAGONAL PATH MEDIUMSIGNWRITING MOVEMENT-WALLPLANE WAVE DIAGONAL PATH LARGESIGNWRITING MOVEMENT-FLOORPLANE CURVE HITTING CEILING SMALLSIGNWRITING MOVEMENT-FLOORPLANE CURVE HITTING CEILING LARGESIGNWRITI\
    \NG MOVEMENT-FLOORPLANE HUMP HITTING CEILING SMALL DOUBLESIGNWRITING MOVEMENT-FLOORPLANE HUMP HITTING CEILING LARGE DOUBLESIGNWRITING MOVEMENT-FLOORPLANE HUMP HITTING CEILING SMALL TRIPLESIGNWRITING MOVEMENT-FLOORPLANE HUMP HITTING CEILING LARGE TRIPLESIGN\
    \WRITING MOVEMENT-FLOORPLANE LOOP HITTING CEILING SMALL SINGLESIGNWRITING MOVEMENT-FLOORPLANE LOOP HITTING CEILING LARGE SINGLESIGNWRITING MOVEMENT-FLOORPLANE LOOP HITTING CEILING SMALL DOUBLESIGNWRITING MOVEMENT-FLOORPLANE LOOP HITTING CEILING LARGE DOUBL\
    \ESIGNWRITING MOVEMENT-FLOORPLANE WAVE HITTING CEILING SMALLSIGNWRITING MOVEMENT-FLOORPLANE WAVE HITTING CEILING LARGESIGNWRITING ROTATION-FLOORPLANE SINGLE HITTING CEILINGSIGNWRITING ROTATION-FLOORPLANE DOUBLE HITTING CEILINGSIGNWRITING ROTATION-FLOORPLAN\
    \E ALTERNATING HITTING CEILINGSIGNWRITING MOVEMENT-FLOORPLANE CURVE HITTING FLOOR SMALLSIGNWRITING MOVEMENT-FLOORPLANE CURVE HITTING FLOOR LARGESIGNWRITING MOVEMENT-FLOORPLANE HUMP HITTING FLOOR SMALL DOUBLESIGNWRITING MOVEMENT-FLOORPLANE HUMP HITTING FLOO\
    \R LARGE DOUBLESIGNWRITING MOVEMENT-FLOORPLANE HUMP HITTING FLOOR TRIPLE SMALL TRIPLESIGNWRITING MOVEMENT-FLOORPLANE HUMP HITTING FLOOR TRIPLE LARGE TRIPLESIGNWRITING MOVEMENT-FLOORPLANE LOOP HITTING FLOOR SMALL SINGLESIGNWRITING MOVEMENT-FLOORPLANE LOOP H\
    \ITTING FLOOR LARGE SINGLESIGNWRITING MOVEMENT-FLOORPLANE LOOP HITTING FLOOR SMALL DOUBLESIGNWRITING MOVEMENT-FLOORPLANE LOOP HITTING FLOOR LARGE DOUBLESIGNWRITING MOVEMENT-FLOORPLANE WAVE HITTING FLOOR SMALLSIGNWRITING MOVEMENT-FLOORPLANE WAVE HITTING FLO\
    \OR LARGESIGNWRITING ROTATION-FLOORPLANE SINGLE HITTING FLOORSIGNWRITING ROTATION-FLOORPLANE DOUBLE HITTING FLOORSIGNWRITING ROTATION-FLOORPLANE ALTERNATING HITTING FLOORSIGNWRITING MOVEMENT-FLOORPLANE CURVE SMALLSIGNWRITING MOVEMENT-FLOORPLANE CURVE MEDIU\
    \MSIGNWRITING MOVEMENT-FLOORPLANE CURVE LARGESIGNWRITING MOVEMENT-FLOORPLANE CURVE LARGESTSIGNWRITING MOVEMENT-FLOORPLANE CURVE COMBINEDSIGNWRITING MOVEMENT-FLOORPLANE HUMP SMALLSIGNWRITING MOVEMENT-FLOORPLANE LOOP SMALLSIGNWRITING MOVEMENT-FLOORPLANE WAVE\
    \ SNAKESIGNWRITING MOVEMENT-FLOORPLANE WAVE SMALLSIGNWRITING MOVEMENT-FLOORPLANE WAVE LARGESIGNWRITING ROTATION-FLOORPLANE SINGLESIGNWRITING ROTATION-FLOORPLANE DOUBLESIGNWRITING ROTATION-FLOORPLANE ALTERNATINGSIGNWRITING MOVEMENT-FLOORPLANE SHAKING PARALL\
    \ELSIGNWRITING MOVEMENT-WALLPLANE ARM CIRCLE SMALL SINGLESIGNWRITING MOVEMENT-WALLPLANE ARM CIRCLE MEDIUM SINGLESIGNWRITING MOVEMENT-WALLPLANE ARM CIRCLE SMALL DOUBLESIGNWRITING MOVEMENT-WALLPLANE ARM CIRCLE MEDIUM DOUBLESIGNWRITING MOVEMENT-FLOORPLANE ARM\
    \ CIRCLE HITTING WALL SMALL SINGLESIGNWRITING MOVEMENT-FLOORPLANE ARM CIRCLE HITTING WALL MEDIUM SINGLESIGNWRITING MOVEMENT-FLOORPLANE ARM CIRCLE HITTING WALL LARGE SINGLESIGNWRITING MOVEMENT-FLOORPLANE ARM CIRCLE HITTING WALL SMALL DOUBLESIGNWRITING MOVEM\
    \ENT-FLOORPLANE ARM CIRCLE HITTING WALL MEDIUM DOUBLESIGNWRITING MOVEMENT-FLOORPLANE ARM CIRCLE HITTING WALL LARGE DOUBLESIGNWRITING MOVEMENT-WALLPLANE WRIST CIRCLE FRONT SINGLESIGNWRITING MOVEMENT-WALLPLANE WRIST CIRCLE FRONT DOUBLESIGNWRITING MOVEMENT-FL\
    \OORPLANE WRIST CIRCLE HITTING WALL SINGLESIGNWRITING MOVEMENT-FLOORPLANE WRIST CIRCLE HITTING WALL DOUBLESIGNWRITING MOVEMENT-WALLPLANE FINGER CIRCLES SINGLESIGNWRITING MOVEMENT-WALLPLANE FINGER CIRCLES DOUBLESIGNWRITING MOVEMENT-FLOORPLANE FINGER CIRCLES\
    \ HITTING WALL SINGLESIGNWRITING MOVEMENT-FLOORPLANE FINGER CIRCLES HITTING WALL DOUBLESIGNWRITING DYNAMIC ARROWHEAD SMALLSIGNWRITING DYNAMIC ARROWHEAD LARGESIGNWRITING DYNAMIC FASTSIGNWRITING DYNAMIC SLOWSIGNWRITING DYNAMIC TENSESIGNWRITING DYNAMIC RELAXE\
    \DSIGNWRITING DYNAMIC SIMULTANEOUSSIGNWRITING DYNAMIC SIMULTANEOUS ALTERNATINGSIGNWRITING DYNAMIC EVERY OTHER TIMESIGNWRITING DYNAMIC GRADUALSIGNWRITING HEADSIGNWRITING HEAD RIMSIGNWRITING HEAD MOVEMENT-WALLPLANE STRAIGHTSIGNWRITING HEAD MOVEMENT-WALLPLANE\
    \ TILTSIGNWRITING HEAD MOVEMENT-FLOORPLANE STRAIGHTSIGNWRITING HEAD MOVEMENT-WALLPLANE CURVESIGNWRITING HEAD MOVEMENT-FLOORPLANE CURVESIGNWRITING HEAD MOVEMENT CIRCLESIGNWRITING FACE DIRECTION POSITION NOSE FORWARD TILTINGSIGNWRITING FACE DIRECTION POSITIO\
    \N NOSE UP OR DOWNSIGNWRITING FACE DIRECTION POSITION NOSE UP OR DOWN TILTINGSIGNWRITING EYEBROWS STRAIGHT UPSIGNWRITING EYEBROWS STRAIGHT NEUTRALSIGNWRITING EYEBROWS STRAIGHT DOWNSIGNWRITING DREAMY EYEBROWS NEUTRAL DOWNSIGNWRITING DREAMY EYEBROWS DOWN NEU\
    \TRALSIGNWRITING DREAMY EYEBROWS UP NEUTRALSIGNWRITING DREAMY EYEBROWS NEUTRAL UPSIGNWRITING FOREHEAD NEUTRALSIGNWRITING FOREHEAD CONTACTSIGNWRITING FOREHEAD WRINKLEDSIGNWRITING EYES OPENSIGNWRITING EYES SQUEEZEDSIGNWRITING EYES CLOSEDSIGNWRITING EYE BLINK\
    \ SINGLESIGNWRITING EYE BLINK MULTIPLESIGNWRITING EYES HALF OPENSIGNWRITING EYES WIDE OPENSIGNWRITING EYES HALF CLOSEDSIGNWRITING EYES WIDENING MOVEMENTSIGNWRITING EYE WINKSIGNWRITING EYELASHES UPSIGNWRITING EYELASHES DOWNSIGNWRITING EYELASHES FLUTTERINGSI\
    \GNWRITING EYEGAZE-WALLPLANE STRAIGHTSIGNWRITING EYEGAZE-WALLPLANE STRAIGHT DOUBLESIGNWRITING EYEGAZE-WALLPLANE STRAIGHT ALTERNATINGSIGNWRITING EYEGAZE-FLOORPLANE STRAIGHTSIGNWRITING EYEGAZE-FLOORPLANE STRAIGHT DOUBLESIGNWRITING EYEGAZE-FLOORPLANE STRAIGHT\
    \ ALTERNATINGSIGNWRITING EYEGAZE-WALLPLANE CURVEDSIGNWRITING EYEGAZE-FLOORPLANE CURVEDSIGNWRITING EYEGAZE-WALLPLANE CIRCLINGSIGNWRITING CHEEKS PUFFEDSIGNWRITING CHEEKS NEUTRALSIGNWRITING CHEEKS SUCKEDSIGNWRITING TENSE CHEEKS HIGHSIGNWRITING TENSE CHEEKS MI\
    \DDLESIGNWRITING TENSE CHEEKS LOWSIGNWRITING EARSSIGNWRITING NOSE NEUTRALSIGNWRITING NOSE CONTACTSIGNWRITING NOSE WRINKLESSIGNWRITING NOSE WIGGLESSIGNWRITING AIR BLOWING OUTSIGNWRITING AIR SUCKING INSIGNWRITING AIR BLOW SMALL ROTATIONSSIGNWRITING AIR SUCK \
    \SMALL ROTATIONSSIGNWRITING BREATH INHALESIGNWRITING BREATH EXHALESIGNWRITING MOUTH CLOSED NEUTRALSIGNWRITING MOUTH CLOSED FORWARDSIGNWRITING MOUTH CLOSED CONTACTSIGNWRITING MOUTH SMILESIGNWRITING MOUTH SMILE WRINKLEDSIGNWRITING MOUTH SMILE OPENSIGNWRITING\
    \ MOUTH FROWNSIGNWRITING MOUTH FROWN WRINKLEDSIGNWRITING MOUTH FROWN OPENSIGNWRITING MOUTH OPEN CIRCLESIGNWRITING MOUTH OPEN FORWARDSIGNWRITING MOUTH OPEN WRINKLEDSIGNWRITING MOUTH OPEN OVALSIGNWRITING MOUTH OPEN OVAL WRINKLEDSIGNWRITING MOUTH OPEN OVAL YA\
    \WNSIGNWRITING MOUTH OPEN RECTANGLESIGNWRITING MOUTH OPEN RECTANGLE WRINKLEDSIGNWRITING MOUTH OPEN RECTANGLE YAWNSIGNWRITING MOUTH KISSSIGNWRITING MOUTH KISS FORWARDSIGNWRITING MOUTH KISS WRINKLEDSIGNWRITING MOUTH TENSESIGNWRITING MOUTH TENSE FORWARDSIGNWR\
    \ITING MOUTH TENSE SUCKEDSIGNWRITING LIPS PRESSED TOGETHERSIGNWRITING LIP LOWER OVER UPPERSIGNWRITING LIP UPPER OVER LOWERSIGNWRITING MOUTH CORNERSSIGNWRITING MOUTH WRINKLES SINGLESIGNWRITING MOUTH WRINKLES DOUBLESIGNWRITING TONGUE STICKING OUT FARSIGNWRIT\
    \ING TONGUE LICKING LIPSSIGNWRITING TONGUE TIP BETWEEN LIPSSIGNWRITING TONGUE TIP TOUCHING INSIDE MOUTHSIGNWRITING TONGUE INSIDE MOUTH RELAXEDSIGNWRITING TONGUE MOVES AGAINST CHEEKSIGNWRITING TONGUE CENTRE STICKING OUTSIGNWRITING TONGUE CENTRE INSIDE MOUTH\
    \SIGNWRITING TEETHSIGNWRITING TEETH MOVEMENTSIGNWRITING TEETH ON TONGUESIGNWRITING TEETH ON TONGUE MOVEMENTSIGNWRITING TEETH ON LIPSSIGNWRITING TEETH ON LIPS MOVEMENTSIGNWRITING TEETH BITE LIPSSIGNWRITING MOVEMENT-WALLPLANE JAWSIGNWRITING MOVEMENT-FLOORPLA\
    \NE JAWSIGNWRITING NECKSIGNWRITING HAIRSIGNWRITING EXCITEMENTSIGNWRITING SHOULDER HIP SPINESIGNWRITING SHOULDER HIP POSITIONSSIGNWRITING WALLPLANE SHOULDER HIP MOVESIGNWRITING FLOORPLANE SHOULDER HIP MOVESIGNWRITING SHOULDER TILTING FROM WAISTSIGNWRITING T\
    \ORSO-WALLPLANE STRAIGHT STRETCHSIGNWRITING TORSO-WALLPLANE CURVED BENDSIGNWRITING TORSO-FLOORPLANE TWISTINGSIGNWRITING UPPER BODY TILTING FROM HIP JOINTSSIGNWRITING LIMB COMBINATIONSIGNWRITING LIMB LENGTH-1SIGNWRITING LIMB LENGTH-2SIGNWRITING LIMB LENGTH-\
    \3SIGNWRITING LIMB LENGTH-4SIGNWRITING LIMB LENGTH-5SIGNWRITING LIMB LENGTH-6SIGNWRITING LIMB LENGTH-7SIGNWRITING FINGERSIGNWRITING LOCATION-WALLPLANE SPACESIGNWRITING LOCATION-FLOORPLANE SPACESIGNWRITING LOCATION HEIGHTSIGNWRITING LOCATION WIDTHSIGNWRITIN\
    \G LOCATION DEPTHSIGNWRITING LOCATION HEAD NECKSIGNWRITING LOCATION TORSOSIGNWRITING LOCATION LIMBS DIGITSSIGNWRITING COMMASIGNWRITING FULL STOPSIGNWRITING SEMICOLONSIGNWRITING COLONSIGNWRITING PARENTHESISSIGNWRITING FILL MODIFIER-2SIGNWRITING FILL MODIFIE\
    \R-3SIGNWRITING FILL MODIFIER-4SIGNWRITING FILL MODIFIER-5SIGNWRITING FILL MODIFIER-6SIGNWRITING ROTATION MODIFIER-2SIGNWRITING ROTATION MODIFIER-3SIGNWRITING ROTATION MODIFIER-4SIGNWRITING ROTATION MODIFIER-5SIGNWRITING ROTATION MODIFIER-6SIGNWRITING ROTA\
    \TION MODIFIER-7SIGNWRITING ROTATION MODIFIER-8SIGNWRITING ROTATION MODIFIER-9SIGNWRITING ROTATION MODIFIER-10SIGNWRITING ROTATION MODIFIER-11SIGNWRITING ROTATION MODIFIER-12SIGNWRITING ROTATION MODIFIER-13SIGNWRITING ROTATION MODIFIER-14SIGNWRITING ROTATI\
    \ON MODIFIER-15SIGNWRITING ROTATION MODIFIER-16LATIN SMALL LETTER FENG DIGRAPH WITH TRILLLATIN SMALL LETTER REVERSED SCRIPT GLATIN LETTER SMALL CAPITAL TURNED GLATIN SMALL LETTER REVERSED KLATIN LETTER SMALL CAPITAL L WITH BELTLATIN SMALL LETTER LEZH WITH \
    \RETROFLEX HOOKLATIN SMALL LETTER TURNED Y WITH BELTLATIN SMALL LETTER REVERSED ENGLATIN SMALL LETTER TURNED R WITH LONG LEG AND RETROFLEX HOOKLATIN SMALL LETTER T WITH HOOK AND RETROFLEX HOOKLATIN LETTER RETROFLEX CLICK WITH RETROFLEX HOOKLATIN SMALL LETT\
    \ER ESH WITH DOUBLE BARLATIN SMALL LETTER ESH WITH DOUBLE BAR AND CURLLATIN SMALL LETTER TURNED T WITH CURLLATIN LETTER INVERTED GLOTTAL STOP WITH CURLLATIN LETTER STRETCHED C WITH CURLLATIN LETTER SMALL CAPITAL TURNED KLATIN SMALL LETTER L WITH FISHHOOKLA\
    \TIN SMALL LETTER DEZH DIGRAPH WITH PALATAL HOOKLATIN SMALL LETTER L WITH BELT AND PALATAL HOOKLATIN SMALL LETTER ENG WITH PALATAL HOOKLATIN SMALL LETTER TURNED R WITH PALATAL HOOKLATIN SMALL LETTER R WITH FISHHOOK AND PALATAL HOOKLATIN SMALL LETTER TESH D\
    \IGRAPH WITH PALATAL HOOKLATIN SMALL LETTER EZH WITH PALATAL HOOKLATIN SMALL LETTER DEZH DIGRAPH WITH RETROFLEX HOOKLATIN SMALL LETTER I WITH STROKE AND RETROFLEX HOOKLATIN SMALL LETTER O WITH RETROFLEX HOOKLATIN SMALL LETTER TESH DIGRAPH WITH RETROFLEX HO\
    \OKLATIN SMALL LETTER C WITH RETROFLEX HOOKLATIN SMALL LETTER S WITH CURLLATIN SMALL LETTER D WITH MID-HEIGHT LEFT HOOKLATIN SMALL LETTER L WITH MID-HEIGHT LEFT HOOKLATIN SMALL LETTER N WITH MID-HEIGHT LEFT HOOKLATIN SMALL LETTER R WITH MID-HEIGHT LEFT HOO\
    \KLATIN SMALL LETTER S WITH MID-HEIGHT LEFT HOOKLATIN SMALL LETTER T WITH MID-HEIGHT LEFT HOOKCOMBINING GLAGOLITIC LETTER AZUCOMBINING GLAGOLITIC LETTER BUKYCOMBINING GLAGOLITIC LETTER VEDECOMBINING GLAGOLITIC LETTER GLAGOLICOMBINING GLAGOLITIC LETTER DOBR\
    \OCOMBINING GLAGOLITIC LETTER YESTUCOMBINING GLAGOLITIC LETTER ZHIVETECOMBINING GLAGOLITIC LETTER ZEMLJACOMBINING GLAGOLITIC LETTER IZHECOMBINING GLAGOLITIC LETTER INITIAL IZHECOMBINING GLAGOLITIC LETTER ICOMBINING GLAGOLITIC LETTER DJERVICOMBINING GLAGOLI\
    \TIC LETTER KAKOCOMBINING GLAGOLITIC LETTER LJUDIJECOMBINING GLAGOLITIC LETTER MYSLITECOMBINING GLAGOLITIC LETTER NASHICOMBINING GLAGOLITIC LETTER ONUCOMBINING GLAGOLITIC LETTER POKOJICOMBINING GLAGOLITIC LETTER RITSICOMBINING GLAGOLITIC LETTER SLOVOCOMBIN\
    \ING GLAGOLITIC LETTER TVRIDOCOMBINING GLAGOLITIC LETTER UKUCOMBINING GLAGOLITIC LETTER FRITUCOMBINING GLAGOLITIC LETTER HERUCOMBINING GLAGOLITIC LETTER SHTACOMBINING GLAGOLITIC LETTER TSICOMBINING GLAGOLITIC LETTER CHRIVICOMBINING GLAGOLITIC LETTER SHACOM\
    \BINING GLAGOLITIC LETTER YERUCOMBINING GLAGOLITIC LETTER YERICOMBINING GLAGOLITIC LETTER YATICOMBINING GLAGOLITIC LETTER YUCOMBINING GLAGOLITIC LETTER SMALL YUSCOMBINING GLAGOLITIC LETTER YOCOMBINING GLAGOLITIC LETTER IOTATED SMALL YUSCOMBINING GLAGOLITIC\
    \ LETTER BIG YUSCOMBINING GLAGOLITIC LETTER IOTATED BIG YUSCOMBINING GLAGOLITIC LETTER FITAMODIFIER LETTER CYRILLIC SMALL AMODIFIER LETTER CYRILLIC SMALL BEMODIFIER LETTER CYRILLIC SMALL VEMODIFIER LETTER CYRILLIC SMALL GHEMODIFIER LETTER CYRILLIC SMALL DE\
    \MODIFIER LETTER CYRILLIC SMALL IEMODIFIER LETTER CYRILLIC SMALL ZHEMODIFIER LETTER CYRILLIC SMALL ZEMODIFIER LETTER CYRILLIC SMALL IMODIFIER LETTER CYRILLIC SMALL KAMODIFIER LETTER CYRILLIC SMALL ELMODIFIER LETTER CYRILLIC SMALL EMMODIFIER LETTER CYRILLIC\
    \ SMALL OMODIFIER LETTER CYRILLIC SMALL PEMODIFIER LETTER CYRILLIC SMALL ERMODIFIER LETTER CYRILLIC SMALL ESMODIFIER LETTER CYRILLIC SMALL TEMODIFIER LETTER CYRILLIC SMALL UMODIFIER LETTER CYRILLIC SMALL EFMODIFIER LETTER CYRILLIC SMALL HAMODIFIER LETTER C\
    \YRILLIC SMALL TSEMODIFIER LETTER CYRILLIC SMALL CHEMODIFIER LETTER CYRILLIC SMALL SHAMODIFIER LETTER CYRILLIC SMALL YERUMODIFIER LETTER CYRILLIC SMALL EMODIFIER LETTER CYRILLIC SMALL YUMODIFIER LETTER CYRILLIC SMALL DZZEMODIFIER LETTER CYRILLIC SMALL SCHW\
    \AMODIFIER LETTER CYRILLIC SMALL BYELORUSSIAN-UKRAINIAN IMODIFIER LETTER CYRILLIC SMALL JEMODIFIER LETTER CYRILLIC SMALL BARRED OMODIFIER LETTER CYRILLIC SMALL STRAIGHT UMODIFIER LETTER CYRILLIC SMALL PALOCHKACYRILLIC SUBSCRIPT SMALL LETTER ACYRILLIC SUBSC\
    \RIPT SMALL LETTER BECYRILLIC SUBSCRIPT SMALL LETTER VECYRILLIC SUBSCRIPT SMALL LETTER GHECYRILLIC SUBSCRIPT SMALL LETTER DECYRILLIC SUBSCRIPT SMALL LETTER IECYRILLIC SUBSCRIPT SMALL LETTER ZHECYRILLIC SUBSCRIPT SMALL LETTER ZECYRILLIC SUBSCRIPT SMALL LETT\
    \ER ICYRILLIC SUBSCRIPT SMALL LETTER KACYRILLIC SUBSCRIPT SMALL LETTER ELCYRILLIC SUBSCRIPT SMALL LETTER OCYRILLIC SUBSCRIPT SMALL LETTER PECYRILLIC SUBSCRIPT SMALL LETTER ESCYRILLIC SUBSCRIPT SMALL LETTER UCYRILLIC SUBSCRIPT SMALL LETTER EFCYRILLIC SUBSCR\
    \IPT SMALL LETTER HACYRILLIC SUBSCRIPT SMALL LETTER TSECYRILLIC SUBSCRIPT SMALL LETTER CHECYRILLIC SUBSCRIPT SMALL LETTER SHACYRILLIC SUBSCRIPT SMALL LETTER HARD SIGNCYRILLIC SUBSCRIPT SMALL LETTER YERUCYRILLIC SUBSCRIPT SMALL LETTER GHE WITH UPTURNCYRILLI\
    \C SUBSCRIPT SMALL LETTER BYELORUSSIAN-UKRAINIAN ICYRILLIC SUBSCRIPT SMALL LETTER DZECYRILLIC SUBSCRIPT SMALL LETTER DZHEMODIFIER LETTER CYRILLIC SMALL ES WITH DESCENDERMODIFIER LETTER CYRILLIC SMALL YERU WITH BACK YERMODIFIER LETTER CYRILLIC SMALL STRAIGH\
    \T U WITH STROKECOMBINING CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN INYIAKENG PUACHUE HMONG LETTER MANYIAKENG PUACHUE HMONG LETTER TSANYIAKENG PUACHUE HMONG LETTER NTANYIAKENG PUACHUE HMONG LETTER TANYIAKENG PUACHUE HMONG LETTER HANYIAKENG PUACHUE HMONG\
    \ LETTER NANYIAKENG PUACHUE HMONG LETTER XANYIAKENG PUACHUE HMONG LETTER NKANYIAKENG PUACHUE HMONG LETTER CANYIAKENG PUACHUE HMONG LETTER LANYIAKENG PUACHUE HMONG LETTER SANYIAKENG PUACHUE HMONG LETTER ZANYIAKENG PUACHUE HMONG LETTER NCANYIAKENG PUACHUE HM\
    \ONG LETTER NTSANYIAKENG PUACHUE HMONG LETTER KANYIAKENG PUACHUE HMONG LETTER DANYIAKENG PUACHUE HMONG LETTER NYANYIAKENG PUACHUE HMONG LETTER NRANYIAKENG PUACHUE HMONG LETTER VANYIAKENG PUACHUE HMONG LETTER NTXANYIAKENG PUACHUE HMONG LETTER TXANYIAKENG PU\
    \ACHUE HMONG LETTER FANYIAKENG PUACHUE HMONG LETTER RANYIAKENG PUACHUE HMONG LETTER QANYIAKENG PUACHUE HMONG LETTER YANYIAKENG PUACHUE HMONG LETTER NQANYIAKENG PUACHUE HMONG LETTER PANYIAKENG PUACHUE HMONG LETTER XYANYIAKENG PUACHUE HMONG LETTER NPANYIAKEN\
    \G PUACHUE HMONG LETTER DLANYIAKENG PUACHUE HMONG LETTER NPLANYIAKENG PUACHUE HMONG LETTER HAHNYIAKENG PUACHUE HMONG LETTER MLANYIAKENG PUACHUE HMONG LETTER PLANYIAKENG PUACHUE HMONG LETTER GANYIAKENG PUACHUE HMONG LETTER RRANYIAKENG PUACHUE HMONG LETTER A\
    \NYIAKENG PUACHUE HMONG LETTER AANYIAKENG PUACHUE HMONG LETTER INYIAKENG PUACHUE HMONG LETTER UNYIAKENG PUACHUE HMONG LETTER ONYIAKENG PUACHUE HMONG LETTER OONYIAKENG PUACHUE HMONG LETTER ENYIAKENG PUACHUE HMONG LETTER EENYIAKENG PUACHUE HMONG LETTER WNYIA\
    \KENG PUACHUE HMONG TONE-BNYIAKENG PUACHUE HMONG TONE-MNYIAKENG PUACHUE HMONG TONE-JNYIAKENG PUACHUE HMONG TONE-VNYIAKENG PUACHUE HMONG TONE-SNYIAKENG PUACHUE HMONG TONE-GNYIAKENG PUACHUE HMONG TONE-DNYIAKENG PUACHUE HMONG SIGN FOR PERSONNYIAKENG PUACHUE H\
    \MONG SIGN FOR THINGNYIAKENG PUACHUE HMONG SIGN FOR LOCATIONNYIAKENG PUACHUE HMONG SIGN FOR ANIMALNYIAKENG PUACHUE HMONG SIGN FOR INVERTEBRATENYIAKENG PUACHUE HMONG SIGN XW XWNYIAKENG PUACHUE HMONG SYLLABLE LENGTHENERNYIAKENG PUACHUE HMONG DIGIT ZERONYIAKE\
    \NG PUACHUE HMONG DIGIT ONENYIAKENG PUACHUE HMONG DIGIT TWONYIAKENG PUACHUE HMONG DIGIT THREENYIAKENG PUACHUE HMONG DIGIT FOURNYIAKENG PUACHUE HMONG DIGIT FIVENYIAKENG PUACHUE HMONG DIGIT SIXNYIAKENG PUACHUE HMONG DIGIT SEVENNYIAKENG PUACHUE HMONG DIGIT EI\
    \GHTNYIAKENG PUACHUE HMONG DIGIT NINENYIAKENG PUACHUE HMONG LOGOGRAM NYAJNYIAKENG PUACHUE HMONG CIRCLED CATOTO LETTER PATOTO LETTER BATOTO LETTER TATOTO LETTER DATOTO LETTER KATOTO LETTER GATOTO LETTER MATOTO LETTER NATOTO LETTER NGATOTO LETTER SATOTO LETT\
    \ER CHATOTO LETTER YATOTO LETTER WATOTO LETTER JATOTO LETTER HATOTO LETTER RATOTO LETTER LATOTO LETTER ITOTO LETTER BREATHY ITOTO LETTER IUTOTO LETTER BREATHY IUTOTO LETTER UTOTO LETTER ETOTO LETTER BREATHY ETOTO LETTER EOTOTO LETTER BREATHY EOTOTO LETTER \
    \OTOTO LETTER AETOTO LETTER BREATHY AETOTO LETTER ATOTO SIGN RISING TONEWANCHO LETTER AAWANCHO LETTER AWANCHO LETTER BAWANCHO LETTER CAWANCHO LETTER DAWANCHO LETTER GAWANCHO LETTER YAWANCHO LETTER PHAWANCHO LETTER LAWANCHO LETTER NAWANCHO LETTER PAWANCHO L\
    \ETTER TAWANCHO LETTER THAWANCHO LETTER FAWANCHO LETTER SAWANCHO LETTER SHAWANCHO LETTER JAWANCHO LETTER ZAWANCHO LETTER WAWANCHO LETTER VAWANCHO LETTER KAWANCHO LETTER OWANCHO LETTER AUWANCHO LETTER RAWANCHO LETTER MAWANCHO LETTER KHAWANCHO LETTER HAWANCH\
    \O LETTER EWANCHO LETTER IWANCHO LETTER NGAWANCHO LETTER UWANCHO LETTER LLHAWANCHO LETTER TSAWANCHO LETTER TRAWANCHO LETTER ONGWANCHO LETTER AANGWANCHO LETTER ANGWANCHO LETTER INGWANCHO LETTER ONWANCHO LETTER ENWANCHO LETTER AANWANCHO LETTER NYAWANCHO LETT\
    \ER UENWANCHO LETTER YIHWANCHO TONE TUPWANCHO TONE TUPNIWANCHO TONE KOIWANCHO TONE KOINIWANCHO DIGIT ZEROWANCHO DIGIT ONEWANCHO DIGIT TWOWANCHO DIGIT THREEWANCHO DIGIT FOURWANCHO DIGIT FIVEWANCHO DIGIT SIXWANCHO DIGIT SEVENWANCHO DIGIT EIGHTWANCHO DIGIT NI\
    \NEWANCHO NGUN SIGNNAG MUNDARI LETTER ONAG MUNDARI LETTER OPNAG MUNDARI LETTER OLNAG MUNDARI LETTER OYNAG MUNDARI LETTER ONGNAG MUNDARI LETTER ANAG MUNDARI LETTER AJNAG MUNDARI LETTER ABNAG MUNDARI LETTER ANYNAG MUNDARI LETTER AHNAG MUNDARI LETTER INAG MUN\
    \DARI LETTER ISNAG MUNDARI LETTER IDDNAG MUNDARI LETTER ITNAG MUNDARI LETTER IHNAG MUNDARI LETTER UNAG MUNDARI LETTER UCNAG MUNDARI LETTER UDNAG MUNDARI LETTER UKNAG MUNDARI LETTER URNAG MUNDARI LETTER ENAG MUNDARI LETTER ENNNAG MUNDARI LETTER EGNAG MUNDAR\
    \I LETTER EMNAG MUNDARI LETTER ENNAG MUNDARI LETTER ETTNAG MUNDARI LETTER ELLNAG MUNDARI SIGN OJODNAG MUNDARI SIGN MUHORNAG MUNDARI SIGN TOYORNAG MUNDARI SIGN IKIRNAG MUNDARI SIGN SUTUHNAG MUNDARI DIGIT ZERONAG MUNDARI DIGIT ONENAG MUNDARI DIGIT TWONAG MUN\
    \DARI DIGIT THREENAG MUNDARI DIGIT FOURNAG MUNDARI DIGIT FIVENAG MUNDARI DIGIT SIXNAG MUNDARI DIGIT SEVENNAG MUNDARI DIGIT EIGHTNAG MUNDARI DIGIT NINEETHIOPIC SYLLABLE HHYAETHIOPIC SYLLABLE HHYUETHIOPIC SYLLABLE HHYIETHIOPIC SYLLABLE HHYAAETHIOPIC SYLLABLE\
    \ HHYEEETHIOPIC SYLLABLE HHYEETHIOPIC SYLLABLE HHYOETHIOPIC SYLLABLE GURAGE HHWAETHIOPIC SYLLABLE HHWIETHIOPIC SYLLABLE HHWEEETHIOPIC SYLLABLE HHWEETHIOPIC SYLLABLE GURAGE MWIETHIOPIC SYLLABLE GURAGE MWEEETHIOPIC SYLLABLE GURAGE QWIETHIOPIC SYLLABLE GURAGE\
    \ QWEEETHIOPIC SYLLABLE GURAGE QWEETHIOPIC SYLLABLE GURAGE BWIETHIOPIC SYLLABLE GURAGE BWEEETHIOPIC SYLLABLE GURAGE KWIETHIOPIC SYLLABLE GURAGE KWEEETHIOPIC SYLLABLE GURAGE KWEETHIOPIC SYLLABLE GURAGE GWIETHIOPIC SYLLABLE GURAGE GWEEETHIOPIC SYLLABLE GURAG\
    \E GWEETHIOPIC SYLLABLE GURAGE FWIETHIOPIC SYLLABLE GURAGE FWEEETHIOPIC SYLLABLE GURAGE PWIETHIOPIC SYLLABLE GURAGE PWEEMENDE KIKAKUI SYLLABLE M001 KIMENDE KIKAKUI SYLLABLE M002 KAMENDE KIKAKUI SYLLABLE M003 KUMENDE KIKAKUI SYLLABLE M065 KEEMENDE KIKAKUI S\
    \YLLABLE M095 KEMENDE KIKAKUI SYLLABLE M076 KOOMENDE KIKAKUI SYLLABLE M048 KOMENDE KIKAKUI SYLLABLE M179 KUAMENDE KIKAKUI SYLLABLE M004 WIMENDE KIKAKUI SYLLABLE M005 WAMENDE KIKAKUI SYLLABLE M006 WUMENDE KIKAKUI SYLLABLE M126 WEEMENDE KIKAKUI SYLLABLE M118\
    \ WEMENDE KIKAKUI SYLLABLE M114 WOOMENDE KIKAKUI SYLLABLE M045 WOMENDE KIKAKUI SYLLABLE M194 WUIMENDE KIKAKUI SYLLABLE M143 WEIMENDE KIKAKUI SYLLABLE M061 WVIMENDE KIKAKUI SYLLABLE M049 WVAMENDE KIKAKUI SYLLABLE M139 WVEMENDE KIKAKUI SYLLABLE M007 MINMENDE\
    \ KIKAKUI SYLLABLE M008 MANMENDE KIKAKUI SYLLABLE M009 MUNMENDE KIKAKUI SYLLABLE M059 MENMENDE KIKAKUI SYLLABLE M094 MONMENDE KIKAKUI SYLLABLE M154 MUANMENDE KIKAKUI SYLLABLE M189 MUENMENDE KIKAKUI SYLLABLE M010 BIMENDE KIKAKUI SYLLABLE M011 BAMENDE KIKAKU\
    \I SYLLABLE M012 BUMENDE KIKAKUI SYLLABLE M150 BEEMENDE KIKAKUI SYLLABLE M097 BEMENDE KIKAKUI SYLLABLE M103 BOOMENDE KIKAKUI SYLLABLE M138 BOMENDE KIKAKUI SYLLABLE M013 IMENDE KIKAKUI SYLLABLE M014 AMENDE KIKAKUI SYLLABLE M015 UMENDE KIKAKUI SYLLABLE M163 \
    \EEMENDE KIKAKUI SYLLABLE M100 EMENDE KIKAKUI SYLLABLE M165 OOMENDE KIKAKUI SYLLABLE M147 OMENDE KIKAKUI SYLLABLE M137 EIMENDE KIKAKUI SYLLABLE M131 INMENDE KIKAKUI SYLLABLE M135 INMENDE KIKAKUI SYLLABLE M195 ANMENDE KIKAKUI SYLLABLE M178 ENMENDE KIKAKUI S\
    \YLLABLE M019 SIMENDE KIKAKUI SYLLABLE M020 SAMENDE KIKAKUI SYLLABLE M021 SUMENDE KIKAKUI SYLLABLE M162 SEEMENDE KIKAKUI SYLLABLE M116 SEMENDE KIKAKUI SYLLABLE M136 SOOMENDE KIKAKUI SYLLABLE M079 SOMENDE KIKAKUI SYLLABLE M196 SIAMENDE KIKAKUI SYLLABLE M025\
    \ LIMENDE KIKAKUI SYLLABLE M026 LAMENDE KIKAKUI SYLLABLE M027 LUMENDE KIKAKUI SYLLABLE M084 LEEMENDE KIKAKUI SYLLABLE M073 LEMENDE KIKAKUI SYLLABLE M054 LOOMENDE KIKAKUI SYLLABLE M153 LOMENDE KIKAKUI SYLLABLE M110 LONG LEMENDE KIKAKUI SYLLABLE M016 DIMENDE\
    \ KIKAKUI SYLLABLE M017 DAMENDE KIKAKUI SYLLABLE M018 DUMENDE KIKAKUI SYLLABLE M089 DEEMENDE KIKAKUI SYLLABLE M180 DOOMENDE KIKAKUI SYLLABLE M181 DOMENDE KIKAKUI SYLLABLE M022 TIMENDE KIKAKUI SYLLABLE M023 TAMENDE KIKAKUI SYLLABLE M024 TUMENDE KIKAKUI SYLL\
    \ABLE M091 TEEMENDE KIKAKUI SYLLABLE M055 TEMENDE KIKAKUI SYLLABLE M104 TOOMENDE KIKAKUI SYLLABLE M069 TOMENDE KIKAKUI SYLLABLE M028 JIMENDE KIKAKUI SYLLABLE M029 JAMENDE KIKAKUI SYLLABLE M030 JUMENDE KIKAKUI SYLLABLE M157 JEEMENDE KIKAKUI SYLLABLE M113 JE\
    \MENDE KIKAKUI SYLLABLE M160 JOOMENDE KIKAKUI SYLLABLE M063 JOMENDE KIKAKUI SYLLABLE M175 LONG JOMENDE KIKAKUI SYLLABLE M031 YIMENDE KIKAKUI SYLLABLE M032 YAMENDE KIKAKUI SYLLABLE M033 YUMENDE KIKAKUI SYLLABLE M109 YEEMENDE KIKAKUI SYLLABLE M080 YEMENDE KI\
    \KAKUI SYLLABLE M141 YOOMENDE KIKAKUI SYLLABLE M121 YOMENDE KIKAKUI SYLLABLE M034 FIMENDE KIKAKUI SYLLABLE M035 FAMENDE KIKAKUI SYLLABLE M036 FUMENDE KIKAKUI SYLLABLE M078 FEEMENDE KIKAKUI SYLLABLE M075 FEMENDE KIKAKUI SYLLABLE M133 FOOMENDE KIKAKUI SYLLAB\
    \LE M088 FOMENDE KIKAKUI SYLLABLE M197 FUAMENDE KIKAKUI SYLLABLE M101 FANMENDE KIKAKUI SYLLABLE M037 NINMENDE KIKAKUI SYLLABLE M038 NANMENDE KIKAKUI SYLLABLE M039 NUNMENDE KIKAKUI SYLLABLE M117 NENMENDE KIKAKUI SYLLABLE M169 NONMENDE KIKAKUI SYLLABLE M176 \
    \HIMENDE KIKAKUI SYLLABLE M041 HAMENDE KIKAKUI SYLLABLE M186 HUMENDE KIKAKUI SYLLABLE M040 HEEMENDE KIKAKUI SYLLABLE M096 HEMENDE KIKAKUI SYLLABLE M042 HOOMENDE KIKAKUI SYLLABLE M140 HOMENDE KIKAKUI SYLLABLE M083 HEEIMENDE KIKAKUI SYLLABLE M128 HOOUMENDE K\
    \IKAKUI SYLLABLE M053 HINMENDE KIKAKUI SYLLABLE M130 HANMENDE KIKAKUI SYLLABLE M087 HUNMENDE KIKAKUI SYLLABLE M052 HENMENDE KIKAKUI SYLLABLE M193 HONMENDE KIKAKUI SYLLABLE M046 HUANMENDE KIKAKUI SYLLABLE M090 NGGIMENDE KIKAKUI SYLLABLE M043 NGGAMENDE KIKAK\
    \UI SYLLABLE M082 NGGUMENDE KIKAKUI SYLLABLE M115 NGGEEMENDE KIKAKUI SYLLABLE M146 NGGEMENDE KIKAKUI SYLLABLE M156 NGGOOMENDE KIKAKUI SYLLABLE M120 NGGOMENDE KIKAKUI SYLLABLE M159 NGGAAMENDE KIKAKUI SYLLABLE M127 NGGUAMENDE KIKAKUI SYLLABLE M086 LONG NGGEM\
    \ENDE KIKAKUI SYLLABLE M106 LONG NGGOOMENDE KIKAKUI SYLLABLE M183 LONG NGGOMENDE KIKAKUI SYLLABLE M155 GIMENDE KIKAKUI SYLLABLE M111 GAMENDE KIKAKUI SYLLABLE M168 GUMENDE KIKAKUI SYLLABLE M190 GEEMENDE KIKAKUI SYLLABLE M166 GUEIMENDE KIKAKUI SYLLABLE M167 \
    \GUANMENDE KIKAKUI SYLLABLE M184 NGENMENDE KIKAKUI SYLLABLE M057 NGONMENDE KIKAKUI SYLLABLE M177 NGUANMENDE KIKAKUI SYLLABLE M068 PIMENDE KIKAKUI SYLLABLE M099 PAMENDE KIKAKUI SYLLABLE M050 PUMENDE KIKAKUI SYLLABLE M081 PEEMENDE KIKAKUI SYLLABLE M051 PEMEN\
    \DE KIKAKUI SYLLABLE M102 POOMENDE KIKAKUI SYLLABLE M066 POMENDE KIKAKUI SYLLABLE M145 MBIMENDE KIKAKUI SYLLABLE M062 MBAMENDE KIKAKUI SYLLABLE M122 MBUMENDE KIKAKUI SYLLABLE M047 MBEEMENDE KIKAKUI SYLLABLE M188 MBEEMENDE KIKAKUI SYLLABLE M072 MBEMENDE KIK\
    \AKUI SYLLABLE M172 MBOOMENDE KIKAKUI SYLLABLE M174 MBOMENDE KIKAKUI SYLLABLE M187 MBUUMENDE KIKAKUI SYLLABLE M161 LONG MBEMENDE KIKAKUI SYLLABLE M105 LONG MBOOMENDE KIKAKUI SYLLABLE M142 LONG MBOMENDE KIKAKUI SYLLABLE M132 KPIMENDE KIKAKUI SYLLABLE M092 K\
    \PAMENDE KIKAKUI SYLLABLE M074 KPUMENDE KIKAKUI SYLLABLE M044 KPEEMENDE KIKAKUI SYLLABLE M108 KPEMENDE KIKAKUI SYLLABLE M112 KPOOMENDE KIKAKUI SYLLABLE M158 KPOMENDE KIKAKUI SYLLABLE M124 GBIMENDE KIKAKUI SYLLABLE M056 GBAMENDE KIKAKUI SYLLABLE M148 GBUMEN\
    \DE KIKAKUI SYLLABLE M093 GBEEMENDE KIKAKUI SYLLABLE M107 GBEMENDE KIKAKUI SYLLABLE M071 GBOOMENDE KIKAKUI SYLLABLE M070 GBOMENDE KIKAKUI SYLLABLE M171 RAMENDE KIKAKUI SYLLABLE M123 NDIMENDE KIKAKUI SYLLABLE M129 NDAMENDE KIKAKUI SYLLABLE M125 NDUMENDE KIK\
    \AKUI SYLLABLE M191 NDEEMENDE KIKAKUI SYLLABLE M119 NDEMENDE KIKAKUI SYLLABLE M067 NDOOMENDE KIKAKUI SYLLABLE M064 NDOMENDE KIKAKUI SYLLABLE M152 NJAMENDE KIKAKUI SYLLABLE M192 NJUMENDE KIKAKUI SYLLABLE M149 NJEEMENDE KIKAKUI SYLLABLE M134 NJOOMENDE KIKAKU\
    \I SYLLABLE M182 VIMENDE KIKAKUI SYLLABLE M185 VAMENDE KIKAKUI SYLLABLE M151 VUMENDE KIKAKUI SYLLABLE M173 VEEMENDE KIKAKUI SYLLABLE M085 VEMENDE KIKAKUI SYLLABLE M144 VOOMENDE KIKAKUI SYLLABLE M077 VOMENDE KIKAKUI SYLLABLE M164 NYINMENDE KIKAKUI SYLLABLE \
    \M058 NYANMENDE KIKAKUI SYLLABLE M170 NYUNMENDE KIKAKUI SYLLABLE M098 NYENMENDE KIKAKUI SYLLABLE M060 NYONMENDE KIKAKUI DIGIT ONEMENDE KIKAKUI DIGIT TWOMENDE KIKAKUI DIGIT THREEMENDE KIKAKUI DIGIT FOURMENDE KIKAKUI DIGIT FIVEMENDE KIKAKUI DIGIT SIXMENDE KI\
    \KAKUI DIGIT SEVENMENDE KIKAKUI DIGIT EIGHTMENDE KIKAKUI DIGIT NINEMENDE KIKAKUI COMBINING NUMBER TEENSMENDE KIKAKUI COMBINING NUMBER TENSMENDE KIKAKUI COMBINING NUMBER HUNDREDSMENDE KIKAKUI COMBINING NUMBER THOUSANDSMENDE KIKAKUI COMBINING NUMBER TEN THOU\
    \SANDSMENDE KIKAKUI COMBINING NUMBER HUNDRED THOUSANDSMENDE KIKAKUI COMBINING NUMBER MILLIONSADLAM CAPITAL LETTER ALIFADLAM CAPITAL LETTER DAALIADLAM CAPITAL LETTER LAAMADLAM CAPITAL LETTER MIIMADLAM CAPITAL LETTER BAADLAM CAPITAL LETTER SINNYIIYHEADLAM CA\
    \PITAL LETTER PEADLAM CAPITAL LETTER BHEADLAM CAPITAL LETTER RAADLAM CAPITAL LETTER EADLAM CAPITAL LETTER FAADLAM CAPITAL LETTER IADLAM CAPITAL LETTER OADLAM CAPITAL LETTER DHAADLAM CAPITAL LETTER YHEADLAM CAPITAL LETTER WAWADLAM CAPITAL LETTER NUNADLAM CA\
    \PITAL LETTER KAFADLAM CAPITAL LETTER YAADLAM CAPITAL LETTER UADLAM CAPITAL LETTER JIIMADLAM CAPITAL LETTER CHIADLAM CAPITAL LETTER HAADLAM CAPITAL LETTER QAAFADLAM CAPITAL LETTER GAADLAM CAPITAL LETTER NYAADLAM CAPITAL LETTER TUADLAM CAPITAL LETTER NHAADL\
    \AM CAPITAL LETTER VAADLAM CAPITAL LETTER KHAADLAM CAPITAL LETTER GBEADLAM CAPITAL LETTER ZALADLAM CAPITAL LETTER KPOADLAM CAPITAL LETTER SHAADLAM SMALL LETTER ALIFADLAM SMALL LETTER DAALIADLAM SMALL LETTER LAAMADLAM SMALL LETTER MIIMADLAM SMALL LETTER BAA\
    \DLAM SMALL LETTER SINNYIIYHEADLAM SMALL LETTER PEADLAM SMALL LETTER BHEADLAM SMALL LETTER RAADLAM SMALL LETTER EADLAM SMALL LETTER FAADLAM SMALL LETTER IADLAM SMALL LETTER OADLAM SMALL LETTER DHAADLAM SMALL LETTER YHEADLAM SMALL LETTER WAWADLAM SMALL LETT\
    \ER NUNADLAM SMALL LETTER KAFADLAM SMALL LETTER YAADLAM SMALL LETTER UADLAM SMALL LETTER JIIMADLAM SMALL LETTER CHIADLAM SMALL LETTER HAADLAM SMALL LETTER QAAFADLAM SMALL LETTER GAADLAM SMALL LETTER NYAADLAM SMALL LETTER TUADLAM SMALL LETTER NHAADLAM SMALL\
    \ LETTER VAADLAM SMALL LETTER KHAADLAM SMALL LETTER GBEADLAM SMALL LETTER ZALADLAM SMALL LETTER KPOADLAM SMALL LETTER SHAADLAM ALIF LENGTHENERADLAM VOWEL LENGTHENERADLAM GEMINATION MARKADLAM HAMZAADLAM CONSONANT MODIFIERADLAM GEMINATE CONSONANT MODIFIERADL\
    \AM NUKTAADLAM NASALIZATION MARKADLAM DIGIT ZEROADLAM DIGIT ONEADLAM DIGIT TWOADLAM DIGIT THREEADLAM DIGIT FOURADLAM DIGIT FIVEADLAM DIGIT SIXADLAM DIGIT SEVENADLAM DIGIT EIGHTADLAM DIGIT NINEADLAM INITIAL EXCLAMATION MARKADLAM INITIAL QUESTION MARKINDIC S\
    \IYAQ NUMBER ONEINDIC SIYAQ NUMBER TWOINDIC SIYAQ NUMBER THREEINDIC SIYAQ NUMBER FOURINDIC SIYAQ NUMBER FIVEINDIC SIYAQ NUMBER SIXINDIC SIYAQ NUMBER SEVENINDIC SIYAQ NUMBER EIGHTINDIC SIYAQ NUMBER NINEINDIC SIYAQ NUMBER TENINDIC SIYAQ NUMBER TWENTYINDIC SI\
    \YAQ NUMBER THIRTYINDIC SIYAQ NUMBER FORTYINDIC SIYAQ NUMBER FIFTYINDIC SIYAQ NUMBER SIXTYINDIC SIYAQ NUMBER SEVENTYINDIC SIYAQ NUMBER EIGHTYINDIC SIYAQ NUMBER NINETYINDIC SIYAQ NUMBER ONE HUNDREDINDIC SIYAQ NUMBER TWO HUNDREDINDIC SIYAQ NUMBER THREE HUNDR\
    \EDINDIC SIYAQ NUMBER FOUR HUNDREDINDIC SIYAQ NUMBER FIVE HUNDREDINDIC SIYAQ NUMBER SIX HUNDREDINDIC SIYAQ NUMBER SEVEN HUNDREDINDIC SIYAQ NUMBER EIGHT HUNDREDINDIC SIYAQ NUMBER NINE HUNDREDINDIC SIYAQ NUMBER ONE THOUSANDINDIC SIYAQ NUMBER TWO THOUSANDINDI\
    \C SIYAQ NUMBER THREE THOUSANDINDIC SIYAQ NUMBER FOUR THOUSANDINDIC SIYAQ NUMBER FIVE THOUSANDINDIC SIYAQ NUMBER SIX THOUSANDINDIC SIYAQ NUMBER SEVEN THOUSANDINDIC SIYAQ NUMBER EIGHT THOUSANDINDIC SIYAQ NUMBER NINE THOUSANDINDIC SIYAQ NUMBER TEN THOUSANDIN\
    \DIC SIYAQ NUMBER TWENTY THOUSANDINDIC SIYAQ NUMBER THIRTY THOUSANDINDIC SIYAQ NUMBER FORTY THOUSANDINDIC SIYAQ NUMBER FIFTY THOUSANDINDIC SIYAQ NUMBER SIXTY THOUSANDINDIC SIYAQ NUMBER SEVENTY THOUSANDINDIC SIYAQ NUMBER EIGHTY THOUSANDINDIC SIYAQ NUMBER NI\
    \NETY THOUSANDINDIC SIYAQ NUMBER LAKHINDIC SIYAQ NUMBER LAKHANINDIC SIYAQ LAKH MARKINDIC SIYAQ NUMBER KARORINDIC SIYAQ NUMBER KARORANINDIC SIYAQ NUMBER PREFIXED ONEINDIC SIYAQ NUMBER PREFIXED TWOINDIC SIYAQ NUMBER PREFIXED THREEINDIC SIYAQ NUMBER PREFIXED \
    \FOURINDIC SIYAQ NUMBER PREFIXED FIVEINDIC SIYAQ NUMBER PREFIXED SIXINDIC SIYAQ NUMBER PREFIXED SEVENINDIC SIYAQ NUMBER PREFIXED EIGHTINDIC SIYAQ NUMBER PREFIXED NINEINDIC SIYAQ PLACEHOLDERINDIC SIYAQ FRACTION ONE QUARTERINDIC SIYAQ FRACTION ONE HALFINDIC \
    \SIYAQ FRACTION THREE QUARTERSINDIC SIYAQ RUPEE MARKINDIC SIYAQ NUMBER ALTERNATE ONEINDIC SIYAQ NUMBER ALTERNATE TWOINDIC SIYAQ NUMBER ALTERNATE TEN THOUSANDINDIC SIYAQ ALTERNATE LAKH MARKOTTOMAN SIYAQ NUMBER ONEOTTOMAN SIYAQ NUMBER TWOOTTOMAN SIYAQ NUMBER\
    \ THREEOTTOMAN SIYAQ NUMBER FOUROTTOMAN SIYAQ NUMBER FIVEOTTOMAN SIYAQ NUMBER SIXOTTOMAN SIYAQ NUMBER SEVENOTTOMAN SIYAQ NUMBER EIGHTOTTOMAN SIYAQ NUMBER NINEOTTOMAN SIYAQ NUMBER TENOTTOMAN SIYAQ NUMBER TWENTYOTTOMAN SIYAQ NUMBER THIRTYOTTOMAN SIYAQ NUMBER\
    \ FORTYOTTOMAN SIYAQ NUMBER FIFTYOTTOMAN SIYAQ NUMBER SIXTYOTTOMAN SIYAQ NUMBER SEVENTYOTTOMAN SIYAQ NUMBER EIGHTYOTTOMAN SIYAQ NUMBER NINETYOTTOMAN SIYAQ NUMBER ONE HUNDREDOTTOMAN SIYAQ NUMBER TWO HUNDREDOTTOMAN SIYAQ NUMBER THREE HUNDREDOTTOMAN SIYAQ NUM\
    \BER FOUR HUNDREDOTTOMAN SIYAQ NUMBER FIVE HUNDREDOTTOMAN SIYAQ NUMBER SIX HUNDREDOTTOMAN SIYAQ NUMBER SEVEN HUNDREDOTTOMAN SIYAQ NUMBER EIGHT HUNDREDOTTOMAN SIYAQ NUMBER NINE HUNDREDOTTOMAN SIYAQ NUMBER ONE THOUSANDOTTOMAN SIYAQ NUMBER TWO THOUSANDOTTOMAN\
    \ SIYAQ NUMBER THREE THOUSANDOTTOMAN SIYAQ NUMBER FOUR THOUSANDOTTOMAN SIYAQ NUMBER FIVE THOUSANDOTTOMAN SIYAQ NUMBER SIX THOUSANDOTTOMAN SIYAQ NUMBER SEVEN THOUSANDOTTOMAN SIYAQ NUMBER EIGHT THOUSANDOTTOMAN SIYAQ NUMBER NINE THOUSANDOTTOMAN SIYAQ NUMBER T\
    \EN THOUSANDOTTOMAN SIYAQ NUMBER TWENTY THOUSANDOTTOMAN SIYAQ NUMBER THIRTY THOUSANDOTTOMAN SIYAQ NUMBER FORTY THOUSANDOTTOMAN SIYAQ NUMBER FIFTY THOUSANDOTTOMAN SIYAQ NUMBER SIXTY THOUSANDOTTOMAN SIYAQ NUMBER SEVENTY THOUSANDOTTOMAN SIYAQ NUMBER EIGHTY TH\
    \OUSANDOTTOMAN SIYAQ NUMBER NINETY THOUSANDOTTOMAN SIYAQ MARRATANOTTOMAN SIYAQ ALTERNATE NUMBER TWOOTTOMAN SIYAQ ALTERNATE NUMBER THREEOTTOMAN SIYAQ ALTERNATE NUMBER FOUROTTOMAN SIYAQ ALTERNATE NUMBER FIVEOTTOMAN SIYAQ ALTERNATE NUMBER SIXOTTOMAN SIYAQ ALT\
    \ERNATE NUMBER SEVENOTTOMAN SIYAQ ALTERNATE NUMBER EIGHTOTTOMAN SIYAQ ALTERNATE NUMBER NINEOTTOMAN SIYAQ ALTERNATE NUMBER TENOTTOMAN SIYAQ ALTERNATE NUMBER FOUR HUNDREDOTTOMAN SIYAQ ALTERNATE NUMBER SIX HUNDREDOTTOMAN SIYAQ ALTERNATE NUMBER TWO THOUSANDOTT\
    \OMAN SIYAQ ALTERNATE NUMBER TEN THOUSANDOTTOMAN SIYAQ FRACTION ONE HALFOTTOMAN SIYAQ FRACTION ONE SIXTHARABIC MATHEMATICAL ALEFARABIC MATHEMATICAL BEHARABIC MATHEMATICAL JEEMARABIC MATHEMATICAL DALARABIC MATHEMATICAL WAWARABIC MATHEMATICAL ZAINARABIC MATH\
    \EMATICAL HAHARABIC MATHEMATICAL TAHARABIC MATHEMATICAL YEHARABIC MATHEMATICAL KAFARABIC MATHEMATICAL LAMARABIC MATHEMATICAL MEEMARABIC MATHEMATICAL NOONARABIC MATHEMATICAL SEENARABIC MATHEMATICAL AINARABIC MATHEMATICAL FEHARABIC MATHEMATICAL SADARABIC MAT\
    \HEMATICAL QAFARABIC MATHEMATICAL REHARABIC MATHEMATICAL SHEENARABIC MATHEMATICAL TEHARABIC MATHEMATICAL THEHARABIC MATHEMATICAL KHAHARABIC MATHEMATICAL THALARABIC MATHEMATICAL DADARABIC MATHEMATICAL ZAHARABIC MATHEMATICAL GHAINARABIC MATHEMATICAL DOTLESS \
    \BEHARABIC MATHEMATICAL DOTLESS NOONARABIC MATHEMATICAL DOTLESS FEHARABIC MATHEMATICAL DOTLESS QAFARABIC MATHEMATICAL INITIAL BEHARABIC MATHEMATICAL INITIAL JEEMARABIC MATHEMATICAL INITIAL HEHARABIC MATHEMATICAL INITIAL HAHARABIC MATHEMATICAL INITIAL YEHAR\
    \ABIC MATHEMATICAL INITIAL KAFARABIC MATHEMATICAL INITIAL LAMARABIC MATHEMATICAL INITIAL MEEMARABIC MATHEMATICAL INITIAL NOONARABIC MATHEMATICAL INITIAL SEENARABIC MATHEMATICAL INITIAL AINARABIC MATHEMATICAL INITIAL FEHARABIC MATHEMATICAL INITIAL SADARABIC\
    \ MATHEMATICAL INITIAL QAFARABIC MATHEMATICAL INITIAL SHEENARABIC MATHEMATICAL INITIAL TEHARABIC MATHEMATICAL INITIAL THEHARABIC MATHEMATICAL INITIAL KHAHARABIC MATHEMATICAL INITIAL DADARABIC MATHEMATICAL INITIAL GHAINARABIC MATHEMATICAL TAILED JEEMARABIC \
    \MATHEMATICAL TAILED HAHARABIC MATHEMATICAL TAILED YEHARABIC MATHEMATICAL TAILED LAMARABIC MATHEMATICAL TAILED NOONARABIC MATHEMATICAL TAILED SEENARABIC MATHEMATICAL TAILED AINARABIC MATHEMATICAL TAILED SADARABIC MATHEMATICAL TAILED QAFARABIC MATHEMATICAL \
    \TAILED SHEENARABIC MATHEMATICAL TAILED KHAHARABIC MATHEMATICAL TAILED DADARABIC MATHEMATICAL TAILED GHAINARABIC MATHEMATICAL TAILED DOTLESS NOONARABIC MATHEMATICAL TAILED DOTLESS QAFARABIC MATHEMATICAL STRETCHED BEHARABIC MATHEMATICAL STRETCHED JEEMARABIC\
    \ MATHEMATICAL STRETCHED HEHARABIC MATHEMATICAL STRETCHED HAHARABIC MATHEMATICAL STRETCHED TAHARABIC MATHEMATICAL STRETCHED YEHARABIC MATHEMATICAL STRETCHED KAFARABIC MATHEMATICAL STRETCHED MEEMARABIC MATHEMATICAL STRETCHED NOONARABIC MATHEMATICAL STRETCHE\
    \D SEENARABIC MATHEMATICAL STRETCHED AINARABIC MATHEMATICAL STRETCHED FEHARABIC MATHEMATICAL STRETCHED SADARABIC MATHEMATICAL STRETCHED QAFARABIC MATHEMATICAL STRETCHED SHEENARABIC MATHEMATICAL STRETCHED TEHARABIC MATHEMATICAL STRETCHED THEHARABIC MATHEMAT\
    \ICAL STRETCHED KHAHARABIC MATHEMATICAL STRETCHED DADARABIC MATHEMATICAL STRETCHED ZAHARABIC MATHEMATICAL STRETCHED GHAINARABIC MATHEMATICAL STRETCHED DOTLESS BEHARABIC MATHEMATICAL STRETCHED DOTLESS FEHARABIC MATHEMATICAL LOOPED ALEFARABIC MATHEMATICAL LO\
    \OPED BEHARABIC MATHEMATICAL LOOPED JEEMARABIC MATHEMATICAL LOOPED DALARABIC MATHEMATICAL LOOPED HEHARABIC MATHEMATICAL LOOPED WAWARABIC MATHEMATICAL LOOPED ZAINARABIC MATHEMATICAL LOOPED HAHARABIC MATHEMATICAL LOOPED TAHARABIC MATHEMATICAL LOOPED YEHARABI\
    \C MATHEMATICAL LOOPED LAMARABIC MATHEMATICAL LOOPED MEEMARABIC MATHEMATICAL LOOPED NOONARABIC MATHEMATICAL LOOPED SEENARABIC MATHEMATICAL LOOPED AINARABIC MATHEMATICAL LOOPED FEHARABIC MATHEMATICAL LOOPED SADARABIC MATHEMATICAL LOOPED QAFARABIC MATHEMATIC\
    \AL LOOPED REHARABIC MATHEMATICAL LOOPED SHEENARABIC MATHEMATICAL LOOPED TEHARABIC MATHEMATICAL LOOPED THEHARABIC MATHEMATICAL LOOPED KHAHARABIC MATHEMATICAL LOOPED THALARABIC MATHEMATICAL LOOPED DADARABIC MATHEMATICAL LOOPED ZAHARABIC MATHEMATICAL LOOPED \
    \GHAINARABIC MATHEMATICAL DOUBLE-STRUCK BEHARABIC MATHEMATICAL DOUBLE-STRUCK JEEMARABIC MATHEMATICAL DOUBLE-STRUCK DALARABIC MATHEMATICAL DOUBLE-STRUCK WAWARABIC MATHEMATICAL DOUBLE-STRUCK ZAINARABIC MATHEMATICAL DOUBLE-STRUCK HAHARABIC MATHEMATICAL DOUBLE\
    \-STRUCK TAHARABIC MATHEMATICAL DOUBLE-STRUCK YEHARABIC MATHEMATICAL DOUBLE-STRUCK LAMARABIC MATHEMATICAL DOUBLE-STRUCK MEEMARABIC MATHEMATICAL DOUBLE-STRUCK NOONARABIC MATHEMATICAL DOUBLE-STRUCK SEENARABIC MATHEMATICAL DOUBLE-STRUCK AINARABIC MATHEMATICAL\
    \ DOUBLE-STRUCK FEHARABIC MATHEMATICAL DOUBLE-STRUCK SADARABIC MATHEMATICAL DOUBLE-STRUCK QAFARABIC MATHEMATICAL DOUBLE-STRUCK REHARABIC MATHEMATICAL DOUBLE-STRUCK SHEENARABIC MATHEMATICAL DOUBLE-STRUCK TEHARABIC MATHEMATICAL DOUBLE-STRUCK THEHARABIC MATHE\
    \MATICAL DOUBLE-STRUCK KHAHARABIC MATHEMATICAL DOUBLE-STRUCK THALARABIC MATHEMATICAL DOUBLE-STRUCK DADARABIC MATHEMATICAL DOUBLE-STRUCK ZAHARABIC MATHEMATICAL DOUBLE-STRUCK GHAINARABIC MATHEMATICAL OPERATOR MEEM WITH HAH WITH TATWEELARABIC MATHEMATICAL OPE\
    \RATOR HAH WITH DALMAHJONG TILE EAST WINDMAHJONG TILE SOUTH WINDMAHJONG TILE WEST WINDMAHJONG TILE NORTH WINDMAHJONG TILE RED DRAGONMAHJONG TILE GREEN DRAGONMAHJONG TILE WHITE DRAGONMAHJONG TILE ONE OF CHARACTERSMAHJONG TILE TWO OF CHARACTERSMAHJONG TILE T\
    \HREE OF CHARACTERSMAHJONG TILE FOUR OF CHARACTERSMAHJONG TILE FIVE OF CHARACTERSMAHJONG TILE SIX OF CHARACTERSMAHJONG TILE SEVEN OF CHARACTERSMAHJONG TILE EIGHT OF CHARACTERSMAHJONG TILE NINE OF CHARACTERSMAHJONG TILE ONE OF BAMBOOSMAHJONG TILE TWO OF BAM\
    \BOOSMAHJONG TILE THREE OF BAMBOOSMAHJONG TILE FOUR OF BAMBOOSMAHJONG TILE FIVE OF BAMBOOSMAHJONG TILE SIX OF BAMBOOSMAHJONG TILE SEVEN OF BAMBOOSMAHJONG TILE EIGHT OF BAMBOOSMAHJONG TILE NINE OF BAMBOOSMAHJONG TILE ONE OF CIRCLESMAHJONG TILE TWO OF CIRCLE\
    \SMAHJONG TILE THREE OF CIRCLESMAHJONG TILE FOUR OF CIRCLESMAHJONG TILE FIVE OF CIRCLESMAHJONG TILE SIX OF CIRCLESMAHJONG TILE SEVEN OF CIRCLESMAHJONG TILE EIGHT OF CIRCLESMAHJONG TILE NINE OF CIRCLESMAHJONG TILE PLUMMAHJONG TILE ORCHIDMAHJONG TILE BAMBOOM\
    \AHJONG TILE CHRYSANTHEMUMMAHJONG TILE SPRINGMAHJONG TILE SUMMERMAHJONG TILE AUTUMNMAHJONG TILE WINTERMAHJONG TILE JOKERMAHJONG TILE BACKDOMINO TILE HORIZONTAL BACKDOMINO TILE HORIZONTAL-00-00DOMINO TILE HORIZONTAL-00-01DOMINO TILE HORIZONTAL-00-02DOMINO T\
    \ILE HORIZONTAL-00-03DOMINO TILE HORIZONTAL-00-04DOMINO TILE HORIZONTAL-00-05DOMINO TILE HORIZONTAL-00-06DOMINO TILE HORIZONTAL-01-00DOMINO TILE HORIZONTAL-01-01DOMINO TILE HORIZONTAL-01-02DOMINO TILE HORIZONTAL-01-03DOMINO TILE HORIZONTAL-01-04DOMINO TILE\
    \ HORIZONTAL-01-05DOMINO TILE HORIZONTAL-01-06DOMINO TILE HORIZONTAL-02-00DOMINO TILE HORIZONTAL-02-01DOMINO TILE HORIZONTAL-02-02DOMINO TILE HORIZONTAL-02-03DOMINO TILE HORIZONTAL-02-04DOMINO TILE HORIZONTAL-02-05DOMINO TILE HORIZONTAL-02-06DOMINO TILE HO\
    \RIZONTAL-03-00DOMINO TILE HORIZONTAL-03-01DOMINO TILE HORIZONTAL-03-02DOMINO TILE HORIZONTAL-03-03DOMINO TILE HORIZONTAL-03-04DOMINO TILE HORIZONTAL-03-05DOMINO TILE HORIZONTAL-03-06DOMINO TILE HORIZONTAL-04-00DOMINO TILE HORIZONTAL-04-01DOMINO TILE HORIZ\
    \ONTAL-04-02DOMINO TILE HORIZONTAL-04-03DOMINO TILE HORIZONTAL-04-04DOMINO TILE HORIZONTAL-04-05DOMINO TILE HORIZONTAL-04-06DOMINO TILE HORIZONTAL-05-00DOMINO TILE HORIZONTAL-05-01DOMINO TILE HORIZONTAL-05-02DOMINO TILE HORIZONTAL-05-03DOMINO TILE HORIZONT\
    \AL-05-04DOMINO TILE HORIZONTAL-05-05DOMINO TILE HORIZONTAL-05-06DOMINO TILE HORIZONTAL-06-00DOMINO TILE HORIZONTAL-06-01DOMINO TILE HORIZONTAL-06-02DOMINO TILE HORIZONTAL-06-03DOMINO TILE HORIZONTAL-06-04DOMINO TILE HORIZONTAL-06-05DOMINO TILE HORIZONTAL-\
    \06-06DOMINO TILE VERTICAL BACKDOMINO TILE VERTICAL-00-00DOMINO TILE VERTICAL-00-01DOMINO TILE VERTICAL-00-02DOMINO TILE VERTICAL-00-03DOMINO TILE VERTICAL-00-04DOMINO TILE VERTICAL-00-05DOMINO TILE VERTICAL-00-06DOMINO TILE VERTICAL-01-00DOMINO TILE VERTI\
    \CAL-01-01DOMINO TILE VERTICAL-01-02DOMINO TILE VERTICAL-01-03DOMINO TILE VERTICAL-01-04DOMINO TILE VERTICAL-01-05DOMINO TILE VERTICAL-01-06DOMINO TILE VERTICAL-02-00DOMINO TILE VERTICAL-02-01DOMINO TILE VERTICAL-02-02DOMINO TILE VERTICAL-02-03DOMINO TILE \
    \VERTICAL-02-04DOMINO TILE VERTICAL-02-05DOMINO TILE VERTICAL-02-06DOMINO TILE VERTICAL-03-00DOMINO TILE VERTICAL-03-01DOMINO TILE VERTICAL-03-02DOMINO TILE VERTICAL-03-03DOMINO TILE VERTICAL-03-04DOMINO TILE VERTICAL-03-05DOMINO TILE VERTICAL-03-06DOMINO \
    \TILE VERTICAL-04-00DOMINO TILE VERTICAL-04-01DOMINO TILE VERTICAL-04-02DOMINO TILE VERTICAL-04-03DOMINO TILE VERTICAL-04-04DOMINO TILE VERTICAL-04-05DOMINO TILE VERTICAL-04-06DOMINO TILE VERTICAL-05-00DOMINO TILE VERTICAL-05-01DOMINO TILE VERTICAL-05-02DO\
    \MINO TILE VERTICAL-05-03DOMINO TILE VERTICAL-05-04DOMINO TILE VERTICAL-05-05DOMINO TILE VERTICAL-05-06DOMINO TILE VERTICAL-06-00DOMINO TILE VERTICAL-06-01DOMINO TILE VERTICAL-06-02DOMINO TILE VERTICAL-06-03DOMINO TILE VERTICAL-06-04DOMINO TILE VERTICAL-06\
    \-05DOMINO TILE VERTICAL-06-06PLAYING CARD BACKPLAYING CARD ACE OF SPADESPLAYING CARD TWO OF SPADESPLAYING CARD THREE OF SPADESPLAYING CARD FOUR OF SPADESPLAYING CARD FIVE OF SPADESPLAYING CARD SIX OF SPADESPLAYING CARD SEVEN OF SPADESPLAYING CARD EIGHT OF\
    \ SPADESPLAYING CARD NINE OF SPADESPLAYING CARD TEN OF SPADESPLAYING CARD JACK OF SPADESPLAYING CARD KNIGHT OF SPADESPLAYING CARD QUEEN OF SPADESPLAYING CARD KING OF SPADESPLAYING CARD ACE OF HEARTSPLAYING CARD TWO OF HEARTSPLAYING CARD THREE OF HEARTSPLAY\
    \ING CARD FOUR OF HEARTSPLAYING CARD FIVE OF HEARTSPLAYING CARD SIX OF HEARTSPLAYING CARD SEVEN OF HEARTSPLAYING CARD EIGHT OF HEARTSPLAYING CARD NINE OF HEARTSPLAYING CARD TEN OF HEARTSPLAYING CARD JACK OF HEARTSPLAYING CARD KNIGHT OF HEARTSPLAYING CARD Q\
    \UEEN OF HEARTSPLAYING CARD KING OF HEARTSPLAYING CARD RED JOKERPLAYING CARD ACE OF DIAMONDSPLAYING CARD TWO OF DIAMONDSPLAYING CARD THREE OF DIAMONDSPLAYING CARD FOUR OF DIAMONDSPLAYING CARD FIVE OF DIAMONDSPLAYING CARD SIX OF DIAMONDSPLAYING CARD SEVEN O\
    \F DIAMONDSPLAYING CARD EIGHT OF DIAMONDSPLAYING CARD NINE OF DIAMONDSPLAYING CARD TEN OF DIAMONDSPLAYING CARD JACK OF DIAMONDSPLAYING CARD KNIGHT OF DIAMONDSPLAYING CARD QUEEN OF DIAMONDSPLAYING CARD KING OF DIAMONDSPLAYING CARD BLACK JOKERPLAYING CARD AC\
    \E OF CLUBSPLAYING CARD TWO OF CLUBSPLAYING CARD THREE OF CLUBSPLAYING CARD FOUR OF CLUBSPLAYING CARD FIVE OF CLUBSPLAYING CARD SIX OF CLUBSPLAYING CARD SEVEN OF CLUBSPLAYING CARD EIGHT OF CLUBSPLAYING CARD NINE OF CLUBSPLAYING CARD TEN OF CLUBSPLAYING CAR\
    \D JACK OF CLUBSPLAYING CARD KNIGHT OF CLUBSPLAYING CARD QUEEN OF CLUBSPLAYING CARD KING OF CLUBSPLAYING CARD WHITE JOKERPLAYING CARD FOOLPLAYING CARD TRUMP-1PLAYING CARD TRUMP-2PLAYING CARD TRUMP-3PLAYING CARD TRUMP-4PLAYING CARD TRUMP-5PLAYING CARD TRUMP\
    \-6PLAYING CARD TRUMP-7PLAYING CARD TRUMP-8PLAYING CARD TRUMP-9PLAYING CARD TRUMP-10PLAYING CARD TRUMP-11PLAYING CARD TRUMP-12PLAYING CARD TRUMP-13PLAYING CARD TRUMP-14PLAYING CARD TRUMP-15PLAYING CARD TRUMP-16PLAYING CARD TRUMP-17PLAYING CARD TRUMP-18PLAY\
    \ING CARD TRUMP-19PLAYING CARD TRUMP-20PLAYING CARD TRUMP-21DIGIT ZERO FULL STOPDIGIT ZERO COMMADIGIT ONE COMMADIGIT TWO COMMADIGIT THREE COMMADIGIT FOUR COMMADIGIT FIVE COMMADIGIT SIX COMMADIGIT SEVEN COMMADIGIT EIGHT COMMADIGIT NINE COMMADINGBAT CIRCLED \
    \SANS-SERIF DIGIT ZERODINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT ZEROCIRCLED ZERO WITH SLASHCIRCLED ANTICLOCKWISE ARROWCIRCLED DOLLAR SIGN WITH OVERLAID BACKSLASHPARENTHESIZED LATIN CAPITAL LETTER APARENTHESIZED LATIN CAPITAL LETTER BPARENTHESIZED LATIN CAP\
    \ITAL LETTER CPARENTHESIZED LATIN CAPITAL LETTER DPARENTHESIZED LATIN CAPITAL LETTER EPARENTHESIZED LATIN CAPITAL LETTER FPARENTHESIZED LATIN CAPITAL LETTER GPARENTHESIZED LATIN CAPITAL LETTER HPARENTHESIZED LATIN CAPITAL LETTER IPARENTHESIZED LATIN CAPITA\
    \L LETTER JPARENTHESIZED LATIN CAPITAL LETTER KPARENTHESIZED LATIN CAPITAL LETTER LPARENTHESIZED LATIN CAPITAL LETTER MPARENTHESIZED LATIN CAPITAL LETTER NPARENTHESIZED LATIN CAPITAL LETTER OPARENTHESIZED LATIN CAPITAL LETTER PPARENTHESIZED LATIN CAPITAL L\
    \ETTER QPARENTHESIZED LATIN CAPITAL LETTER RPARENTHESIZED LATIN CAPITAL LETTER SPARENTHESIZED LATIN CAPITAL LETTER TPARENTHESIZED LATIN CAPITAL LETTER UPARENTHESIZED LATIN CAPITAL LETTER VPARENTHESIZED LATIN CAPITAL LETTER WPARENTHESIZED LATIN CAPITAL LETT\
    \ER XPARENTHESIZED LATIN CAPITAL LETTER YPARENTHESIZED LATIN CAPITAL LETTER ZTORTOISE SHELL BRACKETED LATIN CAPITAL LETTER SCIRCLED ITALIC LATIN CAPITAL LETTER CCIRCLED ITALIC LATIN CAPITAL LETTER RCIRCLED CDCIRCLED WZCOPYLEFT SYMBOLSQUARED LATIN CAPITAL L\
    \ETTER ASQUARED LATIN CAPITAL LETTER BSQUARED LATIN CAPITAL LETTER CSQUARED LATIN CAPITAL LETTER DSQUARED LATIN CAPITAL LETTER ESQUARED LATIN CAPITAL LETTER FSQUARED LATIN CAPITAL LETTER GSQUARED LATIN CAPITAL LETTER HSQUARED LATIN CAPITAL LETTER ISQUARED \
    \LATIN CAPITAL LETTER JSQUARED LATIN CAPITAL LETTER KSQUARED LATIN CAPITAL LETTER LSQUARED LATIN CAPITAL LETTER MSQUARED LATIN CAPITAL LETTER NSQUARED LATIN CAPITAL LETTER OSQUARED LATIN CAPITAL LETTER PSQUARED LATIN CAPITAL LETTER QSQUARED LATIN CAPITAL L\
    \ETTER RSQUARED LATIN CAPITAL LETTER SSQUARED LATIN CAPITAL LETTER TSQUARED LATIN CAPITAL LETTER USQUARED LATIN CAPITAL LETTER VSQUARED LATIN CAPITAL LETTER WSQUARED LATIN CAPITAL LETTER XSQUARED LATIN CAPITAL LETTER YSQUARED LATIN CAPITAL LETTER ZSQUARED \
    \HVSQUARED MVSQUARED SDSQUARED SSSQUARED PPVSQUARED WCNEGATIVE CIRCLED LATIN CAPITAL LETTER ANEGATIVE CIRCLED LATIN CAPITAL LETTER BNEGATIVE CIRCLED LATIN CAPITAL LETTER CNEGATIVE CIRCLED LATIN CAPITAL LETTER DNEGATIVE CIRCLED LATIN CAPITAL LETTER ENEGATIV\
    \E CIRCLED LATIN CAPITAL LETTER FNEGATIVE CIRCLED LATIN CAPITAL LETTER GNEGATIVE CIRCLED LATIN CAPITAL LETTER HNEGATIVE CIRCLED LATIN CAPITAL LETTER INEGATIVE CIRCLED LATIN CAPITAL LETTER JNEGATIVE CIRCLED LATIN CAPITAL LETTER KNEGATIVE CIRCLED LATIN CAPIT\
    \AL LETTER LNEGATIVE CIRCLED LATIN CAPITAL LETTER MNEGATIVE CIRCLED LATIN CAPITAL LETTER NNEGATIVE CIRCLED LATIN CAPITAL LETTER ONEGATIVE CIRCLED LATIN CAPITAL LETTER PNEGATIVE CIRCLED LATIN CAPITAL LETTER QNEGATIVE CIRCLED LATIN CAPITAL LETTER RNEGATIVE C\
    \IRCLED LATIN CAPITAL LETTER SNEGATIVE CIRCLED LATIN CAPITAL LETTER TNEGATIVE CIRCLED LATIN CAPITAL LETTER UNEGATIVE CIRCLED LATIN CAPITAL LETTER VNEGATIVE CIRCLED LATIN CAPITAL LETTER WNEGATIVE CIRCLED LATIN CAPITAL LETTER XNEGATIVE CIRCLED LATIN CAPITAL \
    \LETTER YNEGATIVE CIRCLED LATIN CAPITAL LETTER ZRAISED MC SIGNRAISED MD SIGNRAISED MR SIGNCIRCLED CCCIRCLED C WITH OVERLAID BACKSLASHCIRCLED HUMAN FIGURENEGATIVE SQUARED LATIN CAPITAL LETTER ANEGATIVE SQUARED LATIN CAPITAL LETTER BNEGATIVE SQUARED LATIN CA\
    \PITAL LETTER CNEGATIVE SQUARED LATIN CAPITAL LETTER DNEGATIVE SQUARED LATIN CAPITAL LETTER ENEGATIVE SQUARED LATIN CAPITAL LETTER FNEGATIVE SQUARED LATIN CAPITAL LETTER GNEGATIVE SQUARED LATIN CAPITAL LETTER HNEGATIVE SQUARED LATIN CAPITAL LETTER INEGATIV\
    \E SQUARED LATIN CAPITAL LETTER JNEGATIVE SQUARED LATIN CAPITAL LETTER KNEGATIVE SQUARED LATIN CAPITAL LETTER LNEGATIVE SQUARED LATIN CAPITAL LETTER MNEGATIVE SQUARED LATIN CAPITAL LETTER NNEGATIVE SQUARED LATIN CAPITAL LETTER ONEGATIVE SQUARED LATIN CAPIT\
    \AL LETTER PNEGATIVE SQUARED LATIN CAPITAL LETTER QNEGATIVE SQUARED LATIN CAPITAL LETTER RNEGATIVE SQUARED LATIN CAPITAL LETTER SNEGATIVE SQUARED LATIN CAPITAL LETTER TNEGATIVE SQUARED LATIN CAPITAL LETTER UNEGATIVE SQUARED LATIN CAPITAL LETTER VNEGATIVE S\
    \QUARED LATIN CAPITAL LETTER WNEGATIVE SQUARED LATIN CAPITAL LETTER XNEGATIVE SQUARED LATIN CAPITAL LETTER YNEGATIVE SQUARED LATIN CAPITAL LETTER ZCROSSED NEGATIVE SQUARED LATIN CAPITAL LETTER PNEGATIVE SQUARED ICNEGATIVE SQUARED PANEGATIVE SQUARED SANEGAT\
    \IVE SQUARED ABNEGATIVE SQUARED WCSQUARE DJSQUARED CLSQUARED COOLSQUARED FREESQUARED IDSQUARED NEWSQUARED NGSQUARED OKSQUARED SOSSQUARED UP WITH EXCLAMATION MARKSQUARED VSSQUARED THREE DSQUARED SECOND SCREENSQUARED TWO KSQUARED FOUR KSQUARED EIGHT KSQUARED\
    \ FIVE POINT ONESQUARED SEVEN POINT ONESQUARED TWENTY-TWO POINT TWOSQUARED SIXTY PSQUARED ONE HUNDRED TWENTY PSQUARED LATIN SMALL LETTER DSQUARED HCSQUARED HDRSQUARED HI-RESSQUARED LOSSLESSSQUARED SHVSQUARED UHDSQUARED VODMASK WORK SYMBOLREGIONAL INDICATOR\
    \ SYMBOL LETTER AREGIONAL INDICATOR SYMBOL LETTER BREGIONAL INDICATOR SYMBOL LETTER CREGIONAL INDICATOR SYMBOL LETTER DREGIONAL INDICATOR SYMBOL LETTER EREGIONAL INDICATOR SYMBOL LETTER FREGIONAL INDICATOR SYMBOL LETTER GREGIONAL INDICATOR SYMBOL LETTER HR\
    \EGIONAL INDICATOR SYMBOL LETTER IREGIONAL INDICATOR SYMBOL LETTER JREGIONAL INDICATOR SYMBOL LETTER KREGIONAL INDICATOR SYMBOL LETTER LREGIONAL INDICATOR SYMBOL LETTER MREGIONAL INDICATOR SYMBOL LETTER NREGIONAL INDICATOR SYMBOL LETTER OREGIONAL INDICATOR\
    \ SYMBOL LETTER PREGIONAL INDICATOR SYMBOL LETTER QREGIONAL INDICATOR SYMBOL LETTER RREGIONAL INDICATOR SYMBOL LETTER SREGIONAL INDICATOR SYMBOL LETTER TREGIONAL INDICATOR SYMBOL LETTER UREGIONAL INDICATOR SYMBOL LETTER VREGIONAL INDICATOR SYMBOL LETTER WR\
    \EGIONAL INDICATOR SYMBOL LETTER XREGIONAL INDICATOR SYMBOL LETTER YREGIONAL INDICATOR SYMBOL LETTER ZSQUARE HIRAGANA HOKASQUARED KATAKANA KOKOSQUARED KATAKANA SASQUARED CJK UNIFIED IDEOGRAPH-624BSQUARED CJK UNIFIED IDEOGRAPH-5B57SQUARED CJK UNIFIED IDEOGR\
    \APH-53CCSQUARED KATAKANA DESQUARED CJK UNIFIED IDEOGRAPH-4E8CSQUARED CJK UNIFIED IDEOGRAPH-591ASQUARED CJK UNIFIED IDEOGRAPH-89E3SQUARED CJK UNIFIED IDEOGRAPH-5929SQUARED CJK UNIFIED IDEOGRAPH-4EA4SQUARED CJK UNIFIED IDEOGRAPH-6620SQUARED CJK UNIFIED IDEO\
    \GRAPH-7121SQUARED CJK UNIFIED IDEOGRAPH-6599SQUARED CJK UNIFIED IDEOGRAPH-524DSQUARED CJK UNIFIED IDEOGRAPH-5F8CSQUARED CJK UNIFIED IDEOGRAPH-518DSQUARED CJK UNIFIED IDEOGRAPH-65B0SQUARED CJK UNIFIED IDEOGRAPH-521DSQUARED CJK UNIFIED IDEOGRAPH-7D42SQUARED\
    \ CJK UNIFIED IDEOGRAPH-751FSQUARED CJK UNIFIED IDEOGRAPH-8CA9SQUARED CJK UNIFIED IDEOGRAPH-58F0SQUARED CJK UNIFIED IDEOGRAPH-5439SQUARED CJK UNIFIED IDEOGRAPH-6F14SQUARED CJK UNIFIED IDEOGRAPH-6295SQUARED CJK UNIFIED IDEOGRAPH-6355SQUARED CJK UNIFIED IDEO\
    \GRAPH-4E00SQUARED CJK UNIFIED IDEOGRAPH-4E09SQUARED CJK UNIFIED IDEOGRAPH-904ASQUARED CJK UNIFIED IDEOGRAPH-5DE6SQUARED CJK UNIFIED IDEOGRAPH-4E2DSQUARED CJK UNIFIED IDEOGRAPH-53F3SQUARED CJK UNIFIED IDEOGRAPH-6307SQUARED CJK UNIFIED IDEOGRAPH-8D70SQUARED\
    \ CJK UNIFIED IDEOGRAPH-6253SQUARED CJK UNIFIED IDEOGRAPH-7981SQUARED CJK UNIFIED IDEOGRAPH-7A7ASQUARED CJK UNIFIED IDEOGRAPH-5408SQUARED CJK UNIFIED IDEOGRAPH-6E80SQUARED CJK UNIFIED IDEOGRAPH-6709SQUARED CJK UNIFIED IDEOGRAPH-6708SQUARED CJK UNIFIED IDEO\
    \GRAPH-7533SQUARED CJK UNIFIED IDEOGRAPH-5272SQUARED CJK UNIFIED IDEOGRAPH-55B6SQUARED CJK UNIFIED IDEOGRAPH-914DTORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRAPH-672CTORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRAPH-4E09TORTOISE SHELL BRACKETED CJK UNIFIED IDEO\
    \GRAPH-4E8CTORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRAPH-5B89TORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRAPH-70B9TORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRAPH-6253TORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRAPH-76D7TORTOISE SHELL BRACKETED CJK UNIFIED IDEO\
    \GRAPH-52DDTORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRAPH-6557CIRCLED IDEOGRAPH ADVANTAGECIRCLED IDEOGRAPH ACCEPTROUNDED SYMBOL FOR FUROUNDED SYMBOL FOR LUROUNDED SYMBOL FOR SHOUROUNDED SYMBOL FOR XIROUNDED SYMBOL FOR SHUANGXIROUNDED SYMBOL FOR CAICYCLONEF\
    \OGGYCLOSED UMBRELLANIGHT WITH STARSSUNRISE OVER MOUNTAINSSUNRISECITYSCAPE AT DUSKSUNSET OVER BUILDINGSRAINBOWBRIDGE AT NIGHTWATER WAVEVOLCANOMILKY WAYEARTH GLOBE EUROPE-AFRICAEARTH GLOBE AMERICASEARTH GLOBE ASIA-AUSTRALIAGLOBE WITH MERIDIANSNEW MOON SYMBO\
    \LWAXING CRESCENT MOON SYMBOLFIRST QUARTER MOON SYMBOLWAXING GIBBOUS MOON SYMBOLFULL MOON SYMBOLWANING GIBBOUS MOON SYMBOLLAST QUARTER MOON SYMBOLWANING CRESCENT MOON SYMBOLCRESCENT MOONNEW MOON WITH FACEFIRST QUARTER MOON WITH FACELAST QUARTER MOON WITH F\
    \ACEFULL MOON WITH FACESUN WITH FACEGLOWING STARSHOOTING STARTHERMOMETERBLACK DROPLETWHITE SUNWHITE SUN WITH SMALL CLOUDWHITE SUN BEHIND CLOUDWHITE SUN BEHIND CLOUD WITH RAINCLOUD WITH RAINCLOUD WITH SNOWCLOUD WITH LIGHTNINGCLOUD WITH TORNADOFOGWIND BLOWIN\
    \G FACEHOT DOGTACOBURRITOCHESTNUTSEEDLINGEVERGREEN TREEDECIDUOUS TREEPALM TREECACTUSHOT PEPPERTULIPCHERRY BLOSSOMROSEHIBISCUSSUNFLOWERBLOSSOMEAR OF MAIZEEAR OF RICEHERBFOUR LEAF CLOVERMAPLE LEAFFALLEN LEAFLEAF FLUTTERING IN WINDMUSHROOMTOMATOAUBERGINEGRAPE\
    \SMELONWATERMELONTANGERINELEMONBANANAPINEAPPLERED APPLEGREEN APPLEPEARPEACHCHERRIESSTRAWBERRYHAMBURGERSLICE OF PIZZAMEAT ON BONEPOULTRY LEGRICE CRACKERRICE BALLCOOKED RICECURRY AND RICESTEAMING BOWLSPAGHETTIBREADFRENCH FRIESROASTED SWEET POTATODANGOODENSUS\
    \HIFRIED SHRIMPFISH CAKE WITH SWIRL DESIGNSOFT ICE CREAMSHAVED ICEICE CREAMDOUGHNUTCOOKIECHOCOLATE BARCANDYLOLLIPOPCUSTARDHONEY POTSHORTCAKEBENTO BOXPOT OF FOODCOOKINGFORK AND KNIFETEACUP WITHOUT HANDLESAKE BOTTLE AND CUPWINE GLASSCOCKTAIL GLASSTROPICAL DR\
    \INKBEER MUGCLINKING BEER MUGSBABY BOTTLEFORK AND KNIFE WITH PLATEBOTTLE WITH POPPING CORKPOPCORNRIBBONWRAPPED PRESENTBIRTHDAY CAKEJACK-O-LANTERNCHRISTMAS TREEFATHER CHRISTMASFIREWORKSFIREWORK SPARKLERBALLOONPARTY POPPERCONFETTI BALLTANABATA TREECROSSED FL\
    \AGSPINE DECORATIONJAPANESE DOLLSCARP STREAMERWIND CHIMEMOON VIEWING CEREMONYSCHOOL SATCHELGRADUATION CAPHEART WITH TIP ON THE LEFTBOUQUET OF FLOWERSMILITARY MEDALREMINDER RIBBONMUSICAL KEYBOARD WITH JACKSSTUDIO MICROPHONELEVEL SLIDERCONTROL KNOBSBEAMED AS\
    \CENDING MUSICAL NOTESBEAMED DESCENDING MUSICAL NOTESFILM FRAMESADMISSION TICKETSCAROUSEL HORSEFERRIS WHEELROLLER COASTERFISHING POLE AND FISHMICROPHONEMOVIE CAMERACINEMAHEADPHONEARTIST PALETTETOP HATCIRCUS TENTTICKETCLAPPER BOARDPERFORMING ARTSVIDEO GAMED\
    \IRECT HITSLOT MACHINEBILLIARDSGAME DIEBOWLINGFLOWER PLAYING CARDSMUSICAL NOTEMULTIPLE MUSICAL NOTESSAXOPHONEGUITARMUSICAL KEYBOARDTRUMPETVIOLINMUSICAL SCORERUNNING SHIRT WITH SASHTENNIS RACQUET AND BALLSKI AND SKI BOOTBASKETBALL AND HOOPCHEQUERED FLAGSNOW\
    \BOARDERRUNNERSURFERSPORTS MEDALTROPHYHORSE RACINGAMERICAN FOOTBALLRUGBY FOOTBALLSWIMMERWEIGHT LIFTERGOLFERRACING MOTORCYCLERACING CARCRICKET BAT AND BALLVOLLEYBALLFIELD HOCKEY STICK AND BALLICE HOCKEY STICK AND PUCKTABLE TENNIS PADDLE AND BALLSNOW CAPPED \
    \MOUNTAINCAMPINGBEACH WITH UMBRELLABUILDING CONSTRUCTIONHOUSE BUILDINGSCITYSCAPEDERELICT HOUSE BUILDINGCLASSICAL BUILDINGDESERTDESERT ISLANDNATIONAL PARKSTADIUMHOUSE BUILDINGHOUSE WITH GARDENOFFICE BUILDINGJAPANESE POST OFFICEEUROPEAN POST OFFICEHOSPITALBA\
    \NKAUTOMATED TELLER MACHINEHOTELLOVE HOTELCONVENIENCE STORESCHOOLDEPARTMENT STOREFACTORYIZAKAYA LANTERNJAPANESE CASTLEEUROPEAN CASTLEWHITE PENNANTBLACK PENNANTWAVING WHITE FLAGWAVING BLACK FLAGROSETTEBLACK ROSETTELABELBADMINTON RACQUET AND SHUTTLECOCKBOW A\
    \ND ARROWAMPHORAEMOJI MODIFIER FITZPATRICK TYPE-1-2EMOJI MODIFIER FITZPATRICK TYPE-3EMOJI MODIFIER FITZPATRICK TYPE-4EMOJI MODIFIER FITZPATRICK TYPE-5EMOJI MODIFIER FITZPATRICK TYPE-6RATMOUSEOXWATER BUFFALOCOWTIGERLEOPARDRABBITCATDRAGONCROCODILEWHALESNAILS\
    \NAKEHORSERAMGOATSHEEPMONKEYROOSTERCHICKENDOGPIGBOARELEPHANTOCTOPUSSPIRAL SHELLBUGANTHONEYBEELADY BEETLEFISHTROPICAL FISHBLOWFISHTURTLEHATCHING CHICKBABY CHICKFRONT-FACING BABY CHICKBIRDPENGUINKOALAPOODLEDROMEDARY CAMELBACTRIAN CAMELDOLPHINMOUSE FACECOW FA\
    \CETIGER FACERABBIT FACECAT FACEDRAGON FACESPOUTING WHALEHORSE FACEMONKEY FACEDOG FACEPIG FACEFROG FACEHAMSTER FACEWOLF FACEBEAR FACEPANDA FACEPIG NOSEPAW PRINTSCHIPMUNKEYESEYEEARNOSEMOUTHTONGUEWHITE UP POINTING BACKHAND INDEXWHITE DOWN POINTING BACKHAND I\
    \NDEXWHITE LEFT POINTING BACKHAND INDEXWHITE RIGHT POINTING BACKHAND INDEXFISTED HAND SIGNWAVING HAND SIGNOK HAND SIGNTHUMBS UP SIGNTHUMBS DOWN SIGNCLAPPING HANDS SIGNOPEN HANDS SIGNCROWNWOMANS HATEYEGLASSESNECKTIET-SHIRTJEANSDRESSKIMONOBIKINIWOMANS CLOTHE\
    \SPURSEHANDBAGPOUCHMANS SHOEATHLETIC SHOEHIGH-HEELED SHOEWOMANS SANDALWOMANS BOOTSFOOTPRINTSBUST IN SILHOUETTEBUSTS IN SILHOUETTEBOYGIRLMANWOMANFAMILYMAN AND WOMAN HOLDING HANDSTWO MEN HOLDING HANDSTWO WOMEN HOLDING HANDSPOLICE OFFICERWOMAN WITH BUNNY EARS\
    \BRIDE WITH VEILPERSON WITH BLOND HAIRMAN WITH GUA PI MAOMAN WITH TURBANOLDER MANOLDER WOMANBABYCONSTRUCTION WORKERPRINCESSJAPANESE OGREJAPANESE GOBLINGHOSTBABY ANGELEXTRATERRESTRIAL ALIENALIEN MONSTERIMPSKULLINFORMATION DESK PERSONGUARDSMANDANCERLIPSTICKN\
    \AIL POLISHFACE MASSAGEHAIRCUTBARBER POLESYRINGEPILLKISS MARKLOVE LETTERRINGGEM STONEKISSBOUQUETCOUPLE WITH HEARTWEDDINGBEATING HEARTBROKEN HEARTTWO HEARTSSPARKLING HEARTGROWING HEARTHEART WITH ARROWBLUE HEARTGREEN HEARTYELLOW HEARTPURPLE HEARTHEART WITH R\
    \IBBONREVOLVING HEARTSHEART DECORATIONDIAMOND SHAPE WITH A DOT INSIDEELECTRIC LIGHT BULBANGER SYMBOLBOMBSLEEPING SYMBOLCOLLISION SYMBOLSPLASHING SWEAT SYMBOLDROPLETDASH SYMBOLPILE OF POOFLEXED BICEPSDIZZY SYMBOLSPEECH BALLOONTHOUGHT BALLOONWHITE FLOWERHUND\
    \RED POINTS SYMBOLMONEY BAGCURRENCY EXCHANGEHEAVY DOLLAR SIGNCREDIT CARDBANKNOTE WITH YEN SIGNBANKNOTE WITH DOLLAR SIGNBANKNOTE WITH EURO SIGNBANKNOTE WITH POUND SIGNMONEY WITH WINGSCHART WITH UPWARDS TREND AND YEN SIGNSEATPERSONAL COMPUTERBRIEFCASEMINIDIS\
    \CFLOPPY DISKOPTICAL DISCDVDFILE FOLDEROPEN FILE FOLDERPAGE WITH CURLPAGE FACING UPCALENDARTEAR-OFF CALENDARCARD INDEXCHART WITH UPWARDS TRENDCHART WITH DOWNWARDS TRENDBAR CHARTCLIPBOARDPUSHPINROUND PUSHPINPAPERCLIPSTRAIGHT RULERTRIANGULAR RULERBOOKMARK TA\
    \BSLEDGERNOTEBOOKNOTEBOOK WITH DECORATIVE COVERCLOSED BOOKOPEN BOOKGREEN BOOKBLUE BOOKORANGE BOOKBOOKSNAME BADGESCROLLMEMOTELEPHONE RECEIVERPAGERFAX MACHINESATELLITE ANTENNAPUBLIC ADDRESS LOUDSPEAKERCHEERING MEGAPHONEOUTBOX TRAYINBOX TRAYPACKAGEE-MAIL SYMB\
    \OLINCOMING ENVELOPEENVELOPE WITH DOWNWARDS ARROW ABOVECLOSED MAILBOX WITH LOWERED FLAGCLOSED MAILBOX WITH RAISED FLAGOPEN MAILBOX WITH RAISED FLAGOPEN MAILBOX WITH LOWERED FLAGPOSTBOXPOSTAL HORNNEWSPAPERMOBILE PHONEMOBILE PHONE WITH RIGHTWARDS ARROW AT LE\
    \FTVIBRATION MODEMOBILE PHONE OFFNO MOBILE PHONESANTENNA WITH BARSCAMERACAMERA WITH FLASHVIDEO CAMERATELEVISIONRADIOVIDEOCASSETTEFILM PROJECTORPORTABLE STEREOPRAYER BEADSTWISTED RIGHTWARDS ARROWSCLOCKWISE RIGHTWARDS AND LEFTWARDS OPEN CIRCLE ARROWSCLOCKWIS\
    \E RIGHTWARDS AND LEFTWARDS OPEN CIRCLE ARROWS WITH CIRCLED ONE OVERLAYCLOCKWISE DOWNWARDS AND UPWARDS OPEN CIRCLE ARROWSANTICLOCKWISE DOWNWARDS AND UPWARDS OPEN CIRCLE ARROWSLOW BRIGHTNESS SYMBOLHIGH BRIGHTNESS SYMBOLSPEAKER WITH CANCELLATION STROKESPEAKE\
    \RSPEAKER WITH ONE SOUND WAVESPEAKER WITH THREE SOUND WAVESBATTERYELECTRIC PLUGLEFT-POINTING MAGNIFYING GLASSRIGHT-POINTING MAGNIFYING GLASSLOCK WITH INK PENCLOSED LOCK WITH KEYKEYLOCKOPEN LOCKBELLBELL WITH CANCELLATION STROKEBOOKMARKLINK SYMBOLRADIO BUTTO\
    \NBACK WITH LEFTWARDS ARROW ABOVEEND WITH LEFTWARDS ARROW ABOVEON WITH EXCLAMATION MARK WITH LEFT RIGHT ARROW ABOVESOON WITH RIGHTWARDS ARROW ABOVETOP WITH UPWARDS ARROW ABOVENO ONE UNDER EIGHTEEN SYMBOLKEYCAP TENINPUT SYMBOL FOR LATIN CAPITAL LETTERSINPUT\
    \ SYMBOL FOR LATIN SMALL LETTERSINPUT SYMBOL FOR NUMBERSINPUT SYMBOL FOR SYMBOLSINPUT SYMBOL FOR LATIN LETTERSFIREELECTRIC TORCHWRENCHHAMMERNUT AND BOLTHOCHOPISTOLMICROSCOPETELESCOPECRYSTAL BALLSIX POINTED STAR WITH MIDDLE DOTJAPANESE SYMBOL FOR BEGINNERTR\
    \IDENT EMBLEMBLACK SQUARE BUTTONWHITE SQUARE BUTTONLARGE RED CIRCLELARGE BLUE CIRCLELARGE ORANGE DIAMONDLARGE BLUE DIAMONDSMALL ORANGE DIAMONDSMALL BLUE DIAMONDUP-POINTING RED TRIANGLEDOWN-POINTING RED TRIANGLEUP-POINTING SMALL RED TRIANGLEDOWN-POINTING SM\
    \ALL RED TRIANGLELOWER RIGHT SHADOWED WHITE CIRCLEUPPER RIGHT SHADOWED WHITE CIRCLECIRCLED CROSS POMMEECROSS POMMEE WITH HALF-CIRCLE BELOWCROSS POMMEENOTCHED LEFT SEMICIRCLE WITH THREE DOTSNOTCHED RIGHT SEMICIRCLE WITH THREE DOTSSYMBOL FOR MARKS CHAPTERWHI\
    \TE LATIN CROSSHEAVY LATIN CROSSCELTIC CROSSOM SYMBOLDOVE OF PEACEKAABAMOSQUESYNAGOGUEMENORAH WITH NINE BRANCHESBOWL OF HYGIEIACLOCK FACE ONE OCLOCKCLOCK FACE TWO OCLOCKCLOCK FACE THREE OCLOCKCLOCK FACE FOUR OCLOCKCLOCK FACE FIVE OCLOCKCLOCK FACE SIX OCLOC\
    \KCLOCK FACE SEVEN OCLOCKCLOCK FACE EIGHT OCLOCKCLOCK FACE NINE OCLOCKCLOCK FACE TEN OCLOCKCLOCK FACE ELEVEN OCLOCKCLOCK FACE TWELVE OCLOCKCLOCK FACE ONE-THIRTYCLOCK FACE TWO-THIRTYCLOCK FACE THREE-THIRTYCLOCK FACE FOUR-THIRTYCLOCK FACE FIVE-THIRTYCLOCK FA\
    \CE SIX-THIRTYCLOCK FACE SEVEN-THIRTYCLOCK FACE EIGHT-THIRTYCLOCK FACE NINE-THIRTYCLOCK FACE TEN-THIRTYCLOCK FACE ELEVEN-THIRTYCLOCK FACE TWELVE-THIRTYRIGHT SPEAKERRIGHT SPEAKER WITH ONE SOUND WAVERIGHT SPEAKER WITH THREE SOUND WAVESBULLHORNBULLHORN WITH S\
    \OUND WAVESRINGING BELLBOOKCANDLEMANTELPIECE CLOCKBLACK SKULL AND CROSSBONESNO PIRACYHOLEMAN IN BUSINESS SUIT LEVITATINGSLEUTH OR SPYDARK SUNGLASSESSPIDERSPIDER WEBJOYSTICKMAN DANCINGLEFT HAND TELEPHONE RECEIVERTELEPHONE RECEIVER WITH PAGERIGHT HAND TELEPH\
    \ONE RECEIVERWHITE TOUCHTONE TELEPHONEBLACK TOUCHTONE TELEPHONETELEPHONE ON TOP OF MODEMCLAMSHELL MOBILE PHONEBACK OF ENVELOPESTAMPED ENVELOPEENVELOPE WITH LIGHTNINGFLYING ENVELOPEPEN OVER STAMPED ENVELOPELINKED PAPERCLIPSBLACK PUSHPINLOWER LEFT PENCILLOWE\
    \R LEFT BALLPOINT PENLOWER LEFT FOUNTAIN PENLOWER LEFT PAINTBRUSHLOWER LEFT CRAYONLEFT WRITING HANDTURNED OK HAND SIGNRAISED HAND WITH FINGERS SPLAYEDREVERSED RAISED HAND WITH FINGERS SPLAYEDREVERSED THUMBS UP SIGNREVERSED THUMBS DOWN SIGNREVERSED VICTORY \
    \HANDREVERSED HAND WITH MIDDLE FINGER EXTENDEDRAISED HAND WITH PART BETWEEN MIDDLE AND RING FINGERSWHITE DOWN POINTING LEFT HAND INDEXSIDEWAYS WHITE LEFT POINTING INDEXSIDEWAYS WHITE RIGHT POINTING INDEXSIDEWAYS BLACK LEFT POINTING INDEXSIDEWAYS BLACK RIGH\
    \T POINTING INDEXBLACK LEFT POINTING BACKHAND INDEXBLACK RIGHT POINTING BACKHAND INDEXSIDEWAYS WHITE UP POINTING INDEXSIDEWAYS WHITE DOWN POINTING INDEXSIDEWAYS BLACK UP POINTING INDEXSIDEWAYS BLACK DOWN POINTING INDEXBLACK UP POINTING BACKHAND INDEXBLACK \
    \DOWN POINTING BACKHAND INDEXBLACK HEARTDESKTOP COMPUTERKEYBOARD AND MOUSETHREE NETWORKED COMPUTERSPRINTERPOCKET CALCULATORBLACK HARD SHELL FLOPPY DISKWHITE HARD SHELL FLOPPY DISKSOFT SHELL FLOPPY DISKTAPE CARTRIDGEWIRED KEYBOARDONE BUTTON MOUSETWO BUTTON \
    \MOUSETHREE BUTTON MOUSETRACKBALLOLD PERSONAL COMPUTERHARD DISKSCREENPRINTER ICONFAX ICONOPTICAL DISC ICONDOCUMENT WITH TEXTDOCUMENT WITH TEXT AND PICTUREDOCUMENT WITH PICTUREFRAME WITH PICTUREFRAME WITH TILESFRAME WITH AN XBLACK FOLDERFOLDEROPEN FOLDERCAR\
    \D INDEX DIVIDERSCARD FILE BOXFILE CABINETEMPTY NOTEEMPTY NOTE PAGEEMPTY NOTE PADNOTENOTE PAGENOTE PADEMPTY DOCUMENTEMPTY PAGEEMPTY PAGESDOCUMENTPAGEPAGESWASTEBASKETSPIRAL NOTE PADSPIRAL CALENDAR PADDESKTOP WINDOWMINIMIZEMAXIMIZEOVERLAPCLOCKWISE RIGHT AND \
    \LEFT SEMICIRCLE ARROWSCANCELLATION XINCREASE FONT SIZE SYMBOLDECREASE FONT SIZE SYMBOLCOMPRESSIONOLD KEYROLLED-UP NEWSPAPERPAGE WITH CIRCLED TEXTSTOCK CHARTDAGGER KNIFELIPSSPEAKING HEAD IN SILHOUETTETHREE RAYS ABOVETHREE RAYS BELOWTHREE RAYS LEFTTHREE RAY\
    \S RIGHTLEFT SPEECH BUBBLERIGHT SPEECH BUBBLETWO SPEECH BUBBLESTHREE SPEECH BUBBLESLEFT THOUGHT BUBBLERIGHT THOUGHT BUBBLELEFT ANGER BUBBLERIGHT ANGER BUBBLEMOOD BUBBLELIGHTNING MOOD BUBBLELIGHTNING MOODBALLOT BOX WITH BALLOTBALLOT SCRIPT XBALLOT BOX WITH \
    \SCRIPT XBALLOT BOLD SCRIPT XBALLOT BOX WITH BOLD SCRIPT XLIGHT CHECK MARKBALLOT BOX WITH BOLD CHECKWORLD MAPMOUNT FUJITOKYO TOWERSTATUE OF LIBERTYSILHOUETTE OF JAPANMOYAIGRINNING FACEGRINNING FACE WITH SMILING EYESFACE WITH TEARS OF JOYSMILING FACE WITH O\
    \PEN MOUTHSMILING FACE WITH OPEN MOUTH AND SMILING EYESSMILING FACE WITH OPEN MOUTH AND COLD SWEATSMILING FACE WITH OPEN MOUTH AND TIGHTLY-CLOSED EYESSMILING FACE WITH HALOSMILING FACE WITH HORNSWINKING FACESMILING FACE WITH SMILING EYESFACE SAVOURING DELI\
    \CIOUS FOODRELIEVED FACESMILING FACE WITH HEART-SHAPED EYESSMILING FACE WITH SUNGLASSESSMIRKING FACENEUTRAL FACEEXPRESSIONLESS FACEUNAMUSED FACEFACE WITH COLD SWEATPENSIVE FACECONFUSED FACECONFOUNDED FACEKISSING FACEFACE THROWING A KISSKISSING FACE WITH SM\
    \ILING EYESKISSING FACE WITH CLOSED EYESFACE WITH STUCK-OUT TONGUEFACE WITH STUCK-OUT TONGUE AND WINKING EYEFACE WITH STUCK-OUT TONGUE AND TIGHTLY-CLOSED EYESDISAPPOINTED FACEWORRIED FACEANGRY FACEPOUTING FACECRYING FACEPERSEVERING FACEFACE WITH LOOK OF TR\
    \IUMPHDISAPPOINTED BUT RELIEVED FACEFROWNING FACE WITH OPEN MOUTHANGUISHED FACEFEARFUL FACEWEARY FACESLEEPY FACETIRED FACEGRIMACING FACELOUDLY CRYING FACEFACE WITH OPEN MOUTHHUSHED FACEFACE WITH OPEN MOUTH AND COLD SWEATFACE SCREAMING IN FEARASTONISHED FAC\
    \EFLUSHED FACESLEEPING FACEDIZZY FACEFACE WITHOUT MOUTHFACE WITH MEDICAL MASKGRINNING CAT FACE WITH SMILING EYESCAT FACE WITH TEARS OF JOYSMILING CAT FACE WITH OPEN MOUTHSMILING CAT FACE WITH HEART-SHAPED EYESCAT FACE WITH WRY SMILEKISSING CAT FACE WITH CL\
    \OSED EYESPOUTING CAT FACECRYING CAT FACEWEARY CAT FACESLIGHTLY FROWNING FACESLIGHTLY SMILING FACEUPSIDE-DOWN FACEFACE WITH ROLLING EYESFACE WITH NO GOOD GESTUREFACE WITH OK GESTUREPERSON BOWING DEEPLYSEE-NO-EVIL MONKEYHEAR-NO-EVIL MONKEYSPEAK-NO-EVIL MONK\
    \EYHAPPY PERSON RAISING ONE HANDPERSON RAISING BOTH HANDS IN CELEBRATIONPERSON FROWNINGPERSON WITH POUTING FACEPERSON WITH FOLDED HANDSNORTH WEST POINTING LEAFSOUTH WEST POINTING LEAFNORTH EAST POINTING LEAFSOUTH EAST POINTING LEAFTURNED NORTH WEST POINTIN\
    \G LEAFTURNED SOUTH WEST POINTING LEAFTURNED NORTH EAST POINTING LEAFTURNED SOUTH EAST POINTING LEAFNORTH WEST POINTING VINE LEAFSOUTH WEST POINTING VINE LEAFNORTH EAST POINTING VINE LEAFSOUTH EAST POINTING VINE LEAFHEAVY NORTH WEST POINTING VINE LEAFHEAVY\
    \ SOUTH WEST POINTING VINE LEAFHEAVY NORTH EAST POINTING VINE LEAFHEAVY SOUTH EAST POINTING VINE LEAFNORTH WEST POINTING BUDSOUTH WEST POINTING BUDNORTH EAST POINTING BUDSOUTH EAST POINTING BUDHEAVY NORTH WEST POINTING BUDHEAVY SOUTH WEST POINTING BUDHEAVY\
    \ NORTH EAST POINTING BUDHEAVY SOUTH EAST POINTING BUDHOLLOW QUILT SQUARE ORNAMENTHOLLOW QUILT SQUARE ORNAMENT IN BLACK SQUARESOLID QUILT SQUARE ORNAMENTSOLID QUILT SQUARE ORNAMENT IN BLACK SQUARELEFTWARDS ROCKETUPWARDS ROCKETRIGHTWARDS ROCKETDOWNWARDS ROC\
    \KETSCRIPT LIGATURE ET ORNAMENTHEAVY SCRIPT LIGATURE ET ORNAMENTLIGATURE OPEN ET ORNAMENTHEAVY LIGATURE OPEN ET ORNAMENTHEAVY AMPERSAND ORNAMENTSWASH AMPERSAND ORNAMENTSANS-SERIF HEAVY DOUBLE TURNED COMMA QUOTATION MARK ORNAMENTSANS-SERIF HEAVY DOUBLE COMM\
    \A QUOTATION MARK ORNAMENTSANS-SERIF HEAVY LOW DOUBLE COMMA QUOTATION MARK ORNAMENTHEAVY INTERROBANG ORNAMENTSANS-SERIF INTERROBANG ORNAMENTHEAVY SANS-SERIF INTERROBANG ORNAMENTVERY HEAVY SOLIDUSVERY HEAVY REVERSE SOLIDUSCHECKER BOARDREVERSE CHECKER BOARDR\
    \OCKETHELICOPTERSTEAM LOCOMOTIVERAILWAY CARHIGH-SPEED TRAINHIGH-SPEED TRAIN WITH BULLET NOSETRAINMETROLIGHT RAILSTATIONTRAMTRAM CARBUSONCOMING BUSTROLLEYBUSBUS STOPMINIBUSAMBULANCEFIRE ENGINEPOLICE CARONCOMING POLICE CARTAXIONCOMING TAXIAUTOMOBILEONCOMING \
    \AUTOMOBILERECREATIONAL VEHICLEDELIVERY TRUCKARTICULATED LORRYTRACTORMONORAILMOUNTAIN RAILWAYSUSPENSION RAILWAYMOUNTAIN CABLEWAYAERIAL TRAMWAYSHIPROWBOATSPEEDBOATHORIZONTAL TRAFFIC LIGHTVERTICAL TRAFFIC LIGHTCONSTRUCTION SIGNPOLICE CARS REVOLVING LIGHTTRIA\
    \NGULAR FLAG ON POSTDOORNO ENTRY SIGNSMOKING SYMBOLNO SMOKING SYMBOLPUT LITTER IN ITS PLACE SYMBOLDO NOT LITTER SYMBOLPOTABLE WATER SYMBOLNON-POTABLE WATER SYMBOLBICYCLENO BICYCLESBICYCLISTMOUNTAIN BICYCLISTPEDESTRIANNO PEDESTRIANSCHILDREN CROSSINGMENS SYM\
    \BOLWOMENS SYMBOLRESTROOMBABY SYMBOLTOILETWATER CLOSETSHOWERBATHBATHTUBPASSPORT CONTROLCUSTOMSBAGGAGE CLAIMLEFT LUGGAGETRIANGLE WITH ROUNDED CORNERSPROHIBITED SIGNCIRCLED INFORMATION SOURCEBOYS SYMBOLGIRLS SYMBOLCOUCH AND LAMPSLEEPING ACCOMMODATIONSHOPPING\
    \ BAGSBELLHOP BELLBEDPLACE OF WORSHIPOCTAGONAL SIGNSHOPPING TROLLEYSTUPAPAGODAHINDU TEMPLEHUTELEVATORWIRELESSPLAYGROUND SLIDEWHEELRING BUOYHAMMER AND WRENCHSHIELDOIL DRUMMOTORWAYRAILWAY TRACKMOTOR BOATUP-POINTING MILITARY AIRPLANEUP-POINTING AIRPLANEUP-POI\
    \NTING SMALL AIRPLANESMALL AIRPLANENORTHEAST-POINTING AIRPLANEAIRPLANE DEPARTUREAIRPLANE ARRIVINGSATELLITEONCOMING FIRE ENGINEDIESEL LOCOMOTIVEPASSENGER SHIPSCOOTERMOTOR SCOOTERCANOESLEDFLYING SAUCERSKATEBOARDAUTO RICKSHAWPICKUP TRUCKROLLER SKATEALCHEMICAL\
    \ SYMBOL FOR QUINTESSENCEALCHEMICAL SYMBOL FOR AIRALCHEMICAL SYMBOL FOR FIREALCHEMICAL SYMBOL FOR EARTHALCHEMICAL SYMBOL FOR WATERALCHEMICAL SYMBOL FOR AQUAFORTISALCHEMICAL SYMBOL FOR AQUA REGIAALCHEMICAL SYMBOL FOR AQUA REGIA-2ALCHEMICAL SYMBOL FOR AQUA V\
    \ITAEALCHEMICAL SYMBOL FOR AQUA VITAE-2ALCHEMICAL SYMBOL FOR VINEGARALCHEMICAL SYMBOL FOR VINEGAR-2ALCHEMICAL SYMBOL FOR VINEGAR-3ALCHEMICAL SYMBOL FOR SULFURALCHEMICAL SYMBOL FOR PHILOSOPHERS SULFURALCHEMICAL SYMBOL FOR BLACK SULFURALCHEMICAL SYMBOL FOR M\
    \ERCURY SUBLIMATEALCHEMICAL SYMBOL FOR MERCURY SUBLIMATE-2ALCHEMICAL SYMBOL FOR MERCURY SUBLIMATE-3ALCHEMICAL SYMBOL FOR CINNABARALCHEMICAL SYMBOL FOR SALTALCHEMICAL SYMBOL FOR NITREALCHEMICAL SYMBOL FOR VITRIOLALCHEMICAL SYMBOL FOR VITRIOL-2ALCHEMICAL SYM\
    \BOL FOR ROCK SALTALCHEMICAL SYMBOL FOR ROCK SALT-2ALCHEMICAL SYMBOL FOR GOLDALCHEMICAL SYMBOL FOR SILVERALCHEMICAL SYMBOL FOR IRON OREALCHEMICAL SYMBOL FOR IRON ORE-2ALCHEMICAL SYMBOL FOR CROCUS OF IRONALCHEMICAL SYMBOL FOR REGULUS OF IRONALCHEMICAL SYMBO\
    \L FOR COPPER OREALCHEMICAL SYMBOL FOR IRON-COPPER OREALCHEMICAL SYMBOL FOR SUBLIMATE OF COPPERALCHEMICAL SYMBOL FOR CROCUS OF COPPERALCHEMICAL SYMBOL FOR CROCUS OF COPPER-2ALCHEMICAL SYMBOL FOR COPPER ANTIMONIATEALCHEMICAL SYMBOL FOR SALT OF COPPER ANTIMO\
    \NIATEALCHEMICAL SYMBOL FOR SUBLIMATE OF SALT OF COPPERALCHEMICAL SYMBOL FOR VERDIGRISALCHEMICAL SYMBOL FOR TIN OREALCHEMICAL SYMBOL FOR LEAD OREALCHEMICAL SYMBOL FOR ANTIMONY OREALCHEMICAL SYMBOL FOR SUBLIMATE OF ANTIMONYALCHEMICAL SYMBOL FOR SALT OF ANTI\
    \MONYALCHEMICAL SYMBOL FOR SUBLIMATE OF SALT OF ANTIMONYALCHEMICAL SYMBOL FOR VINEGAR OF ANTIMONYALCHEMICAL SYMBOL FOR REGULUS OF ANTIMONYALCHEMICAL SYMBOL FOR REGULUS OF ANTIMONY-2ALCHEMICAL SYMBOL FOR REGULUSALCHEMICAL SYMBOL FOR REGULUS-2ALCHEMICAL SYMB\
    \OL FOR REGULUS-3ALCHEMICAL SYMBOL FOR REGULUS-4ALCHEMICAL SYMBOL FOR ALKALIALCHEMICAL SYMBOL FOR ALKALI-2ALCHEMICAL SYMBOL FOR MARCASITEALCHEMICAL SYMBOL FOR SAL-AMMONIACALCHEMICAL SYMBOL FOR ARSENICALCHEMICAL SYMBOL FOR REALGARALCHEMICAL SYMBOL FOR REALG\
    \AR-2ALCHEMICAL SYMBOL FOR AURIPIGMENTALCHEMICAL SYMBOL FOR BISMUTH OREALCHEMICAL SYMBOL FOR TARTARALCHEMICAL SYMBOL FOR TARTAR-2ALCHEMICAL SYMBOL FOR QUICK LIMEALCHEMICAL SYMBOL FOR BORAXALCHEMICAL SYMBOL FOR BORAX-2ALCHEMICAL SYMBOL FOR BORAX-3ALCHEMICAL\
    \ SYMBOL FOR ALUMALCHEMICAL SYMBOL FOR OILALCHEMICAL SYMBOL FOR SPIRITALCHEMICAL SYMBOL FOR TINCTUREALCHEMICAL SYMBOL FOR GUMALCHEMICAL SYMBOL FOR WAXALCHEMICAL SYMBOL FOR POWDERALCHEMICAL SYMBOL FOR CALXALCHEMICAL SYMBOL FOR TUTTYALCHEMICAL SYMBOL FOR CAP\
    \UT MORTUUMALCHEMICAL SYMBOL FOR SCEPTER OF JOVEALCHEMICAL SYMBOL FOR CADUCEUSALCHEMICAL SYMBOL FOR TRIDENTALCHEMICAL SYMBOL FOR STARRED TRIDENTALCHEMICAL SYMBOL FOR LODESTONEALCHEMICAL SYMBOL FOR SOAPALCHEMICAL SYMBOL FOR URINEALCHEMICAL SYMBOL FOR HORSE \
    \DUNGALCHEMICAL SYMBOL FOR ASHESALCHEMICAL SYMBOL FOR POT ASHESALCHEMICAL SYMBOL FOR BRICKALCHEMICAL SYMBOL FOR POWDERED BRICKALCHEMICAL SYMBOL FOR AMALGAMALCHEMICAL SYMBOL FOR STRATUM SUPER STRATUMALCHEMICAL SYMBOL FOR STRATUM SUPER STRATUM-2ALCHEMICAL SY\
    \MBOL FOR SUBLIMATIONALCHEMICAL SYMBOL FOR PRECIPITATEALCHEMICAL SYMBOL FOR DISTILLALCHEMICAL SYMBOL FOR DISSOLVEALCHEMICAL SYMBOL FOR DISSOLVE-2ALCHEMICAL SYMBOL FOR PURIFYALCHEMICAL SYMBOL FOR PUTREFACTIONALCHEMICAL SYMBOL FOR CRUCIBLEALCHEMICAL SYMBOL F\
    \OR CRUCIBLE-2ALCHEMICAL SYMBOL FOR CRUCIBLE-3ALCHEMICAL SYMBOL FOR CRUCIBLE-4ALCHEMICAL SYMBOL FOR CRUCIBLE-5ALCHEMICAL SYMBOL FOR ALEMBICALCHEMICAL SYMBOL FOR BATH OF MARYALCHEMICAL SYMBOL FOR BATH OF VAPOURSALCHEMICAL SYMBOL FOR RETORTALCHEMICAL SYMBOL \
    \FOR HOURALCHEMICAL SYMBOL FOR NIGHTALCHEMICAL SYMBOL FOR DAY-NIGHTALCHEMICAL SYMBOL FOR MONTHALCHEMICAL SYMBOL FOR HALF DRAMALCHEMICAL SYMBOL FOR HALF OUNCELOT OF FORTUNEOCCULTATIONLUNAR ECLIPSEHAUMEAMAKEMAKEGONGGONGQUAOARORCUSBLACK LEFT-POINTING ISOSCELE\
    \S RIGHT TRIANGLEBLACK UP-POINTING ISOSCELES RIGHT TRIANGLEBLACK RIGHT-POINTING ISOSCELES RIGHT TRIANGLEBLACK DOWN-POINTING ISOSCELES RIGHT TRIANGLEBLACK SLIGHTLY SMALL CIRCLEMEDIUM BOLD WHITE CIRCLEBOLD WHITE CIRCLEHEAVY WHITE CIRCLEVERY HEAVY WHITE CIRCL\
    \EEXTREMELY HEAVY WHITE CIRCLEWHITE CIRCLE CONTAINING BLACK SMALL CIRCLEROUND TARGETBLACK TINY SQUAREBLACK SLIGHTLY SMALL SQUARELIGHT WHITE SQUAREMEDIUM WHITE SQUAREBOLD WHITE SQUAREHEAVY WHITE SQUAREVERY HEAVY WHITE SQUAREEXTREMELY HEAVY WHITE SQUAREWHITE\
    \ SQUARE CONTAINING BLACK VERY SMALL SQUAREWHITE SQUARE CONTAINING BLACK MEDIUM SQUARESQUARE TARGETBLACK TINY DIAMONDBLACK VERY SMALL DIAMONDBLACK MEDIUM SMALL DIAMONDWHITE DIAMOND CONTAINING BLACK VERY SMALL DIAMONDWHITE DIAMOND CONTAINING BLACK MEDIUM DI\
    \AMONDDIAMOND TARGETBLACK TINY LOZENGEBLACK VERY SMALL LOZENGEBLACK MEDIUM SMALL LOZENGEWHITE LOZENGE CONTAINING BLACK SMALL LOZENGETHIN GREEK CROSSLIGHT GREEK CROSSMEDIUM GREEK CROSSBOLD GREEK CROSSVERY BOLD GREEK CROSSVERY HEAVY GREEK CROSSEXTREMELY HEAV\
    \Y GREEK CROSSTHIN SALTIRELIGHT SALTIREMEDIUM SALTIREBOLD SALTIREHEAVY SALTIREVERY HEAVY SALTIREEXTREMELY HEAVY SALTIRELIGHT FIVE SPOKED ASTERISKMEDIUM FIVE SPOKED ASTERISKBOLD FIVE SPOKED ASTERISKHEAVY FIVE SPOKED ASTERISKVERY HEAVY FIVE SPOKED ASTERISKEX\
    \TREMELY HEAVY FIVE SPOKED ASTERISKLIGHT SIX SPOKED ASTERISKMEDIUM SIX SPOKED ASTERISKBOLD SIX SPOKED ASTERISKHEAVY SIX SPOKED ASTERISKVERY HEAVY SIX SPOKED ASTERISKEXTREMELY HEAVY SIX SPOKED ASTERISKLIGHT EIGHT SPOKED ASTERISKMEDIUM EIGHT SPOKED ASTERISKB\
    \OLD EIGHT SPOKED ASTERISKHEAVY EIGHT SPOKED ASTERISKVERY HEAVY EIGHT SPOKED ASTERISKLIGHT THREE POINTED BLACK STARMEDIUM THREE POINTED BLACK STARTHREE POINTED BLACK STARMEDIUM THREE POINTED PINWHEEL STARLIGHT FOUR POINTED BLACK STARMEDIUM FOUR POINTED BLA\
    \CK STARFOUR POINTED BLACK STARMEDIUM FOUR POINTED PINWHEEL STARREVERSE LIGHT FOUR POINTED PINWHEEL STARLIGHT FIVE POINTED BLACK STARHEAVY FIVE POINTED BLACK STARMEDIUM SIX POINTED BLACK STARHEAVY SIX POINTED BLACK STARSIX POINTED PINWHEEL STARMEDIUM EIGHT\
    \ POINTED BLACK STARHEAVY EIGHT POINTED BLACK STARVERY HEAVY EIGHT POINTED BLACK STARHEAVY EIGHT POINTED PINWHEEL STARLIGHT TWELVE POINTED BLACK STARHEAVY TWELVE POINTED BLACK STARHEAVY TWELVE POINTED PINWHEEL STARCIRCLED TRIANGLENEGATIVE CIRCLED TRIANGLEC\
    \IRCLED SQUARENEGATIVE CIRCLED SQUARENINE POINTED WHITE STARLARGE ORANGE CIRCLELARGE YELLOW CIRCLELARGE GREEN CIRCLELARGE PURPLE CIRCLELARGE BROWN CIRCLELARGE RED SQUARELARGE BLUE SQUARELARGE ORANGE SQUARELARGE YELLOW SQUARELARGE GREEN SQUARELARGE PURPLE S\
    \QUARELARGE BROWN SQUAREHEAVY EQUALS SIGNLEFTWARDS ARROW WITH SMALL TRIANGLE ARROWHEADUPWARDS ARROW WITH SMALL TRIANGLE ARROWHEADRIGHTWARDS ARROW WITH SMALL TRIANGLE ARROWHEADDOWNWARDS ARROW WITH SMALL TRIANGLE ARROWHEADLEFTWARDS ARROW WITH MEDIUM TRIANGLE\
    \ ARROWHEADUPWARDS ARROW WITH MEDIUM TRIANGLE ARROWHEADRIGHTWARDS ARROW WITH MEDIUM TRIANGLE ARROWHEADDOWNWARDS ARROW WITH MEDIUM TRIANGLE ARROWHEADLEFTWARDS ARROW WITH LARGE TRIANGLE ARROWHEADUPWARDS ARROW WITH LARGE TRIANGLE ARROWHEADRIGHTWARDS ARROW WIT\
    \H LARGE TRIANGLE ARROWHEADDOWNWARDS ARROW WITH LARGE TRIANGLE ARROWHEADLEFTWARDS ARROW WITH SMALL EQUILATERAL ARROWHEADUPWARDS ARROW WITH SMALL EQUILATERAL ARROWHEADRIGHTWARDS ARROW WITH SMALL EQUILATERAL ARROWHEADDOWNWARDS ARROW WITH SMALL EQUILATERAL AR\
    \ROWHEADLEFTWARDS ARROW WITH EQUILATERAL ARROWHEADUPWARDS ARROW WITH EQUILATERAL ARROWHEADRIGHTWARDS ARROW WITH EQUILATERAL ARROWHEADDOWNWARDS ARROW WITH EQUILATERAL ARROWHEADHEAVY LEFTWARDS ARROW WITH EQUILATERAL ARROWHEADHEAVY UPWARDS ARROW WITH EQUILATE\
    \RAL ARROWHEADHEAVY RIGHTWARDS ARROW WITH EQUILATERAL ARROWHEADHEAVY DOWNWARDS ARROW WITH EQUILATERAL ARROWHEADHEAVY LEFTWARDS ARROW WITH LARGE EQUILATERAL ARROWHEADHEAVY UPWARDS ARROW WITH LARGE EQUILATERAL ARROWHEADHEAVY RIGHTWARDS ARROW WITH LARGE EQUIL\
    \ATERAL ARROWHEADHEAVY DOWNWARDS ARROW WITH LARGE EQUILATERAL ARROWHEADLEFTWARDS TRIANGLE-HEADED ARROW WITH NARROW SHAFTUPWARDS TRIANGLE-HEADED ARROW WITH NARROW SHAFTRIGHTWARDS TRIANGLE-HEADED ARROW WITH NARROW SHAFTDOWNWARDS TRIANGLE-HEADED ARROW WITH NA\
    \RROW SHAFTLEFTWARDS TRIANGLE-HEADED ARROW WITH MEDIUM SHAFTUPWARDS TRIANGLE-HEADED ARROW WITH MEDIUM SHAFTRIGHTWARDS TRIANGLE-HEADED ARROW WITH MEDIUM SHAFTDOWNWARDS TRIANGLE-HEADED ARROW WITH MEDIUM SHAFTLEFTWARDS TRIANGLE-HEADED ARROW WITH BOLD SHAFTUPW\
    \ARDS TRIANGLE-HEADED ARROW WITH BOLD SHAFTRIGHTWARDS TRIANGLE-HEADED ARROW WITH BOLD SHAFTDOWNWARDS TRIANGLE-HEADED ARROW WITH BOLD SHAFTLEFTWARDS TRIANGLE-HEADED ARROW WITH HEAVY SHAFTUPWARDS TRIANGLE-HEADED ARROW WITH HEAVY SHAFTRIGHTWARDS TRIANGLE-HEAD\
    \ED ARROW WITH HEAVY SHAFTDOWNWARDS TRIANGLE-HEADED ARROW WITH HEAVY SHAFTLEFTWARDS TRIANGLE-HEADED ARROW WITH VERY HEAVY SHAFTUPWARDS TRIANGLE-HEADED ARROW WITH VERY HEAVY SHAFTRIGHTWARDS TRIANGLE-HEADED ARROW WITH VERY HEAVY SHAFTDOWNWARDS TRIANGLE-HEADE\
    \D ARROW WITH VERY HEAVY SHAFTLEFTWARDS FINGER-POST ARROWUPWARDS FINGER-POST ARROWRIGHTWARDS FINGER-POST ARROWDOWNWARDS FINGER-POST ARROWLEFTWARDS SQUARED ARROWUPWARDS SQUARED ARROWRIGHTWARDS SQUARED ARROWDOWNWARDS SQUARED ARROWLEFTWARDS COMPRESSED ARROWUP\
    \WARDS COMPRESSED ARROWRIGHTWARDS COMPRESSED ARROWDOWNWARDS COMPRESSED ARROWLEFTWARDS HEAVY COMPRESSED ARROWUPWARDS HEAVY COMPRESSED ARROWRIGHTWARDS HEAVY COMPRESSED ARROWDOWNWARDS HEAVY COMPRESSED ARROWLEFTWARDS HEAVY ARROWUPWARDS HEAVY ARROWRIGHTWARDS HE\
    \AVY ARROWDOWNWARDS HEAVY ARROWLEFTWARDS SANS-SERIF ARROWUPWARDS SANS-SERIF ARROWRIGHTWARDS SANS-SERIF ARROWDOWNWARDS SANS-SERIF ARROWNORTH WEST SANS-SERIF ARROWNORTH EAST SANS-SERIF ARROWSOUTH EAST SANS-SERIF ARROWSOUTH WEST SANS-SERIF ARROWLEFT RIGHT SAN\
    \S-SERIF ARROWUP DOWN SANS-SERIF ARROWWIDE-HEADED LEFTWARDS LIGHT BARB ARROWWIDE-HEADED UPWARDS LIGHT BARB ARROWWIDE-HEADED RIGHTWARDS LIGHT BARB ARROWWIDE-HEADED DOWNWARDS LIGHT BARB ARROWWIDE-HEADED NORTH WEST LIGHT BARB ARROWWIDE-HEADED NORTH EAST LIGHT\
    \ BARB ARROWWIDE-HEADED SOUTH EAST LIGHT BARB ARROWWIDE-HEADED SOUTH WEST LIGHT BARB ARROWWIDE-HEADED LEFTWARDS BARB ARROWWIDE-HEADED UPWARDS BARB ARROWWIDE-HEADED RIGHTWARDS BARB ARROWWIDE-HEADED DOWNWARDS BARB ARROWWIDE-HEADED NORTH WEST BARB ARROWWIDE-H\
    \EADED NORTH EAST BARB ARROWWIDE-HEADED SOUTH EAST BARB ARROWWIDE-HEADED SOUTH WEST BARB ARROWWIDE-HEADED LEFTWARDS MEDIUM BARB ARROWWIDE-HEADED UPWARDS MEDIUM BARB ARROWWIDE-HEADED RIGHTWARDS MEDIUM BARB ARROWWIDE-HEADED DOWNWARDS MEDIUM BARB ARROWWIDE-HE\
    \ADED NORTH WEST MEDIUM BARB ARROWWIDE-HEADED NORTH EAST MEDIUM BARB ARROWWIDE-HEADED SOUTH EAST MEDIUM BARB ARROWWIDE-HEADED SOUTH WEST MEDIUM BARB ARROWWIDE-HEADED LEFTWARDS HEAVY BARB ARROWWIDE-HEADED UPWARDS HEAVY BARB ARROWWIDE-HEADED RIGHTWARDS HEAVY\
    \ BARB ARROWWIDE-HEADED DOWNWARDS HEAVY BARB ARROWWIDE-HEADED NORTH WEST HEAVY BARB ARROWWIDE-HEADED NORTH EAST HEAVY BARB ARROWWIDE-HEADED SOUTH EAST HEAVY BARB ARROWWIDE-HEADED SOUTH WEST HEAVY BARB ARROWWIDE-HEADED LEFTWARDS VERY HEAVY BARB ARROWWIDE-HE\
    \ADED UPWARDS VERY HEAVY BARB ARROWWIDE-HEADED RIGHTWARDS VERY HEAVY BARB ARROWWIDE-HEADED DOWNWARDS VERY HEAVY BARB ARROWWIDE-HEADED NORTH WEST VERY HEAVY BARB ARROWWIDE-HEADED NORTH EAST VERY HEAVY BARB ARROWWIDE-HEADED SOUTH EAST VERY HEAVY BARB ARROWWI\
    \DE-HEADED SOUTH WEST VERY HEAVY BARB ARROWLEFTWARDS TRIANGLE ARROWHEADUPWARDS TRIANGLE ARROWHEADRIGHTWARDS TRIANGLE ARROWHEADDOWNWARDS TRIANGLE ARROWHEADLEFTWARDS WHITE ARROW WITHIN TRIANGLE ARROWHEADUPWARDS WHITE ARROW WITHIN TRIANGLE ARROWHEADRIGHTWARDS\
    \ WHITE ARROW WITHIN TRIANGLE ARROWHEADDOWNWARDS WHITE ARROW WITHIN TRIANGLE ARROWHEADLEFTWARDS ARROW WITH NOTCHED TAILUPWARDS ARROW WITH NOTCHED TAILRIGHTWARDS ARROW WITH NOTCHED TAILDOWNWARDS ARROW WITH NOTCHED TAILHEAVY ARROW SHAFT WIDTH ONEHEAVY ARROW \
    \SHAFT WIDTH TWO THIRDSHEAVY ARROW SHAFT WIDTH ONE HALFHEAVY ARROW SHAFT WIDTH ONE THIRDLEFTWARDS BOTTOM-SHADED WHITE ARROWRIGHTWARDS BOTTOM SHADED WHITE ARROWLEFTWARDS TOP SHADED WHITE ARROWRIGHTWARDS TOP SHADED WHITE ARROWLEFTWARDS LEFT-SHADED WHITE ARRO\
    \WRIGHTWARDS RIGHT-SHADED WHITE ARROWLEFTWARDS RIGHT-SHADED WHITE ARROWRIGHTWARDS LEFT-SHADED WHITE ARROWLEFTWARDS BACK-TILTED SHADOWED WHITE ARROWRIGHTWARDS BACK-TILTED SHADOWED WHITE ARROWLEFTWARDS FRONT-TILTED SHADOWED WHITE ARROWRIGHTWARDS FRONT-TILTED\
    \ SHADOWED WHITE ARROWWHITE ARROW SHAFT WIDTH ONEWHITE ARROW SHAFT WIDTH TWO THIRDSARROW POINTING UPWARDS THEN NORTH WESTARROW POINTING RIGHTWARDS THEN CURVING SOUTH WESTCIRCLED CROSS FORMEE WITH FOUR DOTSCIRCLED CROSS FORMEE WITH TWO DOTSCIRCLED CROSS FOR\
    \MEELEFT HALF CIRCLE WITH FOUR DOTSLEFT HALF CIRCLE WITH THREE DOTSLEFT HALF CIRCLE WITH TWO DOTSLEFT HALF CIRCLE WITH DOTLEFT HALF CIRCLEDOWNWARD FACING HOOKDOWNWARD FACING NOTCHED HOOKDOWNWARD FACING HOOK WITH DOTDOWNWARD FACING NOTCHED HOOK WITH DOTPINC\
    \HED FINGERSWHITE HEARTBROWN HEARTPINCHING HANDZIPPER-MOUTH FACEMONEY-MOUTH FACEFACE WITH THERMOMETERNERD FACETHINKING FACEFACE WITH HEAD-BANDAGEROBOT FACEHUGGING FACESIGN OF THE HORNSCALL ME HANDRAISED BACK OF HANDLEFT-FACING FISTRIGHT-FACING FISTHANDSHAK\
    \EHAND WITH INDEX AND MIDDLE FINGERS CROSSEDI LOVE YOU HAND SIGNFACE WITH COWBOY HATCLOWN FACENAUSEATED FACEROLLING ON THE FLOOR LAUGHINGDROOLING FACELYING FACEFACE PALMSNEEZING FACEFACE WITH ONE EYEBROW RAISEDGRINNING FACE WITH STAR EYESGRINNING FACE WITH\
    \ ONE LARGE AND ONE SMALL EYEFACE WITH FINGER COVERING CLOSED LIPSSERIOUS FACE WITH SYMBOLS COVERING MOUTHSMILING FACE WITH SMILING EYES AND HAND COVERING MOUTHFACE WITH OPEN MOUTH VOMITINGSHOCKED FACE WITH EXPLODING HEADPREGNANT WOMANBREAST-FEEDINGPALMS U\
    \P TOGETHERSELFIEPRINCEMAN IN TUXEDOMOTHER CHRISTMASSHRUGPERSON DOING CARTWHEELJUGGLINGFENCERMODERN PENTATHLONWRESTLERSWATER POLOHANDBALLDIVING MASKWILTED FLOWERDRUM WITH DRUMSTICKSCLINKING GLASSESTUMBLER GLASSSPOONGOAL NETRIFLEFIRST PLACE MEDALSECOND PLAC\
    \E MEDALTHIRD PLACE MEDALBOXING GLOVEMARTIAL ARTS UNIFORMCURLING STONELACROSSE STICK AND BALLSOFTBALLFLYING DISCCROISSANTAVOCADOCUCUMBERBACONPOTATOCARROTBAGUETTE BREADGREEN SALADSHALLOW PAN OF FOODSTUFFED FLATBREADEGGGLASS OF MILKPEANUTSKIWIFRUITPANCAKESDU\
    \MPLINGFORTUNE COOKIETAKEOUT BOXCHOPSTICKSBOWL WITH SPOONCUP WITH STRAWCOCONUTBROCCOLIPIEPRETZELCUT OF MEATSANDWICHCANNED FOODLEAFY GREENMANGOMOON CAKEBAGELSMILING FACE WITH SMILING EYES AND THREE HEARTSYAWNING FACESMILING FACE WITH TEARFACE WITH PARTY HOR\
    \N AND PARTY HATFACE WITH UNEVEN EYES AND WAVY MOUTHOVERHEATED FACEFREEZING FACENINJADISGUISED FACEFACE HOLDING BACK TEARSFACE WITH PLEADING EYESSARILAB COATGOGGLESHIKING BOOTFLAT SHOECRABLION FACESCORPIONTURKEYUNICORN FACEEAGLEDUCKBATSHARKOWLFOX FACEBUTTE\
    \RFLYDEERGORILLALIZARDRHINOCEROSSHRIMPSQUIDGIRAFFE FACEZEBRA FACEHEDGEHOGSAUROPODT-REXCRICKETKANGAROOLLAMAPEACOCKHIPPOPOTAMUSPARROTRACCOONLOBSTERMOSQUITOMICROBEBADGERSWANMAMMOTHDODOSLOTHOTTERORANGUTANSKUNKFLAMINGOOYSTERBEAVERBISONSEALGUIDE DOGPROBING CANEE\
    \MOJI COMPONENT RED HAIREMOJI COMPONENT CURLY HAIREMOJI COMPONENT BALDEMOJI COMPONENT WHITE HAIRBONELEGFOOTTOOTHSUPERHEROSUPERVILLAINSAFETY VESTEAR WITH HEARING AIDMOTORIZED WHEELCHAIRMANUAL WHEELCHAIRMECHANICAL ARMMECHANICAL LEGCHEESE WEDGECUPCAKESALT SHA\
    \KERBEVERAGE BOXGARLICONIONFALAFELWAFFLEBUTTERMATE DRINKICE CUBEBUBBLE TEATROLLSTANDING PERSONKNEELING PERSONDEAF PERSONFACE WITH MONOCLEADULTCHILDOLDER ADULTBEARDED PERSONPERSON WITH HEADSCARFPERSON IN STEAMY ROOMPERSON CLIMBINGPERSON IN LOTUS POSITIONMAG\
    \EFAIRYVAMPIREMERPERSONELFGENIEZOMBIEBRAINORANGE HEARTBILLED CAPSCARFGLOVESCOATSOCKSRED GIFT ENVELOPEFIRECRACKERJIGSAW PUZZLE PIECETEST TUBEPETRI DISHDNA DOUBLE HELIXCOMPASSABACUSFIRE EXTINGUISHERTOOLBOXBRICKMAGNETLUGGAGELOTION BOTTLESPOOL OF THREADBALL OF\
    \ YARNSAFETY PINTEDDY BEARBROOMBASKETROLL OF PAPERBAR OF SOAPSPONGERECEIPTNAZAR AMULETNEUTRAL CHESS KINGNEUTRAL CHESS QUEENNEUTRAL CHESS ROOKNEUTRAL CHESS BISHOPNEUTRAL CHESS KNIGHTNEUTRAL CHESS PAWNWHITE CHESS KNIGHT ROTATED FORTY-FIVE DEGREESBLACK CHESS \
    \KNIGHT ROTATED FORTY-FIVE DEGREESNEUTRAL CHESS KNIGHT ROTATED FORTY-FIVE DEGREESWHITE CHESS KING ROTATED NINETY DEGREESWHITE CHESS QUEEN ROTATED NINETY DEGREESWHITE CHESS ROOK ROTATED NINETY DEGREESWHITE CHESS BISHOP ROTATED NINETY DEGREESWHITE CHESS KNIG\
    \HT ROTATED NINETY DEGREESWHITE CHESS PAWN ROTATED NINETY DEGREESBLACK CHESS KING ROTATED NINETY DEGREESBLACK CHESS QUEEN ROTATED NINETY DEGREESBLACK CHESS ROOK ROTATED NINETY DEGREESBLACK CHESS BISHOP ROTATED NINETY DEGREESBLACK CHESS KNIGHT ROTATED NINET\
    \Y DEGREESBLACK CHESS PAWN ROTATED NINETY DEGREESNEUTRAL CHESS KING ROTATED NINETY DEGREESNEUTRAL CHESS QUEEN ROTATED NINETY DEGREESNEUTRAL CHESS ROOK ROTATED NINETY DEGREESNEUTRAL CHESS BISHOP ROTATED NINETY DEGREESNEUTRAL CHESS KNIGHT ROTATED NINETY DEGR\
    \EESNEUTRAL CHESS PAWN ROTATED NINETY DEGREESWHITE CHESS KNIGHT ROTATED ONE HUNDRED THIRTY-FIVE DEGREESBLACK CHESS KNIGHT ROTATED ONE HUNDRED THIRTY-FIVE DEGREESNEUTRAL CHESS KNIGHT ROTATED ONE HUNDRED THIRTY-FIVE DEGREESWHITE CHESS TURNED KINGWHITE CHESS \
    \TURNED QUEENWHITE CHESS TURNED ROOKWHITE CHESS TURNED BISHOPWHITE CHESS TURNED KNIGHTWHITE CHESS TURNED PAWNBLACK CHESS TURNED KINGBLACK CHESS TURNED QUEENBLACK CHESS TURNED ROOKBLACK CHESS TURNED BISHOPBLACK CHESS TURNED KNIGHTBLACK CHESS TURNED PAWNNEUT\
    \RAL CHESS TURNED KINGNEUTRAL CHESS TURNED QUEENNEUTRAL CHESS TURNED ROOKNEUTRAL CHESS TURNED BISHOPNEUTRAL CHESS TURNED KNIGHTNEUTRAL CHESS TURNED PAWNWHITE CHESS KNIGHT ROTATED TWO HUNDRED TWENTY-FIVE DEGREESBLACK CHESS KNIGHT ROTATED TWO HUNDRED TWENTY-\
    \FIVE DEGREESNEUTRAL CHESS KNIGHT ROTATED TWO HUNDRED TWENTY-FIVE DEGREESWHITE CHESS KING ROTATED TWO HUNDRED SEVENTY DEGREESWHITE CHESS QUEEN ROTATED TWO HUNDRED SEVENTY DEGREESWHITE CHESS ROOK ROTATED TWO HUNDRED SEVENTY DEGREESWHITE CHESS BISHOP ROTATED\
    \ TWO HUNDRED SEVENTY DEGREESWHITE CHESS KNIGHT ROTATED TWO HUNDRED SEVENTY DEGREESWHITE CHESS PAWN ROTATED TWO HUNDRED SEVENTY DEGREESBLACK CHESS KING ROTATED TWO HUNDRED SEVENTY DEGREESBLACK CHESS QUEEN ROTATED TWO HUNDRED SEVENTY DEGREESBLACK CHESS ROOK\
    \ ROTATED TWO HUNDRED SEVENTY DEGREESBLACK CHESS BISHOP ROTATED TWO HUNDRED SEVENTY DEGREESBLACK CHESS KNIGHT ROTATED TWO HUNDRED SEVENTY DEGREESBLACK CHESS PAWN ROTATED TWO HUNDRED SEVENTY DEGREESNEUTRAL CHESS KING ROTATED TWO HUNDRED SEVENTY DEGREESNEUTR\
    \AL CHESS QUEEN ROTATED TWO HUNDRED SEVENTY DEGREESNEUTRAL CHESS ROOK ROTATED TWO HUNDRED SEVENTY DEGREESNEUTRAL CHESS BISHOP ROTATED TWO HUNDRED SEVENTY DEGREESNEUTRAL CHESS KNIGHT ROTATED TWO HUNDRED SEVENTY DEGREESNEUTRAL CHESS PAWN ROTATED TWO HUNDRED \
    \SEVENTY DEGREESWHITE CHESS KNIGHT ROTATED THREE HUNDRED FIFTEEN DEGREESBLACK CHESS KNIGHT ROTATED THREE HUNDRED FIFTEEN DEGREESNEUTRAL CHESS KNIGHT ROTATED THREE HUNDRED FIFTEEN DEGREESWHITE CHESS EQUIHOPPERBLACK CHESS EQUIHOPPERNEUTRAL CHESS EQUIHOPPERWH\
    \ITE CHESS EQUIHOPPER ROTATED NINETY DEGREESBLACK CHESS EQUIHOPPER ROTATED NINETY DEGREESNEUTRAL CHESS EQUIHOPPER ROTATED NINETY DEGREESWHITE CHESS KNIGHT-QUEENWHITE CHESS KNIGHT-ROOKWHITE CHESS KNIGHT-BISHOPBLACK CHESS KNIGHT-QUEENBLACK CHESS KNIGHT-ROOKB\
    \LACK CHESS KNIGHT-BISHOPXIANGQI RED GENERALXIANGQI RED MANDARINXIANGQI RED ELEPHANTXIANGQI RED HORSEXIANGQI RED CHARIOTXIANGQI RED CANNONXIANGQI RED SOLDIERXIANGQI BLACK GENERALXIANGQI BLACK MANDARINXIANGQI BLACK ELEPHANTXIANGQI BLACK HORSEXIANGQI BLACK C\
    \HARIOTXIANGQI BLACK CANNONXIANGQI BLACK SOLDIERBALLET SHOESONE-PIECE SWIMSUITBRIEFSSHORTSTHONG SANDALLIGHT BLUE HEARTGREY HEARTPINK HEARTDROP OF BLOODADHESIVE BANDAGESTETHOSCOPEX-RAYCRUTCHYO-YOKITEPARACHUTEBOOMERANGMAGIC WANDPINATANESTING DOLLSMARACASFLUT\
    \ERINGED PLANETCHAIRRAZORAXEDIYA LAMPBANJOMILITARY HELMETACCORDIONLONG DRUMCOINCARPENTRY SAWSCREWDRIVERLADDERHOOKMIRRORWINDOWPLUNGERSEWING NEEDLEKNOTBUCKETMOUSE TRAPTOOTHBRUSHHEADSTONEPLACARDROCKMIRROR BALLIDENTIFICATION CARDLOW BATTERYHAMSAFOLDING HAND FA\
    \NHAIR PICKKHANDAFLYWORMBEETLECOCKROACHPOTTED PLANTWOODFEATHERLOTUSCORALEMPTY NESTNEST WITH EGGSHYACINTHJELLYFISHWINGGOOSEANATOMICAL HEARTLUNGSPEOPLE HUGGINGPREGNANT MANPREGNANT PERSONPERSON WITH CROWNMOOSEDONKEYBLUEBERRIESBELL PEPPEROLIVEFLATBREADTAMALEFO\
    \NDUETEAPOTPOURING LIQUIDBEANSJARGINGER ROOTPEA PODMELTING FACESALUTING FACEFACE WITH OPEN EYES AND HAND OVER MOUTHFACE WITH PEEKING EYEFACE WITH DIAGONAL MOUTHDOTTED LINE FACEBITING LIPBUBBLESSHAKING FACEHAND WITH INDEX FINGER AND THUMB CROSSEDRIGHTWARDS \
    \HANDLEFTWARDS HANDPALM DOWN HANDPALM UP HANDINDEX POINTING AT THE VIEWERHEART HANDSLEFTWARDS PUSHING HANDRIGHTWARDS PUSHING HANDBLOCK SEXTANT-1BLOCK SEXTANT-2BLOCK SEXTANT-12BLOCK SEXTANT-3BLOCK SEXTANT-13BLOCK SEXTANT-23BLOCK SEXTANT-123BLOCK SEXTANT-4BL\
    \OCK SEXTANT-14BLOCK SEXTANT-24BLOCK SEXTANT-124BLOCK SEXTANT-34BLOCK SEXTANT-134BLOCK SEXTANT-234BLOCK SEXTANT-1234BLOCK SEXTANT-5BLOCK SEXTANT-15BLOCK SEXTANT-25BLOCK SEXTANT-125BLOCK SEXTANT-35BLOCK SEXTANT-235BLOCK SEXTANT-1235BLOCK SEXTANT-45BLOCK SEX\
    \TANT-145BLOCK SEXTANT-245BLOCK SEXTANT-1245BLOCK SEXTANT-345BLOCK SEXTANT-1345BLOCK SEXTANT-2345BLOCK SEXTANT-12345BLOCK SEXTANT-6BLOCK SEXTANT-16BLOCK SEXTANT-26BLOCK SEXTANT-126BLOCK SEXTANT-36BLOCK SEXTANT-136BLOCK SEXTANT-236BLOCK SEXTANT-1236BLOCK SE\
    \XTANT-46BLOCK SEXTANT-146BLOCK SEXTANT-1246BLOCK SEXTANT-346BLOCK SEXTANT-1346BLOCK SEXTANT-2346BLOCK SEXTANT-12346BLOCK SEXTANT-56BLOCK SEXTANT-156BLOCK SEXTANT-256BLOCK SEXTANT-1256BLOCK SEXTANT-356BLOCK SEXTANT-1356BLOCK SEXTANT-2356BLOCK SEXTANT-12356\
    \BLOCK SEXTANT-456BLOCK SEXTANT-1456BLOCK SEXTANT-2456BLOCK SEXTANT-12456BLOCK SEXTANT-3456BLOCK SEXTANT-13456BLOCK SEXTANT-23456LOWER LEFT BLOCK DIAGONAL LOWER MIDDLE LEFT TO LOWER CENTRELOWER LEFT BLOCK DIAGONAL LOWER MIDDLE LEFT TO LOWER RIGHTLOWER LEFT\
    \ BLOCK DIAGONAL UPPER MIDDLE LEFT TO LOWER CENTRELOWER LEFT BLOCK DIAGONAL UPPER MIDDLE LEFT TO LOWER RIGHTLOWER LEFT BLOCK DIAGONAL UPPER LEFT TO LOWER CENTRELOWER RIGHT BLOCK DIAGONAL UPPER MIDDLE LEFT TO UPPER CENTRELOWER RIGHT BLOCK DIAGONAL UPPER MID\
    \DLE LEFT TO UPPER RIGHTLOWER RIGHT BLOCK DIAGONAL LOWER MIDDLE LEFT TO UPPER CENTRELOWER RIGHT BLOCK DIAGONAL LOWER MIDDLE LEFT TO UPPER RIGHTLOWER RIGHT BLOCK DIAGONAL LOWER LEFT TO UPPER CENTRELOWER RIGHT BLOCK DIAGONAL LOWER MIDDLE LEFT TO UPPER MIDDLE\
    \ RIGHTLOWER RIGHT BLOCK DIAGONAL LOWER CENTRE TO LOWER MIDDLE RIGHTLOWER RIGHT BLOCK DIAGONAL LOWER LEFT TO LOWER MIDDLE RIGHTLOWER RIGHT BLOCK DIAGONAL LOWER CENTRE TO UPPER MIDDLE RIGHTLOWER RIGHT BLOCK DIAGONAL LOWER LEFT TO UPPER MIDDLE RIGHTLOWER RIG\
    \HT BLOCK DIAGONAL LOWER CENTRE TO UPPER RIGHTLOWER LEFT BLOCK DIAGONAL UPPER CENTRE TO UPPER MIDDLE RIGHTLOWER LEFT BLOCK DIAGONAL UPPER LEFT TO UPPER MIDDLE RIGHTLOWER LEFT BLOCK DIAGONAL UPPER CENTRE TO LOWER MIDDLE RIGHTLOWER LEFT BLOCK DIAGONAL UPPER \
    \LEFT TO LOWER MIDDLE RIGHTLOWER LEFT BLOCK DIAGONAL UPPER CENTRE TO LOWER RIGHTLOWER LEFT BLOCK DIAGONAL UPPER MIDDLE LEFT TO LOWER MIDDLE RIGHTUPPER RIGHT BLOCK DIAGONAL LOWER MIDDLE LEFT TO LOWER CENTREUPPER RIGHT BLOCK DIAGONAL LOWER MIDDLE LEFT TO LOW\
    \ER RIGHTUPPER RIGHT BLOCK DIAGONAL UPPER MIDDLE LEFT TO LOWER CENTREUPPER RIGHT BLOCK DIAGONAL UPPER MIDDLE LEFT TO LOWER RIGHTUPPER RIGHT BLOCK DIAGONAL UPPER LEFT TO LOWER CENTREUPPER LEFT BLOCK DIAGONAL UPPER MIDDLE LEFT TO UPPER CENTREUPPER LEFT BLOCK\
    \ DIAGONAL UPPER MIDDLE LEFT TO UPPER RIGHTUPPER LEFT BLOCK DIAGONAL LOWER MIDDLE LEFT TO UPPER CENTREUPPER LEFT BLOCK DIAGONAL LOWER MIDDLE LEFT TO UPPER RIGHTUPPER LEFT BLOCK DIAGONAL LOWER LEFT TO UPPER CENTREUPPER LEFT BLOCK DIAGONAL LOWER MIDDLE LEFT \
    \TO UPPER MIDDLE RIGHTUPPER LEFT BLOCK DIAGONAL LOWER CENTRE TO LOWER MIDDLE RIGHTUPPER LEFT BLOCK DIAGONAL LOWER LEFT TO LOWER MIDDLE RIGHTUPPER LEFT BLOCK DIAGONAL LOWER CENTRE TO UPPER MIDDLE RIGHTUPPER LEFT BLOCK DIAGONAL LOWER LEFT TO UPPER MIDDLE RIG\
    \HTUPPER LEFT BLOCK DIAGONAL LOWER CENTRE TO UPPER RIGHTUPPER RIGHT BLOCK DIAGONAL UPPER CENTRE TO UPPER MIDDLE RIGHTUPPER RIGHT BLOCK DIAGONAL UPPER LEFT TO UPPER MIDDLE RIGHTUPPER RIGHT BLOCK DIAGONAL UPPER CENTRE TO LOWER MIDDLE RIGHTUPPER RIGHT BLOCK D\
    \IAGONAL UPPER LEFT TO LOWER MIDDLE RIGHTUPPER RIGHT BLOCK DIAGONAL UPPER CENTRE TO LOWER RIGHTUPPER RIGHT BLOCK DIAGONAL UPPER MIDDLE LEFT TO LOWER MIDDLE RIGHTUPPER AND RIGHT AND LOWER TRIANGULAR THREE QUARTERS BLOCKLEFT AND LOWER AND RIGHT TRIANGULAR TH\
    \REE QUARTERS BLOCKUPPER AND LEFT AND LOWER TRIANGULAR THREE QUARTERS BLOCKLEFT AND UPPER AND RIGHT TRIANGULAR THREE QUARTERS BLOCKLEFT TRIANGULAR ONE QUARTER BLOCKUPPER TRIANGULAR ONE QUARTER BLOCKRIGHT TRIANGULAR ONE QUARTER BLOCKLOWER TRIANGULAR ONE QUA\
    \RTER BLOCKVERTICAL ONE EIGHTH BLOCK-2VERTICAL ONE EIGHTH BLOCK-3VERTICAL ONE EIGHTH BLOCK-4VERTICAL ONE EIGHTH BLOCK-5VERTICAL ONE EIGHTH BLOCK-6VERTICAL ONE EIGHTH BLOCK-7HORIZONTAL ONE EIGHTH BLOCK-2HORIZONTAL ONE EIGHTH BLOCK-3HORIZONTAL ONE EIGHTH BLO\
    \CK-4HORIZONTAL ONE EIGHTH BLOCK-5HORIZONTAL ONE EIGHTH BLOCK-6HORIZONTAL ONE EIGHTH BLOCK-7LEFT AND LOWER ONE EIGHTH BLOCKLEFT AND UPPER ONE EIGHTH BLOCKRIGHT AND UPPER ONE EIGHTH BLOCKRIGHT AND LOWER ONE EIGHTH BLOCKUPPER AND LOWER ONE EIGHTH BLOCKHORIZO\
    \NTAL ONE EIGHTH BLOCK-1358UPPER ONE QUARTER BLOCKUPPER THREE EIGHTHS BLOCKUPPER FIVE EIGHTHS BLOCKUPPER THREE QUARTERS BLOCKUPPER SEVEN EIGHTHS BLOCKRIGHT ONE QUARTER BLOCKRIGHT THREE EIGHTHS BLOCKRIGHT FIVE EIGHTHS BLOCKRIGHT THREE QUARTERS BLOCKRIGHT SE\
    \VEN EIGHTHS BLOCKLEFT HALF MEDIUM SHADERIGHT HALF MEDIUM SHADEUPPER HALF MEDIUM SHADELOWER HALF MEDIUM SHADEINVERSE MEDIUM SHADEUPPER HALF BLOCK AND LOWER HALF INVERSE MEDIUM SHADEUPPER HALF INVERSE MEDIUM SHADE AND LOWER HALF BLOCKLEFT HALF INVERSE MEDIU\
    \M SHADE AND RIGHT HALF BLOCKCHECKER BOARD FILLINVERSE CHECKER BOARD FILLHEAVY HORIZONTAL FILLUPPER LEFT TO LOWER RIGHT FILLUPPER RIGHT TO LOWER LEFT FILLUPPER AND LOWER TRIANGULAR HALF BLOCKLEFT AND RIGHT TRIANGULAR HALF BLOCKUPPER LEFT TRIANGULAR MEDIUM \
    \SHADEUPPER RIGHT TRIANGULAR MEDIUM SHADELOWER RIGHT TRIANGULAR MEDIUM SHADELOWER LEFT TRIANGULAR MEDIUM SHADEBOX DRAWINGS LIGHT DIAGONAL UPPER CENTRE TO MIDDLE LEFTBOX DRAWINGS LIGHT DIAGONAL UPPER CENTRE TO MIDDLE RIGHTBOX DRAWINGS LIGHT DIAGONAL MIDDLE \
    \LEFT TO LOWER CENTREBOX DRAWINGS LIGHT DIAGONAL MIDDLE RIGHT TO LOWER CENTREBOX DRAWINGS LIGHT DIAGONAL UPPER CENTRE TO MIDDLE LEFT TO LOWER CENTREBOX DRAWINGS LIGHT DIAGONAL UPPER CENTRE TO MIDDLE RIGHT TO LOWER CENTREBOX DRAWINGS LIGHT DIAGONAL MIDDLE L\
    \EFT TO LOWER CENTRE TO MIDDLE RIGHTBOX DRAWINGS LIGHT DIAGONAL MIDDLE LEFT TO UPPER CENTRE TO MIDDLE RIGHTBOX DRAWINGS LIGHT DIAGONAL UPPER CENTRE TO MIDDLE LEFT AND MIDDLE RIGHT TO LOWER CENTREBOX DRAWINGS LIGHT DIAGONAL UPPER CENTRE TO MIDDLE RIGHT AND \
    \MIDDLE LEFT TO LOWER CENTREBOX DRAWINGS LIGHT DIAGONAL UPPER CENTRE TO MIDDLE RIGHT TO LOWER CENTRE TO MIDDLE LEFTBOX DRAWINGS LIGHT DIAGONAL UPPER CENTRE TO MIDDLE LEFT TO LOWER CENTRE TO MIDDLE RIGHTBOX DRAWINGS LIGHT DIAGONAL MIDDLE LEFT TO UPPER CENTR\
    \E TO MIDDLE RIGHT TO LOWER CENTREBOX DRAWINGS LIGHT DIAGONAL MIDDLE RIGHT TO UPPER CENTRE TO MIDDLE LEFT TO LOWER CENTREBOX DRAWINGS LIGHT DIAGONAL DIAMONDBOX DRAWINGS LIGHT HORIZONTAL WITH VERTICAL STROKEARROWHEAD-SHAPED POINTERINVERSE CHECK MARKLEFT HAL\
    \F RUNNING MANRIGHT HALF RUNNING MANINVERSE DOWNWARDS ARROW WITH TIP LEFTWARDSLEFTWARDS ARROW AND UPPER AND LOWER ONE EIGHTH BLOCKRIGHTWARDS ARROW AND UPPER AND LOWER ONE EIGHTH BLOCKDOWNWARDS ARROW AND RIGHT ONE EIGHTH BLOCKUPWARDS ARROW AND RIGHT ONE EIG\
    \HTH BLOCKLEFT HALF FOLDERRIGHT HALF FOLDERVOIDED GREEK CROSSRIGHT OPEN SQUARED DOTNEGATIVE DIAGONAL CROSSNEGATIVE DIAGONAL MIDDLE RIGHT TO LOWER CENTRENEGATIVE DIAGONAL DIAMONDWHITE HEAVY SALTIRE WITH ROUNDED CORNERSLEFT THIRD WHITE RIGHT POINTING INDEXMI\
    \DDLE THIRD WHITE RIGHT POINTING INDEXRIGHT THIRD WHITE RIGHT POINTING INDEXNEGATIVE SQUARED QUESTION MARKSTICK FIGURESTICK FIGURE WITH ARMS RAISEDSTICK FIGURE LEANING LEFTSTICK FIGURE LEANING RIGHTSTICK FIGURE WITH DRESSWHITE UP-POINTING CHEVRONSEGMENTED \
    \DIGIT ZEROSEGMENTED DIGIT ONESEGMENTED DIGIT TWOSEGMENTED DIGIT THREESEGMENTED DIGIT FOURSEGMENTED DIGIT FIVESEGMENTED DIGIT SIXSEGMENTED DIGIT SEVENSEGMENTED DIGIT EIGHTSEGMENTED DIGIT NINELANGUAGE TAGTAG SPACETAG EXCLAMATION MARKTAG QUOTATION MARKTAG NU\
    \MBER SIGNTAG DOLLAR SIGNTAG PERCENT SIGNTAG AMPERSANDTAG APOSTROPHETAG LEFT PARENTHESISTAG RIGHT PARENTHESISTAG ASTERISKTAG PLUS SIGNTAG COMMATAG HYPHEN-MINUSTAG FULL STOPTAG SOLIDUSTAG DIGIT ZEROTAG DIGIT ONETAG DIGIT TWOTAG DIGIT THREETAG DIGIT FOURTAG \
    \DIGIT FIVETAG DIGIT SIXTAG DIGIT SEVENTAG DIGIT EIGHTTAG DIGIT NINETAG COLONTAG SEMICOLONTAG LESS-THAN SIGNTAG EQUALS SIGNTAG GREATER-THAN SIGNTAG QUESTION MARKTAG COMMERCIAL ATTAG LATIN CAPITAL LETTER ATAG LATIN CAPITAL LETTER BTAG LATIN CAPITAL LETTER C\
    \TAG LATIN CAPITAL LETTER DTAG LATIN CAPITAL LETTER ETAG LATIN CAPITAL LETTER FTAG LATIN CAPITAL LETTER GTAG LATIN CAPITAL LETTER HTAG LATIN CAPITAL LETTER ITAG LATIN CAPITAL LETTER JTAG LATIN CAPITAL LETTER KTAG LATIN CAPITAL LETTER LTAG LATIN CAPITAL LET\
    \TER MTAG LATIN CAPITAL LETTER NTAG LATIN CAPITAL LETTER OTAG LATIN CAPITAL LETTER PTAG LATIN CAPITAL LETTER QTAG LATIN CAPITAL LETTER RTAG LATIN CAPITAL LETTER STAG LATIN CAPITAL LETTER TTAG LATIN CAPITAL LETTER UTAG LATIN CAPITAL LETTER VTAG LATIN CAPITA\
    \L LETTER WTAG LATIN CAPITAL LETTER XTAG LATIN CAPITAL LETTER YTAG LATIN CAPITAL LETTER ZTAG LEFT SQUARE BRACKETTAG REVERSE SOLIDUSTAG RIGHT SQUARE BRACKETTAG CIRCUMFLEX ACCENTTAG LOW LINETAG GRAVE ACCENTTAG LATIN SMALL LETTER ATAG LATIN SMALL LETTER BTAG \
    \LATIN SMALL LETTER CTAG LATIN SMALL LETTER DTAG LATIN SMALL LETTER ETAG LATIN SMALL LETTER FTAG LATIN SMALL LETTER GTAG LATIN SMALL LETTER HTAG LATIN SMALL LETTER ITAG LATIN SMALL LETTER JTAG LATIN SMALL LETTER KTAG LATIN SMALL LETTER LTAG LATIN SMALL LET\
    \TER MTAG LATIN SMALL LETTER NTAG LATIN SMALL LETTER OTAG LATIN SMALL LETTER PTAG LATIN SMALL LETTER QTAG LATIN SMALL LETTER RTAG LATIN SMALL LETTER STAG LATIN SMALL LETTER TTAG LATIN SMALL LETTER UTAG LATIN SMALL LETTER VTAG LATIN SMALL LETTER WTAG LATIN \
    \SMALL LETTER XTAG LATIN SMALL LETTER YTAG LATIN SMALL LETTER ZTAG LEFT CURLY BRACKETTAG VERTICAL LINETAG RIGHT CURLY BRACKETTAG TILDECANCEL TAGVARIATION SELECTOR-17VARIATION SELECTOR-18VARIATION SELECTOR-19VARIATION SELECTOR-20VARIATION SELECTOR-21VARIATI\
    \ON SELECTOR-22VARIATION SELECTOR-23VARIATION SELECTOR-24VARIATION SELECTOR-25VARIATION SELECTOR-26VARIATION SELECTOR-27VARIATION SELECTOR-28VARIATION SELECTOR-29VARIATION SELECTOR-30VARIATION SELECTOR-31VARIATION SELECTOR-32VARIATION SELECTOR-33VARIATION \
    \SELECTOR-34VARIATION SELECTOR-35VARIATION SELECTOR-36VARIATION SELECTOR-37VARIATION SELECTOR-38VARIATION SELECTOR-39VARIATION SELECTOR-40VARIATION SELECTOR-41VARIATION SELECTOR-42VARIATION SELECTOR-43VARIATION SELECTOR-44VARIATION SELECTOR-45VARIATION SEL\
    \ECTOR-46VARIATION SELECTOR-47VARIATION SELECTOR-48VARIATION SELECTOR-49VARIATION SELECTOR-50VARIATION SELECTOR-51VARIATION SELECTOR-52VARIATION SELECTOR-53VARIATION SELECTOR-54VARIATION SELECTOR-55VARIATION SELECTOR-56VARIATION SELECTOR-57VARIATION SELECT\
    \OR-58VARIATION SELECTOR-59VARIATION SELECTOR-60VARIATION SELECTOR-61VARIATION SELECTOR-62VARIATION SELECTOR-63VARIATION SELECTOR-64VARIATION SELECTOR-65VARIATION SELECTOR-66VARIATION SELECTOR-67VARIATION SELECTOR-68VARIATION SELECTOR-69VARIATION SELECTOR-\
    \70VARIATION SELECTOR-71VARIATION SELECTOR-72VARIATION SELECTOR-73VARIATION SELECTOR-74VARIATION SELECTOR-75VARIATION SELECTOR-76VARIATION SELECTOR-77VARIATION SELECTOR-78VARIATION SELECTOR-79VARIATION SELECTOR-80VARIATION SELECTOR-81VARIATION SELECTOR-82V\
    \ARIATION SELECTOR-83VARIATION SELECTOR-84VARIATION SELECTOR-85VARIATION SELECTOR-86VARIATION SELECTOR-87VARIATION SELECTOR-88VARIATION SELECTOR-89VARIATION SELECTOR-90VARIATION SELECTOR-91VARIATION SELECTOR-92VARIATION SELECTOR-93VARIATION SELECTOR-94VARI\
    \ATION SELECTOR-95VARIATION SELECTOR-96VARIATION SELECTOR-97VARIATION SELECTOR-98VARIATION SELECTOR-99VARIATION SELECTOR-100VARIATION SELECTOR-101VARIATION SELECTOR-102VARIATION SELECTOR-103VARIATION SELECTOR-104VARIATION SELECTOR-105VARIATION SELECTOR-106\
    \VARIATION SELECTOR-107VARIATION SELECTOR-108VARIATION SELECTOR-109VARIATION SELECTOR-110VARIATION SELECTOR-111VARIATION SELECTOR-112VARIATION SELECTOR-113VARIATION SELECTOR-114VARIATION SELECTOR-115VARIATION SELECTOR-116VARIATION SELECTOR-117VARIATION SEL\
    \ECTOR-118VARIATION SELECTOR-119VARIATION SELECTOR-120VARIATION SELECTOR-121VARIATION SELECTOR-122VARIATION SELECTOR-123VARIATION SELECTOR-124VARIATION SELECTOR-125VARIATION SELECTOR-126VARIATION SELECTOR-127VARIATION SELECTOR-128VARIATION SELECTOR-129VARI\
    \ATION SELECTOR-130VARIATION SELECTOR-131VARIATION SELECTOR-132VARIATION SELECTOR-133VARIATION SELECTOR-134VARIATION SELECTOR-135VARIATION SELECTOR-136VARIATION SELECTOR-137VARIATION SELECTOR-138VARIATION SELECTOR-139VARIATION SELECTOR-140VARIATION SELECTO\
    \R-141VARIATION SELECTOR-142VARIATION SELECTOR-143VARIATION SELECTOR-144VARIATION SELECTOR-145VARIATION SELECTOR-146VARIATION SELECTOR-147VARIATION SELECTOR-148VARIATION SELECTOR-149VARIATION SELECTOR-150VARIATION SELECTOR-151VARIATION SELECTOR-152VARIATIO\
    \N SELECTOR-153VARIATION SELECTOR-154VARIATION SELECTOR-155VARIATION SELECTOR-156VARIATION SELECTOR-157VARIATION SELECTOR-158VARIATION SELECTOR-159VARIATION SELECTOR-160VARIATION SELECTOR-161VARIATION SELECTOR-162VARIATION SELECTOR-163VARIATION SELECTOR-16\
    \4VARIATION SELECTOR-165VARIATION SELECTOR-166VARIATION SELECTOR-167VARIATION SELECTOR-168VARIATION SELECTOR-169VARIATION SELECTOR-170VARIATION SELECTOR-171VARIATION SELECTOR-172VARIATION SELECTOR-173VARIATION SELECTOR-174VARIATION SELECTOR-175VARIATION SE\
    \LECTOR-176VARIATION SELECTOR-177VARIATION SELECTOR-178VARIATION SELECTOR-179VARIATION SELECTOR-180VARIATION SELECTOR-181VARIATION SELECTOR-182VARIATION SELECTOR-183VARIATION SELECTOR-184VARIATION SELECTOR-185VARIATION SELECTOR-186VARIATION SELECTOR-187VAR\
    \IATION SELECTOR-188VARIATION SELECTOR-189VARIATION SELECTOR-190VARIATION SELECTOR-191VARIATION SELECTOR-192VARIATION SELECTOR-193VARIATION SELECTOR-194VARIATION SELECTOR-195VARIATION SELECTOR-196VARIATION SELECTOR-197VARIATION SELECTOR-198VARIATION SELECT\
    \OR-199VARIATION SELECTOR-200VARIATION SELECTOR-201VARIATION SELECTOR-202VARIATION SELECTOR-203VARIATION SELECTOR-204VARIATION SELECTOR-205VARIATION SELECTOR-206VARIATION SELECTOR-207VARIATION SELECTOR-208VARIATION SELECTOR-209VARIATION SELECTOR-210VARIATI\
    \ON SELECTOR-211VARIATION SELECTOR-212VARIATION SELECTOR-213VARIATION SELECTOR-214VARIATION SELECTOR-215VARIATION SELECTOR-216VARIATION SELECTOR-217VARIATION SELECTOR-218VARIATION SELECTOR-219VARIATION SELECTOR-220VARIATION SELECTOR-221VARIATION SELECTOR-2\
    \22VARIATION SELECTOR-223VARIATION SELECTOR-224VARIATION SELECTOR-225VARIATION SELECTOR-226VARIATION SELECTOR-227VARIATION SELECTOR-228VARIATION SELECTOR-229VARIATION SELECTOR-230VARIATION SELECTOR-231VARIATION SELECTOR-232VARIATION SELECTOR-233VARIATION S\
    \ELECTOR-234VARIATION SELECTOR-235VARIATION SELECTOR-236VARIATION SELECTOR-237VARIATION SELECTOR-238VARIATION SELECTOR-239VARIATION SELECTOR-240VARIATION SELECTOR-241VARIATION SELECTOR-242VARIATION SELECTOR-243VARIATION SELECTOR-244VARIATION SELECTOR-245VA\
    \RIATION SELECTOR-246VARIATION SELECTOR-247VARIATION SELECTOR-248VARIATION SELECTOR-249VARIATION SELECTOR-250VARIATION SELECTOR-251VARIATION SELECTOR-252VARIATION SELECTOR-253VARIATION SELECTOR-254VARIATION SELECTOR-255VARIATION SELECTOR-256"#

offsetsBitmap :: Ptr Int32
offsetsBitmap = Ptr
    "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\5\0\0\0\16\5\0\0\14\21\0\0\11\35\0\0\11\46\0\0\12\57\0\0\9\69\0\0\10\78\0\0\16\88\0\0\17\104\0\0\8\121\0\0\9\129\0\0\5\138\0\0\12\143\0\0\9\155\0\0\7\164\0\0\10\171\0\0\9\181\0\0\9\190\0\0\11\199\0\0\10\210\0\0\10\220\0\0\9\230\0\0\11\239\0\0\11\250\0\0\10\5\1\0\5\15\1\0\9\20\1\0\14\29\1\0\11\43\1\0\17\54\1\0\13\71\1\
    \\0\13\84\1\0\22\97\1\0\22\119\1\0\22\141\1\0\22\163\1\0\22\185\1\0\22\207\1\0\22\229\1\0\22\251\1\0\22\17\2\0\22\39\2\0\22\61\2\0\22\83\2\0\22\105\2\0\22\127\2\0\22\149\2\0\22\171\2\0\22\193\2\0\22\215\2\0\22\237\2\0\22\3\3\0\22\25\3\0\22\47\3\0\22\69\3\0\22\91\3\0\22\113\3\0\22\135\3\0\19\157\3\0\15\176\3\0\20\191\3\0\17\211\3\0\8\228\3\0\12\236\3\0\20\248\3\0\20\12\4\0\20\32\4\0\20\52\4\0\20\72\4\0\20\92\4\0\20\112\4\0\20\132\4\0\20\152\4\0\20\172\4\0\20\192\4\0\20\212\4\0\20\232\4\0\20\252\4\0\20\16\5\0\20\36\5\0\20\56\5\0\20\76\5\0\20\96\5\0\20\116\5\0\20\136\5\0\20\156\5\0\20\176\5\0\20\196\5\0\20\216\5\0\20\236\5\0\18\0\6\0\13\18\6\0\19\31\6\0\5\50\6\0\0\0\
    \\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\14\55\6\0\25\69\6\0\9\94\6\0\10\103\6\0\13\113\6\0\8\126\6\0\10\134\6\0\12\144\6\0\9\156\6\0\14\165\6\0\26\179\6\0\41\205\6\0\8\246\6\0\11\254\6\0\15\9\7\0\6\24\7\0\11\30\7\0\15\41\7\0\15\56\7\0\17\71\7\0\12\88\7\0\10\100\7\0\12\110\7\0\10\122\7\0\7\132\7\0\15\139\7\0\27\154\7\0\42\181\7\0\27\223\7\0\24\250\7\0\30\18\8\0\22\
    \\48\8\0\33\70\8\0\33\103\8\0\38\136\8\0\33\174\8\0\37\207\8\0\38\244\8\0\23\26\9\0\35\49\9\0\33\84\9\0\33\117\9\0\38\150\9\0\37\188\9\0\33\225\9\0\33\2\10\0\38\35\10\0\37\73\10\0\24\110\10\0\33\134\10\0\33\167\10\0\33\200\10\0\38\233\10\0\33\15\11\0\37\48\11\0\19\85\11\0\34\104\11\0\33\138\11\0\33\171\11\0\38\204\11\0\37\242\11\0\33\23\12\0\26\56\12\0\26\82\12\0\31\108\12\0\31\139\12\0\36\170\12\0\31\206\12\0\35\237\12\0\36\16\13\0\21\52\13\0\33\73\13\0\31\106\13\0\31\137\13\0\36\168\13\0\35\204\13\0\31\239\13\0\31\14\14\0\36\45\14\0\35\81\14\0\22\116\14\0\31\138\14\0\31\169\14\0\31\200\14\0\36\231\14\0\31\11\15\0\35\42\15\0\13\77\15\0\32\90\15\0\31\122\15\0\31\153\15\0\36\184\15\0\35\220\15\0\31\255\15\0\24\30\16\0\
    \\35\54\16\0\34\89\16\0\32\123\16\0\33\155\16\0\31\188\16\0\34\219\16\0\32\253\16\0\33\29\17\0\31\62\17\0\38\93\17\0\36\131\17\0\37\167\17\0\35\204\17\0\33\239\17\0\31\16\18\0\33\47\18\0\31\80\18\0\34\111\18\0\32\145\18\0\34\177\18\0\32\211\18\0\33\243\18\0\31\20\19\0\37\51\19\0\35\88\19\0\34\123\19\0\32\157\19\0\33\189\19\0\31\222\19\0\38\253\19\0\36\35\20\0\33\71\20\0\31\104\20\0\37\135\20\0\35\172\20\0\35\207\20\0\33\242\20\0\38\19\21\0\36\57\21\0\34\93\21\0\32\127\21\0\33\159\21\0\31\192\21\0\34\223\21\0\32\1\22\0\33\33\22\0\31\66\22\0\34\97\22\0\32\131\22\0\37\163\22\0\28\200\22\0\25\228\22\0\23\253\22\0\38\20\23\0\36\58\23\0\35\94\23\0\33\129\23\0\22\162\23\0\33\184\23\0\31\217\23\0\35\248\23\0\33\27\24\0\33\60\24\0\31\93\24\
    \\0\38\124\24\0\36\162\24\0\34\198\24\0\32\232\24\0\33\8\25\0\31\41\25\0\35\72\25\0\33\107\25\0\33\140\25\0\31\173\25\0\43\204\25\0\24\247\25\0\22\15\26\0\34\37\26\0\32\71\26\0\33\103\26\0\31\136\26\0\40\167\26\0\38\207\26\0\25\245\26\0\23\14\27\0\33\37\27\0\31\70\27\0\35\101\27\0\33\136\27\0\33\169\27\0\31\202\27\0\33\233\27\0\31\10\28\0\38\41\28\0\36\79\28\0\35\115\28\0\33\150\28\0\33\183\28\0\31\216\28\0\35\247\28\0\33\26\29\0\33\59\29\0\31\92\29\0\34\123\29\0\32\157\29\0\33\189\29\0\31\222\29\0\34\253\29\0\32\31\30\0\33\63\30\0\31\96\30\0\38\127\30\0\36\165\30\0\40\201\30\0\38\241\30\0\34\23\31\0\32\57\31\0\38\89\31\0\36\127\31\0\38\163\31\0\36\201\31\0\37\237\31\0\33\18\32\0\31\51\32\0\37\82\32\0\35\119\32\0\33\154\32\0\31\187\
    \\32\0\25\218\32\0\32\243\32\0\32\19\33\0\34\51\33\0\32\85\33\0\29\117\33\0\27\146\33\0\27\173\33\0\32\200\33\0\30\232\33\0\30\6\34\0\32\36\34\0\34\68\34\0\32\102\34\0\31\134\34\0\31\165\34\0\26\196\34\0\27\222\34\0\32\249\34\0\30\25\35\0\32\55\35\0\26\87\35\0\21\113\35\0\25\134\35\0\34\159\35\0\32\193\35\0\30\225\35\0\29\255\35\0\37\28\36\0\29\65\36\0\37\94\36\0\40\131\36\0\40\171\36\0\32\211\36\0\30\243\36\0\23\17\37\0\21\40\37\0\32\61\37\0\30\93\37\0\15\123\37\0\29\138\37\0\27\167\37\0\24\194\37\0\30\218\37\0\38\248\37\0\32\30\38\0\30\62\38\0\42\92\38\0\32\134\38\0\30\166\38\0\28\196\38\0\32\224\38\0\32\0\39\0\30\32\39\0\34\62\39\0\32\96\39\0\24\128\39\0\33\152\39\0\31\185\39\0\32\216\39\0\28\248\39\0\30\20\40\0\28\50\40\0\46\
    \\78\40\0\17\124\40\0\25\141\40\0\26\166\40\0\27\192\40\0\28\219\40\0\34\247\40\0\53\25\41\0\32\78\41\0\23\110\41\0\42\133\41\0\21\175\41\0\23\196\41\0\42\219\41\0\21\5\42\0\33\26\42\0\31\59\42\0\33\90\42\0\31\123\42\0\33\154\42\0\31\187\42\0\33\218\42\0\31\251\42\0\48\26\43\0\46\74\43\0\47\120\43\0\45\167\43\0\47\212\43\0\45\3\44\0\47\48\44\0\45\95\44\0\27\140\44\0\48\167\44\0\46\215\44\0\48\5\45\0\46\53\45\0\35\99\45\0\33\134\45\0\34\167\45\0\32\201\45\0\33\233\45\0\31\10\46\0\33\41\46\0\31\74\46\0\34\105\46\0\32\139\46\0\45\171\46\0\43\216\46\0\35\3\47\0\33\38\47\0\31\71\47\0\23\102\47\0\42\125\47\0\21\167\47\0\33\188\47\0\31\221\47\0\26\252\47\0\25\22\48\0\33\47\48\0\31\80\48\0\48\111\48\0\46\159\48\0\34\205\48\0\32\239\48\0\
    \\44\15\49\0\42\59\49\0\40\101\49\0\38\141\49\0\42\179\49\0\40\221\49\0\40\5\50\0\38\45\50\0\42\83\50\0\40\125\50\0\40\165\50\0\38\205\50\0\42\243\50\0\40\29\51\0\40\69\51\0\38\109\51\0\42\147\51\0\40\189\51\0\40\229\51\0\38\13\52\0\42\51\52\0\40\93\52\0\40\133\52\0\38\173\52\0\42\211\52\0\40\253\52\0\39\37\53\0\37\76\53\0\39\113\53\0\37\152\53\0\25\189\53\0\23\214\53\0\33\237\53\0\31\14\54\0\42\45\54\0\30\87\54\0\23\117\54\0\21\140\54\0\32\161\54\0\30\193\54\0\37\223\54\0\35\4\55\0\35\39\55\0\33\74\55\0\48\107\55\0\46\155\55\0\44\201\55\0\42\245\55\0\37\31\56\0\35\68\56\0\48\103\56\0\46\151\56\0\34\197\56\0\32\231\56\0\30\7\57\0\30\37\57\0\30\67\57\0\28\97\57\0\29\125\57\0\29\154\57\0\34\183\57\0\34\217\57\0\32\251\57\0\31\27\58\
    \\0\43\58\58\0\36\101\58\0\36\137\58\0\33\173\58\0\31\206\58\0\34\237\58\0\26\15\59\0\29\41\59\0\34\70\59\0\32\104\59\0\34\136\59\0\32\170\59\0\43\202\59\0\35\245\59\0\34\24\60\0\32\58\60\0\34\90\60\0\32\124\60\0\27\156\60\0\24\183\60\0\31\207\60\0\30\238\60\0\25\12\61\0\30\37\61\0\30\67\61\0\30\97\61\0\29\127\61\0\24\156\61\0\34\180\61\0\25\214\61\0\34\239\61\0\44\17\62\0\41\61\62\0\40\102\62\0\30\142\62\0\27\172\62\0\28\199\62\0\24\227\62\0\28\251\62\0\27\23\63\0\30\50\63\0\33\80\63\0\32\113\63\0\23\145\63\0\28\168\63\0\38\196\63\0\30\234\63\0\40\8\64\0\23\48\64\0\27\71\64\0\41\98\64\0\30\139\64\0\35\169\64\0\40\204\64\0\28\244\64\0\27\16\65\0\29\43\65\0\31\72\65\0\22\103\65\0\27\125\65\0\41\152\65\0\37\193\65\0\34\230\65\0\30\8\
    \\66\0\34\38\66\0\43\72\66\0\28\115\66\0\37\143\66\0\30\180\66\0\22\210\66\0\49\232\66\0\37\25\67\0\32\62\67\0\27\94\67\0\40\121\67\0\24\161\67\0\26\185\67\0\30\211\67\0\27\241\67\0\27\12\68\0\27\39\68\0\28\66\68\0\40\94\68\0\30\134\68\0\22\164\68\0\32\186\68\0\25\218\68\0\40\243\68\0\34\27\69\0\24\61\69\0\27\85\69\0\28\112\69\0\32\140\69\0\38\172\69\0\28\210\69\0\38\238\69\0\27\20\70\0\28\47\70\0\30\75\70\0\37\105\70\0\46\142\70\0\29\188\70\0\31\217\70\0\39\248\70\0\29\31\71\0\31\60\71\0\39\91\71\0\31\130\71\0\29\161\71\0\29\190\71\0\32\219\71\0\32\251\71\0\41\27\72\0\50\68\72\0\23\118\72\0\33\141\72\0\23\174\72\0\23\197\72\0\30\220\72\0\40\250\72\0\40\34\73\0\23\74\73\0\23\97\73\0\21\120\73\0\28\141\73\0\28\169\73\0\26\197\73\0\30\
    \\223\73\0\31\253\73\0\30\28\74\0\28\58\74\0\37\86\74\0\30\123\74\0\31\153\74\0\28\184\74\0\30\212\74\0\33\242\74\0\5\19\75\0\29\24\75\0\22\53\75\0\28\75\75\0\28\103\75\0\33\131\75\0\26\164\75\0\32\190\75\0\32\222\75\0\32\254\75\0\37\30\76\0\39\67\76\0\38\106\76\0\23\144\76\0\25\167\76\0\25\192\76\0\26\217\76\0\5\243\76\0\9\248\76\0\10\1\77\0\6\11\77\0\11\17\77\0\19\28\77\0\27\47\77\0\28\74\77\0\27\102\77\0\23\129\77\0\23\152\77\0\23\175\77\0\43\198\77\0\35\241\77\0\29\20\78\0\28\49\78\0\28\77\78\0\34\105\78\0\39\139\78\0\40\178\78\0\23\218\78\0\27\241\78\0\33\12\79\0\34\45\79\0\32\79\79\0\34\111\79\0\35\145\79\0\24\180\79\0\35\204\79\0\42\239\79\0\42\25\80\0\25\67\80\0\28\92\80\0\31\120\80\0\29\151\80\0\30\180\80\0\28\210\80\0\
    \\21\238\80\0\26\3\81\0\30\29\81\0\22\59\81\0\22\81\81\0\27\103\81\0\15\130\81\0\16\145\81\0\18\161\81\0\15\179\81\0\19\194\81\0\19\213\81\0\20\232\81\0\20\252\81\0\29\16\82\0\15\45\82\0\29\60\82\0\36\89\82\0\29\125\82\0\21\154\82\0\24\175\82\0\28\199\82\0\21\227\82\0\30\248\82\0\27\22\83\0\28\49\83\0\28\77\83\0\25\105\83\0\26\130\83\0\26\156\83\0\14\182\83\0\30\196\83\0\23\226\83\0\25\249\83\0\25\18\84\0\26\43\84\0\32\69\84\0\30\101\84\0\19\131\84\0\25\150\84\0\20\175\84\0\21\195\84\0\17\216\84\0\16\233\84\0\29\249\84\0\22\22\85\0\36\44\85\0\21\80\85\0\33\101\85\0\21\134\85\0\30\155\85\0\21\185\85\0\22\206\85\0\18\228\85\0\25\246\85\0\23\15\86\0\30\38\86\0\29\68\86\0\31\97\86\0\30\128\86\0\31\158\86\0\31\189\86\0\22\220\86\0\23\242\86\
    \\0\17\9\87\0\24\26\87\0\25\50\87\0\25\75\87\0\25\100\87\0\27\125\87\0\23\152\87\0\31\175\87\0\29\206\87\0\22\235\87\0\27\1\88\0\36\28\88\0\26\64\88\0\25\90\88\0\26\115\88\0\31\141\88\0\32\172\88\0\29\204\88\0\25\233\88\0\31\2\89\0\30\33\89\0\17\63\89\0\17\80\89\0\30\97\89\0\31\127\89\0\48\158\89\0\31\206\89\0\25\237\89\0\24\6\90\0\27\30\90\0\22\57\90\0\28\79\90\0\22\107\90\0\23\129\90\0\29\152\90\0\22\181\90\0\31\203\90\0\39\234\90\0\30\17\91\0\30\47\91\0\30\77\91\0\30\107\91\0\30\137\91\0\30\167\91\0\30\197\91\0\30\227\91\0\30\1\92\0\30\31\92\0\30\61\92\0\30\91\92\0\30\121\92\0\25\151\92\0\23\176\92\0\34\199\92\0\32\233\92\0\18\9\93\0\24\27\93\0\39\51\93\0\37\90\93\0\0\0\0\0\0\0\0\0\19\127\93\0\40\146\93\0\38\186\
    \\93\0\47\224\93\0\19\15\94\0\24\34\94\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\11\58\94\0\21\69\94\0\37\90\94\0\16\127\94\0\39\143\94\0\35\182\94\0\36\217\94\0\0\0\0\0\39\253\94\0\0\0\0\0\39\36\95\0\37\75\95\0\48\112\95\0\26\160\95\0\25\186\95\0\26\211\95\0\26\237\95\0\28\7\96\0\25\35\96\0\24\60\96\0\26\84\96\0\25\110\96\0\26\135\96\0\26\161\96\0\23\187\96\0\23\210\96\0\23\233\96\0\28\0\97\0\23\28\97\0\24\51\97\0\0\0\0\0\26\75\97\0\24\101\97\0\28\125\97\0\24\153\97\0\24\177\97\0\24\201\97\0\26\225\97\0\40\251\97\0\43\35\98\0\35\78\98\0\37\113\98\0\33\150\98\0\34\183\98\0\51\217\98\0\24\12\99\0\23\36\99\0\24\59\99\0\24\83\99\0\26\107\99\0\23\133\99\0\22\156\99\0\24\178\99\0\23\202\99\0\24\225\99\0\24\249\99\0\21\
    \\17\100\0\21\38\100\0\21\59\100\0\26\80\100\0\21\106\100\0\22\127\100\0\30\149\100\0\24\179\100\0\22\203\100\0\26\225\100\0\22\251\100\0\22\17\101\0\22\39\101\0\24\61\101\0\38\85\101\0\41\123\101\0\37\164\101\0\37\201\101\0\35\238\101\0\24\17\102\0\17\41\102\0\18\58\102\0\30\76\102\0\40\106\102\0\44\146\102\0\16\190\102\0\15\206\102\0\16\221\102\0\26\237\102\0\32\7\103\0\19\39\103\0\25\58\103\0\20\83\103\0\26\103\103\0\18\129\103\0\24\147\103\0\18\171\103\0\24\189\103\0\26\213\103\0\24\239\103\0\25\7\104\0\23\32\104\0\26\55\104\0\24\81\104\0\26\105\104\0\24\131\104\0\28\155\104\0\26\183\104\0\27\209\104\0\25\236\104\0\25\5\105\0\23\30\105\0\18\53\105\0\16\71\105\0\25\87\105\0\16\112\105\0\26\128\105\0\27\154\105\0\36\181\105\0\24\217\105\0\22\241\105\0\33\7\106\0\24\40\106\0\22\64\106\0\
    \\28\86\106\0\42\114\106\0\40\156\106\0\49\196\106\0\37\245\106\0\26\26\107\0\27\52\107\0\27\79\107\0\36\106\107\0\27\142\107\0\48\169\107\0\26\217\107\0\26\243\107\0\27\13\108\0\27\40\108\0\28\67\108\0\27\95\108\0\36\122\108\0\31\158\108\0\28\189\108\0\25\217\108\0\26\242\108\0\26\12\109\0\27\38\109\0\26\65\109\0\26\91\109\0\27\117\109\0\26\144\109\0\25\170\109\0\31\195\109\0\26\226\109\0\26\252\109\0\26\22\110\0\26\48\110\0\25\74\110\0\26\99\110\0\26\125\110\0\26\151\110\0\26\177\110\0\25\203\110\0\26\228\110\0\26\254\110\0\27\24\111\0\27\51\111\0\27\78\111\0\29\105\111\0\33\134\111\0\28\167\111\0\33\195\111\0\25\228\111\0\26\253\111\0\26\23\112\0\23\49\112\0\24\72\112\0\24\96\112\0\25\120\112\0\24\145\112\0\24\169\112\0\25\193\112\0\24\218\112\0\23\242\112\0\29\9\113\0\24\38\113\0\24\62\113\
    \\0\24\86\113\0\24\110\113\0\23\134\113\0\24\157\113\0\24\181\113\0\24\205\113\0\24\229\113\0\23\253\113\0\24\20\114\0\24\44\114\0\25\68\114\0\25\93\114\0\25\118\114\0\27\143\114\0\31\170\114\0\26\201\114\0\31\227\114\0\23\2\115\0\24\25\115\0\24\49\115\0\35\73\115\0\24\108\115\0\25\132\115\0\25\157\115\0\34\182\115\0\25\216\115\0\46\241\115\0\24\31\116\0\24\55\116\0\25\79\116\0\25\104\116\0\26\129\116\0\25\155\116\0\34\180\116\0\29\214\116\0\26\243\116\0\29\13\117\0\27\42\117\0\27\69\117\0\25\96\117\0\34\121\117\0\32\155\117\0\34\187\117\0\32\221\117\0\43\253\117\0\41\40\118\0\31\81\118\0\29\112\118\0\40\141\118\0\38\181\118\0\27\219\118\0\25\246\118\0\27\15\119\0\25\42\119\0\28\67\119\0\26\95\119\0\31\121\119\0\29\152\119\0\56\181\119\0\54\237\119\0\26\35\120\0\24\61\120\0\35\85\120\0\33\120\
    \\120\0\40\153\120\0\38\193\120\0\26\231\120\0\24\1\121\0\29\25\121\0\27\54\121\0\23\81\121\0\24\104\121\0\33\128\121\0\33\161\121\0\33\194\121\0\27\227\121\0\41\254\121\0\32\39\122\0\41\71\122\0\39\112\122\0\37\151\122\0\35\188\122\0\36\223\122\0\34\3\123\0\39\37\123\0\37\76\123\0\39\113\123\0\37\152\123\0\44\189\123\0\42\233\123\0\42\19\124\0\40\61\124\0\41\101\124\0\39\142\124\0\41\181\124\0\39\222\124\0\47\5\125\0\45\52\125\0\38\97\125\0\36\135\125\0\34\171\125\0\32\205\125\0\41\237\125\0\39\22\126\0\32\61\126\0\30\93\126\0\43\123\126\0\41\166\126\0\36\207\126\0\34\243\126\0\41\21\127\0\39\62\127\0\41\101\127\0\39\142\127\0\34\181\127\0\32\215\127\0\46\247\127\0\44\37\128\0\41\81\128\0\39\122\128\0\32\161\128\0\30\193\128\0\42\223\128\0\40\9\129\0\48\49\129\0\46\97\129\0\28\143\129\0\26\
    \\171\129\0\37\197\129\0\35\234\129\0\52\13\130\0\50\65\130\0\24\115\130\0\38\139\130\0\36\177\130\0\36\213\130\0\34\249\130\0\36\27\131\0\34\63\131\0\36\97\131\0\34\133\131\0\36\167\131\0\34\203\131\0\38\237\131\0\36\19\132\0\36\55\132\0\34\91\132\0\30\125\132\0\36\155\132\0\34\191\132\0\40\225\132\0\38\9\133\0\30\47\133\0\28\77\133\0\37\105\133\0\35\142\133\0\29\177\133\0\27\206\133\0\44\233\133\0\42\21\134\0\42\63\134\0\40\105\134\0\41\145\134\0\39\186\134\0\37\225\134\0\35\6\135\0\37\41\135\0\35\78\135\0\40\113\135\0\38\153\135\0\40\191\135\0\38\231\135\0\32\13\136\0\30\45\136\0\47\75\136\0\45\122\136\0\40\167\136\0\38\207\136\0\37\245\136\0\35\26\137\0\40\61\137\0\38\101\137\0\43\139\137\0\41\182\137\0\42\223\137\0\40\9\138\0\42\49\138\0\40\91\138\0\43\131\138\0\41\174\138\0\48\215\138\0\
    \\46\7\139\0\36\53\139\0\34\89\139\0\38\123\139\0\36\161\139\0\31\197\139\0\29\228\139\0\32\1\140\0\30\33\140\0\32\63\140\0\30\95\140\0\33\125\140\0\31\158\140\0\32\189\140\0\30\221\140\0\32\251\140\0\30\27\141\0\32\57\141\0\30\89\141\0\32\119\141\0\30\151\141\0\35\181\141\0\33\216\141\0\36\249\141\0\34\29\142\0\27\63\142\0\25\90\142\0\27\115\142\0\25\142\142\0\27\167\142\0\25\194\142\0\26\219\142\0\24\245\142\0\26\13\143\0\24\39\143\0\32\63\143\0\30\95\143\0\43\125\143\0\41\168\143\0\43\209\143\0\41\252\143\0\41\37\144\0\39\78\144\0\43\117\144\0\41\160\144\0\41\201\144\0\39\242\144\0\29\25\145\0\27\54\145\0\28\81\145\0\26\109\145\0\41\135\145\0\39\176\145\0\0\0\0\0\27\215\145\0\27\242\145\0\27\13\146\0\26\40\146\0\27\66\146\0\26\93\146\0\26\119\146\0\26\145\146\0\26\171\146\0\27\197\146\
    \\0\27\224\146\0\28\251\146\0\27\23\147\0\26\50\147\0\27\76\147\0\26\103\147\0\26\129\147\0\28\155\147\0\28\183\147\0\27\211\147\0\26\238\147\0\27\8\148\0\27\35\148\0\26\62\148\0\27\88\148\0\27\115\148\0\28\142\148\0\26\170\148\0\27\196\148\0\27\223\148\0\28\250\148\0\27\22\149\0\26\49\149\0\28\75\149\0\28\103\149\0\27\131\149\0\26\158\149\0\27\184\149\0\0\0\0\0\0\0\0\0\39\211\149\0\19\250\149\0\22\13\150\0\25\35\150\0\14\60\150\0\22\74\150\0\26\96\150\0\32\122\150\0\25\154\150\0\25\179\150\0\25\204\150\0\24\229\150\0\25\253\150\0\24\22\151\0\24\46\151\0\24\70\151\0\24\94\151\0\25\118\151\0\25\143\151\0\26\168\151\0\25\194\151\0\24\219\151\0\25\243\151\0\24\12\152\0\24\36\152\0\26\60\152\0\26\86\152\0\25\112\152\0\24\137\152\0\25\161\152\0\25\186\152\0\24\211\152\0\25\235\152\0\25\4\
    \\153\0\26\29\153\0\24\55\153\0\25\79\153\0\25\104\153\0\26\129\153\0\25\155\153\0\24\180\153\0\26\204\153\0\26\230\153\0\25\0\154\0\24\25\154\0\25\49\154\0\32\74\154\0\36\106\154\0\18\142\154\0\15\160\154\0\0\0\0\0\0\0\0\0\35\175\154\0\34\210\154\0\18\244\154\0\0\0\0\0\21\6\155\0\19\27\155\0\24\46\155\0\25\70\155\0\25\95\155\0\20\120\155\0\19\140\155\0\19\159\155\0\20\178\155\0\19\198\155\0\19\217\155\0\20\236\155\0\27\0\156\0\23\27\156\0\25\50\156\0\28\75\156\0\19\103\156\0\26\122\156\0\19\148\156\0\22\167\156\0\20\189\156\0\27\209\156\0\19\236\156\0\19\255\156\0\28\18\157\0\28\46\157\0\17\74\157\0\18\91\157\0\18\109\157\0\19\127\157\0\25\146\157\0\18\171\157\0\24\189\157\0\24\213\157\0\25\237\157\0\18\6\158\0\18\24\158\0\18\42\158\0\18\60\158\0\19\78\158\0\18\97\158\0\32\
    \\115\158\0\19\147\158\0\28\166\158\0\18\194\158\0\24\212\158\0\17\236\158\0\24\253\158\0\21\21\159\0\20\42\159\0\28\62\159\0\21\90\159\0\21\111\159\0\30\132\159\0\25\162\159\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\18\187\159\0\17\205\159\0\19\222\159\0\19\241\159\0\16\4\160\0\17\20\160\0\19\37\160\0\17\56\160\0\17\73\160\0\17\90\160\0\23\107\160\0\17\130\160\0\19\147\160\0\23\166\160\0\17\189\160\0\23\206\160\0\17\229\160\0\20\246\160\0\18\10\161\0\22\28\161\0\16\50\161\0\25\66\161\0\19\91\161\0\17\110\161\0\18\127\161\0\18\145\161\0\17\163\161\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\19\180\161\0\34\199\161\0\31\233\161\0\34\8\162\0\25\42\162\0\28\67\162\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\
    \\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\18\95\162\0\17\113\162\0\22\130\162\0\17\152\162\0\18\169\162\0\24\187\162\0\22\211\162\0\24\233\162\0\10\1\163\0\27\11\163\0\34\38\163\0\12\72\163\0\12\84\163\0\21\96\163\0\24\117\163\0\17\141\163\0\40\158\163\0\27\198\163\0\30\225\163\0\29\255\163\0\21\28\164\0\21\49\164\0\49\70\164\0\22\119\164\0\18\141\164\0\18\159\164\0\18\177\164\0\16\195\164\0\18\211\164\0\23\229\164\0\34\252\164\0\20\30\165\0\26\50\165\0\19\76\165\0\35\95\165\0\35\130\165\0\34\165\165\0\35\199\165\0\34\234\165\0\18\12\166\0\17\30\166\0\25\47\166\0\17\72\166\0\18\89\166\0\18\107\166\0\17\125\166\0\18\142\166\0\17\160\166\0\18\177\166\0\17\195\166\0\18\212\166\0\18\230\166\0\19\248\166\0\17\11\167\0\17\28\167\0\17\45\167\0\17\62\167\0\17\79\167\
    \\0\19\96\167\0\39\115\167\0\41\154\167\0\39\195\167\0\43\234\167\0\45\21\168\0\14\66\168\0\17\80\168\0\17\97\168\0\17\114\168\0\17\131\168\0\18\148\168\0\18\166\168\0\17\184\168\0\17\201\168\0\26\218\168\0\17\244\168\0\15\5\169\0\15\20\169\0\15\35\169\0\12\50\169\0\12\62\169\0\12\74\169\0\13\86\169\0\12\99\169\0\19\111\169\0\18\130\169\0\18\148\169\0\21\166\169\0\21\187\169\0\23\208\169\0\15\231\169\0\31\246\169\0\40\21\170\0\27\61\170\0\21\88\170\0\26\109\170\0\23\135\170\0\23\158\170\0\22\181\170\0\22\203\170\0\24\225\170\0\23\249\170\0\23\16\171\0\22\39\171\0\24\61\171\0\24\85\171\0\23\109\171\0\19\132\171\0\24\151\171\0\26\175\171\0\24\201\171\0\25\225\171\0\25\250\171\0\30\19\172\0\24\49\172\0\40\73\172\0\40\113\172\0\24\153\172\0\29\177\172\0\28\206\172\0\32\234\172\0\28\10\173\0\18\38\
    \\173\0\20\56\173\0\18\76\173\0\27\94\173\0\49\121\173\0\17\170\173\0\19\187\173\0\19\206\173\0\34\225\173\0\46\3\174\0\18\49\174\0\18\67\174\0\39\85\174\0\19\124\174\0\21\143\174\0\18\164\174\0\27\182\174\0\32\209\174\0\46\241\174\0\19\31\175\0\20\50\175\0\17\70\175\0\49\87\175\0\38\136\175\0\18\174\175\0\30\192\175\0\27\222\175\0\32\249\175\0\36\25\176\0\46\61\176\0\37\107\176\0\17\144\176\0\38\161\176\0\47\199\176\0\40\246\176\0\61\30\177\0\37\91\177\0\39\128\177\0\39\167\177\0\39\206\177\0\25\245\177\0\38\14\178\0\32\52\178\0\17\84\178\0\39\101\178\0\19\140\178\0\32\159\178\0\39\191\178\0\19\230\178\0\23\249\178\0\27\16\179\0\32\43\179\0\16\75\179\0\39\91\179\0\17\130\179\0\27\147\179\0\19\174\179\0\37\193\179\0\18\230\179\0\39\248\179\0\30\31\180\0\32\61\180\0\39\93\180\0\39\132\180\0\33\
    \\171\180\0\25\204\180\0\19\229\180\0\28\248\180\0\40\20\181\0\29\60\181\0\34\89\181\0\32\123\181\0\22\155\181\0\39\177\181\0\30\216\181\0\27\246\181\0\24\17\182\0\16\41\182\0\15\57\182\0\16\72\182\0\24\88\182\0\37\112\182\0\16\149\182\0\23\165\182\0\27\188\182\0\30\215\182\0\32\245\182\0\15\21\183\0\39\36\183\0\24\75\183\0\41\99\183\0\16\140\183\0\16\156\183\0\57\172\183\0\57\229\183\0\35\30\184\0\26\65\184\0\22\91\184\0\28\113\184\0\22\141\184\0\18\163\184\0\27\181\184\0\30\208\184\0\42\238\184\0\38\24\185\0\36\62\185\0\21\98\185\0\23\119\185\0\16\142\185\0\16\158\185\0\21\174\185\0\22\195\185\0\22\217\185\0\28\239\185\0\29\11\186\0\43\40\186\0\21\83\186\0\33\104\186\0\33\137\186\0\32\170\186\0\31\202\186\0\31\233\186\0\33\8\187\0\32\41\187\0\32\73\187\0\31\105\187\0\33\136\187\0\33\169\187\0\
    \\32\202\187\0\34\234\187\0\32\12\188\0\34\44\188\0\28\78\188\0\35\106\188\0\33\141\188\0\23\174\188\0\28\197\188\0\26\225\188\0\24\251\188\0\22\19\189\0\23\41\189\0\24\64\189\0\25\88\189\0\36\113\189\0\35\149\189\0\18\184\189\0\22\202\189\0\25\224\189\0\26\249\189\0\0\0\0\0\24\19\190\0\19\43\190\0\31\62\190\0\18\93\190\0\19\111\190\0\28\130\190\0\20\158\190\0\33\178\190\0\16\211\190\0\17\227\190\0\18\244\190\0\18\6\191\0\18\24\191\0\27\42\191\0\18\69\191\0\21\87\191\0\18\108\191\0\20\126\191\0\17\146\191\0\17\163\191\0\21\180\191\0\27\201\191\0\15\228\191\0\16\243\191\0\25\3\192\0\19\28\192\0\18\47\192\0\18\65\192\0\18\83\192\0\17\101\192\0\27\118\192\0\28\145\192\0\29\173\192\0\19\202\192\0\19\221\192\0\20\240\192\0\19\4\193\0\19\23\193\0\20\42\193\0\18\62\193\0\18\80\193\0\30\98\193\
    \\0\27\128\193\0\18\155\193\0\18\173\193\0\25\191\193\0\18\216\193\0\18\234\193\0\12\252\193\0\19\8\194\0\16\27\194\0\15\43\194\0\30\58\194\0\30\88\194\0\23\118\194\0\23\141\194\0\25\164\194\0\25\189\194\0\12\214\194\0\14\226\194\0\0\0\0\0\0\0\0\0\27\240\194\0\27\11\195\0\24\38\195\0\52\62\195\0\53\114\195\0\56\167\195\0\75\223\195\0\51\42\196\0\45\93\196\0\30\138\196\0\37\168\196\0\56\205\196\0\62\5\197\0\45\67\197\0\29\112\197\0\39\141\197\0\37\180\197\0\58\217\197\0\48\19\198\0\37\67\198\0\56\104\198\0\34\160\198\0\41\194\198\0\58\235\198\0\33\37\199\0\33\70\199\0\38\103\199\0\33\141\199\0\31\174\199\0\26\205\199\0\48\231\199\0\34\23\200\0\49\57\200\0\52\106\200\0\59\158\200\0\60\217\200\0\59\21\201\0\52\80\201\0\61\132\201\0\63\193\201\0\66\0\202\0\68\66\202\0\67\134\202\0\60\201\
    \\202\0\62\5\203\0\67\67\203\0\69\134\203\0\61\203\203\0\62\8\204\0\34\70\204\0\37\104\204\0\17\141\204\0\23\158\204\0\19\181\204\0\17\200\204\0\17\217\204\0\23\234\204\0\19\1\205\0\19\20\205\0\19\39\205\0\19\58\205\0\19\77\205\0\20\96\205\0\18\116\205\0\19\134\205\0\19\153\205\0\23\172\205\0\19\195\205\0\22\214\205\0\22\236\205\0\22\2\206\0\17\24\206\0\22\41\206\0\22\63\206\0\23\85\206\0\18\108\206\0\18\126\206\0\18\144\206\0\20\162\206\0\17\182\206\0\20\199\206\0\20\219\206\0\20\239\206\0\16\3\207\0\16\19\207\0\18\35\207\0\20\53\207\0\19\73\207\0\19\92\207\0\14\111\207\0\16\125\207\0\14\141\207\0\16\155\207\0\14\171\207\0\16\185\207\0\14\201\207\0\16\215\207\0\14\231\207\0\16\245\207\0\12\5\208\0\17\17\208\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\
    \\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\14\34\208\0\13\48\208\0\13\61\208\0\15\74\208\0\14\89\208\0\14\103\208\0\13\117\208\0\15\130\208\0\15\145\208\0\14\160\208\0\12\174\208\0\13\186\208\0\12\199\208\0\12\211\208\0\12\223\208\0\13\235\208\0\12\248\208\0\21\4\209\0\12\25\209\0\13\37\209\0\13\50\209\0\13\63\209\0\13\76\209\0\14\89\209\0\13\103\209\0\13\116\209\0\14\129\209\0\13\143\209\0\14\156\209\0\13\170\209\0\13\183\209\0\13\196\209\0\20\209\209\0\13\229\209\0\14\242\209\0\13\0\210\0\13\13\210\0\13\26\210\0\13\39\210\0\21\52\210\0\18\73\210\0\19\91\210\0\18\110\210\0\29\128\210\0\28\157\210\0\31\185\210\0\34\216\210\0\28\250\210\0\27\22\211\0\30\49\211\0\31\79\211\0\30\110\211\0\24\140\211\0\23\164\211\0\20\187\211\0\21\207\211\0\
    \\9\228\211\0\20\237\211\0\14\1\212\0\0\0\0\0\0\0\0\0\14\15\212\0\15\29\212\0\14\44\212\0\21\58\212\0\20\79\212\0\22\99\212\0\22\121\212\0\19\143\212\0\20\162\212\0\20\182\212\0\19\202\212\0\20\221\212\0\20\241\212\0\21\5\213\0\22\26\213\0\20\48\213\0\20\68\213\0\24\88\213\0\19\112\213\0\19\131\213\0\24\150\213\0\20\174\213\0\21\194\213\0\21\215\213\0\21\236\213\0\17\1\214\0\22\18\214\0\24\40\214\0\21\64\214\0\40\85\214\0\29\125\214\0\27\154\214\0\22\181\214\0\32\203\214\0\28\235\214\0\23\7\215\0\31\30\215\0\27\61\215\0\22\88\215\0\33\110\215\0\28\143\215\0\27\171\215\0\22\198\215\0\27\220\215\0\27\247\215\0\22\18\216\0\22\40\216\0\26\62\216\0\22\88\216\0\0\0\0\0\0\0\0\0\29\110\216\0\28\139\216\0\27\167\216\0\25\194\216\0\28\219\216\0\32\247\216\0\27\23\217\0\35\50\217\
    \\0\27\85\217\0\27\112\217\0\26\139\217\0\26\165\217\0\29\191\217\0\34\220\217\0\28\254\217\0\0\0\0\0\20\26\218\0\17\46\218\0\17\63\218\0\17\80\218\0\17\97\218\0\22\114\218\0\17\136\218\0\17\153\218\0\18\170\218\0\19\188\218\0\17\207\218\0\17\224\218\0\17\241\218\0\17\2\219\0\17\19\219\0\17\36\219\0\17\53\219\0\18\70\219\0\17\88\219\0\17\105\219\0\18\122\219\0\17\140\219\0\23\157\219\0\18\180\219\0\18\198\219\0\24\216\219\0\25\240\219\0\23\9\220\0\0\0\0\0\0\0\0\0\19\32\220\0\0\0\0\0\27\51\220\0\26\78\220\0\27\104\220\0\27\131\220\0\27\158\220\0\28\185\220\0\27\213\220\0\26\240\220\0\27\10\221\0\28\37\221\0\27\65\221\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\38\92\221\0\48\130\221\0\43\178\221\0\42\221\221\0\38\7\222\0\51\45\222\0\48\96\222\0\48\144\
    \\222\0\47\192\222\0\48\239\222\0\33\31\223\0\62\64\223\0\57\126\223\0\65\183\223\0\62\248\223\0\57\54\224\0\65\111\224\0\44\176\224\0\43\220\224\0\36\7\225\0\34\43\225\0\34\77\225\0\22\111\225\0\25\133\225\0\23\158\225\0\40\181\225\0\45\221\225\0\32\10\226\0\39\42\226\0\50\81\226\0\20\131\226\0\0\0\0\0\23\151\226\0\25\174\226\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\29\199\226\0\29\228\226\0\28\1\227\0\29\29\227\0\19\58\227\0\33\77\227\0\20\110\227\0\28\130\227\0\36\158\227\0\34\194\227\0\38\228\227\0\37\10\228\0\53\47\228\0\32\100\228\0\33\132\228\0\40\165\228\0\53\205\228\0\51\2\229\0\27\53\229\0\33\80\229\0\26\113\229\0\22\139\229\0\39\161\229\0\39\200\229\0\38\239\229\0\26\21\230\0\40\47\230\0\39\87\230\0\32\126\230\0\50\158\230\0\39\208\230\0\39\
    \\247\230\0\38\30\231\0\39\68\231\0\58\107\231\0\25\165\231\0\25\190\231\0\26\215\231\0\30\241\231\0\30\15\232\0\31\45\232\0\32\76\232\0\32\108\232\0\41\140\232\0\47\181\232\0\40\228\232\0\40\12\233\0\52\52\233\0\18\104\233\0\22\122\233\0\27\144\233\0\48\171\233\0\26\219\233\0\21\245\233\0\28\10\234\0\28\38\234\0\18\66\234\0\25\84\234\0\42\109\234\0\20\151\234\0\29\171\234\0\21\200\234\0\21\221\234\0\21\242\234\0\33\7\235\0\32\40\235\0\35\72\235\0\31\107\235\0\30\138\235\0\28\168\235\0\26\196\235\0\28\222\235\0\33\250\235\0\28\27\236\0\27\55\236\0\25\82\236\0\18\107\236\0\18\125\236\0\18\143\236\0\21\161\236\0\21\182\236\0\21\203\236\0\25\224\236\0\26\249\236\0\22\19\237\0\25\41\237\0\26\66\237\0\22\92\237\0\20\114\237\0\20\134\237\0\20\154\237\0\21\174\237\0\22\195\237\0\27\217\237\0\27\244\237\0\
    \\27\15\238\0\28\42\238\0\27\70\238\0\28\97\238\0\35\125\238\0\44\160\238\0\37\204\238\0\21\241\238\0\32\6\239\0\36\38\239\0\27\74\239\0\24\101\239\0\23\125\239\0\25\148\239\0\19\173\239\0\20\192\239\0\19\212\239\0\20\231\239\0\19\251\239\0\20\14\240\0\27\34\240\0\27\61\240\0\26\88\240\0\25\114\240\0\19\139\240\0\20\158\240\0\26\178\240\0\25\204\240\0\19\229\240\0\20\248\240\0\20\12\241\0\21\32\241\0\20\53\241\0\21\73\241\0\21\94\241\0\20\115\241\0\21\135\241\0\20\156\241\0\21\176\241\0\21\197\241\0\21\218\241\0\22\239\241\0\21\5\242\0\22\26\242\0\21\48\242\0\20\69\242\0\21\89\242\0\20\110\242\0\21\130\242\0\20\151\242\0\22\171\242\0\20\193\242\0\21\213\242\0\20\234\242\0\21\254\242\0\20\19\243\0\20\39\243\0\20\59\243\0\21\79\243\0\20\100\243\0\21\120\243\0\22\141\243\0\20\163\243\0\21\183\243\
    \\0\21\204\243\0\20\225\243\0\20\245\243\0\24\9\244\0\25\33\244\0\21\58\244\0\24\79\244\0\24\103\244\0\23\127\244\0\24\150\244\0\23\174\244\0\24\197\244\0\31\221\244\0\32\252\244\0\30\28\245\0\29\58\245\0\23\87\245\0\24\110\245\0\30\134\245\0\29\164\245\0\23\193\245\0\24\216\245\0\22\240\245\0\37\6\246\0\24\43\246\0\13\67\246\0\29\80\246\0\31\109\246\0\23\140\246\0\23\163\246\0\35\186\246\0\24\221\246\0\25\245\246\0\20\14\247\0\22\34\247\0\22\56\247\0\20\78\247\0\23\98\247\0\21\121\247\0\20\142\247\0\21\162\247\0\28\183\247\0\28\211\247\0\31\239\247\0\32\14\248\0\16\46\248\0\23\62\248\0\21\85\248\0\20\106\248\0\20\126\248\0\22\146\248\0\21\168\248\0\21\189\248\0\20\210\248\0\22\230\248\0\22\252\248\0\21\18\249\0\28\39\249\0\32\67\249\0\26\99\249\0\20\125\249\0\21\145\249\0\20\166\249\0\20\186\
    \\249\0\21\206\249\0\29\227\249\0\21\0\250\0\26\21\250\0\21\47\250\0\21\68\250\0\30\89\250\0\22\119\250\0\21\141\250\0\12\162\250\0\24\174\250\0\21\198\250\0\20\219\250\0\0\0\0\0\16\239\250\0\17\255\250\0\16\16\251\0\17\32\251\0\16\49\251\0\17\65\251\0\24\82\251\0\24\106\251\0\0\0\0\0\0\0\0\0\16\130\251\0\17\146\251\0\0\0\0\0\0\0\0\0\16\163\251\0\17\179\251\0\17\196\251\0\18\213\251\0\17\231\251\0\18\248\251\0\18\10\252\0\17\28\252\0\18\45\252\0\17\63\252\0\18\80\252\0\18\98\252\0\18\116\252\0\19\134\252\0\18\153\252\0\19\171\252\0\18\190\252\0\17\208\252\0\18\225\252\0\17\243\252\0\18\4\253\0\17\22\253\0\0\0\0\0\17\39\253\0\18\56\253\0\17\74\253\0\18\91\253\0\17\109\253\0\17\126\253\0\17\143\253\0\0\0\0\0\17\160\253\0\0\0\0\0\0\0\0\0\0\0\0\0\18\
    \\177\253\0\18\195\253\0\17\213\253\0\17\230\253\0\0\0\0\0\0\0\0\0\18\247\253\0\21\9\254\0\21\30\254\0\20\51\254\0\21\71\254\0\20\92\254\0\21\112\254\0\28\133\254\0\29\161\254\0\0\0\0\0\0\0\0\0\20\190\254\0\21\210\254\0\0\0\0\0\0\0\0\0\20\231\254\0\21\251\254\0\19\16\255\0\24\35\255\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\22\59\255\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\18\81\255\0\18\99\255\0\0\0\0\0\18\117\255\0\25\135\255\0\25\160\255\0\28\185\255\0\29\213\255\0\0\0\0\0\0\0\0\0\18\242\255\0\17\4\0\1\17\21\0\1\19\38\0\1\18\57\0\1\18\75\0\1\17\93\0\1\19\110\0\1\19\129\0\1\18\148\0\1\38\166\0\1\37\204\0\1\18\241\0\1\18\3\1\1\30\21\1\1\30\51\1\1\
    \\32\81\1\1\31\113\1\1\56\144\1\1\36\200\1\1\14\236\1\1\18\250\1\1\29\12\2\1\25\41\2\1\19\66\2\1\0\0\0\0\0\0\0\0\24\85\2\1\19\109\2\1\21\128\2\1\0\0\0\0\17\149\2\1\18\166\2\1\17\184\2\1\18\201\2\1\17\219\2\1\18\236\2\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\18\254\2\1\18\16\3\1\0\0\0\0\0\0\0\0\18\34\3\1\18\52\3\1\18\70\3\1\19\88\3\1\18\107\3\1\19\125\3\1\19\144\3\1\18\163\3\1\19\181\3\1\18\200\3\1\19\218\3\1\19\237\3\1\19\0\4\1\20\19\4\1\19\39\4\1\20\58\4\1\19\78\4\1\18\97\4\1\19\115\4\1\18\134\4\1\19\152\4\1\18\171\4\1\0\0\0\0\18\189\4\1\19\207\4\1\18\226\4\1\19\244\4\1\18\7\5\1\18\25\5\1\18\43\5\1\0\0\0\0\18\61\5\1\19\79\5\1\0\0\0\0\18\98\5\
    \\1\19\116\5\1\0\0\0\0\18\135\5\1\18\153\5\1\0\0\0\0\0\0\0\0\19\171\5\1\0\0\0\0\22\190\5\1\21\212\5\1\22\233\5\1\21\255\5\1\22\20\6\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\22\42\6\1\22\64\6\1\0\0\0\0\0\0\0\0\22\86\6\1\22\108\6\1\20\130\6\1\0\0\0\0\0\0\0\0\0\0\0\0\19\150\6\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\20\169\6\1\20\189\6\1\18\209\6\1\19\227\6\1\0\0\0\0\18\246\6\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\19\8\7\1\18\27\7\1\18\45\7\1\20\63\7\1\19\83\7\1\19\102\7\1\18\121\7\1\20\139\7\1\20\159\7\1\19\179\7\1\14\198\7\1\14\212\7\1\12\226\7\1\12\238\7\1\17\250\7\1\20\11\
    \\8\1\26\31\8\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\25\57\8\1\22\82\8\1\21\104\8\1\0\0\0\0\17\125\8\1\18\142\8\1\17\160\8\1\18\177\8\1\17\195\8\1\18\212\8\1\25\230\8\1\25\255\8\1\23\24\9\1\0\0\0\0\17\47\9\1\18\64\9\1\23\82\9\1\0\0\0\0\17\105\9\1\18\122\9\1\18\140\9\1\19\158\9\1\18\177\9\1\19\195\9\1\19\214\9\1\18\233\9\1\19\251\9\1\18\14\10\1\19\32\10\1\19\51\10\1\19\70\10\1\20\89\10\1\19\109\10\1\20\128\10\1\19\148\10\1\18\167\10\1\19\185\10\1\18\204\10\1\19\222\10\1\18\241\10\1\0\0\0\0\18\3\11\1\19\21\11\1\18\40\11\1\19\58\11\1\18\77\11\1\18\95\11\1\18\113\11\1\0\0\0\0\18\131\11\1\19\149\11\1\0\0\0\0\18\
    \\168\11\1\19\186\11\1\19\205\11\1\18\224\11\1\18\242\11\1\0\0\0\0\0\0\0\0\19\4\12\1\22\23\12\1\22\45\12\1\21\67\12\1\22\88\12\1\21\110\12\1\22\131\12\1\29\153\12\1\30\182\12\1\28\212\12\1\0\0\0\0\21\240\12\1\22\5\13\1\28\27\13\1\0\0\0\0\21\55\13\1\22\76\13\1\20\98\13\1\0\0\0\0\0\0\0\0\11\118\13\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\26\129\13\1\26\155\13\1\29\181\13\1\30\210\13\1\0\0\0\0\0\0\0\0\19\240\13\1\18\3\14\1\18\21\14\1\20\39\14\1\19\59\14\1\19\78\14\1\18\97\14\1\20\115\14\1\20\135\14\1\19\155\14\1\26\174\14\1\19\200\14\1\0\0\0\0\0\0\0\0\0\0\0\0\
    \\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\19\219\14\1\19\238\14\1\20\1\15\1\20\21\15\1\35\41\15\1\32\76\15\1\36\108\15\1\0\0\0\0\22\144\15\1\19\166\15\1\18\185\15\1\0\0\0\0\14\203\15\1\15\217\15\1\14\232\15\1\15\246\15\1\14\5\16\1\15\19\16\1\22\34\16\1\22\56\16\1\0\0\0\0\0\0\0\0\14\78\16\1\15\92\16\1\0\0\0\0\0\0\0\0\14\107\16\1\15\121\16\1\15\136\16\1\16\151\16\1\15\167\16\1\16\182\16\1\16\198\16\1\15\214\16\1\16\229\16\1\15\245\16\1\16\4\17\1\16\20\17\1\16\36\17\1\17\52\17\1\16\69\17\1\17\85\17\1\16\102\17\1\15\118\17\1\16\133\17\1\15\149\17\1\16\164\17\1\15\180\17\1\0\0\0\0\15\195\17\1\16\210\17\1\15\226\17\1\16\241\17\1\15\1\18\1\15\16\18\1\15\31\18\1\0\0\0\0\15\46\18\1\16\61\18\1\0\0\0\
    \\0\15\77\18\1\16\92\18\1\16\108\18\1\15\124\18\1\15\139\18\1\0\0\0\0\0\0\0\0\16\154\18\1\19\170\18\1\19\189\18\1\18\208\18\1\19\226\18\1\18\245\18\1\19\7\19\1\26\26\19\1\27\52\19\1\0\0\0\0\0\0\0\0\18\79\19\1\19\97\19\1\0\0\0\0\0\0\0\0\18\116\19\1\19\134\19\1\17\153\19\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\19\170\19\1\20\189\19\1\20\209\19\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\16\229\19\1\16\245\19\1\0\0\0\0\16\5\20\1\23\21\20\1\23\44\20\1\26\67\20\1\27\93\20\1\0\0\0\0\0\0\0\0\16\120\20\1\15\136\20\1\15\151\20\1\17\166\20\1\16\183\20\1\16\199\20\1\15\215\20\1\17\230\20\1\17\247\20\1\16\8\21\1\12\24\21\1\15\36\21\1\26\51\21\1\23\77\21\1\29\100\
    \\21\1\28\129\21\1\25\157\21\1\31\182\21\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\19\213\21\1\18\232\21\1\0\0\0\0\14\250\21\1\15\8\22\1\14\23\22\1\15\37\22\1\14\52\22\1\15\66\22\1\0\0\0\0\0\0\0\0\0\0\0\0\14\81\22\1\15\95\22\1\15\110\22\1\0\0\0\0\14\125\22\1\15\139\22\1\15\154\22\1\15\169\22\1\0\0\0\0\0\0\0\0\0\0\0\0\16\184\22\1\15\200\22\1\0\0\0\0\15\215\22\1\0\0\0\0\16\230\22\1\16\246\22\1\0\0\0\0\0\0\0\0\0\0\0\0\16\6\23\1\15\22\23\1\0\0\0\0\0\0\0\0\0\0\0\0\15\37\23\1\17\52\23\1\15\69\23\1\0\0\0\0\0\0\0\0\0\0\0\0\15\84\23\1\15\99\23\1\15\114\23\1\16\129\23\1\15\145\23\1\16\160\23\1\17\
    \\176\23\1\15\193\23\1\16\208\23\1\16\224\23\1\15\240\23\1\15\255\23\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\19\14\24\1\18\33\24\1\19\51\24\1\18\70\24\1\19\88\24\1\0\0\0\0\0\0\0\0\0\0\0\0\18\107\24\1\19\125\24\1\19\144\24\1\0\0\0\0\18\163\24\1\19\181\24\1\19\200\24\1\17\219\24\1\0\0\0\0\0\0\0\0\8\236\24\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\20\244\24\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\16\8\25\1\15\24\25\1\15\39\25\1\17\54\25\1\16\71\25\1\16\87\25\1\15\103\25\1\17\118\25\1\17\135\25\1\16\152\25\1\16\168\25\1\24\184\25\1\25\208\25\1\14\233\25\1\
    \\16\247\25\1\15\7\26\1\16\22\26\1\17\38\26\1\19\55\26\1\16\74\26\1\17\90\26\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\39\107\26\1\23\146\26\1\20\169\26\1\19\189\26\1\36\208\26\1\15\244\26\1\16\3\27\1\15\19\27\1\16\34\27\1\15\50\27\1\16\65\27\1\23\81\27\1\23\104\27\1\0\0\0\0\15\127\27\1\16\142\27\1\16\158\27\1\0\0\0\0\15\174\27\1\16\189\27\1\16\205\27\1\16\221\27\1\17\237\27\1\16\254\27\1\17\14\28\1\17\31\28\1\16\48\28\1\17\64\28\1\16\81\28\1\17\97\28\1\17\114\28\1\17\131\28\1\18\148\28\1\17\166\28\1\18\183\28\1\17\201\28\1\16\218\28\1\17\234\28\1\16\251\28\1\17\11\29\1\16\28\29\1\0\0\0\0\16\44\29\1\17\60\29\1\16\77\29\1\17\93\29\1\16\110\29\1\16\126\29\1\16\142\29\1\17\158\29\1\16\175\29\1\17\191\29\
    \\1\18\208\29\1\16\226\29\1\17\242\29\1\17\3\30\1\16\20\30\1\16\36\30\1\0\0\0\0\0\0\0\0\17\52\30\1\20\69\30\1\20\89\30\1\19\109\30\1\20\128\30\1\19\148\30\1\20\167\30\1\27\187\30\1\28\214\30\1\0\0\0\0\19\242\30\1\20\5\31\1\20\25\31\1\0\0\0\0\19\45\31\1\20\64\31\1\20\84\31\1\18\104\31\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\18\122\31\1\21\140\31\1\0\0\0\0\17\161\31\1\17\178\31\1\18\195\31\1\0\0\0\0\0\0\0\0\27\213\31\1\0\0\0\0\0\0\0\0\24\240\31\1\24\8\32\1\27\32\32\1\28\59\32\1\0\0\0\0\0\0\0\0\17\87\32\1\16\104\32\1\16\120\32\1\18\136\32\1\17\154\32\1\17\171\32\1\16\188\32\1\18\204\32\1\18\222\32\1\17\240\32\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\
    \\0\0\0\0\0\0\0\0\0\0\0\0\0\0\19\1\33\1\49\20\33\1\48\69\33\1\48\117\33\1\50\165\33\1\49\215\33\1\49\8\34\1\51\57\34\1\17\108\34\1\32\125\34\1\24\157\34\1\21\181\34\1\20\202\34\1\20\222\34\1\16\242\34\1\17\2\35\1\16\19\35\1\17\35\35\1\16\52\35\1\17\68\35\1\24\85\35\1\24\109\35\1\0\0\0\0\16\133\35\1\17\149\35\1\17\166\35\1\0\0\0\0\16\183\35\1\17\199\35\1\17\216\35\1\17\233\35\1\18\250\35\1\17\12\36\1\18\29\36\1\18\47\36\1\17\65\36\1\18\82\36\1\17\100\36\1\18\117\36\1\18\135\36\1\18\153\36\1\19\171\36\1\18\190\36\1\19\208\36\1\18\227\36\1\17\245\36\1\18\6\37\1\17\24\37\1\18\41\37\1\17\59\37\1\0\0\0\0\17\76\37\1\18\93\37\1\17\111\37\1\18\128\37\1\17\146\37\1\17\163\37\1\17\180\37\1\18\197\37\1\17\215\37\1\18\
    \\232\37\1\0\0\0\0\17\250\37\1\18\11\38\1\18\29\38\1\17\47\38\1\17\64\38\1\0\0\0\0\0\0\0\0\18\81\38\1\21\99\38\1\21\120\38\1\20\141\38\1\21\161\38\1\20\182\38\1\21\202\38\1\28\223\38\1\29\251\38\1\0\0\0\0\20\24\39\1\21\44\39\1\21\65\39\1\0\0\0\0\20\86\39\1\21\106\39\1\21\127\39\1\19\148\39\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\19\167\39\1\22\186\39\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\28\208\39\1\17\236\39\1\0\0\0\0\25\253\39\1\25\22\40\1\28\47\40\1\29\75\40\1\0\0\0\0\0\0\0\0\18\104\40\1\17\122\40\1\17\139\40\1\19\156\40\1\18\175\40\1\18\193\40\1\17\211\40\1\19\228\40\1\19\247\40\1\18\10\41\1\0\0\0\0\24\28\41\1\24\52\41\1\
    \\43\76\41\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\39\119\41\1\26\158\41\1\23\184\41\1\22\207\41\1\31\229\41\1\18\4\42\1\19\22\42\1\18\41\42\1\19\59\42\1\18\78\42\1\19\96\42\1\26\115\42\1\26\141\42\1\0\0\0\0\18\167\42\1\19\185\42\1\19\204\42\1\0\0\0\0\18\223\42\1\19\241\42\1\19\4\43\1\19\23\43\1\20\42\43\1\19\62\43\1\20\81\43\1\20\101\43\1\19\121\43\1\20\140\43\1\19\160\43\1\20\179\43\1\20\199\43\1\20\219\43\1\21\239\43\1\20\4\44\1\21\24\44\1\20\45\44\1\19\65\44\1\20\84\44\1\19\104\44\1\20\123\44\1\19\143\44\1\21\162\44\1\19\183\44\1\20\202\44\1\19\222\44\1\20\241\44\1\19\5\45\1\19\24\45\1\19\43\45\1\20\62\45\1\19\82\45\
    \\1\20\101\45\1\21\121\45\1\19\142\45\1\20\161\45\1\20\181\45\1\19\201\45\1\19\220\45\1\21\239\45\1\34\4\46\1\30\38\46\1\23\68\46\1\23\91\46\1\22\114\46\1\23\136\46\1\22\159\46\1\23\181\46\1\30\204\46\1\31\234\46\1\0\0\0\0\22\9\47\1\23\31\47\1\23\54\47\1\0\0\0\0\22\77\47\1\23\99\47\1\23\122\47\1\21\145\47\1\25\166\47\1\19\191\47\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\25\210\47\1\25\235\47\1\27\4\48\1\24\31\48\1\47\55\48\1\31\102\48\1\35\133\48\1\32\168\48\1\28\200\48\1\35\228\48\1\28\7\49\1\27\35\49\1\27\62\49\1\27\89\49\1\30\116\49\1\31\146\49\1\0\0\0\0\0\0\0\0\20\177\49\1\19\197\49\1\19\216\49\1\21\235\49\1\20\0\50\1\20\20\50\1\19\40\50\1\21\59\50\1\21\80\50\1\20\101\50\1\20\121\50\1\28\141\50\1\29\169\
    \\50\1\30\198\50\1\27\228\50\1\33\255\50\1\32\32\51\1\29\64\51\1\35\93\51\1\19\128\51\1\26\147\51\1\25\173\51\1\26\198\51\1\25\224\51\1\26\249\51\1\25\19\52\1\0\0\0\0\24\44\52\1\23\68\52\1\22\91\52\1\0\0\0\0\21\113\52\1\22\134\52\1\22\156\52\1\23\178\52\1\21\201\52\1\22\222\52\1\21\244\52\1\22\9\53\1\23\31\53\1\24\54\53\1\23\78\53\1\24\101\53\1\21\125\53\1\22\146\53\1\22\168\53\1\21\190\53\1\22\211\53\1\22\233\53\1\0\0\0\0\0\0\0\0\0\0\0\0\33\255\53\1\34\32\54\1\33\66\54\1\34\99\54\1\33\133\54\1\30\166\54\1\33\196\54\1\34\229\54\1\33\7\55\1\34\40\55\1\33\74\55\1\43\107\55\1\30\150\55\1\34\180\55\1\35\214\55\1\34\249\55\1\35\27\56\1\32\62\56\1\31\94\56\1\33\125\56\1\34\158\56\1\33\192\56\1\34\225\56\1\30\3\57\1\0\
    \\0\0\0\30\33\57\1\33\63\57\1\34\96\57\1\33\130\57\1\34\163\57\1\22\197\57\1\27\219\57\1\22\246\57\1\22\12\58\1\0\0\0\0\30\34\58\1\0\0\0\0\0\0\0\0\22\64\58\1\30\86\58\1\32\116\58\1\30\148\58\1\22\178\58\1\32\200\58\1\22\232\58\1\0\0\0\0\0\0\0\0\0\0\0\0\22\254\58\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\29\20\59\1\35\49\59\1\34\84\59\1\33\118\59\1\32\151\59\1\34\183\59\1\0\0\0\0\33\217\59\1\0\0\0\0\31\250\59\1\26\25\60\1\31\51\60\1\29\82\60\1\41\111\60\1\46\152\60\1\42\198\60\1\30\240\60\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\23\14\61\1\22\37\61\1\22\59\61\1\24\81\61\1\23\105\61\1\23\128\61\1\22\151\61\1\24\173\61\1\24\197\61\1\23\221\61\1\0\0\0\0\0\0\0\0\
    \\36\244\61\1\35\24\62\1\30\59\62\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\21\89\62\1\23\110\62\1\24\133\62\1\24\157\62\1\23\181\62\1\26\204\62\1\22\230\62\1\23\252\62\1\24\19\63\1\24\43\63\1\20\67\63\1\23\87\63\1\22\110\63\1\23\132\63\1\23\155\63\1\23\178\63\1\29\201\63\1\26\230\63\1\21\0\64\1\21\21\64\1\21\42\64\1\24\63\64\1\25\87\64\1\24\112\64\1\20\136\64\1\24\156\64\1\21\180\64\1\24\201\64\1\20\225\64\1\23\245\64\1\21\12\65\1\26\33\65\1\20\59\65\1\21\79\65\1\21\100\65\1\17\121\65\1\22\138\65\1\17\160\65\1\22\177\65\1\22\199\65\1\22\221\65\1\21\243\65\1\21\8\66\1\23\29\66\1\20\52\66\1\24\72\66\1\24\96\66\1\21\120\66\1\27\141\66\
    \\1\22\168\66\1\22\190\66\1\21\212\66\1\22\233\66\1\22\255\66\1\23\21\67\1\21\44\67\1\22\65\67\1\22\87\67\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\25\109\67\1\21\134\67\1\22\155\67\1\21\177\67\1\30\198\67\1\31\228\67\1\26\3\68\1\23\29\68\1\24\52\68\1\21\76\68\1\22\97\68\1\22\119\68\1\27\141\68\1\26\168\68\1\23\194\68\1\23\217\68\1\22\240\68\1\15\6\69\1\14\21\69\1\14\35\69\1\16\49\69\1\15\65\69\1\15\80\69\1\14\95\69\1\16\109\69\1\16\125\69\1\15\141\69\1\25\156\69\1\21\181\69\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\
    \\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\13\202\69\1\19\215\69\1\0\0\0\0\18\234\69\1\0\0\0\0\19\252\69\1\14\15\70\1\13\29\70\1\19\42\70\1\17\61\70\1\0\0\0\0\19\78\70\1\14\97\70\1\19\111\70\1\19\130\70\1\20\149\70\1\19\169\70\1\20\188\70\1\19\208\70\1\13\227\70\1\13\240\70\1\19\253\70\1\18\16\71\1\19\34\71\1\13\53\71\1\13\66\71\1\13\79\71\1\19\92\71\1\17\111\71\1\18\128\71\1\18\146\71\1\19\164\71\1\13\183\71\1\13\196\71\1\18\209\71\1\0\0\0\0\18\227\71\1\0\0\0\0\13\245\71\1\23\2\72\1\23\25\72\1\18\48\72\1\18\66\72\1\19\84\72\1\12\103\72\1\17\115\72\1\12\132\72\1\16\144\72\1\22\
    \\160\72\1\17\182\72\1\17\199\72\1\16\216\72\1\17\232\72\1\16\249\72\1\17\9\73\1\16\26\73\1\17\42\73\1\20\59\73\1\22\79\73\1\21\101\73\1\22\122\73\1\0\0\0\0\0\0\0\0\16\144\73\1\17\160\73\1\16\177\73\1\17\193\73\1\17\210\73\1\0\0\0\0\9\227\73\1\0\0\0\0\15\236\73\1\16\251\73\1\15\11\74\1\19\26\74\1\21\45\74\1\13\66\74\1\12\79\74\1\0\0\0\0\14\91\74\1\13\105\74\1\13\118\74\1\15\131\74\1\14\146\74\1\14\160\74\1\13\174\74\1\15\187\74\1\15\202\74\1\14\217\74\1\0\0\0\0\0\0\0\0\9\231\74\1\9\240\74\1\18\249\74\1\19\11\75\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\
    \\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\19\30\75\1\37\49\75\1\42\86\75\1\43\128\75\1\36\171\75\1\36\207\75\1\39\243\75\1\34\26\76\1\23\60\76\1\26\83\76\1\30\109\76\1\32\139\76\1\34\171\76\1\17\205\76\1\22\222\76\1\23\244\76\1\28\11\77\1\33\39\77\1\27\72\77\1\42\99\77\1\23\141\77\1\32\164\77\1\32\196\77\1\47\228\77\1\35\19\78\1\38\54\78\1\27\92\78\1\28\119\78\1\27\147\78\1\26\174\78\1\27\200\78\1\31\227\78\1\18\2\79\1\17\20\79\1\17\37\79\1\19\54\79\1\18\73\79\1\18\91\79\1\17\109\79\1\19\126\79\1\19\145\79\1\18\164\79\1\22\182\79\1\22\204\79\1\24\226\79\1\23\250\79\1\23\17\80\1\22\40\80\1\24\62\80\
    \\1\24\86\80\1\23\110\80\1\23\133\80\1\24\156\80\1\31\180\80\1\43\211\80\1\34\254\80\1\20\32\81\1\22\52\81\1\27\74\81\1\27\101\81\1\27\128\81\1\27\155\81\1\22\182\81\1\22\204\81\1\17\226\81\1\18\243\81\1\17\5\82\1\18\22\82\1\18\40\82\1\17\58\82\1\18\75\82\1\17\93\82\1\0\0\0\0\18\110\82\1\18\128\82\1\19\146\82\1\18\165\82\1\19\183\82\1\18\202\82\1\17\220\82\1\18\237\82\1\17\255\82\1\18\16\83\1\17\34\83\1\17\51\83\1\18\68\83\1\17\86\83\1\18\103\83\1\17\121\83\1\18\138\83\1\19\156\83\1\18\175\83\1\19\193\83\1\17\212\83\1\18\229\83\1\17\247\83\1\17\8\84\1\17\25\84\1\17\42\84\1\17\59\84\1\18\76\84\1\18\94\84\1\17\112\84\1\17\129\84\1\16\146\84\1\19\162\84\1\28\181\84\1\18\209\84\1\18\227\84\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\
    \\0\0\21\245\84\1\20\10\85\1\21\30\85\1\20\51\85\1\21\71\85\1\28\92\85\1\29\120\85\1\28\149\85\1\29\177\85\1\20\206\85\1\21\226\85\1\20\247\85\1\21\11\86\1\27\32\86\1\22\59\86\1\29\81\86\1\30\110\86\1\27\140\86\1\21\167\86\1\20\188\86\1\19\208\86\1\22\227\86\1\23\249\86\1\24\16\87\1\21\40\87\1\28\61\87\1\28\89\87\1\30\117\87\1\34\147\87\1\31\181\87\1\40\212\87\1\27\252\87\1\28\23\88\1\27\51\88\1\28\78\88\1\28\106\88\1\27\134\88\1\28\161\88\1\27\189\88\1\0\0\0\0\28\216\88\1\28\244\88\1\29\16\89\1\28\45\89\1\29\73\89\1\28\102\89\1\27\130\89\1\28\157\89\1\27\185\89\1\28\212\89\1\27\240\89\1\27\11\90\1\28\38\90\1\27\66\90\1\28\93\90\1\27\121\90\1\28\148\90\1\29\176\90\1\28\205\90\1\29\233\90\1\27\6\91\1\28\33\91\1\27\61\91\1\27\
    \\88\91\1\27\115\91\1\27\142\91\1\27\169\91\1\28\196\91\1\28\224\91\1\27\252\91\1\27\23\92\1\26\50\92\1\29\76\92\1\38\105\92\1\38\143\92\1\38\181\92\1\0\0\0\0\17\219\92\1\30\236\92\1\36\10\93\1\36\46\93\1\35\82\93\1\36\117\93\1\22\153\93\1\22\175\93\1\25\197\93\1\32\222\93\1\22\254\93\1\21\20\94\1\33\41\94\1\33\74\94\1\33\107\94\1\0\0\0\0\31\140\94\1\26\171\94\1\36\197\94\1\35\233\94\1\23\12\95\1\48\35\95\1\48\83\95\1\24\131\95\1\23\155\95\1\34\178\95\1\33\212\95\1\32\245\95\1\33\21\96\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\
    \\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\17\54\96\1\18\71\96\1\17\89\96\1\18\106\96\1\18\124\96\1\17\142\96\1\18\159\96\1\17\177\96\1\18\194\96\1\18\212\96\1\19\230\96\1\18\249\96\1\19\11\97\1\18\30\97\1\19\48\97\1\18\67\97\1\17\85\97\1\18\102\97\1\17\120\97\1\18\137\97\1\17\155\97\1\17\172\97\1\18\189\97\1\17\207\97\1\18\224\97\1\17\242\97\1\17\3\98\1\17\20\98\1\17\37\98\1\17\54\98\1\17\71\98\1\17\88\98\1\18\105\98\1\16\123\98\1\21\139\98\1\16\160\98\1\17\176\98\1\16\193\98\1\17\209\98\1\16\226\98\1\20\242\98\1\16\6\99\1\17\22\99\1\26\39\99\1\21\65\99\1\20\86\99\1\21\106\99\1\20\127\99\
    \\1\21\147\99\1\20\168\99\1\21\188\99\1\25\209\99\1\24\234\99\1\26\2\100\1\21\28\100\1\22\49\100\1\20\71\100\1\19\91\100\1\17\110\100\1\32\127\100\1\32\159\100\1\32\191\100\1\32\223\100\1\23\255\100\1\18\22\101\1\17\40\101\1\17\57\101\1\19\74\101\1\18\93\101\1\18\111\101\1\17\129\101\1\19\146\101\1\19\165\101\1\18\184\101\1\27\202\101\1\20\229\101\1\23\249\101\1\24\16\102\1\29\40\102\1\23\69\102\1\18\92\102\1\18\110\102\1\24\128\102\1\25\152\102\1\24\177\102\1\25\201\102\1\28\226\102\1\29\254\102\1\28\27\103\1\29\55\103\1\22\84\103\1\22\106\103\1\22\128\103\1\22\150\103\1\36\172\103\1\36\208\103\1\36\244\103\1\29\24\104\1\32\53\104\1\34\85\104\1\35\119\104\1\36\154\104\1\36\190\104\1\39\226\104\1\39\9\105\1\37\48\105\1\37\85\105\1\37\122\105\1\37\159\105\1\37\196\105\1\36\233\105\1\36\13\
    \\106\1\37\49\106\1\31\86\106\1\27\117\106\1\26\144\106\1\27\170\106\1\22\197\106\1\23\219\106\1\22\242\106\1\22\8\107\1\22\30\107\1\23\52\107\1\22\75\107\1\22\97\107\1\23\119\107\1\22\142\107\1\22\164\107\1\23\186\107\1\22\209\107\1\37\231\107\1\26\12\108\1\25\38\108\1\31\63\108\1\31\94\108\1\24\125\108\1\24\149\108\1\24\173\108\1\24\197\108\1\32\221\108\1\32\253\108\1\39\29\109\1\31\68\109\1\33\99\109\1\23\132\109\1\22\155\109\1\22\177\109\1\24\199\109\1\23\223\109\1\23\246\109\1\22\13\110\1\24\35\110\1\24\59\110\1\23\83\110\1\26\106\110\1\26\132\110\1\26\158\110\1\27\184\110\1\23\211\110\1\31\234\110\1\26\9\111\1\27\35\111\1\27\62\111\1\27\89\111\1\26\116\111\1\27\142\111\1\27\169\111\1\27\196\111\1\26\223\111\1\27\249\111\1\27\20\112\1\27\47\112\1\27\74\112\1\26\101\112\1\27\127\112\1\28\
    \\154\112\1\27\182\112\1\27\209\112\1\27\236\112\1\26\7\113\1\28\33\113\1\28\61\113\1\28\89\113\1\27\117\113\1\28\144\113\1\28\172\113\1\27\200\113\1\27\227\113\1\27\254\113\1\28\25\114\1\27\53\114\1\28\80\114\1\27\108\114\1\26\135\114\1\27\161\114\1\26\188\114\1\27\214\114\1\27\241\114\1\0\0\0\0\26\12\115\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\27\38\115\1\0\0\0\0\0\0\0\0\18\65\115\1\19\83\115\1\19\102\115\1\19\121\115\1\18\140\115\1\19\158\115\1\19\177\115\1\19\196\115\1\18\215\115\1\19\233\115\1\19\252\115\1\19\15\116\1\19\34\116\1\18\53\116\1\19\71\116\1\20\90\116\1\19\110\116\1\19\129\116\1\19\148\116\1\18\167\116\1\20\185\116\1\20\205\116\1\20\225\116\1\19\245\116\1\20\8\117\1\20\28\117\1\19\48\117\1\19\67\117\1\19\86\117\1\20\105\117\1\19\125\117\1\
    \\20\144\117\1\19\164\117\1\18\183\117\1\19\201\117\1\18\220\117\1\19\238\117\1\19\1\118\1\18\20\118\1\18\38\118\1\21\56\118\1\26\77\118\1\19\103\118\1\28\122\118\1\28\150\118\1\19\178\118\1\25\197\118\1\27\222\118\1\22\249\118\1\27\15\119\1\21\42\119\1\22\63\119\1\27\85\119\1\21\112\119\1\21\133\119\1\21\154\119\1\26\175\119\1\20\201\119\1\25\221\119\1\21\246\119\1\21\11\120\1\26\32\120\1\23\58\120\1\23\81\120\1\23\104\120\1\23\127\120\1\21\150\120\1\28\171\120\1\26\199\120\1\28\225\120\1\27\253\120\1\29\24\121\1\27\53\121\1\26\80\121\1\27\106\121\1\29\133\121\1\27\162\121\1\29\189\121\1\28\218\121\1\27\246\121\1\28\17\122\1\26\45\122\1\33\71\122\1\33\104\122\1\32\137\122\1\31\169\122\1\32\200\122\1\27\232\122\1\29\3\123\1\29\32\123\1\29\61\123\1\29\90\123\1\34\119\123\1\27\153\123\1\26\180\123\
    \\1\27\206\123\1\26\233\123\1\26\3\124\1\26\29\124\1\33\55\124\1\30\88\124\1\26\118\124\1\26\144\124\1\28\170\124\1\28\198\124\1\28\226\124\1\28\254\124\1\26\26\125\1\28\52\125\1\33\80\125\1\31\113\125\1\36\144\125\1\23\180\125\1\28\203\125\1\28\231\125\1\27\3\126\1\27\30\126\1\26\57\126\1\29\83\126\1\26\112\126\1\27\138\126\1\29\165\126\1\29\194\126\1\29\223\126\1\24\252\126\1\27\20\127\1\29\47\127\1\34\76\127\1\32\110\127\1\37\142\127\1\31\179\127\1\29\210\127\1\31\239\127\1\34\14\128\1\29\48\128\1\31\77\128\1\26\108\128\1\27\134\128\1\29\161\128\1\26\190\128\1\27\216\128\1\27\243\128\1\28\14\129\1\22\42\129\1\23\64\129\1\18\87\129\1\19\105\129\1\19\124\129\1\20\143\129\1\19\163\129\1\18\182\129\1\20\200\129\1\19\220\129\1\18\239\129\1\19\1\130\1\20\20\130\1\19\40\130\1\19\59\130\1\18\78\
    \\130\1\20\96\130\1\19\116\130\1\19\135\130\1\19\154\130\1\19\173\130\1\19\192\130\1\18\211\130\1\20\229\130\1\20\249\130\1\21\13\131\1\22\34\131\1\21\56\131\1\21\77\131\1\22\98\131\1\22\120\131\1\22\142\131\1\21\164\131\1\20\185\131\1\21\205\131\1\20\226\131\1\20\246\131\1\22\10\132\1\23\32\132\1\23\55\132\1\21\78\132\1\21\99\132\1\20\120\132\1\21\140\132\1\24\161\132\1\21\185\132\1\20\206\132\1\21\226\132\1\22\247\132\1\21\13\133\1\23\34\133\1\22\57\133\1\21\79\133\1\21\100\133\1\21\121\133\1\22\142\133\1\21\164\133\1\20\185\133\1\21\205\133\1\20\226\133\1\20\246\133\1\21\10\134\1\24\31\