-- autogenerated from https://www.unicode.org/Public/15.0.0/ucd/Scripts.txt
-- |
-- Module      : Unicode.Internal.Char.ScriptExtensions
-- Copyright   : (c) 2022 Composewell Technologies and Contributors
-- License     : Apache-2.0
-- Maintainer  : streamly@composewell.com
-- Stability   : experimental

{-# OPTIONS_HADDOCK hide #-}

module Unicode.Internal.Char.ScriptExtensions
(scriptExtensions)
where

import Data.Char (ord)
import Data.Int (Int8)
import Data.Word (Word16)
import GHC.Exts (Addr#, Int#, Int(..), Ptr(..), andI#, iShiftL#, iShiftRL#, negateInt#, nullAddr#, (+#), (-#))
import Unicode.Internal.Bits.Scripts (lookupWord16AsInt#, lookupWord8AsInt#)

-- | Script extensions of a character.
--
-- Returns a pair:
--
-- * If first value is negative or zero, then its absolute value is a single script,
--   encoded by its index.
-- * Else the first element is the length and the second is the list of scripts,
--   encoded by their index.
--
-- @since 0.1.0
scriptExtensions :: Char -> (# Int#, Addr# #)
scriptExtensions c = case encodedScriptExtensions c of
    164# -> (# 9#, "\0\3\49\81\82\114\123\133\138"# #)
    165# -> (# 7#, "\0\3\49\101\138\149\161"# #)
    166# -> (# 2#, "\3\25"# #)
    167# -> (# 2#, "\3\101"# #)
    168# -> (# 6#, "\3\49\101\138\149\161"# #)
    169# -> (# 2#, "\3\49"# #)
    170# -> (# 2#, "\3\138"# #)
    171# -> (# 3#, "\3\138\149"# #)
    172# -> (# 2#, "\3\149"# #)
    173# -> (# 3#, "\3\149\161"# #)
    174# -> (# 3#, "\10\20\137"# #)
    175# -> (# 2#, "\10\31"# #)
    176# -> (# 21#, "\10\31\33\42\44\45\46\60\68\72\78\80\84\98\115\132\137\144\145\148\153"# #)
    177# -> (# 20#, "\10\31\33\42\44\45\46\60\68\78\80\84\98\115\132\137\144\145\148\153"# #)
    178# -> (# 13#, "\10\31\42\44\46\60\70\80\115\128\145\148\153"# #)
    179# -> (# 12#, "\10\31\42\44\46\60\70\80\115\145\148\153"# #)
    180# -> (# 4#, "\10\31\42\60"# #)
    181# -> (# 8#, "\10\31\42\60\98\115\148\153"# #)
    182# -> (# 5#, "\12\47\48\53\61"# #)
    183# -> (# 6#, "\12\47\48\53\61\162"# #)
    184# -> (# 2#, "\12\47"# #)
    185# -> (# 2#, "\15\58"# #)
    186# -> (# 4#, "\16\50\139\140"# #)
    187# -> (# 3#, "\20\95\141"# #)
    188# -> (# 3#, "\27\28\74"# #)
    189# -> (# 3#, "\27\73\74"# #)
    190# -> (# 2#, "\27\74"# #)
    191# -> (# 2#, "\29\40"# #)
    192# -> (# 2#, "\29\70"# #)
    193# -> (# 2#, "\29\109"# #)
    194# -> (# 2#, "\29\138"# #)
    195# -> (# 14#, "\31\33\44\46\59\60\67\68\78\80\91\98\144\153"# #)
    196# -> (# 13#, "\31\33\44\46\59\60\67\68\78\91\98\144\153"# #)
    197# -> (# 11#, "\31\33\44\46\59\67\68\78\91\144\153"# #)
    198# -> (# 4#, "\31\33\59\78"# #)
    199# -> (# 2#, "\31\42"# #)
    200# -> (# 3#, "\31\42\60"# #)
    201# -> (# 3#, "\31\42\70"# #)
    202# -> (# 6#, "\31\60\80\115\145\148"# #)
    203# -> (# 2#, "\31\98"# #)
    204# -> (# 2#, "\31\128"# #)
    205# -> (# 2#, "\31\145"# #)
    206# -> (# 2#, "\39\70"# #)
    207# -> (# 2#, "\42\145"# #)
    208# -> (# 2#, "\44\67"# #)
    209# -> (# 2#, "\46\94"# #)
    210# -> (# 3#, "\47\53\61"# #)
    211# -> (# 2#, "\47\70"# #)
    212# -> (# 2#, "\53\61"# #)
    213# -> (# 3#, "\63\70\95"# #)
    214# -> (# 2#, "\60\98"# #)
    215# -> (# 2#, "\70\92"# #)
    216# -> (# 2#, "\82\114"# #)
    217# -> (# 2#, "\92\121"# #)
    s    -> (# negateInt# s, nullAddr# #)

{-# INLINE encodedScriptExtensions #-}
encodedScriptExtensions :: Char -> Int#
encodedScriptExtensions c
    -- Planes 0-3
    | cp < 0x323B0 = lookupEncodedScriptExtensionsBitMap cp#
    -- Planes 4-13: 156
    | cp < 0xE0000 = 156#
    -- Plane 14
    | cp < 0xE01F0 = lookupEncodedScriptExtensionsBitMap (cp# -# 0xADC50#)
    -- Default: 156
    | otherwise = 156#
    where
    !cp@(I# cp#) = ord c

{-# INLINE lookupEncodedScriptExtensionsBitMap #-}
lookupEncodedScriptExtensionsBitMap :: Int# -> Int#
lookupEncodedScriptExtensionsBitMap n =
    lookupWord8AsInt# data# (
        lookupWord16AsInt# offsets1# (
            lookupWord16AsInt# offsets2# (
                n `iShiftRL#` 8#
            ) +# ((n `iShiftRL#` 3#) `andI#` maskOffsets)
        ) +# (n `andI#` maskData)
    )
    where
    maskData = (1# `iShiftL#` 3#) -# 1#
    maskOffsets = (1# `iShiftL#` 5#) -# 1#
    !(Ptr data#) = encodedScriptExtensionsDataBitMap
    !(Ptr offsets1#) = encodedScriptExtensionsOffsets1BitMap
    !(Ptr offsets2#) = encodedScriptExtensionsOffsets2BitMap

encodedScriptExtensionsDataBitMap :: Ptr Int8
encodedScriptExtensionsDataBitMap = Ptr
    "\70\70\24\24\156\156\156\156\82\82\82\82\82\216\82\82\82\82\156\123\123\123\123\123\123\123\123\156\156\156\156\156\156\180\31\180\199\31\175\175\204\202\31\204\204\31\31\203\175\31\
    \\31\175\31\31\181\199\200\175\175\10\201\156\156\156\156\156\156\156\143\143\143\143\143\143\143\143\156\156\156\156\156\212\212\212\212\53\53\53\53\53\53\53\156\24\24\24\191\24\24\24\
    \\24\24\24\156\156\156\156\25\25\25\25\25\25\25\156\156\156\156\204\175\31\31\31\31\31\31\156\156\156\156\156\156\98\98\98\98\98\98\156\156\156\32\156\156\32\32\32\32\32\32\
    \\156\32\55\55\43\55\55\43\55\55\55\55\55\55\29\29\29\29\29\29\156\156\156\156\156\156\156\82\82\82\82\82\82\82\156\123\123\123\123\156\156\156\156\156\156\156\160\57\57\57\
    \\57\57\57\57\57\156\156\5\5\5\5\5\5\5\5\156\156\64\64\64\64\64\64\64\64\156\156\156\156\156\156\156\46\46\46\46\156\46\156\156\156\156\156\156\88\88\88\88\88\88\
    \\88\88\45\45\45\45\45\45\45\45\156\45\45\156\156\156\156\156\156\156\124\24\183\183\183\183\183\61\61\61\183\212\61\61\61\61\61\61\61\61\47\47\184\184\184\184\48\48\48\48\
    \\48\48\48\48\24\151\151\156\156\156\156\156\48\48\48\48\48\48\48\156\156\156\156\43\24\43\24\43\43\43\29\70\70\70\70\70\70\70\43\43\43\70\70\70\70\70\24\24\24\156\
    \\156\156\156\156\156\156\148\148\156\92\92\217\217\92\217\92\92\92\92\92\92\92\92\156\156\156\156\156\156\156\156\10\10\156\10\156\156\156\10\10\156\156\10\10\10\10\156\10\10\10\
    \\43\156\156\156\156\156\156\156\132\156\156\156\156\132\132\132\132\132\132\132\132\156\132\156\156\151\151\151\151\151\151\151\151\156\151\151\38\38\38\38\156\38\38\156\53\53\53\53\53\53\
    \\53\53\156\156\61\156\156\60\60\60\156\156\156\156\4\4\4\4\4\4\4\4\156\42\156\156\42\42\42\42\42\156\156\42\42\42\156\156\42\42\42\42\42\42\42\42\156\42\42\42\
    \\42\42\42\42\42\156\156\156\156\156\156\42\42\42\188\188\190\156\156\156\156\189\189\189\189\189\189\189\189\156\156\156\190\190\190\190\190\190\190\190\183\183\24\182\183\183\183\183\183\183\
    \\183\183\24\182\183\183\183\183\182\182\182\182\212\212\212\212\212\24\182\18\18\18\18\18\18\18\18\156\156\156\156\156\156\156\43\43\43\43\24\43\43\43\25\25\25\25\25\25\25\25\
    \\74\74\74\74\74\74\74\74\156\74\74\74\74\74\74\156\156\156\156\156\115\115\115\156\156\115\115\115\115\115\156\156\115\115\115\156\156\115\115\115\115\115\115\115\115\127\127\127\127\127\
    \\127\127\127\156\156\156\156\156\156\127\127\147\156\156\156\156\156\156\156\150\150\150\150\150\150\150\150\156\156\156\156\24\156\156\24\24\156\43\156\43\156\43\156\43\107\107\107\107\107\107\
    \\107\107\156\156\156\156\156\107\107\107\9\9\9\9\9\9\9\9\156\156\156\156\9\9\9\9\46\156\156\46\46\46\156\156\46\156\46\46\156\46\46\156\91\91\91\91\91\91\91\91\
    \\156\156\156\156\156\156\156\90\43\43\70\70\70\70\43\43\55\55\55\55\55\55\24\24\55\55\55\55\24\24\24\24\24\24\24\55\156\156\48\48\48\156\156\156\27\156\27\27\27\27\
    \\27\27\27\27\156\27\17\17\17\17\17\17\17\17\156\156\83\83\83\83\83\83\83\156\11\11\11\11\11\11\11\11\156\156\156\73\73\73\73\73\73\73\73\156\156\21\21\21\21\21\
    \\21\21\21\156\156\194\55\138\55\55\55\55\55\70\70\70\70\70\24\70\70\70\70\70\70\70\70\24\24\24\24\24\24\24\70\24\24\24\24\24\182\182\24\138\138\138\156\156\156\156\
    \\156\52\52\52\52\52\52\52\52\156\52\156\126\126\126\126\126\126\126\126\156\156\72\72\72\72\72\72\72\72\156\141\141\141\141\141\141\156\156\156\142\142\142\142\142\142\142\142\156\156\
    \\142\152\156\156\156\156\156\156\156\152\151\151\151\151\151\24\24\24\70\70\70\70\70\70\70\156\156\156\156\156\156\209\209\209\209\209\209\209\209\86\86\86\86\86\86\86\86\156\156\156\
    \\156\156\156\12\12\12\12\12\12\12\12\96\96\96\96\96\96\96\96\156\173\173\173\173\173\173\173\173\3\3\3\3\3\3\156\156\156\156\156\156\64\64\64\64\64\64\64\64\156\64\
    \\64\156\82\82\82\82\82\82\82\82\156\37\37\37\37\37\37\37\37\156\134\134\134\134\134\134\134\134\156\156\156\156\156\156\156\139\78\78\78\78\78\78\78\78\156\68\68\68\156\156\
    \\156\156\156\156\207\207\26\26\26\26\26\156\156\156\156\156\156\132\132\132\156\156\156\2\2\2\2\2\2\2\2\156\93\93\93\93\93\93\93\93\156\146\146\146\146\146\146\146\146\156\
    \\103\103\103\103\103\103\103\103\156\156\156\156\103\103\154\154\154\154\154\154\154\154\156\149\149\149\149\149\149\149\149\156\156\156\156\156\156\24\24\24\24\24\24\24\24\215\52\52\156\52\
    \\52\156\52\52\52\52\52\156\156\156\156\52\138\138\138\138\138\138\138\138\156\138\10\156\10\10\10\10\10\10\10\10\156\156\10\156\46\46\46\156\46\46\46\46\46\46\46\46\156\156\
    \\156\156\46\156\44\44\44\156\44\44\44\44\44\44\44\44\156\44\156\115\115\115\156\115\115\115\115\115\115\115\115\69\69\69\156\69\69\69\69\69\69\69\69\156\69\156\69\69\156\69\
    \\156\69\69\39\39\39\156\156\39\39\39\206\39\39\39\39\39\39\39\39\156\39\22\22\22\22\22\22\22\22\156\156\139\139\139\139\139\139\139\139\156\156\66\66\66\66\66\66\66\66\
    \\156\156\141\141\141\141\141\141\141\141\156\156\99\99\99\99\156\156\156\156\156\156\156\19\6\6\6\6\6\6\6\6\156\156\156\71\71\156\156\156\71\71\71\71\71\24\70\156\156\24\
    \\24\24\24\47\47\47\47\47\47\47\47\156\156\156\156\156\156\156\3\3\3\3\156\3\156\3\156\156\156\156\162\162\162\162\162\162\162\162\156\156\156\63\63\63\63\63\63\63\63\213\
    \\63\58\58\58\58\58\58\58\58\156\185\156\3\3\156\3\156\156\3\156\156\156\156\3\3\3\3\3\3\167\167\155\155\155\155\155\155\155\155\156\155\110\110\110\110\110\110\110\110\156\
    \\156\156\156\115\115\156\115\117\117\117\117\117\117\117\117\156\156\70\70\156\70\156\70\70\70\70\70\70\156\70\27\156\27\27\27\27\27\27\156\156\156\27\156\156\27\54\54\54\54\54\
    \\54\54\54\156\54\13\13\13\13\13\13\13\13\156\156\113\113\113\113\113\113\113\113\156\156\156\156\156\156\156\13\13\13\13\13\13\156\156\94\94\94\94\94\94\94\94\156\94\94\94\
    \\94\156\94\130\130\130\130\130\130\130\130\156\156\32\32\32\32\32\32\32\32\156\32\32\156\156\156\156\156\156\94\94\156\156\156\156\156\156\98\98\98\98\98\98\98\98\11\156\11\11\
    \\11\11\11\11\11\156\120\120\120\120\120\120\120\120\156\156\156\156\156\156\156\83\83\83\83\83\83\83\83\45\45\156\45\45\45\45\45\45\45\45\62\156\62\62\62\62\62\62\62\62\
    \\207\207\145\207\145\145\145\145\156\156\156\156\156\156\156\145\26\26\26\26\26\26\26\26\156\156\156\35\35\35\35\35\35\35\35\156\156\8\8\8\8\8\8\8\8\156\156\118\118\118\
    \\118\118\118\118\118\156\156\65\65\65\65\65\65\65\65\156\156\34\34\156\156\34\34\34\34\34\34\34\34\156\156\156\156\156\156\156\96\125\156\156\156\156\156\156\156\131\131\131\131\131\
    \\131\131\131\87\87\87\87\87\87\87\87\156\156\87\0\0\0\0\156\156\156\156\0\0\156\3\156\3\156\3\156\3\3\3\156\3\3\3\168\171\3\3\165\43\43\43\43\43\156\43\
    \\43\43\43\43\43\43\43\156\156\43\43\43\156\43\43\3\3\3\170\170\170\170\170\170\3\3\10\156\156\10\10\10\10\10\10\10\10\156\156\174\174\174\174\174\174\174\174\46\156\156\
    \\46\46\46\46\46\46\46\46\44\156\44\44\156\44\44\44\44\44\44\156\156\208\208\208\208\208\208\208\208\115\156\115\115\156\115\115\115\115\115\115\145\156\156\156\156\156\156\145\145\156\
    \\145\145\145\145\156\156\156\145\145\156\156\156\156\145\145\148\156\148\148\148\148\148\148\148\148\156\148\148\60\156\60\60\60\60\60\60\60\60\156\60\60\60\60\156\156\214\214\214\214\214\
    \\214\214\214\80\156\80\80\80\80\80\80\80\80\156\80\80\156\151\151\151\151\151\151\151\156\156\156\38\38\38\38\38\38\38\38\156\156\156\156\156\156\156\13\104\104\104\104\104\104\104\
    \\104\156\156\156\50\50\50\50\50\50\50\50\186\186\156\140\140\140\140\140\140\140\140\156\140\140\72\72\72\72\72\72\72\156\156\156\156\156\118\118\118\142\142\156\156\156\156\156\156\71\
    \\71\71\71\71\71\71\71\24\183\183\182\24\47\47\47\156\47\47\47\47\47\47\47\24\24\24\24\24\24\24\156\24\24\24\24\156\24\24\24\24\24\24\156\24\156\24\24\24\137\137\
    \\137\137\137\137\137\137\156\156\156\3\3\172\3\3\3\3\3\172\3\3\76\76\76\76\76\76\76\76\156\156\156\64\64\64\64\64\64\156\156\156\156\64\59\59\59\156\156\156\156\156\
    \\59\156\156\20\20\20\20\20\20\20\20\156\20\20\67\67\156\67\67\67\67\67\67\67\67\42\156\42\42\156\42\42\42\42\42\42\1\1\1\1\1\1\1\1\156\156\156\156\61\61\
    \\61\61\61\61\212\212\79\79\79\79\79\79\79\79\156\156\156\156\156\156\156\84\156\84\84\156\84\84\84\84\84\84\84\156\156\156\156\156\156\84\84\156\84\84\84\84\84\84\84\84\
    \\118\118\156\118\118\118\118\118\118\156\156\40\156\156\40\40\40\40\40\40\40\156\156\156\156\156\70\70\70\70\70\70\43\70\70\164\3\3\3\3\3\3\3\156\3\3\3\3\3\3\
    \\3\3\156\156\24\156\156\156\156\156\156\4\4\4\4\4\4\4\156\156\4\4\4\170\3\3\3\3\3\3\3\169\3\3\3\81\81\81\81\81\81\81\81\156\156\81\156\44\44\156\
    \\44\44\44\156\156\44\44\44\44\156\145\145\156\145\156\145\145\145\145\156\156\207\207\207\207\207\207\207\207\145\145\145\145\145\145\145\145\148\156\148\148\148\148\156\156\148\148\148\148\156\
    \\156\148\156\156\44\44\156\156\156\156\156\156\156\80\80\80\80\156\156\80\80\69\69\156\156\69\69\69\69\69\69\69\156\156\16\16\16\16\16\16\16\16\156\156\156\156\140\140\140\140\
    \\140\156\140\140\156\156\156\156\66\66\66\66\66\66\156\156\156\156\156\156\156\29\29\29\29\29\29\29\191\72\156\156\156\72\72\72\72\156\156\156\156\99\99\99\99\99\99\99\99\156\
    \\156\156\156\15\15\15\15\15\15\15\15\156\156\15\15\24\24\70\70\24\24\24\24\24\24\43\24\157\157\157\157\157\157\157\157\156\156\156\156\124\124\124\124\124\124\124\124\156\156\156\
    \\156\21\21\21\21\21\21\156\156\21\21\21\21\74\74\74\74\74\74\74\156\74\74\156\74\189\189\189\189\189\189\189\189\156\156\156\190\166\166\166\166\166\166\166\166\156\156\156\156\117\
    \\117\117\117\117\117\156\156\156\156\156\156\60\60\156\156\60\60\60\60\116\116\116\116\116\116\116\116\156\156\156\156\19\19\19\19\19\19\19\19\156\156\156\156\158\158\158\156\158\158\158\
    \\158\158\158\158\158\122\122\122\122\122\122\122\122\156\156\156\122\64\64\64\64\156\64\64\64\64\64\64\64\161\161\156\161\161\161\156\156\156\156\156\156\39\156\156\23\23\23\23\23\23\
    \\23\23\156\156\156\156\100\100\100\100\100\100\100\100\156\100\100\100\33\33\33\33\33\33\33\33\156\156\156\156\102\102\102\102\102\102\102\102\156\156\156\156\199\199\98\156\156\156\156\156\
    \\131\131\131\131\131\156\156\156\156\0\0\0\0\0\0\0\0\156\156\156\156\61\53\53\53\53\53\53\53\24\24\156\156\156\156\156\24\24\24\24\24\156\156\24\24\24\43\43\43\43\
    \\43\43\43\43\156\43\156\43\43\29\29\29\29\29\29\29\29\193\191\192\192\191\3\3\3\3\168\3\3\3\3\3\24\3\3\3\3\3\138\138\138\156\156\138\138\138\138\138\138\156\
    \\138\101\101\101\101\101\101\101\101\156\156\101\101\101\31\175\31\205\31\31\31\31\178\179\55\55\31\31\31\31\31\31\31\31\177\176\198\198\198\198\198\198\198\198\156\132\132\132\156\132\
    \\132\132\132\132\132\132\132\156\156\156\150\150\150\150\150\150\150\150\156\156\156\156\24\156\38\38\38\38\38\38\156\38\38\38\38\156\156\38\38\38\38\38\156\156\156\125\125\125\125\125\
    \\125\125\125\24\24\24\125\125\41\41\41\41\41\41\41\41\156\156\156\156\156\109\109\109\109\109\109\109\109\156\156\156\156\156\158\158\158\156\158\158\158\158\156\158\158\156\158\158\156\156\
    \\156\195\195\195\196\196\196\197\197\156\156\156\156\156\156\51\51\51\51\51\51\51\51\156\51\51\156\156\56\56\56\56\56\56\56\56\156\156\156\156\156\75\75\75\75\75\75\75\75\156\
    \\156\156\156\156\156\156\106\106\106\106\106\106\106\106\156\156\156\156\156\59\59\59\59\59\59\59\59\156\156\156\156\156\68\68\68\68\68\68\68\68\156\156\156\156\156\42\207\42\207\156\
    \\42\42\42\42\42\156\156\42\1\1\1\156\156\1\1\1\1\156\156\156\156\160\160\160\160\160\160\160\160\156\156\156\156\156\135\135\135\135\135\135\135\135\156\156\156\156\156\62\156\62\
    \\62\62\62\62\62\156\156\156\62\62\28\28\28\28\28\28\28\28\156\156\156\156\156\93\93\93\93\93\93\93\156\156\156\156\93\93\146\146\146\146\146\146\146\156\156\156\156\156\156\85\
    \\85\85\85\85\85\85\85\156\156\156\156\156\90\90\90\90\90\90\90\90\156\156\156\156\90\61\61\61\61\156\61\61\61\61\156\61\61\156\24\24\12\12\24\24\24\24\55\55\24\24\
    \\24\24\24\24\55\55\55\55\55\55\55\55\156\156\43\43\156\43\43\43\43\43\25\25\25\25\25\25\46\156\46\46\156\46\46\156\46\46\46\46\46\46\44\156\44\44\156\44\44\44\
    \\156\156\44\44\44\44\187\187\187\187\187\187\187\187\95\95\95\95\95\95\95\95\156\99\99\99\156\156\156\99\99\156\156\156\156\156\156\127\127\127\127\127\127\156\156\156\156\156\156\127\
    \\127\58\58\156\156\156\156\58\58\58\58\58\58\156\185\77\77\77\77\77\77\77\77\156\156\156\156\156\77\49\49\49\49\49\49\49\49\156\156\156\156\156\156\161\161\161\161\161\161\161\
    \\161\156\161\161\161\156\156\133\133\133\133\133\133\133\133\156\156\156\156\156\156\114\114\114\114\114\114\114\114\156\156\156\156\156\156\67\67\156\67\67\67\67\67\156\156\156\156\156\156\42\
    \\156\42\42\156\42\42\42\156\207\207\42\42\42\100\100\100\100\156\100\100\100\156\156\156\156\156\156\153\153\153\153\153\153\153\153\156\156\156\156\156\156\144\144\144\144\144\144\144\144\156\
    \\156\156\156\156\156\62\62\62\156\156\156\62\62\156\156\156\156\156\156\61\61\61\156\156\156\156\156\53\156\156\156\156\156\40\156\156\40\40\40\40\40\156\40\40\156\40\40\40\40\40\
    \\40\40\40\159\159\159\159\159\159\159\159\156\156\156\156\156\159\97\97\97\97\97\97\97\97\156\156\156\156\156\156\7\7\7\7\7\7\7\7\156\156\156\156\156\156\156\152\152\152\152\
    \\152\152\152\152\74\74\74\156\74\74\156\74\74\74\74\74\74\156\156\147\147\147\147\147\147\147\147\102\47\47\65\156\156\156\136\136\136\136\136\136\136\136\105\105\105\105\105\105\105\105\
    \\14\14\14\14\14\14\14\14\47\47\47\47\210\210\47\47\211\211\211\211\211\211\211\211\121\121\121\121\121\121\121\121\30\30\30\30\30\30\30\30\129\129\129\129\129\129\129\129\36\36\
    \\36\36\36\36\36\36\119\119\119\119\119\119\119\119\89\89\89\89\89\89\89\89\112\112\112\112\112\112\112\112\108\108\108\108\108\108\108\108\111\111\111\111\111\111\111\111\128\128\128\128\
    \\128\128\128\128\163\163\163\163\163\163\163\163"#

encodedScriptExtensionsOffsets1BitMap :: Ptr Word16
encodedScriptExtensionsOffsets1BitMap = Ptr
    "\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\85\14\117\1\85\14\149\2\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\
    \\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\203\12\208\12\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\221\6\223\6\
    \\117\1\117\1\117\1\117\1\117\1\123\5\123\5\123\5\125\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\100\3\
    \\101\3\101\3\106\3\100\3\101\3\101\3\106\3\163\4\163\4\163\4\163\4\163\4\114\3\163\4\114\3\101\3\101\3\102\3\101\3\101\3\101\3\102\3\101\3\101\3\101\3\101\3\76\1\71\1\101\3\101\3\101\3\101\3\101\3\77\1\246\2\82\1\101\3\74\1\101\3\101\3\101\3\101\3\101\3\101\3\101\3\75\1\252\2\232\12\232\12\232\12\
    \\232\12\232\12\232\12\87\3\123\5\123\5\123\5\123\5\123\5\123\5\123\5\129\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\117\1\117\1\117\1\117\1\132\14\163\4\163\4\163\4\198\3\101\3\
    \\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\197\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\204\3\237\5\205\3\117\1\117\1\20\9\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\101\3\
    \\101\3\101\3\101\3\101\3\101\3\101\3\101\3\108\3\163\4\163\4\163\4\163\4\104\3\216\12\163\4\163\4\163\4\163\4\163\4\99\0\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\99\0\117\1\163\4\75\8\162\4\163\4\162\4\163\4\162\4\163\4\163\4\163\4\97\0\117\1\123\5\123\5\123\5\123\5\
    \\123\5\123\5\123\5\123\5\123\5\128\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\49\9\117\1\117\1\117\1\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\99\0\117\1\163\4\97\0\163\4\98\0\163\4\92\1\
    \\163\4\163\4\163\4\163\4\163\4\76\8\97\0\157\4\163\4\99\0\163\4\92\1\163\4\92\1\163\4\163\4\163\4\163\4\75\8\117\1\117\1\117\1\163\4\90\1\117\1\117\1\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\
    \\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\97\0\117\1\117\1\117\1\117\1\117\1\117\1\157\4\163\4\163\4\163\4\163\4\75\8\162\4\163\4\163\4\163\4\163\4\163\4\163\4\10\3\8\3\163\4\229\12\1\3\6\3\163\4\163\4\163\4\4\3\163\4\163\4\163\4\163\4\163\4\163\4\163\4\90\1\117\1\117\1\76\8\22\11\
    \\77\8\77\8\163\4\163\4\163\4\80\8\89\8\161\4\86\8\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\97\0\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\75\8\160\4\163\4\163\4\163\4\
    \\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\91\1\163\4\99\0\92\1\117\1\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\85\4\117\1\117\1\117\1\117\1\
    \\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\208\8\208\8\208\8\215\8\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\83\4\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\
    \\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\138\12\138\12\138\12\138\12\138\12\138\12\138\12\138\12\138\12\138\12\138\12\138\12\143\12\117\1\188\1\188\1\188\1\188\1\241\13\117\1\247\13\117\1\117\1\117\1\193\1\117\1\196\8\117\1\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\
    \\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\224\10\126\4\126\4\126\4\126\4\126\4\129\4\126\4\128\4\126\4\132\4\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\
    \\117\1\117\1\117\1\117\1\117\1\117\1\140\4\140\4\140\4\141\4\117\1\117\1\19\14\19\14\19\14\19\14\19\14\19\14\19\14\25\14\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\150\5\
    \\150\5\150\5\153\5\150\5\150\5\150\5\150\5\150\5\150\5\151\5\117\1\19\12\19\12\19\12\19\12\19\12\19\12\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\3\7\241\10\
    \\117\1\0\7\2\7\3\7\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\67\6\67\6\67\6\67\6\67\6\67\6\69\6\67\6\67\6\67\6\67\6\69\6\117\1\117\1\117\1\117\1\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\
    \\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\251\7\251\7\251\7\254\7\179\11\179\11\179\11\179\11\179\11\179\11\179\11\179\11\179\11\184\11\179\11\251\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\181\6\82\4\117\1\
    \\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\85\4\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\122\8\122\8\122\8\125\8\57\2\57\2\57\2\57\2\57\2\57\2\64\2\117\1\77\10\77\10\
    \\77\10\81\10\163\4\163\4\163\4\163\4\9\10\5\10\114\3\163\4\163\4\110\3\163\4\163\4\101\3\101\3\101\3\101\3\101\3\89\1\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\
    \\163\4\163\4\159\4\163\4\98\0\163\4\98\0\249\10\249\10\249\10\249\10\249\10\249\10\249\10\249\10\249\10\22\7\249\10\24\7\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\162\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\98\0\117\1\117\1\
    \\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\236\7\180\1\235\7\236\7\162\4\163\4\163\4\163\4\163\4\163\4\163\4\97\0\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\
    \\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\162\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\98\0\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\101\13\101\13\101\13\101\13\101\13\117\1\101\13\107\13\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\
    \\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\40\9\40\9\40\9\41\9\115\13\115\13\115\13\115\13\115\13\169\10\173\10\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\133\5\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\21\11\163\4\163\4\163\4\
    \\76\8\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\97\0\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\21\11\163\4\163\4\
    \\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\79\8\163\4\163\4\163\4\163\4\163\4\163\4\90\1\117\1\117\1\117\1\117\1\163\4\91\1\191\6\191\6\191\6\191\6\191\6\191\6\191\6\191\6\191\6\191\6\191\6\191\6\191\6\191\6\191\6\191\6\191\6\191\6\191\6\191\6\
    \\191\6\191\6\191\6\191\6\191\6\191\6\191\6\191\6\191\6\191\6\191\6\191\6\193\6\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\33\14\33\14\33\14\33\14\33\14\39\14\76\8\22\11\77\8\77\8\163\4\
    \\163\4\163\4\80\8\89\8\161\4\86\8\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\77\8\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\77\8\167\2\82\8\163\4\92\8\78\8\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\
    \\93\0\163\4\163\4\97\0\117\1\117\1\117\1\117\1\123\5\123\5\123\5\65\8\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\127\5\117\1\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\129\5\117\1\123\5\123\5\123\5\123\5\
    \\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\130\5\117\1\117\1\117\1\163\4\163\4\163\4\163\4\163\4\163\4\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\
    \\232\12\232\12\232\12\232\12\232\12\232\12\232\12\232\12\232\12\232\12\232\12\232\12\232\12\232\12\232\12\232\12\232\12\232\12\232\12\232\12\152\0\232\12\232\12\232\12\92\3\101\3\72\2\70\2\63\1\35\11\57\7\57\7\54\7\57\7\57\7\57\7\57\7\57\7\57\7\57\7\78\2\80\2\57\7\57\7\59\7\59\7\57\7\57\7\57\7\57\7\
    \\59\7\59\7\57\7\174\2\57\7\57\7\57\7\59\7\57\7\57\7\57\7\57\7\57\7\57\7\51\7\57\7\51\7\57\7\61\7\52\7\57\7\57\7\65\7\58\7\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\181\6\82\4\117\1\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\180\6\
    \\180\6\180\6\180\6\180\6\180\6\180\6\83\4\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\99\4\99\4\99\4\99\4\99\4\99\4\99\4\99\4\99\4\99\4\99\4\99\4\99\4\99\4\99\4\99\4\99\4\99\4\99\4\99\4\
    \\99\4\99\4\99\4\99\4\99\4\99\4\99\4\99\4\99\4\99\4\99\4\99\4\100\4\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\211\6\211\6\211\6\211\6\211\6\211\6\211\6\211\6\213\6\117\1\211\6\252\8\252\8\211\6\
    \\211\6\35\8\211\6\211\6\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\177\12\177\12\177\12\177\12\177\12\177\12\177\12\177\12\177\12\177\12\177\12\182\12\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\61\11\56\11\40\9\43\7\40\9\40\9\40\9\40\9\
    \\31\9\73\7\76\7\40\9\252\3\2\4\69\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\73\9\61\11\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\
    \\40\9\40\9\193\5\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\191\5\40\9\40\9\40\9\40\9\40\9\40\9\131\5\117\1\117\1\117\1\117\1\111\8\114\8\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\
    \\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\54\14\108\4\108\4\108\4\109\4\108\4\156\12\117\4\117\4\117\4\117\4\117\4\117\4\117\4\117\4\117\4\118\4\117\4\169\12\201\6\201\6\201\6\203\6\203\6\117\1\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\77\8\163\4\163\4\163\4\163\4\163\4\
    \\163\4\163\4\163\4\77\8\167\2\82\8\163\4\92\8\78\8\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\75\8\117\1\123\5\123\5\73\8\92\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\32\5\32\5\32\5\32\5\34\5\176\10\62\14\62\14\62\14\
    \\62\14\62\14\62\14\62\14\184\3\183\3\184\3\235\7\235\7\236\7\117\1\236\7\236\7\236\7\236\7\236\7\236\7\236\7\236\7\43\11\43\11\43\11\43\11\43\11\43\11\43\11\43\11\43\11\43\11\43\11\43\11\43\11\43\11\43\11\43\11\48\11\43\11\43\11\43\11\43\11\43\11\43\11\43\11\43\11\43\11\43\11\43\11\43\11\43\11\43\11\
    \\43\11\207\1\208\1\208\1\208\1\209\1\207\1\208\1\208\1\208\1\208\1\208\1\61\9\132\3\133\3\133\3\133\3\133\3\133\3\133\3\117\1\133\3\133\3\133\3\180\4\178\4\117\1\67\3\67\3\67\3\67\3\67\3\67\3\67\3\67\3\67\3\67\3\67\3\67\3\67\3\67\3\67\3\67\3\67\3\67\3\67\3\67\3\67\3\67\3\67\3\
    \\67\3\67\3\67\3\67\3\67\3\67\3\67\3\67\3\67\3\68\3\117\1\67\3\67\3\69\3\117\1\67\3\117\1\117\1\117\1\242\5\101\3\101\3\101\3\101\3\101\3\240\5\204\3\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\
    \\123\5\123\5\123\5\123\5\123\5\123\5\123\5\49\9\117\1\117\1\117\1\163\4\163\4\163\4\163\4\163\4\163\4\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\232\12\232\12\232\12\232\12\232\12\232\12\232\12\232\12\232\12\232\12\234\12\232\12\232\12\233\12\117\1\163\4\163\4\
    \\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\99\0\117\1\117\1\117\1\117\1\117\1\117\1\117\1\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\17\10\
    \\17\10\17\10\21\10\117\1\117\1\43\11\43\11\43\11\43\11\43\11\217\9\43\11\43\11\43\11\43\11\43\11\43\11\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\47\14\117\1\200\3\117\1\205\1\128\3\133\3\133\3\134\3\136\3\172\4\133\3\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\
    \\40\9\40\9\40\9\7\4\117\1\190\5\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\5\11\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\
    \\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\241\13\117\1\247\13\117\1\117\1\117\1\193\1\117\1\196\8\117\1\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\220\10\123\5\123\5\
    \\123\5\123\5\123\5\123\5\123\5\123\5\123\5\128\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\125\5\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\
    \\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\235\7\235\7\235\7\235\7\235\7\235\7\235\7\235\7\235\7\163\11\236\7\163\11\235\7\235\7\235\7\235\7\235\7\163\11\235\7\235\7\235\7\235\7\163\11\236\7\163\11\235\7\236\7\235\7\235\7\235\7\235\7\235\7\163\11\235\7\235\7\235\7\235\7\235\7\235\7\235\7\235\7\
    \\166\11\235\7\235\7\235\7\238\7\235\7\235\7\235\7\241\7\42\5\42\5\42\5\42\5\42\5\42\5\42\5\42\5\42\5\42\5\44\5\44\5\163\4\163\4\163\4\163\4\9\10\5\10\114\3\163\4\163\4\110\3\163\4\163\4\101\3\101\3\101\3\101\3\101\3\89\1\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\
    \\163\4\163\4\163\4\220\12\163\4\163\4\163\4\164\4\163\4\163\4\163\4\163\4\163\4\163\4\77\8\163\4\115\5\109\3\163\4\75\8\101\3\202\3\163\4\163\4\163\4\163\4\92\1\117\1\232\12\232\12\232\12\232\12\60\0\117\1\204\14\204\14\204\14\204\14\204\14\117\1\129\13\129\13\129\13\129\13\129\13\135\13\117\1\117\1\143\13\143\13\
    \\143\13\149\13\117\1\117\1\117\1\117\1\184\10\184\10\184\10\188\10\117\1\117\1\37\4\37\4\38\4\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\40\9\40\9\40\9\41\9\115\13\115\13\115\13\115\13\115\13\169\10\173\10\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\133\5\190\12\190\12\190\12\
    \\190\12\190\12\190\12\190\12\190\12\190\12\195\12\190\12\190\12\190\12\190\12\190\12\190\12\190\12\238\2\190\12\190\12\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\92\14\117\1\129\5\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\177\12\177\12\177\12\177\12\177\12\177\12\177\12\177\12\177\12\177\12\177\12\182\12\117\1\117\1\
    \\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\188\4\190\4\188\4\188\4\188\4\188\4\188\4\188\4\188\4\72\11\40\9\40\9\40\9\40\9\40\9\40\9\149\4\149\4\149\4\149\4\149\4\149\4\155\4\117\1\85\11\85\11\85\11\85\11\85\11\85\11\85\11\90\11\144\3\144\3\144\3\144\3\144\3\146\3\144\3\145\3\
    \\81\9\81\9\81\9\85\9\188\4\125\3\40\9\40\9\40\9\41\9\8\4\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\64\11\40\9\40\9\40\9\110\11\110\11\110\11\110\11\110\11\110\11\110\11\110\11\110\11\110\11\105\11\110\11\114\11\120\11\110\11\110\11\136\1\203\4\84\7\200\4\200\4\198\4\126\1\132\1\203\4\133\1\
    \\118\1\121\1\91\7\97\7\200\4\201\4\211\4\221\4\105\7\216\4\216\4\214\4\222\2\218\2\221\4\214\2\236\0\231\0\207\3\213\3\217\4\117\1\229\4\236\4\231\4\234\4\234\4\232\4\116\7\97\9\236\4\93\9\151\9\117\1\123\7\129\7\150\9\233\4\244\4\113\2\110\2\125\2\125\2\247\4\137\7\109\2\113\2\117\2\103\2\219\5\
    \\114\2\125\2\125\2\117\1\153\7\159\7\156\7\105\9\162\7\159\7\125\9\165\7\159\7\156\7\148\7\117\1\74\4\117\9\121\9\169\6\175\7\178\7\173\7\175\7\175\7\173\7\175\7\137\9\178\7\133\9\95\1\142\9\135\9\175\7\93\1\175\7\188\7\191\7\186\7\188\7\188\7\186\7\192\7\101\10\191\7\195\7\96\10\96\10\197\7\203\7\
    \\200\1\117\1\213\7\216\7\211\7\213\7\213\7\213\7\213\7\213\7\216\7\211\7\155\9\213\7\159\9\213\7\213\7\213\7\128\11\157\1\158\1\155\1\157\1\157\1\130\11\161\1\158\1\150\1\160\1\157\1\87\4\157\1\91\4\117\1\156\2\157\2\157\2\157\2\157\2\157\2\157\2\162\2\157\2\157\2\157\2\161\2\117\1\117\1\117\1\117\1\
    \\15\5\1\5\5\5\5\5\9\5\5\5\5\5\172\9\8\5\6\5\5\5\167\9\117\1\117\1\117\1\117\1\169\1\169\1\169\1\169\1\169\1\169\1\169\1\169\1\169\1\168\1\169\1\169\1\169\1\227\7\168\1\169\1\169\1\169\1\169\1\168\1\169\1\169\1\169\1\172\1\169\1\172\1\192\3\48\1\117\1\117\1\117\1\117\1\36\13\36\13\
    \\36\13\36\13\36\13\36\13\36\13\36\13\28\13\34\13\36\13\36\13\36\13\36\13\36\13\36\13\36\13\36\13\36\13\36\13\32\5\32\5\32\5\32\5\34\5\176\10\32\5\32\5\32\5\32\5\32\5\28\5\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\
    \\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\52\5\52\5\54\5\54\4\6\8\6\8\9\8\117\1\180\9\180\9\184\9\117\1\17\8\20\8\196\9\117\1\62\5\62\5\62\5\62\5\62\5\62\5\62\5\62\5\62\5\62\5\62\5\64\5\62\5\208\9\62\5\208\9\103\1\109\1\109\1\115\1\109\1\109\1\
    \\109\1\109\1\109\1\109\1\109\1\109\1\109\1\109\1\109\1\116\1\109\1\109\1\109\1\109\1\109\1\114\1\38\3\38\3\38\3\38\3\38\3\38\3\38\3\38\3\40\3\117\1\154\3\154\3\154\3\155\3\154\3\31\8\154\3\31\8\225\9\154\3\72\5\72\5\72\5\163\3\164\3\117\1\237\9\237\9\237\9\237\9\237\9\82\5\237\9\237\9\
    \\237\9\84\5\237\9\45\13\62\5\62\5\62\5\62\5\249\9\249\9\249\9\253\9\172\3\172\3\172\3\172\3\172\3\172\3\172\3\173\3\172\3\172\3\172\3\175\3\172\3\43\8\172\3\43\8\172\3\174\3\232\12\232\12\232\12\233\12\117\1\117\1\117\1\117\1\117\1\117\1\94\5\94\5\94\5\94\5\94\5\94\5\94\5\94\5\94\5\97\5\
    \\94\5\94\5\94\5\94\5\94\5\95\5\100\14\100\14\100\14\100\14\100\14\100\14\100\14\100\14\198\2\198\2\198\2\198\2\198\2\198\2\202\2\206\2\51\8\51\8\51\8\51\8\51\8\51\8\51\8\107\5\51\8\105\5\108\14\108\14\108\14\108\14\108\14\108\14\43\11\169\0\32\5\32\5\32\5\32\5\32\5\23\5\100\14\117\1\33\0\39\0\
    \\118\0\46\0\52\0\232\10\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\116\14\11\14\11\14\11\14\11\14\11\14\11\14\11\14\11\14\11\14\11\14\11\14\11\14\101\3\101\3\
    \\101\3\101\3\80\2\80\2\80\2\80\2\80\2\80\2\80\2\80\2\80\2\80\2\80\2\80\2\80\2\80\2\110\0\106\0\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\125\5\117\1\117\1\117\1\163\4\99\0\
    \\59\8\32\2\28\2\42\2\122\5\34\1\49\2\124\14\187\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\85\0\80\0\25\1\26\1\26\1\26\1\26\1\26\1\26\1\26\1\26\1\26\1\26\1\21\1\230\3\235\3\235\3\235\3\235\3\235\3\55\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\56\1\
    \\123\5\123\5\235\3\235\3\235\3\235\3\123\5\123\5\123\5\123\5\127\5\117\1\26\1\26\1\40\1\40\1\40\1\56\1\123\5\123\5\123\5\123\5\123\5\163\4\163\4\163\4\40\1\40\1\40\1\41\1\123\5\123\5\123\5\123\5\123\5\123\5\74\8\163\4\123\5\71\8\26\1\26\1\26\1\26\1\26\1\27\1\26\1\26\1\26\1\26\1\
    \\26\1\26\1\26\1\26\1\26\1\26\1\26\1\123\5\123\5\123\5\74\8\120\5\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\163\4\123\5\123\5\123\5\68\8\100\8\100\8\100\8\100\8\100\8\103\8\235\11\241\11\140\14\140\14\140\14\140\14\140\14\140\14\140\14\117\1\133\2\133\2\133\2\133\2\133\2\133\2\
    \\133\2\133\2\135\2\141\2\133\2\139\2\110\11\110\11\98\11\110\11\161\5\161\5\161\5\161\5\161\5\163\5\29\10\29\10\29\10\29\10\33\10\7\1\40\1\40\1\40\1\58\1\171\5\171\5\171\5\171\5\171\5\171\5\171\5\171\5\171\5\173\5\171\5\73\13\36\13\36\13\36\13\37\13\77\3\77\3\77\3\77\3\77\3\77\3\78\3\117\1\
    \\77\3\79\3\77\3\45\10\36\13\36\13\36\13\36\13\68\0\68\0\68\0\68\0\68\0\68\0\68\0\68\0\73\0\117\1\117\1\65\0\221\3\221\3\222\3\117\1\157\11\157\11\157\11\117\1\236\7\236\7\101\3\101\3\101\3\101\3\101\3\97\3\23\9\0\0\42\5\42\5\42\5\42\5\42\5\42\5\42\5\42\5\42\5\42\5\221\3\221\3\
    \\221\3\221\3\221\3\223\3\221\3\227\3\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\40\1\59\1\117\1\40\1\40\1\56\1\53\1\40\1\40\1\40\1\40\1\40\1\59\1\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\
    \\123\5\125\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\123\5\129\5\117\1\117\1\117\1\117\1\232\12\232\12\163\4\91\1\232\12\158\0\163\4\163\4\117\3\163\4\79\8\163\4\75\8\99\0\34\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\40\9\
    \\40\9\43\9\162\4\163\4\163\4\163\4\100\3\101\3\101\3\106\3\100\3\101\3\101\3\106\3\15\1\26\1\25\1\26\1\26\1\26\1\26\1\200\8\40\1\40\1\40\1\56\1\54\1\54\1\54\1\18\3\75\8\75\8\117\1\19\11\88\2\92\2\88\2\88\2\89\2\88\2\88\2\57\10\88\2\97\2\88\2\97\2\117\1\117\1\117\1\117\1\
    \\88\2\88\2\88\2\88\2\88\2\88\2\88\2\88\2\88\2\88\2\88\2\88\2\88\2\88\2\88\2\100\2\2\2\9\2\9\2\9\2\9\2\9\2\13\2\20\2\57\7\57\7\57\7\57\7\57\7\57\7\57\7\57\7\57\7\58\7\163\4\98\0\144\1\117\1\117\1\117\1\117\1\117\1\163\4\163\4\163\4\163\4\163\4\12\3\182\2\182\2\
    \\182\2\182\2\186\2\190\2\192\11\192\11\192\11\197\11\205\11\205\11\205\11\205\11\205\11\210\11\201\5\201\5\201\5\203\5\211\5\211\5\211\5\211\5\215\5\211\5\213\5\117\1\117\1\117\1\117\1\117\1\148\14\148\14\148\14\148\14\148\14\148\14\148\14\148\14\148\14\148\14\156\14\156\14\156\14\156\14\156\14\156\14\227\5\227\5\227\5\229\5\
    \\227\5\93\10\109\10\109\10\109\10\109\10\113\10\109\10\109\10\109\10\109\10\113\10\164\14\164\14\164\14\164\14\164\14\117\1\121\10\121\10\121\10\121\10\121\10\121\10\125\10\86\5\137\10\133\10\137\10\133\10\223\11\137\10\134\10\137\10\134\10\227\11\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\252\5\26\3\28\3\28\3\28\3\28\3\
    \\30\3\1\6\9\6\9\6\11\6\9\6\172\14\172\14\172\14\172\14\243\3\243\3\243\3\244\3\243\6\243\3\117\1\117\1\117\1\117\1\117\1\117\1\249\11\249\11\254\11\246\11\145\10\145\10\145\10\149\10\87\13\87\13\87\13\93\13\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\180\14\180\14\180\14\180\14\244\0\244\0\244\0\240\0\
    \\244\0\244\0\242\0\244\0\244\0\244\0\244\0\244\0\20\4\12\4\157\10\15\4\217\0\217\0\219\0\136\8\217\0\224\0\217\0\224\0\188\14\188\14\188\14\188\14\196\14\196\14\196\14\196\14\117\1\117\1\117\1\117\1\28\4\28\4\28\4\28\4\177\0\5\0\11\0\117\1\207\0\207\0\207\0\207\0\207\0\207\0\209\0\206\0\197\0\197\0\
    \\199\0\197\0\6\12\6\12\11\12\6\12\19\0\19\0\25\0\184\0\117\1\18\0\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\29\6\29\6\29\6\29\6\29\6\29\6\29\6\29\6\29\6\36\6\117\1\117\1\117\1\117\1\117\1\117\1\34\12\34\12\34\12\34\12\34\12\34\12\39\12\117\1\34\12\34\12\34\12\34\12\
    \\34\12\34\12\39\12\32\12\19\6\19\6\19\6\19\6\19\6\19\6\19\6\19\6\19\6\21\6\42\6\19\6\19\6\19\6\21\6\243\7\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\144\8\147\8\46\4\46\4\46\4\53\4\46\4\52\4\155\8\155\8\155\8\155\8\155\8\155\8\158\8\155\8\155\8\117\1\62\4\62\4\62\4\62\4\
    \\63\4\117\1\212\14\212\14\212\14\212\14\212\14\212\14\212\14\212\14\212\14\212\14\212\14\212\14\156\1\157\1\136\11\117\1\169\8\169\8\166\8\169\8\169\8\169\8\169\8\169\8\163\13\117\1\117\1\117\1\117\1\117\1\117\1\117\1\53\6\59\6\52\6\54\6\52\6\94\6\60\12\60\12\60\12\60\12\60\12\60\12\60\12\71\4\60\12\72\4\
    \\73\12\220\1\217\1\232\1\232\1\239\1\177\8\177\13\220\1\224\1\248\1\250\1\221\1\244\1\244\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\196\10\196\10\196\10\196\10\196\10\196\10\196\10\196\10\196\10\196\10\196\10\200\10\191\13\117\1\117\1\117\1\199\13\199\13\
    \\199\13\199\13\199\13\199\13\199\13\199\13\199\13\117\1\199\13\205\13\117\1\117\1\117\1\117\1\230\2\230\2\230\2\230\2\230\2\230\2\230\2\230\2\233\2\117\1\230\2\236\2\109\1\112\1\117\1\117\1\213\13\213\13\213\13\213\13\213\13\213\13\213\13\219\13\213\13\219\13\117\1\117\1\117\1\117\1\117\1\117\1\188\8\188\8\188\8\86\12\
    \\188\8\192\8\188\8\188\8\189\8\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\208\10\208\10\208\10\208\10\208\10\208\10\208\10\212\10\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\99\12\99\12\
    \\99\12\99\12\99\12\99\12\99\12\99\12\99\12\99\12\104\12\189\0\78\6\140\0\81\6\77\6\77\6\77\6\144\0\141\0\78\6\117\1\77\6\86\6\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\102\6\130\0\102\6\102\6\102\6\102\6\102\6\130\0\133\0\117\1\117\1\117\1\220\14\220\14\220\14\220\14\220\14\220\14\220\14\220\14\
    \\220\14\117\1\112\12\112\12\112\12\112\12\112\12\112\12\112\12\112\12\112\12\112\12\117\12\117\1\38\3\38\3\120\6\120\6\120\6\120\6\120\6\120\6\120\6\127\6\110\11\124\0\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\
    \\117\1\117\1\117\1\117\1\117\1\117\1\56\3\110\6\56\3\56\3\56\3\56\3\57\3\56\3\58\3\117\1\56\3\56\3\56\3\59\3\135\6\135\6\135\6\135\6\46\3\135\6\135\6\47\3\48\3\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\228\8\225\8\244\8\244\8\244\8\244\8\228\8\221\8\244\8\117\1\244\8\233\8\
    \\254\0\144\6\252\0\252\0\252\0\253\0\143\6\6\1\252\0\5\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\156\6\156\6\154\6\156\6\156\6\156\6\156\6\130\12\156\6\156\6\156\6\233\13\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\26\12\117\1\125\9\125\9\164\6\125\9\125\9\125\9\
    \\170\6\172\6\235\6\235\6\235\6\235\6\235\6\235\6\235\6\235\6\235\6\235\6\235\6\235\6\235\6\240\6\235\6\238\6\235\6\242\6\235\6\231\6\239\6\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\57\7\57\7\57\7\57\7\57\7\57\7\57\7\57\7\59\7\117\1\117\1\117\1\117\1\117\1\117\1\117\1\
    \\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\163\4\163\4\99\0\117\1\163\4\163\4\99\0\117\1\101\3\101\3\101\3\200\3\17\9\204\3\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\10\9\11\14\
    \\11\14\7\9\5\14\14\9\43\11\43\11\43\11\43\11\43\11\43\11\43\11\164\0\117\1\117\1\117\1\210\9\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\11\7\11\7\11\7\11\7\11\7\11\7\11\7\11\7\11\7\11\7\11\7\11\7\11\7\11\7\11\7\11\7\11\7\11\7\11\7\11\7\
    \\11\7\11\7\11\7\11\7\14\7\11\7\12\7\117\1\117\1\117\1\117\1\117\1\35\9\40\9\40\9\40\9\181\5\39\9\36\9\142\5\186\5\34\7\181\5\32\7\181\5\36\9\36\9\137\5\40\9\37\9\40\9\6\4\38\7\37\9\40\9\6\4\117\1\117\1\117\1\117\1\117\1\117\1\8\4\117\1\12\11\117\1\163\4\163\4\163\4\163\4\
    \\163\4\99\0\163\4\92\1\129\5\117\1\97\0\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1\163\4\99\0\163\4\163\4\163\4\163\4\163\4\163\4\163\4\117\1\163\4\91\1\163\4\163\4\163\4\163\4\163\4\117\1\163\4\163\4\163\4\97\0\91\1\117\1\
    \\117\1\117\1\117\1\117\1\117\1\117\1\117\1\117\1"#

encodedScriptExtensionsOffsets2BitMap :: Ptr Word16
encodedScriptExtensionsOffsets2BitMap = Ptr
    "\141\0\19\1\28\1\26\6\227\7\253\7\0\7\150\10\182\10\214\10\246\10\22\11\54\11\86\11\118\11\150\11\182\11\214\11\155\9\185\9\118\3\118\3\134\3\246\11\22\12\54\12\86\12\118\12\150\12\172\0\19\1\56\6\248\9\234\3\170\1\170\1\158\1\170\1\170\1\170\1\182\12\170\1\170\1\208\4\214\12\199\7\162\5\246\12\22\13\54\13\
    \\86\13\118\13\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\117\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\
    \\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\
    \\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\2\3\2\3\2\3\2\3\17\3\172\8\199\8\244\0\150\13\182\13\214\13\246\13\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\
    \\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\214\11\22\14\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\109\0\
    \\54\14\231\8\28\7\53\7\86\14\118\14\150\14\182\14\202\3\214\14\246\14\22\15\29\8\55\8\54\15\86\15\118\15\150\15\182\15\156\4\176\4\24\10\214\15\246\15\22\16\54\16\86\16\86\3\118\16\150\16\182\16\214\16\246\16\22\17\54\17\86\17\88\2\118\17\51\2\51\2\51\2\64\2\166\3\120\2\84\2\84\2\84\2\84\2\84\2\84\2\
    \\84\2\84\2\84\2\134\2\50\5\50\5\50\5\50\5\72\5\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\144\6\144\6\168\6\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\
    \\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\85\7\85\7\110\7\200\6\84\2\84\2\224\6\87\10\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\1\0\0\0\0\0\0\0\67\0\73\0\33\0\84\2\84\2\
    \\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\35\0\34\9\166\2\180\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\150\17\84\2\84\2\84\2\84\2\84\2\
    \\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\140\8\172\1\210\1\182\17\167\7\137\5\242\1\254\1\248\4\49\3\49\3\64\3\84\2\84\2\84\2\84\2\214\17\246\17\212\2\226\2\84\2\83\5\84\2\84\2\74\4\22\18\33\4\84\2\84\2\51\4\106\4\54\18\84\2\58\1\181\1\86\18\170\1\
    \\170\1\170\1\1\4\19\2\118\18\170\1\126\1\18\5\84\2\84\2\84\2\84\2\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\
    \\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\
    \\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\
    \\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\212\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\204\0\105\0\109\0\109\0\109\0\
    \\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\194\5\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\216\5\84\2\84\2\
    \\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\84\2\109\0\109\0\123\9\84\2\84\2\84\2\84\2\84\2\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\90\1\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\109\0\
    \\109\0\109\0\109\0\100\1\248\5\14\6"#

