{-# LANGUAGE DataKinds #-}
{-# LANGUAGE DefaultSignatures #-}
{-# LANGUAGE DeriveGeneric #-}
{-# LANGUAGE DerivingVia #-}
{-# LANGUAGE EmptyDataDecls #-}
{-# LANGUAGE FlexibleContexts #-}
{-# LANGUAGE FlexibleInstances #-}
{-# LANGUAGE FunctionalDependencies #-}
{-# LANGUAGE GADTs #-}
{-# LANGUAGE GeneralizedNewtypeDeriving #-}
{-# LANGUAGE MultiParamTypeClasses #-}
{-# LANGUAGE OverloadedStrings #-}
{-# LANGUAGE ScopedTypeVariables #-}
{-# LANGUAGE StandaloneDeriving #-}
{-# LANGUAGE TypeFamilies #-}
{-# LANGUAGE TypeOperators #-}
{-# LANGUAGE TypeSynonymInstances #-}
{-# LANGUAGE UndecidableInstances #-}

module Network.URI.Template.Types where

import Control.Arrow (Arrow ((***)))
import Data.Bifunctor (Bifunctor (bimap))
import Data.Char (isUpper, toLower)
import Data.Kind (Type)
import Data.Fixed (Fixed, HasResolution)
import Data.Foldable as F (Foldable (toList))
import Data.Functor.Const (Const)
import Data.Functor.Identity (Identity)
import qualified Data.Text as T
import GHC.Generics
import qualified Data.HashMap.Strict as HS
import Data.Int (Int16, Int32, Int64, Int8)
import Data.List (intercalate)
import qualified Data.List.NonEmpty as NE
import qualified Data.Map.Strict as MS
import Data.Monoid (All, Any, Dual, First, Last, Product, Sum)
import Data.Semigroup (Max, Min)
import qualified Data.String as S
import Data.Tagged (Tagged, unTagged)
import qualified Data.Text as T
import qualified Data.Text.Lazy as TL
import Data.Time (
  Day,
  DayOfWeek,
  LocalTime,
  NominalDiffTime,
  TimeOfDay,
  UTCTime,
  ZonedTime,
 )
import Data.Time.Calendar.Month.Compat (Month)
import Data.Time.Calendar.Quarter.Compat (Quarter, QuarterOfYear)
import qualified Data.UUID.Types as UUID
import qualified Data.Vector as V
import Data.Version (Version)
import Data.Void (Void)
import Data.Word (Word16, Word32, Word64, Word8)
import Numeric.Natural (Natural)
import Web.Cookie (SetCookie)
import Web.HttpApiData (ToHttpApiData (toUrlPiece))


data Single


data Associative


data List


-- | All values must reduce to a single value pair, an associative list of keys and values, or a list of values without keys.
data TemplateValue a where
  Single :: ToHttpApiData a => a -> TemplateValue Single
  Associative :: [(TemplateValue Single, TemplateValue Single)] -> TemplateValue Associative
  List :: [TemplateValue Single] -> TemplateValue List


instance Show (TemplateValue a) where
  show (Single s) = "Single " ++ show (toUrlPiece s)
  show (Associative as) = "Associative [" ++ intercalate ", " (map formatTuple as) ++ "]"
   where
    formatTuple (k, v) = "(" ++ show k ++ ", " ++ show v ++ ")"
  show (List s) = "List [" ++ intercalate ", " (map show s) ++ "]"


instance Semigroup (TemplateValue Associative) where
  Associative l <> Associative r = Associative (l <> r)


instance Monoid (TemplateValue Associative) where
  mempty = Associative mempty


instance Semigroup (TemplateValue List) where
  List l <> List r = List (l <> r)


instance Monoid (TemplateValue List) where
  mempty = List []


data WrappedValue where
  WrappedValue :: TemplateValue a -> WrappedValue


-- | A simple wrapper for interpolating Haskell 98 strings into templates.
newtype TemplateString = String {fromString :: String}
  deriving (Read, Show, Eq, S.IsString, ToHttpApiData)


{- | A simple list of key value pairs. Useful when you want to be able to have multiple duplicate
 keys, which 'Map' and 'HashMap' don't support.
-}
newtype AList k v = AList
  { fromAList :: [(k, v)]
  }
  deriving (Read, Show, Eq, Ord, Semigroup, Monoid)


instance Functor (AList k) where
  fmap f (AList l) = AList $ fmap (fmap f) l


instance Bifunctor AList where
  bimap f g (AList l) = AList $ map (bimap f g) l


class ToTemplateValue a where
  type TemplateRep a :: Type
  type TemplateRep a = Single
  toTemplateValue :: a -> TemplateValue (TemplateRep a)
  default toTemplateValue :: (ToHttpApiData a, TemplateRep a ~ Single) => a -> TemplateValue (TemplateRep a)
  toTemplateValue = Single


{- | A newtype wrapper for deriving 'ToTemplateValue' instances via 'ToHttpApiData'.

This allows you to easily derive 'ToTemplateValue' instances for your custom types
that already have a 'ToHttpApiData' instance using @DerivingVia@.

Example usage:

@
newtype MyId = MyId Int
  deriving (ToHttpApiData) via Int
  deriving (ToTemplateValue) via (ViaHttpApiData MyId)
@

The derived instance will use the 'ToHttpApiData' instance to convert the value
to a 'Single' template value.
-}
newtype ViaHttpApiData a = ViaHttpApiData a


instance (ToHttpApiData a) => ToTemplateValue (ViaHttpApiData a) where
  type TemplateRep (ViaHttpApiData a) = Single
  toTemplateValue (ViaHttpApiData a) = Single a


instance ToTemplateValue Bool


instance ToTemplateValue Char


instance ToTemplateValue Double


instance ToTemplateValue Float


instance ToTemplateValue Int


instance ToTemplateValue Int8


instance ToTemplateValue Int16


instance ToTemplateValue Int32


instance ToTemplateValue Int64


instance ToTemplateValue Integer


instance ToTemplateValue Ordering


instance ToTemplateValue Natural


instance ToTemplateValue Word


instance ToTemplateValue Word8


instance ToTemplateValue Word16


instance ToTemplateValue Word32


instance ToTemplateValue Word64


instance ToTemplateValue ()


instance ToTemplateValue T.Text


instance ToTemplateValue TL.Text


instance ToTemplateValue TemplateString


instance ToTemplateValue Void


instance ToTemplateValue Version


instance ToTemplateValue All


instance ToTemplateValue Any


instance ToTemplateValue UTCTime


instance ToTemplateValue SetCookie


instance ToTemplateValue UUID.UUID


-- TODO these can probably be provided for things where TemplateRep a isn't Single too...
instance (ToTemplateValue a, TemplateRep a ~ Single, ToHttpApiData a) => ToTemplateValue (Min a)


instance (ToTemplateValue a, TemplateRep a ~ Single, ToHttpApiData a) => ToTemplateValue (Max a)


instance (ToTemplateValue a, TemplateRep a ~ Single, ToHttpApiData a) => ToTemplateValue (First a)


instance (ToTemplateValue a, TemplateRep a ~ Single, ToHttpApiData a) => ToTemplateValue (Last a)


instance (ToTemplateValue a, TemplateRep a ~ Single, ToHttpApiData a) => ToTemplateValue (Identity a)


instance (ToTemplateValue a, TemplateRep a ~ Single, ToHttpApiData a) => ToTemplateValue (Dual a)


instance (ToTemplateValue a, TemplateRep a ~ Single, ToHttpApiData a) => ToTemplateValue (Sum a)


instance (ToTemplateValue a, TemplateRep a ~ Single, ToHttpApiData a) => ToTemplateValue (Product a)


instance (ToTemplateValue a, TemplateRep a ~ Single, ToHttpApiData a, HasResolution a) => ToTemplateValue (Fixed a)


instance (ToTemplateValue a, TemplateRep a ~ Single, ToHttpApiData a) => ToTemplateValue (Const a b)


instance (ToTemplateValue a, TemplateRep a ~ Single, ToHttpApiData a) => ToTemplateValue (Tagged b a) where
  toTemplateValue = toTemplateValue . unTagged


instance ToTemplateValue ZonedTime


instance ToTemplateValue LocalTime


instance ToTemplateValue TimeOfDay


instance ToTemplateValue NominalDiffTime


instance ToTemplateValue DayOfWeek


instance ToTemplateValue Day


instance ToTemplateValue QuarterOfYear


instance ToTemplateValue Quarter


instance ToTemplateValue Month


instance (ToTemplateValue a, ToTemplateValue b, TemplateRep a ~ Single, TemplateRep b ~ Single, ToHttpApiData a, ToHttpApiData b) => ToTemplateValue (Either a b)


instance (ToTemplateValue a, TemplateRep a ~ Single) => ToTemplateValue [a] where
  type TemplateRep [a] = List
  toTemplateValue = List . map toTemplateValue


instance (ToTemplateValue a, TemplateRep a ~ Single) => ToTemplateValue (NE.NonEmpty a) where
  type TemplateRep (NE.NonEmpty a) = List
  toTemplateValue = toTemplateValue . NE.toList


instance (ToTemplateValue k, TemplateRep k ~ Single, ToTemplateValue v, TemplateRep v ~ Single) => ToTemplateValue (AList k v) where
  type TemplateRep (AList k v) = Associative
  toTemplateValue = Associative . map (toTemplateValue *** toTemplateValue) . fromAList


instance (ToTemplateValue a, TemplateRep a ~ Single) => ToTemplateValue (V.Vector a) where
  type TemplateRep (V.Vector a) = List
  toTemplateValue = List . F.toList . fmap toTemplateValue


instance (ToTemplateValue a, TemplateRep a ~ Single, ToHttpApiData a) => ToTemplateValue (Maybe a)


instance (ToTemplateValue k, TemplateRep k ~ Single, ToTemplateValue v, TemplateRep v ~ Single) => ToTemplateValue (HS.HashMap k v) where
  type TemplateRep (HS.HashMap k v) = Associative
  toTemplateValue = toTemplateValue . AList . HS.toList


instance (ToTemplateValue k, TemplateRep k ~ Single, ToTemplateValue v, TemplateRep v ~ Single) => ToTemplateValue (MS.Map k v) where
  type TemplateRep (MS.Map k v) = Associative
  toTemplateValue = toTemplateValue . AList . MS.toList


instance
  ( ToTemplateValue a
  , TemplateRep a ~ Single
  , ToTemplateValue b
  , TemplateRep b ~ Single
  ) =>
  ToTemplateValue (a, b)
  where
  type TemplateRep (a, b) = List
  toTemplateValue (a, b) =
    List
      [ toTemplateValue a
      , toTemplateValue b
      ]


instance
  ( ToTemplateValue a
  , TemplateRep a ~ Single
  , ToTemplateValue b
  , TemplateRep b ~ Single
  , ToTemplateValue c
  , TemplateRep c ~ Single
  ) =>
  ToTemplateValue (a, b, c)
  where
  type TemplateRep (a, b, c) = List
  toTemplateValue (a, b, c) =
    List
      [ toTemplateValue a
      , toTemplateValue b
      , toTemplateValue c
      ]


instance
  ( ToTemplateValue a
  , TemplateRep a ~ Single
  , ToTemplateValue b
  , TemplateRep b ~ Single
  , ToTemplateValue c
  , TemplateRep c ~ Single
  , ToTemplateValue d
  , TemplateRep d ~ Single
  ) =>
  ToTemplateValue (a, b, c, d)
  where
  type TemplateRep (a, b, c, d) = List
  toTemplateValue (a, b, c, d) =
    List
      [ toTemplateValue a
      , toTemplateValue b
      , toTemplateValue c
      , toTemplateValue d
      ]


instance
  ( ToTemplateValue a
  , TemplateRep a ~ Single
  , ToTemplateValue b
  , TemplateRep b ~ Single
  , ToTemplateValue c
  , TemplateRep c ~ Single
  , ToTemplateValue d
  , TemplateRep d ~ Single
  , ToTemplateValue e
  , TemplateRep e ~ Single
  ) =>
  ToTemplateValue (a, b, c, d, e)
  where
  type TemplateRep (a, b, c, d, e) = List
  toTemplateValue (a, b, c, d, e) =
    List
      [ toTemplateValue a
      , toTemplateValue b
      , toTemplateValue c
      , toTemplateValue d
      , toTemplateValue e
      ]


instance
  ( ToTemplateValue a
  , TemplateRep a ~ Single
  , ToTemplateValue b
  , TemplateRep b ~ Single
  , ToTemplateValue c
  , TemplateRep c ~ Single
  , ToTemplateValue d
  , TemplateRep d ~ Single
  , ToTemplateValue e
  , TemplateRep e ~ Single
  , ToTemplateValue f
  , TemplateRep f ~ Single
  ) =>
  ToTemplateValue (a, b, c, d, e, f)
  where
  type TemplateRep (a, b, c, d, e, f) = List
  toTemplateValue (a, b, c, d, e, f) =
    List
      [ toTemplateValue a
      , toTemplateValue b
      , toTemplateValue c
      , toTemplateValue d
      , toTemplateValue e
      , toTemplateValue f
      ]


instance
  ( ToTemplateValue a
  , TemplateRep a ~ Single
  , ToTemplateValue b
  , TemplateRep b ~ Single
  , ToTemplateValue c
  , TemplateRep c ~ Single
  , ToTemplateValue d
  , TemplateRep d ~ Single
  , ToTemplateValue e
  , TemplateRep e ~ Single
  , ToTemplateValue f
  , TemplateRep f ~ Single
  , ToTemplateValue g
  , TemplateRep g ~ Single
  ) =>
  ToTemplateValue (a, b, c, d, e, f, g)
  where
  type TemplateRep (a, b, c, d, e, f, g) = List
  toTemplateValue (a, b, c, d, e, f, g) =
    List
      [ toTemplateValue a
      , toTemplateValue b
      , toTemplateValue c
      , toTemplateValue d
      , toTemplateValue e
      , toTemplateValue f
      , toTemplateValue g
      ]


instance
  ( ToTemplateValue a
  , TemplateRep a ~ Single
  , ToTemplateValue b
  , TemplateRep b ~ Single
  , ToTemplateValue c
  , TemplateRep c ~ Single
  , ToTemplateValue d
  , TemplateRep d ~ Single
  , ToTemplateValue e
  , TemplateRep e ~ Single
  , ToTemplateValue f
  , TemplateRep f ~ Single
  , ToTemplateValue g
  , TemplateRep g ~ Single
  , ToTemplateValue h
  , TemplateRep h ~ Single
  ) =>
  ToTemplateValue (a, b, c, d, e, f, g, h)
  where
  type TemplateRep (a, b, c, d, e, f, g, h) = List
  toTemplateValue (a, b, c, d, e, f, g, h) =
    List
      [ toTemplateValue a
      , toTemplateValue b
      , toTemplateValue c
      , toTemplateValue d
      , toTemplateValue e
      , toTemplateValue f
      , toTemplateValue g
      , toTemplateValue h
      ]


instance
  ( ToTemplateValue a
  , TemplateRep a ~ Single
  , ToTemplateValue b
  , TemplateRep b ~ Single
  , ToTemplateValue c
  , TemplateRep c ~ Single
  , ToTemplateValue d
  , TemplateRep d ~ Single
  , ToTemplateValue e
  , TemplateRep e ~ Single
  , ToTemplateValue f
  , TemplateRep f ~ Single
  , ToTemplateValue g
  , TemplateRep g ~ Single
  , ToTemplateValue h
  , TemplateRep h ~ Single
  , ToTemplateValue i
  , TemplateRep i ~ Single
  ) =>
  ToTemplateValue (a, b, c, d, e, f, g, h, i)
  where
  type TemplateRep (a, b, c, d, e, f, g, h, i) = List
  toTemplateValue (a, b, c, d, e, f, g, h, i) =
    List
      [ toTemplateValue a
      , toTemplateValue b
      , toTemplateValue c
      , toTemplateValue d
      , toTemplateValue e
      , toTemplateValue f
      , toTemplateValue g
      , toTemplateValue h
      , toTemplateValue i
      ]


data ValueModifier
  = Normal
  | Explode
  | MaxLength Int
  deriving (Read, Show, Eq)


data Variable = Variable
  { variableName :: T.Text
  , variableValueModifier :: ValueModifier
  }
  deriving (Read, Show, Eq)


data TemplateSegment
  = -- | A literal string. No URI escaping will be performed
    Literal T.Text
  | -- | An interpolation can have multiple variables (separated by commas in the textual format)
    Embed Modifier [Variable]
  deriving (Read, Eq)


instance Show TemplateSegment where
  show (Literal t) = T.unpack t
  show (Embed mod vars) = "{" ++ modifierPrefix mod ++ intercalate "," (map showVariable vars) ++ "}"
    where
      showVariable (Variable name valueMod) = T.unpack name ++ showValueModifier valueMod
      showValueModifier Normal = ""
      showValueModifier Explode = "*"
      showValueModifier (MaxLength n) = ":" ++ show n
      modifierPrefix Simple = ""
      modifierPrefix Reserved = "+"
      modifierPrefix Fragment = "#"
      modifierPrefix Label = "."
      modifierPrefix PathSegment = "/"
      modifierPrefix PathParameter = ";"
      modifierPrefix Query = "?"
      modifierPrefix QueryContinuation = "&"


{- | A URI template is fundamentally a bunch of segments that are either constants
 or else an interpolation
-}
newtype UriTemplate = UriTemplate
  { uriTemplateSegments :: V.Vector TemplateSegment
  }
  deriving (Read, Eq)


instance Show UriTemplate where
  show = renderTemplate


-- | Render a 'UriTemplate' back to its RFC 6570 string representation.
--
-- This is useful for debugging, logging, or storing templates as strings.
--
-- >>> import qualified Data.Vector as V
-- >>> renderTemplate (UriTemplate (V.fromList [Literal "http://example.com/", Embed PathSegment [Variable "path" Normal]]))
-- "http://example.com/{/path}"
renderTemplate :: UriTemplate -> String
renderTemplate = V.foldMap show . uriTemplateSegments


-- | How an interpolated value should be rendered
data Modifier
  = -- | No prefix
    Simple
  | -- | Prefixed by @+@
    Reserved
  | -- | Prefixed by @#@
    Fragment
  | -- | Prefixed by @.@
    Label
  | -- | Prefixed by @/@
    PathSegment
  | -- | Prefixed by @;@
    PathParameter
  | -- | Prefixed by @?@
    Query
  | -- | Prefixed by @&@
    QueryContinuation
  deriving (Read, Show, Eq)


-------------------------------------------------------------------------------
-- Generics-based derivation
-------------------------------------------------------------------------------

-- | Options for generic derivation, matching the TH Options
data GenericOptions = GenericOptions
  { genericFieldLabelModifier :: String -> String
  , genericConstructorTagModifier :: String -> String
  , genericOmitNothingFields :: Bool
  , genericUnwrapUnaryRecords :: Bool
  }

-- | Default options for generic derivation
defaultGenericOptions :: GenericOptions
defaultGenericOptions = GenericOptions
  { genericFieldLabelModifier = id
  , genericConstructorTagModifier = id
  , genericOmitNothingFields = False
  , genericUnwrapUnaryRecords = False
  }

-- | Helper to convert camelCase to snake_case (for use in default options)
camelTo2Generic :: Char -> String -> String
camelTo2Generic _ "" = ""
camelTo2Generic c (x:xs) = toLower x : go xs
  where
    go "" = ""
    go (u:l:rest) | isUpper u && isUpper l = c : toLower u : toLower l : go rest
    go (u:rest) | isUpper u = c : toLower u : go rest
    go (l:rest) = toLower l : go rest

-- | Generic implementation class (only works with record types)
class GToTemplateValue f where
  type GTemplateRep f :: Type
  gToTemplateValue' :: GenericOptions -> f a -> TemplateValue (GTemplateRep f)

-- | Metadata: datatype name
instance (GToTemplateValue f) => GToTemplateValue (D1 c f) where
  type GTemplateRep (D1 c f) = GTemplateRep f
  gToTemplateValue' opts (M1 x) = gToTemplateValue' opts x

-- | Constructor with record fields - always produces Associative
instance (GToTemplateValueRecord f) => GToTemplateValue (C1 c f) where
  type GTemplateRep (C1 c f) = Associative
  gToTemplateValue' opts (M1 x) =
    let pairs = gToTemplateValueRecord opts x
    in Associative pairs

-- | Helper class for record fields
class GToTemplateValueRecord f where
  gToTemplateValueRecord :: GenericOptions -> f a -> [(TemplateValue Single, TemplateValue Single)]

-- | Single field selector
instance (Selector s, ToTemplateValue a, TemplateRep a ~ Single) => GToTemplateValueRecord (S1 s (K1 i a)) where
  gToTemplateValueRecord opts m@(M1 (K1 x)) =
    let fieldName = selName m
        modifiedName = genericFieldLabelModifier opts fieldName
        key = Single (T.pack modifiedName)
        value = toTemplateValue x
    in [(key, value)]

-- | Product of fields (multiple fields)
instance (GToTemplateValueRecord f, GToTemplateValueRecord g) => GToTemplateValueRecord (f :*: g) where
  gToTemplateValueRecord opts (f :*: g) =
    gToTemplateValueRecord opts f ++ gToTemplateValueRecord opts g

-- | Generic to TemplateValue with default options
gToTemplateValue :: (Generic a, GToTemplateValue (Rep a)) => a -> TemplateValue (GTemplateRep (Rep a))
gToTemplateValue = gToTemplateValueWith defaultGenericOptions

-- | Generic to TemplateValue with custom options
gToTemplateValueWith :: (Generic a, GToTemplateValue (Rep a)) => GenericOptions -> a -> TemplateValue (GTemplateRep (Rep a))
gToTemplateValueWith opts = gToTemplateValue' opts . from
