{-# LANGUAGE DeriveGeneric #-}
{-# LANGUAGE OverloadedStrings #-}
{-# LANGUAGE RecordWildCards #-}

-- | Divine with zhouyi
module Game.ZY.Divination where

import Data.ByteString (ByteString)
import Data.Foldable (toList)
import Data.Maybe (catMaybes)
import Data.Sequence (Seq (..), findIndexL, index)
import Data.Text (Text)
import Data.Yaml (FromJSON, decodeEither')
import GHC.Generics (Generic)
import System.Random (StdGen, mkStdGen, randomR)

import Data.Text qualified as T

type Gua = Seq Int
data ZY = ZY
  { symbol :: Gua
  , name :: Text
  , content :: Seq Text
  }
  deriving (Generic, Show)

instance FromJSON ZY

type ZYType = Seq ZY

data Req = Req
  { quiet :: Bool
  , yao :: Bool
  , zhouyiText :: ByteString
  , seed :: StdGen
  }

divine ::
  Req ->
  -- | output info. Left value is the output string. Right value is the error string.
  Either Text Text
divine (Req {..}) = do
  zy <- parseDE $ decodeEither' zhouyiText
  idx <- readGuaNumber g' zy
  let ZY {symbol = symbol', name = name', content = content'} = zy `index` idx
  let idxShow = Just $ T.concat ["卦序：\t", T.show (succ idx), "（1-64）"]
  let nameShow = Just $ T.concat ["卦名：\t", name']
  let genShow =
        if isQY
          then
            Just $
              T.concat ["爻：\t", T.intercalate " " $ toList $ T.show <$> g]
          else Nothing
  let origShow =
        if isQY
          then
            Just $
              T.concat ["本卦：\t", T.intercalate " " $ toList $ T.show <$> g']
          else Nothing
  let zhiShow =
        if isQY
          then
            Just $
              T.concat
                [ "之卦：\t"
                , T.intercalate " " $ toList $ T.show <$> convertToZhiGua g'
                ]
          else Nothing
  let textShow =
        if quiet
          then
            Just $
              T.concat ["卦辞：\n\t", T.intercalate "\n\t" $ toList content']
          else Nothing
  pure $
    T.unlines $
      catMaybes [idxShow, nameShow, genShow, origShow, zhiShow, textShow]
  where
    parseDE (Right z) = pure z
    parseDE (Left e) = Left $ T.show e
    isQY = (quiet && yao) || not quiet
    g = generateGua seed
    g' = convertToGua g

-- | Generate one 爻 with 三变 method
generateYao ::
  -- | seed
  StdGen ->
  -- | return 9, 8, 7, 6.
  (Int, StdGen)
generateYao = randomR (6, 9)

-- | generate 卦
generateGua :: StdGen -> Gua
generateGua = generateGua' Empty
  where
    generateGua' g s =
      let (g', s') = generateYao s
       in if length g < 6
            then generateGua' (g' :<| g) s'
            else g

-- | convert to 本卦.
convertToGua ::
  -- | origin six Yao number
  Gua ->
  -- | convert six Yao number
  Gua
convertToGua = fmap (\y -> if y `mod` 2 == 1 then 9 else 6)

-- | convert to 之卦
convertToZhiGua ::
  -- | origin six Yao number
  Gua ->
  -- | convert six Yao number
  Gua
convertToZhiGua =
  convertToGua
    . fmap
      ( \y ->
          case y of
            9 -> 6
            6 -> 9
            _ -> y
      )

-- | read gua's index from zhouyi
readGuaNumber ::
  -- | converted gua
  Gua ->
  -- | data for zy
  ZYType ->
  -- | number of the gua
  Either Text Int
readGuaNumber g zy = maybe errorLeft return find
  where
    errorLeft = Left $ "Not Found Gua: " `T.append` T.show g
    find = findIndexL eq' zy
    eq' (ZY {symbol}) = g == symbol
