{-|

Copyright:
  This file is part of the package zxcvbn-hs. It is subject to the
  license terms in the LICENSE file found in the top-level directory
  of this distribution and at:

    https://code.devalot.com/sthenauth/zxcvbn-hs

  No part of this package, including this file, may be copied,
  modified, propagated, or distributed except according to the terms
  contained in the LICENSE file.

License: MIT

-}
module Zxcvbn.Encode
  ( header
  , encode
  ) where

--------------------------------------------------------------------------------
-- Imports:
import Data.Binary (Binary)
import Data.Binary.Instances ()
import qualified Codec.Compression.GZip as GZip
import qualified Data.Binary as Binary
import qualified Data.ByteString.Base64.Lazy as Base64

--------------------------------------------------------------------------------
type ModuleName = String
type FunctionName = String
type TypeSig = String

--------------------------------------------------------------------------------
-- | A file header for code using this module.
header :: ModuleName -> String
header m =
  mconcat [ "-- This file is automatically generated, DO NOT EDIT!\n"
          , "{-# LANGUAGE OverloadedStrings #-}\n"
          , "{-# OPTIONS_GHC -Wno-missing-export-lists #-}\n"
          , "{-# OPTIONS_GHC -Wno-unused-imports #-}\n"
          , "module " <> m <> " where\n"
          , "import Data.HashMap.Strict (HashMap)\n"
          , "import qualified Data.HashMap.Strict as HashMap\n"
          , "import Data.Text (Text)\n"
          , "import qualified Codec.Compression.GZip as GZip\n"
          , "import qualified Data.ByteString.Base64.Lazy as Base64\n"
          , "import qualified Data.Binary as Binary\n"
          , "import Data.Binary.Instances ()\n"
          ]

--------------------------------------------------------------------------------
-- | Encode a value and bundle it with a function that can decode it.
encode :: (Binary a) => FunctionName -> TypeSig -> a -> String
encode n t a =
  mconcat [ n <>  " :: " <> t <> "\n"
          , n <> " =\n"
          , "  let decode = " <> decode <> "\n"
          , "  in  decode "
          , show . Base64.encode . GZip.compress . Binary.encode $ a
          , "\n"
          ]
  where
    decode =
      mconcat [ "Binary.decode . "
              , "GZip.decompress . "
              , "Base64.decodeLenient"
              ]
